<?php

namespace Piwik\Plugins\IP2Location\LocationProvider;

use Piwik\Http;
use Piwik\Option;
use Piwik\Piwik;
use Piwik\Plugins\UserCountry\LocationProvider;

class IP2Location extends LocationProvider
{
	public const ID = 'ip2location';
	public const TITLE = 'IP2Location';

	public function __construct()
	{
	}

	/**
	 * Returns information about this location provider.
	 *
	 * @return array
	 */
	public function getInfo()
	{
		$extraMessage = '';

		if (Option::get('IP2Location.LookupMode') == 'IO') {
			$extraMessage = '
				<strong>Lookup Mode: </strong><a href="https://www.ip2location.io" target="_blank">IP2Location.io</a> IP Geolocation Web Service<br/>
				<strong>API Key: </strong>' . Option::get('IP2Location.IOAPIKey');
		} else {
			if ($this->getDatabasePath()) {
				$extraMessage = '
				<strong>Lookup Mode: </strong>BIN Database<br/>
				<strong>Database File: </strong>' . basename($this->getDatabasePath());
			}
		}

		return [
			'id'            => self::ID,
			'title'         => self::TITLE,
			'order'         => 2,
			'description'   => Piwik::translate('IP2Location_InfoDescription'),
			'install_docs'  => 'For BIN database option, please upload IP2Location BIN database file into <strong>Matomo/misc</strong> folder.',
			'extra_message' => $extraMessage,
		];
	}

	/**
	 * Get a visitor's location based on their IP address.
	 *
	 * @param array $info must have an 'ip' field
	 *
	 * @return array
	 */
	public function getLocation($info)
	{
		$ip = $this->getIpFromInfo($info);

		$result = [];

		if (Option::get('IP2Location.LookupMode') == 'WS' && Option::get('IP2Location.APIKey')) {
			$response = Http::sendHttpRequest('https://api.ip2location.com/v2/?key=' . Option::get('IP2Location.APIKey') . '&ip=' . $ip . '&format=json&package=WS6', 30);

			if (($json = json_decode($response)) !== null) {
				if ($json->response == 'OK') {
					$result[self::COUNTRY_CODE_KEY] = $json->country_code;
					$result[self::COUNTRY_NAME_KEY] = $json->country_name;
					$result[self::REGION_CODE_KEY] = $this->getRegionCode($json->country_code, $json->region_name);
					$result[self::REGION_NAME_KEY] = $json->region_name;
					$result[self::CITY_NAME_KEY] = $json->city_name;
					$result[self::LATITUDE_KEY] = $json->latitude;
					$result[self::LONGITUDE_KEY] = $json->longitude;
					$result[self::ISP_KEY] = $json->isp;
				}
			}
		} elseif (Option::get('IP2Location.LookupMode') == 'IO' && Option::get('IP2Location.IOAPIKey')) {
			$response = Http::sendHttpRequest('https://api.ip2location.io/?' . http_build_query([
				'key'    => Option::get('IP2Location.IOAPIKey'),
				'ip'     => $ip,
				'source' => 'matomo',
			]), 30);

			if (($json = json_decode((string) $response)) !== null) {
				$result[self::COUNTRY_CODE_KEY] = $json->country_code;
				$result[self::COUNTRY_NAME_KEY] = $json->country_name;
				$result[self::REGION_CODE_KEY] = $this->getRegionCode($json->country_code, $json->region_name);
				$result[self::REGION_NAME_KEY] = $json->region_name;
				$result[self::CITY_NAME_KEY] = $json->city_name;
				$result[self::LATITUDE_KEY] = $json->latitude;
				$result[self::LONGITUDE_KEY] = $json->longitude;
				$result[self::ISP_KEY] = $json->as;
			}
		} else {
			require_once PIWIK_INCLUDE_PATH . '/plugins/IP2Location/lib/IP2Location.php';

			if (self::getDatabasePath() === false) {
				return;
			}

			$db = new \IP2Location\Database(self::getDatabasePath(), \IP2Location\Database::FILE_IO);
			$response = $db->lookup($ip, \IP2Location\Database::ALL);

			if (!$response) {
				return;
			}

			$result[self::COUNTRY_CODE_KEY] = $response['countryCode'];
			$result[self::COUNTRY_NAME_KEY] = $response['countryName'];

			if (strpos($response['regionName'], 'unavailable') === false) {
				$result[self::REGION_CODE_KEY] = $this->getRegionCode($response['countryCode'], $response['regionName']);
				$result[self::REGION_NAME_KEY] = $response['regionName'];
			}

			if (strpos($response['cityName'], 'unavailable') === false) {
				$result[self::CITY_NAME_KEY] = $response['cityName'];
			}

			if (strpos($response['latitude'], 'unavailable') === false) {
				$result[self::LATITUDE_KEY] = $response['latitude'];
				$result[self::LONGITUDE_KEY] = $response['longitude'];
			}

			if (strpos($response['isp'], 'unavailable') === false) {
				$result[self::ISP_KEY] = $response['isp'];
			}
		}

		$this->completeLocationResult($result);

		return $result;
	}

	/**
	 * Returns an array describing the types of location information this provider will
	 * return.
	 *
	 * @return array
	 */
	public function getSupportedLocationInfo()
	{
		$result = [];

		// Country & continent information always available
		$result[self::COUNTRY_CODE_KEY] = true;
		$result[self::COUNTRY_NAME_KEY] = true;
		$result[self::CONTINENT_CODE_KEY] = true;
		$result[self::CONTINENT_NAME_KEY] = true;

		// All fields are supported with IP2Location WS
		if (Option::get('IP2Location.LookupMode') == 'WS' && Option::get('IP2Location.APIKey')) {
			$result[self::REGION_CODE_KEY] = true;
			$result[self::REGION_NAME_KEY] = true;
			$result[self::CITY_NAME_KEY] = true;
			$result[self::LATITUDE_KEY] = true;
			$result[self::LONGITUDE_KEY] = true;
			$result[self::ISP_KEY] = true;

			return $result;
		}

		// All fields are supported with IP2Location.io Service
		if (Option::get('IP2Location.LookupMode') == 'IO' && Option::get('IP2Location.IOAPIKey')) {
			$result[self::REGION_CODE_KEY] = true;
			$result[self::REGION_NAME_KEY] = true;
			$result[self::CITY_NAME_KEY] = true;
			$result[self::LATITUDE_KEY] = true;
			$result[self::LONGITUDE_KEY] = true;
			$result[self::ISP_KEY] = true;

			return $result;
		}

		require_once PIWIK_INCLUDE_PATH . '/plugins/IP2Location/lib/IP2Location.php';

		$db = new \IP2Location\Database(self::getDatabasePath(), \IP2Location\Database::FILE_IO);
		$response = $db->lookup('8.8.8.8', \IP2Location\Database::ALL);

		if (strpos($response['regionName'], 'unavailable') === false) {
			$result[self::REGION_CODE_KEY] = true;
			$result[self::REGION_NAME_KEY] = true;
		}

		if (strpos($response['cityName'], 'unavailable') === false) {
			$result[self::CITY_NAME_KEY] = true;
		}

		if (strpos($response['latitude'], 'unavailable') === false) {
			$result[self::LATITUDE_KEY] = true;
			$result[self::LONGITUDE_KEY] = true;
		}

		if (strpos($response['isp'], 'unavailable') === false) {
			$result[self::ISP_KEY] = true;
		}

		return $result;
	}

	/**
	 * Returns true if this location provider is available.
	 *
	 * @return bool
	 */
	public function isAvailable()
	{
		switch (Option::get('IP2Location.LookupMode')) {
			case 'BIN':
				return self::getDatabasePath() !== false;

			case 'WS':
				return (bool) preg_match('/^[0-9A-Z]{10}$/', (string) Option::get('IP2Location.APIKey'));

			case 'IO':
				return (bool) preg_match('/^[0-9A-Z]{32}$/', Option::get('IP2Location.IOAPIKey'));
		}

		return false;
	}

	/**
	 * Returns true if this provider has been setup correctly, the error message if
	 * otherwise.
	 *
	 * @return bool|string
	 */
	public function isWorking()
	{
		switch (Option::get('IP2Location.LookupMode')) {
			case 'BIN':
				require_once PIWIK_INCLUDE_PATH . '/plugins/IP2Location/lib/IP2Location.php';

				if (!file_exists(self::getDatabasePath())) {
					return 'IP2Location BIN database not found.';
				}

				$db = new \IP2Location\Database(self::getDatabasePath(), \IP2Location\Database::FILE_IO);
				$response = $db->lookup('8.8.8.8', \IP2Location\Database::ALL);

				if (!isset($response['countryCode'])) {
					return 'IP2Location BIN database is corrupted.';
				}

				if (strpos($response['countryCode'], 'Invalid') !== false) {
					return 'IP2Location BIN database is invalid.';
				}

				return true;

			case 'WS':
				return (bool) preg_match('/^[0-9A-Z]{10}$/', Option::get('IP2Location.APIKey'));

			case 'IO':
				return (bool) preg_match('/^[0-9A-Z]{32}$/', Option::get('IP2Location.IOAPIKey'));
		}

		return false;
	}

	/**
	 * Returns full path for a IP2Location database file.
	 *
	 * @return string
	 */
	public static function getDatabasePath()
	{
		$databasePath = Option::get('IP2Location.DatabasePath');

		if (empty($databasePath)) {
			$databasePath = PIWIK_DOCUMENT_ROOT . '/misc/' . Option::get('IP2Location.BIN');
		}

		return Option::get('IP2Location.DatabasePath');
	}

	/**
	 * Get region code by country code and region name.
	 *
	 * @return false|string
	 */
	private function getRegionCode($countryCode, $regionName)
	{
		$regions = [
			'AD' => ['07' => 'Andorra la Vella', '02' => 'Canillo', '03' => 'Encamp', '08' => 'Escaldes-Engordany', '04' => 'La Massana', '05' => 'Ordino', '06' => 'Sant Julia de Loria'], 	'AE' => ['AJ' => '\'Ajman', 'AZ' => 'Abu Zaby', 'FU' => 'Al Fujayrah', 'SH' => 'Ash Shariqah', 'DU' => 'Dubayy', 'RK' => 'Ra\'s al Khaymah', 'UQ' => 'Umm al Qaywayn'], 	'AF' => ['BDS' => 'Badakhshan', 'BDG' => 'Badghis', 'BGL' => 'Baghlan', 'BAL' => 'Balkh', 'BAM' => 'Bamyan', 'DAY' => 'Daykundi', 'FRA' => 'Farah', 'FYB' => 'Faryab', 'GHA' => 'Ghazni', 'GHO' => 'Ghor', 'HEL' => 'Helmand', 'HER' => 'Herat', 'JOW' => 'Jowzjan', 'KAB' => 'Kabul', 'KAN' => 'Kandahar', 'KHO' => 'Khost', 'KDZ' => 'Kunduz', 'LAG' => 'Laghman', 'LOG' => 'Logar', 'NAN' => 'Nangarhar', 'NIM' => 'Nimroz', 'PKA' => 'Paktika', 'PIA' => 'Paktiya', 'PAR' => 'Parwan', 'SAM' => 'Samangan', 'SAR' => 'Sar-e Pul', 'TAK' => 'Takhar', 'URU' => 'Uruzgan'], 	'AG' => ['10' => 'Barbuda', '03' => 'Saint George', '04' => 'Saint John', '05' => 'Saint Mary', '06' => 'Saint Paul', '07' => 'Saint Peter', '08' => 'Saint Philip'], 	'AI' => ['-' => 'Anguilla'], 	'AL' => ['01' => 'Berat', '09' => 'Diber', '02' => 'Durres', '03' => 'Elbasan', '04' => 'Fier', '05' => 'Gjirokaster', '06' => 'Korce', '07' => 'Kukes', '08' => 'Lezhe', '10' => 'Shkoder', '11' => 'Tirane', '12' => 'Vlore'], 	'AM' => ['AG' => 'Aragacotn', 'AR' => 'Ararat', 'AV' => 'Armavir', 'ER' => 'Erevan', 'GR' => 'Gegark\'unik\'', 'KT' => 'Kotayk\'', 'LO' => 'Lori', 'SH' => 'Sirak', 'SU' => 'Syunik\'', 'TV' => 'Tavus', 'VD' => 'Vayoc Jor'], 	'AO' => ['BGO' => 'Bengo', 'BGU' => 'Benguela', 'BIE' => 'Bie', 'CAB' => 'Cabinda', 'CCU' => 'Cuando Cubango', 'CNO' => 'Cuanza-Norte', 'CUS' => 'Cuanza-Sul', 'HUA' => 'Huambo', 'HUI' => 'Huila', 'LUA' => 'Luanda', 'LNO' => 'Lunda-Norte', 'LSU' => 'Lunda-Sul', 'MAL' => 'Malange', 'MOX' => 'Moxico', 'NAM' => 'Namibe', 'UIG' => 'Uige', 'ZAI' => 'Zaire'], 	'AQ' => ['-' => 'Antarctica'], 	'AR' => ['B' => 'Buenos Aires', 'K' => 'Catamarca', 'H' => 'Chaco', 'U' => 'Chubut', 'C' => 'Ciudad Autonoma de Buenos Aires', 'X' => 'Cordoba', 'W' => 'Corrientes', 'E' => 'Entre Rios', 'P' => 'Formosa', 'Y' => 'Jujuy', 'L' => 'La Pampa', 'F' => 'La Rioja', 'M' => 'Mendoza', 'N' => 'Misiones', 'Q' => 'Neuquen', 'R' => 'Rio Negro', 'A' => 'Salta', 'J' => 'San Juan', 'D' => 'San Luis', 'Z' => 'Santa Cruz', 'S' => 'Santa Fe', 'G' => 'Santiago del Estero', 'V' => 'Tierra del Fuego', 'T' => 'Tucuman'], 	'AS' => ['-' => 'Western District'], 	'AT' => ['1' => 'Burgenland', '2' => 'Karnten', '3' => 'Niederosterreich', '4' => 'Oberosterreich', '5' => 'Salzburg', '6' => 'Steiermark', '7' => 'Tirol', '8' => 'Vorarlberg', '9' => 'Wien'], 	'AU' => ['ACT' => 'Australian Capital Territory', 'NSW' => 'New South Wales', 'NT' => 'Northern Territory', 'QLD' => 'Queensland', 'SA' => 'South Australia', 'TAS' => 'Tasmania', 'VIC' => 'Victoria', 'WA' => 'Western Australia'], 	'AW' => ['-' => 'Aruba'], 	'AX' => ['-' => 'Saltvik'], 	'AZ' => ['ABS' => 'Abseron', 'AGC' => 'Agcabadi', 'AGS' => 'Agdas', 'AGA' => 'Agstafa', 'AGU' => 'Agsu', 'AST' => 'Astara', 'BA' => 'Baki', 'BAL' => 'Balakan', 'BAR' => 'Barda', 'BEY' => 'Beylaqan', 'CAL' => 'Calilabad', 'GAD' => 'Gadabay', 'GA' => 'Ganca', 'GOR' => 'Goranboy', 'GOY' => 'Goycay', 'GYG' => 'Goygol', 'IMI' => 'Imisli', 'ISM' => 'Ismayilli', 'KUR' => 'Kurdamir', 'LA' => 'Lankaran', 'MAS' => 'Masalli', 'MI' => 'Mingacevir', 'NX' => 'Naxcivan', 'NEF' => 'Neftcala', 'OGU' => 'Oguz', 'QAB' => 'Qabala', 'QAZ' => 'Qazax', 'QBA' => 'Quba', 'QUS' => 'Qusar', 'SAT' => 'Saatli', 'SAB' => 'Sabirabad', 'SAK' => 'Saki', 'SAL' => 'Salyan', 'SMI' => 'Samaxi', 'SKR' => 'Samkir', 'SMX' => 'Samux', 'SR' => 'Sirvan', 'SIY' => 'Siyazan', 'SM' => 'Sumqayit', 'TAR' => 'Tartar', 'TOV' => 'Tovuz', 'UCA' => 'Ucar', 'XAC' => 'Xacmaz', 'XIZ' => 'Xizi', 'YAR' => 'Yardimli', 'YEV' => 'Yevlax', 'ZAQ' => 'Zaqatala', 'ZAR' => 'Zardab'], 	'BA' => ['BRC' => 'Brcko distrikt', 'BIH' => 'Federacija Bosne i Hercegovine', 'SRP' => 'Republika Srpska'], 	'BB' => ['01' => 'Christ Church', '02' => 'Saint Andrew', '03' => 'Saint George', '04' => 'Saint James', '05' => 'Saint John', '06' => 'Saint Joseph', '07' => 'Saint Lucy', '08' => 'Saint Michael', '09' => 'Saint Peter', '10' => 'Saint Philip', '11' => 'Saint Thomas'], 	'BD' => ['A' => 'Barishal', 'B' => 'Chattogram', 'C' => 'Dhaka', 'D' => 'Khulna', 'E' => 'Rajshahi', 'F' => 'Rangpur', 'G' => 'Sylhet'], 	'BE' => ['VAN' => 'Antwerpen', 'WBR' => 'Brabant wallon', 'BRU' => 'Brussels Hoofdstedelijk Gewest', 'WHT' => 'Hainaut', 'WLG' => 'Liege', 'VLI' => 'Limburg', 'WLX' => 'Luxembourg', 'WNA' => 'Namur', 'VOV' => 'Oost-Vlaanderen', 'VBR' => 'Vlaams-Brabant', 'VWV' => 'West-Vlaanderen'], 	'BF' => ['BAL' => 'Bale', 'BAM' => 'Bam', 'BAN' => 'Banwa', 'BAZ' => 'Bazega', 'BGR' => 'Bougouriba', 'BLG' => 'Boulgou', 'BLK' => 'Boulkiemde', 'COM' => 'Comoe', 'GAN' => 'Ganzourgou', 'GNA' => 'Gnagna', 'GOU' => 'Gourma', 'HOU' => 'Houet', 'KAD' => 'Kadiogo', 'KEN' => 'Kenedougou', 'KMP' => 'Kompienga', 'KOS' => 'Kossi', 'KOT' => 'Kouritenga', 'KOW' => 'Kourweogo', 'LER' => 'Leraba', 'LOR' => 'Loroum', 'MOU' => 'Mouhoun', 'NAO' => 'Nahouri', 'NAM' => 'Namentenga', 'NAY' => 'Nayala', 'OUB' => 'Oubritenga', 'OUD' => 'Oudalan', 'PAS' => 'Passore', 'SMT' => 'Sanmatenga', 'SEN' => 'Seno', 'SIS' => 'Sissili', 'SOM' => 'Soum', 'SOR' => 'Sourou', 'TAP' => 'Tapoa', 'TUI' => 'Tuy', 'YAT' => 'Yatenga', 'ZIR' => 'Ziro', 'ZON' => 'Zondoma', 'ZOU' => 'Zoundweogo'], 	'BG' => ['01' => 'Blagoevgrad', '02' => 'Burgas', '08' => 'Dobrich', '07' => 'Gabrovo', '26' => 'Haskovo', '09' => 'Kardzhali', '10' => 'Kyustendil', '11' => 'Lovech', '12' => 'Montana', '13' => 'Pazardzhik', '14' => 'Pernik', '15' => 'Pleven', '16' => 'Plovdiv', '17' => 'Razgrad', '18' => 'Ruse', '27' => 'Shumen', '19' => 'Silistra', '20' => 'Sliven', '21' => 'Smolyan', '23' => 'Sofia', '22' => 'Sofia (stolitsa)', '24' => 'Stara Zagora', '25' => 'Targovishte', '03' => 'Varna', '04' => 'Veliko Tarnovo', '05' => 'Vidin', '06' => 'Vratsa', '28' => 'Yambol'], 	'BH' => ['13' => 'Al \'Asimah', '14' => 'Al Janubiyah', '15' => 'Al Muharraq', '17' => 'Ash Shamaliyah'], 	'BI' => ['BM' => 'Bujumbura Mairie', 'BR' => 'Bururi', 'CI' => 'Cibitoke', 'GI' => 'Gitega', 'KI' => 'Kirundo', 'MW' => 'Mwaro', 'NG' => 'Ngozi', 'RM' => 'Rumonge', 'RT' => 'Rutana', 'RY' => 'Ruyigi'], 	'BJ' => ['AK' => 'Atacora', 'AQ' => 'Atlantique', 'BO' => 'Borgou', 'CO' => 'Collines', 'KO' => 'Couffo', 'DO' => 'Donga', 'LI' => 'Littoral', 'MO' => 'Mono', 'OU' => 'Oueme', 'PL' => 'Plateau', 'ZO' => 'Zou'], 	'BL' => ['-' => 'Saint Barthelemy'], 	'BM' => ['-' => 'Saint George'], 	'BN' => ['BE' => 'Belait', 'BM' => 'Brunei-Muara', 'TE' => 'Temburong', 'TU' => 'Tutong'], 	'BO' => ['H' => 'Chuquisaca', 'C' => 'Cochabamba', 'B' => 'El Beni', 'L' => 'La Paz', 'O' => 'Oruro', 'N' => 'Pando', 'P' => 'Potosi', 'S' => 'Santa Cruz', 'T' => 'Tarija'], 	'BQ' => ['BO' => 'Bonaire', 'SA' => 'Saba', 'SE' => 'Sint Eustatius'], 	'BR' => ['AC' => 'Acre', 'AL' => 'Alagoas', 'AP' => 'Amapa', 'AM' => 'Amazonas', 'BA' => 'Bahia', 'CE' => 'Ceara', 'DF' => 'Distrito Federal', 'ES' => 'Espirito Santo', 'GO' => 'Goias', 'MA' => 'Maranhao', 'MT' => 'Mato Grosso', 'MS' => 'Mato Grosso do Sul', 'MG' => 'Minas Gerais', 'PA' => 'Para', 'PB' => 'Paraiba', 'PR' => 'Parana', 'PE' => 'Pernambuco', 'PI' => 'Piaui', 'RN' => 'Rio Grande do Norte', 'RS' => 'Rio Grande do Sul', 'RJ' => 'Rio de Janeiro', 'RO' => 'Rondonia', 'RR' => 'Roraima', 'SC' => 'Santa Catarina', 'SP' => 'Sao Paulo', 'SE' => 'Sergipe', 'TO' => 'Tocantins'], 	'BS' => ['BP' => 'Black Point', 'CO' => 'Central Abaco', 'FP' => 'City of Freeport', 'EG' => 'East Grand Bahama', 'HI' => 'Harbour Island', 'LI' => 'Long Island', 'NP' => 'New Providence', 'NS' => 'North Andros', 'NE' => 'North Eleuthera', 'SS' => 'San Salvador', 'SE' => 'South Eleuthera', 'WG' => 'West Grand Bahama'], 	'BT' => ['33' => 'Bumthang', '12' => 'Chhukha', '22' => 'Dagana', 'GA' => 'Gasa', '44' => 'Lhuentse', '42' => 'Monggar', '11' => 'Paro', '43' => 'Pema Gatshel', '23' => 'Punakha', '45' => 'Samdrup Jongkhar', '14' => 'Samtse', '31' => 'Sarpang', '15' => 'Thimphu', '41' => 'Trashigang', '32' => 'Trongsa', '21' => 'Tsirang', '24' => 'Wangdue Phodrang'], 	'BV' => ['-' => 'Bouvet Island'], 	'BW' => ['CE' => 'Central', 'CH' => 'Chobe', 'GH' => 'Ghanzi', 'KL' => 'Kgatleng', 'KW' => 'Kweneng', 'NE' => 'North East', 'NW' => 'North West', 'SE' => 'South East', 'SO' => 'Southern'], 	'BY' => ['BR' => 'Brestskaya voblasts\'', 'HO' => 'Homyel\'skaya voblasts\'', 'HM' => 'Horad Minsk', 'HR' => 'Hrodzyenskaya voblasts\'', 'MA' => 'Mahilyowskaya voblasts\'', 'MI' => 'Minskaya voblasts\'', 'VI' => 'Vitsyebskaya voblasts\''], 	'BZ' => ['BZ' => 'Belize', 'CY' => 'Cayo', 'CZL' => 'Corozal', 'OW' => 'Orange Walk', 'SC' => 'Stann Creek', 'TOL' => 'Toledo'], 	'CA' => ['AB' => 'Alberta', 'BC' => 'British Columbia', 'MB' => 'Manitoba', 'NB' => 'New Brunswick', 'NL' => 'Newfoundland and Labrador', 'NT' => 'Northwest Territories', 'NS' => 'Nova Scotia', 'NU' => 'Nunavut', 'ON' => 'Ontario', 'PE' => 'Prince Edward Island', 'QC' => 'Quebec', 'SK' => 'Saskatchewan', 'YT' => 'Yukon'], 	'CC' => ['-' => 'Cocos (Keeling) Islands'], 	'CD' => ['EQ' => 'Equateur', 'HK' => 'Haut-Katanga', 'HL' => 'Haut-Lomami', 'HU' => 'Haut-Uele', 'IT' => 'Ituri', 'KS' => 'Kasai', 'KC' => 'Kasai Central', 'KE' => 'Kasai Oriental', 'KN' => 'Kinshasa', 'KL' => 'Kwilu', 'LU' => 'Lualaba', 'MA' => 'Maniema', 'NK' => 'Nord-Kivu', 'NU' => 'Nord-Ubangi', 'SA' => 'Sankuru', 'SK' => 'Sud-Kivu', 'TA' => 'Tanganyika', 'TO' => 'Tshopo'], 	'CF' => ['BB' => 'Bamingui-Bangoran', 'BGF' => 'Bangui', 'KB' => 'Gribingui', 'KG' => 'Kemo-Gribingui', 'LB' => 'Lobaye', 'HS' => 'Mambere-Kadei', 'NM' => 'Nana-Mambere', 'UK' => 'Ouaka', 'AC' => 'Ouham', 'OP' => 'Ouham-Pende', 'SE' => 'Sangha'], 	'CG' => ['11' => 'Bouenza', 'BZV' => 'Brazzaville', '8' => 'Cuvette', '7' => 'Likouala', '9' => 'Niari', '16' => 'Pointe-Noire', '13' => 'Sangha'], 	'CH' => ['AG' => 'Aargau', 'AR' => 'Appenzell Ausserrhoden', 'AI' => 'Appenzell Innerrhoden', 'BL' => 'Basel-Landschaft', 'BS' => 'Basel-Stadt', 'BE' => 'Bern', 'FR' => 'Fribourg', 'GE' => 'Geneve', 'GL' => 'Glarus', 'GR' => 'Graubunden', 'JU' => 'Jura', 'LU' => 'Luzern', 'NE' => 'Neuchatel', 'NW' => 'Nidwalden', 'OW' => 'Obwalden', 'SG' => 'Sankt Gallen', 'SH' => 'Schaffhausen', 'SZ' => 'Schwyz', 'SO' => 'Solothurn', 'TG' => 'Thurgau', 'TI' => 'Ticino', 'UR' => 'Uri', 'VS' => 'Valais', 'VD' => 'Vaud', 'ZG' => 'Zug', 'ZH' => 'Zurich'], 	'CI' => ['AB' => 'Abidjan', 'BS' => 'Bas-Sassandra', 'CM' => 'Comoe', 'DN' => 'Denguele', 'GD' => 'Goh-Djiboua', 'LC' => 'Lacs', 'LG' => 'Lagunes', 'MG' => 'Montagnes', 'SM' => 'Sassandra-Marahoue', 'SV' => 'Savanes', 'VB' => 'Vallee du Bandama', 'WR' => 'Woroba', 'YM' => 'Yamoussoukro', 'ZZ' => 'Zanzan'], 	'CK' => ['-' => 'Cook Islands'], 	'CL' => ['AI' => 'Aisen del General Carlos Ibanez del Campo', 'AN' => 'Antofagasta', 'AP' => 'Arica y Parinacota', 'AT' => 'Atacama', 'BI' => 'Biobio', 'CO' => 'Coquimbo', 'AR' => 'La Araucania', 'LI' => 'Libertador General Bernardo O\'Higgins', 'LL' => 'Los Lagos', 'LR' => 'Los Rios', 'MA' => 'Magallanes', 'ML' => 'Maule', 'NB' => 'Nuble', 'RM' => 'Region Metropolitana de Santiago', 'TA' => 'Tarapaca', 'VS' => 'Valparaiso'], 	'CM' => ['AD' => 'Adamaoua', 'CE' => 'Centre', 'ES' => 'Est', 'EN' => 'Extreme-Nord', 'LT' => 'Littoral', 'NO' => 'Nord', 'NW' => 'Nord-Ouest', 'OU' => 'Ouest', 'SU' => 'Sud', 'SW' => 'Sud-Ouest'], 	'CN' => ['AH' => 'Anhui', 'BJ' => 'Beijing', 'CQ' => 'Chongqing', 'FJ' => 'Fujian', 'GS' => 'Gansu', 'GD' => 'Guangdong', 'GX' => 'Guangxi Zhuangzu', 'GZ' => 'Guizhou', 'HI' => 'Hainan', 'HE' => 'Hebei', 'HL' => 'Heilongjiang', 'HA' => 'Henan', 'HB' => 'Hubei', 'HN' => 'Hunan', 'JS' => 'Jiangsu', 'JX' => 'Jiangxi', 'JL' => 'Jilin', 'LN' => 'Liaoning', 'NM' => 'Nei Mongol', 'NX' => 'Ningxia Huizu', 'QH' => 'Qinghai', 'SN' => 'Shaanxi', 'SD' => 'Shandong', 'SH' => 'Shanghai', 'SX' => 'Shanxi', 'SC' => 'Sichuan', 'TJ' => 'Tianjin', 'XJ' => 'Xinjiang Uygur', 'XZ' => 'Xizang', 'YN' => 'Yunnan', 'ZJ' => 'Zhejiang'], 	'CO' => ['AMA' => 'Amazonas', 'ANT' => 'Antioquia', 'ARA' => 'Arauca', 'ATL' => 'Atlantico', 'BOL' => 'Bolivar', 'BOY' => 'Boyaca', 'CAL' => 'Caldas', 'CAQ' => 'Caqueta', 'CAS' => 'Casanare', 'CAU' => 'Cauca', 'CES' => 'Cesar', 'CHO' => 'Choco', 'COR' => 'Cordoba', 'CUN' => 'Cundinamarca', 'DC' => 'Distrito Capital de Bogota', 'GUA' => 'Guainia', 'GUV' => 'Guaviare', 'HUI' => 'Huila', 'LAG' => 'La Guajira', 'MAG' => 'Magdalena', 'MET' => 'Meta', 'NAR' => 'Narino', 'NSA' => 'Norte de Santander', 'PUT' => 'Putumayo', 'QUI' => 'Quindio', 'RIS' => 'Risaralda', 'SAP' => 'San Andres, Providencia y Santa Catalina', 'SAN' => 'Santander', 'SUC' => 'Sucre', 'TOL' => 'Tolima', 'VAC' => 'Valle del Cauca', 'VID' => 'Vichada'], 	'CR' => ['A' => 'Alajuela', 'C' => 'Cartago', 'G' => 'Guanacaste', 'H' => 'Heredia', 'L' => 'Limon', 'P' => 'Puntarenas', 'SJ' => 'San Jose'], 	'CU' => ['15' => 'Artemisa', '09' => 'Camaguey', '08' => 'Ciego de Avila', '06' => 'Cienfuegos', '12' => 'Granma', '14' => 'Guantanamo', '11' => 'Holguin', '99' => 'Isla de la Juventud', '03' => 'La Habana', '10' => 'Las Tunas', '04' => 'Matanzas', '16' => 'Mayabeque', '01' => 'Pinar del Rio', '07' => 'Sancti Spiritus', '13' => 'Santiago de Cuba', '05' => 'Villa Clara'], 	'CV' => ['BV' => 'Boa Vista', 'PN' => 'Porto Novo', 'PR' => 'Praia', 'RS' => 'Ribeira Grande de Santiago', 'SL' => 'Sal', 'CR' => 'Santa Cruz', 'SS' => 'Sao Salvador do Mundo', 'SV' => 'Sao Vicente', 'TA' => 'Tarrafal'], 	'CW' => ['-' => 'Curacao'], 	'CX' => ['-' => 'Christmas Island'], 	'CY' => ['04' => 'Ammochostos', '06' => 'Keryneia', '03' => 'Larnaka', '01' => 'Lefkosia', '02' => 'Lemesos', '05' => 'Pafos'], 	'CZ' => ['31' => 'Jihocesky kraj', '64' => 'Jihomoravsky kraj', '41' => 'Karlovarsky kraj', '63' => 'Kraj Vysocina', '52' => 'Kralovehradecky kraj', '51' => 'Liberecky kraj', '80' => 'Moravskoslezsky kraj', '71' => 'Olomoucky kraj', '53' => 'Pardubicky kraj', '32' => 'Plzensky kraj', '10' => 'Praha, Hlavni mesto', '20' => 'Stredocesky kraj', '42' => 'Ustecky kraj', '72' => 'Zlinsky kraj'], 	'DE' => ['BW' => 'Baden-Wurttemberg', 'BY' => 'Bayern', 'BE' => 'Berlin', 'BB' => 'Brandenburg', 'HB' => 'Bremen', 'HH' => 'Hamburg', 'HE' => 'Hessen', 'MV' => 'Mecklenburg-Vorpommern', 'NI' => 'Niedersachsen', 'NW' => 'Nordrhein-Westfalen', 'RP' => 'Rheinland-Pfalz', 'SL' => 'Saarland', 'SN' => 'Sachsen', 'ST' => 'Sachsen-Anhalt', 'SH' => 'Schleswig-Holstein', 'TH' => 'Thuringen'], 	'DJ' => ['AR' => 'Arta', 'DI' => 'Dikhil', 'DJ' => 'Djibouti', 'TA' => 'Tadjourah'], 	'DK' => ['84' => 'Hovedstaden', '82' => 'Midtjylland', '81' => 'Nordjylland', '85' => 'Sjaelland', '83' => 'Syddanmark'], 	'DM' => ['02' => 'Saint Andrew', '04' => 'Saint George', '05' => 'Saint John', '06' => 'Saint Joseph', '07' => 'Saint Luke', '09' => 'Saint Patrick', '10' => 'Saint Paul'], 	'DO' => ['02' => 'Azua', '03' => 'Baoruco', '04' => 'Barahona', '05' => 'Dajabon', '01' => 'Distrito Nacional (Santo Domingo)', '06' => 'Duarte', '08' => 'El Seibo', '07' => 'Elias Pina', '09' => 'Espaillat', '30' => 'Hato Mayor', '19' => 'Hermanas Mirabal', '10' => 'Independencia', '11' => 'La Altagracia', '12' => 'La Romana', '13' => 'La Vega', '14' => 'Maria Trinidad Sanchez', '28' => 'Monsenor Nouel', '15' => 'Monte Cristi', '29' => 'Monte Plata', '17' => 'Peravia', '18' => 'Puerto Plata', '20' => 'Samana', '21' => 'San Cristobal', '31' => 'San Jose de Ocoa', '22' => 'San Juan', '23' => 'San Pedro de Macoris', '24' => 'Sanchez Ramirez', '25' => 'Santiago', '26' => 'Santiago Rodriguez', '27' => 'Valverde'], 	'DZ' => ['01' => 'Adrar', '44' => 'Ain Defla', '46' => 'Ain Temouchent', '16' => 'Alger', '23' => 'Annaba', '05' => 'Batna', '08' => 'Bechar', '06' => 'Bejaia', '07' => 'Biskra', '09' => 'Blida', '34' => 'Bordj Bou Arreridj', '10' => 'Bouira', '35' => 'Boumerdes', '02' => 'Chlef', '25' => 'Constantine', '56' => 'Djanet', '17' => 'Djelfa', '32' => 'El Bayadh', '57' => 'El Meghaier', '39' => 'El Oued', '36' => 'El Tarf', '47' => 'Ghardaia', '24' => 'Guelma', '33' => 'Illizi', '53' => 'In Salah', '18' => 'Jijel', '40' => 'Khenchela', '03' => 'Laghouat', '28' => 'M\'sila', '29' => 'Mascara', '26' => 'Medea', '43' => 'Mila', '27' => 'Mostaganem', '45' => 'Naama', '31' => 'Oran', '30' => 'Ouargla', '51' => 'Ouled Djellal', '04' => 'Oum el Bouaghi', '48' => 'Relizane', '20' => 'Saida', '19' => 'Setif', '22' => 'Sidi Bel Abbes', '21' => 'Skikda', '41' => 'Souk Ahras', '11' => 'Tamanrasset', '12' => 'Tebessa', '14' => 'Tiaret', '49' => 'Timimoun', '37' => 'Tindouf', '42' => 'Tipaza', '38' => 'Tissemsilt', '15' => 'Tizi Ouzou', '13' => 'Tlemcen', '55' => 'Touggourt'], 	'EC' => ['A' => 'Azuay', 'B' => 'Bolivar', 'F' => 'Canar', 'C' => 'Carchi', 'H' => 'Chimborazo', 'X' => 'Cotopaxi', 'O' => 'El Oro', 'E' => 'Esmeraldas', 'W' => 'Galapagos', 'G' => 'Guayas', 'I' => 'Imbabura', 'L' => 'Loja', 'R' => 'Los Rios', 'M' => 'Manabi', 'S' => 'Morona Santiago', 'N' => 'Napo', 'D' => 'Orellana', 'Y' => 'Pastaza', 'P' => 'Pichincha', 'SE' => 'Santa Elena', 'SD' => 'Santo Domingo de los Tsachilas', 'U' => 'Sucumbios', 'T' => 'Tungurahua', 'Z' => 'Zamora Chinchipe'], 	'EE' => ['37' => 'Harjumaa', '39' => 'Hiiumaa', '45' => 'Ida-Virumaa', '52' => 'Jarvamaa', '50' => 'Jogevamaa', '60' => 'Laane-Virumaa', '56' => 'Laanemaa', '68' => 'Parnumaa', '64' => 'Polvamaa', '71' => 'Raplamaa', '74' => 'Saaremaa', '79' => 'Tartumaa', '81' => 'Valgamaa', '84' => 'Viljandimaa', '87' => 'Vorumaa'], 	'EG' => ['DK' => 'Ad Daqahliyah', 'BA' => 'Al Bahr al Ahmar', 'BH' => 'Al Buhayrah', 'FYM' => 'Al Fayyum', 'GH' => 'Al Gharbiyah', 'ALX' => 'Al Iskandariyah', 'IS' => 'Al Isma\'iliyah', 'GZ' => 'Al Jizah', 'MNF' => 'Al Minufiyah', 'MN' => 'Al Minya', 'C' => 'Al Qahirah', 'KB' => 'Al Qalyubiyah', 'LX' => 'Al Uqsur', 'WAD' => 'Al Wadi al Jadid', 'SUZ' => 'As Suways', 'SHR' => 'Ash Sharqiyah', 'ASN' => 'Aswan', 'AST' => 'Asyut', 'BNS' => 'Bani Suwayf', 'PTS' => 'Bur Sa\'id', 'DT' => 'Dumyat', 'JS' => 'Janub Sina\'', 'KFS' => 'Kafr ash Shaykh', 'MT' => 'Matruh', 'KN' => 'Qina', 'SIN' => 'Shamal Sina\'', 'SHG' => 'Suhaj'], 	'EH' => ['-' => 'Western Sahara'], 	'ER' => ['MA' => 'Al Awsat', 'GB' => 'Qash-Barkah'], 	'ES' => ['AN' => 'Andalucia', 'AR' => 'Aragon', 'AS' => 'Asturias, Principado de', 'CN' => 'Canarias', 'CB' => 'Cantabria', 'CL' => 'Castilla y Leon', 'CM' => 'Castilla-La Mancha', 'CT' => 'Catalunya', 'CE' => 'Ceuta', 'EX' => 'Extremadura', 'GA' => 'Galicia', 'IB' => 'Illes Balears', 'RI' => 'La Rioja', 'MD' => 'Madrid, Comunidad de', 'ML' => 'Melilla', 'MC' => 'Murcia, Region de', 'NC' => 'Navarra, Comunidad Foral de', 'PV' => 'Pais Vasco', 'VC' => 'Valenciana, Comunidad'], 	'ET' => ['AA' => 'Addis Ababa', 'AF' => 'Afar', 'AM' => 'Amara', 'BE' => 'Benshangul-Gumaz', 'DD' => 'Dire Dawa', 'HA' => 'Harari People', 'OR' => 'Oromia', 'SO' => 'Somali', 'SN' => 'Southern Nations, Nationalities and Peoples', 'TI' => 'Tigrai'], 	'FI' => ['02' => 'Etela-Karjala', '03' => 'Etela-Pohjanmaa', '04' => 'Etela-Savo', '05' => 'Kainuu', '06' => 'Kanta-Hame', '07' => 'Keski-Pohjanmaa', '08' => 'Keski-Suomi', '09' => 'Kymenlaakso', '10' => 'Lappi', '16' => 'Paijat-Hame', '11' => 'Pirkanmaa', '12' => 'Pohjanmaa', '13' => 'Pohjois-Karjala', '14' => 'Pohjois-Pohjanmaa', '15' => 'Pohjois-Savo', '17' => 'Satakunta', '18' => 'Uusimaa', '19' => 'Varsinais-Suomi'], 	'FJ' => ['C' => 'Central', 'E' => 'Eastern', 'N' => 'Northern', 'R' => 'Rotuma', 'W' => 'Western'], 	'FK' => ['-' => 'Falkland Islands (Malvinas)'], 	'FM' => ['TRK' => 'Chuuk', 'KSA' => 'Kosrae', 'PNI' => 'Pohnpei', 'YAP' => 'Yap'], 	'FO' => ['-' => 'Vagar'], 	'FR' => ['ARA' => 'Auvergne-Rhone-Alpes', 'BFC' => 'Bourgogne-Franche-Comte', 'BRE' => 'Bretagne', 'CVL' => 'Centre-Val de Loire', '20R' => 'Corse', 'GES' => 'Grand-Est', 'HDF' => 'Hauts-de-France', 'IDF' => 'Ile-de-France', 'NOR' => 'Normandie', 'NAQ' => 'Nouvelle-Aquitaine', 'OCC' => 'Occitanie', 'PDL' => 'Pays-de-la-Loire', 'PAC' => 'Provence-Alpes-Cote-d\'Azur'], 	'GA' => ['1' => 'Estuaire', '2' => 'Haut-Ogooue', '3' => 'Moyen-Ogooue', '4' => 'Ngounie', '5' => 'Nyanga', '7' => 'Ogooue-Lolo', '8' => 'Ogooue-Maritime', '9' => 'Woleu-Ntem'], 	'GB' => ['ENG' => 'England', 'NIR' => 'Northern Ireland', 'SCT' => 'Scotland', 'WLS' => 'Wales'], 	'GD' => ['01' => 'Saint Andrew', '02' => 'Saint David', '03' => 'Saint George', '04' => 'Saint John', '05' => 'Saint Mark', '06' => 'Saint Patrick', '10' => 'Southern Grenadine Islands'], 	'GE' => ['AB' => 'Abkhazia', 'AJ' => 'Ajaria', 'GU' => 'Guria', 'IM' => 'Imereti', 'KA' => 'K\'akheti', 'KK' => 'Kvemo Kartli', 'MM' => 'Mtskheta-Mtianeti', 'RL' => 'Rach\'a-Lechkhumi-Kvemo Svaneti', 'SZ' => 'Samegrelo-Zemo Svaneti', 'SJ' => 'Samtskhe-Javakheti', 'SK' => 'Shida Kartli', 'TB' => 'Tbilisi'], 	'GF' => ['-' => 'Guyane'], 	'GG' => ['-' => 'Guernsey'], 	'GH' => ['AF' => 'Ahafo', 'AH' => 'Ashanti', 'BO' => 'Bono', 'BE' => 'Bono East', 'CP' => 'Central', 'EP' => 'Eastern', 'AA' => 'Greater Accra', 'NP' => 'Northern', 'UE' => 'Upper East', 'UW' => 'Upper West', 'TV' => 'Volta', 'WP' => 'Western'], 	'GI' => ['-' => 'Gibraltar'], 	'GL' => ['AV' => 'Avannaata Kommunia', 'KU' => 'Kommune Kujalleq', 'QT' => 'Kommune Qeqertalik', 'SM' => 'Kommuneqarfik Sermersooq', 'QE' => 'Qeqqata Kommunia'], 	'GM' => ['B' => 'Banjul', 'M' => 'Central River', 'L' => 'Lower River', 'N' => 'North Bank', 'U' => 'Upper River', 'W' => 'Western'], 	'GN' => ['BF' => 'Boffa', 'BK' => 'Boke', 'C' => 'Conakry', 'DB' => 'Dabola', 'DI' => 'Dinguiraye', 'FR' => 'Fria', 'KA' => 'Kankan', 'KO' => 'Kouroussa', 'LA' => 'Labe', 'MM' => 'Mamou', 'PI' => 'Pita', 'SI' => 'Siguiri'], 	'GP' => ['-' => 'Guadeloupe'], 	'GQ' => ['BN' => 'Bioko Norte', 'CS' => 'Centro Sur', 'LI' => 'Litoral', 'WN' => 'Wele-Nzas'], 	'GR' => ['A' => 'Anatoliki Makedonia kai Thraki', 'I' => 'Attiki', 'G' => 'Dytiki Ellada', 'C' => 'Dytiki Makedonia', 'F' => 'Ionia Nisia', 'D' => 'Ipeiros', 'B' => 'Kentriki Makedonia', 'M' => 'Kriti', 'L' => 'Notio Aigaio', 'J' => 'Peloponnisos', 'H' => 'Sterea Ellada', 'E' => 'Thessalia', 'K' => 'Voreio Aigaio'], 	'GS' => ['-' => 'South Georgia and the South Sandwich Islands'], 	'GT' => ['16' => 'Alta Verapaz', '15' => 'Baja Verapaz', '04' => 'Chimaltenango', '20' => 'Chiquimula', '02' => 'El Progreso', '05' => 'Escuintla', '01' => 'Guatemala', '13' => 'Huehuetenango', '18' => 'Izabal', '21' => 'Jalapa', '22' => 'Jutiapa', '17' => 'Peten', '09' => 'Quetzaltenango', '14' => 'Quiche', '11' => 'Retalhuleu', '03' => 'Sacatepequez', '12' => 'San Marcos', '06' => 'Santa Rosa', '07' => 'Solola', '10' => 'Suchitepequez', '08' => 'Totonicapan', '19' => 'Zacapa'], 	'GU' => ['-' => 'Yona'], 	'GW' => ['BA' => 'Bafata', 'BS' => 'Bissau', 'CA' => 'Cacheu', 'GA' => 'Gabu', 'OI' => 'Oio', 'QU' => 'Quinara'], 	'GY' => ['BA' => 'Barima-Waini', 'CU' => 'Cuyuni-Mazaruni', 'DE' => 'Demerara-Mahaica', 'EB' => 'East Berbice-Corentyne', 'ES' => 'Essequibo Islands-West Demerara', 'MA' => 'Mahaica-Berbice', 'PM' => 'Pomeroon-Supenaam', 'PT' => 'Potaro-Siparuni', 'UD' => 'Upper Demerara-Berbice'], 	'HK' => ['-' => 'Hong Kong'], 	'HM' => ['-' => 'Heard Island and McDonald Islands'], 	'HN' => ['AT' => 'Atlantida', 'CH' => 'Choluteca', 'CL' => 'Colon', 'CM' => 'Comayagua', 'CP' => 'Copan', 'CR' => 'Cortes', 'EP' => 'El Paraiso', 'FM' => 'Francisco Morazan', 'GD' => 'Gracias a Dios', 'IN' => 'Intibuca', 'IB' => 'Islas de la Bahia', 'LP' => 'La Paz', 'LE' => 'Lempira', 'OC' => 'Ocotepeque', 'OL' => 'Olancho', 'SB' => 'Santa Barbara', 'VA' => 'Valle', 'YO' => 'Yoro'], 	'HR' => ['07' => 'Bjelovarsko-bilogorska zupanija', '12' => 'Brodsko-posavska zupanija', '19' => 'Dubrovacko-neretvanska zupanija', '21' => 'Grad Zagreb', '18' => 'Istarska zupanija', '04' => 'Karlovacka zupanija', '06' => 'Koprivnicko-krizevacka zupanija', '02' => 'Krapinsko-zagorska zupanija', '09' => 'Licko-senjska zupanija', '20' => 'Medimurska zupanija', '14' => 'Osjecko-baranjska zupanija', '11' => 'Pozesko-slavonska zupanija', '08' => 'Primorsko-goranska zupanija', '15' => 'Sibensko-kninska zupanija', '03' => 'Sisacko-moslavacka zupanija', '17' => 'Splitsko-dalmatinska zupanija', '05' => 'Varazdinska zupanija', '10' => 'Viroviticko-podravska zupanija', '16' => 'Vukovarsko-srijemska zupanija', '13' => 'Zadarska zupanija', '01' => 'Zagrebacka zupanija'], 	'HT' => ['AR' => 'Artibonite', 'CE' => 'Centre', 'GA' => 'Grande\'Anse', 'ND' => 'Nord', 'NO' => 'Nord-Ouest', 'OU' => 'Ouest', 'SD' => 'Sud', 'SE' => 'Sud-Est'], 	'HU' => ['BK' => 'Bacs-Kiskun', 'BA' => 'Baranya', 'BE' => 'Bekes', 'BZ' => 'Borsod-Abauj-Zemplen', 'BU' => 'Budapest', 'CS' => 'Csongrad-Csanad', 'FE' => 'Fejer', 'GS' => 'Gyor-Moson-Sopron', 'HB' => 'Hajdu-Bihar', 'HE' => 'Heves', 'JN' => 'Jasz-Nagykun-Szolnok', 'KE' => 'Komarom-Esztergom', 'NO' => 'Nograd', 'PE' => 'Pest', 'SO' => 'Somogy', 'SZ' => 'Szabolcs-Szatmar-Bereg', 'TO' => 'Tolna', 'VA' => 'Vas', 'VE' => 'Veszprem', 'ZA' => 'Zala'], 	'ID' => ['AC' => 'Aceh', 'BA' => 'Bali', 'BT' => 'Banten', 'BE' => 'Bengkulu', 'GO' => 'Gorontalo', 'JK' => 'Jakarta Raya', 'JA' => 'Jambi', 'JB' => 'Jawa Barat', 'JT' => 'Jawa Tengah', 'JI' => 'Jawa Timur', 'KB' => 'Kalimantan Barat', 'KS' => 'Kalimantan Selatan', 'KT' => 'Kalimantan Tengah', 'KI' => 'Kalimantan Timur', 'KU' => 'Kalimantan Utara', 'BB' => 'Kepulauan Bangka Belitung', 'KR' => 'Kepulauan Riau', 'LA' => 'Lampung', 'ML' => 'Maluku', 'MU' => 'Maluku Utara', 'NB' => 'Nusa Tenggara Barat', 'NT' => 'Nusa Tenggara Timur', 'PP' => 'Papua', 'PB' => 'Papua Barat', 'PD' => 'Papua Barat Daya', 'PE' => 'Papua Pengunungan', 'PS' => 'Papua Selatan', 'PT' => 'Papua Tengah', 'RI' => 'Riau', 'SR' => 'Sulawesi Barat', 'SN' => 'Sulawesi Selatan', 'ST' => 'Sulawesi Tengah', 'SG' => 'Sulawesi Tenggara', 'SA' => 'Sulawesi Utara', 'SB' => 'Sumatera Barat', 'SS' => 'Sumatera Selatan', 'SU' => 'Sumatera Utara', 'YO' => 'Yogyakarta'], 	'IE' => ['CW' => 'Carlow', 'CN' => 'Cavan', 'CE' => 'Clare', 'CO' => 'Cork', 'DL' => 'Donegal', 'D' => 'Dublin', 'G' => 'Galway', 'KY' => 'Kerry', 'KE' => 'Kildare', 'KK' => 'Kilkenny', 'LS' => 'Laois', 'LM' => 'Leitrim', 'LK' => 'Limerick', 'LD' => 'Longford', 'LH' => 'Louth', 'MO' => 'Mayo', 'MH' => 'Meath', 'MN' => 'Monaghan', 'OY' => 'Offaly', 'RN' => 'Roscommon', 'SO' => 'Sligo', 'TA' => 'Tipperary', 'WD' => 'Waterford', 'WH' => 'Westmeath', 'WX' => 'Wexford', 'WW' => 'Wicklow'], 	'IL' => ['D' => 'HaDarom', 'M' => 'HaMerkaz', 'Z' => 'HaTsafon', 'HA' => 'Hefa', 'TA' => 'Tel Aviv', 'JM' => 'Yerushalayim'], 	'IM' => ['-' => 'Isle of Man'], 	'IN' => ['AN' => 'Andaman and Nicobar Islands', 'AP' => 'Andhra Pradesh', 'AR' => 'Arunachal Pradesh', 'AS' => 'Assam', 'BR' => 'Bihar', 'CH' => 'Chandigarh', 'CG' => 'Chhattisgarh', 'DH' => 'Dadra and Nagar Haveli and Daman and Diu', 'DL' => 'Delhi', 'GA' => 'Goa', 'GJ' => 'Gujarat', 'HR' => 'Haryana', 'HP' => 'Himachal Pradesh', 'JK' => 'Jammu and Kashmir', 'JH' => 'Jharkhand', 'KA' => 'Karnataka', 'KL' => 'Kerala', 'LD' => 'Lakshadweep', 'MP' => 'Madhya Pradesh', 'MH' => 'Maharashtra', 'MN' => 'Manipur', 'ML' => 'Meghalaya', 'MZ' => 'Mizoram', 'NL' => 'Nagaland', 'OD' => 'Odisha', 'PY' => 'Puducherry', 'PB' => 'Punjab', 'RJ' => 'Rajasthan', 'SK' => 'Sikkim', 'TN' => 'Tamil Nadu', 'TS' => 'Telangana', 'TR' => 'Tripura', 'UP' => 'Uttar Pradesh', 'UK' => 'Uttarakhand', 'WB' => 'West Bengal'], 	'IO' => ['-' => 'British Indian Ocean Territory'], 	'IQ' => ['AN' => 'Al Anbar', 'BA' => 'Al Basrah', 'MU' => 'Al Muthanna', 'QA' => 'Al Qadisiyah', 'NA' => 'An Najaf', 'AR' => 'Arbil', 'SU' => 'As Sulaymaniyah', 'BB' => 'Babil', 'BG' => 'Baghdad', 'DA' => 'Dahuk', 'DQ' => 'Dhi Qar', 'DI' => 'Diyala', 'KA' => 'Karbala\'', 'KI' => 'Kirkuk', 'MA' => 'Maysan', 'NI' => 'Ninawa', 'SD' => 'Salah ad Din', 'WA' => 'Wasit'], 	'IR' => ['30' => 'Alborz', '24' => 'Ardabil', '04' => 'Azarbayjan-e Gharbi', '03' => 'Azarbayjan-e Sharqi', '18' => 'Bushehr', '14' => 'Chahar Mahal va Bakhtiari', '10' => 'Esfahan', '07' => 'Fars', '01' => 'Gilan', '27' => 'Golestan', '13' => 'Hamadan', '22' => 'Hormozgan', '16' => 'Ilam', '08' => 'Kerman', '05' => 'Kermanshah', '29' => 'Khorasan-e Jonubi', '09' => 'Khorasan-e Razavi', '28' => 'Khorasan-e Shomali', '06' => 'Khuzestan', '17' => 'Kohgiluyeh va Bowyer Ahmad', '12' => 'Kordestan', '15' => 'Lorestan', '00' => 'Markazi', '02' => 'Mazandaran', '26' => 'Qazvin', '25' => 'Qom', '20' => 'Semnan', '11' => 'Sistan va Baluchestan', '23' => 'Tehran', '21' => 'Yazd', '19' => 'Zanjan'], 	'IS' => ['7' => 'Austurland', '1' => 'Hofudborgarsvaedi', '6' => 'Nordurland eystra', '5' => 'Nordurland vestra', '8' => 'Sudurland', '2' => 'Sudurnes', '4' => 'Vestfirdir', '3' => 'Vesturland'], 	'IT' => ['65' => 'Abruzzo', '77' => 'Basilicata', '78' => 'Calabria', '72' => 'Campania', '45' => 'Emilia-Romagna', '36' => 'Friuli-Venezia Giulia', '62' => 'Lazio', '42' => 'Liguria', '25' => 'Lombardia', '57' => 'Marche', '67' => 'Molise', '21' => 'Piemonte', '75' => 'Puglia', '88' => 'Sardegna', '82' => 'Sicilia', '52' => 'Toscana', '32' => 'Trentino-Alto Adige', '55' => 'Umbria', '23' => 'Valle d\'Aosta', '34' => 'Veneto'], 	'JE' => ['-' => 'Jersey'], 	'JM' => ['13' => 'Clarendon', '09' => 'Hanover', '01' => 'Kingston', '12' => 'Manchester', '04' => 'Portland', '02' => 'Saint Andrew', '06' => 'Saint Ann', '14' => 'Saint Catherine', '11' => 'Saint Elizabeth', '08' => 'Saint James', '05' => 'Saint Mary', '03' => 'Saint Thomas', '07' => 'Trelawny', '10' => 'Westmoreland'], 	'JO' => ['AJ' => '\'Ajlun', 'AQ' => 'Al \'Aqabah', 'AM' => 'Al \'Asimah', 'BA' => 'Al Balqa\'', 'KA' => 'Al Karak', 'MA' => 'Al Mafraq', 'AT' => 'At Tafilah', 'AZ' => 'Az Zarqa\'', 'IR' => 'Irbid', 'JA' => 'Jarash', 'MN' => 'Ma\'an', 'MD' => 'Madaba'], 	'JP' => ['23' => 'Aichi', '05' => 'Akita', '02' => 'Aomori', '12' => 'Chiba', '38' => 'Ehime', '18' => 'Fukui', '40' => 'Fukuoka', '07' => 'Fukushima', '21' => 'Gifu', '10' => 'Gunma', '34' => 'Hiroshima', '01' => 'Hokkaido', '28' => 'Hyogo', '08' => 'Ibaraki', '17' => 'Ishikawa', '03' => 'Iwate', '37' => 'Kagawa', '46' => 'Kagoshima', '14' => 'Kanagawa', '39' => 'Kochi', '43' => 'Kumamoto', '26' => 'Kyoto', '24' => 'Mie', '04' => 'Miyagi', '45' => 'Miyazaki', '20' => 'Nagano', '42' => 'Nagasaki', '29' => 'Nara', '15' => 'Niigata', '44' => 'Oita', '33' => 'Okayama', '47' => 'Okinawa', '27' => 'Osaka', '41' => 'Saga', '11' => 'Saitama', '25' => 'Shiga', '32' => 'Shimane', '22' => 'Shizuoka', '09' => 'Tochigi', '36' => 'Tokushima', '13' => 'Tokyo', '31' => 'Tottori', '16' => 'Toyama', '30' => 'Wakayama', '06' => 'Yamagata', '35' => 'Yamaguchi', '19' => 'Yamanashi'], 	'KE' => ['01' => 'Baringo', '02' => 'Bomet', '03' => 'Bungoma', '04' => 'Busia', '05' => 'Elgeyo\/Marakwet', '06' => 'Embu', '07' => 'Garissa', '08' => 'Homa Bay', '09' => 'Isiolo', '10' => 'Kajiado', '11' => 'Kakamega', '12' => 'Kericho', '13' => 'Kiambu', '14' => 'Kilifi', '15' => 'Kirinyaga', '16' => 'Kisii', '17' => 'Kisumu', '18' => 'Kitui', '19' => 'Kwale', '20' => 'Laikipia', '21' => 'Lamu', '22' => 'Machakos', '23' => 'Makueni', '24' => 'Mandera', '25' => 'Marsabit', '26' => 'Meru', '27' => 'Migori', '28' => 'Mombasa', '29' => 'Murang\'a', '30' => 'Nairobi City', '31' => 'Nakuru', '32' => 'Nandi', '33' => 'Narok', '34' => 'Nyamira', '35' => 'Nyandarua', '36' => 'Nyeri', '37' => 'Samburu', '38' => 'Siaya', '39' => 'Taita\/Taveta', '40' => 'Tana River', '41' => 'Tharaka-Nithi', '42' => 'Trans Nzoia', '43' => 'Turkana', '44' => 'Uasin Gishu', '45' => 'Vihiga', '46' => 'Wajir', '47' => 'West Pokot'], 	'KG' => ['B' => 'Batken', 'GB' => 'Bishkek Shaary', 'C' => 'Chuy', 'J' => 'Jalal-Abad', 'N' => 'Naryn', 'GO' => 'Osh Shaary', 'T' => 'Talas', 'Y' => 'Ysyk-Kol'], 	'KH' => ['2' => 'Baat Dambang', '1' => 'Banteay Mean Choay', '23' => 'Kaeb', '3' => 'Kampong Chaam', '4' => 'Kampong Chhnang', '5' => 'Kampong Spueu', '6' => 'Kampong Thum', '7' => 'Kampot', '8' => 'Kandaal', '9' => 'Kaoh Kong', '10' => 'Kracheh', '11' => 'Mondol Kiri', '24' => 'Pailin', '12' => 'Phnom Penh', '18' => 'Preah Sihanouk', '13' => 'Preah Vihear', '14' => 'Prey Veaeng', '17' => 'Siem Reab', '19' => 'Stueng Traeng', '20' => 'Svaay Rieng', '21' => 'Taakaev'], 	'KI' => ['G' => 'Gilbert Islands', 'L' => 'Line Islands'], 	'KM' => ['G' => 'Grande Comore', 'M' => 'Moheli'], 	'KN' => ['01' => 'Christ Church Nichola Town', '02' => 'Saint Anne Sandy Point', '03' => 'Saint George Basseterre', '05' => 'Saint James Windward', '06' => 'Saint John Capisterre', '07' => 'Saint John Figtree', '08' => 'Saint Mary Cayon', '09' => 'Saint Paul Capisterre', '10' => 'Saint Paul Charlestown', '11' => 'Saint Peter Basseterre', '12' => 'Saint Thomas Lowland', '13' => 'Saint Thomas Middle Island', '15' => 'Trinity Palmetto Point'], 	'KP' => ['01' => 'P\'yongyang'], 	'KR' => ['26' => 'Busan-gwangyeoksi', '43' => 'Chungcheongbuk-do', '44' => 'Chungcheongnam-do', '27' => 'Daegu-gwangyeoksi', '30' => 'Daejeon-gwangyeoksi', '42' => 'Gangwon-teukbyeoljachido', '29' => 'Gwangju-gwangyeoksi', '41' => 'Gyeonggi-do', '47' => 'Gyeongsangbuk-do', '48' => 'Gyeongsangnam-do', '28' => 'Incheon-gwangyeoksi', '49' => 'Jeju-teukbyeoljachido', '45' => 'Jeollabuk-do', '46' => 'Jeollanam-do', '50' => 'Sejong', '11' => 'Seoul-teukbyeolsi', '31' => 'Ulsan-gwangyeoksi'], 	'KW' => ['KU' => 'Al \'Asimah', 'AH' => 'Al Ahmadi', 'FA' => 'Al Farwaniyah', 'JA' => 'Al Jahra\'', 'HA' => 'Hawalli', 'MU' => 'Mubarak al Kabir'], 	'KY' => ['-' => 'Cayman Islands'], 	'KZ' => ['10' => 'Abay oblysy', '75' => 'Almaty', '19' => 'Almaty oblysy', '11' => 'Aqmola oblysy', '15' => 'Aqtobe oblysy', '71' => 'Astana', '23' => 'Atyrau oblysy', '27' => 'Batys Qazaqstan oblysy', '47' => 'Mangghystau oblysy', '55' => 'Pavlodar oblysy', '35' => 'Qaraghandy oblysy', '39' => 'Qostanay oblysy', '43' => 'Qyzylorda oblysy', '63' => 'Shyghys Qazaqstan oblysy', '79' => 'Shymkent', '59' => 'Soltustik Qazaqstan oblysy', '61' => 'Turkistan oblysy', '62' => 'Ulytau oblysy', '31' => 'Zhambyl oblysy', '33' => 'Zhetisu oblysy'], 	'LA' => ['AT' => 'Attapu', 'BK' => 'Bokeo', 'BL' => 'Bolikhamxai', 'CH' => 'Champasak', 'HO' => 'Houaphan', 'KH' => 'Khammouan', 'LP' => 'Louangphabang', 'OU' => 'Oudomxai', 'PH' => 'Phongsali', 'SV' => 'Savannakhet', 'VI' => 'Viangchan', 'XE' => 'Xekong', 'XI' => 'Xiangkhouang'], 	'LB' => ['AK' => 'Aakkar', 'BH' => 'Baalbek-Hermel', 'BI' => 'Beqaa', 'BA' => 'Beyrouth', 'AS' => 'Liban-Nord', 'JA' => 'Liban-Sud', 'JL' => 'Mont-Liban', 'NA' => 'Nabatiye'], 	'LC' => ['01' => 'Anse la Raye', '02' => 'Castries', '03' => 'Choiseul', '05' => 'Dennery', '06' => 'Gros Islet', '07' => 'Laborie', '08' => 'Micoud', '10' => 'Soufriere', '11' => 'Vieux Fort'], 	'LI' => ['01' => 'Balzers', '02' => 'Eschen', '03' => 'Gamprin', '04' => 'Mauren', '06' => 'Ruggell', '07' => 'Schaan', '09' => 'Triesen', '10' => 'Triesenberg', '11' => 'Vaduz'], 	'LK' => ['2' => 'Central Province', '5' => 'Eastern Province', '7' => 'North Central Province', '6' => 'North Western Province', '4' => 'Northern Province', '9' => 'Sabaragamuwa Province', '3' => 'Southern Province', '8' => 'Uva Province', '1' => 'Western Province'], 	'LR' => ['BM' => 'Bomi', 'BG' => 'Bong', 'CM' => 'Grand Cape Mount', 'GG' => 'Grand Gedeh', 'MG' => 'Margibi', 'MO' => 'Montserrado', 'NI' => 'Nimba', 'RI' => 'River Cess', 'RG' => 'River Gee', 'SI' => 'Sinoe'], 	'LS' => ['D' => 'Berea', 'B' => 'Botha-Bothe', 'C' => 'Leribe', 'E' => 'Mafeteng', 'A' => 'Maseru', 'F' => 'Mohale\'s Hoek', 'J' => 'Mokhotlong', 'H' => 'Qacha\'s Nek', 'G' => 'Quthing', 'K' => 'Thaba-Tseka'], 	'LT' => ['AL' => 'Alytaus apskritis', 'KU' => 'Kauno apskritis', 'KL' => 'Klaipedos apskritis', 'MR' => 'Marijampoles apskritis', 'PN' => 'Panevezio apskritis', 'SA' => 'Siauliu apskritis', 'TA' => 'Taurages apskritis', 'TE' => 'Telsiu apskritis', 'UT' => 'Utenos apskritis', 'VL' => 'Vilniaus apskritis'], 	'LU' => ['CA' => 'Capellen', 'CL' => 'Clervaux', 'DI' => 'Diekirch', 'EC' => 'Echternach', 'ES' => 'Esch-sur-Alzette', 'GR' => 'Grevenmacher', 'LU' => 'Luxembourg', 'ME' => 'Mersch', 'RD' => 'Redange', 'RM' => 'Remich', 'VD' => 'Vianden', 'WI' => 'Wiltz'], 	'LV' => ['011' => 'Adazu novads', '002' => 'Aizkraukles novads', '007' => 'Aluksnes novads', '111' => 'Augsdaugavas novads', '015' => 'Balvu novads', '016' => 'Bauskas novads', '022' => 'Cesu novads', 'DGV' => 'Daugavpils', '112' => 'Dienvidkurzemes novads', '026' => 'Dobeles novads', '033' => 'Gulbenes novads', '042' => 'Jekabpils novads', 'JEL' => 'Jelgava', '041' => 'Jelgavas novads', 'JUR' => 'Jurmala', '052' => 'Kekavas novads', '047' => 'Kraslavas novads', '050' => 'Kuldigas novads', 'LPX' => 'Liepaja', '054' => 'Limbazu novads', '056' => 'Livanu novads', '058' => 'Ludzas novads', '059' => 'Madonas novads', '062' => 'Marupes novads', '067' => 'Ogres novads', '068' => 'Olaines novads', '073' => 'Preilu novads', '077' => 'Rezeknes novads', 'RIX' => 'Riga', '080' => 'Ropazu novads', '087' => 'Salaspils novads', '088' => 'Saldus novads', '089' => 'Saulkrastu novads', '091' => 'Siguldas novads', '094' => 'Smiltenes novads', '097' => 'Talsu novads', '099' => 'Tukuma novads', '101' => 'Valkas novads', '113' => 'Valmieras novads', '106' => 'Ventspils novads'], 	'LY' => ['BU' => 'Al Butnan', 'JA' => 'Al Jabal al Akhdar', 'JG' => 'Al Jabal al Gharbi', 'JI' => 'Al Jafarah', 'JU' => 'Al Jufrah', 'KF' => 'Al Kufrah', 'MJ' => 'Al Marj', 'MB' => 'Al Marqab', 'WA' => 'Al Wahat', 'NQ' => 'An Nuqat al Khams', 'ZA' => 'Az Zawiyah', 'BA' => 'Banghazi', 'DR' => 'Darnah', 'MI' => 'Misratah', 'MQ' => 'Murzuq', 'NL' => 'Nalut', 'SB' => 'Sabha', 'SR' => 'Surt', 'TB' => 'Tarabulus', 'WS' => 'Wadi ash Shati\''], 	'MA' => ['05' => 'Beni-Mellal-Khenifra', '06' => 'Casablanca-Settat', '08' => 'Draa-Tafilalet', '03' => 'Fes- Meknes', '10' => 'Guelmim-Oued Noun (EH-partial)', '02' => 'L\'Oriental', '11' => 'Laayoune-Sakia El Hamra (EH-partial)', '07' => 'Marrakech-Safi', '04' => 'Rabat-Sale-Kenitra', '09' => 'Souss-Massa', '01' => 'Tanger-Tetouan-Al Hoceima'], 	'MC' => ['FO' => 'Fontvieille', 'CO' => 'La Condamine', 'MO' => 'Monaco-Ville', 'MG' => 'Moneghetti', 'MC' => 'Monte-Carlo', 'SR' => 'Saint-Roman'], 	'MD' => ['AN' => 'Anenii Noi', 'BA' => 'Balti', 'BS' => 'Basarabeasca', 'BD' => 'Bender', 'BR' => 'Briceni', 'CA' => 'Cahul', 'CL' => 'Calarasi', 'CT' => 'Cantemir', 'CS' => 'Causeni', 'CU' => 'Chisinau', 'CM' => 'Cimislia', 'CR' => 'Criuleni', 'DO' => 'Donduseni', 'DR' => 'Drochia', 'DU' => 'Dubasari', 'ED' => 'Edinet', 'FA' => 'Falesti', 'FL' => 'Floresti', 'GA' => 'Gagauzia, Unitatea teritoriala autonoma', 'GL' => 'Glodeni', 'HI' => 'Hincesti', 'IA' => 'Ialoveni', 'LE' => 'Leova', 'NI' => 'Nisporeni', 'OC' => 'Ocnita', 'OR' => 'Orhei', 'RE' => 'Rezina', 'RI' => 'Riscani', 'SI' => 'Singerei', 'SD' => 'Soldanesti', 'SO' => 'Soroca', 'SV' => 'Stefan Voda', 'SN' => 'Stinga Nistrului, unitatea teritoriala din', 'ST' => 'Straseni', 'TA' => 'Taraclia', 'TE' => 'Telenesti', 'UN' => 'Ungheni'], 	'ME' => ['01' => 'Andrijevica', '02' => 'Bar', '03' => 'Berane', '04' => 'Bijelo Polje', '05' => 'Budva', '06' => 'Cetinje', '07' => 'Danilovgrad', '08' => 'Herceg-Novi', '09' => 'Kolasin', '10' => 'Kotor', '12' => 'Niksic', '13' => 'Plav', '14' => 'Pljevlja', '15' => 'Pluzine', '16' => 'Podgorica', '17' => 'Rozaje', '19' => 'Tivat', '24' => 'Tuzi', '20' => 'Ulcinj', '21' => 'Zabljak', '25' => 'Zeta'], 	'MF' => ['-' => 'Saint Martin (French Part)'], 	'MG' => ['T' => 'Antananarivo', 'D' => 'Antsiranana', 'F' => 'Fianarantsoa', 'M' => 'Mahajanga', 'A' => 'Toamasina', 'U' => 'Toliara'], 	'MH' => ['KWA' => 'Kwajalein', 'MAJ' => 'Majuro'], 	'MK' => ['201' => 'Berovo', '501' => 'Bitola', '401' => 'Bogdanci', '601' => 'Bogovinje', '402' => 'Bosilovo', '602' => 'Brvenica', '803' => 'Butel', '109' => 'Caska', '814' => 'Centar', '313' => 'Centar Zupa', '816' => 'Cucer Sandevo', '303' => 'Debar', '203' => 'Delcevo', '502' => 'Demir Hisar', '103' => 'Demir Kapija', '406' => 'Dojran', '503' => 'Dolneni', '804' => 'Gazi Baba', '405' => 'Gevgelija', '604' => 'Gostivar', '102' => 'Gradsko', '807' => 'Ilinden', '606' => 'Jegunovce', '205' => 'Karbinci', '104' => 'Kavadarci', '307' => 'Kicevo', '809' => 'Kisela Voda', '206' => 'Kocani', '701' => 'Kratovo', '702' => 'Kriva Palanka', '504' => 'Krivogastani', '505' => 'Krusevo', '703' => 'Kumanovo', '704' => 'Lipkovo', '105' => 'Lozovo', '207' => 'Makedonska Kamenica', '308' => 'Makedonski Brod', '607' => 'Mavrovo i Rostusa', '506' => 'Mogila', '106' => 'Negotino', '507' => 'Novaci', '408' => 'Novo Selo', '310' => 'Ohrid', '208' => 'Pehcevo', '810' => 'Petrovec', '508' => 'Prilep', '209' => 'Probistip', '409' => 'Radovis', '705' => 'Rankovce', '509' => 'Resen', '107' => 'Rosoman', '811' => 'Saraj', '211' => 'Stip', '312' => 'Struga', '410' => 'Strumica', '813' => 'Studenicani', '108' => 'Sveti Nikole', '608' => 'Tearce', '609' => 'Tetovo', '403' => 'Valandovo', '404' => 'Vasilevo', '101' => 'Veles', '301' => 'Vevcani', '202' => 'Vinica', '806' => 'Zelenikovo', '605' => 'Zelino'], 	'ML' => ['BKO' => 'Bamako', '7' => 'Gao', '1' => 'Kayes', '8' => 'Kidal', '2' => 'Koulikoro', '5' => 'Mopti', '4' => 'Segou', '3' => 'Sikasso', '6' => 'Tombouctou'], 	'MM' => ['07' => 'Ayeyarwady', '02' => 'Bago', '14' => 'Chin', '11' => 'Kachin', '12' => 'Kayah', '13' => 'Kayin', '03' => 'Magway', '04' => 'Mandalay', '15' => 'Mon', '18' => 'Nay Pyi Taw', '16' => 'Rakhine', '01' => 'Sagaing', '17' => 'Shan', '05' => 'Tanintharyi', '06' => 'Yangon'], 	'MN' => ['069' => 'Bayanhongor', '067' => 'Bulgan', '037' => 'Darhan uul', '061' => 'Dornod', '065' => 'Govi-Altay', '043' => 'Hovd', '041' => 'Hovsgol', '053' => 'Omnogovi', '035' => 'Orhon', '055' => 'Ovorhangay', '047' => 'Tov', '1' => 'Ulaanbaatar'], 	'MO' => ['-' => 'Macao'], 	'MP' => ['-' => 'Northern Mariana Islands'], 	'MQ' => ['-' => 'Martinique'], 	'MR' => ['07' => 'Adrar', '03' => 'Assaba', '05' => 'Brakna', '08' => 'Dakhlet Nouadhibou', '04' => 'Gorgol', '10' => 'Guidimaka', '02' => 'Hodh el Gharbi', '12' => 'Inchiri', '13' => 'Nouakchott Ouest', '09' => 'Tagant', '11' => 'Tiris Zemmour', '06' => 'Trarza'], 	'MS' => ['-' => 'Saint Peter'], 	'MT' => ['01' => 'Attard', '02' => 'Balzan', '03' => 'Birgu', '04' => 'Birkirkara', '05' => 'Birzebbuga', '06' => 'Bormla', '07' => 'Dingli', '08' => 'Fgura', '09' => 'Floriana', '10' => 'Fontana', '13' => 'Ghajnsielem', '14' => 'Gharb', '15' => 'Gharghur', '16' => 'Ghasri', '17' => 'Ghaxaq', '11' => 'Gudja', '12' => 'Gzira', '18' => 'Hamrun', '19' => 'Iklin', '20' => 'Isla', '21' => 'Kalkara', '23' => 'Kirkop', '24' => 'Lija', '25' => 'Luqa', '26' => 'Marsa', '27' => 'Marsaskala', '28' => 'Marsaxlokk', '29' => 'Mdina', '30' => 'Mellieha', '31' => 'Mgarr', '32' => 'Mosta', '33' => 'Mqabba', '34' => 'Msida', '35' => 'Mtarfa', '36' => 'Munxar', '37' => 'Nadur', '38' => 'Naxxar', '39' => 'Paola', '40' => 'Pembroke', '41' => 'Pieta', '42' => 'Qala', '43' => 'Qormi', '44' => 'Qrendi', '45' => 'Rabat Gozo', '46' => 'Rabat Malta', '47' => 'Safi', '49' => 'Saint John', '48' => 'Saint Julian\'s', '50' => 'Saint Lawrence', '53' => 'Saint Lucia\'s', '51' => 'Saint Paul\'s Bay', '52' => 'Sannat', '54' => 'Santa Venera', '55' => 'Siggiewi', '56' => 'Sliema', '57' => 'Swieqi', '58' => 'Ta\' Xbiex', '59' => 'Tarxien', '60' => 'Valletta', '61' => 'Xaghra', '62' => 'Xewkija', '63' => 'Xghajra', '64' => 'Zabbar', '65' => 'Zebbug Gozo', '67' => 'Zejtun', '68' => 'Zurrieq'], 	'MU' => ['BL' => 'Black River', 'FL' => 'Flacq', 'GP' => 'Grand Port', 'MO' => 'Moka', 'PA' => 'Pamplemousses', 'PW' => 'Plaines Wilhems', 'PL' => 'Port Louis', 'RR' => 'Riviere du Rempart', 'RO' => 'Rodrigues Islands', 'SA' => 'Savanne'], 	'MV' => ['01' => 'Addu City', '03' => 'Faadhippolhu', '04' => 'Felidhu Atoll', '05' => 'Hahdhunmathi', 'MLE' => 'Male', '12' => 'Mulaku Atoll', '02' => 'North Ari Atoll', '27' => 'North Huvadhu Atoll', '13' => 'North Maalhosmadulu', '14' => 'North Nilandhe Atoll', '07' => 'North Thiladhunmathi', '00' => 'South Ari Atoll', '28' => 'South Huvadhu Atoll', '20' => 'South Maalhosmadulu', '25' => 'South Miladhunmadulu', '17' => 'South Nilandhe Atoll', '23' => 'South Thiladhunmathi'], 	'MW' => ['BA' => 'Balaka', 'BL' => 'Blantyre', 'CK' => 'Chikwawa', 'CR' => 'Chiradzulu', 'DE' => 'Dedza', 'DO' => 'Dowa', 'KR' => 'Karonga', 'LI' => 'Lilongwe', 'MH' => 'Machinga', 'MG' => 'Mangochi', 'MW' => 'Mwanza', 'MZ' => 'Mzimba', 'NE' => 'Neno', 'NK' => 'Nkhotakota', 'NI' => 'Ntchisi', 'SA' => 'Salima', 'TH' => 'Thyolo', 'ZO' => 'Zomba'], 	'MX' => ['AGU' => 'Aguascalientes', 'BCN' => 'Baja California', 'BCS' => 'Baja California Sur', 'CAM' => 'Campeche', 'CHP' => 'Chiapas', 'CHH' => 'Chihuahua', 'CMX' => 'Ciudad de Mexico', 'COA' => 'Coahuila de Zaragoza', 'COL' => 'Colima', 'DUR' => 'Durango', 'GUA' => 'Guanajuato', 'GRO' => 'Guerrero', 'HID' => 'Hidalgo', 'JAL' => 'Jalisco', 'MEX' => 'Mexico', 'MIC' => 'Michoacan de Ocampo', 'MOR' => 'Morelos', 'NAY' => 'Nayarit', 'NLE' => 'Nuevo Leon', 'OAX' => 'Oaxaca', 'PUE' => 'Puebla', 'QUE' => 'Queretaro', 'ROO' => 'Quintana Roo', 'SLP' => 'San Luis Potosi', 'SIN' => 'Sinaloa', 'SON' => 'Sonora', 'TAB' => 'Tabasco', 'TAM' => 'Tamaulipas', 'TLA' => 'Tlaxcala', 'VER' => 'Veracruz de Ignacio de la Llave', 'YUC' => 'Yucatan', 'ZAC' => 'Zacatecas'], 	'MY' => ['01' => 'Johor', '02' => 'Kedah', '03' => 'Kelantan', '04' => 'Melaka', '05' => 'Negeri Sembilan', '06' => 'Pahang', '08' => 'Perak', '09' => 'Perlis', '07' => 'Pulau Pinang', '12' => 'Sabah', '13' => 'Sarawak', '10' => 'Selangor', '11' => 'Terengganu', '14' => 'Wilayah Persekutuan Kuala Lumpur', '15' => 'Wilayah Persekutuan Labuan', '16' => 'Wilayah Persekutuan Putrajaya'], 	'MZ' => ['P' => 'Cabo Delgado', 'G' => 'Gaza', 'I' => 'Inhambane', 'B' => 'Manica', 'L' => 'Maputo', 'N' => 'Nampula', 'A' => 'Niassa', 'S' => 'Sofala', 'T' => 'Tete', 'Q' => 'Zambezia'], 	'NA' => ['ER' => 'Erongo', 'HA' => 'Hardap', 'KA' => 'Karas', 'KE' => 'Kavango East', 'KW' => 'Kavango West', 'KH' => 'Khomas', 'KU' => 'Kunene', 'OW' => 'Ohangwena', 'OH' => 'Omaheke', 'OS' => 'Omusati', 'ON' => 'Oshana', 'OT' => 'Oshikoto', 'OD' => 'Otjozondjupa', 'CA' => 'Zambezi'], 	'NC' => ['-' => 'Province des iles Loyaute'], 	'NE' => ['1' => 'Agadez', '2' => 'Diffa', '3' => 'Dosso', '4' => 'Maradi', '8' => 'Niamey', '5' => 'Tahoua', '6' => 'Tillaberi', '7' => 'Zinder'], 	'NF' => ['-' => 'Norfolk Island'], 	'NG' => ['AB' => 'Abia', 'FC' => 'Abuja Federal Capital Territory', 'AD' => 'Adamawa', 'AK' => 'Akwa Ibom', 'AN' => 'Anambra', 'BA' => 'Bauchi', 'BY' => 'Bayelsa', 'BE' => 'Benue', 'BO' => 'Borno', 'CR' => 'Cross River', 'DE' => 'Delta', 'EB' => 'Ebonyi', 'ED' => 'Edo', 'EK' => 'Ekiti', 'EN' => 'Enugu', 'GO' => 'Gombe', 'IM' => 'Imo', 'JI' => 'Jigawa', 'KD' => 'Kaduna', 'KN' => 'Kano', 'KT' => 'Katsina', 'KE' => 'Kebbi', 'KO' => 'Kogi', 'KW' => 'Kwara', 'LA' => 'Lagos', 'NA' => 'Nasarawa', 'NI' => 'Niger', 'OG' => 'Ogun', 'ON' => 'Ondo', 'OS' => 'Osun', 'OY' => 'Oyo', 'PL' => 'Plateau', 'RI' => 'Rivers', 'SO' => 'Sokoto', 'TA' => 'Taraba', 'YO' => 'Yobe', 'ZA' => 'Zamfara'], 	'NI' => ['BO' => 'Boaco', 'CA' => 'Carazo', 'CI' => 'Chinandega', 'CO' => 'Chontales', 'AN' => 'Costa Caribe Norte', 'AS' => 'Costa Caribe Sur', 'ES' => 'Esteli', 'GR' => 'Granada', 'JI' => 'Jinotega', 'LE' => 'Leon', 'MD' => 'Madriz', 'MN' => 'Managua', 'MS' => 'Masaya', 'MT' => 'Matagalpa', 'NS' => 'Nueva Segovia', 'SJ' => 'Rio San Juan', 'RI' => 'Rivas'], 	'NL' => ['DR' => 'Drenthe', 'FL' => 'Flevoland', 'FR' => 'Fryslan', 'GE' => 'Gelderland', 'GR' => 'Groningen', 'LI' => 'Limburg', 'NB' => 'Noord-Brabant', 'NH' => 'Noord-Holland', 'OV' => 'Overijssel', 'UT' => 'Utrecht', 'ZE' => 'Zeeland', 'ZH' => 'Zuid-Holland'], 	'NO' => ['42' => 'Agder', '34' => 'Innlandet', '15' => 'More og Romsdal', '18' => 'Nordland', '03' => 'Oslo', '11' => 'Rogaland', '54' => 'Troms og Finnmark', '50' => 'Trondelag', '38' => 'Vestfold og Telemark', '46' => 'Vestland', '30' => 'Viken'], 	'NP' => ['P3' => 'Bagmati', 'P4' => 'Gandaki', 'P6' => 'Karnali', 'P1' => 'Koshi', 'P5' => 'Lumbini', 'P2' => 'Madhesh', ' P7' => 'Sudurpashchim'], 	'NR' => ['01' => 'Aiwo', '03' => 'Anetan', '14' => 'Yaren'], 	'NU' => ['-' => 'Niue'], 	'NZ' => ['AUK' => 'Auckland', 'BOP' => 'Bay of Plenty', 'CAN' => 'Canterbury', 'CIT' => 'Chatham Islands Territory', 'GIS' => 'Gisborne', 'WGN' => 'Greater Wellington', 'HKB' => 'Hawke\'s Bay', 'MWT' => 'Manawatu-Whanganui', 'MBH' => 'Marlborough', 'NSN' => 'Nelson', 'NTL' => 'Northland', 'OTA' => 'Otago', 'STL' => 'Southland', 'TKI' => 'Taranaki', 'TAS' => 'Tasman', 'WKO' => 'Waikato', 'WTC' => 'West Coast'], 	'OM' => ['DA' => 'Ad Dakhiliyah', 'BU' => 'Al Buraymi', 'WU' => 'Al Wusta', 'ZA' => 'Az Zahirah', 'BJ' => 'Janub al Batinah', 'SJ' => 'Janub ash Sharqiyah', 'MA' => 'Masqat', 'MU' => 'Musandam', 'BS' => 'Shamal al Batinah', 'SS' => 'Shamal ash Sharqiyah', 'ZU' => 'Zufar'], 	'PA' => ['1' => 'Bocas del Toro', '4' => 'Chiriqui', '2' => 'Cocle', '3' => 'Colon', '5' => 'Darien', '6' => 'Herrera', '7' => 'Los Santos', 'NB' => 'Ngobe-Bugle', '8' => 'Panama', '9' => 'Veraguas'], 	'PE' => ['AMA' => 'Amazonas', 'ANC' => 'Ancash', 'APU' => 'Apurimac', 'ARE' => 'Arequipa', 'AYA' => 'Ayacucho', 'CAJ' => 'Cajamarca', 'CUS' => 'Cusco', 'CAL' => 'El Callao', 'HUV' => 'Huancavelica', 'HUC' => 'Huanuco', 'ICA' => 'Ica', 'JUN' => 'Junin', 'LAL' => 'La Libertad', 'LAM' => 'Lambayeque', 'LIM' => 'Lima', 'LOR' => 'Loreto', 'MDD' => 'Madre de Dios', 'MOQ' => 'Moquegua', 'PAS' => 'Pasco', 'PIU' => 'Piura', 'PUN' => 'Puno', 'SAM' => 'San Martin', 'TAC' => 'Tacna', 'TUM' => 'Tumbes', 'UCA' => 'Ucayali'], 	'PF' => ['-' => 'Iles du Vent'], 	'PG' => ['NSB' => 'Bougainville', 'CPM' => 'Central', 'EBR' => 'East New Britain', 'ESW' => 'East Sepik', 'EHG' => 'Eastern Highlands', 'EPW' => 'Enga', 'MPM' => 'Madang', 'MRL' => 'Manus', 'MBA' => 'Milne Bay', 'MPL' => 'Morobe', 'NCD' => 'National Capital District (Port Moresby)', 'NIK' => 'New Ireland', 'SHM' => 'Southern Highlands', 'WBK' => 'West New Britain', 'SAN' => 'West Sepik', 'WPD' => 'Western', 'WHM' => 'Western Highlands'], 	'PH' => ['ABR' => 'Abra', 'AGN' => 'Agusan del Norte', 'AGS' => 'Agusan del Sur', 'AKL' => 'Aklan', 'ALB' => 'Albay', 'ANT' => 'Antique', 'APA' => 'Apayao', 'AUR' => 'Aurora', 'BAS' => 'Basilan', 'BAN' => 'Bataan', 'BTN' => 'Batanes', 'BTG' => 'Batangas', 'BEN' => 'Benguet', 'BIL' => 'Biliran', 'BOH' => 'Bohol', 'BUK' => 'Bukidnon', 'BUL' => 'Bulacan', 'CAG' => 'Cagayan', 'CAN' => 'Camarines Norte', 'CAS' => 'Camarines Sur', 'CAM' => 'Camiguin', 'CAP' => 'Capiz', 'CAT' => 'Catanduanes', 'CAV' => 'Cavite', 'CEB' => 'Cebu', 'NCO' => 'Cotabato', 'DAO' => 'Davao Oriental', 'COM' => 'Davao de Oro', 'DAV' => 'Davao del Norte', 'DAS' => 'Davao del Sur', 'DIN' => 'Dinagat Islands', 'EAS' => 'Eastern Samar', 'GUI' => 'Guimaras', 'IFU' => 'Ifugao', 'ILN' => 'Ilocos Norte', 'ILS' => 'Ilocos Sur', 'ILI' => 'Iloilo', 'ISA' => 'Isabela', 'KAL' => 'Kalinga', 'LUN' => 'La Union', 'LAG' => 'Laguna', 'LAN' => 'Lanao del Norte', 'LAS' => 'Lanao del Sur', 'LEY' => 'Leyte', 'MAG' => 'Maguindanao', 'MAD' => 'Marinduque', 'MAS' => 'Masbate', 'MDC' => 'Mindoro Occidental', 'MDR' => 'Mindoro Oriental', 'MSC' => 'Misamis Occidental', 'MSR' => 'Misamis Oriental', 'MOU' => 'Mountain Province', '00' => 'National Capital Region', 'NEC' => 'Negros Occidental', 'NER' => 'Negros Oriental', 'NSA' => 'Northern Samar', 'NUE' => 'Nueva Ecija', 'NUV' => 'Nueva Vizcaya', 'PLW' => 'Palawan', 'PAM' => 'Pampanga', 'PAN' => 'Pangasinan', 'QUE' => 'Quezon', 'QUI' => 'Quirino', 'RIZ' => 'Rizal', 'ROM' => 'Romblon', 'WSA' => 'Samar', 'SAR' => 'Sarangani', 'SIG' => 'Siquijor', 'SOR' => 'Sorsogon', 'SCO' => 'South Cotabato', 'SLE' => 'Southern Leyte', 'SUK' => 'Sultan Kudarat', 'SLU' => 'Sulu', 'SUN' => 'Surigao del Norte', 'SUR' => 'Surigao del Sur', 'TAR' => 'Tarlac', 'TAW' => 'Tawi-Tawi', 'ZMB' => 'Zambales', 'ZSI' => 'Zamboanga Sibugay', 'ZAN' => 'Zamboanga del Norte', 'ZAS' => 'Zamboanga del Sur'], 	'PK' => ['JK' => 'Azad Jammu and Kashmir', 'BA' => 'Balochistan', 'GB' => 'Gilgit-Baltistan', 'IS' => 'Islamabad', 'KP' => 'Khyber Pakhtunkhwa', 'PB' => 'Punjab', 'SD' => 'Sindh'], 	'PL' => ['02' => 'Dolnoslaskie', '04' => 'Kujawsko-Pomorskie', '10' => 'Lodzkie', '06' => 'Lubelskie', '08' => 'Lubuskie', '12' => 'Malopolskie', '14' => 'Mazowieckie', '16' => 'Opolskie', '18' => 'Podkarpackie', '20' => 'Podlaskie', '22' => 'Pomorskie', '24' => 'Slaskie', '26' => 'Swietokrzyskie', '28' => 'Warminsko-Mazurskie', '30' => 'Wielkopolskie', '32' => 'Zachodniopomorskie'], 	'PM' => ['-' => 'Saint Pierre and Miquelon'], 	'PN' => ['-' => 'Pitcairn'], 	'PR' => ['-' => 'Yauco'], 	'PS' => ['BTH' => 'Bethlehem', 'DEB' => 'Deir El Balah', 'GZA' => 'Gaza', 'HBN' => 'Hebron', 'JEN' => 'Jenin', 'JRH' => 'Jericho and Al Aghwar', 'JEM' => 'Jerusalem', 'KYS' => 'Khan Yunis', 'NBS' => 'Nablus', 'QQA' => 'Qalqilya', 'RFH' => 'Rafah', 'RBH' => 'Ramallah', 'SLT' => 'Salfit', 'TBS' => 'Tubas', 'TKM' => 'Tulkarm'], 	'PT' => ['01' => 'Aveiro', '02' => 'Beja', '03' => 'Braga', '04' => 'Braganca', '05' => 'Castelo Branco', '06' => 'Coimbra', '07' => 'Evora', '08' => 'Faro', '09' => 'Guarda', '10' => 'Leiria', '11' => 'Lisboa', '12' => 'Portalegre', '13' => 'Porto', '30' => 'Regiao Autonoma da Madeira', '20' => 'Regiao Autonoma dos Acores', '14' => 'Santarem', '15' => 'Setubal', '16' => 'Viana do Castelo', '17' => 'Vila Real', '18' => 'Viseu'], 	'PW' => ['004' => 'Airai', '150' => 'Koror', '212' => 'Melekeok', '214' => 'Ngaraard', '222' => 'Ngardmau', '224' => 'Ngatpang', '370' => 'Sonsorol'], 	'PY' => ['10' => 'Alto Parana', '13' => 'Amambay', 'ASU' => 'Asuncion', '19' => 'Boqueron', '5' => 'Caaguazu', '6' => 'Caazapa', '14' => 'Canindeyu', '11' => 'Central', '1' => 'Concepcion', '3' => 'Cordillera', '4' => 'Guaira', '7' => 'Itapua', '8' => 'Misiones', '12' => 'Neembucu', '9' => 'Paraguari', '15' => 'Presidente Hayes', '2' => 'San Pedro'], 	'QA' => ['DA' => 'Ad Dawhah', 'KH' => 'Al Khawr wa adh Dhakhirah', 'WA' => 'Al Wakrah', 'RA' => 'Ar Rayyan', 'MS' => 'Ash Shamal', 'ZA' => 'Az Za\'ayin', 'US' => 'Umm Salal'], 	'RE' => ['-' => 'Reunion'], 	'RO' => ['AB' => 'Alba', 'AR' => 'Arad', 'AG' => 'Arges', 'BC' => 'Bacau', 'BH' => 'Bihor', 'BN' => 'Bistrita-Nasaud', 'BT' => 'Botosani', 'BR' => 'Braila', 'BV' => 'Brasov', 'B' => 'Bucuresti', 'BZ' => 'Buzau', 'CL' => 'Calarasi', 'CS' => 'Caras-Severin', 'CJ' => 'Cluj', 'CT' => 'Constanta', 'CV' => 'Covasna', 'DB' => 'Dambovita', 'DJ' => 'Dolj', 'GL' => 'Galati', 'GR' => 'Giurgiu', 'GJ' => 'Gorj', 'HR' => 'Harghita', 'HD' => 'Hunedoara', 'IL' => 'Ialomita', 'IS' => 'Iasi', 'IF' => 'Ilfov', 'MM' => 'Maramures', 'MH' => 'Mehedinti', 'MS' => 'Mures', 'NT' => 'Neamt', 'OT' => 'Olt', 'PH' => 'Prahova', 'SJ' => 'Salaj', 'SM' => 'Satu Mare', 'SB' => 'Sibiu', 'SV' => 'Suceava', 'TR' => 'Teleorman', 'TM' => 'Timis', 'TL' => 'Tulcea', 'VL' => 'Valcea', 'VS' => 'Vaslui', 'VN' => 'Vrancea'], 	'RS' => ['00' => 'Beograd', '14' => 'Borski okrug', '11' => 'Branicevski okrug', '23' => 'Jablanicki okrug', '06' => 'Juznobacki okrug', '04' => 'Juznobanatski okrug', '09' => 'Kolubarski okrug', '28' => 'Kosovsko-Mitrovacki okrug', '08' => 'Macvanski okrug', '17' => 'Moravicki okrug', '20' => 'Nisavski okrug', '24' => 'Pcinjski okrug', '26' => 'Pecki okrug', '22' => 'Pirotski okrug', '10' => 'Podunavski okrug', '13' => 'Pomoravski okrug', '27' => 'Prizrenski okrug', '19' => 'Rasinski okrug', '18' => 'Raski okrug', '01' => 'Severnobacki okrug', '03' => 'Severnobanatski okrug', '02' => 'Srednjebanatski okrug', '07' => 'Sremski okrug', '12' => 'Sumadijski okrug', '21' => 'Toplicki okrug', '15' => 'Zajecarski okrug', '05' => 'Zapadnobacki okrug', '16' => 'Zlatiborski okrug'], 	'RU' => ['AD' => 'Adygeya, Respublika', 'AL' => 'Altay, Respublika', 'ALT' => 'Altayskiy kray', 'AMU' => 'Amurskaya oblast\'', 'ARK' => 'Arkhangel\'skaya oblast\'', 'AST' => 'Astrakhanskaya oblast\'', 'BA' => 'Bashkortostan, Respublika', 'BEL' => 'Belgorodskaya oblast\'', 'BRY' => 'Bryanskaya oblast\'', 'BU' => 'Buryatiya, Respublika', 'CE' => 'Chechenskaya Respublika', 'CHE' => 'Chelyabinskaya oblast\'', 'CHU' => 'Chukotskiy avtonomnyy okrug', 'CU' => 'Chuvashskaya Respublika', 'DA' => 'Dagestan, Respublika', 'IN' => 'Ingushetiya, Respublika', 'IRK' => 'Irkutskaya oblast\'', 'IVA' => 'Ivanovskaya oblast\'', 'KB' => 'Kabardino-Balkarskaya Respublika', 'KGD' => 'Kaliningradskaya oblast\'', 'KL' => 'Kalmykiya, Respublika', 'KLU' => 'Kaluzhskaya oblast\'', 'KAM' => 'Kamchatskiy kray', 'KC' => 'Karachayevo-Cherkesskaya Respublika', 'KR' => 'Kareliya, Respublika', 'KEM' => 'Kemerovskaya oblast\'', 'KHA' => 'Khabarovskiy kray', 'KK' => 'Khakasiya, Respublika', 'KHM' => 'Khanty-Mansiyskiy avtonomnyy okrug', 'KIR' => 'Kirovskaya oblast\'', 'KO' => 'Komi, Respublika', 'KOS' => 'Kostromskaya oblast\'', 'KDA' => 'Krasnodarskiy kray', 'KYA' => 'Krasnoyarskiy kray', 'KGN' => 'Kurganskaya oblast\'', 'KRS' => 'Kurskaya oblast\'', 'LEN' => 'Leningradskaya oblast\'', 'LIP' => 'Lipetskaya oblast\'', 'MAG' => 'Magadanskaya oblast\'', 'ME' => 'Mariy El, Respublika', 'MO' => 'Mordoviya, Respublika', 'MOS' => 'Moskovskaya oblast\'', 'MOW' => 'Moskva', 'MUR' => 'Murmanskaya oblast\'', 'NEN' => 'Nenetskiy avtonomnyy okrug', 'NIZ' => 'Nizhegorodskaya oblast\'', 'NGR' => 'Novgorodskaya oblast\'', 'NVS' => 'Novosibirskaya oblast\'', 'OMS' => 'Omskaya oblast\'', 'ORE' => 'Orenburgskaya oblast\'', 'ORL' => 'Orlovskaya oblast\'', 'PNZ' => 'Penzenskaya oblast\'', 'PER' => 'Permskiy kray', 'PRI' => 'Primorskiy kray', 'PSK' => 'Pskovskaya oblast\'', 'ROS' => 'Rostovskaya oblast\'', 'RYA' => 'Ryazanskaya oblast\'', 'SA' => 'Saha, Respublika', 'SAK' => 'Sakhalinskaya oblast\'', 'SAM' => 'Samarskaya oblast\'', 'SPE' => 'Sankt-Peterburg', 'SAR' => 'Saratovskaya oblast\'', 'SE' => 'Severnaya Osetiya, Respublika', 'SMO' => 'Smolenskaya oblast\'', 'STA' => 'Stavropol\'skiy kray', 'SVE' => 'Sverdlovskaya oblast\'', 'TAM' => 'Tambovskaya oblast\'', 'TA' => 'Tatarstan, Respublika', 'TOM' => 'Tomskaya oblast\'', 'TUL' => 'Tul\'skaya oblast\'', 'TVE' => 'Tverskaya oblast\'', 'TYU' => 'Tyumenskaya oblast\'', 'TY' => 'Tyva, Respublika', 'UD' => 'Udmurtskaya Respublika', 'ULY' => 'Ul\'yanovskaya oblast\'', 'VLA' => 'Vladimirskaya oblast\'', 'VGG' => 'Volgogradskaya oblast\'', 'VLG' => 'Vologodskaya oblast\'', 'VOR' => 'Voronezhskaya oblast\'', 'YAN' => 'Yamalo-Nenetskiy avtonomnyy okrug', 'YAR' => 'Yaroslavskaya oblast\'', 'YEV' => 'Yevreyskaya avtonomnaya oblast\'', 'ZAB' => 'Zabaykal\'skiy kray'], 	'RW' => ['02' => 'Est', '03' => 'Nord', '04' => 'Ouest', '05' => 'Sud', '01' => 'Ville de Kigali'], 	'SA' => ['14' => '\'Asir', '11' => 'Al Bahah', '08' => 'Al Hudud ash Shamaliyah', '12' => 'Al Jawf', '03' => 'Al Madinah al Munawwarah', '05' => 'Al Qasim', '01' => 'Ar Riyad', '04' => 'Ash Sharqiyah', '06' => 'Ha\'il', '09' => 'Jazan', '02' => 'Makkah al Mukarramah', '10' => 'Najran', '07' => 'Tabuk'], 	'SB' => ['GU' => 'Guadalcanal', 'MK' => 'Makira-Ulawa', 'ML' => 'Malaita', 'WE' => 'Western'], 	'SC' => ['02' => 'Anse Boileau', '05' => 'Anse Royale', '01' => 'Anse aux Pins', '06' => 'Baie Lazare', '07' => 'Baie Sainte Anne', '08' => 'Beau Vallon', '10' => 'Bel Ombre', '11' => 'Cascade', '16' => 'English River', '13' => 'Grand Anse Mahe', '14' => 'Grand Anse Praslin', '15' => 'La Digue', '20' => 'Pointe Larue', '23' => 'Takamaka'], 	'SD' => ['NB' => 'Blue Nile', 'DC' => 'Central Darfur', 'GD' => 'Gedaref', 'GZ' => 'Gezira', 'KA' => 'Kassala', 'KH' => 'Khartoum', 'DN' => 'North Darfur', 'KN' => 'North Kordofan', 'NO' => 'Northern', 'RS' => 'Red Sea', 'NR' => 'River Nile', 'SI' => 'Sennar', 'DS' => 'South Darfur', 'KS' => 'South Kordofan', 'DW' => 'West Darfur', 'GK' => 'West Kordofan', 'NW' => 'White Nile'], 	'SE' => ['K' => 'Blekinge lan', 'W' => 'Dalarnas lan', 'X' => 'Gavleborgs lan', 'I' => 'Gotlands lan', 'N' => 'Hallands lan', 'Z' => 'Jamtlands lan', 'F' => 'Jonkopings lan', 'H' => 'Kalmar lan', 'G' => 'Kronobergs lan', 'BD' => 'Norrbottens lan', 'T' => 'Orebro lan', 'E' => 'Ostergotlands lan', 'M' => 'Skane lan', 'D' => 'Sodermanlands lan', 'AB' => 'Stockholms lan', 'C' => 'Uppsala lan', 'S' => 'Varmlands lan', 'AC' => 'Vasterbottens lan', 'Y' => 'Vasternorrlands lan', 'U' => 'Vastmanlands lan', 'O' => 'Vastra Gotalands lan'], 	'SG' => ['-' => 'Singapore'], 	'SH' => ['HL' => 'Saint Helena'], 	'SI' => ['001' => 'Ajdovscina', '213' => 'Ankaran', '195' => 'Apace', '002' => 'Beltinci', '148' => 'Benedikt', '149' => 'Bistrica ob Sotli', '003' => 'Bled', '150' => 'Bloke', '004' => 'Bohinj', '005' => 'Borovnica', '006' => 'Bovec', '151' => 'Braslovce', '007' => 'Brda', '009' => 'Brezice', '008' => 'Brezovica', '152' => 'Cankova', '011' => 'Celje', '012' => 'Cerklje na Gorenjskem', '013' => 'Cerknica', '014' => 'Cerkno', '196' => 'Cirkulane', '015' => 'Crensovci', '017' => 'Crnomelj', '018' => 'Destrnik', '019' => 'Divaca', '154' => 'Dobje', '020' => 'Dobrepolje', '155' => 'Dobrna', '021' => 'Dobrova-Polhov Gradec', '156' => 'Dobrovnik', '023' => 'Domzale', '024' => 'Dornava', '025' => 'Dravograd', '026' => 'Duplek', '207' => 'Gorje', '029' => 'Gornja Radgona', '031' => 'Gornji Petrovci', '158' => 'Grad', '032' => 'Grosuplje', '159' => 'Hajdina', '160' => 'Hoce-Slivnica', '161' => 'Hodos', '162' => 'Horjul', '034' => 'Hrastnik', '035' => 'Hrpelje-Kozina', '036' => 'Idrija', '037' => 'Ig', '038' => 'Ilirska Bistrica', '039' => 'Ivancna Gorica', '040' => 'Izola', '041' => 'Jesenice', '042' => 'Jursinci', '043' => 'Kamnik', '044' => 'Kanal', '045' => 'Kidricevo', '046' => 'Kobarid', '047' => 'Kobilje', '048' => 'Kocevje', '049' => 'Komen', '164' => 'Komenda', '050' => 'Koper', '197' => 'Kosanjevica na Krki', '165' => 'Kostel', '052' => 'Kranj', '053' => 'Kranjska Gora', '166' => 'Krizevci', '054' => 'Krsko', '055' => 'Kungota', '056' => 'Kuzma', '057' => 'Lasko', '058' => 'Lenart', '059' => 'Lendava', '060' => 'Litija', '061' => 'Ljubljana', '063' => 'Ljutomer', '208' => 'Log-Dragomer', '064' => 'Logatec', '065' => 'Loska dolina', '066' => 'Loski Potok', '167' => 'Lovrenc na Pohorju', '067' => 'Luce', '068' => 'Lukovica', '069' => 'Majsperk', '198' => 'Makole', '070' => 'Maribor', '168' => 'Markovci', '071' => 'Medvode', '072' => 'Menges', '073' => 'Metlika', '074' => 'Mezica', '169' => 'Miklavz na Dravskem polju', '075' => 'Miren-Kostanjevica', '212' => 'Mirna', '170' => 'Mirna Pec', '076' => 'Mislinja', '199' => 'Mokronog-Trebelno', '077' => 'Moravce', '079' => 'Mozirje', '080' => 'Murska Sobota', '081' => 'Muta', '082' => 'Naklo', '083' => 'Nazarje', '084' => 'Nova Gorica', '085' => 'Novo Mesto', '086' => 'Odranci', '171' => 'Oplotnica', '087' => 'Ormoz', '090' => 'Piran', '091' => 'Pivka', '092' => 'Podcetrtek', '172' => 'Podlehnik', '200' => 'Poljcane', '173' => 'Polzela', '094' => 'Postojna', '174' => 'Prebold', '095' => 'Preddvor', '175' => 'Prevalje', '096' => 'Ptuj', '097' => 'Puconci', '098' => 'Race-Fram', '099' => 'Radece', '100' => 'Radenci', '101' => 'Radlje ob Dravi', '102' => 'Radovljica', '103' => 'Ravne na Koroskem', '176' => 'Razkrizje', '209' => 'Recica ob Savinji', '201' => 'Rence-Vogrsko', '104' => 'Ribnica', '106' => 'Rogaska Slatina', '105' => 'Rogasovci', '108' => 'Ruse', '033' => 'Salovci', '109' => 'Semic', '183' => 'Sempeter-Vrtojba', '117' => 'Sencur', '118' => 'Sentilj', '119' => 'Sentjernej', '120' => 'Sentjur', '211' => 'Sentrupert', '110' => 'Sevnica', '111' => 'Sezana', '121' => 'Skocjan', '122' => 'Skofja Loka', '123' => 'Skofljica', '112' => 'Slovenj Gradec', '113' => 'Slovenska Bistrica', '114' => 'Slovenske Konjice', '124' => 'Smarje pri Jelsah', '206' => 'Smarjeske Toplice', '125' => 'Smartno ob Paki', '194' => 'Smartno pri Litiji', '179' => 'Sodrazica', '180' => 'Solcava', '126' => 'Sostanj', '115' => 'Starse', '127' => 'Store', '203' => 'Straza', '204' => 'Sveta Trojica v Slovenskih goricah', '182' => 'Sveti Andraz v Slovenskih Goricah', '116' => 'Sveti Jurij ob Scavnici', '210' => 'Sveti Jurij v Slovenskih goricah', '205' => 'Sveti Tomaz', '184' => 'Tabor', '010' => 'Tisina', '128' => 'Tolmin', '129' => 'Trbovlje', '130' => 'Trebnje', '185' => 'Trnovska Vas', '131' => 'Trzic', '186' => 'Trzin', '132' => 'Turnisce', '133' => 'Velenje', '187' => 'Velika Polana', '134' => 'Velike Lasce', '188' => 'Verzej', '135' => 'Videm', '136' => 'Vipava', '137' => 'Vitanje', '138' => 'Vodice', '139' => 'Vojnik', '189' => 'Vransko', '140' => 'Vrhnika', '141' => 'Vuzenica', '142' => 'Zagorje ob Savi', '190' => 'Zalec', '143' => 'Zavrc', '146' => 'Zelezniki', '191' => 'Zetale', '147' => 'Ziri', '144' => 'Zrece', '193' => 'Zuzemberk'], 	'SJ' => ['-' => 'Svalbard and Jan Mayen'], 	'SK' => ['BC' => 'Banskobystricky kraj', 'BL' => 'Bratislavsky kraj', 'KI' => 'Kosicky kraj', 'NI' => 'Nitriansky kraj', 'PV' => 'Presovsky kraj', 'TC' => 'Trenciansky kraj', 'TA' => 'Trnavsky kraj', 'ZI' => 'Zilinsky kraj'], 	'SL' => ['E' => 'Eastern', 'NW' => 'North Western', 'N' => 'Northern', 'S' => 'Southern', 'W' => 'Western Area'], 	'SM' => ['02' => 'Chiesanuova', '07' => 'Citta di San Marino', '04' => 'Faetano', '09' => 'Serravalle'], 	'SN' => ['DK' => 'Dakar', 'DB' => 'Diourbel', 'FK' => 'Fatick', 'KA' => 'Kaffrine', 'KL' => 'Kaolack', 'KE' => 'Kedougou', 'KD' => 'Kolda', 'LG' => 'Louga', 'MT' => 'Matam', 'SL' => 'Saint-Louis', 'SE' => 'Sedhiou', 'TC' => 'Tambacounda', 'TH' => 'Thies', 'ZG' => 'Ziguinchor'], 	'SO' => ['AW' => 'Awdal', 'BN' => 'Banaadir', 'BR' => 'Bari', 'BY' => 'Bay', 'GA' => 'Galguduud', 'HI' => 'Hiiraan', 'JH' => 'Jubbada Hoose', 'MU' => 'Mudug', 'NU' => 'Nugaal', 'SH' => 'Shabeellaha Hoose', 'SO' => 'Sool', 'TO' => 'Togdheer', 'WO' => 'Woqooyi Galbeed'], 	'SR' => ['BR' => 'Brokopondo', 'CM' => 'Commewijne', 'CR' => 'Coronie', 'NI' => 'Nickerie', 'PR' => 'Para', 'PM' => 'Paramaribo', 'SA' => 'Saramacca', 'SI' => 'Sipaliwini', 'WA' => 'Wanica'], 	'SS' => ['EC' => 'Central Equatoria', 'EE' => 'Eastern Equatoria', 'JG' => 'Jonglei', 'BN' => 'Northern Bahr el Ghazal', 'UY' => 'Unity', 'NU' => 'Upper Nile', 'BW' => 'Western Bahr el Ghazal', 'EW' => 'Western Equatoria'], 	'ST' => ['01' => 'Agua Grande', 'P' => 'Principe'], 	'SV' => ['AH' => 'Ahuachapan', 'CA' => 'Cabanas', 'CH' => 'Chalatenango', 'CU' => 'Cuscatlan', 'LI' => 'La Libertad', 'PA' => 'La Paz', 'UN' => 'La Union', 'MO' => 'Morazan', 'SM' => 'San Miguel', 'SS' => 'San Salvador', 'SV' => 'San Vicente', 'SA' => 'Santa Ana', 'SO' => 'Sonsonate', 'US' => 'Usulutan'], 	'SX' => ['-' => 'Sint Maarten (Dutch Part)'], 	'SY' => ['HA' => 'Al Hasakah', 'LA' => 'Al Ladhiqiyah', 'QU' => 'Al Qunaytirah', 'RA' => 'Ar Raqqah', 'SU' => 'As Suwayda\'', 'DR' => 'Dar\'a', 'DY' => 'Dayr az Zawr', 'DI' => 'Dimashq', 'HL' => 'Halab', 'HM' => 'Hamah', 'HI' => 'Hims', 'ID' => 'Idlib', 'RD' => 'Rif Dimashq', 'TA' => 'Tartus'], 	'SZ' => ['HH' => 'Hhohho', 'LU' => 'Lubombo', 'MA' => 'Manzini'], 	'TC' => ['-' => 'Turks and Caicos Islands'], 	'TD' => ['BG' => 'Bahr el Ghazal', 'CB' => 'Chari-Baguirmi', 'GR' => 'Guera', 'LC' => 'Lac', 'LR' => 'Logone-Oriental', 'OD' => 'Ouaddai', 'ND' => 'Ville de Ndjamena'], 	'TF' => ['-' => 'French Southern Territories'], 	'TG' => ['C' => 'Centrale', 'K' => 'Kara', 'M' => 'Maritime', 'P' => 'Plateaux', 'S' => 'Savanes'], 	'TH' => ['37' => 'Amnat Charoen', '15' => 'Ang Thong', '38' => 'Bueng Kan', '31' => 'Buri Ram', '24' => 'Chachoengsao', '18' => 'Chai Nat', '36' => 'Chaiyaphum', '22' => 'Chanthaburi', '50' => 'Chiang Mai', '57' => 'Chiang Rai', '20' => 'Chon Buri', '86' => 'Chumphon', '46' => 'Kalasin', '62' => 'Kamphaeng Phet', '71' => 'Kanchanaburi', '40' => 'Khon Kaen', '81' => 'Krabi', '10' => 'Krung Thep Maha Nakhon', '52' => 'Lampang', '51' => 'Lamphun', '42' => 'Loei', '16' => 'Lop Buri', '58' => 'Mae Hong Son', '44' => 'Maha Sarakham', '49' => 'Mukdahan', '26' => 'Nakhon Nayok', '73' => 'Nakhon Pathom', '48' => 'Nakhon Phanom', '30' => 'Nakhon Ratchasima', '60' => 'Nakhon Sawan', '80' => 'Nakhon Si Thammarat', '55' => 'Nan', '96' => 'Narathiwat', '39' => 'Nong Bua Lam Phu', '43' => 'Nong Khai', '12' => 'Nonthaburi', '13' => 'Pathum Thani', '94' => 'Pattani', '82' => 'Phangnga', '93' => 'Phatthalung', '56' => 'Phayao', '67' => 'Phetchabun', '76' => 'Phetchaburi', '66' => 'Phichit', '65' => 'Phitsanulok', '14' => 'Phra Nakhon Si Ayutthaya', '54' => 'Phrae', '83' => 'Phuket', '25' => 'Prachin Buri', '77' => 'Prachuap Khiri Khan', '85' => 'Ranong', '70' => 'Ratchaburi', '21' => 'Rayong', '45' => 'Roi Et', '27' => 'Sa Kaeo', '47' => 'Sakon Nakhon', '11' => 'Samut Prakan', '74' => 'Samut Sakhon', '75' => 'Samut Songkhram', '19' => 'Saraburi', '91' => 'Satun', '33' => 'Si Sa Ket', '17' => 'Sing Buri', '90' => 'Songkhla', '64' => 'Sukhothai', '72' => 'Suphan Buri', '84' => 'Surat Thani', '32' => 'Surin', '63' => 'Tak', '92' => 'Trang', '23' => 'Trat', '34' => 'Ubon Ratchathani', '41' => 'Udon Thani', '61' => 'Uthai Thani', '53' => 'Uttaradit', '95' => 'Yala', '35' => 'Yasothon'], 	'TJ' => ['DU' => 'Dushanbe', 'KT' => 'Khatlon', 'GB' => 'Kuhistoni Badakhshon', 'RA' => 'Nohiyahoi Tobei Jumhuri', 'SU' => 'Sughd'], 	'TK' => ['-' => 'Tokelau'], 	'TL' => ['AN' => 'Ainaro', 'BO' => 'Bobonaro', 'CO' => 'Cova Lima', 'DI' => 'Dili', 'ER' => 'Ermera', 'LI' => 'Liquica', 'MT' => 'Manatuto', 'VI' => 'Viqueque'], 	'TM' => ['A' => 'Ahal', 'B' => 'Balkan', 'D' => 'Dasoguz', 'L' => 'Lebap', 'M' => 'Mary'], 	'TN' => ['31' => 'Beja', '13' => 'Ben Arous', '23' => 'Bizerte', '81' => 'Gabes', '71' => 'Gafsa', '32' => 'Jendouba', '41' => 'Kairouan', '42' => 'Kasserine', '73' => 'Kebili', '12' => 'L\'Ariana', '14' => 'La Manouba', '33' => 'Le Kef', '53' => 'Mahdia', '82' => 'Medenine', '52' => 'Monastir', '21' => 'Nabeul', '61' => 'Sfax', '43' => 'Sidi Bouzid', '34' => 'Siliana', '51' => 'Sousse', '83' => 'Tataouine', '72' => 'Tozeur', '11' => 'Tunis', '22' => 'Zaghouan'], 	'TO' => ['01' => '\'Eua', '02' => 'Ha\'apai', '03' => 'Niuas', '04' => 'Tongatapu', '05' => 'Vava\'u'], 	'TR' => ['01' => 'Adana', '02' => 'Adiyaman', '03' => 'Afyonkarahisar', '04' => 'Agri', '68' => 'Aksaray', '05' => 'Amasya', '06' => 'Ankara', '07' => 'Antalya', '75' => 'Ardahan', '08' => 'Artvin', '09' => 'Aydin', '10' => 'Balikesir', '74' => 'Bartin', '72' => 'Batman', '69' => 'Bayburt', '11' => 'Bilecik', '12' => 'Bingol', '13' => 'Bitlis', '14' => 'Bolu', '15' => 'Burdur', '16' => 'Bursa', '17' => 'Canakkale', '18' => 'Cankiri', '19' => 'Corum', '20' => 'Denizli', '21' => 'Diyarbakir', '81' => 'Duzce', '22' => 'Edirne', '23' => 'Elazig', '24' => 'Erzincan', '25' => 'Erzurum', '26' => 'Eskisehir', '27' => 'Gaziantep', '28' => 'Giresun', '29' => 'Gumushane', '30' => 'Hakkari', '31' => 'Hatay', '76' => 'Igdir', '32' => 'Isparta', '34' => 'Istanbul', '35' => 'Izmir', '46' => 'Kahramanmaras', '78' => 'Karabuk', '70' => 'Karaman', '36' => 'Kars', '37' => 'Kastamonu', '38' => 'Kayseri', '79' => 'Kilis', '71' => 'Kirikkale', '39' => 'Kirklareli', '40' => 'Kirsehir', '41' => 'Kocaeli', '42' => 'Konya', '43' => 'Kutahya', '44' => 'Malatya', '45' => 'Manisa', '47' => 'Mardin', '33' => 'Mersin', '48' => 'Mugla', '49' => 'Mus', '50' => 'Nevsehir', '51' => 'Nigde', '52' => 'Ordu', '80' => 'Osmaniye', '53' => 'Rize', '54' => 'Sakarya', '55' => 'Samsun', '63' => 'Sanliurfa', '56' => 'Siirt', '57' => 'Sinop', '73' => 'Sirnak', '58' => 'Sivas', '59' => 'Tekirdag', '60' => 'Tokat', '61' => 'Trabzon', '62' => 'Tunceli', '64' => 'Usak', '65' => 'Van', '77' => 'Yalova', '66' => 'Yozgat', '67' => 'Zonguldak'], 	'TT' => ['ARI' => 'Arima', 'CHA' => 'Chaguanas', 'CTT' => 'Couva-Tabaquite-Talparo', 'DMN' => 'Diego Martin', 'MRC' => 'Mayaro-Rio Claro', 'PED' => 'Penal-Debe', 'PTF' => 'Point Fortin', 'POS' => 'Port of Spain', 'PRT' => 'Princes Town', 'SFO' => 'San Fernando', 'SJL' => 'San Juan-Laventille', 'SGE' => 'Sangre Grande', 'SIP' => 'Siparia', 'TOB' => 'Tobago', 'TUP' => 'Tunapuna-Piarco'], 	'TV' => ['FUN' => 'Funafuti'], 	'TW' => ['CHA' => 'Changhua', 'CYQ' => 'Chiayi', 'HSQ' => 'Hsinchu', 'HUA' => 'Hualien', 'KHH' => 'Kaohsiung', 'KEE' => 'Keelung', 'KIN' => 'Kinmen', 'LIE' => 'Lienchiang', 'MIA' => 'Miaoli', 'NAN' => 'Nantou', 'NWT' => 'New Taipei', 'PEN' => 'Penghu', 'PIF' => 'Pingtung', 'TXG' => 'Taichung', 'TNN' => 'Tainan', 'TPE' => 'Taipei', 'TTT' => 'Taitung', 'TAO' => 'Taoyuan', 'ILA' => 'Yilan', 'YUN' => 'Yunlin'], 	'TZ' => ['01' => 'Arusha', '02' => 'Dar es Salaam', '03' => 'Dodoma', '27' => 'Geita', '04' => 'Iringa', '05' => 'Kagera', '06' => 'Kaskazini Pemba', '07' => 'Kaskazini Unguja', '28' => 'Katavi', '08' => 'Kigoma', '09' => 'Kilimanjaro', '10' => 'Kusini Pemba', '11' => 'Kusini Unguja', '12' => 'Lindi', '26' => 'Manyara', '13' => 'Mara', '14' => 'Mbeya', '15' => 'Mjini Magharibi', '16' => 'Morogoro', '17' => 'Mtwara', '18' => 'Mwanza', '29' => 'Njombe', '19' => 'Pwani', '20' => 'Rukwa', '21' => 'Ruvuma', '22' => 'Shinyanga', '30' => 'Simiyu', '23' => 'Singida', '31' => 'Songwe', '24' => 'Tabora', '25' => 'Tanga'], 	'UA' => ['43' => 'Avtonomna Respublika Krym', '71' => 'Cherkaska oblast', '74' => 'Chernihivska oblast', '77' => 'Chernivetska oblast', '12' => 'Dnipropetrovska oblast', '14' => 'Donetska oblast', '26' => 'Ivano-Frankivska oblast', '63' => 'Kharkivska oblast', '65' => 'Khersonska oblast', '68' => 'Khmelnytska oblast', '35' => 'Kirovohradska oblast', '30' => 'Kyiv', '32' => 'Kyivska oblast', '09' => 'Luhanska oblast', '46' => 'Lvivska oblast', '48' => 'Mykolaivska oblast', '51' => 'Odeska oblast', '53' => 'Poltavska oblast', '56' => 'Rivnenska oblast', '40' => 'Sevastopol', '59' => 'Sumska oblast', '61' => 'Ternopilska oblast', '05' => 'Vinnytska oblast', '07' => 'Volynska oblast', '21' => 'Zakarpatska oblast', '23' => 'Zaporizka oblast', '18' => 'Zhytomyrska oblast'], 	'UG' => ['314' => 'Abim', '301' => 'Adjumani', '322' => 'Agago', '323' => 'Alebtong', '302' => 'Apac', '303' => 'Arua', '218' => 'Bududa', '201' => 'Bugiri', '420' => 'Buhweju', '117' => 'Buikwe', '219' => 'Bukedea', '118' => 'Bukomansibi', '225' => 'Bulambuli', '416' => 'Buliisa', '401' => 'Bundibugyo', '402' => 'Bushenyi', '202' => 'Busia', '120' => 'Buvuma', '226' => 'Buyende', '121' => 'Gomba', '304' => 'Gulu', '403' => 'Hoima', '417' => 'Ibanda', '203' => 'Iganga', '418' => 'Isingiro', '204' => 'Jinja', '318' => 'Kaabong', '404' => 'Kabale', '405' => 'Kabarole', '213' => 'Kaberamaido', '101' => 'Kalangala', '222' => 'Kaliro', '122' => 'Kalungu', '102' => 'Kampala', '205' => 'Kamuli', '413' => 'Kamwenge', '414' => 'Kanungu', '206' => 'Kapchorwa', '406' => 'Kasese', '207' => 'Katakwi', '112' => 'Kayunga', '407' => 'Kibaale', '103' => 'Kiboga', '227' => 'Kibuku', '419' => 'Kiruhura', '421' => 'Kiryandongo', '408' => 'Kisoro', '305' => 'Kitgum', '319' => 'Koboko', '325' => 'Kole', '208' => 'Kumi', '228' => 'Kween', '123' => 'Kyankwanzi', '422' => 'Kyegegwa', '415' => 'Kyenjojo', '326' => 'Lamwo', '307' => 'Lira', '229' => 'Luuka', '104' => 'Luwero', '124' => 'Lwengo', '114' => 'Lyantonde', '223' => 'Manafwa', '320' => 'Maracha', '105' => 'Masaka', '409' => 'Masindi', '214' => 'Mayuge', '209' => 'Mbale', '410' => 'Mbarara', '423' => 'Mitooma', '115' => 'Mityana', '308' => 'Moroto', '309' => 'Moyo', '106' => 'Mpigi', '107' => 'Mubende', '108' => 'Mukono', '311' => 'Nakapiripirit', '116' => 'Nakaseke', '109' => 'Nakasongola', '230' => 'Namayingo', '327' => 'Napak', '310' => 'Nebbi', '424' => 'Ntoroko', '411' => 'Ntungamo', '328' => 'Nwoya', '321' => 'Oyam', '312' => 'Pader', '210' => 'Pallisa', '110' => 'Rakai', '425' => 'Rubirizi', '412' => 'Rukungiri', '111' => 'Sembabule', '426' => 'Sheema', '215' => 'Sironko', '211' => 'Soroti', '212' => 'Tororo', '113' => 'Wakiso', '313' => 'Yumbe', '330' => 'Zombo'], 	'UM' => ['95' => 'Palmyra Atoll'], 	'US' => ['AL' => 'Alabama', 'AK' => 'Alaska', 'AZ' => 'Arizona', 'AR' => 'Arkansas', 'CA' => 'California', 'CO' => 'Colorado', 'CT' => 'Connecticut', 'DE' => 'Delaware', 'DC' => 'District of Columbia', 'FL' => 'Florida', 'GA' => 'Georgia', 'HI' => 'Hawaii', 'ID' => 'Idaho', 'IL' => 'Illinois', 'IN' => 'Indiana', 'IA' => 'Iowa', 'KS' => 'Kansas', 'KY' => 'Kentucky', 'LA' => 'Louisiana', 'ME' => 'Maine', 'MD' => 'Maryland', 'MA' => 'Massachusetts', 'MI' => 'Michigan', 'MN' => 'Minnesota', 'MS' => 'Mississippi', 'MO' => 'Missouri', 'MT' => 'Montana', 'NE' => 'Nebraska', 'NV' => 'Nevada', 'NH' => 'New Hampshire', 'NJ' => 'New Jersey', 'NM' => 'New Mexico', 'NY' => 'New York', 'NC' => 'North Carolina', 'ND' => 'North Dakota', 'OH' => 'Ohio', 'OK' => 'Oklahoma', 'OR' => 'Oregon', 'PA' => 'Pennsylvania', 'RI' => 'Rhode Island', 'SC' => 'South Carolina', 'SD' => 'South Dakota', 'TN' => 'Tennessee', 'TX' => 'Texas', 'UT' => 'Utah', 'VT' => 'Vermont', 'VA' => 'Virginia', 'WA' => 'Washington', 'WV' => 'West Virginia', 'WI' => 'Wisconsin', 'WY' => 'Wyoming'], 	'UY' => ['AR' => 'Artigas', 'CA' => 'Canelones', 'CL' => 'Cerro Largo', 'CO' => 'Colonia', 'DU' => 'Durazno', 'FS' => 'Flores', 'FD' => 'Florida', 'LA' => 'Lavalleja', 'MA' => 'Maldonado', 'MO' => 'Montevideo', 'PA' => 'Paysandu', 'RN' => 'Rio Negro', 'RV' => 'Rivera', 'RO' => 'Rocha', 'SA' => 'Salto', 'SJ' => 'San Jose', 'SO' => 'Soriano', 'TA' => 'Tacuarembo', 'TT' => 'Treinta y Tres'], 	'UZ' => ['AN' => 'Andijon', 'BU' => 'Buxoro', 'FA' => 'Farg\'ona', 'JI' => 'Jizzax', 'NG' => 'Namangan', 'NW' => 'Navoiy', 'QA' => 'Qashqadaryo', 'QR' => 'Qoraqalpog\'iston Respublikasi', 'SA' => 'Samarqand', 'SI' => 'Sirdaryo', 'SU' => 'Surxondaryo', 'TK' => 'Toshkent', 'XO' => 'Xorazm'], 	'VA' => ['-' => 'Vatican City'], 	'VC' => ['01' => 'Charlotte', '06' => 'Grenadines', '04' => 'Saint George', '05' => 'Saint Patrick'], 	'VE' => ['Z' => 'Amazonas', 'B' => 'Anzoategui', 'C' => 'Apure', 'D' => 'Aragua', 'E' => 'Barinas', 'F' => 'Bolivar', 'G' => 'Carabobo', 'H' => 'Cojedes', 'Y' => 'Delta Amacuro', 'W' => 'Dependencias Federales', 'A' => 'Distrito Capital', 'I' => 'Falcon', 'J' => 'Guarico', 'X' => 'La Guaira', 'K' => 'Lara', 'L' => 'Merida', 'M' => 'Miranda', 'N' => 'Monagas', 'O' => 'Nueva Esparta', 'P' => 'Portuguesa', 'R' => 'Sucre', 'S' => 'Tachira', 'T' => 'Trujillo', 'U' => 'Yaracuy', 'V' => 'Zulia'], 	'VG' => ['-' => 'Virgin Islands, British'], 	'VI' => ['-' => 'Virgin Islands, U.S.'], 	'VN' => ['44' => 'An Giang', '43' => 'Ba Ria - Vung Tau', '54' => 'Bac Giang', '53' => 'Bac Kan', '55' => 'Bac Lieu', '56' => 'Bac Ninh', '50' => 'Ben Tre', '31' => 'Binh Dinh', '57' => 'Binh Duong', '58' => 'Binh Phuoc', '40' => 'Binh Thuan', '59' => 'Ca Mau', 'CT' => 'Can Tho', '04' => 'Cao Bang', 'DN' => 'Da Nang', '33' => 'Dak Lak', '72' => 'Dak Nong', '71' => 'Dien Bien', '39' => 'Dong Nai', '45' => 'Dong Thap', '30' => 'Gia Lai', '03' => 'Ha Giang', '63' => 'Ha Nam', 'HN' => 'Ha Noi', '23' => 'Ha Tinh', '61' => 'Hai Duong', 'HP' => 'Hai Phong', '73' => 'Hau Giang', 'SG' => 'Ho Chi Minh', '14' => 'Hoa Binh', '66' => 'Hung Yen', '34' => 'Khanh Hoa', '47' => 'Kien Giang', '28' => 'Kon Tum', '01' => 'Lai Chau', '35' => 'Lam Dong', '09' => 'Lang Son', '02' => 'Lao Cai', '41' => 'Long An', '67' => 'Nam Dinh', '22' => 'Nghe An', '18' => 'Ninh Binh', '36' => 'Ninh Thuan', '68' => 'Phu Tho', '32' => 'Phu Yen', '24' => 'Quang Binh', '27' => 'Quang Nam', '29' => 'Quang Ngai', '13' => 'Quang Ninh', '25' => 'Quang Tri', '52' => 'Soc Trang', '05' => 'Son La', '37' => 'Tay Ninh', '20' => 'Thai Binh', '69' => 'Thai Nguyen', '21' => 'Thanh Hoa', '26' => 'Thua Thien-Hue', '46' => 'Tien Giang', '51' => 'Tra Vinh', '07' => 'Tuyen Quang', '49' => 'Vinh Long', '70' => 'Vinh Phuc', '06' => 'Yen Bai'], 	'VU' => ['SAM' => 'Sanma', 'SEE' => 'Shefa', 'TAE' => 'Tafea'], 	'WF' => ['SG' => 'Sigave', 'UV' => 'Uvea'], 	'WS' => ['AT' => 'Atua', 'FA' => 'Fa\'asaleleaga', 'GI' => 'Gagaifomauga', 'TU' => 'Tuamasaga'], 	'YE' => ['AD' => '\'Adan', 'AM' => '\'Amran', 'DA' => 'Ad Dali\'', 'BA' => 'Al Bayda\'', 'HU' => 'Al Hudaydah', 'JA' => 'Al Jawf', 'MR' => 'Al Mahrah', 'MW' => 'Al Mahwit', 'SA' => 'Amanat al \'Asimah', 'DH' => 'Dhamar', 'HD' => 'Hadramawt', 'HJ' => 'Hajjah', 'IB' => 'Ibb', 'LA' => 'Lahij', 'MA' => 'Ma\'rib', 'SD' => 'Sa\'dah', 'SN' => 'San\'a\'', 'SH' => 'Shabwah', 'TA' => 'Ta\'izz'], 	'YT' => ['-' => 'Tsingoni'], 	'ZA' => ['EC' => 'Eastern Cape', 'FS' => 'Free State', 'GP' => 'Gauteng', 'KZN' => 'Kwazulu-Natal', 'LP' => 'Limpopo', 'MP' => 'Mpumalanga', 'NW' => 'North-West', 'NC' => 'Northern Cape', 'WC' => 'Western Cape'], 	'ZM' => ['02' => 'Central', '08' => 'Copperbelt', '03' => 'Eastern', '04' => 'Luapula', '09' => 'Lusaka', '10' => 'Muchinga', '06' => 'North-Western', '05' => 'Northern', '07' => 'Southern', '01' => 'Western'], 	'ZW' => ['BU' => 'Bulawayo', 'HA' => 'Harare', 'MA' => 'Manicaland', 'MC' => 'Mashonaland Central', 'ME' => 'Mashonaland East', 'MW' => 'Mashonaland West', 'MV' => 'Masvingo', 'MN' => 'Matabeleland North', 'MS' => 'Matabeleland South', 'MI' => 'Midlands'],
		];

		if (!isset($regions[$countryCode])) {
			return false;
		}

		foreach ($regions[$countryCode] as $regionCode => $region) {
			if (strtolower($region) == strtolower($regionName)) {
				return $regionCode;
			}
		}

		return false;
	}
}
