# -*- coding: utf-8 -*-
# vim: ts=4 sw=4 tw=100 et ai si
#
# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: BSD-3-Clause
#
# Author: Artem Bityutskiy <artem.bityutskiy@linux.intel.com>

"""
This module provides API to MSR 0x771 (MSR_HWP_CAPABILITIES). This is an architectural MSR found on
many Intel platforms.
"""

from pepclibs.msr import _FeaturedMSR, PMEnable

# The Hardware Power Management Capabilities Model Specific Register.
MSR_HWP_CAPABILITIES = 0x771

# Description of CPU features controlled by the the Power Control MSR. Please, refer to the notes
# for '_FeaturedMSR.FEATURES' for more comments.
FEATURES = {
    "max_perf": {
        "name": "Max. CPU performance",
        "sname": "CPU",
        "iosname": "CPU",
        "help": """The maximum CPU performance (P01).""",
        "cpuflags": {"hwp"},
        "type": "int",
        "bits": (7, 0),
    },
    "base_perf": {
        "name": "Base CPU performance",
        "sname": "CPU",
        "iosname": "CPU",
        "help": """The base (guaranteed) CPU performance (P1).""",
        "cpuflags": {"hwp"},
        "type": "int",
        "bits": (15, 8),
    },
    "eff_perf": {
        "name": "Max. efficiency CPU performance",
        "sname": "CPU",
        "iosname": "CPU",
        "help": """The maximum efficiency CPU performance (Pn).""",
        "cpuflags": {"hwp"},
        "type": "int",
        "bits": (23, 16),
    },
    "min_perf": {
        "name": "Min. CPU performance",
        "sname": "CPU",
        "iosname": "CPU",
        "help": """The minimum CPU performance (Pm).""",
        "cpuflags": {"hwp"},
        "type": "int",
        "bits": (31, 24),
    },
}

class HWPCapabilities(_FeaturedMSR.FeaturedMSR):
    """
    This class provides API to MSR 0x771 (MSR_HWP_CAPABILITIES). This is an architectural MSR found
    on many Intel platforms.
    """

    regaddr = MSR_HWP_CAPABILITIES
    regname = "MSR_HWP_CAPABILITIES"
    vendor = "GenuineIntel"

    def _set_baseclass_attributes(self):
        """Set the attributes the superclass requires."""

        self.features = FEATURES

    def __init__(self, cpuinfo, pman=None, msr=None):
        """
        The class constructor. The argument are as follows.
          * cpuinfo - CPU information object generated by 'CPUInfo.CPUInfo()'.
          * pman - the process manager object that defines the host to run the measurements on.
          * msr - the 'MSR.MSR()' object to use for writing to the MSR register.
        """

        super().__init__(cpuinfo, pman=pman, msr=msr)

        unsupported_cpus = []
        for pkg in self._cpuinfo.get_packages():
            cpus = self._cpuinfo.package_to_cpus(pkg)

            # Make sure the CPU supports HWP and has HWP is enabled.
            cpuflags = self._cpuinfo.info["flags"][cpus[0]]
            if "hwp" in cpuflags:
                if self._msr.read_cpu_bits(PMEnable.MSR_PM_ENABLE,
                                           PMEnable.FEATURES["hwp"]["bits"], cpus[0]):
                    continue

            # If HWP is not supported or not enabled for any CPU in the package, all the other CPUs
            # are expected to be the same.
            unsupported_cpus += cpus

        for finfo in self._features.values():
            if "cpuflags" in finfo and "hwp" in finfo["cpuflags"]:
                for cpu in unsupported_cpus:
                    finfo["supported"][cpu] = False
