# -*- coding: utf-8 -*-
# vim: ts=4 sw=4 tw=100 et ai si
#
# Copyright (C) 2020-2022 Intel Corporation
# SPDX-License-Identifier: BSD-3-Clause
#
# Authors: Artem Bityutskiy <artem.bityutskiy@linux.intel.com>
#          Antti Laakso <antti.laakso@linux.intel.com>
#          Niklas Neronin <niklas.neronin@intel.com>

"""
This class implements per-CPU data caching. The cache is indexed by an element name and CPU number.
"""

from pepclibs.helperlibs import ClassHelpers
from pepclibs.helperlibs.Exceptions import ErrorNotFound

class PerCPUCache:
    """
    This class implements per_CPU data caching. The cache is indexed by data element name and CPU
    number. It takes the CPU scope (global 0, package 3, etc) into account as well. The cache uses
    the write-through policy.
    """

    def is_cached(self, name, cpu):
        """
        Check if '(name, cpu)' exists in the cache. Return 'True' if the item was found and 'False'
        otherwise. Arguments are the same as in 'get()'.
        """

        try:
            self.get(name, cpu)
        except ErrorNotFound:
            return False
        return True

    def get(self, name, cpu):
        """
        Look up the '(name, cpu)' in the cache. Return the value if the item was found, raise
        'ErrorNotFound' otherwise. The argument are as follows.
          * name - name of the data element.
          * cpu - an integer CPU number.
        """

        if not self._enable_cache:
            raise ErrorNotFound("caching is disabled")

        try:
            return self._cache[name][cpu]
        except KeyError:
            raise ErrorNotFound(f"'{name}' is not cached for CPU {cpu}") from None

    def remove(self, name, cpu, sname="CPU"):
        """
        Remove '(name, cpu)' and all the other items sharing the same scope from the cache.
          * name - name of the data element.
          * cpu - an integer CPU number.
          * sname - scope name (e.g. "package", "core").

        Returns 'True' if there are elements in the cache left, and 'False' if there are no more
        elements in the cache.
        """

        if not self._enable_cache:
            return

        if sname == "global":
            del self._cache[name]
            return

        cpus = self._cpuinfo.get_cpu_siblings(cpu, sname)

        for cpu in cpus: # pylint: disable=redefined-argument-from-local
            try:
                del self._cache[name][cpu]
            except KeyError:
                pass

    def add(self, name, cpu, val, sname="CPU"):
        """
        Add value 'val' for item '(name, cpu)' to the cache. Add it also for each CPU sharing the
        same scope. The argument are as follows.
          * name - name of the data element.
          * cpu - an integer CPU number.
          * val - value to get cached.
          * sname - scope name (e.g. "package", "core").

        Return 'val'.
        """

        if not self._enable_cache:
            return val

        cpus = self._cpuinfo.get_cpu_siblings(cpu, sname)

        if name not in self._cache:
            self._cache[name] = {}
        for cpu in cpus: # pylint: disable=redefined-argument-from-local
            self._cache[name][cpu] = val

        return val

    def __init__(self, cpuinfo, enable_cache=True):
        """
        The class constructor. The argument are as follows.
          * cpuinfo - CPU information object generated by 'CPUInfo.CPUInfo()'.
          * enable_cache - this argument can be used to disable caching.
        """

        self._cpuinfo = cpuinfo
        self._enable_cache = enable_cache

        if not self._enable_cache:
            return

        self._cache = {}

    def close(self):
        """Uninitialize the class object."""

        ClassHelpers.close(self, unref_attrs=("_cpuinfo",))
