# -*- coding: utf-8 -*-
# vim: ts=4 sw=4 tw=100 et ai si
#
# Copyright (C) 2020-2023 Intel Corporation
# SPDX-License-Identifier: BSD-3-Clause
#
# Authors: Antti Laakso <antti.laakso@linux.intel.com>
#          Artem Bityutskiy <artem.bityutskiy@linux.intel.com>
#          Niklas Neronin <niklas.neronin@intel.com>

"""
This module provides C-state management API.
"""

from pepclibs.helperlibs import ClassHelpers
from pepclibs.helperlibs.Exceptions import Error, ErrorNotSupported
from pepclibs import _PropsClassBase, CPUIdle
from pepclibs.msr import PowerCtl, PCStateConfigCtl

# Make the exception class be available for users.
from pepclibs._PropsClassBase import ErrorUsePerCPU # pylint: disable=unused-import

# This dictionary describes the C-state properties this module supports. Many of the properties are
# just features controlled by an MSR, such as "c1e_autopromote" from 'PowerCtl.FEATURES'.
#
# While this dictionary is user-visible and can be used, it is not recommended, because it is not
# complete. This dictionary is extended by 'CStates' objects. Use the full dictionary via
# 'CStates.props'.
#
# Some properties have scope name set to 'None' because the scope may be different for different
# systems. In such cases, the scope can be obtained via 'CStates.get_sname()'.
PROPS = {
    "pkg_cstate_limit": {
        "name": "Package C-state limit",
        "type": "str",
        "sname": None,
        "mnames": ("msr",),
        "writable": True,
        "subprops": ("pkg_cstate_limit_lock", "pkg_cstate_limits", "pkg_cstate_limit_aliases"),
    },
    "pkg_cstate_limit_lock": {
        "name": "Package C-state limit lock",
        "type": "bool",
        "sname": None,
        "mnames": ("msr",),
        "writable": False,
    },
    "pkg_cstate_limits": {
        "name": "Available package C-state limits",
        "type": "list[str]",
        # Conceptually this is per-package, but in practice it is global on all current platforms.
        "sname": "global",
        "mnames": ("doc",),
        "writable": False,
    },
    "pkg_cstate_limit_aliases": {
        "name": "Package C-state limit aliases",
        "type": "dict[str,str]",
        # Conceptually this is per-package, but in practice it is global on all current platforms.
        "sname": "global",
        "mnames": ("doc",),
        "writable": False,
    },
    "c1_demotion": {
        "name": "C1 demotion",
        "type": "bool",
        "sname": None,
        "mnames": ("msr",),
        "writable": True,
    },
    "c1_undemotion": {
        "name": "C1 undemotion",
        "type": "bool",
        "sname": None,
        "mnames": ("msr",),
        "writable": True,
    },
    "c1e_autopromote": {
        "name": "C1E autopromote",
        "type": "bool",
        "sname": None,
        "mnames": ("msr",),
        "writable": True,
    },
    "cstate_prewake": {
        "name": "C-state prewake",
        "type": "bool",
        "sname": None,
        "mnames": ("msr",),
        "writable": True,
    },
    "idle_driver": {
        "name": "Idle driver",
        "type": "str",
        "sname": "global",
        "mnames": ("sysfs",),
        "writable": False,
    },
    "governor": {
        "name": "Idle governor",
        "type": "str",
        "sname": "global",
        "mnames": ("sysfs",),
        "writable": True,
    },
    "governors": {
        "name": "Available idle governors",
        "type": "list[str]",
        "sname": "global",
        "mnames": ("sysfs",),
        "writable": False,
    },
}

class CStates(_PropsClassBase.PropsClassBase):
    """
    This class provides C-state management API.

    Public methods overview.
    1. All the get/set property methods defined by the '_PropsClassBase.PropsClassBase' base class
       (refer to its docstring for more information).
    2. Enable or disable multiple C-states for multiple CPUs via Linux sysfs interfaces:
       'enable_cstates()', 'disable_cstates()'.
    3. Get C-state(s) information.
       * For multiple CPUs and multiple C-states: get_cstates_info().
       * For single CPU and multiple C-states: 'get_cpu_cstates_info()'.
       * For single CPU and a single C-state:  'get_cpu_cstate_info()'.
    """

    def _get_cpuidle(self):
        """Returns a 'CPUIdle()' object."""

        if not self._cpuidle:
            self._cpuidle = CPUIdle.CPUIdle(self._pman, cpuinfo=self._cpuinfo,
                                            enable_cache=self._enable_cache)
        return self._cpuidle

    def _get_powerctl(self):
        """Return an instance of 'PowerCtl' class."""

        if not self._powerctl:
            msr = self._get_msr()
            self._powerctl = PowerCtl.PowerCtl(pman=self._pman, cpuinfo=self._cpuinfo, msr=msr)
        return self._powerctl

    def _get_pcstatectl(self):
        """Return an instance of 'PCStateConfigCtl' class."""

        if not self._pcstatectl:
            msr = self._get_msr()
            self._pcstatectl = PCStateConfigCtl.PCStateConfigCtl(pman=self._pman,
                                                                 cpuinfo=self._cpuinfo, msr=msr)
        return self._pcstatectl

    def get_cstates_info(self, cpus="all", csnames="all"):
        """Same as 'CPUIdle.get_cstates_info()'."""

        yield from self._get_cpuidle().get_cstates_info(cpus=cpus, csnames=csnames)

    def get_cpu_cstates_info(self, cpu, csnames="all"):
        """Same as 'CPUIdle.get_cpu_cstates_info()'."""

        return self._get_cpuidle().get_cpu_cstates_info(cpu, csnames=csnames)

    def get_cpu_cstate_info(self, cpu, csname):
        """Same as 'CPUIdle.get_cpu_cstate_info()'."""

        return self._get_cpuidle().get_cpu_cstate_info(cpu, csname)

    def enable_cstates(self, csnames="all", cpus="all", mnames=None):
        """
        Same as 'CPUIdle.enable_cstates()', except for the 'mnames' argument, which is has no
        effect, only checked to to be 'sysfs' on 'None.
        """

        mnames = self._normalize_mnames(mnames, allow_readonly=False)
        if "sysfs" not in mnames:
            mnames = ", ".join(mnames)
            raise ErrorNotSupported(f"cannot disable C-states, unsupported methods: {mnames}.\n"
                                    f"Use the 'sysfs' method instead.")

        return self._get_cpuidle().enable_cstates(csnames=csnames, cpus=cpus)

    def disable_cstates(self, csnames="all", cpus="all", mnames=None):
        """
        Same as 'CPUIdle.disable_cstates()', except for the 'mnames' argument, which is has no
        effect, only checked to to be 'sysfs' on 'None.
        """

        mnames = self._normalize_mnames(mnames, allow_readonly=False)
        if "sysfs" not in mnames:
            mnames = ", ".join(mnames)
            raise ErrorNotSupported(f"cannot disable C-states, unsupported methods: {mnames}.\n"
                                    f"Use the 'sysfs' method instead.")

        return self._get_cpuidle().disable_cstates(csnames=csnames, cpus=cpus)

    def get_prop_from_msr(self, pname, cpus):
        """
        For every CPU in 'cpus', yield '(cpu, val)' pairs, where 'val' is value of property 'pname',
        provided by 'MSR_POWER_CTL' or 'MSR_PKG_CST_CONFIG_CONTROL'.
        """

        if pname in PowerCtl.FEATURES:
            module = self._get_powerctl()
        else:
            module = self._get_pcstatectl()

        yield from module.read_feature(pname, cpus=cpus)

    def _get_pkg_cstate_limit(self, pname, cpus):
        """
        For every CPU in 'cpus', yield '(cpu, val)' pairs, where 'val' is the 'pkg_cstate_limit' or
        a related property value.
        """

        pcstatectl = self._get_pcstatectl()

        if pname == "pkg_cstate_limit_lock":
            yield from pcstatectl.read_feature(pname, cpus=cpus)
        else:
            for cpu, features in pcstatectl.read_feature("pkg_cstate_limit", cpus=cpus):
                yield cpu, features[pname]

    def _get_cpuidle_prop(self, pname, cpus):
        """
        For every CPU in 'cpus', yield '(cpu, val)' pairs, where 'val' is value of property 'pname',
        provided by the 'CPUIdle' module.
        """

        if pname == "idle_driver":
            val = self._get_cpuidle().get_idle_driver()
        elif pname == "governor":
            val = self._get_cpuidle().get_current_governor()
        else:
            val = self._get_cpuidle().get_available_governors()

        # All the properties are global, sor read only once and yield the same value for all CPUs.
        for cpu in cpus:
            yield (cpu, val)

    def _get_prop_cpus(self, pname, cpus, mname):
        """
        For every CPU in 'cpus', yield a '(cpu, val)' tuple, 'val' is property 'pname' value for CPU
        'cpu'. Use mechanism 'mname'.
        """

        if pname.startswith("pkg_cstate_"):
            yield from self._get_pkg_cstate_limit(pname, cpus)
        elif pname in ("idle_driver", "governor", "governors"):
            yield from self._get_cpuidle_prop(pname, cpus)
        elif mname == "msr":
            yield from self.get_prop_from_msr(pname, cpus)
        else:
            raise Error(f"BUG: unsupported property '{pname}'")

    def _set_prop_cpus(self, pname, val, cpus, mname):
        """Set property 'pname' to value 'val' for CPUs in 'cpus'. Use mechanism 'mname'."""

        if mname == "msr":
            if pname in PowerCtl.FEATURES:
                self._get_powerctl().write_feature(pname, val, cpus=cpus)
                return
            if pname in PCStateConfigCtl.FEATURES:
                self._get_pcstatectl().write_feature(pname, val, cpus=cpus)
                return

        if mname == "sysfs":
            if pname == "governor":
                self._get_cpuidle().set_current_governor(val)
                return

        raise Error(f"BUG: unsupported property '{pname}'")

    def _set_sname(self, pname):
        """Set scope name for property 'pname'."""

        prop = self._props[pname]
        if prop["sname"]:
            return

        finfo = None
        if pname in PCStateConfigCtl.FEATURES:
            finfo = self._get_pcstatectl().features
        elif pname in PowerCtl.FEATURES:
            finfo = self._get_powerctl().features

        if finfo:
            prop["sname"] = finfo[pname]["sname"]
            prop["iosname"] = finfo[pname]["iosname"]
            self.props[pname]["sname"] = prop["sname"]
        else:
            raise Error(f"BUG: unexpected property \"{pname}\"")

    def _init_props_dict(self): # pylint: disable=arguments-differ
        """Initialize the 'props' dictionary."""

        super()._init_props_dict(PROPS)

    def __init__(self, pman=None, cpuinfo=None, cpuidle=None, msr=None, enable_cache=True):
        """
        The class constructor. The arguments are as follows.
          * pman - the process manager object that defines the target host.
          * cpuinfo - CPU information object generated by 'CPUInfo.CPUInfo()'.
          * cpuidle - a 'CPUIdle.CPUIdle()' object which should be used for reading and setting
                      requestable C-state properties.
          * msr - an 'MSR.MSR()' object which should be used for accessing MSR registers.
          * enable_cache - this argument can be used to disable caching.
        """

        super().__init__(pman=pman, cpuinfo=cpuinfo, msr=msr, enable_cache=enable_cache)

        self._cpuidle = cpuidle
        self._close_cpuidle = cpuidle is None

        self._powerctl = None
        self._pcstatectl = None

        self._init_props_dict()

    def close(self):
        """Uninitialize the class object."""

        close_attrs = ("_pcstatectl", "_powerctl", "_cpuidle")
        ClassHelpers.close(self, close_attrs=close_attrs)

        super().close()
