<?php

declare(strict_types=1);

namespace Imi\Util;

/**
 * 对象及数组帮助类
 * 智能识别数组和对象，支持对a.b.c这样的name属性进行操作.
 */
class ObjectArrayHelper
{
    use \Imi\Util\Traits\TStaticClass;

    /**
     * 获取值
     */
    public static function get(array|object &$object, string $name, mixed $default = null): mixed
    {
        if ('' !== $name)
        {
            $names = explode('.', $name);
            $result = &$object;
            foreach ($names as $nameItem)
            {
                if (\is_array($result))
                {
                    // 数组
                    if (isset($result[$nameItem]))
                    {
                        $result = &$result[$nameItem];
                    }
                    else
                    {
                        return $default;
                    }
                }
                elseif (\is_object($result))
                {
                    // 对象
                    if (isset($result->{$nameItem}))
                    {
                        $result = &$result->{$nameItem};
                    }
                    else
                    {
                        return $default;
                    }
                }
                else
                {
                    return $default;
                }
            }

            return $result;
        }

        return $default;
    }

    /**
     * 设置值
     */
    public static function set(array|object &$object, string $name, mixed $value): void
    {
        if ('' === $name)
        {
            return;
        }
        $names = explode('.', $name);
        $lastName = array_pop($names);
        $data = &$object;
        if ($names)
        {
            foreach ($names as $nameItem)
            {
                if (\is_array($data))
                {
                    $data[$nameItem] ??= [];
                    $data = &$data[$nameItem];
                }
                elseif (\is_object($data))
                {
                    $data->{$nameItem} ??= new \stdClass();
                    $data = &$data->{$nameItem};
                }
            }
        }
        if (\is_array($data))
        {
            $data[$lastName] = $value;
        }
        elseif (\is_object($data))
        {
            $data->{$lastName} = $value;
        }
    }

    /**
     * 移除值
     */
    public static function remove(array|object &$object, string $name): void
    {
        if ('' === $name)
        {
            return;
        }
        $names = explode('.', $name);
        $lastName = array_pop($names);
        $data = &$object;
        if ($names)
        {
            foreach ($names as $nameItem)
            {
                if (\is_array($data))
                {
                    $data[$nameItem] ??= [];
                    $data = &$data[$nameItem];
                }
                elseif (\is_object($data))
                {
                    $data->{$nameItem} ??= new \stdClass();
                    $data = &$data->{$nameItem};
                }
            }
        }
        if (\is_array($data))
        {
            unset($data[$lastName]);
        }
        elseif (\is_object($data))
        {
            unset($data->{$lastName});
        }
    }

    /**
     * 值是否存在.
     */
    public static function exists(array|object $object, string $name): bool
    {
        return null !== static::get($object, $name);
    }

    /**
     * 将第二纬某字段值放入到一个数组中
     * 功能类似array_column，这个方法也支持对象
     */
    public static function column(array $array, string $columnName): array
    {
        $result = [];
        foreach ($array as $row)
        {
            if (\is_object($row))
            {
                $result[] = $row->{$columnName};
            }
            else
            {
                $result[] = $row[$columnName];
            }
        }

        return $result;
    }

    /**
     * 过滤属性.
     *
     * $mode只允许取值为：allow/deny
     */
    public static function filter(array|object &$object, array $fields, string $mode = 'allow'): void
    {
        if ('allow' === $mode)
        {
            $unsetKeys = [];
            foreach ($object as $field => $value)
            {
                if (!\in_array($field, $fields))
                {
                    $unsetKeys[] = $field;
                }
            }
            foreach ($unsetKeys as $key)
            {
                static::remove($object, $key);
            }
        }
        elseif ('deny' === $mode)
        {
            foreach ($fields as $field)
            {
                static::remove($object, $field);
            }
        }
        else
        {
            throw new \InvalidArgumentException(sprintf('Unknow mode %s', $mode));
        }
    }

    public static function toArray(object $object): array
    {
        if ($object instanceof \ArrayObject)
        {
            return $object->getArrayCopy();
        }
        elseif (is_iterable($object))
        {
            return iterator_to_array($object);
        }
        else
        {
            $array = [];
            foreach ($object as $k => $v)
            {
                $array[$k] = $v;
            }

            return $array;
        }
    }
}
