<?php

declare(strict_types=1);

namespace Imi\Server\DataParser;

use Imi\ConnectionContext;
use Imi\RequestContext;
use Imi\Server\Contract\IServer;
use Imi\Server\Protocol;
use Imi\Server\ServerManager;

/**
 * 数据处理器.
 */
class DataParser
{
    /**
     * 编码为存储格式.
     */
    public function encode(mixed $data, ?string $serverName = null): mixed
    {
        $parserClass = $this->getParserClass($serverName);
        if (null === $parserClass)
        {
            return $data;
        }
        if ($serverName)
        {
            return ServerManager::getServer($serverName)->getBean($parserClass)->encode($data);
        }
        else
        {
            return RequestContext::getServerBean($parserClass)->encode($data);
        }
    }

    /**
     * 解码为php变量.
     */
    public function decode(mixed $data, ?string $serverName = null): mixed
    {
        $parserClass = $this->getParserClass($serverName);
        if (null === $parserClass)
        {
            return $data;
        }
        if ($serverName)
        {
            return ServerManager::getServer($serverName)->getBean($parserClass)->decode($data);
        }
        else
        {
            return RequestContext::getServerBean($parserClass)->decode($data);
        }
    }

    /**
     * 获取处理器类.
     */
    public function getParserClass(?string $serverName = null): ?string
    {
        $requestContext = RequestContext::getContext();
        if ($serverName)
        {
            $server = ServerManager::getServer($serverName);
        }
        else
        {
            $server = $requestContext['server'] ?? null;
        }
        /** @var IServer $server */
        if (Protocol::WEBSOCKET === $server->getProtocol())
        {
            if ($dataParser = ConnectionContext::get('dataParser', null, null, $serverName))
            {
                return $dataParser;
            }
        }

        return $server->getConfig()['dataParser'] ?? JsonObjectParser::class;
    }
}
