<?php

declare(strict_types=1);

namespace Imi\Redis;

use Imi\Log\Log;
use Imi\Pool\BasePoolResource;

class RedisResource extends BasePoolResource
{
    /**
     * 连接配置.
     */
    private array $config = [];

    public function __construct(\Imi\Pool\Interfaces\IPool $pool,
        /**
         * Redis 对象
         */
        private ?RedisHandler $redis, array $config)
    {
        parent::__construct($pool);

        if (isset($config['timeout']))
        {
            $config['timeout'] = (float) $config['timeout'];
        }
        if (isset($config['db']))
        {
            $config['db'] = (int) $config['db'];
        }
        $this->config = $config;
    }

    /**
     * {@inheritDoc}
     */
    public function open(): bool
    {
        RedisManager::initRedisConnection($this->redis, $this->config);

        return true;
    }

    /**
     * {@inheritDoc}
     */
    public function close(): void
    {
        $this->redis->close();
    }

    /**
     * {@inheritDoc}
     */
    public function getInstance(): mixed
    {
        return $this->redis;
    }

    /**
     * {@inheritDoc}
     */
    public function reset(): void
    {
        $config = $this->config;
        $redis = $this->redis;
        if (!$redis->isCluster() && $redis->isConnected() && ($db = $config['db'] ?? 0) !== $redis->getDBNum() && !$redis->select($db))
        {
            throw new \RedisException($redis->getLastError());
        }
        $optScan = $config['options'][\Redis::OPT_SCAN] ?? \Redis::SCAN_RETRY;
        if (!$redis->setOption(\Redis::OPT_SCAN, $optScan))
        {
            throw new \RuntimeException(sprintf('Redis setOption %s=%s failed', \Redis::OPT_SCAN, $optScan));
        }
    }

    /**
     * {@inheritDoc}
     */
    public function checkState(): bool
    {
        $redis = $this->redis;
        if ($redis->isCluster())
        {
            return true;
        }
        else
        {
            try
            {
                $result = $redis->ping();

                // PHPRedis 扩展，5.0.0 版本开始，ping() 返回为 true，旧版本为 +PONG
                return true === $result || '+PONG' === $result;
            }
            catch (\Throwable $th)
            {
                Log::error($th);

                return false;
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public function isOpened(): bool
    {
        $redis = $this->redis;

        return $redis->isCluster() || $redis->isConnected();
    }
}
