<?php

declare(strict_types=1);

namespace Imi\Cron;

use Imi\Aop\Annotation\Inject;
use Imi\Bean\Annotation\Bean;
use Imi\Cron\Contract\ICronManager;
use Imi\Cron\Util\CronUtil;

/**
 * 定时任务工作类.
 */
#[Bean(name: 'CronWorker')]
class CronWorker
{
    #[Inject(name: 'CronManager')]
    protected ICronManager $cronManager;

    /**
     * 执行任务
     */
    public function exec(string $id, mixed $data, string|callable|null $task, string $type): void
    {
        $message = '';
        $success = false;
        try
        {
            if (null === $task)
            {
                $taskObj = $this->cronManager->getTask($id);
                if (!$taskObj)
                {
                    throw new \RuntimeException(sprintf('Can not found task %s', $id));
                }
                $task = $taskObj->getTask();
            }
            $taskCallable = $this->cronManager->getTaskCallable($id, $task, $type);
            if (\is_callable($taskCallable))
            {
                try
                {
                    $taskCallable($id, $data);
                    $success = true;
                }
                catch (\Throwable $th)
                {
                    $code = $th->getCode();
                    throw new \RuntimeException(sprintf('Task %s execution failed, message: %s', $id, $th->getMessage()), \is_int($code) ? $code : 0, $th);
                }
            }
            else
            {
                throw new \RuntimeException(sprintf('Task %s does not a callable', $id));
            }
            $success = true;
        }
        catch (\Throwable $th)
        {
            $message = $th->getMessage();
            throw $th;
        }
        finally
        {
            CronUtil::reportCronResult($id, $success, $message);
        }
    }
}
