#! /usr/bin/awk -f

# Copyright 2009 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.

# CDDL HEADER START
#
# The contents of this file are subject to the terms of the
# Common Development and Distribution License (the "License").
# You may not use this file except in compliance with the License.
#
# You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
# or http://www.opensolaris.org/os/licensing.
# See the License for the specific language governing permissions
# and limitations under the License.
#
# When distributing Covered Code, include this CDDL HEADER in each
# file and include the License file at usr/src/OPENSOLARIS.LICENSE.
# If applicable, add the following below this CDDL HEADER, with the
# fields enclosed by brackets "[]" replaced with your own identifying
# information: Portions Copyright [yyyy] [name of copyright owner]
#
# CDDL HEADER END

#-
#	$NetBSD: usb/devlist2h.awk,v 1.9 2001/01/18 20:28:22 jdolecek Exp $
#  $FreeBSD: src/sys/tools/usbdevs2h.awk,v 1.7 2005/01/07 02:29:25 imp Exp $
#
# Copyright (c) 1995, 1996 Christopher G. Demetriou
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. All advertising materials mentioning features or use of this software
#    must display the following acknowledgement:
#      This product includes software developed by Christopher G. Demetriou.
# 4. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

function usage()
{
	print "usage: usbdevs2h.awk <srcfile> [-d|-D|-h|-H]";
	exit 1;
}

function header(file)
{
	if (os == "NetBSD")
		printf("/*\t\$NetBSD\$\t*/\n\n") > file
	else if (os == "FreeBSD")
		printf("/* \$FreeBSD\$ */\n\n") > file
	else if (os == "OpenBSD")
		printf("/*\t\$OpenBSD\$\t*/\n\n") > file

	printf("/*\n") > file
	printf(" * THIS FILE IS AUTOMATICALLY GENERATED.  DO NOT EDIT.\n") \
	    > file
	printf(" *\n") > file
	printf(" * generated from:\n") > file
	printf(" *  %s\n", VERSION) > file
	printf(" */\n") > file
}

function header_guard(file)
{
	printf("\n") > file
	printf("#ifndef\t_SYS_USB_USBDEVS_H\n") > file
	printf("#define\t_SYS_USB_USBDEVS_H\n\n") > file
}

function header_trailer(file)
{
	printf("\n") > file
	printf("#endif\t/* _SYS_USB_USBDEVS_H */\n") > file
}

function vendor(hfile)
{
	nvendors++

	vendorindex[$2] = nvendors;		# record index for this name, for later.
	vendors[nvendors, 1] = $2;		# name
	vendors[nvendors, 2] = $3;		# id
	if (hfile)
		printf("#define\tUSB_VENDOR_%s\t%s\t", vendors[nvendors, 1],
		    vendors[nvendors, 2]) > hfile

	i = 3; f = 4;

	# comments
	ocomment = oparen = 0
	if (f <= NF) {
		if (hfile)
			printf("\t/* ") > hfile
		ocomment = 1;
	}
	while (f <= NF) {
		if ($f == "#") {
			if (hfile)
				printf("(") > hfile
			oparen = 1
			f++
			continue
		}
		if (oparen) {
			if (hfile)
				printf("%s", $f) > hfile
			if (f < NF && hfile)
				printf(" ") > hfile
			f++
			continue
		}
		vendors[nvendors, i] = $f
		if (hfile)
			printf("%s", vendors[nvendors, i]) > hfile
		if (f < NF && hfile)
			printf(" ") > hfile
		i++; f++;
	}
	if (oparen && hfile)
		printf(")") > hfile
	if (ocomment && hfile)
		printf(" */") > hfile
	if (hfile)
		printf("\n") > hfile
}

function product(hfile)
{
	nproducts++

	products[nproducts, 1] = $2;		# vendor name
	products[nproducts, 2] = $3;		# product id
	products[nproducts, 3] = $4;		# id
	if (hfile)
		printf("#define\tUSB_PRODUCT_%s_%s\t%s\t", \
		  products[nproducts, 1], products[nproducts, 2], \
		  products[nproducts, 3]) > hfile

	i=4; f = 5;

	# comments
	ocomment = oparen = 0
	if (f <= NF) {
		if (hfile)
			printf("\t/* ") > hfile
		ocomment = 1;
	}
	while (f <= NF) {
		if ($f == "#") {
			if (hfile)
				printf("(") > hfile
			oparen = 1
			f++
			continue
		}
		if (oparen) {
			if (hfile)
				printf("%s", $f) > hfile
			if (f < NF && hfile)
				printf(" ") > hfile
			f++
			continue
		}
		products[nproducts, i] = $f
		if (hfile)
			printf("%s", products[nproducts, i]) > hfile
		if (f < NF && hfile)
			printf(" ") > hfile
		i++; f++;
	}
	if (oparen && hfile)
		printf(")") > hfile
	if (ocomment && hfile)
		printf(" */") > hfile
	if (hfile)
		printf("\n") > hfile
}

function dump_dfile(dfile)
{
	printf("\n") > dfile
	printf("const struct usb_knowndev usb_knowndevs[] = {\n") > dfile
	for (i = 1; i <= nproducts; i++) {
		printf("\t{\n") > dfile
		printf("\t    USB_VENDOR_%s, USB_PRODUCT_%s_%s,\n",
		    products[i, 1], products[i, 1], products[i, 2]) > dfile
		printf("\t    ") > dfile
		printf("0") > dfile
		printf(",\n") > dfile

		vendi = vendorindex[products[i, 1]];
		printf("\t    \"") > dfile
		j = 3;
		needspace = 0;
		while (vendors[vendi, j] != "") {
			if (needspace)
				printf(" ") > dfile
			printf("%s", vendors[vendi, j]) > dfile
			needspace = 1
			j++
		}
		printf("\",\n") > dfile

		printf("\t    \"") > dfile
		j = 4;
		needspace = 0;
		while (products[i, j] != "") {
			if (needspace)
				printf(" ") > dfile
			printf("%s", products[i, j]) > dfile
			needspace = 1
			j++
		}
		printf("\",\n") > dfile
		printf("\t},\n") > dfile
	}
	for (i = 1; i <= nvendors; i++) {
		printf("\t{\n") > dfile
		printf("\t    USB_VENDOR_%s, 0,\n", vendors[i, 1]) > dfile
		printf("\t    USB_KNOWNDEV_NOPROD,\n") > dfile
		printf("\t    \"") > dfile
		j = 3;
		needspace = 0;
		while (vendors[i, j] != "") {
			if (needspace)
				printf(" ") > dfile
			printf("%s", vendors[i, j]) > dfile
			needspace = 1
			j++
		}
		printf("\",\n") > dfile
		printf("\t    NULL,\n") > dfile
		printf("\t},\n") > dfile
	}
	printf("\t{ 0, 0, 0, NULL, NULL, }\n") > dfile
	printf("};\n") > dfile
}

BEGIN {

nproducts = nvendors = 0
# Process the command line
for (i = 1; i < ARGC; i++) {
	arg = ARGV[i];
	if (arg !~ /^-[dDhH]+$/ && arg !~ /devs$/)
		usage();
	if (arg ~ /^-.*d/)
		dfile="usbdevs_data.h"
	if (arg ~ /^-.*D/)
		dfile="/dev/stdout"
	if (arg ~ /^-.*h/)
		hfile="usbdevs.h"
	if (arg ~ /^-.*H/)
		hfile="/dev/stdout"
	if (arg ~ /devs$/)
		srcfile = arg;
}
ARGC = 1;
line=0;

while ((getline < srcfile) > 0) {
	line++;
	if (line == 1) {
		VERSION = $0
		gsub("\\$", "", VERSION)
		if (dfile)
			header(dfile)
		if (hfile) {
			header(hfile)
			header_guard(hfile)
		}
		continue;
	}
	if ($1 == "vendor") {
		vendor(hfile)
		continue
	}
	if ($1 == "product") {
		product(hfile)
		continue
	}
	if ($0 == "")
		blanklines++
	if (hfile)
		print $0 > hfile
	if (blanklines < 2 && dfile)
	    print $0 > dfile
}

# close off the header guard

if (hfile)
	header_trailer(hfile)

# print out the match tables

if (dfile)
	dump_dfile(dfile)
}
