/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright (c) 2002-2003, Network Appliance, Inc. All rights reserved.
 */

/*
 * Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

/*
 *
 * MODULE: dapl_evd_qp_async_error_callback.c
 *
 * PURPOSE: implements QP callbacks from verbs
 *
 * $Id: dapl_evd_qp_async_error_callb.c,v 1.17 2003/07/31 13:55:18 hobie16 Exp $
 */

#include "dapl.h"
#include "dapl_evd_util.h"
#include "dapl_adapter_util.h"

/*
 * dapl_evd_qp_async_error_callback
 *
 * The callback function registered with verbs for qp async erros
 *
 * Input:
 * 	ib_cm_handle,
 * 	ib_cm_event
 * 	cause_ptr
 * 	context (evd)
 *
 * Output:
 * 	None
 *
 */

void
dapl_evd_qp_async_error_callback(
	IN	ib_hca_handle_t 	ib_hca_handle,
	IN	ib_qp_handle_t		ib_qp_handle,
	IN	ib_error_record_t	*cause_ptr,
	IN	void			*context) /* ARGSUSED */

{
	/*
	 * This is an affiliated error and hence should be able to
	 * supply us with exact information on the error type and QP.
	 *
	 * However the Mellanox and IBM APIs for registering this callback
	 * are different.
	 *
	 * The IBM API allows consumers to register the callback with
	 *
	 * ib_int32_t
	 * ib_set_qp_async_error_eh_us (
	 * 		ib_hca_handle_t 	hca_handle,
	 * 		ib_qp_async_handler_t 	handler )
	 *
	 * Notice that this function does not take a context. The context is
	 * specified per QP in the call to ib_qp_create_us().
	 *
	 * In contrast the Mellanox API requires that the context be specified
	 * when the funciton is registered:
	 *
	 * VAPI_ret_t
	 * VAPI_set_async_event_handler (
	 * 		IN VAPI_hca_hndl_t 		hca_hndl,
	 * 		IN VAPI_async_event_handler_t 	handler,
	 * 		IN void* 			private_data )
	 *
	 * Therefore we always specify the context as the asyncronous EVD
	 * to be compatible with both APIs.
	 */

	DAPL_IA		*ia_ptr;
	DAPL_EP		*ep_ptr;
	DAPL_EVD		*async_evd;
	DAT_EVENT_NUMBER	async_event;
	DAT_RETURN		dat_status;

	ep_ptr    = (DAPL_EP *) context;
	ia_ptr    = ep_ptr->header.owner_ia;
	async_evd = (DAPL_EVD *) ia_ptr->async_error_evd;

	dapl_dbg_log(DAPL_DBG_TYPE_CALLBACK | DAPL_DBG_TYPE_EXCEPTION,
	    "--> dapl_evd_qp_async_error_callback: ep %p qp %p (%x) state %d\n",
	    ep_ptr,
	    ep_ptr->qp_handle,
	    ep_ptr->qpn,
	    ep_ptr->param.ep_state);

	/*
	 * Transition to ERROR if we are connected; other states need to
	 * complete first (e.g. pending states)
	 */
	if (ep_ptr->param.ep_state == DAT_EP_STATE_CONNECTED) {
		ep_ptr->param.ep_state = DAT_EP_STATE_ERROR;
	}

	dapl_os_assert(async_evd != NULL);

	dat_status = dapls_ib_get_async_event(cause_ptr, &async_event);
	if (dat_status == DAT_SUCCESS) {
		/*
		 * If dapls_ib_get_async_event is not successful,
		 * an event has been generated by the provide that
		 * we are not interested in.
		 */
		(void) dapls_evd_post_async_error_event(async_evd,
		    async_event,
		    async_evd->header.owner_ia);
	}
	dapl_dbg_log(DAPL_DBG_TYPE_CALLBACK | DAPL_DBG_TYPE_EXCEPTION,
	    "dapl_evd_qp_async_error_callback () returns\n");
}

/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 8
 * End:
 */
