/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */
/*
 * Copyright 2011 Nexenta Systems, Inc.  All rights reserved.
 */
/*
 * Copyright 2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

/*
 * __vlog: double precision vector log
 *
 * Algorithm:
 *
 * Write x = 2^n z where 1 - 2^-10 <= z < 2 - 2^-9.  Let m = z
 * rounded to nine significant bits, so m = 1 + 2^-8 k, where
 * 0 <= k <= 255.  Let d = z - m.  Then
 *
 * log(x) = n log(2) + log(m) + log(1+(d/m))
 *
 * Let ln2hi = log(2) rounded to a multiple of 2^-42 and ln2lo
 * ~ log(2) - ln2hi.  From a table, obtain mh and ml such that
 * mh = log(m) rounded to a multiple of 2^-42 and ml ~ log(m) -
 * mh.  From the same table, obtain rh and rl such that rh = 1/m
 * rounded to a multiple of 2^-10 and rl ~ 1/m - rh.  For |y| <=
 * 2^-9, approximate log(1+y) by a polynomial y+p(y) where p(y)
 * := y*y*(-1/2+y*(P3+y*(P4+y*(P5+y*P6)))).  Now letting s =
 * d*rh + d*rl in double precision, we can compute the sum above
 * accurately as
 *
 * (n*ln2hi + mh) + (d*rh + (d*rl + (n*ln2lo + ml) + p(s)))
 *
 * When x is subnormal, we first scale it to the normal range,
 * adjusting n accordingly.
 *
 * Accuracy:
 *
 * The largest error observed is less than 0.8 ulps.
 */

#include <sys/isa_defs.h>

#ifdef _LITTLE_ENDIAN
#define	HI(x)	*(1+(int *)&x)
#define	LO(x)	*(unsigned *)&x
#define	HIWORD	1
#define	LOWORD	0
#else
#define	HI(x)	*(int *)&x
#define	LO(x)	*(1+(unsigned *)&x)
#define	HIWORD	0
#define	LOWORD	1
#endif

#ifdef __RESTRICT
#define	restrict _Restrict
#else
#define	restrict
#endif

static const double TBL[] = {
	1.00000000000000000000e+00,  0.00000000000000000000e+00,
	0.00000000000000000000e+00,  0.00000000000000000000e+00,
	9.96093750000000000000e-01,  1.51994163424124515728e-05,
	3.89864041562759666704e-03,  2.97263469009289512726e-14,
	9.92187500000000000000e-01,  6.05620155038759681518e-05,
	7.78214044203195953742e-03,  2.29894100462035112076e-14,
	9.88281250000000000000e-01,  1.35738416988416988208e-04,
	1.16506172200843138853e-02, -1.09039749717359319029e-13,
	9.84375000000000000000e-01,  2.40384615384615397959e-04,
	1.55041865359635266941e-02,  1.72745674997061065553e-15,
	9.80468750000000000000e-01,  3.74161877394636028203e-04,
	1.93429628432113531744e-02, -8.04185385052258635682e-14,
	9.77539062500000000000e-01, -4.39825858778625927714e-04,
	2.31670592816044518258e-02, -7.00735970431003565857e-14,
	9.73632812500000000000e-01, -2.48782081749049442231e-04,
	2.69765876983001362532e-02, -9.80605051684317662887e-14,
	9.69726562500000000000e-01, -2.95928030303030311244e-05,
	3.07716586667083902285e-02,  4.52981425779092882775e-14,
	9.65820312500000000000e-01,  2.17423349056603779517e-04,
	3.45523815067281248048e-02, -6.83913974232877736961e-14,
	9.62890625000000000000e-01, -4.84609962406015010693e-04,
	3.83188643020275776507e-02,  1.09021543022033016421e-13,
	9.58984375000000000000e-01, -1.82876872659176042957e-04,
	4.20712139207353175152e-02, -4.82631400055112824008e-14,
	9.55078125000000000000e-01,  1.45755597014925360189e-04,
	4.58095360313564015087e-02, -6.21983419947579227529e-14,
	9.52148437500000000000e-01, -4.75575046468401500289e-04,
	4.95339351223265111912e-02, -4.98803091079814255646e-14,
	9.48242187500000000000e-01, -9.40393518518518520526e-05,
	5.32445145188376045553e-02, -2.53216894311744497863e-14,
	9.44335937500000000000e-01,  3.13508994464944631443e-04,
	5.69413764001183153596e-02,  2.01093994355649575698e-14,
	9.41406250000000000000e-01, -2.29779411764705879164e-04,
	6.06246218164869787870e-02, -5.21362063913650408235e-14,
	9.37500000000000000000e-01,  2.28937728937728937530e-04,
	6.42943507054951624013e-02, -9.79051851199021608925e-14,
	9.34570312500000000000e-01, -2.63743156934306572509e-04,
	6.79506619085259444546e-02, -1.81950600301688149235e-14,
	9.30664062500000000000e-01,  2.45028409090909096626e-04,
	7.15936531869374448434e-02,  7.13730822534317801406e-14,
	9.27734375000000000000e-01, -1.98143115942028998078e-04,
	7.52234212375242350390e-02,  6.32906595872454402199e-14,
	9.23828125000000000000e-01,  3.59600631768953083074e-04,
	7.88400617077513743425e-02,  2.46501890617661192316e-14,
	9.20898437500000000000e-01, -3.51281474820143869292e-05,
	8.24436692109884461388e-02,  8.61451293608781447223e-14,
	9.17968750000000000000e-01, -4.06025985663082419983e-04,
	8.60343373417435941519e-02,  5.95592298762564263463e-14,
	9.14062500000000000000e-01,  2.23214285714285707316e-04,
	8.96121586897606903221e-02, -7.35577021943502867846e-14,
	9.11132812500000000000e-01, -1.00784030249110321056e-04,
	9.31772248541165026836e-02,  6.67870851716289831942e-14,
	9.08203125000000000000e-01, -4.01706560283687926730e-04,
	9.67296264584547316190e-02,  9.63806765855227740728e-14,
	9.04296875000000000000e-01,  2.96764575971731443208e-04,
	1.00269453163718935684e-01, -4.37863761707839790971e-14,
	9.01367187500000000000e-01,  4.12632042253521119125e-05,
	1.03796793681567578460e-01,  7.59863659719414144342e-14,
	8.98437500000000000000e-01, -1.91885964912280701945e-04,
	1.07311735789153317455e-01, -6.52667880273107116669e-14,
	8.95507812500000000000e-01, -4.02917395104895122333e-04,
	1.10814366340264314204e-01,  2.57999912830699022513e-14,
	8.91601562500000000000e-01,  3.84500217770034828473e-04,
	1.14304771280103523168e-01, -4.48895335223869926230e-14,
	8.88671875000000000000e-01,  2.17013888888888876842e-04,
	1.17783035656430001836e-01, -4.65472974759844472568e-14,
	8.85742187500000000000e-01,  7.09612889273356431397e-05,
	1.21249243632973957574e-01, -1.04272412782730081647e-13,
	8.82812500000000000000e-01, -5.38793103448275854592e-05,
	1.24703478501032805070e-01, -7.55692068745133691756e-14,
	8.79882812500000000000e-01, -1.57726589347079046649e-04,
	1.28145822691976718488e-01, -4.66803140394579609437e-14,
	8.76953125000000000000e-01, -2.40796232876712315400e-04,
	1.31576357788617315236e-01,  1.01957352237084734958e-13,
	8.74023437500000000000e-01, -3.03300981228668954746e-04,
	1.34995164537485834444e-01,  1.89961580415787680134e-14,
	8.71093750000000000000e-01, -3.45450680272108847594e-04,
	1.38402322859064952354e-01,  5.41833313790089940464e-14,
	8.68164062500000000000e-01, -3.67452330508474583805e-04,
	1.41797911860294334474e-01, -3.69845950669709681858e-14,
	8.65234375000000000000e-01, -3.69510135135135155647e-04,
	1.45182009844575077295e-01, -7.71800133682809851086e-14,
	8.62304687500000000000e-01, -3.51825547138047162871e-04,
	1.48554694323138392065e-01, -1.24915489807515996540e-15,
	8.59375000000000000000e-01, -3.14597315436241590364e-04,
	1.51916042025732167531e-01,  1.09807540998552379211e-13,
	8.56445312500000000000e-01, -2.58021530100334438914e-04,
	1.55266128911080159014e-01,  4.37925082924060541938e-14,
	8.53515625000000000000e-01, -1.82291666666666674979e-04,
	1.58605030176659056451e-01, -2.04723578004619553937e-14,
	8.50585937500000000000e-01, -8.75986295681063168849e-05,
	1.61932820269385047141e-01, -7.17939001929567730476e-14,
	8.47656250000000000000e-01,  2.58692052980132450107e-05,
	1.65249572895390883787e-01, -8.37209109923591205585e-14,
	8.44726562500000000000e-01,  1.57925948844884475120e-04,
	1.68555361029802952544e-01,  3.71439775417047191367e-15,
	8.41796875000000000000e-01,  3.08388157894736824986e-04,
	1.71850256926745714736e-01, -8.64923960721207091374e-14,
	8.38867187500000000000e-01,  4.77074795081967189831e-04,
	1.75134332127754532848e-01,  9.46151658066508147714e-14,
	8.36914062500000000000e-01, -3.12755310457516312941e-04,
	1.78407657472916980623e-01, -9.86835038673494943912e-14,
	8.33984375000000000000e-01, -1.08153501628664488934e-04,
	1.81670303107694053324e-01, -5.93750633338470149673e-14,
	8.31054687500000000000e-01,  1.14143668831168828529e-04,
	1.84922338494061477832e-01, -4.94851676612509959777e-14,
	8.28125000000000000000e-01,  3.53964401294498405386e-04,
	1.88163832418240417610e-01, -5.74307839320075599347e-14,
	8.26171875000000000000e-01, -3.65423387096774205090e-04,
	1.91394852999565046048e-01,  6.44085615069689207389e-14,
	8.23242187500000000000e-01, -9.10620980707395479654e-05,
	1.94615467699577493477e-01,  9.41653814571825038763e-14,
	8.20312500000000000000e-01,  2.00320512820512813563e-04,
	1.97825743329985925811e-01, -6.60454487708238395939e-14,
	8.18359375000000000000e-01, -4.68001198083067100272e-04,
	2.01025746060622623190e-01, -3.18818493754377370219e-14,
	8.15429687500000000000e-01, -1.43063296178343944383e-04,
	2.04215541428766300669e-01, -7.54091651195618882501e-14,
	8.12500000000000000000e-01,  1.98412698412698412526e-04,
	2.07395194345963318483e-01,  1.07268675772897325437e-13,
	8.10546875000000000000e-01, -4.20292721518987358927e-04,
	2.10564769107350002741e-01, -3.65071888317905767114e-16,
	8.07617187500000000000e-01, -4.62095820189274421015e-05,
	2.13724329397791734664e-01, -7.35958018644051430164e-14,
	8.04687500000000000000e-01,  3.43946540880503122493e-04,
	2.16873938300523150247e-01,  9.12093724991498410553e-14,
	8.02734375000000000000e-01, -2.26538009404388704197e-04,
	2.20013658305333592580e-01, -5.14966723414140783686e-14,
	7.99804687500000000000e-01,  1.95312500000000010842e-04,
	2.23143551314251453732e-01, -4.16979658452719528642e-14,
	7.97851562500000000000e-01, -3.43774338006230513552e-04,
	2.26263678650411748094e-01,  4.16412673028722634501e-14,
	7.94921875000000000000e-01,  1.09180900621118015200e-04,
	2.29374101064877322642e-01, -3.14926506519148377243e-14,
	7.92968750000000000000e-01, -3.99090557275541795833e-04,
	2.32474878743005319848e-01,  8.87450729797463158287e-14,
	7.90039062500000000000e-01,  8.43942901234567854386e-05,
	2.35566071312860003673e-01, -9.30945949519688945136e-14,
	7.88085937500000000000e-01, -3.93629807692307670790e-04,
	2.38647737850214980426e-01, -3.99705090953013414198e-14,
	7.85156250000000000000e-01,  1.19823619631901839909e-04,
	2.41719936887193398434e-01, -4.82302894299408858477e-14,
	7.83203125000000000000e-01, -3.28507262996941896190e-04,
	2.44782726417724916246e-01, -3.39998110836183310018e-14,
	7.80273437500000000000e-01,  2.14367378048780488466e-04,
	2.47836163904594286578e-01, -1.30297971733086634357e-14,
	7.78320312500000000000e-01, -2.04810980243161095543e-04,
	2.50880306285807819222e-01,  1.59736634636249040926e-15,
	7.75390625000000000000e-01,  3.66950757575757553416e-04,
	2.53915209980959843961e-01,  3.60017673263733462441e-15,
	7.73437500000000000000e-01, -2.36027190332326283783e-05,
	2.56940930897599173477e-01, -9.87480301596639169955e-14,
	7.71484375000000000000e-01, -4.00037650602409625492e-04,
	2.59957524436913445243e-01,  1.26217293988853160748e-14,
	7.68554687500000000000e-01,  2.14081268768768768606e-04,
	2.62965045500777705456e-01,  1.03646364598966627113e-13,
	7.66601562500000000000e-01, -1.34496631736526949192e-04,
	2.65963548497211377253e-01, -7.34359136986779711761e-14,
	7.64648437500000000000e-01, -4.69333022388059722691e-04,
	2.68953087345607855241e-01, -1.03896307840029875617e-13,
	7.61718750000000000000e-01,  1.86011904761904751579e-04,
	2.71933715483555715764e-01,  8.60430677280873279668e-14,
	7.59765625000000000000e-01, -1.21708086053412463954e-04,
	2.74905485872750432463e-01,  4.88167036467699861016e-14,
	7.57812500000000000000e-01, -4.16050295857988176266e-04,
	2.77868451003541849786e-01, -8.55436000656632193091e-14,
	7.54882812500000000000e-01,  2.79429387905604702334e-04,
	2.80822662900845898548e-01,  4.18860913786370112029e-14,
	7.52929687500000000000e-01,  1.14889705882352939582e-05,
	2.83768173130738432519e-01, -9.38341722366369999987e-14,
	7.50976562500000000000e-01, -2.43424670087976540225e-04,
	2.86705032803865833557e-01,  8.84810960400682115458e-14,
	7.49023437500000000000e-01, -4.85425804093567224515e-04,
	2.89633292582948342897e-01,  9.43339818951269030846e-14,
	7.46093750000000000000e-01,  2.61935131195335281235e-04,
	2.92553002686418039957e-01, -4.05999788601512838979e-14,
	7.44140625000000000000e-01,  4.54215116279069761138e-05,
	2.95464212893875810551e-01, -3.99341638438784391272e-14,
	7.42187500000000000000e-01, -1.58514492753623176778e-04,
	2.98366972551775688771e-01,  2.15926937419734905112e-14,
	7.40234375000000000000e-01, -3.49981936416184958877e-04,
	3.01261330578199704178e-01, -3.79231648020931467980e-14,
	7.37304687500000000000e-01,  4.47473883285302582568e-04,
	3.04147335467405355303e-01, -1.08638286797079129552e-13,
	7.35351562500000000000e-01,  2.80621408045976994047e-04,
	3.07025035294827830512e-01,  8.40315630479242455758e-14,
	7.33398437500000000000e-01,  1.25917800859598846179e-04,
	3.09894477722764349892e-01,  1.00337969820392140548e-13,
	7.31445312500000000000e-01, -1.67410714285714294039e-05,
	3.12755710003784770379e-01,  1.12118007403609819830e-13,
	7.29492187500000000000e-01, -1.47458155270655270810e-04,
	3.15608778986415927648e-01, -1.12592746246808286851e-13,
	7.27539062500000000000e-01, -2.66335227272727253015e-04,
	3.18453731118552241242e-01, -1.76254313121726620573e-14,
	7.25585937500000000000e-01, -3.73472910764872500361e-04,
	3.21290612453822177486e-01, -8.78854276997154463823e-14,
	7.23632812500000000000e-01, -4.68970692090395495540e-04,
	3.24119468654316733591e-01, -1.04757500587765412913e-13,
	7.20703125000000000000e-01,  4.23635563380281667846e-04,
	3.26940344995819032192e-01,  3.42884001266694615699e-14,
	7.18750000000000000000e-01,  3.51123595505617967782e-04,
	3.29753286372579168528e-01, -1.11186713895593226425e-13,
	7.16796875000000000000e-01,  2.89959733893557422817e-04,
	3.32558337300042694551e-01,  3.39068613367222871432e-14,
	7.14843750000000000000e-01,  2.40048882681564236573e-04,
	3.35355541921217081835e-01, -7.92515783138655870267e-14,
	7.12890625000000000000e-01,  2.01297005571030637044e-04,
	3.38144944008718084660e-01, -1.68695012281303904492e-15,
	7.10937500000000000000e-01,  1.73611111111111117737e-04,
	3.40926586970681455568e-01, -8.82452633212564001210e-14,
	7.08984375000000000000e-01,  1.56899238227146807121e-04,
	3.43700513853264055797e-01,  5.43888832989906475149e-14,
	7.07031250000000000000e-01,  1.51070441988950269954e-04,
	3.46466767346100823488e-01,  1.07757430375726404546e-13,
	7.05078125000000000000e-01,  1.56034779614325073201e-04,
	3.49225389785260631470e-01,  2.76727112657366262202e-14,
	7.03125000000000000000e-01,  1.71703296703296716700e-04,
	3.51976423157111639739e-01,  6.65449164332479482515e-14,
	7.01171875000000000000e-01,  1.97988013698630136838e-04,
	3.54719909102868768969e-01,  6.02593863918127820941e-14,
	6.99218750000000000000e-01,  2.34801912568306000561e-04,
	3.57455888921776931966e-01,  2.68422602858563731995e-14,
	6.97265625000000000000e-01,  2.82058923705722061539e-04,
	3.60184403574976386153e-01,  3.14101284357935074430e-14,
	6.95312500000000000000e-01,  3.39673913043478251442e-04,
	3.62905493689368086052e-01,  3.67085697163493829481e-16,
	6.93359375000000000000e-01,  4.07562669376693761502e-04,
	3.65619199561024288414e-01, -5.95770946492931122703e-14,
	6.91406250000000000000e-01,  4.85641891891891918850e-04,
	3.68325561158599157352e-01,  1.08495696229679121506e-13,
	6.90429687500000000000e-01, -4.02733322102425902751e-04,
	3.71024618127876237850e-01, -3.57393774001043846673e-15,
	6.88476562500000000000e-01, -3.04519489247311828540e-04,
	3.73716409793587445165e-01, -3.36434401382552911606e-15,
	6.86523437500000000000e-01, -1.96359752010723855866e-04,
	3.76400975164187912014e-01,  6.51539835645912724894e-14,
	6.84570312500000000000e-01, -7.83338903743315521791e-05,
	3.79078352935039220029e-01, -6.97616377035377091917e-14,
	6.82617187500000000000e-01,  4.94791666666666654379e-05,
	3.81748581490910510183e-01, -6.21703236457339082579e-14,
	6.80664062500000000000e-01,  1.87001329787234041400e-04,
	3.84411698910298582632e-01,  3.34571026954408237380e-14,
	6.78710937500000000000e-01,  3.34155338196286447704e-04,
	3.87067742968383754487e-01,  6.45334117530848658606e-14,
	6.77734375000000000000e-01, -4.85697751322751295790e-04,
	3.89716751139985717600e-01,  3.94957702521028807100e-14,
	6.75781250000000000000e-01, -3.19508575197889187636e-04,
	3.92358760602974143694e-01, -1.10271214775306207128e-13,
	6.73828125000000000000e-01, -1.43914473684210512906e-04,
	3.94993808240769794793e-01,  9.91833135258393974771e-14,
	6.71875000000000000000e-01,  4.10104986876640414256e-05,
	3.97621930647119370406e-01,  1.91186992668509687992e-14,
	6.69921875000000000000e-01,  2.35193062827225135005e-04,
	4.00243164127005002229e-01,  7.70470078193964863175e-15,
	6.67968750000000000000e-01,  4.38560704960835531785e-04,
	4.02857544701191727654e-01, -1.08212998879547184399e-13,
	6.66992187500000000000e-01, -3.25520833333333315263e-04,
	4.05465108108273852849e-01, -1.09470871366066397592e-13,
	6.65039062500000000000e-01, -1.03997564935064929046e-04,
	4.08065889808312931564e-01, -9.11831335065229488419e-14,
	6.63085937500000000000e-01,  1.26497733160621750282e-04,
	4.10659924985338875558e-01, -7.04896239210974659112e-14,
	6.61132812500000000000e-01,  3.65895510335917330171e-04,
	4.13247248550305812387e-01, -8.64814613198628863840e-14,
	6.60156250000000000000e-01, -3.62435567010309291763e-04,
	4.15827895143820569501e-01, -1.09603887929539904968e-13,
	6.58203125000000000000e-01, -1.05438624678663237367e-04,
	4.18401899138871158357e-01,  1.26591539849383157019e-14,
	6.56250000000000000000e-01,  1.60256410256410256271e-04,
	4.20969294644237379543e-01, -1.07743414616095792458e-13,
	6.54296875000000000000e-01,  4.34582800511508948911e-04,
	4.23530115505855064839e-01, -5.17691206942015446275e-14,
	6.53320312500000000000e-01, -2.59088010204081649248e-04,
	4.26084395310908803367e-01, -8.74024251107295313295e-15,
	6.51367187500000000000e-01,  3.23035941475826945284e-05,
	4.28632167389650931000e-01,  4.78292070340653116123e-14,
	6.49414062500000000000e-01,  3.32130393401015248239e-04,
	4.31173464818357388140e-01,  1.39527194700992522593e-14,
	6.48437500000000000000e-01, -3.36234177215189876300e-04,
	4.33708320421601456474e-01, -4.20630377335898599132e-14,
	6.46484375000000000000e-01, -1.97285353535353552123e-05,
	4.36236766774982243078e-01, -6.41727287881571093141e-14,
	6.44531250000000000000e-01,  3.05022040302267011258e-04,
	4.38758836207625790848e-01,  2.14689717834000941735e-15,
	6.43554687500000000000e-01, -3.38607097989949751195e-04,
	4.41274560804913562606e-01, -3.83331165923754571982e-14,
	6.41601562500000000000e-01,  2.44752506265664146815e-06,
	4.43783972410301430500e-01, -4.49328344033376536063e-16,
	6.39648437500000000000e-01,  3.51562499999999986990e-04,
	4.46287102628502907464e-01, -8.33959316905439057284e-14,
	6.38671875000000000000e-01, -2.67884975062344151547e-04,
	4.48783982827080762945e-01, -7.40524322934505657145e-14,
	6.36718750000000000000e-01,  9.71703980099502536783e-05,
	4.51274644139402880683e-01,  5.57044620824077391343e-14,
	6.34765625000000000000e-01,  4.70107009925558303777e-04,
	4.53759117467143369140e-01, -2.28624953086649163255e-14,
	6.33789062500000000000e-01, -1.25696163366336636884e-04,
	4.56237433481646803557e-01, -5.92091761359114736879e-14,
	6.31835937500000000000e-01,  2.62827932098765450035e-04,
	4.58709622626884083729e-01,  9.25811146459912121009e-14,
	6.30859375000000000000e-01, -3.17503078817733981869e-04,
	4.61175715122180918115e-01, -1.07517471912360339462e-14,
	6.28906250000000000000e-01,  8.63789926289926251633e-05,
	4.63635740963127318537e-01, -9.48054446804536471658e-14,
	6.27929687500000000000e-01, -4.78707107843137234706e-04,
	4.66089729924533457961e-01,  6.57665976858006147528e-14,
	6.25976562500000000000e-01, -5.96920843520782368088e-05,
	4.68537711563158154604e-01,  8.11157716400523519546e-14,
	6.24023437500000000000e-01,  3.66806402439024390773e-04,
	4.70979715218845740310e-01, -5.47277630185806178777e-14,
	6.23046875000000000000e-01, -1.75828771289537715006e-04,
	4.73415770016572423629e-01,  9.97077440469968501191e-14,
	6.21093750000000000000e-01,  2.65473300970873776934e-04,
	4.75845904869856894948e-01,  1.07019317621142549209e-13,
	6.20117187500000000000e-01, -2.62465950363196100312e-04,
	4.78270148481442447519e-01,  2.78328646163063623105e-14,
	6.18164062500000000000e-01,  1.93425422705314001282e-04,
	4.80688529345798087888e-01, -4.61802117788209510607e-14,
	6.17187500000000000000e-01, -3.20030120481927722077e-04,
	4.83101075751164898975e-01, -2.90762364463866399448e-14,
	6.15234375000000000000e-01,  1.50240384615384623725e-04,
	4.85507815781602403149e-01,  9.84046527823262695501e-14,
	6.14257812500000000000e-01, -3.48939598321342924619e-04,
	4.87908777319262298988e-01, -2.33257420051882497138e-14,
	6.12304687500000000000e-01,  1.35503887559808614775e-04,
	4.90303988045297955978e-01, -1.04117827384293371195e-13,
	6.11328125000000000000e-01, -3.49604713603818609800e-04,
	4.92693475442592898617e-01, -1.76429214903040463891e-14,
	6.09375000000000000000e-01,  1.48809523809523822947e-04,
	4.95077266797807169496e-01,  4.43451018828153751026e-14,
	6.08398437500000000000e-01, -3.22427998812351533642e-04,
	4.97455389202741571353e-01,  7.73708980421385689768e-14,
	6.06445312500000000000e-01,  1.89758590047393372637e-04,
	4.99827869556384030147e-01,  6.52996738757825591006e-14,
	6.05468750000000000000e-01, -2.67804373522458635890e-04,
	5.02194734566728584468e-01, -1.30901947805436250965e-14,
	6.03515625000000000000e-01,  2.57959905660377355422e-04,
	5.04556010752367001260e-01,  2.82857986090678938760e-14,
	6.02539062500000000000e-01, -1.86121323529411759412e-04,
	5.06911724444762512576e-01,  9.18415373613231066159e-14,
	6.00585937500000000000e-01,  3.53029636150234741275e-04,
	5.09261901789841431309e-01, -3.34845053941249831574e-14,
	5.99609375000000000000e-01, -7.77590749414519956471e-05,
	5.11606568749130019569e-01, -6.79410499533039142111e-14,
	5.97656250000000000000e-01,  4.74591121495327101284e-04,
	5.13945751102255599108e-01, -2.12823065872096837292e-14,
	5.96679687500000000000e-01,  5.69092365967365941461e-05,
	5.16279474448538167053e-01, -8.36708800829965016511e-14,
	5.95703125000000000000e-01, -3.54287790697674440793e-04,
	5.18607764208127264283e-01, -8.16321296891503919914e-14,
	5.93750000000000000000e-01,  2.17517401392111359854e-04,
	5.20930645624275712180e-01, -9.03997701415351032573e-14,
	5.92773437500000000000e-01, -1.80844907407407397368e-04,
	5.23248143764476481010e-01,  7.13555066011812146304e-14,
	5.90820312500000000000e-01,  4.03705975750577367080e-04,
	5.25560283522963800351e-01, -3.64289687078304118459e-14,
	5.89843750000000000000e-01,  1.80011520737327188784e-05,
	5.27867089620940532768e-01, -9.81476542529858082436e-14,
	5.88867187500000000000e-01, -3.61440373563218372236e-04,
	5.30168586609079284244e-01,  4.23335972026522927116e-14,
	5.86914062500000000000e-01,  2.41900802752293591410e-04,
	5.32464798869568767259e-01, -9.69233849737002813365e-14,
	5.85937500000000000000e-01, -1.25143020594965678717e-04,
	5.34755750616113800788e-01, -8.61253103749572066304e-14,
	5.84960937500000000000e-01, -4.86051655251141525530e-04,
	5.37041465896891168086e-01, -7.51351912898166894415e-15,
	5.83007812500000000000e-01,  1.35695472665148063720e-04,
	5.39321968595686485060e-01, -7.76104042041871663206e-14,
	5.82031250000000000000e-01, -2.13068181818181807833e-04,
	5.41597282432803694974e-01, -5.93233971574446149215e-14,
	5.80078125000000000000e-01,  4.20741213151927453007e-04,
	5.43867430967338805203e-01, -5.52875399870574035452e-14,
	5.79101562500000000000e-01,  8.39578619909502261217e-05,
	5.46132437598089381936e-01,  4.62684463909612350375e-14,
	5.78125000000000000000e-01, -2.46896162528216717505e-04,
	5.48392325565600913251e-01, -2.77505026685624314655e-14,
	5.76171875000000000000e-01,  4.04701576576576562902e-04,
	5.50647117952621556469e-01,  4.07227907088846767786e-14,
	5.75195312500000000000e-01,  8.55863764044943823575e-05,
	5.52896837686603248585e-01,  7.44889957023668801898e-14,
	5.74218750000000000000e-01, -2.27718609865470858825e-04,
	5.55141507540611200966e-01, -1.09608250460592783688e-13,
	5.72265625000000000000e-01,  4.41310123042505588354e-04,
	5.57381150134006020380e-01,  3.36669632485986549666e-16,
	5.71289062500000000000e-01,  1.39508928571428563684e-04,
	5.59615787935399566777e-01,  2.31194938380053776320e-14,
	5.70312500000000000000e-01, -1.56597995545657025672e-04,
	5.61845443262654953287e-01,  3.68646286817464054051e-14,
	5.69335937500000000000e-01, -4.47048611111111116653e-04,
	5.64070138284705535625e-01,  9.74304462767037064935e-14,
	5.67382812500000000000e-01,  2.44681956762749441229e-04,
	5.66289895023146527819e-01, -3.06552284854813270707e-14,
	5.66406250000000000000e-01, -3.45685840707964596973e-05,
	5.68504735352689749561e-01, -2.10374825114449422873e-14,
	5.65429687500000000000e-01, -3.08274696467991172252e-04,
	5.70714681003437362961e-01,  3.41818930848065350178e-14,
	5.63476562500000000000e-01,  4.00089482378854644894e-04,
	5.72919753561791367247e-01, -5.85815401264202219115e-15,
	5.62500000000000000000e-01,  1.37362637362637362518e-04,
	5.75119974471363093471e-01,  2.48469505879759890764e-14,
	5.61523437500000000000e-01, -1.19928728070175431939e-04,
	5.77315365034792193910e-01,  3.14104080050449590607e-14,
	5.60546875000000000000e-01, -3.71820295404814028101e-04,
	5.79505946414656136767e-01, -1.39129117330010386790e-14,
	5.58593750000000000000e-01,  3.58215065502183428129e-04,
	5.81691739634607074549e-01,  1.54079711890856738893e-14,
	5.57617187500000000000e-01,  1.17017293028322439969e-04,
	5.83872765580963459797e-01,  1.92193002098161738068e-14,
	5.56640625000000000000e-01, -1.18885869565217396136e-04,
	5.86049045003619539784e-01, -4.13308801481084566682e-14,
	5.55664062500000000000e-01, -3.49528877440347096866e-04,
	5.88220598517182224896e-01, -9.61818609368988642797e-14,
	5.53710937500000000000e-01,  4.01616612554112561388e-04,
	5.90387446602107957006e-01,  6.84176364159146659095e-14,
	5.52734375000000000000e-01,  1.81391738660907137675e-04,
	5.92549609606749072555e-01, -7.74738125310530505286e-14,
	5.51757812500000000000e-01, -3.36745689655172409120e-05,
	5.94707107746671681525e-01,  2.11079891578422983965e-14,
	5.50781250000000000000e-01, -2.43615591397849451990e-04,
	5.96859961107838898897e-01, -4.50623098590974831636e-14,
	5.49804687500000000000e-01, -4.48464324034334772557e-04,
	5.99008189646156097297e-01, -7.26979150253512871478e-14,
	5.47851562500000000000e-01,  3.28309020342612404610e-04,
	6.01151813189289896400e-01,  4.49397919602643900279e-14,
	5.46875000000000000000e-01,  1.33547008547008560445e-04,
	6.03290851438032404985e-01,  5.18573553063418286042e-14,
	5.45898437500000000000e-01, -5.62200159914712159731e-05,
	6.05425323966755968286e-01, -3.90788481567525388100e-14,
	5.44921875000000000000e-01, -2.41023936170212761459e-04,
	6.07555250224550036364e-01, -8.24086314983113070392e-15,
	5.43945312500000000000e-01, -4.20896364118895980992e-04,
	6.09680649536812779843e-01,  4.24936389576037736371e-14,
	5.41992187500000000000e-01,  3.80693855932203405450e-04,
	6.11801541105933210929e-01,  5.96926009653846962309e-14,
	5.41015625000000000000e-01,  2.10590644820295982628e-04,
	6.13917944012428051792e-01, -5.75595951560511011845e-14,
	5.40039062500000000000e-01,  4.53256856540084409344e-05,
	6.16029877215623855591e-01, -1.09835943254384298330e-13,
	5.39062500000000000000e-01, -1.15131578947368418456e-04,
	6.18137359555021248525e-01,  5.74853476805674446129e-14,
	5.38085937500000000000e-01, -2.70811449579831946440e-04,
	6.20240409751886545564e-01, -2.90167125533596631915e-14,
	5.37109375000000000000e-01, -4.21743972746331215531e-04,
	6.22339046408797003096e-01, -1.82614988669165533809e-14,
	5.35156250000000000000e-01,  4.08603556485355630390e-04,
	6.24433288011914555682e-01, -2.10546393306435734475e-14,
	5.34179687500000000000e-01,  2.67076591858037557577e-04,
	6.26523152931440563407e-01, -8.78036279744035513715e-14,
	5.33203125000000000000e-01,  1.30208333333333331526e-04,
	6.28608659422297932906e-01,  7.62048382318937090230e-14,
	5.32226562500000000000e-01, -2.03027546777546788817e-06,
	6.30689825626177480444e-01,  2.12246394140452907525e-14,
	5.31250000000000000000e-01, -1.29668049792531120444e-04,
	6.32766669571083184564e-01, -4.53550186996774688761e-14,
	5.30273437500000000000e-01, -2.52733566252587998902e-04,
	6.34839209172923801816e-01,  8.64101534252508178520e-14,
	5.29296875000000000000e-01, -3.71255165289256208652e-04,
	6.36907462237104482483e-01, -3.52508626243453241145e-14,
	5.28320312500000000000e-01, -4.85260953608247433411e-04,
	6.38971446457844649558e-01,  7.60718216684202016469e-14,
	5.26367187500000000000e-01,  3.81783693415637856959e-04,
	6.41031179420906482846e-01,  2.48082091251967673736e-14,
	5.25390625000000000000e-01,  2.76726129363449673514e-04,
	6.43086678603140171617e-01, -1.12856225215656411367e-13,
	5.24414062500000000000e-01,  1.76101434426229513973e-04,
	6.45137961373620782979e-01, -3.60813136042255739798e-14,
	5.23437500000000000000e-01,  7.98824130879345644567e-05,
	6.47185044995239877608e-01,  6.96725146472247760395e-14,
	5.22460937500000000000e-01, -1.19579081632653055071e-05,
	6.49227946625160257099e-01, -5.04382083563449091526e-14,
	5.21484375000000000000e-01, -9.94462830957230209915e-05,
	6.51266683315043337643e-01, -8.52342468131615437746e-14,
	5.20507812500000000000e-01, -1.82609247967479665321e-04,
	6.53301272012640765752e-01,  1.04873006903856996874e-13,
	5.19531250000000000000e-01, -2.61473123732251517670e-04,
	6.55331729563158660312e-01, -3.10282172335227455825e-14,
	5.18554687500000000000e-01, -3.36064018218623454786e-04,
	6.57358072708348117885e-01,  1.19122567102055698791e-14,
	5.17578125000000000000e-01, -4.06407828282828297722e-04,
	6.59380318089233696810e-01, -1.05870694633429062178e-13,
	5.16601562500000000000e-01, -4.72530241935483884957e-04,
	6.61398482245431296178e-01, -6.62879179039074743232e-14,
	5.14648437500000000000e-01,  4.42105759557344087183e-04,
	6.63412581616967145237e-01,  9.91058598099467920662e-14,
	5.13671875000000000000e-01,  3.84349899598393583006e-04,
	6.65422632545187298092e-01, -9.68491419671810783613e-14,
	5.12695312500000000000e-01,  3.30739604208416838882e-04,
	6.67428651271848139004e-01,  1.08050943383646665619e-13,
	5.11718750000000000000e-01,  2.81249999999999978750e-04,
	6.69430653942526987521e-01,  1.02279777907416200886e-13,
	5.10742187500000000000e-01,  2.35856412175648700539e-04,
	6.71428656605257856427e-01,  4.44668903784876907111e-14,
	5.09765625000000000000e-01,  1.94534362549800786662e-04,
	6.73422675212123067467e-01,  4.36528304869414810551e-14,
	5.08789062500000000000e-01,  1.57259567594433401650e-04,
	6.75412725620162746054e-01,  1.39850267837821649808e-14,
	5.07812500000000000000e-01,  1.24007936507936501053e-04,
	6.77398823591829568613e-01, -2.34278036379790696248e-14,
	5.06835937500000000000e-01,  9.47555693069306959140e-05,
	6.79380984795898257289e-01, -1.00907141981183426552e-13,
	5.05859375000000000000e-01,  6.94787549407114679145e-05,
	6.81359224807920327294e-01, -1.72583456150091690167e-14,
	5.04882812500000000000e-01,  4.81539694280078915244e-05,
	6.83333559111588328960e-01,  3.23592040115024425781e-14,
	5.03906250000000000000e-01,  3.07578740157480310692e-05,
	6.85304003098963221419e-01, -4.38048746232309815355e-14,
	5.02929687500000000000e-01,  1.72673133595284864178e-05,
	6.87270572070929119946e-01,  3.11475515031130920163e-14,
	5.01953125000000000000e-01,  7.65931372549019597214e-06,
	6.89233281238784911693e-01,  2.40686318405286681994e-14,
	5.00976562500000000000e-01,  1.91108121330724059841e-06,
	6.91192145724244255689e-01, -1.02296829368141946888e-13,
};

static const double C[] = {
	6.93147180559890330187e-01,
	5.49792301870837115524e-14,
	-0.5,
	3.33333333332438282293284931714682042701467889609e-0001,
	-2.49999999998669026809069285994497705748522309858e-0001,
	2.00000758613044543658508591796951886624273250472e-0001,
	-1.66667492411916229281646821123333564982955309481e-0001,
	4503599627370496.0,
	0.0
};

#define	ln2hi	C[0]
#define	ln2lo	C[1]
#define	mhalf	C[2]
#define	P3	C[3]
#define	P4	C[4]
#define	P5	C[5]
#define	P6	C[6]
#define	two52	C[7]
#define	zero	C[8]

#define	PROCESS(N)							\
	i##N = (i##N + 0x800) & ~0xfff;					\
	e = (i##N & 0x7ff00000) - 0x3ff00000;				\
	z##N.i[HIWORD] -= e;						\
	w##N.i[HIWORD] = i##N - e;					\
	w##N.i[LOWORD] = 0;						\
	n##N += (e >> 20);						\
	i##N = (i##N >> 10) & 0x3fc;					\
	d##N = z##N.d - w##N.d;					\
	h##N = d##N * TBL[i##N];					\
	l##N = d##N * TBL[i##N+1];					\
	s##N = h##N + l##N;						\
	b##N = (s##N * s##N) * (mhalf + s##N * (P3 + s##N * (P4 +	\
	    s##N * (P5 + s##N * P6))));					\
	*y = (n##N * ln2hi + TBL[i##N+2]) + (h##N + (l##N +		\
	    (n##N * ln2lo + TBL[i##N+3]) + b##N));			\
	y += stridey

#define	PREPROCESS(N, index, label)					\
	i##N = HI(*x);							\
	z##N.d = *x;							\
	x += stridex;							\
	n##N = 0;							\
	if ((i##N & 0x7ff00000) == 0x7ff00000) { /* inf or NaN */	\
		y[index] = z##N.d * ((i##N < 0)? zero : z##N.d);	\
		goto label;						\
	} else if (i##N < 0x00100000) { /* subnormal, negative, zero */	\
		if (((i##N << 1) | z##N.i[LOWORD]) == 0) {		\
			y[index] = mhalf / zero;			\
			goto label;					\
		} else if (i##N < 0) {					\
			y[index] = zero / zero;				\
			goto label;					\
		}							\
		z##N.d *= two52;					\
		n##N = -52;						\
		i##N = z##N.i[HIWORD];				\
	}

void
__vlog(int n, double *restrict x, int stridex, double *restrict y,
	int stridey)
{
	union {
		unsigned	i[2];
		double		d;
	} z0, z1, z2, z3, w0, w1, w2, w3;
	double	b0, b1, b2, b3;
	double	d0, d1, d2, d3;
	double	h0, h1, h2, h3;
	double	l0, l1, l2, l3;
	double	s0, s1, s2, s3;
	int	i0, i1, i2, i3, e;
	int	n0, n1, n2, n3;

	w0.i[LOWORD] = 0;
	w1.i[LOWORD] = 0;
	w2.i[LOWORD] = 0;
	w3.i[LOWORD] = 0;

	y -= stridey;

	for (;;) {
begin:
		y += stridey;

		if (--n < 0)
			break;

		PREPROCESS(0, 0, begin);

		if (--n < 0)
			goto process1;

		PREPROCESS(1, stridey, process1);

		if (--n < 0)
			goto process2;

		PREPROCESS(2, (stridey << 1), process2);

		if (--n < 0)
			goto process3;

		PREPROCESS(3, (stridey << 1) + stridey, process3);

		i0 = (i0 + 0x800) & ~0xfff;
		e = (i0 & 0x7ff00000) - 0x3ff00000;
		z0.i[HIWORD] -= e;
		w0.i[HIWORD] = i0 - e;
		n0 += (e >> 20);
		i0 = (i0 >> 10) & 0x3fc;

		i1 = (i1 + 0x800) & ~0xfff;
		e = (i1 & 0x7ff00000) - 0x3ff00000;
		z1.i[HIWORD] -= e;
		w1.i[HIWORD] = i1 - e;
		n1 += (e >> 20);
		i1 = (i1 >> 10) & 0x3fc;

		i2 = (i2 + 0x800) & ~0xfff;
		e = (i2 & 0x7ff00000) - 0x3ff00000;
		z2.i[HIWORD] -= e;
		w2.i[HIWORD] = i2 - e;
		n2 += (e >> 20);
		i2 = (i2 >> 10) & 0x3fc;

		i3 = (i3 + 0x800) & ~0xfff;
		e = (i3 & 0x7ff00000) - 0x3ff00000;
		z3.i[HIWORD] -= e;
		w3.i[HIWORD] = i3 - e;
		n3 += (e >> 20);
		i3 = (i3 >> 10) & 0x3fc;

		d0 = z0.d - w0.d;
		d1 = z1.d - w1.d;
		d2 = z2.d - w2.d;
		d3 = z3.d - w3.d;

		h0 = d0 * TBL[i0];
		h1 = d1 * TBL[i1];
		h2 = d2 * TBL[i2];
		h3 = d3 * TBL[i3];

		l0 = d0 * TBL[i0+1];
		l1 = d1 * TBL[i1+1];
		l2 = d2 * TBL[i2+1];
		l3 = d3 * TBL[i3+1];

		s0 = h0 + l0;
		s1 = h1 + l1;
		s2 = h2 + l2;
		s3 = h3 + l3;

		b0 = (s0 * s0) * (mhalf + s0 * (P3 + s0 * (P4 +
		    s0 * (P5 + s0 * P6))));
		b1 = (s1 * s1) * (mhalf + s1 * (P3 + s1 * (P4 +
		    s1 * (P5 + s1 * P6))));
		b2 = (s2 * s2) * (mhalf + s2 * (P3 + s2 * (P4 +
		    s2 * (P5 + s2 * P6))));
		b3 = (s3 * s3) * (mhalf + s3 * (P3 + s3 * (P4 +
		    s3 * (P5 + s3 * P6))));

		*y = (n0 * ln2hi + TBL[i0+2]) + (h0 + (l0 +
		    (n0 * ln2lo + TBL[i0+3]) + b0));
		y += stridey;
		*y = (n1 * ln2hi + TBL[i1+2]) + (h1 + (l1 +
		    (n1 * ln2lo + TBL[i1+3]) + b1));
		y += stridey;
		*y = (n2 * ln2hi + TBL[i2+2]) + (h2 + (l2 +
		    (n2 * ln2lo + TBL[i2+3]) + b2));
		y += stridey;
		*y = (n3 * ln2hi + TBL[i3+2]) + (h3 + (l3 +
		    (n3 * ln2lo + TBL[i3+3]) + b3));
		continue;

process1:
		PROCESS(0);
		continue;

process2:
		PROCESS(0);
		PROCESS(1);
		continue;

process3:
		PROCESS(0);
		PROCESS(1);
		PROCESS(2);
	}
}
