/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */
/*
 * Copyright 2011 Nexenta Systems, Inc.  All rights reserved.
 */
/*
 * Copyright 2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#include "libm.h"		/* __k_clog_rl */
#include "complex_wrapper.h"
#include "longdouble.h"

/* INDENT OFF */
/*
 * long double __k_clog_rl(long double x, long double y, long double *e);
 *
 * Compute real part of complex natural logarithm of x+iy in extra precision
 *
 * __k_clog_rl returns log(hypot(x, y)) with a correction term e.
 *
 * Accuracy: quad 140 bits, intel extended 91 bits.
 *
 * Method.
 * Assume X > Y >= 0 . Let X = 2**nx * x, Y = 2**nx * y, where 1 <= x < 2.
 * Let Z = X*X + Y*Y.  Then Z = 2**(nx+nx) * z, where z = x*x + y*y.
 * Note that z < 8.
 * Let Z = x*x + y*y.  Z can be normalized as Z = 2**N * z,  1 <= z < 2.
 * We further break down z into 1 + zk + zh + zt, where
 *	zk = K*(2**-7) matches z to 7.5 significant bits, 0 <= K <= 2**(-7)-1
 *	zh = (z-zk) rounded to half of the current significant bits
 *	zt = (z-zk-zh) rounded.
 *
 *	   z - (1+zk)		(zh+zt)
 * Let s = ------------ = ---------------, then
 *	   z + (1+zk)	  2(1+zk)+zh+zt
 *							z
 * log(Z) = N*log2 + log(z) = N*log2 + log(1+zk) + log(------)
 *							1+zk
 *					1+s
 *	  = N * log2 + log(1 +zk) + log(---)
 *					1-s
 *
 *						3		5
 *	  = N*log2 + log(1+zk) +  2s +  1/12(2s)    +   1/80(2s)  + ...
 *
 *
 * Note 1. For IEEE double precision,  a fifteen degree odd polynomial
 *		2s + P1*(2s)^3 + P2*(2s)^5 + P3*(2s)^7 + ... + P7*(2s)^15
 *	  is generated by a special remez algorithm to
 *	  approx log((1+s)/(1-s)) accurte to 145 bits.
 * Note 2. 2s can be computed accurately as s2h+s2t by
 *	  r = 2/((zh+zt)+2(1+zk))
 *	  s2 = r*(zh+zt)
 *	  s2h = s2 rounded to double;  v = 0.5*s2h;
 *	  s2t = r*((((zh-s2h*(1+zk))-v*zh)+zt)-v*zt)
 */
/* INDENT ON */

static const long double
zero  = 0.0L,
half  = 0.5L,
two   = 2.0L,
two240 = 1.7668470647783843295832975007429185158274839e+72L, /* 2^240 */

/* first 48 bits of ln2 */
ln2_h  = 0.693147180559943620892227045260369777679443359375L,
ln2_t  = 1.68852500507619780679039605677498525525412068e-15L,
P1 = .083333333333333333333333333333333333341023785768375L,
P2 = .01249999999999999999999999999999679085402075766159375L,
P3 = .002232142857142857142857143310092047621284490564671875L,
P4 = .00043402777777777777774746781319264872413156956512109375L,
P5 = .0000887784090909101756336594019277185263940665468935546875L,
P6 = .000018780048055589639895360927834628371268354778446533203125L,
P7 = .000004069227854328982921366736003458838031087153635406494140625L;

/*
 * T[2k, 2k+1] = log(1+k*2**-7) for k = 0, ..., 2**7 - 1,
 * with T[2k] * 2^48 is an int
 */

static const long double TBL_log1k[] = {
0.0000000000000000000000000000000000000000e+00L,
0.0000000000000000000000000000000000000000e+00L,
7.7821404420532758194894995540380477905273e-03L,
1.6731279734005070987158875984584325351222e-15L,
1.5504186535963526694104075431823730468750e-02L,
1.7274567499706106231054091184928671990316e-15L,
2.3167059281533397552266251295804977416992e-02L,
9.8067653290966648493916241687661877474892e-16L,
3.0771658666751022792595904320478439331055e-02L,
2.6655784323032762937247606420524589813624e-15L,
3.8318864302134159061097307130694389343262e-02L,
2.4401326580179931029010027013316092332340e-15L,
4.5809536031292452662455616518855094909668e-02L,
1.7505042236510958082472042641283104263139e-15L,
5.3244514518809182845870964229106903076172e-02L,
3.1000199992295574218738634002122149891138e-15L,
6.0624621816433688081815489567816257476807e-02L,
1.1544987906424726040058093958345197512800e-15L,
6.7950661908504628172522643581032752990723e-02L,
3.1212220426341915966610439115772728417386e-15L,
7.5223421237584631171557703055441379547119e-02L,
2.8945270476369282210350897509258766743153e-15L,
8.2443669211073711267090402543544769287109e-02L,
8.8000106966612476303662698634483335676886e-16L,
8.9612158689686083334891009144484996795654e-02L,
1.0492850604602339995319895311151740799226e-15L,
9.6729626458550654888313147239387035369873e-02L,
4.5740725790924807640164516707244620870662e-16L,
1.0379679368164218544734467286616563796997e-01L,
1.3793787171308978090503366050174239822054e-15L,
1.1081436634028918319927470292896032333374e-01L,
9.3099553146639425160476473362380086036919e-16L,
1.1778303565638026384476688690483570098877e-01L,
3.1906940272225656860040797111813146690890e-15L,
1.2470347850095464536934741772711277008057e-01L,
2.5904940590976537504984110469214193890052e-15L,
1.3157635778871679121948545798659324645996e-01L,
2.4813692306707028899159917911012100567219e-15L,
1.3840232285911824305912887211889028549194e-01L,
8.9262619700148275890190121571708972000380e-16L,
1.4518200984449691759436973370611667633057e-01L,
9.7968756533003444764719201050911636480025e-16L,
1.5191604202583874894116888754069805145264e-01L,
3.2261306345373561864598749471119213018106e-15L,
1.5860503017663774016909883357584476470947e-01L,
8.4392427234104999681053621980394827998735e-16L,
1.6524957289530561865831259638071060180664e-01L,
1.5442172988528965297119225948270579746101e-15L,
1.7185025692665689689420105423778295516968e-01L,
2.3254458978918173643097657009894831132739e-15L,
1.7840765747281750464026117697358131408691e-01L,
7.9247913906453736065426776912520942036896e-16L,
1.8492233849401173984006163664162158966064e-01L,
2.5282384195601762803134514624610774126020e-16L,
1.9139485299962899489401024766266345977783e-01L,
4.5971528855989864541366920731297729269228e-16L,
1.9782574332991842425144568551331758499146e-01L,
1.4561111263856836438840838027526567191527e-15L,
2.0421554142868814096800633706152439117432e-01L,
2.7505358140491347148810394262840919337709e-15L,
2.1056476910734645002776233013719320297241e-01L,
3.1876417904825951583107481283088861928977e-15L,
2.1687393830061196808856038842350244522095e-01L,
2.3915305291373208450532580201045871599499e-15L,
2.2314355131420882116799475625157356262207e-01L,
9.3459830033405826094075253077304795996257e-16L,
2.2937410106484534821902343537658452987671e-01L,
4.8177245728966955534167425511952551974164e-16L,
2.3556607131276408040321257431060075759888e-01L,
2.8286743756446304426525380844720043381780e-15L,
2.4171993688714366044223424978554248809814e-01L,
1.5077020732661279714120052415509585052975e-15L,
2.4783616390458007572306087240576744079590e-01L,
1.1810575418933407573072030113600980623171e-15L,
2.5391520998096339667426946107298135757446e-01L,
4.7463053836833625309891834934881898560705e-17L,
2.5995752443692410338371701072901487350464e-01L,
1.9635883624838132961710716735786266795913e-15L,
2.6596354849713677026556979399174451828003e-01L,
1.1710735561325457988709887923652142233351e-15L,
2.7193371548364098089223261922597885131836e-01L,
7.7793943687530702031066421537496360004376e-16L,
2.7786845100345303194444568362087011337280e-01L,
3.2742419043493025311197092322146237692165e-15L,
2.8376817313064250924981024581938982009888e-01L,
2.0890970909765308649465619266075677112425e-15L,
2.8963329258304071345264674164354801177979e-01L,
1.9634262463138821209582240742801727823629e-15L,
2.9546421289383317798638017848134040832520e-01L,
2.6984003017275736237868564402005801750600e-15L,
3.0126133057816062432721082586795091629028e-01L,
1.1566856647123658045763670687640673680383e-15L,
3.0702503529490954292668902780860662460327e-01L,
2.3191484355127267712770857311812090801833e-15L,
3.1275571000389490450288576539605855941772e-01L,
1.9838833607942922604727420618882220398852e-15L,
3.1845373111853447767316538374871015548706e-01L,
1.3813708182984188944010814590398164268227e-16L,
3.2411946865421015218089451082050800323486e-01L,
1.8239097762496144793489474731253815376404e-15L,
3.2975328637246548169059678912162780761719e-01L,
2.5001238260227991620033344720809714552230e-15L,
3.3535554192113536942088103387504816055298e-01L,
2.4608362985459391180385214539620341910962e-15L,
3.4092658697059263772644044365733861923218e-01L,
5.7257864875612301758921090406373771458003e-16L,
3.4646676734620740489845047704875469207764e-01L,
1.1760200117113770182586341947822306069951e-15L,
3.5197642315717558858523261733353137969971e-01L,
2.5960702148389259075462896448369304790506e-15L,
3.5745588892180180096147523727267980575562e-01L,
1.9732645342528682246686790561260072184839e-15L,
3.6290549368936808605212718248367309570312e-01L,
3.6708569716349381675043725477739939978160e-16L,
3.6832556115870573876236448995769023895264e-01L,
1.9142858656640927085879445412821643247628e-15L,
3.7371640979358389245135185774415731430054e-01L,
1.8836966497497166619234389157276681281343e-16L,
3.7907835293496816575498087331652641296387e-01L,
1.2926358724723144934459175417385013725801e-15L,
3.8441169891033055705520382616668939590454e-01L,
1.4826795862363146014726140088145939341729e-15L,
3.8971675114002479745067830663174390792847e-01L,
4.1591978529737177695912258866565331189698e-16L,
3.9499380824086571806219581048935651779175e-01L,
3.2600441982258756252505182317625310732365e-15L,
4.0024316412701210765590076334774494171143e-01L,
5.9927342433864738622836851475469574662703e-16L,
4.0546510810816371872533636633306741714478e-01L,
6.6325267674913128171942721503283748008372e-16L,
4.1065992498526782128465129062533378601074e-01L,
5.6464965491255048900165082436455718077885e-16L,
4.1582789514371043537721561733633279800415e-01L,
5.3023611327561856950735176370587227509442e-16L,
4.2096929464412724541944044176489114761353e-01L,
2.3907094267197419048248363335257046791153e-15L,
4.2608439531089814522601955104619264602661e-01L,
1.9178985253285492839728700574592375309985e-15L,
4.3117346481836804628073878120630979537964e-01L,
3.2945784336977492852031005044499611665595e-15L,
4.3623676677491474151793227065354585647583e-01L,
3.3288311090524075754441878570852962903891e-15L,
4.4127456080487448275562201160937547683716e-01L,
7.4673387443005192574852544613692268411229e-16L,
4.4628710262841764233598951250314712524414e-01L,
1.8691966006681165218815050615460959199251e-15L,
4.5127464413945617138779198285192251205444e-01L,
2.4137569004002270899666314791611479063976e-15L,
4.5623743348158640742440184112638235092163e-01L,
1.1869564036970375473975162509216610120281e-15L,
4.6117571512216670726047595962882041931152e-01L,
3.4591075239659690349392915732654828400811e-15L,
4.6608972992459740680715185590088367462158e-01L,
1.8177514673916038857252366108673570603067e-15L,
4.7097971521878889689105562865734100341797e-01L,
2.1156558422273990182479555421331461933366e-15L,
4.7584590486996347635795245878398418426514e-01L,
4.3790725712752039722791012358345927696967e-16L,
4.8068852934575190261057286988943815231323e-01L,
5.0660455855585733988956280680891477171499e-18L,
4.8550781578169832641833636444061994552612e-01L,
2.4813834547127501689550526444948043590905e-15L,
4.9030398804519137456736643798649311065674e-01L,
2.4635829797216592537498738468934647345741e-15L,
4.9507726679784980206022737547755241394043e-01L,
1.7125377372093652812514167461480115600063e-15L,
4.9982786955644797899367404170334339141846e-01L,
1.3508276573735437007500942002018098437396e-15L,
5.0455601075239187025545106735080480575562e-01L,
3.4168028574643873701242268618467347998876e-15L,
5.0926190178980590417268103919923305511475e-01L,
2.0426313938800290907697638200502614622891e-15L,
5.1394575110223428282552049495279788970947e-01L,
3.3975485593321419703400672813719873194659e-17L,
5.1860776420804555186805373523384332656860e-01L,
8.0284923261130955371987633083003284697416e-17L,
5.2324814376454753528378205373883247375488e-01L,
3.0123302517119603836788558832352723470118e-16L,
5.2786708962084105678513878956437110900879e-01L,
1.3283287534282139298545497336570406582397e-15L,
5.3246479886946929127589100971817970275879e-01L,
2.5525980327137419625398485590148417041921e-15L,
5.3704146589688050994482182431966066360474e-01L,
3.1446219074198341716354190061340477078626e-15L,
5.4159728243274329884116014000028371810913e-01L,
1.0727353821639001503808606766770295812627e-15L,
5.4613243759813556721383065450936555862427e-01L,
8.3168566554721843605240702438699163825794e-17L,
5.5064711795266063631970610003918409347534e-01L,
1.6429402420791657293666192255419538448840e-15L,
5.5514150754050106684189813677221536636353e-01L,
5.2587358222274368868380660194332415847228e-16L,
5.5961578793542088305912329815328121185303e-01L,
1.8032117652023735453816330571171114110385e-15L,
5.6407013828480145889443519990891218185425e-01L,
1.5071769490901812785299634348367857600711e-15L,
5.6850473535266843327917740680277347564697e-01L,
2.7879956135806418878792935692629147550413e-16L,
5.7291975356178426181941176764667034149170e-01L,
1.2472733449589795907271346997596471822345e-15L,
5.7731536503482061561953742057085037231445e-01L,
2.9886985746409486460291929160223207644146e-15L,
5.8169173963462128540413687005639076232910e-01L,
1.1971164738836689815783808674399742176950e-15L,
5.8604904500357690722012193873524665832520e-01L,
1.3016839974975520776911897855504474452726e-15L,
5.9038744660217545856539800297468900680542e-01L,
9.1607651870514890975077236127894522134392e-16L,
5.9470710774668944509357970673590898513794e-01L,
3.3444207638397932963480545729233567201211e-15L,
5.9900818964608149030937056522816419601440e-01L,
1.9090722294592334873060460706130642200729e-15L,
6.0329085143808214297678205184638500213623e-01L,
2.1193638031348149256035110177854940281795e-15L,
6.0755525022453937822319858241826295852661e-01L,
2.4172778865703728624133665395876418941354e-15L,
6.1180154110599005434778518974781036376953e-01L,
2.8491821045766810044199163148675291775782e-15L,
6.1602987721551372146677749697118997573853e-01L,
2.9818078843122551067455400545109858745295e-16L,
6.2024040975185457114093878772109746932983e-01L,
2.9577105558448461493874424529516311623184e-15L,
6.2443328801189323939979658462107181549072e-01L,
2.6164274215943360130441858075903119505815e-16L,
6.2860865942237253989333112258464097976685e-01L,
1.5978509770831895426601797458058854400463e-15L,
6.3276666957103699928666173946112394332886e-01L,
8.3025912472904245581515990140161946934461e-16L,
6.3690746223706895534633076749742031097412e-01L,
2.7627416365968377888021629180796328536455e-16L,
6.4103117942092779912854894064366817474365e-01L,
3.4919270523937617243719652995048419893186e-15L,
6.4513796137358170312836591619998216629028e-01L,
2.9985368625799347497396478978681548584217e-15L,
6.4922794662510696639401430729776620864868e-01L,
2.8524968256626075449136225882322854909611e-15L,
6.5330127201274379444839723873883485794067e-01L,
1.8443102186424720390266302263929355424008e-15L,
6.5735807270835877602621621917933225631714e-01L,
1.2541156738040666039091970075936624723645e-15L,
6.6139848224536379461824253667145967483521e-01L,
1.2136419933020381912633127333149145382797e-15L,
6.6542263254508782210905337706208229064941e-01L,
2.6268410392329445778904988886114643307320e-15L,
6.6943065394262646350398426875472068786621e-01L,
2.8037949010021747828222575923191438798877e-15L,
6.7342267521216570003161905333399772644043e-01L,
1.0202663413354670195383104149875619397268e-15L,
6.7739882359180469961756898555904626846313e-01L,
1.4411921136244383020300914304078010801275e-15L,
6.8135922480790256372529256623238325119019e-01L,
5.0522277899333570619054540068138110661023e-16L,
6.8530400309891703614084690343588590621948e-01L,
2.3804032011755313470802014258958896193599e-15L,
6.8923328123880622797514661215245723724365e-01L,
2.7523497677256621466659891416404053623832e-15L,
};

/*
 * Compute N*log2 + log(1+zk+zh+zt) in extra precision
 */
static long double k_log_NKzl(int N, int K, long double zh, long double *zt)
{
	long double y, r, w, s2, s2h, s2t, t, zk, v, P;
	double dzk;

#if !defined(__x86)
	unsigned lx, ly;
	int j;
#endif

	((int *)&dzk)[HIWORD] = 0x3ff00000 + (K << 13);
	((int *)&dzk)[LOWORD] = 0;
	t  = zh + (*zt);
	zk = (long double) dzk;
	r = two / (t + two * zk);
	s2h = s2 = r * t;
/* split s2 into correctly rounded half */

#if defined(__x86)
		((unsigned *)&s2h)[0] = 0;	/* 32 bits chopped */
#else

		lx = ((unsigned *)&s2h)[2];	/* 56 bits rounded */
		j  = ((lx >> 24) + 1) >> 1;
		((unsigned *)&s2h)[2] = (j << 25);
		lx = ((unsigned *)&s2h)[1];
		ly = lx + (j >> 7);
		((unsigned *)&s2h)[1] = ly;
		((unsigned *)&s2h)[0] += (ly == 0 && lx != 0);
		((unsigned *)&s2h)[3] = 0;
#endif

	v = half * s2h;
	w = s2 * s2;
	s2t = r * ((((zh - s2h * zk) - v * zh) + (*zt)) - v * (*zt));
	P = s2t + (w * s2) * ((P1 + w * P2) + (w * w) * ((P3 + w * P4)
		+ (w * w) * (P5 + w * P6 + (w * w) * P7)));
	P += N * ln2_t + TBL_log1k[K + K + 1];
	t = N*ln2_h + TBL_log1k[K+K];
	y = t + (P + s2h);
	P -= ((y - t) - s2h);
	*zt = P;
	return (y);
}

long double
__k_clog_rl(long double x, long double y, long double *er)
{
	long double t1, t2, t3, t4, tk, z, wh, w, zh, zk;
	int n, k, ix, iy, iz, nx, ny, nz, i;
	double dk;

#if !defined(__x86)
	int j;
	unsigned lx, ly;
#endif

	ix = HI_XWORD(x) & ~0x80000000;
	iy = HI_XWORD(y) & ~0x80000000;
	y = fabsl(y); x = fabsl(x);
	if (ix < iy || (ix < 0x7fff0000 && ix == iy && x < y)) {
		/* force x >= y */
		tk = x;  x = y;   y = tk;
		n = ix, ix = iy; iy = n;
	}
	*er = zero;
	nx = ix >> 16; ny = iy >> 16;
	if (nx >= 0x7fff) {   	/* x or y is Inf or NaN */
		if (isinfl(x))
			return (x);
		else if (isinfl(y))
			return (y);
		else
			return (x+y);
	}
/*
 * for tiny y:(double y < 2^-35, extended y < 2^-46, quad y < 2^-70)
 *
 *	log(sqrt(1 + y**2)) =  y**2 / 2 - y**4 / 8 + ...  =  y**2 / 2
 */
#if defined(__x86)
	if (x == 1.0L && ny < (0x3fff - 46)) {
#else
	if (x == 1.0L && ny < (0x3fff - 70)) {
#endif

		t2 = y * y;
		if (ny >= 8305) {	/* compute er = tail of t2 */
			dk = (double) y;

#if defined(__x86)
			((unsigned *)&dk)[LOWORD] &= 0xfffe0000;
#endif

			wh = (long double) dk;
			*er = half * ((y - wh) * (y + wh) - (t2 - wh * wh));
		}
		return (half * t2);
	}
/*
 * x or y is subnormal or zero
 */
	if (nx == 0) {
		if (x == 0.0L)
			return (-1.0L / x);
		else {
			x *= two240;
			y *= two240;
			ix = HI_XWORD(x);
			iy = HI_XWORD(y);
			nx = (ix >> 16) - 240;
			ny = (iy >> 16) - 240;
			/* guard subnormal flush to 0 */
			if (x == 0.0L)
				return (-1.0L / x);
		}
	} else if (ny == 0) {	/* y subnormal, scale it */
		y *= two240;
		iy = HI_XWORD(y);
		ny = (iy >> 16) - 240;
	}
	n = nx - ny;
/*
 * When y is zero or when x >> y, i.e.,  n > 62, 78, 122 for DBLE,
 * EXTENDED, QUAD respectively,
 * log(x) = log(sqrt(x * x + y * y)) to 27 extra bits.
 */

#if defined(__x86)
	if (n > 78 || y == 0.0L) {
#else
	if (n > 122 || y == 0.0L) {
#endif

		XFSCALE(x, (0x3fff - (ix >> 16)));
		i = ((ix & 0xffff) + 0x100) >> 9;  /* 7.5 bits of x */
		zk = 1.0L + ((long double) i) * 0.0078125L;
		z = x - zk;
		dk = (double)z;

#if defined(__x86)
		((unsigned *)&dk)[LOWORD] &= 0xfffe0000;
#endif

		zh = (long double)dk;
		k = i & 0x7f;	/* index of zk */
		n = nx - 0x3fff;
		*er = z - zh;
		if (i == 0x80) {	/* if zk = 2.0, adjust scaling */
			n += 1;
			zh *= 0.5L;  *er *= 0.5L;
		}
		w = k_log_NKzl(n, k, zh, er);
	} else {
/*
 * compute z = x*x + y*y
 */
		XFSCALE(x, (0x3fff - (ix >> 16)));
		XFSCALE(y, (0x3fff - n - (iy >> 16)));
		ix = (ix & 0xffff) | 0x3fff0000;
		iy = (iy & 0xffff) | (0x3fff0000 - (n << 16));
		nx -= 0x3fff;
		t1 = x * x; t2 = y * y;
		wh = x;

/* split x into correctly rounded half */
#if defined(__x86)
		((unsigned *)&wh)[0] = 0;	/* 32 bits chopped */
#else
		lx = ((unsigned *)&wh)[2];	/* 56 rounded */
		j  = ((lx >> 24) + 1) >> 1;
		((unsigned *)&wh)[2] = (j << 25);
		lx = ((unsigned *)&wh)[1];
		ly = lx + (j >> 7);
		((unsigned *)&wh)[1] = ly;
		((unsigned *)&wh)[0] += (ly == 0 && lx != 0);
		((unsigned *)&wh)[3] = 0;
#endif

		z = t1+t2;
/*
 * higher precision simulation x*x = t1 + t3, y*y = t2 + t4
 */
		tk = wh - x;
		t3 = tk * tk - (two * wh * tk - (wh * wh - t1));
		wh = y;

/* split y into correctly rounded half */
#if defined(__x86)
		((unsigned *)&wh)[0] = 0;	/* 32 bits chopped */
#else
		ly = ((unsigned *)&wh)[2];	/* 56 bits rounded */
		j  = ((ly >> 24) + 1) >> 1;
		((unsigned *)&wh)[2] = (j << 25);
		lx = ((unsigned *)&wh)[1];
		ly = lx + (j >> 7);
		((unsigned *)&wh)[1] = ly;
		((unsigned *)&wh)[0] += (ly == 0 && lx != 0);
		((unsigned *)&wh)[3] = 0;
#endif

		tk = wh - y;
		t4 = tk * tk - (two * wh * tk - (wh * wh - t2));
/*
 * find zk matches z to 7.5 bits
 */
		iz = HI_XWORD(z);
		k = ((iz & 0xffff) + 0x100) >> 9;  /* 7.5 bits of x */
		nz = (iz >> 16) - 0x3fff + (k >> 7);
		k &= 0x7f;
		zk = 1.0L + ((long double) k) * 0.0078125L;
		if (nz == 1) zk += zk;
		else if (nz == 2) zk *= 4.0L;
		else if (nz == 3) zk *= 8.0L;
/*
 * order t1, t2, t3, t4 according to their size
 */
		if (t2 >= fabsl(t3)) {
			if (fabsl(t3) < fabsl(t4)) {
				wh = t3;  t3 = t4; t4 = wh;
			}
		} else {
			wh = t2; t2 = t3; t3 = wh;
		}
/*
 * higher precision simulation: x * x + y * y = t1 + t2 + t3 + t4
 * = zk(7 bits) + zh(24 bits) + *er(tail) and call k_log_NKz
 */
		tk = t1 - zk;
		zh = ((tk + t2) + t3) + t4;

/* split zh into correctly rounded half */
#if defined(__x86)
		((unsigned *)&zh)[0] = 0;
#else
		ly = ((unsigned *)&zh)[2];
		j  = ((ly >> 24) + 1) >> 1;
		((unsigned *)&zh)[2] = (j << 25);
		lx = ((unsigned *)&zh)[1];
		ly = lx + (j >> 7);
		((unsigned *)&zh)[1] = ly;
		((unsigned *)&zh)[0] += (ly == 0 && lx != 0);
		((unsigned *)&zh)[3] = 0;
#endif

		w = fabsl(zh);
		if (w >= fabsl(t2))
{
			*er = (((tk - zh) + t2) + t3) + t4;
}

		else {

			if (n == 0) {
				wh = half * zk;
				wh = (t1 - wh) - (wh - t2);
			} else
				wh = tk + t2;
			if (w >= fabsl(t3))
				*er = ((wh - zh) + t3) + t4;
			else {
				z = t3;
				t3 += t4;
				t4 -= t3 - z;
				if (w >= fabsl(t3))
					*er = ((wh - zh) + t3) + t4;
				else
					*er = ((wh + t3) - zh) + t4;
			}
		}
		if (nz == 3) {
			zh *= 0.125L; *er *= 0.125L;
		} else if (nz == 2) {
			zh *= 0.25L; *er *= 0.25L;
		} else if (nz == 1) {
			zh *= half; *er *= half;
		}
		nz += nx + nx;
		w = half * k_log_NKzl(nz, k, zh, er);
		*er *= half;
	}
	return (w);
}
