/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2007 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 *
 * ident	"%Z%%M%	%I%	%E% SMI"
 */
package org.opensolaris.os.dtrace;

import java.io.*;
import java.util.EventObject;

/**
 * An event used to pass probe data generated by a DTrace {@link
 * Consumer} to interested listeners.
 *
 * @see ConsumerListener#dataReceived(DataEvent e)
 *
 * @author Tom Erickson
 */
public class DataEvent extends EventObject {
    static final long serialVersionUID = 3068774547474769821L;

    /** @serial */
    private ProbeData probeData;

    /**
     * Creates a {@link ConsumerListener#dataReceived(DataEvent e)
     * dataReceived()} event that conveys data generated by DTrace from
     * a single probe firing.
     *
     * @throws NullPointerException if the given probe data is {@code
     * null}
     */
    public
    DataEvent(Object source, ProbeData generatedData)
    {
	super(source);
	probeData = generatedData;
	validate();
    }

    private final void
    validate()
    {
	if (probeData == null) {
	    throw new NullPointerException("probe data is null");
	}
    }

    /**
     * Gets the data generated by DTrace from a single probe firing.
     *
     * @return non-null probe data generated by DTrace from a single
     * probe firing
     */
    public ProbeData
    getProbeData()
    {
	return probeData;
    }

    private void
    readObject(ObjectInputStream s)
            throws IOException, ClassNotFoundException
    {
	s.defaultReadObject();
	// check invariants
	try {
	    validate();
	} catch (Exception e) {
	    InvalidObjectException x = new InvalidObjectException(
		    e.getMessage());
	    x.initCause(e);
	    throw x;
	}
    }

    /**
     * Gets a string representation of this event useful for logging and
     * not intended for display.  The exact details of the
     * representation are unspecified and subject to change, but the
     * following format may be regarded as typical:
     * <pre><code>
     * class-name[property1 = value1, property2 = value2]
     * </code></pre>
     */
    public String
    toString()
    {
	StringBuilder buf = new StringBuilder();
	buf.append(DataEvent.class.getName());
	buf.append("[source = ");
	buf.append(getSource());
	buf.append(", probeData = ");
	buf.append(probeData);
	buf.append(']');
	return buf.toString();
    }
}
