/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.GatheringByteChannel;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.network.TransportLayer;
import org.apache.kafka.common.record.AbstractRecords;
import org.apache.kafka.common.record.FileLogInputStream;
import org.apache.kafka.common.record.LogEntry;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordsIterator;
import org.apache.kafka.common.utils.Utils;

public class FileRecords
extends AbstractRecords
implements Closeable {
    private final boolean isSlice;
    private final int start;
    private final int end;
    private final Iterable<FileLogInputStream.FileChannelLogEntry> shallowEntries;
    private final Iterable<LogEntry> deepEntries = new Iterable<LogEntry>(){

        @Override
        public Iterator<LogEntry> iterator() {
            return FileRecords.this.deepIterator();
        }
    };
    private final AtomicInteger size;
    private final FileChannel channel;
    private volatile File file;

    public FileRecords(File file, FileChannel channel, int start, int end, boolean isSlice) throws IOException {
        this.file = file;
        this.channel = channel;
        this.start = start;
        this.end = end;
        this.isSlice = isSlice;
        this.size = new AtomicInteger();
        if (isSlice) {
            this.size.set(end - start);
        } else {
            int limit = Math.min((int)channel.size(), end);
            this.size.set(limit - start);
            channel.position(limit);
        }
        this.shallowEntries = this.shallowEntriesFrom(start);
    }

    @Override
    public int sizeInBytes() {
        return this.size.get();
    }

    public File file() {
        return this.file;
    }

    public FileChannel channel() {
        return this.channel;
    }

    public ByteBuffer readInto(ByteBuffer buffer, int position) throws IOException {
        Utils.readFully(this.channel, buffer, position + this.start);
        buffer.flip();
        return buffer;
    }

    public FileRecords read(int position, int size) throws IOException {
        if (position < 0) {
            throw new IllegalArgumentException("Invalid position: " + position);
        }
        if (size < 0) {
            throw new IllegalArgumentException("Invalid size: " + size);
        }
        int end = this.start + position + size < 0 ? this.sizeInBytes() : Math.min(this.start + position + size, this.sizeInBytes());
        return new FileRecords(this.file, this.channel, this.start + position, end, true);
    }

    public int append(MemoryRecords records) throws IOException {
        int written = records.writeFullyTo(this.channel);
        this.size.getAndAdd(written);
        return written;
    }

    public void flush() throws IOException {
        this.channel.force(true);
    }

    @Override
    public void close() throws IOException {
        this.flush();
        this.trim();
        this.channel.close();
    }

    public boolean delete() {
        Utils.closeQuietly(this.channel, "FileChannel");
        return this.file.delete();
    }

    public void trim() throws IOException {
        this.truncateTo(this.sizeInBytes());
    }

    public void setFile(File file) {
        this.file = file;
    }

    public void renameTo(File f) throws IOException {
        try {
            Utils.atomicMoveWithFallback(this.file.toPath(), f.toPath());
        }
        finally {
            this.file = f;
        }
    }

    public int truncateTo(int targetSize) throws IOException {
        int originalSize = this.sizeInBytes();
        if (targetSize > originalSize || targetSize < 0) {
            throw new KafkaException("Attempt to truncate log segment to " + targetSize + " bytes failed, " + " size of this log segment is " + originalSize + " bytes.");
        }
        if (targetSize < (int)this.channel.size()) {
            this.channel.truncate(targetSize);
            this.channel.position(targetSize);
            this.size.set(targetSize);
        }
        return originalSize - targetSize;
    }

    @Override
    public long writeTo(GatheringByteChannel destChannel, long offset, int length) throws IOException {
        long bytesTransferred;
        int oldSize;
        long newSize = Math.min(this.channel.size(), (long)this.end) - (long)this.start;
        if (newSize < (long)(oldSize = this.sizeInBytes())) {
            throw new KafkaException(String.format("Size of FileRecords %s has been truncated during write: old size %d, new size %d", this.file.getAbsolutePath(), oldSize, newSize));
        }
        long position = (long)this.start + offset;
        int count = Math.min(length, oldSize);
        if (destChannel instanceof TransportLayer) {
            TransportLayer tl = (TransportLayer)destChannel;
            bytesTransferred = tl.transferFrom(this.channel, position, count);
        } else {
            bytesTransferred = this.channel.transferTo(position, count, destChannel);
        }
        return bytesTransferred;
    }

    public LogEntryPosition searchForOffsetWithSize(long targetOffset, int startingPosition) {
        for (FileLogInputStream.FileChannelLogEntry entry : this.shallowEntriesFrom(startingPosition)) {
            long offset = entry.offset();
            if (offset < targetOffset) continue;
            return new LogEntryPosition(offset, entry.position(), entry.sizeInBytes());
        }
        return null;
    }

    public TimestampAndOffset searchForTimestamp(long targetTimestamp, int startingPosition) {
        for (LogEntry logEntry : this.shallowEntriesFrom(startingPosition)) {
            Record shallowRecord = logEntry.record();
            if (shallowRecord.timestamp() < targetTimestamp) continue;
            for (LogEntry deepLogEntry : logEntry) {
                long timestamp = deepLogEntry.record().timestamp();
                if (timestamp < targetTimestamp) continue;
                return new TimestampAndOffset(timestamp, deepLogEntry.offset());
            }
            throw new IllegalStateException(String.format("The message set (max timestamp = %s, max offset = %s should contain target timestamp %s, but does not.", shallowRecord.timestamp(), logEntry.offset(), targetTimestamp));
        }
        return null;
    }

    public TimestampAndOffset largestTimestampAfter(int startingPosition) {
        long maxTimestamp = -1L;
        long offsetOfMaxTimestamp = -1L;
        for (LogEntry logEntry : this.shallowEntriesFrom(startingPosition)) {
            long timestamp = logEntry.record().timestamp();
            if (timestamp <= maxTimestamp) continue;
            maxTimestamp = timestamp;
            offsetOfMaxTimestamp = logEntry.offset();
        }
        return new TimestampAndOffset(maxTimestamp, offsetOfMaxTimestamp);
    }

    public Iterable<FileLogInputStream.FileChannelLogEntry> shallowEntries() {
        return this.shallowEntries;
    }

    public Iterable<FileLogInputStream.FileChannelLogEntry> shallowEntries(int maxRecordSize) {
        return this.shallowEntries(maxRecordSize, this.start);
    }

    private Iterable<FileLogInputStream.FileChannelLogEntry> shallowEntriesFrom(int start) {
        return this.shallowEntries(Integer.MAX_VALUE, start);
    }

    private Iterable<FileLogInputStream.FileChannelLogEntry> shallowEntries(final int maxRecordSize, final int start) {
        return new Iterable<FileLogInputStream.FileChannelLogEntry>(){

            @Override
            public Iterator<FileLogInputStream.FileChannelLogEntry> iterator() {
                return FileRecords.this.shallowIterator(maxRecordSize, start);
            }
        };
    }

    private Iterator<FileLogInputStream.FileChannelLogEntry> shallowIterator(int maxRecordSize, int start) {
        int end = this.isSlice ? this.end : this.sizeInBytes();
        FileLogInputStream inputStream = new FileLogInputStream(this.channel, maxRecordSize, start, end);
        return RecordsIterator.shallowIterator(inputStream);
    }

    @Override
    public Iterable<LogEntry> deepEntries() {
        return this.deepEntries;
    }

    private Iterator<LogEntry> deepIterator() {
        int end = this.isSlice ? this.end : this.sizeInBytes();
        FileLogInputStream inputStream = new FileLogInputStream(this.channel, Integer.MAX_VALUE, this.start, end);
        return new RecordsIterator(inputStream, false, false, Integer.MAX_VALUE);
    }

    public static FileRecords open(File file, boolean mutable, boolean fileAlreadyExists, int initFileSize, boolean preallocate) throws IOException {
        FileChannel channel = FileRecords.openChannel(file, mutable, fileAlreadyExists, initFileSize, preallocate);
        int end = !fileAlreadyExists && preallocate ? 0 : Integer.MAX_VALUE;
        return new FileRecords(file, channel, 0, end, false);
    }

    public static FileRecords open(File file, boolean fileAlreadyExists, int initFileSize, boolean preallocate) throws IOException {
        return FileRecords.open(file, true, fileAlreadyExists, initFileSize, preallocate);
    }

    public static FileRecords open(File file, boolean mutable) throws IOException {
        return FileRecords.open(file, mutable, false, 0, false);
    }

    public static FileRecords open(File file) throws IOException {
        return FileRecords.open(file, true);
    }

    private static FileChannel openChannel(File file, boolean mutable, boolean fileAlreadyExists, int initFileSize, boolean preallocate) throws IOException {
        if (mutable) {
            if (fileAlreadyExists) {
                return new RandomAccessFile(file, "rw").getChannel();
            }
            if (preallocate) {
                RandomAccessFile randomAccessFile = new RandomAccessFile(file, "rw");
                randomAccessFile.setLength(initFileSize);
                return randomAccessFile.getChannel();
            }
            return new RandomAccessFile(file, "rw").getChannel();
        }
        return new FileInputStream(file).getChannel();
    }

    public static class TimestampAndOffset {
        public final long timestamp;
        public final long offset;

        public TimestampAndOffset(long timestamp, long offset) {
            this.timestamp = timestamp;
            this.offset = offset;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            TimestampAndOffset that = (TimestampAndOffset)o;
            if (this.timestamp != that.timestamp) {
                return false;
            }
            return this.offset == that.offset;
        }

        public int hashCode() {
            int result = (int)(this.timestamp ^ this.timestamp >>> 32);
            result = 31 * result + (int)(this.offset ^ this.offset >>> 32);
            return result;
        }

        public String toString() {
            return "TimestampAndOffset(timestamp=" + this.timestamp + ", offset=" + this.offset + ')';
        }
    }

    public static class LogEntryPosition {
        public final long offset;
        public final int position;
        public final int size;

        public LogEntryPosition(long offset, int position, int size) {
            this.offset = offset;
            this.position = position;
            this.size = size;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            LogEntryPosition that = (LogEntryPosition)o;
            if (this.offset != that.offset) {
                return false;
            }
            if (this.position != that.position) {
                return false;
            }
            return this.size == that.size;
        }

        public int hashCode() {
            int result = (int)(this.offset ^ this.offset >>> 32);
            result = 31 * result + this.position;
            result = 31 * result + this.size;
            return result;
        }

        public String toString() {
            return "LogEntryPosition(offset=" + this.offset + ", position=" + this.position + ", size=" + this.size + ')';
        }
    }
}

