#ifndef __ROUTING_H__
#define __ROUTING_H__

#include "Arduino.h"
#include <stdint.h>
#include <CircularBuffer.h>
#include "pt/pt.h"

#include "e32_radio.h"
#include "packet.h"

#define MAX_PACKET_HISTORY 5
#define MAX_ROUTES 3
#define ROUTE_UPDATE_FREQ 10000
#define ROUTE_LIFETIME 30000

#define BUF_SIZE 100

#define BROADCAST_ADDR 0xFFFF
#define FLOODING_ADDR 0xFFEE

enum MessageType {
    HELLO,
    RAIN_GUAGE,
    LANDSLIDE
};

typedef E32Radio::Address Address;

typedef struct {
  Address src;
  uint8_t seq_no;
} PktRecord;

typedef struct {
  Address id;
  Address next_hop;
  uint8_t hop_count;
  uint32_t _ts;
  bool valid;
} RouteRecord;

class Routing
{
  public:    

    Routing(E32Radio& radio);
    
    bool available() const { return this->_isPacketAvailable; }
    void send(Address dst, Packet* pkt);
    void forward(Address dst, Address src, uint8_t hop, uint8_t seq_no, Packet* pkt);
    void recv(char* buf, uint8_t* len);
    void loop();
  
    Address getSrc() const { return this->_pktSrc; }
    
    void showPacketHistory();
    void showRtTable();
    uint8_t getRouteCount() const { return this->_rtCount; }

       
  private:  
    E32Radio& _radio;
    bool _isPacketAvailable;
    
    Address _pktSrc;
    uint8_t _pktLen;
    char* _buf;
    
    uint8_t _seqNo; 
    uint8_t _rtCount;
    uint8_t _rtIndex;  
    RouteRecord _rtTable[MAX_ROUTES];
    CircularBuffer<PktRecord, MAX_PACKET_HISTORY> _pktHistory;

    struct pt _ptRxThread;
    struct pt _ptRoutingThread;

    bool checkDuplicatedPacket(Address src, uint8_t seq);  
    void populateRtTable(Address src, E32Radio::Address prev, uint8_t hc);
    void validateRtTable();
    E32Radio::Address getNextHop(Address dst);

    PT_THREAD(_rxThread(struct pt* pt));
    PT_THREAD(_routingThread(struct pt* pt));
};

#endif
