#include "Arduino.h"
#include <stdint.h>
#include <CircularBuffer.h>
#include "pt/pt.h"

#include "e32_radio.h"
#include "routing.h"
#include "packet.h"

#define DEBUG 1

typedef struct {
  Address dst;
  Address src;
  uint8_t hop_count;
  uint8_t seq_no;
  Packet pkt;
} Frame;

Routing::Routing(E32Radio& radio) : _radio(radio)
{
  _rtCount = 0;
  _rtIndex = 0;
  _seqNo = 0;

  PT_INIT(&_ptRxThread);
  PT_INIT(&_ptRoutingThread);
}

/***********************************************
 * 
 */
void Routing::showPacketHistory()
{
  Serial.println("=======================");
  Serial.println("    Packet History");
  Serial.println("=======================");
  for (int i = 0; i < _pktHistory.size(); i++) {
    Serial.print("Src: 0x");
    Serial.print(_pktHistory[i].src, HEX);
    Serial.print(" | Seq: ");
    Serial.println(_pktHistory[i].seq_no);
  }
  Serial.println("=======================");
}

/***********************************************
 * 
 */
void Routing::showRtTable()
{
  Serial.println("-------------------------------------------------");
  Serial.println("                Routing Table");
  Serial.println("-------------------------------------------------");
  for(int i=0; i<_rtIndex; i++) {
    Serial.print("Node: 0x");
    Serial.print(_rtTable[i].id, HEX);
    Serial.print(" | Next: 0x");
    Serial.print(_rtTable[i].next_hop, HEX);
    Serial.print(" | Hop: ");
    Serial.print(_rtTable[i].hop_count);
    Serial.print(" | t: ");
    Serial.print(_rtTable[i]._ts);
    Serial.print(" | Valid: ");
    Serial.print(_rtTable[i].valid);
    Serial.println();
  }
  Serial.println("-------------------------------------------------");
}

/***********************************************
 * 
 */
bool Routing::checkDuplicatedPacket(Address src, uint8_t seq)
{
  for(int i=0; i<_pktHistory.size(); i++) {
    if(src == _pktHistory[i].src && seq == _pktHistory[i].seq_no) {
#ifdef DEBUG
      Serial.println("Duplicated Packet");
#endif
      return true;
    }
  }
  // Add new packet to packet history  
  PktRecord p;
  p.src = src;
  p.seq_no = seq;
  _pktHistory.push(p);
  return false;
}

/***********************************************
 * 
 */
void Routing::validateRtTable()
{
//#ifdef DEBUG
//  Serial.print("Validate Routing Table");
//  Serial.print(" (index: ");
//  Serial.print(_rtIndex);
//  Serial.print(" count: ");
//  Serial.print(_rtCount);
//  Serial.println(")");
//#endif
  
  for(int i=0; i < _rtIndex; i++) {
    
    if(!_rtTable[i].valid)
      continue;
     
    if(millis() - _rtTable[i]._ts > ROUTE_LIFETIME) {
      _rtTable[i].valid = false;
      _rtCount -= 1;
#ifdef DEBUG
      Serial.print("Invalidate route to Node ");
      Serial.println(_rtTable[i].id);
      showRtTable();
#endif    
    }
  }
}

/***********************************************
 * 
 */
void Routing::populateRtTable(Address src, E32Radio::Address prev, uint8_t hc)
{
  for(int i=0; i<_rtIndex; i++) {
    if(_rtTable[i].id == src) {
      if(_rtTable[i].valid && _rtTable[i].hop_count < hc)
        return;
      _rtTable[i].next_hop = prev;  
      _rtTable[i].hop_count = hc;
      _rtTable[i]._ts = millis();
      _rtTable[i].valid = true;
      return;
    }
  }
  // Add New Node Information
  RouteRecord r;
  r.id = src;
  r.next_hop = prev;
  r.hop_count = hc;
  r._ts = millis();
  r.valid = true;

  _rtCount += 1;

  if(_rtIndex < MAX_ROUTES) {
    _rtTable[_rtIndex] = r;
    _rtIndex += 1;
    return;
  }

  unsigned long aged_t = _rtTable[0]._ts;
  uint8_t aged_index = 0;
  
  for(int i=0; i<_rtIndex; i++) {
    if(!_rtTable[i].valid) {
      _rtTable[i] = r;
      return;
    }
    if(_rtTable[i]._ts < aged_t) {
      aged_t = _rtTable[i]._ts;
      aged_index = i;
    }
  }
  _rtTable[aged_index] = r;
}

/***********************************************
 * 
 */
E32Radio::Address Routing::getNextHop(Address dst)
{
  for(int i=0; i < _rtIndex; i++) {
      if(!_rtTable[i].valid)
        continue;
      if(dst == _rtTable[i].id) {
        return _rtTable[i].next_hop;
        break;
      }
  }
  return E32Radio::BROADCAST;
}

/***********************************************
 * 
 */
//void Routing::send(Address dst, char* data, uint8_t len)
void Routing::send(Address dst, Packet* pkt)
{
  E32Radio::Address next_hop = E32Radio::BROADCAST;
  Address src = _radio.getNodeAddr();
  _seqNo += 1;

  if(dst != BROADCAST_ADDR) {
   next_hop = getNextHop(dst);
  }

  Frame frame;
  frame.dst = dst;
  frame.src = src;
  frame.hop_count = 0;
  frame.seq_no = _seqNo;
//  frame.pkt = *((Packet*)data);
  frame.pkt = *pkt;
  _radio.send(next_hop, (char*)&frame, sizeof(frame));
  
#ifdef DEBUG
  Serial.print("TX:NextHop: 0x");
  Serial.print(next_hop, HEX);
  Serial.print(" Dst: 0x");
  Serial.print(frame.dst, HEX);
  Serial.print(" Src: 0x");
  Serial.print(frame.src, HEX);
  Serial.print(" Hop: ");
  Serial.print(frame.hop_count);
  Serial.print(" Seq: ");
  Serial.print(frame.seq_no);
  Serial.println();
#endif 
}

/***********************************************
 * 
 */
void Routing::forward(Address dst, Address src, uint8_t hop, uint8_t seq, Packet* pkt)
{
  Address next_hop = E32Radio::BROADCAST;

  if(dst != BROADCAST_ADDR) {
    next_hop = getNextHop(dst);
  }

  Frame frame;
  frame.dst = dst;
  frame.src = src;
  frame.hop_count = hop+1;
  frame.seq_no = seq;
  frame.pkt = *pkt;
  _radio.send(next_hop, (char*)&frame, sizeof(frame));
  
#ifdef DEBUG
  Serial.print("FW:NextHop: 0x");
  Serial.print(next_hop, HEX);
  Serial.print(" Dst: 0x");
  Serial.print(frame.dst, HEX);
  Serial.print(" Src: 0x");
  Serial.print(frame.src, HEX);
  Serial.print(" Hop: ");
  Serial.print(frame.hop_count);
  Serial.print(" Seq: ");
  Serial.print(frame.seq_no);
  Serial.println();
#endif 
}

/***********************************************
 * 
 */
void Routing::recv(char* buf, uint8_t* len)
{
  if(_isPacketAvailable) {
    buf = _buf;
    *len = _pktLen;
    _isPacketAvailable = false;
  }
  else {
    *len = 0;
  }
}

/***********************************************
 * 
 */
void Routing::loop()
{
  _rxThread(&_ptRxThread);
  _routingThread(&_ptRoutingThread);
}

/***********************************************
 * 
 */
PT_THREAD(Routing::_rxThread(struct pt* pt))
{
 
  PT_BEGIN(pt);

  for (;;)
  {
    PT_WAIT_UNTIL(pt,_radio.available());
    
    E32Radio::Address sender;
    char buf[BUF_SIZE];
    uint8_t len;
    _radio.recv(&sender,buf,&len);

    Frame* frame;
    frame = (Frame*)buf;

    if(!checkDuplicatedPacket(frame->src, frame->seq_no))
    {
      populateRtTable(frame->src, sender, frame->hop_count);
      
#ifdef DEBUG
      showPacketHistory();      
      showRtTable();

      Serial.print("RX:Sender: 0x");
      Serial.print(sender, HEX);
      Serial.print(" Dst: 0x");
      Serial.print(frame->dst, HEX);
      Serial.print(" Src: 0x");
      Serial.print(frame->src, HEX);
      Serial.print(" Hop: ");
      Serial.print(frame->hop_count);
      Serial.print(" Seq: ");
      Serial.print(frame->seq_no);
      Serial.println();
#endif

      if((*frame).dst == _radio.getNodeAddr()) {
        _pktSrc = frame->src;
        _pktLen = sizeof(Packet);
        _buf = (char*) &(frame->pkt);
        _isPacketAvailable = true;
#ifdef DEBUG
        Serial.println("Receive Packet specified for this Node");
#endif
      }
      else if((*frame).dst == BROADCAST_ADDR) {
        forward(frame->dst, frame->src, frame->hop_count, frame->seq_no, &(frame->pkt));
        _pktSrc = frame->src;
        _pktLen = sizeof(Packet);
        _buf = (char*) &(frame->pkt);
        _isPacketAvailable = true;
#ifdef DEBUG
        Serial.println("Receive BROADCAST Packet");
#endif
      }
      else {
        forward(frame->dst, frame->src, frame->hop_count, frame->seq_no, &(frame->pkt));
#ifdef DEBUG
        Serial.println("Forward Packet");
#endif
      }
    }
  }

  PT_END(pt);
}

/***********************************************
 * 
 */
PT_THREAD(Routing::_routingThread(struct pt* pt))
{
   static uint32_t ts; // timestamp
 
  PT_BEGIN(pt);

  ts = millis();
    
  for (;;)
  {
     PT_WAIT_UNTIL(pt,millis() - ts >= ROUTE_UPDATE_FREQ);
     validateRtTable();
     ts = millis();
  }
 
  PT_END(pt);
}

