#ifndef __E32_RADIO_H__
#define __E32_RADIO_H__

#include <stdint.h>
#include <Stream.h>

#include "pt/pt.h"
#include "cirbuf.h"

/***********************************************
 * 
 */
enum class E32Mode
{
  NORMAL,
  WAKEUP,
  POWERSAVE,
  SLEEP
};

/***********************************************
 * 
 */
class E32Radio
{
public:
  
  typedef uint16_t Address;
  static const Address BROADCAST = 0xFFFF;

  static const uint8_t DEFAULT_CHANNEL  = 0x17;
  static const uint8_t DATA_RATE_1KBPS  = 0b000;
  static const uint8_t DATA_RATE_2KBPS  = 0b001;
  static const uint8_t DATA_RATE_5KBPS  = 0b010;
  static const uint8_t DATA_RATE_8KBPS  = 0b011;
  static const uint8_t DATA_RATE_10KBPS = 0b100;
  static const uint8_t DATA_RATE_15KBPS = 0b101;
  static const uint8_t DATA_RATE_20KBPS = 0b110;
  static const uint8_t DATA_RATE_25KBPS = 0b111;

  static const uint8_t WAKEUP_TIME_250MS  = 0b000;
  static const uint8_t WAKEUP_TIME_500MS  = 0b001;
  static const uint8_t WAKEUP_TIME_750MS  = 0b010;
  static const uint8_t WAKEUP_TIME_1000MS = 0b011;
  static const uint8_t WAKEUP_TIME_1250MS = 0b100;
  static const uint8_t WAKEUP_TIME_1500MS = 0b101;
  static const uint8_t WAKEUP_TIME_1750MS = 0b110;
  static const uint8_t WAKEUP_TIME_2000MS = 0b111;

  static const uint8_t TX_POWER_30DBM = 0b00;
  static const uint8_t TX_POWER_27DBM = 0b01;
  static const uint8_t TX_POWER_24DBM = 0b10;
  static const uint8_t TX_POWER_21DBM = 0b11;
  
  E32Radio(Stream& serial,
          uint8_t m0, uint8_t m1, uint8_t aux,
          char* txbuf, uint16_t txBufSize,
          char* rxbuf, uint16_t rxBufSize);
  void init(
    Address nodeAddr,
    uint8_t txPower=TX_POWER_30DBM,
    uint8_t dataRate=DATA_RATE_1KBPS,
    uint8_t channel=0x17,
    uint8_t wakeUpTime=WAKEUP_TIME_250MS
  );
  void setChannel(uint8_t channel);
  uint8_t getChannel() const { return this->_channel; }
  Address getNodeAddr() const { return this->_nodeAddr; }
  void send(Address dst, char* data, uint8_t len);
  bool available() const { return !this->_rxBuf.empty(); }
  void recv(Address* src, char* buf, uint8_t* len);
  void loop();

private:
  Address   _nodeAddr;
  uint8_t   _m0Pin;
  uint8_t   _m1Pin;
  uint8_t   _auxPin;
  uint8_t   _channel;
  CirBuf    _txBuf;
  CirBuf    _rxBuf;
  E32Mode   _mode;
  struct pt _ptTxThread;
  struct pt _ptRxThread;
  Stream&   _serial;

  PT_THREAD(_txThread(struct pt* pt));
  PT_THREAD(_rxThread(struct pt* pt));
};

#endif
