#include <Arduino.h>
#include "cirbuf.h"
#include "e32_radio.h"
#include "pt/pt.h"

//#define E32_TRANSPARENT_MODE
#define E32_FIXED_MODE

#define E32_LOW_POWER

#define FRAME_DELIMETER      0x7E
#define SERIAL_TIMEOUT       100  // timeout in ms
#define TIMER_START(ts)      ts = millis();
#define TIMER_EXPIRED(ts,to) (millis()-ts > to)

#define LO_BYTE(word)  (word & 0xFF)
#define HI_BYTE(word)  (((uint16_t)word) >> 8)

//#define DEBUG

#define NORMAL_MODE()  \
  digitalWrite(_m0Pin,0); \
  digitalWrite(_m1Pin,0); \
  _mode = E32Mode::NORMAL;

#define WAKEUP_MODE()  \
  digitalWrite(_m0Pin,1); \
  digitalWrite(_m1Pin,0); \
  _mode = E32Mode::WAKEUP;
  
#define POWERSAVE_MODE()  \
  digitalWrite(_m0Pin,0); \
  digitalWrite(_m1Pin,1); \
  _mode = E32Mode::POWERSAVE;
  
#define SLEEP_MODE()  \
  digitalWrite(_m0Pin,1); \
  digitalWrite(_m1Pin,1); \
  _mode = E32Mode::SLEEP;

#define PT_DELAY(pt,ms,ts) \
  ts = millis(); \
  PT_WAIT_WHILE(pt, millis()-ts < (ms));

char sbuf[100];
  
/***********************************************
 * 
 */
E32Radio::E32Radio(Stream& serial,
        uint8_t m0, uint8_t m1, uint8_t aux,
        char* txBuf, uint16_t txBufSize,
        char* rxBuf, uint16_t rxBufSize):
  _serial(serial),
  _m0Pin(m0),_m1Pin(m1),_auxPin(aux),
  _txBuf(txBuf,txBufSize),_rxBuf(rxBuf,rxBufSize)
{
  _channel = DEFAULT_CHANNEL;
  PT_INIT(&_ptTxThread);
  PT_INIT(&_ptRxThread);
  pinMode(_m0Pin,OUTPUT);
  pinMode(_m1Pin,OUTPUT);
  pinMode(_auxPin,INPUT);

  while (digitalRead(_auxPin) == LOW)
    ; // wait until the module is ready

  SLEEP_MODE();
}

/***********************************************
 * 
 */
void E32Radio::init(
  Address nodeAddr,
  uint8_t txPower,
  uint8_t dataRate,
  uint8_t channel,
  uint8_t wakeUpTime
)
{
  _nodeAddr = nodeAddr;
  delay(10);
  _serial.write((uint8_t)0xC2);
#ifdef E32_TRANSPARENT_MODE
  _serial.write((uint8_t)0);
  _serial.write((uint8_t)0);
  _serial.write(
    (0b00<<6) |      // UART parity: 8N1
    (0b011<<3) |     // UART baud: 9600bps
    (dataRate << 0)
    );
  _serial.write(_channel);
  _serial.write(
    (0<<7)          |  // Transparent transmission
    (1<<6)          |  // Internal pull-up
    (wakeUpTime<<3) |  //
    (1<<2)          |  // FEC on
    (txPower<<0)
  );
#else
#ifdef E32_FIXED_MODE
  _serial.write(HI_BYTE(_nodeAddr));
  _serial.write(LO_BYTE(_nodeAddr));
  _serial.write(
    (0b00<<6) |      // UART parity: 8N1
    (0b011<<3) |     // UART baud: 9600bps
    (dataRate << 0)
    );
  _serial.write(_channel);
  _serial.write(
    (1<<7)          |  // Fixed transmission
    (1<<6)          |  // Internal pull-up
    (wakeUpTime<<3) |  //
    (1<<2)          |  // FEC on
    (txPower<<0)
  );
#endif
#endif
//  @Begin debug
  delay(50);
  _serial.write(0xC1);
  _serial.write(0xC1);
  _serial.write(0xC1);
  delay(2000);
  while (_serial.available()) {
    Serial.print(_serial.read(),HEX);
    Serial.print(" ");
  }
  Serial.println();
  delay(2000);
//  @End debug
  NORMAL_MODE();
  delay(50);
}

/***********************************************
 * 
 */
void E32Radio::setChannel(uint8_t channel)
{  
  _channel = channel;
}

/***********************************************
 * 
 */
void E32Radio::send(Address dst, char* data, uint8_t len)
{  
  // packet structure to be written to tx buffer
  // 
  //  +---------+-----+----------+
  //  |Dst Addr | Len |   Data   | 
  //  | 2 (LE)  |  1  |   (len)  | 
  //  +---------+-----+----------+
  if (_txBuf.free() < sizeof(dst)+1+len)
    return;
  _txBuf.write((char*)&dst,sizeof(dst));
  _txBuf.writeByte(len);
  _txBuf.write(data,len);
}

/***********************************************
 * 
 */
void E32Radio::recv(Address* src, char* buf, uint8_t* len)
{
  if (!_rxBuf.empty())
  {
    _rxBuf.read((char*)src,sizeof(Address));
    *len = _rxBuf.readByte();
    _rxBuf.read(buf,*len);
  }
  else
  {
    *len = 0;
  }
}

/***********************************************
 * 
 */
void E32Radio::loop()
{
  _txThread(&_ptTxThread);
  _rxThread(&_ptRxThread);
}

/***********************************************
 * 
 */
PT_THREAD(E32Radio::_rxThread(struct pt* pt))
{
  static uint32_t ts; // timestamp for timer
  static Address dst;
 
  PT_BEGIN(pt);

  // @Begin debug
//  for (;;)
//  {
//    PT_WAIT_UNTIL(pt,_serial.available());
//    Serial.println(_serial.read(),HEX);
//  }
  // @End debug

  for (;;)
  {
    // look for a frame delimeter
    PT_WAIT_UNTIL(pt,_serial.available());
    if (_serial.read() != FRAME_DELIMETER) continue;

#ifdef E32_TRANSPARENT_MODE
    //  For TRANSPARENT MODE, data to be read from serial is
    //  +----+---------+----------+-----+----------+--------+
    //  | FD |Dst Addr | Src Addr | Len |   Data   | ChkSum | 
    //  | 1  | 2 (LE)  |  2 (LE)  |  1  |   (len)  |   2    | 
    //  +----+---------+----------+-----+----------+--------+
    TIMER_START(ts);
    PT_WAIT_UNTIL(pt,
            _serial.available() >= 6 || TIMER_EXPIRED(ts,SERIAL_TIMEOUT));
    if (TIMER_EXPIRED(ts,SERIAL_TIMEOUT)) continue;  // restart if timed out

    static Address src;
    static uint8_t len;
    _serial.readBytes((char*)&dst,sizeof(Address));
    _serial.readBytes((char*)&src,sizeof(Address));
    len = _serial.read();

    // make sure packet is fully received before timed out
    TIMER_START(ts);
    PT_WAIT_UNTIL(pt,
            _serial.available() >= len+2 || TIMER_EXPIRED(ts,SERIAL_TIMEOUT));
    if (TIMER_EXPIRED(ts,SERIAL_TIMEOUT)) continue;  // restart if timed out

    // store packet in temporary buffer first to make sure checksum is valid
    uint16_t sum = 0;
    sum += LO_BYTE(dst);
    sum += HI_BYTE(dst);
    sum += LO_BYTE(src);
    sum += HI_BYTE(src);
    sum += len;
#else
#ifdef E32_FIXED_MODE
    // For FIXED TRAMISSION MODE, data to be read from serial is
    //  +----+----------+-----+----------+--------+
    //  | FD | Src Addr | Len |   Data   | ChkSum | 
    //  | 1  |  2 (LE)  |  1  |   (len)  |   2    | 
    //  +----+----------+-----+----------+--------+
    TIMER_START(ts);
    PT_WAIT_UNTIL(pt,
            _serial.available() >= 3 || TIMER_EXPIRED(ts,SERIAL_TIMEOUT));
    if (TIMER_EXPIRED(ts,SERIAL_TIMEOUT)) continue;  // restart if timed out

    static Address src;
    static uint8_t len;
    dst = _nodeAddr;
    _serial.readBytes((char*)&src,sizeof(Address));
    len = _serial.read();

    // make sure packet is fully received before timed out
    TIMER_START(ts);
    PT_WAIT_UNTIL(pt,
            _serial.available() >= len+2 || TIMER_EXPIRED(ts,SERIAL_TIMEOUT));
    if (TIMER_EXPIRED(ts,SERIAL_TIMEOUT)) continue;  // restart if timed out

    // store packet in temporary buffer first to make sure checksum is valid
    uint16_t sum = 0;
    sum += LO_BYTE(src);
    sum += HI_BYTE(src);
    sum += len;
#endif
#endif

    char pkt[100];
    _serial.readBytes(pkt,len);
    for (uint8_t i=0; i<len; i++)
      sum += (uint8_t)pkt[i];
    uint16_t checksum;
    _serial.readBytes((char*)&checksum,sizeof(checksum));
    sum += checksum;

#ifdef DEBUG
    sprintf(sbuf,"RX: src=0x%04x,len=%d,data=",src,len);
    Serial.print(sbuf);
    for (uint8_t i=0; i<len; i++) {
      sprintf(sbuf,"%02hx ",(uint8_t)pkt[i]);
      Serial.print(sbuf);
    }
    sprintf(sbuf,",cs=0x%hx",sum);
    Serial.println(sbuf);
#endif

    // put the packet in the rx buffer only when
    // - checksum is valid
    // - destination address matches its own address or is broadcast
    // - rx buffer has sufficient space
    if (sum != 0) continue;
    if (dst != _nodeAddr && dst != BROADCAST) continue;
    if (_rxBuf.free() < sizeof(src)+sizeof(len)+len) continue;

    _rxBuf.write((char*)&src,sizeof(src));
    _rxBuf.write((char*)&len,sizeof(len));
    _rxBuf.write(pkt,len);
  }

  PT_END(pt);
}

/***********************************************
 * 
 */
PT_THREAD(E32Radio::_txThread(struct pt* pt))
{
  // XXX this may be problematic if serial tx buffer is too small
  static uint32_t ts; // timestamp
 
  PT_BEGIN(pt);

  for (;;)
  {
    // wait until there is a packet in the tx buffer, then read it out and
    // send it to E32 until the buffer is empty
    PT_WAIT_UNTIL(pt,!_txBuf.empty());

    static uint8_t len;
    static Address dst;

    _txBuf.read((char*)&dst,sizeof(dst));
    _txBuf.read((char*)&len,sizeof(len));

#ifdef E32_TRANSPARENT_MODE
    //  For TRANSPARENT MODE, data to be sent over serial is
    //  +----+---------+----------+-----+----------+--------+
    //  | FD |Dst Addr | Src Addr | Len |   Data   | ChkSum | 
    //  | 1  | 2 (LE)  |  2 (LE)  |  1  |   (len)  |   2    | 
    //  +----+---------+----------+-----+----------+--------+
    //  <-------------------- app data --------------------->
    //  <-------------------- E32 data --------------------->

    // make sure serial has enough room for the entire packet+header
    //PT_WAIT_UNTIL(pt,_serial.availableForWrite() >= 2+1+2+1+len+2);

    _serial.write(FRAME_DELIMETER);
    _serial.write(LO_BYTE(dst));
    _serial.write(HI_BYTE(dst));
    _serial.write(LO_BYTE(_nodeAddr));
    _serial.write(HI_BYTE(_nodeAddr));
    _serial.write(len);

    uint16_t checksum = 0;
    checksum -= LO_BYTE(dst);
    checksum -= HI_BYTE(dst);
    checksum -= LO_BYTE(_nodeAddr);
    checksum -= HI_BYTE(_nodeAddr);
    checksum -= len;
#else
#ifdef E32_FIXED_MODE
    // For FIXED TRAMISSION MODE, data to be sent over serial is
    //  +---------+---------+----+----------+-----+----------+--------+
    //  |Dst Addr | Channel | FD | Src Addr | Len |   Data   | ChkSum | 
    //  | 2 (BE)  |    1    | 1  |  2 (LE)  |  1  |   (len)  |   2    | 
    //  +---------+---------+----+----------+-----+----------+--------+
    //                      <---------------- app data --------------->
    //  <----------------------- E32 data ---------------------------->

    // make sure serial has enough room for the entire packet+header
    //PT_WAIT_UNTIL(pt,_serial.availableForWrite() >= 2+1+2+1+len+2);
    _serial.write(HI_BYTE(dst));
    _serial.write(LO_BYTE(dst));
    _serial.write(_channel);
    _serial.write(FRAME_DELIMETER);
    _serial.write(LO_BYTE(_nodeAddr));
    _serial.write(HI_BYTE(_nodeAddr));
    _serial.write(len);

    uint16_t checksum = 0;
    checksum -= LO_BYTE(_nodeAddr);
    checksum -= HI_BYTE(_nodeAddr);
    checksum -= len;

    // @Begin debug
//    Serial.println(HI_BYTE(dst),HEX);
//    Serial.println(LO_BYTE(dst),HEX);
//    Serial.println(_channel,HEX);
//    Serial.println(FRAME_DELIMETER,HEX);
//    Serial.println(LO_BYTE(_nodeAddr),HEX);
//    Serial.println(HI_BYTE(_nodeAddr),HEX);
//    Serial.println(len,HEX);
    // @End debug

#endif
#endif

    char pkt[100];
    _txBuf.read(pkt,len);
    for (uint8_t i=0; i<len; i++)
    {
      _serial.write(pkt[i]);
      checksum -= (uint8_t)pkt[i];
    }
    _serial.write((char*)&checksum,sizeof(checksum));
    
#ifdef DEBUG
    sprintf(sbuf,"TX: src=0x%04x,dst=0x%04x,len=%d,data=",_nodeAddr,dst,len);
    Serial.print(sbuf);
    for (uint8_t i=0; i<len; i++) {
      sprintf(sbuf,"%02hx ",(uint8_t)pkt[i]);
      Serial.print(sbuf);
    }
    sprintf(sbuf,",cs=0x%hx",checksum);
    Serial.println(sbuf);
#endif
  }
 
  PT_END(pt);
}
