/** 
 * @file
 * @brief Circular Buffer API
 *
 * Definitions for creating and maintaining circular buffers.
 *
 * @author Chaiporn Jaikaeo <chaiporn.j@ku.ac.th>
 *
 */
#ifndef __CIRBUF_H__
#define __CIRBUF_H__

#include <stdint.h>

/**
 * Struct for implementing a circular buffer
 */
class CirBuf
{
public:
  CirBuf(char *buffer, uint16_t size);

  /******************************************************************
   * Return the total size of a circular buffer
   */
  uint16_t size() const { return this->_size; }

  /******************************************************************
   * Check if the circular buffer is empty
   * @return true if the buffer is empty; false otherwise
   */
  bool empty() const { return this->_free == this->_size; }

  /******************************************************************
   * Check if the circular buffer is full
   * @return true if the buffer is full; false otherwise
   */
  bool full() const { return this->_free == 0; }

  /******************************************************************
   * Return the number of free bytes left in the buffer
   */
  uint16_t free() const { return this->_free; }

  /******************************************************************
   * Return the number of bytes stored in the buffer
   */
  uint16_t used() const { return this->_size - this->_free; }

  uint8_t writeByte(uint8_t data);
  uint16_t write(const char *data, uint16_t len);
  uint8_t readByte();
  uint16_t read(char *data, uint16_t len);
  void clear();

private:
  uint16_t _firstFree; ///< index of the first free byte
  uint16_t _size;      ///< total usable size of the buffer
  uint16_t _free;      ///< number of remaining free bytes
  char    *_buffer;    ///< pointer to the first byte of buffer mem
};

#endif
