#include <string.h>
#include "cirbuf.h"

/******************************************************************
 * Construct a circular buffer
 *
 * @param buffer pointer to buffer area
 * @param size size of the buffer to be created
 */
CirBuf::CirBuf(char *buffer, uint16_t size)
{
  // make size a power of two for better performance
  size >>= 1;
  size |= (size >> 1);
  size |= (size >> 2);
  size |= (size >> 4);
  size |= (size >> 8);
  size += 1;

  this->_firstFree = 0;
  this->_free      = size;
  this->_size      = size;
  this->_buffer    = (char*)buffer;
}

/******************************************************************
 * Write a single byte to the circular buffer
 *
 * @param data
 * byte value to be written
 *
 * @retval 0
 * write cannot be done because the buffer is already full
 * @retval 1
 * write is successful
 */
uint8_t CirBuf::writeByte(uint8_t data)
{
    if (this->full()) return 0;
    this->_free--;
    this->_buffer[this->_firstFree] = data;
    this->_firstFree++;
    this->_firstFree &= (this->_size-1); // eqv. to modulo this->_size
    return 1;
}

/******************************************************************
 * Write contents to a circular buffer
 *
 * @param data
 * pointer to the first byte of data
 *
 * @param len
 * number of bytes to be written
 *
 * @return
 * the number of bytes successfully written
 */
uint16_t CirBuf::write(const char *data, uint16_t len)
{
    uint16_t bytesToBoundary;

    // overflow protection
    if (this->_free < len) len = this->_free;

    if (len == 0) return 0;

    // Make sure we are not writing beyond the buffer boundary
    bytesToBoundary = this->_size - this->_firstFree;

    if (bytesToBoundary >= len)
        memcpy(&this->_buffer[this->_firstFree], data, len);
    else
    {
        // split write into two chunks
        // (1) from first-free to boundary
        memcpy(&this->_buffer[this->_firstFree], data, bytesToBoundary);

        // (2) from 0 to remaining bytes (after wrap-around)
        memcpy(this->_buffer, data+bytesToBoundary, len-bytesToBoundary);
    }

    // Update properties accordingly
    this->_free -= len;
    // As long as this->_size is a power of two, the following statement is
    // equivalent to:
    //    this->_firstFree = (this->_firstFree + len) % this->_size;
    this->_firstFree = (this->_firstFree + len) & (this->_size-1);

    return len;
}

/******************************************************************
 * Read a single byte from the circular buffer
 *
 * @return
 * next data byte from the buffer.  If the buffer is empty, 0 is return.
 */
uint8_t CirBuf::readByte()
{
    uint16_t firstIndex;
    if (this->empty()) return 0;
    firstIndex = (this->_firstFree + this->_free) & (this->_size-1);
    this->_free++;
    return this->_buffer[firstIndex];
}

/******************************************************************
 * Read contents out of a circular buffer and advance the read pointer
 *
 * @param data
 * pointer to the first byte of data.  If data is NULL, the contents will
 * simply be removed from the circular buffer
 *
 * @param len
 * number of bytes to be copied
 *
 * @return
 * the number of bytes successfully read out
 */
uint16_t CirBuf::read(char *data, uint16_t len)
{
    uint16_t firstIndex;
    uint16_t bytesToBoundary;

    // limit read to only stored bytes
    if ( (this->_size - this->_free) < len ) len = this->_size - this->_free; 
    if (len == 0) return 0;

    // Locate the first stored byte
    // As long as this->_size is a power of two, the following statement is
    // equivalent to:
    //    firstIndex = (this->_firstFree + this->_free) % this->_size;
    firstIndex = (this->_firstFree + this->_free) & (this->_size - 1);

    // Make sure we are not writing beyond the buffer boundary
    bytesToBoundary = this->_size - firstIndex;

    // Store contents in the provided memory location only when it is not NULL
    if (data)
    {
        if (bytesToBoundary >= len)
            memcpy(data, &this->_buffer[firstIndex], len);
        else
        {
            // split read into two chunks
            // (1) from first-_free to boundary
            memcpy(data, &this->_buffer[firstIndex], bytesToBoundary);

            // (2) from 0 to remaining bytes (after wrap-around)
            memcpy(data+bytesToBoundary, this->_buffer, len-bytesToBoundary);
        }
    }

    // Update properties accordingly
    this->_free += len;

    return len;
}

/******************************************************************
 * Clear all data from the circular buffer
 */
void CirBuf::clear()
{
    this->_firstFree = 0;
    this->_free      = this->_size;
}
