/*
 * Copyright 2012 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
module hunt.net.buffer.ByteBufUtil;


import hunt.net.buffer.AbstractByteBuf;
import hunt.net.buffer.ByteBuf;
import hunt.net.buffer.ByteBufAllocator;
import hunt.net.buffer.ByteProcessor;
import hunt.net.buffer.EmptyByteBuf;
import hunt.net.buffer.UnpooledByteBufAllocator;

import hunt.Byte;
import hunt.io.ByteBuffer;
import hunt.Exceptions;
import hunt.Integer;
import hunt.stream.Common;
import hunt.Long;
import hunt.net.Exceptions;
import hunt.Short;
import hunt.util.StringBuilder;
import hunt.text.Charset;
import hunt.util.ByteOrder;

import std.algorithm;
import std.ascii;
import std.conv;
import std.format;
import std.concurrency : initOnce;


/**
 * A collection of utility methods that is related with handling {@link ByteBuf},
 * such as the generation of hex dump and swapping an integer's byte order.
 */
final class ByteBufUtil {

    private static byte[] BYTE_ARRAYS() {
        static byte[] data;
        if(data is null) {
            data = new byte[MAX_TL_ARRAY_LEN];
        }

        return data;
    }

    private enum byte WRITE_UTF_UNKNOWN = '?';
    private enum int MAX_CHAR_BUFFER_SIZE = 16 * 1024;
    private enum int THREAD_LOCAL_BUFFER_SIZE = 0;
    // private enum int MAX_BYTES_PER_CHAR_UTF8 =
    //         (int) CharsetUtil.encoder(CharsetUtil.UTF_8).maxBytesPerChar();

    enum int WRITE_CHUNK_SIZE = 8192;
    __gshared ByteBufAllocator DEFAULT_ALLOCATOR;

    // shared static this() {
    //     UnpooledByteBufAllocator.DEFAULT;
    // }

    // static {
    //     string allocType = SystemPropertyUtil.get(
    //             "io.netty.allocator.type", PlatformDependent.isAndroid() ? "unpooled" : "pooled");
    //     allocType = allocType.toLowerCase(Locale.US).trim();

    //     ByteBufAllocator alloc;
    //     if ("unpooled" == allocType) {
    //         alloc = UnpooledByteBufAllocator.DEFAULT;
    //         logger.debug("-Dio.netty.allocator.type: {}", allocType);
    //     } else if ("pooled" == allocType) {
    //         alloc = PooledByteBufAllocator.DEFAULT;
    //         logger.debug("-Dio.netty.allocator.type: {}", allocType);
    //     } else {
    //         alloc = PooledByteBufAllocator.DEFAULT;
    //         logger.debug("-Dio.netty.allocator.type: pooled (unknown: {})", allocType);
    //     }

    //     DEFAULT_ALLOCATOR = alloc;

    //     THREAD_LOCAL_BUFFER_SIZE = SystemPropertyUtil.getInt("io.netty.threadLocalDirectBufferSize", 0);
    //     logger.debug("-Dio.netty.threadLocalDirectBufferSize: {}", THREAD_LOCAL_BUFFER_SIZE);

    //     MAX_CHAR_BUFFER_SIZE = SystemPropertyUtil.getInt("io.netty.maxThreadLocalCharBufferSize", 16 * 1024);
    //     logger.debug("-Dio.netty.maxThreadLocalCharBufferSize: {}", MAX_CHAR_BUFFER_SIZE);
    // }

    enum int MAX_TL_ARRAY_LEN = 1024;

    /**
     * Allocates a new array if minLength > {@link ByteBufUtil#MAX_TL_ARRAY_LEN}
     */
    static byte[] threadLocalTempArray(int minLength) {
        return minLength <= MAX_TL_ARRAY_LEN ? BYTE_ARRAYS
            : new byte[minLength];
    }

    /**
     * Returns a <a href="http://en.wikipedia.org/wiki/Hex_dump">hex dump</a>
     * of the specified buffer's readable bytes.
     */
    static string hexDump(ByteBuf buffer) {
        return hexDump(buffer, buffer.readerIndex(), buffer.readableBytes());
    }

    /**
     * Returns a <a href="http://en.wikipedia.org/wiki/Hex_dump">hex dump</a>
     * of the specified buffer's sub-region.
     */
    static string hexDump(ByteBuf buffer, int fromIndex, int length) {
        return HexUtil.hexDump(buffer, fromIndex, length);
    }

    /**
     * Returns a <a href="http://en.wikipedia.org/wiki/Hex_dump">hex dump</a>
     * of the specified byte array.
     */
    static string hexDump(byte[] array) {
        return hexDump(array, 0, array.length);
    }

    /**
     * Returns a <a href="http://en.wikipedia.org/wiki/Hex_dump">hex dump</a>
     * of the specified byte array's sub-region.
     */
    static string hexDump(byte[] array, size_t fromIndex, size_t length) {
        return HexUtil.hexDump(array, fromIndex, length);
    }

    // /**
    //  * Decode a 2-digit hex byte from within a string.
    //  */
    // static byte decodeHexByte(CharSequence s, int pos) {
    //     return StringUtil.decodeHexByte(s, pos);
    // }

    // /**
    //  * Decodes a string generated by {@link #hexDump(byte[])}
    //  */
    // static byte[] decodeHexDump(CharSequence hexDump) {
    //     return StringUtil.decodeHexDump(hexDump, 0, hexDump.length());
    // }

    // /**
    //  * Decodes part of a string generated by {@link #hexDump(byte[])}
    //  */
    // static byte[] decodeHexDump(CharSequence hexDump, int fromIndex, int length) {
    //     return StringUtil.decodeHexDump(hexDump, fromIndex, length);
    // }

    /**
     * Used to determine if the return value of {@link ByteBuf#ensureWritable(int, bool)} means that there is
     * adequate space and a write operation will succeed.
     * @param ensureWritableResult The return value from {@link ByteBuf#ensureWritable(int, bool)}.
     * @return {@code true} if {@code ensureWritableResult} means that there is adequate space and a write operation
     * will succeed.
     */
    static bool ensureWritableSuccess(int ensureWritableResult) {
        return ensureWritableResult == 0 || ensureWritableResult == 2;
    }

    /**
     * Calculates the hash code of the specified buffer.  This method is
     * useful when implementing a new buffer type.
     */
    static int toHash(ByteBuf buffer) {
        int aLen = buffer.readableBytes();
        int intCount = aLen >>> 2;
        int byteCount = aLen & 3;

        int hashCode = EmptyByteBuf.EMPTY_BYTE_BUF_HASH_CODE;
        int arrayIndex = buffer.readerIndex();
        if (buffer.order() == ByteOrder.BigEndian) {
            for (int i = intCount; i > 0; i --) {
                hashCode = 31 * hashCode + buffer.getInt(arrayIndex);
                arrayIndex += 4;
            }
        } else {
            for (int i = intCount; i > 0; i --) {
                hashCode = 31 * hashCode + swapInt(buffer.getInt(arrayIndex));
                arrayIndex += 4;
            }
        }

        for (int i = byteCount; i > 0; i --) {
            hashCode = 31 * hashCode + buffer.getByte(arrayIndex ++);
        }

        if (hashCode == 0) {
            hashCode = 1;
        }

        return hashCode;
    }

    /**
     * Returns the reader index of needle in haystack, or -1 if needle is not in haystack.
     */
    static int indexOf(ByteBuf needle, ByteBuf haystack) {
        // TODO: maybe use Boyer Moore for efficiency.
        int attempts = haystack.readableBytes() - needle.readableBytes() + 1;
        for (int i = 0; i < attempts; i++) {
            if (equals(needle, needle.readerIndex(),
                       haystack, haystack.readerIndex() + i,
                       needle.readableBytes())) {
                return haystack.readerIndex() + i;
            }
        }
        return -1;
    }

    /**
     * Returns {@code true} if and only if the two specified buffers are
     * identical to each other for {@code length} bytes starting at {@code aStartIndex}
     * index for the {@code a} buffer and {@code bStartIndex} index for the {@code b} buffer.
     * A more compact way to express this is:
     * <p>
     * {@code a[aStartIndex : aStartIndex + length] == b[bStartIndex : bStartIndex + length]}
     */
    static bool equals(ByteBuf a, int aStartIndex, ByteBuf b, int bStartIndex, int length) {
        if (aStartIndex < 0 || bStartIndex < 0 || length < 0) {
            throw new IllegalArgumentException("All indexes and lengths must be non-negative");
        }
        if (a.writerIndex() - length < aStartIndex || b.writerIndex() - length < bStartIndex) {
            return false;
        }

        int longCount = length >>> 3;
        int byteCount = length & 7;

        if (a.order() == b.order()) {
            for (int i = longCount; i > 0; i --) {
                if (a.getLong(aStartIndex) != b.getLong(bStartIndex)) {
                    return false;
                }
                aStartIndex += 8;
                bStartIndex += 8;
            }
        } else {
            for (int i = longCount; i > 0; i --) {
                if (a.getLong(aStartIndex) != swapLong(b.getLong(bStartIndex))) {
                    return false;
                }
                aStartIndex += 8;
                bStartIndex += 8;
            }
        }

        for (int i = byteCount; i > 0; i --) {
            if (a.getByte(aStartIndex) != b.getByte(bStartIndex)) {
                return false;
            }
            aStartIndex ++;
            bStartIndex ++;
        }

        return true;
    }

    /**
     * Returns {@code true} if and only if the two specified buffers are
     * identical to each other as described in {@link ByteBuf#equals(Object)}.
     * This method is useful when implementing a new buffer type.
     */
    static bool equals(ByteBuf bufferA, ByteBuf bufferB) {
        int aLen = bufferA.readableBytes();
        if (aLen != bufferB.readableBytes()) {
            return false;
        }
        return equals(bufferA, bufferA.readerIndex(), bufferB, bufferB.readerIndex(), aLen);
    }

    /**
     * Compares the two specified buffers as described in {@link ByteBuf#compareTo(ByteBuf)}.
     * This method is useful when implementing a new buffer type.
     */
    static int compare(ByteBuf bufferA, ByteBuf bufferB) {
        int aLen = bufferA.readableBytes();
        int bLen = bufferB.readableBytes();
        int minLength = min(aLen, bLen);
        int uintCount = minLength >>> 2;
        int byteCount = minLength & 3;
        int aIndex = bufferA.readerIndex();
        int bIndex = bufferB.readerIndex();

        if (uintCount > 0) {
            bool bufferAIsBigEndian = bufferA.order() == ByteOrder.BigEndian;
            long res;
            int uintCountIncrement = uintCount << 2;

            if (bufferA.order() == bufferB.order()) {
                res = bufferAIsBigEndian ? compareUintBigEndian(bufferA, bufferB, aIndex, bIndex, uintCountIncrement) :
                        compareUintLittleEndian(bufferA, bufferB, aIndex, bIndex, uintCountIncrement);
            } else {
                res = bufferAIsBigEndian ? compareUintBigEndianA(bufferA, bufferB, aIndex, bIndex, uintCountIncrement) :
                        compareUintBigEndianB(bufferA, bufferB, aIndex, bIndex, uintCountIncrement);
            }
            if (res != 0) {
                // Ensure we not overflow when cast
                return cast(int) min(int.max, max(int.min, res));
            }
            aIndex += uintCountIncrement;
            bIndex += uintCountIncrement;
        }

        for (int aEnd = aIndex + byteCount; aIndex < aEnd; ++aIndex, ++bIndex) {
            int comp = bufferA.getUnsignedByte(aIndex) - bufferB.getUnsignedByte(bIndex);
            if (comp != 0) {
                return comp;
            }
        }

        return aLen - bLen;
    }

    private static long compareUintBigEndian(
            ByteBuf bufferA, ByteBuf bufferB, int aIndex, int bIndex, int uintCountIncrement) {
        for (int aEnd = aIndex + uintCountIncrement; aIndex < aEnd; aIndex += 4, bIndex += 4) {
            long comp = bufferA.getUnsignedInt(aIndex) - bufferB.getUnsignedInt(bIndex);
            if (comp != 0) {
                return comp;
            }
        }
        return 0;
    }

    private static long compareUintLittleEndian(
            ByteBuf bufferA, ByteBuf bufferB, int aIndex, int bIndex, int uintCountIncrement) {
        for (int aEnd = aIndex + uintCountIncrement; aIndex < aEnd; aIndex += 4, bIndex += 4) {
            long comp = bufferA.getUnsignedIntLE(aIndex) - bufferB.getUnsignedIntLE(bIndex);
            if (comp != 0) {
                return comp;
            }
        }
        return 0;
    }

    private static long compareUintBigEndianA(
            ByteBuf bufferA, ByteBuf bufferB, int aIndex, int bIndex, int uintCountIncrement) {
        for (int aEnd = aIndex + uintCountIncrement; aIndex < aEnd; aIndex += 4, bIndex += 4) {
            long comp =  bufferA.getUnsignedInt(aIndex) - bufferB.getUnsignedIntLE(bIndex);
            if (comp != 0) {
                return comp;
            }
        }
        return 0;
    }

    private static long compareUintBigEndianB(
            ByteBuf bufferA, ByteBuf bufferB, int aIndex, int bIndex, int uintCountIncrement) {
        for (int aEnd = aIndex + uintCountIncrement; aIndex < aEnd; aIndex += 4, bIndex += 4) {
            long comp =  bufferA.getUnsignedIntLE(aIndex) - bufferB.getUnsignedInt(bIndex);
            if (comp != 0) {
                return comp;
            }
        }
        return 0;
    }

    /**
     * The default implementation of {@link ByteBuf#indexOf(int, int, byte)}.
     * This method is useful when implementing a new buffer type.
     */
    static int indexOf(ByteBuf buffer, int fromIndex, int toIndex, byte value) {
        if (fromIndex <= toIndex) {
            return firstIndexOf(buffer, fromIndex, toIndex, value);
        } else {
            return lastIndexOf(buffer, fromIndex, toIndex, value);
        }
    }

    /**
     * Toggles the endianness of the specified 16-bit short integer.
     */
    static short swapShort(short value) {
        return Short.reverseBytes(value);
    }

    /**
     * Toggles the endianness of the specified 24-bit medium integer.
     */
    static int swapMedium(int value) {
        int swapped = value << 16 & 0xff0000 | value & 0xff00 | value >>> 16 & 0xff;
        if ((swapped & 0x800000) != 0) {
            swapped |= 0xff000000;
        }
        return swapped;
    }

    /**
     * Toggles the endianness of the specified 32-bit integer.
     */
    static int swapInt(int value) {
        return Integer.reverseBytes(value);
    }

    /**
     * Toggles the endianness of the specified 64-bit long integer.
     */
    static long swapLong(long value) {
        return Long.reverseBytes(value);
    }

    /**
     * Writes a big-endian 16-bit short integer to the buffer.
     */
    // @SuppressWarnings("deprecation")
    // static ByteBuf writeShortBE(ByteBuf buf, int shortValue) {
    //     return buf.order() == ByteProcessor? buf.writeShort(shortValue) : buf.writeShortLE(shortValue);
    // }

    /**
     * Sets a big-endian 16-bit short integer to the buffer.
     */
    // @SuppressWarnings("deprecation")
    // static ByteBuf setShortBE(ByteBuf buf, int index, int shortValue) {
    //     return buf.order() == ByteProcessor? buf.setShort(index, shortValue) : buf.setShortLE(index, shortValue);
    // }

    /**
     * Writes a big-endian 24-bit medium integer to the buffer.
     */
    // @SuppressWarnings("deprecation")
    // static ByteBuf writeMediumBE(ByteBuf buf, int mediumValue) {
    //     return buf.order() == ByteProcessor? buf.writeMedium(mediumValue) : buf.writeMediumLE(mediumValue);
    // }

    /**
     * Read the given amount of bytes into a new {@link ByteBuf} that is allocated from the {@link ByteBufAllocator}.
     */
    static ByteBuf readBytes(ByteBufAllocator alloc, ByteBuf buffer, int length) {
        bool release = true;
        ByteBuf dst = alloc.buffer(length);
        try {
            buffer.readBytes(dst);
            release = false;
            return dst;
        } finally {
            if (release) {
                dst.release();
            }
        }
    }

    private static int firstIndexOf(ByteBuf buffer, int fromIndex, int toIndex, byte value) {
        fromIndex = max(fromIndex, 0);
        if (fromIndex >= toIndex || buffer.capacity() == 0) {
            return -1;
        }

        return buffer.forEachByte(fromIndex, toIndex - fromIndex, new IndexOfProcessor(value));
    }

    private static int lastIndexOf(ByteBuf buffer, int fromIndex, int toIndex, byte value) {
        int capacity = buffer.capacity();
        fromIndex = min(fromIndex, capacity);
        if (fromIndex < 0 || capacity == 0) {
            return -1;
        }

        return buffer.forEachByteDesc(toIndex, fromIndex - toIndex, new IndexOfProcessor(value));
    }

    // private static CharSequence checkCharSequenceBounds(CharSequence seq, int start, int end) {
    //     if (MathUtil.isOutOfBounds(start, end - start, seq.length())) {
    //         throw new IndexOutOfBoundsException("expected: 0 <= start(" ~ start ~ ") <= end (" ~ end
    //                 ~ ") <= seq.length(" ~ seq.length() + ')');
    //     }
    //     return seq;
    // }

    /**
     * Encode a {@link CharSequence} in <a href="http://en.wikipedia.org/wiki/UTF-8">UTF-8</a> and write
     * it to a {@link ByteBuf} allocated with {@code alloc}.
     * @param alloc The allocator used to allocate a new {@link ByteBuf}.
     * @param seq The characters to write into a buffer.
     * @return The {@link ByteBuf} which contains the <a href="http://en.wikipedia.org/wiki/UTF-8">UTF-8</a> encoded
     * result.
     */
    // static ByteBuf writeUtf8(ByteBufAllocator alloc, CharSequence seq) {
    //     // UTF-8 uses max. 3 bytes per char, so calculate the worst case.
    //     ByteBuf buf = alloc.buffer(utf8MaxBytes(seq));
    //     writeUtf8(buf, seq);
    //     return buf;
    // }

    /**
     * Encode a {@link CharSequence} in <a href="http://en.wikipedia.org/wiki/UTF-8">UTF-8</a> and write
     * it to a {@link ByteBuf}.
     * <p>
     * It behaves like {@link #reserveAndWriteUtf8(ByteBuf, CharSequence, int)} with {@code reserveBytes}
     * computed by {@link #utf8MaxBytes(CharSequence)}.<br>
     * This method returns the actual number of bytes written.
     */
    // static int writeUtf8(ByteBuf buf, CharSequence seq) {
    //     int seqLength = seq.length();
    //     return reserveAndWriteUtf8Seq(buf, seq, 0, seqLength, utf8MaxBytes(seqLength));
    // }

    /**
     * Equivalent to <code>{@link #writeUtf8(ByteBuf, CharSequence) writeUtf8(buf, seq.subSequence(start, end))}</code>
     * but avoids subsequence object allocation.
     */
    // static int writeUtf8(ByteBuf buf, CharSequence seq, int start, int end) {
    //     checkCharSequenceBounds(seq, start, end);
    //     return reserveAndWriteUtf8Seq(buf, seq, start, end, utf8MaxBytes(end - start));
    // }

    /**
     * Encode a {@link CharSequence} in <a href="http://en.wikipedia.org/wiki/UTF-8">UTF-8</a> and write
     * it into {@code reserveBytes} of a {@link ByteBuf}.
     * <p>
     * The {@code reserveBytes} must be computed (ie eagerly using {@link #utf8MaxBytes(CharSequence)}
     * or exactly with {@link #utf8Bytes(CharSequence)}) to ensure this method to not fail: for performance reasons
     * the index checks will be performed using just {@code reserveBytes}.<br>
     * This method returns the actual number of bytes written.
     */
    // static int reserveAndWriteUtf8(ByteBuf buf, CharSequence seq, int reserveBytes) {
    //     return reserveAndWriteUtf8Seq(buf, seq, 0, seq.length(), reserveBytes);
    // }

    /**
     * Equivalent to <code>{@link #reserveAndWriteUtf8(ByteBuf, CharSequence, int)
     * reserveAndWriteUtf8(buf, seq.subSequence(start, end), reserveBytes)}</code> but avoids
     * subsequence object allocation if possible.
     *
     * @return actual number of bytes written
     */
    // static int reserveAndWriteUtf8(ByteBuf buf, CharSequence seq, int start, int end, int reserveBytes) {
    //     return reserveAndWriteUtf8Seq(buf, checkCharSequenceBounds(seq, start, end), start, end, reserveBytes);
    // }

    // private static int reserveAndWriteUtf8Seq(ByteBuf buf, CharSequence seq, int start, int end, int reserveBytes) {
    //     for (;;) {
    //         if (buf instanceof WrappedCompositeByteBuf) {
    //             // WrappedCompositeByteBuf is a sub-class of AbstractByteBuf so it needs special handling.
    //             buf = buf.unwrap();
    //         } else if (buf instanceof AbstractByteBuf) {
    //             AbstractByteBuf byteBuf = (AbstractByteBuf) buf;
    //             byteBuf.ensureWritable0(reserveBytes);
    //             int written = writeUtf8(byteBuf, byteBuf.writerIndex, seq, start, end);
    //             byteBuf.writerIndex += written;
    //             return written;
    //         } else if (buf instanceof WrappedByteBuf) {
    //             // Unwrap as the wrapped buffer may be an AbstractByteBuf and so we can use fast-path.
    //             buf = buf.unwrap();
    //         } else {
    //             byte[] bytes = seq.subSequence(start, end).toString().getBytes(CharsetUtil.UTF_8);
    //             buf.writeBytes(bytes);
    //             return bytes.length;
    //         }
    //     }
    // }

    // static int writeUtf8(AbstractByteBuf buffer, int writerIndex, CharSequence seq, int len) {
    //     return writeUtf8(buffer, writerIndex, seq, 0, len);
    // }

    // // Fast-Path implementation
    // static int writeUtf8(AbstractByteBuf buffer, int writerIndex, CharSequence seq, int start, int end) {
    //     int oldWriterIndex = writerIndex;

    //     // We can use the _set methods as these not need to do any index checks and reference checks.
    //     // This is possible as we called ensureWritable(...) before.
    //     for (int i = start; i < end; i++) {
    //         char c = seq[i];
    //         if (c < 0x80) {
    //             buffer._setByte(writerIndex++, (byte) c);
    //         } else if (c < 0x800) {
    //             buffer._setByte(writerIndex++, (byte) (0xc0 | (c >> 6)));
    //             buffer._setByte(writerIndex++, (byte) (0x80 | (c & 0x3f)));
    //         } else if (isSurrogate(c)) {
    //             if (!Character.isHighSurrogate(c)) {
    //                 buffer._setByte(writerIndex++, WRITE_UTF_UNKNOWN);
    //                 continue;
    //             }
    //             // Surrogate Pair consumes 2 characters.
    //             if (++i == end) {
    //                 buffer._setByte(writerIndex++, WRITE_UTF_UNKNOWN);
    //                 break;
    //             }
    //             // Extra method to allow inlining the rest of writeUtf8 which is the most likely code path.
    //             writerIndex = writeUtf8Surrogate(buffer, writerIndex, c, seq[i]);
    //         } else {
    //             buffer._setByte(writerIndex++, (byte) (0xe0 | (c >> 12)));
    //             buffer._setByte(writerIndex++, (byte) (0x80 | ((c >> 6) & 0x3f)));
    //             buffer._setByte(writerIndex++, (byte) (0x80 | (c & 0x3f)));
    //         }
    //     }
    //     return writerIndex - oldWriterIndex;
    // }

    // private static int writeUtf8Surrogate(AbstractByteBuf buffer, int writerIndex, char c, char c2) {
    //     if (!Character.isLowSurrogate(c2)) {
    //         buffer._setByte(writerIndex++, WRITE_UTF_UNKNOWN);
    //         buffer._setByte(writerIndex++, Character.isHighSurrogate(c2) ? WRITE_UTF_UNKNOWN : c2);
    //         return writerIndex;
    //     }
    //     int codePoint = Character.toCodePoint(c, c2);
    //     // See http://www.unicode.org/versions/Unicode7.0.0/ch03.pdf#G2630.
    //     buffer._setByte(writerIndex++, (byte) (0xf0 | (codePoint >> 18)));
    //     buffer._setByte(writerIndex++, (byte) (0x80 | ((codePoint >> 12) & 0x3f)));
    //     buffer._setByte(writerIndex++, (byte) (0x80 | ((codePoint >> 6) & 0x3f)));
    //     buffer._setByte(writerIndex++, (byte) (0x80 | (codePoint & 0x3f)));
    //     return writerIndex;
    // }

    /**
     * Returns max bytes length of UTF8 character sequence of the given length.
     */
    // static int utf8MaxBytes(int seqLength) {
    //     return seqLength * MAX_BYTES_PER_CHAR_UTF8;
    // }

    /**
     * Returns max bytes length of UTF8 character sequence.
     * <p>
     * It behaves like {@link #utf8MaxBytes(int)} applied to {@code seq} {@link CharSequence#length()}.
     */
    // static int utf8MaxBytes(CharSequence seq) {
    //     return utf8MaxBytes(seq.length());
    // }

    /**
     * Returns the exact bytes length of UTF8 character sequence.
     * <p>
     * This method is producing the exact length according to {@link #writeUtf8(ByteBuf, CharSequence)}.
     */
    // static int utf8Bytes(CharSequence seq) {
    //     return utf8ByteCount(seq, 0, seq.length());
    // }

    /**
     * Equivalent to <code>{@link #utf8Bytes(CharSequence) utf8Bytes(seq.subSequence(start, end))}</code>
     * but avoids subsequence object allocation.
     * <p>
     * This method is producing the exact length according to {@link #writeUtf8(ByteBuf, CharSequence, int, int)}.
     */
    // static int utf8Bytes(CharSequence seq, int start, int end) {
    //     return utf8ByteCount(checkCharSequenceBounds(seq, start, end), start, end);
    // }

    // private static int utf8ByteCount(CharSequence seq, int start, int end) {
    //     if (seq instanceof AsciiString) {
    //         return end - start;
    //     }
    //     int i = start;
    //     // ASCII fast path
    //     while (i < end && seq[i] < 0x80) {
    //         ++i;
    //     }
    //     // !ASCII is packed in a separate method to let the ASCII case be smaller
    //     return i < end ? (i - start) + utf8BytesNonAscii(seq, i, end) : i - start;
    // }

    // private static int utf8BytesNonAscii(CharSequence seq, int start, int end) {
    //     int encodedLength = 0;
    //     for (int i = start; i < end; i++) {
    //         char c = seq[i];
    //         // making it 100% branchless isn't rewarding due to the many bit operations necessary!
    //         if (c < 0x800) {
    //             // branchless version of: (c <= 127 ? 0:1) + 1
    //             encodedLength += ((0x7f - c) >>> 31) + 1;
    //         } else if (isSurrogate(c)) {
    //             if (!Character.isHighSurrogate(c)) {
    //                 encodedLength++;
    //                 // WRITE_UTF_UNKNOWN
    //                 continue;
    //             }
    //             // Surrogate Pair consumes 2 characters.
    //             if (++i == end) {
    //                 encodedLength++;
    //                 // WRITE_UTF_UNKNOWN
    //                 break;
    //             }
    //             if (!Character.isLowSurrogate(seq[i])) {
    //                 // WRITE_UTF_UNKNOWN + (Character.isHighSurrogate(c2) ? WRITE_UTF_UNKNOWN : c2)
    //                 encodedLength += 2;
    //                 continue;
    //             }
    //             // See http://www.unicode.org/versions/Unicode7.0.0/ch03.pdf#G2630.
    //             encodedLength += 4;
    //         } else {
    //             encodedLength += 3;
    //         }
    //     }
    //     return encodedLength;
    // }

    /**
     * Encode a {@link CharSequence} in <a href="http://en.wikipedia.org/wiki/ASCII">ASCII</a> and write
     * it to a {@link ByteBuf} allocated with {@code alloc}.
     * @param alloc The allocator used to allocate a new {@link ByteBuf}.
     * @param seq The characters to write into a buffer.
     * @return The {@link ByteBuf} which contains the <a href="http://en.wikipedia.org/wiki/ASCII">ASCII</a> encoded
     * result.
     */
    // static ByteBuf writeAscii(ByteBufAllocator alloc, CharSequence seq) {
    //     // ASCII uses 1 byte per char
    //     ByteBuf buf = alloc.buffer(seq.length());
    //     writeAscii(buf, seq);
    //     return buf;
    // }

    // /**
    //  * Encode a {@link CharSequence} in <a href="http://en.wikipedia.org/wiki/ASCII">ASCII</a> and write it
    //  * to a {@link ByteBuf}.
    //  *
    //  * This method returns the actual number of bytes written.
    //  */
    // static int writeAscii(ByteBuf buf, CharSequence seq) {
    //     // ASCII uses 1 byte per char
    //     int len = seq.length();
    //     if (seq instanceof AsciiString) {
    //         AsciiString asciiString = (AsciiString) seq;
    //         buf.writeBytes(asciiString.array(), asciiString.arrayOffset(), len);
    //     } else {
    //         for (;;) {
    //             if (buf instanceof WrappedCompositeByteBuf) {
    //                 // WrappedCompositeByteBuf is a sub-class of AbstractByteBuf so it needs special handling.
    //                 buf = buf.unwrap();
    //             } else if (buf instanceof AbstractByteBuf) {
    //                 AbstractByteBuf byteBuf = (AbstractByteBuf) buf;
    //                 byteBuf.ensureWritable0(len);
    //                 int written = writeAscii(byteBuf, byteBuf.writerIndex, seq, len);
    //                 byteBuf.writerIndex += written;
    //                 return written;
    //             } else if (buf instanceof WrappedByteBuf) {
    //                 // Unwrap as the wrapped buffer may be an AbstractByteBuf and so we can use fast-path.
    //                 buf = buf.unwrap();
    //             } else {
    //                 byte[] bytes = seq.toString().getBytes(CharsetUtil.US_ASCII);
    //                 buf.writeBytes(bytes);
    //                 return bytes.length;
    //             }
    //         }
    //     }
    //     return len;
    // }

    // // Fast-Path implementation
    // static int writeAscii(AbstractByteBuf buffer, int writerIndex, CharSequence seq, int len) {

    //     // We can use the _set methods as these not need to do any index checks and reference checks.
    //     // This is possible as we called ensureWritable(...) before.
    //     for (int i = 0; i < len; i++) {
    //         buffer._setByte(writerIndex++, AsciiString.c2b(seq[i]));
    //     }
    //     return len;
    // }

    /**
     * Encode the given {@link CharBuffer} using the given {@link Charset} into a new {@link ByteBuf} which
     * is allocated via the {@link ByteBufAllocator}.
     */
    // static ByteBuf encodeString(ByteBufAllocator alloc, CharBuffer src, Charset charset) {
    //     return encodeString0(alloc, false, src, charset, 0);
    // }

    // /**
    //  * Encode the given {@link CharBuffer} using the given {@link Charset} into a new {@link ByteBuf} which
    //  * is allocated via the {@link ByteBufAllocator}.
    //  *
    //  * @param alloc The {@link ByteBufAllocator} to allocate {@link ByteBuf}.
    //  * @param src The {@link CharBuffer} to encode.
    //  * @param charset The specified {@link Charset}.
    //  * @param extraCapacity the extra capacity to alloc except the space for decoding.
    //  */
    // static ByteBuf encodeString(ByteBufAllocator alloc, CharBuffer src, Charset charset, int extraCapacity) {
    //     return encodeString0(alloc, false, src, charset, extraCapacity);
    // }

    // static ByteBuf encodeString0(ByteBufAllocator alloc, bool enforceHeap, CharBuffer src, Charset charset,
    //                              int extraCapacity) {
    //     CharsetEncoder encoder = CharsetUtil.encoder(charset);
    //     int length = cast(int) ((double) src.remaining() * encoder.maxBytesPerChar()) + extraCapacity;
    //     bool release = true;
    //     ByteBuf dst;
    //     if (enforceHeap) {
    //         dst = alloc.heapBuffer(length);
    //     } else {
    //         dst = alloc.buffer(length);
    //     }
    //     try {
    //         ByteBuffer dstBuf = dst.internalNioBuffer(dst.readerIndex(), length);
    //         int pos = dstBuf.position();
    //         CoderResult cr = encoder.encode(src, dstBuf, true);
    //         if (!cr.isUnderflow()) {
    //             cr.throwException();
    //         }
    //         cr = encoder.flush(dstBuf);
    //         if (!cr.isUnderflow()) {
    //             cr.throwException();
    //         }
    //         dst.writerIndex(dst.writerIndex() + dstBuf.position() - pos);
    //         release = false;
    //         return dst;
    //     } catch (CharacterCodingException x) {
    //         throw new IllegalStateException(x);
    //     } finally {
    //         if (release) {
    //             dst.release();
    //         }
    //     }
    // }

    static string decodeString(ByteBuf src, int readerIndex, int len, Charset charset) {
        if (len == 0) {
            return "";
        }
        byte[] array;
        int offset;

        if (src.hasArray()) {
            array = src.array();
            offset = src.arrayOffset() + readerIndex;
        } else {
            array = threadLocalTempArray(len);
            offset = 0;
            src.getBytes(readerIndex, array, 0, len);
        }
        // if (CharsetUtil.US_ASCII == charset) {
        //     // Fast-path for US-ASCII which is used frequently.
        //     return new string(array, 0, offset, len);
        // }
        // FIXME: Needing refactor or cleanup -@zxp at 8/20/2019, 4:45:20 PM
        // dup here for safe
        return cast(string) array[offset .. offset+len].idup;
    }

    /**
     * Returns a cached thread-local direct buffer, if available.
     *
     * @return a cached thread-local direct buffer, if available.  {@code null} otherwise.
     */
    // static ByteBuf threadLocalDirectBuffer() {
    //     if (THREAD_LOCAL_BUFFER_SIZE <= 0) {
    //         return null;
    //     }

    //     if (PlatformDependent.hasUnsafe()) {
    //         return ThreadLocalUnsafeDirectByteBuf.newInstance();
    //     } else {
    //         return ThreadLocalDirectByteBuf.newInstance();
    //     }
    // }

    /**
     * Create a copy of the underlying storage from {@code buf} into a byte array.
     * The copy will start at {@link ByteBuf#readerIndex()} and copy {@link ByteBuf#readableBytes()} bytes.
     */
    static byte[] getBytes(ByteBuf buf) {
        return getBytes(buf,  buf.readerIndex(), buf.readableBytes());
    }

    /**
     * Create a copy of the underlying storage from {@code buf} into a byte array.
     * The copy will start at {@code start} and copy {@code length} bytes.
     */
    static byte[] getBytes(ByteBuf buf, int start, int length) {
        return getBytes(buf, start, length, true);
    }

    /**
     * Return an array of the underlying storage from {@code buf} into a byte array.
     * The copy will start at {@code start} and copy {@code length} bytes.
     * If {@code copy} is true a copy will be made of the memory.
     * If {@code copy} is false the underlying storage will be shared, if possible.
     */
    static byte[] getBytes(ByteBuf buf, int start, int length, bool copy) {
        int capacity = buf.capacity();
        if (isOutOfBounds(start, length, capacity)) {
            string msg = format("expected: " ~ "0 <= start(%d) <= start + length(%d) <= " ~ 
                "buf.capacity(%d)", start, length, capacity);
            throw new IndexOutOfBoundsException(msg);
        }

        if (buf.hasArray()) {
            if (copy || start != 0 || length != capacity) {
                int baseOffset = buf.arrayOffset() + start;
                byte[] b = buf.array();
                return b[baseOffset .. baseOffset + length].dup;
                // return Arrays.copyOfRange(buf.array(), baseOffset, baseOffset + length);
            } else {
                return buf.array();
            }
        }

        byte[] v = new byte[length]; // PlatformDependent.allocateUninitializedArray(length);
        buf.getBytes(start, v);
        return v;
    }

    // /**
    //  * Copies the all content of {@code src} to a {@link ByteBuf} using {@link ByteBuf#writeBytes(byte[], int, int)}.
    //  *
    //  * @param src the source string to copy
    //  * @param dst the destination buffer
    //  */
    // static void copy(AsciiString src, ByteBuf dst) {
    //     copy(src, 0, dst, src.length());
    // }

    // /**
    //  * Copies the content of {@code src} to a {@link ByteBuf} using {@link ByteBuf#setBytes(int, byte[], int, int)}.
    //  * Unlike the {@link #copy(AsciiString, ByteBuf)} and {@link #copy(AsciiString, int, ByteBuf, int)} methods,
    //  * this method do not increase a {@code writerIndex} of {@code dst} buffer.
    //  *
    //  * @param src the source string to copy
    //  * @param srcIdx the starting offset of characters to copy
    //  * @param dst the destination buffer
    //  * @param dstIdx the starting offset in the destination buffer
    //  * @param length the number of characters to copy
    //  */
    // static void copy(AsciiString src, int srcIdx, ByteBuf dst, int dstIdx, int length) {
    //     if (isOutOfBounds(srcIdx, length, cast(int)src.length) {
    //         string msg = format("expected: " ~ "0 <= srcIdx(%d) <= srcIdx + length(%d) <= srcLen(%d)", 
    //             srcIdx, length, src.length)
    //         throw new IndexOutOfBoundsException(msg);
    //     }

    //     checkNotNull(dst, "dst").setBytes(dstIdx, src.array(), srcIdx + src.arrayOffset(), length);
    // }

    // /**
    //  * Copies the content of {@code src} to a {@link ByteBuf} using {@link ByteBuf#writeBytes(byte[], int, int)}.
    //  *
    //  * @param src the source string to copy
    //  * @param srcIdx the starting offset of characters to copy
    //  * @param dst the destination buffer
    //  * @param length the number of characters to copy
    //  */
    // static void copy(AsciiString src, int srcIdx, ByteBuf dst, int length) {
    //     if (isOutOfBounds(srcIdx, length, src.length())) {
    //         throw new IndexOutOfBoundsException("expected: " ~ "0 <= srcIdx(" ~ srcIdx ~ ") <= srcIdx + length("
    //                         + length ~ ") <= srcLen(" ~ src.length() + ')');
    //     }

    //     checkNotNull(dst, "dst").writeBytes(src.array(), srcIdx + src.arrayOffset(), length);
    // }

    // /**
    //  * Returns a multi-line hexadecimal dump of the specified {@link ByteBuf} that is easy to read by humans.
    //  */
    // static string prettyHexDump(ByteBuf buffer) {
    //     return prettyHexDump(buffer, buffer.readerIndex(), buffer.readableBytes());
    // }

    // /**
    //  * Returns a multi-line hexadecimal dump of the specified {@link ByteBuf} that is easy to read by humans,
    //  * starting at the given {@code offset} using the given {@code length}.
    //  */
    // static string prettyHexDump(ByteBuf buffer, int offset, int length) {
    //     return HexUtil.prettyHexDump(buffer, offset, length);
    // }

    // /**
    //  * Appends the prettified multi-line hexadecimal dump of the specified {@link ByteBuf} to the specified
    //  * {@link StringBuilder} that is easy to read by humans.
    //  */
    // static void appendPrettyHexDump(StringBuilder dump, ByteBuf buf) {
    //     appendPrettyHexDump(dump, buf, buf.readerIndex(), buf.readableBytes());
    // }

    // /**
    //  * Appends the prettified multi-line hexadecimal dump of the specified {@link ByteBuf} to the specified
    //  * {@link StringBuilder} that is easy to read by humans, starting at the given {@code offset} using
    //  * the given {@code length}.
    //  */
    // static void appendPrettyHexDump(StringBuilder dump, ByteBuf buf, int offset, int length) {
    //     HexUtil.appendPrettyHexDump(dump, buf, offset, length);
    // }


    // static final class ThreadLocalUnsafeDirectByteBuf : UnpooledUnsafeDirectByteBuf {

    //     private static final Recycler!(ThreadLocalUnsafeDirectByteBuf) RECYCLER =
    //             new Recycler!(ThreadLocalUnsafeDirectByteBuf)() {
    //                 override
    //                 protected ThreadLocalUnsafeDirectByteBuf newObject(Handle!(ThreadLocalUnsafeDirectByteBuf) handle) {
    //                     return new ThreadLocalUnsafeDirectByteBuf(handle);
    //                 }
    //             };

    //     static ThreadLocalUnsafeDirectByteBuf newInstance() {
    //         ThreadLocalUnsafeDirectByteBuf buf = RECYCLER.get();
    //         buf.resetRefCnt();
    //         return buf;
    //     }

    //     private Handle!(ThreadLocalUnsafeDirectByteBuf) handle;

    //     private ThreadLocalUnsafeDirectByteBuf(Handle!(ThreadLocalUnsafeDirectByteBuf) handle) {
    //         super(UnpooledByteBufAllocator.DEFAULT, 256, int.max);
    //         this.handle = handle;
    //     }

    //     override
    //     protected void deallocate() {
    //         if (capacity() > THREAD_LOCAL_BUFFER_SIZE) {
    //             super.deallocate();
    //         } else {
    //             clear();
    //             handle.recycle(this);
    //         }
    //     }
    // }

    // static final class ThreadLocalDirectByteBuf : UnpooledDirectByteBuf {

    //     private static final Recycler!(ThreadLocalDirectByteBuf) RECYCLER = new Recycler!(ThreadLocalDirectByteBuf)() {
    //         override
    //         protected ThreadLocalDirectByteBuf newObject(Handle!(ThreadLocalDirectByteBuf) handle) {
    //             return new ThreadLocalDirectByteBuf(handle);
    //         }
    //     };

    //     static ThreadLocalDirectByteBuf newInstance() {
    //         ThreadLocalDirectByteBuf buf = RECYCLER.get();
    //         buf.resetRefCnt();
    //         return buf;
    //     }

    //     private Handle!(ThreadLocalDirectByteBuf) handle;

    //     private ThreadLocalDirectByteBuf(Handle!(ThreadLocalDirectByteBuf) handle) {
    //         super(UnpooledByteBufAllocator.DEFAULT, 256, int.max);
    //         this.handle = handle;
    //     }

    //     override
    //     protected void deallocate() {
    //         if (capacity() > THREAD_LOCAL_BUFFER_SIZE) {
    //             super.deallocate();
    //         } else {
    //             clear();
    //             handle.recycle(this);
    //         }
    //     }
    // }

    /**
     * Returns {@code true} if the given {@link ByteBuf} is valid text using the given {@link Charset},
     * otherwise return {@code false}.
     *
     * @param buf The given {@link ByteBuf}.
     * @param charset The specified {@link Charset}.
     */
    // static bool isText(ByteBuf buf, Charset charset) {
    //     return isText(buf, buf.readerIndex(), buf.readableBytes(), charset);
    // }

    /**
     * Returns {@code true} if the specified {@link ByteBuf} starting at {@code index} with {@code length} is valid
     * text using the given {@link Charset}, otherwise return {@code false}.
     *
     * @param buf The given {@link ByteBuf}.
     * @param index The start index of the specified buffer.
     * @param length The length of the specified buffer.
     * @param charset The specified {@link Charset}.
     *
     * @throws IndexOutOfBoundsException if {@code index} + {@code length} is greater than {@code buf.readableBytes}
     */
    // static bool isText(ByteBuf buf, int index, int length, Charset charset) {
    //     checkNotNull(buf, "buf");
    //     checkNotNull(charset, "charset");
    //     int maxIndex = buf.readerIndex() + buf.readableBytes();
    //     if (index < 0 || length < 0 || index > maxIndex - length) {
    //         throw new IndexOutOfBoundsException("index: " ~ index ~ " length: " ~ length);
    //     }
    //     if (charset == CharsetUtil.UTF_8) {
    //         return isUtf8(buf, index, length);
    //     } else if (charset == CharsetUtil.US_ASCII) {
    //         return isAscii(buf, index, length);
    //     } else {
    //         CharsetDecoder decoder = CharsetUtil.decoder(charset, CodingErrorAction.REPORT, CodingErrorAction.REPORT);
    //         try {
    //             if (buf.nioBufferCount() == 1) {
    //                 decoder.decode(buf.nioBuffer(index, length));
    //             } else {
    //                 ByteBuf heapBuffer = buf.alloc().heapBuffer(length);
    //                 try {
    //                     heapBuffer.writeBytes(buf, index, length);
    //                     decoder.decode(heapBuffer.internalNioBuffer(heapBuffer.readerIndex(), length));
    //                 } finally {
    //                     heapBuffer.release();
    //                 }
    //             }
    //             return true;
    //         } catch (CharacterCodingException ignore) {
    //             return false;
    //         }
    //     }
    // }

    /**
     * Aborts on a byte which is not a valid ASCII character.
     */
    // private static final ByteProcessor FIND_NON_ASCII = new ByteProcessor() {
    //     override
    //     bool process(byte value) {
    //         return value >= 0;
    //     }
    // };

    /**
     * Returns {@code true} if the specified {@link ByteBuf} starting at {@code index} with {@code length} is valid
     * ASCII text, otherwise return {@code false}.
     *
     * @param buf    The given {@link ByteBuf}.
     * @param index  The start index of the specified buffer.
     * @param length The length of the specified buffer.
     */
    // private static bool isAscii(ByteBuf buf, int index, int length) {
    //     return buf.forEachByte(index, length, FIND_NON_ASCII) == -1;
    // }

    /**
     * Returns {@code true} if the specified {@link ByteBuf} starting at {@code index} with {@code length} is valid
     * UTF8 text, otherwise return {@code false}.
     *
     * @param buf The given {@link ByteBuf}.
     * @param index The start index of the specified buffer.
     * @param length The length of the specified buffer.
     *
     * @see
     * <a href=http://www.ietf.org/rfc/rfc3629.txt>UTF-8 Definition</a>
     *
     * <pre>
     * 1. Bytes format of UTF-8
     *
     * The table below summarizes the format of these different octet types.
     * The letter x indicates bits available for encoding bits of the character number.
     *
     * Char. number range  |        UTF-8 octet sequence
     *    (hexadecimal)    |              (binary)
     * --------------------+---------------------------------------------
     * 0000 0000-0000 007F | 0xxxxxxx
     * 0000 0080-0000 07FF | 110xxxxx 10xxxxxx
     * 0000 0800-0000 FFFF | 1110xxxx 10xxxxxx 10xxxxxx
     * 0001 0000-0010 FFFF | 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
     * </pre>
     *
     * <pre>
     * 2. Syntax of UTF-8 Byte Sequences
     *
     * UTF8-octets = *( UTF8-char )
     * UTF8-char   = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4
     * UTF8-1      = %x00-7F
     * UTF8-2      = %xC2-DF UTF8-tail
     * UTF8-3      = %xE0 %xA0-BF UTF8-tail /
     *               %xE1-EC 2( UTF8-tail ) /
     *               %xED %x80-9F UTF8-tail /
     *               %xEE-EF 2( UTF8-tail )
     * UTF8-4      = %xF0 %x90-BF 2( UTF8-tail ) /
     *               %xF1-F3 3( UTF8-tail ) /
     *               %xF4 %x80-8F 2( UTF8-tail )
     * UTF8-tail   = %x80-BF
     * </pre>
     */
    // private static bool isUtf8(ByteBuf buf, int index, int length) {
    //     int endIndex = index + length;
    //     while (index < endIndex) {
    //         byte b1 = buf.getByte(index++);
    //         byte b2, b3, b4;
    //         if ((b1 & 0x80) == 0) {
    //             // 1 byte
    //             continue;
    //         }
    //         if ((b1 & 0xE0) == 0xC0) {
    //             // 2 bytes
    //             //
    //             // Bit/Byte pattern
    //             // 110xxxxx    10xxxxxx
    //             // C2..DF      80..BF
    //             if (index >= endIndex) { // no enough bytes
    //                 return false;
    //             }
    //             b2 = buf.getByte(index++);
    //             if ((b2 & 0xC0) != 0x80) { // 2nd byte not starts with 10
    //                 return false;
    //             }
    //             if ((b1 & 0xFF) < 0xC2) { // out of lower bound
    //                 return false;
    //             }
    //         } else if ((b1 & 0xF0) == 0xE0) {
    //             // 3 bytes
    //             //
    //             // Bit/Byte pattern
    //             // 1110xxxx    10xxxxxx    10xxxxxx
    //             // E0          A0..BF      80..BF
    //             // E1..EC      80..BF      80..BF
    //             // ED          80..9F      80..BF
    //             // E1..EF      80..BF      80..BF
    //             if (index > endIndex - 2) { // no enough bytes
    //                 return false;
    //             }
    //             b2 = buf.getByte(index++);
    //             b3 = buf.getByte(index++);
    //             if ((b2 & 0xC0) != 0x80 || (b3 & 0xC0) != 0x80) { // 2nd or 3rd bytes not start with 10
    //                 return false;
    //             }
    //             if ((b1 & 0x0F) == 0x00 && (b2 & 0xFF) < 0xA0) { // out of lower bound
    //                 return false;
    //             }
    //             if ((b1 & 0x0F) == 0x0D && (b2 & 0xFF) > 0x9F) { // out of upper bound
    //                 return false;
    //             }
    //         } else if ((b1 & 0xF8) == 0xF0) {
    //             // 4 bytes
    //             //
    //             // Bit/Byte pattern
    //             // 11110xxx    10xxxxxx    10xxxxxx    10xxxxxx
    //             // F0          90..BF      80..BF      80..BF
    //             // F1..F3      80..BF      80..BF      80..BF
    //             // F4          80..8F      80..BF      80..BF
    //             if (index > endIndex - 3) { // no enough bytes
    //                 return false;
    //             }
    //             b2 = buf.getByte(index++);
    //             b3 = buf.getByte(index++);
    //             b4 = buf.getByte(index++);
    //             if ((b2 & 0xC0) != 0x80 || (b3 & 0xC0) != 0x80 || (b4 & 0xC0) != 0x80) {
    //                 // 2nd, 3rd or 4th bytes not start with 10
    //                 return false;
    //             }
    //             if ((b1 & 0xFF) > 0xF4 // b1 invalid
    //                     || (b1 & 0xFF) == 0xF0 && (b2 & 0xFF) < 0x90    // b2 out of lower bound
    //                     || (b1 & 0xFF) == 0xF4 && (b2 & 0xFF) > 0x8F) { // b2 out of upper bound
    //                 return false;
    //             }
    //         } else {
    //             return false;
    //         }
    //     }
    //     return true;
    // }

    /**
     * Read bytes from the given {@link ByteBuffer} into the given {@link OutputStream} using the {@code position} and
     * {@code length}. The position and limit of the given {@link ByteBuffer} may be adjusted.
     */
    // static void readBytes(ByteBufAllocator allocator, ByteBuffer buffer, int position, int length, OutputStream out) {
    //     if (buffer.hasArray()) {
    //         out.write(buffer.array(), position + buffer.arrayOffset(), length);
    //     } else {
    //         int chunkLen = min(length, WRITE_CHUNK_SIZE);
    //         buffer.clear().position(position);

    //         if (length <= MAX_TL_ARRAY_LEN || !allocator.isDirectBufferPooled()) {
    //             getBytes(buffer, threadLocalTempArray(chunkLen), 0, chunkLen, out, length);
    //         } else {
    //             // if direct buffers are pooled chances are good that heap buffers are pooled as well.
    //             ByteBuf tmpBuf = allocator.heapBuffer(chunkLen);
    //             try {
    //                 byte[] tmp = tmpBuf.array();
    //                 int offset = tmpBuf.arrayOffset();
    //                 getBytes(buffer, tmp, offset, chunkLen, out, length);
    //             } finally {
    //                 tmpBuf.release();
    //             }
    //         }
    //     }
    // }

    private static void getBytes(ByteBuffer inBuffer, byte[] inBytes, 
            int inOffset, int inLen, OutputStream outStream, int outLen) {
        do {
            int len = min(inLen, outLen);
            inBuffer.get(inBytes, inOffset, len);
            outStream.write(inBytes, inOffset, len);
            outLen -= len;
        } while (outLen > 0);
    }

    private this() { }
}



/* Separate class so that the expensive static initialization is only done when needed */
final class HexUtil {

    private __gshared char[] BYTE2CHAR;
    private __gshared char[] HEXDUMP_TABLE;
    private __gshared string[] HEXPADDING;
    private __gshared string[] HEXDUMP_ROWPREFIXES;
    private __gshared string[] BYTE2HEX;
    private __gshared string[] BYTEPADDING;

    shared static this() {
        BYTE2CHAR = new char[256];
        HEXDUMP_TABLE = new char[256 * 4];
        HEXPADDING = new string[16];
        HEXDUMP_ROWPREFIXES = new string[65536 >>> 4];
        BYTE2HEX = new string[256];
        BYTEPADDING = new string[16];

        for (int i = 0; i < 256; i ++) {
            HEXDUMP_TABLE[ i << 1     ] = lowerHexDigits[i >>> 4 & 0x0F];
            HEXDUMP_TABLE[(i << 1) + 1] = lowerHexDigits[i       & 0x0F];
        }

        size_t i;

        // Generate the lookup table for hex dump paddings
        for (i = 0; i < HEXPADDING.length; i ++) {
            size_t padding = HEXPADDING.length - i;
            StringBuilder buf = new StringBuilder(padding * 3);
            for (size_t j = 0; j < padding; j ++) {
                buf.append("   ");
            }
            HEXPADDING[i] = buf.toString();
        }

        // Generate the lookup table for the start-offset header in each row (up to 64KiB).
        for (i = 0; i < HEXDUMP_ROWPREFIXES.length; i ++) {
            StringBuilder buf = new StringBuilder(12);
            buf.append(newline);
            buf.append(Long.toHexString(i << 4 & 0xFFFFFFFFL | 0x100000000L));
            buf.setCharAt(buf.length() - 9, '|');
            buf.append('|');
            HEXDUMP_ROWPREFIXES[i] = buf.toString();
        }

        // Generate the lookup table for byte-to-hex-dump conversion
        for (i = 0; i < BYTE2HEX.length; i ++) {
            BYTE2HEX[i] = " " ~ format("%02x", i); // StringUtil.byteToHexStringPadded(i);
        }

        // Generate the lookup table for byte dump paddings
        for (i = 0; i < BYTEPADDING.length; i ++) {
            size_t padding = BYTEPADDING.length - i;
            StringBuilder buf = new StringBuilder(padding);
            for (size_t j = 0; j < padding; j ++) {
                buf.append(' ');
            }
            BYTEPADDING[i] = buf.toString();
        }

        // Generate the lookup table for byte-to-char conversion
        for (i = 0; i < BYTE2CHAR.length; i ++) {
            if (i <= 0x1f || i >= 0x7f) {
                BYTE2CHAR[i] = '.';
            } else {
                BYTE2CHAR[i] = cast(char) i;
            }
        }
    }

    private static string hexDump(ByteBuf buffer, size_t fromIndex, size_t length) {
        checkPositiveOrZero(cast(int)length, "length");
        if (length == 0) {
          return "";
        }

        size_t endIndex = fromIndex + length;
        char[] buf = new char[length << 1];

        size_t srcIdx = fromIndex;
        size_t dstIdx = 0;
        for (; srcIdx < endIndex; srcIdx++, dstIdx += 2) {
        //   System.arraycopy(
        //       HEXDUMP_TABLE, buffer.getUnsignedByte(srcIdx) << 1,
        //       buf, dstIdx, 2);
            size_t srcPos = buffer.getUnsignedByte(cast(int)srcIdx) << 1;
            buf[dstIdx .. dstIdx+2] = HEXDUMP_TABLE[srcPos .. srcPos + 2];
        }

        return cast(string)(buf);
    }

    private static string hexDump(byte[] array, size_t fromIndex, size_t length) {
        checkPositiveOrZero(cast(int)length, "length");
        if (length == 0) {
            return "";
        }

        size_t endIndex = fromIndex + length;
        char[] buf = new char[length << 1];

        size_t srcIdx = fromIndex;
        size_t dstIdx = 0;
        for (; srcIdx < endIndex; srcIdx ++, dstIdx += 2) {
            // System.arraycopy(
            //     HEXDUMP_TABLE, (array[srcIdx] & 0xFF) << 1,
            //     buf, dstIdx, 2);
            
            size_t srcPos = (array[srcIdx] & 0xFF) << 1;
            buf[dstIdx .. dstIdx+2] = HEXDUMP_TABLE[srcPos .. srcPos + 2];
        }

        return cast(string)(buf);
    }

    private static string prettyHexDump(ByteBuf buffer, size_t offset, size_t length) {
        if (length == 0) {
          return "";
        } else {
            size_t rows = length / 16 + ((length & 15) == 0? 0 : 1) + 4;
            StringBuilder buf = new StringBuilder(rows * 80);
            appendPrettyHexDump(buf, buffer, offset, length);
            return buf.toString();
        }
    }

    private static void appendPrettyHexDump(StringBuilder dump, ByteBuf buf, size_t offset, size_t length) {
        if (isOutOfBounds(cast(int)offset, cast(int)length, buf.capacity())) {
            string msg = format("expected: " ~ "0 <= offset(%d) <= offset + length(%d) <= buf.capacity(%d)", 
                    offset, length, buf.capacity());
            throw new IndexOutOfBoundsException(msg);
        }
        if (length == 0) {
            return;
        }
        dump.append(
                          "         +-------------------------------------------------+" ~
                newline ~ "         |  0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f |" ~
                newline ~ "+--------+-------------------------------------------------+----------------+");

        size_t startIndex = offset;
        size_t fullRows = length >>> 4;
        size_t remainder = length & 0xF;

        // Dump the rows which have 16 bytes.
        for (size_t row = 0; row < fullRows; row ++) {
            size_t rowStartIndex = (row << 4) + startIndex;

            // Per-row prefix.
            appendHexDumpRowPrefix(dump, row, rowStartIndex);

            // Hex dump
            size_t rowEndIndex = rowStartIndex + 16;
            for (size_t j = rowStartIndex; j < rowEndIndex; j ++) {
                dump.append(BYTE2HEX[buf.getUnsignedByte(cast(int)j)]);
            }
            dump.append(" |");

            // ASCII dump
            for (size_t j = rowStartIndex; j < rowEndIndex; j ++) {
                dump.append(BYTE2CHAR[buf.getUnsignedByte(cast(int)j)]);
            }
            dump.append('|');
        }

        // Dump the last row which has less than 16 bytes.
        if (remainder != 0) {
            size_t rowStartIndex = (fullRows << 4) + startIndex;
            appendHexDumpRowPrefix(dump, fullRows, rowStartIndex);

            // Hex dump
            size_t rowEndIndex = rowStartIndex + remainder;
            for (size_t j = rowStartIndex; j < rowEndIndex; j ++) {
                dump.append(BYTE2HEX[buf.getUnsignedByte(cast(int)j)]);
            }
            dump.append(HEXPADDING[remainder]);
            dump.append(" |");

            // Ascii dump
            for (size_t j = rowStartIndex; j < rowEndIndex; j ++) {
                dump.append(BYTE2CHAR[buf.getUnsignedByte(cast(int)j)]);
            }
            dump.append(BYTEPADDING[remainder]);
            dump.append('|');
        }

        dump.append(newline ~
                    "+--------+-------------------------------------------------+----------------+");
    }

    private static void appendHexDumpRowPrefix(StringBuilder dump, size_t row, size_t rowStartIndex) {
        if (row < HEXDUMP_ROWPREFIXES.length) {
            dump.append(HEXDUMP_ROWPREFIXES[row]);
        } else {
            dump.append(newline);
            dump.append(Long.toHexString(rowStartIndex & 0xFFFFFFFFL | 0x100000000L));
            dump.setCharAt(dump.length() - 9, '|');
            dump.append('|');
        }
    }
}
