/* @brief Convert the Abstract Syntax Tree generated by mpc for the DBC file
 * into an equivalent XML file.
 * @copyright Richard James Howe (2018)
 * @license MIT *
 */
#include "2xml.h"
#include "util.h"
#include <assert.h>
#include <time.h>

/*
Add:
	<?xml-stylesheet type="text/xsl" href="yourxsl.xsl"?>


 */

static int print_escaped(FILE *o, const char *string)
{
	assert(o);
	assert(string);
	char c;
	int r = 0;
	while ((c = *(string)++)) {
		switch(c) {
		case '"':  r = fputs("&quot;", o); break;
		case '\'': r = fputs("&apos;", o); break;
		case '<':  r = fputs("&lt;",   o); break;
		case '>':  r = fputs("&gt;",   o); break;
		case '&':  r = fputs("&amp;",  o); break;
		default:
			   r = fputc(c, o);
		}
		if (r < 0)
			return -1;
	}
	return 0;
}

static int indent(FILE *o, unsigned depth)
{
	assert(o);
	while (depth--)
		if (fputc('\t', o) != '\t')
			return -1;
	return 0;
}

static int pnode(FILE *o, unsigned depth, const char *node, const char *fmt, ...)
{
	assert(o);
	assert(node);
	assert(fmt);
	va_list args;
	assert(o && node && fmt);
	errno = 0;
	if (indent(o, depth) < 0)
		goto warn;
	if (fprintf(o, "<%s>", node) < 0)
		goto warn;
	assert(fmt);
	va_start(args, fmt);
	int r = vfprintf(o, fmt, args);
	va_end(args);
	if (r < 0)
		goto warn;
	if (fprintf(o, "</%s>\n", node) < 0)
		goto warn;
	return 0;
warn:
	warning("XML node generation, problem writing to FILE* <%p>: %s", o, emsg());
	return -1;
}

static int comment(FILE *o, unsigned depth, const char *fmt, ...)
{
	assert(o);
	assert(fmt);
	va_list args;
	assert(o && fmt);
	errno = 0;
	if (indent(o, depth) < 0)
		goto warn;
	if (fputs("<!-- ", o) < 0)
		goto warn;
	assert(fmt);
	va_start(args, fmt);
	int r = vfprintf(o, fmt, args);
	va_end(args);
	if (r < 0)
		goto warn;
	if (fputs(" -->\n", o) < 0)
		goto warn;
	return 0;
warn:
	warning("XML comment generation, problem writing to FILE* <%p>: %s", o, emsg());
	return -1;
}

static int signal2xml(signal_t *sig, FILE *o, unsigned depth)
{
	assert(sig);
	assert(o);
	indent(o, depth);
	fprintf(o, "<signal>\n");
	pnode(o, depth+1, "name",      "%s", sig->name);
	pnode(o, depth+1, "startbit",  "%u", sig->start_bit);
	pnode(o, depth+1, "bitlength", "%u", sig->bit_length);
	pnode(o, depth+1, "endianess", "%s", sig->endianess == endianess_motorola_e ? "motorola" : "intel");
	pnode(o, depth+1, "scaling",   "%g", sig->scaling);
	pnode(o, depth+1, "offset",    "%g", sig->offset);
	pnode(o, depth+1, "minimum",   "%g", sig->minimum);
	pnode(o, depth+1, "maximum",   "%g", sig->maximum);
	pnode(o, depth+1, "signed",    "%s", sig->is_signed ? "true" : "false");
	pnode(o, depth+1, "floating",  "%u", sig->is_floating ? sig->sigval : 0);

	indent(o, depth+1);
	fprintf(o, "<units>");
	/*indent(o, depth+2);*/
	print_escaped(o, sig->units);
	/*indent(o, depth+1);*/
	fprintf(o, "</units>\n");

	indent(o, depth);
	if (fprintf(o, "</signal>\n") < 0)
		return -1;
	return 0;
}

static int msg2xml(can_msg_t *msg, FILE *o, unsigned depth)
{
	assert(msg);
	assert(o);
	indent(o, depth);
	fprintf(o, "<message>\n");
	pnode(o, depth+1, "name", "%s", msg->name);
	pnode(o, depth+1, "id",   "%u", msg->id);
	pnode(o, depth+1, "dlc",  "%u", msg->dlc);
	pnode(o, depth+1, "extended",  "%d", (int)msg->is_extended);

	signal_t *multiplexor = NULL;
	for (size_t i = 0; i < msg->signal_count; i++) {
		signal_t *sig = msg->sigs[i];
		if (sig->is_multiplexor) {
			if (multiplexor) {
				error("multiple multiplexor values detected (only one per CAN msg is allowed) for %s", msg->name);
				return -1;
			}
			multiplexor = sig;
			continue;
		}
		if (sig->is_multiplexed)
			continue;
		if (signal2xml(sig, o, depth+1) < 0)
			return -1;
	}

	if (multiplexor) {
		indent(o, depth+1);
		fprintf(o, "<multiplexor-group>\n");
		indent(o, depth+2);
		fprintf(o, "<multiplexor>\n");
		if (signal2xml(multiplexor, o, depth+2) < 0)
			return -1;
		indent(o, depth+2);
		fprintf(o, "</multiplexor>\n");

		for (size_t i = 0; i < msg->signal_count; i++) {
			signal_t *sig = msg->sigs[i];
			if (!(sig->is_multiplexed))
				continue;
			indent(o, depth+2);
			fprintf(o, "<multiplexed>\n");
			pnode(o, depth+3, "multiplexed-on", "%u",  sig->switchval);
			if (signal2xml(sig, o, depth+3) < 0)
				return -1;
			indent(o, depth+2);
			fprintf(o, "</multiplexed>\n");
		}
		indent(o, depth+2);
		fprintf(o, "</multiplexor-group>\n");
	}

	indent(o, depth);
	if (fprintf(o, "</message>\n") < 0)
		return -1;
	return 0;
}

int dbc2xml(dbc_t *dbc, FILE *output, bool use_time_stamps)
{
	assert(dbc);
	assert(output);
	time_t rawtime = time(NULL);
	struct tm *timeinfo = localtime(&rawtime);

	fprintf(output, "<?xml version=\"1.0\"?>\n");
	fprintf(output, "<?xml-stylesheet type=\"text/xsl\" href=\"%s\"?>\n",
		"https://raw.githubusercontent.com/howerj/dbcc/master/dbcc.xslt");

	comment(output, 0, "Generated by dbcc (see https://github.com/howerj/dbcc)");
	if (use_time_stamps)
		comment(output, 0, "Generated on: %s", asctime(timeinfo));

	fprintf(output, "<candb>\n");
	for (size_t i = 0; i < dbc->message_count; i++)
		if (msg2xml(dbc->messages[i], output, 1) < 0)
			return -1;
	if (fprintf(output, "</candb>\n") < 0)
		return -1;
	return 0;
}

