
Dealing with loops usually comes down to two scenarios:

1. Aggregation. Accumulating data over multiple iterations and handling it after the loop. String reversal is just one example of this.
2. Executing a loop until you get the required result and then ending it. For example, if the job of the loop is to find prime numbers. Remember that a prime number is a number that can only be divided by itself and one.

Let's look at a simple algorithm to check prime numbers. We'll divide the number,`x`  by all the numbers in the range from two to `x - 1` and see the remainder. If we don't find a divisor `x` can be divided by without a remainder in this range, then we're looking at a prime number.

If you think about it, it's actually enough to check numbers not up tov `x - 1`, but up to half the given number. For example, 11 isn't divisible by 2, 3, 4, 5. But it's also guaranteed that it can't be divided by numbers greater than its half. So, we can do a little optimization and check the division only up to `x / 2`.

```php
<?php

function isPrime($number)
{
    if ($number < 2) {
        return false;
    }

    $divisor = 2;

    while ($divisor <= $number / 2) {
        if ($number % $divisor === 0) {
            return false;
        }

        $divisor += 1;
    }

    return true;
}

isPrime(1); // false
isPrime(2); // true
isPrime(3); // true
isPrime(4); // false
```

The algorithm is built like so: if during the successive division by numbers up to `x / 2` there is at least one result without a remainder, then the given argument is not a prime number, and therefore further calculations will be pointless. At this point, it returns `false`.

And only if the entire loop is completed can we say that the number is prime, since no number by which it can be divided without a remainder can be found.
