Многие языки в дополнение к условной конструкции *if* включают в себя **switch**. В этом уроке мы познакомимся с ним.

Это специализированная версия *if*, созданная для некоторых особых ситуаций. Например, ее имеет смысл использовать там, где есть цепочка *if else* с проверками на равенство. Например:

```php
<?php

if ($status === 'processing') {
    // Делаем раз
} elseif ($status === 'paid') {
    // Делаем два
} elseif ($status === 'new') {
    // Делаем три
} else {
    // Делаем четыре
}
```

Эта составная проверка обладает одной отличительной чертой, каждая ветка здесь, это проверка значения переменной `status`. Switch позволяет записать этот код короче и выразительнее:

```php
<?php

switch ($status) {
    case 'processing': // status == processing
        // Делаем раз
        break;
    case 'paid': // status == paid
        // Делаем два
        break;
    case 'new': // status == new
        // Делаем три
        break;
    default: // else
        // Делаем четыре
}
```

Свитч довольно сложная конструкция с точки зрения количества элементов из которых она состоит:

* Внешнее описание, в которое входит ключевое слово `switch`. Переменная, по значениям которой *switch* будет выбирать поведение. И фигурные скобки для вариантов выбора.
* Конструкции `case` и `default`, внутри которых описывается поведение для разных значений рассматриваемой переменной. Каждый `case` соответствует `if` в примере выше. `default` это особая ситуация, соответствующая ветке `else` в условных конструкциях. Как и `else`, указывать `default` не обязательно.
* `break` нужен для предотвращения "проваливания". Если его не указать, то после выполнения нужного `case` выполнение перейдет к следующему `case` и так либо до ближайшего `break`, либо до конца *switch*.

Фигурные скобки в *switch* не определяют блок кода как это было в других местах. Внутри допустим только тот синтаксис, который показан выше. То есть там можно использовать `case` или `default`. А вот внутри каждого `case` (и `default`) ситуация другая. Здесь можно выполнять любой произвольный код:

```php
<?php

switch ($count) {
    case 1:
        // Делаем что-то полезное
        break;
    case 2:
        // Делаем что-то полезное
        break;
    default:
        // Что-то делаем
}
```

Иногда результат, полученный внутри `case` — это конец выполнения функции, содержащей *switch*. В таком случае его нужно как-то вернуть наружу. Для решения этой задачи есть два способа.

**Первый способ** — создать переменную перед *switch*, заполнить ее в *case* и затем, в конце, вернуть значение этой переменной наружу:

```php
<?php

function doSomethingGood($count)
{
    // Заполняем
    switch ($count) {
        case 1:
            $result = 'one';
            break;
        case 2:
            $result = 'two';
            break;
        default:
            $result = null;
    }

    // Возвращаем
    return $result;
}
```

**Второй способ** проще и короче. Вместо создания переменной, *case* позволяет внутри себя делать обычный возврат из функции. После `return` никакой код не выполняется, поэтому мы можем избавиться от `break`:

```php
<?php

function doSomethingGood($count)
{
    switch ($count) {
        case 1:
            return 'one';
        case 2:
            return 'two';
        default:
            return null;
    }
}
```

Switch хоть и встречается в коде, но технически всегда можно обойтись без него. Ключевая польза при его использовании в том, что он лучше выражает намерение программиста, когда нужно проверять конкретные значения переменной. Хотя кода и стало физически больше, читать его легче, чем блоки с *elseif*.
