Логические выражения могут объединяться друг с другом, создавая все более хитрые проверки.

Хороший пример — проверка пароля. Как вы знаете, некоторые сайты при регистрации хотят пароль от 8 до 20 символов в длину.

В математике мы бы написали `8 >= x <= 20`, где `x` — это длина конкретного пароля. Но в PHP такой трюк не пройдет.

Нам придется сделать два отдельных логических выражения и соединить их специальным оператором «И»:

```php
<?php

function hasSpecialChars($str)
{
    // Проверяет содержание специальных символов в строке
}

// Функция принимает пароль и проверяет, соответствует ли он условиям
function isCorrectPassword($password)
{
    $length = strlen($password);
    // Скобки задают приоритет, чтобы показать, что к чему относится
    return ($length >= 8 && $length <= 20) && hasSpecialChars($password);
}

isCorrectPassword('qwerty'); // false
isCorrectPassword('qwerty1234'); // true
isCorrectPassword('zxcvbnmasdfghjkqwertyui'); // false
```

Оператор `&&` означает «И» — в математической логике это называют **конъюнкцией**. Все выражение считается истинным только в том случае, когда истинен каждый операнд — каждое из составных выражений. Иными словами, `&&` означает «и то, и другое». Приоритет этого оператора ниже, чем приоритет операторов сравнения, поэтому выражение отрабатывает правильно без скобок.

Кроме `&&`, часто используется оператор `||` — «ИЛИ» (**дизъюнкция**). Он означает «или то, или другое, или оба». Операторы можно комбинировать в любом количестве и любой последовательности, но когда одновременно встречаются `&&` и `||`, то приоритет лучше задавать скобками.

Область математики, в которой изучаются логические операторы, называется булевой алгеброй. Ниже увидите **таблицы истинности** — по ним можно определить, каким будет результат, если применить оператора:

### И `&&`

| A     | B     | A `&&` B |
| ----- | ----- | -------- |
| TRUE  | TRUE  | **TRUE** |
| TRUE  | FALSE | FALSE    |
| FALSE | TRUE  | FALSE    |
| FALSE | FALSE | FALSE    |

### ИЛИ `||`

| A     | B     | A `‖` B  |
| ----- | ----- | -------- |
| TRUE  | TRUE  | **TRUE** |
| TRUE  | FALSE | **TRUE** |
| FALSE | TRUE  | **TRUE** |
| FALSE | FALSE | FALSE    |
