//! JVM methods.

use bitflags::bitflags;

use super::{references::MethodRef, Method};

/// A generic type signature for a method.
pub type Signature = String;

impl Method {
    /// The method of a static initializer block.
    pub const CLASS_INITIALIZER_NAME: &'static str = "<clinit>";
    /// The method of a constructor.
    pub const CONSTRUCTOR_NAME: &'static str = "<init>";

    /// Checks if the method is a constructor.
    #[must_use]
    pub fn is_constructor(&self) -> bool {
        self.name == Self::CONSTRUCTOR_NAME
    }

    /// Checks if the method is a static initializer block.
    #[must_use]
    pub fn is_static_initializer_block(&self) -> bool {
        self.name == Self::CLASS_INITIALIZER_NAME
    }

    /// Creates a [`MethodRef`] pointting to this method.
    #[must_use]
    pub fn as_ref(&self) -> MethodRef {
        MethodRef {
            owner: self.owner.clone(),
            name: self.name.clone(),
            descriptor: self.descriptor.clone(),
        }
    }
}

/// The information of a method parameter.
#[derive(Debug, Clone)]
pub struct ParameterInfo {
    /// The name of the parameter.
    pub name: Option<String>,
    /// The access flags of the parameter.
    pub access_flags: ParameterAccessFlags,
}

bitflags! {
    /// Access flags for a [`Method`].
    #[derive(Debug, PartialEq, Eq, Clone, Copy)]
    pub struct AccessFlags: u16 {
        /// Declared `public`; may be accessed from outside its package.
        const PUBLIC = 0x0001;
        /// Declared `private`; accessible only within the defining class and other classes belonging to the same nest.
        const PRIVATE = 0x0002;
        /// Declared `protected`; may be accessed within subclasses.
        const PROTECTED = 0x0004;
        /// Declared `static`.
        const STATIC = 0x0008;
        /// Declared `final`; must not be overridden.
        const FINAL = 0x0010;
        /// Declared `synchronized`; invocation is wrapped by a monitor use.
        const SYNCHRONIZED = 0x0020;
        /// A bridge method, generated by the compiler.
        const BRIDGE = 0x0040;
        /// Declared with variable number of arguments.
        const VARARGS = 0x0080;
        /// Declared `native`; implemented in a language other than Java.
        const NATIVE = 0x0100;
        /// Declared `abstract`; no implementation is provided.
        const ABSTRACT = 0x0400;
        /// In a `class` file whose major version is at least 46 and at most 60; Declared `strictfp`.
        const STRICT = 0x0800;
        /// Declared synthetic; not present in the source code.
        const SYNTHETIC = 0x1000;
    }
}

bitflags! {
    /// The access flags for a method parameter.
    #[derive(Debug, PartialEq, Eq, Clone, Copy)]
    pub struct ParameterAccessFlags: u16 {
        /// Declared `final`; may not be assigned to after initialization.
        const FINAL = 0x0010;
        /// Declared synthetic; not present in the source code.
        const SYNTHETIC = 0x1000;
        /// Declared as either `mandated` or `optional`.
        const MANDATED = 0x8000;
    }
}

#[cfg(test)]
mod tests {
    use crate::{jvm::references::ClassRef, tests::arb_identifier};

    use super::*;
    use proptest::prelude::*;

    fn empty_method(name: String) -> Method {
        Method {
            name,
            access_flags: AccessFlags::PUBLIC,
            descriptor: "()V".parse().unwrap(),
            owner: ClassRef::new("org/mokapot/Test"),
            body: None,
            exceptions: vec![],
            runtime_visible_annotations: vec![],
            runtime_invisible_annotations: vec![],
            runtime_visible_type_annotations: vec![],
            runtime_invisible_type_annotations: vec![],
            runtime_visible_parameter_annotations: vec![],
            runtime_invisible_parameter_annotations: vec![],
            annotation_default: None,
            parameters: vec![],
            is_synthetic: false,
            is_deprecated: false,
            signature: None,
            free_attributes: vec![],
        }
    }

    proptest! {
        #[test]
        fn not_a_constructor(name in arb_identifier()) {
            prop_assume!(name != Method::CONSTRUCTOR_NAME);
            let method = empty_method(name);
            assert!(!method.is_constructor());
        }

        #[test]
        fn not_a_static_init(name in arb_identifier()) {
            prop_assume!(name != Method::CLASS_INITIALIZER_NAME);
            let method = empty_method(name);
            assert!(!method.is_static_initializer_block());
        }
    }

    fn arb_access_flag() -> impl Strategy<Value = AccessFlags> {
        prop_oneof![
            Just(AccessFlags::PUBLIC),
            Just(AccessFlags::PRIVATE),
            Just(AccessFlags::PROTECTED),
            Just(AccessFlags::STATIC),
            Just(AccessFlags::FINAL),
            Just(AccessFlags::SYNCHRONIZED),
            Just(AccessFlags::BRIDGE),
            Just(AccessFlags::VARARGS),
            Just(AccessFlags::NATIVE),
            Just(AccessFlags::ABSTRACT),
            Just(AccessFlags::STRICT),
            Just(AccessFlags::SYNTHETIC)
        ]
    }

    fn arb_parameter_access_flag() -> impl Strategy<Value = ParameterAccessFlags> {
        prop_oneof![
            Just(ParameterAccessFlags::FINAL),
            Just(ParameterAccessFlags::SYNTHETIC),
            Just(ParameterAccessFlags::MANDATED)
        ]
    }

    proptest! {

        #[test]
        fn access_flags_bit_no_overlap(
            lhs in arb_access_flag(),
            rhs in arb_access_flag()
        ){
            prop_assume!(lhs != rhs);
            assert_eq!(lhs.bits() & rhs.bits(), 0);
        }

        #[test]
        fn parameter_access_flags_bit_no_overlap(
            lhs in arb_parameter_access_flag(),
            rhs in arb_parameter_access_flag()
        ){
            prop_assume!(lhs != rhs);
            assert_eq!(lhs.bits() & rhs.bits(), 0);
        }
    }
}
