# Copyright 2024 The Langfun Authors
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Video tests."""
import base64
import io
import unittest
from unittest import mock

from langfun.core.modalities import ms_office as ms_office_lib
from langfun.core.modalities import pdf as pdf_lib
import pyglove as pg


def sample_docx_bytes():
  import docx  # pylint: disable=g-import-not-at-top

  # Create a new Document
  doc = docx.Document()
  doc.add_heading('Sample Document', level=1)
  doc.add_paragraph('This is a sample paragraph.')

  # Save the document to a BytesIO object
  file_stream = io.BytesIO()
  doc.save(file_stream)
  file_stream.seek(0)  # Move the cursor to the beginning of the stream

  # Get the bytes of the document
  content_bytes = file_stream.read()
  return content_bytes


expected_docx_xml = """<w:document xmlns:wpc="http://schemas.microsoft.com/office/word/2010/wordprocessingCanvas" xmlns:mo="http://schemas.microsoft.com/office/mac/office/2008/main" xmlns:mc="http://schemas.openxmlformats.org/markup-compatibility/2006" xmlns:mv="urn:schemas-microsoft-com:mac:vml" xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships" xmlns:m="http://schemas.openxmlformats.org/officeDocument/2006/math" xmlns:v="urn:schemas-microsoft-com:vml" xmlns:wp14="http://schemas.microsoft.com/office/word/2010/wordprocessingDrawing" xmlns:wp="http://schemas.openxmlformats.org/drawingml/2006/wordprocessingDrawing" xmlns:w10="urn:schemas-microsoft-com:office:word" xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main" xmlns:w14="http://schemas.microsoft.com/office/word/2010/wordml" xmlns:wpg="http://schemas.microsoft.com/office/word/2010/wordprocessingGroup" xmlns:wpi="http://schemas.microsoft.com/office/word/2010/wordprocessingInk" xmlns:wne="http://schemas.microsoft.com/office/word/2006/wordml" xmlns:wps="http://schemas.microsoft.com/office/word/2010/wordprocessingShape" mc:Ignorable="w14 wp14">
  <w:body>
    <w:p>
      <w:pPr>
        <w:pStyle w:val="Heading1"/>
      </w:pPr>
      <w:r>
        <w:t>Sample Document</w:t>
      </w:r>
    </w:p>
    <w:p>
      <w:r>
        <w:t>This is a sample paragraph.</w:t>
      </w:r>
    </w:p>
    <w:sectPr w:rsidR="00FC693F" w:rsidRPr="0006063C" w:rsidSect="00034616">
      <w:pgSz w:w="12240" w:h="15840"/>
      <w:pgMar w:top="1440" w:right="1800" w:bottom="1440" w:left="1800" w:header="720" w:footer="720" w:gutter="0"/>
      <w:cols w:space="720"/>
      <w:docGrid w:linePitch="360"/>
    </w:sectPr>
  </w:body>
</w:document>
"""


def sample_xlsx_bytes():
  import pandas as pd  # pylint: disable=g-import-not-at-top

  # Create a sample DataFrame
  data = {'Column1': [1, 2, 3, 4], 'Column2': ['A', 'B', 'C', 'D']}
  df = pd.DataFrame(data)

  # Save the DataFrame to a BytesIO object
  file_stream = io.BytesIO()
  df.to_excel(file_stream, index=False, engine='openpyxl')
  file_stream.seek(0)  # Move the cursor to the beginning of the stream

  # Get the bytes of the Excel file
  content_bytes = file_stream.read()
  return content_bytes


expected_xlsx_html = """<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th>Column1</th>
      <th>Column2</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th>0</th>
      <td>1</td>
      <td>A</td>
    </tr>
    <tr>
      <th>1</th>
      <td>2</td>
      <td>B</td>
    </tr>
    <tr>
      <th>2</th>
      <td>3</td>
      <td>C</td>
    </tr>
    <tr>
      <th>3</th>
      <td>4</td>
      <td>D</td>
    </tr>
  </tbody>
</table>"""

docx_bytes = sample_docx_bytes()
xlsx_bytes = sample_xlsx_bytes()

pptx_bytes = (
    b"PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\xc6\xaf\xc4g\xb4\x01\x00\x00\xba\x0c\x00\x00\x13\x00\x00\x00[Content_Types].xml\xcd\x97\xc9N\xc30\x10\x86\xef<E\x94K\x0e\xa8q\xd9\x175\xe5\xc0rb\xa9\x04<\x80I\xa6\xad\xc1\xb1-\xcf\xb4\xd0\xb7g\x92.\xaa\xd8R\x96\n_\x12\xd93\xf3\xff\x9f'Q4\xe9\x9c\xbc\x94:\x1a\x83GeM\x96l\xa5\xed$\x02\x93\xdbB\x99A\x96\xdc\xdf]\xb4\x0e\x93\x08I\x9aBjk"
    b' K&\x80\xc9Iw\xa3s7q\x80\x11\x17\x1b\xcc\xe2!\x91;\x16\x02\xf3!\x94\x12S\xeb\xc0p\xa4o})\x89\x97~'
    b' \x9c\xcc\x9f\xe4\x00\xc4v\xbb\xbd/rk\x08\x0c\xb5\xa8\xd2\x88\xbb\x9d3\xe8\xcb\x91\xa6\xe8\xfc\x85\xb7k\x90\xf8A\x998:\x9d\xe6UVY,\x9d\xd3*\x97\xc4a16\xc5\x1b\x93\x96\xed\xf7U\x0e\x85\xcdG%\x97\xa4\xce\x03\xf2\xbdN/5/\x15K\xf9['
    b' \xe2\x83a,>4}t0x\xe3\xaa\xca\x8a\xba\x0e|\\\xe3A\xe3\xf7Hg\xadH\xb9\xb2\xce\xc1\xa1r\xb8\xc9\t\x9f8T\x91\xcf\rfu7\xfc\x08\xbd*'
    b' \xeaIO\xd7\xb2\xe4,\xc1\xcd\xe8y\xebPp~\xfa\xb5JsC\xa1\x02*\xa0h9\x96\x04O\n\x16\xcc_z\xe7\xd6\xc3\xf7\xcd\xe7=\xaa\xaaWtt\x8eD\xf5\xd4k\xdb_\x1f\xf7\xdd\xfb3\x13^\x85`^\xb7\x0e\x88\x85v)\x95i\x82A\xcd\x9b\x97rbG\x84\xcb\x8b\xad\xbf&[\xd2\xfe1S;D\xa8\x10;\xb5\x1d'
    b' \xd3N\x80L\xbb\x012\xed\x05\xc8\xb4\x1f'
    b' \xd3A\x80L\x87\x012\x1d\xfd7\xd3\x95D\x9e\xabpy\xb1\x9eo\xe6T{%\xa6\x19\xcdz8\x9a\x08H>h\xb8\xa5\x89\x86?\x1fB\x96\xa4\x1b)x\x10\x87\xe9\xf5\xf7m\xa8e\x9a\x1c\xc7\n\x9e\xd72z-\x84\xe7\x04\xa2\xfe\xf5\xe8\xbe\x02PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\xf1\r7\xec\x00\x01\x00\x00\xe1\x02\x00\x00\x0b\x00\x00\x00_rels/.rels\xad\x92\xcfN\x03!\x10\x87\xef>\x05\xd9\x0b\xa7.\xdbj\x8c1e{1&\xbd\x19S\x1f`\x84\xe9.u\x81\tLM\xfb\xf6\xa2\x89\x7fj\xb6M\x0f=\xc2\xfc\xf8\xe6\x1b`\xbe\xd8\xf9A\xbcc\xca.\x06-\xa7u#\x05\x06\x13\xad\x0b\x9d\x96/\xab\xc7\xc9\x9d\x14\x99!X\x18b@-\xf7\x98\xe5\xa2\xbd\x9a?\xe3\x00\\\xce\xe4\xdeQ\x16\x05\x12\xb2\xaezf\xbaW*\x9b\x1e=\xe4:\x12\x86RY\xc7\xe4\x81\xcb2u\x8a\xc0\xbcA\x87j\xd64\xb7*\xfdeT\xed\x01S,\xad\xae\xd2\xd2N+\xb1\xda\x13\x9e\xc3\x8e\xeb\xb53\xf8\x10\xcd\xd6c\xe0\x91\x16\xff\x12\x85\x0c\xa9C\xd6\x15\x11+J\x98\xcb\xe6W\xba.\xe4J\x8d\x0b\xcd\xce\x17:>\xac\xf2\xc8`\x81Aq\xbf\xf5\xaf\x01\xdc\xf0kc\xa3yJ\xb1\x84~j\xf5\x86\xb0;&t}Y!\x13\x13N\xa8\xf4\xc7\xc4\x0e\xf3\x88\xd6g\xe2\xd4\r\xdd\\\xf2\xc9p\xc7\x18,\xda\xd3J@\xf4m\xa4\x0e~f\xfb\x01PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x8b\x14\xfc\xe3y\x01\x00\x00\xdb\x02\x00\x00\x11\x00\x00\x00docProps/core.xml\x8d\x92\xcdN\xc30\x10\x84\xef<E\xd4KN\xa9\xe3\x16J\x89\x92'
    b' \x01\xe2\x04\x12R\x8b@\xdc\x8c\xbdM\r\x89m\xd9\xdb\xa6y{\x9c\xa4M\xf9\xe9\x81[V3\xfbi<\x9b\xf4zW\x95\xc1\x16\xac\x93Ze!\x1d\xc7a\x00\x8ak!U\x91\x85\xcf\xcb\xfbh\x1e\x06\x0e\x99\x12\xac\xd4\n\xb2\xb0\x01\x17^\xe7g)7\t\xd7\x16\x9e\xac6`Q\x82\x0b<H\xb9\x84\x9bl\xb4F4\t!\x8e\xaf\xa1bn\xec\x1d\xca\x8b+m+\x86~\xb4\x051\x8c\x7f\xb2\x02\xc8$\x8eg\xa4\x02d\x82!#-02\x03q\xb4G\n>'
    b' \xcd\xc6\x96\x1d@p\x02%T\xa0\xd0\x11:\xa6\xe4\xe8E\xb0\x95;\xb9\xd0)\xdf\x9c\x95\xc4\xc6\xc0I\xebA\x1c\xdc;\'\x07c]\xd7\xe3z\xdaY}~J^\x1f\x1f\x16\xddS#\xa9\xda\xaa8\x8c\xf2T\xf0\x04%\x96@\xbaO\xb7y\xff\x00\x8e\xfd\xc0-0\xd4\xd6\x0f\xbe\xc4Ohjm\x85\xeb%\x01\x8e[i\xd0\x1f#/@\x81e\x08"\xd88\x7f\x8d\xc04\xb8\xd6*2\x06w)\xf9\xe5mI%s\xf8\xe8\x0f\xb7\x92'
    b" n\x9a|\x81\xb0\x85\xe0\x96)\xd5\xa4\xe4\xaf\xdcnX\xd8\xca\xf6\xee9\xed\x1c\xc3\x98\xee[\xec\x93\xfa\x00\xfe\xf5I\xdf\xd5Ay\x99\xde\xde-\xefG\xf9$\xa6\xd3(\xa6\xd1\xe4r\x19_%\xf4<\xa1\xb3\xb76\xdd\x8f\xfd#\xb0\xda\x07\xf8?\xf1\"\xb9\x98\x7f#\x1e\x00]~\xee\xe1\x85\xb6\x8d\xef\x8e\xfc\xf9\x1f\xf3/PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x9e\xd0\x8ey\xef\x01\x00\x00m\x04\x00\x00\x10\x00\x00\x00docProps/app.xml\x9dT\xc1\x8e\xd30\x10\xbd#\xf1\x0f\x96Oph\x93B\x85P\xe5f\x05]\xadz\xa04R\xb3\xcby\xb0'\x8d\x85cG\xb6\xe9n\xf9z&\t\xc9\xa6P!AN\xef\xcd<\xbd\x19\xcf\xd8\x117O\xb5a'\xf4A;\xbb\xe6\x8by\xca\x19Z\xe9\x94\xb6\xc75\xbf/\xeef\xef9\x0b\x11\xac\x02\xe3,\xae\xf9\x19\x03\xbf\xc9^\xbe\x10\xb9w\r\xfa\xa810\xb2\xb0a\xcd\xab\x18\x9bU\x92\x04Ya\raNiK\x99\xd2\xf9\x1a\"Q\x7fL\\Yj\x89\xb7N~\xaf\xd1\xc6\xe4M\x9a\xbeK\xf0)\xa2U\xa8f\xcdh\xc8{\xc7\xd5)\xfe\xaf\xa9r\xb2\xed/<\x14\xe7\x86\xfc2Q\xb8\x08\xa6\xd05f\x0b\x91<\x13\xf1\xc5y\x15\xb2T$=\x10\x1f\x9a\xc6h\t\x91\xa6\x91\xed\xb4\xf4.\xb82\xb2\x1dHm\xa3\x0b\x15\xcb\xdd#\xfa\xdc\x11\x13\xc9TK\xe3\xc0@\xe5;v\xd7u\x97\xed\xed,H\x8fh\xd9\xa1r\x8f\xec\xd5r\xf5\xf6\xb5H\xae\x08E\x0e\x1e\x8e\x1e\x9a\xaake\xc2\xc4\xc1h\x85]\xf4\x17\x12\x9f]\xec\x03=\x10[\xad\x14\xdag\xdd\x05\x17\xbb\xdd\xc6\xe8\xa6K\x0cP\x1c$\x18\xdc\xd0x\xb2\x12L@\xb2\x1e\x03b\x8b\xd0\xae>\x07\xedIy\x8a\xab\x13\xca\xe8<\x0b\xfa\x07-\x7f\xc9\xd9W\x08\xd8\x0eu\xcdO\xe05\xd8\xc8{YO:l\x9a\x10}V\xd0\xc2\xc8{\xe4\x1d\x9c\xca\xa6X/\xdb\xbd\xf4\xe0\xaf\xc2\xde\xab;\x1d+t4\x18\xfe\xa1Dz\xbdD2\x1e\x93\xf0\xe5\x00\xfa\x12\xfb\x92V\x12\xaf\xccc1\x9dG\xd7\x03\x9ft\xb9\xef.&\xbb>\x88\xa1\xdeo\x15v`\xe1\x88mbD\x1bW7`\xcf\x14\x1a\xd1'm\xbf\x85\xfb\xa6p\xb7\x10q\xd8\xe2eP\x1c*\xf0\xa8\xe8Y\x8c[\x1e\x03bK\r{C\xfa\x8f\xd4}{\xe8K>\xd2\xb0\xa9\xc0\x1eQ\r\x16\x7f&\xda\x07\xf3\xd0\xff=\xb2\xc5r\x9e\xd2\xd7=\x8c!\xd6\xde\xf7\xe1Yg?\x01PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x05w\x9c\x0f;\x02\x00\x00\xb4\x0c\x00\x00\x14\x00\x00\x00ppt/presentation.xml\xed\x97\xdfn\xda0\x14\xc6\xef\xf7\x14\x96o\xb8\x98h\xfe\x10\x924\xc2TZ'\xa4I\x9d\x84\n}\x00\xd79@T\xc7\x89l\x87A\x9f~vpH`\x9a\xd4\x07\xc8\x9d\xeds\xbe\xef\x9c\xfclY\xce\xe2\xe9Trt\x04\xa9\x8aJ\x90I\xf0\xe0O\x10\x08V\xe5\x85\xd8\x93\xc9\xdbv5M'Hi*r\xca+\x01dr\x065yZ~[\xd4Y-A\x81\xd0T\x1b%2.Be\x94\xe0\x83\xd6u\xe6y\x8a\x1d\xa0\xa4\xea\xa1\xaaA\x98\xd8\xae\x92%\xd5f*\xf7^.\xe9\x1f\xe3^r/\xf4\xfd\xd8+i!\xb0\xd3\xcb\xaf\xe8\xab\xdd\xae`\xf0\xb3bMi\xca_L$\xf0\xb6\x0fu(j\xd5\xb9\xd5_q\x1b~\xc5mK\x8a\x1ea\xd3\xbc+\xd0\xabJhEp\x80\x11mt\xf5\\\x95V\xa4\xd6\x05\xd3\x8d\x19\x10\xec\xe3\xa5\xe1\xa1x\xfe\x9b*\r\xf2W\xfe\xa2\xf4\xdd\n*r\x82\xc3"
    b" J\xa2t\x16G)F2\xb3+&\x12`o\xb9\xf0\xfe#\xbf\x1d_L\xe6\xf1@\x9d\xf4\xeaa\xee\xe6\x13\xb1\x13\xc1\x8fA\x14\xf9\xbe\x8f\x11;\x13\x1c\xa7\xf3\xb4\x9d\xe8s\r\x04+&\x01Dt\x9aY\x87:\x13\x95\x06\xe5d\xd7L+\xeb<\xda\xac\x1cv\xb4\xe1z\x0b'\xbd\xd1g\x0e\xcb\x05\xb5k\xeb\xb5t\xa3\xd7\xb5D\x9c\x9a\xb3\x83AL\xdf6mw\xc3\x14~\xe4AmrJ*_,8D\xf9^\x10\xcc129[\xfa\xbe\xf9$8\x9a'\xa1\xad.5oS\x80\xbe\x88\x1f\xf2\xa3\xdd\x00\xbb\xcd\xc2MM\xe8`J\x99\xb3\xb4n\x04\xd36>\xe8B\x19\xa7"
    b' \xb5>\x1f'
    b' M\x89\xc0z\xda\xb8\xaax\x91\xaf\n\xce\xdb\x89=\x19\xf0\xcc%:RSM\x9f\x02\xd7\xf2MV[\xb5\xe5\xb6\xa3\xcc\xb0\xfb^\x8a)\xd76\x93f@\xef\x02@/\x01\xa6\xee\x02L\xf58^-\x0e\xef\xca\xc3\xa1\t{4\x1d\x84\x91O\xd8\xf3\x99\xf5|.\xc7r\xe4s\x81\xe2\xf8D=\x9f`\x96\x04\xf1\x08\xa8\xa3\xe2\x00\xcd\x07\x80\xd20MG@\x1d\x15\x07(\xee\x01\x85a\x1a\xfb#\xa0\x8e\x8a\x03\x94\x0c\x00%\xd1l\xbc\xa3\xafT\x1c\xa0\xb4\x07d\xe9\x8c\x97\xf4\x95\x8a\x03\xf48\x00\x14\xcf\x93\xf1\x92\xbeRi_\xb2\xff>1\xbd\xdb\x7f\x8d\xe5_PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdXR\x9cP\xc9\x1c\x01\x00\x00q\x04\x00\x00\x1f\x00\x00\x00ppt/_rels/presentation.xml.rels\xad\x94\xc1N\xc30\x0c\x86\xef<E\x94KN4\xed\x80\x81\xd0\xd2]\x10\xd2\x0eH\x88\x8d\x07\xc8Z\xb7\x8dH\x93(\x0e\x83\xbd=\x11L[[m\x15\x87\x1e\xfd\xdb\xfe\xfd\xc9\x8a\xb3X~\xb7\x9a\xec\xc0\xa3\xb2F\xb0,I\x19\x01S\xd8R\x99Z\xb0\xf7\xcd\xf3\xf5\x03#\x18\xa4)\xa5\xb6\x06\x04\xdb\x03\xb2e~\xb5x\x03-C\xec\xc1F9$\xd1\xc4\xa0\xa0M\x08\xee\x91s,\x1ah%&\xd6\x81\x89\x99\xca\xfaV\x86\x18\xfa\x9a;Y|\xc8\x1a\xf8,M\xe7\xdcw=h\xde\xf3$\xabRP\xbf*3J6{\x07\xff\xf1\xb6U\xa5\nx\xb2\xc5g\x0b&\x9c\x19\xc1Q\xab\x12^$\x06\xf0\xd1V\xfa\x1a\x82\xa0\x1d\xb1W\x91%\xd1\x9f\xf2\xf3X\xb3)\xb1\x9cW&\x0e\\C\x08q\xedxB\x1b$\x86\x85Y\xb2U\xe6\x12\xe1\xcd\xb4\x84\x80\xaf\xde\xba\x1e\xdbA\x1a[\xd3\xed\x94\x10;\x05_\x03\x88\xa34\x06q7%D\x88\xbdp\x02\xf8\r\xff\xc4\xd1\xf72\x9f\x94An5\xac\xc3^Cg\x15\x1dq\x0c\xe4~\xf2{\x1a\\\xd2A=n\x83\xf7~\x8a\xfc\x07PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\\\x9cG\x14D\x01\x00\x00\x89\x02\x00\x00\x11\x00\x00\x00ppt/presProps.xml\xb5\x92\xcbN\xc30\x10E\xf7H\xfcC\xe4\xbdk;I\xf3R\x93*i\x82\x84\xc4\x82\x05|\x80\x958\xad\xa5\xf8!\xdb}'
    b' \xc4\xbf\x13B\n\x146\xdd\xb0\x9b\xd1\xe8\xde9w4\xab\xf5I\x0c\xde\x81\x19\xcb\x95\xcc\x01Y`\xe01\xd9\xaa\x8e\xcbm\x0e\x9e\x9f\xee`\x02<\xeb\xa8\xec\xe8\xa0$\xcb\xc1\x0b\xb3`]\xdc\xde\xact\xa6\r\xb3L:\xeaF\xe9\xa3\xf1F#i3\x9a\x83\x9ds:C\xc8\xb6;&\xa8](\xcd\xe48\xeb\x95\x11\xd4\x8d\xad\xd9\xa2\xce\xd0\xe3\xb8@\x0c\xc8\xc78B\x82r\tf\xbd\xb9F\xaf\xfa\x9e\xb7\xacV\xed^\x8c\x00\x9f&\x86\r\x13\x89\xddqm\xcfn\xfa\x1a\xb7\x9f9.\x90\x8a1$;\xb9\x07\xeb\xe6\xca\xdb\x1b\x9e\x83\xd7&\x8e6M\x1a\x960\xc2\xc1\x06\x86$\xf4a\x956\x15\x8cj\x12\xc4\x18\x13\\\xfa\xf1\xdb\x87\x9a\x84Y\xc7mKMw/\xe8\x965\x1dw5u\xf4\x0cG\xc2?x\x82\xb7FY\xd5\xbbE\xab\xc4\x9c\x13iudF+>E%x\xbe\xd7\x81\x0e9\xc0\x00\x15+4\xc1]2\xd6\x01)q\xe4\x970N\x93\x12\x86\x81\x9f\xc2\xb2\xaakXUe\xb2\x8c"\x1f/\t\xfebd=\xdd\x0fnb\xac5\xff/<\xf4}M\xf4\xfb{\x8awPK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdXg3&\x8d\x9b\x01\x00\x00\x82\x03\x00\x00\x11\x00\x00\x00ppt/viewProps.xml\x8dS\xc1N\xe30\x10\xbd\xaf\xc4?X\xbe\x83\x93\x08B\x89\x9arApAZ\xa4\x86\xbd\x1bg\x9a\x1a9\xb6\xe5qK\xcb\xd7\xef$ni\x0b=p\x9b7\xe3y~o\xc6\x9e\xdeoz\xc3\xd6\x10P;[\xf3\xfc*\xe3\x0c\xacr\xad\xb6]\xcd_\x9b\xc7\xcb\tg\x18\xa5m\xa5q\x16j\xbe\x05\xe4\xf7\xb3\x8b?S_\xad5|\xbc\x04F\x04\x16+Y\xf3e\x8c\xbe\x12\x02\xd5\x12z\x89W\xce\x83\xa5\xda\xc2\x85^F\x82\xa1\x13m\x90\x1fD\xdc\x1bQdY)z\xa9-\xdf\xf5\x87\xdf\xf4\xbb\xc5B+xpj\xd5\x83\x8d\x89$\x80\x91\x91\xc4\xe3R{\xdc\xb3\xf9\xdf\xb0\xf9\x00H4c\xf7\xa9$#1\xfe#w5G\xd36\xcbU\xfff\xa56C\x86\xcf\xc8\xb8\x1dHF\xf8\x12\x06L<\xd1\x05h\x9fa\x11\x19~\xd2\x18o\xca"\xe3\xe2\xb8\xd68?\x96\xee\xae\xcbr,\x89\x9f<ht\x0b\x07\xa8\xe6\xa6M\x88\xa1\x95\xbeqOA\xb75\xa7\r%\xf8\xf7\xed\x1dTD\xbanT\xa5vg\xd72\xcc\x954\xb0\xcf\xe3\x00fSY\xe1\x86\r+.\xae9#\x9a<\x1bePz{&-\xbe\xfa|\xe5\x82\xee\xb4e\x9b\x9a_\xe67y\xc1\xd9v\x88(H\xe7\xd4Aq\xb7"\x03\xcf\x18\xbfbF\xbd4b\xda\x86\x0b\x9f\x9cyGj\x8b\xbc\xdc\xcd&\x1dI\xc9\xc9d\x7f\xef\x81D\x1c\xcf'
    b" i:\x9d\x90u\x11\xb0\x81M<\x1a\xda\xd18\xbf\x19'g\xe7\x8c\x9f\xa6\xcf\x1b\xcfF\xd3\xd9w\xc7\xe2\xac\x84\x8e\xd64\xf7R\xd1Kg\x8a\x9ao\xe91\x10\x81\xda\xee\xc3\xc4\x92\xbe\xcf\xec?PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x93\nmu!\x06\x00\x00\xe7\x1d\x00\x00\x14\x00\x00\x00ppt/theme/theme1.xml\xedYMo\xdb6\x18\xbe\x0f\xd8\x7f"
    b' toe\xd9V\xea\x04u\x8a\xd8\xb1\xdb\xadM\x1b$n\x87\x1ei\x89\x96\xd8P\xa2@\xd2I|\x1b\xda\xe3\x80\x01\xc3\xbaa\x97\x01\xbb\xed0l+\xd0\x02\xbbt\xbf&[\x87\xad\x03\xfa\x17\xf6\xea\xc32e\xd3\x89\xd3f[\x81\xd6\x07\x9b\xa4\x9e\xf7\xfb\x83\xa4|\xf5\xdaq\xc4\xd0!\x11\x92\xf2\xb8m9\x97k\x16"\xb1\xc7}\x1a\x07m\xeb\xee\xa0\x7f\xa9e!\xa9p\xecc\xc6c\xd2\xb6&DZ\xd76?\xfc\xe0*\xdeP!\x89\x08\x02\xfaXn\xe0\xb6\x15*\x95l\xd8\xb6\xf4`\x19\xcb\xcb<!1<\x1bq\x11a\x05S\x11\xd8\xbe\xc0G\xc07bv\xbdV[\xb3#Lc\x0b\xc58\x02\xb6wF#\xea\x114HYZ\x9bS\xe6=\x06_\xb1\x92\xe9\x82\xc7\xc4\xbe\x97I\xd4)2\xac\x7f\xe0\xa4?r"\xbbL\xa0C\xcc\xda\x16\xc8\xf1\xf9\xd1\x80\x1c+\x0b1,\x15<h[\xb5\xecc\xd9\x9bW\xed\x92\x88\xa9%\xb4\x1a]?\xfb\x14t\x05\x81\x7fP\xcf\xe8D0,\t\x9d~s\xfd\xcav\xc9\xbf\x9e\xf3_\xc4\xf5z\xbdn\xcf)\xf9e\x00\xecy`\xa9\xb3\x80m\xf6[Ng\xcaS\x03\xe5\xc3E\xde\xdd\x9a[kV\xf1\x1a\xff\xc6\x02~\xbd\xd3\xe9\xb8\xeb\x15|c\x86o.\xe0[\xb5\xb5\xe6V\xbd\x82o\xce\xf0\xee\xa2\xfe\x9d\xadnw\xad\x82wg\xf8\xb5\x05|\xff\xca\xfaZ\xb3\x8a\xcf@!\xa3\xf1\xc1\x02:\x8dg\x19\x99\x122\xe2\xec\x86\x11\xde\x02xk\x9a\x003\x94\xadeWN\x1f\xabe\xb9\x16\xe1\x07\\\xf4\x01\x90\x05\x17+\x1a#5I\xc8\x08{\x80\xebbF\x87\x82\xa6\x02\xf0\x06\xc1\xda\x93|\xc9\x93\x0bK\xa9,$=A\x13\xd5\xb6>N0T\xc4\x0c\xf2\xea\xf9\x8f\xaf\x9e?E\xaf\x9e?9y\xf8\xec\xe4\xe1/\'\x8f\x1e\x9d<\xfc\xd9@x\x03\xc7\x81N\xf8\xf2\xfb/\xfe\xfe\xf6S\xf4\xd7\xd3\xef^>\xfe\xca\x8c\x97:\xfe\xf7\x9f>\xfb\xed\xd7/\xcd@\xa5\x03_|\xfd\xe4\x8fgO^|\xf3\xf9\x9f?<6\xc0\xb7\x04\x1e\xea\xf0\x01\x8d\x88D\xb7\xc9\x11\xda\xe3\x11\xd8f\x10@\x86\xe2|\x14\x83\x10S\x9db+\x0e$\x8eqJc@\xf7TXA\xdf\x9e`\x86\r\xb8\x0e\xa9z\xf0\x9e\x80.`\x02^\x1f?\xa8(\xbc\x1f\x8a\xb1\xa2\x06\xe0\xcd0\xaa\x00w8g\x1d.\x8c6\xddLe\xe9^\x18\xc7\x81Y\xb8\x18\xeb\xb8=\x8c\x0fM\xb2\xbbs\xf1\xed\x8d\x13Hgjb\xd9\rIE\xcd]\x06!\xc7\x01\x89\x89B\xe93~@\x88\x81\xec>\xa5\x15\xbf\xeePOp\xc9G\n\xdd\xa7\xa8\x83\xa9\xd1%\x03:Tf\xa2\x1b4\x82\xb8LL\nB\xbc+\xbe\xd9\xb9\x87:\x9c\x99\xd8o\x93\xc3*\x12\xaa\x023\x13K\xc2*n\xbc\x8e\xc7\nGF\x8dq\xc4t\xe4-\xacB\x93\x92\xfb\x13\xe1U\x1c.\x15D:'
    b' \x8c\xa3\x9eO\xa44\xd1\xdc\x11\x93\x8a\xba7\xa1{\x98\xc3\xbe\xc3&Q\x15)\x14=0!oa\xceu\xe46?\xe8\x868J\x8c:\xd38\xd4\xb1\x1f\xc9\x03HQ\x8cv\xb92*\xc1\xab\x15\x92\xce!\x0e8^\x1a\xee{\x94\xa8\xf3\xd5\xf6]\x1a\x84\xe6\x04I\x9f\x8c\x85\xa9$\x08\xaf\xd6\xe3\x84\x8d0\x89\x8b&_i\xd7\x11\x8d\xdf\xf7\xee\x95{\xf7\x96\xa0\xc6\xe2\x99\xef\xd8\xcbp\xf3}\xba\xcb\x85O\xdf\xfe6\xbd\x8d\xc7\xf1.\x81\xcax\xdf\xa5\xdfw\xe9w\xb1K/\xab\xe7\x8b\xef\xcd\xb3vl\xeb\x87\xee\x8cM\xb4\xf4\x04>\xa2\x8c\xed\xab\t#\xb7d\xd6\xc8%\x98\xe7\xf7a1\x9bdD\xe5\x81?\taX\x88\xab\xe0\x02\x81\xb31\x12\\}BU\xb8\x1f\xe2\x04\xc48\x99\x84@\x16\xac\x03\x89\x12.\xe1\x9aa-\xe5\x9d\xddU)\xd8\x9c\xad\xb9\xd3\x0b&\xa0\xb1\xda\xe1~\xbe\xdc\xd0/\x9e%\x9bl\x16H]P#e\xb0\xaa\xb0\xc6\x957\x13\xe6\xe4\xc0\x15\xa59\xaeY\x9a{\xaa4[\xf3&\xd4\r\xc2\xe9k\x05g\xad\x9e\x8b\x86D\xc1\x8c\xf8\xa9\xdfs\x06\xd3\xb0\xfc\x8b!rjZ\x8cB\xec\x13\xc3\xb2f\x9f\xd3\xf8W\xbc\xe9\x9eK\x89\x8bqrm\xc1\xc9\xf6b5\xb1\xb8:CGmk\xdd\xad\xbb\x16\xf2p\xd2\xb6Fpl\x82a\x94\x00?\x99v\x1a\xcc\x82\xb8my*7\xf0\xecZ\x9c\xb3x\xdd\x9cUN\xcd]fpED"\xa4\xda\xc62\xcc\xa9\xb2G\xd3\xd7*\xf1L\xff\xba\xdbL\xfdp1\x06\x18\x9a\xc9jZ4Z\xce\xff\xa8\x85=\x1fZ2\x1a\x11O-Y\x99M\x8bg|\xac\x88\xd8\x0f\xfd#4dc\xb1\x87A\xeff\x9e]>\x95\xd0\xe9\xeb\xd3\x89\x80\xdcn\x16\x89W-\xdc\xa26\xe6_\xdf\x145\x83Y\x12\xe2"\xdb[Z\xecsx6.u\xc8f\x9az\xf6\x12\xdd_\xd3\x94\xc6\x05\x9a\xe2\xbe\xbb\xa6\xa4\x99\x0b\xe7\xd3\x86\x9f\xdd\x9e`\x17\x17\x18\xa59\xda\xb6\xb8P!\x87.\x94\x84\xd4\xeb\x0b\xd8\xf73Y\xa0\x17\x82\xb2HUB,}\x19\x9d\xeaJ\x0eg}+\xe7\x917\xb9'
    b' T{4@\x82B\xa7S\xa1'
    b' dW\x15v\x9e\xc1\xcc\xa9\xeb\xdb\xe3\x94Q\xd1gJue\x92\xff\x0e\xc9!a\x83\xb4z\xd7R\xfb-\x14N\xbbI\xe1\x88\x0c7\x1f4\xdbT]\xc3\xa0\xff\x16\x1f\\\x9a\xaf\xb5\xf1\xcc\x045\xcf\xb3\xf95\xb5\xa6\xafm\x05\xebo\xa6\xc2*\x1b\xb0&\xaen\xb6\xb8\xee.\xddy\xe6\xb7\xda\x04n\x19(\xfd\x82\xc6M\x85\xc7f\xc7\xd3\x01\xdf\x83\xe8\xa3r\x9fG\x90\x88\x97ZE\xf9\x95\x8bC\xd0\xb9\xa5\x19\x97\xb2\xfa\xafNA\xad%\xf1\xbe\xc8\xb3\xa3\xe6\xec\xc6\x12g\x9f.\xee\xf5\x9d\xed\x1a|\xed\x9e\xeej{\xb1Dm\xed\x1e\x92\xcd\x16\xfe\x94\xe2\xc3\x07'
    b' {\x1b\xae7c\x96\xaf\xc8\x04f\xf9`Wd\x06\x0f\xb9?)\x86L\xe6-!w\xc4\xb4\xa5\xb3x\x8f\x8c\x10\xf5\x8f\xa7a\x9d\xf3h\xf1\xafO\xb9\x99\xef\xe5\x02R\xdbK\xc2\xc6\xd9\x84\x05~\xb6\x89\x94\xc4\xf5\xb3\x89K\x8a\xe9\x1d\xaf$\xcenq&\x06l&9\xc7\xe7Q.[d\xe9)\x16\xbf\x89\xcbVP\xde\xec2c\xf6\xae\xea\xb2\x15\x02\xf5\x1a.S\xc7\xa7\xbb\xac\xf0\x94mJ<r\xac\x04\xeeN\xff\xc6\x82\xfc\xb5g)\xbb\xf9\x0fPK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\xd8\xfd\x8d\x8f\xa5\x00\x00\x00\xb6\x00\x00\x00\x13\x00\x00\x00ppt/tableStyles.xml\r\xccI\x0e\x820\x18@\xe1\xbd\x89wh\xfe}-CQ$\x14\xc2'
    b' +w\xea\x01*\x94!\xe9@h\xa3\x12\xe3\xdde\xf9\xf2\x92/\xcd?J\xa2\x97X\xecd4\x03\xff\xe0\x01\x12\xba5\xdd\xa4\x07\x06\x8f{\x83c@\xd6q\xddqi\xb4`\xb0\n\x0by\xb6\xdf\xa5<qOys\xab\x14W\xeb\xd0\xa6h\x9bp\x06\xa3ssB\x88mG\xa1\xb8=\x98Y\xe8\xed\xf5fQ\xdcm\xb9\x0c\xa4[\xf8{\xd3\x95$\x81\xe7\x1d\x89\xe2\x93\x06\xd4\x89\x9e\xc17\xaa\x82'
    b' \xa2\xb4\xc0\xa7\xcb\xe5\x88iH\x03\\z4\xc6qT\xd6\xd5\xb9\xa9\xfd*,~@\xb2?PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\xa6-\xa25\xee\x06\x00\x00\xd2.\x00\x00!\x00\x00\x00ppt/slideMasters/slideMaster1.xml\xedZ\xefn\xe36\x12\xff~O!\xe8>\xe4\xc3\xc1+\x89"\xf5\xc7X\xa7\x88\x9duo\x81t\x1b4\xe9\x03\xd0\x12m\xebBK:\x8aN\x93=\x14\xd8w\xe8\x1b\xf4-\xda\xfbv\x8f\xb2OrCJ\xb4d\xc7\x89\x13\xac\xd3\xae\xef\x0c,,j8\x1a\xce\xccofHN\xf6\xed7w\x0bn\xdd2QeE>8\xf1\xde\xb8\'\x16\xcb\x93"\xcd\xf2\xd9\xe0\xe4\xc7\xebq/:\xb1*I\xf3\x94\xf2"g\x83\x93{V\x9d|s\xfa\x97\xb7e\xbf\xe2\xe9w\xb4\x92LX'
    b' "\xaf\xfat`\xcf\xa5,\xfb\x8eS%s\xb6\xa0\xd5\x9b\xa2d9\xccM\x0b\xb1\xa0\x12^\xc5\xccI\x05\xfd\tD/\xb8\x83\\7p\x164\xcb\xed\xe6{\xf1\x9c\xef\x8b\xe94K\xd8y\x91,\x17,\x97\xb5\x10\xc18\x95\xa0~5\xcf\xca\xcaH+\x9f#\xad\x14\xac\x021\xfa\xeb5\x95N\xc1\xbe\xe4\x8a\xa7\xea9\x99\xd5\xbf?\xb0\xa9\x95\xa5w\x03\xdbs]\x0f8h_Kf#.\xac[\xca\x07\xf6d\xe6\xd9\xce\xe9[\xa7anF\xea\xe3\xaa\xbc\x16\x8c\xa9Q~\xfb\xad(\xaf\xcaK\xa1W\xf8p{)@&\x88\xb4\xad\x9c.\xd8\xc0V\x02\xf4D\xc3\xe6\xd4\x1f\xe9\x81\xb3\xf1\xf9\xcc\x0ci\xffn*\x16\xea\t\xee\xb1@C\xd7\xb6\xee\xd5\xaf\xa3h\xecNZIMLZj2\xff~\x0bo2\x7f\xb7\x85\xdb1\x0b8\x9dE\x95U\xb5r\x0f\xcdA\xc6\x9c\xebLrf]r\x9a\xb0y\xc1S\x88\x15oe\xa1\xd1\xbd*/\x8a\xe4\xa6\xb2\xf2\x02lS\xae\xa8M]q\xd4\xf6\xabg9\xb7\xe4}\tb\xa5\x12k\x1b\x97\xa8I\xa7\xabH\xb5\xdd+\x98\x84\x80\xb06\x17\x858\xf0\xa3u\xffD\x08\xc5\x81\xdb\xd8\xedy\xd8w\xddu\xebi\xbf\x14\x95\xfc\x96\x15\x0bK\r\x06\xb6`\x89\xd4\x81@o/*Y\xb3\x1a\x16\xadR\xd5($\xef\x86Ez\xaf8\'\xf0\x04\'A\xc2\xc1\xf7\xf3B|\xb4-\xfe>\xaf\x06v\xeca\x0ckK\xfd\xa25\xb5-\xd1\x9d\x99\xac\xcdH>*\xb8F\x89\xe6\t\xc8\x19\xd8\x89\x14Z\x97\x1c\xe2\xfbl)\x8bi\xd6hT/\xa9\xa6x%\xaf\xe4=g\xda\xecR\xfdh\xb2\x00\x858\x85|\xb7Y\xde\xfb\xf1\xca\xb6\xaa\x85\x1cqF\xf3UX\xc8\xd3\x11\xcf\x92\x1bK\x16\x16K3i5y\xafa\x80\xea\x00"\xd5BR/\xa7E\xb2<\xbd\xa4\x82\xfe\xb0!\xb9q\x91\xf6\x8d\xf1\x89c\x02\xe9\xf1p\xf2W\xe1\xa4\xb0\xeaF\x13\xdaG4)\x07\xd9Mj\x7fIPy\x10=\xc8u\x9f\x88*L\x10\x89\x03\xff\xeb\x8f\xaa\x17\x07R\xa9\x90\xbe\xe5\xab\x88\xf9\xc2\xc0R\xde\xd3qU\xad\x05\x96cV[[\xd2{\xe1\x92W,)\xf2\xd4\xe2\xec\x96\xf1g\x88G/\x14\x7f=\xcf\xc4\xf3\xa5\xfb/\x94>.\x96B\xce\x9f-\x1e\xbfT|6\xdd*}\xdf)\x8dMJ\x9fS\xb9\xbeA\xf8\xfbH\xe9T\x82u\x1f!\x17(\x9f6\xa9\x8d\xbe$\xb5\x03\x9f\xc0\xbf\x8d\xd4F\x9e\xef\xafR\xdb\x0f\x88\x87\xc8\xd7\x9f\xd9k\xfb\x85\xd3Mf=\xbe\xe5\x9e\x8a\x1d\xcag\x10\x15\\+\x9b\xb2\xa9\x02]\xb9\xd3S\xfe\xd0\x90\x14<K\xc7\x19\xe7[\x8eA\xf2\xae>\x1d\xc9,\x975%$\xedV\xbab\xae\xdfZ9\x8eYI\x0f\x1bE\xeaqGA\x1d\xddS\x9e\xea'
    b' \xfa\x17\x19\x8e\xce\xce\xdd\x88\xf4\xdeEgA/\x8a0\xe9\r\xcf\xf1\xbb\xdeh\x88G\xa33\x97\xc4\xe3\x11\xfe\xd961\x01\x91&\xb3\x05\x1bg\xb3\xa5`\xdf/k(\x9e\x93\x14\x9e\x83B\xc7\xf3\xdb\x84\x98\xaa\x93\xe1\xbeS\x82\x98\x94\x18\x17\x85*\x82\xdd\xa4\xc0\xfbH\x8a)'
    b' \xaea\xfc\xe7\x92\nX\xa1I\x0c\xff\xc5\x89\xe1{\x08?\x9d\x19QL\xfe\xa73\xc3\x1c\xb6\xbe\xbe\xdc\xd8oL\x06&&\xaf@\x17f}X.&\x1b\x91I\xf6\x11\x99p\x95\x04\xd1\xdb\x82\x13\xbf88\x03B\xfc\xff\xef\xb2\xfd\xb5\x86\xe6\xaal\x8f\xbc\xf188?\x8b{\xae\x1b\x8d{\xd1\x10G\xbd\x18A\x01\x1f\x06\x04N\xcb\x11\x0e\xa3\xe1xU\xb6+\x15y9D\xc7s\xab\xf5\xe7O\xbf\xfd\xf5\xf3\xa7\xdf\xf7P\xad\x9d\xee\xcd\x1d\xc2\x07\xd0oF\xd6Rd`\xc8p\x18\x07h\x14\r{C\x0f\x8f{\xf8<\x0e{g\xe3\x80\xf4\xc6\xc4\xc7x4\x8c\xceF\xfe\xbb\x9fU3\xc1\xc3\xfdD0\xddgx\x9f\x9a\x0e\x85\x87\x1f\xf4(\x16Y"\x8a\xaa\x98\xca7I\xb1h\x9a\x1dNY\xfc\xc4DYd\xba\xdf\xe1\xb9M\xd3DC\x84\x90\x1b\xc7aH\xbc\xb8\xc9\x13\xd0\xcd<\xb5\xb6N\xdb\xc7H\xb8\xf8\x8e\x96\xd6d\xe6\xc1\xce.=\xf0\xef\x1d\x8c\xd2\x1b\x18MfH\xd1\x90\xa2!E\x83\x11M\x12\x96K\xe0h\x06\x86\x82\x0ce\xc5\xe3\x1b\x8ao(\xd8P\xb0\xa1\x10C!\x86\x12\x18\n\xd4\x989\xcf\xf2\x1bp\x86z\xd8\xd6\xb4\xe0\x7f\xaf\tfT\xd7\x18\xa8\x12\x17\xf4\xbeX\xca\xf7i\x83D\x87R\xf7\x1d<\x1c\xe2\xc8\x0fp\x0c\xb9\xd3W\x14\xf1>\xf5\x1e|\xbd\xc6K\xdc\x0e/\xda\xc1\xebux\xfd\x1d\xbc\xa8\xc3\x8bw\xf0\xfa\x1d^\xb2\x83\x17wx\x83\x1d\xbc\xa4\xc3\x1b\xee\xe0\r:\xbc\xd1\x0e\xde\xb0\xc3\x1b\xef\xe0\x8d\xbaX\xb8;\x98\xd7\x803[\xc7C\xe0\xe5\x9d.-\x95\x1e\xab.\xc4\x13\xfb\xb4\x05\xf5\xe9\x9aN\xae>\xb6\'z\xa8\xab\xba\xa82z\x91\x0f\xc5\x8d\xee\xbf\xa9\x1eb\xde\xbc\xc2\xd4\x1cJD\x96\xcf.\x97y"\xd5|\xbd\xb3%C\xd5\xd7\xd3\xa3\xcb\xa4)\x93\xab\x12\xb9\x9a\x9d,?\x14y}9\xeeTa(\xef'
    b' \xf7\x86\x89\xfc\x05\x15\xd9\xd9\xac\xb7`\xa1RT\x17\xc7)l\xc3\x03\xfbo\x8b\x7f\xf4\xb8l\xf68\xba1\xc1h\xd3\xd8\xab6&\x92\xaa\x91\xbd\xb5z\xaf{\xb5\xd4\xfb\xd9\x03\x17/\xa8\xb8\x80\x1d\x14\xa3X\x19\x96\xe5P\xa6\xc1U=C0w\x88\xd7\xf6?Ht\xb7a0.`#k\x8d>\x13\x19\xe5\xb53&\xcb\xd1\x9c\n+\x81\x9f\x81\xfd\xf9\xd3\xaf\xf6&T\xf5\x01\xe25\xa0\xca\x1f\x83*\x7f\x0c\xaa\xfci\xa8\xf4\x10\xb5p\x84\xe0}\xd2\x85\x03E$$\x87\x04\xc7/\x0f\xe0@\xd1\x01\xc0\x81Z8\xfc\x16\x0e\xd3G\xee\xe0\x81\xa2\xe0\xc0\xd3\x03\xbdZ%\xdb#\x1e~\x8b\x07\xee\xe0\xd1\xf4h\x0f\x18\x8f-\xf9\xe1\x1e\x00\x1e\xb8\xc5\x83\xb4x'
    b' \x97\x84\xf8\x90\xf1\xf8\xcf\xbf\x0f\x13\x0e\xd2\xc2\x11t\xe0'
    b' \x1e\x0e\x0e\x19\x8e\xad\xe5\xea\x10\xf0\x08Z<\xc2\x0e\x1eq\xe8EG<\xfe\x04<\xc2\x16\x8fh\xf3\xb0{\xc4\xe3\x8f\xc7#j\xf1\x88;xDQp\xe0\xdb\xf9\x81\xe2\x11\x9b\x8bb\xe7jX\xf6\x0b9gbuQ\x84/.k\xd4\x1a\xeb\x1e\xf6\xddZ\x96\xf5[\xe5\xab'
    b' \xd8m\x89\x1e\xc2\x95b\xfb\r\xcf8\xe1\xe8\x9f\xedW.\xddH?\xfa\xe7\xf1+\x90\x1fz\xafT"\x0f\xcdA\xdb\xef$^\x84\xa2\xe8\xe8\xa0\'n\tz\x8f=:\xe8\xf1c{\x88\xfdc\x8d~\xea\x1c\r\xea\x1e\x8b\xf4S\x07\xdb\x80\x84\xc7"\xbd~\xd2\xec\x1e.\x9d\xee\xdf\x80\x9c\xce\x7fF?\xfd/PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x19\xcb\xf1\xf9\r\x01\x00\x00\xc6\x07\x00\x00,\x00\x00\x00ppt/slideMasters/_rels/slideMaster1.xml.rels\xc5\xd5Mk\xc3'
    b" \x18\x07\xf0\xfb>\x85x\xf1\xd4\x18\xd36MKM/cP\xd8it\x1f@\xe2\x93\x17\x96\xa8\xa8-\xcb\xb7\x9fl0\x1a(\xb2\xc3\xc0\x8b\xe0\xcb\xf3\x7f~'\x9f\xe3\xe9s\x1a\xd1\r\xac\x1b\xb4\xe2\x84e9A\xa0\x1a-\x07\xd5q\xf2~yYU\x049/\x94\x14\xa3V\xc0\xc9\x0c\x8e\x9c\xea\xa7\xe3\x1b\x8c\xc2\x87\x1a\xd7\x0f\xc6\xa1\x10\xa2\x1c\xc7\xbd\xf7\xe6@\xa9kz\x98\x84\xcb\xb4\x01\x15nZm'\xe1\xc3\xd6v\xd4\x88\xe6Ct@\x8b</\xa9\xbd\xcf\xc0\xf5\"\x13\x9d%\xc7\xf6,\x19F\x97\xd9\xc0_\xb2u\xdb\x0e\r<\xeb\xe6:\x81\xf2\x0fZP7\x0e\x12^\xc5\xac\xaf>\xc4\n\xdb\x81\xe78\xcb\xee\xcf\x17\x8fX\x16Z`\xfaXV\xa4\x94\x151\xd9:\xa5l\x1d\x93mR\xca61\xd96\xa5l\x1b\x93\x95)eeL\xb6K)\xdb\xc5dUJY\x15\x93\xedS\xca\xf61\x19\xcb\x93~\xb5y\xd4\x96v\x0cD\xe7\x00\xfb\xd7A\xe0C-,T\xdf'?\xeb\xaf\x83.\xc6o\xfd\x05PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdXK\x89PW\xc0\x03\x00\x00\xad\x0c\x00\x00\"\x00\x00\x00ppt/slideLayouts/slideLayout11.xml\xb5W\xd1\x92\x9b6\x14}\xefWh\xe8\x83\x9fX\x01\x06\x8c=\xf1f\x0c^:\x9d\xd9dwj'\xef\n\xc8k&\x02QIv\xect2\x93\xdfj?'_\xd2+\x01^\xdb\xeb\xa4\xf6\xd4y1"
    b' \xae\x8e\xce=\xe7\n]\xbfz\xbd)\x19ZS!\x0b^\x8d{\xee\x8d\xd3C\xb4\xcax^TO\xe3\xde\xbbyjG=$\x15\xa9r\xc2xE\xc7\xbd-\x95\xbd\xd7\xb7\xbf\xbc\xaaG\x92\xe5\xf7d\xcbW\n\x01D%Gdl-\x95\xaaG\x18\xcblIK"oxM+x\xb7\xe0\xa2$\n\x1e\xc5\x13\xce\x05\xf9\x04\xd0%\xc3\x9e\xe3\x84\xb8$Ee\xb5\xf3\xc59\xf3\xf9bQdt\xca\xb3UI+\xd5\x80\x08\xca\x88\x02\xfarY\xd4\xb2C\xab\xcfA\xab\x05\x95\x00cf\x1fRR\xdb\x9a\x8e-\xd0E\xcd\x0b\xc5\xe8\xa4\xca\xe7\x1b\x0b\x99x\xb1\x867\xaeu\x0b\x12d3\x96\xa3\x8a\x940\xf0\x1eB\x8b\x8c0d\xe2\x11\x08\x86\xe6t\xa3L\x98\xac\xe7\x82R}W\xad\x7f\x13\xf5\xac~\x14f\xf6\xdb\xf5\xa3@E\xae\xd1Z\x14\x0b\xb7/\xda0\xdcL27\xf8h\xfaSwKF\x9b\x85(\xf5\x15\xd4A\x9b\xb1\xe5Xh\xab\x7f\xb1\x1e\x03\x12(k\x06\xb3\xe7\xd1l\xf9p"6[\xde\x9d\x88\xc6\xdd\x02xoQ\x9dUC\xeee:\x9euZ\x14w\x97^G\\\xd6\xf7<\xfb(Q\xc5!1\xadC\x93\xe7.\xa2I^_\xebe\xeb\x89\xd2P\x16\xe2\xa2\x00\xe7\x1a\x8b\xacN\x1d\x1d\x8a\xf79\xc9\xd3\x02\x85\xa17\xf4\x9d&uo\xe0\x87\xfd\xe8P+\xcf\t\x06\xe6\xbd\xd6'
    b' \x88\x027\xf0\x82c%d\xbb\x84\xda\xc4<\xdf\xea\xd9\x1f\xe0\n\nhFc\x8b\x92\xf7-32bR\xcd\xd4\x96Q\xf3P\xeb\x1fCJ@0#\xb0\xcf,Z\xd9\xeff\x16\x92\xa5J\x18%\xd5\xce\x0fu\x9b\xb0"\xfb\x88\x14G4/\x14zC\xa4\xa2\x02\x19\t`W\x02\xa4\xa6\xa4\x0c1\x03I\xab\xfc\x91\x08\xf2\xc7\x11rC\xbd6\xbc;\xbe\xb8s\xf0\xfb>\xf6_\xfa\xa8\x15zd$\xa3K\xcer\xa0\xe2]\xc3R-\xdc\x91\xa3\xb0\xfe\xe6y\xf2\xf9\xce\xfa\xc1\xc0\xfb\x81\xb1\xa1\xe3\x0e\xa3\x9fil\xad\x95_\xb3\x9d\x83\xff\xd3h\xcd\xdb\xf8,\x0f\x8c\xc6\xddj\x07K\xba\x17.9\xa3\x19\x87\xcf\x14\xa3k\xca\xce\x80\xf7.\x84\x9f/\x0bq>z\xffB\xf4\x94\xaf\x84Z\x9e\r\xef_\n_,N\xa2_{\x8b\xf9\xdd\x16\x9b\x12E\x0fvV\xff\x1a;+\x87\x9d$?\xc3QH\xd8\xa2\xdbS\xce\x8f7\x15>U\xfb\xdf\xa9\xf6\x05\x1c\x7f:\x8b\xbf\x828\x99L\x9d(\xb0\xef\xa2IhG\x91\x1f\xd8\xf1\xd4\xbf\xb3\x93\xd8O\x92\x89\x13\x0c\xd3\xc4\xff\xd2\x9d\xaa9\xa4\xaa\x8a\x92\xa6\xc5\xd3J\xd0\x87\x95>$\xcfs\xc5\xc5\xde\x00\xbb\xfdgG\x80\xc0\xf5=\t:OR\xce\xf5.\xdcw\xc5\xbf\x86+\x0b%\x1a[\xfe\\\x11\x01+t\xce\xfc\xc7\xe7\xee\x12g\xae\xabH\xd8)2cEN\xd1\xdbU\xf9\xe1H\x97\xe0\x1a\xba@G\t\xd0\'\xa5\xf1~B\xd1&n\x9a\x86\xd3\xc9\xd0v\x9c\x08\xfa\xdc\xd8\x8f\xec\xa1\x07\xe5\x1b\x87\x81\xe7\r#\x7f\x10\xc5\xe9\xaeh\xa5\xce\xbc\x02v\xe7\xd6\xea\xb7\xaf\x7f\xff\xfa\xed\xeb?W\xa8U\xbc\xdfA\xc2\x89p/U{\x87V\xa2\x80D\xe2x\x18zI\x14\xdb\xb1\xeb\xa7\xb6?\x1d\x0e\xecI\x1a\x06v\x1a\xf4}?\x89\xa3I\xd2\xbf\xfb\xa2;Q\xd7\x1fe\x82\x9av\xf7\xf7\xbck\x94]\xffE\xab\\\x16\x99\xe0\x92/\xd4M\xc6\xcb\xb6\xe7\xc65\xffDE\xcd\x0b\xd3v\xbbN\xdb(\xaf\x89\xfex\x87\xae\xe7y\xfd\xc1\xb0\xb3\t\xb8uW\xc3\x167\xbd\xb2)\x11&\xde\x90\xfaam\x8a\xa44\xe7\\b\x86j\xf8_\xd0\xd6\xc8s\x08\xde\xfb\x9fq\xfb/PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00-\x00\x00\x00ppt/slideLayouts/_rels/slideLayout11.xml.rels\x8d\xcf\xbd\x0e\xc2'
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x00\xfd\xec\r*\x04\x00\x00\x05\x11\x00\x00!\x00\x00\x00ppt/slideLayouts/slideLayout1.xml\xcdX]\x8e\xdb6\x10~\xef)\x08\xf5\xc1O\n\xf5CI\xb4\x11o`\xc9\xab\xa2\xc0fw\x11o\x0e\xc0\x95h[\x08%\xaa$\xed\xd8)\x02\xe4Z\xedqr\x92R\x94dy\x7f\xda:\x80\x03\xf8\xc5\xa2\xa8\x99\xe17\xf3\xcd\x90\x1c\xbf}\xb7+\x19\xd8R!\x0b^MG\xee\x1bg\x04h\x95\xf1\xbc\xa8V\xd3\xd1\xc7\x87\xd4\xc6#'
    b" \x15\xa9r\xc2xE\xa7\xa3=\x95\xa3wW\xbf\xbc\xad'\x92\xe57d\xcf7\nh\x13\x95\x9c\x90\xa9\xb5V\xaa\x9e@(\xb35-\x89|\xc3kZ\xe9oK.J\xa2\xf4\xabX\xc1\\\x90\xcf\xdat\xc9\xa0\xe78!,IQY\x9d\xbe8E\x9f/\x97EF\xe7<\xdb\x94\xb4R\xad\x11A\x19Q\x1a\xbe\\\x17\xb5\xec\xad\xd5\xa7X\xab\x05\x95\xda\x8c\xd1~\nI\xedk:\xb5T\xa1\x18\xb5\x80\x11\x13[=\xe1ZW\xda\xf3l\xc1rP\x91RO<4\x12`\xc1\x8a\x9c\x9aO\xb2~\x10\x946\xa3j\xfb\x9b\xa8\x17\xf5\xbd0\x1a\xb7\xdb{\x01\x8a\xbc\xb1\xd0iZ\xb0\xfb\xd0\x89\xc1V\xc9\x0c\xe03\xf5U?$\x93\xddR\x94\xcdS\x07\x02\xec\xa6\x96c\x81}\xf3\x0b\x9b9\xbaS"
    b' k\'\xb3a6[\xdf\xbd"\x9b\xad\xaf_\x91\x86\xfd\x02\xf0h\xd1\xc6\xab\x16\xdcKw<\xebI'
    b' \xdc\x83W=^Y\xdf\xf0\xec\x93\x04\x15\xd7\xfe4\xee\xb7\xee\x1d$Z\x9f\x9bg\xbd\xee\xa2\x9e)a\xacY}$\x9a\xef\xf0x}\xf9z0B\x1c`\xa7\xf5\xd2s}\x07y\xc1\xd3\xb8DQ\xe4!\xa7\xf3\xd7E\x91\xe3\xb4\x12\xc7^\xcbn\t\xb5\x8by\xbeo\xb4\x1f\xf5\xd3\xb0B&L\xaa\x85\xda3j^\xea\xe6\xc7\xc0\x10:\x18\x8c\xe8\x82\xb1he\x7f\\X@\x96*a\x94T\x87h\xab\xab\x84\x15\xd9\'\xa08\xa0y\xa1\xc0{"\x15\x15\xc0\xe4\x97./m\xb2\x01\xa1\x0c\x14c\x92V\xf9=\x11\xe4\xc33\xcb-\xd8\xda'
    b' \xed\x11\xc2\x9e\x9f\x7fg\xc9\xefYZl\x1e\xdb5\xbds\x10%7\x8f-Qz\x91\xdd\xa0r:a\xae\x1f\xb9a\xc7\x98\x8fq\xa8\x0b\xf0)c\xa1\xa6\x0b\x1f\x18\x8b\x02/t^\xe4\xe9I\x8c\x99\xf1\x96\xb9Z\x16\x94D\xdc\x98\xb4/\xaa\\W\xbf\x19\x12\xb6\xaaL\xe6Y\xc6\xc0\xe6Vov\xc6@N\x97\x1f\xba\x00q]\xe5i\xc1\x98yi6\x15\x9a0\x01\xb6\x84\xe9\x8db\xe7\x1aEUT\xaa\x9d\x89\x02\xe7\x00\xf5'
    b' \xdc\xbe\rv\xe0`\x1f\x1e\xf0uP\xbd\x01*\n\xa2&2\x17\x88\xd7\x1b\xf0\xfa\x03\xde\xb1\x8b\xd0e\xe2\xf5\x07\xbch\xc0{H\xc3\xcb\x03\x8c\x06\xc0\xc1\x11`\xeca|\x99\x80\x83\x01p8\x00\xf6<\x1c:\x97\t8\x1c\x00GG\x80#\xe4_h\xcdE\x03`<\x00n\xd0^h\xd1\xe1\x01\xf0\xf8\x08p\x18D\x17Zt\xe3\xba\x1f\x1f\x9d\x1eg8\xeee\x7f\xfa\xfe\xfc\x13\x1f\xf5\'\xfe\x9c(\n\xee\x19\xc9\xe8\x9a\xb3\\\x83\xf0\xcfq\xf2\xe7J{\xfdE_\xb1\t[\xf6\xa7\xbf\xf3\xdf\xc7?\xfc\x81[\xd5R\xdf\xaf\x1b/\xfe\x0c\xe2d6wp`_\xe3Yhc\x8c\x02;\x9e\xa3k;\x89Q\x92\xcc\x9c`\x9c&\xe8k\x7f[\xcf\xb5\xab\xaa(iZ\xac6\x82\xdem\x94u*[.\xf4"\xe8\xfa\x03#\x1a\xc0\xf99\tzNR\xce\x9bt8f\x05\x9d\x83\x95\xa5.\x1cC\xcb\x1f\x1b"\xf4\n=3\xffs1\xfb\x11f\xce\x1b\x91\xf0p/m\x1a(p\xbb)\x1f\x9f\xc5%8\xcb=\x95\xe5\xda\xf4\xab\xa1\xf1~B\xd2&n\x9a\x86\xf3\xd9\xd8\xd6wW\xdd?\xc7\x08\xdbcO\xa7o\x1c\x06\x9e7\xc6(\xc2qzHZ\xd9x^it\xa7\xe6\xea\xf7o\x7f\xfd\xfa\xfd\xdb\xdfg\xc8Ux\xdc\xae\xea\x1b\xf7\x8dT\xdd\x08lD\xa1\x1d\x89\xe3q\xe8%8\xb6c\x17\xa56\x9a\x8f#{\x96\x86\x81\x9d\x06>BI\x8cg\x89\x7f\xfd\xb5i{]4\xc9\x045m\xf4\xefy\xdf\x80\xbb\xe8E\x0b^\x16\x99\xe0\x92/\xd5\x9b\x8c\x97]/\x0fk\xfe\x99\x8a\x9a\x17\xa6\x9dw\x9d\xae\x017\xdb\xb7\xef\x86\xd8\x89\x82\x00\xfb\x1dM\x1a[\xff4ha\xdb\x8c\x9b\x14a\xe2=\xa9\xef\xb6&IJ\xb3\xe1&f\xaa.\xaaU\x97#\x83\x08<\xfa\xff\xe2\xea\x1fPK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00ppt/slideLayouts/_rels/slideLayout1.xml.rels\x8d\xcf\xbd\x0e\xc2'
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x01W\xe8\x8bm\x03\x00\x00\x96\x0b\x00\x00!\x00\x00\x00ppt/slideLayouts/slideLayout2.xml\xb5V\xd1r\x9b:\x10}\xbf_\xa1\xa1\x0f~"\x02\x0c\x0e\xf6\xd4\xe9\x18\x1c\xee\xdc\x99\xb4\xc9\xd4\xe9\x07('
    b' \x82Z\x81t%\xd9\xb5\xdb\xe9L\x7f\xab\xfd\x9c~I%\x01\x8e\x9d\xa63\xce\x94\xbe'
    b' !Vgw\xcf\x1e\xa4}\xf9j[S\xb0\xc1B\x12\xd6\xccG\xfe\x997\x02\xb8\xc9YA\x9a\xfb\xf9\xe8\xddm\xe6\xc6#'
    b" \x15j\nDY\x83\xe7\xa3\x1d\x96\xa3W\x17\xff\xbc\xe43I\x8b+\xb4ck\x054D#gh\xeeTJ\xf1\x19\x842\xafp\x8d\xe4\x19\xe3\xb8\xd1\xdfJ&j\xa4\xf4\xab\xb8\x87\x85@\x1f5tMa\xe0y\x13X#\xd28\xdd~q\xca~V\x96$\xc7K\x96\xafk\xdc\xa8\x16D`\x8a\x94\x0e_V\x84\xcb\x1e\x8d\x9f\x82\xc6\x05\x96\x1a\xc6\xee>\x0eI\xed8\x9e;\xec\xee\xbd\x03\xac\x91\xd8\xe8W\xdf\xb9\xd0y\xe7+Z\x80\x06\xd5z\xe1\x96(\x8a\x81&\x07\xa4\xacQ\x1a\xc9\x1aH~+06\xb3f\xf3\xaf\xe0+~#\xec\xbe7\x9b\x1b\x01Hap\xba\xfd\x0e\xec>tf\xb0\xddd'\xf0\xd1\xf6\xfb~\x8af\xdbR\xd4f\xd4d\x80\xed\xdc\xf1\x1c\xb03Oh\xd6\xf0V\x81\xbc]\xcc\x1fV\xf3\xea\xfa\t\xdb\xbc\xba|\xc2\x1a\xf6\x0e\xe0\x81S\x93U\x1b\xdc\xaf\xe9\x04\xce\x11\x1d\xfe>\xab>^\xc9\xafX\xfeA\x82\x86\xe9|L\xfamz{\x8b6g3\xf2\xaac^\x19(\xa7\xa7\xc1|\x84\x87\xceeO\x96\xda&\xac\xd8\x19'wz\xb4\x8bhF\xa5Z\xa9\x1d\xc5\xf6\x85\x9b\x87\rC\xe8x)\xd2\xbavp\xe3\xbe[9@\xd6*\xa5\x185{B\xd4EJI\xfe\x01(\x06pA\x14x\x8d\xa4\xc2\x02\xd8`\xf4_\xa0!\r;\xcard!qS\xdc"
    b' \x81\xde>BnY\xe46\xe8>B\xd8S\xf8{"\xc7=\x91\x9d\x9a\xc0\rE9\xae\x18-t\x10\xc1\x9f\xd1J\x8a\xed\x83\xc9\x00\x8cr\x93\xf2\x86\xee\xa9\xfbC\x86\x8dl-\xc1\xf2\x88a\xd8{;r\xe9?\xd3\xe5\n\xe7L\xff\xa3\x14o0=\x01>x&\xfcmE\xc4\xe9\xe8\xe3g\xa2gl-Tu2|\xf8\\xR>\x89>\xb4\xb6\xc3^\xdbK\xa4\xf0\x91\xb0\xc7C\x9c\x17\x85\xd2\xd9}\xd2g>\xa2\xa5\xd3\x89\xdd\x1bN\xed\xa5>\xf2M\x16\x9f\xa3$],\xbd8r/\xe3\xc5\xc4\x8d\xe30r\x93ex\xe9\xa6I\x98\xa6\x0b/\x9afi\xf8\xa5\xbf>\n\x9d\xaa"5\xce\xc8\xfdZ\xe0\xeb\xb5\xb9\x1eN\xab\x8a\x0f\x83s\xe8\x8f\x1f*\xa2\x03\x18\xbe&Q_\x93\x8c1\xf3\x17\x1eV%\x1c\xa2*\xa5\x12mY\xfe_#\xa1=\xf4\x95\x19\xf0\x1c\x1a\x96\x91I\xcf\xc8\x8a\x92\x02\x837\xeb\xfa\xee\x11/\xd1\x10\xbc\xe8\xd6IC?IM\xf0\x17D\x9b\xfaY6Y.\xa6\xae\xe7\xc5\xba\xa1K\xc2\xd8\x9d\x06Z\xbe\xc9$\n\x82i\x1c\x9e\xc7I\xb6\x17\xad4\x997:\xbaS\xb5\xfa\xe3\xeb\xb7\x17?\xbe~\x1f@\xab\xf0\xb0w\xd27\xc2\x95T\xdd\x0c\xac\x05\xd1\x89$\xc9t\x12\xa4q\xe2&~\x98\xb9\xe1rz\xee.\xb2I\xe4f\xd18\x0c\xd3$^\xa4\xe3\xcb/\xa6\x07\xf3\xc3Y.\xb0\xed\xeb\xfe+\xfa\x8e\xd0\x0f\x7f\xe9\tk\x92\x0b&Y\xa9\xcerVw\xcd%\xe4\xec#\x16\x9c\x11\xdb_\xfa^\xd7\x11n\x90\xb9\x1a&~8\xf6\xc3'
    b' \x8a\xbb2\xe9\xd8\xfa\xd1F\x0b\xdb\xfe\xd0J\x84\x8a\xd7\x88_o\xacHj{\xcf\xa5v\x89\xeb\x06\xb8\xd3\xc8\x83\t<h\xa8/~\x02PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00ppt/slideLayouts/_rels/slideLayout2.xml.rels\x8d\xcf\xbd\x0e\xc2'
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x8b`\xedZc\x04\x00\x00X\x11\x00\x00!\x00\x00\x00ppt/slideLayouts/slideLayout3.xml\xcdX\xdbn\xdb6\x18\xbe\xdfS\x08\xea\x85\xaf\x14R\x12u\n\xea\x14\x96\x1cm\x03\xd2$\xa8\xd3\x07`$\xda\x16J\x1dF\xd2\xae\xbd\xa1@_k{\x9c>\xc9HJ\xb2\x1c7i\xdd\xce\x0br#R\xd4\x7f\xf8\xfe\x03\xf9\xf3\xd7\xeb7\x9b\x92\x1ak\xc2xQW\xe3\x91}\x06G\x06\xa9\xb2:/\xaa\xc5x\xf4\xfe.\xb5\xc2\x91\xc1\x05\xaerL\xeb\x8a\x8cG[\xc2Go.~y\xdd\x9cs\x9a_\xe1m\xbd\x12\x86\x14Q\xf1s<6\x97B4\xe7\x00\xf0lIJ\xcc\xcf\xea\x86T\xf2\xdb\xbcf%\x16\xf2\x95-@\xce\xf0G)\xba\xa4\xc0\x81\xd0\x07%.*\xb3\xe3g\xc7\xf0\xd7\xf3y\x91\x91i\x9d\xadJR\x89V\x08#\x14\x0b\t\x9f/\x8b\x86\xf7\xd2\x9ac\xa45\x8cp)Fs?\x84$\xb6\r\x19\x9b\x9cd\xbf\x11\x9c\x9b\x86&dk\xb9d\x9b\x17\xd2\xf6lFs\xa3\xc2\xa5\\\x98\x91L\xb1\x1b\x8a\x900\xfd\x957w\x8c\x105\xab\xd6\xbf\xb2f\xd6\xdc2\xcdt\xbd\xbeeF\x91+!\x1d\xb3\t\xba\x0f\x1d\x19h\x99\xf4\x04\x1c\xb0/\xfa)>\xdf\xccY\xa9F\xe9\rc36\xa1il\xd5\x13\xa85\xb2\x11F\xd6.f\xc3j\xb6\xbcy\x846[^>B\rz\x05`O\xa9\xb2\xaa\x05\xf7\xb59No\xce]!(1\xec\x9dU=^\xde\\\xd5\xd9\x07nT\xb5\xb4G\x99\xdf\x9a\xb7\xa3hmVc\xb3\xec\\/\x94(\xb3w\x83\xfa\x08\xf6\x95\xf3\xc7=\x118\x8ek\xbb\xdaD\x84\xa0\x1f\xc1\x03\xa7\x04A\xe0'
    b' \xd8\x19k\xbb\xbe\x03\x03\xef\xd0d\xde\xa9\x10\x9b\xb8\xce\xb7\x8a\xfb^\x8e\xd2T\\e\xcbZf\xa9heR.fbK\x89\x9e\xaf\xa9\xdd(\x12\xba\xa8\xc6&5\xd5ZN\xe6\xef\xe4\x12\xffSb\x81J\xe7\xbd\x0e|\x86\xa5\x070\xa5\x9d\xda\x8e\xb3\x9d\xefIl\xd4C\x9b\xc8\xa4\x10\x8a\xe5v4Ie\xbd\x9f\x99\x06/EB\t\xaeva\x14\x17\t-\xb2\x0f\x86\xa8\r\x92\x17\xc2x\x8b\xb9'
    b' \xcc\xd0.\x94\x9bWJT\xd2\x85\xd6\xa1E\x92*\xbf\xc5\x0c\xbf;\x90\xdc"j\xb4\x17z\xebA\x1f\xf8\xa7\xc3\xef\xee\xc2\xaf\xdc|KqF\x965\x95\x9b\xc1pN\x91\t\xca\xfb\xa6T\xb4\x19\xc8\x7f*!\x9c\x08\xfa\x81\x9c\x7f#!<\x08\xed0\xf8\xe1\x84\xb8\x7f:!J\xcc\xae\xf4\xee*\xaa\\\x9e4j\xaa\x05\xac\xae\xe5i\n\x0e\xd2\xc4Qi\xa2\xbdT\xd3"O\x0bJ\xf5\x8b:\xbfHB\x99\xb1\xc6Tf\xdf\xc6\xd64\xa2\xa8D\xbb\x12x\x10\xf6\x1bwG\xdc\xbe\rr@\xaf\xe9a\xd6\xe9\xa93'
    b" E^\xe0\xc0#\xe1\xda\xe13\xc2u\x06\xb8\xee\x007\xb2\x11:\x1a\xae\xff\x8cp\xdd\x01.\x1a\xe0\xdan\xa0Q\x1c\x87\x17=#^4\xe0\xf5\xf6\xf0\x86N\x18\xbeH\xbc\xde\x80\xd7\x1f\xf0:N\xe8\xc3\x17\x89\xd7\x1f\xf0\x06{x\x03\xe4\x1e\xbf\xdd\x9e\x13o0\xe0\r\x07\xbc\n\xec\xf1\xfb\xed9\xf1\x86\x03\xdeh\x0f\xaf\xef\x05/s\xbfEO\xd6|\x85^\x12\xec\x8a\xfb\x7f\xbc\x03\xa8B\xa7\xaf\x00\xfc\xc1\x1d\xe0g\xea<\xea\xeb\xfc\x14\x0b\xf2\xa0\xce\xbb\xa7\xa8\xf3\xb90u\x1c\x96\x98\xce\xfbz\x0f\xbf]\xf0\xc1ce\xf9A-\x06;\xbf\xce\xe5\x8d]Y\xf1\x97\x17'\x93)\x0c=\xeb2\x9c\xf8V\x18\"\xcf\x8a\xa7\xe8\xd2Jb\x94$\x13\xe8Ei\x82>\xf5\x1d@.M\x15EI\xd2b\xb1b\xe4f%\xccc\xc3a\x03'\x00\xb6;x]\x028\xfd\xdd\xcb\xebc\x92\xd6\xb5\x8a\xf7~T\xd0)\xa22\x17\xac\r\xcb\x1f+\xcc\xa4\x86>2\xdf\xb9\x8a\xfdHdN\xeb\x11\xbf\xf7\xc8L\xee>b\\\xaf\xca\xfb\x03\xbfx\xa7\xf0\x8b\xec~\xa5\xe8G]\xe3\xfc\x0fI\x9b\xd8i\xeaO'\x91\x05a({\xf2\x18\x85V\xe4\xc8\xf4\x8d}\xcfq\xa2\x10\x05a\x9c\xee\x92\x96+\xcb+\x89\xee\xd8\\\xfd\xf2\xf9\xefW_>\xffs\x82\\\x05\xfb\xdd\xaf<{\xae\xb8\xe8f\xc6\x8a\x15\xd2\x908\x8e|'\tc+\xb6Qj\xa1i\x14X\x93\xd4\xf7\xac\xd4s\x11J\xe2p\x92\xb8\x97\x9fT\x17m\xa3\xf3\x8c\x11\xdd\x9a\xff\x9e\xf7M\xbd\x8d\xbej\xeb\xcb\"c5\xaf\xe7\xe2,\xab\xcb\xee\xff\x00h\xea\x8f\x845u\xa1\x7f\x11\xd8\xb0k\xea\xf5y\x1d\xf9\xd0G\xa1\xdb\xf5}\x1aZ?j\xb0\xa0\xed\xeeu\x86P\xf6\x1677k\x9d#\xa5>P\x13\xbd\xd4\x14\xd5\xa2K\x91\x81\x04\xec\xfd\x12\xb9\xf8\x17PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00ppt/slideLayouts/_rels/slideLayout3.xml.rels\x8d\xcf\xbd\x0e\xc2"
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdXO\xca\x82\x1c\x08\x04\x00\x00h\x12\x00\x00!\x00\x00\x00ppt/slideLayouts/slideLayout4.xml\xedX\xddr\xda8\x14\xbe\xdf\xa7\xd0\xb8\x17\\9\xb2\x8dl\x0cS\xd2\xc1&\xde\xd9\x99\xb4\xc9\x14\xfa\x00\x8a-\x82\xb7\xb2\xe5\x95\x04\x81\xeet\xa6\xaf\xb5\xfb8}\x92\x95\x84\x8d!\xa1\x05\xb6\\\xe6\x06\x0b\xf9\xd3w\xfe\x8f\xed\xf3\xf6\xdd\xaa\xa0`I\xb8\xc8Y9\xec\xb8WN\x07\x902eY^>\x0e;\x9f\xa6\x89\x1dv\x80\x90\xb8\xcc0e%\x19v\xd6Dt\xde]\xff\xf6\xb6\x1a\x08\x9a\xdd\xe25[H\xa0(J1\xc0Ck.e5\x80P\xa4sR`q\xc5*R\xaa{3\xc6\x0b,\xd5_\xfe\x083\x8e\x9f\x14uA\xa1\xe78\x01,p^Z\xf5y~\xcay6\x9b\xe5)\x19\xb3tQ\x90RnH8\xa1X*\xf5\xc5<\xafD\xc3V\x9d\xc2Vq"\x14\x8d9\xbd\xaf\x92\\Wdh\xc9\'v\xf7\xf0\xa7\x05\x0c\x8e/\xd5\x8ek]+\xd3\xd3\t\xcd@\x89\x0b\xb51}b'
    b" f\xa5T4\xe6\x96\xa8\xa6\x9c\x10\xbd*\x97\xbf\xf3jR\xddss\xe2\xc3\xf2\x9e\x83<\xd3\x0c\xf5I\x0b\xd67j\x18\xdc\x1c2\x0b\xf8\xec\xf8c\xb3\xc4\x83\xd5\x8c\x17\xfa\xaa<\x01VC\xcb\xb1\xc0Z\xffB\xbdGV\x12\xa4\x9b\xcd\xb4\xddM\xe7w\x07\xb0\xe9\xfc\xe6\x00\x1a6\x02\xe0\x8ePm\xd5F\xb9\x97\xe6x\x8d9\xd3\\R\x02\xdc\xadU\x8d\xbe\xa2\xbae\xe9g\x01J\xa6\xec\xd1\xe6o\xcc\xdb\"66\xebk5o\xdc\xae\xa9\xac\xc6\r\xfa&\xdc\x15.\x1ag\xc9U\xc4\xb2\xb5\x16\xf2\xa0\xaef\x13\x0f\xa8\x90\x13\xb9\xa6\xc4\xfc\xa9\xf4\x8fQ\x83+})VIm\x91\xd2\xfe4\xb1\x80(dL\t.\xb7\x0e\x91\xd71\xcd\xd3\xcf@2@\xb2\\\x82\xf7XH\xc2\x81QF\x95\x80\xa2\xd4\xde\x91\xc6G\x86\x92\x94\xd9=\xe6\xf8\xe33\xe6\x8d\x17+\xa3t\xa3!l\\\xf8cGv\x1bG\xd6\xd9\x04\xee)N\xc9\x9c\xd1L)\xe1\xfd\x9a[\xc5\x17U\r\x98\xce,%i\xd5\x82\x7f\xe0\xdb\x03Y\x86\xfc\x9e*\x0e\x93>n\xe08z\xbd\x97p\xc8\xe9\x86\x81S'\x12\xf2=\xbf\x1ft\x9f\xa7\x93\xa8E\xfc4jf\xbd\xa4n\xadFFf\xda\xbdZ\x7f/t\x9a\x0c\xdd\x01\xa8\xa5w\x00\x8bv\xb1^\x8b\xed\x1e\xc0:\xbb\xd8n\x8bE/\xb1\xee\x9e\x0e\xa8\xc5\xfa\xc7\xb0~\x8b\r\x8ea\x83\x16\xdb;\x86\xed\xb5\xd8\xf0\x186l\xb1\xfdc\xd8\r\x00\xee\x07\xc6TS\xa5\xd3}I\xb7e\xf3\x8b\xd5\xa53\xc8\x14\x97\xd8\xab.\xd8H\xdb\x13\xe9\x9e)rBRVf\x80\x92%\xa1'\xd0{g\xd2O\xe79?\x9d\xbd{&{\xc2\x16\\\xceO\xa6G\xe7\xd2\xe7\xb3\x83\xec\x97\xeek\xe8g}\xad{\xd1\xbe\xe6\x9d\xdf\xd7\x02\x14\xbe6\xb6\xd7\xc6\xf6\xda\xd8^\x1b\xdb\xb9\x8d\xcdo\x1a\xdb\x18K\xb2\xd7\xd5\xd0%^\x823i\xbdxos.\xf7R<S_0\xda\x8a\xbf\xfd(\x1e\x8d\x9d\xd0\xb7o\xc2Q`\x87!\xf2\xedh\x8cn\xec8Bq<r\xfc~\x12\xa3\xaf\xcd\x07Q\xa6L\x95yA\x92\xfcq\xc1\xc9\xddB\x7f\xf3\x9c\x16\x15\x17z=\xe8v\xdb\x88(\x05.\x1f\x93\xa0\x89I\xc2\x98\xae\xc2\xdd\xa8\xf8\x97\x88\xcaL\xf2MX\xfeZ`\xae$4\x919\xf2J}Nd.\xeb\x91^\xe3\x91\t\xcd3\x02>,\x8a\x87g~\t.\xe1\x17A3E}\xd05G\x9e\xca\xff+ic7I\x82\xf1\xa8o;N\x98\xd8a\x84B\xbb\xef\xa9\xf4\x8d\x02\xdf\xf3\xfa!\xea\x85Q\xb2MZ\xa1-/\x95v\xa7\xe6\xea\xf7o\xff\xbc\xf9\xfe\xed\xdf\x0b\xe4*\xdc\x1d\x08\xa8'\xc2\xad\x90\xf5\n,x\xae\x0c\x89\xa2~\xe0\xc5adG.Jl4\xee\xf7\xecQ\x12\xf8v\xe2w\x11\x8a\xa3p\x14wo\xbe\xea\xc1\x82\x8b\x06)'fR\xf1G\xd6\xcc8\\\xf4b\xcaQ\xe4)g\x82\xcd\xe4U\xca\x8az\\\x02+\xf6Dx\xc5r31q\x9dz\xc6\xb1\xc4\xfa\xd1\xd0\x0b=\x0f\xa1>\xea\xd5aR\xba5W\xa3-\xdc\x8c;L\x8aP\xfe\x1eWwK\x93$\x85y\xce\xc5f\xab\xca\xcb\xc7:GZ\x08\xdc\x19\x11]\xff\x07PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00ppt/slideLayouts/_rels/slideLayout4.xml.rels\x8d\xcf\xbd\x0e\xc2"
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\xe9\xa4\xc4\x8f\xe3\x04\x00\x006\x1c\x00\x00!\x00\x00\x00ppt/slideLayouts/slideLayout5.xml\xedY\xdd\x92\xa28\x14\xbe\xdf\xa7\xa0\xd8\x0b\xaf\x18\x08\x04\x08\xd6\xd8S-\xddnmUOw\xd7\xe8<@\x1ab\xcb\x0e\x106\x89\xb6\xce\xd6T\xcdk\xed>\xce<\xc9&\x08\xa2\xb6\xda8z\xb1U\xeb\r\xc4p\xf2\xe5\xfc~\x1c\xc9\xfb\x0f\xf3,\xd5f\x84\xf1\x84\xe6\xbd\x0exgu4\x92G4N\xf2\xe7^\xe7\xf3h`\xa0\x8e\xc6\x05\xcec\x9c\xd2\x9c\xf4:\x0b\xc2;\x1f\xae~y_ty\x1a\xdf\xe1\x05\x9d\nMB\xe4\xbc\x8b{\xfaD\x88\xa2k\x9a<\x9a\x90\x0c\xf3w\xb4'
    b' \xb9|6\xa6,\xc3B\xfed\xcff\xcc\xf0\x8b\x84\xceR\xd3\xb6,\xcf\xccp\x92\xeb\xd5z\xd6f=\x1d\x8f\x93\x88\xdc\xd0h\x9a\x91\\,A\x18I\xb1\x90\xea\xf3IR\xf0\x1a\xadh\x83V0\xc2%L\xb9zS%\xb1(HO\x17/t4\x1f\xbd\xd0\x87\xa7?t\xad\x14f39\r\xf4+i\x7f4Lc-\xc7\x99\x9c\x08iV`\x96p\x9a\x97Ox1b\x84\xa8Q>\xfb\x8d\x15\xc3\xe2\x91\x95\x0b\xeeg\x8fLKb\x05P-\xd4\xcd\xeaA%f.\x17\x95\x03sk\xf9s=\xc4\xdd\xf9\x98e\xea.\xbd\xa1\xcd{\xba\xa5k\x0bu5\xd5\x1c\x99\x0b-ZNF\xcdl4y\xd8!\x1bMnwH\x9b\xf5\x06\xe6\xda\xa6\xca\xaa\xa5r\xaf\xcd\xb1ksF\x89H\x89\x06VV\xd5\xfa\xf2\xe2\x8eF_\xb8\x96Si\x8f2\x7fi\xdeJbi\xb3\xba\x17\x93\xda\xf5\nJ\xaf\xdd\xa0\x1e\x9a\xeb\x9b\xf3\xdaYb\xde\xa7\xf1Bm\xf2$\xef\xe5$\xee\xa6\\\x0c\xc5"%\xe5x\x96\x82J\x8d\x98\x8c?-]\xbb6mn\x8a\x17\xeaRJ3i]\x8ae\x19\xe8$7>\x0fu\x8dg"L\t\xceW\xee\x13Wa\x9aD_4A5\x12\'B\xfb\x88\xb9'
    b' L+U\x97E#\x11\x15\xba(\xf7(!I\x1e?b\x86?m!/5*J\x13k{\xcc\xda\xe1\xfb\xdd\xee\xac\xdc\xaeb\xfe\x98\xe2\x88Lh\x1aK\r\xecsD@\xf9S\x97\x1b\xcd\x1b\xf1=\x81\xd8\x91\x92\xd0\xf5e5\x95\xb9\x06\\\xc7\x05\xc0\xd9\xccNhA\x0b'
    b' \xb4\xcc:\xcf\t|\xcf\xdeN=^\xed\xb0\x1da\r\xe7\xd1\x84J\xb6x\xd2\xf7\x05[\xcb0\xbb+\x93:\xc9cY\xe0jX\x02L\xef%\x89\x99M.h\xfc\xabL_\xa84}\xaa\xcd\xdcH\x199\xb4\x1b\xc0\xda\xaaV\xa8\xd6kT\xbbAu\x1a\xd4\x00@\xd8\x16\x15\xa0\xd7\xa8N\x83\n\x1bT\xe0\xf8\xc0k\r\xeb\xbd\x86\x85\r\xac\xbb\x06\x8bl\x84N\x81u\x1bX\xaf\x81\xb5m\xe4Y\xa7\xc0z\r\xac\xbf\x06\xebC\xa7u\xc4v\xc1\xfa\r,j`\x15f\xfb\x90\xed\x80E\rl\xb0\x06\xeb\xb9\xfeI!\x0b\xf62\x9a\xdaD\n\xac\xa8\xebD\x86Se\\\x12\x1c\xdf`\xb8\x9fa1\xa8\xaf^\xa2\xb9\x90Vo\x10\x99s\x1a\x91)?Mp:\xaeh\xcc>\x85\xc6l\xe0C\xe4\xbb\x07h\xcc\t\\'
    b" \x8b\xa3-\x8f\xbd\xfd\xa6j\xd8i\x1f/\xed\xe2\x9c}l\xb3\x8bI\xf6q\xc8\xae\\\xdbG\x0c\x07e\xb7\xaa\xfd\xa0\xecV\t\x1f\x94\xdd\xaa\xcb\x83\xb2[\xc5vP\xf6\xbfQA\xdb[\x82#\xb7\x1c\x92\x88\xe6\xb1\x96\x92\x19I[\xc0\xdbG\xc2\x8f&\tk\x8f\xee\x1c\x89>\xa0S&&\xad\xe1\xe1\xb1\xf0\xc9x'\xfa\xb9\xbb3wow\x06\xcf\xd7\x9d\xa9\x04\xfes\x8a\x99L\xa9\x8a\xe3\x9c\xe39\xce\x83\xaee\xbb\x07{5\xe0K\xe6\xbb\xf4j\x97^\xed\xd2\xab\xfd\x9f{5\xefP\xaf\xe6\x9e\xde\xabmR\x19<\x89\xca\xf6\xf5k\r\x95]\xfa\xb5K\xbfv\xe9\xd7.\xfd\xda\x92\xdb\xfc\x9a\xdbn\xb0"
    b' \x1b\xc4\xe6\x9d\xa3_\x8b\x85\xbe\xfdw\x14X\xa7~\xdf4W\xee\x1d\xa7qi\xc5_n?\xbc\xbe\xb1\x90k\xdc\xa2k\xcf@\x08\xbaF\xff\x06\xde\x1aa\x1f\x86\xe1\xb5\xe5\x06\x83\x10~\xab\xbfo\xc7\xd2T\x91dd\x90<O\x19y\x98\n\xbdmT\x80i\xfb&p\x9a\x88H\x05\xce\x1f\x13T\xc7d@\xa9\xaa\xc2\xf5\xa8\xf8\xe7\x88\xcaX\xb0]M4x\xe3\x83\xe71\x919\xafG\x82\xda#\xc34\x89\x89v?\xcd\x9e\xb6\xfc\x82\xce\xe1\x17\x9e\xc6\x12z\xa7k\xde\xf8\x88\xf2SI\x1b\x82\xc1\xc0\xbb\xb9\x0e\x0c\xcbB\x03\x03\xf5!2\x02[\xa6o\xdfsm;@\xd0G\xfd\xc1*i\xb9\xb2<\x97\xda\xb5\xcd\xd5\x1f\xdf\xff\xfe\xf5\xc7\xf7\x7f\xce\x90\xab\xe6\xfa\xd9\x8e|#\xdcqQ\x8d\xb4)K\xa4!\xfd~\xe0\xd9!\xea\x1b}\x00\x07\x06\xbc\t|\xe3z\xe0\xb9\xc6\xc0u'
    b' \x0c\xfb\xe8:tn\xbf\xa93"\x00\xbb\x11#\xe5\xc1\xd3\xefq}d\x05\xe0\xabC\xab,\x89\x18\xe5t,\xdeE4\xabN\xbf\xcc\x82\xbe\x10V\xd0\xa4<\x00\x03Vud5\xc3\x92]\x83\xc0\x02.\xf2\x1d\xaf\x8a\x92T\xad\xbe\x97\xca\x9a\xcbs\xab2CR\xf6\x11\x17\x0f\xb32G\xb2\xf25\x17\x96SE\x92?W)\xd2\x88\x98k\x07~W\xff\x02PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00ppt/slideLayouts/_rels/slideLayout5.xml.rels\x8d\xcf\xbd\x0e\xc2'
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX-\xb4&\xf5\x12\x03\x00\x00\xb8\x08\x00\x00!\x00\x00\x00ppt/slideLayouts/slideLayout6.xml\xb5V\xddn\xda0\x14\xbe\xdfSX\xd9\x05W\xa9\x93\x10'
    b' \xa0\xc1DB3MjG5\xda\x07\xf0\x12\x03\xd1\x1c\xdb\xb3\r\x83M\x95\xf6Z\xdb\xe3\xf4Iv\xec\x10\xca\xbaN\xea\x05\xbb\x89\x9d\xe3\xf3\xf3\x9d\xef\x1c\xe7\xe4\xcd\xdb]\xcd\xd0\x96*]\t>\xee\x84\x17A\x07Q^\x88\xb2\xe2\xabq\xe7\xee6\xf7\x93\x0e\xd2\x86\xf0\x920\xc1\xe9\xb8\xb3\xa7\xba\xf3v\xf2\xea\x8d\x1ciV^\x91\xbd\xd8\x18\x04.\xb8\x1e\x91\xb1\xb76F\x8e0\xd6\xc5\x9a\xd6D_\x08I9\x9c-\x85\xaa\x89\x81W\xb5\xc2\xa5"_\xc1u\xcdp\x14\x04}\\\x93\x8a{\x07{\xf5\x12{\xb1\\V\x05\x9d\x89bSSn\x1a\'\x8a2b\x00\xbe^WR\xb7\xde\xe4K\xbcIE5\xb8q\xd6\x7fB2{I\xc7\x9e\xa9\x0c\xa3s\xce\xf6\x1er\xaaj\x0b\xc2\xd0\x9b@\xf6\xc5\x82\x95\x88\x93\x1a\x04\xb7V\x0b95{\xa2\xe5\xad\xa2\xd4\xee\xf8\xf6\x9d\x92\x0by\xa3\x9c\xc1\x87\xed\x8dBUi\x1d\x1c\x0c=|88\xa8\xe1\xc6\xc8m\xf0\x13\xf3U\xbb%\xa3\xddR\xd5v\x05.\xd0n\xec\x05\x1e\xda\xdb\'\xb62\xba3\xa8h\x84\xc5\xa3\xb4X\xcf\x9f\xd1-\xd6\x97\xcfh\xe36\x00>\tj\xb3j\xc0\xfd\x9dN\xe4\xfd\xc1Cx\xcc\xaa\xc5\xab\xe5\x95(>k\xc4\x05\xe4c\xd3o\xd2;j49\xdbU\xaeO\x89\xf7Z\x1a\xec!>\r\xae[\xb2\xcc.\x15\xe5\xde\x06\xf9\x04\xab\x13\x92\x11\xd3fa\xf6\x8c\xba\x17i\x1f\x0e\x86\x02\xbc\x8c@[{\x94\xfbw\x0b\x0f\xe9\xdad\x8c\x12~$\xc4L2V\x15\x9f\x91\x11\x88\x96\x95A\xd7D\x1b\xaa\x90\x03\x03\x97\x00\\Zv\x8c\xe3\xc8\xb9\xa4\xbc\xbc!\x8a||\xe2\xb9aQ:\xd0-B\xdcR\xf8o"\xbb-\x913b(\xbaa\xa4\xa0k\xc1J@\x10\x9d\x83\xd3\xd2@\xca\xdf\xe0Z\x10\xb6\xf4'
    b'  \xd4=\x0c\xce\xc7\xf1\x12\xee\x83\xcd\xe2{/\xcd\xa6\xb3'
    b" \xe9\xf9\x97\xc9\xb4\xef'I\xdc\xf3\xd3Y|\xe9gi\x9ce\xd3\xa07\xcc\xb3\xf8\xbe\xbda%\xa4j\xaa\x9a\xe6\xd5j\xa3\xe8|c\xbc\x97\x96*\xc4\xd1\x00\x87\xdd\xc7\x8a\x00\x80\xf3\xd7$nk\x92\x0ba{\xe1\xb4*\xddsTeiTS\x96/\x1b\xa2"
    b' B[\x99\xf0|\x959/#\xbd\x96\x91\x05\xabJ\x8a>l\xeaOOx\x89\xcf\xc1\x0bL\x17p\xfd,5\xd1\x7fh\xda,\xcc\xf3\xfel:\xf4\x83'
    b' \x81\x99\x97\xc6\x89?\x8c\xa0}\xd3~/\x8a\x86I<H\xd2\xfc\xd8\xb4\xdaf\xce\x01\xddK{\xf5\xe1\xc7\xcf\xd7\x0f?~\x9d\xa1W\xf1\xe9|\x81\x8f\xfd\x956\x87\x1d\xda\xa8\n\x12I\xd3a?\xca\x92\xd4O\xc38\xf7\xe3\xd9p\xe0O\xf3~\xcf\xcf{\xdd8\xce\xd2d\x9au/\xef\xed\x9c\n\xe3Q\xa1\xa8\x1b}\xef\xcbvh\x86\xf1_c\xb3\xae\n%\xb4X\x9a\x8bB\xd4\x87\xf9\x8b\xa5\xf8J\x95\x14\x95\x1b\xc1ap\x18\x9a[\xc2\xc6\xde'
    b' \x1a\x04\xd1`pl`\x80\xd6\xae\x0e,nf\xa7\xeb\x10\xa6\xae\x89\x9co]\x8f\xd4\xeec\x9b9\x91\x84_\x84C\x8b<\xaa\xe0\x93_\x8e\xc9oPK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00ppt/slideLayouts/_rels/slideLayout6.xml.rels\x8d\xcf\xbd\x0e\xc2'
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\xeb\x17\x9fw\xe6\x02\x00\x00g\x07\x00\x00!\x00\x00\x00ppt/slideLayouts/slideLayout7.xml\xb5U\xd1n\x9b0\x14}\xdfW'
    b" \xf6\x90'j"
    b' $\x85\xa8I\x15H\x99&um\xb4\xb4\x1f\xe0\x82IP\xc1\xf6l\'K6U\xeaom\x9f\xd3/\xd9\xb5\x814k;\xa9\x0f\xd9\x0b\xb6/\xf7^\xdfs\xce\xd5\xf5\xd9\xf9\xb6\xae\xac\r\x11\xb2dt\xdc\xf3N\xdc\x9eEh\xc6\xf2\x92.\xc7\xbd\xdb\x9b\xd4\t{\x96T\x98\xe6\xb8b\x94\x8c{;"{\xe7\x93\x0fg|$\xab\xfc\x12\xef\xd8ZY\x90\x82\xca\x11\x1e\xdb+\xa5\xf8\x08!\x99\xadH\x8d\xe5\t\xe3\x84\xc2\xbf\x82\x89\x1a+8\x8a%\xca\x05\xfe\x0e\xa9\xeb\n\xf9\xae;D5.\xa9\xdd\xc6\x8b\xf7\xc4\xb3\xa2(32c\xd9\xba&T5I\x04\xa9\xb0\x82\xf2\xe5\xaa\xe4\xb2\xcb\xc6\xdf\x93\x8d\x0b"!\x8d\x89\xfe\xbb$\xb5\xe3dl\xdfU\x98\xde\xdb\x96q\x13\x1b0x\xf6\x04\x90g\x8b*\xb7(\xae\xc1\x10\x1b\x0fm\x94\xfcF\x10\xa2wt\xf3I\xf0\x05\x9f\x0b\xe3{\xb5\x99\x0b\xab\xccul\x1bc\xa3\xf6G\xeb\x86\x9a'
    b' \xb3A/\xc2\x97\xdd\x16\x8f\xb6\x85\xa8\xf5\n\x14X\xdb\xb1\xed\xda\xd6N\x7f\x91\xb6\x91\xad\xb2\xb2\xc6\x98=[\xb3\xd5\xf5\x1b\xbe\xd9\xea\xe2\ro\xd4]\x80\x0e.\xd5\xa8\x9a\xe2^\xc3\xf1;83\xac\x885\xafpFV\xac\xca\x89\xb0\xbc=\xc0\xaet\xc9/Yv/-\xca\x00\x9af\xa2A\xba\xf7h\xe0\xeb\x95\xafZ\xease[\xf2\x07\x88\x88\xab\xc2\x86\x0b\xa1\\\xcf\xb5;\x86\xb43:\xacKv<\xaam\xcc\xf2\x9d\xbe\xf4\x0eVc\xc4\xa3J\xaa\x85\xdaU\xc4\x1c\xb8\xfe\x14\xa0\xa0F\xf1s\x10\'\xd3\x99\x1b\x0e\x9c\x8bp:t\xc20\x188\xf1,\xb8p\x928H\x92\xa9;\x88\xd2$x\xe8\xfa!\x07\xa8\xaa\xacIZ.\xd7\x82\\\xaf\x95\xads\t`\x04\xda`9\xb6\tun\x17Pw\xad\x92\x8a`\xba\xa7\\M<\xe4\x9f"\xaf\xafiV\x86l(\xc0\x08G\xf39\x16\xf8\xeb\x8b\x14\x8d'
    b" \xdc\x80\xec\x10\xa1N\x8d\x7fk\xd2\xef4I\x19S\xa0\xc4\xa1*\xfe1T)\x94hd\xf9\xb6\xc6\x02n\xe8\x94\xf1\x8e\xa7\xccq\x19\t:F\x16U\x99\x13\xebj]\xdf\xbd\xe0\xa5\x7f\x0c^`\x16B\xea7\xa9\xf1\xffC\xd3&^\x9a\x0eg\xd3\xc8q\xdd\x10&t\x1c\x84N\xe4C\xfb\xc6\xc3\x81\xefGap\x1a\xc6\xe9\xbei\xa5FN\xa1\xba\xf7\xf6\xea\xd3\xe3\xaf\x8fO\x8f\xbf\x8f\xd0\xab\xe8p,\xc2\x8c\xba\x94\xaa\xddYkQ\x02\x908\x8e\x86~\x12\xc6N\xec\x05\xa9\x13\xcc\xa2Sg\x9a\x0e\x07N:\xe8\x07A\x12\x87\xd3\xa4\x7f\xf1\xa0\xc7\xab\x17\x8c2A\xcc\xa0\xfe\x9cw#\xde\x0b^\r\xf9\xba\xcc\x04\x93\xacP'\x19\xab\xdb\xd7\x02q\xf6\x9d\x08\xceJ\xf3`xn;\xe27\xb8\xd2\xf2x~\x14E\xa1\x17\xb62Am\xddj\xaaE\xcd\xb87-R\x89/\x98_oL\x93\xc0e"
    b' rbL\x1c^\xb4\xb6G\x9e]\xd0\xc1\x0b9\xf9\x03PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00ppt/slideLayouts/_rels/slideLayout7.xml.rels\x8d\xcf\xbd\x0e\xc2'
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\xcd\xca\x8a\xd5\xb2\x04\x00\x00\xc2\x12\x00\x00!\x00\x00\x00ppt/slideLayouts/slideLayout8.xml\xcdX\xddr\xa36\x18\xbd\xefS0\xf4\xc2W\x04\x04\xe2/\xb3\xce\x8e!\xa1\xd3\x99l\x92Yg\x1f@\x01\xd9\xa6\x0b\x88J\xb2\xd7ngg\xf6\xb5\xda\xc7\xd9\'\xa9$\xc0v\x1c\xc7\xc6\x89/zcd\xf9\xe8H\xdfw>\x1da}\xf8\xb8,\x0bm\x81)\xcbI5\x1c\x80\x0bk\xa0\xe1*%Y^M\x87\x83/\x8f\x89\x11\x0c4\xc6Q\x95\xa1\x82Tx8Xa6\xf8x\xf5\xcb\x87\xfa\x92\x15\xd9-Z\x919\xd7\x04E\xc5.\xd1P\x9fq^_\x9a&Kg\xb8D\xec\x82\xd4\xb8\x12\xbfM\x08-\x11\x17_\xe9\xd4\xcc(\xfa&\xa8\xcb\xc2\xb4-\xcb3K\x94Wz;\x9e\xf6\x19O&\x93<\xc5\xd7$\x9d\x97\xb8\xe2\r\t\xc5\x05\xe2b\xf9l\x96\xd7\xacc\xab\xfb\xb0\xd5\x143A\xa3F?_\x12_\xd5x\xa8\x93\xa7?\x1e\x97\xba\xa6`t!:\x80~%"O\xc7E\xa6U\xa8\x14\x1d1\xa9\xb8`\xd0\xbe\xe5|\xa6\xc5\xa8\x96L\n\xc3\xeaG\x8a\xb1lU\x8b\xdfh=\xae\x1f\xa8\x1az\xb7x\xa0Z\x9eI\xaa\x96B7\xdb\x1fZ\x98\xd9\x0cR\rsg\xf8\xb4k\xa2\xcb\xe5\x84\x96\xf2)2\xa2-\x87\xba\xa5k+\xf9i\xca>\xbc\xe4Z\xdat\xa6\x9b\xdetv\xbf\x07\x9b\xcen\xf6\xa0\xcdn\x02skR\x19U\xb3\xb8\x97\xe1\xd8]8\x8f9/\xb0\x06\xd6Qu\xebe\xf5-I\xbf2\xad""\x1e\x19~\x13\xde\x1a\xd1\xc4,\x9f\xf5\xacM?\x97Tz\x97\x06\xf9\xa3\xb9=9\xdb\x9f\t\xe8\xfaBH\x15\xa2\xed;\x96\xbb\x93\x13\xc7\xb2\x02\x078M\xac\x00xv\x8b\xd8\x8e\x98\xb53\xf0eD\xb2\x95\x1c\xfd$\x9e"RT\xa53"\n\xf5\xa9\xe1,\x18\x1f\xf3U\x81U{Q\x80ZB\x8ai5\xd4\x0b]\xf6ex\xf2Yt\xb1\xbf\xc4R,\xb9\xa6\xa7.\xf05\xbeio\xf1\xd4\xf2C\xc5E\xc5\xd0\x02\x89}\xa8\xe3\xca\xf82\xd65V\xf2\xb8\xc0\xa8Zk\xc7\xaf\xe2"O\xbfj\x9ch8\xcb\xb9\xf6\t1\x8e\xa9\xa6\xf2&v\xad`\x94\xec\\\xcd\xa1(q\x95='
    b' \x8a>\xef07+\xaaU\xec]\xccf\xa7\xf6\xeb\x9a;\xfa\xce.x(P\x8ag\xa4\xc8\xc4"\xec\xf7U@\x9e-7\x90\xfe\xe2;\xae\xefJA_S\xdf\x05\x00\xf8n[\xe9n\xe0:@\x94BO\xf5_\x93|GiGV\xdf\x8e\xc6\xaai\xbf\xc4\xda\xc16\xd6\xde`\x9d=X\xb8\x8du6X\xb8\x07kmc\xe1\x06\xeb\x1e\xc3\xba\x1b\xacw\x0c\xebm\xb0\xfe1\xac\xbf\xc1\x06\xc7\xb0\xc1\x06\x1b\x1e\xc3\x86\xaf\xee!\xb9\x19\x05`\xbdY\xde\xb9\xa7d\x05\xa9-\xc5\x9e\xed)\xb3\x9b\xed\xd9\x94\xe0\xc4)\xc78%U\xa6\x15x\x81\x8b\x1e\xf4\xf6\x89\xf4\x8f\xb3\x9c\xf6gwNdO\xc8\x9c\x8a\xd3\xaf/=<\x95>\x9f\xece?\xb7\x9b\xc1\xf5\t&\xa5\xde\xb62\xe7\x1c\x87\x99\xf4\x10]\x15\xf0\x0c\x15\x13\xbd18\xfb=\xa7\x1b\x80\x8e\x0b\xacC\xc7\x1b\xf4B`y\xef68\xadD\xf4V\xbd\x1f\xe4U&|^6\xd5\xa8\xf9\x9dx\'4w\xf6\'\x80\x07\xfc\xaf\xa5\xea\xa2\xe8\xc5g\x1f\xf0\xc8\x96/\x04\x10\xf6\xe6\xb3\x0e\xf8h\xcb\x07\x1c\x1fx}\t\xc3\x03^\xdb\xf1\x05v\x10\xbc\x89o\xc7\x8f[>\xdb\x0e<\xebM|;\x9e\xdd\xf1\xf9\xd0\xe9-Hx\xc0\xd7[>I\xd6[\x90\xf0\x80\xf7w|\x9e\xeb\xbfM\x8f\xff\xc7\xf9p\x9a\x13\xb9\x9d\x13]#\x8e\x9f9\x11<\x87\x13e\xfc\x85\x0f\x01\xeb\xb0\x11\x99G\xed\xc2\\\xe7u"\xfe\x1c\xc9(\xfev\xa3xtm\x05\xaeq\x13\x8c<#\x08\xa0kD\xd7\xf0\xc6\x88#\x18\xc7#\xcb\r\x93\x18~\xef\xfeje"T\x9e\x978\xc9\xa7s\x8a\xef\xe7\\\xef+\x070m\xdf\x04\xce&\xebb\x01\xe7?\x1d\xbcN\x93\x84\x10\xa9\xf7\xb6*\xee9T\x99p\xda\xc8\xf2\xe7\x1cQ1C\xa7\xcc\x91w\xe0S\x949oF\xfc.#\xe3"\xcf\xb0v7/\x9fv\xf2\xe2\x9d#/\xac\xc8\x04\xf5\xde\xd4\x1c9=\xdfT\xb41H\x12\xefz\x14\x1a\xe2\x1cM\x8c'
    b' \x82\x81\x11\xda\xa2|#\xcf\xb5\xed0\x80~\x10%\xeb\xa2e2\xf2J\xac\xaeo\xad\xfe\xfc\xf1\xcf\xaf?\x7f\xfc{\x86Z5\xb7\xaf\x18\x84\xf7\xdc2\xde\xb6\xb49\xcdE'
    b' Q\x14zv\x1cDF\x04`b\xc0\xeb\xd07F\x89\xe7\x1a\x89\xeb@\x18G\xc1(vn\xbe\xcb\xab\n\x00/S\x8a\xd5\x1d\xc8\xefYw{\x02\xe0\x8b\xfb\x932O)ad\xc2/RR\xb6\x171fM\xbeaZ\x93\\\xdd\xc5\x00\xab\xbd=Y'
    b' \xf9\x0e\x1c@\xcb\xb7=\xd7\xeb\xbcE\xac\xad{\xaa\xd5\x9a\xcdM\x8a*\x91\x82~B\xf5\xfdB\x15I\xa9\x1c5V]u^M\xdb\x1a\xd9@\xcc\xad\xcb\xa7\xab\xff\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00ppt/slideLayouts/_rels/slideLayout8.xml.rels\x8d\xcf\xbd\x0e\xc2'
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdXZ\xd3\xb4\x92y\x04\x00\x001\x12\x00\x00!\x00\x00\x00ppt/slideLayouts/slideLayout9.xml\xbdX\xddr\x9b8\x14\xbe\xdf\xa7`\xe8\x85\xaf\x88\xf8\x11'
    b' 2u:\x06\xc7;;\x93&\x99&}\x00\x05d\x9b)\x7f+\xc9\x8e\xbd;\x9d\xe9k\xed>N\x9f\xa4\x92\x00C\x9c\xa4a]fo\x8c,\x8e>\x9d\xef\x9c\xa3OB\xef?\xec\xf2L\xdb\x12\xca\xd2\xb2\x98N\xac3s\xa2\x91".\x93\xb4XM\'\x9f\xef\x17\x06\x9ah\x8c\xe3"\xc1YY\x90\xe9dO\xd8\xe4\xc3\xc5o\xef\xabs\x96%Wx_n\xb8&'
    b' \nv\x8e\xa7\xfa\x9a\xf3\xea\x1c\x00\x16\xafI\x8e\xd9YY\x91B\xbc[\x964\xc7\\\xfc\xa5+\x90P\xfc(\xa0\xf3\x0c\xd8\xa6\xe9\x81\x1c\xa7\x85\xde\x8c\xa7C\xc6\x97\xcbe\x1a\x93y\x19orR\xf0\x1a\x84\x92\x0cs\xe1>[\xa7\x15k\xd1\xaa!h\x15%L\xc0\xa8\xd1O]\xe2\xfb\x8aL\xf5*\x8d\xefw\xba\xa6\xcc\xe8VtX\xfa\x85`\x1e\xdfe\x89V\xe0\\t\xdc\xa61\xdfP\xa2=\xa6|\xadE\xb8\x92H\xca\x86U\xf7\x94\x10\xd9*\xb6\xbf\xd3\xea\xae\xba\xa5j\xe8\xf5\xf6\x96ji"\xa1\x1a\x08\x1d4/\x1a3P\x0fR\rp4|\xd56\xf1\xf9nIs\xf9\x14\x11\xd1vS\xdd\xd4\xb5\xbd\xfc\x05\xb2\x8f\xec\xb8\x16\xd7\x9dq\xd7\x1b\xafo^\xb0\x8d\xd7\x97/X\x83v\x02\xd0\x9bT\xb2\xaa\x9d{N\xc7n\xe9\xdc\xa7<#\x9au`\xd5\xfa\xcb\xaa\xab2\xfe\xc2\xb4\xa2\x14|$\xfd\x9a\xde\xc1\xa2\xe6,\x9f\xd5\xba\t?\x97Pz\x1b\x06\xf9\x12\xf4\'g/G\xc2\xf2\x03\xdbFHq\x84H\xa4\xd4<\x8a\x8a\x0b\x91\x07\xcd\x86\xad\xeby\xbe\x83\x8e)\xb3f\n\xbe\x0b\xcbd/\x07?\x88\xa7\xa0\x8a\x8bx]\x8aJ}\xa8!3\xc6\xef\xf8>#\xaa\xbd\xcd\xacJ\x9ad\xabb\xaag\xba\xecK\xc8\xf2\x93\xe8b\x7f\x89\x00\x99r\xca\x87\x96\xf9\xc1\xben\xf7p*\xf9\xa3\x88Q14\xc3b!\xea\xa40>\xdf\xe9\x1a\xcby\x94\x11\\\x1c\x92\xc7/\xa2,\x8d\xbfh\xbc\xd4H\x92r\xed#f\x9cPM\x05N,[\x81(\xd1\xb9\x9aCA\x92"\xb9\xc5\x14\x7f:B\xae=\xaa\x14\xf7\x963h\xd3\xfdz\xd2\x1d\xfdh\x19\xdcf8&\xeb2K\x84\x13\xf6\x18%'
    b" V\xa0.\xa6\xdau\xd6\xa7\x15\x82g\xd9\xbe\xef\xfe\xa4\x0e\xa0e\xc9b\x19Z\x08\xaff?\xc7\xf4J-\xa5\xb4H\x84\xb4\xc8\xa6\x1a\xb5\xb9\x16\xf2\t\x8ej\xc2\xb1\x0f3\x1e\xaaA5\xed\x0e\n\xba\xbe\xb4\x1a\x84g\xa3>\x9e\xdd\xe19\x1d^`A8\x18\x0f\xf6\xf1\x9c\x0e\x0fvx\x96\xe3[\xde`@\xb3\x0f\x08;@\xb7\x07\x88D\xd2N\x03t;@\xaf\x03\x14E\xe0\x99\xa7\x01z\x1d\xa0\xdf\x03\xf4\xa13<'O\x00\xfd\x0e\x10u\x80\x12mxR\x9e\x00\xa2\x0e0\xe8\x01z\xae\x7fbR\x82W5i\\\xed\x80\x87\rC\xae\xc7\xbep8c\x08\x87\\\xa6\xba\xa2\xb7\xc6\xd9\xb2\xd1\x10\xfb\x974\xc4u\xc4VQ\xef\x15\xaf\x88\x082\xc5?\xfb\xff\xd5\x10\x0b\x8e\xab!\x96=\xae\x86X\xe6\xc8\x1a\x12\x8c,!\xc1\xc8\n\x12\x8c,"
    b' \xc1\xc8\xfa\x11\x8c,\x1f\xc10\xf5\x90\xe8\xc2\xe0pt\xf9\xc5\x13\x8e\\\x7f\xea\x80\xc3\x9e\x9cpNQ"\xb7U\xa29\xe6O\x8f0p\x0c%J\xf83\x1d\xb2\xcc\x9f\x0b\x11xS.\xc0!\xaeK\xf1-"Y\xfc\xed\x86\xd1ln"\xd7\xb8D3\xcf@\x08\xbaF8\x87\x97F\x14\xc2(\x9a\x99n\xb0\x88\xe0\xd7\xf6\xcb&\x11Ty\x9a\x93E\xba\x12\xe7\xb6\x9b\r\xd7\x87\xa6\xc3\x02\xb6\x0f,\xa7\x8b\xbap`\xfc\xdd\xc1ks\xb2(K\x99\xef~V\xdc1\xb2\xb2\xe4\xb4N\xcb\x9f\x1bL\xc5\x0cmf\xde8f\xfe\x97\xcc\x8c\x1b\x11\xbf\x8d\xc8]\x96&D\xbb\xde\xe4\x0fGq\xf1\xc6\x88\x8b\xf8\xaa\x17\xd0/\x86\xe6\x8d\xdd\xf3\xa4\xa2\x8d\xac\xc5\xc2\x9b\xcf\x02\xc34\xd1\xc2@!DF`\x8b\xf2\r=\xd7\xb6\x03\x04}\x14.\x0eE\xcb$\xf3Bx7\xb4V\xbf\x7f\xfb\xe7\xdd\xf7o\xff\x8eP\xab\xa0\xffE/\xb4\xe7\x8a\xf1\xa6\xa5mh*\x88\x84a\xe0\xd9\x11\n\x8d\xd0\x82\x0b\x03\xce\x03\xdf\x98-<\xd7X\xb8\x0e\x84Q\x88f\x91s\xf9U\xde\x0cX\xf0<\xa6D]9\xfc\x91\xb4\x97\x15\x16|v]\x91\xa71-Y\xb9\xe4gq\x997\xf7\x1e\xa0*\x1f\t\xad\xcaT]}XfsY\xb1\xc5BV\x1d\x84\x02\xdb\xf1\x02\'h\xd2$|k\x9f\xca[P_\\\xa8\x12\xc9\xe8G\\\xddlU\x91\xe4JQ#\xd5U\xa5\xc5\xaa\xa9\x91\xce\x04\xf4\xeez.~\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00ppt/slideLayouts/_rels/slideLayout9.xml.rels\x8d\xcf\xbd\x0e\xc2'
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX7\xc65\xf8\x8d\x03\x00\x00\xcd\x0b\x00\x00"\x00\x00\x00ppt/slideLayouts/slideLayout10.xml\xb5V\xc1\x8e\xdb6\x10\xbd\xf7+\x08\xf5\xe0\x93\x96\x92,{e#\xde\xc0\x92WE\x81MvQ;\xbd3\x12\xbd&B\x89,I;v\x8a\x00\xf9\xad\xf6s\xf2%\x1dR\x92\xd7\xde\xec\x02v\xeb^D\x8a\x1a\xbey\xf3f(\xce\x9b\xb7\xdb\x8a\xa3\rU\x9a\x89z\xd2\x0b\xaf\x82\x1e\xa2u!JV?Nz\x1f\x16\xb9\x9f\xf4\x906\xa4.\t\x175\x9d\xf4vT\xf7\xde\xde\xfc\xf4F\x8e5/\xef\xc8N\xac\r\x02\x88Z\x8f\xc9\xc4[\x19#\xc7\x18\xebbE+\xa2\xaf\x84\xa45|[\nU\x11\x03\xaf\xea\x11\x97\x8a|\x06\xe8\x8a\xe3(\x08\x86\xb8"\xac\xf6\xda\xfd\xea\x94\xfdb\xb9d\x05\x9d\x89b]\xd1\xda4'
    b" \x8arb\x80\xbe^1\xa9;4y\n\x9aTT\x03\x8c\xdb}L\xc9\xec$\x9dx\xa0\x8bYl=\xe4\xec\xd4\x06VB\xef\x06B/\xe6\xbcD5\xa9`a\xc1\x0c\xa7\x08\xf4A\xbf\x831+\x08G\x0b\xba5\xceL\xcb\x85\xa2\xd4\xce\xea\xcd/J\xce\xe5\x83r\xbb\xdfo\x1e\x14b\xa5EkQ<\xdc~h\xcdp\xb3\xc9M\xf0\xb3\xed\x8f\xdd\x94\x8c\xb7KU\xd9\x11TA\xdb\x89\x17xhg\x9f\xd8\xae\x01\tT4\x8b\xc5\xd3j\xb1\xba\x7f\xc1\xb6X\xdd\xbe`\x8d;\x07\xf8\xc0\xa9\x8d\xaa!\xf7c8\x91w$J\xb8\x8f\xaa\xe3\xab\xe5\x9d(>iT\x0b\x88\xc7\x86\xdf\x84\xb7\xb7hb\xb6\xa3\\\xb5)0\x16\xca\xebd\xb0\x1f\xf1\xa1s\xdd\x89e\xb6\xa9(w\xd6\xc9G\x18\xdd\"\x19sm\xe6f\xc7\xa9{\x91\xf6\xe1h(\xe0\xcb\t\x14\xb8Gk\xff\xc3\xdcC\xba2\x19\xa7\xa4\xde\x0bbn2\xce\x8aO\xc8\x08DKf\xd0;\xa2\rU\xc8\x91\x81\xe3\x00\x90V\x1d\xe34r\x90\xb4.\x1f\x88\"\xbf=CnT\x94\x8et\xc7\x10w\x12\xbe.d\xbf\x13\xf2\xa8\xa6\xd0\x03'\x05]\t^\x02\x95\xe8\x12\xe2Z\xa9<$\x14\x83C\xd0T\xbb\x07\xfe\xb7O\x9b\xcfQ\xdc\xfeE\x00\x85\x12K\xda{E\x7fi\x05\xda\xf0\xbd\xd0\xff1\x1fV\x15\x97\x0e}\x94\x0f\xdcy;r\x19\x9e\xe9rN\x0b\x01\xe7\x9a\xd3\r\xe5'\xc0Gg\xc2/VL\x9d\x8e\xde?\x13=\x17keV'\xc3\xc7\xe7\xc2\xb3\xe5\x8b\xe8\x97>\tqw\x12f\xc4\xd0\xa3\x03\xd0\xbf\xc4\x01(\xa1\xe0\xf5\x17\xb8*\x08_v\xa5\x1f\\\xeeo\xb3\x84k\xc2F\xf1\xe7"
    b' \xcd\xa6\xb3'
    b' \x19\xf8\xb7\xc9t\xe8\'I<\xf0\xd3Y|\xebgi\x9ce\xd3`0\xca\xb3\xf8kw\xeb\x94\x10\xaaa\x15\xcd\xd9\xe3Z\xd1\xfb\xb5\xbdLN\xcbJ\x88\xa3k\x1c\xf6\x9f2\x02\x04.\x9f\x93A\x97\x93\\\x08{\n\x0f\xb3\x12_"+K\xa3\x9a\xb4\xfc\xb1&\n<t\x99\xf97\x7f\xa5W2sYE\x86\x9d"s\xceJ\x8a\xde\xaf\xab\x8f\xcft\x19\\B\x17\xe8\xb8\x00\xfaEi\xa2\xff\xa1h\xb30\xcf\x87\xb3\xe9\xc8\x0f\x82\x04\xfa\xc04N\xfcQ\x04\xe5\x9b\x0e\x07Q4J\xe2\xeb$\xcd\xf7E\xabm\xe45\xb0;\xb5V\xbf\x7f\xfb\xeb\xe7\xef\xdf\xfe\xbe@\xad\xe2\xc3N\x0bn\x84;m\xda\x19Z+\x06\x81\xa4\xe9h\x18eI\xea\xa7a\x9c\xfb\xf1lt\xedO\xf3\xe1\xc0\xcf\x07\xfd8\xce\xd2d\x9a\xf5o\xbf\xda\x8e-\x8c\xc7\x85\xa2\xae\x1d\xfc\xb5\xec\x1a\xc90\xfe\xa1\x95\xacX\xa1\x84\x16KsU\x88\xaa\xedI\xb1\x14\x9f\xa9\x92\x82\xb9\xb64\x0c\xdaFrC\xec\xd50\n\x83Qt=\x1a\xc6m\x9a\x80[7:\xb6\xb8\xe9)]\x89p\xf5\x8e\xc8\xfb\x8d+\x92\xca\xdds\x99[\x92\xd07\xb75\xf2d\x82\x0f\xfa\xf0\x9b\x7f\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00-\x00\x00\x00ppt/slideLayouts/_rels/slideLayout10.xml.rels\x8d\xcf\xbd\x0e\xc2'
    b' \x10\x07\xf0\xdd\xa7'
    b' ,LB\xeb`\x8c)\xedbL\x1c\\\x8c>\xc0\x05\xae-\xb1\x05\xc2\xa1\xd1\xb7\x97\xd1&\x0e\x8e\xf7\xf5\xfb\xe7\x9a\xee5O\xec\x89\x89\\\xf0Z\xd4\xb2\x12\x0c\xbd\t\xd6\xf9A\x8b\xdb\xf5\xb8\xde\tF\x19\xbc\x85)x\xd4\xe2\x8d$\xbav\xd5\\p\x82\\nht\x91XA<i>\xe6\x1c\xf7J\x91\x19q\x06\x92!\xa2/\x93>\xa4\x19r)\xd3\xa0"\x98;\x0c\xa86U\xb5U\xe9\xdb\xe0\xed\xc2d\'\xaby:\xd9\x9a\xb3\xeb;\xe2?v\xe8{g\xf0\x10\xcccF\x9f\x7fD(\x9a\x9c\xc53P\xc6TXH\x03f\xcd\xa5\xfc\xee/\x96jY"\xb8j\x1b\xb5x\xb7\xfd\x00PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\xe8\xe4I\xd19\x03\x00\x00\xb3$\x00\x00(\x00\x00\x00ppt/printerSettings/printerSettings1.bin\xedY\xcfn\xda0\x18\xcfz+o\xb0[\x96;1PV\xd8\x94R1(\x1a\x12m\xa3\x12*\xedT\xb9\x89\xcb\xdc\x868r\xcc\x18{\xa4\xbd\xdf\xees\x02\x01\x130\x84\x1d\xd6$\xea\xa1Up\xec/\xbf?\xf6\x17\xfb\xcb\x89\xa2(\xef\xf8\xdf\xef\xf7\x8ab\\\xfe\x9c\xb8\xea\x0fD\x03L\xbc\x0b\xad\xaaW4\x15y6q\xb07\xbe\xd0FV\xaf\xdc\xd4.[%\xe3C\xf7\xb6c}3\xafT\xdf\xc5\x01S\xcd\xd1\x97A\xbf\xa3je\x00\xda\xbe\xef"\x00\xbaVW5\x07\xfd\xa1\xa5\xf2\x18\x00\\\xddh\xaa\xf6\x9d1\xff3\x00\xb3\xd9L\x87a/\xdd&\x93\xb0c\x00LJ|D\xd9|\xc0\x83\x95\xf9\x00\xdda\x8e\xc6\x1f\xb3\x88\xbe\x01\x87\xb7:\xd8f\xad\xd2\xa9\xf1\x82\xe6-\x1eb\x19\xcc\xa7\xd8c\xba\t\xc7\xa8G\xe8\x04\xf2\xcb\xeb\xaf\x84\xe2_\xc4c\xd0\xbdC\x81\x01\xc2\xfe|\xd8r\xf8\xee\xf1\x0c\xdb/\x88\xe96E\x90\x11\x1a\x8f95\x02\xc6o\x8f\x85\xee\xcf\xe4q\xd1\xd7\x00\xcb{\x07Bb\x86&mJ\xe1|\x1d\x14\x86?\xc3\xab5(I\x8c\xc3\xb4\xc2\x91\x1c\xb4\xdbj\xd4\x0c\x10]\xc8\xa3-\x11\x05\x0c2\xd4s\xe1X\x8c\xc1\xef\xa31\xa2\xad\x8a\x01\xe2\xcb\x08'
    b' X\xc9\x06b\xd8\xab\xb6\xc3\x90o)F\x1c0\xe36\x16\xc7\x87\x1d\xa4D\x05\xab\x9b\nf\xc5\x8a\xa1\r].SqlH\x10Z-\x84j\x06\xd7\xc1=\xcfr\xd8.X>\xdaA*\xdb\xd9(\x06\\\xb8\xa5'
    b' !\x96\xb5%\x11L\x1f\xad\xc5s|\xc8\xdf\xfb\x0f\xd8{"\x0f\xb1f\xbb\xbc0\xafM\xb3k\x86};\xc4A7p\x82\xd6R\xad\xf49\xc6\xb5\xb4\xb6\x1d\xe9\x9bh\x9c\xe8\xdcA\x16\x02\xa2\x01b\x0c\xd1\r\x10\xc7{%5KpK\xb0K\xf4p\x85\xd4\xa2\xd0\x0b\xdc\xe8\xf56\x8c\xb0D\xd0s-~\nJ\x02\xbc\xd1P\xcd\x86\x19\x16&c\x98s\xf5%\x1c\x04<\x1e,\xbb\x19\x90{o\x82m;\xcf\xd3\x80!\'l\xbcC6\xcb\xa3\x17\xffF0\x11u\x8f\xca\xfbo-v\x05gu\xf1M\x147\x7fl\x9co4\x0b&ev\x1e\xf0\t]\xf0\x89\x90d\xb8=\x13"\xcb\xca\xd5\xe6.O%\xcd\x8d\xc6\xee\x19\xf0\xa9\x9e\xe5\x19\xc0\xa5\xe8\xf3\xbd\n\x97\'\xd7\xd9\xf88byH\xd1k\xfc#\x0f\x162G\xa7b\xf8\x96\xa4%:\x15*K\xa7\xa3\xf8\x96\xa6u\xdfw\x8a\x9b\xaae\xe4\x04\xa0Y8\xbdH\x9e\xd46\xfb\xf7\x8b\xb2\xfeV%\xa5R\xd1k\x95\xb4\xb5\x136\xf7\xd1V\x04\xa9hV\xd2\xac\xfdu\n)V\x19\xd4\xb4H\xe5@\xe3\x1dX\x12i\x0c\xd4\x00\xd17\x91V\xe9DQ\x94?\xa5\x02|\xb1\xe9\x12{:A\xde\x92qX\xcf\xf5\tq\x17*\xe4\xba2\x97\x86\x98\xb0X\xc3\xa1\xd8\x8ej\x13\xc0w\x9e6W\xed+\x16N\xc3\xffC\x9eH8\x96\x80\x93\xe8\x10\x1f\xe7\xbdx\xbd\x97\x92\xa8^\x86>\xe1lc\x9ew\x88\xeb\xf2g\x16\xcd\x8b$\xafp(\xa3S\x04\xb2\xe6A\x0f\xd3\x80\x85)\xbbP\x0el\xb1\xca\xc7\x82\x18\xc0\x02z\x91$%*X\xab\xd6\x1b\xf5\xe6\xd9y\xbd\x91YO\xa2\xf3)\xf4\nf\xca\x16\xab\xe4IK\xbaZ\xd2\x98\'\x9e\xa4^\xcf\xc9\xff\xbf\xf3\x15E>\xb8\xf9\xfd\x0bPK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x16\xcb\x03`\x89\x01\x00\x00\xd1\x03\x00\x00\x15\x00\x00\x00ppt/slides/slide1.xml\xadS\xdbj\xc30\x0c}\xdfW\x98\xbc\xe4iu\xdb\xc1\x18\xa5ia\x1d\xdb\xcb.\x85v\x1f\xe0\xc5jc\xe6\xd8\xc2\xf2\xb2\xf6\xefg;\xc9Zv\x81\xc2\x06A\xbe\xe8\xe8\x1cI\x91\xa7\xf3]\xadY\x03\x8e\x945E>\x1a\x0cs\x06\xa6\xb4R\x99m\x91?\xafo\xcf\xafrF^\x18)\xb45P\xe4{\xa0|>;\x9b\xe2\x84\xb4d!\xd8\xd0D\x14Y\xe5=N8\xa7\xb2\x82Z\xd0\xc0"\x98\xe0\xdbXW\x0b\x1f\x8en\xcb\xa5\x13\xef\x81\xb4\xd6|<\x1c^\xf2Z(\x93u\xf1xJ<:'
    b' 0^\xf8\x90\xe8O$\xee\x14\x12\xbb\xd9\xa8\x12nl\xf9V\x07\xae\x96\xc4\x81N\xa4T)\xa4l\x16*+WZ\xc6\x95p\xed\x00\xe2\xce4w\x0eW\xb8t\xc9\xfd\xd8,\x1dS\xb2\xc8F\x193\xa2\x86"\xcbx\xe7\xe8`\xbc\rJ\x1b\xfe%|{\x04!l\x81\xdf\xa9\xc7=\xf5Zy\rl\xf4\xa9\xd0BE\x08\xbd\xb7\xe5+1c\x03wL\xa5\x95\xfaD\xb4\xfaq\xc5\x8a\xf9=\x06*\x1f\xa9\xb2>\xa5\xe8\xe4\xc7\xe2\xd4g\xe5w\xd7V\xee\xa3\xc8KX\xd3\xa5\x98h\xf2+\xbf\xd7\x90\x0e\x18MJ\xc3\xcfV\xa2\xc6\x90\xe1\xf2\xe8\xffLytD\xeb\x92\xc5\xa4\xd4\xd3\xf2\xbe\xee\xdf\xab\xbf\xe8\xab_X\xe3\x03+[jQBe\xb5\x04\xc7\xc6\x7f\xeb\x85\x92\xbb\x03\xe4\xff\xda\xb0\xae\x14\xb1\xf0\tFmCH+\t\x83\x93Z\xc1\x0f\xb3\xc6\x0f\xe3Wj\xf7'
    b' \xf0\xa9I\x02a\xa0=\xb8E\xba\xc2\xf0\x8e\xba\n\x0f\x10\x9e^\xe4\xec\x03PK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdX\x1eAN\x95\xb7\x00\x00\x006\x01\x00\x00'
    b' \x00\x00\x00ppt/slides/_rels/slide1.xml.rels\x8d\xcf\xbd\n\xc20\x10\x07\xf0\xdd\xa7\x08Y2\x99T\x07\x11i\xdaE\x04\xc1I\xf4\x01\x8e\xe4\xda\x06\xdb$\xe4\xa2\xd8\xb77\xa3\x05\x07\xc7\xfb\xfa\xfd\xb9\xba}O#{a"\x17\xbc\x16\x1bY\t\x86\xde\x04\xeb|\xaf\xc5\xfdvZ\xef\x05\xa3\x0c\xde\xc2\x18<j1#\x89\xb6Y\xd5W\x1c!\x97\x1b\x1a\\$V\x10O\x9a\x0f9\xc7\x83Rd\x06\x9c\x80d\x88\xe8\xcb\xa4\x0bi\x82\\\xca\xd4\xab\x08\xe6\x01=\xaamU\xedT\xfa6x\xb30\xd9\xd9j\x9e\xcev\xc3\xd9m\x8e\xf8\x8f\x1d\xba\xce\x19<\x06\xf3\x9c\xd0\xe7\x1f\x11\x8aFg\xf1\x02sx\xe6\xc2B\xea1k.\xe5w\x7f\xb1\xb4\x95%\x82\xab\xa6V\x8bw\x9b\x0fPK\x03\x04\x14\x00\x00\x00\x08\x00{z\xbdXZ\xa0\x0e\xad\xa3\x05\x00\x00\xe3\x0f\x00\x00\x17\x00\x00\x00docProps/thumbnail.jpeg\xedVkp\x13U\x14>\xbb{7)m\xcd\x10(-\x14\x07\xc2\xbb2\xc0\xa4-B+\x02&i\xda\xa6\x946\xa4-\xafq\x86I\x93M\x13\x9a&aw\xd3\x96N\x9d\x91\xfa@\xfd#\x0f\xdf\x7f,\x05\x15\x1dg\x1cT\xb4\xa0\x8e\xad"\xa0\xa3\x03\x88\x05\n\x0cc\x11\xb5\xf8\x1a\x1e\x8a\xaf\x81x\xeen^@\x10\x94\xbf\xbd;{\xef\x97s\xbe{\xee\xf9\xce\xbds7\x91c\x91\xafaxE\xa9\xad\x14\x18\x86\x81r|'
    b' rZ\xdbm\xb3XV\xd9\x1d\xd5\xa5\xb6J+:\x00\xb4\xdb\x9d\xa1\x90\x9f5\x004\x05d\xd1Qf6,_\xb1\xd2\xa0\xed\x07\x16\xc6@\x06\xe4B\x86\xd3%\x85Lv{%`\xa3\\\xb8\xae]:\x02\x0c\x1d\x0f\xcfL\xed\xff\xd7\x96\xe1\x16$\x17\x00\x93\x868\xe8\x96\\M\x88[\x01x\xbf+$\xca\x00\x9a3h/j\x91C\x88\xb5w"\xce\x121A\xc4F\x8a\x1bT\\Bq\xbd\x8a\x97+\x9cZ\x87\x051\xcdE\xe7\xf2:\xdd\x88\xdb\x11\xcf\xa8O\xb27$a5\x07\xa5e\x95\t\x01A\xf4\xb9\x0c\xb4\x16v1\xe8\xf1\xf9\x85\xa4to\xe2\xbe\xc5\xd6\xe4\x0f\xc7\xd6\x1b\x87o\xa6\xd4X\xb3\x08\xc7<\xaa\xdd+\x96;\xa2\xb8\xd3\xe5\xb4\xd6'
    b' \x9e\x8cx\x7fH6S\xfbT\xc4?\x85\x1b\xebL\x88\xa7\x03\xb0#<bi\x9d\xcag\xefm\xf3\xd6.C\x9c\x8d\xd8\xed\x93m\xb5Q{[\xa0\xbe\xaaZ\x9d\xcbv5\x06\x179\xa2\x9c\xfd.\xc9\x825\x83\x89\x88Oy\x05[\xa5\x9a\x0f\x07n\xa1\xc4J\xeb\x85x\x9c7\\\x1e\x8d\xcfUH\xcd5\xd6X\x9c6\xaf\xa5J\x8d\xc3\x89\xab\x9d\x15v\xc4\xb9\x88\x1f\x13\x83\x8ej5g\xaeK\xf0\x979\xd4\xf8\xdc\xde\x90l\x8f\xe6\xc0\r\x06\xfcU\x95jL\xa2\x13$E\xa3b\x97\xbd\xb5\xe5\xea\\2G\xc6MT\xe7\x92\x95\x1e_\xa9-\xcao\x0f\xf9\x95\xb3\x88\xb9\x91mb\xd8Q\x17\xe5\x1ct\x8a\xd625\x0e\xb9'
    b' \x04\xea\xa21\xf9\xd1ng\t\xad\xed,\xc4\x0b`)\xe3\x04\x01\x82P\x8f\xbd\x0b\x02p\x19\x0c\xe0\x8020\xe3\x18\x02\x11=\x1e\xf0\x81\x1f-\x02z\x05\xb4\xf8\x98;\xa0\x11m\xa9yv\x85\xa3\xe2\x04\xa3A\x99=Hg\xe3*\xa99\xea\ng\xa3\x9c'
    b' \xc9!FR\x88\xef<RI\xe6\x93"R\x0c\x06\xb2\x90\xdcG\x16\x90\x12\xb4\x16\x93y\xf1\xb9\xf6\xa4\xf5\xe9Zg\xe3q\xd6@\x18\xa3R\xdeR0o\xc8\r\xe7%\xd6\xeb\x10W\xf9\\\x07\x9e<w\xd5\xec\xe0u9\x0b\xb1|\x92+\x00\x12V'
    b' \xc6\x9c\x99\xac\x7f_\xfb\xfb\xa3\x131\xfaH\xf7\xfd\x873\xf7\xb5C\xf5\xcd\xea\xcb\x9f\xe1\x07\xf9>\xec\xfb\xf9\x93\t\x06\x7f\x82?\x89O?\x9807\xbf\x92Q\x13\xbe>%\x0f))\x83d\r\xbd\xf8\xca\xe0\xc4~\x00y\xc1$\xdeU+z\x02\x1br\x13\x1eZ\ta}\xd5\xa5*\xe8\x98\x91\xb0\x1a\x8f\x1a\x7f6\xf6\x19\xb7\x18\xb7\x19\x7f\xbc\xa6\xca)\xab\xc4m\xe6vr\x1fp\xbb\xb8\xdd\xdc\xe7`\xe0z\xb8^\xeeCn/\xf7\x06\xf7^\xd2^\xdd\xf8|\xc4\xf7^\xd1\x1bSK=\xa9j-\x80_g\xd6\x8d\xd5M\xd2\x95\xe8\xc6\xeb\xa6\xe8*\x13\xf1t9\xba|]\xb9n\x1az\xc6\xc6\xf7-y\xbdd->X\x81}\xac\xaa\xa9\xd7Ryu\xe8\xf5A\x8b\xa2@R*\x1c\x80\xb5\xd7\x9c\xff\xe8l2\x8e\xe4\x13\xdb5\xa7\xb6\x88\x9e\xe5\x18Cc\xd5\x94hL`\xd0L\xd7\x14k\xf25\x15\x14\xc7\xf2\xd3LC_1\xf6\xd6\xabN\x9d\xeb\x06\n\x84$V\xb2\xce\x99\xca\xa9\xa3g\x95\xcenV|\x12\x08\xb2\xd0*\xd3\x8b\xd6\x12\x0c\xad\x15}\r^\xd9P`4\xce5\x98\xf0S%\x18l\x01\xd7\xac\x19\x06\xa7\xdfoP\\\x92A\x14$Al\x16\xdc\xb3\x80~\x07\xd5+\xfa\xa2C\xf9\xbe1\xd9\x07\x126y1\xc0\xfc_\xf0\xce:\x98\xb0\xad\x0c\x03\xbc.\x01\xe4\xccN\xd8\xf2\xf0N\x1c\xf5"@\xf7\x1cWXl\x8e\xde\xf9\x0c\xf3\x05\x80\xe4),P\x7fe\x9a\xf1n:\x15\x89\\\xc4\xfbJ\xbb\t\xe0\xf2\xc6H\xe4\xef\xaeH\xe4\xf2V\x8c\x7f\x12\xa0\xc7\x1f\x19\x00\xd9\xda\xea\xf3\x00,^Lo}H\x03\xc2\xe4\x02Og\xe3\xbb\x80\x19\x1b\xc7\xa5L\x1e^\xe0\x14\xb3\x00\xeb}@\xa2\xf6\xaa\xe8\xda\xe5\xd1\xdf\xea\xc8v\xb21\x06\x03\x9d\\\x9c\xddC\xa9\x91\x13`\xa1\xff\x1en\xab\xd1'
    b' \xb7\x1b\x83\x89\xf4\x80>\rz\x18\xe0\x18=\xb0z\x86\xd33\x91=0\x1es\xe5UB\xec\xc3\xca\xb0\x1c\xe15\xda\xb4a\xe9\x19H\xd89\x1cX\x86\xe3X\xc2\xf1<Ai\xcc\x03\xe8\x07\xa2\xe7GL\xc87iF.qj\'\xae\xc9*X\xb7qK\xda$\xf3\x8e\xdeQ\x8eC\xe7&\x17\xd6\x8b\x1d\xc3\xd2\xb3sF\x8f\xc9\x9d2uZ\xde]\xd3g\xdf=gnQ\xf1=\x96\x12kiY\xb9\xad\xa2\xa6\xb6n\xe92\xdc^\x97[\xf04x}\xab%9\xdc\xdc\xd2\xba\xb6\xed\xa1\x87\x1fyt\xfdc\x8f?\xb1i\xf3SO?\xf3\xecs\xcf\xbf\xd0\xb5u\xdbK/\xbf\xb2\xfd\xd5\xd7\xde|\xeb\xed\x9d\xef\xbc\xdb\xbdk\xf7G\x1f\xef\xf9d\xef\xbe\xfd\x9f~\xf6\xe5\xe1\xaf\xfa\x8e\x1c=\xd6\x7f|\xe0\xf47g\xbe\xfd\xee\xfb\xc1\xb3?\x9c\xbfp\xf1\xd7\xdf.\xfd\xfe\xc7\x9f\x7fQ]Tg\xac\xa5\xd4\x85E`XB8\xa2\xa5\xba\x18\xb6\x85\x12\xf4\x84\x9f\x90\xaf\x19aZ\xa2u\xae\x199\xb1`]Z\x96y\xe3\x96\x1d\xbd\xc3&\x15:\xce\x8d\xaa\x17\x0f\xa5gO\x9e=0\xe5<\x95\xa6(\xbb5a\x1d\xffKY\\XB\xd7q\xc8\xe4\xf0\xc0\xe99=,\x84+W\xf2\xa0\x93}0\x1d\x86\x86\xa1ah\x18\x1a\x86\x86\xff8D\xfa\xff\x01PK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\xc6\xaf\xc4g\xb4\x01\x00\x00\xba\x0c\x00\x00\x13\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x00\x00\x00\x00[Content_Types].xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\xf1\r7\xec\x00\x01\x00\x00\xe1\x02\x00\x00\x0b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xe5\x01\x00\x00_rels/.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x8b\x14\xfc\xe3y\x01\x00\x00\xdb\x02\x00\x00\x11\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x0e\x03\x00\x00docProps/core.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x9e\xd0\x8ey\xef\x01\x00\x00m\x04\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xb6\x04\x00\x00docProps/app.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x05w\x9c\x0f;\x02\x00\x00\xb4\x0c\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xd3\x06\x00\x00ppt/presentation.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdXR\x9cP\xc9\x1c\x01\x00\x00q\x04\x00\x00\x1f\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01@\t\x00\x00ppt/_rels/presentation.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\\\x9cG\x14D\x01\x00\x00\x89\x02\x00\x00\x11\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x99\n\x00\x00ppt/presProps.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdXg3&\x8d\x9b\x01\x00\x00\x82\x03\x00\x00\x11\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x0c\x0c\x00\x00ppt/viewProps.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x93\nmu!\x06\x00\x00\xe7\x1d\x00\x00\x14\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xd6\r\x00\x00ppt/theme/theme1.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\xd8\xfd\x8d\x8f\xa5\x00\x00\x00\xb6\x00\x00\x00\x13\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01)\x14\x00\x00ppt/tableStyles.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\xa6-\xa25\xee\x06\x00\x00\xd2.\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xff\x14\x00\x00ppt/slideMasters/slideMaster1.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x19\xcb\xf1\xf9\r\x01\x00\x00\xc6\x07\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01,\x1c\x00\x00ppt/slideMasters/_rels/slideMaster1.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdXK\x89PW\xc0\x03\x00\x00\xad\x0c\x00\x00"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x83\x1d\x00\x00ppt/slideLayouts/slideLayout11.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00-\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x83!\x00\x00ppt/slideLayouts/_rels/slideLayout11.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x00\xfd\xec\r*\x04\x00\x00\x05\x11\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x85"\x00\x00ppt/slideLayouts/slideLayout1.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xee&\x00\x00ppt/slideLayouts/_rels/slideLayout1.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x01W\xe8\x8bm\x03\x00\x00\x96\x0b\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xef\'\x00\x00ppt/slideLayouts/slideLayout2.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x9b+\x00\x00ppt/slideLayouts/_rels/slideLayout2.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x8b`\xedZc\x04\x00\x00X\x11\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x9c,\x00\x00ppt/slideLayouts/slideLayout3.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01>1\x00\x00ppt/slideLayouts/_rels/slideLayout3.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdXO\xca\x82\x1c\x08\x04\x00\x00h\x12\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01?2\x00\x00ppt/slideLayouts/slideLayout4.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x866\x00\x00ppt/slideLayouts/_rels/slideLayout4.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\xe9\xa4\xc4\x8f\xe3\x04\x00\x006\x1c\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x877\x00\x00ppt/slideLayouts/slideLayout5.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xa9<\x00\x00ppt/slideLayouts/_rels/slideLayout5.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX-\xb4&\xf5\x12\x03\x00\x00\xb8\x08\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xaa=\x00\x00ppt/slideLayouts/slideLayout6.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xfb@\x00\x00ppt/slideLayouts/_rels/slideLayout6.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\xeb\x17\x9fw\xe6\x02\x00\x00g\x07\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xfcA\x00\x00ppt/slideLayouts/slideLayout7.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01!E\x00\x00ppt/slideLayouts/_rels/slideLayout7.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\xcd\xca\x8a\xd5\xb2\x04\x00\x00\xc2\x12\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01"F\x00\x00ppt/slideLayouts/slideLayout8.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x13K\x00\x00ppt/slideLayouts/_rels/slideLayout8.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdXZ\xd3\xb4\x92y\x04\x00\x001\x12\x00\x00!\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x14L\x00\x00ppt/slideLayouts/slideLayout9.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00,\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xccP\x00\x00ppt/slideLayouts/_rels/slideLayout9.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX7\xc65\xf8\x8d\x03\x00\x00\xcd\x0b\x00\x00"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xcdQ\x00\x00ppt/slideLayouts/slideLayout10.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x80e\xe1\x88\xb7\x00\x00\x006\x01\x00\x00-\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x9aU\x00\x00ppt/slideLayouts/_rels/slideLayout10.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\xe8\xe4I\xd19\x03\x00\x00\xb3$\x00\x00(\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x9cV\x00\x00ppt/printerSettings/printerSettings1.binPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x16\xcb\x03`\x89\x01\x00\x00\xd1\x03\x00\x00\x15\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\x1bZ\x00\x00ppt/slides/slide1.xmlPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdX\x1eAN\x95\xb7\x00\x00\x006\x01\x00\x00'
    b' \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xd7[\x00\x00ppt/slides/_rels/slide1.xml.relsPK\x01\x02\x14\x03\x14\x00\x00\x00\x08\x00{z\xbdXZ\xa0\x0e\xad\xa3\x05\x00\x00\xe3\x0f\x00\x00\x17\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x01\xcc\\\x00\x00docProps/thumbnail.jpegPK\x05\x06\x00\x00\x00\x00&\x00&\x00\xa3\x0b\x00\x00\xa4b\x00\x00\x00\x00'
)


def docx_mock_request(*args, **kwargs):
  del args, kwargs
  return pg.Dict(content=docx_bytes)


def xlsx_mock_request(*args, **kwargs):
  del args, kwargs
  return pg.Dict(content=xlsx_bytes)


def pptx_mock_request(*args, **kwargs):
  del args, kwargs
  return pg.Dict(content=pptx_bytes)


pdf_bytes = (
    b'%PDF-1.1\n%\xc2\xa5\xc2\xb1\xc3\xab\n\n1 0 obj\n'
    b'<< /Type /Catalog\n     /Pages 2 0 R\n  >>\nendobj\n\n2 0 obj\n '
    b'<< /Type /Pages\n     /Kids [3 0 R]\n     '
    b'/Count 1\n     /MediaBox [0 0 300 144]\n  '
    b'>>\nendobj\n\n3 0 obj\n  '
    b'<<  /Type /Page\n      /Parent 2 0 R\n      /Resources\n       '
    b'<< /Font\n'
    b'<< /F1\n'
    b'<< /Type /Font\n'
    b'/Subtype /Type1\n'
    b'/BaseFont /Times-Roman\n'
    b'>>\n>>\n>>\n      '
    b'/Contents 4 0 R\n  >>\nendobj\n\n4 0 obj\n  '
    b'<< /Length 55 >>\nstream\n  BT\n    /F1 18 Tf\n    0 0 Td\n    '
    b'(Hello World) Tj\n  ET\nendstream\nendobj\n\nxref\n0 5\n0000000000 '
    b'65535 f \n0000000018 00000 n \n0000000077 00000 n \n0000000178 00000 n '
    b'\n0000000457 00000 n \ntrailer\n  <<  /Root 1 0 R\n      /Size 5\n  '
    b'>>\nstartxref\n565\n%%EOF\n'
)


def convert_mock_request(*args, **kwargs):
  del args, kwargs

  class Result:
    def json(self):
      return {
          'Files': [
              {
                  'FileData': base64.b64encode(pdf_bytes).decode()
              }
          ]
      }
  return Result()


class DocxTest(unittest.TestCase):

  def test_from_bytes(self):
    content = ms_office_lib.Docx.from_bytes(docx_bytes)
    self.assertIn(
        content.mime_type,
        (
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/octet-stream',
        ),
    )
    self.assertEqual(content.to_bytes(), docx_bytes)
    self.assertTrue(content.is_compatible('text/plain'))
    self.assertFalse(content.is_compatible('application/pdf'))
    self.assertEqual(
        content.make_compatible(['image/png', 'text/plain']).mime_type,
        'text/plain'
    )

  def test_from_uri(self):
    content = ms_office_lib.Docx.from_uri('http://mock/web/a.docx')
    with mock.patch('requests.get') as mock_requests_get:
      mock_requests_get.side_effect = docx_mock_request
      self.assertIn(
          content.mime_type,
          (
              'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
              'application/octet-stream',
          ),
      )
      self.assertEqual(content.to_bytes(), docx_bytes)
      self.assertEqual(content.to_xml(), expected_docx_xml)


class XlsxTest(unittest.TestCase):

  def test_from_bytes(self):
    content = ms_office_lib.Xlsx.from_bytes(xlsx_bytes)
    self.assertIn(
        content.mime_type,
        (
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'application/octet-stream',
        ),
    )
    self.assertEqual(content.to_bytes(), xlsx_bytes)
    self.assertTrue(content.is_compatible('text/plain'))
    self.assertFalse(content.is_compatible('application/pdf'))
    self.assertEqual(
        content.make_compatible('text/plain').mime_type,
        'text/html'
    )

  def test_from_uri(self):
    content = ms_office_lib.Xlsx.from_uri('http://mock/web/a.xlsx')
    with mock.patch('requests.get') as mock_requests_get:
      mock_requests_get.side_effect = xlsx_mock_request
      self.assertIn(
          content.mime_type,
          (
              'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
              'application/octet-stream',
          ),
      )
      self.assertEqual(content.to_bytes(), xlsx_bytes)
      self.assertEqual(content._raw_html(), expected_xlsx_html)


class PptxTest(unittest.TestCase):

  def test_content(self):
    content = ms_office_lib.Pptx.from_bytes(pptx_bytes)
    self.assertIn(
        content.mime_type,
        (
            'application/vnd.openxmlformats-officedocument.presentationml.presentation',
            'application/octet-stream',
        ),
    )
    self.assertEqual(content.to_bytes(), pptx_bytes)

  def test_file(self):
    content = ms_office_lib.Pptx.from_uri('http://mock/web/a.pptx')
    self.assertFalse(content.is_compatible('text/plain'))
    self.assertTrue(content.is_compatible('application/pdf'))
    with mock.patch('requests.get') as mock_requests_get:
      mock_requests_get.side_effect = pptx_mock_request
      self.assertIn(
          content.mime_type,
          (
              'application/vnd.openxmlformats-officedocument.presentationml.presentation',
              'application/octet-stream',
          ),
      )
      self.assertEqual(content.to_bytes(), pptx_bytes)

    with mock.patch('requests.post') as mock_requests_post:
      mock_requests_post.side_effect = convert_mock_request
      self.assertIsInstance(
          content.make_compatible('application/pdf'), pdf_lib.PDF
      )


if __name__ == '__main__':
  unittest.main()
