import {resolve} from 'path'
import type {ConfigEnv, UserConfig} from 'vite'
import {loadEnv} from 'vite'
import Vue from '@vitejs/plugin-vue'
import WindiCSS from 'vite-plugin-windicss'
import VueJsx from '@vitejs/plugin-vue-jsx'
import EslintPlugin from 'vite-plugin-eslint'
import VueI18n from '@intlify/vite-plugin-vue-i18n'
import styleImport, {ElementPlusResolve} from 'vite-plugin-style-import'
import {createSvgIconsPlugin} from 'vite-plugin-svg-icons'
import PurgeIcons from 'vite-plugin-purge-icons'
import {viteMockServe} from 'vite-plugin-mock'
import DefineOptions from 'unplugin-vue-define-options/vite'
import {createHtmlPlugin} from 'vite-plugin-html'

// https://vitejs.dev/config/
const root = process.cwd()

function pathResolve(dir: string) {
    return resolve(root, '.', dir)
}

export default ({command, mode}: ConfigEnv): UserConfig => {
    let env = {} as any
    const isBuild = command === 'build'
    if (!isBuild) {
        env = loadEnv((process.argv[3] === '--mode' ? process.argv[4] : process.argv[3]), root)
    } else {
        env = loadEnv(mode, root)
    }
    return {
        base: env.VITE_BASE_PATH,
        plugins: [
            Vue(),
            VueJsx(),
            WindiCSS(),
            styleImport({
                resolves: [ElementPlusResolve()],
                libs: [{
                    libraryName: 'element-plus',
                    esModule: true,
                    resolveStyle: (name) => {
                        return `element-plus/es/components/${name.substring(3)}/style/css`
                    }
                }]
            }),
            EslintPlugin({
                cache: false,
                include: ['src/**/*.vue', 'src/**/*.ts', 'src/**/*.tsx'] // 检查的文件
            }),
            VueI18n({
                runtimeOnly: true,
                compositionOnly: true,
                include: [resolve(__dirname, 'src/locales/**')]
            }),
            createSvgIconsPlugin({
                iconDirs: [pathResolve('src/assets/svgs')],
                symbolId: 'icon-[dir]-[name]',
                svgoOptions: true
            }),
            PurgeIcons(),
            viteMockServe({
                ignore: /^\_/,
                mockPath: 'mock',
                localEnabled: !isBuild,
                prodEnabled: isBuild,
                injectCode: `
          import { setupProdMockServer } from '../mock/_createProductionServer'

          setupProdMockServer()
          `
            }),
            DefineOptions(),
            createHtmlPlugin({
                inject: {
                    data: {
                        title: env.VITE_APP_TITLE,
                        injectScript: `<script src="./inject.js"></script>`,
                    }
                }
            })
        ],

        css: {
            preprocessorOptions: {
                less: {
                    additionalData: '@import "./src/styles/variables.module.less";',
                    javascriptEnabled: true
                }
            }
        },
        resolve: {
            extensions: ['.mjs', '.js', '.ts', '.jsx', '.tsx', '.json', '.less', '.css'],
            alias: [
                {
                    find: 'vue-i18n',
                    replacement: 'vue-i18n/dist/vue-i18n.cjs.js'
                },
                {
                    find: /\@\//,
                    replacement: `${pathResolve('src')}/`
                }
            ]
        },
        build: {
            minify: 'terser',
            outDir: env.VITE_OUT_DIR || 'dist',
            sourcemap: env.VITE_SOURCEMAP === 'true' ? 'inline' : false,
            // brotliSize: false,
            terserOptions: {
                compress: {
                    drop_debugger: env.VITE_DROP_DEBUGGER === 'true',
                    drop_console: env.VITE_DROP_CONSOLE === 'true'
                }
            }
        },
        server: {
            port: 18090,
            proxy: {
                '/api': {
                    target: env.VITE_BACKEND_HOST,
                    changeOrigin: true,
                }
            },
            hmr: {
                overlay: false
            },
            host: '0.0.0.0'
        },
        optimizeDeps: {
            include: [
                'vue',
                'vue-router',
                'vue-types',
                'element-plus/es/locale/lang/zh-cn',
                'element-plus/es/locale/lang/en',
                '@iconify/iconify',
                '@vueuse/core',
                'axios',
                'qs',
                'echarts',
                'echarts-wordcloud',
                'intro.js',
                'qrcode',
                '@wangeditor/editor',
                '@wangeditor/editor-for-vue'
            ]
        }
    }
}
