# :dog: **Pretrain Transformers Models in PyTorch using Hugging Face Transformers**

## **Pretrain 67 transformers models on your custom dataset.**

[![Open In Colab](https://colab.research.google.com/assets/colab-badge.svg)](https://colab.research.google.com/github/gmihaila/ml_things/blob/master/notebooks/pytorch/pretrain_transformers_pytorch.ipynb) &nbsp;
[![Generic badge](https://img.shields.io/badge/GitHub-Source-greensvg)](https://github.com/gmihaila/ml_things/blob/master/notebooks/pytorch/pretrain_transformers_pytorch.ipynb)
[![Generic badge](https://img.shields.io/badge/Download-Notebook-red.svg)](https://www.dropbox.com/s/rkq79hwzhqa6x8k/pretrain_transformers_pytorch.ipynb?dl=1)
[![Generic badge](https://img.shields.io/badge/Article-Medium-black.svg)](https://gmihaila.medium.com/pretrain-transformers-models-in-pytorch-using-transformers-ecaaec00fbaa)
[![License](https://img.shields.io/badge/License-Apache%202.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)


<br>

**Disclaimer:** *The format of this tutorial notebook is very similar with my other tutorial notebooks. This is done intentionally in order to keep readers familiar with my format.*

<br>

This notebook is used to pretrain transformers models using [Huggingface](https://huggingface.co/transformers/) on your own custom dataset.

**What do I mean by *pretrain transformers*? The [definition of pretraining](https://www.merriam-webster.com/dictionary/pretraining) is *to train in advance*. That is exactly what I mean! Train a transformer model to use it as a pretrained transformers model which can be used to fine-tune it on a specific task!**

**I also use the term fine-tune where I mean to continue training a pretrained model on a custom dataset. I know it is confusing and I hope I'm not making it worse. 
At the end of the day you are training a transformer model that was previously trained or not!**

With the AutoClasses functionality we can reuse the code on a large number of transformers models!

This notebook is designed to:

* **Use an already pretrained transformers model and fine-tune (continue training) it on your custom dataset.**

* **Train a transformer model from scratch on a custom dataset.** This requires an already trained (pretrained) tokenizer. This notebook will use by default the pretrained tokenizer if an already trained tokenizer is no provided.

This notebook is **heavily inspired** from the Hugging Face script used for training language models: [transformers/tree/master/examples/language-modeling](https://github.com/huggingface/transformers/tree/master/examples/language-modeling). I basically adapted that script to work nicely in a notebook with a lot more comments.

**Notes from [transformers/tree/master/examples/language-modeling](https://github.com/huggingface/transformers/tree/master/examples/language-modeling):** *Fine-tuning the library models for language modeling on a text file (GPT, GPT-2, CTRL, BERT, RoBERTa, XLNet).
GPT, GPT-2 and CTRL are fine-tuned using a causal language modeling (CLM) loss. BERT and RoBERTa are fine-tuned
using a masked language modeling (MLM) loss. XLNet is fine-tuned using a permutation language modeling (PLM) loss.*


<br>

## **What should I know for this notebook?**

Since I am using PyTorch to fine-tune our transformers models any knowledge on PyTorch is very useful.

Knowing a little bit about the [transformers](https://github.com/huggingface/transformers) library helps too.

In this notebook **I am using raw text data to pretrain / train / fine-tune transformers models**. There is no need for labeled data since we are not doing classification. The Transformers library handles the text files in same way as the original implementation of each model did.

<br>

## **How to use this notebook?**


Like with every project, I built this notebook with reusability in mind. This notebook uses a custom dataset from `.txt` files. Since the dataset does not come in a single `.txt` file I created a custom function `movie_reviews_to_file` that reads the dataset and creates the `text` file. The way I load the `.txt` files can be easily reused for any other dataset.

The only modifications needed to use your own dataset will be in the paths provided to the train `.txt` file and evaluation `.txt` file.

All parameters that need to be changed are under the **Parameters Setup** section. Each parameter is nicely commented and structured to be as intuitive as possible.

<br>


## **What transformers models work with this notebook?**

A lot of people will probably use it for Bert. When there is a need to run a different transformer model architecture, which one would work with this code?
Since the name of the notebooks is **pretrain_transformers** it should work with more than one type of transformers.

I ran this notebook across all the pretrained models found on Hugging Face Transformer. This way you know ahead of time if the model you plan to use works with this code without any modifications.

The list of pretrained transformers models that work with this notebook can be found [here](https://github.com/gmihaila/ml_things/blob/master/notebooks/pytorch/pretrain_transformers_pytorch_status_models.md). There are **67 models that worked** 😄 and 39 models that failed to work 😢 with this notebook. *Remember these are pretrained models and fine-tuned on custom dataset.*


<br>

## **Dataset**

This notebook will cover pretraining transformers on a custom dataset. I will use the well known movies reviews positive - negative labeled [Large Movie Review Dataset](https://ai.stanford.edu/~amaas/data/sentiment/).

The description provided on the Stanford website:

*This is a dataset for binary sentiment classification containing substantially more data than previous benchmark datasets. We provide a set of 25,000 highly polar movie reviews for training, and 25,000 for testing. There is additional unlabeled data for use as well. Raw text and already processed bag of words formats are provided. See the README file contained in the release for more details.*

**Why this dataset?** I believe is an easy to understand and use dataset for classification. I think sentiment data is always fun to work with.

<br>

## **Coding**

Now let's do some coding! We will go through each coding cell in the notebook and describe what it does, what's the code, and when is relevant - show the output.

I made this format to be easy to follow if you decide to run each code cell in your own python notebook.

When I learn from a tutorial I always try to replicate the results. I believe it's easy to follow along if you have the code next to the explanations.

<br>
 

## **Downloads**

Download the *Large Movie Review Dataset* and unzip it locally.



```python
# Download the dataset.
!wget -q -nc http://ai.stanford.edu/~amaas/data/sentiment/aclImdb_v1.tar.gz
# Unzip the dataset.
!tar -zxf /content/aclImdb_v1.tar.gz
```

## **Installs**

* **[transformers](https://github.com/huggingface/transformers)** library needs to be installed to use all the awesome code from Hugging Face. To get the latest version I will install it straight from GitHub.

* **[ml_things](https://github.com/gmihaila/ml_things)** library used for various machine learning related tasks. I created this library to reduce the amount of code I need to write for each machine learning project.



```python
# Install transformers library.
!pip install -q git+https://github.com/huggingface/transformers.git
# Install helper functions.
!pip install -q git+https://github.com/gmihaila/ml_things.git
```

    Installing build dependencies ... done
    Getting requirements to build wheel ... done
    Preparing wheel metadata ... done
     |████████████████████████████████| 2.9MB 6.7MB/s 
     |████████████████████████████████| 890kB 48.9MB/s 
     |████████████████████████████████| 1.1MB 49.0MB/s 
    Building wheel for transformers (PEP 517) ... done
    Building wheel for sacremoses (setup.py) ... done
     |████████████████████████████████| 71kB 5.2MB/s 
    Building wheel for ml-things (setup.py) ... done
    Building wheel for ftfy (setup.py) ... done


## **Imports**

Import all needed libraries for this notebook.

Declare basic parameters used for this notebook:

* `set_seed(123)` - Always good to set a fixed seed for reproducibility.

* `device` - Look for gpu to use. I will use cpu by default if no gpu found.


```python
import io
import os
import math
import torch
import warnings
from tqdm.notebook import tqdm
from ml_things import plot_dict, fix_text
from transformers import (
                          CONFIG_MAPPING,
                          MODEL_FOR_MASKED_LM_MAPPING,
                          MODEL_FOR_CAUSAL_LM_MAPPING,
                          PreTrainedTokenizer,
                          TrainingArguments,
                          AutoConfig,
                          AutoTokenizer,
                          AutoModelWithLMHead,
                          AutoModelForCausalLM,
                          AutoModelForMaskedLM,
                          LineByLineTextDataset,
                          TextDataset,
                          DataCollatorForLanguageModeling,
                          DataCollatorForWholeWordMask,
                          DataCollatorForPermutationLanguageModeling,
                          PretrainedConfig,
                          Trainer,
                          set_seed,
                          )

# Set seed for reproducibility,
set_seed(123)

# Look for gpu to use. Will use `cpu` by default if no gpu found.
device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')
```

## **Helper Functions**

I like to keep all Classes and functions that will be used in this notebook under this section to help maintain a clean look of the notebook:

**movie_reviews_to_file(path_data: str, path_texts_file: str)**

As I mentioned before, we will need `.txt` files to run this notebook. Since the Large Movie Review Dataset comes in multiple files with different labels I created this function to put together all data in a single `.txt` file. Examples are saved on each line of the file. The `path_data` points to the path where data files are present and `path_texts_file` will be the `.txt` file containing all data.

<br>

**ModelDataArguments**

This class follows similar format as the [transformers]((https://github.com/huggingface/transformers) library. The main difference is the way I combined multiple types of arguments into one and used rules to make sure the arguments used are correctly set. Here are all argument detailed (they are also mentioned in the class documentation):

* `train_data_file`: 
  *Path to your `.txt` file dataset. If you have an example on each line of the file make sure to use `line_by_line=True`. If the data file contains all text data without any special grouping use `line_by_line=False` to move a `block_size` window across the text file.*

* `eval_data_file`: 
  *Path to evaluation `.txt` file. It has the same format as `train_data_file`.*

* `line_by_line`: 
  *If the `train_data_file` and `eval_data_file` contains separate examples 
  on each line set `line_by_line=True`.
  If there is no separation between examples and `train_data_file` and 
  `eval_data_file` contains continuous text then `line_by_line=False` and a 
  window of `block_size` will be moved across the files to acquire examples.*

* `mlm`: 
  *Is a flag that changes loss function depending on model architecture. 
  This variable needs to be set to `True` when working with masked language 
  models like bert or roberta and set to `False` otherwise. There are functions that will raise ValueError if this argument is not set accordingly.*

* `whole_word_mask`:
  *Used as flag to determine if we decide to use whole word masking or not. Whole word masking means that whole words will be masked during training instead of tokens which can be chunks of words.*

* `mlm_probability`: 
  *Used when training masked language models. Needs to have `mlm=True`. 
  It represents the probability of masking tokens when training model.*

* `plm_probability`: 
  *Flag to define the ratio of length of a span of masked 
  tokens to surrounding context length for permutation language modeling. Used for XLNet.*

* `max_span_length`: 
  *Flag may also be used to limit the length of a span of 
  masked tokens used for permutation language modeling.
  Used for XLNet.*

* `block_size`: 
  *It refers to the windows size that is moved across the text file. Set to -1 to use maximum allowed length.*

* `overwrite_cache`: 
  *If there are any cached files, overwrite them.*

* `model_type`: 
  *Type of model used: bert, roberta, gpt2. 
  More details [here](https://huggingface.co/transformers/pretrained_models.html).*

* `model_config_name`:
  *Config of model used: bert, roberta, gpt2. 
  More details [here](https://huggingface.co/transformers/pretrained_models.html).*

* `tokenizer_name`:
  *Tokenizer used to process data for training the model. 
  It usually has same name as `model_name_or_path`: bert-base-cased, 
  roberta-base, gpt2 etc.*

* `model_name_or_path`: 
  *Path to existing transformers model or name of 
  transformer model to be used: bert-base-cased, roberta-base, gpt2 etc. 
  More details [here](https://huggingface.co/transformers/pretrained_models.html).*

* `model_cache_dir`: 
  *Path to cache files. It helps to save time when re-running code.*

<br>

**get_model_config(args: ModelDataArguments)**

Get model configuration. Using the ModelDataArguments to return the model configuration. Here are all argument detailed:

* `args`: *Model and data configuration arguments needed to perform pretraining.*

* Returns: *Model transformers configuration.*

* Raises: *ValueError: If `mlm=True` and `model_type` is NOT in ["bert", "roberta", "distilbert", "camembert"]. We need to use a masked language model in order to set `mlm=True`.*

<br>

**get_tokenizer(args: ModelDataArguments)**

Get model tokenizer.Using the ModelDataArguments return the model tokenizer and change `block_size` form `args` if needed. Here are all argument detailed:

* `args`: *Model and data configuration arugments needed to perform pretraining.*

* Returns: *Model transformers tokenizer.*

<br>

**get_model(args: ModelDataArguments, model_config)**

Get model. Using the ModelDataArguments return the actual model. Here are all argument detailed:

* `args`: *Model and data configuration arguments needed to perform pretraining.*

* `model_config`: *Model transformers configuration.*

* Returns: *PyTorch model.*

<br>

**get_dataset(args: ModelDataArguments, tokenizer: PreTrainedTokenizer, evaluate: bool=False)**

Process dataset file into PyTorch Dataset. Using the ModelDataArguments return the actual model. Here are all argument detailed:

* `args`: *Model and data configuration arguments needed to perform pretraining.*

* `tokenizer`: *Model transformers tokenizer.*

* `evaluate`: *If set to `True` the test / validation file is being handled. If set to `False` the train file is being handled.*

* Returns: *PyTorch Dataset that contains file's data.*

<br>

**get_collator(args: ModelDataArguments, model_config: PretrainedConfig, tokenizer: PreTrainedTokenizer)**

Get appropriate collator function.
Collator function will be used to collate a PyTorch Dataset object. Here are all argument detailed:

* `args`: *Model and data configuration arguments needed to perform pretraining.*

* `model_config`: *Model transformers configuration.*

* `tokenizer`: *Model transformers tokenizer.*

* Returns: *Transformers specific data collator.*


```python
def movie_reviews_to_file(path_data: str, path_texts_file: str):
  r"""Reading in all data from path and saving it into a single `.txt` file.
  
  In the pretraining process of our transformers model we require a text file.

  This function is designed to work for the Movie Reviews Dataset. 
  You wil have to create your own function to move all examples into a text 
  file if you don't already have a text file with all your unlabeled data.

  Arguments:

      path_data (:obj:`str`):
        Path to the Movie Review Dataset partition. We only have `\train` and 
        `test` partitions.

      path_texts_file (:obj:`str`):
        File path of the generated `.txt` file that contains one example / line.

  """

  # Check if path exists.
  if not os.path.isdir(path_data):
    # Raise error if path is invalid.
    raise ValueError('Invalid `path` variable! Needs to be a directory')
  # Check max sequence length.
  texts = []
  print('Reading `%s` partition...' % (os.path.basename(path_data)))
  # Since the labels are defined by folders with data we loop 
  # through each label.
  for label  in ['neg', 'pos']:
    sentiment_path = os.path.join(path_data, label)

    # Get all files from path.
    files_names = os.listdir(sentiment_path)#[:30] # SAMPLE FOR DEBUGGING.
    # Go through each file and read its content.
    for file_name in tqdm(files_names, desc=label, unit='files'):
      file_path = os.path.join(sentiment_path, file_name)

      # Read content.
      content = io.open(file_path, mode='r', encoding='utf-8').read()
      # Fix any unicode issues.
      content = fix_text(content)
      # Save content.
      texts.append(content)
  # Move list to single string.
  all_texts = '\n'.join(texts)
  # Send all texts string to single file.
  io.open(file=path_texts_file, mode='w', encoding='utf-8').write(all_texts)
  # Print when done.
  print('`.txt` file saved in `%s`\n' % path_texts_file)

  return


class ModelDataArguments(object):
  r"""Define model and data configuration needed to perform pretraining.

  Eve though all arguments are optional there still needs to be a certain 
  number of arguments that require values attributed.
  
  Arguments:

    train_data_file (:obj:`str`, `optional`): 
      Path to your .txt file dataset. If you have an example on each line of 
      the file make sure to use line_by_line=True. If the data file contains 
      all text data without any special grouping use line_by_line=False to move 
      a block_size window across the text file.
      This argument is optional and it will have a `None` value attributed 
      inside the function.

    eval_data_file (:obj:`str`, `optional`): 
      Path to evaluation .txt file. It has the same format as train_data_file.
      This argument is optional and it will have a `None` value attributed 
      inside the function.

    line_by_line (:obj:`bool`, `optional`, defaults to :obj:`False`): 
      If the train_data_file and eval_data_file contains separate examples on 
      each line then line_by_line=True. If there is no separation between 
      examples and train_data_file and eval_data_file contains continuous text 
      then line_by_line=False and a window of block_size will be moved across 
      the files to acquire examples.
      This argument is optional and it has a default value.

    mlm (:obj:`bool`, `optional`, defaults to :obj:`False`): 
      Is a flag that changes loss function depending on model architecture. 
      This variable needs to be set to True when working with masked language 
      models like bert or roberta and set to False otherwise. There are 
      functions that will raise ValueError if this argument is 
      not set accordingly.
      This argument is optional and it has a default value.

    whole_word_mask (:obj:`bool`, `optional`, defaults to :obj:`False`):
      Used as flag to determine if we decide to use whole word masking or not. 
      Whole word masking means that whole words will be masked during training 
      instead of tokens which can be chunks of words.
      This argument is optional and it has a default value.

    mlm_probability(:obj:`float`, `optional`, defaults to :obj:`0.15`): 
      Used when training masked language models. Needs to have mlm set to True. 
      It represents the probability of masking tokens when training model.
      This argument is optional and it has a default value.

    plm_probability (:obj:`float`, `optional`, defaults to :obj:`float(1/6)`): 
      Flag to define the ratio of length of a span of masked tokens to 
      surrounding context length for permutation language modeling. 
      Used for XLNet.
      This argument is optional and it has a default value.

    max_span_length (:obj:`int`, `optional`, defaults to :obj:`5`): 
      Flag may also be used to limit the length of a span of masked tokens used 
      for permutation language modeling. Used for XLNet.
      This argument is optional and it has a default value.

    block_size (:obj:`int`, `optional`, defaults to :obj:`-1`): 
      It refers to the windows size that is moved across the text file. 
      Set to -1 to use maximum allowed length.
      This argument is optional and it has a default value.

    overwrite_cache (:obj:`bool`, `optional`, defaults to :obj:`False`): 
      If there are any cached files, overwrite them.
      This argument is optional and it has a default value.

    model_type (:obj:`str`, `optional`): 
      Type of model used: bert, roberta, gpt2. 
      More details: https://huggingface.co/transformers/pretrained_models.html
      This argument is optional and it will have a `None` value attributed 
      inside the function.

    model_config_name (:obj:`str`, `optional`):
      Config of model used: bert, roberta, gpt2. 
      More details: https://huggingface.co/transformers/pretrained_models.html
      This argument is optional and it will have a `None` value attributed 
      inside the function.

    tokenizer_name: (:obj:`str`, `optional`)
      Tokenizer used to process data for training the model. 
      It usually has same name as model_name_or_path: bert-base-cased, 
      roberta-base, gpt2 etc.
      This argument is optional and it will have a `None` value attributed 
      inside the function.

    model_name_or_path (:obj:`str`, `optional`): 
      Path to existing transformers model or name of 
      transformer model to be used: bert-base-cased, roberta-base, gpt2 etc. 
      More details: https://huggingface.co/transformers/pretrained_models.html
      This argument is optional and it will have a `None` value attributed 
      inside the function.

    model_cache_dir (:obj:`str`, `optional`): 
      Path to cache files to save time when re-running code.
      This argument is optional and it will have a `None` value attributed 
      inside the function.

  Raises:

        ValueError: If `CONFIG_MAPPING` is not loaded in global variables.

        ValueError: If `model_type` is not present in `CONFIG_MAPPING.keys()`.

        ValueError: If `model_type`, `model_config_name` and 
          `model_name_or_path` variables are all `None`. At least one of them 
          needs to be set.

        warnings: If `model_config_name` and `model_name_or_path` are both 
          `None`, the model will be trained from scratch.

        ValueError: If `tokenizer_name` and `model_name_or_path` are both 
          `None`. We need at least one of them set to load tokenizer.
    
  """

  def __init__(self, train_data_file=None, eval_data_file=None, 
               line_by_line=False, mlm=False, mlm_probability=0.15, 
               whole_word_mask=False, plm_probability=float(1/6), 
               max_span_length=5, block_size=-1, overwrite_cache=False, 
               model_type=None, model_config_name=None, tokenizer_name=None, 
               model_name_or_path=None, model_cache_dir=None):
    
    # Make sure CONFIG_MAPPING is imported from transformers module.
    if 'CONFIG_MAPPING' not in globals():
      raise ValueError('Could not find `CONFIG_MAPPING` imported! Make sure' \
                       ' to import it from `transformers` module!')

    # Make sure model_type is valid.
    if (model_type is not None) and (model_type not in CONFIG_MAPPING.keys()):
      raise ValueError('Invalid `model_type`! Use one of the following: %s' % 
                       (str(list(CONFIG_MAPPING.keys()))))
      
    # Make sure that model_type, model_config_name and model_name_or_path 
    # variables are not all `None`.
    if not any([model_type, model_config_name, model_name_or_path]):
      raise ValueError('You can`t have all `model_type`, `model_config_name`,' \
                       ' `model_name_or_path` be `None`! You need to have' \
                       'at least one of them set!')
    
    # Check if a new model will be loaded from scratch.
    if not any([model_config_name, model_name_or_path]):
      # Setup warning to show pretty. This is an overkill
      warnings.formatwarning = lambda message,category,*args,**kwargs: \
                               '%s: %s\n' % (category.__name__, message)
      # Display warning.
      warnings.warn('You are planning to train a model from scratch! 🙀')

    # Check if a new tokenizer wants to be loaded.
    # This feature is not supported!
    if not any([tokenizer_name, model_name_or_path]):
      # Can't train tokenizer from scratch here! Raise error.
      raise ValueError('You want to train tokenizer from scratch! ' \
                    'That is not possible yet! You can train your own ' \
                    'tokenizer separately and use path here to load it!')
      
    # Set all data related arguments.
    self.train_data_file = train_data_file
    self.eval_data_file = eval_data_file
    self.line_by_line = line_by_line
    self.mlm = mlm
    self.whole_word_mask = whole_word_mask
    self.mlm_probability = mlm_probability
    self.plm_probability = plm_probability
    self.max_span_length = max_span_length
    self.block_size = block_size
    self.overwrite_cache = overwrite_cache

    # Set all model and tokenizer arguments.
    self.model_type = model_type
    self.model_config_name = model_config_name
    self.tokenizer_name = tokenizer_name
    self.model_name_or_path = model_name_or_path
    self.model_cache_dir = model_cache_dir
    return


def get_model_config(args: ModelDataArguments):
  r"""
  Get model configuration.

  Using the ModelDataArguments return the model configuration.

  Arguments:

    args (:obj:`ModelDataArguments`):
      Model and data configuration arguments needed to perform pretraining.

  Returns:

    :obj:`PretrainedConfig`: Model transformers configuration.

  Raises:

    ValueError: If `mlm=True` and `model_type` is NOT in ["bert", 
          "roberta", "distilbert", "camembert"]. We need to use a masked 
          language model in order to set `mlm=True`.
  """

  # Check model configuration.
  if args.model_config_name is not None:
    # Use model configure name if defined.
    model_config = AutoConfig.from_pretrained(args.model_config_name, 
                                      cache_dir=args.model_cache_dir)

  elif args.model_name_or_path is not None:
    # Use model name or path if defined.
    model_config = AutoConfig.from_pretrained(args.model_name_or_path, 
                                      cache_dir=args.model_cache_dir)

  else:
    # Use config mapping if building model from scratch.
    model_config = CONFIG_MAPPING[args.model_type]()

  # Make sure `mlm` flag is set for Masked Language Models (MLM).
  if (model_config.model_type in ["bert", "roberta", "distilbert", 
                                  "camembert"]) and (args.mlm is False):
    raise ValueError('BERT and RoBERTa-like models do not have LM heads ' \
                    'butmasked LM heads. They must be run setting `mlm=True`')
  
  # Adjust block size for xlnet.
  if model_config.model_type == "xlnet":
    # xlnet used 512 tokens when training.
    args.block_size = 512
    # setup memory length
    model_config.mem_len = 1024
  
  return model_config


def get_tokenizer(args: ModelDataArguments):
  r"""
  Get model tokenizer.

  Using the ModelDataArguments return the model tokenizer and change 
  `block_size` form `args` if needed.

  Arguments:

    args (:obj:`ModelDataArguments`):
      Model and data configuration arguments needed to perform pretraining.

  Returns:

    :obj:`PreTrainedTokenizer`: Model transformers tokenizer.

  """

  # Check tokenizer configuration.
  if args.tokenizer_name:
    # Use tokenizer name if define.
    tokenizer = AutoTokenizer.from_pretrained(args.tokenizer_name, 
                                              cache_dir=args.model_cache_dir)

  elif args.model_name_or_path:
    # Use tokenizer name of path if defined.
    tokenizer = AutoTokenizer.from_pretrained(args.model_name_or_path, 
                                              cache_dir=args.model_cache_dir)
    
  # Setp data block size.
  if args.block_size <= 0:
    # Set block size to maximum length of tokenizer.
    # Input block size will be the max possible for the model.
    # Some max lengths are very large and will cause a
    args.block_size = tokenizer.model_max_length
  else:
    # Never go beyond tokenizer maximum length.
    args.block_size = min(args.block_size, tokenizer.model_max_length)

  return tokenizer
  

def get_model(args: ModelDataArguments, model_config):
  r"""
  Get model.

  Using the ModelDataArguments return the actual model.

  Arguments:

    args (:obj:`ModelDataArguments`):
      Model and data configuration arguments needed to perform pretraining.

    model_config (:obj:`PretrainedConfig`):
      Model transformers configuration.

  Returns:

    :obj:`torch.nn.Module`: PyTorch model.

  """

  # Make sure MODEL_FOR_MASKED_LM_MAPPING and MODEL_FOR_CAUSAL_LM_MAPPING are 
  # imported from transformers module.
  if ('MODEL_FOR_MASKED_LM_MAPPING' not in globals()) and \
                ('MODEL_FOR_CAUSAL_LM_MAPPING' not in globals()):
    raise ValueError('Could not find `MODEL_FOR_MASKED_LM_MAPPING` and' \
                     ' `MODEL_FOR_MASKED_LM_MAPPING` imported! Make sure to' \
                     ' import them from `transformers` module!')
    
  # Check if using pre-trained model or train from scratch.
  if args.model_name_or_path:
    # Use pre-trained model.
    if type(model_config) in MODEL_FOR_MASKED_LM_MAPPING.keys():
      # Masked language modeling head.
      return AutoModelForMaskedLM.from_pretrained(
                        args.model_name_or_path,
                        from_tf=bool(".ckpt" in args.model_name_or_path),
                        config=model_config,
                        cache_dir=args.model_cache_dir,
                        )
    elif type(model_config) in MODEL_FOR_CAUSAL_LM_MAPPING.keys():
      # Causal language modeling head.
      return AutoModelForCausalLM.from_pretrained(
                                          args.model_name_or_path, 
                                          from_tf=bool(".ckpt" in 
                                                        args.model_name_or_path),
                                          config=model_config, 
                                          cache_dir=args.model_cache_dir)
    else:
      raise ValueError(
          'Invalid `model_name_or_path`! It should be in %s or %s!' % 
          (str(MODEL_FOR_MASKED_LM_MAPPING.keys()), 
           str(MODEL_FOR_CAUSAL_LM_MAPPING.keys())))
    
  else:
    # Use model from configuration - train from scratch.
      print("Training new model from scratch!")
      return AutoModelWithLMHead.from_config(config)


def get_dataset(args: ModelDataArguments, tokenizer: PreTrainedTokenizer, 
                evaluate: bool=False):
  r"""
  Process dataset file into PyTorch Dataset.

  Using the ModelDataArguments return the actual model.

  Arguments:

    args (:obj:`ModelDataArguments`):
      Model and data configuration arguments needed to perform pretraining.

    tokenizer (:obj:`PreTrainedTokenizer`):
      Model transformers tokenizer.

    evaluate (:obj:`bool`, `optional`, defaults to :obj:`False`):
      If set to `True` the test / validation file is being handled.
      If set to `False` the train file is being handled.

  Returns:

    :obj:`Dataset`: PyTorch Dataset that contains file's data.

  """

  # Get file path for either train or evaluate.
  file_path = args.eval_data_file if evaluate else args.train_data_file

  # Check if `line_by_line` flag is set to `True`.
  if args.line_by_line:
    # Each example in data file is on each line.
    return LineByLineTextDataset(tokenizer=tokenizer, file_path=file_path, 
                                 block_size=args.block_size)
    
  else:
    # All data in file is put together without any separation.
    return TextDataset(tokenizer=tokenizer, file_path=file_path, 
                       block_size=args.block_size, 
                       overwrite_cache=args.overwrite_cache)


def get_collator(args: ModelDataArguments, model_config: PretrainedConfig, 
                 tokenizer: PreTrainedTokenizer):
  r"""
  Get appropriate collator function.

  Collator function will be used to collate a PyTorch Dataset object.

  Arguments:

    args (:obj:`ModelDataArguments`):
      Model and data configuration arguments needed to perform pretraining.

    model_config (:obj:`PretrainedConfig`):
      Model transformers configuration.

    tokenizer (:obj:`PreTrainedTokenizer`):
      Model transformers tokenizer.

  Returns:

    :obj:`data_collator`: Transformers specific data collator.

  """

  # Special dataset handle depending on model type.
  if model_config.model_type == "xlnet":
    # Configure collator for XLNET.
    return DataCollatorForPermutationLanguageModeling(
                                          tokenizer=tokenizer,
                                          plm_probability=args.plm_probability,
                                          max_span_length=args.max_span_length,
                                          )
  else:
    # Configure data for rest of model types.
    if args.mlm and args.whole_word_mask:
      # Use whole word masking.
      return DataCollatorForWholeWordMask(
                                          tokenizer=tokenizer, 
                                          mlm_probability=args.mlm_probability,
                                          )
    else:
      # Regular language modeling.
      return DataCollatorForLanguageModeling(
                                          tokenizer=tokenizer, 
                                          mlm=args.mlm, 
                                          mlm_probability=args.mlm_probability,
                                          )

```

## **Parameters Setup**

Declare the rest of the parameters used for this notebook:

* `model_data_args` contains all arguments needed to setup dataset, model configuration, model tokenizer and the actual model. This is created using the `ModelDataArguments` class.

* `training_args` contain all arguments needed to use the [Trainer]() functionality from Transformers that allows us to train transformers models in PyTorch very easy. You can find the complete documentation [here](https://huggingface.co/transformers/main_classes/trainer.html#trainingarguments). There are a lot of parameters that can be set to allow multiple functionalities. I only used the following parameters (the comments are inspired from the HuggingFace documentation of [TrainingArguments](https://huggingface.co/transformers/main_classes/trainer.html#trainingarguments):
 
  * `output_dir`: *The output directory where the model predictions and checkpoints will be written. I set it up to `pretrained_bert_model` where the model and will be saved.*

  * `overwrite_output_dir`: *Overwrite the content of the output directory. I set it to `True` in case I run the notebook multiple times I only care about the last run.*

  * `do_train`: *Whether to run training or not. I set this parameter to `True` because I want to train the model on my custom dataset.* 

  * `do_eval`: *Whether to run evaluation on the evaluation files or not.
I set it to `True` since I have test data file and I want to evaluate how well the model trains.*

  * `per_device_train_batch_size`: *Batch size GPU/TPU core/CPU training. I set it to `2` for this example. I recommend setting it up as high as your GPU memory allows you.*

  * `per_device_eval_batch_size`: *Batch size  GPU/TPU core/CPU for evaluation.I set this value to `100` since it's not dealing with gradients.*

  * `evaluation_strategy`: *Evaluation strategy to adopt during training: `no`: No evaluation during training; `steps`: Evaluate every `eval_steps; `epoch`: Evaluate every end of epoch. I set it to 'steps' since I want to evaluate model more often.*

  * `logging_steps`: *How often to show logs. I will se this to plot history loss and calculate perplexity. I set this to `20` just as an example. If your evaluate data is large you might not want to run it that often because it will significantly slow down training time.*

  * `eval_steps`: *Number of update steps between two evaluations if evaluation_strategy="steps". Will default to the same value as logging_steps if not set. Since I want to evaluate model ever`logging_steps` I will set this to `None` since it will inherit same value as `logging_steps`.*

  * `prediction_loss_only`: *Set prediction loss to `True` in order to return loss for perplexity calculation. Since I want to calculate perplexity I set this to `True` since I want to monitor loss and perplexity (which is exp(loss)).*

  * `learning_rate`: *The initial learning rate for Adam. Defaults is set to `5e-5`.*

  * `weight_decay`: *The weight decay to apply (if not zero)Defaults is set to `0`.*

  * `adam_epsilon`: *Epsilon for the Adam optimizer. Defaults to `1e-8`.*

  * `max_grad_norm`: *Maximum gradient norm (for gradient clipping). Defaults to `0`.*

  * `num_train_epochs`: *Total number of training epochs to perform (if not an integer, will perform the decimal part percents of the last epoch before stopping training). I set it to `2` at most. Since the custom dataset will be a lot smaller than the original dataset the model was trained on we don't want to overfit.*

  * `save_steps`: *Number of updates steps before two checkpoint saves. Defaults to `500`.*


```python
# Define arguments for data, tokenizer and model arguments.
# See comments in `ModelDataArguments` class.
model_data_args = ModelDataArguments(
                                    train_data_file='/content/train.txt', 
                                    eval_data_file='/content/test.txt', 
                                    line_by_line=True, 
                                    mlm=True,
                                    whole_word_mask=True,
                                    mlm_probability=0.15,
                                    plm_probability=float(1/6), 
                                    max_span_length=5,
                                    block_size=50, 
                                    overwrite_cache=False, 
                                    model_type='bert', 
                                    model_config_name='bert-base-cased', 
                                    tokenizer_name='bert-base-cased', 
                                    model_name_or_path='bert-base-cased', 
                                    model_cache_dir=None,
                                    )

# Define arguments for training
# Note: I only used the arguments I care about. `TrainingArguments` contains
# a lot more arguments. For more details check the awesome documentation:
# https://huggingface.co/transformers/main_classes/trainer.html#trainingarguments
training_args = TrainingArguments(
                          # The output directory where the model predictions 
                          # and checkpoints will be written.
                          output_dir='pretrain_bert',

                          # Overwrite the content of the output directory.
                          overwrite_output_dir=True,

                          # Whether to run training or not.
                          do_train=True, 
                          
                          # Whether to run evaluation on the dev or not.
                          do_eval=True,
                          
                          # Batch size GPU/TPU core/CPU training.
                          per_device_train_batch_size=10,
                          
                          # Batch size  GPU/TPU core/CPU for evaluation.
                          per_device_eval_batch_size=100,

                          # evaluation strategy to adopt during training
                          # `no`: No evaluation during training.
                          # `steps`: Evaluate every `eval_steps`.
                          # `epoch`: Evaluate every end of epoch.
                          evaluation_strategy='steps',

                          # How often to show logs. I will se this to 
                          # plot history loss and calculate perplexity.
                          logging_steps=700,

                          # Number of update steps between two 
                          # evaluations if evaluation_strategy="steps".
                          # Will default to the same value as l
                          # logging_steps if not set.
                          eval_steps = None,
                          
                          # Set prediction loss to `True` in order to 
                          # return loss for perplexity calculation.
                          prediction_loss_only=True,

                          # The initial learning rate for Adam. 
                          # Defaults to 5e-5.
                          learning_rate = 5e-5,

                          # The weight decay to apply (if not zero).
                          weight_decay=0,

                          # Epsilon for the Adam optimizer. 
                          # Defaults to 1e-8
                          adam_epsilon = 1e-8,

                          # Maximum gradient norm (for gradient 
                          # clipping). Defaults to 0.
                          max_grad_norm = 1.0,
                          # Total number of training epochs to perform 
                          # (if not an integer, will perform the 
                          # decimal part percents of
                          # the last epoch before stopping training).
                          num_train_epochs = 2,

                          # Number of updates steps before two checkpoint saves. 
                          # Defaults to 500
                          save_steps = -1,
                          )
```

## **Load Configuration, Tokenizer and Model**

Loading the three essential parts of the pretrained transformers: configuration, tokenizer and model. 

Since I use the AutoClass functionality from Hugging Face I only need to worry about the model's name as input and the rest is handled by the transformers library.

I will be calling each three functions created in the **Helper Functions** tab that help return `config` of the model, `tokenizer` of the model and the actual PyTorch `model`.

After `model` is loaded is always good practice to resize the model depending on the `tokenizer` size. This means that the tokenizer's vocabulary will be aligned with the models embedding layer. This is very useful when we have a different tokenizer that the pretrained one or we train a transformer model from scratch.




```python
# Load model configuration.
print('Loading model configuration...')
config = get_model_config(model_data_args)

# Load model tokenizer.
print('Loading model`s tokenizer...')
tokenizer = get_tokenizer(model_data_args)

# Loading model.
print('Loading actual model...')
model = get_model(model_data_args, config)

# Resize model to fit all tokens in tokenizer.
model.resize_token_embeddings(len(tokenizer))
```


    Loading model configuration...
    Downloading: 100%|████████████████████████████████|433/433 [00:01<00:00, 285B/s]
    
    Loading model`s tokenizer...
    Downloading: 100%|████████████████████████████████|433/433 [00:01<00:00, 285B/s]
    
    Loading actual model...
    Downloading: 100% |████████████████████████████████|436M/436M [00:36<00:00, 11.9MB/s]
    
    Some weights of the model checkpoint at bert-base-cased were not used when initializing BertForMaskedLM: ['cls.seq_relationship.weight', 'cls.seq_relationship.bias']
    - This IS expected if you are initializing BertForMaskedLM from the checkpoint of a model trained on another task or with another architecture (e.g. initializing a BertForSequenceClassification model from a BertForPretraining model).
    - This IS NOT expected if you are initializing BertForMaskedLM from the checkpoint of a model that you expect to be exactly identical (initializing a BertForSequenceClassification model from a BertForSequenceClassification model).
    Embedding(28996, 768, padding_idx=0)

<br>

## **Dataset and Collator**

This is where I create the PyTorch Dataset and data collator objects that will be used to feed data into our model.

This is where I use the MovieReviewsDataset text files created with the `movie_reviews_to_file` function. Since data is partitioned for both train and test I will create two text files: one used for train and one used for evaluation. 

I strongly recommend to use a validation text file in order to determine how much training is needed in order to avoid overfitting. After you figure out what parameters yield the best results, the validation file can be incorporated in train and run a final train with the whole dataset.

The data collator is used to format the PyTorch Dataset outputs to match the output of our specific transformers model: i.e. for Bert it will created the masked tokens needed to train.


```python
# Create texts file from train data.
movie_reviews_to_file(path_data='/content/aclImdb/train', path_texts_file='/content/train.txt')
# Create texts file from test data.
movie_reviews_to_file(path_data='/content/aclImdb/test', path_texts_file='/content/test.txt')


# Setup train dataset if `do_train` is set.
print('Creating train dataset...')
train_dataset = get_dataset(model_data_args, tokenizer=tokenizer, evaluate=False) if training_args.do_train else None

# Setup evaluation dataset if `do_eval` is set.
print('Creating evaluate dataset...')
eval_dataset = get_dataset(model_data_args, tokenizer=tokenizer, evaluate=True) if training_args.do_eval else None

# Get data collator to modify data format depending on type of model used.
data_collator = get_collator(model_data_args, config, tokenizer)

# Check how many logging prints you'll have. This is to avoid overflowing the 
# notebook with a lot of prints. Display warning to user if the logging steps 
# that will be displayed is larger than 100.
if (len(train_dataset) // training_args.per_device_train_batch_size \
    // training_args.logging_steps * training_args.num_train_epochs) > 100:
  # Display warning.
  warnings.warn('Your `logging_steps` value will will do a lot of printing!' \
                ' Consider increasing `logging_steps` to avoid overflowing' \
                ' the notebook with a lot of prints!')
```

    Reading `train` partition...
    neg: 100%|████████████████████████████████|12500/12500 [00:55<00:00, 224.11files/s]
    pos: 100%|████████████████████████████████|12500/12500 [00:55<00:00, 224.11files/s]
    `.txt` file saved in `/content/train.txt`

    Reading `test` partition...
    neg: 100%|████████████████████████████████|12500/12500 [00:55<00:00, 224.11files/s]
    pos: 100%|████████████████████████████████|12500/12500 [00:55<00:00, 224.11files/s]
    `.txt` file saved in `/content/train.txt`
    
    Creating train dataset...
    Creating evaluate dataset...

<br>

## **Train**

Hugging Face was very nice to us for creating the `Trainer` class. This helps make PyTorch model training of transformers very easy! We just need to make sure we loaded the proper parameters and everything else is taking care of!

At the end of the training the tokenizer is saved along with the model so you can easily re-use it later or even load in on Hugging Face Models.

I configured the arguments to display both train and validation loss at every `logging_steps`. It gives us a sense of how well the model is trained.


```python
# Initialize Trainer.
print('Loading `trainer`...')
trainer = Trainer(model=model,
                  args=training_args,
                  data_collator=data_collator,
                  train_dataset=train_dataset,
                  eval_dataset=eval_dataset,
                  )


# Check model path to save.
if training_args.do_train:
  print('Start training...')

  # Setup model path if the model to train loaded from a local path.
  model_path = (model_data_args.model_name_or_path 
                if model_data_args.model_name_or_path is not None and 
                os.path.isdir(model_data_args.model_name_or_path) 
                else None
                )
  # Run training.
  trainer.train(model_path=model_path)
  # Save model.
  trainer.save_model()

  # For convenience, we also re-save the tokenizer to the same directory,
  # so that you can share your model easily on huggingface.co/models =).
  if trainer.is_world_process_zero():
    tokenizer.save_pretrained(training_args.output_dir)
```

    Loading `trainer`...
    Start training...
    |████████████████████████████████|[5000/5000 09:43, Epoch 2/2]

    Step	Training Loss	Validation Loss
    700	    2.804672	2.600590
    1400	2.666996	2.548267
    2100	2.625075	2.502431
    2800	2.545872	2.485056
    3500	2.470102	2.444808
    4200	2.464950	2.420487
    4900	2.436973	2.410310


<br>

## **Plot Train**

The `Trainer` class is so useful that it will record the log history for us. I use this to access the train and validation losses recorded at each `logging_steps` during training.

Since we are training / fine-tuning / extended training or pretraining (depending what terminology you use) a language model, we want to compute the perplexity. 

This is what [Wikipedia](https://en.wikipedia.org/wiki/Perplexity) says about perplexity: *In information theory, perplexity is a measurement of how well a probability distribution or probability model predicts a sample. It may be used to compare probability models. A low perplexity indicates the probability distribution is good at predicting the sample.*

We can look at the perplexity plot in the same way we look at the loss plot: the lower the better and if the validation perplexity starts to increase we are starting to overfit the model.

**Note:** It looks from the plots that the train loss is higher than validation loss. That means that our validation data is too easy for the model and we should use a different validation dataset. Since the purpose of this notebook is to show how to train transformers models and provide tools to evaluate such process I will leave the results *as is*.


```python
# Keep track of train and evaluate loss.
loss_history = {'train_loss':[], 'eval_loss':[]}

# Keep track of train and evaluate perplexity.
# This is a metric useful to track for language models.
perplexity_history = {'train_perplexity':[], 'eval_perplexity':[]}

# Loop through each log history.
for log_history in trainer.state.log_history:

  if 'loss' in log_history.keys():
    # Deal with trianing loss.
    loss_history['train_loss'].append(log_history['loss'])
    perplexity_history['train_perplexity'].append(math.exp(log_history['loss']))
    
  elif 'eval_loss' in log_history.keys():
    # Deal with eval loss.
    loss_history['eval_loss'].append(log_history['eval_loss'])
    perplexity_history['eval_perplexity'].append(math.exp(log_history['eval_loss']))

# Plot Losses.
plot_dict(loss_history, start_step=training_args.logging_steps, 
          step_size=training_args.logging_steps, use_title='Loss', 
          use_xlabel='Train Steps', use_ylabel='Values', magnify=2)

print()

# Plot Perplexities.
plot_dict(perplexity_history, start_step=training_args.logging_steps, 
          step_size=training_args.logging_steps, use_title='Perplexity', 
          use_xlabel='Train Steps', use_ylabel='Values', magnify=2)
```

![train_valid_loss](data:image/png;base64,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)

![train_valid_perplexity](data:image/png;base64,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)


## **Evaluate**

For the final evaluation we can have a separate test set that we use to do our final perplexity evaluation. For simplicity I used the same validation text file for the final evaluation. That is the reason I get the same results as the last validation perplexity plot value.


```python
# check if `do_eval` flag is set.
if training_args.do_eval:
  
  # capture output if trainer evaluate.
  eval_output = trainer.evaluate()
  # compute perplexity from model loss.
  perplexity = math.exp(eval_output["eval_loss"])
  print('\nEvaluate Perplexity: {:10,.2f}'.format(perplexity))
else:
  print('No evaluation needed. No evaluation data provided, `do_eval=False`!')
```

    |████████████████████████████████|[250/250 00:25]
    
    Evaluate Perplexity:      11.01


## **Final Note**

If you made it this far **Congrats!** 🎊 and **Thank you!** 🙏 for your interest in my tutorial!

I've been using this code for a while now and I feel it got to a point where is nicely documented and easy to follow.

Of course is easy for me to follow because I built it. That is why any feedback is welcome and it helps me improve my future tutorials!

If you see something wrong please let me know by opening an issue on my [ml_things GitHub repository](https://github.com/gmihaila/ml_things/issues)!

A lot of tutorials out there are mostly a one-time thing and are not being maintained. I plan on keeping my tutorials up to date as much as I can.

## **Contact** 🎣

🦊 GitHub: [gmihaila](https://github.com/gmihaila)

🌐 Website: [gmihaila.github.io](https://gmihaila.github.io/)

👔 LinkedIn: [mihailageorge](https://medium.com/r/?url=https%3A%2F%2Fwww.linkedin.com%2Fin%2Fmihailageorge)

📬 Email: [georgemihaila@my.unt.edu.com](mailto:georgemihaila@my.unt.edu.com?subject=GitHub%20Website)

<br>