import url from "node:url";
import path from "node:path";
import _ from "lodash";

import { VITE_RUN_UUID_ROUTE, WORKER_ENV_BY_RUN_UUID } from "./constants";

import type { Connect } from "vite";
import type { WorkerInitializePayload } from "./browser-modules/types";

type TestInfoFromViteRequest = {
    routeName: string;
    runUuid: string;
    env: WorkerInitializePayload;
};

// TODO: use import.meta.url after migrate to esm
export const getImportMetaUrl = (path: string): string => {
    return url.pathToFileURL(path).toString();
};

export const getNodeModulePath = async ({
    moduleName,
    rootDir = process.cwd(),
    parent = "node_modules",
}: {
    moduleName: string;
    rootDir?: string;
    parent?: string;
}): Promise<string> => {
    const rootFileUrl = url.pathToFileURL(rootDir).href;

    // TODO: use import at the beginning of the file after migrate to esm
    const { resolve } = await eval(`import("import-meta-resolve")`);

    return resolve(moduleName, path.join(rootFileUrl, parent));
};

// TODO: use from browser code after migrate to esm
export const prepareError = (error: Error): Error => {
    return JSON.parse(JSON.stringify(error, Object.getOwnPropertyNames(error)));
};

export const getTestInfoFromViteRequest = (req: Connect.IncomingMessage): TestInfoFromViteRequest | null => {
    if (!req.url?.endsWith("index.html") || !req.originalUrl) {
        return null;
    }

    const parsedUrl = url.parse(req.originalUrl);
    const [routeName, runUuid] = _.compact(parsedUrl.pathname?.split("/"));

    if (routeName !== VITE_RUN_UUID_ROUTE || !runUuid) {
        throw new Error(`Pathname must be in "/${VITE_RUN_UUID_ROUTE}/:uuid" format, but got: ${req.originalUrl}`);
    }

    const env = WORKER_ENV_BY_RUN_UUID.get(runUuid);
    if (!env) {
        throw new Error(
            `Worker environment is not found by "${runUuid}". ` +
                "This is possible if:\n" +
                '  - "runUuid" is not generated by Testplane\n' +
                "  - the test has already been completed\n" +
                "  - worker was disconnected",
        );
    }

    return { routeName, runUuid, env };
};

export const getPathWithoutExtName = (fsPath: string): string => {
    const extname = path.extname(fsPath);

    if (!extname) {
        return fsPath;
    }

    return fsPath.slice(0, -extname.length);
};
