import _ from "lodash";
import fs from "fs-extra";
import path from "path";
import { getOsPackagesDir, type DownloadProgressCallback, browserInstallerDebug } from "../utils";
import { installUbuntuPackages } from "./apt";
import { getUbuntuMilestone } from "./utils";
import logger from "../../utils/logger";
import { LINUX_RUNTIME_LIBRARIES_PATH_ENV_NAME, LINUX_UBUNTU_RELEASE_ID } from "../constants";
import registry from "../registry";

export { isUbuntu, getUbuntuMilestone, ensureUnixBinaryExists } from "./utils";

const getDependenciesArrayFilePath = (ubuntuMilestone: string): string =>
    path.join(__dirname, "autogenerated", `ubuntu-${ubuntuMilestone}-dependencies.json`);

const readUbuntuPackageDependencies = async (ubuntuMilestone: string): Promise<string[]> => {
    try {
        return await fs.readJSON(getDependenciesArrayFilePath(ubuntuMilestone));
    } catch (_) {
        logger.warn(
            [
                `Unable to read ubuntu dependencies for Ubuntu@${ubuntuMilestone}, as this version currently not supported`,
                `Assuming all necessary packages are installed already`,
            ].join("\n"),
        );

        return [];
    }
};

export const writeUbuntuPackageDependencies = async (ubuntuMilestone: string, deps: string[]): Promise<void> => {
    const currentPackages = await readUbuntuPackageDependencies(ubuntuMilestone).catch(() => [] as string[]);

    const packagesToWrite = _.uniq(currentPackages.concat(deps)).sort();

    await fs.outputJSON(getDependenciesArrayFilePath(ubuntuMilestone), packagesToWrite, { spaces: 4 });
};

export const installUbuntuPackageDependencies = async (): Promise<string> => {
    const ubuntuMilestone = await getUbuntuMilestone();

    browserInstallerDebug(`installing ubuntu${ubuntuMilestone} dependencies`);

    if (registry.hasOsPackages(LINUX_UBUNTU_RELEASE_ID, ubuntuMilestone)) {
        browserInstallerDebug(`installing ubuntu${ubuntuMilestone} dependencies`);

        return registry.getOsPackagesPath(LINUX_UBUNTU_RELEASE_ID, ubuntuMilestone);
    }

    const downloadFn = async (downloadProgressCallback: DownloadProgressCallback): Promise<string> => {
        const ubuntuPackageDependencies = await readUbuntuPackageDependencies(ubuntuMilestone);
        const ubuntuPackagesDir = getOsPackagesDir(LINUX_UBUNTU_RELEASE_ID, ubuntuMilestone);

        await installUbuntuPackages(ubuntuPackageDependencies, ubuntuPackagesDir, { downloadProgressCallback });

        return ubuntuPackagesDir;
    };

    return registry.installOsPackages(LINUX_UBUNTU_RELEASE_ID, ubuntuMilestone, downloadFn);
};

const listDirsAbsolutePath = async (dirBasePath: string, ...prefix: string[]): Promise<string[]> => {
    const fullDirPath = path.join(dirBasePath, ...prefix);

    if (!fs.existsSync(fullDirPath)) {
        return [];
    }

    const dirContents = await fs.readdir(fullDirPath);
    const dirContentsAbsPaths = dirContents.map(obj => path.join(fullDirPath, obj));

    const directories = [] as string[];

    await Promise.all(
        dirContentsAbsPaths.map(obj =>
            fs.stat(obj).then(stat => {
                if (stat.isDirectory()) {
                    directories.push(obj);
                }
            }),
        ),
    );

    return directories;
};

const getUbuntuLinkerEnvRaw = async (): Promise<Record<string, string>> => {
    const ubuntuMilestone = await getUbuntuMilestone();

    if (!registry.hasOsPackages(LINUX_UBUNTU_RELEASE_ID, ubuntuMilestone)) {
        return {};
    }

    const ubuntuPackagesDir = await registry.getOsPackagesPath(LINUX_UBUNTU_RELEASE_ID, ubuntuMilestone);

    const currentRuntimeLibrariesEnvValue = process.env[LINUX_RUNTIME_LIBRARIES_PATH_ENV_NAME];

    const [libDirs, usrLibDirs] = await Promise.all([
        listDirsAbsolutePath(ubuntuPackagesDir, "lib"),
        listDirsAbsolutePath(ubuntuPackagesDir, "usr", "lib"),
    ]);

    const libraryPaths = [...libDirs, ...usrLibDirs, currentRuntimeLibrariesEnvValue].filter(Boolean);

    return { [LINUX_RUNTIME_LIBRARIES_PATH_ENV_NAME]: libraryPaths.join(":") };
};

export const getUbuntuLinkerEnv = _.once(getUbuntuLinkerEnvRaw);
