import fs from "fs-extra";
import _ from "lodash";
import path from "path";
import { getMilestone } from "../../../utils";
import type { BrowserWithVersion } from "../utils";

type SharedObjectName = string;
type PackageName = string;

type BrowserName = string;
type BrowserVersion = string;

export type CacheData = {
    /** Different for each ubuntu version */
    sharedObjectsMap: Record<SharedObjectName, PackageName>;

    /** Mutual for all linux versions */
    processedBrowsers: {
        downloadedBrowsers: Record<BrowserName, BrowserVersion[]>;
        sharedObjects: string[];
    };
};

const sortObject = <T = unknown>(obj: T): T => {
    if (_.isArray(obj)) {
        return obj.sort();
    }

    if (!_.isPlainObject(obj)) {
        return obj;
    }

    const sourceObj = obj as T & Record<string, unknown>;
    const result = {} as T;

    const sortedKeys = Object.keys(sourceObj).sort() as Array<keyof T>;

    for (const key of sortedKeys) {
        result[key] = sortObject(sourceObj[key]);
    }

    return result;
};

export class Cache {
    private _sharedObjectsMapPath: string;
    private _processedBrowsersCachePath: string;
    private _cache: CacheData = {
        sharedObjectsMap: {},
        processedBrowsers: { downloadedBrowsers: {}, sharedObjects: [] },
    };

    constructor(osVersion: string) {
        const autoGeneratedDirPath = path.join(__dirname, "autogenerated");

        this._sharedObjectsMapPath = path.join(autoGeneratedDirPath, `shared-objects-map-ubuntu-${osVersion}.json`);
        this._processedBrowsersCachePath = path.join(autoGeneratedDirPath, "processed-browsers-linux.json");
    }

    async read(): Promise<this> {
        const [sharedObjectsMap, processedBrowsers] = await Promise.all([
            fs.readJSON(this._sharedObjectsMapPath).catch(() => null),
            fs.readJSON(this._processedBrowsersCachePath).catch(() => null),
        ]);

        if (sharedObjectsMap) {
            this._cache.sharedObjectsMap = sharedObjectsMap;
        }

        if (processedBrowsers) {
            this._cache.processedBrowsers = processedBrowsers;
        }

        return this;
    }

    async write(): Promise<void> {
        const { sharedObjectsMap, processedBrowsers } = this._cache;

        const uniqSharedObjects = _.uniq(Object.keys(sharedObjectsMap).concat(processedBrowsers.sharedObjects));

        processedBrowsers.sharedObjects = uniqSharedObjects;

        await fs.outputJSON(this._sharedObjectsMapPath, sortObject(sharedObjectsMap), { spaces: 4 });
        await fs.outputJSON(this._processedBrowsersCachePath, sortObject(processedBrowsers), { spaces: 4 });
    }

    private hasProcessedBrowser({ browserName, browserVersion }: BrowserWithVersion): boolean {
        const processedBrowserVersions = this._cache.processedBrowsers.downloadedBrowsers[browserName];
        const milestone = getMilestone(browserVersion);

        return Boolean(processedBrowserVersions && processedBrowserVersions.includes(milestone));
    }

    filterProcessedBrowsers(browsers: BrowserWithVersion[]): BrowserWithVersion[] {
        return browsers.filter(browser => !this.hasProcessedBrowser(browser));
    }

    private saveProcessedBrowser({ browserName, browserVersion }: BrowserWithVersion): void {
        const browserCache = (this._cache.processedBrowsers.downloadedBrowsers[browserName] ||= []);
        const milestone = getMilestone(browserVersion);

        if (!browserCache.includes(milestone)) {
            browserCache.push(milestone);
        }
    }

    saveProcessedBrowsers(browsers: BrowserWithVersion[]): void {
        browsers.forEach(browser => this.saveProcessedBrowser(browser));
    }

    hasResolvedPackageName(sharedObjectName: string): boolean {
        return Boolean(this._cache.sharedObjectsMap[sharedObjectName]);
    }

    getResolvedPackageName(sharedObjectName: string): string {
        if (!this.hasResolvedPackageName(sharedObjectName)) {
            throw new Error(`shared object [${sharedObjectName}] is not cached`);
        }

        return this._cache.sharedObjectsMap[sharedObjectName];
    }

    savePackageName(sharedObjectName: string, packageName: string): void {
        this._cache.sharedObjectsMap[sharedObjectName] = packageName;
    }

    getUnresolvedSharedObjects(): string[] {
        const resolvedSharedObjects = Object.keys(this._cache.sharedObjectsMap);

        return _.difference(this._cache.processedBrowsers.sharedObjects, resolvedSharedObjects);
    }
}
