# Copyright (c) 2021 The Regents of the University of California
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met: redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer;
# redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution;
# neither the name of the copyright holders nor the names of its
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""
This gem5 configuation script creates a simple board to run an X86
"hello world" binary.

This is setup is the close to the simplest setup possible using the gem5
library. It does not contain any kind of caching, IO, or any non-essential
components.

Usage
-----

```
gem5-x86 01-gem5-hello-world.py
```
"""

import m5
from m5.objects import Root

from gem5.isas import ISA
from gem5.utils.requires import requires
from gem5.resources.resource import Resource
from gem5.components.memory import SingleChannelDDR3_1600
from gem5.components.processors.cpu_types import CPUTypes
from gem5.components.boards.simple_board import SimpleBoard
from gem5.components.cachehierarchies.classic.no_cache import NoCache
from gem5.components.processors.simple_processor import SimpleProcessor
from gem5.simulate.simulator import Simulator

if __name__ == "__m5_main__":

    # This check ensures the gem5 binary is compiled to the X86 ISA target. If not,
    # an exception will be thrown.
    requires(isa_required=ISA.X86)

    # In this setup we don't have a cache. `NoCache` can be used for such setups.
    cache_hierarchy = NoCache()

    # We use a single channel DDR3_1600 memory system
    memory = SingleChannelDDR3_1600(size="32MB")

    # We use a simple Timing processor with one core.
    processor = SimpleProcessor(cpu_type=CPUTypes.TIMING, isa=ISA.X86, num_cores=1)

    # The gem5 library simple board which can be used to run simple SE-mode
    # simulations.
    board = SimpleBoard(
        clk_freq="3GHz",
        processor=processor,
        memory=memory,
        cache_hierarchy=cache_hierarchy,
    )

    # Here we set the workload. In this case we want to run a simple "Hello World!"
    # program compiled to the X86 ISA. The `Resource` class will automatically
    # download the binary from the gem5 Resources cloud bucket if it's not already
    # present.
    board.set_se_binary_workload(
        # The `Resource` class reads the `resources.json` file from the gem5
        # resources repository:
        # https://gem5.googlesource.com/public/gem5-resource.
        # Any resource specified in this file will be automatically retrieved.
        # At the time of writing, this file is a WIP and does not contain all
        # resources. Jira ticket: https://gem5.atlassian.net/browse/GEM5-1096
        Resource("x86-hello64-static")
    )

    # Lastly we run the simulation.
    root = Root(full_system=False, system=board)
    m5.instantiate()
    exit_event = (
        m5.simulate()
    )  # m5.simulate() without a parameter will run the simulation until the end

    print(f"Exiting @ tick {m5.curTick()} because {exit_event.getCause()}.")
    print()
