# Copyleft (C) 2021-2024, Marek Gagolewski <https://www.gagolewski.com>
# Adding author and other metadata to every Rd file;
# There is one and only one official manual. Ad- and tracker-free.
# Enjoy the free internet.

library("roxygen2")

postprocess_contents <- function(contents)
{
    stopifnot(is.character(contents), length(contents)==1, !is.na(contents))

    author <- "\\href{https://www.gagolewski.com/}{Marek Gagolewski} and other contributors\n"

    seealso <- stringi::stri_paste(
        "The official online manual of \\pkg{stringi} at ",
        "\\url{https://stringi.gagolewski.com/}\n",
        "\n",
        "Gagolewski M., ",
        "\\pkg{stringi}: Fast and portable character string processing in R, ",
        "\\emph{Journal of Statistical Software} 103(2), 2022, 1-59, ",
        "\\doi{10.18637/jss.v103.i02}\n",
        "\n"
    )

    if (!stringi::stri_detect_regex(
        contents, "% Please edit documentation in R/.*\\\\name", dotall=TRUE)  # fixed comment generated by roxugen2
    ) return(contents)

    if (!stringi::stri_detect_fixed(contents, "\\author")) {
        contents <- stringi::stri_paste(contents,
            "\\author{\n",
            author,
            "}\n")
    }

    if (!stringi::stri_detect_fixed(contents, "\\seealso{\n")) {
        contents <- stringi::stri_paste(
            contents,
            stringi::stri_paste("\\seealso{\n", seealso, "}\n")
        )
    }
    else {
        contents <- stringi::stri_replace_first_fixed(
            contents,
            "\\seealso{\n",
            stringi::stri_paste("\\seealso{\n", seealso)
        )
    }

    contents
}

# ########################################################################### #

# taken from roxygen2 7.3.2 by Hadley Wickham et al.
write_if_different <- function (path, contents, command=NULL, check=TRUE)
{
    if (!file.exists(dirname(path))) {
        dir.create(dirname(path), showWarnings = FALSE)
    }
    name <- basename(path)
    if (check && !made_by_roxygen(path)) {
        cli::cli_inform(c(x = "Skipping {.path {name}}", i = "It already exists and was not generated by roxygen2."))
        return(FALSE)
    }
    line_ending <- detect_line_ending(path)
    contents <- paste0(paste0(contents, collapse = line_ending),
        line_ending)
    contents <- enc2utf8(gsub("\r?\n", line_ending, contents))

    contents <- postprocess_contents(contents)  ######################## PATCH

    if (same_contents(path, contents))
        return(FALSE)


    if (!str_detect(name, "^[a-zA-Z][a-zA-Z0-9_.-]*$")) {
        cli::cli_inform(c(x = "Skipping {.path {name}}", i = "Invalid file name"))
        FALSE
    }
    else {
        if (!is.null(command)) {
            scheme <- "x-r-run"
            url <- paste0(scheme, ":", command)
            name <- cli::style_hyperlink(name, url)
        }
        cli::cli_inform("Writing {.path {name}}")
        writeBin(charToRaw(contents), path)
        TRUE
    }
}

# ########################################################################### #

environment(postprocess_contents) <- environment(roxygen2:::write_if_different)
environment(write_if_different) <- environment(roxygen2:::write_if_different)
unlockBinding("write_if_different", getNamespace("roxygen2"))
assign("write_if_different", write_if_different, getNamespace("roxygen2"))
