import { fromHex, toHex, toHexFormatted, toHexSeparated } from "./hex";

function byteArraysEqual(a: Uint8Array, b: Uint8Array): boolean {
  if (a.length !== b.length) {
    return false;
  }

  for (let i = 0; i < a.length; i++) {
    if (a[i] !== b[i]) {
      return false;
    }
  }

  return true;
}

describe("fromHex", () => {
  it("should convert hex strings to byte arrays", () => {
    expect(byteArraysEqual(fromHex(""), new Uint8Array())).toEqual(true);
    expect(byteArraysEqual(fromHex("00"), new Uint8Array([0]))).toEqual(true);
    expect(byteArraysEqual(fromHex("01"), new Uint8Array([1]))).toEqual(true);
    expect(byteArraysEqual(fromHex("0f"), new Uint8Array([15]))).toEqual(true);
    expect(byteArraysEqual(fromHex("10"), new Uint8Array([16]))).toEqual(true);
    expect(byteArraysEqual(fromHex("ff"), new Uint8Array([255]))).toEqual(true);
    expect(byteArraysEqual(fromHex("fffe"), new Uint8Array([255, 254]))).toEqual(true);
    expect(byteArraysEqual(fromHex("ffff"), new Uint8Array([255, 255]))).toEqual(true);
    expect(byteArraysEqual(fromHex("01020304"), new Uint8Array([1, 2, 3, 4]))).toEqual(true);
    expect(byteArraysEqual(fromHex("52545053"), new Uint8Array([0x52, 0x54, 0x50, 0x53]))).toEqual(true); // prettier-ignore
  });
});

describe("toHex", () => {
  it("should convert byte arrays to hex", () => {
    expect(toHex(new Uint8Array())).toEqual("");
    expect(toHex(new Uint8Array([0]))).toEqual("00");
    expect(toHex(new Uint8Array([1]))).toEqual("01");
    expect(toHex(new Uint8Array([15]))).toEqual("0f");
    expect(toHex(new Uint8Array([16]))).toEqual("10");
    expect(toHex(new Uint8Array([255]))).toEqual("ff");
    expect(toHex(new Uint8Array([255, 254]))).toEqual("fffe");
    expect(toHex(new Uint8Array([255, 255]))).toEqual("ffff");
    expect(toHex(new Uint8Array([1, 2, 3, 4]))).toEqual("01020304");
    expect(toHex(new Uint8Array([0x52, 0x54, 0x50, 0x53]))).toEqual("52545053");
  });
});

describe("toHexSeparated", () => {
  it("should convert byte arrays to hex", () => {
    expect(toHexSeparated(new Uint8Array())).toEqual("");
    expect(toHexSeparated(new Uint8Array(), "!")).toEqual("");
    expect(toHexSeparated(new Uint8Array([0]))).toEqual("00");
    expect(toHexSeparated(new Uint8Array([1]), " ")).toEqual("01");
    expect(toHexSeparated(new Uint8Array([255, 254]))).toEqual("ff:fe");
    expect(toHexSeparated(new Uint8Array([255, 255]), "!")).toEqual("ff!ff");
    expect(toHexSeparated(new Uint8Array([1, 2, 3, 4]), "")).toEqual("01020304");
    expect(toHexSeparated(new Uint8Array([0x52, 0x54, 0x50, 0x53]), "  ")).toEqual(
      "52  54  50  53",
    );
  });
});

describe("toHexFormatted", () => {
  it("should convert a long byte array to a formatted hex dump", () => {
    const data = new Uint8Array([
      0x52, 0x54, 0x50, 0x53, 0x02, 0x01, 0x01, 0x10, 0x5a, 0xb8, 0x10, 0x01, 0x16, 0x36, 0xc7,
      0xd5, 0x18, 0x95, 0xc5, 0x4e, 0x09, 0x01, 0x08, 0x00, 0xb3, 0xee, 0xe9, 0x60, 0xbc, 0xb4,
      0xe1, 0x48, 0x15, 0x05, 0xf8, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x01, 0x00, 0xc2, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
      0x2c, 0x00, 0x10, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x65, 0x6e, 0x63, 0x6c, 0x61, 0x76, 0x65,
      0x3d, 0x2f, 0x3b, 0x00, 0x00, 0x15, 0x00, 0x04, 0x00, 0x02, 0x01, 0x00, 0x00, 0x16, 0x00,
      0x04, 0x00, 0x01, 0x10, 0x00, 0x00, 0x02, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x50, 0x00, 0x10, 0x00, 0x5a, 0xb8, 0x10, 0x01, 0x16, 0x36, 0xc7, 0xd5,
      0x18, 0x95, 0xc5, 0x4e, 0x00, 0x00, 0x01, 0xc1, 0x58, 0x00, 0x04, 0x00, 0x3f, 0x0c, 0x00,
      0x00, 0x0f, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x31, 0x00, 0x18, 0x00, 0x01, 0x00,
      0x00, 0x00, 0xd8, 0xe4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x2e, 0x32, 0x00, 0x18, 0x00, 0x01, 0x00, 0x00, 0x00,
      0xd8, 0xe4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x0a, 0x00, 0x00, 0x2e, 0x07, 0x80, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2c, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2c,
      0x00, 0x00, 0x00, 0x4a, 0x6f, 0x68, 0x6e, 0x73, 0x2d, 0x4d, 0x61, 0x63, 0x42, 0x6f, 0x6f,
      0x6b, 0x2d, 0x50, 0x72, 0x6f, 0x2e, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x2f, 0x30, 0x2e, 0x38,
      0x2e, 0x30, 0x2f, 0x44, 0x61, 0x72, 0x77, 0x69, 0x6e, 0x2f, 0x44, 0x61, 0x72, 0x77, 0x69,
      0x6e, 0x00, 0x19, 0x80, 0x04, 0x00, 0x00, 0x00, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00,
    ]);
    expect(toHexFormatted(data)).toEqual(`52 54 50 53 02 01 01 10  5a b8 10 01 16 36 c7 d5
18 95 c5 4e 09 01 08 00  b3 ee e9 60 bc b4 e1 48
15 05 f8 00 00 00 10 00  00 00 00 00 00 01 00 c2
00 00 00 00 01 00 00 00  00 03 00 00 2c 00 10 00
0a 00 00 00 65 6e 63 6c  61 76 65 3d 2f 3b 00 00
15 00 04 00 02 01 00 00  16 00 04 00 01 10 00 00
02 00 08 00 0a 00 00 00  00 00 00 00 50 00 10 00
5a b8 10 01 16 36 c7 d5  18 95 c5 4e 00 00 01 c1
58 00 04 00 3f 0c 00 00  0f 00 04 00 00 00 00 00
31 00 18 00 01 00 00 00  d8 e4 00 00 00 00 00 00
00 00 00 00 00 00 00 00  0a 00 00 2e 32 00 18 00
01 00 00 00 d8 e4 00 00  00 00 00 00 00 00 00 00
00 00 00 00 0a 00 00 2e  07 80 44 00 00 00 00 00
2c 00 00 00 00 00 00 00  00 00 00 00 00 00 00 00
2c 00 00 00 4a 6f 68 6e  73 2d 4d 61 63 42 6f 6f
6b 2d 50 72 6f 2e 6c 6f  63 61 6c 2f 30 2e 38 2e
30 2f 44 61 72 77 69 6e  2f 44 61 72 77 69 6e 00
19 80 04 00 00 00 10 00  01 00 00 00`);
  });
});
