"Utility macro the vanilla aspect.dev rules_jest to supply common dependencies."

load("@aspect_rules_jest//jest:defs.bzl", _jest_test = "jest_test")
load("@aspect_rules_ts//ts:defs.bzl", "ts_config", "ts_project")

def jest_test(name, data, size = "small", jest_config = "//:jest.config", node_modules = "//:node_modules", snapshots = [], flaky = False, tags = [], no_copy_to_bin = [], **kwargs):
    """A macro around the autogenerated jest_test rule.

    The target to update the snapshot is `{name}_update_snapshots`.

    Args:
        name: target name
        data: list of data sources and dependencies.
        size: test size
        snapshots: snapshot files
        jest_config: jest.config.js file, default to the root one
        node_modules: the node_modules label, default to the root one
        flaky: Whether this test is flaky
        tags: tags
        no_copy_to_bin: no_copy_to_bin
        **kwargs: the rest
    """
    data = data + [
        "//:node_modules/@swc/jest",
        "//:node_modules/@types/jest",
        "//:node_modules/@swc/helpers",
        "//:swcrc",
    ]

    tags = [
        # Need to set the pwd to avoid jest needing a runfiles helper
        # Windows users with permissions can use --enable_runfiles
        # to make this test work
        "no-bazelci-windows",
        # TODO: why does this fail almost all the time, but pass on local Mac?
        "no-bazelci-mac",
    ] + tags

    _jest_test(
        name = name,
        config = jest_config,
        node_modules = node_modules,
        data = data,
        size = size,
        snapshots = snapshots,
        flaky = flaky,
        tags = tags,
        node_options = [
            "--experimental-vm-modules",
        ],
        no_copy_to_bin = no_copy_to_bin,
        **kwargs
    )

def jest_test_v2(name, srcs = [], deps = [], size = "small", tsconfig = "//:tsconfig.es6", jest_config = "//:jest-no-transpile.config", node_modules = "//:node_modules", snapshots = [], flaky = False, tags = [], no_copy_to_bin = [], **kwargs):
    """A macro around the autogenerated jest_test rule.

    The target to update the snapshot is `{name}_update_snapshots`.

    Args:
        name: target name
        srcs: test src files
        deps: deps
        size: test size
        tsconfig: tsconfig.json file, default to the jest one
        snapshots: snapshot files
        jest_config: jest.config.js file, default to the root one
        node_modules: the node_modules label, default to the root one
        flaky: Whether this test is flaky
        tags: tags
        no_copy_to_bin: no_copy_to_bin
        **kwargs: the rest
    """
    deps = deps + [
        "//:node_modules/@types/jest",
    ]

    if tsconfig.endswith(".json"):
        f = tsconfig
        tsconfig = ":%s" % f.replace(".json", "")
        ts_config(
            name = f.replace(".json", ""),
            src = f,
            visibility = ["//visibility:public"],
            deps = ["//:tsconfig"],
            testonly = True,
        )

    ts_project(
        name = "%s-base" % name,
        srcs = srcs,
        deps = deps,
        tsconfig = tsconfig,
        resolve_json_module = True,
        source_map = True,
        declaration = True,
        testonly = True,
    )

    tags = [
        # Need to set the pwd to avoid jest needing a runfiles helper
        # Windows users with permissions can use --enable_runfiles
        # to make this test work
        "no-bazelci-windows",
        # TODO: why does this fail almost all the time, but pass on local Mac?
        "no-bazelci-mac",
    ] + tags

    _jest_test(
        name = name,
        config = jest_config,
        node_modules = node_modules,
        data = [":%s-base" % name] + deps,
        size = size,
        snapshots = snapshots,
        flaky = flaky,
        tags = tags,
        node_options = [
            "--experimental-vm-modules",
        ],
        no_copy_to_bin = no_copy_to_bin,
        **kwargs
    )
