#include "fd_quic_test_helpers.h"
#include "../../../util/net/fd_pcapng.h"
#include <errno.h>
#include <net/if.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include "../../../ballet/txn/fd_txn.h" /* FD_TXN_MTU */
#include "../../../util/net/fd_eth.h"
#include "../../../util/net/fd_ip4.h"

#if defined(__linux__)
#include <linux/if_link.h>
#endif

FILE * fd_quic_test_pcap;

/* Mac address counter, incremented for each new QUIC */
static ulong test_mac_addr_seq = 0x0A0000000000;
/* IP address counter, incremented for each new QUIC */
static uint  test_ip_addr_seq  = FD_IP4_ADDR( 127, 10, 0, 0 );

/* Default implementations of callbacks */

static void
fd_quic_test_cb_conn_new( fd_quic_conn_t * conn,
                          void *           quic_ctx ) {
  FD_LOG_DEBUG(( "cb_conn_new(conn=%p, quic_ctx=%p)",
                 (void *)conn, (void *)quic_ctx ));
}

static void
fd_quic_test_cb_conn_handshake_complete( fd_quic_conn_t * conn,
                                         void *           quic_ctx ) {
  FD_LOG_DEBUG(( "cb_conn_handshake_complete(conn=%p, quic_ctx=%p)",
                 (void *)conn, (void *)quic_ctx ));
}

static void
fd_quic_test_cb_conn_final( fd_quic_conn_t * conn,
                            void *           quic_ctx ) {
  FD_LOG_DEBUG(( "cb_conn_final(conn=%p, quic_ctx=%p)",
                 (void *)conn, (void *)quic_ctx ));
}

static void
fd_quic_test_cb_stream_notify( fd_quic_stream_t * stream,
                               void *             quic_ctx,
                               int                notify_type ) {
  FD_LOG_DEBUG(( "cb_stream_notify(stream=%lu, quic_ctx=%p, notify_type=%d)",
                 stream->stream_id, quic_ctx, notify_type ));
}

static int
fd_quic_test_cb_stream_rx( fd_quic_conn_t * conn,
                           ulong            stream_id,
                           ulong            offset,
                           uchar const *    data,
                           ulong            data_sz,
                           int              fin ) {
  FD_LOG_DEBUG(( "cb_stream_rx(conn=%p, stream=%lu, offset=%lu, data=%p, data_sz=%lu, fin=%d)",
                 (void *)conn, stream_id, offset, (void const *)data, data_sz, fin ));
  return FD_QUIC_SUCCESS;
}

void
fd_quic_test_cb_tls_keylog( void *       quic_ctx,
                            char const * line ) {
  (void)quic_ctx;
  if( fd_quic_test_pcap )
    fd_pcapng_fwrite_tls_key_log( (uchar const *)line, (uint)strlen( line ), fd_quic_test_pcap );
}

ulong
fd_quic_test_now( void * context ) {
  (void)context;
  return (ulong)fd_log_wallclock();
}

/* Test runtime */

void
fd_quic_test_boot( int *    pargc,
                   char *** pargv ) {
  char const * _pcap = fd_env_strip_cmdline_cstr( pargc, pargv, "--pcap", NULL, NULL );

  if( _pcap ) {
    FD_LOG_NOTICE(( "Logging to --pcap %s", _pcap ));
    fd_quic_test_pcap = fopen( _pcap, "ab" );
    FD_TEST( fd_quic_test_pcap );
  }
}

void
fd_quic_test_halt( void ) {
  if( fd_quic_test_pcap ) {
    FD_TEST( 0==fclose( fd_quic_test_pcap ) );
    fd_quic_test_pcap = NULL;
  }
}

/* QUIC creation helper */

void
fd_quic_config_anonymous( fd_quic_t * quic,
                          int         role ) {

  fd_quic_config_t * config = &quic->config;
  config->role = role;

  /* Generate MAC address */
  test_mac_addr_seq++;
  ulong mac_addr_be = fd_ulong_bswap( test_mac_addr_seq )>>16UL;
  memcpy( config->link.src_mac_addr, &mac_addr_be, 6 );

  /* Set destination MAC to dummy */
  static uchar const dst_mac_addr[6] = "\x06\x00\xde\xad\xbe\xef";
  memcpy( config->link.dst_mac_addr, dst_mac_addr, 6 );

  /* Generate IP address */
  test_ip_addr_seq = fd_uint_bswap( fd_uint_bswap( test_ip_addr_seq ) + 1 );
  config->net.ip_addr = test_ip_addr_seq;

  config->net.listen_udp_port   =  9000;
  config->net.ephem_udp_port.lo = 10000;
  config->net.ephem_udp_port.hi = 10100;

  /* Default settings */
  config->idle_timeout     = FD_QUIC_DEFAULT_IDLE_TIMEOUT;
  config->ack_delay        = FD_QUIC_DEFAULT_ACK_DELAY;
  config->ack_threshold    = FD_QUIC_DEFAULT_ACK_THRESHOLD;
  config->initial_rx_max_stream_data = FD_TXN_MTU;

  /* Default callbacks */
  quic->cb.conn_new         = fd_quic_test_cb_conn_new;
  quic->cb.conn_hs_complete = fd_quic_test_cb_conn_handshake_complete;
  quic->cb.conn_final       = fd_quic_test_cb_conn_final;
  quic->cb.stream_notify    = fd_quic_test_cb_stream_notify;
  quic->cb.stream_rx        = fd_quic_test_cb_stream_rx;
  quic->cb.tls_keylog       = fd_quic_test_cb_tls_keylog;
  quic->cb.now              = fd_quic_test_now;
  quic->cb.now_ctx          = NULL;
}

void
fd_quic_config_test_signer( fd_quic_t *              quic,
                            fd_tls_test_sign_ctx_t * sign_ctx ) {
  fd_quic_config_t * config = &quic->config;
  fd_memcpy( config->identity_public_key, sign_ctx->public_key, 32UL );
  config->sign_ctx = sign_ctx;
  config->sign     = fd_tls_test_sign_sign;
}

fd_quic_t *
fd_quic_new_anonymous( fd_wksp_t *              wksp,
                       fd_quic_limits_t const * limits,
                       int                      role,
                       fd_rng_t *               rng ) {
  void * shquic = fd_quic_new( fd_wksp_alloc_laddr( wksp, fd_quic_align(), fd_quic_footprint( limits ), 1UL ), limits );
  FD_TEST( shquic );

  fd_quic_t * quic = fd_quic_join( shquic );
  FD_TEST( quic );

  fd_quic_config_anonymous( quic, role );

  fd_tls_test_sign_ctx_t * sign_ctx = fd_wksp_alloc_laddr( wksp, alignof(fd_tls_test_sign_ctx_t), sizeof(fd_tls_test_sign_ctx_t), 1UL );
  *sign_ctx = fd_tls_test_sign_ctx( rng );
  fd_quic_config_test_signer( quic, sign_ctx );

  return quic;
}

fd_quic_t *
fd_quic_new_anonymous_small( fd_wksp_t * wksp,
                             int         role,
                             fd_rng_t *  rng ) {

  fd_quic_limits_t quic_limits = {
    .conn_cnt           = 1UL,
    .handshake_cnt      = 1UL,
    .conn_id_cnt        = 4UL,
    .inflight_pkt_cnt   = 64UL,
    .tx_buf_sz          = 1UL<<15UL,
    .stream_pool_cnt    = 1024
  };

  return fd_quic_new_anonymous( wksp, &quic_limits, role, rng );
}

static void
fd_quic_virtual_pair_direct( fd_quic_virtual_pair_t * pair,
                             fd_quic_t *              quic_a,
                             fd_quic_t *              quic_b ) {

  pair->quic_a = quic_a;
  pair->quic_b = quic_b;

  fd_aio_t const * rx_a = fd_quic_get_aio_net_rx( quic_a );
  fd_aio_t const * rx_b = fd_quic_get_aio_net_rx( quic_b );

  fd_quic_set_aio_net_tx( quic_a, rx_b );
  fd_quic_set_aio_net_tx( quic_b, rx_a );
}

static void
fd_quic_virtual_pair_pcap( fd_quic_virtual_pair_t * pair,
                           fd_quic_t *              quic_a,
                           fd_quic_t *              quic_b,
                           FILE *                   pcap ) {

  pair->quic_a = quic_a;
  pair->quic_b = quic_b;

  fd_aio_t const * rx_a = fd_quic_get_aio_net_rx( quic_a );
  fd_aio_t const * rx_b = fd_quic_get_aio_net_rx( quic_b );

  /* Write pcapng header */

  FD_TEST( 1UL==fd_aio_pcapng_start( pcap ) );

  /* Install captures */

  FD_TEST( fd_aio_pcapng_join( &pair->quic_b2a, rx_a, pcap ) );
  FD_TEST( fd_aio_pcapng_join( &pair->quic_a2b, rx_b, pcap ) );

  /* Set send target */

  fd_quic_set_aio_net_tx( quic_a, fd_aio_pcapng_get_aio( &pair->quic_a2b ) );
  fd_quic_set_aio_net_tx( quic_b, fd_aio_pcapng_get_aio( &pair->quic_b2a ) );
}

void
fd_quic_virtual_pair_init( fd_quic_virtual_pair_t * pair,
                           fd_quic_t * quic_a,
                           fd_quic_t * quic_b ) {
  memset( pair, 0, sizeof(fd_quic_virtual_pair_t) );
  if( !fd_quic_test_pcap )
    fd_quic_virtual_pair_direct( pair, quic_a, quic_b );
  else
    fd_quic_virtual_pair_pcap  ( pair, quic_a, quic_b, fd_quic_test_pcap );
}

void
fd_quic_virtual_pair_fini( fd_quic_virtual_pair_t * pair ) {
  if( pair->quic_a2b.pcapng ) {
    FD_TEST( fd_aio_pcapng_leave( &pair->quic_a2b ) );
    FD_TEST( fd_aio_pcapng_leave( &pair->quic_b2a ) );
  }
  fd_quic_set_aio_net_tx( pair->quic_a, NULL );
  fd_quic_set_aio_net_tx( pair->quic_b, NULL );
}

fd_quic_udpsock_t *
fd_quic_client_create_udpsock(fd_quic_udpsock_t * udpsock,
                              fd_wksp_t *      wksp,
                              fd_aio_t const * rx_aio,
                              uint listen_ip) {
  ulong        mtu          = 2048UL;
  ulong        rx_depth     = 1024UL;
  ulong        tx_depth     = 1024UL;

  int sock_fd = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP );
  if( FD_UNLIKELY( sock_fd<0 ) ) {
    FD_LOG_WARNING(( "socket(AF_INET,SOCK_DGRAM,IPPROTO_UDP) failed (%i-%s)", errno, fd_io_strerror( errno ) ));
    return NULL;
  }

  struct sockaddr_in listen_addr = {
      .sin_family = AF_INET,
      .sin_addr   = { .s_addr = listen_ip },
      .sin_port   = 0,
  };
  if( FD_UNLIKELY( 0!=bind( sock_fd, (struct sockaddr const *)fd_type_pun_const( &listen_addr ), sizeof(struct sockaddr_in) ) ) ) {
    FD_LOG_WARNING(( "bind(sock_fd) failed (%i-%s)", errno, fd_io_strerror( errno ) ));
    close( sock_fd );
    return NULL;
  }

  void * sock_mem = fd_wksp_alloc_laddr( wksp, fd_udpsock_align(),
                                         fd_udpsock_footprint( mtu, rx_depth, tx_depth ),
                                         1UL );
  if( FD_UNLIKELY( !sock_mem ) ) {
    FD_LOG_WARNING(( "fd_wksp_alloc_laddr() failed" ));
    close( sock_fd );
    return NULL;
  }

  fd_udpsock_t * sock = fd_udpsock_join( fd_udpsock_new( sock_mem, mtu, rx_depth, tx_depth ), sock_fd );
  if( FD_UNLIKELY( !sock ) ) {
    FD_LOG_WARNING(( "fd_udpsock_join() failed" ));
    close( sock_fd );
    fd_wksp_free_laddr( sock_mem );
    return NULL;
  }

  udpsock->type            = FD_QUIC_UDPSOCK_TYPE_UDPSOCK;
  udpsock->wksp            = wksp;
  udpsock->udpsock.sock    = sock;
  udpsock->udpsock.sock_fd = sock_fd;
  udpsock->aio             = fd_udpsock_get_tx( sock );
  udpsock->listen_ip       = fd_udpsock_get_ip4_address( sock );
  udpsock->listen_port     = (ushort)fd_udpsock_get_listen_port( sock );
  fd_udpsock_set_rx( sock, rx_aio );

  FD_LOG_NOTICE(( "UDP socket listening on " FD_IP4_ADDR_FMT ":%u",
      FD_IP4_ADDR_FMT_ARGS( udpsock->listen_ip ), udpsock->listen_port ));
  return udpsock;
}

fd_quic_udpsock_t *
fd_quic_udpsock_create( void *           _sock,
                        int *            pargc,
                        char ***         pargv,
                        fd_wksp_t *      wksp,
                        fd_aio_t const * rx_aio ) {

  /* This kinda sucks, should be cleaned up */

  fd_quic_udpsock_t * quic_sock = (fd_quic_udpsock_t *)_sock;

  char const * if_name       = fd_env_strip_cmdline_cstr  ( pargc, pargv, "--iface",        NULL,      NULL );
  uint         if_queue      = fd_env_strip_cmdline_uint  ( pargc, pargv, "--ifqueue",      NULL,       0U  );
  char const * _src_mac      = fd_env_strip_cmdline_cstr  ( pargc, pargv, "--src-mac",      NULL,      NULL );
  ulong        mtu           = fd_env_strip_cmdline_ulong ( pargc, pargv, "--mtu",          NULL,    2048UL );
  ulong        rx_depth      = fd_env_strip_cmdline_ulong ( pargc, pargv, "--rx-depth",     NULL,    1024UL );
  ulong        tx_depth      = fd_env_strip_cmdline_ulong ( pargc, pargv, "--tx-depth",     NULL,    1024UL );
  ulong        xsk_pkt_cnt   = fd_env_strip_cmdline_ulong ( pargc, pargv, "--xsk-pkt-cnt",  NULL,      32UL );
  char const * _listen_ip    = fd_env_strip_cmdline_cstr  ( pargc, pargv, "--listen-ip",    NULL, "0.0.0.0" );
  ushort       listen_port   = fd_env_strip_cmdline_ushort( pargc, pargv, "--listen-port",  NULL,     9090U );
  char const * net_mode_cstr = fd_env_strip_cmdline_cstr  ( pargc, pargv, "--net-mode",     NULL,  "socket" );
  char const * xdp_mode_cstr = fd_env_strip_cmdline_cstr  ( pargc, pargv, "--xdp-mode",     NULL,     "skb" );

  uint listen_ip = 0;
  if( FD_UNLIKELY( !fd_cstr_to_ip4_addr( _listen_ip, &listen_ip ) ) ) FD_LOG_ERR(( "invalid --listen-ip" ));

  if( FD_UNLIKELY( !if_name ) ) FD_LOG_ERR(( "Missing --iface" ));  /* TODO relax this for regular sockets? */
  uint if_idx = if_nametoindex( if_name );
  if( FD_UNLIKELY( !if_idx ) ) FD_LOG_ERR(( "invalid --iface: %s", if_name ));

  static char mac_cstr[18];
  if( FD_UNLIKELY( !_src_mac ) ) do {
    char path[ 22 + IF_NAMESIZE ];
    FILE * address_file;

    snprintf( path, sizeof(path), "/sys/class/net/%s/address", if_name );

    address_file = fopen( path, "r" );
    if( FD_UNLIKELY( !address_file ) )
      goto detect_fail;

    if( FD_UNLIKELY( 17!=fread( mac_cstr, 1, 17UL, address_file ) ) )
      goto detect_fail;
    mac_cstr[17] = '\0';

    _src_mac = mac_cstr;
    fclose( address_file );
    break;
detect_fail:
    FD_LOG_ERR(( "Missing --src-mac (auto detection failed)" ));
  } while(0);

  int sock_type = 0;
  if( 0==strcmp( net_mode_cstr, "xdp" ) ) {
    sock_type = FD_QUIC_UDPSOCK_TYPE_XSK;
  } else if( 0==strcmp( net_mode_cstr, "socket" ) ) {
    sock_type = FD_QUIC_UDPSOCK_TYPE_UDPSOCK;
  } else {
    FD_LOG_ERR(( "Invalid --net-mode \"%s\"", net_mode_cstr ));
  }
  FD_LOG_NOTICE(( "Using --net-mode %s", net_mode_cstr ));

  quic_sock->listen_ip   = listen_ip;
  quic_sock->listen_port = listen_port;

  if( sock_type == FD_QUIC_UDPSOCK_TYPE_XSK ) {
#   if defined(__linux__)
    uint xdp_mode = 0;
    if( 0==strcmp( xdp_mode_cstr, "skb" ) ) {
      xdp_mode = XDP_FLAGS_SKB_MODE;
    } else if( 0==strcmp( xdp_mode_cstr, "drv" ) ) {
      xdp_mode = XDP_FLAGS_DRV_MODE;
    } else {
      FD_LOG_ERR(( "Invalid --xdp-mode %s", xdp_mode_cstr ));
    }

    FD_TEST( _src_mac );
    if( FD_UNLIKELY( !fd_cstr_to_mac_addr( _src_mac, quic_sock->self_mac ) ) ) FD_LOG_ERR(( "invalid --src-mac" ));

    fd_xdp_session_t * session = fd_xdp_session_init( &quic_sock->xdp.session );
    if( FD_UNLIKELY( !session ) ) {
      FD_LOG_WARNING(( "fd_xdp_session_init failed" ));
      return NULL;
    }

    FD_LOG_NOTICE(( "Adding UDP listener (" FD_IP4_ADDR_FMT ":%u)",
                    FD_IP4_ADDR_FMT_ARGS( quic_sock->listen_ip ), quic_sock->listen_port ));
    if( FD_UNLIKELY( 0!=fd_xdp_listen_udp_port( session, quic_sock->listen_ip, (ushort)quic_sock->listen_port, 0 ) ) ) {
      FD_LOG_WARNING(( "fd_xdp_listen_udp_port failed" ));
      fd_xdp_session_fini( session );
      return NULL;
    }

    ulong xsk_sz = fd_xsk_footprint( mtu, rx_depth, rx_depth, tx_depth, tx_depth );
    if( FD_UNLIKELY( !xsk_sz ) ) {
      FD_LOG_WARNING(( "invalid XSK command-line params" ));
      return NULL;
    }

    FD_LOG_NOTICE(( "Creating XSK" ));
    void * xsk_mem = fd_wksp_alloc_laddr( wksp, fd_xsk_align(), xsk_sz, 1UL );
    fd_xsk_t * xsk = fd_xsk_join( fd_xsk_new( xsk_mem, mtu, rx_depth, rx_depth, tx_depth, tx_depth ) );
    if( FD_UNLIKELY( !xsk ) ) {
      FD_LOG_WARNING(( "fd_xsk_new failed" ));
      fd_wksp_free_laddr( xsk_mem );
      fd_xdp_session_fini( session );
      return NULL;
    }

    /* TODO merge this into new? */
    FD_LOG_NOTICE(( "Binding XSK (--iface %s, --ifqueue %u)", if_name, if_queue ));
    if( FD_UNLIKELY( !fd_xsk_init( xsk, if_idx, if_queue, 0 /* flags */ ) ) ) {
      FD_LOG_WARNING(( "fd_xsk_init failed" ));
      return NULL;
    }

    FD_LOG_NOTICE(( "Creating fd_xsk_aio" ));
    void * xsk_aio_mem =
      fd_wksp_alloc_laddr( wksp, fd_xsk_aio_align(), fd_xsk_aio_footprint( tx_depth, xsk_pkt_cnt ), 1UL );
    fd_xsk_aio_t * xsk_aio = fd_xsk_aio_join( fd_xsk_aio_new( xsk_aio_mem, tx_depth, xsk_pkt_cnt ), xsk );
    if( FD_UNLIKELY( !xsk_aio ) ) {
      FD_LOG_WARNING(( "failed to create fd_xsk_aio" ));
      fd_wksp_free_laddr( xsk_aio );
      fd_wksp_free_laddr( fd_xsk_delete( fd_xsk_leave( xsk ) ) );
      fd_xdp_session_fini( session );
      return NULL;
    }

    FD_LOG_NOTICE(( "Joining fd_xsk_aio" ));

    if( FD_UNLIKELY( !xsk_aio ) ) {
      FD_LOG_WARNING(( "failed to join fd_xsk_aio" ));
      fd_wksp_free_laddr( fd_xsk_aio_delete( fd_xsk_aio_leave( xsk_aio ) ) );
      fd_wksp_free_laddr( fd_xsk_delete( fd_xsk_leave( xsk ) ) );
      fd_xdp_session_fini( session );
      return NULL;
    }

    quic_sock->type        = FD_QUIC_UDPSOCK_TYPE_XSK;
    quic_sock->wksp        = wksp;
    quic_sock->xdp.xsk     = xsk;
    quic_sock->xdp.xsk_aio = xsk_aio;
    quic_sock->aio         = fd_xsk_aio_get_tx( quic_sock->xdp.xsk_aio );
    fd_xsk_aio_set_rx( xsk_aio, rx_aio );

    fd_xdp_link_session_t * link_session =
        fd_xdp_link_session_init( &quic_sock->xdp.link_session, session, if_idx, xdp_mode );
    if( FD_UNLIKELY( !link_session ) ) {
      FD_LOG_WARNING(( "failed to hook iface" ));
      fd_wksp_free_laddr( fd_xsk_aio_delete( fd_xsk_aio_leave( xsk_aio ) ) );
      fd_wksp_free_laddr( fd_xsk_delete( fd_xsk_leave( xsk ) ) );
      fd_xdp_session_fini( session );
      return NULL;
    }

    FD_LOG_NOTICE(( "AF_XDP listening on " FD_IP4_ADDR_FMT ":%u",
                    FD_IP4_ADDR_FMT_ARGS( quic_sock->listen_ip ), quic_sock->listen_port ));
#   else
    (void)if_queue; (void)_src_mac; (void)xdp_mode_cstr; (void)xsk_pkt_cnt;
    FD_LOG_ERR(( "AF_XDP not supported on this platform" ));
#   endif
  } else {
    int sock_fd = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP );
    if( FD_UNLIKELY( sock_fd<0 ) ) {
      FD_LOG_WARNING(( "socket(AF_INET,SOCK_DGRAM,IPPROTO_UDP) failed (%i-%s)", errno, fd_io_strerror( errno ) ));
      return NULL;
    }

    struct sockaddr_in listen_addr = {
      .sin_family = AF_INET,
      .sin_addr   = { .s_addr = listen_ip },
      .sin_port   = (ushort)fd_ushort_bswap( (ushort)listen_port ),
    };
    if( FD_UNLIKELY( 0!=bind( sock_fd, (struct sockaddr const *)fd_type_pun_const( &listen_addr ), sizeof(struct sockaddr_in) ) ) ) {
      FD_LOG_WARNING(( "bind(sock_fd) failed (%i-%s)", errno, fd_io_strerror( errno ) ));
      close( sock_fd );
      return NULL;
    }

    void * sock_mem = fd_wksp_alloc_laddr( wksp, fd_udpsock_align(),
                        fd_udpsock_footprint( mtu, rx_depth, tx_depth ),
                        1UL );
    if( FD_UNLIKELY( !sock_mem ) ) {
      FD_LOG_WARNING(( "fd_wksp_alloc_laddr() failed" ));
      close( sock_fd );
      return NULL;
    }

    fd_udpsock_t * sock = fd_udpsock_join( fd_udpsock_new( sock_mem, mtu, rx_depth, tx_depth ), sock_fd );
    if( FD_UNLIKELY( !sock ) ) {
      FD_LOG_WARNING(( "fd_udpsock_join() failed" ));
      close( sock_fd );
      fd_wksp_free_laddr( sock_mem );
      return NULL;
    }

    quic_sock->type            = FD_QUIC_UDPSOCK_TYPE_UDPSOCK;
    quic_sock->wksp            = wksp;
    quic_sock->udpsock.sock    = sock;
    quic_sock->udpsock.sock_fd = sock_fd;
    quic_sock->aio             = fd_udpsock_get_tx( sock );
    quic_sock->listen_ip       = fd_udpsock_get_ip4_address( sock );
    quic_sock->listen_port     = (ushort)fd_udpsock_get_listen_port( sock );
    fd_udpsock_set_rx( sock, rx_aio );

    FD_LOG_NOTICE(( "UDP socket listening on " FD_IP4_ADDR_FMT ":%u",
                    FD_IP4_ADDR_FMT_ARGS( quic_sock->listen_ip ), quic_sock->listen_port ));
  }

  return quic_sock;
}

void *
fd_quic_udpsock_destroy( fd_quic_udpsock_t * udpsock ) {
  if( FD_UNLIKELY( !udpsock ) )
    return NULL;

  switch( udpsock->type ) {
# if defined(__linux__)
  case FD_QUIC_UDPSOCK_TYPE_XSK:
    fd_xdp_link_session_fini( &udpsock->xdp.link_session );
    fd_xdp_session_fini     ( &udpsock->xdp.session      );
    fd_wksp_free_laddr( fd_xsk_aio_delete( fd_xsk_aio_leave( udpsock->xdp.xsk_aio ) ) );
    fd_wksp_free_laddr( fd_xsk_delete    ( fd_xsk_leave    ( udpsock->xdp.xsk     ) ) );
    break;
# endif
  case FD_QUIC_UDPSOCK_TYPE_UDPSOCK:
    fd_wksp_free_laddr( fd_udpsock_delete( fd_udpsock_leave( udpsock->udpsock.sock ) ) );
    close( udpsock->udpsock.sock_fd );
    break;
  }

  return udpsock;
}

void
fd_quic_udpsock_service( fd_quic_udpsock_t const * udpsock ) {
  switch( udpsock->type ) {
# if defined(__linux__)
  case FD_QUIC_UDPSOCK_TYPE_XSK:
    fd_xsk_aio_service( udpsock->xdp.xsk_aio );
    break;
# endif
  case FD_QUIC_UDPSOCK_TYPE_UDPSOCK:
    fd_udpsock_service( udpsock->udpsock.sock );
    break;
  }
}


fd_quic_netem_t *
fd_quic_netem_init( fd_quic_netem_t * netem,
                    float             thres_drop,
                    float             thres_reorder ) {
  *netem = (fd_quic_netem_t) {
    .thresh_drop    = thres_drop,
    .thresh_reorder = thres_reorder,
  };
  fd_aio_new( &netem->local, netem, fd_quic_netem_send );
  return netem;
}

int
fd_quic_netem_send( void *                    ctx, /* fd_quic_net_em_t */
                    fd_aio_pkt_info_t const * batch,
                    ulong                     batch_cnt,
                    ulong *                   opt_batch_idx,
                    int                       flush ) {
  (void)opt_batch_idx; /* FIXME fd_aio compliance */

  fd_quic_netem_t * mitm_ctx = (fd_quic_netem_t *)ctx;

  /* go packet by packet */
  for( ulong j = 0UL; j < batch_cnt; ++j ) {
    /* generate a random number and compare with threshold, and either pass thru or drop */
    static FD_TL uint seed = 0;
    ulong l = fd_rng_private_expand( seed++ );
    float rnd_num = (float)l * (float)0x1p-64;

    if( rnd_num < mitm_ctx->thresh_drop ) {
      /* dropping behaves as-if the send was successful */
      continue;
    }

    if( rnd_num < mitm_ctx->thresh_reorder ) {
      /* reorder */

      /* logic:
           if we already have a reordered buffer, delay it another packet
           else store the current packet into the reorder buffer */
      if( mitm_ctx->reorder_sz > 0UL ) {
        fd_aio_pkt_info_t lcl_batch[1] = { batch[j] };
        fd_aio_send( mitm_ctx->dst, lcl_batch, 1UL, NULL, flush );

        /* clear buffer */
        mitm_ctx->reorder_sz = 0UL;
      } else {
        fd_memcpy( mitm_ctx->reorder_buf, batch[j].buf, batch[j].buf_sz );
        mitm_ctx->reorder_sz = batch[j].buf_sz;
      }
      continue;
    }

    /* send new packet */
    fd_aio_pkt_info_t batch_0[1] = { batch[j] };
    fd_aio_send( mitm_ctx->dst, batch_0, 1UL, NULL, flush );

    /* we aren't dropping or reordering, but we might have a prior reorder */
    if( mitm_ctx->reorder_sz > 0UL ) {
      fd_aio_pkt_info_t batch_1[1] = {{ .buf = mitm_ctx->reorder_buf, .buf_sz = (ushort)mitm_ctx->reorder_sz }};
      fd_aio_send( mitm_ctx->dst, batch_1, 1UL, NULL, flush );

      /* clear the sent buffer */
      mitm_ctx->reorder_sz = 0UL;
    }
  }

  return FD_AIO_SUCCESS;
}
