#include "fd_exec_slot_ctx.h"
#include "fd_exec_epoch_ctx.h"
#include "../sysvar/fd_sysvar_epoch_schedule.h"
#include "../program/fd_vote_program.h"

#include <assert.h>
#include <time.h>

void *
fd_exec_slot_ctx_new( void *      mem,
                      fd_valloc_t valloc ) {
  if( FD_UNLIKELY( !mem ) ) {
    FD_LOG_WARNING(( "NULL mem" ));
    return NULL;
  }

  if( FD_UNLIKELY( !fd_ulong_is_aligned( (ulong)mem, FD_EXEC_SLOT_CTX_ALIGN ) ) ) {
    FD_LOG_WARNING(( "misaligned mem" ));
    return NULL;
  }

  fd_memset( mem, 0, sizeof(fd_exec_slot_ctx_t) );

  fd_exec_slot_ctx_t * self = (fd_exec_slot_ctx_t *) mem;
  self->valloc = valloc;
  fd_slot_bank_new(&self->slot_bank);

  self->sysvar_cache = fd_sysvar_cache_new( fd_valloc_malloc( valloc, fd_sysvar_cache_align(), fd_sysvar_cache_footprint() ), valloc );
  self->account_compute_table = fd_account_compute_table_join( fd_account_compute_table_new( fd_valloc_malloc( valloc, fd_account_compute_table_align(), fd_account_compute_table_footprint( 10000 ) ), 10000, 0 ) );

  /* This is inactive by default */
  self->epoch_reward_status.discriminant = fd_epoch_reward_status_enum_Inactive;

  FD_COMPILER_MFENCE();
  self->magic = FD_EXEC_SLOT_CTX_MAGIC;
  FD_COMPILER_MFENCE();

  return mem;
}

fd_exec_slot_ctx_t *
fd_exec_slot_ctx_join( void * mem ) {
  if( FD_UNLIKELY( !mem ) ) {
    FD_LOG_WARNING(( "NULL block" ));
    return NULL;
  }

  fd_exec_slot_ctx_t * ctx = (fd_exec_slot_ctx_t *) mem;

  if( FD_UNLIKELY( ctx->magic!=FD_EXEC_SLOT_CTX_MAGIC ) ) {
    FD_LOG_WARNING(( "bad magic" ));
    return NULL;
  }

  return ctx;
}

void *
fd_exec_slot_ctx_leave( fd_exec_slot_ctx_t * ctx) {
  if( FD_UNLIKELY( !ctx ) ) {
    FD_LOG_WARNING(( "NULL block" ));
    return NULL;
  }

  if( FD_UNLIKELY( ctx->magic!=FD_EXEC_SLOT_CTX_MAGIC ) ) {
    FD_LOG_WARNING(( "bad magic" ));
    return NULL;
  }

  return (void *) ctx;
}

void *
fd_exec_slot_ctx_delete( void * mem ) {
  if( FD_UNLIKELY( !mem ) ) {
    FD_LOG_WARNING(( "NULL mem" ));
    return NULL;
  }

  if( FD_UNLIKELY( !fd_ulong_is_aligned( (ulong)mem, FD_EXEC_SLOT_CTX_ALIGN) ) )  {
    FD_LOG_WARNING(( "misaligned mem" ));
    return NULL;
  }

  fd_exec_slot_ctx_t * hdr = (fd_exec_slot_ctx_t *)mem;
  if( FD_UNLIKELY( hdr->magic!=FD_EXEC_SLOT_CTX_MAGIC ) ) {
    FD_LOG_WARNING(( "bad magic" ));
    return NULL;
  }

  fd_bincode_destroy_ctx_t ctx = { .valloc = hdr->valloc };
  fd_slot_bank_destroy(&hdr->slot_bank, &ctx);

  fd_valloc_free( hdr->valloc, fd_sysvar_cache_delete( hdr->sysvar_cache ) );
  hdr->sysvar_cache = NULL;
  fd_valloc_free( hdr->valloc, fd_account_compute_table_delete( fd_account_compute_table_leave( hdr->account_compute_table ) ) );
  hdr->account_compute_table = NULL;

  FD_COMPILER_MFENCE();
  FD_VOLATILE( hdr->magic ) = 0UL;
  FD_COMPILER_MFENCE();

  return mem;
}

/* recover_clock recovers PoH/wallclock synchronization.  Walks all vote
   accounts in current epoch stakes. */

static int
recover_clock( fd_exec_slot_ctx_t * slot_ctx ) {

  fd_epoch_bank_t const * epoch_bank = fd_exec_epoch_ctx_epoch_bank( slot_ctx->epoch_ctx );
  fd_vote_accounts_t const * vote_accounts = &epoch_bank->stakes.vote_accounts;

  fd_vote_accounts_pair_t_mapnode_t * vote_accounts_pool = vote_accounts->vote_accounts_pool;
  fd_vote_accounts_pair_t_mapnode_t * vote_accounts_root = vote_accounts->vote_accounts_root;

  for( fd_vote_accounts_pair_t_mapnode_t * n = fd_vote_accounts_pair_t_map_minimum(vote_accounts_pool, vote_accounts_root);
       n;
       n = fd_vote_accounts_pair_t_map_successor( vote_accounts_pool, n ) ) {
    /* Extract vote timestamp of account */

    fd_vote_block_timestamp_t vote_state_timestamp = {
      .timestamp = n->elem.value.last_timestamp_ts,
      .slot      = n->elem.value.last_timestamp_slot
    };

    /* Record timestamp */
    if( vote_state_timestamp.slot != 0 || n->elem.stake != 0 ) {
      fd_vote_record_timestamp_vote_with_slot(slot_ctx, &n->elem.key, vote_state_timestamp.timestamp, vote_state_timestamp.slot);
    }
  }

  return 1;
}

/* Implementation note: fd_exec_slot_ctx_recover moves objects from
   manifest to slot_ctx.  This function must not share pointers between
   slot_ctx and manifest.  Otherwise, would cause a use-after-free. */

static fd_exec_slot_ctx_t *
fd_exec_slot_ctx_recover_( fd_exec_slot_ctx_t *   slot_ctx,
                           fd_solana_manifest_t * manifest ) {

  fd_exec_epoch_ctx_t * epoch_ctx   = slot_ctx->epoch_ctx;
  fd_epoch_bank_t *     epoch_bank  = fd_exec_epoch_ctx_epoch_bank( epoch_ctx );
  fd_valloc_t           slot_valloc = slot_ctx->valloc;

  /* Clean out prior bank */
  fd_bincode_destroy_ctx_t destroy = { .valloc = slot_valloc };
  fd_slot_bank_t * slot_bank = &slot_ctx->slot_bank;
  fd_slot_bank_destroy( slot_bank, &destroy );
  fd_slot_bank_new( slot_bank );

  for ( fd_vote_accounts_pair_t_mapnode_t * n = fd_vote_accounts_pair_t_map_minimum(
          epoch_bank->stakes.vote_accounts.vote_accounts_pool,
          epoch_bank->stakes.vote_accounts.vote_accounts_root );
          n;
          n = fd_vote_accounts_pair_t_map_successor( epoch_bank->stakes.vote_accounts.vote_accounts_pool, n ) ) {

      const fd_pubkey_t null_pubkey = {{ 0 }};
      if ( memcmp( &n->elem.key, &null_pubkey, FD_PUBKEY_FOOTPRINT ) == 0 ) {
        continue;
      }
  }

  fd_deserializable_versioned_bank_t * oldbank = &manifest->bank;

  /* Populate the epoch context, using the already-allocated statically allocated memory */
  /* Copy stakes */
  epoch_bank->stakes.epoch = oldbank->stakes.epoch;

  /* Copy stakes->vote_accounts */
  for ( fd_vote_accounts_pair_t_mapnode_t * n = fd_vote_accounts_pair_t_map_minimum(
          oldbank->stakes.vote_accounts.vote_accounts_pool,
          oldbank->stakes.vote_accounts.vote_accounts_root );
              n;
              n = fd_vote_accounts_pair_t_map_successor( oldbank->stakes.vote_accounts.vote_accounts_pool, n ) ) {

      const fd_pubkey_t null_pubkey = {{ 0 }};
      if ( memcmp( &n->elem.key, &null_pubkey, FD_PUBKEY_FOOTPRINT ) == 0 ) {
        continue;
      }

      FD_TEST( fd_vote_accounts_pair_t_map_free( epoch_bank->stakes.vote_accounts.vote_accounts_pool ) );
      fd_vote_accounts_pair_t_mapnode_t * new_node = fd_vote_accounts_pair_t_map_acquire( epoch_bank->stakes.vote_accounts.vote_accounts_pool );
      FD_TEST( new_node );
      fd_memcpy( &new_node->elem, &n->elem, FD_VOTE_ACCOUNTS_PAIR_FOOTPRINT );
      fd_vote_accounts_pair_t_map_insert(
        epoch_bank->stakes.vote_accounts.vote_accounts_pool,
        &epoch_bank->stakes.vote_accounts.vote_accounts_root,
        new_node
      );
  }

  /* Copy stakes->stake_delegations */
  for ( fd_delegation_pair_t_mapnode_t * n = fd_delegation_pair_t_map_minimum(
          oldbank->stakes.stake_delegations_pool,
          oldbank->stakes.stake_delegations_root );
          n;
          n = fd_delegation_pair_t_map_successor( oldbank->stakes.stake_delegations_pool, n ) ) {

      const fd_pubkey_t null_pubkey = {{ 0 }};
      if ( memcmp( &n->elem.account, &null_pubkey, FD_PUBKEY_FOOTPRINT ) == 0 ) {
        continue;
      }

      fd_delegation_pair_t_mapnode_t * new_node = fd_delegation_pair_t_map_acquire( epoch_bank->stakes.stake_delegations_pool );
      FD_TEST( new_node );
      fd_memcpy( &new_node->elem, &n->elem, FD_DELEGATION_PAIR_FOOTPRINT );
      fd_delegation_pair_t_map_insert(
        epoch_bank->stakes.stake_delegations_pool,
        &epoch_bank->stakes.stake_delegations_root,
        new_node
      );
  }

  /* Copy stakes->stake_history */
  for ( fd_stake_history_treap_fwd_iter_t iter = fd_stake_history_treap_fwd_iter_init(
          oldbank->stakes.stake_history.treap,
          oldbank->stakes.stake_history.pool );
          !fd_stake_history_treap_fwd_iter_done( iter );
          iter = fd_stake_history_treap_fwd_iter_next( iter, oldbank->stakes.stake_history.pool ) ) {

    fd_stake_history_entry_t const * ele = fd_stake_history_treap_fwd_iter_ele_const( iter, oldbank->stakes.stake_history.pool );

    FD_TEST( fd_stake_history_pool_free( epoch_bank->stakes.stake_history.pool ) );
    fd_stake_history_entry_t * new_ele = fd_stake_history_pool_ele_acquire( epoch_bank->stakes.stake_history.pool );

    new_ele->epoch        = ele->epoch;
    new_ele->activating   = ele->activating;
    new_ele->deactivating = ele->deactivating;
    new_ele->effective    = ele->effective;

    epoch_bank->stakes.stake_history.treap = fd_stake_history_treap_ele_insert(
      epoch_bank->stakes.stake_history.treap,
      new_ele,
      epoch_bank->stakes.stake_history.pool
    );
  }

  fd_stakes_destroy( &oldbank->stakes, &destroy );

  /* Index vote accounts */

  /* Copy over fields */

  if( oldbank->blockhash_queue.last_hash )
    slot_bank->poh = *oldbank->blockhash_queue.last_hash;
  slot_bank->slot = oldbank->slot;
  slot_bank->prev_slot = oldbank->parent_slot;
  fd_memcpy(&slot_bank->banks_hash, &oldbank->hash, sizeof(oldbank->hash));
  fd_memcpy(&slot_bank->fee_rate_governor, &oldbank->fee_rate_governor, sizeof(oldbank->fee_rate_governor));
  slot_bank->lamports_per_signature = manifest->lamports_per_signature;
  slot_ctx->prev_lamports_per_signature = manifest->lamports_per_signature;
  slot_ctx->parent_signature_cnt = oldbank->signature_count;
  if( oldbank->hashes_per_tick )
    epoch_bank->hashes_per_tick = *oldbank->hashes_per_tick;
  else
    epoch_bank->hashes_per_tick = 0;
  epoch_bank->ticks_per_slot = oldbank->ticks_per_slot;
  fd_memcpy(&epoch_bank->ns_per_slot, &oldbank->ns_per_slot, sizeof(oldbank->ns_per_slot));
  epoch_bank->genesis_creation_time = oldbank->genesis_creation_time;
  epoch_bank->slots_per_year = oldbank->slots_per_year;
  slot_bank->max_tick_height = oldbank->max_tick_height;
  fd_memcpy( &epoch_bank->inflation, &oldbank->inflation, FD_INFLATION_FOOTPRINT );
  fd_memcpy( &epoch_bank->epoch_schedule, &oldbank->epoch_schedule, FD_EPOCH_SCHEDULE_FOOTPRINT );
  epoch_bank->rent = oldbank->rent_collector.rent;
  fd_memcpy( &epoch_bank->rent, &oldbank->rent_collector.rent, FD_RENT_FOOTPRINT );
  fd_memcpy( &epoch_bank->rent_epoch_schedule, &oldbank->rent_collector.epoch_schedule, FD_EPOCH_SCHEDULE_FOOTPRINT );

  if( manifest->epoch_account_hash )
    slot_bank->epoch_account_hash = *manifest->epoch_account_hash;

  slot_bank->collected_rent = oldbank->collected_rent;
  // did they not change the bank?!
  slot_bank->collected_execution_fees = oldbank->collector_fees;
  slot_bank->collected_priority_fees = 0;
  slot_bank->capitalization = oldbank->capitalization;
  slot_bank->block_height = oldbank->block_height;
  slot_bank->transaction_count = oldbank->transaction_count;
  if ( oldbank->blockhash_queue.last_hash ) {
    slot_bank->block_hash_queue.last_hash = fd_valloc_malloc( slot_ctx->valloc, FD_HASH_ALIGN, FD_HASH_FOOTPRINT );
    fd_memcpy( slot_bank->block_hash_queue.last_hash, oldbank->blockhash_queue.last_hash, sizeof(fd_hash_t) );
  } else {
    slot_bank->block_hash_queue.last_hash = NULL;
  }
  slot_bank->block_hash_queue.last_hash_index = oldbank->blockhash_queue.last_hash_index;
  slot_bank->block_hash_queue.max_age = oldbank->blockhash_queue.max_age;
  slot_bank->block_hash_queue.ages_root = NULL;
  slot_bank->block_hash_queue.ages_pool = fd_hash_hash_age_pair_t_map_alloc( slot_ctx->valloc, 400 );
  for ( ulong i = 0; i < oldbank->blockhash_queue.ages_len; i++ ) {
    fd_hash_hash_age_pair_t * elem = &oldbank->blockhash_queue.ages[i];
    fd_hash_hash_age_pair_t_mapnode_t * node = fd_hash_hash_age_pair_t_map_acquire( slot_bank->block_hash_queue.ages_pool );
    fd_memcpy( &node->elem, elem, FD_HASH_HASH_AGE_PAIR_FOOTPRINT );
    fd_hash_hash_age_pair_t_map_insert( slot_bank->block_hash_queue.ages_pool, &slot_bank->block_hash_queue.ages_root, node );
  }

  recover_clock( slot_ctx );

  /* Update last restart slot
     https://github.com/solana-labs/solana/blob/30531d7a5b74f914dde53bfbb0bc2144f2ac92bb/runtime/src/bank.rs#L2152

     oldbank->hard_forks is sorted ascending by slot number.
     To find the last restart slot, take the highest hard fork slot
     number that is less or equal than the current slot number.
     (There might be some hard forks in the future, ignore these) */
  do {
    slot_bank->last_restart_slot.slot = 0UL;
    if( FD_UNLIKELY( oldbank->hard_forks.hard_forks_len == 0 ) ) {
      /* SIMD-0047: The first restart slot should be `0` */
      break;
    }

    fd_slot_pair_t const * head = oldbank->hard_forks.hard_forks;
    fd_slot_pair_t const * tail = head + oldbank->hard_forks.hard_forks_len - 1UL;

    for( fd_slot_pair_t const *pair = tail; pair >= head; pair-- ) {
      if( pair->slot <= slot_bank->slot ) {
        slot_bank->last_restart_slot.slot = pair->slot;
        break;
      }
    }
  } while (0);

  /* Move EpochStakes */
  do {
    ulong epoch = fd_slot_to_epoch( &epoch_bank->epoch_schedule, slot_bank->slot, NULL );

    /* We need to save the vote accounts for the current epoch and the next 
       epoch as it is used to calculate the leader schedule at the epoch
       boundary. */

    fd_vote_accounts_t curr_stakes = { .vote_accounts_pool = NULL, .vote_accounts_root = NULL };
    fd_vote_accounts_t next_stakes = { .vote_accounts_pool = NULL, .vote_accounts_root = NULL };

    for( ulong i=0UL; i<manifest->bank.epoch_stakes_len; i++ ) {
      if( manifest->bank.epoch_stakes[i].key == epoch ) {
        curr_stakes.vote_accounts_pool = manifest->bank.epoch_stakes[i].value.stakes.vote_accounts.vote_accounts_pool;
        curr_stakes.vote_accounts_root = manifest->bank.epoch_stakes[i].value.stakes.vote_accounts.vote_accounts_root;
      }
      if( manifest->bank.epoch_stakes[i].key == epoch+1UL ) {
        next_stakes.vote_accounts_pool = manifest->bank.epoch_stakes[i].value.stakes.vote_accounts.vote_accounts_pool;
        next_stakes.vote_accounts_root = manifest->bank.epoch_stakes[i].value.stakes.vote_accounts.vote_accounts_root;
      }

      /* When loading from a snapshot, Agave's stake caches mean that we have to special-case the epoch stakes
         that are used for the second epoch E+2 after the snapshot epoch E.

         If the snapshot contains the epoch stakes for E+2, we should use those.

         If the snapshot does not, we should use the stakes at the end of the E-1 epoch, instead of E-2 as we do for
         all other epochs. */

      if( manifest->bank.epoch_stakes[i].key==epoch+2UL ) {
        slot_ctx->slot_bank.has_use_preceeding_epoch_stakes = 0;
      }
    }

    for( ulong i=0UL; i<manifest->versioned_epoch_stakes_len; i++ ) {
      if( manifest->versioned_epoch_stakes[i].epoch == epoch ) {
        curr_stakes.vote_accounts_pool = manifest->versioned_epoch_stakes[i].val.inner.Current.stakes.vote_accounts.vote_accounts_pool;
        curr_stakes.vote_accounts_root = manifest->versioned_epoch_stakes[i].val.inner.Current.stakes.vote_accounts.vote_accounts_root;
      }
      if( manifest->versioned_epoch_stakes[i].epoch == epoch+1UL ) {
        next_stakes.vote_accounts_pool = manifest->versioned_epoch_stakes[i].val.inner.Current.stakes.vote_accounts.vote_accounts_pool;
        next_stakes.vote_accounts_root = manifest->versioned_epoch_stakes[i].val.inner.Current.stakes.vote_accounts.vote_accounts_root;
      }

      if( manifest->versioned_epoch_stakes[i].epoch==epoch+2UL ) {
        slot_ctx->slot_bank.has_use_preceeding_epoch_stakes = 0;
      }
    }

    slot_ctx->slot_bank.has_use_preceeding_epoch_stakes = 1;
    slot_ctx->slot_bank.use_preceeding_epoch_stakes     = epoch + 2UL;

    if( FD_UNLIKELY( (!curr_stakes.vote_accounts_root) | (!next_stakes.vote_accounts_root) ) ) {
      FD_LOG_WARNING(( "snapshot missing EpochStakes for epochs %lu and/or %lu", epoch, epoch+1UL ));
      return 0;
    }

    /* Move current EpochStakes */

    slot_ctx->slot_bank.epoch_stakes.vote_accounts_pool =
      fd_vote_accounts_pair_t_map_alloc( slot_ctx->valloc, 100000 );  /* FIXME remove magic constant */
    slot_ctx->slot_bank.epoch_stakes.vote_accounts_root = NULL;

    for ( fd_vote_accounts_pair_t_mapnode_t * n = fd_vote_accounts_pair_t_map_minimum(
          curr_stakes.vote_accounts_pool,
          curr_stakes.vote_accounts_root );
          n;
          n = fd_vote_accounts_pair_t_map_successor( curr_stakes.vote_accounts_pool, n ) ) {

        fd_vote_accounts_pair_t_mapnode_t * elem = fd_vote_accounts_pair_t_map_acquire(
          slot_ctx->slot_bank.epoch_stakes.vote_accounts_pool );
        FD_TEST( elem );

        fd_memcpy( &elem->elem, &n->elem, sizeof(fd_vote_accounts_pair_t));

        fd_vote_accounts_pair_t_map_insert(
          slot_ctx->slot_bank.epoch_stakes.vote_accounts_pool,
          &slot_ctx->slot_bank.epoch_stakes.vote_accounts_root,
          elem );
    }

    fd_vote_accounts_destroy( &curr_stakes, &destroy );

    /* Move next EpochStakes
       TODO Can we derive this instead of trusting the snapshot? */

    fd_vote_accounts_pair_t_mapnode_t * pool = next_stakes.vote_accounts_pool;
    fd_vote_accounts_pair_t_mapnode_t * root = next_stakes.vote_accounts_root;

    for ( fd_vote_accounts_pair_t_mapnode_t * n = fd_vote_accounts_pair_t_map_minimum(pool, root);
          n;
          n = fd_vote_accounts_pair_t_map_successor(pool, n) ) {

      fd_vote_accounts_pair_t_mapnode_t * elem = fd_vote_accounts_pair_t_map_acquire(
        epoch_bank->next_epoch_stakes.vote_accounts_pool );
      FD_TEST( elem );

      fd_memcpy( &elem->elem, &n->elem, sizeof(fd_vote_accounts_pair_t));

      fd_vote_accounts_pair_t_map_insert(
        epoch_bank->next_epoch_stakes.vote_accounts_pool,
        &epoch_bank->next_epoch_stakes.vote_accounts_root,
        elem );

    }

    fd_vote_accounts_destroy( &next_stakes, &destroy );
  } while(0);

  // TODO Backup to database
  //int result = fd_runtime_save_epoch_bank(slot_ctx);
  //if( result != FD_EXECUTOR_INSTR_SUCCESS ) {
  //  FD_LOG_WARNING(("save epoch bank failed"));
  //  return result;
  //}
  //
  //return fd_runtime_save_slot_bank(slot_ctx);

  return slot_ctx;
}

fd_exec_slot_ctx_t *
fd_exec_slot_ctx_recover( fd_exec_slot_ctx_t *   slot_ctx,
                          fd_solana_manifest_t * manifest ) {

  fd_exec_slot_ctx_t * res = fd_exec_slot_ctx_recover_( slot_ctx, manifest );

  /* Regardless of result, always destroy manifest */
  fd_bincode_destroy_ctx_t destroy = { .valloc = slot_ctx->valloc };
  fd_solana_manifest_destroy( manifest, &destroy );
  fd_memset( manifest, 0, sizeof(fd_solana_manifest_t) );

  return res;
}

fd_exec_slot_ctx_t *
fd_exec_slot_ctx_recover_status_cache( fd_exec_slot_ctx_t *    ctx,
                                       fd_bank_slot_deltas_t * slot_deltas ) {
  fd_txncache_t * status_cache = ctx->status_cache;
  if( !status_cache ) {
    FD_LOG_WARNING(("No status cache in slot ctx"));
    return NULL;
  }

  FD_SCRATCH_SCOPE_BEGIN {
    ulong num_entries = 0;
    for( ulong i = 0; i < slot_deltas->slot_deltas_len; i++ ) {
      fd_slot_delta_t * slot_delta = &slot_deltas->slot_deltas[i];
      for( ulong j = 0; j < slot_delta->slot_delta_vec_len; j++ ) {
        num_entries += slot_delta->slot_delta_vec[j].value.statuses_len;
      }
    }
    fd_txncache_insert_t * insert_vals = fd_scratch_alloc( alignof(fd_txncache_insert_t), num_entries * sizeof(fd_txncache_insert_t) );

    /* Dumb sort for 300 slot entries to insert in order. */
    fd_slot_delta_t ** deltas = fd_scratch_alloc(alignof(fd_slot_delta_t *), slot_deltas->slot_deltas_len * sizeof(fd_slot_delta_t *));

    ulong curr = 0;
    for( ulong i = 0; i < slot_deltas->slot_deltas_len; i++ ) {
      ulong curr_min = ULONG_MAX;
      ulong curr_min_idx = ULONG_MAX;
      for( ulong j = 0; j < slot_deltas->slot_deltas_len; j++ ) {
        fd_slot_delta_t * slot_delta = &slot_deltas->slot_deltas[j];
        if( slot_delta->slot <= curr ) continue;

        if( curr_min > slot_delta->slot ) {
          curr_min = slot_delta->slot;
          curr_min_idx = j;
        }
      }
      deltas[i] = &slot_deltas->slot_deltas[curr_min_idx];
      curr = slot_deltas->slot_deltas[curr_min_idx].slot;
    }

    ulong idx = 0;
    for( ulong i = 0; i < slot_deltas->slot_deltas_len; i++ ) {
      fd_slot_delta_t * slot_delta = deltas[i];
      ulong slot = slot_delta->slot;
      if( slot_delta->is_root ) {
        fd_txncache_register_root_slot( ctx->status_cache, slot );
      }
      for( ulong j = 0; j < slot_delta->slot_delta_vec_len; j++ ) {
        fd_status_pair_t * pair = &slot_delta->slot_delta_vec[j];
        fd_hash_t * blockhash = &pair->hash;

        for( ulong k = 0; k < pair->value.statuses_len; k++ ) {
          fd_cache_status_t * status = &pair->value.statuses[k];
          uchar result = (uchar)status->result.discriminant;
          insert_vals[idx++] = (fd_txncache_insert_t){
            .blockhash = blockhash->uc,
            .slot = slot,
            .txnhash = status->key_slice,
            .result = &result
          };
        }
      }
    }
    fd_txncache_insert_batch( ctx->status_cache, insert_vals, num_entries );

    for( ulong i = 0; i < slot_deltas->slot_deltas_len; i++ ) {
      fd_slot_delta_t * slot_delta = deltas[i];
      ulong slot = slot_delta->slot;
      for( ulong j = 0; j < slot_delta->slot_delta_vec_len; j++ ) {
        fd_status_pair_t * pair = &slot_delta->slot_delta_vec[j];
        fd_hash_t * blockhash = &pair->hash;
        fd_txncache_set_txnhash_offset( ctx->status_cache, slot, blockhash->uc, pair->value.txn_idx );
      }
    }
  } FD_SCRATCH_SCOPE_END;
  return ctx;
}

void
fd_exec_slot_ctx_free( fd_exec_slot_ctx_t * slot_ctx ) {
  fd_bincode_destroy_ctx_t ctx;
  ctx.valloc = slot_ctx->valloc;
  fd_slot_bank_destroy( &slot_ctx->slot_bank, &ctx );

  /* leader points to a caller-allocated leader schedule */
  fd_exec_slot_ctx_delete( fd_exec_slot_ctx_leave( slot_ctx ) );
}
