#ifndef HEADER_fd_src_ballet_shred_fd_wsample_h
#define HEADER_fd_src_ballet_shred_fd_wsample_h

/* This header defines methods for computing weighted "random" samples
   of the specific type used by Solana for computing the leader
   schedule and the Turbine tree.

   In particular, those random samples are generated by:

   1. Seeding ChaCha20 with a 32B seed (different for leader schedule vs
      Turbine).
   2. Using the ChaCha20 random stream to generate a uniformly
      distributed random integer x in [0, total_stake_weight).
   3. Returning the first index such that the cumulative sum of stake is
      at least x. */

#include "../fd_ballet_base.h"
#include "../chacha20/fd_chacha20rng.h"

struct fd_wsample_private;
typedef struct fd_wsample_private fd_wsample_t;

#define FD_WSAMPLE_ALIGN (64UL)
/* fd_leaders really wants a compile time-compatible footprint... The
   internal count is 1/8 * (9^ceil(log_9(ele_cnt)) - 1) */
#define FD_WSAMPLE_FOOTPRINT( ele_cnt, restore_enabled )                   \
   (192UL + 64UL*((restore_enabled)?2UL:1UL)*(                             \
                 ((ele_cnt)<=         1UL)?        0UL :                   \
                 ((ele_cnt)<=         9UL)?        1UL :                   \
                 ((ele_cnt)<=        81UL)?       10UL :                   \
                 ((ele_cnt)<=       729UL)?       91UL :                   \
                 ((ele_cnt)<=      6561UL)?      820UL :                   \
                 ((ele_cnt)<=     59049UL)?     7381UL :                   \
                 ((ele_cnt)<=    531441UL)?    66430UL :                   \
                 ((ele_cnt)<=   4782969UL)?   597871UL :                   \
                 ((ele_cnt)<=  43046721UL)?  5380840UL :                   \
                 ((ele_cnt)<= 387420489UL)? 48427561UL :                   \
                 ((ele_cnt)<=3486784401UL)?435848050UL : 3922632451UL ))   \

/* fd_wsample_{align, footprint} give the alignment and footprint
   respectively required to create a weighted sampler with at most
   ele_cnt stake weights.  If restore_enabled is zero, calls to
   wsample_restore_all will be no-ops, but the footprint required will
   be smaller. ele_cnt in [0, UINT_MAX-2) (note, not ULONG MAX).

   fd_wsample_{join,leave} join and leave a memory region formatted as a
   weighted sampler, respectively.  They both are simple casts.

   fd_wsample_delete unformats a memory region used as a weighted
   sampler.  Releases all interest in rng. */

FD_FN_CONST ulong fd_wsample_align    ( void          );
FD_FN_CONST ulong fd_wsample_footprint( ulong ele_cnt, int restore_enabled );
fd_wsample_t *    fd_wsample_join     ( void * shmem  );
void *            fd_wsample_leave    ( fd_wsample_t * sampler );
void *            fd_wsample_delete   ( void * shmem  );


/* FD_WSAMPLE_HINT_*: Hints that can be passed to fd_wsample_new in the
   opt_hint field.  The hint specifies the distribution of weights:
   FLAT: The all weights are approximately constant.
   POWERLAW: The weights are sorted largest to smallest and decay with a
       power law distribution.  At the moment, this assumes it's
       proportional to 1/x.

   The hint can also specify whether sampling will be done with deleting
   or without deleting:
   REMOVE: Sampling will be done without replacement, i.e. mostly with
       fd_wsample_sample_and_remove and/or
       fd_wsample_sample_and_remove_many.
   NOREMOVE: Sampling will be done with replacement, i.e. mostly with
       fd_wsample_sample and/or
       fd_wsample_sample_many. */
#define FD_WSAMPLE_HINT_FLAT              0
#define FD_WSAMPLE_HINT_POWERLAW_NOREMOVE 2
#define FD_WSAMPLE_HINT_POWERLAW_REMOVE   3

/* fd_wsample_new_init, fd_wsample_new_add_weight, and
   fd_wsample_new_fini format a memory region with the appropriate
   alignment and footprint to be usable as a weighted sampler.  This
   multi-function initialization process prevents needing to construct a
   flat array of weights, which is often inconvenient.

   The caller must first call fd_wsample_new_init, then new_add_weight
   ele_cnt times, and finally new_fini.  Only at that point will the
   region of memory be ready to be joined.

   fd_wsample_new_init begins the formatting a memory region.  shmem is
   a pointer to the first byte of the memory region to use.  rng must be
   a local join of a ChaCha20 RNG struct.  The weighted sampler will use
   rng to generate random numbers.  It may seem more natural for the
   weighted sampler to own its own rng, but this is done to facilitate
   sharing of rngs between weighted samplers, which is useful for
   Turbine.  ele_cnt specifies the number of elements that can be
   sampled from and must be less than UINT_MAX.  If restore_enabled is
   set to 0, fd_wsample_restore_all will not work but the required
   footprint is smaller.  opt_hint gives a hint of the shape of the
   weights and the style of queries that will be most common; this hint
   impacts query performance but not correctness.  opt_hint must be one
   of FD_WSAMPLE_HINT_*.

   fd_wsample_new_add adds a weight to a partially formatted memory
   region.  shmem must be a partially constructed region of memory, as
   returned by fd_wsample_new_init or fd_wsample_new_add_weight, weight
   must be strictly positive, and the cumulative sum of this weight and
   all other weights must be no more than ULONG_MAX.

   fd_wsample_new_fini finalizes the formatting of a partially formatted
   memory region.  shmem must be a partially constructed region of
   memory, as returned by fd_wsample_new_add_weight (or
   fd_wsample_new_init if ele_cnt==0).  If poisoned_weight is non-zero,
   the weighted sampler will end with a poisoned region representing an
   indeterminate number of unknown elements with total weight equal to
   poisoned_weight.  This is useful for chopping off a long tail so that
   the number of elements can be bounded easily.  The sum of all weights
   and poisoned_weight must be no more than ULONG_MAX.

   Retains read/write interest in rng.

   Each function returns shmem on success and NULL on failure.  It's
   safe to pass NULL as shmem, in which case NULL will be returned, so
   you only need to check the final result.

   On successful completion of the formatting process, the weighted
   sampler will contain an element corresponding to each provided
   weight.  Caller is not joined on return. */
void * fd_wsample_new_init( void             * shmem,
                            fd_chacha20rng_t * rng,
                            ulong              ele_cnt,
                            int                restore_enabled,
                            int                opt_hint );
void * fd_wsample_new_add ( void * shmem, ulong weight          );
void * fd_wsample_new_fini( void * shmem, ulong poisoned_weight );

/* fd_wsample_get_rng returns the value provided for rng in new. */
fd_chacha20rng_t * fd_wsample_get_rng( fd_wsample_t * sampler );

/* fd_wsample_seed_rng seeds the ChaCha20 rng with the provided seed in
   preparation for sampling.  This function is compatible with Solana's
   ChaChaRng::from_seed. */
void fd_wsample_seed_rng( fd_chacha20rng_t * rng, uchar seed[static 32] );

/* fd_wsample_sample{_and_remove}{,_many} produces one or cnt (in the
   _many case) weighted random samples from the sampler.  If the
   _and_remove variant of the function is called, the returned node will
   be temporarily removed from the sampler, i.e. for sampling without
   replacement.  Random samples are produced using the Solana-required
   method.  Sampler's RNG must be seeded appropriately
   prior to using these functions.

   The _many variants of the function store the ith index they sampled
   in idxs[i] for i in [0, cnt).  The other variants of the function
   simply return the sampled index.

   If the sampler has no unremoved elements, these functions will
   return/store FD_WSAMPLE_EMPTY.

   If the RNG sample lands in the poisoned region, these functions will
   return/store FD_WSAMPLE_INDETERMINATE.  If such a sample is produced
   in the without replacement mode, all subsequent calls (with and
   without replacement) will also return FD_WSAMPLE_INDETERMINATE until
   the next call to fd_wsample_restore_all.  This is necessary because
   the poisoned region represents an indeterminate number of elements,
   so it's not possible to know how removing one of them will affect the
   total weight, and thus all subsequent random samples.

   For each index i, fd_wsample_sample returns i with
   probability weights[i]/sum(weights), only considering weights of
   elements that have not been removed. */
#define FD_WSAMPLE_EMPTY          UINT_MAX
#define FD_WSAMPLE_INDETERMINATE (UINT_MAX-1UL)
ulong fd_wsample_sample                ( fd_wsample_t * sampler );
ulong fd_wsample_sample_and_remove     ( fd_wsample_t * sampler );
void  fd_wsample_sample_many           ( fd_wsample_t * sampler, ulong * idxs, ulong cnt );
void  fd_wsample_sample_and_remove_many( fd_wsample_t * sampler, ulong * idxs, ulong cnt );

/* fd_wsample_remove_idx removes an element by index as if it had been selected
   for sampling without replacement.  Unless restore_all is called, this
   index will no longer be returned by any of the sample methods, and
   its weight will be excluded from the remaining sampling operations.
   Removing an element that has already been removed is a no-op.  idx
   must be in [0, ele_cnt). */
void fd_wsample_remove_idx( fd_wsample_t * sampler, ulong idx );

/* fd_wsample_restore_all restores all elements removed with one of the
   sample_and_remove functions or with fd_wsample_remove_idx.  The
   elements with have their original weight.  This is faster than
   recreating the weighted sampler, even if all elements have been
   removed.  Returns sampler on success and NULL on failure.  The only
   error case is if sampler was constructed with restore_enabled set to 0,
   in which case no elements are restored. */
fd_wsample_t * fd_wsample_restore_all( fd_wsample_t * sampler );



#endif /* HEADER_fd_src_ballet_shred_fd_wsample_h */
