<?php

/**
 * @copyright Copyright (C) 2015-2021 AIZAWA Hina
 * @license https://github.com/fetus-hina/stat.ink/blob/master/LICENSE MIT
 * @author AIZAWA Hina <hina@fetus.jp>
 */

namespace app\models;

use DateTimeZone;
use Yii;
use app\components\helpers\DateTimeFormatter;
use app\components\helpers\Translator;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\helpers\ArrayHelper;

use function array_map;
use function array_merge;
use function strtotime;

use const SORT_ASC;

/**
 * This is the model class for table "map".
 *
 * @property integer $id
 * @property string $key
 * @property string $name
 * @property integer $area
 * @property string $release_at
 * @property string $short_name
 *
 * @property Battle[] $battles
 * @property PeriodMap[] $periodMaps
 * @property SplapiMap[] $splapiMaps
 */
final class Map extends ActiveRecord
{
    use SafeFindOneTrait;
    use openapi\Util;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'map';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['key', 'name', 'short_name'], 'required'],
            [['area'], 'integer'],
            [['release_at'], 'safe'],
            [['key', 'short_name'], 'string', 'max' => 16],
            [['name'], 'string', 'max' => 32],
            [['key'], 'unique'],
            [['name'], 'unique'],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'key' => 'Key',
            'name' => 'Name',
            'area' => 'Area',
            'release_at' => 'Release At',
            'short_name' => 'Short Name',
        ];
    }

    /**
     * @return ActiveQuery
     */
    public function getBattles()
    {
        return $this->hasMany(Battle::class, ['map_id' => 'id']);
    }

    /**
     * @return ActiveQuery
     */
    public function getPeriodMaps()
    {
        return $this->hasMany(PeriodMap::class, ['map_id' => 'id']);
    }

    /**
     * @return ActiveQuery
     */
    public function getSplapiMaps()
    {
        return $this->hasMany(SplapiMap::class, ['map_id' => 'id']);
    }

    public function toJsonArray()
    {
        $t = $this->release_at ? strtotime($this->release_at) : null;
        return [
            'key' => $this->key,
            'name' => Translator::translateToAll('app-map', $this->name),
            'area' => $this->area,
            'release_at' => $t
                ? DateTimeFormatter::unixTimeToJsonArray(
                    $t,
                    new DateTimeZone('Etc/UTC'),
                )
                : null,
        ];
    }

    public static function openApiSchema(): array
    {
        $values = static::find()
            ->orderBy(['key' => SORT_ASC])
            ->all();
        return [
            'type' => 'object',
            'description' => Yii::t('app-apidoc1', 'Mode information'),
            'properties' => [
                'key' => static::oapiKey(
                    static::oapiKeyValueTable(
                        Yii::t('app-apidoc1', 'Stage'),
                        'app-map',
                        $values,
                    ),
                    ArrayHelper::getColumn($values, 'key', false),
                ),
                'name' => static::oapiRef(openapi\Name::class),
                'area' => [
                    'type' => 'integer',
                    'format' => 'int32',
                    'nullable' => true,
                    'description' => Yii::t('app-apidoc1', 'Total area'),
                ],
                'release_at' => array_merge(openapi\DateTime::openApiSchema(), [
                    'description' => Yii::t('app-apidoc1', 'Date and time when ready to play'),
                    'nullable' => true,
                ]),
            ],
            'example' => $values[0]->toJsonArray(),
        ];
    }

    public static function openApiDepends(): array
    {
        return [
            openapi\Name::class,
        ];
    }

    public static function openapiExample(): array
    {
        $values = static::find()
            ->orderBy(['key' => SORT_ASC])
            ->all();
        return array_map(
            fn (self $model): array => $model->toJsonArray(),
            $values,
        );
    }
}
