<?php
/**
 * FecShop file.
 *
 * @link http://www.fecshop.com/
 * @copyright Copyright (c) 2016 FecShop Software LLC
 * @license http://www.fecshop.com/license/
 */

namespace fecshop\models\mysqldb;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

/**
 * User model.
 *
 * @property int $id
 * @property string $username
 * @property string $password_hash
 * @property string $password_reset_token
 * @property string $email
 * @property string $auth_key
 * @property int $status
 * @property int $created_at
 * @property int $updated_at
 * @property string $password write-only password
 * @property int $access_token_created_at
 *
 * @author Terry Zhao <2358269014@qq.com>
 * @since 1.0
 */
class Customer extends ActiveRecord implements IdentityInterface
{
    const STATUS_DELETED = 10;
    const STATUS_ACTIVE = 1;
    const STATUS_REGISTER_DISABLE = 2;

    public static function tableName()
    {
        return '{{%customer}}';
    }

    public function rules()
    {
        return [
            ['status', 'default', 'value' => self::STATUS_ACTIVE],
            ['status', 'in', 'range' => [self::STATUS_ACTIVE, self::STATUS_REGISTER_DISABLE, self::STATUS_DELETED]],
        ];
    }

    /**
     * @param $id | Int , 用户id
     * 通过id 找到identity（状态有效）
     */
    public static function findIdentity($id)
    {
        return static::findOne(['id' => $id, 'status' => self::STATUS_ACTIVE]);
    }

    /**
     * @param $token | String
     * 通过access_token 找到identity
     */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        return static::findOne(['access_token' => $token, 'status' => self::STATUS_ACTIVE]);
    }

    /**
     * 生成access_token
     */
    public function generateAccessToken()
    {
        $this->access_token = Yii::$app->security->generateRandomString();
    }

    /**
     * Finds user by username.
     *
     * @param  string      $username
     * @return static|null
     */
    public static function findByEmail($email)
    {
        return static::findOne(['email' => $email, 'status' => self::STATUS_ACTIVE]);
    }
    
    public static function findAvailableByEmail($email)
    {
        return static::find()->where(['email' => $email])->andWhere(['in', 'status', [
            self::STATUS_ACTIVE,
            self::STATUS_REGISTER_DISABLE
        ]])->one();
    }

    /**
     * Finds user by password reset token.
     *
     * @param string $token password reset token
     * @return static|null
     */
    public static function findByPasswordResetToken($token)
    {
        if (!static::isPasswordResetTokenValid($token)) {
            return null;
        }

        return static::findOne([
            'password_reset_token' => $token,
            'status' => self::STATUS_ACTIVE,
        ]);
    }
    
    /**
     * Finds user by password reset token.
     *
     * @param string $token password reset token
     * @return static|null
     */
    public static function findByRegisterEnableToken($token)
    {
        if (!static::isRegisterEnableTokenValid($token)) {
            return null;
        }

        return static::findOne([
            'register_enable_token' => $token,
            'status' => self::STATUS_REGISTER_DISABLE,
        ]);
    }
    
    /**
     * Finds out if password reset token is valid.
     *
     * @param  string $token password reset token
     * @return bool
     */
    public static function isRegisterEnableTokenValid($token)
    {
        if (empty($token)) {
            return false;
        }
        $timestamp = (int) substr($token, strrpos($token, '_') + 1);
        //$expire = Yii::$app->params['user.passwordResetTokenExpire'];
        $expire = Yii::$service->email->customer->getRegisterEnableTokenExpire();

        return $timestamp + $expire >= time();
    }
    

    /**
     * Finds out if password reset token is valid.
     *
     * @param  string $token password reset token
     * @return bool
     */
    public static function isPasswordResetTokenValid($token)
    {
        if (empty($token)) {
            return false;
        }
        $timestamp = (int) substr($token, strrpos($token, '_') + 1);
        //$expire = Yii::$app->params['user.passwordResetTokenExpire'];
        $expire = Yii::$service->email->customer->getPasswordResetTokenExpire();

        return $timestamp + $expire >= time();
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->getPrimaryKey();
    }

    /**
     * {@inheritdoc}
     */
    public function getAuthKey()
    {
        return $this->auth_key;
    }

    /**
     * {@inheritdoc}
     */
    public function validateAuthKey($authKey)
    {
        return $this->getAuthKey() === $authKey;
    }

    /**
     * Validates password.
     *
     * @param  string $password password to validate
     * @return bool   if password provided is valid for current user
     */
    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    /**
     * Generates password hash from password and sets it to the model.
     *
     * @param string $password
     */
    public function setPassword($password)
    {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password, 6);
    }

    /**
     * Generates "remember me" authentication key.
     */
    public function generateAuthKey()
    {
        $this->auth_key = Yii::$app->security->generateRandomString();
    }

    /**
     * Generates new password reset token.
     */
    public function generatePasswordResetToken()
    {
        $this->password_reset_token = Yii::$app->security->generateRandomString() . '_' . time();
    }

    /**
     * Removes password reset token.
     */
    public function removePasswordResetToken()
    {
        $this->password_reset_token = null;
    }
    
    /**
     * Generates new password reset token.
     */
    public function generateRegisterEnableToken()
    {
        $this->register_enable_token = Yii::$app->security->generateRandomString() . '_' . time();
    }

    /**
     * Removes password reset token.
     */
    public function removeRegisterEnableToken()
    {
        $this->register_enable_token = null;
    }
}
