---
type: slides
---

# 機械学習パイプライン

Notes: それでは、`nlp`オブジェクトを強化していきましょう！

この章では、spaCyの機械学習パイプラインを扱う方法を学びます。

---

# 機械学習パイプラインとは？

- _文脈から_ 言語の特徴を抽出するための手法
    - 品詞タグ付け
    - 統語的依存関係解析
    - 固有表現抽出
- ラベル付けされたデータを用いて訓練します
- データを追加することで、予測結果を調整することができます

Notes: 言語の特徴が文脈に依存することはよくあります。
例えば、ある単語が動詞かどうかの判別や、テキストのある区間が人の名前を示すかどうかの判別などです。
（訳注：英語は日本語と異なり、単語の品詞は文脈に強く依存します。）

spaCyは機械学習パイプラインのコンポネントによって、文脈に依存する特徴を抽出することができます。
品詞タグ付けや依存関係解析、固有表現抽出などがそれに当たります。

機械学習パイプラインは、大量のラベル付きデータによって訓練されます。

データを追加することで、予測結果の調整をすることができます。
例えば、特定の分野のデータを用いることで、その分野でのパフォーマンスをあげることができます。

---

# パイプラインパッケージ

<img src="/package_ja.png" alt="A package with the label ja_core_news_sm" width="30%" align="right" />

```bash
$ python -m spacy download ja_core_news_sm
```

```python
import spacy

nlp = spacy.load("ja_core_news_sm")
```

- バイナリ化されたモデルパラメータ
- 語彙データ
- メタデータ（言語、パイプライン）
- 設定ファイル

Notes: spaCyには、`spacy download`コマンドを使ってダウンロードできる学習済みパイプラインがたくさんあります。
例えば、「ja_core_news_sm」パッケージは、spaCyの中心的な機能が全て詰まった、Web文章で訓練された小サイズの日本語用パイプラインです。

`spacy.load`は、パイプラインパッケージをロードし、`nlp`オブジェクトを返す関数です。

パッケージにはパイプラインのパラメータが含まれており、spaCyはこれを用いて予測を行います。

パイプラインパッケージにはこれらの他にも、語彙データやメタデータが含まれています。
メタデータは、spaCyにどの言語クラスを使うかを伝えたり、処理パイプラインの設定方法が記載されています。

---

# 品詞タグの予測

```python
import spacy

# 日本語のパイプライン（小サイズ）をロード
nlp = spacy.load("ja_core_news_sm")

# テキストを処理
doc = nlp("彼女はおにぎりを食べた")

# tokenを順に処理
for token in doc:
    # テキストと、品詞タグの予測結果をプリント
    print(token.text, token.pos_)
```

```out
彼女 PRON
は ADP
おにぎり NOUN
を ADP
食べ VERB
た AUX
```

Notes: それでは、パイプラインの予測結果をみていきましょう。
この例では、spaCyを使って品詞タグ（文脈に依存した単語のタイプ）を予測しています。

まずはじめに、小サイズの日本語パイプラインをロードし、`nlp`変数に格納します。

次に、「彼女はおにぎりを食べた」というテキストを解析します。

最後に`doc`内のそれぞれの`token`に対して、テキストと、品詞タグの予測結果が格納されている`.pos_`属性を出力します。

spaCyでは、文字列が格納されている属性の名前は通常、アンダースコア「_」で終わり、
アンダースコアの無い属性は数字からなるIDを返します。

この例では、モデルは「食べ」を動詞、「おにぎり」を名詞と正しく判断できています。

---

# 依存関係の解析

```python
for token in doc:
    print(token.text, token.pos_, token.dep_, token.head.text)
```

```out
彼女 PRON nsubj 食べ
は ADP case 彼女
おにぎり NOUN obj 食べ
を ADP case おにぎり
食べ VERB ROOT 食べ
た AUX aux 食べ
```

Notes: 品詞タグ付けに加えて、単語間の依存関係も解析できます。
例えば、文章中の単語が主語か目的語かなどを予測できます。

依存関係ラベルの予測結果は、`.dep_`属性で取得することができます。

`.head`属性は、係り先のトークン、つまり構文木における親のトークンを表しています。

---

# 依存関係ラベルの定義

<img src="/dep_example_ja.png" alt="'彼女はおにぎりを食べた'という文の構文木のグラフ" />

| ラベル | 説明 | 例|
| --------- | -------------------- | ------- |
| **nsubj** | 名詞句主語 | 彼女     |
| **obj**  | 目的語 | おにぎり   |
| **aux**  | 助動詞 | た   |

Notes: 依存関係解析の結果は、標準化されたラベルで表現されます。
ここに記載されているのは、よく使われるラベルのうちいくつかの例です。

「彼女」という代名詞は、動詞「食べ」に係る名詞句主語です。

「おにぎり」は動詞「食べ」に係る目的語です。「おにぎり」は、主語である「彼女」によって食べられます。

助動詞「た」も動詞「食べ」に係ります。

---

# 固有表現抽出

<img src="/ner_example_ja.png" alt="'アップルはイギリスのスタートアップを10億円で買いそうだ'という文の固有表現抽出結果" width="80%" />

```python
# テキストを処理
doc = nlp("アップルはイギリスのスタートアップを10億円で買いそうだ")

# 抽出された固有表現をイテレート
for ent in doc.ents:
    # 固有表現のテキストとラベルをプリント
    print(ent.text, ent.label_)
```

```out
アップル ORG
イギリス GPE
10億円 MONEY
```

Notes: 固有表現とは、例えば人、国、組織のような、名前のついた実世界の実体のことです。

固有表現抽出の解析結果は、`doc.ents`プロパティから取得できます。

`doc.ents`プロパティは`Span`オブジェクトのイテレータを返します。
そしてそれぞれの`Span`に対して、`.text`でテキストを、`.label_`で固有表現のラベルを取得できます。

このケースでは、「アップル」を組織、「イギリス」を地理名称、「10億円」を金額、と正しく解析されています。

---

# Tip：explain関数

よく使われるタグやラベルの定義を確認してみましょう。

```python
spacy.explain("GPE")
```

```out
'Countries, cities, states'
```

```python
spacy.explain("NNP")
```

```out
'noun, proper singular'
```

```python
spacy.explain("dobj")
```

```out
'direct object'
```

Notes: よく使われるタグやラベルの定義を確認したいときは、`spacy.explain`ヘルパー関数が便利です。

例えば、「GPE」は地理的な実体を示しますが、ちょっとわかりづらいです。
そこで、`spacy.explain`を使えば、国や町を意味していることをすぐに確認できます。

この関数は品詞タグや依存関係ラベルにも使えます。

---

# Let's practice!

Notes: さて、ここからは手を動かしていきましょう。
spaCyの機械学習パイプラインと、解析結果を実際にみていきます。
