unit OAuth.GMail;

interface

uses
  Quick.OAuth,
  Quick.HttpServer.Request;

type
  TOAuthGMail = class (TOAuthBase)
  private
    fClientID: string;
    fSecretID: string;
  protected
    function CreateAuthorizationRequest: string; override;
    function CreateAuthToAccessRequest (const aAuthToken: string): string; override;
    function CreateRefreshRequest (const aRefreshToken: string): string; override;

    function CreateAuthorizationHTMLPage (const aAuthorised: boolean): string; override;
  public
    constructor Create(const aClientID, aSecretID: string);
  end;

implementation

{ TOAuthGMail }

uses
  Quick.OAuth.Utils,
  SysUtils;

constructor TOAuthGMail.Create(const aClientID, aSecretID: string);
begin
  inherited Create;
  fClientID:=aClientID;
  fSecretID:=aSecretID;
  AuthCodeParam:='code';
  AuthErrorParam:='error';
  AccessTokenParam:='access_token';
  ExpirationParam:='expires_in';
  RefreshTokenParam:='refresh_token';
end;

function TOAuthGMail.CreateAuthorizationHTMLPage(
  const aAuthorised: boolean): string;
const
  DENY_IMAGE = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAABACAYAAACqaXHeA'+
  'AAACXBIWXMAAA7DAAAOwwHHb6hkAAABtElEQVR4nO2bUY7CMAxEhz0NB+EgrDjXCg6yB+E4+9NKCAiN'+
  '7XHcjf1+ED+Z6ahNHScFiqIoBNwvp+v9crpG+3hG6+sgFQFwXv7ejj+/31JBDyy+ugN4ElkJD8Hqqyu'+
  'AhohYjA3D12YAGyIiMSYsXx8D6BTpFmPB9NUMQCjSJcaA7ettAEqRTTErHr5eAjCKfBSz4OXryzhgiz'+
  'OzWCJd/Fs8HoFHzHeCtxf2JNgt3MMID8zXoMpAtDarEDIbidJklMJUQ6MDty6GNDSNRdxtluWwhReDU'+
  'fONKADAx2jkZCsOAOAaXn4pY2let6oAAN/qTIG61lAHAOwmBFO1aQoACA/BXGqbAwDCQqCsOCkBAMND'+
  'oC23aQEAw0Kg9hqoAQDuIdAbLfQAALcQXFptLgEA9BDc+oxeLbF/Qz0C7AFTT4KpX4OpC6HUpXDqxVD'+
  'q5fBOLn5lbEMkdUssdVM0dVs89cZI6q2x1JujqbfHUx+Q2MPFj/LidUqsKajB05NXS4y6Zl/GurHGe6'+
  'QOSpLF5jgqqxSb67C0UGzO4/KdYnN/MLEhluOTmYZY+MWvDPU122dzRVHk5g+X6Lw5aVkK9AAAAABJRU5ErkJggg==';

  GRANT_IMAGE = 'data:image/png;base64,' +
'iVBORw0KGgoAAAANSUhEUgAAAgAAAAIACAYAAAD0eNT6AAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAOxAAADsQBlSsOGwAAABl0' +
'RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAACAASURBVHic7N11dFRXuwbwZzKZuCdEIZBAPLg7xaFIi1tKKVparIJT' +
'pEWLlSIt7u5Q3N2DhgQJGoi7TGYyM/ePfLmlBsnsPXPmzLy/te5ad63mvOf9wmT2c/Y5Z2+AEEIIIYQQQgghhBBCCCGEEEIIIYQQ' +
'QgghhBBCCCGEEEIIIYQQQgghhkKip/N4APgIQB0AwQD8AJQCYAvAQk89EEIIIYZEASAHQBKAZwAeArgC4DSARF2fXJcBwAVALwAR' +
'AGrp8DyEEEKIMdEAuAZgA4AtAFJ1cRJdBABvAN8BGITCK3xCCCGEaCcbwHIAcwG85VlYyrGWOYBhAHYDaAya2ieEEEJYWQCoC+DL' +
'//3/lwCoeBTmNQMQAGAbgKqc6hFCCCHkn24C6A7gKWshM/Ze0BHADdDgTwghhOhadRSGgPashVhvAfQFsBGANWsjhBBCCCkWKwDd' +
'UPhMwC1ti7AEgAEAVjHWIIQQQkjJmQFoB+AVgEhtCmj7DEBHALtAgz8hhBAipAIAnwI4WNIDtQkAFVB4/8FBi2MJIYQQwlcWgBoA' +
'HpXkoJI+BCgDsB00+BNCCCGGwh7AJhS+jl9sJZ3C/w6FK/sRQgghxHB4A0hH4VLCxVKSWwDeKJxeoNX9CCGEEMOTBSAQQHxxfrgk' +
'twC+Aw3+hBBCiKGyB/BNcX+4uDMALgBeggIAIYQQYsiyAfgCSPvQDxZ3BqAXaPAnhBBCDJ0dgB7F+cHizgBcA1BT63beEVrBFf27' +
'hqNFPV+U9XGArbWMR1lCCCFEVHLylHgRl4njl15i1Y77iHqSwqv0FRRuIPRexQkAHihcbpBp4yALmRQ/j2mEIT0qQSrVxS7EhBBC' +
'iDipVBos23IHo+ech0LJvNmfBoVjd9L7fqg4twA+AofB/8DvHfFV78o0+BNCCCF/I5VK8HWfKjjwe0dYyJgX2ZWgcOx+r+IEgDqs' +
'nfw8phGa1fVlLUMIIYQYtWZ1fTFndEMepT44dhcnAASxdBBawRVDelRiKUEIIYSYjC97VkZoeRfWMh8cu4sTAAJYOujfNZym/Qkh' +
'hJBikkol+KJrOGuZD47dxQkATiwdtKhfluVwQgghxOS0qMc8djp/6AeKEwDsWTrw9WI6nBBCCDE5vt7MY+cHCxQnAFiwdGBnQ+/5' +
'E0IIISVhb8s09AKA5Yd+oKTbARNCCCHECFAAIIQQQkwQBQBCCCHEBFEAIIQQQkwQBQBCCCHEBFEAIIQQQkwQBQBCCCHEBJkL3cCH' +
'mIcsZDq+4OFIOj8hhBDyNzQDQAghhJggCgCEEEKICaIAQAghhJggCgCEEEKICaIAQAghhJggCgCEEEKICaIAQAghhJggCgCEEEKI' +
'CaIAQAghhJggCgCEEEKICaIAQAghhJggCgCEEEKICaIAQAghhJggCgCEEEKICaIAQAghhJggSTF+RsNyAtqP3nTk5CnRYcg+nL32' +
'WuhWCCGEfGCMpxkAwgUN/oQQIi4UAAgzGvwJIUR8KAAQJjT4E0KIOFEAIFqjwZ8QQsSLAgDRCg3+hBAibhQASInR4E8IIeJHAYCU' +
'CA3+hBBiHCgAkGKjwZ8QQowHBQBSLDT4E0KIcaEAQD6IBn9CCDE+FADIe9HgTwghxokCAPlPNPgTQojxogBA/hUN/oQQYtwoAJB/' +
'oMGfEEKMHwUA8hc0+BNCiGl4717B/6PReRfE6NWsVRubtmyHnZ2d0K0QQogoeHu4spZ47xhPMwBE52jwJ4QQw0MBgOgUDf6EEGKY' +
'KAAQnaHBnxBCDBcFAKITNPgTQohhowBAuKPBnxBCDB8FAMIVDf6EECIOFAAINzT4E0KIeFAAIFzQ4E8IIeJCAYAwo8GfEELEhwIA' +
'YUKDPyGEiBMFAKI1GvwJIUS8KAAQrdDgTwgh4kYBgJQYDf6EECJ+FABIidDgTwghxoECACk2GvwJIcR4mOv6BG8SUnR9CkIIITrA' +
'YT96k2bo4x/NABBCCCEmiAIAIYQQYoIoABBCCCEmiAIAIYQQYoIoABBCCCEmiAIAIYQQYoIoABBCCCEmiAIAIYQQYoIoABBCCCEm' +
'iAIAIYQQYoIoABBCCCEmiAIAIYQQYoIoABBCCCEmiAIAIYQQYoIoABBCCCEmyFzoBojhSkxMREz0Q8TERCMmOhrPn8UiMysTmRmZ' +
'yMrKQnZ2NgDAzs4O9vb2cHRyhL2dPcr5+SMoOBhBQcEICg6Bu7u7wP9LCCHaMPT97AkbSTF+RsNyAvoAiUdubi4uXjiH48eO4dTJ' +
'E3jzJo5LXTc3NzT5qClatGyNxo2bwMHRkUtdQggxZt4erqwl3jvGUwAwcQpFPv44eABbNm3E1atXoFQqdXo+mUyG2nXqolfvPvi4' +
'XXvIZBY6PR8hhIgVBQCiEwnx8di4YR3WrV2N5ORkQXpwc3ND9569EfFZX/j6lhWkB0IIMVQUAAhX8W/fYs6cmdixbStUKpXQ7QAA' +
'pFIpunXvidFjxsHD01PodgghxCBQACBc5ObmYs2qFfhl4fz/f3jP0FhbW+OLAYMwfMQo2NvbC90OIYQIigIAYXboj4MYO/pbwab6' +
'S6pUqVKY/fN8tG7TVuhWCCFEMLoOALQOgBGTy+WYNHEcBnzRVzSDPwAkJSXhi88jMPzrL5GTkyN0O4QQYpQoABipyMhbaPZRQ6xa' +
'sVzoVrS2c8d2tGzeBLdvRwrdCiGEGB0KAEbo0B8H0aljOzyLjRW6FWbPYmPRsV1b7Nu7W+hWCCHEqFAAMDIrV/yOQQP6IT8/X+hW' +
'uFEqFRg6ZBCWLvlV6FYIIcRoUAAwEhqNBvN+no0fJo6HWq0Wuh3uNBoNfpo2BZMmjjPK/32EEKJvtBeAkZjyw0SsWP6b0G3o3KoV' +
'y6EqUGHGrDlCt0IIIaJGMwBGYOmSX01i8C+yds0q/LZsidBtEEKIqFEAELl9e3djxk/ThG5D736cOhk7tm8Tug1CCBEtCgAiduHC' +
'eQz/+iuTvCeu0Wjw3TcjceHCeaFbIYQQUaKVAEXqyePHaN+uNTLS03V+LncPT4RXqgrfcv4o41sO7h6esLGxhZW1DQBAnpeL3Nwc' +
'JCbE49XL53j5PBb37kQiKTFe5705OjnhwMEjqBAQoPNzEUJKhsNKdqLGOv7RUsDkH1JTU9CuTSs8f/5MZ+fw8PRCs5Yfo1bdBvD2' +
'KaNVjbjXL3HtygWcOnYICfFvOXf4p3J+fvjj8DE4O7vo7ByEkJKjAEABgOVw8je5ubno0a0zbly/ppP6AUGh6NK9D6pUrwWJpDgf' +
'jw/TaDS4ffMadm3fiEfRUVxq/l2t2nWwZdtOWFtb66Q+IaTkKABQAGA5nLxDocjHZ3164dzZM9xre3h6o3ffgahdryG3gf/vNBoN' +
'rl46j03rViAh/g33+k0+aop1GzZBJrPgXpsQUnIUACgAsBxO/kelUmHokIE4sH8f99qNPmqBAV+OgJWVfq6eFYp8bFq3EocP8F/e' +
't3Wbtli+cg3MzWmJC0KERgGAAgDL4QSFA+aXgwfi8KE/uNa1tLLC16PGonbdhlzrFtfVy+exeMEs5MvlXOu2a98BS5b9TjMBhAiM' +
'AoBhBwB6DdDA5eXl4fPP+nAf/O3tHTBx2hzBBn8AqF23IabMmA9HRyeudQ8e2I/ePbrRVsKEEPIeFAAMWFpaKrp37YQzp09xrevk' +
'7IKffl6MoOAwrnW1Ub5CEKbNXgQnzk/wX7hwHn16ddfLa5KEECJGdAvAQD1/9gwRfXrg6ZMnXOtaW9tgyoz58CtvWO/Nv3zxDJPH' +
'jkROTjbXuuX8/LB+wxZaJ4DolEajQUJ8PF69eomXL1/g1atXSExIQGpqClKSk5GalobsrCxkZmUCAJQKBXJzcwEA9vb2MJNKYWVp' +
'CSsra9jY2sLd3R1ubqXg6uoKNzc3lC5TBuX8/OFXzg+OTnxnzHSJbgEY9i0ACgAG6ML5cxg4oB/3q1epVIqxk6ajcrWaXOvycu/O' +
'LcyYMg4qVQHXuk5Ozli+ag0aNBDudgcxHjk5Obhz5zaiHtzHw6goREXdR0x0NOScn2X5L87OLvD390doWDgqVa6MypWrICg4BDKZ' +
'TC/nLwkKABQAWA43KRqNBqtWLsePUydDqVRyr99/yHC0atuRe12eTp84gmWLfuZeVyqVYuSobzHq2+9hZkZ3vkjxJSUl4dLF87h+' +
'/RquX7uKqAcPoFKphG7rLywsLBEeHo469eqjQYOGqF2nrkGsiUEBgAIAy+EmIzU1BcO/HopTJ0/opH77T7oi4oshOqnN24Y1v+PA' +
'nu06qd2gYSMsXvo73N3ddVKfiJ9KpcKtmzdw6uQJnD59Evfu3oVGw/Q1qHcymQWq16iBxk0+Qus2bREUFCxIHxQAKACwHG4Sjh45' +
'jDHff4PExESd1K9Rqx6+nzBNZwv88KbRaDB35mRcv3JRJ/U9PDwwZ+58tGjZWif1ifio1WrcuH4NBw7sw4F9e3X2tygUX9+yaNGq' +
'Fdq374iatWrr7buAAgAFAJbDjVpmRgZ++nEqNm5Yp7NzBAaHYuK0OXpb5IcXhSIfM6aMQ9T9Ozo7R/sOHTFrzlzaQ8CEPX78CJs3' +
'bsDePbuQkJAgdDt64efvjx49eqNrt+7w9PLS6bkoAFAAYDncKKnVauzYthUzpk9DUlKSzs7jXyEQP/w0FzY2tjo7hy7l5eZi2sTv' +
'8PRJjM7O4e7ujvETJ6Nrt+6imSEhbPLz83Fg/z5s2rgeV69cFrodwUilUjT5qCl69OyN1m3aQiqVcj8HBQAKACyHG53IyFuYOG4M' +
'IiNv6fQ8ZXzLYcqM+bB3cNTpeXQtKysTU8aNwquXz3V6nurVa+CnGbNRuUoVnZ6HCCclJRlbN2/CypXLkRCv+62qxaRMGV981vdz' +
'RHz2ORwc+X1nUACgAMByuNF49CgGc+fMwh8HD+j8gSKfMr744ce5cHYxjj++tLQUTJv4HeJevdTpeSQSCdp36Ihvvx+DgIBAnZ6L' +
'6M/zZ8+wdMmv2LF9K/Lz84Vux6A5ODigT0RfDBw0BB6enkK3Y/IoAIjco0cxWLxoIfbs3qWXV4f8ygdgwpRZcOC8vK7QMjPSMX3K' +
'WDx7+ljn55JKpejUuQu+Hj6SgoCIxcW9xi8L5mPrlk0oKOC7toSxk8ks0L1HT3w3eiy9MSMgCgAide3qFSz5dRFOnDimt1eIKgQG' +
'Y/zkWbCzt9fL+fQtNzcHs6aNR3TUfb2cTyKRoEHDRhgwcBCat2hFzwiIRFJSEhbM+xmbNm6AUqkQuh1Rs7W1Rf8BgzBk6FdwcnIW' +
'uh2TQwFARBITE7FzxzZs27IZjx8/0uu5q9aojW9G/wBLKyu9nlff8vPzMX/2VETeuKrX8wYGBqF7z17o0rU7SpUqpddzk+JRKpVY' +
't3Y15s6ZhczMTKHbMSqOTk749rvR+Lxff9pqW48oABi49PQ0HDt6FH8c2I/Tp08KMtXYpn0n9O3/pcmscKdWq7Fu1TIcPrBb7+eW' +
'yWRo8lEztGvfAS1atqSrIgNx7OgRTJk8Ec+fPRO6FaMWEhqG6TNmoU7dekK3YhIoABgYjUaDh1EPcOHCeZw4fgxXLl8S7P6iVCrF' +
'F4OGoUWb9oKcX2jHDx/A6uW/CrYsq7m5OerWq4/mLVqiQYOGCA4JpdsEepaQkICJ48fgj4MHhG7FZEgkEnT8pBOm/TQDbm5uQrdj' +
'1CgACCwjPR13797Bvbt3cPt2JC5fuoSUlGSh24KDoxNGfDcBFStXE7oVQd27cwu/zJ2OzAzht/11dXVD3Xr1UKVKVVSqXAUVK1YS' +
'1c5tYqLRaLBzx3ZM+WEi0tJShW7HJDk6OWHCxB/QJ6Kv0K0YLQoAOqDRaJCZkQGgcGev9PQ0JPxv686E+Hg8e/YMz5/FIvZZLN6+' +
'eSNwt/8UElYJI76bABdXSt8AkJqSjF/mTsfDB3eFbuUfvL194OfnV7iVq58fPDw94erqBnd3dzg5OcPWtnCRJgdHR5o9KKbExESM' +
'HP4Vzpw+JXQrBEDbj9th5uy59GyMDph8ADD1hSTeJZFI0Lrdp/jsi8GQSulBnHep1Wrs2rYBO7duEN3GLbpkiAGcxbGjR/DtqBEG' +
'MQv3b+ytZSjjagt3J2t4OBb+n4ONBRysZbC3lsHexgJmEsBcagYrWeHKe7n5BVBrNFAUqJGXX4CMPCXSc/KRlq1Aek4+4lJz8SY1' +
'B2/SciFXGNYuhEVcXFyxcNFiNG/RUuhWjAoFAAoAAIBS7p74cvh3CK9UVehWDNr9u5FYtmgukhJppTfAeAKAQpGPHyZOwPp1a4Ru' +
'5f852lggyMcRwT5O8PdwQDl3O7jYWer0nMlZcjyNz8TjN5l49DYDj99mIEduGGscSCQSfDn0a4wZNwEymUzodowCBQATDwASiQTN' +
'Wn6MiC8Gw9raRuh2RCE/Px87t67H/t3bTH42wBgCwJu4OAzs/7nOl8/+ECuZFJXKuaB6eTdU8nWBj6vwe2xoNBo8jc/C7ecpuP0s' +
'BdFx6VAUqAXtqXqNmvht+Ur4+JQWtA9jQAHAhAOAh6c3Bn/9DV31a+nenVtYvmQBEuIN7zkOfRF7ALh48QKGDOwv2JS/vbUMdQLd' +
'0SDEE+G+zpBJDftV23ylCjeeJuPCw3jceJqMfKUwtwycnV2wcvVa1K1XX5DzGwsKACYYACwtLdGhU3d80rknZBYWQrcjagUFBTh2' +
'eD+2bVyDvLxcodvROzEHgPXr1mDi+LF6f81WaiZBnUB3NKvkg6p+rpCaifPhzHylCteeJOHEnTjcfp6q99kwmcwCM2fPQa/eEXo9' +
'rzGhAGBiAaBewyaI6DcErm70RC1PKclJWL/6N1y+cEboVvRKjAFArVZjxk/TsHTJr3o9r7ujNdpUK41mFX3gZGtcwTshPQ/H7rzG' +
'ibtvkJat3w2RBg8Ziok/TNHJdsPGjgKAiQSAoJBw9PqsP0LCKgndilF78igau7ZtwM3rV4RuRS/EFgDy8/Mx7KshOHhgv97OGeDl' +
'iE9rl0XdIA/RXu0Xl1Klxpn7b7H76nPEpeTo7bytWrfBb8tXwdJStw9JGhsKAEYeAAKDQ9Gjzxd0n1/PYqIfYNvGNbh/N1LoVnRK' +
'TAEgJycHn0f0wsWLF/RyvuDSTujdsAIql3PRy/kMiUajwZVHidh2MRaxCVl6OWf9+g2wZv0m2NnZ6eV8xoACgBEGAIlEgmo1aqNt' +
'h84mv5Kf0GKfPMKhA7tx4exJqNXCPj2tC2IJAJkZGejdqztu3riu83P5e9gjonEAqpenhbQ0AC5ExWPjuSd4m6b7Z2SqVq2GTVu3' +
'0x4axUQBwIgCgJWVNRo3a4mPO3SGp5ePXs+tVCrp3dz3ePsmDof278LZU8cgl+cJ3Q43YggAqakp6Nm9C+7d1e1Kjk62FohoHIDm' +
'lbxp1cW/Uak1OHr7NTade4KsPKVOzxUcHIJtO/fQyoHFQAHACAKAf4VANG/1Meo3airIu/z370Zi6S9zMGr0JAQEher9/GKiUOTj' +
'5vUrOHHkIO7fjRT9OgKGHgAyMzLQtcsnOh38zSQSdKjpix4NysPGklbQfJ/MPCXWnnqEk3fj2L74PyAkNAy79uyjmYAPoAAg0gBQ' +
'xrccatVtiEYftYCXt36v9otoNBrs370Nm9evhEajgZOzC2bOW0pvGBTT27jXOHfmBK5dPo9XL58L3Y5WDDkA5OTkoEe3zjqd9i/n' +
'bo9hbUMR4OWos3MYo4ev07H0SBReJGXr7BxVqlTFtp17YG9vr7NziB0FAJEEADMzM5QPCEatOvVRq04DeAm8ClZWViYWzZuBO7f+' +
'+uXq518B02b9AksrK4E6E6e3ca9x7coFXLtyEU8fR4vmeQFDDQByuRwRvbrr7IE/qZkE3er5o1t9f6N/sl9XlCo1Np59gr3XXuhs' +
'JqxO3XrYtGU7rK2tdVJf7CgAGGgAkEgkKO1bDuGVqqJi5aoICasEW1vDeLo1JvoBFs75ESnJSf/63+s2aIKR30+k+6BaysnJRtT9' +
'O7h/9zbu343E65fPDfZWgSEGALVajSGD+uvsVT8PJ2t826Eign1oK2Ye7r9Mw8KD95GYoZtnY5q3aIk16zbSOgH/ggKAAQQAc3Nz' +
'eHr5oJx/BfiVD4B/+UD4la8AGxvh1wJ/l1KhwPYt63Bgz/YPXqF26/U5uvSgFbp4yM3NwbOnjxH79DGePX2MZ7GPkfD2jd5XsPs3' +
'hhgAfpo2RWeL/NQP9sCwtmF0r5+z3PwCLDx4H1ceJeqkfv+Bg/DjTzN1UlvMKADoIABIJBLY/O9q3draGjKZDA4OTrB3cISDoyOc' +
'nF3g6loKnt4+8PTyhqubO8zMDHsN8BfPY7Fk4Ww8j31SrJ+XSCQY8d1E1GvYRLeNmSi1Wo3kpETEv41Dwts3SElJQnpaKjIzMpCV' +
'mYHMzHQoFArI5XIAQG5Otk5mEQwtAGzetAHffTOSe10ziQQRTSqgUx2/Yn2pkZLTADh4/SVWn4qBSs3/szp9xmz06z+Ae10xowDA' +
'+AvYvv8k0/GGTqVS4eDeHdi2aU2JrzgtLCwxddYClK8QpKPuCKtuHZoxHW9IAeDixQvo2a0z95kRe2sZxnxaGZXKmt6CPkK48zwF' +
'c/be5f66oLm5Odau34SmzZpzrStmug4Ahn1ZS97r3p1bGD1iEDatW6HVl6pCkY95M6cgIz1NB90R8qc3cXEYMrA/98Hfw8kasyNq' +
'0eCvR5XLuWJe39rwduH7SnNBQQH6RvRC966dsGP7NmRn6+4NBFKIAoAIpSQnYfGCWfhx0vfMr6clJyVi7ozJUCp1u/gHMV0KRT4G' +
'9v+c+5a+gd6OmNu3Nkq7GtazOKbA09kGs/rU4v56pUqlwvlzZzFi2FBUqRiC4V9/ibt37nA9B/kTBQARUSjysW/XVowa2g/nTh/n' +
'Vjcm+gGWL5nPrR4h7/ph4gRERt7iWrOirwt+6lkDjjbGtWufmDjZWmB67xqo5q+bB7Vzc3Oxc8d2tG7ZFF0+7YDjx48a7Ns2YkUB' +
'QARUqgKcOHoQI7/8HJvWrdDJUrVnTx3DgT3budclpu3UyRPYsH4t15rV/N0wuXs1WFnQa2NCs5JJMbFLVdQK0O3iYpcuXUTfPr3w' +
'UaP6OH78qE7PZUooABgwjUaDyxfPYtTQL7B8yQIkJ+nmFZwiG9cuN5ltconuJSQkYMSwr7hetVXzd8OELlVgYU5fXYbCXGqGMZ9W' +
'Rs0Kul9h9NGjGPTt0ws9u3dB1IP7Oj+fsaO/IgOkUqlw9tQxDB/8GRbMnob4t3F6Oa9UKkVaKt/7tMQ0aTQajBrxNdf7/hV9XTC+' +
'cxXIpPS1ZWhkUjOM61RZbzssnj1zGq1aNMVP06b8/6u0pOToL8mA5Obm4NCB3Rg+OAJLFs5GQvwbvZ3bxdUNk2fMR/NW7fR2TmK8' +
'tm7ZhDOnT3GrF+jtiEldq9KVvwErmgkI9NbPvgsqlQpLl/yKFs0a4/q1q3o5p7GhvyYD8Dz2CZYvWYDBfbti7YolSEpM0Ov5Q8Iq' +
'Ydb8ZQgKDtPreYlxSkhIwLSpk7nV83SyxqSuVemevwhYyaSY1LUqfDi/Ivg+T588wacd2+Hn2TOhUqn0dl5jQOtlCiQnJxtXL53H' +
'mZNHEB0lzL0siUSC9p92Q8+I/rQON+Fm4vgxyEhP51LL3lqGH7pVo6f9RcTRxgJTulfHd+uuIiNXoZdzqtVqLJg/F7du3cSSZb/D' +
'xUX4JeTFgAKAHikVCty6cQXnz55E5I2rgr577+TsgiHDvkW1GnUE64EYn2NHj+CPgwe41JKaSTDm08r0nr8IeThZY/SnlfDDlps6' +
'WTb4v5w9cxqtmjfF2vUbERZeUW/nFSsKADqWm5uDu5E3cevGFVy7fAG5uTlCt4TadRti0FejYO9Ae6QTfpRKBaZOmcStXt8mAbTC' +
'n4hV9HVBv6aBWHkiRq/njYt7jU86fIxVa9ajUeMmej232FAA0IGE+Le4ef0ybl67jIcP7hrErnAAYG1jg4h+g+lBP6ITK37/Dc9i' +
'Y7nUqh/sgY61y3GpRYTToWZZPInPxJn7b/V63pycHPSN6IWlv61Am7Yf6/XcYkIBgJFcnoenjx/hcUwUHsc8xKOYKINcWz+8UlV8' +
'Ofx7lHL3ELoVYoSSkpLwy0I+q0l6OlljWNsw2tXPSAxtHYrY+Cy8TNbv2v75+fkYNKAfFi/9DR0/6aTXc4sFBYASyMvNxcsXsXj+' +
'7Cmexz7Fk8fRePXiGdRqtdCt/SdHJ2d89sUQNGjcDBIJfaUS3Vgw72dkZWUx15GaSfBN+4qwsaSvJmNhJZNizKeV8O3aq5Ar9fuU' +
'vkqlwvCvh8LW1g7NW7TU67nFgP7KPmDHlvV4+TwWz589QWJCvGjWopZIJGjZpgN6RHwBW1s7odshRuzVq5fYtHEDl1rd6vkjuLQT' +
'l1rEcJRxs8PSwfVx93kq7r1MxZ1nqUjO0s8CPkqlEoMG9MOmLdtRt159vZxTLCgAfMCOLeuEbqHE/PwrYMCXIxAQFCp0K8QELJg/' +
'F0ol++te5dzt0a2+P4eOiCFys7dC04reaFrRGwAQm5CFG0+TcPZBPF7p+PaAXC5Hv759cOjICfiXL6/Tc4kJBQAj4upWCp2790HT' +
'Fm1hZkZrPBHdexYbi53btzHXMZNIMKxtKKRmdJvKVPh72MPfwx7d6vkj+nU6jtx+jXNR8ShQ6eaWamZmJr74PAIHDx+DnR3NigIU' +
'AIyCnb09OnbqgbbtO0FmQQumEP1ZtnQxl7dcOtT05b63PBGP4NJOCC7thF4Ny2PXlec4fidOJ0Hg0aMYjBg2FCtXr6NnokBLAYua' +
'paUlOnbugV+Xb0THzj1o8Cd6lZycjB3btzLXcbK1QI8GNC1LAHdHa3zZKgSLB9RDNX/dbCx0+NAfWLH8N53UFhuaARAha2sbNGne' +
'Gp907gFnWvKSCGTdmlXIz89nrhPROICe+id/4e1igyndq+FidAKWHYlCZh7fVVNnzfgJTZs2R4WA2LD0hQAAIABJREFUAK51xYb+' +
'6kTE0ckZLdu0R9sOnenJfiIouVyOtWtWM9fx97BH80reHDoixqh+sAdCSjthwYH7uPM8hVtduVyOEcOGYt/BwzA3N91hkG4BiECZ' +
'sn4YMuw7LFu9FV179qXBnwju4IH9SElJZq4T0TiA7sWS93Kxs8S0HtXwKeeVISMjb2H578u41hQb040+Bk4mk6FGrXpo3rodwitV' +
'pS9JYlA2bVzPXCO4tBOql9fNfV5iXCQSCfo1DYSPqy2WHYnitsHQLwvmoWu3HihVqhSXemJDAcDAePuUwUfNW6Npiza0WQ8xSI8f' +
'P8LVK5eZ6/RuWIFDN8SUtKzsAxsLKebtv8clBGRlZWH2zOmYO38hh+7EhwKAAXB1K4VadRuibv3GCAoJo6t9YtA2c1j1L9DbEZXL' +
'0U5/pOQahHgCAObuuwc1h5VZt23djC/6D0BoWDhzLbGhACAQGvSNh1yeh8sXzuKj5q2FbkXn1Go19u7dzVznk1plOXRDTFWDEE+k' +
'ZSuw4kQ0cy2VSoUF8+dixaq17I2JDAUAPZFKpSjrVx7Va9ZB9Zp14VeeHn4yBvlyOWb/OAEP7t3Bm7hX6N13oNAt6dTVK5eREB/P' +
'VMPd0Rp1g2hXSsKmfU1fxKXm4NCtV8y1Dh/6A08ePza51wIpAOhYq487okrVmgirVAVWVtZCt0M4ypfLMevH8Xhw7w4AYN+uwkVx' +
'jDkE7Nu7h7lGm2qlaclfwsXAFsF4Ep+JR28ymOqo1WosXfor5i9YxKkzcaDXAHWs/+DhqF6rLg3+Rubvg3+Rfbu2YtO6FQJ1pVsq' +
'lQqH/jjAVENqJkGzij6cOiKmrmj7aCsLKXOtXTt2IDExkUNX4kEBgJAS+q/Bv4ixhoBbN28gOZnt3f86ge5wsqUlqwk/3i426NuE' +
'fepeqVRg187tHDoSDwoAhJTAhwb/IsYYAk6dPMFco1kluvon/LWpVgb+HvbMdbZv3cKhG/GgAEBIMRV38C9ibCHg9OmTTMfbW8tQ' +
'1Y/2riD8mUkkGNwyhLlOTEw0IiNvcehIHCgAEFIMJR38ixhLCEhKSsK9u3eZatQJdKeH/4jOhJR2QjV/9oC5n8ODrmJBAYCQD9B2' +
'8C9iDCHg0sXz0DAuulK0gAshutKjPvu20idPHufQiThQACDkPVgH/yJiDwHXr19jOt7KQopwX2dO3RDy74JLOyHYx4mpxpPHj/Hy' +
'5QtOHRk2gw8AUinb6x1qtZrx/GxLJahUBUzHE+HwGvyLlDQEqFQqpvPx3Ob0+rWrTMdXKusCmdTgv26IEWhZhf1B05Mn2B94FQOD' +
'/4tkH4DZvkRlMhnT8Uqlkul4Igy5PA/Tp4zhNvgX2bdrK06fOFysn1UzfnZZw3OR7OxsPIyKYqpR3Z92/SP60TDEEzaWbOMGj82u' +
'xMDgA4C5OeMMAAUAUkJFy/tGR93nXjsoJBx16jcu1s8WMM4esX52i9y9ewcFBWy9VCpLG/8Q/bCUSVGNMXDevXObUzeGTQQBgHEG' +
'QM04jcr4JVpAAUBUeE/7vysoJBzjp8yEtbVNsX6eNbyam/MJAA/u32M63sFaBm9XWy69EFIctSqUYjr+xYvnSEtL5dSN4RJBAGD7' +
'ElMxXrlYWLCtWkYzAOJhSIM/ABQwBwA+twAePmSb/g8p7Qx6+Y/oU7XybkyfOY1Gw/zaqxgYfACQWbAFgPz8fKbjWWcAlAoF0/FE' +
'Pwxt8AcARb6c6bwWFpZMxxeJZrz/H1zakUsfhBSXg7UM3i4l+3v7u6dPn3LqxnAZfACwtWWbOpTL85iOt5CxzQDkM36JE90zxMEf' +
'APLy2D67rH87QOFbNDExbHuu+7mzL9FKSEkFeLMFz7jX7NsMGzqDDwB2dnZMx+fl5TIdb2FpxXR8ZibbNpVEtwx18AfYP7usfzsA' +
'kJiQwBxEylEAIAJgDZ6vKQAIz86W7UtMzvjl5ejEtqhERnoa0/FEdwx58AcAeS5bALDlEABevXrJdLyDtQwudnxuRZCSkStUkCvY' +
'niMRMw8nti3YX7+iACA4O3u2FJebm8N0vKMjWwDIzEhnOp7ohqEP/gD7LQAeMwCsK6KVpqf/BSFXqDBl201M2XbTZEOAuyNbAEgz' +
'gYs3ww8AdmwBIDszk+l4B0e25UszKAAYHDEM/gCQlcX22eURAF4xXgW5M16FkZIrGvyjXqcj6nW6yYYAJ1u257dyGWfgxMDgA4CT' +
'M+MUPOMA7ODI9iAJ3QIwLGIZ/AH2z46zM/viO/Fv3zId78F4FUZKRq5U4ccdkYh6/ef3XtTrdPyw7SbyFKa1LLmVjO01WNYZODEw' +
'+ADg6sq2ohPrFLwj4wwA3QIwHGIa/AH2z46rG/vyu+mMIYR1GpYUn1yhwpStN3Hv5T8XsIl+nY6p226Z1EyAtQXbInJ5NAMgPFdX' +
'tv2dWZ/Cd3RiCwDJyUlMxxM+xDb4A+yzV24cAkBKcjLT8azTsKR4/u3K/+9MbSZAasa2/JQpLOJm8AHAzY1tSUfWaVTWtwAS3r5h' +
'3kedsBHj4A9wmAFgnD0DgLR0th7srfgsR0z+W25+ASb/x5X/35niTAD5bwYfAFhnAFJT2a5g3Eq5QyLRPkkqFPlITWHrgWhPrIM/' +
'AKSmsM0e8ZgBSE9jC9D2NjQDoEuZeUpM3HwDD99z5f93pvxgIPkrgw8A7u4eTMcnJyYwHW9lZQ0nJ7aHqeLfxjEdT7Qj5sFfo9Ew' +
'3z5yK8U2ewYUBlgWdlZs92HJf0vLzsf4jdfxJL7kb4tQCCCACAKAl7c30xV4fn4+shifA/D09mE6/u2b10zHk5IT8+APFE7/s+wj' +
'YWZmBi8vL+Y+FIx7WcikBv8VI0ovk7Px/fpreJmcrXUNU3smgPyTwf91WllZMd8GSE5KZDreizkA0AyAPol98AeAJMaZq1Lu7lw2' +
'A2INAOYUALi78zwVYzZcQ2IG+2tq0a/TMXnbLQoBJkoUf52lS5dhOp71y9TLuzTT8W/jaAZAX4xh8AeAlGS20Frah+0zW4T1SWhz' +
'KW0EzNPhW68wZdtN5Mj5Ddj0YKDpEkUA8CnN9mXGeg+e9RbA82dPmI4nxWMsgz8AvGG8beTDGJqLqFRsg4IZw+078idFgRq/HnqA' +
'ZUcfQqXm/1YR3Q4wTaIIAKwzAKzbOrLOACQnJSI97cOv6BDtGdPgD7DPGpVmDM3EcMSl5OD7dVdx/I5ubyVGv07HlG23uM4uEMMm' +
'igBQtlw5puPfvmELAJ5e3pBK2ZaVfPb0MdPx5L8Z2+APsIdW1r8ZYhhO3XuDUWuv4Flill7O9/B1OkZvuIbkLLlezkeEJYoAUKFC' +
'ANPxrF+mFhaWKF2mLFONp08eMR1P/p0xDv4Ae2hl/ZshwkrLzsdPOyOx8OB9vd+bf5WcjbEbriMuhW0nVWL4TCIAZGVmMO+s5l8h' +
'iOn4WAoA3Bnr4J+RkY7sLLYrPv/yFTh1Q/RJA+Do7dcYuuIirj0WbhnxxIw8jNlwDdFxtJeJMRNFAPDw9GTe2vTl81im4/0ZQ8iT' +
'x9FMx5O/MtbBHwBePHvKdLyDgwM8PNgW0CL69zwxC+M3XseSw1EGcR8+M0+J8ZtuYN+1F6DFzI2TKJbpkkgk8PP3x727d7Wu8Tz2' +
'CcIqVtH6+PJazACYm5vDv0IggkLCERwaDo1Gw7SoESlkzIM/UPhZZVGerv5FJStPia0XnuKPm6+gNrB9QwpUaqw6GYN7L1Mx4uNw' +
'2FvT3g7GRBQBAAACA4PZAgDjVVVZv/IwNzdHQcF/J3MrK2sEBIUgODQcwaEVERQSxmUxFvInYx/8AeDFc7bPakBgIKdOiC7l5hfg' +
'wI2X2HP1OXLzhb/if59rj5MwYvVljGoXjopl2ZZGJ4ZDNAEgNCwMu3ZqfzxrAJDJZChT1u8vT/M7O7siODQcQaHhCA4Jh1/5ALrC' +
'1yFTGPwB4Hks22c1LLwip06ILuQpCgf+fddeICtPPFvOJmfKMXHzDTQK80K/poFwsaOLG7ETTQAIZ/xSe/3yBZQKBWQW2u9OVqde' +
'IwQGhyI4pPAK35Vxq2JSfKYy+CsU+XgTx/YGQGhYOKduCE9pOfk4cus1Dt58KaqB/10aAGcfvMXVR4n4tHY5dK3nR8s9i5h4AkBF' +
'tgCgUhUgNvYxgoLDtK7xaddeTD0Q7ZjK4A8ATx/HMK++F04BwGBoANx7kYqjt1/jUnSCTlbxE4JcqcKWC09x7mE8utTxQ5NwL0jN' +
'aPZTbEQT3ZydXeDl7c1U43H0Q07dEH0xpcEfAB7FsH1GS5cuA0cnJ07dEG3Fp+dh64WnGLTsPCZuvoHzUfF6HfydHe2xcf4E1K2m' +
'/QVPccSl5OCXP+5j0LLzOHDjJfKVtJ+AmIhmBgAAwsMq4u2bN1of/yjmAYAu/BoiOmVqgz8API6JYjo+NEy3X/jkv8Wl5uJyTAIu' +
'RifgaTzbuiMswgLLYeO8CfAr44WP6lRFl68n4/KtBzo9Z1KmHCuOR2P7xVg0CfdC4zAvVPB00Ok5CTtRBYAq1arh+PGjWh8f81C3' +
'fwSEH1Mc/AHgUTRbAKhevSanTsiHZOYp8eBlKiKfpSDyWQoS0tm352XVpU1jLPphGGysrQAAtjZW2L10Gj77biaOX7ih8/Nn5Cqw' +
'79oL7Lv2AmVcbdEozAsNQzzh7WJ4f2tEZAGgRg22L7e01BQkJSaglDstkmLITHXwT4h/y7xpVLUaNTh1Q96VIy/Ai+RsvEjMQsyb' +
'DMTEpSMuNVfotv6ftZUl5owZjIhPW/7rf9u0YCKGTJyH3UfP662nVyk52HTuCTadewI3BytU9HVBRV9nhJd1gaeTtd76IP9NVAGg' +
'arXqkEqlTA9JPbh3G02ateLYFeHJVAd/ALh/N5LpeKlUiipVqnLqxvjJlSoUqNRQawoX48nKUyA7T4nMPCUSMvKQmJGHxAw53qbm' +
'GvTmOGGB5bB69hgE+f33rqkWMnOsmPk9HO3tsGbnYT12Vyg5U47T99/g9P3CW7iONhbwdrGBj4stvF1s4O1iA1c7K1hZSGEpk8LO' +
'yhxWMim9YaBjogoAdnZ2CAwMwsOH2k+T3r8bSQHAQJny4A8UhlMWIaFhsLW15dSNfigK1IiNz8Sjtxl49CYDCRl5SM6UI1+pRrZc' +
'nK/K6Yu5VIoR/Tpj9KCesLT48Ap9UjMzLJj4FYL8ymDi/FUoYHzbhEVGrgIZuQo8fP3hvQbsrGSwlJmhlIM13B2tEOTthABvB/h7' +
'OMDCnAICC1EFAACoXqMmUwC4d+cWx24IL6Y++ANA1H22/+2st8j0JSNXgfNR8Tj3MB6P32QYzatx+hRc3hfLfhyFqqEl36NkSO8O' +
'CK7gi36jZyMtQz/bDLPIliuRLQdSsvIRHQeci4oHAEjNJAj0dkTDEE80DPWEo432a7yYKtEFgNp16mDjhnVaH5+WmoK41y/hU9qX' +
'Y1eEBQ3+QNzrl0hNSWaqUat2bU7d8KcBcCUmAcfuxOH2sxQa9LVkbWWJbwd0w/C+nWEh0/7ru0ntKji1cT4ivp2B+4+ecexQf1Rq' +
'DR6+TsfD1+lYdTIGVf1c0aJKadQJdAetSFA8ogsA9es3ZK5x9/ZNCgAGggb/Qncj2Z7QlkgkqFuvAadu+LrzPAVrTz8W9NU4Y9C6' +
'US3MHjMYZX34PMTsV8YLpzcvwLyV2zFn+RaoRRzKVGoNbjxNxo2nyShbyg49GpRHvWB62PtDRBcAPL28UL5CBTx9ov2OabduXEWb' +
'dp9y7Ipogwb/P926cZXp+ICAQIPcAvibNVfwhAZ+JmGB5TB1RD80r1+de22ZuTnGDumFulXD8OUPC/AmgW0WyhC8SMrG7D13aB2C' +
'YhDlExQNGjRiOj7q3m3I5cK/s2vKaPD/U75cznz/v35D9pkxXaDBX3s+nm5YOm0kzm/9VSeD/7sa166MSzsWo1+XNjAzkiV96bP3' +
'YaIMAPUbsH3ZKZVK5leuiPZo8P+ru7dvQqlke+Kdx60xYhh8PN0w6/tBuLlvOXp1aK63AdnJwQ4LJn6F05sWokbFIL2ckwhLpAGg' +
'AaRSKVONW9fZplyJdmjw/yfW6X+pVIp69etz6oYIxa+MFxZM/AqRB1ZiSO8OsLIU5qn2yiHlcXTdz5g3fig83JwF6YHohygDgLOz' +
'C6pWY5sSu371ItRqNaeOSHHQ4P9ParUaN65eZKpRvUZNODnRF7VY1akSirU/j8WNfb+jX5c2TE/38yI1M0P/bm1x++BKzPp+ENxd' +
'aYMpYyTKAAAAzVv8c8nLkshIT0N01D1O3ZAPocH/3z18cBcZGR9eDOV9mjdvwakboi+O9rYY2L0dLmz/FUfWzsEnLRpAamZ4X8fW' +
'VpYY0rsDbu5fjvFD+1AQMDKG94krJh5felcunuPQCX+nTxwxqocU5fI8TJ8yRieDf3BoOCZMnSXKwR8Arlxi/ww2b0krW4qB1MwM' +
'TWpXwe/Tv0XMiQ34edwQhAf6Cd1Wsdjb2mD0oB64d3gNlv04CpWCywvdEuGgOE+XML0c+iYhheXw96pZrTLi4l5rfbyTswt+X7sd' +
'EonhPPW6Y8s67NiyHkHBYRgv4oGtCF35/zeNRoMh/bojLVX7vxEfn9K4fov/77aIt4erzmqbAnOpFPWqh+OTFg3QoXk9uDk7Ct0S' +
'N1cio7D5wEnsO34BGVk5QrejE7ocv4qDw9/fewc34W82MWjarDk2rF+r9fHpaamIjrqPkLCK/JpiUDT4A0BM9APMmDxW1CGABv/3' +
'i466xzT4A0ALuvo3OKVcnNCsfjW0bFATTetWhZODndAt6USdqqGoUzUUP48dgmPnr2P7oTM4eekWcvMMd+Mk8leiDgAft2vPFAAA' +
'4PyZEwYRAN4d/IuIOQTQ4P9h506fYK7Rtl17Dp0QFl7urqhXLQx1q4WhQfWKCPIvY1CzirpmaSFD+2b10L5ZPeQrlLgc+QAnLt7E' +
'iQs3ER37Uuj2yHuI+hZAQUEBqlYKQwrDGuo2NrZYvn4HLCwsOXZWMv82+L9LbLcDaPD/MKVCgUF9uyInJ1vrGi4urrh9Lwrm5rrL' +
'8XQL4E+WFjL4+3ojoFxphAWUQ5XQCqgSUoFelXsPpyrthG6BCd0CMGDm5uZo2ao1tmzeqHWN3Nwc3Lx+BXXrN+bYWfF9aPAHxDUT' +
'QIN/8Vy/eolp8AeAth+30+ngb+ysLC3+/117e1sbONrbwsnBDk4Odijl4gQvd1eU8SoFL3dXlPPxRBlvd4N8Up8QbYn+26Nd+w5M' +
'AQAAzp0+LkgAKM7gX0QMIYAG/+I7d/o4c42PjWj6387WGq0b1UKjmpUQHuQPX293ONrbolSNT5jqpt8+yKlDIoSkG3uRkZWDF3EJ' +
'uP/oGc5du4Oj568jO8d43pISkugDQIOGjeDk5Iz09DSta9y+eR1paSlwdtbfdGdJBv8ihhwCaPAvvrTUFNyJvM5Uw8nJGfXqG+bu' +
'fyXh6eaC0YN7omf7prC2Eu42HDFMMnNzuDk7ws3ZEdXDA9G3UyvkyfOx49AZLFyzE7Gv3grdoqiJfj5LJpOhfYeOTDVUqgKcOnaY' +
'U0cfps3gX6QoBOTl5XLuSns0+JfMqeOHoVKpmGp0/ORTyGQyTh3pn5WlBcYP7YNbB5bji65taPAnxWZtZYnPOrXC1T3LMGXE57C0' +
'EO/fgdBEHwAAoEvXbsw1Thw9qJelgVkG/yKGFAJo8C8ZjUaDU8fZwyaPz7xQ/Mt44dTG+Rg9qAdsrK2EboeIlMzcHCP7dcH+FTNo' +
'hUItGUUAqFmrNvzLs61MlZKchMib1zh19O94DP5FDCEE0OBfcrdvXkNSYjxTDT9/f1SrXoNTR/pVMcgfR9f9jNCAckK3QoxE7coh' +
'OL1pIUIqlBW6FdExigAAAJ06d2WucfzwAQ6d/LvTJw5zG/yLxEQ/wMyp4wRZNpiW99XO8SPsD6V17dZDlO+Z+5fxwu5l01DKha7W' +
'CF8+nm7YsXgKzQSUkNEEgC5duzF/KUbevIq3DEsLv0+9Bh8hrGJl7nWjo+5j+g9j9DoTkC+XY/aPExAddZ977aCQcIybPBNWVtbc' +
'awst/m0cbl6/zFRDIpGgcxf2sKtvVpYWWDd3HA3+RGdKe5bCxgUT6ZmAEjCaAODrWxb1GzRkqqHRaHDowG5OHf2VpZUVxk6aoZMQ' +
'oM/bATTtr71D+3dDo2FaVwsNGzVGmTK+nDrSn2/6d0PFIH+h2yBGrlalYIwd0kvoNkTDaAIAAER89jlzjTMnjyArK5O9mX8h9hBA' +
'g7/2cnKycebkUeY6PD7j+ubp5oKvI9je5yekuL6K+BR+ZbyEbkMUjCoAtGn7MTw8PZlq5Ofn48RR3S0eItYQQIM/mxNHDjI/q+Hu' +
'7o6WrVpz6kh/Rg/uSU/7E72xkBW+HUA+zKgCgLm5OXr0YJ/+OXJgL5QKBYeO/p3YQgAN/myUSiUOH9zDXKdHz96ie/ff3tYGPds3' +
'FboNYmK6tmkMWxsKnR9iVAEAAHpHfAapVMpUIy0tBadO6HZhILGEABr82Z09dRSpDBtWAYCZmRl6R3zGqSP9adWoJtMiP6xr7xcw' +
'LrhEtMf6u2f5t7extkLLBjWZzm8KjC4AlC5dhss06b5dW1FQUMCho/9m6CGABn92arUa+3dvZ67TslVrUT7816hmJabjLRif6JbL' +
'dTeTR96P9XfP+m/fuDb/71VjY3QBAAAGDf6SuUZyUiLOn2Hfr/1DDDUE0ODPx4WzJxH/No65Do/PtBDCGZ/8t5CxbVci1+GtPPJ+' +
'efn5TMezvs4XRotNfZBRBoDadeqicpUqzHX27tzMvGZ7cRhaCKDBnw+1Wo09Ozcz16lUuTLq1K3HoSP9K+vjwXQ86yAgz6cAIBTW' +
'3z3rv3250mwPhJsCowwAAJ8rprdv4nD21DEO3XyYoYQAGvz5OX/mBOJevWSuM3jIUA7dCMPelu3fWsY6A0ABQDC5ctYZAAum4x3s' +
'bJmONwVGGwDate8Ib28f5jo7t67X6RsB7xI6BNDgz49KVYCdW9mXfvby9ka79my7XbKQydi+hFnZWLE9yZ3HOAgR7bE+A2BtKexn' +
'z8LC+HeoNNoAIJPJMPhL9iun5KREHDuiuz0C/k6oEECDP18njv6BhHj2vcoHDxkq6Kt/dnZ2TMdn5bC9ieLq5MB0fEZWDtPxRHsZ' +
'WdlMx7s4s/3bZ2az/duzfvbFwGgDAAD0ieiLUqVKMdfZu2OzXtfa13cIoMGfL4UiH7u3b2Ku4+rqhj4RfTl0pD1bW7Zp1BdxCUzH' +
'uzIOAm8SU5iOJ9p7k8D2u2fdN+L5a7ZdNykAiJy1tTWXZwEyMtKxb9dWDh0Vn75CAA3+/B3Ysx1pqewDz6DBX8LGRtjfnYurC9Px' +
'92JimY53c3ZkOv4tBQDBvE1i+92zzv7cf/SM6XgXF7bPvhgYdQAAgH79B8LFxZW5zoG9O5CUyHY1U1KWVlYYM2k6gkPDudcu2kqY' +
'tvTlKyM9Dfv3sL/37+DggL6f9+PQERt//wpMx5+7xvbZYp4BSGBbgIloj3UGgDUAnL3K9tnzL8/22RcDow8ANjY26D9wEHMdpUKB' +
'zetXcuioZKysrDFhymydbSVMW/rytWXDKuTlst8uGjBwMBwc2a5+eagQEMB0/NHz15kexGOdBqZbAMJ5k8gWvtxctP/85+TKcezC' +
'Dabzs372xcDoAwBQOJXq5ubGXOfS+dM6GTA/RJe3A3gz1Wl/AHjx7CmXHf8cHB0xcNAQDh2xCwoKZjo+OycPOw6d0fr4Ml7uTOen' +
'GQDhsM4A+Hpr/2+/4/AZ5ObJmc7P+tkXA5MIALa2thj69XDmOhqNButWLmXe010bYggBpjz4azQarF6+GGq1mrnWsGEj4OjEduXL' +
'S526dSGRSJhqLFyzE0otl9VmXcyF9UEwor3ncWy/+3I+2v3b5yuUWLB6B9O5zczMUKduXaYaYmASAQAA+n0xgMu6AE+fxOC4Hl8L' +
'fJchhwBTHvwB4OypY3j44C5zHQ8PD3wxgP2WFS+urm4IDg5hqhH76i2WbNir1bHaDgJFUtIzEZ+UylSDlNybhGSkZWRpfbxEItE6' +
'/C1ev4f57ZPQsHA4O9NDgEbD0tISI7/5lkutTetWIC1NmHuLhhgCTH3wz8nOwqa1y7nUGjHqW1hbG9azE40aN2GuMXPZJly987DE' +
'x9nZWjM/B8D6NDgpufuPnzMd7+7qBBvrki8CdSUyCrN/Z19+m8dnXgxMJgAAhfup+5cvz1wnLzcXm9au4NCRdgwpBJj64A8Am9ev' +
'REZGOnOdsmXLoXefCA4d8fVppy7MNfIVSkR8Mx1x8SW/J896G+AB42BESu4BY+jSZubn1dtERHw7Awol+y6unTp1Zq4hBiYVAMzN' +
'zfHD5Glcap0/cwL37tziUksbhhACaPAvfJ3yxNE/uNSaMGmy4Evv/ptKlStzeSAqMSUdXb6eXOIQEFyebRtk1sGIlBxr6Crpv/nr' +
'+CR0/XoKklLZg3hIaBhCw/i/em2ITCoAAIX7qjdu8hFzHY1Gg99+nafXFQL/TsgQQIM/oFQq8fuv87g8FFqjZi183K49h650o0ev' +
'3lzqPHzyAh/1Hlmi2wEVGbcUZp2OJiV3P4YtdIUH+RX7Z69ERuGjXqMQ/ZR94y0A6Mnpsy4GJhcAAGDqj9Nhbs62yxgAJCXGY+uG' +
'1Rw60p4QIYAG/0LbN6/F61cvmOuYmZlh2k8zmJ+216U+EX25PRSVmJKODgPHY8HqHcWarq3EGAAexb5i3pOAFF9GVg4eP3/NVKNi' +
'4If/zRXKAsxbuR0dBo3ncuUPFD702qu34d2G0xWTDACBgUHo2asPl1pH/tjL5elvFvoMATT4F3oW+wQH9+7kUqtL126oUqUql1q6' +
'Ymtry2VBrSL5CiWmLlqHWp8OwbrdR9/7znZYYDmmcFSgUuHSrQdaH09K5vKtB1AxvA4rkUgQFljuP/97Tq4ca3cdQc1PBuPHxeu5' +
'3PMvMnDwEMGX39bwhUjYAAAgAElEQVSn4vxVMc1vsi4GoSvJycloWL82MtLZk6OXtw/mLFwOS8atS1npcl1/gAb/IkqlEuO++RIv' +
'X7DfW7azs8P5i1fh4cn2oJs+ZGZkoEG9WkhO5r+4jq2NFVo2qIlGtSqhYpA/yvp4wNHeDhaywpm6qu0H4tkr7XdXHNa3E34c9QWv' +
'dsl7jJ+7Eks3avfaJwD4lfFC5IHCh6wVygJkZGXjRVwC7kbH4ty1Ozh24QbzIj//xsPDA+cuXoW9vT332try9mBexv69Y7zJBgAA' +
'WL9uDcaO/o5Lreat2mHQV6O41GKhqxBAg/+f1q1ahj/28bn6n/rjdINZ9a84tm/bipHDvxK6jRKrEloBZzYvFLoNk9Cg2zBRvnq5' +
'ZNnvXN544UnXAcAkbwEU6RPRF9Vr1ORS68TRg7hy8RyXWix0cTuABv8/3btzC4f27+JSKyQkFJ/368+llr507dYdderWE7qNErsX' +
'HYv0TLb96cmHpaZnIerJc6HbKLH69Rvgk09N49W/d5l0ADAzM8OcufO5PBAIACuWLRRsgaB38QwBNPj/KScnG0t/mcPlqf+iz55M' +
'JuPQmf5IJBIsXLTYIDYqKgmVWo2LN/W/j4epuXjzHtRq/S+VzsLJyRkLFi026IdwdcWkAwBQeBXW74sBXGplZWZg6S8/C7JXwN/x' +
'CAE0+P9Jo9Hgt0VzkZKcxKVer94R3Gaf9M3Xtyx+XbxUdF+Yh85cEboFo/fHaXH9jiUSCeYvXITSpcsI3YogTD4AAMCYcRNQrlzx' +
'3zt9nzu3rmPPDvalKHlgCQE0+P/V4YN7cPXyeS61PDw8MGHiD1xqCaVFy9ZcNtjSp0Nnrmq9KRH5MIWyAIfPXhW6jRIZPmIUWrdp' +
'K3QbgqEAAMDGxgY/z1/I7Ypm26Y1gq4S+C5tQgAN/n/19EkMt7X+AWDm7LkGs9sfi/ETJnFbIEgf0jKycO6asK/sGrNTl28hIytH' +
'6DaKrXOXbhg9drzQbQiKAsD/1K/fAL37fMallkajwaJ5M5CWKvzzAEDJQgAN/n+Vk5ONhXN+hFKp5FKvc5duRnPFIZFIMHvOPLRs' +
'1VroVopt34mLQrdgtPafuCR0C8XWqnUbLPjlV9HdxuKNAsA7Jk2eymXLYADISE/DL3OnQ6UyjCnH4oQAGvz/SqPRYPGCWUiI1/79' +
'83eVKlUK036azqWWoZDJZFi1Zr1oVk87ePISClQqodswOsoC8Uz/d+3WHctXruH28LeYUQB4h729PeZyvBUQdf8O1q5cyqUWD+8L' +
'ATT4/9P2zetw89plbvVmzZlnlHuMS6VS/DxvAb4aNsLgr6hSM7Jw+nKk0G0YnZMXbyEtI0voNt5LIpFgxMhvsHDREtG9faMrFAD+' +
'pslHTdGvP5+3AgDg6B/7cOLoQW71WP1bCKDB/5+uX7mI3ds3cqvXs1cftGn7Mbd6hkYikWDCxB+wZt1GODk5C93Oe63ZeUToFozO' +
'6h2HhG7hvRwcHPD7itUYM26CwYdUfTLplQD/i0KRjzatWuBhFJ/1w6VSc/zw088ICavEpR4PRSsGFhSoaPD/m7hXLzH++6+Ql8tn' +
'A5myZcvh+KmzsLOz41LP0L169RIjh3+Ny5cM83671MwMdw+tho+nm9CtGIWXbxJQpd0Ag33/v379BliwaLEoX/WjlQAFYGFhicVL' +
'lsHCwpJLPZWqAPNnT0NSYjyXejxYWllhzKTpmDB1Fg3+78hIT8PMaeO4Df7m5uZY8ttykxn8AaBMGV/s2rMf6zZsgpe3t9Dt/INK' +
'rca63TQLwMuanUcMcvB3d3fHL78uxfZde0U5+OsDBYD/EBIahvETJ3Grl5GehplTxyMnx3CWI7WysoaVlbXQbRiM/Px8zP5pIhIT' +
'+AW1Ud9+j2rVqnOrJyYtWrbG6bMXMXrseLi4MF/JcLV+9zFaE4ADhbIAG/ceF7qNv3B1dcPY8RNx4fJ1dO3Wnab834MCwHsMHDSE' +
'6ytbr1+9wJyfJnF7pYzwo9FosHj+TDx5FM2tZr169TF8hPAbRAnJwcEBI0d9i6s3IjH1x+kICQkVuiUAQHxyquhWrTNE+09cRFIq' +
'+46qPISEhmHaTzNw9UYkho8YZVKzbtqiZwA+ID09Da2aN8WrVy+51WzctCWGjhhNydSArF+9DAf38tnhDyhc7e/YybMoVaoUt5rG' +
'4sH9e9izexfOnj2Dh1EPoGbYO55F1dAAnN68QJBzGwONRoNGPUbgXkysIOc3MzNDSGgYGjf5CJ06dUZoWLggfegSbQdsAG7fjsQn' +
'7T+GQpHPrWab9p3Qb6D4tlU1Rvt2bcWmdSu41TMzM8OWbTvRsFFjbjWNVWpqCq5euYKYmGg8fvQIsbFPkJ6WjoyMDOTk5ECpVOj0' +
'/Nt+nYxWDcW5J4PQDp25gl4jf9LpOWQyC9ja2sLR0RFOzk4oXz4AFQICEBQUjDp16xrla7XvMvkAwPoL4BVAVq1YjkkTx3GpVaR3' +
'34Ho2LkH15qkZM6eOsZth78iY8ZNwIiR33CrR/5dt86f4MIFtv0ZqoRWwOlNC2g2TgtNe4/CrQePmWo0bNQY23bs5tSR8aG3AAxE' +
'/4GDuO8XvXn9Spw8+gfXmqT4rl4+j2WL+O7e2KJFKwwbPpJbPfLfvhgwiLnG7agnOEULA5XYkXPXmAd/AOjP4d+QaI8CQAnMW/AL' +
'Klbi9y6/RqPBimULcen8GW41SfHcibyBRXOnc73/HBAQiMXLfoeZGf1Z6UPLVq3h61uWuc6s3wxj904x+Xn5VuYaZcr4olnzFhy6' +
'Idqib6oSsLa2xvoNW+Dh6cmtplqtxqJ503Hx3CluNcn7PXxwD3NnTub6Noa9vT1WrF4Le3t7bjXJ+5mZmeGzz/sx17l+NxqHztAb' +
'AcV14OQl3Lz/iLlOv/4DIJVKOXREtEUBoIQ8PD2xYuUabosEAYUh4Nf5M3Hx/GluNcm/i3l4HzOnjkO+XM6tppmZGZYsW47AwCBu' +
'NUnx9OnzGWxs2BeyGj93JeT5un3g0BjI8xWYOH8Vcx1ra2v06NmLQ0eEBQUALdSoWQvTZ87iWlOtVmPx/Fm4fsUwl081BtFR9zF9' +
'8ljI5Xlc6078YQqat2jJtSYpHgdHR3TnMJA8fx2PJRv2cujIuP26fjdexCUw1+nd5zOD3zPCFFAA0FLvPp/hq2EjuNYsXDJ4Ks6e' +
'Osa1LgEe3LuDmVPHcR/8e/WOwJAv6XVOIQ0f8Q0sLdln5Oau3IbX8UkcOjJObxKSsWA1+1oZlpaWGPrVMA4dEVYUABiMnzAJnbt0' +
'41pTpVJh2aKfcfKYYe+uJSY3r1/BjKljkZfHZ33/Ik2bNcesOXO51iQl5+HhgR49ezPXyZPnY9qidRw6Mk6TFqxGbh77rbPeEZ/B' +
'08uLQ0eEFQUABhKJBPMX/oL69RtwratWq7F8yXwc2LuDa11TdPH8acydMRlKBd/7u8HBIVj62wqYm5tzrUu0M2z4SMhkFsx1dhw+' +
'izNXb3PoyLicvhKJ3UfZ1lwACjda++qr4Rw6IjxQAGAkk1lg+ao1qBAQwLWuRqPBhtW/YdumtVzfUzclhw/uwaK506FS8d30xcen' +
'NDZv3QEHBweudYn2vH18uDxUptFo8OWkBUjPNJxNu4SWmZ2DYVMWcfke6tW7j0HuEGmqKABw4Ozsgu079qBMGV/utXdt24Clv8zh' +
'PogZM41Ggx1b1mHN8sXcw5OLiyu2bNtJU5gGaOSob2Ftzb675dvEFIz7md/S0GI3etbvXJ6NsLGxMfnNsQwNBQBOPL28sG3Hbri7' +
'u3OvffbUMcyYwm+PemNWUFCAxQtmYseW9dxr29vbY/PWHdxnewgfXt7e3B4u23LgJPafoDdyDp25gq0H+axR8tWwERScDQztBcDZ' +
'vbt30bVzR2RmZnKv7Vc+AGMnTYezge2tbihysrMwb9ZU3L/Lf2lXS0tLbNq6A/Xq1edem/Ajl8vRsF5txMW9Zq7l5uyIy7uWoJSL' +
'E4fOxCclPRN1Ow9FYgr7dr9e3t44f/EqlzUbTAntBSAyFStVwtr1m7hMRf7ds6ePMfabL/H0/9q78/CYrj4O4N/JZCKRzRZLIhHR' +
'SEIR+5ogEWIvWmttLapvF+qlRVuKVnVF37YoLSq1tLprLVWitMVbbXVB7cTW0ipijWTeP9J5O43ILOece2fmfj/P43mKub9zqbn3' +
'd7bf2feL9Nje7sTxHEwcd7+Sl7/FYsFrC97gy98LBAYG4pEJj0qJdebsOQyf8DzydTquWE8FBVbc8+gLUl7+APDY45P58vdATAAU' +
'aNa8BZYufwfBwcHSY5/943dMGj+KtQLs/PD9Djw67gGclNDrK8psNuOll19FRvtM6bFJjZ69bkdycn0psbK3fY8Zc4x3VsCTr7yJ' +
'9V/ukBKrQYOG0g9SIzmYACjStFlzLFychcDAQOmx8/Ly8OrsZ7FsyeuG3yHw6cfv4ekpE3Ax94L02P7+/pgzbwG639ZTemxSx8/P' +
'D8+9MFPaFs3nF6ww1HqAT7O3Sin4AxR+h6bPeI7HLXsoj18D4O02f7EJgwf2xxWJteft1a6TjNHjHkO4wcpqXrt2FQvmzEb252uV' +
'xLf1/Hv0vF1JfFJv6hOTMHfOK1JihQQH4fOsF5FQPVpKPE+1/8hxpA0Yg/O5F6XE+9f9D+KxxydLiWVEqtcAMAHQwBebsjF08J24' +
'fFluGVqbChEVMeaRybilZqKS+J7m+LGjeOHpJ3As54iS+BaLBf95ZQ66de+hJD5p4/Lly0hvk4LDhw9JiZcYF4N1bz6HsBD5U3ue' +
'4HzuRWQMHItfDuVIiRdbvTo2ZG9RMgpqFFwE6ANSW7fBshUrlRWOOXP6N0waPwprP/lQSXxP8vWWbEz8933KXv6lSpXCgoWL+fL3' +
'AUFBQXh+5mxpw897Dh5Fv1HTfPLUwCtXr6HvqGnSXv4mkwnPPPsCX/4ejgmARpo0bYaV732I8uUrKIl//fp1vD7vJTw3fRIuXJC/' +
'BVFv165dxcL5r2Dms9Ok1/S3KV26NBYvWYqMjA5K4pP2WrRoKeWcAJsvd/yEEY++4FM7A/ILCjBswnP4asdP0mL2HzAQKamtpcUj' +
'NTgFoLH9+/ahb+9eOHHiuLI2wsuUxf2jH0G9Bo2VtaGlnCOHMOv5p5BzRM5QbnHCwsPx1tIVaNjIN/7O6G+XLl1Ch4y2OLB/v7SY' +
'Q3plYtbj90uLp6eHZ8zFa8tXSYsXW7061q3PRkhIiLSYRsU1AD4oJ+coBvTrjf379ilrw2Qy4bZe/XBH/8Fee2BNQUEBVn3wDpZn' +
'vYHr19WVQo6MjELWshVITExS1gbp64edO9G1cyby8uQN348f2R/jR4qfP6Cnp17JwnPzl0uLFxBQCh9/sgZ16taVFtPIuAbAB0VH' +
'x+DjVWvQrHkLZW1YrVa8v3IpHnloJA7s977CQb+eOompj41F1qLXlL78ExOT8NGq1Xz5+7i69eph/EQ5BYJsZsxdiidmL5IaU0uz' +
'Fq6U+vIHgImPPc6XvxfhCICOrl27ilEP3I8PP3hPaTtmsxldbrsDfQYM9fjRAKvVis/XfYLFr8/BVUVbJ21atUrBgoVv8lQ/g7Ba' +
'rRg8sD/Wfya3iNaoIb0wZfRQqTFVm/5qFp59Te7Lv21aOrKWruCef4k4BeDjCgoKMGXy45j/2lzlbcXG3YKR9/8bcbfUVN6WO47l' +
'HMH8V2di988/Km+r1+298eKs2VLOkCfvcebMGXRsny7lrAB7I/t3w9Pjhnv8y89qteKRZ+ZJnfMHgKpVo7F63Xpli5yNigmAQWQt' +
'WYxHJ4yXOkdZHJPJhPT2nTFw6D0I8pDa3NeuXcWH7y7HByuXIS8vT2lbJpMJY/49DmPGPuzxD2tS4+effkT3rp1wSfLpmn06t8Ur' +
'U0fD32yWGleW/IICjJ72Mpa8L3cEJDAwEB989Cnq1qsnNS4xATCUr776EiPuHoo//lD/d1aufAUMHX4fmrZIVd5WSXZ+9w1en/sS' +
'Tp1UtyvCJjg4GC+/Og8dMjsqb4s82yerPsaIYUOll9LOaNUIbzzzMEKDPSO5trl46QpGPPo8Ptm4VWpck8mEOfPms26GIkwADObo' +
'0SMYMnAA9uzZrUl7yQ0aY/CwfyGqaowm7dn8euokliyci+1fb9GkvZiYalj4ZhaSkmpp0h55vqenT8N/Zs+SHrdWfCyWz34cMZGV' +
'pMd2x4lfz6Df6GnYufuA9NijRo+Rdvoi3YgJgAHl5ubioVH345NVH2vSntnsjw6duuGOfoMQHBKqtK0rVy7jvbffwicfrlQ+3G+T' +
'ktoac+bNR7lywl8m8iEFBQW4e+ggrF2zWnrsiHJlsOTFiWiWrG/C+fW3P2Pgv6fjzNlz0mNnduyEBW8shp8fN5OpwgTAoKxWK16b' +
'NwdPTZuidBucvdDQMNzebxDapHWQvj7g3Lk/8cWGdVj1wUqcPavNvwmTyYQHRz2EsQ+Ph9lD52VJX1euXEHf3r2wfZvcoXEAKBVg' +
'wYuP3ocB3dtJj+2MN99bi7FPz8G1PPnPj6bNmmPZipUs9asYEwCD275tK+4ZcTd+PXVKszZLBQaiWYtUNG/VGkm16yIoyL1k4M+z' +
'f+DHnd9h+9eb8c32r5Gfr00iAwDhZcrgpZdfZVlfcuj8+fPo1aMbfv5Jze6T2zJaYfakBxAeqs0hQhcuXsK4p+di+aoNSuInJibh' +
'/Q9XIbxMGSXx6W9MAAinT5/GfSOHY8uWzZq3bTabUSM+AdXj4hEZVRWRUdEIDQtH6eBgBAYGIT8/H1cuX8aVq5dx5vRvOHn8GE6e' +
'OIb9e/cg5+hhze8XAOolJ2Pe/DcQE1NNl/bJ+/x66hS6demInJyjSuJHV6mI+U+PVT4lsOOnvRg+4TkczDmpJH5kZBQ+WrUakVFR' +
'SuLTPxk+ARD9C/CVBMRqteL1Ba9h2pQnlG8V9FYmkwl3DRuOSZOncH8/uezA/v3oeVsXnD59Wkl8i78/Jtw7AKOH3g4/P7lbUPML' +
'CvDi62/jmbnLcD0/X2psm4iICLz/4SeIq1FDSXy6ERMAJgD/sPP77/Gve4fj0MGDet+KR4mIiMCsl15B27R0vW+FvNi+fXtxR8/u' +
'+O2335S10bxBbcx6/H4kVI+WEm/PgaMY/eTL2PrdLinxilOpUiW8/e4HiI/3zCJivopnAdA/1EtOxtrPNqJ3n75634rHyOzYCdmb' +
'v+LLn4TFx9fEex+sQuUqVZS18fW3P6PVHQ9g/LOv4eIl98tdX7l6DTPmLkXrfqOUvvwrVqyIFSvf58vfB3EEwIut/2wdxo19SNMF' +
'gp4kLCwMj016AncOHKz3rZCPyck5itt7dFe2JsCmWlQlPDfhXrRv1cil69Zu/i/GPT0XR0/8qujOClWJjMTKdz9E9bg4pe1Q8TgF' +
'wASgROfPncOT06Yga8livW9FU23T0vH8C7NQJTJS71shH3X48CG0aOrai9ldXdNbYNIDgxAfW7XEz+09dAxTXlokvaLfzXy17RvE' +
'xlbXpC26ERMAJgBO+WTVx5jwyFicOXNG71tRKiwsDFOmPoU+/bz7HHbyDhIewE7z8zOhW3pLPDFqCGKrVv7H7x0/dQbPzV+OrA8+' +
'U7bIrzhGeX56KiYATACcdvbsH5g6ZTLeXr5Meo1zT9Ct+22YMm06KlXyjBKr5Pu0TABsAiz+6N+tHcbf2x8Wf3/8Z/F7mLv0I1y5' +
'qv3uHyM9Pz0REwAmAC7bvm0rHhn3b/zyyx69b0WKmJhqmD7jWaSl61NRjYxLjwTApnRQYZW9S5fdXygoyojPT0/CXQDksiZNm2Hd' +
'5xvxyIRHvbpUp8USgAdGjUb25q/48ifDuXT5iq4vf/J9TAB8lMUSgFGjxyB781fo3KWr3rfjsnYZ7bFx0xZMmPi4VycxRESeigmA' +
'j4uJqYb5ry/Cu+9/hNq31tH7dhyKq1EDb2Ytw5tZy1hxjIhIISYABtG8RUusWfc5nn3+RVSoUEHv27lBlchITH1yOjZu+hLtMtrr' +
'fTtERD6PiwAN6Nq1q1i7Zg2ylizGls1f6LZjwM/PDy1bpeDOgYPRsVNn+Pv763IfRDej5yJAT8Dnp764C4AJgFIHDxzA0reW4NNP' +
'V+HwoUPK2zOZTKh9ax106JCJvv0HICqq5MInRHpiAsDnp56YADAB0Mzhw4ewKXsjNmVvxJdbNuPChQtS4pYtWw4pqalo2zYdbdLS' +
'uY+fvAYTAD4/9cQEgAmALqxWK44dy8G+vXvxyy97sG/fXhw9fBi5F3Nx7s9zuHDhAnJzcwEAoaGhCA0NRVh4GMJCwxBTLRY1ExKQ' +
'mJiEmjUTlB6sQqQSEwA+P/Vk+ASAiEgvog/g8mXC8Puf5yXdjfZt8/mtLxYCIiLyUt9+PB+jh96OwFIBmrVp8ffHkF6Z2P7BXM3a' +
'JO/EEQAior9cvnwZ3/x3O9atW4M1n36K48ePCcX78/tVAIBjp07jyZeXYMUnG5XuuslMbYLp44YjLrpw2q1McheheBEREWjdpi0y' +
'2meides2CAsPl3Gb5CROARARKXTkyGF8tm4tPlu3Ftu2bsW1a1elxbYlADb//WEPHn1+Abb/IPecjnpJNTB97HC0bHjrP35dNAGw' +
'ZzabUfvWW5GR0QEZ7TNRp25dmEzOvELIXUwAiIgkys3NxeYvNmHD5+uxccPnOHHiuLK2iiYAAFBQYMV7a7/AzIXv4Oe9h4Xi164Z' +
'izF39UaP9inw87vxcS4zASgqMjIKbdPSkZbeDimprRESEqKsLaNiAkBEJEhlL78kxSUA9rZ+twtPz30Lm7btdClus+RaGD30dnRI' +
'bVxiL1xlAmCPowNqMAEgInKRlr38kmxaNhv1khyfafHVjp8wc+FKrP9yx03XCJhMJrRr2RBj7roDzRvUdhhz5+4DaN1vlMv3LANH' +
'B+RgAkBE5ITdu3dh44bPsfHz9di+fRvy8vL0viV0bNMUy2Y97vTnf/zlIF7J+gDvr92Mq9cK779UgAU9OqTgvjtvQ52EOKdj9Rs9' +
'Dauzt7l8z7JZLBY0adIUbdPS0Ta9HZKSaul9S17DExKAqwDc3sOy98ARZn9EJJ2n9PJLYjKZkL10llOjAPZO//EnFr27BgAwpFcm' +
'IsqVcen6nbsPoE3/0bqd81GSKpGRSEtrh7T0dmiVkorQ0FC9b8kj5ebmomaNaiIhrgIo8Sx1ZxKAMwDcTkOyN3+FmjUT3L2ciOj/' +
'du/ehY1/vfA9pZfviKujADJ4Su/fEY4O3NyePbuR1rqVSIgzACJK+oAzx6+dhUACsCl7IxMAInKLN/TyHVmzaTt27j7g8iiAu3bu' +
'PoA1m7Zr0paovLw8fPnlFnz55RY8OW3K/0cH2qalIyW1taFHBzZlbxQNcdbRB5wZAVgNINPdO0hISMT6jV/AbDa7G4KIDMQbe/mO' +
'aDkK4C29f0eMPDqQn5+P9DYp2Lv3F5EwnwLoXNIHnEkAZgEQWkr65FMzcNew4SIhiMhH+UIv3xF31wK4ypPn/kUZaXRgwfx5mPTY' +
'RNEwMwGMKekDziQAfQAsF7kLi8WCrKUrkJLaWiQMEfkIX+zlO6LFKICv9P4dsVgsaNy4CdLS2/nc6MDmLzbhzv59ZHwnegN4p6QP' +
'OJMAVARwysnP3pTFEoDJT0zF4KF3cTqAyGCM0Mt3RPUogC/3/h3xhdGB69evY9HC1zFtymQZL/8CAJVQuBDwppx9qW8F0FT0jgCg' +
'Zs0E9L9zIFJbt0F0dAyCg4NlhCUiD2PEXr4jKkcBjNL7d+QfowNp6Uiq5bhokl4uXryInJyj2JS9EcveyhKd87f3FYCWjj7kbAJw' +
'H4CXhW6HiIiw4a2ZaFA7XmrMH/YcQOt+xuz9U7HuBeDwPGg/J4MtA3BR6HaIiLxUVEgQ+iVEi82D/uW5+UJLqor19Nylwi9/E4B+' +
'CdGICgmSc1OklwsAVjjzQWfqAADAHwDmwcGKQiIiX5OV2Rh9a1YFAOTmXcfHB08KxZNdF0DWvv8ucVWwpEMjAMDyvcdw55r/Csck' +
'XcyBEzUAAOdHAADgeQC5bt0OEZGXstgdszupSaLwKIDVasWMeUsFo/xtxjw5vf9JTRL//3NLMUcLk1c4D+AFZz/sSgJwEsAUl2+H' +
'iMiLHcu9/P//rl+xDLrEVRGOuTp7G779eZ9wnB/2yOn9d42rgvoV/z5vIOfC5RI+TR5sEoDfnP2wKwkAAMwGsMPFa4iIvNbx3Cv/' +
'+LmMUQBAzloAWXP/j9v1/gHgxMUrxX+YPNl2AK+4coGrCUAeCgsDnXPxOiIizVnMfogJF9sTnnPh0j9+LmsUwLYWwF0y5/7te//A' +
'jX9mV8WEh8JidvX1QgL+BNAXwHVXLnLn/9ABAANdbYiISAtRYSEY1rA23u3bGWfGj8ALmSlC8eynAGw8YS2Airl/m+L+zK54ITMF' +
'Z8aPwLt9O2NYw9qICuOR8ApdBzAAwCFXL3R2F0BRHwO4B8ACCFYIJCISYTH7oWVMJDJvqYaONauhbqUK//j96HCxl0/RKQDg71EA' +
'vXYEqOz9A8X/mV0RHR6CsFIB6FmrBnrWKvyz/fDrGazeewRr9h/Bl0dPIC+/QKgNAgBYAQxD4cE/LnM3AQCAN1CYeSwAYBGIQ0Tk' +
'korBQehwSzV0TayOjBoxKBNY6qafrSrY+zxx8TLyrVaYTf/s60xumohVB09CpA9utVrxzLxlWDrrMZeuk9H7B4BHG994VHu+1YoT' +
'F8VGAKKLmXapW6kC6laqgEdSGuLitTx8nXMKH/9yEB/sPoij5y4ItWdQ+QBGAljsbgCRBAAA3kRhjYAsAOGCsYiIiuWol1+SyiHB' +
'8Pfzw/UC93qc1wusOHXxyg0FcpIj5IwCrN60zaVRAFm9/65xVdCoUtkbfv3UxSu4XuB+cuHv54dKwaVL/ExwgAXtakSjXY1ozO7U' +
'mqMDrvsThcP+bvX8bUQTAABYBaABCisPNZIQj4gIUWEh6BhfDZnx1U3uZf8AABIbSURBVJBRIwZhpQLcimP2MyEyNFiol3k898YE' +
'ACicP5cxCjBj3lKnzwhQOfcPiA//R4YGw+xiHQH70YHzV69h/YEcrN53GGv2HcGx8yw/U8R2FC74c3nOvygZCQAAHATQHIVnBkwF' +
'ECYpLhEZhNnPhOTKEeiSUB1dE6qjQZWKMElaYVQ1PEQoAcjJvYQmuLG3LGstgK0ugKMzAlTt+7d3VHAHgOiai6JrBw6ePYeP9xzC' +
'qr2H8MXhE7iWny8U34tdBDANhYV+pCzCl5UAAIU3NBvAcgBjUTg3waWfRHRTsnr5joiuAyhpVbyMUQCgsC6Ao1EAVfv+7R0XnP+X' +
'veI/rmw4RjVPxqjmyUYdHbiAwvK+L8CFIj/OkJkA2PwKYByA6QD6oXDLYFNwtwCR4VnMfmgRXQUd42NdnssXUdyiNFeUNCyu1Y4A' +
'1Sv/bcR3AIj9XZfEQDsLrAC+BrAEhdPrTtX2d5WKBMDmLIBX//oRAaAtgGYAEgHEAagAIBSAmpSfiDyCVr38koiOADgqjKPFWgDV' +
'c/82okWARP+uXeEDaweuobCHfwaFU+m7AWwFsPGvX1NKZQJg7zSAt//6QUTy3Y3CYULdt+SqnMt3l8opAED9WgAt5v5tcgSLAImu' +
'AXCXl6wduA5gNFws2auKVgkAEalhAjD5rx+6cWVfvh5UFAMqSkZdAAB4fv6KG+oCyJj7B4CJxez7L+q4aAIQpm4KwBX2awc8qO6A' +
'P4CXAdQEMAaFe/l1wwSAyHuVRuGi2656NN6gSkXcXvsWdKoZi3qVtZnLd5eqYkD2VNUFUL3v316+1YqTggcBRYUFC12vQnF1Bz7d' +
'exgrf96PHSekrqtz1oMAYlG4Tk5szkUAEwAi7xSGwhocYoXuXdQoqiLuqB2P22vfgriy3lP7S1UxoKJUrAXQau4fkFMEqHKI5yUA' +
'RdnWDoxPaYSDZ89h5c/78c7P+/DNcU2TgW4AVqMwgT+vZcM2TACIvE9ZFD44mmrRWCl/M7olxGFMi/poFl1ZiyalM/uZUCU0GDkK' +
'igHZk70WwN/sp9ncPyB+CJA7RYD0Flc2HA+3aoiHWzXEdydPY+5/f0TWzj24lKfJeXepADYAyIQGi/6KYgJA5F0qoHCF8K2qG6oc' +
'UhqjmidjWMPaqFC65BefN4gODxFKAG5WDKgomXUBACjf929PNAGoqtMCQFnqV4nAvG5peKpdcyzY8TNmf/09TuUqH6FviMIkoC2A' +
'31U3Zo8JAJH3CAewFopf/mWDSmFcy4Z4sFk9BAfovqlAGtU7AWxk1gWQwdG+f3vCCYCPHPtboXQQxqc0wgNN62H21u/x3JZv8eeV' +
'qyqbrANgDYB0aDgd4KdVQ0QkJBjAJyg8d0OJIIs/xqc0woHRQzAhtZFPvfwBtcWAiprUJFG48pnVatVs7t/Gk4sA6SE4wIKJqY1x' +
'8KEheCSlIQL9zSqba4TCdT0ln6QkERMAIs/nD+AdAC1VNdC2elXs/Fd/PJ3RAmWDPGsLnyyqiwHZs40C6M2V3j/gXUWAtFQ2qBRm' +
'ZLTED/cNQJvqVVU2lYLC77rSTMOGCQCR55sJoKOKwGUCS2FetzR8PqQn4ss7/6LwRlpNAdhMbio+CiDqUSf2/dvz1iJAWokvXwYb' +
'hvTE4p4ZKF86UFUznQC8qCq4PSYARJ7t/r9+SJdRIwa7HxyIEY1u1b1Snxa0KAZkLzmiDLrqOArQzYl9/0WdEJwC8NURAHsmEzAo' +
'OQk/3X8n2tWIVtXMgyg8XVcpJgBEnisdwCzZQc1+JkxJa4Y1g7qjcohm0426k1UMyBWPS1gL4A5XVv7b5FutOCF4EqAREgCbyiGl' +
'sXbQbXiibVNVWx9nAUhTEdiGCQCRZ4oE8BYkzwVWDimNdYN6YFKbJvAzQrffjq0YkLtsxYBcUb+iPqMAzu77t2eUIkAy+ZlMmNy2' +
'KdYOug2V5CfT/gCWAlD2D4gJAJHnMQN4E0AlmUFrRZTDtnv6IC1O6SImj2UrBiTCnVXyk3RYC+BMzf+ijFgESJb0uGhsG9EHSRHl' +
'ZIeuBGAZFC0KZAJA5HmmoHD4X5qUapHYMuwOxPjYNi1Xia4DyHGjKIzWawHcmfsHWARIVLUyodgy7Ha0jJH+/7o1gEmygwJMAIg8' +
'TVMA42UG7FXrFqwb3MNnt/e5QuudADZarQVwZ+7fhkWAxJULCsRng3ugx18HOUk0EUBj2UGZABB5jlIA3oDE4b6etWpgee9M1QVM' +
'vIaWxYDsaVUXwJ25f5tjF8QSAKOPLtkEWfzxTp9O6HNrTZlh/VE4LSi1JjcTACLPMR1ALVnBuiRUx/I7OgotfPM1WhYDKkpGdcCS' +
'iPT+AfERgCiOAPyf2c+EN3tloFPNWJlhE1E4PSgNnwxEnqEOCvf+SpEeF413+nSCxcyvuD29pgAA9aMAIr1/gEWAZAswm/Fe387I' +
'jK8mM+wYAPVkBePTgUh/JgBzIOlwrnqVK+DD/l047F8MrYsBFaVqFEC09w+wCJAKpfzNeKdPJ9StVEFWSDOAlwA5/4yYABDpbxAk' +
'1fkvXzoQ7/Xr7HMH+ciiRzEge6pGAUR7/zKKAPnaQUCyhARY8NGArogIljZ9nwqgv4xATACI9BUMYIaMQBazH97t2xlxZcNlhPNJ' +
'ehQDKurORPnlYwcIxpRRBKhSsHGqSrqqWplQvN27o8wpuWch4dRAJgBE+noQQGUZgWZmpqJ1bJSMUD5Lr2JANkt2H8Vdn+0Qar84' +
'd322A0t2H3X7etGpjSoGLgLkrDbVq2JmZqqscJGQcEYIEwAi/ZQBMFZGoMz4avhXk7oyQvk8PYoBXcnPx0ObfsDQz3bgYl6+UPvF' +
'uZiXj6Gf7cCQde7FPyp4DHAMFwA65b6mddElobqscOMBuF7xyQ4TACL9jAUgXDs0IjgIi3pkGOJEPxm03gmw+48LaLI8G//ZeUCo' +
'XWdk7TmK5m9nY/cfF1y67rjg/D+3ADpvfvd0WesByqJwV4DbmAAQ6SMMko75ndO1rYqDSHyWlsWAsvYcRfMV2dj1+3mhNl2x6/fz' +
'aLJ8Ixb8dNjpa1gESDuVQ0pjfndplb4fBOD2oh8mAET6GAGBL67NgLoJ6FXrFgm3YxxaFAO6dD0fw9Z/iyHrdiA377pQe+64fD0f' +
'Izd8h2Hrv8Wl646nBFgESFvdE+PQt46USoFhAO5292ImAETas0BC0Z+QAAueaS9l96ChqJ4C+OVsLlq+vQmLdh0RakeGRbuOoMmy' +
'jfjJwQgEiwBp7/kOrWRt130IQIA7FzIBINJeHwDCe8EmpjZmz8sNoi+rkhKAJbuPosnyDfjxzDmhNmTac/YCWr6dXeIugeM8CEhz' +
'UWEhGJ/SSEaoqgB6uXMhEwAi7Y0QDRBXNhwPtagv414MR7gYUO6VG4oBqV7lL6qkXQL5VitOCtY2YBEg94xr2QDx5d0v4GRnuDsX' +
'MQEg0lYCgFaiQaalN2OpXzeJFgPKt/6zGNAuDVf5i7LtEthlt0uARYD0U8rfjEltmsgI1QaAy4uBmAAQaetuCNbxrlEuHL1vjZd0' +
'O8YjsxhQ4Sr/jZqu8he16/fzaGq3S0B0AWAkiwAJ6VcnQcYogAnAXa5eJOXwESJyih+AO0WDTExtrNkRv6ZJLwldb50q7YBDqaLD' +
'Q5BzzrW98vZ+OXsBr/10yCMW+rnDtktg66k/0DpK7KCaqh68ANAb/v2a/UwY16oBRny4QTTUIACPAShw9gKOABBppyUAoZNgYsJD' +
'MbCe2KlvJL4OYOSG77z25W9v0a4jGLnhO6EYXAAobnBykoy/xygAzVy5gAkAkXbcWqlr74Fm9WQeKGJYog/bq/lOd7LcEhxgwcIe' +
'GVjYI0P5yY6ifxYmAOICzGbc37SejFAuPWP4JCHShglAD5EA/n5+uJO9fyk8edV6YoWy+Hp4bwypn4Qh9ZPwzci+qFOpvN63dVOs' +
'AijH4PpJMpL7O+DCGiMmAETaqAcgRiRAl4TqqMySv1J4aq91cPKNL3xbQjA4OUnHO7s51qKQo3JIaXSMjxUNEw2gjrMfZgJApI0M' +
'0QB3N6gl4z4InpcABFn8MatjKhb1LH7IPzjAgkU9M7D4Jr+vJ1YBlOcuOd/xds5+kAkAkTaETv+ICA5CZnw1WfdieJ700qoVUQ7f' +
'jOyLUc2THX52UHIStt/TB7UihA+RlMbTkilv1rlmLMqXDhQNwwSAyIOUgmDxn47xsZpt/TMC0WJAsgx244VeK6Ictt/TxyOmBPz9' +
'/FA5RKymAv3N389PxjRAKpw8G0D/bwCR76sPQOgp2TWhuqRbIUBOMSARjob8HfGUKQEWAZKvc81Y0RDBABwPJ4EJAJEWGotcbDH7' +
'IaOG0PpBKoZe0wCJFcpi24g+Tg35OzKomEWDWvKkqRRf0TE+VsZuAKeeOUwAiNQTOvKrVUwkwgPdOu2TSqDH3HVxq/xF6blLgDsA' +
'5AsPDEDzaKF6YYCTzxwmAETqCY0AtIgRfhhQMbRMAEpb/LGwR4bbQ/6O2KYEFvbIQGmLdhXeuQBQjZbi33kmAEQeIABATZEATatW' +
'lnQrZE+rYkCJFcpi64g+GFJffQ99SP0k7Li3n2ZTAiwCpEbjqEqiIRIBOMw0mQAQqVUDgNC5vY0iK0q6FbKnRe91YL1EzefobWsM' +
'RjS6VXlbnnwQkDdrJp70+wOIdfQhJgBEagmd2xsTHqrranVfpjIBCPQ3Y1bHVLzZq70uq/SDLP6Y1y1N+S4BTgGoUSU0WMb6Cocj' +
'j0wAiNQSSgBu9eAa8N5O1Qp2Vwr7qDYoOQnbRvRWVjiICYA6EkaNHCYA2q0WITImof171cuGyboPKsJWDOh6gbyT/QbWS8Scrm2d' +
'7nVrcV597Yrl8c3Ivhi9+gu89s1PQu3ZYxEgtWLLCH/3ox19gCMARGoJreapFs4EQBWZxYD0HvJ3RMWUAIsAqVWtjPACS4fPHiYA' +
'RGoJreDjCIBaMqYBEiuUxfZ7PGPI3xGZhYNYBEgtCd99h88eJgBEakWIXMxtVmqJzmGrKOyjmqzCQSwCpJaE0T8mAEQ6KytyMSsA' +
'quVuAqC6sI9qMgoHcQGgWhK++w6fPUwAiNQSOtvTG18u3sSdl5iWhX1UEykcpFUhJaOS8N0v5egDTACI1HL4JSyJlmVdjcjVl5ge' +
'hX1Uc7dwENcAqBVsEU4AHHY+mAAQqSWUAEh4CFAJnB0B8PRV/qLc2SXAKQC1JPw7c5gAsHtBpJbQt7iUv1AVYXLAmV5srYhyeLtP' +
'R9Su6Du9/psZlJyEhpEV0XvFauw6/UeJn2UCoFag+Hff4SICjgAQkWHZigHdzMB6idh+Tx9DvPxtbIWDhje8+ZQAiwD5BiYARGRY' +
'NysG5OtD/o4EWfzxWvebTwmwCJBvYAJARIZWdBrAmwr7qHazwkFcAOgbmAAQkaHZz2X74ip/UbZdAvZTAlXDuAXQFzABICJDiw4P' +
'NfyQvyNFpwQ4AuAbuAuAiAytdWwUhtZPMtRCP3fZdgkcPHte71shCZgAEJGhdU2orvcteJXaFcszWfIRTACIiHRinfqg3rdABsY1' +
'AERERAbEBICIiMiAmAAQEREZEBMAIiIiA2ICQEREZEBMAIiIiAyICQAREZEBMQEgIiIyICYAREREBsQEgIiIyICYABARERkQEwAi' +
'IiIDYgJARERkQEwAiIiIDIgJABERkQH5630DRERGZZr0ktD11qkPSroTMiKOABARERkQEwAiIiIDYgJARERkQEwAiIiIDIgJABER' +
'kQExASAiIjIgJgBEREQGxASAiIjIgJgAEBERGRATACIiIgNiAkBERGRATACIiIgMiAkAERGRATEBICIiMiAmAERERAbkr/cNEBEZ' +
'lXXqg3rfAhkYRwCIiIgMiAkAERGRATEBICIiMiAmAERERAbEBICIiMiAmAAQEREZEBMAIiIiA2ICQEREZEBMAIiIiAyICQAREZEB' +
'MQEgIiIyICYAREREBsQEgIiIyICYABARERkQEwAiIiIDMul9A0Q+zqr3DRCRYZX4jucIABERkQExASAiIjIgJgBEREQGxASAiIjI' +
'gJgAEBERGRATACIiIgNiAkBERGRATACIiIgMiAkAERGRATEBICIiMiAmAERERAbEBICIiMiAmAAQEREZEBMAIiIiA2ICQERERERE' +
'RERERERERERERERERERERERERERERERERERERESko/8BXdk/jkRh35gAAAAASUVORK5CYII=';

  RESPONSE_PAGE =
  '<!doctype html>'+sLineBreak+
  '<html lang="en">'+sLineBreak+
  '<head>'+sLineBreak+
  '  <title>Authentication Result</title>'+sLineBreak+
  '  <meta name="viewport" content="width=device-width, initial-scale=1">'+sLineBreak+
  '</head>'+sLineBreak+
  '<body>'+sLineBreak+
  '<style>'+sLineBreak+
  '  hr'+sLineBreak+
  '  {'+sLineBreak+
  '    border: none;'+sLineBreak+
  '    height: 1px;'+sLineBreak+
  '    background-color: rgb(171, 171, 171);'+sLineBreak+
  '  }'+sLineBreak+
  '  div.container'+sLineBreak+
  '  {'+sLineBreak+
  '    position: fixed;'+sLineBreak+
  '    font-family: Arial;'+sLineBreak+
  '    font-size: 12pt;'+sLineBreak+
  '    max-width: 100%;'+sLineBreak+
  '    max-height: 100%;'+sLineBreak+
  '    top: 50%;'+sLineBreak+
  '    left: 50%;'+sLineBreak+
  '    transform: translate(-50%, -50%);'+sLineBreak+
  '  }'+sLineBreak+
  '  div.section'+sLineBreak+
  '  {'+sLineBreak+
  '    display: inline-block;'+sLineBreak+
  '    margin: 15px;'+sLineBreak+
  '    padding: 5px;'+sLineBreak+
  '    float: left;'+sLineBreak+
  '  }'+sLineBreak+
  '  div.image img'+sLineBreak+
  '  {'+sLineBreak+
  '    height: 100%;'+sLineBreak+
  '  }'+sLineBreak+
  '  div.text'+sLineBreak+
  '  {'+sLineBreak+
  '    padding-top: 15px;'+sLineBreak+
  '  }'+sLineBreak+
  '  div.text2'+sLineBreak+
  '  {'+sLineBreak+
  '    font-size: 18pt;'+sLineBreak+
  '  }'+sLineBreak+
  '  span.title'+sLineBreak+
  '  {'+sLineBreak+
  '    font-size: 26px;'+sLineBreak+
  '  }'+sLineBreak+
  '</style>'+sLineBreak+
  '  <div class="container">'+sLineBreak+
  '  <div class="section image">'+sLineBreak+
  '    <img src="#ADD_IMAGE_HERE#" width="256" height="256">'+sLineBreak+
  '  </div>'+sLineBreak+
  '  <div class="section text">'+sLineBreak+
  '  #ADD_TEXT_HERE#'+sLineBreak+
  '  <p>#ADD_TEXT2_HERE#' + sLineBreak +
  '  </div>'+sLineBreak+
  '  </div>'+sLineBreak+
  ' </body>'+sLineBreak+
  '</html>';

begin
  if aAuthorised then
    result:=RESPONSE_PAGE.Replace('#ADD_IMAGE_HERE#', GRANT_IMAGE)
            .Replace('#ADD_TEXT_HERE#', 'Authorisation Completed. '+
                        'You can close this page and return to the application')
            .Replace('#ADD_TEXT2_HERE#', 'Icons made by ' +
              '<a href="https://creativemarket.com/eucalyp" title="Eucalyp">' +
              'Eucalyp</a> from <a href="https://www.flaticon.com/" title="Flaticon">' +
              ' www.flaticon.com</a>')
  else
    result:=RESPONSE_PAGE.Replace('#ADD_IMAGE_HERE#', DENY_IMAGE)
            .Replace('#ADD_TEXT_HERE#', 'Authorisation Completed. '+
                        'Authorisation was denied');
end;

function TOAuthGMail.CreateAuthorizationRequest: string;
var
  scopes: string;
begin
  inherited;
  scopes:='https://www.googleapis.com/auth/userinfo.profile+' +
          'https://www.googleapis.com/auth/userinfo.email';
  result:='https://accounts.google.com/o/oauth2/v2/auth'+
          '?scope=' + scopes
        + '&state=profile'
        + '&redirect_uri='+ EncodeURL(CallbackURL)
        + '&response_type=code'
        + '&client_id=' + fClientID
        + '&approval_prompt=force'
        + '&access_type=offline'
        + '&hl=en';
  result:='GET '+ result;
end;

function TOAuthGMail.CreateAuthToAccessRequest(
  const aAuthToken: string): string;
begin
  inherited;
  result:='https://oauth2.googleapis.com/token' +
          '?client_id=' + fClientID +
          '&client_secret=' + fSecretID +
          '&redirect_uri=' + EncodeURL(CallbackURL) +
          '&code=' + aAuthToken +
          '&grant_type=authorization_code';
  result:='POST ' + result;
end;

function TOAuthGMail.CreateRefreshRequest(const aRefreshToken: string): string;
begin
  inherited;
  result:='https://oauth2.googleapis.com/token' +
          '?client_id=' + fClientID +
          '&client_secret=' + fSecretID +
          '&refresh_token=' + aRefreshToken +
          '&grant_type=refresh_token';
  result:='POST ' + result;
end;

end.



