/**
 * Copyright (c) 2016 DeepCortex GmbH <legal@eventql.io>
 * Authors:
 *   - Paul Asmuth <paul@eventql.io>
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License ("the license") as
 * published by the Free Software Foundation, either version 3 of the License,
 * or any later version.
 *
 * In accordance with Section 7(e) of the license, the licensing of the Program
 * under the license does not imply a trademark license. Therefore any rights,
 * title and interest in our trademarks remain entirely with us.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the license for more details.
 *
 * You can be released from the requirements of the license by purchasing a
 * commercial license. Buying such a license is mandatory as soon as you develop
 * commercial activities involving this program without disclosing the source
 * code of your own applications
 */
#ifndef _STX_REFLECT_METHODCALL_H
#define _STX_REFLECT_METHODCALL_H
#include <functional>
#include <memory>
#include <string>
#include <vector>
#include "eventql/util/reflect/indexsequence.h"

namespace reflect {

template <typename _ClassType, typename _ReturnType, typename... ArgTypes>
class MethodCall {
public:
  typedef std::tuple<ArgTypes...> ArgPackType;
  typedef _ClassType ClassType;
  typedef _ReturnType ReturnType;

  template <typename... ArgNameTypes>
  MethodCall(
    const std::string& name,
    ReturnType (ClassType::* fn)(ArgTypes...),
    ArgNameTypes... names);

  ReturnType call(ClassType* klass, ArgTypes... args) const;
  ReturnType call(ClassType* klass, const ArgPackType& args) const;

  template <class ArgListType>
  ReturnType call(ClassType* klass, const ArgListType& args) const;

  const std::string& name() const;

protected:

  template <int... I>
  ReturnType call(
      ClassType* klass,
      const ArgPackType& args,
      IndexSequence<I...>) const;

  template <class ArgListType, int... I>
  ReturnType call(
      ClassType* klass,
      const ArgListType& args,
      IndexSequence<I...>) const;

  std::string name_;
  ReturnType (ClassType::* fn_)(ArgTypes...);
  std::vector<std::string> arg_names_;
};

template <typename TargetType>
class MethodCallProxy {
public:
  MethodCallProxy(TargetType* target);

  template <typename MethodType, typename... ArgNameTypes>
  void method(
      const std::string& method_name,
      MethodType method_call,
      ArgNameTypes... arg_names);

  template <typename MethodType, typename... ArgNameTypes>
  void rpc(
      const std::string& method_name,
      MethodType method_call,
      ArgNameTypes... arg_names);

protected:
  TargetType* target_;
};

template <typename ClassType, typename ReturnType, typename... ArgTypes>
class MethodCallLookup {
public:
  MethodCallLookup(ReturnType (ClassType::* subject)(ArgTypes...));

  template <typename T1, typename... ArgNameTypes>
  void tryMethod(
      T1 m1,
      T1 m2,
      const std::string& method_name,
      ArgNameTypes... arg_names);

  template <typename T1, typename T2, typename... ArgNameTypes>
  void tryMethod(
      T1 m1,
      T2 m2,
      const std::string& method_name,
      ArgNameTypes... arg_names);

  template <typename MethodType, typename... ArgNameTypes>
  void method(
      const std::string& method_name,
      MethodType method_call,
      ArgNameTypes... arg_names);

  template <typename MethodType, typename... ArgNameTypes>
  void rpc(
      const std::string& method_name,
      MethodType method_call,
      ArgNameTypes... arg_names);

  MethodCall<ClassType, ReturnType, ArgTypes...> get() const;

protected:
  std::unique_ptr<MethodCall<ClassType, ReturnType, ArgTypes...>> method_call_;
  ReturnType (ClassType::* subject_)(ArgTypes...);
};

template <typename ClassType, typename ReturnType, typename... ArgTypes>
MethodCall<ClassType, ReturnType, ArgTypes...> reflectMethodImpl(
    ReturnType (ClassType::* method)(ArgTypes...));

template <typename MethodType>
auto reflectMethod(MethodType method) -> decltype(reflectMethodImpl(method))
    const*;

template <typename MethodCallType>
class RPCCall {
public:
  typedef typename std::tuple_element<0,
      typename std::remove_pointer<
          MethodCallType>::type::ArgPackType>::type RPCType;

  typedef typename std::remove_pointer<RPCType>::type::ArgPackType
      RPCArgPackType;

  typedef typename std::remove_pointer<RPCType>::type::ResultType
      RPCReturnType;

  typedef
      typename std::remove_pointer<MethodCallType>::type::ClassType
      RPCServiceType;

  template <typename... ArgNameTypes>
  RPCCall(MethodCallType method, ArgNameTypes... arg_names);

  MethodCallType method() const;

  template <typename ArgListType>
  RPCArgPackType getArgs(const ArgListType& args) const;

  template <typename ArgListType, int... I>
  RPCArgPackType getArgs(
      const ArgListType& args,
      IndexSequence<I...>) const;

protected:
  MethodCallType method_;
  std::vector<std::string> arg_names_;
};

}

#include "methodcall_impl.h"
#endif
