%%%===================================================================
%%% @copyright (C) 2012, Erlang Solutions Ltd.
%%% @doc Suite for testing s2s connection
%%% @end
%%%===================================================================

-module(s2s_SUITE).
-compile([export_all, nowarn_export_all]).

-include_lib("escalus/include/escalus.hrl").
-include_lib("exml/include/exml.hrl").
-include_lib("exml/include/exml_stream.hrl").
-include_lib("eunit/include/eunit.hrl").
-include_lib("kernel/include/inet.hrl").

%% Module aliases
-define(dh, distributed_helper).
-import(distributed_helper, [mim/0, rpc_spec/1, rpc/4]).

%%%===================================================================
%%% Suite configuration
%%%===================================================================

all() ->
    [
     {group, both_plain},
     {group, both_tls_optional}, %% default MongooseIM config
     {group, both_tls_required},

     {group, node1_tls_optional_node2_tls_required},
     {group, node1_tls_required_node2_tls_optional},

     {group, node1_tls_required_trusted_node2_tls_optional},
     {group, node1_tls_optional_node2_tls_required_trusted_with_cachain},

     {group, node1_tls_false_node2_tls_optional},
     {group, node1_tls_optional_node2_tls_false},

     {group, node1_tls_false_node2_tls_required},
     {group, node1_tls_required_node2_tls_false},

     {group, dialback}
    ].

groups() ->
    [{both_plain, [sequence], all_tests()},
     {both_tls_optional, [], essentials()},
     {both_tls_required, [], essentials()},

     {node1_tls_optional_node2_tls_required, [], essentials()},
     {node1_tls_required_node2_tls_optional, [], essentials()},

     %% Node1 closes connection from nodes with invalid certs
     {node1_tls_required_trusted_node2_tls_optional, [], negative()},

     %% Node1 accepts connection provided the cert can be verified
     {node1_tls_optional_node2_tls_required_trusted_with_cachain, [parallel],
      essentials() ++ connection_cases()},

     {node1_tls_false_node2_tls_optional, [], essentials()},
     {node1_tls_optional_node2_tls_false, [], essentials()},

     {node1_tls_false_node2_tls_required, [], negative()},
     {node1_tls_required_node2_tls_false, [], negative()},
     {dialback, [], [dialback_key_is_synchronized_on_different_nodes]}].

essentials() ->
    [simple_message].

all_tests() ->
    [connections_info, dns_discovery, dns_discovery_ip_fail, nonexistent_user,
     unknown_domain, malformed_jid, dialback_with_wrong_key].

negative() ->
    [timeout_waiting_for_message].

connection_cases() ->
    [successful_external_auth_with_valid_cert,
     start_stream_fails_for_wrong_namespace,
     start_stream_fails_for_wrong_version,
     start_stream_fails_without_version,
     start_stream_fails_without_host,
     start_stream_fails_for_unknown_host,
     starttls_fails_for_unknown_host,
     only_messages_from_authenticated_domain_users_are_accepted,
     auth_with_valid_cert_fails_when_requested_name_is_not_in_the_cert,
     auth_with_valid_cert_fails_for_other_mechanism_than_external].

suite() ->
    distributed_helper:require_rpc_nodes([mim, mim2, fed]) ++ escalus:suite().

users() ->
    [alice2, alice, bob].

%%%===================================================================
%%% Init & teardown
%%%===================================================================

init_per_suite(Config) ->
    instrument_helper:start(tested_events()),
    mongoose_helper:inject_module(?MODULE, reload),
    Config1 = s2s_helper:init_s2s(escalus:init_per_suite(Config)),
    escalus:create_users(Config1, escalus:get_users(users())).

end_per_suite(Config) ->
    escalus_fresh:clean(),
    s2s_helper:end_s2s(Config),
    escalus:delete_users(Config, escalus:get_users(users())),
    escalus:end_per_suite(Config),
    instrument_helper:stop().

init_per_group(dialback, Config) ->
    %% Tell mnesia that mim and mim2 nodes are clustered
    distributed_helper:add_node_to_cluster(distributed_helper:mim2(), Config);
init_per_group(GroupName, Config) ->
    Config1 = s2s_helper:configure_s2s(GroupName, Config),
    [{requires_tls, group_with_tls(GroupName)}, {group, GroupName} | Config1].

end_per_group(_GroupName, _Config) ->
    ok.

init_per_testcase(dns_discovery = CaseName, Config) ->
    meck_inet_res("_xmpp-server._tcp.fed2"),
    ok = rpc(mim(), meck, new, [inet, [no_link, unstick, passthrough]]),
    ok = rpc(mim(), meck, expect,
             [inet, getaddr,
                fun ("fed2", inet) ->
                        {ok, {127, 0, 0, 1}};
                    (Address, Family) ->
                        meck:passthrough([Address, Family])
                end]),
    Config1 = escalus_users:update_userspec(Config, alice2, server, <<"fed2">>),
    escalus:init_per_testcase(CaseName, Config1);
init_per_testcase(dns_discovery_ip_fail = CaseName, Config) ->
    meck_inet_res("_xmpp-server._tcp.fed3"),
    ok = rpc(mim(), meck, new, [inet, [no_link, unstick, passthrough]]),
    escalus:init_per_testcase(CaseName, Config);
init_per_testcase(CaseName, Config) ->
    escalus:init_per_testcase(CaseName, Config).

meck_inet_res(Domain) ->
    ok = rpc(mim(), meck, new, [inet_res, [no_link, unstick, passthrough]]),
    ok = rpc(mim(), meck, expect,
             [inet_res, getbyname,
                fun (Domain1, srv, _Timeout) when Domain1 == Domain ->
                        {ok, {hostent, Domain, [], srv, 1,
                         [{30, 10, 5299, "localhost"}]}};
                    (Name, Type, Timeout) ->
                        meck:passthrough([Name, Type, Timeout])
                end]).

end_per_testcase(CaseName, Config) when CaseName =:= dns_discovery;
                                        CaseName =:= dns_discovery_ip_fail ->
    rpc(mim(), meck, unload, []),
    escalus:end_per_testcase(CaseName, Config);
end_per_testcase(CaseName, Config) ->
    escalus:end_per_testcase(CaseName, Config).

%%%===================================================================
%%% Server-to-server communication test
%%%===================================================================

simple_message(Config) ->
    escalus:fresh_story(Config, [{alice2, 1}, {alice, 1}], fun(Alice2, Alice1) ->
        TS = instrument_helper:timestamp(),

        %% User on the main server sends a message to a user on a federated server
        escalus:send(Alice1, escalus_stanza:chat_to(Alice2, <<"Hi, foreign Alice!">>)),

        %% User on the federated server receives the message
        Stanza = escalus:wait_for_stanza(Alice2, 10000),
        escalus:assert(is_chat_message, [<<"Hi, foreign Alice!">>], Stanza),

        %% User on the federated server sends a message to the main server
        escalus:send(Alice2, escalus_stanza:chat_to(Alice1, <<"Nice to meet you!">>)),

        %% User on the main server receives the message
        Stanza2 = escalus:wait_for_stanza(Alice1, 10000),
        escalus:assert(is_chat_message, [<<"Nice to meet you!">>], Stanza2),

        % Instrumentation events are executed
        assert_events(TS, Config)
    end).

timeout_waiting_for_message(Config) ->
    try
        simple_message(Config),
        ct:fail("got message but shouldn't")
    catch
        error:timeout_when_waiting_for_stanza ->
            ok
    end.

connections_info(Config) ->
    simple_message(Config),
    FedDomain = ct:get_config({hosts, fed, domain}),
    %% there should be at least one in and at least one out connection
    [_ | _] = get_s2s_connections(?dh:mim(), FedDomain, in),
    [_ | _] = get_s2s_connections(?dh:mim(), FedDomain, out),
    ok.

dns_discovery(Config) ->
    simple_message(Config),
    %% Ensure that the mocked DNS discovery for connecting to the other server
    History = rpc(mim(), meck, history, [inet_res]),
    ?assertEqual(length(History), 2),
    ?assertEqual(s2s_helper:has_xmpp_server(History, "fed2"), true),
    ok.


dns_discovery_ip_fail(Config) ->
    escalus:fresh_story(Config, [{alice, 1}], fun(Alice1) ->

        escalus:send(Alice1, escalus_stanza:chat_to(
            <<"alice2@fed3">>,
            <<"Hello, second Alice!">>)),

        Stanza = escalus:wait_for_stanza(Alice1, 10000),
        escalus:assert(is_error, [<<"cancel">>, <<"remote-server-not-found">>], Stanza),
        History = rpc(mim(), meck, history, [inet]),
        ?assertEqual(s2s_helper:has_inet_errors(History, "fed3"), true)
    end).

get_s2s_connections(RPCSpec, Domain, Type) ->
    AllS2SConnections = ?dh:rpc(RPCSpec, mongoose_s2s_info, get_connections, [Type]),
    DomainS2SConnections =
        [Connection || Connection <- AllS2SConnections,
                       Type =/= in orelse [Domain] =:= maps:get(domains, Connection),
                       Type =/= out orelse Domain =:= maps:get(server, Connection)],
    ct:pal("Node = ~p,  ConnectionType = ~p, Domain = ~s~nDomainS2SConnections(~p): ~p",
           [maps:get(node, RPCSpec), Type, Domain, length(DomainS2SConnections),
            DomainS2SConnections]),
    DomainS2SConnections.

nonexistent_user(Config) ->
    escalus:fresh_story(Config, [{alice, 1}, {alice2, 1}], fun(Alice1, Alice2) ->

        %% Alice@localhost1 sends message to Xyz@localhost2
        RemoteServer = escalus_client:server(Alice2),
        Fake = <<"xyz@", RemoteServer/binary>>,
        escalus:send(Alice1, escalus_stanza:chat_to(Fake,
                                                    <<"Hello, nonexistent!">>)),

        %% Alice@localhost1 receives stanza error: service-unavailable
        Stanza = escalus:wait_for_stanza(Alice1),
        escalus:assert(is_error, [<<"cancel">>, <<"service-unavailable">>], Stanza)

    end).

unknown_domain(Config) ->
    escalus:fresh_story(Config, [{alice, 1}], fun(Alice1) ->

        %% Alice@localhost1 sends message to Xyz@localhost3
        escalus:send(Alice1, escalus_stanza:chat_to(
            <<"xyz@somebogushost">>,
            <<"Hello, unreachable!">>)),

        %% Alice@localhost1 receives stanza error: remote-server-not-found
        Stanza = escalus:wait_for_stanza(Alice1, 10000),
        escalus:assert(is_error, [<<"cancel">>, <<"remote-server-not-found">>], Stanza)

    end).

malformed_jid(Config) ->
    escalus:fresh_story(Config, [{alice, 1}], fun(Alice1) ->

        %% Alice@localhost1 sends message to Xyz@localhost3
        escalus:send(Alice1, escalus_stanza:chat_to(
            <<"not a jid">>,
            <<"Hello, unreachable!">>)),

        %% Alice@localhost1 receives stanza error: remote-server-not-found
        Stanza = escalus:wait_for_stanza(Alice1, 10000),
        escalus:assert(is_error, [<<"cancel">>, <<"remote-server-not-found">>], Stanza)

    end).

dialback_with_wrong_key(_Config) ->
    HostType = domain_helper:host_type(mim),
    MimDomain = domain_helper:domain(mim),
    FedDomain = domain_helper:domain(fed),
    FromTo = {MimDomain, FedDomain},
    Key = <<"123456">>, %% wrong key
    StreamId = <<"sdfdsferrr">>,
    StartType = {verify, self(), Key, StreamId},
    {ok, _} = rpc(rpc_spec(mim), ejabberd_s2s_out, start, [FromTo, StartType]),
    receive
        %% Remote server (fed1) rejected out request
        {'$gen_event', {validity_from_s2s_out, false, FromTo}} ->
            ok
        after 5000 ->
            ct:fail(timeout)
    end.

nonascii_addr(Config) ->
    escalus:fresh_story(Config, [{alice, 1}, {bob2, 1}], fun(Alice, Bob) ->

        %% Bob@localhost2 sends message to Alice@localhost1
        escalus:send(Bob, escalus_stanza:chat_to(Alice, <<"Cześć Alice!">>)),

        %% Alice@localhost1 receives message from Bob@localhost2
        Stanza = escalus:wait_for_stanza(Alice, 10000),
        escalus:assert(is_chat_message, [<<"Cześć Alice!">>], Stanza),

        %% Alice@localhost1 sends message to Bob@localhost2
        escalus:send(Alice, escalus_stanza:chat_to(Bob, <<"Miło Cię poznać">>)),

        %% Bob@localhost2 receives message from Alice@localhost1
        Stanza2 = escalus:wait_for_stanza(Bob, 10000),
        escalus:assert(is_chat_message, [<<"Miło Cię poznać">>], Stanza2)

    end).

successful_external_auth_with_valid_cert(Config) ->
    ConnectionArgs = connection_args("localhost.bis", <<"localhost">>, Config),
    {ok, Client, _Features} = escalus_connection:start(ConnectionArgs,
                                                       [fun s2s_start_stream/2,
                                                        fun s2s_starttls/2,
                                                        fun s2s_external_auth/2]),
    escalus_connection:stop(Client).

start_stream_fails_for_wrong_namespace(Config) ->
    start_stream_fails(Config, <<"invalid-namespace">>,
                       [fun s2s_start_stream_with_wrong_namespace/2]).

start_stream_fails_for_wrong_version(Config) ->
    %% TLS authentication requires version 1.0
    start_stream_fails(Config, <<"invalid-xml">>,
                       [fun s2s_start_stream_with_wrong_version/2]).

start_stream_fails_without_version(Config) ->
    %% TLS authentication requires version 1.0
    start_stream_fails(Config, <<"invalid-xml">>,
                       [fun s2s_start_stream_without_version/2]).

start_stream_fails_without_host(Config) ->
    start_stream_fails(Config, <<"improper-addressing">>,
                       [fun s2s_start_stream_without_host/2]).

start_stream_fails_for_unknown_host(Config) ->
    start_stream_fails(Config, <<"host-unknown">>,
                       [fun s2s_start_stream_to_wrong_host/2]).

starttls_fails_for_unknown_host(Config) ->
    start_stream_fails(Config, <<"host-unknown">>,
                       [fun s2s_start_stream/2,
                        fun s2s_starttls_to_wrong_host/2]).

start_stream_fails(Config, ErrorType, ConnectionSteps) ->
    ConnectionArgs = connection_args("localhost.bis", <<"localhost">>, Config),
    {ok, Client, _} = escalus_connection:start(ConnectionArgs, ConnectionSteps),
    [Start, Error, End] = escalus:wait_for_stanzas(Client, 3),
    escalus:assert(is_stream_start, Start),
    escalus:assert(is_stream_error, [ErrorType, <<>>], Error),
    escalus:assert(is_stream_end, End).

only_messages_from_authenticated_domain_users_are_accepted(Config) ->
    ConnectionArgs = connection_args("localhost.bis", <<"localhost">>, Config),
    {ok, Client, _Features} = escalus_connection:start(ConnectionArgs,
                                                       [fun s2s_start_stream/2,
                                                        fun s2s_starttls/2,
                                                        fun s2s_external_auth/2]),
    escalus:fresh_story(Config, [{alice2, 1}], fun(Alice) ->

        UserInWrongDomain = <<"a_user@this_is_not_my.domain.com">>,
        ChatToAliceFromUserInWrongDomain = escalus_stanza:chat(UserInWrongDomain,
                                                               Alice, <<"Miło Cię poznać">>),
        %% Client is a s2s connection established and authenticated for domain "localhost"
        %% Now we try to send a message from other domain than "localhost"
        %% over the established s2s connection
        escalus:send(Client, ChatToAliceFromUserInWrongDomain),

        %% Alice@fed1 does not receives message from a_user@this_is_not_my.domain.com
        timer:sleep(timer:seconds(5)),
        escalus_assert:has_no_stanzas(Alice)

    end),

    escalus_connection:stop(Client).

auth_with_valid_cert_fails_when_requested_name_is_not_in_the_cert(Config) ->
    ConnectionArgs = connection_args("not_in_cert_domain", <<"not_in_cert_domain">>, Config),
    {ok, Client, _Features} = escalus_connection:start(ConnectionArgs,
                                                       [fun s2s_start_stream/2,
                                                        fun s2s_starttls/2]),

    try
        escalus_auth:auth_sasl_external(Client, Client#client.props),
        ct:fail("Authenitcated but MUST NOT")
    catch throw:{auth_failed, _, _} ->
              escalus_connection:wait_for_close(Client, timer:seconds(5))
    end.

auth_with_valid_cert_fails_for_other_mechanism_than_external(Config) ->
    ConnectionArgs = connection_args("localhost", <<"localhost">>, Config),
    {ok, Client, _Features} = escalus_connection:start(ConnectionArgs,
                                                       [fun s2s_start_stream/2,
                                                        fun s2s_starttls/2
                                                       ]),

    Stanza = escalus_stanza:auth(<<"ANONYMOUS">>),
    ok = escalus_connection:send(Client, Stanza),
    #xmlel{name = <<"failure">>} = escalus_connection:get_stanza(Client, wait_for_auth_reply),

    escalus_connection:wait_for_close(Client, timer:seconds(5)).

connection_args(FromServer, RequestedName, Config) ->
    {KeyFile, CertFile} = get_main_key_and_cert_files(Config),
    [{host, "localhost"},
     {to_server, "fed1"},
     {from_server, FromServer},
     {requested_name, RequestedName},
     {starttls, required},
     {port, ct:get_config({hosts, fed, incoming_s2s_port})},
     {ssl_opts, [{versions, ['tlsv1.2']}, {verify, verify_none}, {certfile, CertFile}, {keyfile, KeyFile}]}].

s2s_start_stream_with_wrong_namespace(Conn = #client{props = Props}, Features) ->
    Start = s2s_stream_start_stanza(Props, fun(Attrs) -> Attrs#{<<"xmlns">> => <<"42">>} end),
    ok = escalus_connection:send(Conn, Start),
    {Conn, Features}.

s2s_start_stream_with_wrong_version(Conn = #client{props = Props}, Features) ->
    Start = s2s_stream_start_stanza(Props, fun(Attrs) -> Attrs#{<<"version">> => <<"42">>} end),
    ok = escalus_connection:send(Conn, Start),
    {Conn, Features}.

s2s_start_stream_without_version(Conn = #client{props = Props}, Features) ->
    Start = s2s_stream_start_stanza(Props, fun(Attrs) -> maps:remove(<<"version">>, Attrs) end),
    ok = escalus_connection:send(Conn, Start),
    {Conn, Features}.

s2s_start_stream_without_host(Conn = #client{props = Props}, Features) ->
    Start = s2s_stream_start_stanza(Props, fun(Attrs) -> maps:remove(<<"to">>, Attrs) end),
    ok = escalus_connection:send(Conn, Start),
    {Conn, Features}.

s2s_start_stream_to_wrong_host(Conn = #client{props = Props}, Features) ->
    Start = s2s_stream_start_stanza(Props, fun(Attrs) -> Attrs#{<<"to">> => <<"42">>} end),
    ok = escalus_connection:send(Conn, Start),
    {Conn, Features}.

s2s_start_stream(Conn = #client{props = Props}, []) ->
    StreamStartRep = s2s_start_stream_and_wait_for_response(Conn),

    #xmlstreamstart{attrs = Attrs} = StreamStartRep,
    Id = proplists:get_value(<<"id">>, Attrs),

    escalus_session:stream_features(Conn#client{props = [{sid, Id} | Props]}, []).

s2s_start_stream_and_wait_for_response(Conn = #client{props = Props}) ->
    StreamStart = s2s_stream_start_stanza(Props, fun(Attrs) -> Attrs end),
    ok = escalus_connection:send(Conn, StreamStart),
    escalus_connection:get_stanza(Conn, wait_for_stream).

s2s_stream_start_stanza(Props, F) ->
    Attrs = (stream_start_attrs())#{<<"to">> => proplists:get_value(to_server, Props),
                                    <<"from">> => proplists:get_value(from_server, Props)},
    #xmlstreamstart{name = <<"stream:stream">>, attrs = maps:to_list(F(Attrs))}.

stream_start_attrs() ->
    #{<<"xmlns">> => <<"jabber:server">>,
      <<"xmlns:stream">> => <<"http://etherx.jabber.org/streams">>,
      <<"version">> => <<"1.0">>}.

s2s_starttls(Client, Features, StartStreamF) ->
    case proplists:get_value(starttls, Features) of
        false ->
            ct:fail("The server does not offer STARTTLS");
        _ ->
            ok
    end,

    escalus_connection:send(Client, escalus_stanza:starttls()),
    escalus_connection:get_stanza(Client, proceed),
    escalus_connection:upgrade_to_tls(Client),
    StartStreamF(Client, []).

s2s_starttls(Client, Features) ->
    s2s_starttls(Client, Features, fun s2s_start_stream/2).

s2s_starttls_to_wrong_host(Client, Features) ->
    s2s_starttls(Client, Features, fun s2s_start_stream_to_wrong_host/2).

s2s_external_auth(Client = #client{props = Props}, Features) ->
    case proplists:get_value(sasl_mechanisms, Features) of
        [<<"EXTERNAL">>] ->
            ok;
        SASL ->
            ct:fail("Server does not provide EXTERNAL auth: ~p", [SASL])
    end,
    escalus_auth:auth_sasl_external(Client, Props),
    s2s_start_stream(Client, []).

get_main_key_and_cert_files(Config) ->
    CertFile = get_main_file_path(Config, "cert.pem"),
    KeyFile = get_main_file_path(Config, "key.pem"),
    {KeyFile, CertFile}.

get_main_file_path(Config, File) ->
    filename:join([path_helper:repo_dir(Config),
                   "tools", "ssl", "mongooseim", File]).

dialback_key_is_synchronized_on_different_nodes(_Config) ->
    configure_secret_and_restart_s2s(mim),
    configure_secret_and_restart_s2s(mim2),
    Key1 = get_shared_secret(mim),
    Key2 = get_shared_secret(mim2),
    ?assertEqual(Key1, Key2),
    %% Node 2 is restarted later, so both nodes should have the key.
    ?assertEqual(Key2, {ok, <<"9e438f25e81cf347100b">>}).

get_shared_secret(NodeKey) ->
    HostType = domain_helper:host_type(mim),
    rpc(rpc_spec(NodeKey), mongoose_s2s_backend, get_shared_secret, [HostType]).

set_opt(Spec, Opt, Value) ->
    rpc(Spec, mongoose_config, set_opt, [Opt, Value]).

configure_secret_and_restart_s2s(NodeKey) ->
    HostType = domain_helper:host_type(mim),
    Spec = rpc_spec(NodeKey),
    set_opt(Spec, [{s2s, HostType}, shared], shared_secret(NodeKey)),
    ok = rpc(Spec, supervisor, terminate_child, [ejabberd_sup, ejabberd_s2s]),
    {ok, _} = rpc(Spec, supervisor, restart_child, [ejabberd_sup, ejabberd_s2s]).

shared_secret(mim) -> <<"f623e54a0741269be7dd">>; %% Some random key
shared_secret(mim2) -> <<"9e438f25e81cf347100b">>.

assert_events(TS, Config) ->
    TLS = proplists:get_value(requires_tls, Config),
    instrument_helper:assert(s2s_xmpp_element_size_in, #{}, fun(#{byte_size := S}) -> S > 0 end,
        #{expected_count => element_count(in, TLS), min_timestamp => TS}),
    instrument_helper:assert(s2s_xmpp_element_size_out, #{}, fun(#{byte_size := S}) -> S > 0 end,
        #{expected_count => element_count(out, TLS), min_timestamp => TS}),
    instrument_helper:assert(s2s_tcp_data_in, #{}, fun(#{byte_size := S}) -> S > 0 end,
        #{min_timestamp => TS}),
    instrument_helper:assert(s2s_tcp_data_out, #{}, fun(#{byte_size := S}) -> S > 0 end,
        #{min_timestamp => TS}).

element_count(_Dir, true) ->
    % TLS tests are not checking a specific number of events, because the numbers are flaky
    positive;
element_count(in, false) ->
    % Some of these steps happen asynchronously, so the order may be different.
    % Since S2S connections are unidirectional, mim1 acts both as initiating,
    % and receiving (and authoritative) server in the dialback procedure.
    %   1. Stream start response from fed1 (as initiating server)
    %   2. Stream start from fed1 (as receiving server)
    %   3. Dialback key (step 1, as receiving server)
    %   4. Dialback verification request (step 2, as authoritative server)
    %   5. Dialback result (step 4, as initiating server)
    % New s2s process is started to verify fed1 as an authoritative server
    % This process sends a new stream header, as it opens a new connection to fed1,
    % now acting as an authoritative server. Also see comment on L360 in ejabberd_s2s.
    %   6. Stream start response from fed1
    %   7. Dialback verification response (step 3, as receiving server)
    %   8. Message from federated Alice
    % The number can be seen as the sum of all arrows from the dialback diagram, since mim
    % acts as all three roles in the two dialback procedures that occur:
    % https://xmpp.org/extensions/xep-0220.html#intro-howitworks
    % (6 arrows) + one for stream header response + one for the actual message
    8;
element_count(out, false) ->
    % Since S2S connections are unidirectional, mim1 acts both as initiating,
    % and receiving (and authoritative) server in the dialback procedure.
    %   1. Dialback key (step 1, as initiating server)
    %   2. Dialback verification response (step 3, as authoritative server)
    %   3. Dialback request (step 2, as receiving server)
    %   4. Message from Alice
    %   5. Dialback result (step 4, as receiving server)
    % The number calculation corresponds to in_element, however the stream headers are not
    % sent as XML elements, but straight as text, and so these three events do not appear:
    %  - open stream to fed1,
    %  - stream response for fed1->mim stream,
    %  - open stream to fed1 as authoritative server.
    5.

group_with_tls(both_tls_optional) -> true;
group_with_tls(both_tls_required) -> true;
group_with_tls(node1_tls_optional_node2_tls_required) -> true;
group_with_tls(node1_tls_required_node2_tls_optional) -> true;
group_with_tls(node1_tls_required_trusted_node2_tls_optional) -> true;
group_with_tls(node1_tls_optional_node2_tls_required_trusted_with_cachain) -> true;
group_with_tls(_GN) -> false.

tested_events() ->
    Names = [s2s_xmpp_element_size_in, s2s_xmpp_element_size_out,
             s2s_tcp_data_in, s2s_tcp_data_out],
    [{Name, #{}} || Name <- Names].
