import { get, PROPERTY_DID_CHANGE } from '@ember/-internals/metal';
import type { PropertyDidChange } from '@ember/-internals/metal/lib/property_events';
import { getOwner } from '@ember/-internals/owner';
import { TargetActionSupport } from '@ember/-internals/runtime';
import {
  ActionSupport,
  ChildViewsSupport,
  ClassNamesSupport,
  CoreView,
  EventDispatcher,
  getViewElement,
  ViewMixin,
  ViewStateSupport,
} from '@ember/-internals/views';
import { assert } from '@ember/debug';
import { DEBUG } from '@glimmer/env';
import type { Environment, Template, TemplateFactory } from '@glimmer/interfaces';
import { setInternalComponentManager } from '@glimmer/manager';
import { isUpdatableRef, updateRef } from '@glimmer/reference';
import { normalizeProperty } from '@glimmer/runtime';
import type { DirtyableTag } from '@glimmer/validator';
import { createTag, dirtyTag } from '@glimmer/validator';
import type { SimpleElement } from '@simple-dom/interface';
import {
  ARGS,
  BOUNDS,
  CURLY_COMPONENT_MANAGER,
  DIRTY_TAG,
  IS_DISPATCHING_ATTRS,
} from './component-managers/curly';

// Keep track of which component classes have already been processed for lazy event setup.
let lazyEventsProcessed = new WeakMap<EventDispatcher, WeakSet<object>>();

/**
@module @ember/component
*/

interface ComponentMethods {
  // Overrideable methods are defined here since you can't `declare` a method in a class

  /**
   Called when the attributes passed into the component have been updated.
    Called both during the initial render of a container and during a rerender.
    Can be used in place of an observer; code placed here will be executed
    every time any attribute updates.
    @method didReceiveAttrs
    @public
    @since 1.13.0
    */
  didReceiveAttrs(): void;

  /**
   Called when the attributes passed into the component have been updated.
    Called both during the initial render of a container and during a rerender.
    Can be used in place of an observer; code placed here will be executed
    every time any attribute updates.
    @event didReceiveAttrs
    @public
    @since 1.13.0
    */

  /**
   Called after a component has been rendered, both on initial render and
    in subsequent rerenders.
    @method didRender
    @public
    @since 1.13.0
    */
  didRender(): void;

  /**
   Called after a component has been rendered, both on initial render and
    in subsequent rerenders.
    @event didRender
    @public
    @since 1.13.0
    */

  /**
   Called before a component has been rendered, both on initial render and
    in subsequent rerenders.
    @method willRender
    @public
    @since 1.13.0
    */
  willRender(): void;

  /**
   Called before a component has been rendered, both on initial render and
    in subsequent rerenders.
    @event willRender
    @public
    @since 1.13.0
    */

  /**
   Called when the attributes passed into the component have been changed.
    Called only during a rerender, not during an initial render.
    @method didUpdateAttrs
    @public
    @since 1.13.0
    */
  didUpdateAttrs(): void;

  /**
   Called when the attributes passed into the component have been changed.
    Called only during a rerender, not during an initial render.
    @event didUpdateAttrs
    @public
    @since 1.13.0
    */

  /**
   Called when the component is about to update and rerender itself.
    Called only during a rerender, not during an initial render.
    @method willUpdate
    @public
    @since 1.13.0
    */
  willUpdate(): void;

  /**
   Called when the component is about to update and rerender itself.
    Called only during a rerender, not during an initial render.
    @event willUpdate
    @public
    @since 1.13.0
    */

  /**
   Called when the component has updated and rerendered itself.
    Called only during a rerender, not during an initial render.
    @method didUpdate
    @public
    @since 1.13.0
    */
  didUpdate(): void;

  /**
   Called when the component has updated and rerendered itself.
    Called only during a rerender, not during an initial render.
    @event didUpdate
    @public
    @since 1.13.0
    */

  /**
    The HTML `id` of the component's element in the DOM. You can provide this
    value yourself but it must be unique (just as in HTML):

    ```handlebars
    {{my-component elementId="a-really-cool-id"}}
    ```

    ```handlebars
    <MyComponent @elementId="a-really-cool-id" />
    ```
    If not manually set a default value will be provided by the framework.
    Once rendered an element's `elementId` is considered immutable and you
    should never change it. If you need to compute a dynamic value for the
    `elementId`, you should do this when the component or element is being
    instantiated:

    ```javascript
    export default Component.extend({
      init() {
        this._super(...arguments);

        var index = this.get('index');
        this.set('elementId', `component-id${index}`);
      }
    });
    ```

    @property elementId
    @type String
    @public
  */
  layoutName?: string;
}

// A zero-runtime-overhead private symbol to use in branding the component to
// preserve its type parameter.
declare const SIGNATURE: unique symbol;

/**
  A component is a reusable UI element that consists of a `.hbs` template and an
  optional JavaScript class that defines its behavior. For example, someone
  might make a `button` in the template and handle the click behavior in the
  JavaScript file that shares the same name as the template.

  Components are broken down into two categories:

  - Components _without_ JavaScript, that are based only on a template. These
    are called Template-only or TO components.
  - Components _with_ JavaScript, which consist of a template and a backing
    class.

  Ember ships with two types of JavaScript classes for components:

  1. Glimmer components, imported from `@glimmer/component`, which are the
     default component's for Ember Octane (3.15) and more recent editions.
  2. Classic components, imported from `@ember/component`, which were the
     default for older editions of Ember (pre 3.15).

  Below is the documentation for Classic components. If you are looking for the
  API documentation for Template-only or Glimmer components, it is [available
  here](/ember/release/modules/@glimmer%2Fcomponent).

  ## Defining a Classic Component

  If you want to customize the component in order to handle events, transform
  arguments or maintain internal state, you implement a subclass of `Component`.

  One example is to add computed properties to your component:

  ```app/components/person-profile.js
  import Component from '@ember/component';

  export default Component.extend({
    displayName: computed('person.title', 'person.firstName', 'person.lastName', function() {
      let { title, firstName, lastName } = this.person;

      if (title) {
        return `${title} ${lastName}`;
      } else {
        return `${firstName} ${lastName}`;
      }
    })
  });
  ```

  And then use it in the component's template:

  ```app/templates/components/person-profile.hbs
  <h1>{{this.displayName}}</h1>
  {{yield}}
  ```

  ## Customizing a Classic Component's HTML Element in JavaScript

  ### HTML Tag

  The default HTML tag name used for a component's HTML representation is `div`.
  This can be customized by setting the `tagName` property.

  Consider the following component class:

  ```app/components/emphasized-paragraph.js
  import Component from '@ember/component';

  export default Component.extend({
    tagName: 'em'
  });
  ```

  When invoked, this component would produce output that looks something like
  this:

  ```html
  <em id="ember1" class="ember-view"></em>
  ```

  ### HTML `class` Attribute

  The HTML `class` attribute of a component's tag can be set by providing a
  `classNames` property that is set to an array of strings:

  ```app/components/my-widget.js
  import Component from '@ember/component';

  export default Component.extend({
    classNames: ['my-class', 'my-other-class']
  });
  ```

  Invoking this component will produce output that looks like this:

  ```html
  <div id="ember1" class="ember-view my-class my-other-class"></div>
  ```

  `class` attribute values can also be set by providing a `classNameBindings`
  property set to an array of properties names for the component. The return
  value of these properties will be added as part of the value for the
  components's `class` attribute. These properties can be computed properties:

  ```app/components/my-widget.js
  import Component from '@ember/component';
  import { computed } from '@ember/object';

  export default Component.extend({
    classNames: ['my-class', 'my-other-class'],
    classNameBindings: ['propertyA', 'propertyB'],

    propertyA: 'from-a',
    propertyB: computed(function() {
      if (someLogic) { return 'from-b'; }
    })
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <div id="ember1" class="ember-view my-class my-other-class from-a from-b"></div>
  ```

  Note that `classNames` and `classNameBindings` is in addition to the `class`
  attribute passed with the angle bracket invocation syntax. Therefore, if this
  component was invoked like so:

  ```handlebars
  <MyWidget class="from-invocation" />
  ```

  The resulting HTML will look similar to this:

  ```html
  <div id="ember1" class="from-invocation ember-view my-class my-other-class from-a from-b"></div>
  ```

  If the value of a class name binding returns a boolean the property name
  itself will be used as the class name if the property is true. The class name
  will not be added if the value is `false` or `undefined`.

  ```app/components/my-widget.js
  import Component from '@ember/component';

  export default Component.extend({
    classNameBindings: ['hovered'],

    hovered: true
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <div id="ember1" class="ember-view hovered"></div>
  ```

  ### Custom Class Names for Boolean Values

  When using boolean class name bindings you can supply a string value other
  than the property name for use as the `class` HTML attribute by appending the
  preferred value after a ":" character when defining the binding:

  ```app/components/my-widget.js
  import Component from '@ember/component';

  export default Component.extend({
    classNameBindings: ['awesome:so-very-cool'],

    awesome: true
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <div id="ember1" class="ember-view so-very-cool"></div>
  ```

  Boolean value class name bindings whose property names are in a
  camelCase-style format will be converted to a dasherized format:

  ```app/components/my-widget.js
  import Component from '@ember/component';

  export default Component.extend({
    classNameBindings: ['isUrgent'],

    isUrgent: true
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <div id="ember1" class="ember-view is-urgent"></div>
  ```

  Class name bindings can also refer to object values that are found by
  traversing a path relative to the component itself:

  ```app/components/my-widget.js
  import Component from '@ember/component';
  import EmberObject from '@ember/object';

  export default Component.extend({
    classNameBindings: ['messages.empty'],

    messages: EmberObject.create({
      empty: true
    })
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <div id="ember1" class="ember-view empty"></div>
  ```

  If you want to add a class name for a property which evaluates to true and and
  a different class name if it evaluates to false, you can pass a binding like
  this:

  ```app/components/my-widget.js
  import Component from '@ember/component';

  export default Component.extend({
    classNameBindings: ['isEnabled:enabled:disabled'],
    isEnabled: true
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <div id="ember1" class="ember-view enabled"></div>
  ```

  When isEnabled is `false`, the resulting HTML representation looks like this:

  ```html
  <div id="ember1" class="ember-view disabled"></div>
  ```

  This syntax offers the convenience to add a class if a property is `false`:

  ```app/components/my-widget.js
  import Component from '@ember/component';

  // Applies no class when isEnabled is true and class 'disabled' when isEnabled is false
  export default Component.extend({
    classNameBindings: ['isEnabled::disabled'],
    isEnabled: true
  });
  ```

  Invoking this component when the `isEnabled` property is true will produce
  HTML that looks like:

  ```html
  <div id="ember1" class="ember-view"></div>
  ```

  Invoking it when the `isEnabled` property on the component is `false` will
  produce HTML that looks like:

  ```html
  <div id="ember1" class="ember-view disabled"></div>
  ```

  Updates to the value of a class name binding will result in automatic update
  of the  HTML `class` attribute in the component's rendered HTML
  representation. If the value becomes `false` or `undefined` the class name
  will be removed.

  Both `classNames` and `classNameBindings` are concatenated properties. See
  [EmberObject](/ember/release/classes/EmberObject) documentation for more
  information about concatenated properties.

  ### Other HTML Attributes

  The HTML attribute section of a component's tag can be set by providing an
  `attributeBindings` property set to an array of property names on the
  component. The return value of these properties will be used as the value of
  the component's HTML associated attribute:

  ```app/components/my-anchor.js
  import Component from '@ember/component';

  export default Component.extend({
    tagName: 'a',
    attributeBindings: ['href'],

    href: 'http://google.com'
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <a id="ember1" class="ember-view" href="http://google.com"></a>
  ```

  One property can be mapped on to another by placing a ":" between the source
  property and the destination property:

  ```app/components/my-anchor.js
  import Component from '@ember/component';

  export default Component.extend({
    tagName: 'a',
    attributeBindings: ['url:href'],

    url: 'http://google.com'
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <a id="ember1" class="ember-view" href="http://google.com"></a>
  ```

  HTML attributes passed with angle bracket invocations will take precedence
  over those specified in `attributeBindings`. Therefore, if this component was
  invoked like so:

  ```handlebars
  <MyAnchor href="http://bing.com" @url="http://google.com" />
  ```

  The resulting HTML will looks like this:

  ```html
  <a id="ember1" class="ember-view" href="http://bing.com"></a>
  ```

  Note that the `href` attribute is ultimately set to `http://bing.com`, despite
  it having attribute binidng to the `url` property, which was set to
  `http://google.com`.

  Namespaced attributes (e.g. `xlink:href`) are supported, but have to be
  mapped, since `:` is not a valid character for properties in Javascript:

  ```app/components/my-use.js
  import Component from '@ember/component';

  export default Component.extend({
    tagName: 'use',
    attributeBindings: ['xlinkHref:xlink:href'],

    xlinkHref: '#triangle'
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <use xlink:href="#triangle"></use>
  ```

  If the value of a property monitored by `attributeBindings` is a boolean, the
  attribute will be present or absent depending on the value:

  ```app/components/my-text-input.js
  import Component from '@ember/component';

  export default Component.extend({
    tagName: 'input',
    attributeBindings: ['disabled'],

    disabled: false
  });
  ```

  Invoking this component will produce HTML that looks like:

  ```html
  <input id="ember1" class="ember-view" />
  ```

  `attributeBindings` can refer to computed properties:

  ```app/components/my-text-input.js
  import Component from '@ember/component';
  import { computed } from '@ember/object';

  export default Component.extend({
    tagName: 'input',
    attributeBindings: ['disabled'],

    disabled: computed(function() {
      if (someLogic) {
        return true;
      } else {
        return false;
      }
    })
  });
  ```

  To prevent setting an attribute altogether, use `null` or `undefined` as the
  value of the property used in `attributeBindings`:

  ```app/components/my-text-input.js
  import Component from '@ember/component';

  export default Component.extend({
    tagName: 'form',
    attributeBindings: ['novalidate'],
    novalidate: null
  });
  ```

  Updates to the property of an attribute binding will result in automatic
  update of the  HTML attribute in the component's HTML output.

  `attributeBindings` is a concatenated property. See
  [EmberObject](/ember/release/classes/EmberObject) documentation for more
  information about concatenated properties.

  ## Layouts

  The `layout` property can be used to dynamically specify a template associated
  with a component class, instead of relying on Ember to link together a
  component class and a template based on file names.

  In general, applications should not use this feature, but it's commonly used
  in addons for historical reasons.

  The `layout` property should be set to the default export of a template
  module, which is the name of a template file without the `.hbs` extension.

  ```app/templates/components/person-profile.hbs
  <h1>Person's Title</h1>
  <div class='details'>{{yield}}</div>
  ```

  ```app/components/person-profile.js
    import Component from '@ember/component';
    import layout from '../templates/components/person-profile';

    export default Component.extend({
      layout
    });
  ```

  If you invoke the component:

  ```handlebars
  <PersonProfile>
    <h2>Chief Basket Weaver</h2>
    <h3>Fisherman Industries</h3>
  </PersonProfile>
  ```

  or

  ```handlebars
  {{#person-profile}}
    <h2>Chief Basket Weaver</h2>
    <h3>Fisherman Industries</h3>
  {{/person-profile}}
  ```

  It will result in the following HTML output:

  ```html
  <h1>Person's Title</h1>
    <div class="details">
    <h2>Chief Basket Weaver</h2>
    <h3>Fisherman Industries</h3>
  </div>
  ```

  ## Handling Browser Events

  There are two ways to handle user-initiated events:

  ### Using the `on` modifier to capture browser events

  In a component's template, you can attach an event handler to any element with the `on` modifier:

  ```handlebars
  <button {{on 'click' this.doSomething}} />
  ```

  This will call the function on your component:

  ```js
  import Component from '@ember/component';

  export default class ExampleComponent extends Component {
    doSomething = (event) => {
      // `event` is the native click Event
      console.log('clicked on the button');
    };
  });
  ```

  See the [Guide on Component event
  handlers](https://guides.emberjs.com/release/components/component-state-and-actions/#toc_html-modifiers-and-actions)
  and the [API docs for `on`](../Ember.Templates.helpers/methods/on?anchor=on)
  for more details.

  ### Event Handler Methods

  Components can also respond to user-initiated events by implementing a method
  that matches the event name. This approach is appropriate when the same event
  should be handled by all instances of the same component.

  An event object will be passed as the argument to the event handler method.

  ```app/components/my-widget.js
  import Component from '@ember/component';

  export default Component.extend({
    click(event) {
      // `event.target` is either the component's element or one of its children
      let tag = event.target.tagName.toLowerCase();
      console.log('clicked on a `<${tag}>` HTML element!');
    }
  });
  ```

  In this example, whenever the user clicked anywhere inside the component, it
  will log a message to the console.

  It is possible to handle event types other than `click` by implementing the
  following event handler methods. In addition, custom events can be registered
  by using `Application.customEvents`.

  Touch events:

  * `touchStart`
  * `touchMove`
  * `touchEnd`
  * `touchCancel`

  Keyboard events:

  * `keyDown`
  * `keyUp`
  * `keyPress`

  Mouse events:

  * `mouseDown`
  * `mouseUp`
  * `contextMenu`
  * `click`
  * `doubleClick`
  * `focusIn`
  * `focusOut`

  Form events:

  * `submit`
  * `change`
  * `focusIn`
  * `focusOut`
  * `input`

  Drag and drop events:

  * `dragStart`
  * `drag`
  * `dragEnter`
  * `dragLeave`
  * `dragOver`
  * `dragEnd`
  * `drop`

  @class Component
  @extends Ember.CoreView
  @uses Ember.TargetActionSupport
  @uses Ember.ClassNamesSupport
  @uses Ember.ActionSupport
  @uses Ember.ViewMixin
  @uses Ember.ViewStateSupport
  @public
*/
// This type param is used in the class, so must appear here.
// eslint-disable-next-line @typescript-eslint/no-unused-vars
interface Component<S = unknown>
  extends CoreView,
    ChildViewsSupport,
    ViewStateSupport,
    ClassNamesSupport,
    TargetActionSupport,
    ActionSupport,
    ViewMixin,
    ComponentMethods {}

class Component<S = unknown>
  extends CoreView.extend(
    ChildViewsSupport,
    ViewStateSupport,
    ClassNamesSupport,
    TargetActionSupport,
    ActionSupport,
    ViewMixin,
    {
      // These need to be overridable via extend/create but should still
      // have a default. Defining them here is the best way to achieve that.
      didReceiveAttrs() {},
      didRender() {},
      didUpdate() {},
      didUpdateAttrs() {},
      willRender() {},
      willUpdate() {},
    } as ComponentMethods
  )
  implements PropertyDidChange
{
  isComponent = true;

  // SAFETY: this has no runtime existence whatsoever; it is a "phantom type"
  // here to preserve the type param.
  private declare [SIGNATURE]: S;

  // SAFTEY: This is set in `init`.
  declare _superRerender: ViewMixin['rerender'];

  declare [IS_DISPATCHING_ATTRS]: boolean;
  declare [DIRTY_TAG]: DirtyableTag;

  init(properties?: object | undefined) {
    super.init(properties);

    // Handle methods from ViewMixin.
    // The native class inheritance will not work for mixins. To work around this,
    // we copy the existing rerender method provided by the mixin and swap in the
    // new rerender method from our class.
    this._superRerender = this.rerender;
    this.rerender = this._rerender;

    this[IS_DISPATCHING_ATTRS] = false;
    this[DIRTY_TAG] = createTag();
    this[BOUNDS] = null;

    const eventDispatcher = this._dispatcher;
    if (eventDispatcher) {
      let lazyEventsProcessedForComponentClass = lazyEventsProcessed.get(eventDispatcher);
      if (!lazyEventsProcessedForComponentClass) {
        lazyEventsProcessedForComponentClass = new WeakSet<object>();
        lazyEventsProcessed.set(eventDispatcher, lazyEventsProcessedForComponentClass);
      }

      let proto = Object.getPrototypeOf(this);
      if (!lazyEventsProcessedForComponentClass.has(proto)) {
        let lazyEvents = eventDispatcher.lazyEvents;

        lazyEvents.forEach((mappedEventName, event) => {
          if (mappedEventName !== null && typeof (this as any)[mappedEventName] === 'function') {
            eventDispatcher.setupHandlerForBrowserEvent(event);
          }
        });

        lazyEventsProcessedForComponentClass.add(proto);
      }
    }

    if (DEBUG && eventDispatcher && this.renderer._isInteractive && this.tagName === '') {
      let eventNames = [];
      let events = eventDispatcher.finalEventNameMapping;

      for (let key in events) {
        let methodName = events[key];

        if (methodName && typeof (this as any)[methodName] === 'function') {
          eventNames.push(methodName);
        }
      }
      // If in a tagless component, assert that no event handlers are defined
      assert(
        `You can not define \`${eventNames}\` function(s) to handle DOM event in the \`${this}\` tagless component since it doesn't have any DOM element.`,
        !eventNames.length
      );
    }
  }

  __dispatcher?: EventDispatcher | null;

  get _dispatcher(): EventDispatcher | null {
    if (this.__dispatcher === undefined) {
      let owner = getOwner(this);
      assert('Component is unexpectedly missing an owner', owner);

      if ((owner.lookup('-environment:main') as Environment)!.isInteractive) {
        let dispatcher = owner.lookup('event_dispatcher:main');
        assert(
          'Expected dispatcher to be an EventDispatcher',
          dispatcher instanceof EventDispatcher
        );
        this.__dispatcher = dispatcher;
      } else {
        // In FastBoot we have no EventDispatcher. Set to null to not try again to look it up.
        this.__dispatcher = null;
      }
    }

    return this.__dispatcher;
  }

  on<Target>(
    name: string,
    target: Target,
    method: string | ((this: Target, ...args: any[]) => void)
  ): this;
  on(name: string, method: ((...args: any[]) => void) | string): this;
  on(name: string, target: any, method?: any) {
    this._dispatcher?.setupHandlerForEmberEvent(name);
    // The `on` method here comes from the Evented mixin. Since this mixin
    // is applied to the parent of this class, however, we are still able
    // to use `super`.
    return super.on(name, target, method);
  }

  // Changed to `rerender` on init
  _rerender() {
    dirtyTag(this[DIRTY_TAG]);
    this._superRerender();
  }

  [PROPERTY_DID_CHANGE](key: string, value?: unknown) {
    if (this[IS_DISPATCHING_ATTRS]) {
      return;
    }

    let args = (this as any)[ARGS];
    let reference = args !== undefined ? args[key] : undefined;

    if (reference !== undefined && isUpdatableRef(reference)) {
      updateRef(reference, arguments.length === 2 ? value : get(this, key));
    }
  }

  getAttr(key: string) {
    // TODO Intimate API should be deprecated
    return this.get(key);
  }

  /**
    Normally, Ember's component model is "write-only". The component takes a
    bunch of attributes that it got passed in, and uses them to render its
    template.

    One nice thing about this model is that if you try to set a value to the
    same thing as last time, Ember (through HTMLBars) will avoid doing any
    work on the DOM.

    This is not just a performance optimization. If an attribute has not
    changed, it is important not to clobber the element's "hidden state".
    For example, if you set an input's `value` to the same value as before,
    it will clobber selection state and cursor position. In other words,
    setting an attribute is not **always** idempotent.

    This method provides a way to read an element's attribute and also
    update the last value Ember knows about at the same time. This makes
    setting an attribute idempotent.

    In particular, what this means is that if you get an `<input>` element's
    `value` attribute and then re-render the template with the same value,
    it will avoid clobbering the cursor and selection position.
    Since most attribute sets are idempotent in the browser, you typically
    can get away with reading attributes using jQuery, but the most reliable
    way to do so is through this method.
    @method readDOMAttr

    @param {String} name the name of the attribute
    @return String
    @public
    */
  readDOMAttr(name: string) {
    // TODO revisit this
    let _element = getViewElement(this);

    assert(
      `Cannot call \`readDOMAttr\` on ${this} which does not have an element`,
      _element !== null
    );

    let element = _element;
    let isSVG = element.namespaceURI === 'http://www.w3.org/2000/svg';
    let { type, normalized } = normalizeProperty(element as unknown as SimpleElement, name);

    if (isSVG || type === 'attr') {
      return element.getAttribute(normalized);
    }

    return (element as any)[normalized];
  }

  // --- Declarations which support mixins ---
  // We use `declare` on these properties, even though they are optional, so
  // that they do not get created on the class *at all* when emitting the
  // transpiled code. Otherwise, since declared class properties are equivalent
  // to calling `defineProperty` in the class constructor, they would "stomp"
  // the properties supplied by mixins.

  declare attributeBindings?: string[];

  /**
   Enables components to take a list of parameters as arguments.
    For example, a component that takes two parameters with the names
    `name` and `age`:

    ```app/components/my-component.js
    import Component from '@ember/component';

    let MyComponent = Component.extend();

    MyComponent.reopenClass({
      positionalParams: ['name', 'age']
    });

    export default MyComponent;
    ```

    It can then be invoked like this:

    ```hbs
    {{my-component "John" 38}}
    ```

    The parameters can be referred to just like named parameters:

    ```hbs
    Name: {{name}}, Age: {{age}}.
    ```

    Using a string instead of an array allows for an arbitrary number of
    parameters:

    ```app/components/my-component.js
    import Component from '@ember/component';

    let MyComponent = Component.extend();

    MyComponent.reopenClass({
      positionalParams: 'names'
    });

    export default MyComponent;
    ```

    It can then be invoked like this:

    ```hbs
    {{my-component "John" "Michael" "Scott"}}
    ```
    The parameters can then be referred to by enumerating over the list:

    ```hbs
    {{#each names as |name|}}{{name}}{{/each}}
    ```

    @static
    @public
    @property positionalParams
    @since 1.13.0
    */ /**
   Enables components to take a list of parameters as arguments.
    For example, a component that takes two parameters with the names
    `name` and `age`:

    ```app/components/my-component.js
    import Component from '@ember/component';

    let MyComponent = Component.extend();

    MyComponent.reopenClass({
      positionalParams: ['name', 'age']
    });

    export default MyComponent;
    ```

    It can then be invoked like this:

    ```hbs
    {{my-component "John" 38}}
    ```

    The parameters can be referred to just like named parameters:

    ```hbs
    Name: {{name}}, Age: {{age}}.
    ```

    Using a string instead of an array allows for an arbitrary number of
    parameters:

    ```app/components/my-component.js
    import Component from '@ember/component';

    let MyComponent = Component.extend();

    MyComponent.reopenClass({
      positionalParams: 'names'
    });

    export default MyComponent;
    ```

    It can then be invoked like this:

    ```hbs
    {{my-component "John" "Michael" "Scott"}}
    ```
    The parameters can then be referred to by enumerating over the list:

    ```hbs
    {{#each names as |name|}}{{name}}{{/each}}
    ```

    @static
    @public
    @property positionalParams
    @since 1.13.0
    */
  declare static positionalParams: string | string[];

  /**
    Layout can be used to wrap content in a component.
    @property layout
    @type Function
    @public
  */
  declare layout?: TemplateFactory | Template;

  /**
    The name of the layout to lookup if no layout is provided.
    By default `Component` will lookup a template with this name in
    `Ember.TEMPLATES` (a shared global object).
    @property layoutName
    @type String
    @default undefined
    @private
  */
  declare layoutName?: string;

  /**
   The WAI-ARIA role of the control represented by this view. For example, a
    button may have a role of type 'button', or a pane may have a role of
    type 'alertdialog'. This property is used by assistive software to help
    visually challenged users navigate rich web applications.

    The full list of valid WAI-ARIA roles is available at:
    [https://www.w3.org/TR/wai-aria/#roles_categorization](https://www.w3.org/TR/wai-aria/#roles_categorization)

    @property ariaRole
    @type String
    @default undefined
    @public
    */
  declare ariaRole?: string;

  static isComponentFactory = true;

  static toString() {
    return '@ember/component';
  }
}

// We continue to use reopenClass here so that positionalParams can be overridden with reopenClass in subclasses.
Component.reopenClass({
  positionalParams: [],
});

setInternalComponentManager(CURLY_COMPONENT_MANAGER, Component);

export default Component;
