### 后端校验登录逻辑：
1. 获取`token`，去`token`表中查询是否存在、过期，
2. 如果是过期了，返回code = 1
3. 如果为过期，返回`token`信息
4. 后端拿到用户信息进行相关性操作
```ts
// src\common\parse-token.pipe.ts
@Injectable()
export class ParseTokenPipe implements PipeTransform {
  @Inject()
  private tokenService: TokenService;

  async transform(value: string, metadata: ArgumentMetadata) {
    // console.log('token:', value);
    const data = await this.tokenService.getTokenInfo(value);
    // console.log('uid:', data);

    if (data === null) {
      throw new AppException(1, '用户未登录！');
    }
    return data;
  }
}
```

### 后端登录接口逻辑：
1. 先判断用户密码是否正确
2. 如果不正确，返回`code=-1`（后续有空在换）
3. 如果正确，生成`token`，`token`里面包含一个随机字符串+过期时间（每次登录都需要校验是否过期）
4. 后端返回`token`
```ts
// src\user\user.controller.ts
@Post('/login')
  async login(@Body() loginDto: LoginDto) {
    const ok = await this.userService.checkPassword(
      loginDto.uid,
      loginDto.password,
    );
    if (ok) {
      const token = await this.tokenService.createToken(
        loginDto.uid,
        2 * 60 * 60,
      );
      const user = await this.userService.getUser(loginDto.uid);
      const data = { token, uid: loginDto.uid, name: user.name };
      return data;
    }
    // 密码错误，返回state = 1；
    throw new Error('密码错误，登录失败');
  }
```



### 前端登录逻辑
1. 把登录请求到的`token`存入`localStorage`里面
```js
// client\src\js\service.js
// 登录请求方法
export const login = async (data) => {
  try {
    const response = await axios({
      ...API.LOGIN_USER,
      data: {
        uid: data.uid,
        password: data.password,
      }
    })
    const userInfo = JSON.stringify(response.data.data);
    console.log(response);
    window.localStorage.setItem(TOKEN, response.data.data.token);
    window.localStorage.setItem(USER_INFO, userInfo);
    return response.data;
  } catch (err) {
    console.error(err);
  }
}

```


### 前端会话保持逻辑
1. 使用`localStorage.getItem()`获取token
2. 使用`axios`请求拦截器，在接口调用时自动填充token
3. 使用`axios`响应拦截器,实现登录过期自动跳转逻辑
4. 使用`router.push(/login)`自动跳转登录页
```js
// client\src\js\service.js
// 添加请求拦截器
axios.interceptors.request.use(function (config) {
  // 在发送请求之前做些什么
  const token = window.localStorage.getItem(TOKEN);
  if(token) {
    config.headers.token = token || '';
  }
  // console.log('config.headers.token', token);
  return config;
}, function (error) {
  // 对请求错误做些什么
  return Promise.reject(error);
});

// 添加响应拦截器
axios.interceptors.response.use((response) => {
  // 2xx 范围内的状态码都会触发该函数。
  // 对响应数据，token过期，或者无token时跳转登录
  const code = response.data.code;
  if(code === 1) {
    router.push('/login');
  }

  if(code !== 0) {
    Message({
      message: response.data.message,
      type: 'error'
    });
  }
  // console.log('response', response)
  return response;
}, function (error) {
  // 超出 2xx 范围的状态码都会触发该函数。
  // 对响应错误做点什么
  return Promise.reject(error);
});
```