import sqlite3
from .config import config
from .log import logger_group
from os import path
from logbook import Logger
import pandas as pd
from typing import Callable, Any
from uuid import uuid4

logger = Logger("bot.db")
logger_group.add_logger(logger)

_db_path = path.join(config.get("db_location"), config.get("db_file")) # type: ignore

_db_schema = f"""
-- The schema of the database

CREATE TABLE rooms (
    -- rooms table
    id_room INTEGER PRIMARY KEY,
    matrix_room_id TEXT NOT NULL UNIQUE
);

CREATE TABLE sessions (
    -- sessions table
    id_session INTEGER PRIMARY KEY,
    room_id INTEGER NOT NULL,
    session_hash TEXT NOT NULL UNIQUE, -- uuid4.hex hash generated by application
    session_desc TEXT NOT NULL,
    archived INTEGER NOT NULL CHECK(archived >= 0 AND archived < 2), -- boolean, 1 or 0
    FOREIGN KEY(room_id) REFERENCES rooms(id_room)
);

CREATE TABLE messages (
    -- messages table
    id_message INTEGER PRIMARY KEY,
    session_id INTEGER NOT NULL,
    user_message INTEGER NOT NULL CHECK(user_message >= 0 AND user_message < 2), -- boolean, 1 or 0
    content TEXT NOT NULL,
    timestamp TEXT,
    FOREIGN KEY(session_id) REFERENCES sessions(id_session)
);

CREATE TRIGGER insert_message_timestamp_trigger
AFTER INSERT ON messages
BEGIN
    UPDATE messages SET timestamp = STRFTIME('%Y-%m-%d %H:%M:%f', 'NOW') WHERE id_message = NEW.id_message;
END;
"""

def _db_work(work_func: Callable[[sqlite3.Connection], Any]) -> Any:
    with sqlite3.connect(_db_path) as con:
        result = work_func(con)
        return result

def _prepare_db(con: sqlite3.Connection):
    cur: sqlite3.Cursor = con.cursor()
    try:
        cur.executescript(_db_schema)
        cur.execute("PRAGMA user_version = 1")
    except:
        logger.critical("There was an error while preparing the schema in the db... exiting.")
        exit(1)
    logger.info("Created db schema.")
    cur.close()

if not path.isfile(_db_path):
    logger.debug("Creating db file.")
    _db_work(_prepare_db)
else:
    logger.debug("Database file exists.")

logger.debug(f"""The db is at {_db_path}""")

def get_session_list(room_id: str):
    statement = f"""
SELECT
    SUBSTR(s.session_hash, 0, 12),
    SUBSTR(session_desc, 0, 100) || '...',
    m.timestamp
FROM rooms AS r
INNER JOIN sessions AS s
ON r.id_room = s.room_id
INNER JOIN messages AS m
ON s.id_session = m.session_id
WHERE s.archived = 0
AND r.matrix_room_id = ?
GROUP BY s.session_hash
HAVING m.timestamp = MAX(m.timestamp)
ORDER BY m.timestamp DESC
"""
    result = []
    with sqlite3.connect(_db_path) as con:
        cur: sqlite3.Cursor = con.cursor()
        res = cur.execute(statement, (room_id,))
        result = res.fetchall()
        cur.close()
    return result

def get_sessions_by_hash(room_id: str, session_hash: str):
    statement = f"""
SELECT
    s.session_hash,
    SUBSTR(session_desc, 0, 20) || '...',
    m.timestamp
FROM rooms AS r
INNER JOIN sessions AS s
ON r.id_room = s.room_id
INNER JOIN messages AS m
ON s.id_session = m.session_id
WHERE s.archived = 0
AND r.matrix_room_id = ?
AND s.session_hash LIKE ?
GROUP BY s.session_hash
HAVING m.timestamp = MAX(m.timestamp)
ORDER BY m.timestamp DESC
"""
    result = []
    with sqlite3.connect(_db_path) as con:
        cur: sqlite3.Cursor = con.cursor()
        res = cur.execute(statement, (room_id, session_hash + '%',))
        result = res.fetchall()
        cur.close()
    return result

def hash_exists(session_hash: str):
    statement = f"""
SELECT COUNT(id_session)
FROM sessions
WHERE session_hash = ?
"""
    result = False
    with sqlite3.connect(_db_path) as con:
        cur: sqlite3.Cursor = con.cursor()
        res = cur.execute(statement, (session_hash,))
        count = res.fetchone()[0]
        if count > 0:
            result = True
        cur.close()
    return result

def delete_session(room_id: str, session_hash: str):
    # Delete session and change to last active if there is one
    statement = f"""
UPDATE sessions SET archived = 1
WHERE id_session = (
    SELECT
    s.id_session
    FROM rooms AS r
    INNER JOIN sessions AS s
    ON r.id_room = s.room_id
    WHERE s.archived = 0
    AND r.matrix_room_id = ?
    AND s.session_hash LIKE ?
    GROUP BY s.session_hash
    LIMIT 1
)
"""
    statement_two = f"""
UPDATE messages SET timestamp = STRFTIME('%Y-%m-%d %H:%M:%f', 'NOW')
WHERE id_message = (
    SELECT m.id_message FROM
    rooms AS r
    INNER JOIN sessions AS s
    ON r.id_room = s.room_id
    INNER JOIN messages AS m
    ON s.id_session = m.session_id
    WHERE s.archived = 0
    AND r.matrix_room_id = ?
    GROUP BY s.session_hash
    HAVING m.timestamp = MAX(m.timestamp)
    ORDER BY m.timestamp DESC LIMIT 1
)
"""
    statement_three = f"""
SELECT
    SUBSTR(s.session_hash, 0, 12),
    SUBSTR(session_desc, 0, 20) || '...',
    m.timestamp
FROM rooms AS r
INNER JOIN sessions AS s
ON r.id_room = s.room_id
INNER JOIN messages AS m
ON s.id_session = m.session_id
WHERE s.archived = 0
AND r.matrix_room_id = ?
GROUP BY s.session_hash
HAVING m.timestamp = MAX(m.timestamp)
ORDER BY m.timestamp DESC LIMIT 1
"""

    deleted = False
    newly_active = None
    with sqlite3.connect(_db_path) as con:
        cur: sqlite3.Cursor = con.cursor()
        res = cur.execute(statement, (room_id, session_hash + '%',))
        if res.rowcount == 1:
            deleted = True
        res = cur.execute(statement_two, (room_id,))
        if res.rowcount == 1:
            res = cur.execute(statement_three, (room_id,))
            newly_active = res.fetchone()
        cur.close()
    return (deleted, newly_active)

def activate_session(room_id: str, session_hash: str):
    statement = f"""
UPDATE messages SET timestamp = STRFTIME('%Y-%m-%d %H:%M:%f', 'NOW')
WHERE id_message = (
    SELECT m.id_message FROM
    rooms AS r
    INNER JOIN sessions AS s
    ON r.id_room = s.room_id
    INNER JOIN messages AS m
    ON s.id_session = m.session_id
    WHERE s.archived = 0
    AND r.matrix_room_id = ?
    AND s.session_hash LIKE ?
    GROUP BY s.session_hash
    HAVING m.timestamp = MAX(m.timestamp)
    ORDER BY m.timestamp DESC LIMIT 1
)
"""
    statement_two = f"""
SELECT
    SUBSTR(s.session_hash, 0, 12),
    SUBSTR(session_desc, 0, 20) || '...',
    m.timestamp
FROM rooms AS r
INNER JOIN sessions AS s
ON r.id_room = s.room_id
INNER JOIN messages AS m
ON s.id_session = m.session_id
WHERE s.archived = 0
AND r.matrix_room_id = ?
GROUP BY s.session_hash
HAVING m.timestamp = MAX(m.timestamp)
ORDER BY m.timestamp DESC LIMIT 1
"""

    updated = False
    newly_active = None
    with sqlite3.connect(_db_path) as con:
        cur: sqlite3.Cursor = con.cursor()
        res = cur.execute(statement, (room_id, session_hash + '%',))
        if res.rowcount == 1:
            updated = True
        res = cur.execute(statement_two, (room_id,))
        newly_active = res.fetchone()
        cur.close()
    return (updated, newly_active)

def get_active_session(room_id: str):
    statement = f"""
SELECT
    SUBSTR(s.session_hash, 0, 12),
    SUBSTR(session_desc, 0, 100) || '...',
    m.timestamp
FROM rooms AS r
INNER JOIN sessions AS s
ON r.id_room = s.room_id
INNER JOIN messages AS m
ON s.id_session = m.session_id
WHERE s.archived = 0
AND r.matrix_room_id = ?
GROUP BY s.session_hash
HAVING m.timestamp = MAX(m.timestamp)
ORDER BY m.timestamp DESC LIMIT 1
"""

    result = None
    with sqlite3.connect(_db_path) as con:
        cur: sqlite3.Cursor = con.cursor()
        res = cur.execute(statement, (room_id,))
        result = res.fetchone()
        cur.close()
    return result

def create_room_if_not_exists(room_id: str):
    statement = f"""
INSERT INTO rooms(matrix_room_id)
SELECT ?
WHERE NOT EXISTS(SELECT 1 FROM rooms WHERE matrix_room_id = ?)
"""
    with sqlite3.connect(_db_path) as con:
        cur: sqlite3.Cursor = con.cursor()
        res = cur.execute(statement, (room_id, room_id,))
        cur.close()

def get_conversation(room_id: str, limit: int = 50):
    statement = f"""
SELECT
    m.user_message,
    m.content
FROM rooms AS r
INNER JOIN sessions AS s
ON r.id_room = s.room_id
INNER JOIN messages AS m
ON s.id_session = m.session_id
WHERE s.archived = 0
AND r.matrix_room_id = ?
AND s.session_hash = (
    SELECT
        s.session_hash
    FROM rooms AS r
    INNER JOIN sessions AS s
    ON r.id_room = s.room_id
    INNER JOIN messages AS m
    ON s.id_session = m.session_id
    WHERE s.archived = 0
    AND r.matrix_room_id = ?
    GROUP BY s.session_hash
    HAVING m.timestamp = MAX(m.timestamp)
    ORDER BY m.timestamp DESC LIMIT 1
)
ORDER BY m.timestamp DESC LIMIT ?
"""

    result = []
    with sqlite3.connect(_db_path) as con:
        cur: sqlite3.Cursor = con.cursor()
        res = cur.execute(statement, (room_id, room_id, limit))
        result = res.fetchall()
        cur.close()
    return result

def create_new_message(room_id: str, message: str, from_user: bool):
    statement = f"""
INSERT INTO messages(session_id, user_message, content)
SELECT s.id_session, ?, ?
FROM sessions AS s
INNER JOIN rooms AS r
ON r.id_room = s.room_id
WHERE r.matrix_room_id = ?
AND s.session_hash = (
    SELECT
        s.session_hash
    FROM rooms AS r
    INNER JOIN sessions AS s
    ON r.id_room = s.room_id
    INNER JOIN messages AS m
    ON s.id_session = m.session_id
    WHERE s.archived = 0
    AND r.matrix_room_id = ?
    GROUP BY s.session_hash
    HAVING m.timestamp = MAX(m.timestamp)
    ORDER BY m.timestamp DESC LIMIT 1
)
"""
    with sqlite3.connect(_db_path) as con:
        cur: sqlite3.Cursor = con.cursor()
        res = cur.execute(statement, (1 if from_user else 0, message, room_id, room_id,))
        cur.close()

def get_or_create_conversation(room_id: str, message: str, limit: int = 50, new = False):
    result = get_active_session(room_id)
    if not result or new:

        # Create unique hash
        hash = uuid4().hex
        while hash_exists(hash):
            hash = uuid4().hex

        # Create session row
        statement = f"""
INSERT INTO sessions(room_id, session_hash, session_desc, archived)
SELECT rooms.id_room, ?, SUBSTR(?, 0, 100), 0
FROM rooms
WHERE rooms.matrix_room_id = ?
"""
        with sqlite3.connect(_db_path) as con:
            cur: sqlite3.Cursor = con.cursor()
            res = cur.execute(statement, (hash, message, room_id,))
            cur.close()
        
        # Create first user message
        statement = f"""
INSERT INTO messages(session_id, user_message, content)
SELECT s.id_session, 1, ?
FROM sessions AS s
INNER JOIN rooms AS r
ON r.id_room = s.room_id
WHERE r.matrix_room_id = ?
AND s.session_hash = ?
"""
        with sqlite3.connect(_db_path) as con:
            cur: sqlite3.Cursor = con.cursor()
            res = cur.execute(statement, (message, room_id, hash,))
            cur.close()

    # Select all messages of a session in a room
    conversation = get_conversation(room_id)
    if len(conversation) == 0:
        raise Exception("At this point, at least one message should exist in the conversation...")
    if conversation[0][0] == 0:
        # Last message was from assistant. Let's insert user message here.
        create_new_message(room_id, message, True)
    return get_conversation(room_id)