# 🧬 3D Mesh

DocArray supports many different modalities including `3D Mesh`.
This section will show you how to load and handle 3D data using DocArray.

A 3D mesh is the structural build of a 3D model consisting of polygons. Most 3D meshes are created via professional software packages, such as commercial suites like [Unity](https://unity.com/), or the open-source [Blender](https://www.blender.org/).

!!! note
    This feature requires `trimesh`. You can install all necessary dependencies via:

    ```cmd
    pip install "docarray[mesh]"
    ```

## Vertices and Faces representation

A 3D mesh can be represented by its vertices and faces:

- **Vertices** are points in a 3D space, represented as a tensor of shape `(n_points, 3)`. 
- **Faces** are triangular surfaces that are defined by three points in 3D space, corresponding to the three vertices of a triangle. They can be represented as a tensor of shape `(n_faces, 3)`. Each number in that tensor refers to an index of a vertex in the tensor of vertices.

### Load vertices and faces

First, let's define our class `MyMesh3D`, which extends [`BaseDoc`][docarray.base_doc.doc.BaseDoc] and provides attributes to store our 3D data:

- The `mesh_url` attribute of type [`Mesh3DUrl`][docarray.typing.url.url_3d.mesh_url.Mesh3DUrl]. 
- The optional `tensors` attribute, of type [`VerticesAndFaces`][docarray.documents.mesh.vertices_and_faces.VerticesAndFaces]
  - The `VerticesAndFaces` class has the attributes `vertices` and `faces`, both of type [`AnyTensor`](../../../../api_references/typing/tensor/tensor). This especially comes in handy later when we want to display our 3D mesh.
 
!!! tip
    Check out our predefined [`Mesh3D`](#getting-started-predefined-docs) to get started and play around with our 3D features.

But for now, let's create a `MyMesh3D` instance with a URL to a remote `.obj` file:

```python
from typing import Optional

from docarray import BaseDoc
from docarray.documents.mesh.vertices_and_faces import VerticesAndFaces
from docarray.typing import Mesh3DUrl


class MyMesh3D(BaseDoc):
    mesh_url: Mesh3DUrl
    tensors: Optional[VerticesAndFaces] = None


doc = MyMesh3D(mesh_url="https://people.sc.fsu.edu/~jburkardt/data/obj/al.obj")
```

To load the vertices and faces information, you can call [`.load()`][docarray.typing.url.url_3d.mesh_url.Mesh3DUrl.load] on the [`Mesh3DUrl`][docarray.typing.url.url_3d.mesh_url.Mesh3DUrl] instance. This will return a [`VerticesAndFaces`][docarray.documents.mesh.vertices_and_faces.VerticesAndFaces] object:

```python
doc.tensors = doc.mesh_url.load()
doc.summary()
```

<details>
    <summary>Output</summary>
    ``` { .text .no-copy }
    📄 MyMesh3D : 9d8c26f ...
    ╭─────────────────────┬────────────────────────────────────────────────────────╮
    │ Attribute           │ Value                                                  │
    ├─────────────────────┼────────────────────────────────────────────────────────┤
    │ mesh_url: Mesh3DUrl │ https://people.sc.fsu.edu/~jburkardt/data/obj/al.o ... │
    │                     │ (length: 52)                                           │
    ╰─────────────────────┴────────────────────────────────────────────────────────╯
    └── 🔶 tensors: VerticesAndFaces
        └── 📄 VerticesAndFaces : 8cae4c4 ...
            ╭───────────────────┬──────────────────────────────────────────────────╮
            │ Attribute         │ Value                                            │
            ├───────────────────┼──────────────────────────────────────────────────┤
            │ vertices: NdArray │ NdArray of shape (3980, 3), dtype: float64       │
            │ faces: NdArray    │ NdArray of shape (7152, 3), dtype: int64         │
            ╰───────────────────┴──────────────────────────────────────────────────╯
    ```
</details>

### Display 3D mesh in notebook

You can display your 3D mesh interactively from its URL as well as a [`VerticesAndFaces`][docarray.documents.mesh.vertices_and_faces.VerticesAndFaces] instance, by calling `.display()` on either one. The latter will always display without color, whereas the display from the URL will show with color if this information is included in the file content.

``` { .python }
doc.mesh_url.display()
```

<iframe srcdoc="<!DOCTYPE html>
<html lang=&quot;en&quot;>
  <head>
    <title>trimesh: threejs viewer</title>
    <meta charset=&quot;utf-8&quot;>
    <meta name=&quot;viewport&quot; content=&quot;width=device-width,
		   user-scalable=no,
		   minimum-scale=1.0,
		   maximum-scale=1.0&quot;>
    <style>
      body {
      margin: 0px;
      overflow: hidden;
      }
    </style>
  </head>
  <body>
    <div id=&quot;container&quot;></div>
    <script>// threejs.org/license
(function(k,Ea){&quot;object&quot;===typeof exports&&&quot;undefined&quot;!==typeof module?Ea(exports):&quot;function&quot;===typeof define&&define.amd?define([&quot;exports&quot;],Ea):(k=k||self,Ea(k.THREE={}))})(this,function(k){function Ea(){}function t(a,b){this.x=a||0;this.y=b||0}function wa(){this.elements=[1,0,0,0,1,0,0,0,1];0<arguments.length&&console.error(&quot;THREE.Matrix3: the constructor no longer reads arguments. use .set() instead.&quot;)}function V(a,b,c,d,e,f,g,h,l,m){Object.defineProperty(this,&quot;id&quot;,{value:pj++});this.uuid=L.generateUUID();
this.name=&quot;&quot;;this.image=void 0!==a?a:V.DEFAULT_IMAGE;this.mipmaps=[];this.mapping=void 0!==b?b:V.DEFAULT_MAPPING;this.wrapS=void 0!==c?c:1001;this.wrapT=void 0!==d?d:1001;this.magFilter=void 0!==e?e:1006;this.minFilter=void 0!==f?f:1008;this.anisotropy=void 0!==l?l:1;this.format=void 0!==g?g:1023;this.internalFormat=null;this.type=void 0!==h?h:1009;this.offset=new t(0,0);this.repeat=new t(1,1);this.center=new t(0,0);this.rotation=0;this.matrixAutoUpdate=!0;this.matrix=new wa;this.generateMipmaps=
!0;this.premultiplyAlpha=!1;this.flipY=!0;this.unpackAlignment=4;this.encoding=void 0!==m?m:3E3;this.version=0;this.onUpdate=null}function ka(a,b,c,d){this.x=a||0;this.y=b||0;this.z=c||0;this.w=void 0!==d?d:1}function Ha(a,b,c){this.width=a;this.height=b;this.scissor=new ka(0,0,a,b);this.scissorTest=!1;this.viewport=new ka(0,0,a,b);c=c||{};this.texture=new V(void 0,c.mapping,c.wrapS,c.wrapT,c.magFilter,c.minFilter,c.format,c.type,c.anisotropy,c.encoding);this.texture.image={};this.texture.image.width=
a;this.texture.image.height=b;this.texture.generateMipmaps=void 0!==c.generateMipmaps?c.generateMipmaps:!1;this.texture.minFilter=void 0!==c.minFilter?c.minFilter:1006;this.depthBuffer=void 0!==c.depthBuffer?c.depthBuffer:!0;this.stencilBuffer=void 0!==c.stencilBuffer?c.stencilBuffer:!0;this.depthTexture=void 0!==c.depthTexture?c.depthTexture:null}function Zf(a,b,c){Ha.call(this,a,b,c);this.samples=4}function Aa(a,b,c,d){this._x=a||0;this._y=b||0;this._z=c||0;this._w=void 0!==d?d:1}function n(a,b,
c){this.x=a||0;this.y=b||0;this.z=c||0}function P(){this.elements=[1,0,0,0,0,1,0,0,0,0,1,0,0,0,0,1];0<arguments.length&&console.error(&quot;THREE.Matrix4: the constructor no longer reads arguments. use .set() instead.&quot;)}function Tb(a,b,c,d){this._x=a||0;this._y=b||0;this._z=c||0;this._order=d||Tb.DefaultOrder}function He(){this.mask=1}function F(){Object.defineProperty(this,&quot;id&quot;,{value:qj++});this.uuid=L.generateUUID();this.name=&quot;&quot;;this.type=&quot;Object3D&quot;;this.parent=null;this.children=[];this.up=F.DefaultUp.clone();
var a=new n,b=new Tb,c=new Aa,d=new n(1,1,1);b._onChange(function(){c.setFromEuler(b,!1)});c._onChange(function(){b.setFromQuaternion(c,void 0,!1)});Object.defineProperties(this,{position:{configurable:!0,enumerable:!0,value:a},rotation:{configurable:!0,enumerable:!0,value:b},quaternion:{configurable:!0,enumerable:!0,value:c},scale:{configurable:!0,enumerable:!0,value:d},modelViewMatrix:{value:new P},normalMatrix:{value:new wa}});this.matrix=new P;this.matrixWorld=new P;this.matrixAutoUpdate=F.DefaultMatrixAutoUpdate;
this.matrixWorldNeedsUpdate=!1;this.layers=new He;this.visible=!0;this.receiveShadow=this.castShadow=!1;this.frustumCulled=!0;this.renderOrder=0;this.userData={}}function ob(){F.call(this);this.type=&quot;Scene&quot;;this.overrideMaterial=this.fog=this.environment=this.background=null;this.autoUpdate=!0;&quot;undefined&quot;!==typeof __THREE_DEVTOOLS__&&__THREE_DEVTOOLS__.dispatchEvent(new CustomEvent(&quot;observe&quot;,{detail:this}))}function Sa(a,b){this.min=void 0!==a?a:new n(Infinity,Infinity,Infinity);this.max=void 0!==
b?b:new n(-Infinity,-Infinity,-Infinity)}function $f(a,b,c,d,e){var f;var g=0;for(f=a.length-3;g<=f;g+=3){Ub.fromArray(a,g);var h=e.x*Math.abs(Ub.x)+e.y*Math.abs(Ub.y)+e.z*Math.abs(Ub.z),l=b.dot(Ub),m=c.dot(Ub),u=d.dot(Ub);if(Math.max(-Math.max(l,m,u),Math.min(l,m,u))>h)return!1}return!0}function pb(a,b){this.center=void 0!==a?a:new n;this.radius=void 0!==b?b:0}function Vb(a,b){this.origin=void 0!==a?a:new n;this.direction=void 0!==b?b:new n(0,0,-1)}function Ta(a,b){this.normal=void 0!==a?a:new n(1,
0,0);this.constant=void 0!==b?b:0}function oa(a,b,c){this.a=void 0!==a?a:new n;this.b=void 0!==b?b:new n;this.c=void 0!==c?c:new n}function A(a,b,c){return void 0===b&&void 0===c?this.set(a):this.setRGB(a,b,c)}function ag(a,b,c){0>c&&(c+=1);1<c&&--c;return c<1/6?a+6*(b-a)*c:.5>c?b:c<2/3?a+6*(b-a)*(2/3-c):a}function bg(a){return.04045>a?.0773993808*a:Math.pow(.9478672986*a+.0521327014,2.4)}function cg(a){return.0031308>a?12.92*a:1.055*Math.pow(a,.41666)-.055}function Bc(a,b,c,d,e,f){this.a=a;this.b=
b;this.c=c;this.normal=d&&d.isVector3?d:new n;this.vertexNormals=Array.isArray(d)?d:[];this.color=e&&e.isColor?e:new A;this.vertexColors=Array.isArray(e)?e:[];this.materialIndex=void 0!==f?f:0}function K(){Object.defineProperty(this,&quot;id&quot;,{value:rj++});this.uuid=L.generateUUID();this.name=&quot;&quot;;this.type=&quot;Material&quot;;this.fog=!0;this.blending=1;this.side=0;this.vertexColors=this.flatShading=!1;this.opacity=1;this.transparent=!1;this.blendSrc=204;this.blendDst=205;this.blendEquation=100;this.blendEquationAlpha=
this.blendDstAlpha=this.blendSrcAlpha=null;this.depthFunc=3;this.depthWrite=this.depthTest=!0;this.stencilWriteMask=255;this.stencilFunc=519;this.stencilRef=0;this.stencilFuncMask=255;this.stencilZPass=this.stencilZFail=this.stencilFail=7680;this.stencilWrite=!1;this.clippingPlanes=null;this.clipShadows=this.clipIntersection=!1;this.shadowSide=null;this.colorWrite=!0;this.precision=null;this.polygonOffset=!1;this.polygonOffsetUnits=this.polygonOffsetFactor=0;this.dithering=!1;this.alphaTest=0;this.premultipliedAlpha=
!1;this.toneMapped=this.visible=!0;this.userData={};this.version=0}function Oa(a){K.call(this);this.type=&quot;MeshBasicMaterial&quot;;this.color=new A(16777215);this.lightMap=this.map=null;this.lightMapIntensity=1;this.aoMap=null;this.aoMapIntensity=1;this.envMap=this.alphaMap=this.specularMap=null;this.combine=0;this.reflectivity=1;this.refractionRatio=.98;this.wireframe=!1;this.wireframeLinewidth=1;this.wireframeLinejoin=this.wireframeLinecap=&quot;round&quot;;this.morphTargets=this.skinning=!1;this.setValues(a)}
function M(a,b,c){if(Array.isArray(a))throw new TypeError(&quot;THREE.BufferAttribute: array should be a Typed Array.&quot;);this.name=&quot;&quot;;this.array=a;this.itemSize=b;this.count=void 0!==a?a.length/b:0;this.normalized=!0===c;this.usage=35044;this.updateRange={offset:0,count:-1};this.version=0}function Bd(a,b,c){M.call(this,new Int8Array(a),b,c)}function Cd(a,b,c){M.call(this,new Uint8Array(a),b,c)}function Dd(a,b,c){M.call(this,new Uint8ClampedArray(a),b,c)}function Ed(a,b,c){M.call(this,new Int16Array(a),
b,c)}function Wb(a,b,c){M.call(this,new Uint16Array(a),b,c)}function Fd(a,b,c){M.call(this,new Int32Array(a),b,c)}function Xb(a,b,c){M.call(this,new Uint32Array(a),b,c)}function y(a,b,c){M.call(this,new Float32Array(a),b,c)}function Gd(a,b,c){M.call(this,new Float64Array(a),b,c)}function xh(){this.vertices=[];this.normals=[];this.colors=[];this.uvs=[];this.uvs2=[];this.groups=[];this.morphTargets={};this.skinWeights=[];this.skinIndices=[];this.boundingSphere=this.boundingBox=null;this.groupsNeedUpdate=
this.uvsNeedUpdate=this.colorsNeedUpdate=this.normalsNeedUpdate=this.verticesNeedUpdate=!1}function yh(a){if(0===a.length)return-Infinity;for(var b=a[0],c=1,d=a.length;c<d;++c)a[c]>b&&(b=a[c]);return b}function C(){Object.defineProperty(this,&quot;id&quot;,{value:sj+=2});this.uuid=L.generateUUID();this.name=&quot;&quot;;this.type=&quot;BufferGeometry&quot;;this.index=null;this.attributes={};this.morphAttributes={};this.morphTargetsRelative=!1;this.groups=[];this.boundingSphere=this.boundingBox=null;this.drawRange={start:0,count:Infinity};
this.userData={}}function S(a,b){F.call(this);this.type=&quot;Mesh&quot;;this.geometry=void 0!==a?a:new C;this.material=void 0!==b?b:new Oa;this.updateMorphTargets()}function zh(a,b,c,d,e,f,g,h){if(null===(1===b.side?d.intersectTriangle(g,f,e,!0,h):d.intersectTriangle(e,f,g,2!==b.side,h)))return null;Ie.copy(h);Ie.applyMatrix4(a.matrixWorld);b=c.ray.origin.distanceTo(Ie);return b<c.near||b>c.far?null:{distance:b,point:Ie.clone(),object:a}}function Je(a,b,c,d,e,f,g,h,l,m,u,p){Yb.fromBufferAttribute(e,m);Zb.fromBufferAttribute(e,
u);$b.fromBufferAttribute(e,p);e=a.morphTargetInfluences;if(b.morphTargets&&f&&e){Ke.set(0,0,0);Le.set(0,0,0);Me.set(0,0,0);for(var x=0,r=f.length;x<r;x++){var k=e[x],v=f[x];0!==k&&(dg.fromBufferAttribute(v,m),eg.fromBufferAttribute(v,u),fg.fromBufferAttribute(v,p),g?(Ke.addScaledVector(dg,k),Le.addScaledVector(eg,k),Me.addScaledVector(fg,k)):(Ke.addScaledVector(dg.sub(Yb),k),Le.addScaledVector(eg.sub(Zb),k),Me.addScaledVector(fg.sub($b),k)))}Yb.add(Ke);Zb.add(Le);$b.add(Me)}if(a=zh(a,b,c,d,Yb,Zb,
$b,Hd))h&&(Cc.fromBufferAttribute(h,m),Dc.fromBufferAttribute(h,u),Ec.fromBufferAttribute(h,p),a.uv=oa.getUV(Hd,Yb,Zb,$b,Cc,Dc,Ec,new t)),l&&(Cc.fromBufferAttribute(l,m),Dc.fromBufferAttribute(l,u),Ec.fromBufferAttribute(l,p),a.uv2=oa.getUV(Hd,Yb,Zb,$b,Cc,Dc,Ec,new t)),h=new Bc(m,u,p),oa.getNormal(Yb,Zb,$b,h.normal),a.face=h;return a}function N(){Object.defineProperty(this,&quot;id&quot;,{value:tj+=2});this.uuid=L.generateUUID();this.name=&quot;&quot;;this.type=&quot;Geometry&quot;;this.vertices=[];this.colors=[];this.faces=[];
this.faceVertexUvs=[[]];this.morphTargets=[];this.morphNormals=[];this.skinWeights=[];this.skinIndices=[];this.lineDistances=[];this.boundingSphere=this.boundingBox=null;this.groupsNeedUpdate=this.lineDistancesNeedUpdate=this.colorsNeedUpdate=this.normalsNeedUpdate=this.uvsNeedUpdate=this.verticesNeedUpdate=this.elementsNeedUpdate=!1}function Fc(a){var b={},c;for(c in a){b[c]={};for(var d in a[c]){var e=a[c][d];e&&(e.isColor||e.isMatrix3||e.isMatrix4||e.isVector2||e.isVector3||e.isVector4||e.isTexture)?
b[c][d]=e.clone():Array.isArray(e)?b[c][d]=e.slice():b[c][d]=e}}return b}function va(a){for(var b={},c=0;c<a.length;c++){var d=Fc(a[c]),e;for(e in d)b[e]=d[e]}return b}function Ba(a){K.call(this);this.type=&quot;ShaderMaterial&quot;;this.defines={};this.uniforms={};this.vertexShader=&quot;void main() {\n\tgl_Position = projectionMatrix * modelViewMatrix * vec4( position, 1.0 );\n}&quot;;this.fragmentShader=&quot;void main() {\n\tgl_FragColor = vec4( 1.0, 0.0, 0.0, 1.0 );\n}&quot;;this.linewidth=1;this.wireframe=!1;this.wireframeLinewidth=
1;this.morphNormals=this.morphTargets=this.skinning=this.clipping=this.lights=this.fog=!1;this.extensions={derivatives:!1,fragDepth:!1,drawBuffers:!1,shaderTextureLOD:!1};this.defaultAttributeValues={color:[1,1,1],uv:[0,0],uv2:[0,0]};this.index0AttributeName=void 0;this.uniformsNeedUpdate=!1;void 0!==a&&(void 0!==a.attributes&&console.error(&quot;THREE.ShaderMaterial: attributes should now be defined in THREE.BufferGeometry instead.&quot;),this.setValues(a))}function db(){F.call(this);this.type=&quot;Camera&quot;;this.matrixWorldInverse=
new P;this.projectionMatrix=new P;this.projectionMatrixInverse=new P}function aa(a,b,c,d){db.call(this);this.type=&quot;PerspectiveCamera&quot;;this.fov=void 0!==a?a:50;this.zoom=1;this.near=void 0!==c?c:.1;this.far=void 0!==d?d:2E3;this.focus=10;this.aspect=void 0!==b?b:1;this.view=null;this.filmGauge=35;this.filmOffset=0;this.updateProjectionMatrix()}function Gc(a,b,c,d){F.call(this);this.type=&quot;CubeCamera&quot;;var e=new aa(90,1,a,b);e.up.set(0,-1,0);e.lookAt(new n(1,0,0));this.add(e);var f=new aa(90,1,a,b);f.up.set(0,
-1,0);f.lookAt(new n(-1,0,0));this.add(f);var g=new aa(90,1,a,b);g.up.set(0,0,1);g.lookAt(new n(0,1,0));this.add(g);var h=new aa(90,1,a,b);h.up.set(0,0,-1);h.lookAt(new n(0,-1,0));this.add(h);var l=new aa(90,1,a,b);l.up.set(0,-1,0);l.lookAt(new n(0,0,1));this.add(l);var m=new aa(90,1,a,b);m.up.set(0,-1,0);m.lookAt(new n(0,0,-1));this.add(m);d=d||{format:1022,magFilter:1006,minFilter:1006};this.renderTarget=new Db(c,d);this.renderTarget.texture.name=&quot;CubeCamera&quot;;this.update=function(a,b){null===this.parent&&
this.updateMatrixWorld();var c=a.getRenderTarget(),d=this.renderTarget,p=d.texture.generateMipmaps;d.texture.generateMipmaps=!1;a.setRenderTarget(d,0);a.render(b,e);a.setRenderTarget(d,1);a.render(b,f);a.setRenderTarget(d,2);a.render(b,g);a.setRenderTarget(d,3);a.render(b,h);a.setRenderTarget(d,4);a.render(b,l);d.texture.generateMipmaps=p;a.setRenderTarget(d,5);a.render(b,m);a.setRenderTarget(c)};this.clear=function(a,b,c,d){for(var e=a.getRenderTarget(),f=this.renderTarget,g=0;6>g;g++)a.setRenderTarget(f,
g),a.clear(b,c,d);a.setRenderTarget(e)}}function Db(a,b,c){Number.isInteger(b)&&(console.warn(&quot;THREE.WebGLCubeRenderTarget: constructor signature is now WebGLCubeRenderTarget( size, options )&quot;),b=c);Ha.call(this,a,a,b)}function ac(a,b,c,d,e,f,g,h,l,m,u,p){V.call(this,null,f,g,h,l,m,d,e,u,p);this.image={data:a||null,width:b||1,height:c||1};this.magFilter=void 0!==l?l:1003;this.minFilter=void 0!==m?m:1003;this.flipY=this.generateMipmaps=!1;this.unpackAlignment=1;this.needsUpdate=!0}function Hc(a,b,
c,d,e,f){this.planes=[void 0!==a?a:new Ta,void 0!==b?b:new Ta,void 0!==c?c:new Ta,void 0!==d?d:new Ta,void 0!==e?e:new Ta,void 0!==f?f:new Ta]}function Ah(){function a(e,f){!1!==c&&(d(e,f),b.requestAnimationFrame(a))}var b=null,c=!1,d=null;return{start:function(){!0!==c&&null!==d&&(b.requestAnimationFrame(a),c=!0)},stop:function(){c=!1},setAnimationLoop:function(a){d=a},setContext:function(a){b=a}}}function uj(a,b){function c(b,c){var d=b.array,e=b.usage,f=a.createBuffer();a.bindBuffer(c,f);a.bufferData(c,
d,e);b.onUploadCallback();c=5126;d instanceof Float32Array?c=5126:d instanceof Float64Array?console.warn(&quot;THREE.WebGLAttributes: Unsupported data buffer format: Float64Array.&quot;):d instanceof Uint16Array?c=5123:d instanceof Int16Array?c=5122:d instanceof Uint32Array?c=5125:d instanceof Int32Array?c=5124:d instanceof Int8Array?c=5120:d instanceof Uint8Array&&(c=5121);return{buffer:f,type:c,bytesPerElement:d.BYTES_PER_ELEMENT,version:b.version}}var d=b.isWebGL2,e=new WeakMap;return{get:function(a){a.isInterleavedBufferAttribute&&
(a=a.data);return e.get(a)},remove:function(b){b.isInterleavedBufferAttribute&&(b=b.data);var c=e.get(b);c&&(a.deleteBuffer(c.buffer),e.delete(b))},update:function(b,g){b.isInterleavedBufferAttribute&&(b=b.data);var f=e.get(b);if(void 0===f)e.set(b,c(b,g));else if(f.version<b.version){var l=b.array,m=b.updateRange;a.bindBuffer(g,f.buffer);-1===m.count?a.bufferSubData(g,0,l):(d?a.bufferSubData(g,m.offset*l.BYTES_PER_ELEMENT,l,m.offset,m.count):a.bufferSubData(g,m.offset*l.BYTES_PER_ELEMENT,l.subarray(m.offset,
m.offset+m.count)),m.count=-1);f.version=b.version}}}}function Id(a,b,c,d){N.call(this);this.type=&quot;PlaneGeometry&quot;;this.parameters={width:a,height:b,widthSegments:c,heightSegments:d};this.fromBufferGeometry(new bc(a,b,c,d));this.mergeVertices()}function bc(a,b,c,d){C.call(this);this.type=&quot;PlaneBufferGeometry&quot;;this.parameters={width:a,height:b,widthSegments:c,heightSegments:d};a=a||1;b=b||1;var e=a/2,f=b/2;c=Math.floor(c)||1;d=Math.floor(d)||1;var g=c+1,h=d+1,l=a/c,m=b/d,u=[],p=[],k=[],r=[];for(a=0;a<
h;a++){var q=a*m-f;for(b=0;b<g;b++)p.push(b*l-e,-q,0),k.push(0,0,1),r.push(b/c),r.push(1-a/d)}for(a=0;a<d;a++)for(b=0;b<c;b++)e=b+g*(a+1),f=b+1+g*(a+1),h=b+1+g*a,u.push(b+g*a,e,h),u.push(e,f,h);this.setIndex(u);this.setAttribute(&quot;position&quot;,new y(p,3));this.setAttribute(&quot;normal&quot;,new y(k,3));this.setAttribute(&quot;uv&quot;,new y(r,2))}function vj(a,b,c,d){function e(a,c){b.buffers.color.setClear(a.r,a.g,a.b,c,d)}var f=new A(0),g=0,h,l,m=null,u=0,p=null;return{getClearColor:function(){return f},setClearColor:function(a,
b){f.set(a);g=void 0!==b?b:1;e(f,g)},getClearAlpha:function(){return g},setClearAlpha:function(a){g=a;e(f,g)},render:function(b,d,k,v){d=d.background;k=a.xr;(k=k.getSession&&k.getSession())&&&quot;additive&quot;===k.environmentBlendMode&&(d=null);null===d?e(f,g):d&&d.isColor&&(e(d,1),v=!0);(a.autoClear||v)&&a.clear(a.autoClearColor,a.autoClearDepth,a.autoClearStencil);if(d&&(d.isCubeTexture||d.isWebGLCubeRenderTarget||306===d.mapping)){void 0===l&&(l=new S(new Jd(1,1,1),new Ba({type:&quot;BackgroundCubeMaterial&quot;,
uniforms:Fc(eb.cube.uniforms),vertexShader:eb.cube.vertexShader,fragmentShader:eb.cube.fragmentShader,side:1,depthTest:!1,depthWrite:!1,fog:!1})),l.geometry.deleteAttribute(&quot;normal&quot;),l.geometry.deleteAttribute(&quot;uv&quot;),l.onBeforeRender=function(a,b,c){this.matrixWorld.copyPosition(c.matrixWorld)},Object.defineProperty(l.material,&quot;envMap&quot;,{get:function(){return this.uniforms.envMap.value}}),c.update(l));v=d.isWebGLCubeRenderTarget?d.texture:d;l.material.uniforms.envMap.value=v;l.material.uniforms.flipEnvMap.value=
v.isCubeTexture?-1:1;if(m!==d||u!==v.version||p!==a.toneMapping)l.material.needsUpdate=!0,m=d,u=v.version,p=a.toneMapping;b.unshift(l,l.geometry,l.material,0,0,null)}else if(d&&d.isTexture){void 0===h&&(h=new S(new bc(2,2),new Ba({type:&quot;BackgroundMaterial&quot;,uniforms:Fc(eb.background.uniforms),vertexShader:eb.background.vertexShader,fragmentShader:eb.background.fragmentShader,side:0,depthTest:!1,depthWrite:!1,fog:!1})),h.geometry.deleteAttribute(&quot;normal&quot;),Object.defineProperty(h.material,&quot;map&quot;,{get:function(){return this.uniforms.t2D.value}}),
c.update(h));h.material.uniforms.t2D.value=d;!0===d.matrixAutoUpdate&&d.updateMatrix();h.material.uniforms.uvTransform.value.copy(d.matrix);if(m!==d||u!==d.version||p!==a.toneMapping)h.material.needsUpdate=!0,m=d,u=d.version,p=a.toneMapping;b.unshift(h,h.geometry,h.material,0,0,null)}}}}function wj(a,b,c,d){var e=d.isWebGL2,f;this.setMode=function(a){f=a};this.render=function(b,d){a.drawArrays(f,b,d);c.update(d,f)};this.renderInstances=function(d,h,l,m){if(0!==m){if(e){d=a;var g=&quot;drawArraysInstanced&quot;}else if(d=
b.get(&quot;ANGLE_instanced_arrays&quot;),g=&quot;drawArraysInstancedANGLE&quot;,null===d){console.error(&quot;THREE.WebGLBufferRenderer: using THREE.InstancedBufferGeometry but hardware does not support extension ANGLE_instanced_arrays.&quot;);return}d[g](f,h,l,m);c.update(l,f,m)}}}function xj(a,b,c){function d(b){if(&quot;highp&quot;===b){if(0<a.getShaderPrecisionFormat(35633,36338).precision&&0<a.getShaderPrecisionFormat(35632,36338).precision)return&quot;highp&quot;;b=&quot;mediump&quot;}return&quot;mediump&quot;===b&&0<a.getShaderPrecisionFormat(35633,36337).precision&&
0<a.getShaderPrecisionFormat(35632,36337).precision?&quot;mediump&quot;:&quot;lowp&quot;}var e,f=&quot;undefined&quot;!==typeof WebGL2RenderingContext&&a instanceof WebGL2RenderingContext||&quot;undefined&quot;!==typeof WebGL2ComputeRenderingContext&&a instanceof WebGL2ComputeRenderingContext,g=void 0!==c.precision?c.precision:&quot;highp&quot;,h=d(g);h!==g&&(console.warn(&quot;THREE.WebGLRenderer:&quot;,g,&quot;not supported, using&quot;,h,&quot;instead.&quot;),g=h);c=!0===c.logarithmicDepthBuffer;h=a.getParameter(34930);var l=a.getParameter(35660),m=a.getParameter(3379),u=
a.getParameter(34076),p=a.getParameter(34921),k=a.getParameter(36347),r=a.getParameter(36348),q=a.getParameter(36349),v=0<l,n=f||!!b.get(&quot;OES_texture_float&quot;),w=v&&n,z=f?a.getParameter(36183):0;return{isWebGL2:f,getMaxAnisotropy:function(){if(void 0!==e)return e;var c=b.get(&quot;EXT_texture_filter_anisotropic&quot;);return e=null!==c?a.getParameter(c.MAX_TEXTURE_MAX_ANISOTROPY_EXT):0},getMaxPrecision:d,precision:g,logarithmicDepthBuffer:c,maxTextures:h,maxVertexTextures:l,maxTextureSize:m,maxCubemapSize:u,
maxAttributes:p,maxVertexUniforms:k,maxVaryings:r,maxFragmentUniforms:q,vertexTextures:v,floatFragmentTextures:n,floatVertexTextures:w,maxSamples:z}}function yj(){function a(){m.value!==d&&(m.value=d,m.needsUpdate=0<e);c.numPlanes=e;c.numIntersection=0}function b(a,b,d,e){var f=null!==a?a.length:0,g=null;if(0!==f){g=m.value;if(!0!==e||null===g){e=d+4*f;b=b.matrixWorldInverse;l.getNormalMatrix(b);if(null===g||g.length<e)g=new Float32Array(e);for(e=0;e!==f;++e,d+=4)h.copy(a[e]).applyMatrix4(b,l),h.normal.toArray(g,
d),g[d+3]=h.constant}m.value=g;m.needsUpdate=!0}c.numPlanes=f;c.numIntersection=0;return g}var c=this,d=null,e=0,f=!1,g=!1,h=new Ta,l=new wa,m={value:null,needsUpdate:!1};this.uniform=m;this.numIntersection=this.numPlanes=0;this.init=function(a,c,g){var h=0!==a.length||c||0!==e||f;f=c;d=b(a,g,0);e=a.length;return h};this.beginShadows=function(){g=!0;b(null)};this.endShadows=function(){g=!1;a()};this.setState=function(c,h,l,k,q,v){if(!f||null===c||0===c.length||g&&!l)g?b(null):a();else{l=g?0:e;var p=
4*l,u=q.clippingState||null;m.value=u;u=b(c,k,p,v);for(c=0;c!==p;++c)u[c]=d[c];q.clippingState=u;this.numIntersection=h?this.numPlanes:0;this.numPlanes+=l}}}function zj(a){var b={};return{get:function(c){if(void 0!==b[c])return b[c];switch(c){case &quot;WEBGL_depth_texture&quot;:var d=a.getExtension(&quot;WEBGL_depth_texture&quot;)||a.getExtension(&quot;MOZ_WEBGL_depth_texture&quot;)||a.getExtension(&quot;WEBKIT_WEBGL_depth_texture&quot;);break;case &quot;EXT_texture_filter_anisotropic&quot;:d=a.getExtension(&quot;EXT_texture_filter_anisotropic&quot;)||a.getExtension(&quot;MOZ_EXT_texture_filter_anisotropic&quot;)||
a.getExtension(&quot;WEBKIT_EXT_texture_filter_anisotropic&quot;);break;case &quot;WEBGL_compressed_texture_s3tc&quot;:d=a.getExtension(&quot;WEBGL_compressed_texture_s3tc&quot;)||a.getExtension(&quot;MOZ_WEBGL_compressed_texture_s3tc&quot;)||a.getExtension(&quot;WEBKIT_WEBGL_compressed_texture_s3tc&quot;);break;case &quot;WEBGL_compressed_texture_pvrtc&quot;:d=a.getExtension(&quot;WEBGL_compressed_texture_pvrtc&quot;)||a.getExtension(&quot;WEBKIT_WEBGL_compressed_texture_pvrtc&quot;);break;default:d=a.getExtension(c)}null===d&&console.warn(&quot;THREE.WebGLRenderer: &quot;+c+&quot; extension not supported.&quot;);
return b[c]=d}}}function Aj(a,b,c){function d(a){var e=a.target;a=f.get(e);null!==a.index&&b.remove(a.index);for(var h in a.attributes)b.remove(a.attributes[h]);e.removeEventListener(&quot;dispose&quot;,d);f.delete(e);if(h=g.get(a))b.remove(h),g.delete(a);c.memory.geometries--}function e(a){var c=[],d=a.index,e=a.attributes.position;if(null!==d){var f=d.array;d=d.version;e=0;for(var h=f.length;e<h;e+=3){var k=f[e+0],q=f[e+1],v=f[e+2];c.push(k,q,q,v,v,k)}}else for(f=e.array,d=e.version,e=0,h=f.length/3-1;e<
h;e+=3)k=e+0,q=e+1,v=e+2,c.push(k,q,q,v,v,k);c=new (65535<yh(c)?Xb:Wb)(c,1);c.version=d;b.update(c,34963);(f=g.get(a))&&b.remove(f);g.set(a,c)}var f=new WeakMap,g=new WeakMap;return{get:function(a,b){var e=f.get(b);if(e)return e;b.addEventListener(&quot;dispose&quot;,d);b.isBufferGeometry?e=b:b.isGeometry&&(void 0===b._bufferGeometry&&(b._bufferGeometry=(new C).setFromObject(a)),e=b._bufferGeometry);f.set(b,e);c.memory.geometries++;return e},update:function(a){var c=a.index,d=a.attributes;null!==c&&b.update(c,
34963);for(var e in d)b.update(d[e],34962);a=a.morphAttributes;for(e in a){c=a[e];d=0;for(var f=c.length;d<f;d++)b.update(c[d],34962)}},getWireframeAttribute:function(a){var b=g.get(a);if(b){var c=a.index;null!==c&&b.version<c.version&&e(a)}else e(a);return g.get(a)}}}function Bj(a,b,c,d){var e=d.isWebGL2,f,g,h;this.setMode=function(a){f=a};this.setIndex=function(a){g=a.type;h=a.bytesPerElement};this.render=function(b,d){a.drawElements(f,d,g,b*h);c.update(d,f)};this.renderInstances=function(d,m,u,
p){if(0!==p){if(e){d=a;var l=&quot;drawElementsInstanced&quot;}else if(d=b.get(&quot;ANGLE_instanced_arrays&quot;),l=&quot;drawElementsInstancedANGLE&quot;,null===d){console.error(&quot;THREE.WebGLIndexedBufferRenderer: using THREE.InstancedBufferGeometry but hardware does not support extension ANGLE_instanced_arrays.&quot;);return}d[l](f,u,g,m*h,p);c.update(u,f,p)}}}function Cj(a){var b={frame:0,calls:0,triangles:0,points:0,lines:0};return{memory:{geometries:0,textures:0},render:b,programs:null,autoReset:!0,reset:function(){b.frame++;
b.calls=0;b.triangles=0;b.points=0;b.lines=0},update:function(a,d,e){e=e||1;b.calls++;switch(d){case 4:b.triangles+=a/3*e;break;case 1:b.lines+=a/2*e;break;case 3:b.lines+=e*(a-1);break;case 2:b.lines+=e*a;break;case 0:b.points+=e*a;break;default:console.error(&quot;THREE.WebGLInfo: Unknown draw mode:&quot;,d)}}}}function Dj(a,b){return Math.abs(b[1])-Math.abs(a[1])}function Ej(a){var b={},c=new Float32Array(8);return{update:function(d,e,f,g){var h=d.morphTargetInfluences,l=void 0===h?0:h.length;d=b[e.id];
if(void 0===d){d=[];for(var m=0;m<l;m++)d[m]=[m,0];b[e.id]=d}var u=f.morphTargets&&e.morphAttributes.position;f=f.morphNormals&&e.morphAttributes.normal;for(m=0;m<l;m++){var p=d[m];0!==p[1]&&(u&&e.deleteAttribute(&quot;morphTarget&quot;+m),f&&e.deleteAttribute(&quot;morphNormal&quot;+m))}for(m=0;m<l;m++)p=d[m],p[0]=m,p[1]=h[m];d.sort(Dj);for(m=h=0;8>m;m++){if(p=d[m])if(l=p[0],p=p[1]){u&&e.setAttribute(&quot;morphTarget&quot;+m,u[l]);f&&e.setAttribute(&quot;morphNormal&quot;+m,f[l]);c[m]=p;h+=p;continue}c[m]=0}e=e.morphTargetsRelative?1:
1-h;g.getUniforms().setValue(a,&quot;morphTargetBaseInfluence&quot;,e);g.getUniforms().setValue(a,&quot;morphTargetInfluences&quot;,c)}}}function Fj(a,b,c,d){var e=new WeakMap;return{update:function(a){var f=d.render.frame,h=a.geometry,l=b.get(a,h);e.get(l)!==f&&(h.isGeometry&&l.updateFromObject(a),b.update(l),e.set(l,f));a.isInstancedMesh&&c.update(a.instanceMatrix,34962);return l},dispose:function(){e=new WeakMap}}}function qb(a,b,c,d,e,f,g,h,l,m){a=void 0!==a?a:[];V.call(this,a,void 0!==b?b:301,c,d,e,f,void 0!==g?
g:1022,h,l,m);this.flipY=!1}function Ic(a,b,c,d){V.call(this,null);this.image={data:a||null,width:b||1,height:c||1,depth:d||1};this.minFilter=this.magFilter=1003;this.wrapR=1001;this.flipY=this.generateMipmaps=!1;this.needsUpdate=!0}function Jc(a,b,c,d){V.call(this,null);this.image={data:a||null,width:b||1,height:c||1,depth:d||1};this.minFilter=this.magFilter=1003;this.wrapR=1001;this.flipY=this.generateMipmaps=!1;this.needsUpdate=!0}function Kc(a,b,c){var d=a[0];if(0>=d||0<d)return a;var e=b*c,f=
Bh[e];void 0===f&&(f=new Float32Array(e),Bh[e]=f);if(0!==b)for(d.toArray(f,0),d=1,e=0;d!==b;++d)e+=c,a[d].toArray(f,e);return f}function Pa(a,b){if(a.length!==b.length)return!1;for(var c=0,d=a.length;c<d;c++)if(a[c]!==b[c])return!1;return!0}function Ia(a,b){for(var c=0,d=b.length;c<d;c++)a[c]=b[c]}function Ch(a,b){var c=Dh[b];void 0===c&&(c=new Int32Array(b),Dh[b]=c);for(var d=0;d!==b;++d)c[d]=a.allocateTextureUnit();return c}function Gj(a,b){var c=this.cache;c[0]!==b&&(a.uniform1f(this.addr,b),c[0]=
b)}function Hj(a,b){var c=this.cache;if(void 0!==b.x){if(c[0]!==b.x||c[1]!==b.y)a.uniform2f(this.addr,b.x,b.y),c[0]=b.x,c[1]=b.y}else Pa(c,b)||(a.uniform2fv(this.addr,b),Ia(c,b))}function Ij(a,b){var c=this.cache;if(void 0!==b.x){if(c[0]!==b.x||c[1]!==b.y||c[2]!==b.z)a.uniform3f(this.addr,b.x,b.y,b.z),c[0]=b.x,c[1]=b.y,c[2]=b.z}else if(void 0!==b.r){if(c[0]!==b.r||c[1]!==b.g||c[2]!==b.b)a.uniform3f(this.addr,b.r,b.g,b.b),c[0]=b.r,c[1]=b.g,c[2]=b.b}else Pa(c,b)||(a.uniform3fv(this.addr,b),Ia(c,b))}
function Jj(a,b){var c=this.cache;if(void 0!==b.x){if(c[0]!==b.x||c[1]!==b.y||c[2]!==b.z||c[3]!==b.w)a.uniform4f(this.addr,b.x,b.y,b.z,b.w),c[0]=b.x,c[1]=b.y,c[2]=b.z,c[3]=b.w}else Pa(c,b)||(a.uniform4fv(this.addr,b),Ia(c,b))}function Kj(a,b){var c=this.cache,d=b.elements;void 0===d?Pa(c,b)||(a.uniformMatrix2fv(this.addr,!1,b),Ia(c,b)):Pa(c,d)||(Eh.set(d),a.uniformMatrix2fv(this.addr,!1,Eh),Ia(c,d))}function Lj(a,b){var c=this.cache,d=b.elements;void 0===d?Pa(c,b)||(a.uniformMatrix3fv(this.addr,!1,
b),Ia(c,b)):Pa(c,d)||(Fh.set(d),a.uniformMatrix3fv(this.addr,!1,Fh),Ia(c,d))}function Mj(a,b){var c=this.cache,d=b.elements;void 0===d?Pa(c,b)||(a.uniformMatrix4fv(this.addr,!1,b),Ia(c,b)):Pa(c,d)||(Gh.set(d),a.uniformMatrix4fv(this.addr,!1,Gh),Ia(c,d))}function Nj(a,b,c){var d=this.cache,e=c.allocateTextureUnit();d[0]!==e&&(a.uniform1i(this.addr,e),d[0]=e);c.safeSetTexture2D(b||Hh,e)}function Oj(a,b,c){var d=this.cache,e=c.allocateTextureUnit();d[0]!==e&&(a.uniform1i(this.addr,e),d[0]=e);c.setTexture2DArray(b||
Pj,e)}function Qj(a,b,c){var d=this.cache,e=c.allocateTextureUnit();d[0]!==e&&(a.uniform1i(this.addr,e),d[0]=e);c.setTexture3D(b||Rj,e)}function Sj(a,b,c){var d=this.cache,e=c.allocateTextureUnit();d[0]!==e&&(a.uniform1i(this.addr,e),d[0]=e);c.safeSetTextureCube(b||Ih,e)}function Tj(a,b){var c=this.cache;c[0]!==b&&(a.uniform1i(this.addr,b),c[0]=b)}function Uj(a,b){var c=this.cache;Pa(c,b)||(a.uniform2iv(this.addr,b),Ia(c,b))}function Vj(a,b){var c=this.cache;Pa(c,b)||(a.uniform3iv(this.addr,b),Ia(c,
b))}function Wj(a,b){var c=this.cache;Pa(c,b)||(a.uniform4iv(this.addr,b),Ia(c,b))}function Xj(a,b){var c=this.cache;c[0]!==b&&(a.uniform1ui(this.addr,b),c[0]=b)}function Yj(a){switch(a){case 5126:return Gj;case 35664:return Hj;case 35665:return Ij;case 35666:return Jj;case 35674:return Kj;case 35675:return Lj;case 35676:return Mj;case 5124:case 35670:return Tj;case 35667:case 35671:return Uj;case 35668:case 35672:return Vj;case 35669:case 35673:return Wj;case 5125:return Xj;case 35678:case 36198:case 36298:case 36306:case 35682:return Nj;
case 35679:case 36299:case 36307:return Qj;case 35680:case 36300:case 36308:case 36293:return Sj;case 36289:case 36303:case 36311:case 36292:return Oj}}function Zj(a,b){a.uniform1fv(this.addr,b)}function ak(a,b){a.uniform1iv(this.addr,b)}function bk(a,b){a.uniform2iv(this.addr,b)}function ck(a,b){a.uniform3iv(this.addr,b)}function dk(a,b){a.uniform4iv(this.addr,b)}function ek(a,b){b=Kc(b,this.size,2);a.uniform2fv(this.addr,b)}function fk(a,b){b=Kc(b,this.size,3);a.uniform3fv(this.addr,b)}function gk(a,
b){b=Kc(b,this.size,4);a.uniform4fv(this.addr,b)}function hk(a,b){b=Kc(b,this.size,4);a.uniformMatrix2fv(this.addr,!1,b)}function ik(a,b){b=Kc(b,this.size,9);a.uniformMatrix3fv(this.addr,!1,b)}function jk(a,b){b=Kc(b,this.size,16);a.uniformMatrix4fv(this.addr,!1,b)}function kk(a,b,c){var d=b.length,e=Ch(c,d);a.uniform1iv(this.addr,e);for(a=0;a!==d;++a)c.safeSetTexture2D(b[a]||Hh,e[a])}function lk(a,b,c){var d=b.length,e=Ch(c,d);a.uniform1iv(this.addr,e);for(a=0;a!==d;++a)c.safeSetTextureCube(b[a]||
Ih,e[a])}function mk(a){switch(a){case 5126:return Zj;case 35664:return ek;case 35665:return fk;case 35666:return gk;case 35674:return hk;case 35675:return ik;case 35676:return jk;case 5124:case 35670:return ak;case 35667:case 35671:return bk;case 35668:case 35672:return ck;case 35669:case 35673:return dk;case 35678:case 36198:case 36298:case 36306:case 35682:return kk;case 35680:case 36300:case 36308:case 36293:return lk}}function nk(a,b,c){this.id=a;this.addr=c;this.cache=[];this.setValue=Yj(b.type)}
function Jh(a,b,c){this.id=a;this.addr=c;this.cache=[];this.size=b.size;this.setValue=mk(b.type)}function Kh(a){this.id=a;this.seq=[];this.map={}}function Eb(a,b){this.seq=[];this.map={};for(var c=a.getProgramParameter(b,35718),d=0;d<c;++d){var e=a.getActiveUniform(b,d),f=a.getUniformLocation(b,e.name),g=this,h=e.name,l=h.length;for(gg.lastIndex=0;;){var m=gg.exec(h),u=gg.lastIndex,p=m[1],k=m[3];&quot;]&quot;===m[2]&&(p|=0);if(void 0===k||&quot;[&quot;===k&&u+2===l){h=g;e=void 0===k?new nk(p,e,f):new Jh(p,e,f);h.seq.push(e);
h.map[e.id]=e;break}else k=g.map[p],void 0===k&&(k=new Kh(p),p=g,g=k,p.seq.push(g),p.map[g.id]=g),g=k}}}function Lh(a,b,c){b=a.createShader(b);a.shaderSource(b,c);a.compileShader(b);return b}function Mh(a){switch(a){case 3E3:return[&quot;Linear&quot;,&quot;( value )&quot;];case 3001:return[&quot;sRGB&quot;,&quot;( value )&quot;];case 3002:return[&quot;RGBE&quot;,&quot;( value )&quot;];case 3004:return[&quot;RGBM&quot;,&quot;( value, 7.0 )&quot;];case 3005:return[&quot;RGBM&quot;,&quot;( value, 16.0 )&quot;];case 3006:return[&quot;RGBD&quot;,&quot;( value, 256.0 )&quot;];case 3007:return[&quot;Gamma&quot;,&quot;( value, float( GAMMA_FACTOR ) )&quot;];
case 3003:return[&quot;LogLuv&quot;,&quot;( value )&quot;];default:throw Error(&quot;unsupported encoding: &quot;+a);}}function Nh(a,b,c){var d=a.getShaderParameter(b,35713),e=a.getShaderInfoLog(b).trim();if(d&&&quot;&quot;===e)return&quot;&quot;;a=a.getShaderSource(b).split(&quot;\n&quot;);for(b=0;b<a.length;b++)a[b]=b+1+&quot;: &quot;+a[b];a=a.join(&quot;\n&quot;);return&quot;THREE.WebGLShader: gl.getShaderInfoLog() &quot;+c+&quot;\n&quot;+e+a}function Kd(a,b){b=Mh(b);return&quot;vec4 &quot;+a+&quot;( vec4 value ) { return &quot;+b[0]+&quot;ToLinear&quot;+b[1]+&quot;; }&quot;}function ok(a,b){b=Mh(b);return&quot;vec4 &quot;+a+&quot;( vec4 value ) { return LinearTo&quot;+
b[0]+b[1]+&quot;; }&quot;}function pk(a,b){switch(b){case 1:b=&quot;Linear&quot;;break;case 2:b=&quot;Reinhard&quot;;break;case 3:b=&quot;Uncharted2&quot;;break;case 4:b=&quot;OptimizedCineon&quot;;break;case 5:b=&quot;ACESFilmic&quot;;break;default:throw Error(&quot;unsupported toneMapping: &quot;+b);}return&quot;vec3 &quot;+a+&quot;( vec3 color ) { return &quot;+b+&quot;ToneMapping( color ); }&quot;}function qk(a){var b=[],c;for(c in a){var d=a[c];!1!==d&&b.push(&quot;#define &quot;+c+&quot; &quot;+d)}return b.join(&quot;\n&quot;)}function Ld(a){return&quot;&quot;!==a}function Oh(a,b){return a.replace(/NUM_DIR_LIGHTS/g,b.numDirLights).replace(/NUM_SPOT_LIGHTS/g,
b.numSpotLights).replace(/NUM_RECT_AREA_LIGHTS/g,b.numRectAreaLights).replace(/NUM_POINT_LIGHTS/g,b.numPointLights).replace(/NUM_HEMI_LIGHTS/g,b.numHemiLights).replace(/NUM_DIR_LIGHT_SHADOWS/g,b.numDirLightShadows).replace(/NUM_SPOT_LIGHT_SHADOWS/g,b.numSpotLightShadows).replace(/NUM_POINT_LIGHT_SHADOWS/g,b.numPointLightShadows)}function Ph(a,b){return a.replace(/NUM_CLIPPING_PLANES/g,b.numClippingPlanes).replace(/UNION_CLIPPING_PLANES/g,b.numClippingPlanes-b.numClipIntersection)}function hg(a,b){a=
O[b];if(void 0===a)throw Error(&quot;Can not resolve #include <&quot;+b+&quot;>&quot;);return a.replace(ig,hg)}function Qh(a,b,c,d){console.warn(&quot;WebGLProgram: #pragma unroll_loop shader syntax is deprecated. Please use #pragma unroll_loop_start syntax instead.&quot;);return jg(a,b,c,d)}function jg(a,b,c,d){a=&quot;&quot;;for(b=parseInt(b);b<parseInt(c);b++)a+=d.replace(/\[ i \]/g,&quot;[ &quot;+b+&quot; ]&quot;).replace(/UNROLLED_LOOP_INDEX/g,b);return a}function Rh(a){var b=&quot;precision &quot;+a.precision+&quot; float;\nprecision &quot;+a.precision+&quot; int;&quot;;&quot;highp&quot;===
a.precision?b+=&quot;\n#define HIGH_PRECISION&quot;:&quot;mediump&quot;===a.precision?b+=&quot;\n#define MEDIUM_PRECISION&quot;:&quot;lowp&quot;===a.precision&&(b+=&quot;\n#define LOW_PRECISION&quot;);return b}function rk(a){var b=&quot;SHADOWMAP_TYPE_BASIC&quot;;1===a.shadowMapType?b=&quot;SHADOWMAP_TYPE_PCF&quot;:2===a.shadowMapType?b=&quot;SHADOWMAP_TYPE_PCF_SOFT&quot;:3===a.shadowMapType&&(b=&quot;SHADOWMAP_TYPE_VSM&quot;);return b}function sk(a){var b=&quot;ENVMAP_TYPE_CUBE&quot;;if(a.envMap)switch(a.envMapMode){case 301:case 302:b=&quot;ENVMAP_TYPE_CUBE&quot;;break;case 306:case 307:b=&quot;ENVMAP_TYPE_CUBE_UV&quot;;
break;case 303:case 304:b=&quot;ENVMAP_TYPE_EQUIREC&quot;;break;case 305:b=&quot;ENVMAP_TYPE_SPHERE&quot;}return b}function tk(a){var b=&quot;ENVMAP_MODE_REFLECTION&quot;;if(a.envMap)switch(a.envMapMode){case 302:case 304:b=&quot;ENVMAP_MODE_REFRACTION&quot;}return b}function uk(a){var b=&quot;ENVMAP_BLENDING_NONE&quot;;if(a.envMap)switch(a.combine){case 0:b=&quot;ENVMAP_BLENDING_MULTIPLY&quot;;break;case 1:b=&quot;ENVMAP_BLENDING_MIX&quot;;break;case 2:b=&quot;ENVMAP_BLENDING_ADD&quot;}return b}function vk(a,b,c){var d=a.getContext(),e=c.defines,f=c.vertexShader,g=c.fragmentShader,
h=rk(c),l=sk(c),m=tk(c),u=uk(c),p=0<a.gammaFactor?a.gammaFactor:1,k=c.isWebGL2?&quot;&quot;:[c.extensionDerivatives||c.envMapCubeUV||c.bumpMap||c.tangentSpaceNormalMap||c.clearcoatNormalMap||c.flatShading||&quot;physical&quot;===c.shaderID?&quot;#extension GL_OES_standard_derivatives : enable&quot;:&quot;&quot;,(c.extensionFragDepth||c.logarithmicDepthBuffer)&&c.rendererExtensionFragDepth?&quot;#extension GL_EXT_frag_depth : enable&quot;:&quot;&quot;,c.extensionDrawBuffers&&c.rendererExtensionDrawBuffers?&quot;#extension GL_EXT_draw_buffers : require&quot;:&quot;&quot;,(c.extensionShaderTextureLOD||
c.envMap)&&c.rendererExtensionShaderTextureLod?&quot;#extension GL_EXT_shader_texture_lod : enable&quot;:&quot;&quot;].filter(Ld).join(&quot;\n&quot;),r=qk(e),q=d.createProgram();c.isRawShaderMaterial?(e=[r].filter(Ld).join(&quot;\n&quot;),0<e.length&&(e+=&quot;\n&quot;),h=[k,r].filter(Ld).join(&quot;\n&quot;),0<h.length&&(h+=&quot;\n&quot;)):(e=[Rh(c),&quot;#define SHADER_NAME &quot;+c.shaderName,r,c.instancing?&quot;#define USE_INSTANCING&quot;:&quot;&quot;,c.supportsVertexTextures?&quot;#define VERTEX_TEXTURES&quot;:&quot;&quot;,&quot;#define GAMMA_FACTOR &quot;+p,&quot;#define MAX_BONES &quot;+c.maxBones,c.useFog&&c.fog?&quot;#define USE_FOG&quot;:
&quot;&quot;,c.useFog&&c.fogExp2?&quot;#define FOG_EXP2&quot;:&quot;&quot;,c.map?&quot;#define USE_MAP&quot;:&quot;&quot;,c.envMap?&quot;#define USE_ENVMAP&quot;:&quot;&quot;,c.envMap?&quot;#define &quot;+m:&quot;&quot;,c.lightMap?&quot;#define USE_LIGHTMAP&quot;:&quot;&quot;,c.aoMap?&quot;#define USE_AOMAP&quot;:&quot;&quot;,c.emissiveMap?&quot;#define USE_EMISSIVEMAP&quot;:&quot;&quot;,c.bumpMap?&quot;#define USE_BUMPMAP&quot;:&quot;&quot;,c.normalMap?&quot;#define USE_NORMALMAP&quot;:&quot;&quot;,c.normalMap&&c.objectSpaceNormalMap?&quot;#define OBJECTSPACE_NORMALMAP&quot;:&quot;&quot;,c.normalMap&&c.tangentSpaceNormalMap?&quot;#define TANGENTSPACE_NORMALMAP&quot;:&quot;&quot;,c.clearcoatMap?&quot;#define USE_CLEARCOATMAP&quot;:
&quot;&quot;,c.clearcoatRoughnessMap?&quot;#define USE_CLEARCOAT_ROUGHNESSMAP&quot;:&quot;&quot;,c.clearcoatNormalMap?&quot;#define USE_CLEARCOAT_NORMALMAP&quot;:&quot;&quot;,c.displacementMap&&c.supportsVertexTextures?&quot;#define USE_DISPLACEMENTMAP&quot;:&quot;&quot;,c.specularMap?&quot;#define USE_SPECULARMAP&quot;:&quot;&quot;,c.roughnessMap?&quot;#define USE_ROUGHNESSMAP&quot;:&quot;&quot;,c.metalnessMap?&quot;#define USE_METALNESSMAP&quot;:&quot;&quot;,c.alphaMap?&quot;#define USE_ALPHAMAP&quot;:&quot;&quot;,c.vertexTangents?&quot;#define USE_TANGENT&quot;:&quot;&quot;,c.vertexColors?&quot;#define USE_COLOR&quot;:&quot;&quot;,c.vertexUvs?&quot;#define USE_UV&quot;:&quot;&quot;,c.uvsVertexOnly?&quot;#define UVS_VERTEX_ONLY&quot;:
&quot;&quot;,c.flatShading?&quot;#define FLAT_SHADED&quot;:&quot;&quot;,c.skinning?&quot;#define USE_SKINNING&quot;:&quot;&quot;,c.useVertexTexture?&quot;#define BONE_TEXTURE&quot;:&quot;&quot;,c.morphTargets?&quot;#define USE_MORPHTARGETS&quot;:&quot;&quot;,c.morphNormals&&!1===c.flatShading?&quot;#define USE_MORPHNORMALS&quot;:&quot;&quot;,c.doubleSided?&quot;#define DOUBLE_SIDED&quot;:&quot;&quot;,c.flipSided?&quot;#define FLIP_SIDED&quot;:&quot;&quot;,c.shadowMapEnabled?&quot;#define USE_SHADOWMAP&quot;:&quot;&quot;,c.shadowMapEnabled?&quot;#define &quot;+h:&quot;&quot;,c.sizeAttenuation?&quot;#define USE_SIZEATTENUATION&quot;:&quot;&quot;,c.logarithmicDepthBuffer?&quot;#define USE_LOGDEPTHBUF&quot;:&quot;&quot;,c.logarithmicDepthBuffer&&
c.rendererExtensionFragDepth?&quot;#define USE_LOGDEPTHBUF_EXT&quot;:&quot;&quot;,&quot;uniform mat4 modelMatrix;&quot;,&quot;uniform mat4 modelViewMatrix;&quot;,&quot;uniform mat4 projectionMatrix;&quot;,&quot;uniform mat4 viewMatrix;&quot;,&quot;uniform mat3 normalMatrix;&quot;,&quot;uniform vec3 cameraPosition;&quot;,&quot;uniform bool isOrthographic;&quot;,&quot;#ifdef USE_INSTANCING&quot;,&quot; attribute mat4 instanceMatrix;&quot;,&quot;#endif&quot;,&quot;attribute vec3 position;&quot;,&quot;attribute vec3 normal;&quot;,&quot;attribute vec2 uv;&quot;,&quot;#ifdef USE_TANGENT&quot;,&quot;\tattribute vec4 tangent;&quot;,&quot;#endif&quot;,&quot;#ifdef USE_COLOR&quot;,&quot;\tattribute vec3 color;&quot;,
&quot;#endif&quot;,&quot;#ifdef USE_MORPHTARGETS&quot;,&quot;\tattribute vec3 morphTarget0;&quot;,&quot;\tattribute vec3 morphTarget1;&quot;,&quot;\tattribute vec3 morphTarget2;&quot;,&quot;\tattribute vec3 morphTarget3;&quot;,&quot;\t#ifdef USE_MORPHNORMALS&quot;,&quot;\t\tattribute vec3 morphNormal0;&quot;,&quot;\t\tattribute vec3 morphNormal1;&quot;,&quot;\t\tattribute vec3 morphNormal2;&quot;,&quot;\t\tattribute vec3 morphNormal3;&quot;,&quot;\t#else&quot;,&quot;\t\tattribute vec3 morphTarget4;&quot;,&quot;\t\tattribute vec3 morphTarget5;&quot;,&quot;\t\tattribute vec3 morphTarget6;&quot;,&quot;\t\tattribute vec3 morphTarget7;&quot;,&quot;\t#endif&quot;,&quot;#endif&quot;,
&quot;#ifdef USE_SKINNING&quot;,&quot;\tattribute vec4 skinIndex;&quot;,&quot;\tattribute vec4 skinWeight;&quot;,&quot;#endif&quot;,&quot;\n&quot;].filter(Ld).join(&quot;\n&quot;),h=[k,Rh(c),&quot;#define SHADER_NAME &quot;+c.shaderName,r,c.alphaTest?&quot;#define ALPHATEST &quot;+c.alphaTest+(c.alphaTest%1?&quot;&quot;:&quot;.0&quot;):&quot;&quot;,&quot;#define GAMMA_FACTOR &quot;+p,c.useFog&&c.fog?&quot;#define USE_FOG&quot;:&quot;&quot;,c.useFog&&c.fogExp2?&quot;#define FOG_EXP2&quot;:&quot;&quot;,c.map?&quot;#define USE_MAP&quot;:&quot;&quot;,c.matcap?&quot;#define USE_MATCAP&quot;:&quot;&quot;,c.envMap?&quot;#define USE_ENVMAP&quot;:&quot;&quot;,c.envMap?&quot;#define &quot;+l:&quot;&quot;,c.envMap?&quot;#define &quot;+m:&quot;&quot;,c.envMap?&quot;#define &quot;+
u:&quot;&quot;,c.lightMap?&quot;#define USE_LIGHTMAP&quot;:&quot;&quot;,c.aoMap?&quot;#define USE_AOMAP&quot;:&quot;&quot;,c.emissiveMap?&quot;#define USE_EMISSIVEMAP&quot;:&quot;&quot;,c.bumpMap?&quot;#define USE_BUMPMAP&quot;:&quot;&quot;,c.normalMap?&quot;#define USE_NORMALMAP&quot;:&quot;&quot;,c.normalMap&&c.objectSpaceNormalMap?&quot;#define OBJECTSPACE_NORMALMAP&quot;:&quot;&quot;,c.normalMap&&c.tangentSpaceNormalMap?&quot;#define TANGENTSPACE_NORMALMAP&quot;:&quot;&quot;,c.clearcoatMap?&quot;#define USE_CLEARCOATMAP&quot;:&quot;&quot;,c.clearcoatRoughnessMap?&quot;#define USE_CLEARCOAT_ROUGHNESSMAP&quot;:&quot;&quot;,c.clearcoatNormalMap?&quot;#define USE_CLEARCOAT_NORMALMAP&quot;:&quot;&quot;,
c.specularMap?&quot;#define USE_SPECULARMAP&quot;:&quot;&quot;,c.roughnessMap?&quot;#define USE_ROUGHNESSMAP&quot;:&quot;&quot;,c.metalnessMap?&quot;#define USE_METALNESSMAP&quot;:&quot;&quot;,c.alphaMap?&quot;#define USE_ALPHAMAP&quot;:&quot;&quot;,c.sheen?&quot;#define USE_SHEEN&quot;:&quot;&quot;,c.vertexTangents?&quot;#define USE_TANGENT&quot;:&quot;&quot;,c.vertexColors?&quot;#define USE_COLOR&quot;:&quot;&quot;,c.vertexUvs?&quot;#define USE_UV&quot;:&quot;&quot;,c.uvsVertexOnly?&quot;#define UVS_VERTEX_ONLY&quot;:&quot;&quot;,c.gradientMap?&quot;#define USE_GRADIENTMAP&quot;:&quot;&quot;,c.flatShading?&quot;#define FLAT_SHADED&quot;:&quot;&quot;,c.doubleSided?&quot;#define DOUBLE_SIDED&quot;:&quot;&quot;,c.flipSided?&quot;#define FLIP_SIDED&quot;:
&quot;&quot;,c.shadowMapEnabled?&quot;#define USE_SHADOWMAP&quot;:&quot;&quot;,c.shadowMapEnabled?&quot;#define &quot;+h:&quot;&quot;,c.premultipliedAlpha?&quot;#define PREMULTIPLIED_ALPHA&quot;:&quot;&quot;,c.physicallyCorrectLights?&quot;#define PHYSICALLY_CORRECT_LIGHTS&quot;:&quot;&quot;,c.logarithmicDepthBuffer?&quot;#define USE_LOGDEPTHBUF&quot;:&quot;&quot;,c.logarithmicDepthBuffer&&c.rendererExtensionFragDepth?&quot;#define USE_LOGDEPTHBUF_EXT&quot;:&quot;&quot;,(c.extensionShaderTextureLOD||c.envMap)&&c.rendererExtensionShaderTextureLod?&quot;#define TEXTURE_LOD_EXT&quot;:&quot;&quot;,&quot;uniform mat4 viewMatrix;&quot;,&quot;uniform vec3 cameraPosition;&quot;,
&quot;uniform bool isOrthographic;&quot;,0!==c.toneMapping?&quot;#define TONE_MAPPING&quot;:&quot;&quot;,0!==c.toneMapping?O.tonemapping_pars_fragment:&quot;&quot;,0!==c.toneMapping?pk(&quot;toneMapping&quot;,c.toneMapping):&quot;&quot;,c.dithering?&quot;#define DITHERING&quot;:&quot;&quot;,c.outputEncoding||c.mapEncoding||c.matcapEncoding||c.envMapEncoding||c.emissiveMapEncoding||c.lightMapEncoding?O.encodings_pars_fragment:&quot;&quot;,c.mapEncoding?Kd(&quot;mapTexelToLinear&quot;,c.mapEncoding):&quot;&quot;,c.matcapEncoding?Kd(&quot;matcapTexelToLinear&quot;,c.matcapEncoding):&quot;&quot;,c.envMapEncoding?Kd(&quot;envMapTexelToLinear&quot;,
c.envMapEncoding):&quot;&quot;,c.emissiveMapEncoding?Kd(&quot;emissiveMapTexelToLinear&quot;,c.emissiveMapEncoding):&quot;&quot;,c.lightMapEncoding?Kd(&quot;lightMapTexelToLinear&quot;,c.lightMapEncoding):&quot;&quot;,c.outputEncoding?ok(&quot;linearToOutputTexel&quot;,c.outputEncoding):&quot;&quot;,c.depthPacking?&quot;#define DEPTH_PACKING &quot;+c.depthPacking:&quot;&quot;,&quot;\n&quot;].filter(Ld).join(&quot;\n&quot;));f=f.replace(ig,hg);f=Oh(f,c);f=Ph(f,c);g=g.replace(ig,hg);g=Oh(g,c);g=Ph(g,c);f=f.replace(Sh,jg).replace(Th,Qh);g=g.replace(Sh,jg).replace(Th,Qh);c.isWebGL2&&!c.isRawShaderMaterial&&(l=
!1,m=/^\s*#version\s+300\s+es\s*\n/,c.isShaderMaterial&&null!==f.match(m)&&null!==g.match(m)&&(l=!0,f=f.replace(m,&quot;&quot;),g=g.replace(m,&quot;&quot;)),e=&quot;#version 300 es\n\n#define attribute in\n#define varying out\n#define texture2D texture\n&quot;+e,h=[&quot;#version 300 es\n\n#define varying in&quot;,l?&quot;&quot;:&quot;out highp vec4 pc_fragColor;&quot;,l?&quot;&quot;:&quot;#define gl_FragColor pc_fragColor&quot;,&quot;#define gl_FragDepthEXT gl_FragDepth\n#define texture2D texture\n#define textureCube texture\n#define texture2DProj textureProj\n#define texture2DLodEXT textureLod\n#define texture2DProjLodEXT textureProjLod\n#define textureCubeLodEXT textureLod\n#define texture2DGradEXT textureGrad\n#define texture2DProjGradEXT textureProjGrad\n#define textureCubeGradEXT textureGrad&quot;].join(&quot;\n&quot;)+
&quot;\n&quot;+h);g=h+g;f=Lh(d,35633,e+f);g=Lh(d,35632,g);d.attachShader(q,f);d.attachShader(q,g);void 0!==c.index0AttributeName?d.bindAttribLocation(q,0,c.index0AttributeName):!0===c.morphTargets&&d.bindAttribLocation(q,0,&quot;position&quot;);d.linkProgram(q);if(a.debug.checkShaderErrors){a=d.getProgramInfoLog(q).trim();l=d.getShaderInfoLog(f).trim();m=d.getShaderInfoLog(g).trim();p=u=!0;if(!1===d.getProgramParameter(q,35714))u=!1,k=Nh(d,f,&quot;vertex&quot;),r=Nh(d,g,&quot;fragment&quot;),console.error(&quot;THREE.WebGLProgram: shader error: &quot;,
d.getError(),&quot;35715&quot;,d.getProgramParameter(q,35715),&quot;gl.getProgramInfoLog&quot;,a,k,r);else if(&quot;&quot;!==a)console.warn(&quot;THREE.WebGLProgram: gl.getProgramInfoLog()&quot;,a);else if(&quot;&quot;===l||&quot;&quot;===m)p=!1;p&&(this.diagnostics={runnable:u,programLog:a,vertexShader:{log:l,prefix:e},fragmentShader:{log:m,prefix:h}})}d.detachShader(q,f);d.detachShader(q,g);d.deleteShader(f);d.deleteShader(g);var v;this.getUniforms=function(){void 0===v&&(v=new Eb(d,q));return v};var n;this.getAttributes=function(){if(void 0===n){for(var a=
{},b=d.getProgramParameter(q,35721),c=0;c<b;c++){var e=d.getActiveAttrib(q,c).name;a[e]=d.getAttribLocation(q,e)}n=a}return n};this.destroy=function(){d.deleteProgram(q);this.program=void 0};this.name=c.shaderName;this.id=wk++;this.cacheKey=b;this.usedTimes=1;this.program=q;this.vertexShader=f;this.fragmentShader=g;return this}function xk(a,b,c){function d(a){if(a)a.isTexture?b=a.encoding:a.isWebGLRenderTarget&&(console.warn(&quot;THREE.WebGLPrograms.getTextureEncodingFromMap: don't use render targets as textures. Use their .texture property instead.&quot;),
b=a.texture.encoding);else var b=3E3;return b}var e=[],f=c.isWebGL2,g=c.logarithmicDepthBuffer,h=c.floatVertexTextures,l=c.precision,m=c.maxVertexUniforms,u=c.vertexTextures,p={MeshDepthMaterial:&quot;depth&quot;,MeshDistanceMaterial:&quot;distanceRGBA&quot;,MeshNormalMaterial:&quot;normal&quot;,MeshBasicMaterial:&quot;basic&quot;,MeshLambertMaterial:&quot;lambert&quot;,MeshPhongMaterial:&quot;phong&quot;,MeshToonMaterial:&quot;toon&quot;,MeshStandardMaterial:&quot;physical&quot;,MeshPhysicalMaterial:&quot;physical&quot;,MeshMatcapMaterial:&quot;matcap&quot;,LineBasicMaterial:&quot;basic&quot;,LineDashedMaterial:&quot;dashed&quot;,
PointsMaterial:&quot;points&quot;,ShadowMaterial:&quot;shadow&quot;,SpriteMaterial:&quot;sprite&quot;},k=&quot;precision isWebGL2 supportsVertexTextures outputEncoding instancing map mapEncoding matcap matcapEncoding envMap envMapMode envMapEncoding envMapCubeUV lightMap lightMapEncoding aoMap emissiveMap emissiveMapEncoding bumpMap normalMap objectSpaceNormalMap tangentSpaceNormalMap clearcoatMap clearcoatRoughnessMap clearcoatNormalMap displacementMap specularMap roughnessMap metalnessMap gradientMap alphaMap combine vertexColors vertexTangents vertexUvs uvsVertexOnly fog useFog fogExp2 flatShading sizeAttenuation logarithmicDepthBuffer skinning maxBones useVertexTexture morphTargets morphNormals maxMorphTargets maxMorphNormals premultipliedAlpha numDirLights numPointLights numSpotLights numHemiLights numRectAreaLights numDirLightShadows numPointLightShadows numSpotLightShadows shadowMapEnabled shadowMapType toneMapping physicallyCorrectLights alphaTest doubleSided flipSided numClippingPlanes numClipIntersection depthPacking dithering sheen&quot;.split(&quot; &quot;);
this.getParameters=function(e,k,x,n,w,z,ha){var q=n.fog;n=e.isMeshStandardMaterial?n.environment:null;n=e.envMap||n;var r=p[e.type];if(ha.isSkinnedMesh){var v=ha.skeleton.bones;if(h)v=1024;else{var E=Math.min(Math.floor((m-20)/4),v.length);E<v.length?(console.warn(&quot;THREE.WebGLRenderer: Skeleton has &quot;+v.length+&quot; bones. This GPU supports &quot;+E+&quot;.&quot;),v=0):v=E}}else v=0;null!==e.precision&&(l=c.getMaxPrecision(e.precision),l!==e.precision&&console.warn(&quot;THREE.WebGLProgram.getParameters:&quot;,e.precision,&quot;not supported, using&quot;,
l,&quot;instead.&quot;));r?(E=eb[r],E={name:e.type,uniforms:Uh.clone(E.uniforms),vertexShader:E.vertexShader,fragmentShader:E.fragmentShader}):E={name:e.type,uniforms:e.uniforms,vertexShader:e.vertexShader,fragmentShader:e.fragmentShader};e.onBeforeCompile(E,a);var ia=a.getRenderTarget();return{isWebGL2:f,shaderID:r,shaderName:E.name,uniforms:E.uniforms,vertexShader:E.vertexShader,fragmentShader:E.fragmentShader,defines:e.defines,isRawShaderMaterial:e.isRawShaderMaterial,isShaderMaterial:e.isShaderMaterial,
precision:l,instancing:!0===ha.isInstancedMesh,supportsVertexTextures:u,outputEncoding:null!==ia?d(ia.texture):a.outputEncoding,map:!!e.map,mapEncoding:d(e.map),matcap:!!e.matcap,matcapEncoding:d(e.matcap),envMap:!!n,envMapMode:n&&n.mapping,envMapEncoding:d(n),envMapCubeUV:!!n&&(306===n.mapping||307===n.mapping),lightMap:!!e.lightMap,lightMapEncoding:d(e.lightMap),aoMap:!!e.aoMap,emissiveMap:!!e.emissiveMap,emissiveMapEncoding:d(e.emissiveMap),bumpMap:!!e.bumpMap,normalMap:!!e.normalMap,objectSpaceNormalMap:1===
e.normalMapType,tangentSpaceNormalMap:0===e.normalMapType,clearcoatMap:!!e.clearcoatMap,clearcoatRoughnessMap:!!e.clearcoatRoughnessMap,clearcoatNormalMap:!!e.clearcoatNormalMap,displacementMap:!!e.displacementMap,roughnessMap:!!e.roughnessMap,metalnessMap:!!e.metalnessMap,specularMap:!!e.specularMap,alphaMap:!!e.alphaMap,gradientMap:!!e.gradientMap,sheen:!!e.sheen,combine:e.combine,vertexTangents:e.normalMap&&e.vertexTangents,vertexColors:e.vertexColors,vertexUvs:!!e.map||!!e.bumpMap||!!e.normalMap||
!!e.specularMap||!!e.alphaMap||!!e.emissiveMap||!!e.roughnessMap||!!e.metalnessMap||!!e.clearcoatMap||!!e.clearcoatRoughnessMap||!!e.clearcoatNormalMap||!!e.displacementMap,uvsVertexOnly:!(e.map||e.bumpMap||e.normalMap||e.specularMap||e.alphaMap||e.emissiveMap||e.roughnessMap||e.metalnessMap||e.clearcoatNormalMap)&&!!e.displacementMap,fog:!!q,useFog:e.fog,fogExp2:q&&q.isFogExp2,flatShading:e.flatShading,sizeAttenuation:e.sizeAttenuation,logarithmicDepthBuffer:g,skinning:e.skinning&&0<v,maxBones:v,
useVertexTexture:h,morphTargets:e.morphTargets,morphNormals:e.morphNormals,maxMorphTargets:a.maxMorphTargets,maxMorphNormals:a.maxMorphNormals,numDirLights:k.directional.length,numPointLights:k.point.length,numSpotLights:k.spot.length,numRectAreaLights:k.rectArea.length,numHemiLights:k.hemi.length,numDirLightShadows:k.directionalShadowMap.length,numPointLightShadows:k.pointShadowMap.length,numSpotLightShadows:k.spotShadowMap.length,numClippingPlanes:w,numClipIntersection:z,dithering:e.dithering,shadowMapEnabled:a.shadowMap.enabled&&
0<x.length,shadowMapType:a.shadowMap.type,toneMapping:e.toneMapped?a.toneMapping:0,physicallyCorrectLights:a.physicallyCorrectLights,premultipliedAlpha:e.premultipliedAlpha,alphaTest:e.alphaTest,doubleSided:2===e.side,flipSided:1===e.side,depthPacking:void 0!==e.depthPacking?e.depthPacking:!1,index0AttributeName:e.index0AttributeName,extensionDerivatives:e.extensions&&e.extensions.derivatives,extensionFragDepth:e.extensions&&e.extensions.fragDepth,extensionDrawBuffers:e.extensions&&e.extensions.drawBuffers,
extensionShaderTextureLOD:e.extensions&&e.extensions.shaderTextureLOD,rendererExtensionFragDepth:f||null!==b.get(&quot;EXT_frag_depth&quot;),rendererExtensionDrawBuffers:f||null!==b.get(&quot;WEBGL_draw_buffers&quot;),rendererExtensionShaderTextureLod:f||null!==b.get(&quot;EXT_shader_texture_lod&quot;),onBeforeCompile:e.onBeforeCompile}};this.getProgramCacheKey=function(b){var c=[];b.shaderID?c.push(b.shaderID):(c.push(b.fragmentShader),c.push(b.vertexShader));if(void 0!==b.defines)for(var d in b.defines)c.push(d),c.push(b.defines[d]);
if(void 0===b.isRawShaderMaterial){for(d=0;d<k.length;d++)c.push(b[k[d]]);c.push(a.outputEncoding);c.push(a.gammaFactor)}c.push(b.onBeforeCompile.toString());return c.join()};this.acquireProgram=function(b,c){for(var d,f=0,g=e.length;f<g;f++){var h=e[f];if(h.cacheKey===c){d=h;++d.usedTimes;break}}void 0===d&&(d=new vk(a,c,b),e.push(d));return d};this.releaseProgram=function(a){if(0===--a.usedTimes){var b=e.indexOf(a);e[b]=e[e.length-1];e.pop();a.destroy()}};this.programs=e}function yk(){var a=new WeakMap;
return{get:function(b){var c=a.get(b);void 0===c&&(c={},a.set(b,c));return c},remove:function(b){a.delete(b)},update:function(b,c,d){a.get(b)[c]=d},dispose:function(){a=new WeakMap}}}function zk(a,b){return a.groupOrder!==b.groupOrder?a.groupOrder-b.groupOrder:a.renderOrder!==b.renderOrder?a.renderOrder-b.renderOrder:a.program!==b.program?a.program.id-b.program.id:a.material.id!==b.material.id?a.material.id-b.material.id:a.z!==b.z?a.z-b.z:a.id-b.id}function Ak(a,b){return a.groupOrder!==b.groupOrder?
a.groupOrder-b.groupOrder:a.renderOrder!==b.renderOrder?a.renderOrder-b.renderOrder:a.z!==b.z?b.z-a.z:a.id-b.id}function Vh(){function a(a,d,e,m,k,p){var g=b[c];void 0===g?(g={id:a.id,object:a,geometry:d,material:e,program:e.program||f,groupOrder:m,renderOrder:a.renderOrder,z:k,group:p},b[c]=g):(g.id=a.id,g.object=a,g.geometry=d,g.material=e,g.program=e.program||f,g.groupOrder=m,g.renderOrder=a.renderOrder,g.z=k,g.group=p);c++;return g}var b=[],c=0,d=[],e=[],f={id:-1};return{opaque:d,transparent:e,
init:function(){c=0;d.length=0;e.length=0},push:function(b,c,f,m,k,p){b=a(b,c,f,m,k,p);(!0===f.transparent?e:d).push(b)},unshift:function(b,c,f,m,k,p){b=a(b,c,f,m,k,p);(!0===f.transparent?e:d).unshift(b)},finish:function(){for(var a=c,d=b.length;a<d;a++){var e=b[a];if(null===e.id)break;e.id=null;e.object=null;e.geometry=null;e.material=null;e.program=null;e.group=null}},sort:function(a,b){1<d.length&&d.sort(a||zk);1<e.length&&e.sort(b||Ak)}}}function Bk(){function a(c){c=c.target;c.removeEventListener(&quot;dispose&quot;,
a);b.delete(c)}var b=new WeakMap;return{get:function(c,d){var e=b.get(c);if(void 0===e){var f=new Vh;b.set(c,new WeakMap);b.get(c).set(d,f);c.addEventListener(&quot;dispose&quot;,a)}else f=e.get(d),void 0===f&&(f=new Vh,e.set(d,f));return f},dispose:function(){b=new WeakMap}}}function Ck(){var a={};return{get:function(b){if(void 0!==a[b.id])return a[b.id];switch(b.type){case &quot;DirectionalLight&quot;:var c={direction:new n,color:new A};break;case &quot;SpotLight&quot;:c={position:new n,direction:new n,color:new A,distance:0,
coneCos:0,penumbraCos:0,decay:0};break;case &quot;PointLight&quot;:c={position:new n,color:new A,distance:0,decay:0};break;case &quot;HemisphereLight&quot;:c={direction:new n,skyColor:new A,groundColor:new A};break;case &quot;RectAreaLight&quot;:c={color:new A,position:new n,halfWidth:new n,halfHeight:new n}}return a[b.id]=c}}}function Dk(){var a={};return{get:function(b){if(void 0!==a[b.id])return a[b.id];switch(b.type){case &quot;DirectionalLight&quot;:var c={shadowBias:0,shadowRadius:1,shadowMapSize:new t};break;case &quot;SpotLight&quot;:c={shadowBias:0,
shadowRadius:1,shadowMapSize:new t};break;case &quot;PointLight&quot;:c={shadowBias:0,shadowRadius:1,shadowMapSize:new t,shadowCameraNear:1,shadowCameraFar:1E3}}return a[b.id]=c}}}function Ek(a,b){return(b.castShadow?1:0)-(a.castShadow?1:0)}function Fk(){for(var a=new Ck,b=Dk(),c={version:0,hash:{directionalLength:-1,pointLength:-1,spotLength:-1,rectAreaLength:-1,hemiLength:-1,numDirectionalShadows:-1,numPointShadows:-1,numSpotShadows:-1},ambient:[0,0,0],probe:[],directional:[],directionalShadow:[],directionalShadowMap:[],
directionalShadowMatrix:[],spot:[],spotShadow:[],spotShadowMap:[],spotShadowMatrix:[],rectArea:[],point:[],pointShadow:[],pointShadowMap:[],pointShadowMatrix:[],hemi:[]},d=0;9>d;d++)c.probe.push(new n);var e=new n,f=new P,g=new P;return{setup:function(d,l,m){for(var h=0,p=0,k=0,r=0;9>r;r++)c.probe[r].set(0,0,0);var q=l=0,v=0,n=0,w=0,z=0,ha=0,U=0;m=m.matrixWorldInverse;d.sort(Ek);r=0;for(var ca=d.length;r<ca;r++){var B=d[r],t=B.color,ia=B.intensity,Ca=B.distance,Ja=B.shadow&&B.shadow.map?B.shadow.map.texture:
null;if(B.isAmbientLight)h+=t.r*ia,p+=t.g*ia,k+=t.b*ia;else if(B.isLightProbe)for(Ja=0;9>Ja;Ja++)c.probe[Ja].addScaledVector(B.sh.coefficients[Ja],ia);else if(B.isDirectionalLight){var H=a.get(B);H.color.copy(B.color).multiplyScalar(B.intensity);H.direction.setFromMatrixPosition(B.matrixWorld);e.setFromMatrixPosition(B.target.matrixWorld);H.direction.sub(e);H.direction.transformDirection(m);B.castShadow&&(ia=B.shadow,t=b.get(B),t.shadowBias=ia.bias,t.shadowRadius=ia.radius,t.shadowMapSize=ia.mapSize,
c.directionalShadow[l]=t,c.directionalShadowMap[l]=Ja,c.directionalShadowMatrix[l]=B.shadow.matrix,z++);c.directional[l]=H;l++}else B.isSpotLight?(H=a.get(B),H.position.setFromMatrixPosition(B.matrixWorld),H.position.applyMatrix4(m),H.color.copy(t).multiplyScalar(ia),H.distance=Ca,H.direction.setFromMatrixPosition(B.matrixWorld),e.setFromMatrixPosition(B.target.matrixWorld),H.direction.sub(e),H.direction.transformDirection(m),H.coneCos=Math.cos(B.angle),H.penumbraCos=Math.cos(B.angle*(1-B.penumbra)),
H.decay=B.decay,B.castShadow&&(ia=B.shadow,t=b.get(B),t.shadowBias=ia.bias,t.shadowRadius=ia.radius,t.shadowMapSize=ia.mapSize,c.spotShadow[v]=t,c.spotShadowMap[v]=Ja,c.spotShadowMatrix[v]=B.shadow.matrix,U++),c.spot[v]=H,v++):B.isRectAreaLight?(H=a.get(B),H.color.copy(t).multiplyScalar(ia),H.position.setFromMatrixPosition(B.matrixWorld),H.position.applyMatrix4(m),g.identity(),f.copy(B.matrixWorld),f.premultiply(m),g.extractRotation(f),H.halfWidth.set(.5*B.width,0,0),H.halfHeight.set(0,.5*B.height,
0),H.halfWidth.applyMatrix4(g),H.halfHeight.applyMatrix4(g),c.rectArea[n]=H,n++):B.isPointLight?(H=a.get(B),H.position.setFromMatrixPosition(B.matrixWorld),H.position.applyMatrix4(m),H.color.copy(B.color).multiplyScalar(B.intensity),H.distance=B.distance,H.decay=B.decay,B.castShadow&&(ia=B.shadow,t=b.get(B),t.shadowBias=ia.bias,t.shadowRadius=ia.radius,t.shadowMapSize=ia.mapSize,t.shadowCameraNear=ia.camera.near,t.shadowCameraFar=ia.camera.far,c.pointShadow[q]=t,c.pointShadowMap[q]=Ja,c.pointShadowMatrix[q]=
B.shadow.matrix,ha++),c.point[q]=H,q++):B.isHemisphereLight&&(H=a.get(B),H.direction.setFromMatrixPosition(B.matrixWorld),H.direction.transformDirection(m),H.direction.normalize(),H.skyColor.copy(B.color).multiplyScalar(ia),H.groundColor.copy(B.groundColor).multiplyScalar(ia),c.hemi[w]=H,w++)}c.ambient[0]=h;c.ambient[1]=p;c.ambient[2]=k;d=c.hash;if(d.directionalLength!==l||d.pointLength!==q||d.spotLength!==v||d.rectAreaLength!==n||d.hemiLength!==w||d.numDirectionalShadows!==z||d.numPointShadows!==
ha||d.numSpotShadows!==U)c.directional.length=l,c.spot.length=v,c.rectArea.length=n,c.point.length=q,c.hemi.length=w,c.directionalShadow.length=z,c.directionalShadowMap.length=z,c.pointShadow.length=ha,c.pointShadowMap.length=ha,c.spotShadow.length=U,c.spotShadowMap.length=U,c.directionalShadowMatrix.length=z,c.pointShadowMatrix.length=ha,c.spotShadowMatrix.length=U,d.directionalLength=l,d.pointLength=q,d.spotLength=v,d.rectAreaLength=n,d.hemiLength=w,d.numDirectionalShadows=z,d.numPointShadows=ha,
d.numSpotShadows=U,c.version=Gk++},state:c}}function Wh(){var a=new Fk,b=[],c=[];return{init:function(){b.length=0;c.length=0},state:{lightsArray:b,shadowsArray:c,lights:a},setupLights:function(d){a.setup(b,c,d)},pushLight:function(a){b.push(a)},pushShadow:function(a){c.push(a)}}}function Hk(){function a(c){c=c.target;c.removeEventListener(&quot;dispose&quot;,a);b.delete(c)}var b=new WeakMap;return{get:function(c,d){if(!1===b.has(c)){var e=new Wh;b.set(c,new WeakMap);b.get(c).set(d,e);c.addEventListener(&quot;dispose&quot;,
a)}else!1===b.get(c).has(d)?(e=new Wh,b.get(c).set(d,e)):e=b.get(c).get(d);return e},dispose:function(){b=new WeakMap}}}function Fb(a){K.call(this);this.type=&quot;MeshDepthMaterial&quot;;this.depthPacking=3200;this.morphTargets=this.skinning=!1;this.displacementMap=this.alphaMap=this.map=null;this.displacementScale=1;this.displacementBias=0;this.wireframe=!1;this.wireframeLinewidth=1;this.fog=!1;this.setValues(a)}function Gb(a){K.call(this);this.type=&quot;MeshDistanceMaterial&quot;;this.referencePosition=new n;this.nearDistance=
1;this.farDistance=1E3;this.morphTargets=this.skinning=!1;this.displacementMap=this.alphaMap=this.map=null;this.displacementScale=1;this.displacementBias=0;this.fog=!1;this.setValues(a)}function Xh(a,b,c){function d(a,b,c){c=a<<0|b<<1|c<<2;var d=p[c];void 0===d&&(d=new Fb({depthPacking:3201,morphTargets:a,skinning:b}),p[c]=d);return d}function e(a,b,c){c=a<<0|b<<1|c<<2;var d=x[c];void 0===d&&(d=new Gb({morphTargets:a,skinning:b}),x[c]=d);return d}function f(b,c,f,g,h,l){var m=b.geometry,p=d,k=b.customDepthMaterial;
!0===f.isPointLight&&(p=e,k=b.customDistanceMaterial);void 0===k?(k=!1,!0===c.morphTargets&&(!0===m.isBufferGeometry?k=m.morphAttributes&&m.morphAttributes.position&&0<m.morphAttributes.position.length:!0===m.isGeometry&&(k=m.morphTargets&&0<m.morphTargets.length)),m=!1,!0===b.isSkinnedMesh&&(!0===c.skinning?m=!0:console.warn(&quot;THREE.WebGLShadowMap: THREE.SkinnedMesh with material.skinning set to false:&quot;,b)),b=p(k,m,!0===b.isInstancedMesh)):b=k;a.localClippingEnabled&&!0===c.clipShadows&&0!==c.clippingPlanes.length&&
(k=b.uuid,p=c.uuid,m=r[k],void 0===m&&(m={},r[k]=m),k=m[p],void 0===k&&(k=b.clone(),m[p]=k),b=k);b.visible=c.visible;b.wireframe=c.wireframe;b.side=3===l?null!==c.shadowSide?c.shadowSide:c.side:null!==c.shadowSide?c.shadowSide:q[c.side];b.clipShadows=c.clipShadows;b.clippingPlanes=c.clippingPlanes;b.clipIntersection=c.clipIntersection;b.wireframeLinewidth=c.wireframeLinewidth;b.linewidth=c.linewidth;!0===f.isPointLight&&!0===b.isMeshDistanceMaterial&&(b.referencePosition.setFromMatrixPosition(f.matrixWorld),
b.nearDistance=g,b.farDistance=h);return b}function g(c,d,e,l,m){if(!1!==c.visible){if(c.layers.test(d.layers)&&(c.isMesh||c.isLine||c.isPoints)&&(c.castShadow||c.receiveShadow&&3===m)&&(!c.frustumCulled||h.intersectsObject(c))){c.modelViewMatrix.multiplyMatrices(e.matrixWorldInverse,c.matrixWorld);var p=b.update(c),k=c.material;if(Array.isArray(k))for(var u=p.groups,x=0,q=u.length;x<q;x++){var r=u[x],v=k[r.materialIndex];v&&v.visible&&(v=f(c,v,l,e.near,e.far,m),a.renderBufferDirect(e,null,p,v,c,
r))}else k.visible&&(v=f(c,k,l,e.near,e.far,m),a.renderBufferDirect(e,null,p,v,c,null))}c=c.children;p=0;for(k=c.length;p<k;p++)g(c[p],d,e,l,m)}}var h=new Hc,l=new t,m=new t,k=new ka,p=[],x=[],r={},q={0:1,1:0,2:2},v=new Ba({defines:{SAMPLE_RATE:.25,HALF_SAMPLE_RATE:.125},uniforms:{shadow_pass:{value:null},resolution:{value:new t},radius:{value:4}},vertexShader:&quot;void main() {\n\tgl_Position = vec4( position, 1.0 );\n}&quot;,fragmentShader:&quot;uniform sampler2D shadow_pass;\nuniform vec2 resolution;\nuniform float radius;\n#include <packing>\nvoid main() {\n  float mean = 0.0;\n  float squared_mean = 0.0;\n\tfloat depth = unpackRGBAToDepth( texture2D( shadow_pass, ( gl_FragCoord.xy  ) / resolution ) );\n  for ( float i = -1.0; i < 1.0 ; i += SAMPLE_RATE) {\n    #ifdef HORIZONAL_PASS\n      vec2 distribution = unpackRGBATo2Half( texture2D( shadow_pass, ( gl_FragCoord.xy + vec2( i, 0.0 ) * radius ) / resolution ) );\n      mean += distribution.x;\n      squared_mean += distribution.y * distribution.y + distribution.x * distribution.x;\n    #else\n      float depth = unpackRGBAToDepth( texture2D( shadow_pass, ( gl_FragCoord.xy + vec2( 0.0,  i )  * radius ) / resolution ) );\n      mean += depth;\n      squared_mean += depth * depth;\n    #endif\n  }\n  mean = mean * HALF_SAMPLE_RATE;\n  squared_mean = squared_mean * HALF_SAMPLE_RATE;\n  float std_dev = sqrt( squared_mean - mean * mean );\n  gl_FragColor = pack2HalfToRGBA( vec2( mean, std_dev ) );\n}&quot;}),
n=v.clone();n.defines.HORIZONAL_PASS=1;var w=new C;w.setAttribute(&quot;position&quot;,new M(new Float32Array([-1,-1,.5,3,-1,.5,-1,3,.5]),3));var z=new S(w,v),ha=this;this.enabled=!1;this.autoUpdate=!0;this.needsUpdate=!1;this.type=1;this.render=function(d,e,f){if(!1!==ha.enabled&&(!1!==ha.autoUpdate||!1!==ha.needsUpdate)&&0!==d.length){var p=a.getRenderTarget(),u=a.getActiveCubeFace(),x=a.getActiveMipmapLevel(),q=a.state;q.setBlending(0);q.buffers.color.setClear(1,1,1,1);q.buffers.depth.setTest(!0);q.setScissorTest(!1);
for(var r=0,E=d.length;r<E;r++){var w=d[r],B=w.shadow;if(void 0===B)console.warn(&quot;THREE.WebGLShadowMap:&quot;,w,&quot;has no shadow.&quot;);else{l.copy(B.mapSize);var t=B.getFrameExtents();l.multiply(t);m.copy(B.mapSize);if(l.x>c||l.y>c)console.warn(&quot;THREE.WebGLShadowMap:&quot;,w,&quot;has shadow exceeding max texture size, reducing&quot;),l.x>c&&(m.x=Math.floor(c/t.x),l.x=m.x*t.x,B.mapSize.x=m.x),l.y>c&&(m.y=Math.floor(c/t.y),l.y=m.y*t.y,B.mapSize.y=m.y);null!==B.map||B.isPointLightShadow||3!==this.type||(t={minFilter:1006,magFilter:1006,
format:1023},B.map=new Ha(l.x,l.y,t),B.map.texture.name=w.name+&quot;.shadowMap&quot;,B.mapPass=new Ha(l.x,l.y,t),B.camera.updateProjectionMatrix());null===B.map&&(t={minFilter:1003,magFilter:1003,format:1023},B.map=new Ha(l.x,l.y,t),B.map.texture.name=w.name+&quot;.shadowMap&quot;,B.camera.updateProjectionMatrix());a.setRenderTarget(B.map);a.clear();t=B.getViewportCount();for(var ca=0;ca<t;ca++){var U=B.getViewport(ca);k.set(m.x*U.x,m.y*U.y,m.x*U.z,m.y*U.w);q.viewport(k);B.updateMatrices(w,ca);h=B.getFrustum();g(e,
f,B.camera,w,this.type)}B.isPointLightShadow||3!==this.type||(w=B,B=f,t=b.update(z),v.uniforms.shadow_pass.value=w.map.texture,v.uniforms.resolution.value=w.mapSize,v.uniforms.radius.value=w.radius,a.setRenderTarget(w.mapPass),a.clear(),a.renderBufferDirect(B,null,t,v,z,null),n.uniforms.shadow_pass.value=w.mapPass.texture,n.uniforms.resolution.value=w.mapSize,n.uniforms.radius.value=w.radius,a.setRenderTarget(w.map),a.clear(),a.renderBufferDirect(B,null,t,n,z,null))}}ha.needsUpdate=!1;a.setRenderTarget(p,
u,x)}}}function Ik(a,b,c){function d(b,c,d){var e=new Uint8Array(4),f=a.createTexture();a.bindTexture(b,f);a.texParameteri(b,10241,9728);a.texParameteri(b,10240,9728);for(b=0;b<d;b++)a.texImage2D(c+b,0,6408,1,1,0,6408,5121,e);return f}function e(c,d){n[c]=1;0===w[c]&&(a.enableVertexAttribArray(c),w[c]=1);z[c]!==d&&((x?a:b.get(&quot;ANGLE_instanced_arrays&quot;))[x?&quot;vertexAttribDivisor&quot;:&quot;vertexAttribDivisorANGLE&quot;](c,d),z[c]=d)}function f(b){!0!==t[b]&&(a.enable(b),t[b]=!0)}function g(b){!1!==t[b]&&(a.disable(b),
t[b]=!1)}function h(b,c,d,e,h,l,m,p){if(0===b)ca&&(g(3042),ca=!1);else if(ca||(f(3042),ca=!0),5!==b){if(b!==B||p!==F){if(100!==y||100!==Ja)a.blendEquation(32774),Ja=y=100;if(p)switch(b){case 1:a.blendFuncSeparate(1,771,1,771);break;case 2:a.blendFunc(1,1);break;case 3:a.blendFuncSeparate(0,0,769,771);break;case 4:a.blendFuncSeparate(0,768,0,770);break;default:console.error(&quot;THREE.WebGLState: Invalid blending: &quot;,b)}else switch(b){case 1:a.blendFuncSeparate(770,771,1,771);break;case 2:a.blendFunc(770,
1);break;case 3:a.blendFunc(0,769);break;case 4:a.blendFunc(0,768);break;default:console.error(&quot;THREE.WebGLState: Invalid blending: &quot;,b)}D=H=Ca=ia=null;B=b;F=p}}else{h=h||c;l=l||d;m=m||e;if(c!==y||h!==Ja)a.blendEquationSeparate(Lc[c],Lc[h]),y=c,Ja=h;if(d!==ia||e!==Ca||l!==H||m!==D)a.blendFuncSeparate(J[d],J[e],J[l],J[m]),ia=d,Ca=e,H=l,D=m;B=b;F=null}}function l(b){A!==b&&(b?a.frontFace(2304):a.frontFace(2305),A=b)}function m(b){0!==b?(f(2884),b!==C&&(1===b?a.cullFace(1029):2===b?a.cullFace(1028):
a.cullFace(1032))):g(2884);C=b}function k(b,c,d){if(b){if(f(32823),K!==c||M!==d)a.polygonOffset(c,d),K=c,M=d}else g(32823)}function p(b){void 0===b&&(b=33984+fa-1);L!==b&&(a.activeTexture(b),L=b)}var x=c.isWebGL2,r=new function(){var b=!1,c=new ka,d=null,e=new ka(0,0,0,0);return{setMask:function(c){d===c||b||(a.colorMask(c,c,c,c),d=c)},setLocked:function(a){b=a},setClear:function(b,d,f,g,h){!0===h&&(b*=g,d*=g,f*=g);c.set(b,d,f,g);!1===e.equals(c)&&(a.clearColor(b,d,f,g),e.copy(c))},reset:function(){b=
!1;d=null;e.set(-1,0,0,0)}}},q=new function(){var b=!1,c=null,d=null,e=null;return{setTest:function(a){a?f(2929):g(2929)},setMask:function(d){c===d||b||(a.depthMask(d),c=d)},setFunc:function(b){if(d!==b){if(b)switch(b){case 0:a.depthFunc(512);break;case 1:a.depthFunc(519);break;case 2:a.depthFunc(513);break;case 3:a.depthFunc(515);break;case 4:a.depthFunc(514);break;case 5:a.depthFunc(518);break;case 6:a.depthFunc(516);break;case 7:a.depthFunc(517);break;default:a.depthFunc(515)}else a.depthFunc(515);
d=b}},setLocked:function(a){b=a},setClear:function(b){e!==b&&(a.clearDepth(b),e=b)},reset:function(){b=!1;e=d=c=null}}},v=new function(){var b=!1,c=null,d=null,e=null,h=null,l=null,m=null,p=null,k=null;return{setTest:function(a){b||(a?f(2960):g(2960))},setMask:function(d){c===d||b||(a.stencilMask(d),c=d)},setFunc:function(b,c,f){if(d!==b||e!==c||h!==f)a.stencilFunc(b,c,f),d=b,e=c,h=f},setOp:function(b,c,d){if(l!==b||m!==c||p!==d)a.stencilOp(b,c,d),l=b,m=c,p=d},setLocked:function(a){b=a},setClear:function(b){k!==
b&&(a.clearStencil(b),k=b)},reset:function(){b=!1;k=p=m=l=h=e=d=c=null}}};c=a.getParameter(34921);var n=new Uint8Array(c),w=new Uint8Array(c),z=new Uint8Array(c),t={},U=null,ca=null,B=null,y=null,ia=null,Ca=null,Ja=null,H=null,D=null,F=!1,A=null,C=null,G=null,K=null,M=null,fa=a.getParameter(35661),N=!1;c=0;c=a.getParameter(7938);-1!==c.indexOf(&quot;WebGL&quot;)?(c=parseFloat(/^WebGL ([0-9])/.exec(c)[1]),N=1<=c):-1!==c.indexOf(&quot;OpenGL ES&quot;)&&(c=parseFloat(/^OpenGL ES ([0-9])/.exec(c)[1]),N=2<=c);var L=null,
Nd={},Z=new ka,Yh=new ka,ng={};ng[3553]=d(3553,3553,1);ng[34067]=d(34067,34069,6);r.setClear(0,0,0,1);q.setClear(1);v.setClear(0);f(2929);q.setFunc(3);l(!1);m(1);f(2884);h(0);var Lc={100:32774,101:32778,102:32779};x?(Lc[103]=32775,Lc[104]=32776):(c=b.get(&quot;EXT_blend_minmax&quot;),null!==c&&(Lc[103]=c.MIN_EXT,Lc[104]=c.MAX_EXT));var J={200:0,201:1,202:768,204:770,210:776,208:774,206:772,203:769,205:771,209:775,207:773};return{buffers:{color:r,depth:q,stencil:v},initAttributes:function(){for(var a=0,b=n.length;a<
b;a++)n[a]=0},enableAttribute:function(a){e(a,0)},enableAttributeAndDivisor:e,disableUnusedAttributes:function(){for(var b=0,c=w.length;b!==c;++b)w[b]!==n[b]&&(a.disableVertexAttribArray(b),w[b]=0)},enable:f,disable:g,useProgram:function(b){return U!==b?(a.useProgram(b),U=b,!0):!1},setBlending:h,setMaterial:function(a,b){2===a.side?g(2884):f(2884);var c=1===a.side;b&&(c=!c);l(c);1===a.blending&&!1===a.transparent?h(0):h(a.blending,a.blendEquation,a.blendSrc,a.blendDst,a.blendEquationAlpha,a.blendSrcAlpha,
a.blendDstAlpha,a.premultipliedAlpha);q.setFunc(a.depthFunc);q.setTest(a.depthTest);q.setMask(a.depthWrite);r.setMask(a.colorWrite);b=a.stencilWrite;v.setTest(b);b&&(v.setMask(a.stencilWriteMask),v.setFunc(a.stencilFunc,a.stencilRef,a.stencilFuncMask),v.setOp(a.stencilFail,a.stencilZFail,a.stencilZPass));k(a.polygonOffset,a.polygonOffsetFactor,a.polygonOffsetUnits)},setFlipSided:l,setCullFace:m,setLineWidth:function(b){b!==G&&(N&&a.lineWidth(b),G=b)},setPolygonOffset:k,setScissorTest:function(a){a?
f(3089):g(3089)},activeTexture:p,bindTexture:function(b,c){null===L&&p();var d=Nd[L];void 0===d&&(d={type:void 0,texture:void 0},Nd[L]=d);if(d.type!==b||d.texture!==c)a.bindTexture(b,c||ng[b]),d.type=b,d.texture=c},unbindTexture:function(){var b=Nd[L];void 0!==b&&void 0!==b.type&&(a.bindTexture(b.type,null),b.type=void 0,b.texture=void 0)},compressedTexImage2D:function(){try{a.compressedTexImage2D.apply(a,arguments)}catch(Q){console.error(&quot;THREE.WebGLState:&quot;,Q)}},texImage2D:function(){try{a.texImage2D.apply(a,
arguments)}catch(Q){console.error(&quot;THREE.WebGLState:&quot;,Q)}},texImage3D:function(){try{a.texImage3D.apply(a,arguments)}catch(Q){console.error(&quot;THREE.WebGLState:&quot;,Q)}},scissor:function(b){!1===Z.equals(b)&&(a.scissor(b.x,b.y,b.z,b.w),Z.copy(b))},viewport:function(b){!1===Yh.equals(b)&&(a.viewport(b.x,b.y,b.z,b.w),Yh.copy(b))},reset:function(){for(var b=0;b<w.length;b++)1===w[b]&&(a.disableVertexAttribArray(b),w[b]=0);t={};L=null;Nd={};C=A=B=U=null;r.reset();q.reset();v.reset()}}}function Jk(a,b,c,d,
e,f,g){function h(a,b){return K?new OffscreenCanvas(a,b):document.createElementNS(&quot;http://www.w3.org/1999/xhtml&quot;,&quot;canvas&quot;)}function l(a,b,c,d){var e=1;if(a.width>d||a.height>d)e=d/Math.max(a.width,a.height);if(1>e||!0===b){if(&quot;undefined&quot;!==typeof HTMLImageElement&&a instanceof HTMLImageElement||&quot;undefined&quot;!==typeof HTMLCanvasElement&&a instanceof HTMLCanvasElement||&quot;undefined&quot;!==typeof ImageBitmap&&a instanceof ImageBitmap)return d=b?L.floorPowerOfTwo:Math.floor,b=d(e*a.width),e=d(e*a.height),void 0===
G&&(G=h(b,e)),c=c?h(b,e):G,c.width=b,c.height=e,c.getContext(&quot;2d&quot;).drawImage(a,0,0,b,e),console.warn(&quot;THREE.WebGLRenderer: Texture has been resized from (&quot;+a.width+&quot;x&quot;+a.height+&quot;) to (&quot;+b+&quot;x&quot;+e+&quot;).&quot;),c;&quot;data&quot;in a&&console.warn(&quot;THREE.WebGLRenderer: Image in DataTexture is too big (&quot;+a.width+&quot;x&quot;+a.height+&quot;).&quot;)}return a}function m(a){return L.isPowerOfTwo(a.width)&&L.isPowerOfTwo(a.height)}function k(a,b){return a.generateMipmaps&&b&&1003!==a.minFilter&&1006!==a.minFilter}function p(b,c,e,f){a.generateMipmap(b);
d.get(c).__maxMipLevel=Math.log(Math.max(e,f))*Math.LOG2E}function x(c,d,e){if(!1===Ca)return d;if(null!==c){if(void 0!==a[c])return a[c];console.warn(&quot;THREE.WebGLRenderer: Attempt to use non-existing WebGL internal format '&quot;+c+&quot;'&quot;)}c=d;6403===d&&(5126===e&&(c=33326),5131===e&&(c=33325),5121===e&&(c=33321));6407===d&&(5126===e&&(c=34837),5131===e&&(c=34843),5121===e&&(c=32849));6408===d&&(5126===e&&(c=34836),5131===e&&(c=34842),5121===e&&(c=32856));33325!==c&&33326!==c&&34842!==c&&34836!==c||b.get(&quot;EXT_color_buffer_float&quot;);
return c}function r(a){return 1003===a||1004===a||1005===a?9728:9729}function q(b){b=b.target;b.removeEventListener(&quot;dispose&quot;,q);var c=d.get(b);void 0!==c.__webglInit&&(a.deleteTexture(c.__webglTexture),d.remove(b));b.isVideoTexture&&C.delete(b);g.memory.textures--}function v(b){b=b.target;b.removeEventListener(&quot;dispose&quot;,v);var c=d.get(b),e=d.get(b.texture);if(b){void 0!==e.__webglTexture&&a.deleteTexture(e.__webglTexture);b.depthTexture&&b.depthTexture.dispose();if(b.isWebGLCubeRenderTarget)for(e=
0;6>e;e++)a.deleteFramebuffer(c.__webglFramebuffer[e]),c.__webglDepthbuffer&&a.deleteRenderbuffer(c.__webglDepthbuffer[e]);else a.deleteFramebuffer(c.__webglFramebuffer),c.__webglDepthbuffer&&a.deleteRenderbuffer(c.__webglDepthbuffer),c.__webglMultisampledFramebuffer&&a.deleteFramebuffer(c.__webglMultisampledFramebuffer),c.__webglColorRenderbuffer&&a.deleteRenderbuffer(c.__webglColorRenderbuffer),c.__webglDepthRenderbuffer&&a.deleteRenderbuffer(c.__webglDepthRenderbuffer);d.remove(b.texture);d.remove(b)}g.memory.textures--}
function n(a,b){var e=d.get(a);if(a.isVideoTexture){var f=g.render.frame;C.get(a)!==f&&(C.set(a,f),a.update())}if(0<a.version&&e.__version!==a.version)if(f=a.image,void 0===f)console.warn(&quot;THREE.WebGLRenderer: Texture marked for update but image is undefined&quot;);else if(!1===f.complete)console.warn(&quot;THREE.WebGLRenderer: Texture marked for update but image is incomplete&quot;);else{ca(e,a,b);return}c.activeTexture(33984+b);c.bindTexture(3553,e.__webglTexture)}function w(b,e){if(6===b.image.length){var g=
d.get(b);if(0<b.version&&g.__version!==b.version){U(g,b);c.activeTexture(33984+e);c.bindTexture(34067,g.__webglTexture);a.pixelStorei(37440,b.flipY);var h=b&&(b.isCompressedTexture||b.image[0].isCompressedTexture);e=b.image[0]&&b.image[0].isDataTexture;for(var u=[],q=0;6>q;q++)u[q]=h||e?e?b.image[q].image:b.image[q]:l(b.image[q],!1,!0,H);var r=u[0],v=m(r)||Ca,n=f.convert(b.format),w=f.convert(b.type),E=x(b.internalFormat,n,w);t(34067,b,v);if(h){for(q=0;6>q;q++){var Z=u[q].mipmaps;for(h=0;h<Z.length;h++){var z=
Z[h];1023!==b.format&&1022!==b.format?null!==n?c.compressedTexImage2D(34069+q,h,E,z.width,z.height,0,z.data):console.warn(&quot;THREE.WebGLRenderer: Attempt to load unsupported compressed texture format in .setTextureCube()&quot;):c.texImage2D(34069+q,h,E,z.width,z.height,0,n,w,z.data)}}g.__maxMipLevel=Z.length-1}else{Z=b.mipmaps;for(q=0;6>q;q++)if(e)for(c.texImage2D(34069+q,0,E,u[q].width,u[q].height,0,n,w,u[q].data),h=0;h<Z.length;h++)z=Z[h],z=z.image[q].image,c.texImage2D(34069+q,h+1,E,z.width,z.height,
0,n,w,z.data);else for(c.texImage2D(34069+q,0,E,n,w,u[q]),h=0;h<Z.length;h++)z=Z[h],c.texImage2D(34069+q,h+1,E,n,w,z.image[q]);g.__maxMipLevel=Z.length}k(b,v)&&p(34067,b,r.width,r.height);g.__version=b.version;if(b.onUpdate)b.onUpdate(b)}else c.activeTexture(33984+e),c.bindTexture(34067,g.__webglTexture)}}function z(a,b){c.activeTexture(33984+b);c.bindTexture(34067,d.get(a).__webglTexture)}function t(c,f,g){g?(a.texParameteri(c,10242,N[f.wrapS]),a.texParameteri(c,10243,N[f.wrapT]),32879!==c&&35866!==
c||a.texParameteri(c,32882,N[f.wrapR]),a.texParameteri(c,10240,fa[f.magFilter]),a.texParameteri(c,10241,fa[f.minFilter])):(a.texParameteri(c,10242,33071),a.texParameteri(c,10243,33071),32879!==c&&35866!==c||a.texParameteri(c,32882,33071),1001===f.wrapS&&1001===f.wrapT||console.warn(&quot;THREE.WebGLRenderer: Texture is not power of two. Texture.wrapS and Texture.wrapT should be set to THREE.ClampToEdgeWrapping.&quot;),a.texParameteri(c,10240,r(f.magFilter)),a.texParameteri(c,10241,r(f.minFilter)),1003!==f.minFilter&&
1006!==f.minFilter&&console.warn(&quot;THREE.WebGLRenderer: Texture is not power of two. Texture.minFilter should be set to THREE.NearestFilter or THREE.LinearFilter.&quot;));!(g=b.get(&quot;EXT_texture_filter_anisotropic&quot;))||1015===f.type&&null===b.get(&quot;OES_texture_float_linear&quot;)||1016===f.type&&null===(Ca||b.get(&quot;OES_texture_half_float_linear&quot;))||!(1<f.anisotropy||d.get(f).__currentAnisotropy)||(a.texParameterf(c,g.TEXTURE_MAX_ANISOTROPY_EXT,Math.min(f.anisotropy,e.getMaxAnisotropy())),d.get(f).__currentAnisotropy=
f.anisotropy)}function U(b,c){void 0===b.__webglInit&&(b.__webglInit=!0,c.addEventListener(&quot;dispose&quot;,q),b.__webglTexture=a.createTexture(),g.memory.textures++)}function ca(b,d,e){var g=3553;d.isDataTexture2DArray&&(g=35866);d.isDataTexture3D&&(g=32879);U(b,d);c.activeTexture(33984+e);c.bindTexture(g,b.__webglTexture);a.pixelStorei(37440,d.flipY);a.pixelStorei(37441,d.premultiplyAlpha);a.pixelStorei(3317,d.unpackAlignment);e=Ca?!1:1001!==d.wrapS||1001!==d.wrapT||1003!==d.minFilter&&1006!==d.minFilter;
e=e&&!1===m(d.image);e=l(d.image,e,!1,F);var h=m(e)||Ca,u=f.convert(d.format),q=f.convert(d.type),r=x(d.internalFormat,u,q);t(g,d,h);var v=d.mipmaps;if(d.isDepthTexture)r=6402,Ca?r=1015===d.type?36012:1014===d.type?33190:1020===d.type?35056:33189:1015===d.type&&console.error(&quot;WebGLRenderer: Floating point depth texture requires WebGL2.&quot;),1026===d.format&&6402===r&&1012!==d.type&&1014!==d.type&&(console.warn(&quot;THREE.WebGLRenderer: Use UnsignedShortType or UnsignedIntType for DepthFormat DepthTexture.&quot;),
d.type=1012,q=f.convert(d.type)),1027===d.format&&6402===r&&(r=34041,1020!==d.type&&(console.warn(&quot;THREE.WebGLRenderer: Use UnsignedInt248Type for DepthStencilFormat DepthTexture.&quot;),d.type=1020,q=f.convert(d.type))),c.texImage2D(3553,0,r,e.width,e.height,0,u,q,null);else if(d.isDataTexture)if(0<v.length&&h){for(var n=0,w=v.length;n<w;n++){var E=v[n];c.texImage2D(3553,n,r,E.width,E.height,0,u,q,E.data)}d.generateMipmaps=!1;b.__maxMipLevel=v.length-1}else c.texImage2D(3553,0,r,e.width,e.height,0,u,
q,e.data),b.__maxMipLevel=0;else if(d.isCompressedTexture){n=0;for(w=v.length;n<w;n++)E=v[n],1023!==d.format&&1022!==d.format?null!==u?c.compressedTexImage2D(3553,n,r,E.width,E.height,0,E.data):console.warn(&quot;THREE.WebGLRenderer: Attempt to load unsupported compressed texture format in .uploadTexture()&quot;):c.texImage2D(3553,n,r,E.width,E.height,0,u,q,E.data);b.__maxMipLevel=v.length-1}else if(d.isDataTexture2DArray)c.texImage3D(35866,0,r,e.width,e.height,e.depth,0,u,q,e.data),b.__maxMipLevel=0;else if(d.isDataTexture3D)c.texImage3D(32879,
0,r,e.width,e.height,e.depth,0,u,q,e.data),b.__maxMipLevel=0;else if(0<v.length&&h){n=0;for(w=v.length;n<w;n++)E=v[n],c.texImage2D(3553,n,r,u,q,E);d.generateMipmaps=!1;b.__maxMipLevel=v.length-1}else c.texImage2D(3553,0,r,u,q,e),b.__maxMipLevel=0;k(d,h)&&p(g,d,e.width,e.height);b.__version=d.version;if(d.onUpdate)d.onUpdate(d)}function B(b,e,g,h){var l=f.convert(e.texture.format),m=f.convert(e.texture.type),p=x(e.texture.internalFormat,l,m);c.texImage2D(h,0,p,e.width,e.height,0,l,m,null);a.bindFramebuffer(36160,
b);a.framebufferTexture2D(36160,g,h,d.get(e.texture).__webglTexture,0);a.bindFramebuffer(36160,null)}function y(b,c,d){a.bindRenderbuffer(36161,b);if(c.depthBuffer&&!c.stencilBuffer){var e=33189;d?((d=c.depthTexture)&&d.isDepthTexture&&(1015===d.type?e=36012:1014===d.type&&(e=33190)),d=ia(c),a.renderbufferStorageMultisample(36161,d,e,c.width,c.height)):a.renderbufferStorage(36161,e,c.width,c.height);a.framebufferRenderbuffer(36160,36096,36161,b)}else c.depthBuffer&&c.stencilBuffer?(d?(d=ia(c),a.renderbufferStorageMultisample(36161,
d,35056,c.width,c.height)):a.renderbufferStorage(36161,34041,c.width,c.height),a.framebufferRenderbuffer(36160,33306,36161,b)):(b=f.convert(c.texture.format),e=f.convert(c.texture.type),e=x(c.texture.internalFormat,b,e),d?(d=ia(c),a.renderbufferStorageMultisample(36161,d,e,c.width,c.height)):a.renderbufferStorage(36161,e,c.width,c.height));a.bindRenderbuffer(36161,null)}function ia(a){return Ca&&a.isWebGLMultisampleRenderTarget?Math.min(A,a.samples):0}var Ca=e.isWebGL2,D=e.maxTextures,H=e.maxCubemapSize,
F=e.maxTextureSize,A=e.maxSamples,C=new WeakMap,G,K=!1;try{K=&quot;undefined&quot;!==typeof OffscreenCanvas&&null!==(new OffscreenCanvas(1,1)).getContext(&quot;2d&quot;)}catch(Nd){}var M=0,N={1E3:10497,1001:33071,1002:33648},fa={1003:9728,1004:9984,1005:9986,1006:9729,1007:9985,1008:9987},P=!1,O=!1;this.allocateTextureUnit=function(){var a=M;a>=D&&console.warn(&quot;THREE.WebGLTextures: Trying to use &quot;+a+&quot; texture units while this GPU supports only &quot;+D);M+=1;return a};this.resetTextureUnits=function(){M=0};this.setTexture2D=
n;this.setTexture2DArray=function(a,b){var e=d.get(a);0<a.version&&e.__version!==a.version?ca(e,a,b):(c.activeTexture(33984+b),c.bindTexture(35866,e.__webglTexture))};this.setTexture3D=function(a,b){var e=d.get(a);0<a.version&&e.__version!==a.version?ca(e,a,b):(c.activeTexture(33984+b),c.bindTexture(32879,e.__webglTexture))};this.setTextureCube=w;this.setTextureCubeDynamic=z;this.setupRenderTarget=function(b){var e=d.get(b),h=d.get(b.texture);b.addEventListener(&quot;dispose&quot;,v);h.__webglTexture=a.createTexture();
g.memory.textures++;var l=!0===b.isWebGLCubeRenderTarget,u=!0===b.isWebGLMultisampleRenderTarget,q=m(b)||Ca;!Ca||1022!==b.texture.format||1015!==b.texture.type&&1016!==b.texture.type||(b.texture.format=1023,console.warn(&quot;THREE.WebGLRenderer: Rendering to textures with RGB format is not supported. Using RGBA format instead.&quot;));if(l)for(e.__webglFramebuffer=[],u=0;6>u;u++)e.__webglFramebuffer[u]=a.createFramebuffer();else if(e.__webglFramebuffer=a.createFramebuffer(),u)if(Ca){e.__webglMultisampledFramebuffer=
a.createFramebuffer();e.__webglColorRenderbuffer=a.createRenderbuffer();a.bindRenderbuffer(36161,e.__webglColorRenderbuffer);u=f.convert(b.texture.format);var r=f.convert(b.texture.type);u=x(b.texture.internalFormat,u,r);r=ia(b);a.renderbufferStorageMultisample(36161,r,u,b.width,b.height);a.bindFramebuffer(36160,e.__webglMultisampledFramebuffer);a.framebufferRenderbuffer(36160,36064,36161,e.__webglColorRenderbuffer);a.bindRenderbuffer(36161,null);b.depthBuffer&&(e.__webglDepthRenderbuffer=a.createRenderbuffer(),
y(e.__webglDepthRenderbuffer,b,!0));a.bindFramebuffer(36160,null)}else console.warn(&quot;THREE.WebGLRenderer: WebGLMultisampleRenderTarget can only be used with WebGL2.&quot;);if(l){c.bindTexture(34067,h.__webglTexture);t(34067,b.texture,q);for(u=0;6>u;u++)B(e.__webglFramebuffer[u],b,36064,34069+u);k(b.texture,q)&&p(34067,b.texture,b.width,b.height);c.bindTexture(34067,null)}else c.bindTexture(3553,h.__webglTexture),t(3553,b.texture,q),B(e.__webglFramebuffer,b,36064,3553),k(b.texture,q)&&p(3553,b.texture,
b.width,b.height),c.bindTexture(3553,null);if(b.depthBuffer){e=d.get(b);h=!0===b.isWebGLCubeRenderTarget;if(b.depthTexture){if(h)throw Error(&quot;target.depthTexture not supported in Cube render targets&quot;);if(b&&b.isWebGLCubeRenderTarget)throw Error(&quot;Depth Texture with cube render targets is not supported&quot;);a.bindFramebuffer(36160,e.__webglFramebuffer);if(!b.depthTexture||!b.depthTexture.isDepthTexture)throw Error(&quot;renderTarget.depthTexture must be an instance of THREE.DepthTexture&quot;);d.get(b.depthTexture).__webglTexture&&
b.depthTexture.image.width===b.width&&b.depthTexture.image.height===b.height||(b.depthTexture.image.width=b.width,b.depthTexture.image.height=b.height,b.depthTexture.needsUpdate=!0);n(b.depthTexture,0);e=d.get(b.depthTexture).__webglTexture;if(1026===b.depthTexture.format)a.framebufferTexture2D(36160,36096,3553,e,0);else if(1027===b.depthTexture.format)a.framebufferTexture2D(36160,33306,3553,e,0);else throw Error(&quot;Unknown depthTexture format&quot;);}else if(h)for(e.__webglDepthbuffer=[],h=0;6>h;h++)a.bindFramebuffer(36160,
e.__webglFramebuffer[h]),e.__webglDepthbuffer[h]=a.createRenderbuffer(),y(e.__webglDepthbuffer[h],b,!1);else a.bindFramebuffer(36160,e.__webglFramebuffer),e.__webglDepthbuffer=a.createRenderbuffer(),y(e.__webglDepthbuffer,b,!1);a.bindFramebuffer(36160,null)}};this.updateRenderTargetMipmap=function(a){var b=a.texture,e=m(a)||Ca;if(k(b,e)){e=a.isWebGLCubeRenderTarget?34067:3553;var f=d.get(b).__webglTexture;c.bindTexture(e,f);p(e,b,a.width,a.height);c.bindTexture(e,null)}};this.updateMultisampleRenderTarget=
function(b){if(b.isWebGLMultisampleRenderTarget)if(Ca){var c=d.get(b);a.bindFramebuffer(36008,c.__webglMultisampledFramebuffer);a.bindFramebuffer(36009,c.__webglFramebuffer);var e=b.width,f=b.height,g=16384;b.depthBuffer&&(g|=256);b.stencilBuffer&&(g|=1024);a.blitFramebuffer(0,0,e,f,0,0,e,f,g,9728);a.bindFramebuffer(36160,c.__webglMultisampledFramebuffer)}else console.warn(&quot;THREE.WebGLRenderer: WebGLMultisampleRenderTarget can only be used with WebGL2.&quot;)};this.safeSetTexture2D=function(a,b){a&&a.isWebGLRenderTarget&&
(!1===P&&(console.warn(&quot;THREE.WebGLTextures.safeSetTexture2D: don't use render targets as textures. Use their .texture property instead.&quot;),P=!0),a=a.texture);n(a,b)};this.safeSetTextureCube=function(a,b){a&&a.isWebGLCubeRenderTarget&&(!1===O&&(console.warn(&quot;THREE.WebGLTextures.safeSetTextureCube: don't use cube render targets as textures. Use their .texture property instead.&quot;),O=!0),a=a.texture);a&&a.isCubeTexture||Array.isArray(a.image)&&6===a.image.length?w(a,b):z(a,b)}}function Zh(a,b,c){var d=
c.isWebGL2;return{convert:function(a){if(1009===a)return 5121;if(1017===a)return 32819;if(1018===a)return 32820;if(1019===a)return 33635;if(1010===a)return 5120;if(1011===a)return 5122;if(1012===a)return 5123;if(1013===a)return 5124;if(1014===a)return 5125;if(1015===a)return 5126;if(1016===a){if(d)return 5131;var c=b.get(&quot;OES_texture_half_float&quot;);return null!==c?c.HALF_FLOAT_OES:null}if(1021===a)return 6406;if(1022===a)return 6407;if(1023===a)return 6408;if(1024===a)return 6409;if(1025===a)return 6410;
if(1026===a)return 6402;if(1027===a)return 34041;if(1028===a)return 6403;if(1029===a)return 36244;if(1030===a)return 33319;if(1031===a)return 33320;if(1032===a)return 36248;if(1033===a)return 36249;if(33776===a||33777===a||33778===a||33779===a)if(c=b.get(&quot;WEBGL_compressed_texture_s3tc&quot;),null!==c){if(33776===a)return c.COMPRESSED_RGB_S3TC_DXT1_EXT;if(33777===a)return c.COMPRESSED_RGBA_S3TC_DXT1_EXT;if(33778===a)return c.COMPRESSED_RGBA_S3TC_DXT3_EXT;if(33779===a)return c.COMPRESSED_RGBA_S3TC_DXT5_EXT}else return null;
if(35840===a||35841===a||35842===a||35843===a)if(c=b.get(&quot;WEBGL_compressed_texture_pvrtc&quot;),null!==c){if(35840===a)return c.COMPRESSED_RGB_PVRTC_4BPPV1_IMG;if(35841===a)return c.COMPRESSED_RGB_PVRTC_2BPPV1_IMG;if(35842===a)return c.COMPRESSED_RGBA_PVRTC_4BPPV1_IMG;if(35843===a)return c.COMPRESSED_RGBA_PVRTC_2BPPV1_IMG}else return null;if(36196===a)return c=b.get(&quot;WEBGL_compressed_texture_etc1&quot;),null!==c?c.COMPRESSED_RGB_ETC1_WEBGL:null;if(37492===a||37496===a)if(c=b.get(&quot;WEBGL_compressed_texture_etc&quot;),
null!==c){if(37492===a)return c.COMPRESSED_RGB8_ETC2;if(37496===a)return c.COMPRESSED_RGBA8_ETC2_EAC}if(37808===a||37809===a||37810===a||37811===a||37812===a||37813===a||37814===a||37815===a||37816===a||37817===a||37818===a||37819===a||37820===a||37821===a||37840===a||37841===a||37842===a||37843===a||37844===a||37845===a||37846===a||37847===a||37848===a||37849===a||37850===a||37851===a||37852===a||37853===a)return c=b.get(&quot;WEBGL_compressed_texture_astc&quot;),null!==c?a:null;if(36492===a)return c=b.get(&quot;EXT_texture_compression_bptc&quot;),
null!==c?a:null;if(1020===a){if(d)return 34042;c=b.get(&quot;WEBGL_depth_texture&quot;);return null!==c?c.UNSIGNED_INT_24_8_WEBGL:null}}}}function Pe(a){aa.call(this);this.cameras=a||[]}function Mc(){F.call(this);this.type=&quot;Group&quot;}function $h(a,b){function c(a){var b=q.get(a.inputSource);b&&(b.targetRay&&b.targetRay.dispatchEvent({type:a.type}),b.grip&&b.grip.dispatchEvent({type:a.type}))}function d(){q.forEach(function(a,b){a.targetRay&&(a.targetRay.dispatchEvent({type:&quot;disconnected&quot;,data:b}),a.targetRay.visible=
!1);a.grip&&(a.grip.dispatchEvent({type:&quot;disconnected&quot;,data:b}),a.grip.visible=!1)});q.clear();a.setFramebuffer(null);a.setRenderTarget(a.getRenderTarget());y.stop();h.isPresenting=!1;h.dispatchEvent({type:&quot;sessionend&quot;})}function e(a){k=a;y.setContext(l);y.start();h.isPresenting=!0;h.dispatchEvent({type:&quot;sessionstart&quot;})}function f(a){for(var b=l.inputSources,c=0;c<r.length;c++)q.set(b[c],r[c]);for(c=0;c<a.removed.length;c++){b=a.removed[c];var d=q.get(b);d&&(d.targetRay&&d.targetRay.dispatchEvent({type:&quot;disconnected&quot;,
data:b}),d.grip&&d.grip.dispatchEvent({type:&quot;disconnected&quot;,data:b}),q.delete(b))}for(c=0;c<a.added.length;c++)if(b=a.added[c],d=q.get(b))d.targetRay&&d.targetRay.dispatchEvent({type:&quot;connected&quot;,data:b}),d.grip&&d.grip.dispatchEvent({type:&quot;connected&quot;,data:b})}function g(a,b){null===b?a.matrixWorld.copy(a.matrix):a.matrixWorld.multiplyMatrices(b.matrixWorld,a.matrix);a.matrixWorldInverse.getInverse(a.matrixWorld)}var h=this,l=null,m=1,k=null,p=&quot;local-floor&quot;,x=null,r=[],q=new Map,v=new aa;v.layers.enable(1);
v.viewport=new ka;var E=new aa;E.layers.enable(2);E.viewport=new ka;var w=new Pe([v,E]);w.layers.enable(1);w.layers.enable(2);var z=null,t=null;this.isPresenting=this.enabled=!1;this.getController=function(a){var b=r[a];void 0===b&&(b={},r[a]=b);void 0===b.targetRay&&(b.targetRay=new Mc,b.targetRay.matrixAutoUpdate=!1,b.targetRay.visible=!1);return b.targetRay};this.getControllerGrip=function(a){var b=r[a];void 0===b&&(b={},r[a]=b);void 0===b.grip&&(b.grip=new Mc,b.grip.matrixAutoUpdate=!1,b.grip.visible=
!1);return b.grip};this.setFramebufferScaleFactor=function(a){m=a;1==h.isPresenting&&console.warn(&quot;WebXRManager: Cannot change framebuffer scale while presenting VR content&quot;)};this.setReferenceSpaceType=function(a){p=a};this.getReferenceSpace=function(){return k};this.getSession=function(){return l};this.setSession=function(a){l=a;null!==l&&(l.addEventListener(&quot;select&quot;,c),l.addEventListener(&quot;selectstart&quot;,c),l.addEventListener(&quot;selectend&quot;,c),l.addEventListener(&quot;squeeze&quot;,c),l.addEventListener(&quot;squeezestart&quot;,
c),l.addEventListener(&quot;squeezeend&quot;,c),l.addEventListener(&quot;end&quot;,d),a=b.getContextAttributes(),a=new XRWebGLLayer(l,b,{antialias:a.antialias,alpha:a.alpha,depth:a.depth,stencil:a.stencil,framebufferScaleFactor:m}),l.updateRenderState({baseLayer:a}),l.requestReferenceSpace(p).then(e),l.addEventListener(&quot;inputsourceschange&quot;,f))};var U=new n,ca=new n;this.getCamera=function(a){w.near=E.near=v.near=a.near;w.far=E.far=v.far=a.far;if(z!==w.near||t!==w.far)l.updateRenderState({depthNear:w.near,depthFar:w.far}),
z=w.near,t=w.far;var b=a.parent,c=w.cameras;g(w,b);for(var d=0;d<c.length;d++)g(c[d],b);a.matrixWorld.copy(w.matrixWorld);a=a.children;d=0;for(b=a.length;d<b;d++)a[d].updateMatrixWorld(!0);U.setFromMatrixPosition(v.matrixWorld);ca.setFromMatrixPosition(E.matrixWorld);d=U.distanceTo(ca);var e=v.projectionMatrix.elements,f=E.projectionMatrix.elements,h=e[14]/(e[10]-1);a=e[14]/(e[10]+1);b=(e[9]+1)/e[5];c=(e[9]-1)/e[5];var m=(e[8]-1)/e[0],p=(f[8]+1)/f[0];f=h*m;e=h*p;p=d/(-m+p);m=p*-m;v.matrixWorld.decompose(w.position,
w.quaternion,w.scale);w.translateX(m);w.translateZ(p);w.matrixWorld.compose(w.position,w.quaternion,w.scale);w.matrixWorldInverse.getInverse(w.matrixWorld);h+=p;p=a+p;w.projectionMatrix.makePerspective(f-m,e+(d-m),b*a/p*h,c*a/p*h,h,p);return w};var B=null,y=new Ah;y.setAnimationLoop(function(b,c){x=c.getViewerPose(k);if(null!==x){var d=x.views,e=l.renderState.baseLayer;a.setFramebuffer(e.framebuffer);for(var f=0;f<d.length;f++){var g=d[f],h=e.getViewport(g),m=w.cameras[f];m.matrix.fromArray(g.transform.matrix);
m.projectionMatrix.fromArray(g.projectionMatrix);m.viewport.set(h.x,h.y,h.width,h.height);0===f&&w.matrix.copy(m.matrix)}}d=l.inputSources;for(f=0;f<r.length;f++)e=r[f],g=d[f],m=h=null,g&&(e.targetRay&&(h=c.getPose(g.targetRaySpace,k),null!==h&&(e.targetRay.matrix.fromArray(h.transform.matrix),e.targetRay.matrix.decompose(e.targetRay.position,e.targetRay.rotation,e.targetRay.scale))),e.grip&&g.gripSpace&&(m=c.getPose(g.gripSpace,k),null!==m&&(e.grip.matrix.fromArray(m.transform.matrix),e.grip.matrix.decompose(e.grip.position,
e.grip.rotation,e.grip.scale)))),e.targetRay&&(e.targetRay.visible=null!==h),e.grip&&(e.grip.visible=null!==m);B&&B(b,c)});this.setAnimationLoop=function(a){B=a};this.dispose=function(){}}function og(a){var b;function c(){ra=new zj(I);Fa=new xj(I,ra,a);!1===Fa.isWebGL2&&(ra.get(&quot;WEBGL_depth_texture&quot;),ra.get(&quot;OES_texture_float&quot;),ra.get(&quot;OES_texture_half_float&quot;),ra.get(&quot;OES_texture_half_float_linear&quot;),ra.get(&quot;OES_standard_derivatives&quot;),ra.get(&quot;OES_element_index_uint&quot;),ra.get(&quot;ANGLE_instanced_arrays&quot;));
ra.get(&quot;OES_texture_float_linear&quot;);qa=new Zh(I,ra,Fa);Y=new Ik(I,ra,Fa);Y.scissor(T.copy(ja).multiplyScalar(Q).floor());Y.viewport(Z.copy(R).multiplyScalar(Q).floor());aa=new Cj(I);X=new yk;ea=new Jk(I,ra,Y,X,Fa,qa,aa);oa=new uj(I,Fa);xa=new Aj(I,oa,aa);sa=new Fj(I,xa,oa,aa);ya=new Ej(I);ta=new xk(H,ra,Fa);wa=new Bk;va=new Hk;pa=new vj(H,Y,sa,ca);Aa=new wj(I,ra,aa,Fa);Ba=new Bj(I,ra,aa,Fa);aa.programs=ta.programs;H.capabilities=Fa;H.extensions=ra;H.properties=X;H.renderLists=wa;H.state=Y;H.info=aa}
function d(a){a.preventDefault();console.log(&quot;THREE.WebGLRenderer: Context Lost.&quot;);G=!0}function e(){console.log(&quot;THREE.WebGLRenderer: Context Restored.&quot;);G=!1;c()}function f(a){a=a.target;a.removeEventListener(&quot;dispose&quot;,f);g(a);X.remove(a)}function g(a){var b=X.get(a).program;a.program=void 0;void 0!==b&&ta.releaseProgram(b)}function h(a,b){a.render(function(a){H.renderBufferImmediate(a,b)})}function l(a,b,c,d){if(!1!==a.visible){if(a.layers.test(b.layers))if(a.isGroup)c=a.renderOrder;else if(a.isLOD)!0===
a.autoUpdate&&a.update(b);else if(a.isLight)A.pushLight(a),a.castShadow&&A.pushShadow(a);else if(a.isSprite){if(!a.frustumCulled||lg.intersectsSprite(a)){d&&Hb.setFromMatrixPosition(a.matrixWorld).applyMatrix4(Md);var e=sa.update(a),f=a.material;f.visible&&C.push(a,e,f,c,Hb.z,null)}}else if(a.isImmediateRenderObject)d&&Hb.setFromMatrixPosition(a.matrixWorld).applyMatrix4(Md),C.push(a,null,a.material,c,Hb.z,null);else if(a.isMesh||a.isLine||a.isPoints)if(a.isSkinnedMesh&&a.skeleton.frame!==aa.render.frame&&
(a.skeleton.update(),a.skeleton.frame=aa.render.frame),!a.frustumCulled||lg.intersectsObject(a))if(d&&Hb.setFromMatrixPosition(a.matrixWorld).applyMatrix4(Md),e=sa.update(a),f=a.material,Array.isArray(f))for(var g=e.groups,h=0,m=g.length;h<m;h++){var p=g[h],k=f[p.materialIndex];k&&k.visible&&C.push(a,e,k,c,Hb.z,p)}else f.visible&&C.push(a,e,f,c,Hb.z,null);a=a.children;h=0;for(m=a.length;h<m;h++)l(a[h],b,c,d)}}function m(a,b,c,d){for(var e=0,f=a.length;e<f;e++){var g=a[e],h=g.object,l=g.geometry,m=
void 0===d?g.material:d;g=g.group;if(c.isArrayCamera){ba=c;for(var p=c.cameras,u=0,q=p.length;u<q;u++){var x=p[u];h.layers.test(x.layers)&&(Y.viewport(Z.copy(x.viewport)),A.setupLights(x),k(h,b,x,l,m,g))}}else ba=null,k(h,b,c,l,m,g)}}function k(a,c,d,e,f,g){a.onBeforeRender(H,c,d,e,f,g);A=va.get(c,ba||d);a.modelViewMatrix.multiplyMatrices(d.matrixWorldInverse,a.matrixWorld);a.normalMatrix.getNormalMatrix(a.modelViewMatrix);if(a.isImmediateRenderObject){var l=x(d,c,f,a);Y.setMaterial(f);fa=b=null;
Oe=!1;h(a,l)}else H.renderBufferDirect(d,c,e,f,a,g);a.onAfterRender(H,c,d,e,f,g);A=va.get(c,ba||d)}function p(a,b,c){var d=X.get(a),e=A.state.lights,h=e.state.version;c=ta.getParameters(a,e.state,A.state.shadowsArray,b,Ua.numPlanes,Ua.numIntersection,c);var l=ta.getProgramCacheKey(c),m=d.program,p=!0;if(void 0===m)a.addEventListener(&quot;dispose&quot;,f);else if(m.cacheKey!==l)g(a);else{if(d.lightsStateVersion!==h)d.lightsStateVersion=h;else if(void 0!==c.shaderID)return;p=!1}p&&(m=ta.acquireProgram(c,l),
d.program=m,d.uniforms=c.uniforms,d.environment=a.isMeshStandardMaterial?b.environment:null,d.outputEncoding=H.outputEncoding,a.program=m);c=m.getAttributes();if(a.morphTargets)for(l=a.numSupportedMorphTargets=0;l<H.maxMorphTargets;l++)0<=c[&quot;morphTarget&quot;+l]&&a.numSupportedMorphTargets++;if(a.morphNormals)for(l=a.numSupportedMorphNormals=0;l<H.maxMorphNormals;l++)0<=c[&quot;morphNormal&quot;+l]&&a.numSupportedMorphNormals++;c=d.uniforms;if(!a.isShaderMaterial&&!a.isRawShaderMaterial||!0===a.clipping)d.numClippingPlanes=
Ua.numPlanes,d.numIntersection=Ua.numIntersection,c.clippingPlanes=Ua.uniform;d.fog=b.fog;d.needsLights=a.isMeshLambertMaterial||a.isMeshToonMaterial||a.isMeshPhongMaterial||a.isMeshStandardMaterial||a.isShadowMaterial||a.isShaderMaterial&&!0===a.lights;d.lightsStateVersion=h;d.needsLights&&(c.ambientLightColor.value=e.state.ambient,c.lightProbe.value=e.state.probe,c.directionalLights.value=e.state.directional,c.directionalLightShadows.value=e.state.directionalShadow,c.spotLights.value=e.state.spot,
c.spotLightShadows.value=e.state.spotShadow,c.rectAreaLights.value=e.state.rectArea,c.pointLights.value=e.state.point,c.pointLightShadows.value=e.state.pointShadow,c.hemisphereLights.value=e.state.hemi,c.directionalShadowMap.value=e.state.directionalShadowMap,c.directionalShadowMatrix.value=e.state.directionalShadowMatrix,c.spotShadowMap.value=e.state.spotShadowMap,c.spotShadowMatrix.value=e.state.spotShadowMatrix,c.pointShadowMap.value=e.state.pointShadowMap,c.pointShadowMatrix.value=e.state.pointShadowMatrix);
a=d.program.getUniforms();a=Eb.seqWithValue(a.seq,c);d.uniformsList=a}function x(a,b,c,d){ea.resetTextureUnits();var e=b.fog,f=c.isMeshStandardMaterial?b.environment:null,g=X.get(c),h=A.state.lights;na&&(mg||a!==V)&&Ua.setState(c.clippingPlanes,c.clipIntersection,c.clipShadows,a,g,a===V&&c.id===Ne);c.version===g.__version?void 0===g.program?p(c,b,d):c.fog&&g.fog!==e?p(c,b,d):g.environment!==f?p(c,b,d):g.needsLights&&g.lightsStateVersion!==h.state.version?p(c,b,d):void 0===g.numClippingPlanes||g.numClippingPlanes===
Ua.numPlanes&&g.numIntersection===Ua.numIntersection?g.outputEncoding!==H.outputEncoding&&p(c,b,d):p(c,b,d):(p(c,b,d),g.__version=c.version);var l=!1,m=!1,k=!1;b=g.program;h=b.getUniforms();var u=g.uniforms;Y.useProgram(b.program)&&(k=m=l=!0);c.id!==Ne&&(Ne=c.id,m=!0);if(l||V!==a){h.setValue(I,&quot;projectionMatrix&quot;,a.projectionMatrix);Fa.logarithmicDepthBuffer&&h.setValue(I,&quot;logDepthBufFC&quot;,2/(Math.log(a.far+1)/Math.LN2));V!==a&&(V=a,k=m=!0);if(c.isShaderMaterial||c.isMeshPhongMaterial||c.isMeshToonMaterial||
c.isMeshStandardMaterial||c.envMap)l=h.map.cameraPosition,void 0!==l&&l.setValue(I,Hb.setFromMatrixPosition(a.matrixWorld));(c.isMeshPhongMaterial||c.isMeshToonMaterial||c.isMeshLambertMaterial||c.isMeshBasicMaterial||c.isMeshStandardMaterial||c.isShaderMaterial)&&h.setValue(I,&quot;isOrthographic&quot;,!0===a.isOrthographicCamera);(c.isMeshPhongMaterial||c.isMeshToonMaterial||c.isMeshLambertMaterial||c.isMeshBasicMaterial||c.isMeshStandardMaterial||c.isShaderMaterial||c.skinning)&&h.setValue(I,&quot;viewMatrix&quot;,
a.matrixWorldInverse)}if(c.skinning&&(h.setOptional(I,d,&quot;bindMatrix&quot;),h.setOptional(I,d,&quot;bindMatrixInverse&quot;),a=d.skeleton))if(l=a.bones,Fa.floatVertexTextures){if(void 0===a.boneTexture){l=Math.sqrt(4*l.length);l=L.ceilPowerOfTwo(l);l=Math.max(l,4);var x=new Float32Array(l*l*4);x.set(a.boneMatrices);var v=new ac(x,l,l,1023,1015);a.boneMatrices=x;a.boneTexture=v;a.boneTextureSize=l}h.setValue(I,&quot;boneTexture&quot;,a.boneTexture,ea);h.setValue(I,&quot;boneTextureSize&quot;,a.boneTextureSize)}else h.setOptional(I,a,
&quot;boneMatrices&quot;);if(m||g.receiveShadow!==d.receiveShadow)g.receiveShadow=d.receiveShadow,h.setValue(I,&quot;receiveShadow&quot;,d.receiveShadow);if(m){h.setValue(I,&quot;toneMappingExposure&quot;,H.toneMappingExposure);h.setValue(I,&quot;toneMappingWhitePoint&quot;,H.toneMappingWhitePoint);g.needsLights&&(m=k,u.ambientLightColor.needsUpdate=m,u.lightProbe.needsUpdate=m,u.directionalLights.needsUpdate=m,u.directionalLightShadows.needsUpdate=m,u.pointLights.needsUpdate=m,u.pointLightShadows.needsUpdate=m,u.spotLights.needsUpdate=
m,u.spotLightShadows.needsUpdate=m,u.rectAreaLights.needsUpdate=m,u.hemisphereLights.needsUpdate=m);e&&c.fog&&(u.fogColor.value.copy(e.color),e.isFog?(u.fogNear.value=e.near,u.fogFar.value=e.far):e.isFogExp2&&(u.fogDensity.value=e.density));if(c.isMeshBasicMaterial)r(u,c);else if(c.isMeshLambertMaterial)r(u,c),c.emissiveMap&&(u.emissiveMap.value=c.emissiveMap);else if(c.isMeshToonMaterial)r(u,c),u.specular.value.copy(c.specular),u.shininess.value=Math.max(c.shininess,1E-4),c.gradientMap&&(u.gradientMap.value=
c.gradientMap),c.emissiveMap&&(u.emissiveMap.value=c.emissiveMap),c.bumpMap&&(u.bumpMap.value=c.bumpMap,u.bumpScale.value=c.bumpScale,1===c.side&&(u.bumpScale.value*=-1)),c.normalMap&&(u.normalMap.value=c.normalMap,u.normalScale.value.copy(c.normalScale),1===c.side&&u.normalScale.value.negate()),c.displacementMap&&(u.displacementMap.value=c.displacementMap,u.displacementScale.value=c.displacementScale,u.displacementBias.value=c.displacementBias);else if(c.isMeshPhongMaterial)r(u,c),u.specular.value.copy(c.specular),
u.shininess.value=Math.max(c.shininess,1E-4),c.emissiveMap&&(u.emissiveMap.value=c.emissiveMap),c.bumpMap&&(u.bumpMap.value=c.bumpMap,u.bumpScale.value=c.bumpScale,1===c.side&&(u.bumpScale.value*=-1)),c.normalMap&&(u.normalMap.value=c.normalMap,u.normalScale.value.copy(c.normalScale),1===c.side&&u.normalScale.value.negate()),c.displacementMap&&(u.displacementMap.value=c.displacementMap,u.displacementScale.value=c.displacementScale,u.displacementBias.value=c.displacementBias);else if(c.isMeshStandardMaterial)r(u,
c,f),c.isMeshPhysicalMaterial?(q(u,c,f),u.reflectivity.value=c.reflectivity,u.clearcoat.value=c.clearcoat,u.clearcoatRoughness.value=c.clearcoatRoughness,c.sheen&&u.sheen.value.copy(c.sheen),c.clearcoatMap&&(u.clearcoatMap.value=c.clearcoatMap),c.clearcoatRoughnessMap&&(u.clearcoatRoughnessMap.value=c.clearcoatRoughnessMap),c.clearcoatNormalMap&&(u.clearcoatNormalScale.value.copy(c.clearcoatNormalScale),u.clearcoatNormalMap.value=c.clearcoatNormalMap,1===c.side&&u.clearcoatNormalScale.value.negate()),
u.transparency.value=c.transparency):q(u,c,f);else if(c.isMeshMatcapMaterial)r(u,c),c.matcap&&(u.matcap.value=c.matcap),c.bumpMap&&(u.bumpMap.value=c.bumpMap,u.bumpScale.value=c.bumpScale,1===c.side&&(u.bumpScale.value*=-1)),c.normalMap&&(u.normalMap.value=c.normalMap,u.normalScale.value.copy(c.normalScale),1===c.side&&u.normalScale.value.negate()),c.displacementMap&&(u.displacementMap.value=c.displacementMap,u.displacementScale.value=c.displacementScale,u.displacementBias.value=c.displacementBias);
else if(c.isMeshDepthMaterial)r(u,c),c.displacementMap&&(u.displacementMap.value=c.displacementMap,u.displacementScale.value=c.displacementScale,u.displacementBias.value=c.displacementBias);else if(c.isMeshDistanceMaterial)r(u,c),c.displacementMap&&(u.displacementMap.value=c.displacementMap,u.displacementScale.value=c.displacementScale,u.displacementBias.value=c.displacementBias),u.referencePosition.value.copy(c.referencePosition),u.nearDistance.value=c.nearDistance,u.farDistance.value=c.farDistance;
else if(c.isMeshNormalMaterial)r(u,c),c.bumpMap&&(u.bumpMap.value=c.bumpMap,u.bumpScale.value=c.bumpScale,1===c.side&&(u.bumpScale.value*=-1)),c.normalMap&&(u.normalMap.value=c.normalMap,u.normalScale.value.copy(c.normalScale),1===c.side&&u.normalScale.value.negate()),c.displacementMap&&(u.displacementMap.value=c.displacementMap,u.displacementScale.value=c.displacementScale,u.displacementBias.value=c.displacementBias);else if(c.isLineBasicMaterial)u.diffuse.value.copy(c.color),u.opacity.value=c.opacity,
c.isLineDashedMaterial&&(u.dashSize.value=c.dashSize,u.totalSize.value=c.dashSize+c.gapSize,u.scale.value=c.scale);else if(c.isPointsMaterial){u.diffuse.value.copy(c.color);u.opacity.value=c.opacity;u.size.value=c.size*Q;u.scale.value=.5*J;c.map&&(u.map.value=c.map);c.alphaMap&&(u.alphaMap.value=c.alphaMap);if(c.map)var n=c.map;else c.alphaMap&&(n=c.alphaMap);void 0!==n&&(!0===n.matrixAutoUpdate&&n.updateMatrix(),u.uvTransform.value.copy(n.matrix))}else if(c.isSpriteMaterial){u.diffuse.value.copy(c.color);
u.opacity.value=c.opacity;u.rotation.value=c.rotation;c.map&&(u.map.value=c.map);c.alphaMap&&(u.alphaMap.value=c.alphaMap);if(c.map)var w=c.map;else c.alphaMap&&(w=c.alphaMap);void 0!==w&&(!0===w.matrixAutoUpdate&&w.updateMatrix(),u.uvTransform.value.copy(w.matrix))}else c.isShadowMaterial&&(u.color.value.copy(c.color),u.opacity.value=c.opacity);void 0!==u.ltc_1&&(u.ltc_1.value=D.LTC_1);void 0!==u.ltc_2&&(u.ltc_2.value=D.LTC_2);Eb.upload(I,g.uniformsList,u,ea);c.isShaderMaterial&&(c.uniformsNeedUpdate=
!1)}c.isShaderMaterial&&!0===c.uniformsNeedUpdate&&(Eb.upload(I,g.uniformsList,u,ea),c.uniformsNeedUpdate=!1);c.isSpriteMaterial&&h.setValue(I,&quot;center&quot;,d.center);h.setValue(I,&quot;modelViewMatrix&quot;,d.modelViewMatrix);h.setValue(I,&quot;normalMatrix&quot;,d.normalMatrix);h.setValue(I,&quot;modelMatrix&quot;,d.matrixWorld);return b}function r(a,b,c){a.opacity.value=b.opacity;b.color&&a.diffuse.value.copy(b.color);b.emissive&&a.emissive.value.copy(b.emissive).multiplyScalar(b.emissiveIntensity);b.map&&(a.map.value=b.map);b.alphaMap&&
(a.alphaMap.value=b.alphaMap);b.specularMap&&(a.specularMap.value=b.specularMap);if(c=b.envMap||c)a.envMap.value=c,a.flipEnvMap.value=c.isCubeTexture?-1:1,a.reflectivity.value=b.reflectivity,a.refractionRatio.value=b.refractionRatio,a.maxMipLevel.value=X.get(c).__maxMipLevel;b.lightMap&&(a.lightMap.value=b.lightMap,a.lightMapIntensity.value=b.lightMapIntensity);b.aoMap&&(a.aoMap.value=b.aoMap,a.aoMapIntensity.value=b.aoMapIntensity);if(b.map)var d=b.map;else b.specularMap?d=b.specularMap:b.displacementMap?
d=b.displacementMap:b.normalMap?d=b.normalMap:b.bumpMap?d=b.bumpMap:b.roughnessMap?d=b.roughnessMap:b.metalnessMap?d=b.metalnessMap:b.alphaMap?d=b.alphaMap:b.emissiveMap&&(d=b.emissiveMap);void 0!==d&&(d.isWebGLRenderTarget&&(d=d.texture),!0===d.matrixAutoUpdate&&d.updateMatrix(),a.uvTransform.value.copy(d.matrix));if(b.aoMap)var e=b.aoMap;else b.lightMap&&(e=b.lightMap);void 0!==e&&(e.isWebGLRenderTarget&&(e=e.texture),!0===e.matrixAutoUpdate&&e.updateMatrix(),a.uv2Transform.value.copy(e.matrix))}
function q(a,b,c){a.roughness.value=b.roughness;a.metalness.value=b.metalness;b.roughnessMap&&(a.roughnessMap.value=b.roughnessMap);b.metalnessMap&&(a.metalnessMap.value=b.metalnessMap);b.emissiveMap&&(a.emissiveMap.value=b.emissiveMap);b.bumpMap&&(a.bumpMap.value=b.bumpMap,a.bumpScale.value=b.bumpScale,1===b.side&&(a.bumpScale.value*=-1));b.normalMap&&(a.normalMap.value=b.normalMap,a.normalScale.value.copy(b.normalScale),1===b.side&&a.normalScale.value.negate());b.displacementMap&&(a.displacementMap.value=
b.displacementMap,a.displacementScale.value=b.displacementScale,a.displacementBias.value=b.displacementBias);if(b.envMap||c)a.envMapIntensity.value=b.envMapIntensity}a=a||{};var v=void 0!==a.canvas?a.canvas:document.createElementNS(&quot;http://www.w3.org/1999/xhtml&quot;,&quot;canvas&quot;),E=void 0!==a.context?a.context:null,w=void 0!==a.alpha?a.alpha:!1,z=void 0!==a.depth?a.depth:!0,ha=void 0!==a.stencil?a.stencil:!0,U=void 0!==a.antialias?a.antialias:!1,ca=void 0!==a.premultipliedAlpha?a.premultipliedAlpha:!0,B=
void 0!==a.preserveDrawingBuffer?a.preserveDrawingBuffer:!1,y=void 0!==a.powerPreference?a.powerPreference:&quot;default&quot;,F=void 0!==a.failIfMajorPerformanceCaveat?a.failIfMajorPerformanceCaveat:!1,C=null,A=null;this.domElement=v;this.debug={checkShaderErrors:!0};this.sortObjects=this.autoClearStencil=this.autoClearDepth=this.autoClearColor=this.autoClear=!0;this.clippingPlanes=[];this.localClippingEnabled=!1;this.gammaFactor=2;this.outputEncoding=3E3;this.physicallyCorrectLights=!1;this.toneMappingWhitePoint=
this.toneMappingExposure=this.toneMapping=1;this.maxMorphTargets=8;this.maxMorphNormals=4;var H=this,G=!1,M=null,N=0,K=0,O=null,W=null,Ne=-1;var fa=b=null;var Oe=!1;var V=null,ba=null,Z=new ka,T=new ka,da=null,S=v.width,J=v.height,Q=1,la=null,ma=null,R=new ka(0,0,S,J),ja=new ka(0,0,S,J),kg=!1,lg=new Hc,Ua=new yj,na=!1,mg=!1,Md=new P,Hb=new n;try{w={alpha:w,depth:z,stencil:ha,antialias:U,premultipliedAlpha:ca,preserveDrawingBuffer:B,powerPreference:y,failIfMajorPerformanceCaveat:F,xrCompatible:!0};
v.addEventListener(&quot;webglcontextlost&quot;,d,!1);v.addEventListener(&quot;webglcontextrestored&quot;,e,!1);var I=E||v.getContext(&quot;webgl&quot;,w)||v.getContext(&quot;experimental-webgl&quot;,w);if(null===I){if(null!==v.getContext(&quot;webgl&quot;))throw Error(&quot;Error creating WebGL context with your selected attributes.&quot;);throw Error(&quot;Error creating WebGL context.&quot;);}void 0===I.getShaderPrecisionFormat&&(I.getShaderPrecisionFormat=function(){return{rangeMin:1,rangeMax:1,precision:1}})}catch(ai){throw console.error(&quot;THREE.WebGLRenderer: &quot;+
ai.message),ai;}var ra,Fa,Y,aa,X,ea,oa,xa,sa,ta,wa,va,pa,ya,Aa,Ba,qa;c();var ua=new $h(H,I);this.xr=ua;var Ea=new Xh(H,sa,Fa.maxTextureSize);this.shadowMap=Ea;this.getContext=function(){return I};this.getContextAttributes=function(){return I.getContextAttributes()};this.forceContextLoss=function(){var a=ra.get(&quot;WEBGL_lose_context&quot;);a&&a.loseContext()};this.forceContextRestore=function(){var a=ra.get(&quot;WEBGL_lose_context&quot;);a&&a.restoreContext()};this.getPixelRatio=function(){return Q};this.setPixelRatio=
function(a){void 0!==a&&(Q=a,this.setSize(S,J,!1))};this.getSize=function(a){void 0===a&&(console.warn(&quot;WebGLRenderer: .getsize() now requires a Vector2 as an argument&quot;),a=new t);return a.set(S,J)};this.setSize=function(a,b,c){ua.isPresenting?console.warn(&quot;THREE.WebGLRenderer: Can't change size while VR device is presenting.&quot;):(S=a,J=b,v.width=Math.floor(a*Q),v.height=Math.floor(b*Q),!1!==c&&(v.style.width=a+&quot;px&quot;,v.style.height=b+&quot;px&quot;),this.setViewport(0,0,a,b))};this.getDrawingBufferSize=function(a){void 0===
a&&(console.warn(&quot;WebGLRenderer: .getdrawingBufferSize() now requires a Vector2 as an argument&quot;),a=new t);return a.set(S*Q,J*Q).floor()};this.setDrawingBufferSize=function(a,b,c){S=a;J=b;Q=c;v.width=Math.floor(a*c);v.height=Math.floor(b*c);this.setViewport(0,0,a,b)};this.getCurrentViewport=function(a){void 0===a&&(console.warn(&quot;WebGLRenderer: .getCurrentViewport() now requires a Vector4 as an argument&quot;),a=new ka);return a.copy(Z)};this.getViewport=function(a){return a.copy(R)};this.setViewport=function(a,
b,c,d){a.isVector4?R.set(a.x,a.y,a.z,a.w):R.set(a,b,c,d);Y.viewport(Z.copy(R).multiplyScalar(Q).floor())};this.getScissor=function(a){return a.copy(ja)};this.setScissor=function(a,b,c,d){a.isVector4?ja.set(a.x,a.y,a.z,a.w):ja.set(a,b,c,d);Y.scissor(T.copy(ja).multiplyScalar(Q).floor())};this.getScissorTest=function(){return kg};this.setScissorTest=function(a){Y.setScissorTest(kg=a)};this.setOpaqueSort=function(a){la=a};this.setTransparentSort=function(a){ma=a};this.getClearColor=function(){return pa.getClearColor()};
this.setClearColor=function(){pa.setClearColor.apply(pa,arguments)};this.getClearAlpha=function(){return pa.getClearAlpha()};this.setClearAlpha=function(){pa.setClearAlpha.apply(pa,arguments)};this.clear=function(a,b,c){var d=0;if(void 0===a||a)d|=16384;if(void 0===b||b)d|=256;if(void 0===c||c)d|=1024;I.clear(d)};this.clearColor=function(){this.clear(!0,!1,!1)};this.clearDepth=function(){this.clear(!1,!0,!1)};this.clearStencil=function(){this.clear(!1,!1,!0)};this.dispose=function(){v.removeEventListener(&quot;webglcontextlost&quot;,
d,!1);v.removeEventListener(&quot;webglcontextrestored&quot;,e,!1);wa.dispose();va.dispose();X.dispose();sa.dispose();ua.dispose();za.stop();this.forceContextLoss()};this.renderBufferImmediate=function(a,b){Y.initAttributes();var c=X.get(a);a.hasPositions&&!c.position&&(c.position=I.createBuffer());a.hasNormals&&!c.normal&&(c.normal=I.createBuffer());a.hasUvs&&!c.uv&&(c.uv=I.createBuffer());a.hasColors&&!c.color&&(c.color=I.createBuffer());b=b.getAttributes();a.hasPositions&&(I.bindBuffer(34962,c.position),
I.bufferData(34962,a.positionArray,35048),Y.enableAttribute(b.position),I.vertexAttribPointer(b.position,3,5126,!1,0,0));a.hasNormals&&(I.bindBuffer(34962,c.normal),I.bufferData(34962,a.normalArray,35048),Y.enableAttribute(b.normal),I.vertexAttribPointer(b.normal,3,5126,!1,0,0));a.hasUvs&&(I.bindBuffer(34962,c.uv),I.bufferData(34962,a.uvArray,35048),Y.enableAttribute(b.uv),I.vertexAttribPointer(b.uv,2,5126,!1,0,0));a.hasColors&&(I.bindBuffer(34962,c.color),I.bufferData(34962,a.colorArray,35048),Y.enableAttribute(b.color),
I.vertexAttribPointer(b.color,3,5126,!1,0,0));Y.disableUnusedAttributes();I.drawArrays(4,0,a.count);a.count=0};var Ga=new ob;this.renderBufferDirect=function(a,c,d,e,f,g){null===c&&(c=Ga);var h=f.isMesh&&0>f.matrixWorld.determinant(),l=x(a,c,e,f);Y.setMaterial(e,h);var m=!1;if(b!==d.id||fa!==l.id||Oe!==(!0===e.wireframe))b=d.id,fa=l.id,Oe=!0===e.wireframe,m=!0;if(e.morphTargets||e.morphNormals)ya.update(f,d,e,l),m=!0;a=d.index;c=d.attributes.position;if(null===a){if(void 0===c||0===c.count)return}else if(0===
a.count)return;var u=1;!0===e.wireframe&&(a=xa.getWireframeAttribute(d),u=2);h=Aa;if(null!==a){var p=oa.get(a);h=Ba;h.setIndex(p)}if(m){if(!1!==Fa.isWebGL2||!f.isInstancedMesh&&!d.isInstancedBufferGeometry||null!==ra.get(&quot;ANGLE_instanced_arrays&quot;)){Y.initAttributes();m=d.attributes;l=l.getAttributes();var k=e.defaultAttributeValues;for(ha in l){var q=l[ha];if(0<=q){var r=m[ha];if(void 0!==r){var n=r.normalized,v=r.itemSize,w=oa.get(r);if(void 0!==w){var E=w.buffer,z=w.type;w=w.bytesPerElement;if(r.isInterleavedBufferAttribute){var B=
r.data,t=B.stride;r=r.offset;B&&B.isInstancedInterleavedBuffer?(Y.enableAttributeAndDivisor(q,B.meshPerAttribute),void 0===d.maxInstancedCount&&(d.maxInstancedCount=B.meshPerAttribute*B.count)):Y.enableAttribute(q);I.bindBuffer(34962,E);I.vertexAttribPointer(q,v,z,n,t*w,r*w)}else r.isInstancedBufferAttribute?(Y.enableAttributeAndDivisor(q,r.meshPerAttribute),void 0===d.maxInstancedCount&&(d.maxInstancedCount=r.meshPerAttribute*r.count)):Y.enableAttribute(q),I.bindBuffer(34962,E),I.vertexAttribPointer(q,
v,z,n,0,0)}}else if(&quot;instanceMatrix&quot;===ha)w=oa.get(f.instanceMatrix),void 0!==w&&(E=w.buffer,z=w.type,Y.enableAttributeAndDivisor(q+0,1),Y.enableAttributeAndDivisor(q+1,1),Y.enableAttributeAndDivisor(q+2,1),Y.enableAttributeAndDivisor(q+3,1),I.bindBuffer(34962,E),I.vertexAttribPointer(q+0,4,z,!1,64,0),I.vertexAttribPointer(q+1,4,z,!1,64,16),I.vertexAttribPointer(q+2,4,z,!1,64,32),I.vertexAttribPointer(q+3,4,z,!1,64,48));else if(void 0!==k&&(n=k[ha],void 0!==n))switch(n.length){case 2:I.vertexAttrib2fv(q,
n);break;case 3:I.vertexAttrib3fv(q,n);break;case 4:I.vertexAttrib4fv(q,n);break;default:I.vertexAttrib1fv(q,n)}}}Y.disableUnusedAttributes()}null!==a&&I.bindBuffer(34963,p.buffer)}var ha=d.drawRange.start*u;m=null!==g?g.start*u:0;p=Math.max(ha,m);g=Math.max(0,Math.min(null!==a?a.count:c.count,ha+d.drawRange.count*u,m+(null!==g?g.count*u:Infinity))-1-p+1);0!==g&&(f.isMesh?!0===e.wireframe?(Y.setLineWidth(e.wireframeLinewidth*(null===O?Q:1)),h.setMode(1)):h.setMode(4):f.isLine?(e=e.linewidth,void 0===
e&&(e=1),Y.setLineWidth(e*(null===O?Q:1)),f.isLineSegments?h.setMode(1):f.isLineLoop?h.setMode(2):h.setMode(3)):f.isPoints?h.setMode(0):f.isSprite&&h.setMode(4),f.isInstancedMesh?h.renderInstances(d,p,g,f.count):d.isInstancedBufferGeometry?h.renderInstances(d,p,g,d.maxInstancedCount):h.render(p,g))};this.compile=function(a,b){A=va.get(a,b);A.init();a.traverse(function(a){a.isLight&&(A.pushLight(a),a.castShadow&&A.pushShadow(a))});A.setupLights(b);var c={};a.traverse(function(b){if(b.material)if(Array.isArray(b.material))for(var d=
0;d<b.material.length;d++)!1===b.material[d].uuid in c&&(p(b.material[d],a,b),c[b.material[d].uuid]=!0);else!1===b.material.uuid in c&&(p(b.material,a,b),c[b.material.uuid]=!0)})};var Da=null,za=new Ah;za.setAnimationLoop(function(a){ua.isPresenting||Da&&Da(a)});&quot;undefined&quot;!==typeof window&&za.setContext(window);this.setAnimationLoop=function(a){Da=a;ua.setAnimationLoop(a);za.start()};this.render=function(a,c,d,e){if(void 0!==d){console.warn(&quot;THREE.WebGLRenderer.render(): the renderTarget argument has been removed. Use .setRenderTarget() instead.&quot;);
var f=d}if(void 0!==e){console.warn(&quot;THREE.WebGLRenderer.render(): the forceClear argument has been removed. Use .clear() instead.&quot;);var g=e}c&&c.isCamera?G||(fa=b=null,Oe=!1,Ne=-1,V=null,!0===a.autoUpdate&&a.updateMatrixWorld(),null===c.parent&&c.updateMatrixWorld(),ua.enabled&&ua.isPresenting&&(c=ua.getCamera(c)),A=va.get(a,c),A.init(),a.onBeforeRender(H,a,c,f||O),Md.multiplyMatrices(c.projectionMatrix,c.matrixWorldInverse),lg.setFromProjectionMatrix(Md),mg=this.localClippingEnabled,na=Ua.init(this.clippingPlanes,
mg,c),C=wa.get(a,c),C.init(),l(a,c,0,H.sortObjects),C.finish(),!0===H.sortObjects&&C.sort(la,ma),na&&Ua.beginShadows(),Ea.render(A.state.shadowsArray,a,c),A.setupLights(c),na&&Ua.endShadows(),this.info.autoReset&&this.info.reset(),void 0!==f&&this.setRenderTarget(f),pa.render(C,a,c,g),d=C.opaque,e=C.transparent,a.overrideMaterial?(f=a.overrideMaterial,d.length&&m(d,a,c,f),e.length&&m(e,a,c,f)):(d.length&&m(d,a,c),e.length&&m(e,a,c)),a.onAfterRender(H,a,c),null!==O&&(ea.updateRenderTargetMipmap(O),
ea.updateMultisampleRenderTarget(O)),Y.buffers.depth.setTest(!0),Y.buffers.depth.setMask(!0),Y.buffers.color.setMask(!0),Y.setPolygonOffset(!1),A=C=null):console.error(&quot;THREE.WebGLRenderer.render: camera is not an instance of THREE.Camera.&quot;)};this.setFramebuffer=function(a){M!==a&&null===O&&I.bindFramebuffer(36160,a);M=a};this.getActiveCubeFace=function(){return N};this.getActiveMipmapLevel=function(){return K};this.getRenderTarget=function(){return O};this.setRenderTarget=function(a,b,c){O=a;N=b;
K=c;a&&void 0===X.get(a).__webglFramebuffer&&ea.setupRenderTarget(a);var d=M,e=!1;a?(d=X.get(a).__webglFramebuffer,a.isWebGLCubeRenderTarget?(d=d[b||0],e=!0):d=a.isWebGLMultisampleRenderTarget?X.get(a).__webglMultisampledFramebuffer:d,Z.copy(a.viewport),T.copy(a.scissor),da=a.scissorTest):(Z.copy(R).multiplyScalar(Q).floor(),T.copy(ja).multiplyScalar(Q).floor(),da=kg);W!==d&&(I.bindFramebuffer(36160,d),W=d);Y.viewport(Z);Y.scissor(T);Y.setScissorTest(da);e&&(a=X.get(a.texture),I.framebufferTexture2D(36160,
36064,34069+(b||0),a.__webglTexture,c||0))};this.readRenderTargetPixels=function(a,b,c,d,e,f,g){if(a&&a.isWebGLRenderTarget){var h=X.get(a).__webglFramebuffer;a.isWebGLCubeRenderTarget&&void 0!==g&&(h=h[g]);if(h){g=!1;h!==W&&(I.bindFramebuffer(36160,h),g=!0);try{var l=a.texture,m=l.format,u=l.type;1023!==m&&qa.convert(m)!==I.getParameter(35739)?console.error(&quot;THREE.WebGLRenderer.readRenderTargetPixels: renderTarget is not in RGBA or implementation defined format.&quot;):1009===u||qa.convert(u)===I.getParameter(35738)||
1015===u&&(Fa.isWebGL2||ra.get(&quot;OES_texture_float&quot;)||ra.get(&quot;WEBGL_color_buffer_float&quot;))||1016===u&&(Fa.isWebGL2?ra.get(&quot;EXT_color_buffer_float&quot;):ra.get(&quot;EXT_color_buffer_half_float&quot;))?36053===I.checkFramebufferStatus(36160)?0<=b&&b<=a.width-d&&0<=c&&c<=a.height-e&&I.readPixels(b,c,d,e,qa.convert(m),qa.convert(u),f):console.error(&quot;THREE.WebGLRenderer.readRenderTargetPixels: readPixels from renderTarget failed. Framebuffer not complete.&quot;):console.error(&quot;THREE.WebGLRenderer.readRenderTargetPixels: renderTarget is not in UnsignedByteType or implementation defined type.&quot;)}finally{g&&
I.bindFramebuffer(36160,W)}}}else console.error(&quot;THREE.WebGLRenderer.readRenderTargetPixels: renderTarget is not THREE.WebGLRenderTarget.&quot;)};this.copyFramebufferToTexture=function(a,b,c){void 0===c&&(c=0);var d=Math.pow(2,-c),e=Math.floor(b.image.width*d);d=Math.floor(b.image.height*d);var f=qa.convert(b.format);ea.setTexture2D(b,0);I.copyTexImage2D(3553,c,f,a.x,a.y,e,d,0);Y.unbindTexture()};this.copyTextureToTexture=function(a,b,c,d){var e=b.image.width,f=b.image.height,g=qa.convert(c.format),h=
qa.convert(c.type);ea.setTexture2D(c,0);b.isDataTexture?I.texSubImage2D(3553,d||0,a.x,a.y,e,f,g,h,b.image.data):I.texSubImage2D(3553,d||0,a.x,a.y,g,h,b.image);Y.unbindTexture()};this.initTexture=function(a){ea.setTexture2D(a,0);Y.unbindTexture()};&quot;undefined&quot;!==typeof __THREE_DEVTOOLS__&&__THREE_DEVTOOLS__.dispatchEvent(new CustomEvent(&quot;observe&quot;,{detail:this}))}function Qe(a,b){this.name=&quot;&quot;;this.color=new A(a);this.density=void 0!==b?b:2.5E-4}function Re(a,b,c){this.name=&quot;&quot;;this.color=new A(a);this.near=
void 0!==b?b:1;this.far=void 0!==c?c:1E3}function rb(a,b){this.array=a;this.stride=b;this.count=void 0!==a?a.length/b:0;this.usage=35044;this.updateRange={offset:0,count:-1};this.version=0}function Od(a,b,c,d){this.data=a;this.itemSize=b;this.offset=c;this.normalized=!0===d}function Ib(a){K.call(this);this.type=&quot;SpriteMaterial&quot;;this.color=new A(16777215);this.alphaMap=this.map=null;this.rotation=0;this.transparent=this.sizeAttenuation=!0;this.setValues(a)}function Pd(a){F.call(this);this.type=&quot;Sprite&quot;;
if(void 0===Nc){Nc=new C;var b=new Float32Array([-.5,-.5,0,0,0,.5,-.5,0,1,0,.5,.5,0,1,1,-.5,.5,0,0,1]);b=new rb(b,5);Nc.setIndex([0,1,2,0,2,3]);Nc.setAttribute(&quot;position&quot;,new Od(b,3,0,!1));Nc.setAttribute(&quot;uv&quot;,new Od(b,2,3,!1))}this.geometry=Nc;this.material=void 0!==a?a:new Ib;this.center=new t(.5,.5)}function Se(a,b,c,d,e,f){Oc.subVectors(a,c).addScalar(.5).multiply(d);void 0!==e?(Qd.x=f*Oc.x-e*Oc.y,Qd.y=e*Oc.x+f*Oc.y):Qd.copy(Oc);a.copy(b);a.x+=Qd.x;a.y+=Qd.y;a.applyMatrix4(bi)}function Rd(){F.call(this);
this._currentLevel=0;this.type=&quot;LOD&quot;;Object.defineProperties(this,{levels:{enumerable:!0,value:[]}});this.autoUpdate=!0}function Te(a,b){a&&a.isGeometry&&console.error(&quot;THREE.SkinnedMesh no longer supports THREE.Geometry. Use THREE.BufferGeometry instead.&quot;);S.call(this,a,b);this.type=&quot;SkinnedMesh&quot;;this.bindMode=&quot;attached&quot;;this.bindMatrix=new P;this.bindMatrixInverse=new P}function Ue(a,b){a=a||[];this.bones=a.slice(0);this.boneMatrices=new Float32Array(16*this.bones.length);this.frame=-1;if(void 0===
b)this.calculateInverses();else if(this.bones.length===b.length)this.boneInverses=b.slice(0);else for(console.warn(&quot;THREE.Skeleton boneInverses is the wrong length.&quot;),this.boneInverses=[],a=0,b=this.bones.length;a<b;a++)this.boneInverses.push(new P)}function pg(){F.call(this);this.type=&quot;Bone&quot;}function Ve(a,b,c){S.call(this,a,b);this.instanceMatrix=new M(new Float32Array(16*c),16);this.count=c;this.frustumCulled=!1}function la(a){K.call(this);this.type=&quot;LineBasicMaterial&quot;;this.color=new A(16777215);
this.linewidth=1;this.linejoin=this.linecap=&quot;round&quot;;this.setValues(a)}function Ka(a,b,c){1===c&&console.error(&quot;THREE.Line: parameter THREE.LinePieces no longer supported. Use THREE.LineSegments instead.&quot;);F.call(this);this.type=&quot;Line&quot;;this.geometry=void 0!==a?a:new C;this.material=void 0!==b?b:new la}function ma(a,b){Ka.call(this,a,b);this.type=&quot;LineSegments&quot;}function We(a,b){Ka.call(this,a,b);this.type=&quot;LineLoop&quot;}function Va(a){K.call(this);this.type=&quot;PointsMaterial&quot;;this.color=new A(16777215);this.alphaMap=
this.map=null;this.size=1;this.sizeAttenuation=!0;this.morphTargets=!1;this.setValues(a)}function Pc(a,b){F.call(this);this.type=&quot;Points&quot;;this.geometry=void 0!==a?a:new C;this.material=void 0!==b?b:new Va;this.updateMorphTargets()}function qg(a,b,c,d,e,f,g){var h=rg.distanceSqToPoint(a);h<c&&(c=new n,rg.closestPointToPoint(a,c),c.applyMatrix4(d),a=e.ray.origin.distanceTo(c),a<e.near||a>e.far||f.push({distance:a,distanceToRay:Math.sqrt(h),point:c,index:b,face:null,object:g}))}function sg(a,b,c,d,e,
f,g,h,l){V.call(this,a,b,c,d,e,f,g,h,l);this.format=void 0!==g?g:1022;this.minFilter=void 0!==f?f:1006;this.magFilter=void 0!==e?e:1006;this.generateMipmaps=!1}function Qc(a,b,c,d,e,f,g,h,l,m,u,p){V.call(this,null,f,g,h,l,m,d,e,u,p);this.image={width:b,height:c};this.mipmaps=a;this.generateMipmaps=this.flipY=!1}function Sd(a,b,c,d,e,f,g,h,l){V.call(this,a,b,c,d,e,f,g,h,l);this.needsUpdate=!0}function Td(a,b,c,d,e,f,g,h,l,m){m=void 0!==m?m:1026;if(1026!==m&&1027!==m)throw Error(&quot;DepthTexture format must be either THREE.DepthFormat or THREE.DepthStencilFormat&quot;);
void 0===c&&1026===m&&(c=1012);void 0===c&&1027===m&&(c=1020);V.call(this,null,d,e,f,g,h,m,c,l);this.image={width:a,height:b};this.magFilter=void 0!==g?g:1003;this.minFilter=void 0!==h?h:1003;this.generateMipmaps=this.flipY=!1}function Rc(a){C.call(this);this.type=&quot;WireframeGeometry&quot;;var b=[],c,d,e,f=[0,0],g={},h=[&quot;a&quot;,&quot;b&quot;,&quot;c&quot;];if(a&&a.isGeometry){var l=a.faces;var m=0;for(d=l.length;m<d;m++){var u=l[m];for(c=0;3>c;c++){var p=u[h[c]];var k=u[h[(c+1)%3]];f[0]=Math.min(p,k);f[1]=Math.max(p,k);p=f[0]+
&quot;,&quot;+f[1];void 0===g[p]&&(g[p]={index1:f[0],index2:f[1]})}}for(p in g)m=g[p],h=a.vertices[m.index1],b.push(h.x,h.y,h.z),h=a.vertices[m.index2],b.push(h.x,h.y,h.z)}else if(a&&a.isBufferGeometry)if(h=new n,null!==a.index){l=a.attributes.position;u=a.index;var r=a.groups;0===r.length&&(r=[{start:0,count:u.count,materialIndex:0}]);a=0;for(e=r.length;a<e;++a)for(m=r[a],c=m.start,d=m.count,m=c,d=c+d;m<d;m+=3)for(c=0;3>c;c++)p=u.getX(m+c),k=u.getX(m+(c+1)%3),f[0]=Math.min(p,k),f[1]=Math.max(p,k),p=f[0]+&quot;,&quot;+
f[1],void 0===g[p]&&(g[p]={index1:f[0],index2:f[1]});for(p in g)m=g[p],h.fromBufferAttribute(l,m.index1),b.push(h.x,h.y,h.z),h.fromBufferAttribute(l,m.index2),b.push(h.x,h.y,h.z)}else for(l=a.attributes.position,m=0,d=l.count/3;m<d;m++)for(c=0;3>c;c++)g=3*m+c,h.fromBufferAttribute(l,g),b.push(h.x,h.y,h.z),g=3*m+(c+1)%3,h.fromBufferAttribute(l,g),b.push(h.x,h.y,h.z);this.setAttribute(&quot;position&quot;,new y(b,3))}function Ud(a,b,c){N.call(this);this.type=&quot;ParametricGeometry&quot;;this.parameters={func:a,slices:b,
stacks:c};this.fromBufferGeometry(new Sc(a,b,c));this.mergeVertices()}function Sc(a,b,c){C.call(this);this.type=&quot;ParametricBufferGeometry&quot;;this.parameters={func:a,slices:b,stacks:c};var d=[],e=[],f=[],g=[],h=new n,l=new n,m=new n,u=new n,p=new n,k,r;3>a.length&&console.error(&quot;THREE.ParametricGeometry: Function must now modify a Vector3 as third parameter.&quot;);var q=b+1;for(k=0;k<=c;k++){var v=k/c;for(r=0;r<=b;r++){var E=r/b;a(E,v,l);e.push(l.x,l.y,l.z);0<=E-1E-5?(a(E-1E-5,v,m),u.subVectors(l,m)):(a(E+
1E-5,v,m),u.subVectors(m,l));0<=v-1E-5?(a(E,v-1E-5,m),p.subVectors(l,m)):(a(E,v+1E-5,m),p.subVectors(m,l));h.crossVectors(u,p).normalize();f.push(h.x,h.y,h.z);g.push(E,v)}}for(k=0;k<c;k++)for(r=0;r<b;r++)a=k*q+r+1,h=(k+1)*q+r+1,l=(k+1)*q+r,d.push(k*q+r,a,l),d.push(a,h,l);this.setIndex(d);this.setAttribute(&quot;position&quot;,new y(e,3));this.setAttribute(&quot;normal&quot;,new y(f,3));this.setAttribute(&quot;uv&quot;,new y(g,2))}function Vd(a,b,c,d){N.call(this);this.type=&quot;PolyhedronGeometry&quot;;this.parameters={vertices:a,indices:b,
radius:c,detail:d};this.fromBufferGeometry(new Ga(a,b,c,d));this.mergeVertices()}function Ga(a,b,c,d){function e(a){h.push(a.x,a.y,a.z)}function f(b,c){b*=3;c.x=a[b+0];c.y=a[b+1];c.z=a[b+2]}function g(a,b,c,d){0>d&&1===a.x&&(l[b]=a.x-1);0===c.x&&0===c.z&&(l[b]=d/2/Math.PI+.5)}C.call(this);this.type=&quot;PolyhedronBufferGeometry&quot;;this.parameters={vertices:a,indices:b,radius:c,detail:d};c=c||1;d=d||0;var h=[],l=[];(function(a){for(var c=new n,d=new n,g=new n,h=0;h<b.length;h+=3){f(b[h+0],c);f(b[h+1],d);
f(b[h+2],g);var l,m,k=c,w=d,z=g,t=Math.pow(2,a),U=[];for(m=0;m<=t;m++){U[m]=[];var ca=k.clone().lerp(z,m/t),B=w.clone().lerp(z,m/t),A=t-m;for(l=0;l<=A;l++)U[m][l]=0===l&&m===t?ca:ca.clone().lerp(B,l/A)}for(m=0;m<t;m++)for(l=0;l<2*(t-m)-1;l++)k=Math.floor(l/2),0===l%2?(e(U[m][k+1]),e(U[m+1][k]),e(U[m][k])):(e(U[m][k+1]),e(U[m+1][k+1]),e(U[m+1][k]))}})(d);(function(a){for(var b=new n,c=0;c<h.length;c+=3)b.x=h[c+0],b.y=h[c+1],b.z=h[c+2],b.normalize().multiplyScalar(a),h[c+0]=b.x,h[c+1]=b.y,h[c+2]=b.z})(c);
(function(){for(var a=new n,b=0;b<h.length;b+=3)a.x=h[b+0],a.y=h[b+1],a.z=h[b+2],l.push(Math.atan2(a.z,-a.x)/2/Math.PI+.5,1-(Math.atan2(-a.y,Math.sqrt(a.x*a.x+a.z*a.z))/Math.PI+.5));a=new n;b=new n;for(var c=new n,d=new n,e=new t,f=new t,k=new t,E=0,w=0;E<h.length;E+=9,w+=6){a.set(h[E+0],h[E+1],h[E+2]);b.set(h[E+3],h[E+4],h[E+5]);c.set(h[E+6],h[E+7],h[E+8]);e.set(l[w+0],l[w+1]);f.set(l[w+2],l[w+3]);k.set(l[w+4],l[w+5]);d.copy(a).add(b).add(c).divideScalar(3);var z=Math.atan2(d.z,-d.x);g(e,w+0,a,z);
g(f,w+2,b,z);g(k,w+4,c,z)}for(a=0;a<l.length;a+=6)b=l[a+0],c=l[a+2],d=l[a+4],e=Math.min(b,c,d),.9<Math.max(b,c,d)&&.1>e&&(.2>b&&(l[a+0]+=1),.2>c&&(l[a+2]+=1),.2>d&&(l[a+4]+=1))})();this.setAttribute(&quot;position&quot;,new y(h,3));this.setAttribute(&quot;normal&quot;,new y(h.slice(),3));this.setAttribute(&quot;uv&quot;,new y(l,2));0===d?this.computeVertexNormals():this.normalizeNormals()}function Wd(a,b){N.call(this);this.type=&quot;TetrahedronGeometry&quot;;this.parameters={radius:a,detail:b};this.fromBufferGeometry(new Tc(a,b));this.mergeVertices()}
function Tc(a,b){Ga.call(this,[1,1,1,-1,-1,1,-1,1,-1,1,-1,-1],[2,1,0,0,3,2,1,3,0,2,3,1],a,b);this.type=&quot;TetrahedronBufferGeometry&quot;;this.parameters={radius:a,detail:b}}function Xd(a,b){N.call(this);this.type=&quot;OctahedronGeometry&quot;;this.parameters={radius:a,detail:b};this.fromBufferGeometry(new cc(a,b));this.mergeVertices()}function cc(a,b){Ga.call(this,[1,0,0,-1,0,0,0,1,0,0,-1,0,0,0,1,0,0,-1],[0,2,4,0,4,3,0,3,5,0,5,2,1,2,5,1,5,3,1,3,4,1,4,2],a,b);this.type=&quot;OctahedronBufferGeometry&quot;;this.parameters=
{radius:a,detail:b}}function Yd(a,b){N.call(this);this.type=&quot;IcosahedronGeometry&quot;;this.parameters={radius:a,detail:b};this.fromBufferGeometry(new Uc(a,b));this.mergeVertices()}function Uc(a,b){var c=(1+Math.sqrt(5))/2;Ga.call(this,[-1,c,0,1,c,0,-1,-c,0,1,-c,0,0,-1,c,0,1,c,0,-1,-c,0,1,-c,c,0,-1,c,0,1,-c,0,-1,-c,0,1],[0,11,5,0,5,1,0,1,7,0,7,10,0,10,11,1,5,9,5,11,4,11,10,2,10,7,6,7,1,8,3,9,4,3,4,2,3,2,6,3,6,8,3,8,9,4,9,5,2,4,11,6,2,10,8,6,7,9,8,1],a,b);this.type=&quot;IcosahedronBufferGeometry&quot;;this.parameters=
{radius:a,detail:b}}function Zd(a,b){N.call(this);this.type=&quot;DodecahedronGeometry&quot;;this.parameters={radius:a,detail:b};this.fromBufferGeometry(new Vc(a,b));this.mergeVertices()}function Vc(a,b){var c=(1+Math.sqrt(5))/2,d=1/c;Ga.call(this,[-1,-1,-1,-1,-1,1,-1,1,-1,-1,1,1,1,-1,-1,1,-1,1,1,1,-1,1,1,1,0,-d,-c,0,-d,c,0,d,-c,0,d,c,-d,-c,0,-d,c,0,d,-c,0,d,c,0,-c,0,-d,c,0,-d,-c,0,d,c,0,d],[3,11,7,3,7,15,3,15,13,7,19,17,7,17,6,7,6,15,17,4,8,17,8,10,17,10,6,8,0,16,8,16,2,8,2,10,0,12,1,0,1,18,0,18,16,6,10,2,
6,2,13,6,13,15,2,16,18,2,18,3,2,3,13,18,1,9,18,9,11,18,11,3,4,14,12,4,12,0,4,0,8,11,9,5,11,5,19,11,19,7,19,5,14,19,14,4,19,4,17,1,12,14,1,14,5,1,5,9],a,b);this.type=&quot;DodecahedronBufferGeometry&quot;;this.parameters={radius:a,detail:b}}function $d(a,b,c,d,e,f){N.call(this);this.type=&quot;TubeGeometry&quot;;this.parameters={path:a,tubularSegments:b,radius:c,radialSegments:d,closed:e};void 0!==f&&console.warn(&quot;THREE.TubeGeometry: taper has been removed.&quot;);a=new dc(a,b,c,d,e);this.tangents=a.tangents;this.normals=
a.normals;this.binormals=a.binormals;this.fromBufferGeometry(a);this.mergeVertices()}function dc(a,b,c,d,e){function f(e){u=a.getPointAt(e/b,u);var f=g.normals[e];e=g.binormals[e];for(x=0;x<=d;x++){var m=x/d*Math.PI*2,k=Math.sin(m);m=-Math.cos(m);l.x=m*f.x+k*e.x;l.y=m*f.y+k*e.y;l.z=m*f.z+k*e.z;l.normalize();q.push(l.x,l.y,l.z);h.x=u.x+c*l.x;h.y=u.y+c*l.y;h.z=u.z+c*l.z;r.push(h.x,h.y,h.z)}}C.call(this);this.type=&quot;TubeBufferGeometry&quot;;this.parameters={path:a,tubularSegments:b,radius:c,radialSegments:d,
closed:e};b=b||64;c=c||1;d=d||8;e=e||!1;var g=a.computeFrenetFrames(b,e);this.tangents=g.tangents;this.normals=g.normals;this.binormals=g.binormals;var h=new n,l=new n,m=new t,u=new n,k,x,r=[],q=[],v=[],E=[];for(k=0;k<b;k++)f(k);f(!1===e?b:0);for(k=0;k<=b;k++)for(x=0;x<=d;x++)m.x=k/b,m.y=x/d,v.push(m.x,m.y);(function(){for(x=1;x<=b;x++)for(k=1;k<=d;k++){var a=(d+1)*x+(k-1),c=(d+1)*x+k,e=(d+1)*(x-1)+k;E.push((d+1)*(x-1)+(k-1),a,e);E.push(a,c,e)}})();this.setIndex(E);this.setAttribute(&quot;position&quot;,new y(r,
3));this.setAttribute(&quot;normal&quot;,new y(q,3));this.setAttribute(&quot;uv&quot;,new y(v,2))}function ae(a,b,c,d,e,f,g){N.call(this);this.type=&quot;TorusKnotGeometry&quot;;this.parameters={radius:a,tube:b,tubularSegments:c,radialSegments:d,p:e,q:f};void 0!==g&&console.warn(&quot;THREE.TorusKnotGeometry: heightScale has been deprecated. Use .scale( x, y, z ) instead.&quot;);this.fromBufferGeometry(new Wc(a,b,c,d,e,f));this.mergeVertices()}function Wc(a,b,c,d,e,f){function g(a,b,c,d,e){var f=Math.sin(a);b=c/b*a;c=Math.cos(b);e.x=d*
(2+c)*.5*Math.cos(a);e.y=d*(2+c)*f*.5;e.z=d*Math.sin(b)*.5}C.call(this);this.type=&quot;TorusKnotBufferGeometry&quot;;this.parameters={radius:a,tube:b,tubularSegments:c,radialSegments:d,p:e,q:f};a=a||1;b=b||.4;c=Math.floor(c)||64;d=Math.floor(d)||8;e=e||2;f=f||3;var h=[],l=[],m=[],k=[],p,x=new n,r=new n,q=new n,v=new n,E=new n,w=new n,z=new n;for(p=0;p<=c;++p){var t=p/c*e*Math.PI*2;g(t,e,f,a,q);g(t+.01,e,f,a,v);w.subVectors(v,q);z.addVectors(v,q);E.crossVectors(w,z);z.crossVectors(E,w);E.normalize();z.normalize();
for(t=0;t<=d;++t){var U=t/d*Math.PI*2,ca=-b*Math.cos(U);U=b*Math.sin(U);x.x=q.x+(ca*z.x+U*E.x);x.y=q.y+(ca*z.y+U*E.y);x.z=q.z+(ca*z.z+U*E.z);l.push(x.x,x.y,x.z);r.subVectors(x,q).normalize();m.push(r.x,r.y,r.z);k.push(p/c);k.push(t/d)}}for(t=1;t<=c;t++)for(p=1;p<=d;p++)a=(d+1)*t+(p-1),b=(d+1)*t+p,e=(d+1)*(t-1)+p,h.push((d+1)*(t-1)+(p-1),a,e),h.push(a,b,e);this.setIndex(h);this.setAttribute(&quot;position&quot;,new y(l,3));this.setAttribute(&quot;normal&quot;,new y(m,3));this.setAttribute(&quot;uv&quot;,new y(k,2))}function be(a,
b,c,d,e){N.call(this);this.type=&quot;TorusGeometry&quot;;this.parameters={radius:a,tube:b,radialSegments:c,tubularSegments:d,arc:e};this.fromBufferGeometry(new Xc(a,b,c,d,e));this.mergeVertices()}function Xc(a,b,c,d,e){C.call(this);this.type=&quot;TorusBufferGeometry&quot;;this.parameters={radius:a,tube:b,radialSegments:c,tubularSegments:d,arc:e};a=a||1;b=b||.4;c=Math.floor(c)||8;d=Math.floor(d)||6;e=e||2*Math.PI;var f=[],g=[],h=[],l=[],m=new n,k=new n,p=new n,x,r;for(x=0;x<=c;x++)for(r=0;r<=d;r++){var q=r/d*e,v=x/
c*Math.PI*2;k.x=(a+b*Math.cos(v))*Math.cos(q);k.y=(a+b*Math.cos(v))*Math.sin(q);k.z=b*Math.sin(v);g.push(k.x,k.y,k.z);m.x=a*Math.cos(q);m.y=a*Math.sin(q);p.subVectors(k,m).normalize();h.push(p.x,p.y,p.z);l.push(r/d);l.push(x/c)}for(x=1;x<=c;x++)for(r=1;r<=d;r++)a=(d+1)*(x-1)+r-1,b=(d+1)*(x-1)+r,e=(d+1)*x+r,f.push((d+1)*x+r-1,a,e),f.push(a,b,e);this.setIndex(f);this.setAttribute(&quot;position&quot;,new y(g,3));this.setAttribute(&quot;normal&quot;,new y(h,3));this.setAttribute(&quot;uv&quot;,new y(l,2))}function ci(a,b,c,d,e){for(var f,
g=0,h=b,l=c-d;h<c;h+=d)g+=(a[l]-a[h])*(a[h+1]+a[l+1]),l=h;if(e===0<g)for(e=b;e<c;e+=d)f=di(e,a[e],a[e+1],f);else for(e=c-d;e>=b;e-=d)f=di(e,a[e],a[e+1],f);f&&ec(f,f.next)&&(ce(f),f=f.next);return f}function de(a,b){if(!a)return a;b||(b=a);do{var c=!1;if(a.steiner||!ec(a,a.next)&&0!==xa(a.prev,a,a.next))a=a.next;else{ce(a);a=b=a.prev;if(a===a.next)break;c=!0}}while(c||a!==b);return b}function ee(a,b,c,d,e,f,g){if(a){if(!g&&f){var h=a,l=h;do null===l.z&&(l.z=tg(l.x,l.y,d,e,f)),l.prevZ=l.prev,l=l.nextZ=
l.next;while(l!==h);l.prevZ.nextZ=null;l.prevZ=null;h=l;var m,k,p,x,r=1;do{l=h;var q=h=null;for(k=0;l;){k++;var n=l;for(m=p=0;m<r&&(p++,n=n.nextZ,n);m++);for(x=r;0<p||0<x&&n;)0!==p&&(0===x||!n||l.z<=n.z)?(m=l,l=l.nextZ,p--):(m=n,n=n.nextZ,x--),q?q.nextZ=m:h=m,m.prevZ=q,q=m;l=n}q.nextZ=null;r*=2}while(1<k)}for(h=a;a.prev!==a.next;){l=a.prev;n=a.next;if(f)q=Kk(a,d,e,f);else a:if(q=a,k=q.prev,p=q,r=q.next,0<=xa(k,p,r))q=!1;else{for(m=q.next.next;m!==q.prev;){if(Yc(k.x,k.y,p.x,p.y,r.x,r.y,m.x,m.y)&&0<=
xa(m.prev,m,m.next)){q=!1;break a}m=m.next}q=!0}if(q)b.push(l.i/c),b.push(a.i/c),b.push(n.i/c),ce(a),h=a=n.next;else if(a=n,a===h){if(!g)ee(de(a),b,c,d,e,f,1);else if(1===g){g=b;h=c;l=a;do n=l.prev,q=l.next.next,!ec(n,q)&&ei(n,l,l.next,q)&&fe(n,q)&&fe(q,n)&&(g.push(n.i/h),g.push(l.i/h),g.push(q.i/h),ce(l),ce(l.next),l=a=q),l=l.next;while(l!==a);a=l;ee(a,b,c,d,e,f,2)}else if(2===g)a:{g=a;do{for(h=g.next.next;h!==g.prev;){if(l=g.i!==h.i){l=g;n=h;if(q=l.next.i!==n.i&&l.prev.i!==n.i){b:{q=l;do{if(q.i!==
l.i&&q.next.i!==l.i&&q.i!==n.i&&q.next.i!==n.i&&ei(q,q.next,l,n)){q=!0;break b}q=q.next}while(q!==l);q=!1}q=!q}if(q=q&&fe(l,n)&&fe(n,l)){q=l;k=!1;p=(l.x+n.x)/2;n=(l.y+n.y)/2;do q.y>n!==q.next.y>n&&q.next.y!==q.y&&p<(q.next.x-q.x)*(n-q.y)/(q.next.y-q.y)+q.x&&(k=!k),q=q.next;while(q!==l);q=k}l=q}if(l){a=fi(g,h);g=de(g,g.next);a=de(a,a.next);ee(g,b,c,d,e,f);ee(a,b,c,d,e,f);break a}h=h.next}g=g.next}while(g!==a)}break}}}}function Kk(a,b,c,d){var e=a.prev,f=a.next;if(0<=xa(e,a,f))return!1;var g=e.x>a.x?
e.x>f.x?e.x:f.x:a.x>f.x?a.x:f.x,h=e.y>a.y?e.y>f.y?e.y:f.y:a.y>f.y?a.y:f.y,l=tg(e.x<a.x?e.x<f.x?e.x:f.x:a.x<f.x?a.x:f.x,e.y<a.y?e.y<f.y?e.y:f.y:a.y<f.y?a.y:f.y,b,c,d);b=tg(g,h,b,c,d);c=a.prevZ;for(d=a.nextZ;c&&c.z>=l&&d&&d.z<=b;){if(c!==a.prev&&c!==a.next&&Yc(e.x,e.y,a.x,a.y,f.x,f.y,c.x,c.y)&&0<=xa(c.prev,c,c.next))return!1;c=c.prevZ;if(d!==a.prev&&d!==a.next&&Yc(e.x,e.y,a.x,a.y,f.x,f.y,d.x,d.y)&&0<=xa(d.prev,d,d.next))return!1;d=d.nextZ}for(;c&&c.z>=l;){if(c!==a.prev&&c!==a.next&&Yc(e.x,e.y,a.x,a.y,
f.x,f.y,c.x,c.y)&&0<=xa(c.prev,c,c.next))return!1;c=c.prevZ}for(;d&&d.z<=b;){if(d!==a.prev&&d!==a.next&&Yc(e.x,e.y,a.x,a.y,f.x,f.y,d.x,d.y)&&0<=xa(d.prev,d,d.next))return!1;d=d.nextZ}return!0}function Lk(a,b){return a.x-b.x}function Mk(a,b){var c=b,d=a.x,e=a.y,f=-Infinity;do{if(e<=c.y&&e>=c.next.y&&c.next.y!==c.y){var g=c.x+(e-c.y)*(c.next.x-c.x)/(c.next.y-c.y);if(g<=d&&g>f){f=g;if(g===d){if(e===c.y)return c;if(e===c.next.y)return c.next}var h=c.x<c.next.x?c:c.next}}c=c.next}while(c!==b);if(!h)return null;
if(d===f)return h.prev;b=h;g=h.x;var l=h.y,m=Infinity;for(c=h.next;c!==b;){if(d>=c.x&&c.x>=g&&d!==c.x&&Yc(e<l?d:f,e,g,l,e<l?f:d,e,c.x,c.y)){var k=Math.abs(e-c.y)/(d-c.x);(k<m||k===m&&c.x>h.x)&&fe(c,a)&&(h=c,m=k)}c=c.next}return h}function tg(a,b,c,d,e){a=32767*(a-c)*e;b=32767*(b-d)*e;a=(a|a<<8)&16711935;a=(a|a<<4)&252645135;a=(a|a<<2)&858993459;b=(b|b<<8)&16711935;b=(b|b<<4)&252645135;b=(b|b<<2)&858993459;return(a|a<<1)&1431655765|((b|b<<1)&1431655765)<<1}function Nk(a){var b=a,c=a;do{if(b.x<c.x||
b.x===c.x&&b.y<c.y)c=b;b=b.next}while(b!==a);return c}function Yc(a,b,c,d,e,f,g,h){return 0<=(e-g)*(b-h)-(a-g)*(f-h)&&0<=(a-g)*(d-h)-(c-g)*(b-h)&&0<=(c-g)*(f-h)-(e-g)*(d-h)}function xa(a,b,c){return(b.y-a.y)*(c.x-b.x)-(b.x-a.x)*(c.y-b.y)}function ec(a,b){return a.x===b.x&&a.y===b.y}function ei(a,b,c,d){return ec(a,c)&&ec(b,d)||ec(a,d)&&ec(c,b)?!0:0<xa(a,b,c)!==0<xa(a,b,d)&&0<xa(c,d,a)!==0<xa(c,d,b)}function fe(a,b){return 0>xa(a.prev,a,a.next)?0<=xa(a,b,a.next)&&0<=xa(a,a.prev,b):0>xa(a,b,a.prev)||
0>xa(a,a.next,b)}function fi(a,b){var c=new ug(a.i,a.x,a.y),d=new ug(b.i,b.x,b.y),e=a.next,f=b.prev;a.next=b;b.prev=a;c.next=e;e.prev=c;d.next=c;c.prev=d;f.next=d;d.prev=f;return d}function di(a,b,c,d){a=new ug(a,b,c);d?(a.next=d.next,a.prev=d,d.next.prev=a,d.next=a):(a.prev=a,a.next=a);return a}function ce(a){a.next.prev=a.prev;a.prev.next=a.next;a.prevZ&&(a.prevZ.nextZ=a.nextZ);a.nextZ&&(a.nextZ.prevZ=a.prevZ)}function ug(a,b,c){this.i=a;this.x=b;this.y=c;this.nextZ=this.prevZ=this.z=this.next=
this.prev=null;this.steiner=!1}function gi(a){var b=a.length;2<b&&a[b-1].equals(a[0])&&a.pop()}function hi(a,b){for(var c=0;c<b.length;c++)a.push(b[c].x),a.push(b[c].y)}function fc(a,b){N.call(this);this.type=&quot;ExtrudeGeometry&quot;;this.parameters={shapes:a,options:b};this.fromBufferGeometry(new fb(a,b));this.mergeVertices()}function fb(a,b){function c(a){function c(a,b,c){b||console.error(&quot;THREE.ExtrudeGeometry: vec does not exist&quot;);return b.clone().multiplyScalar(c).add(a)}function g(a,b,c){var d=a.x-
b.x;var e=a.y-b.y;var f=c.x-a.x;var g=c.y-a.y,h=d*d+e*e;if(Math.abs(d*g-e*f)>Number.EPSILON){var l=Math.sqrt(h),m=Math.sqrt(f*f+g*g);h=b.x-e/l;b=b.y+d/l;g=((c.x-g/m-h)*g-(c.y+f/m-b)*f)/(d*g-e*f);f=h+d*g-a.x;d=b+e*g-a.y;e=f*f+d*d;if(2>=e)return new t(f,d);e=Math.sqrt(e/2)}else a=!1,d>Number.EPSILON?f>Number.EPSILON&&(a=!0):d<-Number.EPSILON?f<-Number.EPSILON&&(a=!0):Math.sign(e)===Math.sign(g)&&(a=!0),a?(f=-e,e=Math.sqrt(h)):(f=d,d=e,e=Math.sqrt(h/2));return new t(f/e,d/e)}function h(a,b){for(J=a.length;0<=
--J;){var c=J;var f=J-1;0>f&&(f=a.length-1);var g,h=z+2*y;for(g=0;g<h;g++){var l=W*g,m=W*(g+1),k=b+f+l,u=b+f+m;m=b+c+m;q(b+c+l);q(k);q(m);q(k);q(u);q(m);l=e.length/3;l=F.generateSideWallUV(d,e,l-6,l-3,l-2,l-1);v(l[0]);v(l[1]);v(l[3]);v(l[1]);v(l[2]);v(l[3])}}}function l(a,b,c){E.push(a);E.push(b);E.push(c)}function k(a,b,c){q(a);q(b);q(c);a=e.length/3;a=F.generateTopUV(d,e,a-3,a-2,a-1);v(a[0]);v(a[1]);v(a[2])}function q(a){e.push(E[3*a]);e.push(E[3*a+1]);e.push(E[3*a+2])}function v(a){f.push(a.x);
f.push(a.y)}var E=[],w=void 0!==b.curveSegments?b.curveSegments:12,z=void 0!==b.steps?b.steps:1,ha=void 0!==b.depth?b.depth:100,U=void 0!==b.bevelEnabled?b.bevelEnabled:!0,ca=void 0!==b.bevelThickness?b.bevelThickness:6,B=void 0!==b.bevelSize?b.bevelSize:ca-2,A=void 0!==b.bevelOffset?b.bevelOffset:0,y=void 0!==b.bevelSegments?b.bevelSegments:3,C=b.extrudePath,F=void 0!==b.UVGenerator?b.UVGenerator:Ok;void 0!==b.amount&&(console.warn(&quot;THREE.ExtrudeBufferGeometry: amount has been renamed to depth.&quot;),
ha=b.amount);var H=!1;if(C){var D=C.getSpacedPoints(z);H=!0;U=!1;var G=C.computeFrenetFrames(z,!1);var M=new n;var N=new n;var K=new n}U||(A=B=ca=y=0);var O;w=a.extractPoints(w);a=w.shape;var L=w.holes;if(!sb.isClockWise(a)){a=a.reverse();var fa=0;for(O=L.length;fa<O;fa++){var P=L[fa];sb.isClockWise(P)&&(L[fa]=P.reverse())}}var X=sb.triangulateShape(a,L),V=a;fa=0;for(O=L.length;fa<O;fa++)P=L[fa],a=a.concat(P);var Z,W=a.length,T,ba=X.length;w=[];var J=0;var Q=V.length;var S=Q-1;for(Z=J+1;J<Q;J++,S++,
Z++)S===Q&&(S=0),Z===Q&&(Z=0),w[J]=g(V[J],V[S],V[Z]);C=[];var ea=w.concat();fa=0;for(O=L.length;fa<O;fa++){P=L[fa];var aa=[];J=0;Q=P.length;S=Q-1;for(Z=J+1;J<Q;J++,S++,Z++)S===Q&&(S=0),Z===Q&&(Z=0),aa[J]=g(P[J],P[S],P[Z]);C.push(aa);ea=ea.concat(aa)}for(S=0;S<y;S++){Q=S/y;var da=ca*Math.cos(Q*Math.PI/2);Z=B*Math.sin(Q*Math.PI/2)+A;J=0;for(Q=V.length;J<Q;J++){var R=c(V[J],w[J],Z);l(R.x,R.y,-da)}fa=0;for(O=L.length;fa<O;fa++)for(P=L[fa],aa=C[fa],J=0,Q=P.length;J<Q;J++)R=c(P[J],aa[J],Z),l(R.x,R.y,-da)}Z=
B+A;for(J=0;J<W;J++)R=U?c(a[J],ea[J],Z):a[J],H?(N.copy(G.normals[0]).multiplyScalar(R.x),M.copy(G.binormals[0]).multiplyScalar(R.y),K.copy(D[0]).add(N).add(M),l(K.x,K.y,K.z)):l(R.x,R.y,0);for(Q=1;Q<=z;Q++)for(J=0;J<W;J++)R=U?c(a[J],ea[J],Z):a[J],H?(N.copy(G.normals[Q]).multiplyScalar(R.x),M.copy(G.binormals[Q]).multiplyScalar(R.y),K.copy(D[Q]).add(N).add(M),l(K.x,K.y,K.z)):l(R.x,R.y,ha/z*Q);for(S=y-1;0<=S;S--){Q=S/y;da=ca*Math.cos(Q*Math.PI/2);Z=B*Math.sin(Q*Math.PI/2)+A;J=0;for(Q=V.length;J<Q;J++)R=
c(V[J],w[J],Z),l(R.x,R.y,ha+da);fa=0;for(O=L.length;fa<O;fa++)for(P=L[fa],aa=C[fa],J=0,Q=P.length;J<Q;J++)R=c(P[J],aa[J],Z),H?l(R.x,R.y+D[z-1].y,D[z-1].x+da):l(R.x,R.y,ha+da)}(function(){var a=e.length/3;if(U){var b=0*W;for(J=0;J<ba;J++)T=X[J],k(T[2]+b,T[1]+b,T[0]+b);b=W*(z+2*y);for(J=0;J<ba;J++)T=X[J],k(T[0]+b,T[1]+b,T[2]+b)}else{for(J=0;J<ba;J++)T=X[J],k(T[2],T[1],T[0]);for(J=0;J<ba;J++)T=X[J],k(T[0]+W*z,T[1]+W*z,T[2]+W*z)}d.addGroup(a,e.length/3-a,0)})();(function(){var a=e.length/3,b=0;h(V,b);
b+=V.length;fa=0;for(O=L.length;fa<O;fa++)P=L[fa],h(P,b),b+=P.length;d.addGroup(a,e.length/3-a,1)})()}C.call(this);this.type=&quot;ExtrudeBufferGeometry&quot;;this.parameters={shapes:a,options:b};a=Array.isArray(a)?a:[a];for(var d=this,e=[],f=[],g=0,h=a.length;g<h;g++)c(a[g]);this.setAttribute(&quot;position&quot;,new y(e,3));this.setAttribute(&quot;uv&quot;,new y(f,2));this.computeVertexNormals()}function ii(a,b,c){c.shapes=[];if(Array.isArray(a))for(var d=0,e=a.length;d<e;d++)c.shapes.push(a[d].uuid);else c.shapes.push(a.uuid);
void 0!==b.extrudePath&&(c.options.extrudePath=b.extrudePath.toJSON());return c}function ge(a,b){N.call(this);this.type=&quot;TextGeometry&quot;;this.parameters={text:a,parameters:b};this.fromBufferGeometry(new Zc(a,b));this.mergeVertices()}function Zc(a,b){b=b||{};var c=b.font;if(!c||!c.isFont)return console.error(&quot;THREE.TextGeometry: font parameter is not an instance of THREE.Font.&quot;),new N;a=c.generateShapes(a,b.size);b.depth=void 0!==b.height?b.height:50;void 0===b.bevelThickness&&(b.bevelThickness=10);
void 0===b.bevelSize&&(b.bevelSize=8);void 0===b.bevelEnabled&&(b.bevelEnabled=!1);fb.call(this,a,b);this.type=&quot;TextBufferGeometry&quot;}function he(a,b,c,d,e,f,g){N.call(this);this.type=&quot;SphereGeometry&quot;;this.parameters={radius:a,widthSegments:b,heightSegments:c,phiStart:d,phiLength:e,thetaStart:f,thetaLength:g};this.fromBufferGeometry(new gc(a,b,c,d,e,f,g));this.mergeVertices()}function gc(a,b,c,d,e,f,g){C.call(this);this.type=&quot;SphereBufferGeometry&quot;;this.parameters={radius:a,widthSegments:b,heightSegments:c,
phiStart:d,phiLength:e,thetaStart:f,thetaLength:g};a=a||1;b=Math.max(3,Math.floor(b)||8);c=Math.max(2,Math.floor(c)||6);d=void 0!==d?d:0;e=void 0!==e?e:2*Math.PI;f=void 0!==f?f:0;g=void 0!==g?g:Math.PI;var h=Math.min(f+g,Math.PI),l,m,k=0,p=[],x=new n,r=new n,q=[],v=[],E=[],w=[];for(m=0;m<=c;m++){var t=[],ha=m/c,U=0;0==m&&0==f?U=.5/b:m==c&&h==Math.PI&&(U=-.5/b);for(l=0;l<=b;l++){var ca=l/b;x.x=-a*Math.cos(d+ca*e)*Math.sin(f+ha*g);x.y=a*Math.cos(f+ha*g);x.z=a*Math.sin(d+ca*e)*Math.sin(f+ha*g);v.push(x.x,
x.y,x.z);r.copy(x).normalize();E.push(r.x,r.y,r.z);w.push(ca+U,1-ha);t.push(k++)}p.push(t)}for(m=0;m<c;m++)for(l=0;l<b;l++)a=p[m][l+1],d=p[m][l],e=p[m+1][l],g=p[m+1][l+1],(0!==m||0<f)&&q.push(a,d,g),(m!==c-1||h<Math.PI)&&q.push(d,e,g);this.setIndex(q);this.setAttribute(&quot;position&quot;,new y(v,3));this.setAttribute(&quot;normal&quot;,new y(E,3));this.setAttribute(&quot;uv&quot;,new y(w,2))}function ie(a,b,c,d,e,f){N.call(this);this.type=&quot;RingGeometry&quot;;this.parameters={innerRadius:a,outerRadius:b,thetaSegments:c,phiSegments:d,
thetaStart:e,thetaLength:f};this.fromBufferGeometry(new $c(a,b,c,d,e,f));this.mergeVertices()}function $c(a,b,c,d,e,f){C.call(this);this.type=&quot;RingBufferGeometry&quot;;this.parameters={innerRadius:a,outerRadius:b,thetaSegments:c,phiSegments:d,thetaStart:e,thetaLength:f};a=a||.5;b=b||1;e=void 0!==e?e:0;f=void 0!==f?f:2*Math.PI;c=void 0!==c?Math.max(3,c):8;d=void 0!==d?Math.max(1,d):1;var g=[],h=[],l=[],m=[],k=a,p=(b-a)/d,x=new n,r=new t,q,v;for(q=0;q<=d;q++){for(v=0;v<=c;v++)a=e+v/c*f,x.x=k*Math.cos(a),
x.y=k*Math.sin(a),h.push(x.x,x.y,x.z),l.push(0,0,1),r.x=(x.x/b+1)/2,r.y=(x.y/b+1)/2,m.push(r.x,r.y);k+=p}for(q=0;q<d;q++)for(b=q*(c+1),v=0;v<c;v++)a=v+b,e=a+c+1,f=a+c+2,k=a+1,g.push(a,e,k),g.push(e,f,k);this.setIndex(g);this.setAttribute(&quot;position&quot;,new y(h,3));this.setAttribute(&quot;normal&quot;,new y(l,3));this.setAttribute(&quot;uv&quot;,new y(m,2))}function je(a,b,c,d){N.call(this);this.type=&quot;LatheGeometry&quot;;this.parameters={points:a,segments:b,phiStart:c,phiLength:d};this.fromBufferGeometry(new ad(a,b,c,d));this.mergeVertices()}
function ad(a,b,c,d){C.call(this);this.type=&quot;LatheBufferGeometry&quot;;this.parameters={points:a,segments:b,phiStart:c,phiLength:d};b=Math.floor(b)||12;c=c||0;d=d||2*Math.PI;d=L.clamp(d,0,2*Math.PI);var e=[],f=[],g=[],h=1/b,l=new n,m=new t,k;for(k=0;k<=b;k++){var p=c+k*h*d;var x=Math.sin(p),r=Math.cos(p);for(p=0;p<=a.length-1;p++)l.x=a[p].x*x,l.y=a[p].y,l.z=a[p].x*r,f.push(l.x,l.y,l.z),m.x=k/b,m.y=p/(a.length-1),g.push(m.x,m.y)}for(k=0;k<b;k++)for(p=0;p<a.length-1;p++)c=p+k*a.length,h=c+a.length,l=c+a.length+
1,m=c+1,e.push(c,h,m),e.push(h,l,m);this.setIndex(e);this.setAttribute(&quot;position&quot;,new y(f,3));this.setAttribute(&quot;uv&quot;,new y(g,2));this.computeVertexNormals();if(d===2*Math.PI)for(d=this.attributes.normal.array,e=new n,f=new n,g=new n,c=b*a.length*3,p=k=0;k<a.length;k++,p+=3)e.x=d[p+0],e.y=d[p+1],e.z=d[p+2],f.x=d[c+p+0],f.y=d[c+p+1],f.z=d[c+p+2],g.addVectors(e,f).normalize(),d[p+0]=d[c+p+0]=g.x,d[p+1]=d[c+p+1]=g.y,d[p+2]=d[c+p+2]=g.z}function hc(a,b){N.call(this);this.type=&quot;ShapeGeometry&quot;;&quot;object&quot;===
typeof b&&(console.warn(&quot;THREE.ShapeGeometry: Options parameter has been removed.&quot;),b=b.curveSegments);this.parameters={shapes:a,curveSegments:b};this.fromBufferGeometry(new ic(a,b));this.mergeVertices()}function ic(a,b){function c(a){var c,h=e.length/3;a=a.extractPoints(b);var m=a.shape,k=a.holes;!1===sb.isClockWise(m)&&(m=m.reverse());a=0;for(c=k.length;a<c;a++){var u=k[a];!0===sb.isClockWise(u)&&(k[a]=u.reverse())}var n=sb.triangulateShape(m,k);a=0;for(c=k.length;a<c;a++)u=k[a],m=m.concat(u);a=
0;for(c=m.length;a<c;a++)u=m[a],e.push(u.x,u.y,0),f.push(0,0,1),g.push(u.x,u.y);a=0;for(c=n.length;a<c;a++)m=n[a],d.push(m[0]+h,m[1]+h,m[2]+h),l+=3}C.call(this);this.type=&quot;ShapeBufferGeometry&quot;;this.parameters={shapes:a,curveSegments:b};b=b||12;var d=[],e=[],f=[],g=[],h=0,l=0;if(!1===Array.isArray(a))c(a);else for(var m=0;m<a.length;m++)c(a[m]),this.addGroup(h,l,m),h+=l,l=0;this.setIndex(d);this.setAttribute(&quot;position&quot;,new y(e,3));this.setAttribute(&quot;normal&quot;,new y(f,3));this.setAttribute(&quot;uv&quot;,new y(g,
2))}function ji(a,b){b.shapes=[];if(Array.isArray(a))for(var c=0,d=a.length;c<d;c++)b.shapes.push(a[c].uuid);else b.shapes.push(a.uuid);return b}function bd(a,b){C.call(this);this.type=&quot;EdgesGeometry&quot;;this.parameters={thresholdAngle:b};var c=[];b=Math.cos(L.DEG2RAD*(void 0!==b?b:1));var d=[0,0],e={},f=[&quot;a&quot;,&quot;b&quot;,&quot;c&quot;];if(a.isBufferGeometry){var g=new N;g.fromBufferGeometry(a)}else g=a.clone();g.mergeVertices();g.computeFaceNormals();a=g.vertices;g=g.faces;for(var h=0,l=g.length;h<l;h++)for(var m=g[h],
k=0;3>k;k++){var p=m[f[k]];var n=m[f[(k+1)%3]];d[0]=Math.min(p,n);d[1]=Math.max(p,n);p=d[0]+&quot;,&quot;+d[1];void 0===e[p]?e[p]={index1:d[0],index2:d[1],face1:h,face2:void 0}:e[p].face2=h}for(p in e)if(d=e[p],void 0===d.face2||g[d.face1].normal.dot(g[d.face2].normal)<=b)f=a[d.index1],c.push(f.x,f.y,f.z),f=a[d.index2],c.push(f.x,f.y,f.z);this.setAttribute(&quot;position&quot;,new y(c,3))}function jc(a,b,c,d,e,f,g,h){N.call(this);this.type=&quot;CylinderGeometry&quot;;this.parameters={radiusTop:a,radiusBottom:b,height:c,radialSegments:d,
heightSegments:e,openEnded:f,thetaStart:g,thetaLength:h};this.fromBufferGeometry(new tb(a,b,c,d,e,f,g,h));this.mergeVertices()}function tb(a,b,c,d,e,f,g,h){function l(c){var e,f=new t,l=new n,u=0,v=!0===c?a:b,z=!0===c?1:-1;var A=q;for(e=1;e<=d;e++)p.push(0,E*z,0),x.push(0,z,0),r.push(.5,.5),q++;var y=q;for(e=0;e<=d;e++){var C=e/d*h+g,D=Math.cos(C);C=Math.sin(C);l.x=v*C;l.y=E*z;l.z=v*D;p.push(l.x,l.y,l.z);x.push(0,z,0);f.x=.5*D+.5;f.y=.5*C*z+.5;r.push(f.x,f.y);q++}for(e=0;e<d;e++)f=A+e,l=y+e,!0===
c?k.push(l,l+1,f):k.push(l+1,l,f),u+=3;m.addGroup(w,u,!0===c?1:2);w+=u}C.call(this);this.type=&quot;CylinderBufferGeometry&quot;;this.parameters={radiusTop:a,radiusBottom:b,height:c,radialSegments:d,heightSegments:e,openEnded:f,thetaStart:g,thetaLength:h};var m=this;a=void 0!==a?a:1;b=void 0!==b?b:1;c=c||1;d=Math.floor(d)||8;e=Math.floor(e)||1;f=void 0!==f?f:!1;g=void 0!==g?g:0;h=void 0!==h?h:2*Math.PI;var k=[],p=[],x=[],r=[],q=0,v=[],E=c/2,w=0;(function(){var f,l,u=new n,t=new n,B=0,A=(b-a)/c;for(l=0;l<=e;l++){var C=
[],y=l/e,D=y*(b-a)+a;for(f=0;f<=d;f++){var H=f/d,F=H*h+g,G=Math.sin(F);F=Math.cos(F);t.x=D*G;t.y=-y*c+E;t.z=D*F;p.push(t.x,t.y,t.z);u.set(G,A,F).normalize();x.push(u.x,u.y,u.z);r.push(H,1-y);C.push(q++)}v.push(C)}for(f=0;f<d;f++)for(l=0;l<e;l++)u=v[l+1][f],t=v[l+1][f+1],A=v[l][f+1],k.push(v[l][f],u,A),k.push(u,t,A),B+=6;m.addGroup(w,B,0);w+=B})();!1===f&&(0<a&&l(!0),0<b&&l(!1));this.setIndex(k);this.setAttribute(&quot;position&quot;,new y(p,3));this.setAttribute(&quot;normal&quot;,new y(x,3));this.setAttribute(&quot;uv&quot;,
new y(r,2))}function ke(a,b,c,d,e,f,g){jc.call(this,0,a,b,c,d,e,f,g);this.type=&quot;ConeGeometry&quot;;this.parameters={radius:a,height:b,radialSegments:c,heightSegments:d,openEnded:e,thetaStart:f,thetaLength:g}}function le(a,b,c,d,e,f,g){tb.call(this,0,a,b,c,d,e,f,g);this.type=&quot;ConeBufferGeometry&quot;;this.parameters={radius:a,height:b,radialSegments:c,heightSegments:d,openEnded:e,thetaStart:f,thetaLength:g}}function me(a,b,c,d){N.call(this);this.type=&quot;CircleGeometry&quot;;this.parameters={radius:a,segments:b,thetaStart:c,
thetaLength:d};this.fromBufferGeometry(new cd(a,b,c,d));this.mergeVertices()}function cd(a,b,c,d){C.call(this);this.type=&quot;CircleBufferGeometry&quot;;this.parameters={radius:a,segments:b,thetaStart:c,thetaLength:d};a=a||1;b=void 0!==b?Math.max(3,b):8;c=void 0!==c?c:0;d=void 0!==d?d:2*Math.PI;var e=[],f=[],g=[],h=[],l,m=new n,k=new t;f.push(0,0,0);g.push(0,0,1);h.push(.5,.5);var p=0;for(l=3;p<=b;p++,l+=3){var x=c+p/b*d;m.x=a*Math.cos(x);m.y=a*Math.sin(x);f.push(m.x,m.y,m.z);g.push(0,0,1);k.x=(f[l]/a+1)/
2;k.y=(f[l+1]/a+1)/2;h.push(k.x,k.y)}for(l=1;l<=b;l++)e.push(l,l+1,0);this.setIndex(e);this.setAttribute(&quot;position&quot;,new y(f,3));this.setAttribute(&quot;normal&quot;,new y(g,3));this.setAttribute(&quot;uv&quot;,new y(h,2))}function kc(a){K.call(this);this.type=&quot;ShadowMaterial&quot;;this.color=new A(0);this.transparent=!0;this.setValues(a)}function ub(a){Ba.call(this,a);this.type=&quot;RawShaderMaterial&quot;}function gb(a){K.call(this);this.defines={STANDARD:&quot;&quot;};this.type=&quot;MeshStandardMaterial&quot;;this.color=new A(16777215);this.roughness=
1;this.metalness=0;this.lightMap=this.map=null;this.lightMapIntensity=1;this.aoMap=null;this.aoMapIntensity=1;this.emissive=new A(0);this.emissiveIntensity=1;this.bumpMap=this.emissiveMap=null;this.bumpScale=1;this.normalMap=null;this.normalMapType=0;this.normalScale=new t(1,1);this.displacementMap=null;this.displacementScale=1;this.displacementBias=0;this.envMap=this.alphaMap=this.metalnessMap=this.roughnessMap=null;this.envMapIntensity=1;this.refractionRatio=.98;this.wireframe=!1;this.wireframeLinewidth=
1;this.wireframeLinejoin=this.wireframeLinecap=&quot;round&quot;;this.vertexTangents=this.morphNormals=this.morphTargets=this.skinning=!1;this.setValues(a)}function lc(a){gb.call(this);this.defines={STANDARD:&quot;&quot;,PHYSICAL:&quot;&quot;};this.type=&quot;MeshPhysicalMaterial&quot;;this.clearcoat=0;this.clearcoatMap=null;this.clearcoatRoughness=0;this.clearcoatRoughnessMap=null;this.clearcoatNormalScale=new t(1,1);this.clearcoatNormalMap=null;this.reflectivity=.5;this.sheen=null;this.transparency=0;this.setValues(a)}function Jb(a){K.call(this);
this.type=&quot;MeshPhongMaterial&quot;;this.color=new A(16777215);this.specular=new A(1118481);this.shininess=30;this.lightMap=this.map=null;this.lightMapIntensity=1;this.aoMap=null;this.aoMapIntensity=1;this.emissive=new A(0);this.emissiveIntensity=1;this.bumpMap=this.emissiveMap=null;this.bumpScale=1;this.normalMap=null;this.normalMapType=0;this.normalScale=new t(1,1);this.displacementMap=null;this.displacementScale=1;this.displacementBias=0;this.envMap=this.alphaMap=this.specularMap=null;this.combine=0;
this.reflectivity=1;this.refractionRatio=.98;this.wireframe=!1;this.wireframeLinewidth=1;this.wireframeLinejoin=this.wireframeLinecap=&quot;round&quot;;this.morphNormals=this.morphTargets=this.skinning=!1;this.setValues(a)}function mc(a){K.call(this);this.defines={TOON:&quot;&quot;};this.type=&quot;MeshToonMaterial&quot;;this.color=new A(16777215);this.specular=new A(1118481);this.shininess=30;this.lightMap=this.gradientMap=this.map=null;this.lightMapIntensity=1;this.aoMap=null;this.aoMapIntensity=1;this.emissive=new A(0);this.emissiveIntensity=
1;this.bumpMap=this.emissiveMap=null;this.bumpScale=1;this.normalMap=null;this.normalMapType=0;this.normalScale=new t(1,1);this.displacementMap=null;this.displacementScale=1;this.displacementBias=0;this.alphaMap=this.specularMap=null;this.wireframe=!1;this.wireframeLinewidth=1;this.wireframeLinejoin=this.wireframeLinecap=&quot;round&quot;;this.morphNormals=this.morphTargets=this.skinning=!1;this.setValues(a)}function nc(a){K.call(this);this.type=&quot;MeshNormalMaterial&quot;;this.bumpMap=null;this.bumpScale=1;this.normalMap=
null;this.normalMapType=0;this.normalScale=new t(1,1);this.displacementMap=null;this.displacementScale=1;this.displacementBias=0;this.wireframe=!1;this.wireframeLinewidth=1;this.morphNormals=this.morphTargets=this.skinning=this.fog=!1;this.setValues(a)}function oc(a){K.call(this);this.type=&quot;MeshLambertMaterial&quot;;this.color=new A(16777215);this.lightMap=this.map=null;this.lightMapIntensity=1;this.aoMap=null;this.aoMapIntensity=1;this.emissive=new A(0);this.emissiveIntensity=1;this.envMap=this.alphaMap=
this.specularMap=this.emissiveMap=null;this.combine=0;this.reflectivity=1;this.refractionRatio=.98;this.wireframe=!1;this.wireframeLinewidth=1;this.wireframeLinejoin=this.wireframeLinecap=&quot;round&quot;;this.morphNormals=this.morphTargets=this.skinning=!1;this.setValues(a)}function pc(a){K.call(this);this.defines={MATCAP:&quot;&quot;};this.type=&quot;MeshMatcapMaterial&quot;;this.color=new A(16777215);this.bumpMap=this.map=this.matcap=null;this.bumpScale=1;this.normalMap=null;this.normalMapType=0;this.normalScale=new t(1,1);
this.displacementMap=null;this.displacementScale=1;this.displacementBias=0;this.alphaMap=null;this.morphNormals=this.morphTargets=this.skinning=!1;this.setValues(a)}function qc(a){la.call(this);this.type=&quot;LineDashedMaterial&quot;;this.scale=1;this.dashSize=3;this.gapSize=1;this.setValues(a)}function La(a,b,c,d){this.parameterPositions=a;this._cachedIndex=0;this.resultBuffer=void 0!==d?d:new b.constructor(c);this.sampleValues=b;this.valueSize=c}function Xe(a,b,c,d){La.call(this,a,b,c,d);this._offsetNext=
this._weightNext=this._offsetPrev=this._weightPrev=-0}function ne(a,b,c,d){La.call(this,a,b,c,d)}function Ye(a,b,c,d){La.call(this,a,b,c,d)}function sa(a,b,c,d){if(void 0===a)throw Error(&quot;THREE.KeyframeTrack: track name is undefined&quot;);if(void 0===b||0===b.length)throw Error(&quot;THREE.KeyframeTrack: no keyframes in track named &quot;+a);this.name=a;this.times=R.convertArray(b,this.TimeBufferType);this.values=R.convertArray(c,this.ValueBufferType);this.setInterpolation(d||this.DefaultInterpolation)}function Ze(a,
b,c){sa.call(this,a,b,c)}function $e(a,b,c,d){sa.call(this,a,b,c,d)}function dd(a,b,c,d){sa.call(this,a,b,c,d)}function af(a,b,c,d){La.call(this,a,b,c,d)}function oe(a,b,c,d){sa.call(this,a,b,c,d)}function bf(a,b,c,d){sa.call(this,a,b,c,d)}function ed(a,b,c,d){sa.call(this,a,b,c,d)}function Qa(a,b,c){this.name=a;this.tracks=c;this.duration=void 0!==b?b:-1;this.uuid=L.generateUUID();0>this.duration&&this.resetDuration()}function Pk(a){switch(a.toLowerCase()){case &quot;scalar&quot;:case &quot;double&quot;:case &quot;float&quot;:case &quot;number&quot;:case &quot;integer&quot;:return dd;
case &quot;vector&quot;:case &quot;vector2&quot;:case &quot;vector3&quot;:case &quot;vector4&quot;:return ed;case &quot;color&quot;:return $e;case &quot;quaternion&quot;:return oe;case &quot;bool&quot;:case &quot;boolean&quot;:return Ze;case &quot;string&quot;:return bf}throw Error(&quot;THREE.KeyframeTrack: Unsupported typeName: &quot;+a);}function Qk(a){if(void 0===a.type)throw Error(&quot;THREE.KeyframeTrack: track type undefined, can not parse&quot;);var b=Pk(a.type);if(void 0===a.times){var c=[],d=[];R.flattenJSON(a.keys,c,d,&quot;value&quot;);a.times=c;a.values=d}return void 0!==b.parse?b.parse(a):new b(a.name,
a.times,a.values,a.interpolation)}function vg(a,b,c){var d=this,e=!1,f=0,g=0,h=void 0,l=[];this.onStart=void 0;this.onLoad=a;this.onProgress=b;this.onError=c;this.itemStart=function(a){g++;if(!1===e&&void 0!==d.onStart)d.onStart(a,f,g);e=!0};this.itemEnd=function(a){f++;if(void 0!==d.onProgress)d.onProgress(a,f,g);if(f===g&&(e=!1,void 0!==d.onLoad))d.onLoad()};this.itemError=function(a){if(void 0!==d.onError)d.onError(a)};this.resolveURL=function(a){return h?h(a):a};this.setURLModifier=function(a){h=
a;return this};this.addHandler=function(a,b){l.push(a,b);return this};this.removeHandler=function(a){a=l.indexOf(a);-1!==a&&l.splice(a,2);return this};this.getHandler=function(a){for(var b=0,c=l.length;b<c;b+=2){var d=l[b],e=l[b+1];d.global&&(d.lastIndex=0);if(d.test(a))return e}return null}}function W(a){this.manager=void 0!==a?a:ki;this.crossOrigin=&quot;anonymous&quot;;this.resourcePath=this.path=&quot;&quot;}function Ra(a){W.call(this,a)}function wg(a){W.call(this,a)}function xg(a){W.call(this,a)}function cf(a){W.call(this,
a)}function fd(a){W.call(this,a)}function df(a){W.call(this,a)}function ef(a){W.call(this,a)}function G(){this.type=&quot;Curve&quot;;this.arcLengthDivisions=200}function Ma(a,b,c,d,e,f,g,h){G.call(this);this.type=&quot;EllipseCurve&quot;;this.aX=a||0;this.aY=b||0;this.xRadius=c||1;this.yRadius=d||1;this.aStartAngle=e||0;this.aEndAngle=f||2*Math.PI;this.aClockwise=g||!1;this.aRotation=h||0}function gd(a,b,c,d,e,f){Ma.call(this,a,b,c,c,d,e,f);this.type=&quot;ArcCurve&quot;}function yg(){var a=0,b=0,c=0,d=0;return{initCatmullRom:function(e,
f,g,h,l){e=l*(g-e);h=l*(h-f);a=f;b=e;c=-3*f+3*g-2*e-h;d=2*f-2*g+e+h},initNonuniformCatmullRom:function(e,f,g,h,l,m,k){e=((f-e)/l-(g-e)/(l+m)+(g-f)/m)*m;h=((g-f)/m-(h-f)/(m+k)+(h-g)/k)*m;a=f;b=e;c=-3*f+3*g-2*e-h;d=2*f-2*g+e+h},calc:function(e){var f=e*e;return a+b*e+c*f+d*f*e}}}function pa(a,b,c,d){G.call(this);this.type=&quot;CatmullRomCurve3&quot;;this.points=a||[];this.closed=b||!1;this.curveType=c||&quot;centripetal&quot;;this.tension=d||.5}function li(a,b,c,d,e){b=.5*(d-b);e=.5*(e-c);var f=a*a;return(2*c-2*d+b+e)*
a*f+(-3*c+3*d-2*b-e)*f+b*a+c}function pe(a,b,c,d){var e=1-a;return e*e*b+2*(1-a)*a*c+a*a*d}function qe(a,b,c,d,e){var f=1-a,g=1-a;return f*f*f*b+3*g*g*a*c+3*(1-a)*a*a*d+a*a*a*e}function Wa(a,b,c,d){G.call(this);this.type=&quot;CubicBezierCurve&quot;;this.v0=a||new t;this.v1=b||new t;this.v2=c||new t;this.v3=d||new t}function hb(a,b,c,d){G.call(this);this.type=&quot;CubicBezierCurve3&quot;;this.v0=a||new n;this.v1=b||new n;this.v2=c||new n;this.v3=d||new n}function Da(a,b){G.call(this);this.type=&quot;LineCurve&quot;;this.v1=a||
new t;this.v2=b||new t}function Xa(a,b){G.call(this);this.type=&quot;LineCurve3&quot;;this.v1=a||new n;this.v2=b||new n}function Ya(a,b,c){G.call(this);this.type=&quot;QuadraticBezierCurve&quot;;this.v0=a||new t;this.v1=b||new t;this.v2=c||new t}function ib(a,b,c){G.call(this);this.type=&quot;QuadraticBezierCurve3&quot;;this.v0=a||new n;this.v1=b||new n;this.v2=c||new n}function Za(a){G.call(this);this.type=&quot;SplineCurve&quot;;this.points=a||[]}function vb(){G.call(this);this.type=&quot;CurvePath&quot;;this.curves=[];this.autoClose=!1}function $a(a){vb.call(this);
this.type=&quot;Path&quot;;this.currentPoint=new t;a&&this.setFromPoints(a)}function Kb(a){$a.call(this,a);this.uuid=L.generateUUID();this.type=&quot;Shape&quot;;this.holes=[]}function da(a,b){F.call(this);this.type=&quot;Light&quot;;this.color=new A(a);this.intensity=void 0!==b?b:1;this.receiveShadow=void 0}function ff(a,b,c){da.call(this,a,c);this.type=&quot;HemisphereLight&quot;;this.castShadow=void 0;this.position.copy(F.DefaultUp);this.updateMatrix();this.groundColor=new A(b)}function jb(a){this.camera=a;this.bias=0;this.radius=1;
this.mapSize=new t(512,512);this.mapPass=this.map=null;this.matrix=new P;this._frustum=new Hc;this._frameExtents=new t(1,1);this._viewportCount=1;this._viewports=[new ka(0,0,1,1)]}function gf(){jb.call(this,new aa(50,1,.5,500))}function hf(a,b,c,d,e,f){da.call(this,a,b);this.type=&quot;SpotLight&quot;;this.position.copy(F.DefaultUp);this.updateMatrix();this.target=new F;Object.defineProperty(this,&quot;power&quot;,{get:function(){return this.intensity*Math.PI},set:function(a){this.intensity=a/Math.PI}});this.distance=
void 0!==c?c:0;this.angle=void 0!==d?d:Math.PI/3;this.penumbra=void 0!==e?e:0;this.decay=void 0!==f?f:1;this.shadow=new gf}function zg(){jb.call(this,new aa(90,1,.5,500));this._frameExtents=new t(4,2);this._viewportCount=6;this._viewports=[new ka(2,1,1,1),new ka(0,1,1,1),new ka(3,1,1,1),new ka(1,1,1,1),new ka(3,0,1,1),new ka(1,0,1,1)];this._cubeDirections=[new n(1,0,0),new n(-1,0,0),new n(0,0,1),new n(0,0,-1),new n(0,1,0),new n(0,-1,0)];this._cubeUps=[new n(0,1,0),new n(0,1,0),new n(0,1,0),new n(0,
1,0),new n(0,0,1),new n(0,0,-1)]}function jf(a,b,c,d){da.call(this,a,b);this.type=&quot;PointLight&quot;;Object.defineProperty(this,&quot;power&quot;,{get:function(){return 4*this.intensity*Math.PI},set:function(a){this.intensity=a/(4*Math.PI)}});this.distance=void 0!==c?c:0;this.decay=void 0!==d?d:1;this.shadow=new zg}function hd(a,b,c,d,e,f){db.call(this);this.type=&quot;OrthographicCamera&quot;;this.zoom=1;this.view=null;this.left=void 0!==a?a:-1;this.right=void 0!==b?b:1;this.top=void 0!==c?c:1;this.bottom=void 0!==d?d:-1;
this.near=void 0!==e?e:.1;this.far=void 0!==f?f:2E3;this.updateProjectionMatrix()}function kf(){jb.call(this,new hd(-5,5,5,-5,.5,500))}function lf(a,b){da.call(this,a,b);this.type=&quot;DirectionalLight&quot;;this.position.copy(F.DefaultUp);this.updateMatrix();this.target=new F;this.shadow=new kf}function mf(a,b){da.call(this,a,b);this.type=&quot;AmbientLight&quot;;this.castShadow=void 0}function nf(a,b,c,d){da.call(this,a,b);this.type=&quot;RectAreaLight&quot;;this.width=void 0!==c?c:10;this.height=void 0!==d?d:10}function of(a){W.call(this,
a);this.textures={}}function pf(){C.call(this);this.type=&quot;InstancedBufferGeometry&quot;;this.maxInstancedCount=void 0}function qf(a,b,c,d){&quot;number&quot;===typeof c&&(d=c,c=!1,console.error(&quot;THREE.InstancedBufferAttribute: The constructor now expects normalized as the third argument.&quot;));M.call(this,a,b,c);this.meshPerAttribute=d||1}function rf(a){W.call(this,a)}function sf(a){W.call(this,a)}function Ag(a){&quot;undefined&quot;===typeof createImageBitmap&&console.warn(&quot;THREE.ImageBitmapLoader: createImageBitmap() not supported.&quot;);
&quot;undefined&quot;===typeof fetch&&console.warn(&quot;THREE.ImageBitmapLoader: fetch() not supported.&quot;);W.call(this,a);this.options=void 0}function Bg(){this.type=&quot;ShapePath&quot;;this.color=new A;this.subPaths=[];this.currentPath=null}function Cg(a){this.type=&quot;Font&quot;;this.data=a}function Dg(a){W.call(this,a)}function tf(a){W.call(this,a)}function uf(){this.coefficients=[];for(var a=0;9>a;a++)this.coefficients.push(new n)}function ab(a,b){da.call(this,void 0,b);this.sh=void 0!==a?a:new uf}function Eg(a,b,c){ab.call(this,
void 0,c);a=(new A).set(a);c=(new A).set(b);b=new n(a.r,a.g,a.b);a=new n(c.r,c.g,c.b);c=Math.sqrt(Math.PI);var d=c*Math.sqrt(.75);this.sh.coefficients[0].copy(b).add(a).multiplyScalar(c);this.sh.coefficients[1].copy(b).sub(a).multiplyScalar(d)}function Fg(a,b){ab.call(this,void 0,b);a=(new A).set(a);this.sh.coefficients[0].set(a.r,a.g,a.b).multiplyScalar(2*Math.sqrt(Math.PI))}function mi(){this.type=&quot;StereoCamera&quot;;this.aspect=1;this.eyeSep=.064;this.cameraL=new aa;this.cameraL.layers.enable(1);this.cameraL.matrixAutoUpdate=
!1;this.cameraR=new aa;this.cameraR.layers.enable(2);this.cameraR.matrixAutoUpdate=!1;this._cache={focus:null,fov:null,aspect:null,near:null,far:null,zoom:null,eyeSep:null}}function Gg(a){this.autoStart=void 0!==a?a:!0;this.elapsedTime=this.oldTime=this.startTime=0;this.running=!1}function Hg(){F.call(this);this.type=&quot;AudioListener&quot;;this.context=Ig.getContext();this.gain=this.context.createGain();this.gain.connect(this.context.destination);this.filter=null;this.timeDelta=0;this._clock=new Gg}function id(a){F.call(this);
this.type=&quot;Audio&quot;;this.listener=a;this.context=a.context;this.gain=this.context.createGain();this.gain.connect(a.getInput());this.autoplay=!1;this.buffer=null;this.detune=0;this.loop=!1;this.offset=this.loopEnd=this.loopStart=0;this.duration=void 0;this.playbackRate=1;this.isPlaying=!1;this.hasPlaybackControl=!0;this.sourceType=&quot;empty&quot;;this._pausedAt=this._startedAt=0;this.filters=[]}function Jg(a){id.call(this,a);this.panner=this.context.createPanner();this.panner.panningModel=&quot;HRTF&quot;;this.panner.connect(this.gain)}
function Kg(a,b){this.analyser=a.context.createAnalyser();this.analyser.fftSize=void 0!==b?b:2048;this.data=new Uint8Array(this.analyser.frequencyBinCount);a.getOutput().connect(this.analyser)}function Lg(a,b,c){this.binding=a;this.valueSize=c;a=Float64Array;switch(b){case &quot;quaternion&quot;:b=this._slerp;break;case &quot;string&quot;:case &quot;bool&quot;:a=Array;b=this._select;break;default:b=this._lerp}this.buffer=new a(4*c);this._mixBufferRegion=b;this.referenceCount=this.useCount=this.cumulativeWeight=0}function ni(a,
b,c){c=c||ya.parseTrackName(b);this._targetGroup=a;this._bindings=a.subscribe_(b,c)}function ya(a,b,c){this.path=b;this.parsedPath=c||ya.parseTrackName(b);this.node=ya.findNode(a,this.parsedPath.nodeName)||a;this.rootNode=a}function oi(){this.uuid=L.generateUUID();this._objects=Array.prototype.slice.call(arguments);this.nCachedObjects_=0;var a={};this._indicesByUUID=a;for(var b=0,c=arguments.length;b!==c;++b)a[arguments[b].uuid]=b;this._paths=[];this._parsedPaths=[];this._bindings=[];this._bindingsIndicesByPath=
{};var d=this;this.stats={objects:{get total(){return d._objects.length},get inUse(){return this.total-d.nCachedObjects_}},get bindingsPerObject(){return d._bindings.length}}}function pi(a,b,c){this._mixer=a;this._clip=b;this._localRoot=c||null;a=b.tracks;b=a.length;c=Array(b);for(var d={endingStart:2400,endingEnd:2400},e=0;e!==b;++e){var f=a[e].createInterpolant(null);c[e]=f;f.settings=d}this._interpolantSettings=d;this._interpolants=c;this._propertyBindings=Array(b);this._weightInterpolant=this._timeScaleInterpolant=
this._byClipCacheIndex=this._cacheIndex=null;this.loop=2201;this._loopCount=-1;this._startTime=null;this.time=0;this._effectiveWeight=this.weight=this._effectiveTimeScale=this.timeScale=1;this.repetitions=Infinity;this.paused=!1;this.enabled=!0;this.clampWhenFinished=!1;this.zeroSlopeAtEnd=this.zeroSlopeAtStart=!0}function Mg(a){this._root=a;this._initMemoryManager();this.time=this._accuIndex=0;this.timeScale=1}function vf(a,b){&quot;string&quot;===typeof a&&(console.warn(&quot;THREE.Uniform: Type parameter is no longer needed.&quot;),
a=b);this.value=a}function Ng(a,b,c){rb.call(this,a,b);this.meshPerAttribute=c||1}function Og(a,b,c,d){this.ray=new Vb(a,b);this.near=c||0;this.far=d||Infinity;this.camera=null;this.layers=new He;this.params={Mesh:{},Line:{threshold:1},LOD:{},Points:{threshold:1},Sprite:{}};Object.defineProperties(this.params,{PointCloud:{get:function(){console.warn(&quot;THREE.Raycaster: params.PointCloud has been renamed to params.Points.&quot;);return this.Points}}})}function qi(a,b){return a.distance-b.distance}function Pg(a,
b,c,d){a.layers.test(b.layers)&&a.raycast(b,c);if(!0===d){a=a.children;d=0;for(var e=a.length;d<e;d++)Pg(a[d],b,c,!0)}}function ri(a,b,c){this.radius=void 0!==a?a:1;this.phi=void 0!==b?b:0;this.theta=void 0!==c?c:0;return this}function si(a,b,c){this.radius=void 0!==a?a:1;this.theta=void 0!==b?b:0;this.y=void 0!==c?c:0;return this}function Qg(a,b){this.min=void 0!==a?a:new t(Infinity,Infinity);this.max=void 0!==b?b:new t(-Infinity,-Infinity)}function Rg(a,b){this.start=void 0!==a?a:new n;this.end=
void 0!==b?b:new n}function re(a){F.call(this);this.material=a;this.render=function(){}}function jd(a,b){F.call(this);this.light=a;this.light.updateMatrixWorld();this.matrix=a.matrixWorld;this.matrixAutoUpdate=!1;this.color=b;a=new C;b=[0,0,0,0,0,1,0,0,0,1,0,1,0,0,0,-1,0,1,0,0,0,0,1,1,0,0,0,0,-1,1];for(var c=0,d=1;32>c;c++,d++){var e=c/32*Math.PI*2,f=d/32*Math.PI*2;b.push(Math.cos(e),Math.sin(e),1,Math.cos(f),Math.sin(f),1)}a.setAttribute(&quot;position&quot;,new y(b,3));b=new la({fog:!1,toneMapped:!1});this.cone=
new ma(a,b);this.add(this.cone);this.update()}function ti(a){var b=[];a&&a.isBone&&b.push(a);for(var c=0;c<a.children.length;c++)b.push.apply(b,ti(a.children[c]));return b}function rc(a){for(var b=ti(a),c=new C,d=[],e=[],f=new A(0,0,1),g=new A(0,1,0),h=0;h<b.length;h++){var l=b[h];l.parent&&l.parent.isBone&&(d.push(0,0,0),d.push(0,0,0),e.push(f.r,f.g,f.b),e.push(g.r,g.g,g.b))}c.setAttribute(&quot;position&quot;,new y(d,3));c.setAttribute(&quot;color&quot;,new y(e,3));d=new la({vertexColors:!0,depthTest:!1,depthWrite:!1,
toneMapped:!1,transparent:!0});ma.call(this,c,d);this.root=a;this.bones=b;this.matrix=a.matrixWorld;this.matrixAutoUpdate=!1}function kd(a,b,c){this.light=a;this.light.updateMatrixWorld();this.color=c;a=new gc(b,4,2);b=new Oa({wireframe:!0,fog:!1,toneMapped:!1});S.call(this,a,b);this.matrix=this.light.matrixWorld;this.matrixAutoUpdate=!1;this.update()}function ld(a,b,c){F.call(this);this.light=a;this.light.updateMatrixWorld();this.matrix=a.matrixWorld;this.matrixAutoUpdate=!1;this.color=c;a=new cc(b);
a.rotateY(.5*Math.PI);this.material=new Oa({wireframe:!0,fog:!1,toneMapped:!1});void 0===this.color&&(this.material.vertexColors=!0);b=a.getAttribute(&quot;position&quot;);b=new Float32Array(3*b.count);a.setAttribute(&quot;color&quot;,new M(b,3));this.add(new S(a,this.material));this.update()}function wf(a,b,c,d){a=a||10;b=b||10;c=new A(void 0!==c?c:4473924);d=new A(void 0!==d?d:8947848);var e=b/2,f=a/b,g=a/2;a=[];for(var h=[],l=0,m=0,k=-g;l<=b;l++,k+=f){a.push(-g,0,k,g,0,k);a.push(k,0,-g,k,0,g);var p=l===e?c:d;p.toArray(h,
m);m+=3;p.toArray(h,m);m+=3;p.toArray(h,m);m+=3;p.toArray(h,m);m+=3}b=new C;b.setAttribute(&quot;position&quot;,new y(a,3));b.setAttribute(&quot;color&quot;,new y(h,3));c=new la({vertexColors:!0,toneMapped:!1});ma.call(this,b,c)}function xf(a,b,c,d,e,f){a=a||10;b=b||16;c=c||8;d=d||64;e=new A(void 0!==e?e:4473924);f=new A(void 0!==f?f:8947848);var g=[],h=[],l;for(l=0;l<=b;l++){var m=l/b*2*Math.PI;var k=Math.sin(m)*a;m=Math.cos(m)*a;g.push(0,0,0);g.push(k,0,m);var p=l&1?e:f;h.push(p.r,p.g,p.b);h.push(p.r,p.g,p.b)}for(l=
0;l<=c;l++){p=l&1?e:f;var n=a-a/c*l;for(b=0;b<d;b++)m=b/d*2*Math.PI,k=Math.sin(m)*n,m=Math.cos(m)*n,g.push(k,0,m),h.push(p.r,p.g,p.b),m=(b+1)/d*2*Math.PI,k=Math.sin(m)*n,m=Math.cos(m)*n,g.push(k,0,m),h.push(p.r,p.g,p.b)}a=new C;a.setAttribute(&quot;position&quot;,new y(g,3));a.setAttribute(&quot;color&quot;,new y(h,3));g=new la({vertexColors:!0,toneMapped:!1});ma.call(this,a,g)}function md(a,b,c){F.call(this);this.light=a;this.light.updateMatrixWorld();this.matrix=a.matrixWorld;this.matrixAutoUpdate=!1;this.color=c;
void 0===b&&(b=1);a=new C;a.setAttribute(&quot;position&quot;,new y([-b,b,0,b,b,0,b,-b,0,-b,-b,0,-b,b,0],3));b=new la({fog:!1,toneMapped:!1});this.lightPlane=new Ka(a,b);this.add(this.lightPlane);a=new C;a.setAttribute(&quot;position&quot;,new y([0,0,0,0,0,1],3));this.targetLine=new Ka(a,b);this.add(this.targetLine);this.update()}function se(a){function b(a,b,d){c(a,d);c(b,d)}function c(a,b){f.push(0,0,0);g.push(b.r,b.g,b.b);void 0===h[a]&&(h[a]=[]);h[a].push(f.length/3-1)}var d=new C,e=new la({color:16777215,vertexColors:!0,
toneMapped:!1}),f=[],g=[],h={},l=new A(16755200),m=new A(16711680),k=new A(43775),p=new A(16777215),n=new A(3355443);b(&quot;n1&quot;,&quot;n2&quot;,l);b(&quot;n2&quot;,&quot;n4&quot;,l);b(&quot;n4&quot;,&quot;n3&quot;,l);b(&quot;n3&quot;,&quot;n1&quot;,l);b(&quot;f1&quot;,&quot;f2&quot;,l);b(&quot;f2&quot;,&quot;f4&quot;,l);b(&quot;f4&quot;,&quot;f3&quot;,l);b(&quot;f3&quot;,&quot;f1&quot;,l);b(&quot;n1&quot;,&quot;f1&quot;,l);b(&quot;n2&quot;,&quot;f2&quot;,l);b(&quot;n3&quot;,&quot;f3&quot;,l);b(&quot;n4&quot;,&quot;f4&quot;,l);b(&quot;p&quot;,&quot;n1&quot;,m);b(&quot;p&quot;,&quot;n2&quot;,m);b(&quot;p&quot;,&quot;n3&quot;,m);b(&quot;p&quot;,&quot;n4&quot;,m);b(&quot;u1&quot;,&quot;u2&quot;,k);b(&quot;u2&quot;,&quot;u3&quot;,k);b(&quot;u3&quot;,&quot;u1&quot;,k);b(&quot;c&quot;,&quot;t&quot;,p);b(&quot;p&quot;,&quot;c&quot;,n);b(&quot;cn1&quot;,&quot;cn2&quot;,n);b(&quot;cn3&quot;,&quot;cn4&quot;,n);b(&quot;cf1&quot;,&quot;cf2&quot;,n);b(&quot;cf3&quot;,&quot;cf4&quot;,n);d.setAttribute(&quot;position&quot;,
new y(f,3));d.setAttribute(&quot;color&quot;,new y(g,3));ma.call(this,d,e);this.camera=a;this.camera.updateProjectionMatrix&&this.camera.updateProjectionMatrix();this.matrix=a.matrixWorld;this.matrixAutoUpdate=!1;this.pointMap=h;this.update()}function na(a,b,c,d,e,f,g){yf.set(e,f,g).unproject(d);a=b[a];if(void 0!==a)for(c=c.getAttribute(&quot;position&quot;),b=0,d=a.length;b<d;b++)c.setXYZ(a[b],yf.x,yf.y,yf.z)}function wb(a,b){this.object=a;void 0===b&&(b=16776960);a=new Uint16Array([0,1,1,2,2,3,3,0,4,5,5,6,6,7,7,4,
0,4,1,5,2,6,3,7]);var c=new Float32Array(24),d=new C;d.setIndex(new M(a,1));d.setAttribute(&quot;position&quot;,new M(c,3));ma.call(this,d,new la({color:b,toneMapped:!1}));this.matrixAutoUpdate=!1;this.update()}function te(a,b){this.type=&quot;Box3Helper&quot;;this.box=a;b=b||16776960;a=new Uint16Array([0,1,1,2,2,3,3,0,4,5,5,6,6,7,7,4,0,4,1,5,2,6,3,7]);var c=new C;c.setIndex(new M(a,1));c.setAttribute(&quot;position&quot;,new y([1,1,1,-1,1,1,-1,-1,1,1,-1,1,1,1,-1,-1,1,-1,-1,-1,-1,1,-1,-1],3));ma.call(this,c,new la({color:b,toneMapped:!1}));
this.geometry.computeBoundingSphere()}function ue(a,b,c){this.type=&quot;PlaneHelper&quot;;this.plane=a;this.size=void 0===b?1:b;a=void 0!==c?c:16776960;b=new C;b.setAttribute(&quot;position&quot;,new y([1,-1,1,-1,1,1,-1,-1,1,1,1,1,-1,1,1,-1,-1,1,1,-1,1,1,1,1,0,0,1,0,0,0],3));b.computeBoundingSphere();Ka.call(this,b,new la({color:a,toneMapped:!1}));b=new C;b.setAttribute(&quot;position&quot;,new y([1,1,1,-1,1,1,-1,-1,1,1,1,1,-1,-1,1,1,-1,1],3));b.computeBoundingSphere();this.add(new S(b,new Oa({color:a,opacity:.2,transparent:!0,
depthWrite:!1,toneMapped:!1})))}function xb(a,b,c,d,e,f){F.call(this);void 0===a&&(a=new n(0,0,1));void 0===b&&(b=new n(0,0,0));void 0===c&&(c=1);void 0===d&&(d=16776960);void 0===e&&(e=.2*c);void 0===f&&(f=.2*e);void 0===zf&&(zf=new C,zf.setAttribute(&quot;position&quot;,new y([0,0,0,0,1,0],3)),Sg=new tb(0,.5,1,5,1),Sg.translate(0,-.5,0));this.position.copy(b);this.line=new Ka(zf,new la({color:d,toneMapped:!1}));this.line.matrixAutoUpdate=!1;this.add(this.line);this.cone=new S(Sg,new Oa({color:d,toneMapped:!1}));
this.cone.matrixAutoUpdate=!1;this.add(this.cone);this.setDirection(a);this.setLength(c,e,f)}function ve(a){a=a||1;var b=[0,0,0,a,0,0,0,0,0,0,a,0,0,0,0,0,0,a];a=new C;a.setAttribute(&quot;position&quot;,new y(b,3));a.setAttribute(&quot;color&quot;,new y([1,0,0,1,.6,0,0,1,0,.6,1,0,0,0,1,0,.6,1],3));b=new la({vertexColors:!0,toneMapped:!1});ma.call(this,a,b)}function Tg(a){T=a;Ug(Af)}function ui(a){var b={magFilter:1003,minFilter:1003,generateMipmaps:!1,type:a?a.type:1009,format:a?a.format:1023,encoding:a?a.encoding:3002,
depthBuffer:!1,stencilBuffer:!1},c=vi(b);c.depthBuffer=a?!1:!0;Bf=vi(b);return c}function wi(a){Bf.dispose();T.setRenderTarget(Vg);a.scissorTest=!1;a.setSize(a.width,a.height)}function Ug(a){var b=new ob;b.add(new S(we[0],a));T.compile(b,Wg)}function vi(a){a=new Ha(3*kb,3*kb,a);a.texture.mapping=306;a.texture.name=&quot;PMREM.cubeUv&quot;;a.scissorTest=!0;return a}function Xg(a,b,c,d,e){a.viewport.set(b,c,d,e);a.scissor.set(b,c,d,e)}function xi(a){var b=T.autoClear;T.autoClear=!1;for(var c=1;c<yi;c++)zi(a,
c-1,c,Math.sqrt(Cf[c]*Cf[c]-Cf[c-1]*Cf[c-1]),Ai[(c-1)%Ai.length]);T.autoClear=b}function zi(a,b,c,d,e){Bi(a,Bf,b,c,d,&quot;latitudinal&quot;,e);Bi(Bf,a,c,c,d,&quot;longitudinal&quot;,e)}function Bi(a,b,c,d,e,f,g){&quot;latitudinal&quot;!==f&&&quot;longitudinal&quot;!==f&&console.error(&quot;blur direction must be either latitudinal or longitudinal!&quot;);var h=new ob;h.add(new S(we[d],Af));var l=Af.uniforms,m=Ci[c]-1;m=isFinite(e)?Math.PI/(2*m):2*Math.PI/39;var k=e/m,p=isFinite(e)?1+Math.floor(3*k):20;20<p&&console.warn(&quot;sigmaRadians, &quot;+e+&quot;, is too large and will clip, as it requested &quot;+
p+&quot; samples when the maximum is set to 20&quot;);e=[];for(var n=0,r=0;20>r;++r){var q=r/k;q=Math.exp(-q*q/2);e.push(q);0==r?n+=q:r<p&&(n+=2*q)}for(r=0;r<e.length;r++)e[r]/=n;l.envMap.value=a.texture;l.samples.value=p;l.weights.value=e;l.latitudinal.value=&quot;latitudinal&quot;===f;g&&(l.poleAxis.value=g);l.dTheta.value=m;l.mipInt.value=8-c;l.inputEncoding.value=lb[a.texture.encoding];l.outputEncoding.value=lb[a.texture.encoding];a=Ci[d];q=3*Math.max(0,kb-2*a);Xg(b,q,(0===d?0:2*kb)+2*a*(4<d?d-8+4:0),3*a,2*a);T.setRenderTarget(b);
T.render(h,Wg)}function Di(){var a=new t(1,1);a=new ub({uniforms:{envMap:{value:null},texelSize:{value:a},inputEncoding:{value:lb[3E3]},outputEncoding:{value:lb[3E3]}},vertexShader:Yg(),fragmentShader:&quot;\nprecision mediump float;\nprecision mediump int;\nvarying vec3 vOutputDirection;\nuniform sampler2D envMap;\nuniform vec2 texelSize;\n\n&quot;+Zg()+&quot;\n\n#define RECIPROCAL_PI 0.31830988618\n#define RECIPROCAL_PI2 0.15915494\n\nvoid main() {\n\tgl_FragColor = vec4(0.0);\n\tvec3 outputDirection = normalize(vOutputDirection);\n\tvec2 uv;\n\tuv.y = asin(clamp(outputDirection.y, -1.0, 1.0)) * RECIPROCAL_PI + 0.5;\n\tuv.x = atan(outputDirection.z, outputDirection.x) * RECIPROCAL_PI2 + 0.5;\n\tvec2 f = fract(uv / texelSize - 0.5);\n\tuv -= f * texelSize;\n\tvec3 tl = envMapTexelToLinear(texture2D(envMap, uv)).rgb;\n\tuv.x += texelSize.x;\n\tvec3 tr = envMapTexelToLinear(texture2D(envMap, uv)).rgb;\n\tuv.y += texelSize.y;\n\tvec3 br = envMapTexelToLinear(texture2D(envMap, uv)).rgb;\n\tuv.x -= texelSize.x;\n\tvec3 bl = envMapTexelToLinear(texture2D(envMap, uv)).rgb;\n\tvec3 tm = mix(tl, tr, f.x);\n\tvec3 bm = mix(bl, br, f.x);\n\tgl_FragColor.rgb = mix(tm, bm, f.y);\n\tgl_FragColor = linearToOutputTexel(gl_FragColor);\n}\n\t\t&quot;,
blending:0,depthTest:!1,depthWrite:!1});a.type=&quot;EquirectangularToCubeUV&quot;;return a}function Ei(){var a=new ub({uniforms:{envMap:{value:null},inputEncoding:{value:lb[3E3]},outputEncoding:{value:lb[3E3]}},vertexShader:Yg(),fragmentShader:&quot;\nprecision mediump float;\nprecision mediump int;\nvarying vec3 vOutputDirection;\nuniform samplerCube envMap;\n\n&quot;+Zg()+&quot;\n\nvoid main() {\n\tgl_FragColor = vec4(0.0);\n\tgl_FragColor.rgb = envMapTexelToLinear(textureCube(envMap, vec3( - vOutputDirection.x, vOutputDirection.yz ))).rgb;\n\tgl_FragColor = linearToOutputTexel(gl_FragColor);\n}\n\t\t&quot;,
blending:0,depthTest:!1,depthWrite:!1});a.type=&quot;CubemapToCubeUV&quot;;return a}function Yg(){return&quot;\nprecision mediump float;\nprecision mediump int;\nattribute vec3 position;\nattribute vec2 uv;\nattribute float faceIndex;\nvarying vec3 vOutputDirection;\nvec3 getDirection(vec2 uv, float face) {\n\tuv = 2.0 * uv - 1.0;\n\tvec3 direction = vec3(uv, 1.0);\n\tif (face == 0.0) {\n\t\tdirection = direction.zyx;\n\t\tdirection.z *= -1.0;\n\t} else if (face == 1.0) {\n\t\tdirection = direction.xzy;\n\t\tdirection.z *= -1.0;\n\t} else if (face == 3.0) {\n\t\tdirection = direction.zyx;\n\t\tdirection.x *= -1.0;\n\t} else if (face == 4.0) {\n\t\tdirection = direction.xzy;\n\t\tdirection.y *= -1.0;\n\t} else if (face == 5.0) {\n\t\tdirection.xz *= -1.0;\n\t}\n\treturn direction;\n}\nvoid main() {\n\tvOutputDirection = getDirection(uv, faceIndex);\n\tgl_Position = vec4( position, 1.0 );\n}\n\t&quot;}
function Zg(){return&quot;\nuniform int inputEncoding;\nuniform int outputEncoding;\n\n#include <encodings_pars_fragment>\n\nvec4 inputTexelToLinear(vec4 value){\n\tif(inputEncoding == 0){\n\t\treturn value;\n\t}else if(inputEncoding == 1){\n\t\treturn sRGBToLinear(value);\n\t}else if(inputEncoding == 2){\n\t\treturn RGBEToLinear(value);\n\t}else if(inputEncoding == 3){\n\t\treturn RGBMToLinear(value, 7.0);\n\t}else if(inputEncoding == 4){\n\t\treturn RGBMToLinear(value, 16.0);\n\t}else if(inputEncoding == 5){\n\t\treturn RGBDToLinear(value, 256.0);\n\t}else{\n\t\treturn GammaToLinear(value, 2.2);\n\t}\n}\n\nvec4 linearToOutputTexel(vec4 value){\n\tif(outputEncoding == 0){\n\t\treturn value;\n\t}else if(outputEncoding == 1){\n\t\treturn LinearTosRGB(value);\n\t}else if(outputEncoding == 2){\n\t\treturn LinearToRGBE(value);\n\t}else if(outputEncoding == 3){\n\t\treturn LinearToRGBM(value, 7.0);\n\t}else if(outputEncoding == 4){\n\t\treturn LinearToRGBM(value, 16.0);\n\t}else if(outputEncoding == 5){\n\t\treturn LinearToRGBD(value, 256.0);\n\t}else{\n\t\treturn LinearToGamma(value, 2.2);\n\t}\n}\n\nvec4 envMapTexelToLinear(vec4 color) {\n\treturn inputTexelToLinear(color);\n}\n\t&quot;}
function Fi(a){console.warn(&quot;THREE.ClosedSplineCurve3 has been deprecated. Use THREE.CatmullRomCurve3 instead.&quot;);pa.call(this,a);this.type=&quot;catmullrom&quot;;this.closed=!0}function Gi(a){console.warn(&quot;THREE.SplineCurve3 has been deprecated. Use THREE.CatmullRomCurve3 instead.&quot;);pa.call(this,a);this.type=&quot;catmullrom&quot;}function $g(a){console.warn(&quot;THREE.Spline has been removed. Use THREE.CatmullRomCurve3 instead.&quot;);pa.call(this,a);this.type=&quot;catmullrom&quot;}void 0===Number.EPSILON&&(Number.EPSILON=Math.pow(2,
-52));void 0===Number.isInteger&&(Number.isInteger=function(a){return&quot;number&quot;===typeof a&&isFinite(a)&&Math.floor(a)===a});void 0===Math.sign&&(Math.sign=function(a){return 0>a?-1:0<a?1:+a});!1===&quot;name&quot;in Function.prototype&&Object.defineProperty(Function.prototype,&quot;name&quot;,{get:function(){return this.toString().match(/^\s*function\s*([^\(\s]*)/)[1]}});void 0===Object.assign&&(Object.assign=function(a){if(void 0===a||null===a)throw new TypeError(&quot;Cannot convert undefined or null to object&quot;);for(var b=
Object(a),c=1;c<arguments.length;c++){var d=arguments[c];if(void 0!==d&&null!==d)for(var e in d)Object.prototype.hasOwnProperty.call(d,e)&&(b[e]=d[e])}return b});Object.assign(Ea.prototype,{addEventListener:function(a,b){void 0===this._listeners&&(this._listeners={});var c=this._listeners;void 0===c[a]&&(c[a]=[]);-1===c[a].indexOf(b)&&c[a].push(b)},hasEventListener:function(a,b){if(void 0===this._listeners)return!1;var c=this._listeners;return void 0!==c[a]&&-1!==c[a].indexOf(b)},removeEventListener:function(a,
b){void 0!==this._listeners&&(a=this._listeners[a],void 0!==a&&(b=a.indexOf(b),-1!==b&&a.splice(b,1)))},dispatchEvent:function(a){if(void 0!==this._listeners){var b=this._listeners[a.type];if(void 0!==b){a.target=this;b=b.slice(0);for(var c=0,d=b.length;c<d;c++)b[c].call(this,a)}}}});for(var ta=[],xe=0;256>xe;xe++)ta[xe]=(16>xe?&quot;0&quot;:&quot;&quot;)+xe.toString(16);var L={DEG2RAD:Math.PI/180,RAD2DEG:180/Math.PI,generateUUID:function(){var a=4294967295*Math.random()|0,b=4294967295*Math.random()|0,c=4294967295*Math.random()|
0,d=4294967295*Math.random()|0;return(ta[a&255]+ta[a>>8&255]+ta[a>>16&255]+ta[a>>24&255]+&quot;-&quot;+ta[b&255]+ta[b>>8&255]+&quot;-&quot;+ta[b>>16&15|64]+ta[b>>24&255]+&quot;-&quot;+ta[c&63|128]+ta[c>>8&255]+&quot;-&quot;+ta[c>>16&255]+ta[c>>24&255]+ta[d&255]+ta[d>>8&255]+ta[d>>16&255]+ta[d>>24&255]).toUpperCase()},clamp:function(a,b,c){return Math.max(b,Math.min(c,a))},euclideanModulo:function(a,b){return(a%b+b)%b},mapLinear:function(a,b,c,d,e){return d+(a-b)*(e-d)/(c-b)},lerp:function(a,b,c){return(1-c)*a+c*b},smoothstep:function(a,
b,c){if(a<=b)return 0;if(a>=c)return 1;a=(a-b)/(c-b);return a*a*(3-2*a)},smootherstep:function(a,b,c){if(a<=b)return 0;if(a>=c)return 1;a=(a-b)/(c-b);return a*a*a*(a*(6*a-15)+10)},randInt:function(a,b){return a+Math.floor(Math.random()*(b-a+1))},randFloat:function(a,b){return a+Math.random()*(b-a)},randFloatSpread:function(a){return a*(.5-Math.random())},degToRad:function(a){return a*L.DEG2RAD},radToDeg:function(a){return a*L.RAD2DEG},isPowerOfTwo:function(a){return 0===(a&a-1)&&0!==a},ceilPowerOfTwo:function(a){return Math.pow(2,
Math.ceil(Math.log(a)/Math.LN2))},floorPowerOfTwo:function(a){return Math.pow(2,Math.floor(Math.log(a)/Math.LN2))},setQuaternionFromProperEuler:function(a,b,c,d,e){var f=Math.cos,g=Math.sin,h=f(c/2);c=g(c/2);var l=f((b+d)/2),m=g((b+d)/2),k=f((b-d)/2),p=g((b-d)/2);f=f((d-b)/2);b=g((d-b)/2);&quot;XYX&quot;===e?a.set(h*m,c*k,c*p,h*l):&quot;YZY&quot;===e?a.set(c*p,h*m,c*k,h*l):&quot;ZXZ&quot;===e?a.set(c*k,c*p,h*m,h*l):&quot;XZX&quot;===e?a.set(h*m,c*b,c*f,h*l):&quot;YXY&quot;===e?a.set(c*f,h*m,c*b,h*l):&quot;ZYZ&quot;===e?a.set(c*b,c*f,h*m,h*l):console.warn(&quot;THREE.MathUtils: .setQuaternionFromProperEuler() encountered an unknown order.&quot;)}};
Object.defineProperties(t.prototype,{width:{get:function(){return this.x},set:function(a){this.x=a}},height:{get:function(){return this.y},set:function(a){this.y=a}}});Object.assign(t.prototype,{isVector2:!0,set:function(a,b){this.x=a;this.y=b;return this},setScalar:function(a){this.y=this.x=a;return this},setX:function(a){this.x=a;return this},setY:function(a){this.y=a;return this},setComponent:function(a,b){switch(a){case 0:this.x=b;break;case 1:this.y=b;break;default:throw Error(&quot;index is out of range: &quot;+
a);}return this},getComponent:function(a){switch(a){case 0:return this.x;case 1:return this.y;default:throw Error(&quot;index is out of range: &quot;+a);}},clone:function(){return new this.constructor(this.x,this.y)},copy:function(a){this.x=a.x;this.y=a.y;return this},add:function(a,b){if(void 0!==b)return console.warn(&quot;THREE.Vector2: .add() now only accepts one argument. Use .addVectors( a, b ) instead.&quot;),this.addVectors(a,b);this.x+=a.x;this.y+=a.y;return this},addScalar:function(a){this.x+=a;this.y+=a;return this},
addVectors:function(a,b){this.x=a.x+b.x;this.y=a.y+b.y;return this},addScaledVector:function(a,b){this.x+=a.x*b;this.y+=a.y*b;return this},sub:function(a,b){if(void 0!==b)return console.warn(&quot;THREE.Vector2: .sub() now only accepts one argument. Use .subVectors( a, b ) instead.&quot;),this.subVectors(a,b);this.x-=a.x;this.y-=a.y;return this},subScalar:function(a){this.x-=a;this.y-=a;return this},subVectors:function(a,b){this.x=a.x-b.x;this.y=a.y-b.y;return this},multiply:function(a){this.x*=a.x;this.y*=
a.y;return this},multiplyScalar:function(a){this.x*=a;this.y*=a;return this},divide:function(a){this.x/=a.x;this.y/=a.y;return this},divideScalar:function(a){return this.multiplyScalar(1/a)},applyMatrix3:function(a){var b=this.x,c=this.y;a=a.elements;this.x=a[0]*b+a[3]*c+a[6];this.y=a[1]*b+a[4]*c+a[7];return this},min:function(a){this.x=Math.min(this.x,a.x);this.y=Math.min(this.y,a.y);return this},max:function(a){this.x=Math.max(this.x,a.x);this.y=Math.max(this.y,a.y);return this},clamp:function(a,
b){this.x=Math.max(a.x,Math.min(b.x,this.x));this.y=Math.max(a.y,Math.min(b.y,this.y));return this},clampScalar:function(a,b){this.x=Math.max(a,Math.min(b,this.x));this.y=Math.max(a,Math.min(b,this.y));return this},clampLength:function(a,b){var c=this.length();return this.divideScalar(c||1).multiplyScalar(Math.max(a,Math.min(b,c)))},floor:function(){this.x=Math.floor(this.x);this.y=Math.floor(this.y);return this},ceil:function(){this.x=Math.ceil(this.x);this.y=Math.ceil(this.y);return this},round:function(){this.x=
Math.round(this.x);this.y=Math.round(this.y);return this},roundToZero:function(){this.x=0>this.x?Math.ceil(this.x):Math.floor(this.x);this.y=0>this.y?Math.ceil(this.y):Math.floor(this.y);return this},negate:function(){this.x=-this.x;this.y=-this.y;return this},dot:function(a){return this.x*a.x+this.y*a.y},cross:function(a){return this.x*a.y-this.y*a.x},lengthSq:function(){return this.x*this.x+this.y*this.y},length:function(){return Math.sqrt(this.x*this.x+this.y*this.y)},manhattanLength:function(){return Math.abs(this.x)+
Math.abs(this.y)},normalize:function(){return this.divideScalar(this.length()||1)},angle:function(){return Math.atan2(-this.y,-this.x)+Math.PI},distanceTo:function(a){return Math.sqrt(this.distanceToSquared(a))},distanceToSquared:function(a){var b=this.x-a.x;a=this.y-a.y;return b*b+a*a},manhattanDistanceTo:function(a){return Math.abs(this.x-a.x)+Math.abs(this.y-a.y)},setLength:function(a){return this.normalize().multiplyScalar(a)},lerp:function(a,b){this.x+=(a.x-this.x)*b;this.y+=(a.y-this.y)*b;return this},
lerpVectors:function(a,b,c){return this.subVectors(b,a).multiplyScalar(c).add(a)},equals:function(a){return a.x===this.x&&a.y===this.y},fromArray:function(a,b){void 0===b&&(b=0);this.x=a[b];this.y=a[b+1];return this},toArray:function(a,b){void 0===a&&(a=[]);void 0===b&&(b=0);a[b]=this.x;a[b+1]=this.y;return a},fromBufferAttribute:function(a,b,c){void 0!==c&&console.warn(&quot;THREE.Vector2: offset has been removed from .fromBufferAttribute().&quot;);this.x=a.getX(b);this.y=a.getY(b);return this},rotateAround:function(a,
b){var c=Math.cos(b);b=Math.sin(b);var d=this.x-a.x,e=this.y-a.y;this.x=d*c-e*b+a.x;this.y=d*b+e*c+a.y;return this}});Object.assign(wa.prototype,{isMatrix3:!0,set:function(a,b,c,d,e,f,g,h,l){var m=this.elements;m[0]=a;m[1]=d;m[2]=g;m[3]=b;m[4]=e;m[5]=h;m[6]=c;m[7]=f;m[8]=l;return this},identity:function(){this.set(1,0,0,0,1,0,0,0,1);return this},clone:function(){return(new this.constructor).fromArray(this.elements)},copy:function(a){var b=this.elements;a=a.elements;b[0]=a[0];b[1]=a[1];b[2]=a[2];b[3]=
a[3];b[4]=a[4];b[5]=a[5];b[6]=a[6];b[7]=a[7];b[8]=a[8];return this},extractBasis:function(a,b,c){a.setFromMatrix3Column(this,0);b.setFromMatrix3Column(this,1);c.setFromMatrix3Column(this,2);return this},setFromMatrix4:function(a){a=a.elements;this.set(a[0],a[4],a[8],a[1],a[5],a[9],a[2],a[6],a[10]);return this},multiply:function(a){return this.multiplyMatrices(this,a)},premultiply:function(a){return this.multiplyMatrices(a,this)},multiplyMatrices:function(a,b){var c=a.elements,d=b.elements;b=this.elements;
a=c[0];var e=c[3],f=c[6],g=c[1],h=c[4],l=c[7],m=c[2],k=c[5];c=c[8];var p=d[0],n=d[3],r=d[6],q=d[1],v=d[4],E=d[7],w=d[2],t=d[5];d=d[8];b[0]=a*p+e*q+f*w;b[3]=a*n+e*v+f*t;b[6]=a*r+e*E+f*d;b[1]=g*p+h*q+l*w;b[4]=g*n+h*v+l*t;b[7]=g*r+h*E+l*d;b[2]=m*p+k*q+c*w;b[5]=m*n+k*v+c*t;b[8]=m*r+k*E+c*d;return this},multiplyScalar:function(a){var b=this.elements;b[0]*=a;b[3]*=a;b[6]*=a;b[1]*=a;b[4]*=a;b[7]*=a;b[2]*=a;b[5]*=a;b[8]*=a;return this},determinant:function(){var a=this.elements,b=a[0],c=a[1],d=a[2],e=a[3],
f=a[4],g=a[5],h=a[6],l=a[7];a=a[8];return b*f*a-b*g*l-c*e*a+c*g*h+d*e*l-d*f*h},getInverse:function(a,b){void 0!==b&&console.warn(&quot;THREE.Matrix3: .getInverse() can no longer be configured to throw on degenerate.&quot;);var c=a.elements;a=this.elements;b=c[0];var d=c[1],e=c[2],f=c[3],g=c[4],h=c[5],l=c[6],m=c[7];c=c[8];var k=c*g-h*m,p=h*l-c*f,n=m*f-g*l,r=b*k+d*p+e*n;if(0===r)return this.set(0,0,0,0,0,0,0,0,0);r=1/r;a[0]=k*r;a[1]=(e*m-c*d)*r;a[2]=(h*d-e*g)*r;a[3]=p*r;a[4]=(c*b-e*l)*r;a[5]=(e*f-h*b)*r;a[6]=
n*r;a[7]=(d*l-m*b)*r;a[8]=(g*b-d*f)*r;return this},transpose:function(){var a=this.elements;var b=a[1];a[1]=a[3];a[3]=b;b=a[2];a[2]=a[6];a[6]=b;b=a[5];a[5]=a[7];a[7]=b;return this},getNormalMatrix:function(a){return this.setFromMatrix4(a).getInverse(this).transpose()},transposeIntoArray:function(a){var b=this.elements;a[0]=b[0];a[1]=b[3];a[2]=b[6];a[3]=b[1];a[4]=b[4];a[5]=b[7];a[6]=b[2];a[7]=b[5];a[8]=b[8];return this},setUvTransform:function(a,b,c,d,e,f,g){var h=Math.cos(e);e=Math.sin(e);this.set(c*
h,c*e,-c*(h*f+e*g)+f+a,-d*e,d*h,-d*(-e*f+h*g)+g+b,0,0,1)},scale:function(a,b){var c=this.elements;c[0]*=a;c[3]*=a;c[6]*=a;c[1]*=b;c[4]*=b;c[7]*=b;return this},rotate:function(a){var b=Math.cos(a);a=Math.sin(a);var c=this.elements,d=c[0],e=c[3],f=c[6],g=c[1],h=c[4],l=c[7];c[0]=b*d+a*g;c[3]=b*e+a*h;c[6]=b*f+a*l;c[1]=-a*d+b*g;c[4]=-a*e+b*h;c[7]=-a*f+b*l;return this},translate:function(a,b){var c=this.elements;c[0]+=a*c[2];c[3]+=a*c[5];c[6]+=a*c[8];c[1]+=b*c[2];c[4]+=b*c[5];c[7]+=b*c[8];return this},
equals:function(a){var b=this.elements;a=a.elements;for(var c=0;9>c;c++)if(b[c]!==a[c])return!1;return!0},fromArray:function(a,b){void 0===b&&(b=0);for(var c=0;9>c;c++)this.elements[c]=a[c+b];return this},toArray:function(a,b){void 0===a&&(a=[]);void 0===b&&(b=0);var c=this.elements;a[b]=c[0];a[b+1]=c[1];a[b+2]=c[2];a[b+3]=c[3];a[b+4]=c[4];a[b+5]=c[5];a[b+6]=c[6];a[b+7]=c[7];a[b+8]=c[8];return a}});var nd,Lb={getDataURL:function(a){if(&quot;undefined&quot;==typeof HTMLCanvasElement)return a.src;if(!(a instanceof
HTMLCanvasElement)){void 0===nd&&(nd=document.createElementNS(&quot;http://www.w3.org/1999/xhtml&quot;,&quot;canvas&quot;));nd.width=a.width;nd.height=a.height;var b=nd.getContext(&quot;2d&quot;);a instanceof ImageData?b.putImageData(a,0,0):b.drawImage(a,0,0,a.width,a.height);a=nd}return 2048<a.width||2048<a.height?a.toDataURL(&quot;image/jpeg&quot;,.6):a.toDataURL(&quot;image/png&quot;)}},pj=0;V.DEFAULT_IMAGE=void 0;V.DEFAULT_MAPPING=300;V.prototype=Object.assign(Object.create(Ea.prototype),{constructor:V,isTexture:!0,updateMatrix:function(){this.matrix.setUvTransform(this.offset.x,
this.offset.y,this.repeat.x,this.repeat.y,this.rotation,this.center.x,this.center.y)},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.name=a.name;this.image=a.image;this.mipmaps=a.mipmaps.slice(0);this.mapping=a.mapping;this.wrapS=a.wrapS;this.wrapT=a.wrapT;this.magFilter=a.magFilter;this.minFilter=a.minFilter;this.anisotropy=a.anisotropy;this.format=a.format;this.internalFormat=a.internalFormat;this.type=a.type;this.offset.copy(a.offset);this.repeat.copy(a.repeat);
this.center.copy(a.center);this.rotation=a.rotation;this.matrixAutoUpdate=a.matrixAutoUpdate;this.matrix.copy(a.matrix);this.generateMipmaps=a.generateMipmaps;this.premultiplyAlpha=a.premultiplyAlpha;this.flipY=a.flipY;this.unpackAlignment=a.unpackAlignment;this.encoding=a.encoding;return this},toJSON:function(a){var b=void 0===a||&quot;string&quot;===typeof a;if(!b&&void 0!==a.textures[this.uuid])return a.textures[this.uuid];var c={metadata:{version:4.5,type:&quot;Texture&quot;,generator:&quot;Texture.toJSON&quot;},uuid:this.uuid,
name:this.name,mapping:this.mapping,repeat:[this.repeat.x,this.repeat.y],offset:[this.offset.x,this.offset.y],center:[this.center.x,this.center.y],rotation:this.rotation,wrap:[this.wrapS,this.wrapT],format:this.format,type:this.type,encoding:this.encoding,minFilter:this.minFilter,magFilter:this.magFilter,anisotropy:this.anisotropy,flipY:this.flipY,premultiplyAlpha:this.premultiplyAlpha,unpackAlignment:this.unpackAlignment};if(void 0!==this.image){var d=this.image;void 0===d.uuid&&(d.uuid=L.generateUUID());
if(!b&&void 0===a.images[d.uuid]){if(Array.isArray(d)){var e=[];for(var f=0,g=d.length;f<g;f++)e.push(Lb.getDataURL(d[f]))}else e=Lb.getDataURL(d);a.images[d.uuid]={uuid:d.uuid,url:e}}c.image=d.uuid}b||(a.textures[this.uuid]=c);return c},dispose:function(){this.dispatchEvent({type:&quot;dispose&quot;})},transformUv:function(a){if(300!==this.mapping)return a;a.applyMatrix3(this.matrix);if(0>a.x||1<a.x)switch(this.wrapS){case 1E3:a.x-=Math.floor(a.x);break;case 1001:a.x=0>a.x?0:1;break;case 1002:a.x=1===Math.abs(Math.floor(a.x)%
2)?Math.ceil(a.x)-a.x:a.x-Math.floor(a.x)}if(0>a.y||1<a.y)switch(this.wrapT){case 1E3:a.y-=Math.floor(a.y);break;case 1001:a.y=0>a.y?0:1;break;case 1002:a.y=1===Math.abs(Math.floor(a.y)%2)?Math.ceil(a.y)-a.y:a.y-Math.floor(a.y)}this.flipY&&(a.y=1-a.y);return a}});Object.defineProperty(V.prototype,&quot;needsUpdate&quot;,{set:function(a){!0===a&&this.version++}});Object.defineProperties(ka.prototype,{width:{get:function(){return this.z},set:function(a){this.z=a}},height:{get:function(){return this.w},set:function(a){this.w=
a}}});Object.assign(ka.prototype,{isVector4:!0,set:function(a,b,c,d){this.x=a;this.y=b;this.z=c;this.w=d;return this},setScalar:function(a){this.w=this.z=this.y=this.x=a;return this},setX:function(a){this.x=a;return this},setY:function(a){this.y=a;return this},setZ:function(a){this.z=a;return this},setW:function(a){this.w=a;return this},setComponent:function(a,b){switch(a){case 0:this.x=b;break;case 1:this.y=b;break;case 2:this.z=b;break;case 3:this.w=b;break;default:throw Error(&quot;index is out of range: &quot;+
a);}return this},getComponent:function(a){switch(a){case 0:return this.x;case 1:return this.y;case 2:return this.z;case 3:return this.w;default:throw Error(&quot;index is out of range: &quot;+a);}},clone:function(){return new this.constructor(this.x,this.y,this.z,this.w)},copy:function(a){this.x=a.x;this.y=a.y;this.z=a.z;this.w=void 0!==a.w?a.w:1;return this},add:function(a,b){if(void 0!==b)return console.warn(&quot;THREE.Vector4: .add() now only accepts one argument. Use .addVectors( a, b ) instead.&quot;),this.addVectors(a,
b);this.x+=a.x;this.y+=a.y;this.z+=a.z;this.w+=a.w;return this},addScalar:function(a){this.x+=a;this.y+=a;this.z+=a;this.w+=a;return this},addVectors:function(a,b){this.x=a.x+b.x;this.y=a.y+b.y;this.z=a.z+b.z;this.w=a.w+b.w;return this},addScaledVector:function(a,b){this.x+=a.x*b;this.y+=a.y*b;this.z+=a.z*b;this.w+=a.w*b;return this},sub:function(a,b){if(void 0!==b)return console.warn(&quot;THREE.Vector4: .sub() now only accepts one argument. Use .subVectors( a, b ) instead.&quot;),this.subVectors(a,b);this.x-=
a.x;this.y-=a.y;this.z-=a.z;this.w-=a.w;return this},subScalar:function(a){this.x-=a;this.y-=a;this.z-=a;this.w-=a;return this},subVectors:function(a,b){this.x=a.x-b.x;this.y=a.y-b.y;this.z=a.z-b.z;this.w=a.w-b.w;return this},multiplyScalar:function(a){this.x*=a;this.y*=a;this.z*=a;this.w*=a;return this},applyMatrix4:function(a){var b=this.x,c=this.y,d=this.z,e=this.w;a=a.elements;this.x=a[0]*b+a[4]*c+a[8]*d+a[12]*e;this.y=a[1]*b+a[5]*c+a[9]*d+a[13]*e;this.z=a[2]*b+a[6]*c+a[10]*d+a[14]*e;this.w=a[3]*
b+a[7]*c+a[11]*d+a[15]*e;return this},divideScalar:function(a){return this.multiplyScalar(1/a)},setAxisAngleFromQuaternion:function(a){this.w=2*Math.acos(a.w);var b=Math.sqrt(1-a.w*a.w);1E-4>b?(this.x=1,this.z=this.y=0):(this.x=a.x/b,this.y=a.y/b,this.z=a.z/b);return this},setAxisAngleFromRotationMatrix:function(a){a=a.elements;var b=a[0];var c=a[4];var d=a[8],e=a[1],f=a[5],g=a[9];var h=a[2];var l=a[6];var m=a[10];if(.01>Math.abs(c-e)&&.01>Math.abs(d-h)&&.01>Math.abs(g-l)){if(.1>Math.abs(c+e)&&.1>
Math.abs(d+h)&&.1>Math.abs(g+l)&&.1>Math.abs(b+f+m-3))return this.set(1,0,0,0),this;a=Math.PI;b=(b+1)/2;f=(f+1)/2;m=(m+1)/2;c=(c+e)/4;d=(d+h)/4;g=(g+l)/4;b>f&&b>m?.01>b?(l=0,c=h=.707106781):(l=Math.sqrt(b),h=c/l,c=d/l):f>m?.01>f?(l=.707106781,h=0,c=.707106781):(h=Math.sqrt(f),l=c/h,c=g/h):.01>m?(h=l=.707106781,c=0):(c=Math.sqrt(m),l=d/c,h=g/c);this.set(l,h,c,a);return this}a=Math.sqrt((l-g)*(l-g)+(d-h)*(d-h)+(e-c)*(e-c));.001>Math.abs(a)&&(a=1);this.x=(l-g)/a;this.y=(d-h)/a;this.z=(e-c)/a;this.w=
Math.acos((b+f+m-1)/2);return this},min:function(a){this.x=Math.min(this.x,a.x);this.y=Math.min(this.y,a.y);this.z=Math.min(this.z,a.z);this.w=Math.min(this.w,a.w);return this},max:function(a){this.x=Math.max(this.x,a.x);this.y=Math.max(this.y,a.y);this.z=Math.max(this.z,a.z);this.w=Math.max(this.w,a.w);return this},clamp:function(a,b){this.x=Math.max(a.x,Math.min(b.x,this.x));this.y=Math.max(a.y,Math.min(b.y,this.y));this.z=Math.max(a.z,Math.min(b.z,this.z));this.w=Math.max(a.w,Math.min(b.w,this.w));
return this},clampScalar:function(a,b){this.x=Math.max(a,Math.min(b,this.x));this.y=Math.max(a,Math.min(b,this.y));this.z=Math.max(a,Math.min(b,this.z));this.w=Math.max(a,Math.min(b,this.w));return this},clampLength:function(a,b){var c=this.length();return this.divideScalar(c||1).multiplyScalar(Math.max(a,Math.min(b,c)))},floor:function(){this.x=Math.floor(this.x);this.y=Math.floor(this.y);this.z=Math.floor(this.z);this.w=Math.floor(this.w);return this},ceil:function(){this.x=Math.ceil(this.x);this.y=
Math.ceil(this.y);this.z=Math.ceil(this.z);this.w=Math.ceil(this.w);return this},round:function(){this.x=Math.round(this.x);this.y=Math.round(this.y);this.z=Math.round(this.z);this.w=Math.round(this.w);return this},roundToZero:function(){this.x=0>this.x?Math.ceil(this.x):Math.floor(this.x);this.y=0>this.y?Math.ceil(this.y):Math.floor(this.y);this.z=0>this.z?Math.ceil(this.z):Math.floor(this.z);this.w=0>this.w?Math.ceil(this.w):Math.floor(this.w);return this},negate:function(){this.x=-this.x;this.y=
-this.y;this.z=-this.z;this.w=-this.w;return this},dot:function(a){return this.x*a.x+this.y*a.y+this.z*a.z+this.w*a.w},lengthSq:function(){return this.x*this.x+this.y*this.y+this.z*this.z+this.w*this.w},length:function(){return Math.sqrt(this.x*this.x+this.y*this.y+this.z*this.z+this.w*this.w)},manhattanLength:function(){return Math.abs(this.x)+Math.abs(this.y)+Math.abs(this.z)+Math.abs(this.w)},normalize:function(){return this.divideScalar(this.length()||1)},setLength:function(a){return this.normalize().multiplyScalar(a)},
lerp:function(a,b){this.x+=(a.x-this.x)*b;this.y+=(a.y-this.y)*b;this.z+=(a.z-this.z)*b;this.w+=(a.w-this.w)*b;return this},lerpVectors:function(a,b,c){return this.subVectors(b,a).multiplyScalar(c).add(a)},equals:function(a){return a.x===this.x&&a.y===this.y&&a.z===this.z&&a.w===this.w},fromArray:function(a,b){void 0===b&&(b=0);this.x=a[b];this.y=a[b+1];this.z=a[b+2];this.w=a[b+3];return this},toArray:function(a,b){void 0===a&&(a=[]);void 0===b&&(b=0);a[b]=this.x;a[b+1]=this.y;a[b+2]=this.z;a[b+3]=
this.w;return a},fromBufferAttribute:function(a,b,c){void 0!==c&&console.warn(&quot;THREE.Vector4: offset has been removed from .fromBufferAttribute().&quot;);this.x=a.getX(b);this.y=a.getY(b);this.z=a.getZ(b);this.w=a.getW(b);return this}});Ha.prototype=Object.assign(Object.create(Ea.prototype),{constructor:Ha,isWebGLRenderTarget:!0,setSize:function(a,b){if(this.width!==a||this.height!==b)this.width=a,this.height=b,this.texture.image.width=a,this.texture.image.height=b,this.dispose();this.viewport.set(0,0,
a,b);this.scissor.set(0,0,a,b)},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.width=a.width;this.height=a.height;this.viewport.copy(a.viewport);this.texture=a.texture.clone();this.depthBuffer=a.depthBuffer;this.stencilBuffer=a.stencilBuffer;this.depthTexture=a.depthTexture;return this},dispose:function(){this.dispatchEvent({type:&quot;dispose&quot;})}});Zf.prototype=Object.assign(Object.create(Ha.prototype),{constructor:Zf,isWebGLMultisampleRenderTarget:!0,copy:function(a){Ha.prototype.copy.call(this,
a);this.samples=a.samples;return this}});Object.assign(Aa,{slerp:function(a,b,c,d){return c.copy(a).slerp(b,d)},slerpFlat:function(a,b,c,d,e,f,g){var h=c[d+0],l=c[d+1],m=c[d+2];c=c[d+3];d=e[f+0];var k=e[f+1],p=e[f+2];e=e[f+3];if(c!==e||h!==d||l!==k||m!==p){f=1-g;var n=h*d+l*k+m*p+c*e,r=0<=n?1:-1,q=1-n*n;q>Number.EPSILON&&(q=Math.sqrt(q),n=Math.atan2(q,n*r),f=Math.sin(f*n)/q,g=Math.sin(g*n)/q);r*=g;h=h*f+d*r;l=l*f+k*r;m=m*f+p*r;c=c*f+e*r;f===1-g&&(g=1/Math.sqrt(h*h+l*l+m*m+c*c),h*=g,l*=g,m*=g,c*=g)}a[b]=
h;a[b+1]=l;a[b+2]=m;a[b+3]=c}});Object.defineProperties(Aa.prototype,{x:{get:function(){return this._x},set:function(a){this._x=a;this._onChangeCallback()}},y:{get:function(){return this._y},set:function(a){this._y=a;this._onChangeCallback()}},z:{get:function(){return this._z},set:function(a){this._z=a;this._onChangeCallback()}},w:{get:function(){return this._w},set:function(a){this._w=a;this._onChangeCallback()}}});Object.assign(Aa.prototype,{isQuaternion:!0,set:function(a,b,c,d){this._x=a;this._y=
b;this._z=c;this._w=d;this._onChangeCallback();return this},clone:function(){return new this.constructor(this._x,this._y,this._z,this._w)},copy:function(a){this._x=a.x;this._y=a.y;this._z=a.z;this._w=a.w;this._onChangeCallback();return this},setFromEuler:function(a,b){if(!a||!a.isEuler)throw Error(&quot;THREE.Quaternion: .setFromEuler() now expects an Euler rotation rather than a Vector3 and order.&quot;);var c=a._x,d=a._y,e=a._z;a=a.order;var f=Math.cos,g=Math.sin,h=f(c/2),l=f(d/2);f=f(e/2);c=g(c/2);d=g(d/
2);e=g(e/2);&quot;XYZ&quot;===a?(this._x=c*l*f+h*d*e,this._y=h*d*f-c*l*e,this._z=h*l*e+c*d*f,this._w=h*l*f-c*d*e):&quot;YXZ&quot;===a?(this._x=c*l*f+h*d*e,this._y=h*d*f-c*l*e,this._z=h*l*e-c*d*f,this._w=h*l*f+c*d*e):&quot;ZXY&quot;===a?(this._x=c*l*f-h*d*e,this._y=h*d*f+c*l*e,this._z=h*l*e+c*d*f,this._w=h*l*f-c*d*e):&quot;ZYX&quot;===a?(this._x=c*l*f-h*d*e,this._y=h*d*f+c*l*e,this._z=h*l*e-c*d*f,this._w=h*l*f+c*d*e):&quot;YZX&quot;===a?(this._x=c*l*f+h*d*e,this._y=h*d*f+c*l*e,this._z=h*l*e-c*d*f,this._w=h*l*f-c*d*e):&quot;XZY&quot;===a&&(this._x=c*l*f-h*d*
e,this._y=h*d*f-c*l*e,this._z=h*l*e+c*d*f,this._w=h*l*f+c*d*e);!1!==b&&this._onChangeCallback();return this},setFromAxisAngle:function(a,b){b/=2;var c=Math.sin(b);this._x=a.x*c;this._y=a.y*c;this._z=a.z*c;this._w=Math.cos(b);this._onChangeCallback();return this},setFromRotationMatrix:function(a){var b=a.elements,c=b[0];a=b[4];var d=b[8],e=b[1],f=b[5],g=b[9],h=b[2],l=b[6];b=b[10];var m=c+f+b;0<m?(c=.5/Math.sqrt(m+1),this._w=.25/c,this._x=(l-g)*c,this._y=(d-h)*c,this._z=(e-a)*c):c>f&&c>b?(c=2*Math.sqrt(1+
c-f-b),this._w=(l-g)/c,this._x=.25*c,this._y=(a+e)/c,this._z=(d+h)/c):f>b?(c=2*Math.sqrt(1+f-c-b),this._w=(d-h)/c,this._x=(a+e)/c,this._y=.25*c,this._z=(g+l)/c):(c=2*Math.sqrt(1+b-c-f),this._w=(e-a)/c,this._x=(d+h)/c,this._y=(g+l)/c,this._z=.25*c);this._onChangeCallback();return this},setFromUnitVectors:function(a,b){var c=a.dot(b)+1;1E-6>c?(c=0,Math.abs(a.x)>Math.abs(a.z)?(this._x=-a.y,this._y=a.x,this._z=0):(this._x=0,this._y=-a.z,this._z=a.y)):(this._x=a.y*b.z-a.z*b.y,this._y=a.z*b.x-a.x*b.z,this._z=
a.x*b.y-a.y*b.x);this._w=c;return this.normalize()},angleTo:function(a){return 2*Math.acos(Math.abs(L.clamp(this.dot(a),-1,1)))},rotateTowards:function(a,b){var c=this.angleTo(a);if(0===c)return this;this.slerp(a,Math.min(1,b/c));return this},inverse:function(){return this.conjugate()},conjugate:function(){this._x*=-1;this._y*=-1;this._z*=-1;this._onChangeCallback();return this},dot:function(a){return this._x*a._x+this._y*a._y+this._z*a._z+this._w*a._w},lengthSq:function(){return this._x*this._x+
this._y*this._y+this._z*this._z+this._w*this._w},length:function(){return Math.sqrt(this._x*this._x+this._y*this._y+this._z*this._z+this._w*this._w)},normalize:function(){var a=this.length();0===a?(this._z=this._y=this._x=0,this._w=1):(a=1/a,this._x*=a,this._y*=a,this._z*=a,this._w*=a);this._onChangeCallback();return this},multiply:function(a,b){return void 0!==b?(console.warn(&quot;THREE.Quaternion: .multiply() now only accepts one argument. Use .multiplyQuaternions( a, b ) instead.&quot;),this.multiplyQuaternions(a,
b)):this.multiplyQuaternions(this,a)},premultiply:function(a){return this.multiplyQuaternions(a,this)},multiplyQuaternions:function(a,b){var c=a._x,d=a._y,e=a._z;a=a._w;var f=b._x,g=b._y,h=b._z;b=b._w;this._x=c*b+a*f+d*h-e*g;this._y=d*b+a*g+e*f-c*h;this._z=e*b+a*h+c*g-d*f;this._w=a*b-c*f-d*g-e*h;this._onChangeCallback();return this},slerp:function(a,b){if(0===b)return this;if(1===b)return this.copy(a);var c=this._x,d=this._y,e=this._z,f=this._w,g=f*a._w+c*a._x+d*a._y+e*a._z;0>g?(this._w=-a._w,this._x=
-a._x,this._y=-a._y,this._z=-a._z,g=-g):this.copy(a);if(1<=g)return this._w=f,this._x=c,this._y=d,this._z=e,this;a=1-g*g;if(a<=Number.EPSILON)return g=1-b,this._w=g*f+b*this._w,this._x=g*c+b*this._x,this._y=g*d+b*this._y,this._z=g*e+b*this._z,this.normalize(),this._onChangeCallback(),this;a=Math.sqrt(a);var h=Math.atan2(a,g);g=Math.sin((1-b)*h)/a;b=Math.sin(b*h)/a;this._w=f*g+this._w*b;this._x=c*g+this._x*b;this._y=d*g+this._y*b;this._z=e*g+this._z*b;this._onChangeCallback();return this},equals:function(a){return a._x===
this._x&&a._y===this._y&&a._z===this._z&&a._w===this._w},fromArray:function(a,b){void 0===b&&(b=0);this._x=a[b];this._y=a[b+1];this._z=a[b+2];this._w=a[b+3];this._onChangeCallback();return this},toArray:function(a,b){void 0===a&&(a=[]);void 0===b&&(b=0);a[b]=this._x;a[b+1]=this._y;a[b+2]=this._z;a[b+3]=this._w;return a},fromBufferAttribute:function(a,b){this._x=a.getX(b);this._y=a.getY(b);this._z=a.getZ(b);this._w=a.getW(b);return this},_onChange:function(a){this._onChangeCallback=a;return this},
_onChangeCallback:function(){}});var ah=new n,Hi=new Aa;Object.assign(n.prototype,{isVector3:!0,set:function(a,b,c){this.x=a;this.y=b;this.z=c;return this},setScalar:function(a){this.z=this.y=this.x=a;return this},setX:function(a){this.x=a;return this},setY:function(a){this.y=a;return this},setZ:function(a){this.z=a;return this},setComponent:function(a,b){switch(a){case 0:this.x=b;break;case 1:this.y=b;break;case 2:this.z=b;break;default:throw Error(&quot;index is out of range: &quot;+a);}return this},getComponent:function(a){switch(a){case 0:return this.x;
case 1:return this.y;case 2:return this.z;default:throw Error(&quot;index is out of range: &quot;+a);}},clone:function(){return new this.constructor(this.x,this.y,this.z)},copy:function(a){this.x=a.x;this.y=a.y;this.z=a.z;return this},add:function(a,b){if(void 0!==b)return console.warn(&quot;THREE.Vector3: .add() now only accepts one argument. Use .addVectors( a, b ) instead.&quot;),this.addVectors(a,b);this.x+=a.x;this.y+=a.y;this.z+=a.z;return this},addScalar:function(a){this.x+=a;this.y+=a;this.z+=a;return this},
addVectors:function(a,b){this.x=a.x+b.x;this.y=a.y+b.y;this.z=a.z+b.z;return this},addScaledVector:function(a,b){this.x+=a.x*b;this.y+=a.y*b;this.z+=a.z*b;return this},sub:function(a,b){if(void 0!==b)return console.warn(&quot;THREE.Vector3: .sub() now only accepts one argument. Use .subVectors( a, b ) instead.&quot;),this.subVectors(a,b);this.x-=a.x;this.y-=a.y;this.z-=a.z;return this},subScalar:function(a){this.x-=a;this.y-=a;this.z-=a;return this},subVectors:function(a,b){this.x=a.x-b.x;this.y=a.y-b.y;this.z=
a.z-b.z;return this},multiply:function(a,b){if(void 0!==b)return console.warn(&quot;THREE.Vector3: .multiply() now only accepts one argument. Use .multiplyVectors( a, b ) instead.&quot;),this.multiplyVectors(a,b);this.x*=a.x;this.y*=a.y;this.z*=a.z;return this},multiplyScalar:function(a){this.x*=a;this.y*=a;this.z*=a;return this},multiplyVectors:function(a,b){this.x=a.x*b.x;this.y=a.y*b.y;this.z=a.z*b.z;return this},applyEuler:function(a){a&&a.isEuler||console.error(&quot;THREE.Vector3: .applyEuler() now expects an Euler rotation rather than a Vector3 and order.&quot;);
return this.applyQuaternion(Hi.setFromEuler(a))},applyAxisAngle:function(a,b){return this.applyQuaternion(Hi.setFromAxisAngle(a,b))},applyMatrix3:function(a){var b=this.x,c=this.y,d=this.z;a=a.elements;this.x=a[0]*b+a[3]*c+a[6]*d;this.y=a[1]*b+a[4]*c+a[7]*d;this.z=a[2]*b+a[5]*c+a[8]*d;return this},applyNormalMatrix:function(a){return this.applyMatrix3(a).normalize()},applyMatrix4:function(a){var b=this.x,c=this.y,d=this.z;a=a.elements;var e=1/(a[3]*b+a[7]*c+a[11]*d+a[15]);this.x=(a[0]*b+a[4]*c+a[8]*
d+a[12])*e;this.y=(a[1]*b+a[5]*c+a[9]*d+a[13])*e;this.z=(a[2]*b+a[6]*c+a[10]*d+a[14])*e;return this},applyQuaternion:function(a){var b=this.x,c=this.y,d=this.z,e=a.x,f=a.y,g=a.z;a=a.w;var h=a*b+f*d-g*c,l=a*c+g*b-e*d,m=a*d+e*c-f*b;b=-e*b-f*c-g*d;this.x=h*a+b*-e+l*-g-m*-f;this.y=l*a+b*-f+m*-e-h*-g;this.z=m*a+b*-g+h*-f-l*-e;return this},project:function(a){return this.applyMatrix4(a.matrixWorldInverse).applyMatrix4(a.projectionMatrix)},unproject:function(a){return this.applyMatrix4(a.projectionMatrixInverse).applyMatrix4(a.matrixWorld)},
transformDirection:function(a){var b=this.x,c=this.y,d=this.z;a=a.elements;this.x=a[0]*b+a[4]*c+a[8]*d;this.y=a[1]*b+a[5]*c+a[9]*d;this.z=a[2]*b+a[6]*c+a[10]*d;return this.normalize()},divide:function(a){this.x/=a.x;this.y/=a.y;this.z/=a.z;return this},divideScalar:function(a){return this.multiplyScalar(1/a)},min:function(a){this.x=Math.min(this.x,a.x);this.y=Math.min(this.y,a.y);this.z=Math.min(this.z,a.z);return this},max:function(a){this.x=Math.max(this.x,a.x);this.y=Math.max(this.y,a.y);this.z=
Math.max(this.z,a.z);return this},clamp:function(a,b){this.x=Math.max(a.x,Math.min(b.x,this.x));this.y=Math.max(a.y,Math.min(b.y,this.y));this.z=Math.max(a.z,Math.min(b.z,this.z));return this},clampScalar:function(a,b){this.x=Math.max(a,Math.min(b,this.x));this.y=Math.max(a,Math.min(b,this.y));this.z=Math.max(a,Math.min(b,this.z));return this},clampLength:function(a,b){var c=this.length();return this.divideScalar(c||1).multiplyScalar(Math.max(a,Math.min(b,c)))},floor:function(){this.x=Math.floor(this.x);
this.y=Math.floor(this.y);this.z=Math.floor(this.z);return this},ceil:function(){this.x=Math.ceil(this.x);this.y=Math.ceil(this.y);this.z=Math.ceil(this.z);return this},round:function(){this.x=Math.round(this.x);this.y=Math.round(this.y);this.z=Math.round(this.z);return this},roundToZero:function(){this.x=0>this.x?Math.ceil(this.x):Math.floor(this.x);this.y=0>this.y?Math.ceil(this.y):Math.floor(this.y);this.z=0>this.z?Math.ceil(this.z):Math.floor(this.z);return this},negate:function(){this.x=-this.x;
this.y=-this.y;this.z=-this.z;return this},dot:function(a){return this.x*a.x+this.y*a.y+this.z*a.z},lengthSq:function(){return this.x*this.x+this.y*this.y+this.z*this.z},length:function(){return Math.sqrt(this.x*this.x+this.y*this.y+this.z*this.z)},manhattanLength:function(){return Math.abs(this.x)+Math.abs(this.y)+Math.abs(this.z)},normalize:function(){return this.divideScalar(this.length()||1)},setLength:function(a){return this.normalize().multiplyScalar(a)},lerp:function(a,b){this.x+=(a.x-this.x)*
b;this.y+=(a.y-this.y)*b;this.z+=(a.z-this.z)*b;return this},lerpVectors:function(a,b,c){return this.subVectors(b,a).multiplyScalar(c).add(a)},cross:function(a,b){return void 0!==b?(console.warn(&quot;THREE.Vector3: .cross() now only accepts one argument. Use .crossVectors( a, b ) instead.&quot;),this.crossVectors(a,b)):this.crossVectors(this,a)},crossVectors:function(a,b){var c=a.x,d=a.y;a=a.z;var e=b.x,f=b.y;b=b.z;this.x=d*b-a*f;this.y=a*e-c*b;this.z=c*f-d*e;return this},projectOnVector:function(a){var b=
a.lengthSq();if(0===b)return this.set(0,0,0);b=a.dot(this)/b;return this.copy(a).multiplyScalar(b)},projectOnPlane:function(a){ah.copy(this).projectOnVector(a);return this.sub(ah)},reflect:function(a){return this.sub(ah.copy(a).multiplyScalar(2*this.dot(a)))},angleTo:function(a){var b=Math.sqrt(this.lengthSq()*a.lengthSq());if(0===b)return Math.PI/2;a=this.dot(a)/b;return Math.acos(L.clamp(a,-1,1))},distanceTo:function(a){return Math.sqrt(this.distanceToSquared(a))},distanceToSquared:function(a){var b=
this.x-a.x,c=this.y-a.y;a=this.z-a.z;return b*b+c*c+a*a},manhattanDistanceTo:function(a){return Math.abs(this.x-a.x)+Math.abs(this.y-a.y)+Math.abs(this.z-a.z)},setFromSpherical:function(a){return this.setFromSphericalCoords(a.radius,a.phi,a.theta)},setFromSphericalCoords:function(a,b,c){var d=Math.sin(b)*a;this.x=d*Math.sin(c);this.y=Math.cos(b)*a;this.z=d*Math.cos(c);return this},setFromCylindrical:function(a){return this.setFromCylindricalCoords(a.radius,a.theta,a.y)},setFromCylindricalCoords:function(a,
b,c){this.x=a*Math.sin(b);this.y=c;this.z=a*Math.cos(b);return this},setFromMatrixPosition:function(a){a=a.elements;this.x=a[12];this.y=a[13];this.z=a[14];return this},setFromMatrixScale:function(a){var b=this.setFromMatrixColumn(a,0).length(),c=this.setFromMatrixColumn(a,1).length();a=this.setFromMatrixColumn(a,2).length();this.x=b;this.y=c;this.z=a;return this},setFromMatrixColumn:function(a,b){return this.fromArray(a.elements,4*b)},setFromMatrix3Column:function(a,b){return this.fromArray(a.elements,
3*b)},equals:function(a){return a.x===this.x&&a.y===this.y&&a.z===this.z},fromArray:function(a,b){void 0===b&&(b=0);this.x=a[b];this.y=a[b+1];this.z=a[b+2];return this},toArray:function(a,b){void 0===a&&(a=[]);void 0===b&&(b=0);a[b]=this.x;a[b+1]=this.y;a[b+2]=this.z;return a},fromBufferAttribute:function(a,b,c){void 0!==c&&console.warn(&quot;THREE.Vector3: offset has been removed from .fromBufferAttribute().&quot;);this.x=a.getX(b);this.y=a.getY(b);this.z=a.getZ(b);return this}});var od=new n,X=new P,Rk=new n(0,
0,0),Sk=new n(1,1,1),Mb=new n,Df=new n,qa=new n;Object.assign(P.prototype,{isMatrix4:!0,set:function(a,b,c,d,e,f,g,h,l,m,k,p,n,r,q,v){var u=this.elements;u[0]=a;u[4]=b;u[8]=c;u[12]=d;u[1]=e;u[5]=f;u[9]=g;u[13]=h;u[2]=l;u[6]=m;u[10]=k;u[14]=p;u[3]=n;u[7]=r;u[11]=q;u[15]=v;return this},identity:function(){this.set(1,0,0,0,0,1,0,0,0,0,1,0,0,0,0,1);return this},clone:function(){return(new P).fromArray(this.elements)},copy:function(a){var b=this.elements;a=a.elements;b[0]=a[0];b[1]=a[1];b[2]=a[2];b[3]=
a[3];b[4]=a[4];b[5]=a[5];b[6]=a[6];b[7]=a[7];b[8]=a[8];b[9]=a[9];b[10]=a[10];b[11]=a[11];b[12]=a[12];b[13]=a[13];b[14]=a[14];b[15]=a[15];return this},copyPosition:function(a){var b=this.elements;a=a.elements;b[12]=a[12];b[13]=a[13];b[14]=a[14];return this},extractBasis:function(a,b,c){a.setFromMatrixColumn(this,0);b.setFromMatrixColumn(this,1);c.setFromMatrixColumn(this,2);return this},makeBasis:function(a,b,c){this.set(a.x,b.x,c.x,0,a.y,b.y,c.y,0,a.z,b.z,c.z,0,0,0,0,1);return this},extractRotation:function(a){var b=
this.elements,c=a.elements,d=1/od.setFromMatrixColumn(a,0).length(),e=1/od.setFromMatrixColumn(a,1).length();a=1/od.setFromMatrixColumn(a,2).length();b[0]=c[0]*d;b[1]=c[1]*d;b[2]=c[2]*d;b[3]=0;b[4]=c[4]*e;b[5]=c[5]*e;b[6]=c[6]*e;b[7]=0;b[8]=c[8]*a;b[9]=c[9]*a;b[10]=c[10]*a;b[11]=0;b[12]=0;b[13]=0;b[14]=0;b[15]=1;return this},makeRotationFromEuler:function(a){a&&a.isEuler||console.error(&quot;THREE.Matrix4: .makeRotationFromEuler() now expects a Euler rotation rather than a Vector3 and order.&quot;);var b=this.elements,
c=a.x,d=a.y,e=a.z,f=Math.cos(c);c=Math.sin(c);var g=Math.cos(d);d=Math.sin(d);var h=Math.cos(e);e=Math.sin(e);if(&quot;XYZ&quot;===a.order){a=f*h;var l=f*e,m=c*h,k=c*e;b[0]=g*h;b[4]=-g*e;b[8]=d;b[1]=l+m*d;b[5]=a-k*d;b[9]=-c*g;b[2]=k-a*d;b[6]=m+l*d;b[10]=f*g}else&quot;YXZ&quot;===a.order?(a=g*h,l=g*e,m=d*h,k=d*e,b[0]=a+k*c,b[4]=m*c-l,b[8]=f*d,b[1]=f*e,b[5]=f*h,b[9]=-c,b[2]=l*c-m,b[6]=k+a*c,b[10]=f*g):&quot;ZXY&quot;===a.order?(a=g*h,l=g*e,m=d*h,k=d*e,b[0]=a-k*c,b[4]=-f*e,b[8]=m+l*c,b[1]=l+m*c,b[5]=f*h,b[9]=k-a*c,b[2]=-f*d,b[6]=
c,b[10]=f*g):&quot;ZYX&quot;===a.order?(a=f*h,l=f*e,m=c*h,k=c*e,b[0]=g*h,b[4]=m*d-l,b[8]=a*d+k,b[1]=g*e,b[5]=k*d+a,b[9]=l*d-m,b[2]=-d,b[6]=c*g,b[10]=f*g):&quot;YZX&quot;===a.order?(a=f*g,l=f*d,m=c*g,k=c*d,b[0]=g*h,b[4]=k-a*e,b[8]=m*e+l,b[1]=e,b[5]=f*h,b[9]=-c*h,b[2]=-d*h,b[6]=l*e+m,b[10]=a-k*e):&quot;XZY&quot;===a.order&&(a=f*g,l=f*d,m=c*g,k=c*d,b[0]=g*h,b[4]=-e,b[8]=d*h,b[1]=a*e+k,b[5]=f*h,b[9]=l*e-m,b[2]=m*e-l,b[6]=c*h,b[10]=k*e+a);b[3]=0;b[7]=0;b[11]=0;b[12]=0;b[13]=0;b[14]=0;b[15]=1;return this},makeRotationFromQuaternion:function(a){return this.compose(Rk,
a,Sk)},lookAt:function(a,b,c){var d=this.elements;qa.subVectors(a,b);0===qa.lengthSq()&&(qa.z=1);qa.normalize();Mb.crossVectors(c,qa);0===Mb.lengthSq()&&(1===Math.abs(c.z)?qa.x+=1E-4:qa.z+=1E-4,qa.normalize(),Mb.crossVectors(c,qa));Mb.normalize();Df.crossVectors(qa,Mb);d[0]=Mb.x;d[4]=Df.x;d[8]=qa.x;d[1]=Mb.y;d[5]=Df.y;d[9]=qa.y;d[2]=Mb.z;d[6]=Df.z;d[10]=qa.z;return this},multiply:function(a,b){return void 0!==b?(console.warn(&quot;THREE.Matrix4: .multiply() now only accepts one argument. Use .multiplyMatrices( a, b ) instead.&quot;),
this.multiplyMatrices(a,b)):this.multiplyMatrices(this,a)},premultiply:function(a){return this.multiplyMatrices(a,this)},multiplyMatrices:function(a,b){var c=a.elements,d=b.elements;b=this.elements;a=c[0];var e=c[4],f=c[8],g=c[12],h=c[1],l=c[5],m=c[9],k=c[13],p=c[2],n=c[6],r=c[10],q=c[14],v=c[3],t=c[7],w=c[11];c=c[15];var z=d[0],A=d[4],C=d[8],y=d[12],B=d[1],D=d[5],F=d[9],G=d[13],K=d[2],H=d[6],L=d[10],M=d[14],N=d[3],O=d[7],P=d[11];d=d[15];b[0]=a*z+e*B+f*K+g*N;b[4]=a*A+e*D+f*H+g*O;b[8]=a*C+e*F+f*L+
g*P;b[12]=a*y+e*G+f*M+g*d;b[1]=h*z+l*B+m*K+k*N;b[5]=h*A+l*D+m*H+k*O;b[9]=h*C+l*F+m*L+k*P;b[13]=h*y+l*G+m*M+k*d;b[2]=p*z+n*B+r*K+q*N;b[6]=p*A+n*D+r*H+q*O;b[10]=p*C+n*F+r*L+q*P;b[14]=p*y+n*G+r*M+q*d;b[3]=v*z+t*B+w*K+c*N;b[7]=v*A+t*D+w*H+c*O;b[11]=v*C+t*F+w*L+c*P;b[15]=v*y+t*G+w*M+c*d;return this},multiplyScalar:function(a){var b=this.elements;b[0]*=a;b[4]*=a;b[8]*=a;b[12]*=a;b[1]*=a;b[5]*=a;b[9]*=a;b[13]*=a;b[2]*=a;b[6]*=a;b[10]*=a;b[14]*=a;b[3]*=a;b[7]*=a;b[11]*=a;b[15]*=a;return this},determinant:function(){var a=
this.elements,b=a[0],c=a[4],d=a[8],e=a[12],f=a[1],g=a[5],h=a[9],l=a[13],m=a[2],k=a[6],p=a[10],n=a[14];return a[3]*(+e*h*k-d*l*k-e*g*p+c*l*p+d*g*n-c*h*n)+a[7]*(+b*h*n-b*l*p+e*f*p-d*f*n+d*l*m-e*h*m)+a[11]*(+b*l*k-b*g*n-e*f*k+c*f*n+e*g*m-c*l*m)+a[15]*(-d*g*m-b*h*k+b*g*p+d*f*k-c*f*p+c*h*m)},transpose:function(){var a=this.elements;var b=a[1];a[1]=a[4];a[4]=b;b=a[2];a[2]=a[8];a[8]=b;b=a[6];a[6]=a[9];a[9]=b;b=a[3];a[3]=a[12];a[12]=b;b=a[7];a[7]=a[13];a[13]=b;b=a[11];a[11]=a[14];a[14]=b;return this},setPosition:function(a,
b,c){var d=this.elements;a.isVector3?(d[12]=a.x,d[13]=a.y,d[14]=a.z):(d[12]=a,d[13]=b,d[14]=c);return this},getInverse:function(a,b){void 0!==b&&console.warn(&quot;THREE.Matrix4: .getInverse() can no longer be configured to throw on degenerate.&quot;);b=this.elements;var c=a.elements;a=c[0];var d=c[1],e=c[2],f=c[3],g=c[4],h=c[5],l=c[6],m=c[7],k=c[8],p=c[9],n=c[10],r=c[11],q=c[12],v=c[13],t=c[14];c=c[15];var w=p*t*m-v*n*m+v*l*r-h*t*r-p*l*c+h*n*c,z=q*n*m-k*t*m-q*l*r+g*t*r+k*l*c-g*n*c,A=k*v*m-q*p*m+q*h*r-g*v*
r-k*h*c+g*p*c,C=q*p*l-k*v*l-q*h*n+g*v*n+k*h*t-g*p*t,y=a*w+d*z+e*A+f*C;if(0===y)return this.set(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0);y=1/y;b[0]=w*y;b[1]=(v*n*f-p*t*f-v*e*r+d*t*r+p*e*c-d*n*c)*y;b[2]=(h*t*f-v*l*f+v*e*m-d*t*m-h*e*c+d*l*c)*y;b[3]=(p*l*f-h*n*f-p*e*m+d*n*m+h*e*r-d*l*r)*y;b[4]=z*y;b[5]=(k*t*f-q*n*f+q*e*r-a*t*r-k*e*c+a*n*c)*y;b[6]=(q*l*f-g*t*f-q*e*m+a*t*m+g*e*c-a*l*c)*y;b[7]=(g*n*f-k*l*f+k*e*m-a*n*m-g*e*r+a*l*r)*y;b[8]=A*y;b[9]=(q*p*f-k*v*f-q*d*r+a*v*r+k*d*c-a*p*c)*y;b[10]=(g*v*f-q*h*f+q*d*m-
a*v*m-g*d*c+a*h*c)*y;b[11]=(k*h*f-g*p*f-k*d*m+a*p*m+g*d*r-a*h*r)*y;b[12]=C*y;b[13]=(k*v*e-q*p*e+q*d*n-a*v*n-k*d*t+a*p*t)*y;b[14]=(q*h*e-g*v*e-q*d*l+a*v*l+g*d*t-a*h*t)*y;b[15]=(g*p*e-k*h*e+k*d*l-a*p*l-g*d*n+a*h*n)*y;return this},scale:function(a){var b=this.elements,c=a.x,d=a.y;a=a.z;b[0]*=c;b[4]*=d;b[8]*=a;b[1]*=c;b[5]*=d;b[9]*=a;b[2]*=c;b[6]*=d;b[10]*=a;b[3]*=c;b[7]*=d;b[11]*=a;return this},getMaxScaleOnAxis:function(){var a=this.elements;return Math.sqrt(Math.max(a[0]*a[0]+a[1]*a[1]+a[2]*a[2],a[4]*
a[4]+a[5]*a[5]+a[6]*a[6],a[8]*a[8]+a[9]*a[9]+a[10]*a[10]))},makeTranslation:function(a,b,c){this.set(1,0,0,a,0,1,0,b,0,0,1,c,0,0,0,1);return this},makeRotationX:function(a){var b=Math.cos(a);a=Math.sin(a);this.set(1,0,0,0,0,b,-a,0,0,a,b,0,0,0,0,1);return this},makeRotationY:function(a){var b=Math.cos(a);a=Math.sin(a);this.set(b,0,a,0,0,1,0,0,-a,0,b,0,0,0,0,1);return this},makeRotationZ:function(a){var b=Math.cos(a);a=Math.sin(a);this.set(b,-a,0,0,a,b,0,0,0,0,1,0,0,0,0,1);return this},makeRotationAxis:function(a,
b){var c=Math.cos(b);b=Math.sin(b);var d=1-c,e=a.x,f=a.y;a=a.z;var g=d*e,h=d*f;this.set(g*e+c,g*f-b*a,g*a+b*f,0,g*f+b*a,h*f+c,h*a-b*e,0,g*a-b*f,h*a+b*e,d*a*a+c,0,0,0,0,1);return this},makeScale:function(a,b,c){this.set(a,0,0,0,0,b,0,0,0,0,c,0,0,0,0,1);return this},makeShear:function(a,b,c){this.set(1,b,c,0,a,1,c,0,a,b,1,0,0,0,0,1);return this},compose:function(a,b,c){var d=this.elements,e=b._x,f=b._y,g=b._z,h=b._w,l=e+e,m=f+f,k=g+g;b=e*l;var p=e*m;e*=k;var n=f*m;f*=k;g*=k;l*=h;m*=h;h*=k;k=c.x;var r=
c.y;c=c.z;d[0]=(1-(n+g))*k;d[1]=(p+h)*k;d[2]=(e-m)*k;d[3]=0;d[4]=(p-h)*r;d[5]=(1-(b+g))*r;d[6]=(f+l)*r;d[7]=0;d[8]=(e+m)*c;d[9]=(f-l)*c;d[10]=(1-(b+n))*c;d[11]=0;d[12]=a.x;d[13]=a.y;d[14]=a.z;d[15]=1;return this},decompose:function(a,b,c){var d=this.elements,e=od.set(d[0],d[1],d[2]).length(),f=od.set(d[4],d[5],d[6]).length(),g=od.set(d[8],d[9],d[10]).length();0>this.determinant()&&(e=-e);a.x=d[12];a.y=d[13];a.z=d[14];X.copy(this);a=1/e;d=1/f;var h=1/g;X.elements[0]*=a;X.elements[1]*=a;X.elements[2]*=
a;X.elements[4]*=d;X.elements[5]*=d;X.elements[6]*=d;X.elements[8]*=h;X.elements[9]*=h;X.elements[10]*=h;b.setFromRotationMatrix(X);c.x=e;c.y=f;c.z=g;return this},makePerspective:function(a,b,c,d,e,f){void 0===f&&console.warn(&quot;THREE.Matrix4: .makePerspective() has been redefined and has a new signature. Please check the docs.&quot;);var g=this.elements;g[0]=2*e/(b-a);g[4]=0;g[8]=(b+a)/(b-a);g[12]=0;g[1]=0;g[5]=2*e/(c-d);g[9]=(c+d)/(c-d);g[13]=0;g[2]=0;g[6]=0;g[10]=-(f+e)/(f-e);g[14]=-2*f*e/(f-e);g[3]=
0;g[7]=0;g[11]=-1;g[15]=0;return this},makeOrthographic:function(a,b,c,d,e,f){var g=this.elements,h=1/(b-a),l=1/(c-d),m=1/(f-e);g[0]=2*h;g[4]=0;g[8]=0;g[12]=-((b+a)*h);g[1]=0;g[5]=2*l;g[9]=0;g[13]=-((c+d)*l);g[2]=0;g[6]=0;g[10]=-2*m;g[14]=-((f+e)*m);g[3]=0;g[7]=0;g[11]=0;g[15]=1;return this},equals:function(a){var b=this.elements;a=a.elements;for(var c=0;16>c;c++)if(b[c]!==a[c])return!1;return!0},fromArray:function(a,b){void 0===b&&(b=0);for(var c=0;16>c;c++)this.elements[c]=a[c+b];return this},toArray:function(a,
b){void 0===a&&(a=[]);void 0===b&&(b=0);var c=this.elements;a[b]=c[0];a[b+1]=c[1];a[b+2]=c[2];a[b+3]=c[3];a[b+4]=c[4];a[b+5]=c[5];a[b+6]=c[6];a[b+7]=c[7];a[b+8]=c[8];a[b+9]=c[9];a[b+10]=c[10];a[b+11]=c[11];a[b+12]=c[12];a[b+13]=c[13];a[b+14]=c[14];a[b+15]=c[15];return a}});var Ii=new P,Ji=new Aa;Tb.RotationOrders=&quot;XYZ YZX ZXY XZY YXZ ZYX&quot;.split(&quot; &quot;);Tb.DefaultOrder=&quot;XYZ&quot;;Object.defineProperties(Tb.prototype,{x:{get:function(){return this._x},set:function(a){this._x=a;this._onChangeCallback()}},y:{get:function(){return this._y},
set:function(a){this._y=a;this._onChangeCallback()}},z:{get:function(){return this._z},set:function(a){this._z=a;this._onChangeCallback()}},order:{get:function(){return this._order},set:function(a){this._order=a;this._onChangeCallback()}}});Object.assign(Tb.prototype,{isEuler:!0,set:function(a,b,c,d){this._x=a;this._y=b;this._z=c;this._order=d||this._order;this._onChangeCallback();return this},clone:function(){return new this.constructor(this._x,this._y,this._z,this._order)},copy:function(a){this._x=
a._x;this._y=a._y;this._z=a._z;this._order=a._order;this._onChangeCallback();return this},setFromRotationMatrix:function(a,b,c){var d=L.clamp,e=a.elements;a=e[0];var f=e[4],g=e[8],h=e[1],l=e[5],m=e[9],k=e[2],p=e[6];e=e[10];b=b||this._order;&quot;XYZ&quot;===b?(this._y=Math.asin(d(g,-1,1)),.9999999>Math.abs(g)?(this._x=Math.atan2(-m,e),this._z=Math.atan2(-f,a)):(this._x=Math.atan2(p,l),this._z=0)):&quot;YXZ&quot;===b?(this._x=Math.asin(-d(m,-1,1)),.9999999>Math.abs(m)?(this._y=Math.atan2(g,e),this._z=Math.atan2(h,l)):
(this._y=Math.atan2(-k,a),this._z=0)):&quot;ZXY&quot;===b?(this._x=Math.asin(d(p,-1,1)),.9999999>Math.abs(p)?(this._y=Math.atan2(-k,e),this._z=Math.atan2(-f,l)):(this._y=0,this._z=Math.atan2(h,a))):&quot;ZYX&quot;===b?(this._y=Math.asin(-d(k,-1,1)),.9999999>Math.abs(k)?(this._x=Math.atan2(p,e),this._z=Math.atan2(h,a)):(this._x=0,this._z=Math.atan2(-f,l))):&quot;YZX&quot;===b?(this._z=Math.asin(d(h,-1,1)),.9999999>Math.abs(h)?(this._x=Math.atan2(-m,l),this._y=Math.atan2(-k,a)):(this._x=0,this._y=Math.atan2(g,e))):&quot;XZY&quot;===b?(this._z=
Math.asin(-d(f,-1,1)),.9999999>Math.abs(f)?(this._x=Math.atan2(p,l),this._y=Math.atan2(g,a)):(this._x=Math.atan2(-m,e),this._y=0)):console.warn(&quot;THREE.Euler: .setFromRotationMatrix() given unsupported order: &quot;+b);this._order=b;!1!==c&&this._onChangeCallback();return this},setFromQuaternion:function(a,b,c){Ii.makeRotationFromQuaternion(a);return this.setFromRotationMatrix(Ii,b,c)},setFromVector3:function(a,b){return this.set(a.x,a.y,a.z,b||this._order)},reorder:function(a){Ji.setFromEuler(this);return this.setFromQuaternion(Ji,
a)},equals:function(a){return a._x===this._x&&a._y===this._y&&a._z===this._z&&a._order===this._order},fromArray:function(a){this._x=a[0];this._y=a[1];this._z=a[2];void 0!==a[3]&&(this._order=a[3]);this._onChangeCallback();return this},toArray:function(a,b){void 0===a&&(a=[]);void 0===b&&(b=0);a[b]=this._x;a[b+1]=this._y;a[b+2]=this._z;a[b+3]=this._order;return a},toVector3:function(a){return a?a.set(this._x,this._y,this._z):new n(this._x,this._y,this._z)},_onChange:function(a){this._onChangeCallback=
a;return this},_onChangeCallback:function(){}});Object.assign(He.prototype,{set:function(a){this.mask=1<<a|0},enable:function(a){this.mask=this.mask|1<<a|0},enableAll:function(){this.mask=-1},toggle:function(a){this.mask^=1<<a|0},disable:function(a){this.mask&=~(1<<a|0)},disableAll:function(){this.mask=0},test:function(a){return 0!==(this.mask&a.mask)}});var qj=0,Ki=new n,pd=new Aa,yb=new P,Ef=new n,ye=new n,Tk=new n,Uk=new Aa,Li=new n(1,0,0),Mi=new n(0,1,0),Ni=new n(0,0,1),Vk={type:&quot;added&quot;},Wk={type:&quot;removed&quot;};
F.DefaultUp=new n(0,1,0);F.DefaultMatrixAutoUpdate=!0;F.prototype=Object.assign(Object.create(Ea.prototype),{constructor:F,isObject3D:!0,onBeforeRender:function(){},onAfterRender:function(){},applyMatrix4:function(a){this.matrixAutoUpdate&&this.updateMatrix();this.matrix.premultiply(a);this.matrix.decompose(this.position,this.quaternion,this.scale)},applyQuaternion:function(a){this.quaternion.premultiply(a);return this},setRotationFromAxisAngle:function(a,b){this.quaternion.setFromAxisAngle(a,b)},
setRotationFromEuler:function(a){this.quaternion.setFromEuler(a,!0)},setRotationFromMatrix:function(a){this.quaternion.setFromRotationMatrix(a)},setRotationFromQuaternion:function(a){this.quaternion.copy(a)},rotateOnAxis:function(a,b){pd.setFromAxisAngle(a,b);this.quaternion.multiply(pd);return this},rotateOnWorldAxis:function(a,b){pd.setFromAxisAngle(a,b);this.quaternion.premultiply(pd);return this},rotateX:function(a){return this.rotateOnAxis(Li,a)},rotateY:function(a){return this.rotateOnAxis(Mi,
a)},rotateZ:function(a){return this.rotateOnAxis(Ni,a)},translateOnAxis:function(a,b){Ki.copy(a).applyQuaternion(this.quaternion);this.position.add(Ki.multiplyScalar(b));return this},translateX:function(a){return this.translateOnAxis(Li,a)},translateY:function(a){return this.translateOnAxis(Mi,a)},translateZ:function(a){return this.translateOnAxis(Ni,a)},localToWorld:function(a){return a.applyMatrix4(this.matrixWorld)},worldToLocal:function(a){return a.applyMatrix4(yb.getInverse(this.matrixWorld))},
lookAt:function(a,b,c){a.isVector3?Ef.copy(a):Ef.set(a,b,c);a=this.parent;this.updateWorldMatrix(!0,!1);ye.setFromMatrixPosition(this.matrixWorld);this.isCamera||this.isLight?yb.lookAt(ye,Ef,this.up):yb.lookAt(Ef,ye,this.up);this.quaternion.setFromRotationMatrix(yb);a&&(yb.extractRotation(a.matrixWorld),pd.setFromRotationMatrix(yb),this.quaternion.premultiply(pd.inverse()))},add:function(a){if(1<arguments.length){for(var b=0;b<arguments.length;b++)this.add(arguments[b]);return this}if(a===this)return console.error(&quot;THREE.Object3D.add: object can't be added as a child of itself.&quot;,
a),this;a&&a.isObject3D?(null!==a.parent&&a.parent.remove(a),a.parent=this,this.children.push(a),a.dispatchEvent(Vk)):console.error(&quot;THREE.Object3D.add: object not an instance of THREE.Object3D.&quot;,a);return this},remove:function(a){if(1<arguments.length){for(var b=0;b<arguments.length;b++)this.remove(arguments[b]);return this}b=this.children.indexOf(a);-1!==b&&(a.parent=null,this.children.splice(b,1),a.dispatchEvent(Wk));return this},attach:function(a){this.updateWorldMatrix(!0,!1);yb.getInverse(this.matrixWorld);
null!==a.parent&&(a.parent.updateWorldMatrix(!0,!1),yb.multiply(a.parent.matrixWorld));a.applyMatrix4(yb);a.updateWorldMatrix(!1,!1);this.add(a);return this},getObjectById:function(a){return this.getObjectByProperty(&quot;id&quot;,a)},getObjectByName:function(a){return this.getObjectByProperty(&quot;name&quot;,a)},getObjectByProperty:function(a,b){if(this[a]===b)return this;for(var c=0,d=this.children.length;c<d;c++){var e=this.children[c].getObjectByProperty(a,b);if(void 0!==e)return e}},getWorldPosition:function(a){void 0===
a&&(console.warn(&quot;THREE.Object3D: .getWorldPosition() target is now required&quot;),a=new n);this.updateMatrixWorld(!0);return a.setFromMatrixPosition(this.matrixWorld)},getWorldQuaternion:function(a){void 0===a&&(console.warn(&quot;THREE.Object3D: .getWorldQuaternion() target is now required&quot;),a=new Aa);this.updateMatrixWorld(!0);this.matrixWorld.decompose(ye,a,Tk);return a},getWorldScale:function(a){void 0===a&&(console.warn(&quot;THREE.Object3D: .getWorldScale() target is now required&quot;),a=new n);this.updateMatrixWorld(!0);
this.matrixWorld.decompose(ye,Uk,a);return a},getWorldDirection:function(a){void 0===a&&(console.warn(&quot;THREE.Object3D: .getWorldDirection() target is now required&quot;),a=new n);this.updateMatrixWorld(!0);var b=this.matrixWorld.elements;return a.set(b[8],b[9],b[10]).normalize()},raycast:function(){},traverse:function(a){a(this);for(var b=this.children,c=0,d=b.length;c<d;c++)b[c].traverse(a)},traverseVisible:function(a){if(!1!==this.visible){a(this);for(var b=this.children,c=0,d=b.length;c<d;c++)b[c].traverseVisible(a)}},
traverseAncestors:function(a){var b=this.parent;null!==b&&(a(b),b.traverseAncestors(a))},updateMatrix:function(){this.matrix.compose(this.position,this.quaternion,this.scale);this.matrixWorldNeedsUpdate=!0},updateMatrixWorld:function(a){this.matrixAutoUpdate&&this.updateMatrix();if(this.matrixWorldNeedsUpdate||a)null===this.parent?this.matrixWorld.copy(this.matrix):this.matrixWorld.multiplyMatrices(this.parent.matrixWorld,this.matrix),this.matrixWorldNeedsUpdate=!1,a=!0;for(var b=this.children,c=
0,d=b.length;c<d;c++)b[c].updateMatrixWorld(a)},updateWorldMatrix:function(a,b){var c=this.parent;!0===a&&null!==c&&c.updateWorldMatrix(!0,!1);this.matrixAutoUpdate&&this.updateMatrix();null===this.parent?this.matrixWorld.copy(this.matrix):this.matrixWorld.multiplyMatrices(this.parent.matrixWorld,this.matrix);if(!0===b)for(a=this.children,b=0,c=a.length;b<c;b++)a[b].updateWorldMatrix(!1,!0)},toJSON:function(a){function b(b,c){void 0===b[c.uuid]&&(b[c.uuid]=c.toJSON(a));return c.uuid}function c(a){var b=
[],c;for(c in a){var d=a[c];delete d.metadata;b.push(d)}return b}var d=void 0===a||&quot;string&quot;===typeof a,e={};d&&(a={geometries:{},materials:{},textures:{},images:{},shapes:{}},e.metadata={version:4.5,type:&quot;Object&quot;,generator:&quot;Object3D.toJSON&quot;});var f={};f.uuid=this.uuid;f.type=this.type;&quot;&quot;!==this.name&&(f.name=this.name);!0===this.castShadow&&(f.castShadow=!0);!0===this.receiveShadow&&(f.receiveShadow=!0);!1===this.visible&&(f.visible=!1);!1===this.frustumCulled&&(f.frustumCulled=!1);0!==this.renderOrder&&
(f.renderOrder=this.renderOrder);&quot;{}&quot;!==JSON.stringify(this.userData)&&(f.userData=this.userData);f.layers=this.layers.mask;f.matrix=this.matrix.toArray();!1===this.matrixAutoUpdate&&(f.matrixAutoUpdate=!1);this.isInstancedMesh&&(f.type=&quot;InstancedMesh&quot;,f.count=this.count,f.instanceMatrix=this.instanceMatrix.toJSON());if(this.isMesh||this.isLine||this.isPoints){f.geometry=b(a.geometries,this.geometry);var g=this.geometry.parameters;if(void 0!==g&&void 0!==g.shapes)if(g=g.shapes,Array.isArray(g))for(var h=
0,l=g.length;h<l;h++)b(a.shapes,g[h]);else b(a.shapes,g)}if(void 0!==this.material)if(Array.isArray(this.material)){g=[];h=0;for(l=this.material.length;h<l;h++)g.push(b(a.materials,this.material[h]));f.material=g}else f.material=b(a.materials,this.material);if(0<this.children.length)for(f.children=[],h=0;h<this.children.length;h++)f.children.push(this.children[h].toJSON(a).object);if(d){d=c(a.geometries);h=c(a.materials);l=c(a.textures);var m=c(a.images);g=c(a.shapes);0<d.length&&(e.geometries=d);
0<h.length&&(e.materials=h);0<l.length&&(e.textures=l);0<m.length&&(e.images=m);0<g.length&&(e.shapes=g)}e.object=f;return e},clone:function(a){return(new this.constructor).copy(this,a)},copy:function(a,b){void 0===b&&(b=!0);this.name=a.name;this.up.copy(a.up);this.position.copy(a.position);this.quaternion.copy(a.quaternion);this.scale.copy(a.scale);this.matrix.copy(a.matrix);this.matrixWorld.copy(a.matrixWorld);this.matrixAutoUpdate=a.matrixAutoUpdate;this.matrixWorldNeedsUpdate=a.matrixWorldNeedsUpdate;
this.layers.mask=a.layers.mask;this.visible=a.visible;this.castShadow=a.castShadow;this.receiveShadow=a.receiveShadow;this.frustumCulled=a.frustumCulled;this.renderOrder=a.renderOrder;this.userData=JSON.parse(JSON.stringify(a.userData));if(!0===b)for(b=0;b<a.children.length;b++)this.add(a.children[b].clone());return this}});ob.prototype=Object.assign(Object.create(F.prototype),{constructor:ob,isScene:!0,copy:function(a,b){F.prototype.copy.call(this,a,b);null!==a.background&&(this.background=a.background.clone());
null!==a.environment&&(this.environment=a.environment.clone());null!==a.fog&&(this.fog=a.fog.clone());null!==a.overrideMaterial&&(this.overrideMaterial=a.overrideMaterial.clone());this.autoUpdate=a.autoUpdate;this.matrixAutoUpdate=a.matrixAutoUpdate;return this},toJSON:function(a){var b=F.prototype.toJSON.call(this,a);null!==this.background&&(b.object.background=this.background.toJSON(a));null!==this.environment&&(b.object.environment=this.environment.toJSON(a));null!==this.fog&&(b.object.fog=this.fog.toJSON());
return b},dispose:function(){this.dispatchEvent({type:&quot;dispose&quot;})}});var zb=[new n,new n,new n,new n,new n,new n,new n,new n],ze=new n,bh=new Sa,qd=new n,rd=new n,sd=new n,Nb=new n,Ob=new n,sc=new n,Ae=new n,Ff=new n,Gf=new n,Ub=new n;Object.assign(Sa.prototype,{isBox3:!0,set:function(a,b){this.min.copy(a);this.max.copy(b);return this},setFromArray:function(a){for(var b=Infinity,c=Infinity,d=Infinity,e=-Infinity,f=-Infinity,g=-Infinity,h=0,l=a.length;h<l;h+=3){var m=a[h],k=a[h+1],p=a[h+2];m<b&&(b=
m);k<c&&(c=k);p<d&&(d=p);m>e&&(e=m);k>f&&(f=k);p>g&&(g=p)}this.min.set(b,c,d);this.max.set(e,f,g);return this},setFromBufferAttribute:function(a){for(var b=Infinity,c=Infinity,d=Infinity,e=-Infinity,f=-Infinity,g=-Infinity,h=0,l=a.count;h<l;h++){var m=a.getX(h),k=a.getY(h),p=a.getZ(h);m<b&&(b=m);k<c&&(c=k);p<d&&(d=p);m>e&&(e=m);k>f&&(f=k);p>g&&(g=p)}this.min.set(b,c,d);this.max.set(e,f,g);return this},setFromPoints:function(a){this.makeEmpty();for(var b=0,c=a.length;b<c;b++)this.expandByPoint(a[b]);
return this},setFromCenterAndSize:function(a,b){b=ze.copy(b).multiplyScalar(.5);this.min.copy(a).sub(b);this.max.copy(a).add(b);return this},setFromObject:function(a){this.makeEmpty();return this.expandByObject(a)},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.min.copy(a.min);this.max.copy(a.max);return this},makeEmpty:function(){this.min.x=this.min.y=this.min.z=Infinity;this.max.x=this.max.y=this.max.z=-Infinity;return this},isEmpty:function(){return this.max.x<
this.min.x||this.max.y<this.min.y||this.max.z<this.min.z},getCenter:function(a){void 0===a&&(console.warn(&quot;THREE.Box3: .getCenter() target is now required&quot;),a=new n);return this.isEmpty()?a.set(0,0,0):a.addVectors(this.min,this.max).multiplyScalar(.5)},getSize:function(a){void 0===a&&(console.warn(&quot;THREE.Box3: .getSize() target is now required&quot;),a=new n);return this.isEmpty()?a.set(0,0,0):a.subVectors(this.max,this.min)},expandByPoint:function(a){this.min.min(a);this.max.max(a);return this},expandByVector:function(a){this.min.sub(a);
this.max.add(a);return this},expandByScalar:function(a){this.min.addScalar(-a);this.max.addScalar(a);return this},expandByObject:function(a){a.updateWorldMatrix(!1,!1);var b=a.geometry;void 0!==b&&(null===b.boundingBox&&b.computeBoundingBox(),bh.copy(b.boundingBox),bh.applyMatrix4(a.matrixWorld),this.union(bh));a=a.children;b=0;for(var c=a.length;b<c;b++)this.expandByObject(a[b]);return this},containsPoint:function(a){return a.x<this.min.x||a.x>this.max.x||a.y<this.min.y||a.y>this.max.y||a.z<this.min.z||
a.z>this.max.z?!1:!0},containsBox:function(a){return this.min.x<=a.min.x&&a.max.x<=this.max.x&&this.min.y<=a.min.y&&a.max.y<=this.max.y&&this.min.z<=a.min.z&&a.max.z<=this.max.z},getParameter:function(a,b){void 0===b&&(console.warn(&quot;THREE.Box3: .getParameter() target is now required&quot;),b=new n);return b.set((a.x-this.min.x)/(this.max.x-this.min.x),(a.y-this.min.y)/(this.max.y-this.min.y),(a.z-this.min.z)/(this.max.z-this.min.z))},intersectsBox:function(a){return a.max.x<this.min.x||a.min.x>this.max.x||
a.max.y<this.min.y||a.min.y>this.max.y||a.max.z<this.min.z||a.min.z>this.max.z?!1:!0},intersectsSphere:function(a){this.clampPoint(a.center,ze);return ze.distanceToSquared(a.center)<=a.radius*a.radius},intersectsPlane:function(a){if(0<a.normal.x){var b=a.normal.x*this.min.x;var c=a.normal.x*this.max.x}else b=a.normal.x*this.max.x,c=a.normal.x*this.min.x;0<a.normal.y?(b+=a.normal.y*this.min.y,c+=a.normal.y*this.max.y):(b+=a.normal.y*this.max.y,c+=a.normal.y*this.min.y);0<a.normal.z?(b+=a.normal.z*
this.min.z,c+=a.normal.z*this.max.z):(b+=a.normal.z*this.max.z,c+=a.normal.z*this.min.z);return b<=-a.constant&&c>=-a.constant},intersectsTriangle:function(a){if(this.isEmpty())return!1;this.getCenter(Ae);Ff.subVectors(this.max,Ae);qd.subVectors(a.a,Ae);rd.subVectors(a.b,Ae);sd.subVectors(a.c,Ae);Nb.subVectors(rd,qd);Ob.subVectors(sd,rd);sc.subVectors(qd,sd);a=[0,-Nb.z,Nb.y,0,-Ob.z,Ob.y,0,-sc.z,sc.y,Nb.z,0,-Nb.x,Ob.z,0,-Ob.x,sc.z,0,-sc.x,-Nb.y,Nb.x,0,-Ob.y,Ob.x,0,-sc.y,sc.x,0];if(!$f(a,qd,rd,sd,Ff))return!1;
a=[1,0,0,0,1,0,0,0,1];if(!$f(a,qd,rd,sd,Ff))return!1;Gf.crossVectors(Nb,Ob);a=[Gf.x,Gf.y,Gf.z];return $f(a,qd,rd,sd,Ff)},clampPoint:function(a,b){void 0===b&&(console.warn(&quot;THREE.Box3: .clampPoint() target is now required&quot;),b=new n);return b.copy(a).clamp(this.min,this.max)},distanceToPoint:function(a){return ze.copy(a).clamp(this.min,this.max).sub(a).length()},getBoundingSphere:function(a){void 0===a&&console.error(&quot;THREE.Box3: .getBoundingSphere() target is now required&quot;);this.getCenter(a.center);
a.radius=.5*this.getSize(ze).length();return a},intersect:function(a){this.min.max(a.min);this.max.min(a.max);this.isEmpty()&&this.makeEmpty();return this},union:function(a){this.min.min(a.min);this.max.max(a.max);return this},applyMatrix4:function(a){if(this.isEmpty())return this;zb[0].set(this.min.x,this.min.y,this.min.z).applyMatrix4(a);zb[1].set(this.min.x,this.min.y,this.max.z).applyMatrix4(a);zb[2].set(this.min.x,this.max.y,this.min.z).applyMatrix4(a);zb[3].set(this.min.x,this.max.y,this.max.z).applyMatrix4(a);
zb[4].set(this.max.x,this.min.y,this.min.z).applyMatrix4(a);zb[5].set(this.max.x,this.min.y,this.max.z).applyMatrix4(a);zb[6].set(this.max.x,this.max.y,this.min.z).applyMatrix4(a);zb[7].set(this.max.x,this.max.y,this.max.z).applyMatrix4(a);this.setFromPoints(zb);return this},translate:function(a){this.min.add(a);this.max.add(a);return this},equals:function(a){return a.min.equals(this.min)&&a.max.equals(this.max)}});var Xk=new Sa;Object.assign(pb.prototype,{set:function(a,b){this.center.copy(a);this.radius=
b;return this},setFromPoints:function(a,b){var c=this.center;void 0!==b?c.copy(b):Xk.setFromPoints(a).getCenter(c);for(var d=b=0,e=a.length;d<e;d++)b=Math.max(b,c.distanceToSquared(a[d]));this.radius=Math.sqrt(b);return this},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.center.copy(a.center);this.radius=a.radius;return this},empty:function(){return 0>=this.radius},containsPoint:function(a){return a.distanceToSquared(this.center)<=this.radius*this.radius},distanceToPoint:function(a){return a.distanceTo(this.center)-
this.radius},intersectsSphere:function(a){var b=this.radius+a.radius;return a.center.distanceToSquared(this.center)<=b*b},intersectsBox:function(a){return a.intersectsSphere(this)},intersectsPlane:function(a){return Math.abs(a.distanceToPoint(this.center))<=this.radius},clampPoint:function(a,b){var c=this.center.distanceToSquared(a);void 0===b&&(console.warn(&quot;THREE.Sphere: .clampPoint() target is now required&quot;),b=new n);b.copy(a);c>this.radius*this.radius&&(b.sub(this.center).normalize(),b.multiplyScalar(this.radius).add(this.center));
return b},getBoundingBox:function(a){void 0===a&&(console.warn(&quot;THREE.Sphere: .getBoundingBox() target is now required&quot;),a=new Sa);a.set(this.center,this.center);a.expandByScalar(this.radius);return a},applyMatrix4:function(a){this.center.applyMatrix4(a);this.radius*=a.getMaxScaleOnAxis();return this},translate:function(a){this.center.add(a);return this},equals:function(a){return a.center.equals(this.center)&&a.radius===this.radius}});var Ab=new n,ch=new n,Hf=new n,Pb=new n,dh=new n,If=new n,eh=new n;
Object.assign(Vb.prototype,{set:function(a,b){this.origin.copy(a);this.direction.copy(b);return this},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.origin.copy(a.origin);this.direction.copy(a.direction);return this},at:function(a,b){void 0===b&&(console.warn(&quot;THREE.Ray: .at() target is now required&quot;),b=new n);return b.copy(this.direction).multiplyScalar(a).add(this.origin)},lookAt:function(a){this.direction.copy(a).sub(this.origin).normalize();return this},recast:function(a){this.origin.copy(this.at(a,
Ab));return this},closestPointToPoint:function(a,b){void 0===b&&(console.warn(&quot;THREE.Ray: .closestPointToPoint() target is now required&quot;),b=new n);b.subVectors(a,this.origin);a=b.dot(this.direction);return 0>a?b.copy(this.origin):b.copy(this.direction).multiplyScalar(a).add(this.origin)},distanceToPoint:function(a){return Math.sqrt(this.distanceSqToPoint(a))},distanceSqToPoint:function(a){var b=Ab.subVectors(a,this.origin).dot(this.direction);if(0>b)return this.origin.distanceToSquared(a);Ab.copy(this.direction).multiplyScalar(b).add(this.origin);
return Ab.distanceToSquared(a)},distanceSqToSegment:function(a,b,c,d){ch.copy(a).add(b).multiplyScalar(.5);Hf.copy(b).sub(a).normalize();Pb.copy(this.origin).sub(ch);var e=.5*a.distanceTo(b),f=-this.direction.dot(Hf),g=Pb.dot(this.direction),h=-Pb.dot(Hf),l=Pb.lengthSq(),m=Math.abs(1-f*f);if(0<m){a=f*h-g;b=f*g-h;var k=e*m;0<=a?b>=-k?b<=k?(e=1/m,a*=e,b*=e,f=a*(a+f*b+2*g)+b*(f*a+b+2*h)+l):(b=e,a=Math.max(0,-(f*b+g)),f=-a*a+b*(b+2*h)+l):(b=-e,a=Math.max(0,-(f*b+g)),f=-a*a+b*(b+2*h)+l):b<=-k?(a=Math.max(0,
-(-f*e+g)),b=0<a?-e:Math.min(Math.max(-e,-h),e),f=-a*a+b*(b+2*h)+l):b<=k?(a=0,b=Math.min(Math.max(-e,-h),e),f=b*(b+2*h)+l):(a=Math.max(0,-(f*e+g)),b=0<a?e:Math.min(Math.max(-e,-h),e),f=-a*a+b*(b+2*h)+l)}else b=0<f?-e:e,a=Math.max(0,-(f*b+g)),f=-a*a+b*(b+2*h)+l;c&&c.copy(this.direction).multiplyScalar(a).add(this.origin);d&&d.copy(Hf).multiplyScalar(b).add(ch);return f},intersectSphere:function(a,b){Ab.subVectors(a.center,this.origin);var c=Ab.dot(this.direction),d=Ab.dot(Ab)-c*c;a=a.radius*a.radius;
if(d>a)return null;a=Math.sqrt(a-d);d=c-a;c+=a;return 0>d&&0>c?null:0>d?this.at(c,b):this.at(d,b)},intersectsSphere:function(a){return this.distanceSqToPoint(a.center)<=a.radius*a.radius},distanceToPlane:function(a){var b=a.normal.dot(this.direction);if(0===b)return 0===a.distanceToPoint(this.origin)?0:null;a=-(this.origin.dot(a.normal)+a.constant)/b;return 0<=a?a:null},intersectPlane:function(a,b){a=this.distanceToPlane(a);return null===a?null:this.at(a,b)},intersectsPlane:function(a){var b=a.distanceToPoint(this.origin);
return 0===b||0>a.normal.dot(this.direction)*b?!0:!1},intersectBox:function(a,b){var c=1/this.direction.x;var d=1/this.direction.y;var e=1/this.direction.z,f=this.origin;if(0<=c){var g=(a.min.x-f.x)*c;c*=a.max.x-f.x}else g=(a.max.x-f.x)*c,c*=a.min.x-f.x;if(0<=d){var h=(a.min.y-f.y)*d;d*=a.max.y-f.y}else h=(a.max.y-f.y)*d,d*=a.min.y-f.y;if(g>d||h>c)return null;if(h>g||g!==g)g=h;if(d<c||c!==c)c=d;0<=e?(h=(a.min.z-f.z)*e,a=(a.max.z-f.z)*e):(h=(a.max.z-f.z)*e,a=(a.min.z-f.z)*e);if(g>a||h>c)return null;
if(h>g||g!==g)g=h;if(a<c||c!==c)c=a;return 0>c?null:this.at(0<=g?g:c,b)},intersectsBox:function(a){return null!==this.intersectBox(a,Ab)},intersectTriangle:function(a,b,c,d,e){dh.subVectors(b,a);If.subVectors(c,a);eh.crossVectors(dh,If);b=this.direction.dot(eh);if(0<b){if(d)return null;d=1}else if(0>b)d=-1,b=-b;else return null;Pb.subVectors(this.origin,a);a=d*this.direction.dot(If.crossVectors(Pb,If));if(0>a)return null;c=d*this.direction.dot(dh.cross(Pb));if(0>c||a+c>b)return null;a=-d*Pb.dot(eh);
return 0>a?null:this.at(a/b,e)},applyMatrix4:function(a){this.origin.applyMatrix4(a);this.direction.transformDirection(a);return this},equals:function(a){return a.origin.equals(this.origin)&&a.direction.equals(this.direction)}});var fh=new n,Yk=new n,Zk=new wa;Object.assign(Ta.prototype,{isPlane:!0,set:function(a,b){this.normal.copy(a);this.constant=b;return this},setComponents:function(a,b,c,d){this.normal.set(a,b,c);this.constant=d;return this},setFromNormalAndCoplanarPoint:function(a,b){this.normal.copy(a);
this.constant=-b.dot(this.normal);return this},setFromCoplanarPoints:function(a,b,c){b=fh.subVectors(c,b).cross(Yk.subVectors(a,b)).normalize();this.setFromNormalAndCoplanarPoint(b,a);return this},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.normal.copy(a.normal);this.constant=a.constant;return this},normalize:function(){var a=1/this.normal.length();this.normal.multiplyScalar(a);this.constant*=a;return this},negate:function(){this.constant*=-1;this.normal.negate();
return this},distanceToPoint:function(a){return this.normal.dot(a)+this.constant},distanceToSphere:function(a){return this.distanceToPoint(a.center)-a.radius},projectPoint:function(a,b){void 0===b&&(console.warn(&quot;THREE.Plane: .projectPoint() target is now required&quot;),b=new n);return b.copy(this.normal).multiplyScalar(-this.distanceToPoint(a)).add(a)},intersectLine:function(a,b){void 0===b&&(console.warn(&quot;THREE.Plane: .intersectLine() target is now required&quot;),b=new n);var c=a.delta(fh),d=this.normal.dot(c);
if(0===d){if(0===this.distanceToPoint(a.start))return b.copy(a.start)}else if(d=-(a.start.dot(this.normal)+this.constant)/d,!(0>d||1<d))return b.copy(c).multiplyScalar(d).add(a.start)},intersectsLine:function(a){var b=this.distanceToPoint(a.start);a=this.distanceToPoint(a.end);return 0>b&&0<a||0>a&&0<b},intersectsBox:function(a){return a.intersectsPlane(this)},intersectsSphere:function(a){return a.intersectsPlane(this)},coplanarPoint:function(a){void 0===a&&(console.warn(&quot;THREE.Plane: .coplanarPoint() target is now required&quot;),
a=new n);return a.copy(this.normal).multiplyScalar(-this.constant)},applyMatrix4:function(a,b){b=b||Zk.getNormalMatrix(a);a=this.coplanarPoint(fh).applyMatrix4(a);b=this.normal.applyMatrix3(b).normalize();this.constant=-a.dot(b);return this},translate:function(a){this.constant-=a.dot(this.normal);return this},equals:function(a){return a.normal.equals(this.normal)&&a.constant===this.constant}});var bb=new n,Bb=new n,gh=new n,Cb=new n,td=new n,ud=new n,Oi=new n,hh=new n,ih=new n,jh=new n;Object.assign(oa,
{getNormal:function(a,b,c,d){void 0===d&&(console.warn(&quot;THREE.Triangle: .getNormal() target is now required&quot;),d=new n);d.subVectors(c,b);bb.subVectors(a,b);d.cross(bb);a=d.lengthSq();return 0<a?d.multiplyScalar(1/Math.sqrt(a)):d.set(0,0,0)},getBarycoord:function(a,b,c,d,e){bb.subVectors(d,b);Bb.subVectors(c,b);gh.subVectors(a,b);a=bb.dot(bb);b=bb.dot(Bb);c=bb.dot(gh);var f=Bb.dot(Bb);d=Bb.dot(gh);var g=a*f-b*b;void 0===e&&(console.warn(&quot;THREE.Triangle: .getBarycoord() target is now required&quot;),e=new n);
if(0===g)return e.set(-2,-1,-1);g=1/g;f=(f*c-b*d)*g;a=(a*d-b*c)*g;return e.set(1-f-a,a,f)},containsPoint:function(a,b,c,d){oa.getBarycoord(a,b,c,d,Cb);return 0<=Cb.x&&0<=Cb.y&&1>=Cb.x+Cb.y},getUV:function(a,b,c,d,e,f,g,h){this.getBarycoord(a,b,c,d,Cb);h.set(0,0);h.addScaledVector(e,Cb.x);h.addScaledVector(f,Cb.y);h.addScaledVector(g,Cb.z);return h},isFrontFacing:function(a,b,c,d){bb.subVectors(c,b);Bb.subVectors(a,b);return 0>bb.cross(Bb).dot(d)?!0:!1}});Object.assign(oa.prototype,{set:function(a,
b,c){this.a.copy(a);this.b.copy(b);this.c.copy(c);return this},setFromPointsAndIndices:function(a,b,c,d){this.a.copy(a[b]);this.b.copy(a[c]);this.c.copy(a[d]);return this},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.a.copy(a.a);this.b.copy(a.b);this.c.copy(a.c);return this},getArea:function(){bb.subVectors(this.c,this.b);Bb.subVectors(this.a,this.b);return.5*bb.cross(Bb).length()},getMidpoint:function(a){void 0===a&&(console.warn(&quot;THREE.Triangle: .getMidpoint() target is now required&quot;),
a=new n);return a.addVectors(this.a,this.b).add(this.c).multiplyScalar(1/3)},getNormal:function(a){return oa.getNormal(this.a,this.b,this.c,a)},getPlane:function(a){void 0===a&&(console.warn(&quot;THREE.Triangle: .getPlane() target is now required&quot;),a=new Ta);return a.setFromCoplanarPoints(this.a,this.b,this.c)},getBarycoord:function(a,b){return oa.getBarycoord(a,this.a,this.b,this.c,b)},getUV:function(a,b,c,d,e){return oa.getUV(a,this.a,this.b,this.c,b,c,d,e)},containsPoint:function(a){return oa.containsPoint(a,
this.a,this.b,this.c)},isFrontFacing:function(a){return oa.isFrontFacing(this.a,this.b,this.c,a)},intersectsBox:function(a){return a.intersectsTriangle(this)},closestPointToPoint:function(a,b){void 0===b&&(console.warn(&quot;THREE.Triangle: .closestPointToPoint() target is now required&quot;),b=new n);var c=this.a,d=this.b,e=this.c;td.subVectors(d,c);ud.subVectors(e,c);hh.subVectors(a,c);var f=td.dot(hh),g=ud.dot(hh);if(0>=f&&0>=g)return b.copy(c);ih.subVectors(a,d);var h=td.dot(ih),l=ud.dot(ih);if(0<=h&&l<=
h)return b.copy(d);var m=f*l-h*g;if(0>=m&&0<=f&&0>=h)return d=f/(f-h),b.copy(c).addScaledVector(td,d);jh.subVectors(a,e);a=td.dot(jh);var k=ud.dot(jh);if(0<=k&&a<=k)return b.copy(e);f=a*g-f*k;if(0>=f&&0<=g&&0>=k)return m=g/(g-k),b.copy(c).addScaledVector(ud,m);g=h*k-a*l;if(0>=g&&0<=l-h&&0<=a-k)return Oi.subVectors(e,d),m=(l-h)/(l-h+(a-k)),b.copy(d).addScaledVector(Oi,m);e=1/(g+f+m);d=f*e;m*=e;return b.copy(c).addScaledVector(td,d).addScaledVector(ud,m)},equals:function(a){return a.a.equals(this.a)&&
a.b.equals(this.b)&&a.c.equals(this.c)}});var Pi={aliceblue:15792383,antiquewhite:16444375,aqua:65535,aquamarine:8388564,azure:15794175,beige:16119260,bisque:16770244,black:0,blanchedalmond:16772045,blue:255,blueviolet:9055202,brown:10824234,burlywood:14596231,cadetblue:6266528,chartreuse:8388352,chocolate:13789470,coral:16744272,cornflowerblue:6591981,cornsilk:16775388,crimson:14423100,cyan:65535,darkblue:139,darkcyan:35723,darkgoldenrod:12092939,darkgray:11119017,darkgreen:25600,darkgrey:11119017,
darkkhaki:12433259,darkmagenta:9109643,darkolivegreen:5597999,darkorange:16747520,darkorchid:10040012,darkred:9109504,darksalmon:15308410,darkseagreen:9419919,darkslateblue:4734347,darkslategray:3100495,darkslategrey:3100495,darkturquoise:52945,darkviolet:9699539,deeppink:16716947,deepskyblue:49151,dimgray:6908265,dimgrey:6908265,dodgerblue:2003199,firebrick:11674146,floralwhite:16775920,forestgreen:2263842,fuchsia:16711935,gainsboro:14474460,ghostwhite:16316671,gold:16766720,goldenrod:14329120,gray:8421504,
green:32768,greenyellow:11403055,grey:8421504,honeydew:15794160,hotpink:16738740,indianred:13458524,indigo:4915330,ivory:16777200,khaki:15787660,lavender:15132410,lavenderblush:16773365,lawngreen:8190976,lemonchiffon:16775885,lightblue:11393254,lightcoral:15761536,lightcyan:14745599,lightgoldenrodyellow:16448210,lightgray:13882323,lightgreen:9498256,lightgrey:13882323,lightpink:16758465,lightsalmon:16752762,lightseagreen:2142890,lightskyblue:8900346,lightslategray:7833753,lightslategrey:7833753,lightsteelblue:11584734,
lightyellow:16777184,lime:65280,limegreen:3329330,linen:16445670,magenta:16711935,maroon:8388608,mediumaquamarine:6737322,mediumblue:205,mediumorchid:12211667,mediumpurple:9662683,mediumseagreen:3978097,mediumslateblue:8087790,mediumspringgreen:64154,mediumturquoise:4772300,mediumvioletred:13047173,midnightblue:1644912,mintcream:16121850,mistyrose:16770273,moccasin:16770229,navajowhite:16768685,navy:128,oldlace:16643558,olive:8421376,olivedrab:7048739,orange:16753920,orangered:16729344,orchid:14315734,
palegoldenrod:15657130,palegreen:10025880,paleturquoise:11529966,palevioletred:14381203,papayawhip:16773077,peachpuff:16767673,peru:13468991,pink:16761035,plum:14524637,powderblue:11591910,purple:8388736,rebeccapurple:6697881,red:16711680,rosybrown:12357519,royalblue:4286945,saddlebrown:9127187,salmon:16416882,sandybrown:16032864,seagreen:3050327,seashell:16774638,sienna:10506797,silver:12632256,skyblue:8900331,slateblue:6970061,slategray:7372944,slategrey:7372944,snow:16775930,springgreen:65407,
steelblue:4620980,tan:13808780,teal:32896,thistle:14204888,tomato:16737095,turquoise:4251856,violet:15631086,wheat:16113331,white:16777215,whitesmoke:16119285,yellow:16776960,yellowgreen:10145074},za={h:0,s:0,l:0},Jf={h:0,s:0,l:0};Object.assign(A.prototype,{isColor:!0,r:1,g:1,b:1,set:function(a){a&&a.isColor?this.copy(a):&quot;number&quot;===typeof a?this.setHex(a):&quot;string&quot;===typeof a&&this.setStyle(a);return this},setScalar:function(a){this.b=this.g=this.r=a;return this},setHex:function(a){a=Math.floor(a);
this.r=(a>>16&255)/255;this.g=(a>>8&255)/255;this.b=(a&255)/255;return this},setRGB:function(a,b,c){this.r=a;this.g=b;this.b=c;return this},setHSL:function(a,b,c){a=L.euclideanModulo(a,1);b=L.clamp(b,0,1);c=L.clamp(c,0,1);0===b?this.r=this.g=this.b=c:(b=.5>=c?c*(1+b):c+b-c*b,c=2*c-b,this.r=ag(c,b,a+1/3),this.g=ag(c,b,a),this.b=ag(c,b,a-1/3));return this},setStyle:function(a){function b(b){void 0!==b&&1>parseFloat(b)&&console.warn(&quot;THREE.Color: Alpha component of &quot;+a+&quot; will be ignored.&quot;)}var c;if(c=
/^((?:rgb|hsl)a?)\(\s*([^\)]*)\)/.exec(a)){var d=c[2];switch(c[1]){case &quot;rgb&quot;:case &quot;rgba&quot;:if(c=/^(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*(,\s*([0-9]*\.?[0-9]+)\s*)?$/.exec(d))return this.r=Math.min(255,parseInt(c[1],10))/255,this.g=Math.min(255,parseInt(c[2],10))/255,this.b=Math.min(255,parseInt(c[3],10))/255,b(c[5]),this;if(c=/^(\d+)%\s*,\s*(\d+)%\s*,\s*(\d+)%\s*(,\s*([0-9]*\.?[0-9]+)\s*)?$/.exec(d))return this.r=Math.min(100,parseInt(c[1],10))/100,this.g=Math.min(100,parseInt(c[2],10))/100,this.b=Math.min(100,
parseInt(c[3],10))/100,b(c[5]),this;break;case &quot;hsl&quot;:case &quot;hsla&quot;:if(c=/^([0-9]*\.?[0-9]+)\s*,\s*(\d+)%\s*,\s*(\d+)%\s*(,\s*([0-9]*\.?[0-9]+)\s*)?$/.exec(d)){d=parseFloat(c[1])/360;var e=parseInt(c[2],10)/100,f=parseInt(c[3],10)/100;b(c[5]);return this.setHSL(d,e,f)}}}else if(c=/^#([A-Fa-f0-9]+)$/.exec(a)){c=c[1];d=c.length;if(3===d)return this.r=parseInt(c.charAt(0)+c.charAt(0),16)/255,this.g=parseInt(c.charAt(1)+c.charAt(1),16)/255,this.b=parseInt(c.charAt(2)+c.charAt(2),16)/255,this;if(6===d)return this.r=
parseInt(c.charAt(0)+c.charAt(1),16)/255,this.g=parseInt(c.charAt(2)+c.charAt(3),16)/255,this.b=parseInt(c.charAt(4)+c.charAt(5),16)/255,this}return a&&0<a.length?this.setColorName(a):this},setColorName:function(a){var b=Pi[a];void 0!==b?this.setHex(b):console.warn(&quot;THREE.Color: Unknown color &quot;+a);return this},clone:function(){return new this.constructor(this.r,this.g,this.b)},copy:function(a){this.r=a.r;this.g=a.g;this.b=a.b;return this},copyGammaToLinear:function(a,b){void 0===b&&(b=2);this.r=Math.pow(a.r,
b);this.g=Math.pow(a.g,b);this.b=Math.pow(a.b,b);return this},copyLinearToGamma:function(a,b){void 0===b&&(b=2);b=0<b?1/b:1;this.r=Math.pow(a.r,b);this.g=Math.pow(a.g,b);this.b=Math.pow(a.b,b);return this},convertGammaToLinear:function(a){this.copyGammaToLinear(this,a);return this},convertLinearToGamma:function(a){this.copyLinearToGamma(this,a);return this},copySRGBToLinear:function(a){this.r=bg(a.r);this.g=bg(a.g);this.b=bg(a.b);return this},copyLinearToSRGB:function(a){this.r=cg(a.r);this.g=cg(a.g);
this.b=cg(a.b);return this},convertSRGBToLinear:function(){this.copySRGBToLinear(this);return this},convertLinearToSRGB:function(){this.copyLinearToSRGB(this);return this},getHex:function(){return 255*this.r<<16^255*this.g<<8^255*this.b<<0},getHexString:function(){return(&quot;000000&quot;+this.getHex().toString(16)).slice(-6)},getHSL:function(a){void 0===a&&(console.warn(&quot;THREE.Color: .getHSL() target is now required&quot;),a={h:0,s:0,l:0});var b=this.r,c=this.g,d=this.b,e=Math.max(b,c,d),f=Math.min(b,c,d),g,h=
(f+e)/2;if(f===e)f=g=0;else{var l=e-f;f=.5>=h?l/(e+f):l/(2-e-f);switch(e){case b:g=(c-d)/l+(c<d?6:0);break;case c:g=(d-b)/l+2;break;case d:g=(b-c)/l+4}g/=6}a.h=g;a.s=f;a.l=h;return a},getStyle:function(){return&quot;rgb(&quot;+(255*this.r|0)+&quot;,&quot;+(255*this.g|0)+&quot;,&quot;+(255*this.b|0)+&quot;)&quot;},offsetHSL:function(a,b,c){this.getHSL(za);za.h+=a;za.s+=b;za.l+=c;this.setHSL(za.h,za.s,za.l);return this},add:function(a){this.r+=a.r;this.g+=a.g;this.b+=a.b;return this},addColors:function(a,b){this.r=a.r+b.r;this.g=a.g+b.g;
this.b=a.b+b.b;return this},addScalar:function(a){this.r+=a;this.g+=a;this.b+=a;return this},sub:function(a){this.r=Math.max(0,this.r-a.r);this.g=Math.max(0,this.g-a.g);this.b=Math.max(0,this.b-a.b);return this},multiply:function(a){this.r*=a.r;this.g*=a.g;this.b*=a.b;return this},multiplyScalar:function(a){this.r*=a;this.g*=a;this.b*=a;return this},lerp:function(a,b){this.r+=(a.r-this.r)*b;this.g+=(a.g-this.g)*b;this.b+=(a.b-this.b)*b;return this},lerpHSL:function(a,b){this.getHSL(za);a.getHSL(Jf);
a=L.lerp(za.h,Jf.h,b);var c=L.lerp(za.s,Jf.s,b);b=L.lerp(za.l,Jf.l,b);this.setHSL(a,c,b);return this},equals:function(a){return a.r===this.r&&a.g===this.g&&a.b===this.b},fromArray:function(a,b){void 0===b&&(b=0);this.r=a[b];this.g=a[b+1];this.b=a[b+2];return this},toArray:function(a,b){void 0===a&&(a=[]);void 0===b&&(b=0);a[b]=this.r;a[b+1]=this.g;a[b+2]=this.b;return a},toJSON:function(){return this.getHex()}});A.NAMES=Pi;Object.assign(Bc.prototype,{clone:function(){return(new this.constructor).copy(this)},
copy:function(a){this.a=a.a;this.b=a.b;this.c=a.c;this.normal.copy(a.normal);this.color.copy(a.color);this.materialIndex=a.materialIndex;for(var b=0,c=a.vertexNormals.length;b<c;b++)this.vertexNormals[b]=a.vertexNormals[b].clone();b=0;for(c=a.vertexColors.length;b<c;b++)this.vertexColors[b]=a.vertexColors[b].clone();return this}});var rj=0;K.prototype=Object.assign(Object.create(Ea.prototype),{constructor:K,isMaterial:!0,onBeforeCompile:function(){},setValues:function(a){if(void 0!==a)for(var b in a){var c=
a[b];if(void 0===c)console.warn(&quot;THREE.Material: '&quot;+b+&quot;' parameter is undefined.&quot;);else if(&quot;shading&quot;===b)console.warn(&quot;THREE.&quot;+this.type+&quot;: .shading has been removed. Use the boolean .flatShading instead.&quot;),this.flatShading=1===c?!0:!1;else{var d=this[b];void 0===d?console.warn(&quot;THREE.&quot;+this.type+&quot;: '&quot;+b+&quot;' is not a property of this material.&quot;):d&&d.isColor?d.set(c):d&&d.isVector3&&c&&c.isVector3?d.copy(c):this[b]=c}}},toJSON:function(a){function b(a){var b=[],c;for(c in a){var d=a[c];delete d.metadata;
b.push(d)}return b}var c=void 0===a||&quot;string&quot;===typeof a;c&&(a={textures:{},images:{}});var d={metadata:{version:4.5,type:&quot;Material&quot;,generator:&quot;Material.toJSON&quot;}};d.uuid=this.uuid;d.type=this.type;&quot;&quot;!==this.name&&(d.name=this.name);this.color&&this.color.isColor&&(d.color=this.color.getHex());void 0!==this.roughness&&(d.roughness=this.roughness);void 0!==this.metalness&&(d.metalness=this.metalness);this.sheen&&this.sheen.isColor&&(d.sheen=this.sheen.getHex());this.emissive&&this.emissive.isColor&&
(d.emissive=this.emissive.getHex());this.emissiveIntensity&&1!==this.emissiveIntensity&&(d.emissiveIntensity=this.emissiveIntensity);this.specular&&this.specular.isColor&&(d.specular=this.specular.getHex());void 0!==this.shininess&&(d.shininess=this.shininess);void 0!==this.clearcoat&&(d.clearcoat=this.clearcoat);void 0!==this.clearcoatRoughness&&(d.clearcoatRoughness=this.clearcoatRoughness);this.clearcoatMap&&this.clearcoatMap.isTexture&&(d.clearcoatMap=this.clearcoatMap.toJSON(a).uuid);this.clearcoatRoughnessMap&&
this.clearcoatRoughnessMap.isTexture&&(d.clearcoatRoughnessMap=this.clearcoatRoughnessMap.toJSON(a).uuid);this.clearcoatNormalMap&&this.clearcoatNormalMap.isTexture&&(d.clearcoatNormalMap=this.clearcoatNormalMap.toJSON(a).uuid,d.clearcoatNormalScale=this.clearcoatNormalScale.toArray());this.map&&this.map.isTexture&&(d.map=this.map.toJSON(a).uuid);this.matcap&&this.matcap.isTexture&&(d.matcap=this.matcap.toJSON(a).uuid);this.alphaMap&&this.alphaMap.isTexture&&(d.alphaMap=this.alphaMap.toJSON(a).uuid);
this.lightMap&&this.lightMap.isTexture&&(d.lightMap=this.lightMap.toJSON(a).uuid);this.aoMap&&this.aoMap.isTexture&&(d.aoMap=this.aoMap.toJSON(a).uuid,d.aoMapIntensity=this.aoMapIntensity);this.bumpMap&&this.bumpMap.isTexture&&(d.bumpMap=this.bumpMap.toJSON(a).uuid,d.bumpScale=this.bumpScale);this.normalMap&&this.normalMap.isTexture&&(d.normalMap=this.normalMap.toJSON(a).uuid,d.normalMapType=this.normalMapType,d.normalScale=this.normalScale.toArray());this.displacementMap&&this.displacementMap.isTexture&&
(d.displacementMap=this.displacementMap.toJSON(a).uuid,d.displacementScale=this.displacementScale,d.displacementBias=this.displacementBias);this.roughnessMap&&this.roughnessMap.isTexture&&(d.roughnessMap=this.roughnessMap.toJSON(a).uuid);this.metalnessMap&&this.metalnessMap.isTexture&&(d.metalnessMap=this.metalnessMap.toJSON(a).uuid);this.emissiveMap&&this.emissiveMap.isTexture&&(d.emissiveMap=this.emissiveMap.toJSON(a).uuid);this.specularMap&&this.specularMap.isTexture&&(d.specularMap=this.specularMap.toJSON(a).uuid);
this.envMap&&this.envMap.isTexture&&(d.envMap=this.envMap.toJSON(a).uuid,d.reflectivity=this.reflectivity,d.refractionRatio=this.refractionRatio,void 0!==this.combine&&(d.combine=this.combine),void 0!==this.envMapIntensity&&(d.envMapIntensity=this.envMapIntensity));this.gradientMap&&this.gradientMap.isTexture&&(d.gradientMap=this.gradientMap.toJSON(a).uuid);void 0!==this.size&&(d.size=this.size);void 0!==this.sizeAttenuation&&(d.sizeAttenuation=this.sizeAttenuation);1!==this.blending&&(d.blending=
this.blending);!0===this.flatShading&&(d.flatShading=this.flatShading);0!==this.side&&(d.side=this.side);this.vertexColors&&(d.vertexColors=!0);1>this.opacity&&(d.opacity=this.opacity);!0===this.transparent&&(d.transparent=this.transparent);d.depthFunc=this.depthFunc;d.depthTest=this.depthTest;d.depthWrite=this.depthWrite;d.stencilWrite=this.stencilWrite;d.stencilWriteMask=this.stencilWriteMask;d.stencilFunc=this.stencilFunc;d.stencilRef=this.stencilRef;d.stencilFuncMask=this.stencilFuncMask;d.stencilFail=
this.stencilFail;d.stencilZFail=this.stencilZFail;d.stencilZPass=this.stencilZPass;this.rotation&&0!==this.rotation&&(d.rotation=this.rotation);!0===this.polygonOffset&&(d.polygonOffset=!0);0!==this.polygonOffsetFactor&&(d.polygonOffsetFactor=this.polygonOffsetFactor);0!==this.polygonOffsetUnits&&(d.polygonOffsetUnits=this.polygonOffsetUnits);this.linewidth&&1!==this.linewidth&&(d.linewidth=this.linewidth);void 0!==this.dashSize&&(d.dashSize=this.dashSize);void 0!==this.gapSize&&(d.gapSize=this.gapSize);
void 0!==this.scale&&(d.scale=this.scale);!0===this.dithering&&(d.dithering=!0);0<this.alphaTest&&(d.alphaTest=this.alphaTest);!0===this.premultipliedAlpha&&(d.premultipliedAlpha=this.premultipliedAlpha);!0===this.wireframe&&(d.wireframe=this.wireframe);1<this.wireframeLinewidth&&(d.wireframeLinewidth=this.wireframeLinewidth);&quot;round&quot;!==this.wireframeLinecap&&(d.wireframeLinecap=this.wireframeLinecap);&quot;round&quot;!==this.wireframeLinejoin&&(d.wireframeLinejoin=this.wireframeLinejoin);!0===this.morphTargets&&
(d.morphTargets=!0);!0===this.morphNormals&&(d.morphNormals=!0);!0===this.skinning&&(d.skinning=!0);!1===this.visible&&(d.visible=!1);!1===this.toneMapped&&(d.toneMapped=!1);&quot;{}&quot;!==JSON.stringify(this.userData)&&(d.userData=this.userData);c&&(c=b(a.textures),a=b(a.images),0<c.length&&(d.textures=c),0<a.length&&(d.images=a));return d},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.name=a.name;this.fog=a.fog;this.blending=a.blending;this.side=a.side;this.flatShading=
a.flatShading;this.vertexColors=a.vertexColors;this.opacity=a.opacity;this.transparent=a.transparent;this.blendSrc=a.blendSrc;this.blendDst=a.blendDst;this.blendEquation=a.blendEquation;this.blendSrcAlpha=a.blendSrcAlpha;this.blendDstAlpha=a.blendDstAlpha;this.blendEquationAlpha=a.blendEquationAlpha;this.depthFunc=a.depthFunc;this.depthTest=a.depthTest;this.depthWrite=a.depthWrite;this.stencilWriteMask=a.stencilWriteMask;this.stencilFunc=a.stencilFunc;this.stencilRef=a.stencilRef;this.stencilFuncMask=
a.stencilFuncMask;this.stencilFail=a.stencilFail;this.stencilZFail=a.stencilZFail;this.stencilZPass=a.stencilZPass;this.stencilWrite=a.stencilWrite;var b=a.clippingPlanes,c=null;if(null!==b){var d=b.length;c=Array(d);for(var e=0;e!==d;++e)c[e]=b[e].clone()}this.clippingPlanes=c;this.clipIntersection=a.clipIntersection;this.clipShadows=a.clipShadows;this.shadowSide=a.shadowSide;this.colorWrite=a.colorWrite;this.precision=a.precision;this.polygonOffset=a.polygonOffset;this.polygonOffsetFactor=a.polygonOffsetFactor;
this.polygonOffsetUnits=a.polygonOffsetUnits;this.dithering=a.dithering;this.alphaTest=a.alphaTest;this.premultipliedAlpha=a.premultipliedAlpha;this.visible=a.visible;this.toneMapped=a.toneMapped;this.userData=JSON.parse(JSON.stringify(a.userData));return this},dispose:function(){this.dispatchEvent({type:&quot;dispose&quot;})}});Object.defineProperty(K.prototype,&quot;needsUpdate&quot;,{set:function(a){!0===a&&this.version++}});Oa.prototype=Object.create(K.prototype);Oa.prototype.constructor=Oa;Oa.prototype.isMeshBasicMaterial=
!0;Oa.prototype.copy=function(a){K.prototype.copy.call(this,a);this.color.copy(a.color);this.map=a.map;this.lightMap=a.lightMap;this.lightMapIntensity=a.lightMapIntensity;this.aoMap=a.aoMap;this.aoMapIntensity=a.aoMapIntensity;this.specularMap=a.specularMap;this.alphaMap=a.alphaMap;this.envMap=a.envMap;this.combine=a.combine;this.reflectivity=a.reflectivity;this.refractionRatio=a.refractionRatio;this.wireframe=a.wireframe;this.wireframeLinewidth=a.wireframeLinewidth;this.wireframeLinecap=a.wireframeLinecap;
this.wireframeLinejoin=a.wireframeLinejoin;this.skinning=a.skinning;this.morphTargets=a.morphTargets;return this};var ba=new n;Object.defineProperty(M.prototype,&quot;needsUpdate&quot;,{set:function(a){!0===a&&this.version++}});Object.assign(M.prototype,{isBufferAttribute:!0,onUploadCallback:function(){},setUsage:function(a){this.usage=a;return this},copy:function(a){this.name=a.name;this.array=new a.array.constructor(a.array);this.itemSize=a.itemSize;this.count=a.count;this.normalized=a.normalized;this.usage=
a.usage;return this},copyAt:function(a,b,c){a*=this.itemSize;c*=b.itemSize;for(var d=0,e=this.itemSize;d<e;d++)this.array[a+d]=b.array[c+d];return this},copyArray:function(a){this.array.set(a);return this},copyColorsArray:function(a){for(var b=this.array,c=0,d=0,e=a.length;d<e;d++){var f=a[d];void 0===f&&(console.warn(&quot;THREE.BufferAttribute.copyColorsArray(): color is undefined&quot;,d),f=new A);b[c++]=f.r;b[c++]=f.g;b[c++]=f.b}return this},copyVector2sArray:function(a){for(var b=this.array,c=0,d=0,e=
a.length;d<e;d++){var f=a[d];void 0===f&&(console.warn(&quot;THREE.BufferAttribute.copyVector2sArray(): vector is undefined&quot;,d),f=new t);b[c++]=f.x;b[c++]=f.y}return this},copyVector3sArray:function(a){for(var b=this.array,c=0,d=0,e=a.length;d<e;d++){var f=a[d];void 0===f&&(console.warn(&quot;THREE.BufferAttribute.copyVector3sArray(): vector is undefined&quot;,d),f=new n);b[c++]=f.x;b[c++]=f.y;b[c++]=f.z}return this},copyVector4sArray:function(a){for(var b=this.array,c=0,d=0,e=a.length;d<e;d++){var f=a[d];void 0===
f&&(console.warn(&quot;THREE.BufferAttribute.copyVector4sArray(): vector is undefined&quot;,d),f=new ka);b[c++]=f.x;b[c++]=f.y;b[c++]=f.z;b[c++]=f.w}return this},applyMatrix3:function(a){for(var b=0,c=this.count;b<c;b++)ba.x=this.getX(b),ba.y=this.getY(b),ba.z=this.getZ(b),ba.applyMatrix3(a),this.setXYZ(b,ba.x,ba.y,ba.z);return this},applyMatrix4:function(a){for(var b=0,c=this.count;b<c;b++)ba.x=this.getX(b),ba.y=this.getY(b),ba.z=this.getZ(b),ba.applyMatrix4(a),this.setXYZ(b,ba.x,ba.y,ba.z);return this},applyNormalMatrix:function(a){for(var b=
0,c=this.count;b<c;b++)ba.x=this.getX(b),ba.y=this.getY(b),ba.z=this.getZ(b),ba.applyNormalMatrix(a),this.setXYZ(b,ba.x,ba.y,ba.z);return this},transformDirection:function(a){for(var b=0,c=this.count;b<c;b++)ba.x=this.getX(b),ba.y=this.getY(b),ba.z=this.getZ(b),ba.transformDirection(a),this.setXYZ(b,ba.x,ba.y,ba.z);return this},set:function(a,b){void 0===b&&(b=0);this.array.set(a,b);return this},getX:function(a){return this.array[a*this.itemSize]},setX:function(a,b){this.array[a*this.itemSize]=b;
return this},getY:function(a){return this.array[a*this.itemSize+1]},setY:function(a,b){this.array[a*this.itemSize+1]=b;return this},getZ:function(a){return this.array[a*this.itemSize+2]},setZ:function(a,b){this.array[a*this.itemSize+2]=b;return this},getW:function(a){return this.array[a*this.itemSize+3]},setW:function(a,b){this.array[a*this.itemSize+3]=b;return this},setXY:function(a,b,c){a*=this.itemSize;this.array[a+0]=b;this.array[a+1]=c;return this},setXYZ:function(a,b,c,d){a*=this.itemSize;this.array[a+
0]=b;this.array[a+1]=c;this.array[a+2]=d;return this},setXYZW:function(a,b,c,d,e){a*=this.itemSize;this.array[a+0]=b;this.array[a+1]=c;this.array[a+2]=d;this.array[a+3]=e;return this},onUpload:function(a){this.onUploadCallback=a;return this},clone:function(){return(new this.constructor(this.array,this.itemSize)).copy(this)},toJSON:function(){return{itemSize:this.itemSize,type:this.array.constructor.name,array:Array.prototype.slice.call(this.array),normalized:this.normalized}}});Bd.prototype=Object.create(M.prototype);
Bd.prototype.constructor=Bd;Cd.prototype=Object.create(M.prototype);Cd.prototype.constructor=Cd;Dd.prototype=Object.create(M.prototype);Dd.prototype.constructor=Dd;Ed.prototype=Object.create(M.prototype);Ed.prototype.constructor=Ed;Wb.prototype=Object.create(M.prototype);Wb.prototype.constructor=Wb;Fd.prototype=Object.create(M.prototype);Fd.prototype.constructor=Fd;Xb.prototype=Object.create(M.prototype);Xb.prototype.constructor=Xb;y.prototype=Object.create(M.prototype);y.prototype.constructor=y;
Gd.prototype=Object.create(M.prototype);Gd.prototype.constructor=Gd;Object.assign(xh.prototype,{computeGroups:function(a){var b=[],c=void 0;a=a.faces;for(var d=0;d<a.length;d++){var e=a[d];if(e.materialIndex!==c){c=e.materialIndex;void 0!==f&&(f.count=3*d-f.start,b.push(f));var f={start:3*d,materialIndex:c}}}void 0!==f&&(f.count=3*d-f.start,b.push(f));this.groups=b},fromGeometry:function(a){var b=a.faces,c=a.vertices,d=a.faceVertexUvs,e=d[0]&&0<d[0].length,f=d[1]&&0<d[1].length,g=a.morphTargets,h=
g.length;if(0<h){var l=[];for(var m=0;m<h;m++)l[m]={name:g[m].name,data:[]};this.morphTargets.position=l}var k=a.morphNormals,p=k.length;if(0<p){var n=[];for(m=0;m<p;m++)n[m]={name:k[m].name,data:[]};this.morphTargets.normal=n}var r=a.skinIndices,q=a.skinWeights,v=r.length===c.length,E=q.length===c.length;0<c.length&&0===b.length&&console.error(&quot;THREE.DirectGeometry: Faceless geometries are not supported.&quot;);for(m=0;m<b.length;m++){var w=b[m];this.vertices.push(c[w.a],c[w.b],c[w.c]);var z=w.vertexNormals;
3===z.length?this.normals.push(z[0],z[1],z[2]):(z=w.normal,this.normals.push(z,z,z));z=w.vertexColors;3===z.length?this.colors.push(z[0],z[1],z[2]):(z=w.color,this.colors.push(z,z,z));!0===e&&(z=d[0][m],void 0!==z?this.uvs.push(z[0],z[1],z[2]):(console.warn(&quot;THREE.DirectGeometry.fromGeometry(): Undefined vertexUv &quot;,m),this.uvs.push(new t,new t,new t)));!0===f&&(z=d[1][m],void 0!==z?this.uvs2.push(z[0],z[1],z[2]):(console.warn(&quot;THREE.DirectGeometry.fromGeometry(): Undefined vertexUv2 &quot;,m),this.uvs2.push(new t,
new t,new t)));for(z=0;z<h;z++){var y=g[z].vertices;l[z].data.push(y[w.a],y[w.b],y[w.c])}for(z=0;z<p;z++)y=k[z].vertexNormals[m],n[z].data.push(y.a,y.b,y.c);v&&this.skinIndices.push(r[w.a],r[w.b],r[w.c]);E&&this.skinWeights.push(q[w.a],q[w.b],q[w.c])}this.computeGroups(a);this.verticesNeedUpdate=a.verticesNeedUpdate;this.normalsNeedUpdate=a.normalsNeedUpdate;this.colorsNeedUpdate=a.colorsNeedUpdate;this.uvsNeedUpdate=a.uvsNeedUpdate;this.groupsNeedUpdate=a.groupsNeedUpdate;null!==a.boundingSphere&&
(this.boundingSphere=a.boundingSphere.clone());null!==a.boundingBox&&(this.boundingBox=a.boundingBox.clone());return this}});var sj=1,mb=new P,kh=new F,vd=new n,Na=new Sa,Be=new Sa,ea=new n;C.prototype=Object.assign(Object.create(Ea.prototype),{constructor:C,isBufferGeometry:!0,getIndex:function(){return this.index},setIndex:function(a){Array.isArray(a)?this.index=new (65535<yh(a)?Xb:Wb)(a,1):this.index=a},getAttribute:function(a){return this.attributes[a]},setAttribute:function(a,b){this.attributes[a]=
b;return this},deleteAttribute:function(a){delete this.attributes[a];return this},addGroup:function(a,b,c){this.groups.push({start:a,count:b,materialIndex:void 0!==c?c:0})},clearGroups:function(){this.groups=[]},setDrawRange:function(a,b){this.drawRange.start=a;this.drawRange.count=b},applyMatrix4:function(a){var b=this.attributes.position;void 0!==b&&(b.applyMatrix4(a),b.needsUpdate=!0);b=this.attributes.normal;if(void 0!==b){var c=(new wa).getNormalMatrix(a);b.applyNormalMatrix(c);b.needsUpdate=
!0}b=this.attributes.tangent;void 0!==b&&(b.transformDirection(a),b.needsUpdate=!0);null!==this.boundingBox&&this.computeBoundingBox();null!==this.boundingSphere&&this.computeBoundingSphere();return this},rotateX:function(a){mb.makeRotationX(a);this.applyMatrix4(mb);return this},rotateY:function(a){mb.makeRotationY(a);this.applyMatrix4(mb);return this},rotateZ:function(a){mb.makeRotationZ(a);this.applyMatrix4(mb);return this},translate:function(a,b,c){mb.makeTranslation(a,b,c);this.applyMatrix4(mb);
return this},scale:function(a,b,c){mb.makeScale(a,b,c);this.applyMatrix4(mb);return this},lookAt:function(a){kh.lookAt(a);kh.updateMatrix();this.applyMatrix4(kh.matrix);return this},center:function(){this.computeBoundingBox();this.boundingBox.getCenter(vd).negate();this.translate(vd.x,vd.y,vd.z);return this},setFromObject:function(a){var b=a.geometry;if(a.isPoints||a.isLine){a=new y(3*b.vertices.length,3);var c=new y(3*b.colors.length,3);this.setAttribute(&quot;position&quot;,a.copyVector3sArray(b.vertices));
this.setAttribute(&quot;color&quot;,c.copyColorsArray(b.colors));b.lineDistances&&b.lineDistances.length===b.vertices.length&&(a=new y(b.lineDistances.length,1),this.setAttribute(&quot;lineDistance&quot;,a.copyArray(b.lineDistances)));null!==b.boundingSphere&&(this.boundingSphere=b.boundingSphere.clone());null!==b.boundingBox&&(this.boundingBox=b.boundingBox.clone())}else a.isMesh&&b&&b.isGeometry&&this.fromGeometry(b);return this},setFromPoints:function(a){for(var b=[],c=0,d=a.length;c<d;c++){var e=a[c];b.push(e.x,
e.y,e.z||0)}this.setAttribute(&quot;position&quot;,new y(b,3));return this},updateFromObject:function(a){var b=a.geometry;if(a.isMesh){var c=b.__directGeometry;!0===b.elementsNeedUpdate&&(c=void 0,b.elementsNeedUpdate=!1);if(void 0===c)return this.fromGeometry(b);c.verticesNeedUpdate=b.verticesNeedUpdate;c.normalsNeedUpdate=b.normalsNeedUpdate;c.colorsNeedUpdate=b.colorsNeedUpdate;c.uvsNeedUpdate=b.uvsNeedUpdate;c.groupsNeedUpdate=b.groupsNeedUpdate;b.verticesNeedUpdate=!1;b.normalsNeedUpdate=!1;b.colorsNeedUpdate=
!1;b.uvsNeedUpdate=!1;b.groupsNeedUpdate=!1;b=c}!0===b.verticesNeedUpdate&&(c=this.attributes.position,void 0!==c&&(c.copyVector3sArray(b.vertices),c.needsUpdate=!0),b.verticesNeedUpdate=!1);!0===b.normalsNeedUpdate&&(c=this.attributes.normal,void 0!==c&&(c.copyVector3sArray(b.normals),c.needsUpdate=!0),b.normalsNeedUpdate=!1);!0===b.colorsNeedUpdate&&(c=this.attributes.color,void 0!==c&&(c.copyColorsArray(b.colors),c.needsUpdate=!0),b.colorsNeedUpdate=!1);b.uvsNeedUpdate&&(c=this.attributes.uv,void 0!==
c&&(c.copyVector2sArray(b.uvs),c.needsUpdate=!0),b.uvsNeedUpdate=!1);b.lineDistancesNeedUpdate&&(c=this.attributes.lineDistance,void 0!==c&&(c.copyArray(b.lineDistances),c.needsUpdate=!0),b.lineDistancesNeedUpdate=!1);b.groupsNeedUpdate&&(b.computeGroups(a.geometry),this.groups=b.groups,b.groupsNeedUpdate=!1);return this},fromGeometry:function(a){a.__directGeometry=(new xh).fromGeometry(a);return this.fromDirectGeometry(a.__directGeometry)},fromDirectGeometry:function(a){var b=new Float32Array(3*
a.vertices.length);this.setAttribute(&quot;position&quot;,(new M(b,3)).copyVector3sArray(a.vertices));0<a.normals.length&&(b=new Float32Array(3*a.normals.length),this.setAttribute(&quot;normal&quot;,(new M(b,3)).copyVector3sArray(a.normals)));0<a.colors.length&&(b=new Float32Array(3*a.colors.length),this.setAttribute(&quot;color&quot;,(new M(b,3)).copyColorsArray(a.colors)));0<a.uvs.length&&(b=new Float32Array(2*a.uvs.length),this.setAttribute(&quot;uv&quot;,(new M(b,2)).copyVector2sArray(a.uvs)));0<a.uvs2.length&&(b=new Float32Array(2*
a.uvs2.length),this.setAttribute(&quot;uv2&quot;,(new M(b,2)).copyVector2sArray(a.uvs2)));this.groups=a.groups;for(var c in a.morphTargets){b=[];for(var d=a.morphTargets[c],e=0,f=d.length;e<f;e++){var g=d[e],h=new y(3*g.data.length,3);h.name=g.name;b.push(h.copyVector3sArray(g.data))}this.morphAttributes[c]=b}0<a.skinIndices.length&&(c=new y(4*a.skinIndices.length,4),this.setAttribute(&quot;skinIndex&quot;,c.copyVector4sArray(a.skinIndices)));0<a.skinWeights.length&&(c=new y(4*a.skinWeights.length,4),this.setAttribute(&quot;skinWeight&quot;,
c.copyVector4sArray(a.skinWeights)));null!==a.boundingSphere&&(this.boundingSphere=a.boundingSphere.clone());null!==a.boundingBox&&(this.boundingBox=a.boundingBox.clone());return this},computeBoundingBox:function(){null===this.boundingBox&&(this.boundingBox=new Sa);var a=this.attributes.position,b=this.morphAttributes.position;if(void 0!==a){if(this.boundingBox.setFromBufferAttribute(a),b){a=0;for(var c=b.length;a<c;a++)Na.setFromBufferAttribute(b[a]),this.morphTargetsRelative?(ea.addVectors(this.boundingBox.min,
Na.min),this.boundingBox.expandByPoint(ea),ea.addVectors(this.boundingBox.max,Na.max),this.boundingBox.expandByPoint(ea)):(this.boundingBox.expandByPoint(Na.min),this.boundingBox.expandByPoint(Na.max))}}else this.boundingBox.makeEmpty();(isNaN(this.boundingBox.min.x)||isNaN(this.boundingBox.min.y)||isNaN(this.boundingBox.min.z))&&console.error('THREE.BufferGeometry.computeBoundingBox: Computed min/max have NaN values. The &quot;position&quot; attribute is likely to have NaN values.',this)},computeBoundingSphere:function(){null===
this.boundingSphere&&(this.boundingSphere=new pb);var a=this.attributes.position,b=this.morphAttributes.position;if(a){var c=this.boundingSphere.center;Na.setFromBufferAttribute(a);if(b)for(var d=0,e=b.length;d<e;d++){var f=b[d];Be.setFromBufferAttribute(f);this.morphTargetsRelative?(ea.addVectors(Na.min,Be.min),Na.expandByPoint(ea),ea.addVectors(Na.max,Be.max),Na.expandByPoint(ea)):(Na.expandByPoint(Be.min),Na.expandByPoint(Be.max))}Na.getCenter(c);var g=0;d=0;for(e=a.count;d<e;d++)ea.fromBufferAttribute(a,
d),g=Math.max(g,c.distanceToSquared(ea));if(b)for(d=0,e=b.length;d<e;d++){f=b[d];for(var h=this.morphTargetsRelative,l=0,m=f.count;l<m;l++)ea.fromBufferAttribute(f,l),h&&(vd.fromBufferAttribute(a,l),ea.add(vd)),g=Math.max(g,c.distanceToSquared(ea))}this.boundingSphere.radius=Math.sqrt(g);isNaN(this.boundingSphere.radius)&&console.error('THREE.BufferGeometry.computeBoundingSphere(): Computed radius is NaN. The &quot;position&quot; attribute is likely to have NaN values.',this)}},computeFaceNormals:function(){},
computeVertexNormals:function(){var a=this.index,b=this.attributes;if(b.position){var c=b.position.array;if(void 0===b.normal)this.setAttribute(&quot;normal&quot;,new M(new Float32Array(c.length),3));else for(var d=b.normal.array,e=0,f=d.length;e<f;e++)d[e]=0;d=b.normal.array;var g=new n,h=new n,l=new n,m=new n,k=new n;if(a){var p=a.array;e=0;for(f=a.count;e<f;e+=3){a=3*p[e+0];var x=3*p[e+1];var r=3*p[e+2];g.fromArray(c,a);h.fromArray(c,x);l.fromArray(c,r);m.subVectors(l,h);k.subVectors(g,h);m.cross(k);d[a]+=
m.x;d[a+1]+=m.y;d[a+2]+=m.z;d[x]+=m.x;d[x+1]+=m.y;d[x+2]+=m.z;d[r]+=m.x;d[r+1]+=m.y;d[r+2]+=m.z}}else for(e=0,f=c.length;e<f;e+=9)g.fromArray(c,e),h.fromArray(c,e+3),l.fromArray(c,e+6),m.subVectors(l,h),k.subVectors(g,h),m.cross(k),d[e]=m.x,d[e+1]=m.y,d[e+2]=m.z,d[e+3]=m.x,d[e+4]=m.y,d[e+5]=m.z,d[e+6]=m.x,d[e+7]=m.y,d[e+8]=m.z;this.normalizeNormals();b.normal.needsUpdate=!0}},merge:function(a,b){if(a&&a.isBufferGeometry){void 0===b&&(b=0,console.warn(&quot;THREE.BufferGeometry.merge(): Overwriting original geometry, starting at offset=0. Use BufferGeometryUtils.mergeBufferGeometries() for lossless merge.&quot;));
var c=this.attributes,d;for(d in c)if(void 0!==a.attributes[d]){var e=c[d].array,f=a.attributes[d],g=f.array,h=f.itemSize*b;f=Math.min(g.length,e.length-h);for(var l=0;l<f;l++,h++)e[h]=g[l]}return this}console.error(&quot;THREE.BufferGeometry.merge(): geometry not an instance of THREE.BufferGeometry.&quot;,a)},normalizeNormals:function(){for(var a=this.attributes.normal,b=0,c=a.count;b<c;b++)ea.x=a.getX(b),ea.y=a.getY(b),ea.z=a.getZ(b),ea.normalize(),a.setXYZ(b,ea.x,ea.y,ea.z)},toNonIndexed:function(){function a(a,
b){var c=a.array;a=a.itemSize;for(var d=new c.constructor(b.length*a),e,f=0,g=0,h=b.length;g<h;g++){e=b[g]*a;for(var l=0;l<a;l++)d[f++]=c[e++]}return new M(d,a)}if(null===this.index)return console.warn(&quot;THREE.BufferGeometry.toNonIndexed(): Geometry is already non-indexed.&quot;),this;var b=new C,c=this.index.array,d=this.attributes,e;for(e in d){var f=d[e];f=a(f,c);b.setAttribute(e,f)}var g=this.morphAttributes;for(e in g){var h=[],l=g[e];d=0;for(var m=l.length;d<m;d++)f=l[d],f=a(f,c),h.push(f);b.morphAttributes[e]=
h}b.morphTargetsRelative=this.morphTargetsRelative;c=this.groups;d=0;for(e=c.length;d<e;d++)f=c[d],b.addGroup(f.start,f.count,f.materialIndex);return b},toJSON:function(){var a={metadata:{version:4.5,type:&quot;BufferGeometry&quot;,generator:&quot;BufferGeometry.toJSON&quot;}};a.uuid=this.uuid;a.type=this.type;&quot;&quot;!==this.name&&(a.name=this.name);0<Object.keys(this.userData).length&&(a.userData=this.userData);if(void 0!==this.parameters){var b=this.parameters;for(m in b)void 0!==b[m]&&(a[m]=b[m]);return a}a.data={attributes:{}};
b=this.index;null!==b&&(a.data.index={type:b.array.constructor.name,array:Array.prototype.slice.call(b.array)});var c=this.attributes;for(m in c){b=c[m];var d=b.toJSON();&quot;&quot;!==b.name&&(d.name=b.name);a.data.attributes[m]=d}c={};var e=!1;for(m in this.morphAttributes){for(var f=this.morphAttributes[m],g=[],h=0,l=f.length;h<l;h++)b=f[h],d=b.toJSON(),&quot;&quot;!==b.name&&(d.name=b.name),g.push(d);0<g.length&&(c[m]=g,e=!0)}e&&(a.data.morphAttributes=c,a.data.morphTargetsRelative=this.morphTargetsRelative);var m=
this.groups;0<m.length&&(a.data.groups=JSON.parse(JSON.stringify(m)));m=this.boundingSphere;null!==m&&(a.data.boundingSphere={center:m.center.toArray(),radius:m.radius});return a},clone:function(){return(new C).copy(this)},copy:function(a){var b;this.index=null;this.attributes={};this.morphAttributes={};this.groups=[];this.boundingSphere=this.boundingBox=null;this.name=a.name;var c=a.index;null!==c&&this.setIndex(c.clone());c=a.attributes;for(g in c)this.setAttribute(g,c[g].clone());var d=a.morphAttributes;
for(g in d){var e=[],f=d[g];c=0;for(b=f.length;c<b;c++)e.push(f[c].clone());this.morphAttributes[g]=e}this.morphTargetsRelative=a.morphTargetsRelative;var g=a.groups;c=0;for(b=g.length;c<b;c++)d=g[c],this.addGroup(d.start,d.count,d.materialIndex);g=a.boundingBox;null!==g&&(this.boundingBox=g.clone());g=a.boundingSphere;null!==g&&(this.boundingSphere=g.clone());this.drawRange.start=a.drawRange.start;this.drawRange.count=a.drawRange.count;this.userData=a.userData;return this},dispose:function(){this.dispatchEvent({type:&quot;dispose&quot;})}});
var Qi=new P,tc=new Vb,lh=new pb,Yb=new n,Zb=new n,$b=new n,dg=new n,eg=new n,fg=new n,Ke=new n,Le=new n,Me=new n,Cc=new t,Dc=new t,Ec=new t,Hd=new n,Ie=new n;S.prototype=Object.assign(Object.create(F.prototype),{constructor:S,isMesh:!0,copy:function(a){F.prototype.copy.call(this,a);void 0!==a.morphTargetInfluences&&(this.morphTargetInfluences=a.morphTargetInfluences.slice());void 0!==a.morphTargetDictionary&&(this.morphTargetDictionary=Object.assign({},a.morphTargetDictionary));return this},updateMorphTargets:function(){var a=
this.geometry;if(a.isBufferGeometry){a=a.morphAttributes;var b=Object.keys(a);if(0<b.length){var c=a[b[0]];if(void 0!==c)for(this.morphTargetInfluences=[],this.morphTargetDictionary={},a=0,b=c.length;a<b;a++){var d=c[a].name||String(a);this.morphTargetInfluences.push(0);this.morphTargetDictionary[d]=a}}}else a=a.morphTargets,void 0!==a&&0<a.length&&console.error(&quot;THREE.Mesh.updateMorphTargets() no longer supports THREE.Geometry. Use THREE.BufferGeometry instead.&quot;)},raycast:function(a,b){var c=this.geometry,
d=this.material,e=this.matrixWorld;if(void 0!==d&&(null===c.boundingSphere&&c.computeBoundingSphere(),lh.copy(c.boundingSphere),lh.applyMatrix4(e),!1!==a.ray.intersectsSphere(lh)&&(Qi.getInverse(e),tc.copy(a.ray).applyMatrix4(Qi),null===c.boundingBox||!1!==tc.intersectsBox(c.boundingBox))))if(c.isBufferGeometry){var f=c.index;e=c.attributes.position;var g=c.morphAttributes.position,h=c.morphTargetsRelative,l=c.attributes.uv,m=c.attributes.uv2,k=c.groups,p=c.drawRange,n,r;if(null!==f)if(Array.isArray(d)){var q=
0;for(n=k.length;q<n;q++){var v=k[q];var E=d[v.materialIndex];var w=Math.max(v.start,p.start);for(r=c=Math.min(v.start+v.count,p.start+p.count);w<r;w+=3){c=f.getX(w);var z=f.getX(w+1);var y=f.getX(w+2);if(c=Je(this,E,a,tc,e,g,h,l,m,c,z,y))c.faceIndex=Math.floor(w/3),c.face.materialIndex=v.materialIndex,b.push(c)}}}else for(w=Math.max(0,p.start),c=Math.min(f.count,p.start+p.count),q=w,n=c;q<n;q+=3){if(c=f.getX(q),z=f.getX(q+1),y=f.getX(q+2),c=Je(this,d,a,tc,e,g,h,l,m,c,z,y))c.faceIndex=Math.floor(q/
3),b.push(c)}else if(void 0!==e)if(Array.isArray(d))for(q=0,n=k.length;q<n;q++)for(v=k[q],E=d[v.materialIndex],w=Math.max(v.start,p.start),r=c=Math.min(v.start+v.count,p.start+p.count);w<r;w+=3){if(c=w,z=w+1,y=w+2,c=Je(this,E,a,tc,e,g,h,l,m,c,z,y))c.faceIndex=Math.floor(w/3),c.face.materialIndex=v.materialIndex,b.push(c)}else for(w=Math.max(0,p.start),c=Math.min(e.count,p.start+p.count),q=w,n=c;q<n;q+=3)if(c=q,z=q+1,y=q+2,c=Je(this,d,a,tc,e,g,h,l,m,c,z,y))c.faceIndex=Math.floor(q/3),b.push(c)}else if(c.isGeometry)for(e=
Array.isArray(d),g=c.vertices,h=c.faces,c=c.faceVertexUvs[0],0<c.length&&(f=c),p=0,q=h.length;p<q;p++)if(n=h[p],c=e?d[n.materialIndex]:d,void 0!==c&&(l=g[n.a],m=g[n.b],k=g[n.c],c=zh(this,c,a,tc,l,m,k,Hd)))f&&f[p]&&(v=f[p],Cc.copy(v[0]),Dc.copy(v[1]),Ec.copy(v[2]),c.uv=oa.getUV(Hd,l,m,k,Cc,Dc,Ec,new t)),c.face=n,c.faceIndex=p,b.push(c)},clone:function(){return(new this.constructor(this.geometry,this.material)).copy(this)}});var tj=0,nb=new P,mh=new F,Kf=new n;N.prototype=Object.assign(Object.create(Ea.prototype),
{constructor:N,isGeometry:!0,applyMatrix4:function(a){for(var b=(new wa).getNormalMatrix(a),c=0,d=this.vertices.length;c<d;c++)this.vertices[c].applyMatrix4(a);c=0;for(d=this.faces.length;c<d;c++){a=this.faces[c];a.normal.applyMatrix3(b).normalize();for(var e=0,f=a.vertexNormals.length;e<f;e++)a.vertexNormals[e].applyMatrix3(b).normalize()}null!==this.boundingBox&&this.computeBoundingBox();null!==this.boundingSphere&&this.computeBoundingSphere();this.normalsNeedUpdate=this.verticesNeedUpdate=!0;return this},
rotateX:function(a){nb.makeRotationX(a);this.applyMatrix4(nb);return this},rotateY:function(a){nb.makeRotationY(a);this.applyMatrix4(nb);return this},rotateZ:function(a){nb.makeRotationZ(a);this.applyMatrix4(nb);return this},translate:function(a,b,c){nb.makeTranslation(a,b,c);this.applyMatrix4(nb);return this},scale:function(a,b,c){nb.makeScale(a,b,c);this.applyMatrix4(nb);return this},lookAt:function(a){mh.lookAt(a);mh.updateMatrix();this.applyMatrix4(mh.matrix);return this},fromBufferGeometry:function(a){function b(a,
b,d,e){var f=void 0===h?[]:[c.colors[a].clone(),c.colors[b].clone(),c.colors[d].clone()],k=void 0===g?[]:[(new n).fromArray(g,3*a),(new n).fromArray(g,3*b),(new n).fromArray(g,3*d)];e=new Bc(a,b,d,k,f,e);c.faces.push(e);void 0!==l&&c.faceVertexUvs[0].push([(new t).fromArray(l,2*a),(new t).fromArray(l,2*b),(new t).fromArray(l,2*d)]);void 0!==m&&c.faceVertexUvs[1].push([(new t).fromArray(m,2*a),(new t).fromArray(m,2*b),(new t).fromArray(m,2*d)])}var c=this,d=null!==a.index?a.index.array:void 0,e=a.attributes;
if(void 0===e.position)return console.error(&quot;THREE.Geometry.fromBufferGeometry(): Position attribute required for conversion.&quot;),this;var f=e.position.array,g=void 0!==e.normal?e.normal.array:void 0,h=void 0!==e.color?e.color.array:void 0,l=void 0!==e.uv?e.uv.array:void 0,m=void 0!==e.uv2?e.uv2.array:void 0;void 0!==m&&(this.faceVertexUvs[1]=[]);for(e=0;e<f.length;e+=3)c.vertices.push((new n).fromArray(f,e)),void 0!==h&&c.colors.push((new A).fromArray(h,e));var k=a.groups;if(0<k.length)for(e=0;e<k.length;e++){f=
k[e];var p=f.start,x=p;for(p+=f.count;x<p;x+=3)void 0!==d?b(d[x],d[x+1],d[x+2],f.materialIndex):b(x,x+1,x+2,f.materialIndex)}else if(void 0!==d)for(e=0;e<d.length;e+=3)b(d[e],d[e+1],d[e+2]);else for(e=0;e<f.length/3;e+=3)b(e,e+1,e+2);this.computeFaceNormals();null!==a.boundingBox&&(this.boundingBox=a.boundingBox.clone());null!==a.boundingSphere&&(this.boundingSphere=a.boundingSphere.clone());return this},center:function(){this.computeBoundingBox();this.boundingBox.getCenter(Kf).negate();this.translate(Kf.x,
Kf.y,Kf.z);return this},normalize:function(){this.computeBoundingSphere();var a=this.boundingSphere.center,b=this.boundingSphere.radius;b=0===b?1:1/b;var c=new P;c.set(b,0,0,-b*a.x,0,b,0,-b*a.y,0,0,b,-b*a.z,0,0,0,1);this.applyMatrix4(c);return this},computeFaceNormals:function(){for(var a=new n,b=new n,c=0,d=this.faces.length;c<d;c++){var e=this.faces[c],f=this.vertices[e.a],g=this.vertices[e.b];a.subVectors(this.vertices[e.c],g);b.subVectors(f,g);a.cross(b);a.normalize();e.normal.copy(a)}},computeVertexNormals:function(a){void 0===
a&&(a=!0);var b;var c=Array(this.vertices.length);var d=0;for(b=this.vertices.length;d<b;d++)c[d]=new n;if(a){var e=new n,f=new n;a=0;for(d=this.faces.length;a<d;a++){b=this.faces[a];var g=this.vertices[b.a];var h=this.vertices[b.b];var l=this.vertices[b.c];e.subVectors(l,h);f.subVectors(g,h);e.cross(f);c[b.a].add(e);c[b.b].add(e);c[b.c].add(e)}}else for(this.computeFaceNormals(),a=0,d=this.faces.length;a<d;a++)b=this.faces[a],c[b.a].add(b.normal),c[b.b].add(b.normal),c[b.c].add(b.normal);d=0;for(b=
this.vertices.length;d<b;d++)c[d].normalize();a=0;for(d=this.faces.length;a<d;a++)b=this.faces[a],g=b.vertexNormals,3===g.length?(g[0].copy(c[b.a]),g[1].copy(c[b.b]),g[2].copy(c[b.c])):(g[0]=c[b.a].clone(),g[1]=c[b.b].clone(),g[2]=c[b.c].clone());0<this.faces.length&&(this.normalsNeedUpdate=!0)},computeFlatVertexNormals:function(){var a;this.computeFaceNormals();var b=0;for(a=this.faces.length;b<a;b++){var c=this.faces[b];var d=c.vertexNormals;3===d.length?(d[0].copy(c.normal),d[1].copy(c.normal),
d[2].copy(c.normal)):(d[0]=c.normal.clone(),d[1]=c.normal.clone(),d[2]=c.normal.clone())}0<this.faces.length&&(this.normalsNeedUpdate=!0)},computeMorphNormals:function(){var a,b;var c=0;for(b=this.faces.length;c<b;c++){var d=this.faces[c];d.__originalFaceNormal?d.__originalFaceNormal.copy(d.normal):d.__originalFaceNormal=d.normal.clone();d.__originalVertexNormals||(d.__originalVertexNormals=[]);var e=0;for(a=d.vertexNormals.length;e<a;e++)d.__originalVertexNormals[e]?d.__originalVertexNormals[e].copy(d.vertexNormals[e]):
d.__originalVertexNormals[e]=d.vertexNormals[e].clone()}var f=new N;f.faces=this.faces;e=0;for(a=this.morphTargets.length;e<a;e++){if(!this.morphNormals[e]){this.morphNormals[e]={};this.morphNormals[e].faceNormals=[];this.morphNormals[e].vertexNormals=[];d=this.morphNormals[e].faceNormals;var g=this.morphNormals[e].vertexNormals;c=0;for(b=this.faces.length;c<b;c++){var h=new n;var l={a:new n,b:new n,c:new n};d.push(h);g.push(l)}}g=this.morphNormals[e];f.vertices=this.morphTargets[e].vertices;f.computeFaceNormals();
f.computeVertexNormals();c=0;for(b=this.faces.length;c<b;c++)d=this.faces[c],h=g.faceNormals[c],l=g.vertexNormals[c],h.copy(d.normal),l.a.copy(d.vertexNormals[0]),l.b.copy(d.vertexNormals[1]),l.c.copy(d.vertexNormals[2])}c=0;for(b=this.faces.length;c<b;c++)d=this.faces[c],d.normal=d.__originalFaceNormal,d.vertexNormals=d.__originalVertexNormals},computeBoundingBox:function(){null===this.boundingBox&&(this.boundingBox=new Sa);this.boundingBox.setFromPoints(this.vertices)},computeBoundingSphere:function(){null===
this.boundingSphere&&(this.boundingSphere=new pb);this.boundingSphere.setFromPoints(this.vertices)},merge:function(a,b,c){if(a&&a.isGeometry){var d,e=this.vertices.length,f=this.vertices,g=a.vertices,h=this.faces,l=a.faces,m=this.colors,k=a.colors;void 0===c&&(c=0);void 0!==b&&(d=(new wa).getNormalMatrix(b));for(var p=0,n=g.length;p<n;p++){var r=g[p].clone();void 0!==b&&r.applyMatrix4(b);f.push(r)}p=0;for(n=k.length;p<n;p++)m.push(k[p].clone());p=0;for(n=l.length;p<n;p++){g=l[p];var q=g.vertexNormals;
k=g.vertexColors;m=new Bc(g.a+e,g.b+e,g.c+e);m.normal.copy(g.normal);void 0!==d&&m.normal.applyMatrix3(d).normalize();b=0;for(f=q.length;b<f;b++)r=q[b].clone(),void 0!==d&&r.applyMatrix3(d).normalize(),m.vertexNormals.push(r);m.color.copy(g.color);b=0;for(f=k.length;b<f;b++)r=k[b],m.vertexColors.push(r.clone());m.materialIndex=g.materialIndex+c;h.push(m)}p=0;for(n=a.faceVertexUvs.length;p<n;p++)for(c=a.faceVertexUvs[p],void 0===this.faceVertexUvs[p]&&(this.faceVertexUvs[p]=[]),b=0,f=c.length;b<f;b++){d=
c[b];e=[];h=0;for(l=d.length;h<l;h++)e.push(d[h].clone());this.faceVertexUvs[p].push(e)}}else console.error(&quot;THREE.Geometry.merge(): geometry not an instance of THREE.Geometry.&quot;,a)},mergeMesh:function(a){a&&a.isMesh?(a.matrixAutoUpdate&&a.updateMatrix(),this.merge(a.geometry,a.matrix)):console.error(&quot;THREE.Geometry.mergeMesh(): mesh not an instance of THREE.Mesh.&quot;,a)},mergeVertices:function(){var a={},b=[],c=[],d=Math.pow(10,4),e;var f=0;for(e=this.vertices.length;f<e;f++){var g=this.vertices[f];
g=Math.round(g.x*d)+&quot;_&quot;+Math.round(g.y*d)+&quot;_&quot;+Math.round(g.z*d);void 0===a[g]?(a[g]=f,b.push(this.vertices[f]),c[f]=b.length-1):c[f]=c[a[g]]}a=[];f=0;for(e=this.faces.length;f<e;f++)for(d=this.faces[f],d.a=c[d.a],d.b=c[d.b],d.c=c[d.c],d=[d.a,d.b,d.c],g=0;3>g;g++)if(d[g]===d[(g+1)%3]){a.push(f);break}for(f=a.length-1;0<=f;f--)for(d=a[f],this.faces.splice(d,1),c=0,e=this.faceVertexUvs.length;c<e;c++)this.faceVertexUvs[c].splice(d,1);f=this.vertices.length-b.length;this.vertices=b;return f},setFromPoints:function(a){this.vertices=
[];for(var b=0,c=a.length;b<c;b++){var d=a[b];this.vertices.push(new n(d.x,d.y,d.z||0))}return this},sortFacesByMaterialIndex:function(){for(var a=this.faces,b=a.length,c=0;c<b;c++)a[c]._id=c;a.sort(function(a,b){return a.materialIndex-b.materialIndex});var d=this.faceVertexUvs[0],e=this.faceVertexUvs[1],f,g;d&&d.length===b&&(f=[]);e&&e.length===b&&(g=[]);for(c=0;c<b;c++){var h=a[c]._id;f&&f.push(d[h]);g&&g.push(e[h])}f&&(this.faceVertexUvs[0]=f);g&&(this.faceVertexUvs[1]=g)},toJSON:function(){function a(a,
b,c){return c?a|1<<b:a&~(1<<b)}function b(a){var b=a.x.toString()+a.y.toString()+a.z.toString();if(void 0!==m[b])return m[b];m[b]=l.length/3;l.push(a.x,a.y,a.z);return m[b]}function c(a){var b=a.r.toString()+a.g.toString()+a.b.toString();if(void 0!==p[b])return p[b];p[b]=k.length;k.push(a.getHex());return p[b]}function d(a){var b=a.x.toString()+a.y.toString();if(void 0!==r[b])return r[b];r[b]=n.length/2;n.push(a.x,a.y);return r[b]}var e={metadata:{version:4.5,type:&quot;Geometry&quot;,generator:&quot;Geometry.toJSON&quot;}};
e.uuid=this.uuid;e.type=this.type;&quot;&quot;!==this.name&&(e.name=this.name);if(void 0!==this.parameters){var f=this.parameters,g;for(g in f)void 0!==f[g]&&(e[g]=f[g]);return e}f=[];for(g=0;g<this.vertices.length;g++){var h=this.vertices[g];f.push(h.x,h.y,h.z)}h=[];var l=[],m={},k=[],p={},n=[],r={};for(g=0;g<this.faces.length;g++){var q=this.faces[g],v=void 0!==this.faceVertexUvs[0][g],t=0<q.normal.length(),w=0<q.vertexNormals.length,z=1!==q.color.r||1!==q.color.g||1!==q.color.b,y=0<q.vertexColors.length,
A=0;A=a(A,0,0);A=a(A,1,!0);A=a(A,2,!1);A=a(A,3,v);A=a(A,4,t);A=a(A,5,w);A=a(A,6,z);A=a(A,7,y);h.push(A);h.push(q.a,q.b,q.c);h.push(q.materialIndex);v&&(v=this.faceVertexUvs[0][g],h.push(d(v[0]),d(v[1]),d(v[2])));t&&h.push(b(q.normal));w&&(t=q.vertexNormals,h.push(b(t[0]),b(t[1]),b(t[2])));z&&h.push(c(q.color));y&&(q=q.vertexColors,h.push(c(q[0]),c(q[1]),c(q[2])))}e.data={};e.data.vertices=f;e.data.normals=l;0<k.length&&(e.data.colors=k);0<n.length&&(e.data.uvs=[n]);e.data.faces=h;return e},clone:function(){return(new N).copy(this)},
copy:function(a){var b,c,d;this.vertices=[];this.colors=[];this.faces=[];this.faceVertexUvs=[[]];this.morphTargets=[];this.morphNormals=[];this.skinWeights=[];this.skinIndices=[];this.lineDistances=[];this.boundingSphere=this.boundingBox=null;this.name=a.name;var e=a.vertices;var f=0;for(b=e.length;f<b;f++)this.vertices.push(e[f].clone());e=a.colors;f=0;for(b=e.length;f<b;f++)this.colors.push(e[f].clone());e=a.faces;f=0;for(b=e.length;f<b;f++)this.faces.push(e[f].clone());f=0;for(b=a.faceVertexUvs.length;f<
b;f++){var g=a.faceVertexUvs[f];void 0===this.faceVertexUvs[f]&&(this.faceVertexUvs[f]=[]);e=0;for(c=g.length;e<c;e++){var h=g[e],l=[];var m=0;for(d=h.length;m<d;m++)l.push(h[m].clone());this.faceVertexUvs[f].push(l)}}m=a.morphTargets;f=0;for(b=m.length;f<b;f++){d={};d.name=m[f].name;if(void 0!==m[f].vertices)for(d.vertices=[],e=0,c=m[f].vertices.length;e<c;e++)d.vertices.push(m[f].vertices[e].clone());if(void 0!==m[f].normals)for(d.normals=[],e=0,c=m[f].normals.length;e<c;e++)d.normals.push(m[f].normals[e].clone());
this.morphTargets.push(d)}m=a.morphNormals;f=0;for(b=m.length;f<b;f++){d={};if(void 0!==m[f].vertexNormals)for(d.vertexNormals=[],e=0,c=m[f].vertexNormals.length;e<c;e++)g=m[f].vertexNormals[e],h={},h.a=g.a.clone(),h.b=g.b.clone(),h.c=g.c.clone(),d.vertexNormals.push(h);if(void 0!==m[f].faceNormals)for(d.faceNormals=[],e=0,c=m[f].faceNormals.length;e<c;e++)d.faceNormals.push(m[f].faceNormals[e].clone());this.morphNormals.push(d)}e=a.skinWeights;f=0;for(b=e.length;f<b;f++)this.skinWeights.push(e[f].clone());
e=a.skinIndices;f=0;for(b=e.length;f<b;f++)this.skinIndices.push(e[f].clone());e=a.lineDistances;f=0;for(b=e.length;f<b;f++)this.lineDistances.push(e[f]);f=a.boundingBox;null!==f&&(this.boundingBox=f.clone());f=a.boundingSphere;null!==f&&(this.boundingSphere=f.clone());this.elementsNeedUpdate=a.elementsNeedUpdate;this.verticesNeedUpdate=a.verticesNeedUpdate;this.uvsNeedUpdate=a.uvsNeedUpdate;this.normalsNeedUpdate=a.normalsNeedUpdate;this.colorsNeedUpdate=a.colorsNeedUpdate;this.lineDistancesNeedUpdate=
a.lineDistancesNeedUpdate;this.groupsNeedUpdate=a.groupsNeedUpdate;return this},dispose:function(){this.dispatchEvent({type:&quot;dispose&quot;})}});var nh=function(a){function b(b,d,e,f,g,h){a.call(this);this.type=&quot;BoxGeometry&quot;;this.parameters={width:b,height:d,depth:e,widthSegments:f,heightSegments:g,depthSegments:h};this.fromBufferGeometry(new Jd(b,d,e,f,g,h));this.mergeVertices()}a&&(b.__proto__=a);b.prototype=Object.create(a&&a.prototype);return b.prototype.constructor=b}(N),Jd=function(a){function b(b,
d,e,f,g,h){function c(a,b,c,d,e,f,g,h,l,u,t){var w=f/l,z=g/u,y=f/2,E=g/2,B=h/2;g=l+1;var A=u+1,C=f=0,D,F,G=new n;for(F=0;F<A;F++){var K=F*z-E;for(D=0;D<g;D++)G[a]=(D*w-y)*d,G[b]=K*e,G[c]=B,p.push(G.x,G.y,G.z),G[a]=0,G[b]=0,G[c]=0<h?1:-1,x.push(G.x,G.y,G.z),r.push(D/l),r.push(1-F/u),f+=1}for(F=0;F<u;F++)for(D=0;D<l;D++)a=q+D+g*(F+1),b=q+(D+1)+g*(F+1),c=q+(D+1)+g*F,k.push(q+D+g*F,a,c),k.push(a,b,c),C+=6;m.addGroup(v,C,t);v+=C;q+=f}a.call(this);this.type=&quot;BoxBufferGeometry&quot;;this.parameters={width:b,
height:d,depth:e,widthSegments:f,heightSegments:g,depthSegments:h};var m=this;b=b||1;d=d||1;e=e||1;f=Math.floor(f)||1;g=Math.floor(g)||1;h=Math.floor(h)||1;var k=[],p=[],x=[],r=[],q=0,v=0;c(&quot;z&quot;,&quot;y&quot;,&quot;x&quot;,-1,-1,e,d,b,h,g,0);c(&quot;z&quot;,&quot;y&quot;,&quot;x&quot;,1,-1,e,d,-b,h,g,1);c(&quot;x&quot;,&quot;z&quot;,&quot;y&quot;,1,1,b,e,d,f,h,2);c(&quot;x&quot;,&quot;z&quot;,&quot;y&quot;,1,-1,b,e,-d,f,h,3);c(&quot;x&quot;,&quot;y&quot;,&quot;z&quot;,1,-1,b,d,e,f,g,4);c(&quot;x&quot;,&quot;y&quot;,&quot;z&quot;,-1,-1,b,d,-e,f,g,5);this.setIndex(k);this.setAttribute(&quot;position&quot;,new y(p,3));this.setAttribute(&quot;normal&quot;,new y(x,3));this.setAttribute(&quot;uv&quot;,
new y(r,2))}a&&(b.__proto__=a);b.prototype=Object.create(a&&a.prototype);return b.prototype.constructor=b}(C),Uh={clone:Fc,merge:va};Ba.prototype=Object.create(K.prototype);Ba.prototype.constructor=Ba;Ba.prototype.isShaderMaterial=!0;Ba.prototype.copy=function(a){K.prototype.copy.call(this,a);this.fragmentShader=a.fragmentShader;this.vertexShader=a.vertexShader;this.uniforms=Fc(a.uniforms);this.defines=Object.assign({},a.defines);this.wireframe=a.wireframe;this.wireframeLinewidth=a.wireframeLinewidth;
this.lights=a.lights;this.clipping=a.clipping;this.skinning=a.skinning;this.morphTargets=a.morphTargets;this.morphNormals=a.morphNormals;this.extensions=a.extensions;return this};Ba.prototype.toJSON=function(a){var b=K.prototype.toJSON.call(this,a);b.uniforms={};for(var c in this.uniforms){var d=this.uniforms[c].value;b.uniforms[c]=d&&d.isTexture?{type:&quot;t&quot;,value:d.toJSON(a).uuid}:d&&d.isColor?{type:&quot;c&quot;,value:d.getHex()}:d&&d.isVector2?{type:&quot;v2&quot;,value:d.toArray()}:d&&d.isVector3?{type:&quot;v3&quot;,value:d.toArray()}:
d&&d.isVector4?{type:&quot;v4&quot;,value:d.toArray()}:d&&d.isMatrix3?{type:&quot;m3&quot;,value:d.toArray()}:d&&d.isMatrix4?{type:&quot;m4&quot;,value:d.toArray()}:{value:d}}0<Object.keys(this.defines).length&&(b.defines=this.defines);b.vertexShader=this.vertexShader;b.fragmentShader=this.fragmentShader;a={};for(var e in this.extensions)!0===this.extensions[e]&&(a[e]=!0);0<Object.keys(a).length&&(b.extensions=a);return b};db.prototype=Object.assign(Object.create(F.prototype),{constructor:db,isCamera:!0,copy:function(a,b){F.prototype.copy.call(this,
a,b);this.matrixWorldInverse.copy(a.matrixWorldInverse);this.projectionMatrix.copy(a.projectionMatrix);this.projectionMatrixInverse.copy(a.projectionMatrixInverse);return this},getWorldDirection:function(a){void 0===a&&(console.warn(&quot;THREE.Camera: .getWorldDirection() target is now required&quot;),a=new n);this.updateMatrixWorld(!0);var b=this.matrixWorld.elements;return a.set(-b[8],-b[9],-b[10]).normalize()},updateMatrixWorld:function(a){F.prototype.updateMatrixWorld.call(this,a);this.matrixWorldInverse.getInverse(this.matrixWorld)},
updateWorldMatrix:function(a,b){F.prototype.updateWorldMatrix.call(this,a,b);this.matrixWorldInverse.getInverse(this.matrixWorld)},clone:function(){return(new this.constructor).copy(this)}});aa.prototype=Object.assign(Object.create(db.prototype),{constructor:aa,isPerspectiveCamera:!0,copy:function(a,b){db.prototype.copy.call(this,a,b);this.fov=a.fov;this.zoom=a.zoom;this.near=a.near;this.far=a.far;this.focus=a.focus;this.aspect=a.aspect;this.view=null===a.view?null:Object.assign({},a.view);this.filmGauge=
a.filmGauge;this.filmOffset=a.filmOffset;return this},setFocalLength:function(a){a=.5*this.getFilmHeight()/a;this.fov=2*L.RAD2DEG*Math.atan(a);this.updateProjectionMatrix()},getFocalLength:function(){var a=Math.tan(.5*L.DEG2RAD*this.fov);return.5*this.getFilmHeight()/a},getEffectiveFOV:function(){return 2*L.RAD2DEG*Math.atan(Math.tan(.5*L.DEG2RAD*this.fov)/this.zoom)},getFilmWidth:function(){return this.filmGauge*Math.min(this.aspect,1)},getFilmHeight:function(){return this.filmGauge/Math.max(this.aspect,
1)},setViewOffset:function(a,b,c,d,e,f){this.aspect=a/b;null===this.view&&(this.view={enabled:!0,fullWidth:1,fullHeight:1,offsetX:0,offsetY:0,width:1,height:1});this.view.enabled=!0;this.view.fullWidth=a;this.view.fullHeight=b;this.view.offsetX=c;this.view.offsetY=d;this.view.width=e;this.view.height=f;this.updateProjectionMatrix()},clearViewOffset:function(){null!==this.view&&(this.view.enabled=!1);this.updateProjectionMatrix()},updateProjectionMatrix:function(){var a=this.near,b=a*Math.tan(.5*L.DEG2RAD*
this.fov)/this.zoom,c=2*b,d=this.aspect*c,e=-.5*d,f=this.view;if(null!==this.view&&this.view.enabled){var g=f.fullWidth,h=f.fullHeight;e+=f.offsetX*d/g;b-=f.offsetY*c/h;d*=f.width/g;c*=f.height/h}f=this.filmOffset;0!==f&&(e+=a*f/this.getFilmWidth());this.projectionMatrix.makePerspective(e,e+d,b,b-c,a,this.far);this.projectionMatrixInverse.getInverse(this.projectionMatrix)},toJSON:function(a){a=F.prototype.toJSON.call(this,a);a.object.fov=this.fov;a.object.zoom=this.zoom;a.object.near=this.near;a.object.far=
this.far;a.object.focus=this.focus;a.object.aspect=this.aspect;null!==this.view&&(a.object.view=Object.assign({},this.view));a.object.filmGauge=this.filmGauge;a.object.filmOffset=this.filmOffset;return a}});Gc.prototype=Object.create(F.prototype);Gc.prototype.constructor=Gc;Db.prototype=Object.create(Ha.prototype);Db.prototype.constructor=Db;Db.prototype.isWebGLCubeRenderTarget=!0;Db.prototype.fromEquirectangularTexture=function(a,b){this.texture.type=b.type;this.texture.format=b.format;this.texture.encoding=
b.encoding;var c=new ob,d=new Ba({type:&quot;CubemapFromEquirect&quot;,uniforms:Fc({tEquirect:{value:null}}),vertexShader:&quot;varying vec3 vWorldDirection;\nvec3 transformDirection( in vec3 dir, in mat4 matrix ) {\n\treturn normalize( ( matrix * vec4( dir, 0.0 ) ).xyz );\n}\nvoid main() {\n\tvWorldDirection = transformDirection( position, modelMatrix );\n\t#include <begin_vertex>\n\t#include <project_vertex>\n}&quot;,fragmentShader:&quot;uniform sampler2D tEquirect;\nvarying vec3 vWorldDirection;\n#define RECIPROCAL_PI 0.31830988618\n#define RECIPROCAL_PI2 0.15915494\nvoid main() {\n\tvec3 direction = normalize( vWorldDirection );\n\tvec2 sampleUV;\n\tsampleUV.y = asin( clamp( direction.y, - 1.0, 1.0 ) ) * RECIPROCAL_PI + 0.5;\n\tsampleUV.x = atan( direction.z, direction.x ) * RECIPROCAL_PI2 + 0.5;\n\tgl_FragColor = texture2D( tEquirect, sampleUV );\n}&quot;,
side:1,blending:0});d.uniforms.tEquirect.value=b;b=new S(new Jd(5,5,5),d);c.add(b);d=new Gc(1,10,1);d.renderTarget=this;d.renderTarget.texture.name=&quot;CubeCameraTexture&quot;;d.update(a,c);b.geometry.dispose();b.material.dispose();return this};ac.prototype=Object.create(V.prototype);ac.prototype.constructor=ac;ac.prototype.isDataTexture=!0;var wd=new pb,Lf=new n;Object.assign(Hc.prototype,{set:function(a,b,c,d,e,f){var g=this.planes;g[0].copy(a);g[1].copy(b);g[2].copy(c);g[3].copy(d);g[4].copy(e);g[5].copy(f);
return this},clone:function(){return(new this.constructor).copy(this)},copy:function(a){for(var b=this.planes,c=0;6>c;c++)b[c].copy(a.planes[c]);return this},setFromProjectionMatrix:function(a){var b=this.planes,c=a.elements;a=c[0];var d=c[1],e=c[2],f=c[3],g=c[4],h=c[5],l=c[6],m=c[7],k=c[8],p=c[9],n=c[10],r=c[11],q=c[12],v=c[13],t=c[14];c=c[15];b[0].setComponents(f-a,m-g,r-k,c-q).normalize();b[1].setComponents(f+a,m+g,r+k,c+q).normalize();b[2].setComponents(f+d,m+h,r+p,c+v).normalize();b[3].setComponents(f-
d,m-h,r-p,c-v).normalize();b[4].setComponents(f-e,m-l,r-n,c-t).normalize();b[5].setComponents(f+e,m+l,r+n,c+t).normalize();return this},intersectsObject:function(a){var b=a.geometry;null===b.boundingSphere&&b.computeBoundingSphere();wd.copy(b.boundingSphere).applyMatrix4(a.matrixWorld);return this.intersectsSphere(wd)},intersectsSprite:function(a){wd.center.set(0,0,0);wd.radius=.7071067811865476;wd.applyMatrix4(a.matrixWorld);return this.intersectsSphere(wd)},intersectsSphere:function(a){var b=this.planes,
c=a.center;a=-a.radius;for(var d=0;6>d;d++)if(b[d].distanceToPoint(c)<a)return!1;return!0},intersectsBox:function(a){for(var b=this.planes,c=0;6>c;c++){var d=b[c];Lf.x=0<d.normal.x?a.max.x:a.min.x;Lf.y=0<d.normal.y?a.max.y:a.min.y;Lf.z=0<d.normal.z?a.max.z:a.min.z;if(0>d.distanceToPoint(Lf))return!1}return!0},containsPoint:function(a){for(var b=this.planes,c=0;6>c;c++)if(0>b[c].distanceToPoint(a))return!1;return!0}});var D={common:{diffuse:{value:new A(15658734)},opacity:{value:1},map:{value:null},
uvTransform:{value:new wa},uv2Transform:{value:new wa},alphaMap:{value:null}},specularmap:{specularMap:{value:null}},envmap:{envMap:{value:null},flipEnvMap:{value:-1},reflectivity:{value:1},refractionRatio:{value:.98},maxMipLevel:{value:0}},aomap:{aoMap:{value:null},aoMapIntensity:{value:1}},lightmap:{lightMap:{value:null},lightMapIntensity:{value:1}},emissivemap:{emissiveMap:{value:null}},bumpmap:{bumpMap:{value:null},bumpScale:{value:1}},normalmap:{normalMap:{value:null},normalScale:{value:new t(1,
1)}},displacementmap:{displacementMap:{value:null},displacementScale:{value:1},displacementBias:{value:0}},roughnessmap:{roughnessMap:{value:null}},metalnessmap:{metalnessMap:{value:null}},gradientmap:{gradientMap:{value:null}},fog:{fogDensity:{value:2.5E-4},fogNear:{value:1},fogFar:{value:2E3},fogColor:{value:new A(16777215)}},lights:{ambientLightColor:{value:[]},lightProbe:{value:[]},directionalLights:{value:[],properties:{direction:{},color:{}}},directionalLightShadows:{value:[],properties:{shadowBias:{},
shadowRadius:{},shadowMapSize:{}}},directionalShadowMap:{value:[]},directionalShadowMatrix:{value:[]},spotLights:{value:[],properties:{color:{},position:{},direction:{},distance:{},coneCos:{},penumbraCos:{},decay:{}}},spotLightShadows:{value:[],properties:{shadowBias:{},shadowRadius:{},shadowMapSize:{}}},spotShadowMap:{value:[]},spotShadowMatrix:{value:[]},pointLights:{value:[],properties:{color:{},position:{},decay:{},distance:{}}},pointLightShadows:{value:[],properties:{shadowBias:{},shadowRadius:{},
shadowMapSize:{},shadowCameraNear:{},shadowCameraFar:{}}},pointShadowMap:{value:[]},pointShadowMatrix:{value:[]},hemisphereLights:{value:[],properties:{direction:{},skyColor:{},groundColor:{}}},rectAreaLights:{value:[],properties:{color:{},position:{},width:{},height:{}}}},points:{diffuse:{value:new A(15658734)},opacity:{value:1},size:{value:1},scale:{value:1},map:{value:null},alphaMap:{value:null},uvTransform:{value:new wa}},sprite:{diffuse:{value:new A(15658734)},opacity:{value:1},center:{value:new t(.5,
.5)},rotation:{value:0},map:{value:null},alphaMap:{value:null},uvTransform:{value:new wa}}};Id.prototype=Object.create(N.prototype);Id.prototype.constructor=Id;bc.prototype=Object.create(C.prototype);bc.prototype.constructor=bc;var O={alphamap_fragment:&quot;#ifdef USE_ALPHAMAP\n\tdiffuseColor.a *= texture2D( alphaMap, vUv ).g;\n#endif&quot;,alphamap_pars_fragment:&quot;#ifdef USE_ALPHAMAP\n\tuniform sampler2D alphaMap;\n#endif&quot;,alphatest_fragment:&quot;#ifdef ALPHATEST\n\tif ( diffuseColor.a < ALPHATEST ) discard;\n#endif&quot;,
aomap_fragment:&quot;#ifdef USE_AOMAP\n\tfloat ambientOcclusion = ( texture2D( aoMap, vUv2 ).r - 1.0 ) * aoMapIntensity + 1.0;\n\treflectedLight.indirectDiffuse *= ambientOcclusion;\n\t#if defined( USE_ENVMAP ) && defined( STANDARD )\n\t\tfloat dotNV = saturate( dot( geometry.normal, geometry.viewDir ) );\n\t\treflectedLight.indirectSpecular *= computeSpecularOcclusion( dotNV, ambientOcclusion, material.specularRoughness );\n\t#endif\n#endif&quot;,aomap_pars_fragment:&quot;#ifdef USE_AOMAP\n\tuniform sampler2D aoMap;\n\tuniform float aoMapIntensity;\n#endif&quot;,
begin_vertex:&quot;vec3 transformed = vec3( position );&quot;,beginnormal_vertex:&quot;vec3 objectNormal = vec3( normal );\n#ifdef USE_TANGENT\n\tvec3 objectTangent = vec3( tangent.xyz );\n#endif&quot;,bsdfs:&quot;vec2 integrateSpecularBRDF( const in float dotNV, const in float roughness ) {\n\tconst vec4 c0 = vec4( - 1, - 0.0275, - 0.572, 0.022 );\n\tconst vec4 c1 = vec4( 1, 0.0425, 1.04, - 0.04 );\n\tvec4 r = roughness * c0 + c1;\n\tfloat a004 = min( r.x * r.x, exp2( - 9.28 * dotNV ) ) * r.x + r.y;\n\treturn vec2( -1.04, 1.04 ) * a004 + r.zw;\n}\nfloat punctualLightIntensityToIrradianceFactor( const in float lightDistance, const in float cutoffDistance, const in float decayExponent ) {\n#if defined ( PHYSICALLY_CORRECT_LIGHTS )\n\tfloat distanceFalloff = 1.0 / max( pow( lightDistance, decayExponent ), 0.01 );\n\tif( cutoffDistance > 0.0 ) {\n\t\tdistanceFalloff *= pow2( saturate( 1.0 - pow4( lightDistance / cutoffDistance ) ) );\n\t}\n\treturn distanceFalloff;\n#else\n\tif( cutoffDistance > 0.0 && decayExponent > 0.0 ) {\n\t\treturn pow( saturate( -lightDistance / cutoffDistance + 1.0 ), decayExponent );\n\t}\n\treturn 1.0;\n#endif\n}\nvec3 BRDF_Diffuse_Lambert( const in vec3 diffuseColor ) {\n\treturn RECIPROCAL_PI * diffuseColor;\n}\nvec3 F_Schlick( const in vec3 specularColor, const in float dotLH ) {\n\tfloat fresnel = exp2( ( -5.55473 * dotLH - 6.98316 ) * dotLH );\n\treturn ( 1.0 - specularColor ) * fresnel + specularColor;\n}\nvec3 F_Schlick_RoughnessDependent( const in vec3 F0, const in float dotNV, const in float roughness ) {\n\tfloat fresnel = exp2( ( -5.55473 * dotNV - 6.98316 ) * dotNV );\n\tvec3 Fr = max( vec3( 1.0 - roughness ), F0 ) - F0;\n\treturn Fr * fresnel + F0;\n}\nfloat G_GGX_Smith( const in float alpha, const in float dotNL, const in float dotNV ) {\n\tfloat a2 = pow2( alpha );\n\tfloat gl = dotNL + sqrt( a2 + ( 1.0 - a2 ) * pow2( dotNL ) );\n\tfloat gv = dotNV + sqrt( a2 + ( 1.0 - a2 ) * pow2( dotNV ) );\n\treturn 1.0 / ( gl * gv );\n}\nfloat G_GGX_SmithCorrelated( const in float alpha, const in float dotNL, const in float dotNV ) {\n\tfloat a2 = pow2( alpha );\n\tfloat gv = dotNL * sqrt( a2 + ( 1.0 - a2 ) * pow2( dotNV ) );\n\tfloat gl = dotNV * sqrt( a2 + ( 1.0 - a2 ) * pow2( dotNL ) );\n\treturn 0.5 / max( gv + gl, EPSILON );\n}\nfloat D_GGX( const in float alpha, const in float dotNH ) {\n\tfloat a2 = pow2( alpha );\n\tfloat denom = pow2( dotNH ) * ( a2 - 1.0 ) + 1.0;\n\treturn RECIPROCAL_PI * a2 / pow2( denom );\n}\nvec3 BRDF_Specular_GGX( const in IncidentLight incidentLight, const in vec3 viewDir, const in vec3 normal, const in vec3 specularColor, const in float roughness ) {\n\tfloat alpha = pow2( roughness );\n\tvec3 halfDir = normalize( incidentLight.direction + viewDir );\n\tfloat dotNL = saturate( dot( normal, incidentLight.direction ) );\n\tfloat dotNV = saturate( dot( normal, viewDir ) );\n\tfloat dotNH = saturate( dot( normal, halfDir ) );\n\tfloat dotLH = saturate( dot( incidentLight.direction, halfDir ) );\n\tvec3 F = F_Schlick( specularColor, dotLH );\n\tfloat G = G_GGX_SmithCorrelated( alpha, dotNL, dotNV );\n\tfloat D = D_GGX( alpha, dotNH );\n\treturn F * ( G * D );\n}\nvec2 LTC_Uv( const in vec3 N, const in vec3 V, const in float roughness ) {\n\tconst float LUT_SIZE  = 64.0;\n\tconst float LUT_SCALE = ( LUT_SIZE - 1.0 ) / LUT_SIZE;\n\tconst float LUT_BIAS  = 0.5 / LUT_SIZE;\n\tfloat dotNV = saturate( dot( N, V ) );\n\tvec2 uv = vec2( roughness, sqrt( 1.0 - dotNV ) );\n\tuv = uv * LUT_SCALE + LUT_BIAS;\n\treturn uv;\n}\nfloat LTC_ClippedSphereFormFactor( const in vec3 f ) {\n\tfloat l = length( f );\n\treturn max( ( l * l + f.z ) / ( l + 1.0 ), 0.0 );\n}\nvec3 LTC_EdgeVectorFormFactor( const in vec3 v1, const in vec3 v2 ) {\n\tfloat x = dot( v1, v2 );\n\tfloat y = abs( x );\n\tfloat a = 0.8543985 + ( 0.4965155 + 0.0145206 * y ) * y;\n\tfloat b = 3.4175940 + ( 4.1616724 + y ) * y;\n\tfloat v = a / b;\n\tfloat theta_sintheta = ( x > 0.0 ) ? v : 0.5 * inversesqrt( max( 1.0 - x * x, 1e-7 ) ) - v;\n\treturn cross( v1, v2 ) * theta_sintheta;\n}\nvec3 LTC_Evaluate( const in vec3 N, const in vec3 V, const in vec3 P, const in mat3 mInv, const in vec3 rectCoords[ 4 ] ) {\n\tvec3 v1 = rectCoords[ 1 ] - rectCoords[ 0 ];\n\tvec3 v2 = rectCoords[ 3 ] - rectCoords[ 0 ];\n\tvec3 lightNormal = cross( v1, v2 );\n\tif( dot( lightNormal, P - rectCoords[ 0 ] ) < 0.0 ) return vec3( 0.0 );\n\tvec3 T1, T2;\n\tT1 = normalize( V - N * dot( V, N ) );\n\tT2 = - cross( N, T1 );\n\tmat3 mat = mInv * transposeMat3( mat3( T1, T2, N ) );\n\tvec3 coords[ 4 ];\n\tcoords[ 0 ] = mat * ( rectCoords[ 0 ] - P );\n\tcoords[ 1 ] = mat * ( rectCoords[ 1 ] - P );\n\tcoords[ 2 ] = mat * ( rectCoords[ 2 ] - P );\n\tcoords[ 3 ] = mat * ( rectCoords[ 3 ] - P );\n\tcoords[ 0 ] = normalize( coords[ 0 ] );\n\tcoords[ 1 ] = normalize( coords[ 1 ] );\n\tcoords[ 2 ] = normalize( coords[ 2 ] );\n\tcoords[ 3 ] = normalize( coords[ 3 ] );\n\tvec3 vectorFormFactor = vec3( 0.0 );\n\tvectorFormFactor += LTC_EdgeVectorFormFactor( coords[ 0 ], coords[ 1 ] );\n\tvectorFormFactor += LTC_EdgeVectorFormFactor( coords[ 1 ], coords[ 2 ] );\n\tvectorFormFactor += LTC_EdgeVectorFormFactor( coords[ 2 ], coords[ 3 ] );\n\tvectorFormFactor += LTC_EdgeVectorFormFactor( coords[ 3 ], coords[ 0 ] );\n\tfloat result = LTC_ClippedSphereFormFactor( vectorFormFactor );\n\treturn vec3( result );\n}\nvec3 BRDF_Specular_GGX_Environment( const in vec3 viewDir, const in vec3 normal, const in vec3 specularColor, const in float roughness ) {\n\tfloat dotNV = saturate( dot( normal, viewDir ) );\n\tvec2 brdf = integrateSpecularBRDF( dotNV, roughness );\n\treturn specularColor * brdf.x + brdf.y;\n}\nvoid BRDF_Specular_Multiscattering_Environment( const in GeometricContext geometry, const in vec3 specularColor, const in float roughness, inout vec3 singleScatter, inout vec3 multiScatter ) {\n\tfloat dotNV = saturate( dot( geometry.normal, geometry.viewDir ) );\n\tvec3 F = F_Schlick_RoughnessDependent( specularColor, dotNV, roughness );\n\tvec2 brdf = integrateSpecularBRDF( dotNV, roughness );\n\tvec3 FssEss = F * brdf.x + brdf.y;\n\tfloat Ess = brdf.x + brdf.y;\n\tfloat Ems = 1.0 - Ess;\n\tvec3 Favg = specularColor + ( 1.0 - specularColor ) * 0.047619;\tvec3 Fms = FssEss * Favg / ( 1.0 - Ems * Favg );\n\tsingleScatter += FssEss;\n\tmultiScatter += Fms * Ems;\n}\nfloat G_BlinnPhong_Implicit( ) {\n\treturn 0.25;\n}\nfloat D_BlinnPhong( const in float shininess, const in float dotNH ) {\n\treturn RECIPROCAL_PI * ( shininess * 0.5 + 1.0 ) * pow( dotNH, shininess );\n}\nvec3 BRDF_Specular_BlinnPhong( const in IncidentLight incidentLight, const in GeometricContext geometry, const in vec3 specularColor, const in float shininess ) {\n\tvec3 halfDir = normalize( incidentLight.direction + geometry.viewDir );\n\tfloat dotNH = saturate( dot( geometry.normal, halfDir ) );\n\tfloat dotLH = saturate( dot( incidentLight.direction, halfDir ) );\n\tvec3 F = F_Schlick( specularColor, dotLH );\n\tfloat G = G_BlinnPhong_Implicit( );\n\tfloat D = D_BlinnPhong( shininess, dotNH );\n\treturn F * ( G * D );\n}\nfloat GGXRoughnessToBlinnExponent( const in float ggxRoughness ) {\n\treturn ( 2.0 / pow2( ggxRoughness + 0.0001 ) - 2.0 );\n}\nfloat BlinnExponentToGGXRoughness( const in float blinnExponent ) {\n\treturn sqrt( 2.0 / ( blinnExponent + 2.0 ) );\n}\n#if defined( USE_SHEEN )\nfloat D_Charlie(float roughness, float NoH) {\n\tfloat invAlpha  = 1.0 / roughness;\n\tfloat cos2h = NoH * NoH;\n\tfloat sin2h = max(1.0 - cos2h, 0.0078125);\treturn (2.0 + invAlpha) * pow(sin2h, invAlpha * 0.5) / (2.0 * PI);\n}\nfloat V_Neubelt(float NoV, float NoL) {\n\treturn saturate(1.0 / (4.0 * (NoL + NoV - NoL * NoV)));\n}\nvec3 BRDF_Specular_Sheen( const in float roughness, const in vec3 L, const in GeometricContext geometry, vec3 specularColor ) {\n\tvec3 N = geometry.normal;\n\tvec3 V = geometry.viewDir;\n\tvec3 H = normalize( V + L );\n\tfloat dotNH = saturate( dot( N, H ) );\n\treturn specularColor * D_Charlie( roughness, dotNH ) * V_Neubelt( dot(N, V), dot(N, L) );\n}\n#endif&quot;,
bumpmap_pars_fragment:&quot;#ifdef USE_BUMPMAP\n\tuniform sampler2D bumpMap;\n\tuniform float bumpScale;\n\tvec2 dHdxy_fwd() {\n\t\tvec2 dSTdx = dFdx( vUv );\n\t\tvec2 dSTdy = dFdy( vUv );\n\t\tfloat Hll = bumpScale * texture2D( bumpMap, vUv ).x;\n\t\tfloat dBx = bumpScale * texture2D( bumpMap, vUv + dSTdx ).x - Hll;\n\t\tfloat dBy = bumpScale * texture2D( bumpMap, vUv + dSTdy ).x - Hll;\n\t\treturn vec2( dBx, dBy );\n\t}\n\tvec3 perturbNormalArb( vec3 surf_pos, vec3 surf_norm, vec2 dHdxy ) {\n\t\tvec3 vSigmaX = vec3( dFdx( surf_pos.x ), dFdx( surf_pos.y ), dFdx( surf_pos.z ) );\n\t\tvec3 vSigmaY = vec3( dFdy( surf_pos.x ), dFdy( surf_pos.y ), dFdy( surf_pos.z ) );\n\t\tvec3 vN = surf_norm;\n\t\tvec3 R1 = cross( vSigmaY, vN );\n\t\tvec3 R2 = cross( vN, vSigmaX );\n\t\tfloat fDet = dot( vSigmaX, R1 );\n\t\tfDet *= ( float( gl_FrontFacing ) * 2.0 - 1.0 );\n\t\tvec3 vGrad = sign( fDet ) * ( dHdxy.x * R1 + dHdxy.y * R2 );\n\t\treturn normalize( abs( fDet ) * surf_norm - vGrad );\n\t}\n#endif&quot;,
clipping_planes_fragment:&quot;#if NUM_CLIPPING_PLANES > 0\n\tvec4 plane;\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < UNION_CLIPPING_PLANES; i ++ ) {\n\t\tplane = clippingPlanes[ i ];\n\t\tif ( dot( vClipPosition, plane.xyz ) > plane.w ) discard;\n\t}\n\t#pragma unroll_loop_end\n\t#if UNION_CLIPPING_PLANES < NUM_CLIPPING_PLANES\n\t\tbool clipped = true;\n\t\t#pragma unroll_loop_start\n\t\tfor ( int i = UNION_CLIPPING_PLANES; i < NUM_CLIPPING_PLANES; i ++ ) {\n\t\t\tplane = clippingPlanes[ i ];\n\t\t\tclipped = ( dot( vClipPosition, plane.xyz ) > plane.w ) && clipped;\n\t\t}\n\t\t#pragma unroll_loop_end\n\t\tif ( clipped ) discard;\n\t#endif\n#endif&quot;,
clipping_planes_pars_fragment:&quot;#if NUM_CLIPPING_PLANES > 0\n\tvarying vec3 vClipPosition;\n\tuniform vec4 clippingPlanes[ NUM_CLIPPING_PLANES ];\n#endif&quot;,clipping_planes_pars_vertex:&quot;#if NUM_CLIPPING_PLANES > 0\n\tvarying vec3 vClipPosition;\n#endif&quot;,clipping_planes_vertex:&quot;#if NUM_CLIPPING_PLANES > 0\n\tvClipPosition = - mvPosition.xyz;\n#endif&quot;,color_fragment:&quot;#ifdef USE_COLOR\n\tdiffuseColor.rgb *= vColor;\n#endif&quot;,color_pars_fragment:&quot;#ifdef USE_COLOR\n\tvarying vec3 vColor;\n#endif&quot;,color_pars_vertex:&quot;#ifdef USE_COLOR\n\tvarying vec3 vColor;\n#endif&quot;,
color_vertex:&quot;#ifdef USE_COLOR\n\tvColor.xyz = color.xyz;\n#endif&quot;,common:&quot;#define PI 3.14159265359\n#define PI2 6.28318530718\n#define PI_HALF 1.5707963267949\n#define RECIPROCAL_PI 0.31830988618\n#define RECIPROCAL_PI2 0.15915494\n#define LOG2 1.442695\n#define EPSILON 1e-6\n#ifndef saturate\n#define saturate(a) clamp( a, 0.0, 1.0 )\n#endif\n#define whiteComplement(a) ( 1.0 - saturate( a ) )\nfloat pow2( const in float x ) { return x*x; }\nfloat pow3( const in float x ) { return x*x*x; }\nfloat pow4( const in float x ) { float x2 = x*x; return x2*x2; }\nfloat average( const in vec3 color ) { return dot( color, vec3( 0.3333 ) ); }\nhighp float rand( const in vec2 uv ) {\n\tconst highp float a = 12.9898, b = 78.233, c = 43758.5453;\n\thighp float dt = dot( uv.xy, vec2( a,b ) ), sn = mod( dt, PI );\n\treturn fract(sin(sn) * c);\n}\n#ifdef HIGH_PRECISION\n\tfloat precisionSafeLength( vec3 v ) { return length( v ); }\n#else\n\tfloat max3( vec3 v ) { return max( max( v.x, v.y ), v.z ); }\n\tfloat precisionSafeLength( vec3 v ) {\n\t\tfloat maxComponent = max3( abs( v ) );\n\t\treturn length( v / maxComponent ) * maxComponent;\n\t}\n#endif\nstruct IncidentLight {\n\tvec3 color;\n\tvec3 direction;\n\tbool visible;\n};\nstruct ReflectedLight {\n\tvec3 directDiffuse;\n\tvec3 directSpecular;\n\tvec3 indirectDiffuse;\n\tvec3 indirectSpecular;\n};\nstruct GeometricContext {\n\tvec3 position;\n\tvec3 normal;\n\tvec3 viewDir;\n#ifdef CLEARCOAT\n\tvec3 clearcoatNormal;\n#endif\n};\nvec3 transformDirection( in vec3 dir, in mat4 matrix ) {\n\treturn normalize( ( matrix * vec4( dir, 0.0 ) ).xyz );\n}\nvec3 inverseTransformDirection( in vec3 dir, in mat4 matrix ) {\n\treturn normalize( ( vec4( dir, 0.0 ) * matrix ).xyz );\n}\nvec3 projectOnPlane(in vec3 point, in vec3 pointOnPlane, in vec3 planeNormal ) {\n\tfloat distance = dot( planeNormal, point - pointOnPlane );\n\treturn - distance * planeNormal + point;\n}\nfloat sideOfPlane( in vec3 point, in vec3 pointOnPlane, in vec3 planeNormal ) {\n\treturn sign( dot( point - pointOnPlane, planeNormal ) );\n}\nvec3 linePlaneIntersect( in vec3 pointOnLine, in vec3 lineDirection, in vec3 pointOnPlane, in vec3 planeNormal ) {\n\treturn lineDirection * ( dot( planeNormal, pointOnPlane - pointOnLine ) / dot( planeNormal, lineDirection ) ) + pointOnLine;\n}\nmat3 transposeMat3( const in mat3 m ) {\n\tmat3 tmp;\n\ttmp[ 0 ] = vec3( m[ 0 ].x, m[ 1 ].x, m[ 2 ].x );\n\ttmp[ 1 ] = vec3( m[ 0 ].y, m[ 1 ].y, m[ 2 ].y );\n\ttmp[ 2 ] = vec3( m[ 0 ].z, m[ 1 ].z, m[ 2 ].z );\n\treturn tmp;\n}\nfloat linearToRelativeLuminance( const in vec3 color ) {\n\tvec3 weights = vec3( 0.2126, 0.7152, 0.0722 );\n\treturn dot( weights, color.rgb );\n}\nbool isPerspectiveMatrix( mat4 m ) {\n  return m[ 2 ][ 3 ] == - 1.0;\n}&quot;,
cube_uv_reflection_fragment:&quot;#ifdef ENVMAP_TYPE_CUBE_UV\n#define cubeUV_maxMipLevel 8.0\n#define cubeUV_minMipLevel 4.0\n#define cubeUV_maxTileSize 256.0\n#define cubeUV_minTileSize 16.0\nfloat getFace(vec3 direction) {\n    vec3 absDirection = abs(direction);\n    float face = -1.0;\n    if (absDirection.x > absDirection.z) {\n      if (absDirection.x > absDirection.y)\n        face = direction.x > 0.0 ? 0.0 : 3.0;\n      else\n        face = direction.y > 0.0 ? 1.0 : 4.0;\n    } else {\n      if (absDirection.z > absDirection.y)\n        face = direction.z > 0.0 ? 2.0 : 5.0;\n      else\n        face = direction.y > 0.0 ? 1.0 : 4.0;\n    }\n    return face;\n}\nvec2 getUV(vec3 direction, float face) {\n    vec2 uv;\n    if (face == 0.0) {\n      uv = vec2(-direction.z, direction.y) / abs(direction.x);\n    } else if (face == 1.0) {\n      uv = vec2(direction.x, -direction.z) / abs(direction.y);\n    } else if (face == 2.0) {\n      uv = direction.xy / abs(direction.z);\n    } else if (face == 3.0) {\n      uv = vec2(direction.z, direction.y) / abs(direction.x);\n    } else if (face == 4.0) {\n      uv = direction.xz / abs(direction.y);\n    } else {\n      uv = vec2(-direction.x, direction.y) / abs(direction.z);\n    }\n    return 0.5 * (uv + 1.0);\n}\nvec3 bilinearCubeUV(sampler2D envMap, vec3 direction, float mipInt) {\n  float face = getFace(direction);\n  float filterInt = max(cubeUV_minMipLevel - mipInt, 0.0);\n  mipInt = max(mipInt, cubeUV_minMipLevel);\n  float faceSize = exp2(mipInt);\n  float texelSize = 1.0 / (3.0 * cubeUV_maxTileSize);\n  vec2 uv = getUV(direction, face) * (faceSize - 1.0);\n  vec2 f = fract(uv);\n  uv += 0.5 - f;\n  if (face > 2.0) {\n    uv.y += faceSize;\n    face -= 3.0;\n  }\n  uv.x += face * faceSize;\n  if(mipInt < cubeUV_maxMipLevel){\n    uv.y += 2.0 * cubeUV_maxTileSize;\n  }\n  uv.y += filterInt * 2.0 * cubeUV_minTileSize;\n  uv.x += 3.0 * max(0.0, cubeUV_maxTileSize - 2.0 * faceSize);\n  uv *= texelSize;\n  vec3 tl = envMapTexelToLinear(texture2D(envMap, uv)).rgb;\n  uv.x += texelSize;\n  vec3 tr = envMapTexelToLinear(texture2D(envMap, uv)).rgb;\n  uv.y += texelSize;\n  vec3 br = envMapTexelToLinear(texture2D(envMap, uv)).rgb;\n  uv.x -= texelSize;\n  vec3 bl = envMapTexelToLinear(texture2D(envMap, uv)).rgb;\n  vec3 tm = mix(tl, tr, f.x);\n  vec3 bm = mix(bl, br, f.x);\n  return mix(tm, bm, f.y);\n}\n#define r0 1.0\n#define v0 0.339\n#define m0 -2.0\n#define r1 0.8\n#define v1 0.276\n#define m1 -1.0\n#define r4 0.4\n#define v4 0.046\n#define m4 2.0\n#define r5 0.305\n#define v5 0.016\n#define m5 3.0\n#define r6 0.21\n#define v6 0.0038\n#define m6 4.0\nfloat roughnessToMip(float roughness) {\n  float mip = 0.0;\n  if (roughness >= r1) {\n    mip = (r0 - roughness) * (m1 - m0) / (r0 - r1) + m0;\n  } else if (roughness >= r4) {\n    mip = (r1 - roughness) * (m4 - m1) / (r1 - r4) + m1;\n  } else if (roughness >= r5) {\n    mip = (r4 - roughness) * (m5 - m4) / (r4 - r5) + m4;\n  } else if (roughness >= r6) {\n    mip = (r5 - roughness) * (m6 - m5) / (r5 - r6) + m5;\n  } else {\n    mip = -2.0 * log2(1.16 * roughness);  }\n  return mip;\n}\nvec4 textureCubeUV(sampler2D envMap, vec3 sampleDir, float roughness) {\n  float mip = clamp(roughnessToMip(roughness), m0, cubeUV_maxMipLevel);\n  float mipF = fract(mip);\n  float mipInt = floor(mip);\n  vec3 color0 = bilinearCubeUV(envMap, sampleDir, mipInt);\n  if (mipF == 0.0) {\n    return vec4(color0, 1.0);\n  } else {\n    vec3 color1 = bilinearCubeUV(envMap, sampleDir, mipInt + 1.0);\n    return vec4(mix(color0, color1, mipF), 1.0);\n  }\n}\n#endif&quot;,
defaultnormal_vertex:&quot;vec3 transformedNormal = objectNormal;\n#ifdef USE_INSTANCING\n\tmat3 m = mat3( instanceMatrix );\n\ttransformedNormal /= vec3( dot( m[ 0 ], m[ 0 ] ), dot( m[ 1 ], m[ 1 ] ), dot( m[ 2 ], m[ 2 ] ) );\n\ttransformedNormal = m * transformedNormal;\n#endif\ntransformedNormal = normalMatrix * transformedNormal;\n#ifdef FLIP_SIDED\n\ttransformedNormal = - transformedNormal;\n#endif\n#ifdef USE_TANGENT\n\tvec3 transformedTangent = ( modelViewMatrix * vec4( objectTangent, 0.0 ) ).xyz;\n\t#ifdef FLIP_SIDED\n\t\ttransformedTangent = - transformedTangent;\n\t#endif\n#endif&quot;,
displacementmap_pars_vertex:&quot;#ifdef USE_DISPLACEMENTMAP\n\tuniform sampler2D displacementMap;\n\tuniform float displacementScale;\n\tuniform float displacementBias;\n#endif&quot;,displacementmap_vertex:&quot;#ifdef USE_DISPLACEMENTMAP\n\ttransformed += normalize( objectNormal ) * ( texture2D( displacementMap, vUv ).x * displacementScale + displacementBias );\n#endif&quot;,emissivemap_fragment:&quot;#ifdef USE_EMISSIVEMAP\n\tvec4 emissiveColor = texture2D( emissiveMap, vUv );\n\temissiveColor.rgb = emissiveMapTexelToLinear( emissiveColor ).rgb;\n\ttotalEmissiveRadiance *= emissiveColor.rgb;\n#endif&quot;,
emissivemap_pars_fragment:&quot;#ifdef USE_EMISSIVEMAP\n\tuniform sampler2D emissiveMap;\n#endif&quot;,encodings_fragment:&quot;gl_FragColor = linearToOutputTexel( gl_FragColor );&quot;,encodings_pars_fragment:&quot;\nvec4 LinearToLinear( in vec4 value ) {\n\treturn value;\n}\nvec4 GammaToLinear( in vec4 value, in float gammaFactor ) {\n\treturn vec4( pow( value.rgb, vec3( gammaFactor ) ), value.a );\n}\nvec4 LinearToGamma( in vec4 value, in float gammaFactor ) {\n\treturn vec4( pow( value.rgb, vec3( 1.0 / gammaFactor ) ), value.a );\n}\nvec4 sRGBToLinear( in vec4 value ) {\n\treturn vec4( mix( pow( value.rgb * 0.9478672986 + vec3( 0.0521327014 ), vec3( 2.4 ) ), value.rgb * 0.0773993808, vec3( lessThanEqual( value.rgb, vec3( 0.04045 ) ) ) ), value.a );\n}\nvec4 LinearTosRGB( in vec4 value ) {\n\treturn vec4( mix( pow( value.rgb, vec3( 0.41666 ) ) * 1.055 - vec3( 0.055 ), value.rgb * 12.92, vec3( lessThanEqual( value.rgb, vec3( 0.0031308 ) ) ) ), value.a );\n}\nvec4 RGBEToLinear( in vec4 value ) {\n\treturn vec4( value.rgb * exp2( value.a * 255.0 - 128.0 ), 1.0 );\n}\nvec4 LinearToRGBE( in vec4 value ) {\n\tfloat maxComponent = max( max( value.r, value.g ), value.b );\n\tfloat fExp = clamp( ceil( log2( maxComponent ) ), -128.0, 127.0 );\n\treturn vec4( value.rgb / exp2( fExp ), ( fExp + 128.0 ) / 255.0 );\n}\nvec4 RGBMToLinear( in vec4 value, in float maxRange ) {\n\treturn vec4( value.rgb * value.a * maxRange, 1.0 );\n}\nvec4 LinearToRGBM( in vec4 value, in float maxRange ) {\n\tfloat maxRGB = max( value.r, max( value.g, value.b ) );\n\tfloat M = clamp( maxRGB / maxRange, 0.0, 1.0 );\n\tM = ceil( M * 255.0 ) / 255.0;\n\treturn vec4( value.rgb / ( M * maxRange ), M );\n}\nvec4 RGBDToLinear( in vec4 value, in float maxRange ) {\n\treturn vec4( value.rgb * ( ( maxRange / 255.0 ) / value.a ), 1.0 );\n}\nvec4 LinearToRGBD( in vec4 value, in float maxRange ) {\n\tfloat maxRGB = max( value.r, max( value.g, value.b ) );\n\tfloat D = max( maxRange / maxRGB, 1.0 );\n\tD = clamp( floor( D ) / 255.0, 0.0, 1.0 );\n\treturn vec4( value.rgb * ( D * ( 255.0 / maxRange ) ), D );\n}\nconst mat3 cLogLuvM = mat3( 0.2209, 0.3390, 0.4184, 0.1138, 0.6780, 0.7319, 0.0102, 0.1130, 0.2969 );\nvec4 LinearToLogLuv( in vec4 value )  {\n\tvec3 Xp_Y_XYZp = cLogLuvM * value.rgb;\n\tXp_Y_XYZp = max( Xp_Y_XYZp, vec3( 1e-6, 1e-6, 1e-6 ) );\n\tvec4 vResult;\n\tvResult.xy = Xp_Y_XYZp.xy / Xp_Y_XYZp.z;\n\tfloat Le = 2.0 * log2(Xp_Y_XYZp.y) + 127.0;\n\tvResult.w = fract( Le );\n\tvResult.z = ( Le - ( floor( vResult.w * 255.0 ) ) / 255.0 ) / 255.0;\n\treturn vResult;\n}\nconst mat3 cLogLuvInverseM = mat3( 6.0014, -2.7008, -1.7996, -1.3320, 3.1029, -5.7721, 0.3008, -1.0882, 5.6268 );\nvec4 LogLuvToLinear( in vec4 value ) {\n\tfloat Le = value.z * 255.0 + value.w;\n\tvec3 Xp_Y_XYZp;\n\tXp_Y_XYZp.y = exp2( ( Le - 127.0 ) / 2.0 );\n\tXp_Y_XYZp.z = Xp_Y_XYZp.y / value.y;\n\tXp_Y_XYZp.x = value.x * Xp_Y_XYZp.z;\n\tvec3 vRGB = cLogLuvInverseM * Xp_Y_XYZp.rgb;\n\treturn vec4( max( vRGB, 0.0 ), 1.0 );\n}&quot;,
envmap_fragment:&quot;#ifdef USE_ENVMAP\n\t#ifdef ENV_WORLDPOS\n\t\tvec3 cameraToFrag;\n\t\t\n\t\tif ( isOrthographic ) {\n\t\t\tcameraToFrag = normalize( vec3( - viewMatrix[ 0 ][ 2 ], - viewMatrix[ 1 ][ 2 ], - viewMatrix[ 2 ][ 2 ] ) );\n\t\t}  else {\n\t\t\tcameraToFrag = normalize( vWorldPosition - cameraPosition );\n\t\t}\n\t\tvec3 worldNormal = inverseTransformDirection( normal, viewMatrix );\n\t\t#ifdef ENVMAP_MODE_REFLECTION\n\t\t\tvec3 reflectVec = reflect( cameraToFrag, worldNormal );\n\t\t#else\n\t\t\tvec3 reflectVec = refract( cameraToFrag, worldNormal, refractionRatio );\n\t\t#endif\n\t#else\n\t\tvec3 reflectVec = vReflect;\n\t#endif\n\t#ifdef ENVMAP_TYPE_CUBE\n\t\tvec4 envColor = textureCube( envMap, vec3( flipEnvMap * reflectVec.x, reflectVec.yz ) );\n\t#elif defined( ENVMAP_TYPE_CUBE_UV )\n\t\tvec4 envColor = textureCubeUV( envMap, reflectVec, 0.0 );\n\t#elif defined( ENVMAP_TYPE_EQUIREC )\n\t\tvec2 sampleUV;\n\t\treflectVec = normalize( reflectVec );\n\t\tsampleUV.y = asin( clamp( reflectVec.y, - 1.0, 1.0 ) ) * RECIPROCAL_PI + 0.5;\n\t\tsampleUV.x = atan( reflectVec.z, reflectVec.x ) * RECIPROCAL_PI2 + 0.5;\n\t\tvec4 envColor = texture2D( envMap, sampleUV );\n\t#elif defined( ENVMAP_TYPE_SPHERE )\n\t\treflectVec = normalize( reflectVec );\n\t\tvec3 reflectView = normalize( ( viewMatrix * vec4( reflectVec, 0.0 ) ).xyz + vec3( 0.0, 0.0, 1.0 ) );\n\t\tvec4 envColor = texture2D( envMap, reflectView.xy * 0.5 + 0.5 );\n\t#else\n\t\tvec4 envColor = vec4( 0.0 );\n\t#endif\n\t#ifndef ENVMAP_TYPE_CUBE_UV\n\t\tenvColor = envMapTexelToLinear( envColor );\n\t#endif\n\t#ifdef ENVMAP_BLENDING_MULTIPLY\n\t\toutgoingLight = mix( outgoingLight, outgoingLight * envColor.xyz, specularStrength * reflectivity );\n\t#elif defined( ENVMAP_BLENDING_MIX )\n\t\toutgoingLight = mix( outgoingLight, envColor.xyz, specularStrength * reflectivity );\n\t#elif defined( ENVMAP_BLENDING_ADD )\n\t\toutgoingLight += envColor.xyz * specularStrength * reflectivity;\n\t#endif\n#endif&quot;,
envmap_common_pars_fragment:&quot;#ifdef USE_ENVMAP\n\tuniform float envMapIntensity;\n\tuniform float flipEnvMap;\n\tuniform int maxMipLevel;\n\t#ifdef ENVMAP_TYPE_CUBE\n\t\tuniform samplerCube envMap;\n\t#else\n\t\tuniform sampler2D envMap;\n\t#endif\n\t\n#endif&quot;,envmap_pars_fragment:&quot;#ifdef USE_ENVMAP\n\tuniform float reflectivity;\n\t#if defined( USE_BUMPMAP ) || defined( USE_NORMALMAP ) || defined( PHONG )\n\t\t#define ENV_WORLDPOS\n\t#endif\n\t#ifdef ENV_WORLDPOS\n\t\tvarying vec3 vWorldPosition;\n\t\tuniform float refractionRatio;\n\t#else\n\t\tvarying vec3 vReflect;\n\t#endif\n#endif&quot;,
envmap_pars_vertex:&quot;#ifdef USE_ENVMAP\n\t#if defined( USE_BUMPMAP ) || defined( USE_NORMALMAP ) ||defined( PHONG )\n\t\t#define ENV_WORLDPOS\n\t#endif\n\t#ifdef ENV_WORLDPOS\n\t\t\n\t\tvarying vec3 vWorldPosition;\n\t#else\n\t\tvarying vec3 vReflect;\n\t\tuniform float refractionRatio;\n\t#endif\n#endif&quot;,envmap_physical_pars_fragment:&quot;#if defined( USE_ENVMAP )\n\t#ifdef ENVMAP_MODE_REFRACTION\n\t\tuniform float refractionRatio;\n\t#endif\n\tvec3 getLightProbeIndirectIrradiance( const in GeometricContext geometry, const in int maxMIPLevel ) {\n\t\tvec3 worldNormal = inverseTransformDirection( geometry.normal, viewMatrix );\n\t\t#ifdef ENVMAP_TYPE_CUBE\n\t\t\tvec3 queryVec = vec3( flipEnvMap * worldNormal.x, worldNormal.yz );\n\t\t\t#ifdef TEXTURE_LOD_EXT\n\t\t\t\tvec4 envMapColor = textureCubeLodEXT( envMap, queryVec, float( maxMIPLevel ) );\n\t\t\t#else\n\t\t\t\tvec4 envMapColor = textureCube( envMap, queryVec, float( maxMIPLevel ) );\n\t\t\t#endif\n\t\t\tenvMapColor.rgb = envMapTexelToLinear( envMapColor ).rgb;\n\t\t#elif defined( ENVMAP_TYPE_CUBE_UV )\n\t\t\tvec4 envMapColor = textureCubeUV( envMap, worldNormal, 1.0 );\n\t\t#else\n\t\t\tvec4 envMapColor = vec4( 0.0 );\n\t\t#endif\n\t\treturn PI * envMapColor.rgb * envMapIntensity;\n\t}\n\tfloat getSpecularMIPLevel( const in float roughness, const in int maxMIPLevel ) {\n\t\tfloat maxMIPLevelScalar = float( maxMIPLevel );\n\t\tfloat sigma = PI * roughness * roughness / ( 1.0 + roughness );\n\t\tfloat desiredMIPLevel = maxMIPLevelScalar + log2( sigma );\n\t\treturn clamp( desiredMIPLevel, 0.0, maxMIPLevelScalar );\n\t}\n\tvec3 getLightProbeIndirectRadiance( const in vec3 viewDir, const in vec3 normal, const in float roughness, const in int maxMIPLevel ) {\n\t\t#ifdef ENVMAP_MODE_REFLECTION\n\t\t  vec3 reflectVec = reflect( -viewDir, normal );\n\t\t  reflectVec = normalize( mix( reflectVec, normal, roughness * roughness) );\n\t\t#else\n\t\t  vec3 reflectVec = refract( -viewDir, normal, refractionRatio );\n\t\t#endif\n\t\treflectVec = inverseTransformDirection( reflectVec, viewMatrix );\n\t\tfloat specularMIPLevel = getSpecularMIPLevel( roughness, maxMIPLevel );\n\t\t#ifdef ENVMAP_TYPE_CUBE\n\t\t\tvec3 queryReflectVec = vec3( flipEnvMap * reflectVec.x, reflectVec.yz );\n\t\t\t#ifdef TEXTURE_LOD_EXT\n\t\t\t\tvec4 envMapColor = textureCubeLodEXT( envMap, queryReflectVec, specularMIPLevel );\n\t\t\t#else\n\t\t\t\tvec4 envMapColor = textureCube( envMap, queryReflectVec, specularMIPLevel );\n\t\t\t#endif\n\t\t\tenvMapColor.rgb = envMapTexelToLinear( envMapColor ).rgb;\n\t\t#elif defined( ENVMAP_TYPE_CUBE_UV )\n\t\t\tvec4 envMapColor = textureCubeUV( envMap, reflectVec, roughness );\n\t\t#elif defined( ENVMAP_TYPE_EQUIREC )\n\t\t\tvec2 sampleUV;\n\t\t\tsampleUV.y = asin( clamp( reflectVec.y, - 1.0, 1.0 ) ) * RECIPROCAL_PI + 0.5;\n\t\t\tsampleUV.x = atan( reflectVec.z, reflectVec.x ) * RECIPROCAL_PI2 + 0.5;\n\t\t\t#ifdef TEXTURE_LOD_EXT\n\t\t\t\tvec4 envMapColor = texture2DLodEXT( envMap, sampleUV, specularMIPLevel );\n\t\t\t#else\n\t\t\t\tvec4 envMapColor = texture2D( envMap, sampleUV, specularMIPLevel );\n\t\t\t#endif\n\t\t\tenvMapColor.rgb = envMapTexelToLinear( envMapColor ).rgb;\n\t\t#elif defined( ENVMAP_TYPE_SPHERE )\n\t\t\tvec3 reflectView = normalize( ( viewMatrix * vec4( reflectVec, 0.0 ) ).xyz + vec3( 0.0,0.0,1.0 ) );\n\t\t\t#ifdef TEXTURE_LOD_EXT\n\t\t\t\tvec4 envMapColor = texture2DLodEXT( envMap, reflectView.xy * 0.5 + 0.5, specularMIPLevel );\n\t\t\t#else\n\t\t\t\tvec4 envMapColor = texture2D( envMap, reflectView.xy * 0.5 + 0.5, specularMIPLevel );\n\t\t\t#endif\n\t\t\tenvMapColor.rgb = envMapTexelToLinear( envMapColor ).rgb;\n\t\t#endif\n\t\treturn envMapColor.rgb * envMapIntensity;\n\t}\n#endif&quot;,
envmap_vertex:&quot;#ifdef USE_ENVMAP\n\t#ifdef ENV_WORLDPOS\n\t\tvWorldPosition = worldPosition.xyz;\n\t#else\n\t\tvec3 cameraToVertex;\n\t\tif ( isOrthographic ) { \n\t\t\tcameraToVertex = normalize( vec3( - viewMatrix[ 0 ][ 2 ], - viewMatrix[ 1 ][ 2 ], - viewMatrix[ 2 ][ 2 ] ) );\n\t\t} else {\n\t\t\tcameraToVertex = normalize( worldPosition.xyz - cameraPosition );\n\t\t}\n\t\tvec3 worldNormal = inverseTransformDirection( transformedNormal, viewMatrix );\n\t\t#ifdef ENVMAP_MODE_REFLECTION\n\t\t\tvReflect = reflect( cameraToVertex, worldNormal );\n\t\t#else\n\t\t\tvReflect = refract( cameraToVertex, worldNormal, refractionRatio );\n\t\t#endif\n\t#endif\n#endif&quot;,
fog_vertex:&quot;#ifdef USE_FOG\n\tfogDepth = -mvPosition.z;\n#endif&quot;,fog_pars_vertex:&quot;#ifdef USE_FOG\n\tvarying float fogDepth;\n#endif&quot;,fog_fragment:&quot;#ifdef USE_FOG\n\t#ifdef FOG_EXP2\n\t\tfloat fogFactor = 1.0 - exp( - fogDensity * fogDensity * fogDepth * fogDepth );\n\t#else\n\t\tfloat fogFactor = smoothstep( fogNear, fogFar, fogDepth );\n\t#endif\n\tgl_FragColor.rgb = mix( gl_FragColor.rgb, fogColor, fogFactor );\n#endif&quot;,fog_pars_fragment:&quot;#ifdef USE_FOG\n\tuniform vec3 fogColor;\n\tvarying float fogDepth;\n\t#ifdef FOG_EXP2\n\t\tuniform float fogDensity;\n\t#else\n\t\tuniform float fogNear;\n\t\tuniform float fogFar;\n\t#endif\n#endif&quot;,
gradientmap_pars_fragment:&quot;#ifdef USE_GRADIENTMAP\n\tuniform sampler2D gradientMap;\n#endif\nvec3 getGradientIrradiance( vec3 normal, vec3 lightDirection ) {\n\tfloat dotNL = dot( normal, lightDirection );\n\tvec2 coord = vec2( dotNL * 0.5 + 0.5, 0.0 );\n\t#ifdef USE_GRADIENTMAP\n\t\treturn texture2D( gradientMap, coord ).rgb;\n\t#else\n\t\treturn ( coord.x < 0.7 ) ? vec3( 0.7 ) : vec3( 1.0 );\n\t#endif\n}&quot;,lightmap_fragment:&quot;#ifdef USE_LIGHTMAP\n\tvec4 lightMapTexel= texture2D( lightMap, vUv2 );\n\treflectedLight.indirectDiffuse += PI * lightMapTexelToLinear( lightMapTexel ).rgb * lightMapIntensity;\n#endif&quot;,
lightmap_pars_fragment:&quot;#ifdef USE_LIGHTMAP\n\tuniform sampler2D lightMap;\n\tuniform float lightMapIntensity;\n#endif&quot;,lights_lambert_vertex:&quot;vec3 diffuse = vec3( 1.0 );\nGeometricContext geometry;\ngeometry.position = mvPosition.xyz;\ngeometry.normal = normalize( transformedNormal );\ngeometry.viewDir = ( isOrthographic ) ? vec3( 0, 0, 1 ) : normalize( -mvPosition.xyz );\nGeometricContext backGeometry;\nbackGeometry.position = geometry.position;\nbackGeometry.normal = -geometry.normal;\nbackGeometry.viewDir = geometry.viewDir;\nvLightFront = vec3( 0.0 );\nvIndirectFront = vec3( 0.0 );\n#ifdef DOUBLE_SIDED\n\tvLightBack = vec3( 0.0 );\n\tvIndirectBack = vec3( 0.0 );\n#endif\nIncidentLight directLight;\nfloat dotNL;\nvec3 directLightColor_Diffuse;\n#if NUM_POINT_LIGHTS > 0\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_POINT_LIGHTS; i ++ ) {\n\t\tgetPointDirectLightIrradiance( pointLights[ i ], geometry, directLight );\n\t\tdotNL = dot( geometry.normal, directLight.direction );\n\t\tdirectLightColor_Diffuse = PI * directLight.color;\n\t\tvLightFront += saturate( dotNL ) * directLightColor_Diffuse;\n\t\t#ifdef DOUBLE_SIDED\n\t\t\tvLightBack += saturate( -dotNL ) * directLightColor_Diffuse;\n\t\t#endif\n\t}\n\t#pragma unroll_loop_end\n#endif\n#if NUM_SPOT_LIGHTS > 0\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_SPOT_LIGHTS; i ++ ) {\n\t\tgetSpotDirectLightIrradiance( spotLights[ i ], geometry, directLight );\n\t\tdotNL = dot( geometry.normal, directLight.direction );\n\t\tdirectLightColor_Diffuse = PI * directLight.color;\n\t\tvLightFront += saturate( dotNL ) * directLightColor_Diffuse;\n\t\t#ifdef DOUBLE_SIDED\n\t\t\tvLightBack += saturate( -dotNL ) * directLightColor_Diffuse;\n\t\t#endif\n\t}\n\t#pragma unroll_loop_end\n#endif\n#if NUM_DIR_LIGHTS > 0\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_DIR_LIGHTS; i ++ ) {\n\t\tgetDirectionalDirectLightIrradiance( directionalLights[ i ], geometry, directLight );\n\t\tdotNL = dot( geometry.normal, directLight.direction );\n\t\tdirectLightColor_Diffuse = PI * directLight.color;\n\t\tvLightFront += saturate( dotNL ) * directLightColor_Diffuse;\n\t\t#ifdef DOUBLE_SIDED\n\t\t\tvLightBack += saturate( -dotNL ) * directLightColor_Diffuse;\n\t\t#endif\n\t}\n\t#pragma unroll_loop_end\n#endif\n#if NUM_HEMI_LIGHTS > 0\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_HEMI_LIGHTS; i ++ ) {\n\t\tvIndirectFront += getHemisphereLightIrradiance( hemisphereLights[ i ], geometry );\n\t\t#ifdef DOUBLE_SIDED\n\t\t\tvIndirectBack += getHemisphereLightIrradiance( hemisphereLights[ i ], backGeometry );\n\t\t#endif\n\t}\n\t#pragma unroll_loop_end\n#endif&quot;,
lights_pars_begin:&quot;uniform bool receiveShadow;\nuniform vec3 ambientLightColor;\nuniform vec3 lightProbe[ 9 ];\nvec3 shGetIrradianceAt( in vec3 normal, in vec3 shCoefficients[ 9 ] ) {\n\tfloat x = normal.x, y = normal.y, z = normal.z;\n\tvec3 result = shCoefficients[ 0 ] * 0.886227;\n\tresult += shCoefficients[ 1 ] * 2.0 * 0.511664 * y;\n\tresult += shCoefficients[ 2 ] * 2.0 * 0.511664 * z;\n\tresult += shCoefficients[ 3 ] * 2.0 * 0.511664 * x;\n\tresult += shCoefficients[ 4 ] * 2.0 * 0.429043 * x * y;\n\tresult += shCoefficients[ 5 ] * 2.0 * 0.429043 * y * z;\n\tresult += shCoefficients[ 6 ] * ( 0.743125 * z * z - 0.247708 );\n\tresult += shCoefficients[ 7 ] * 2.0 * 0.429043 * x * z;\n\tresult += shCoefficients[ 8 ] * 0.429043 * ( x * x - y * y );\n\treturn result;\n}\nvec3 getLightProbeIrradiance( const in vec3 lightProbe[ 9 ], const in GeometricContext geometry ) {\n\tvec3 worldNormal = inverseTransformDirection( geometry.normal, viewMatrix );\n\tvec3 irradiance = shGetIrradianceAt( worldNormal, lightProbe );\n\treturn irradiance;\n}\nvec3 getAmbientLightIrradiance( const in vec3 ambientLightColor ) {\n\tvec3 irradiance = ambientLightColor;\n\t#ifndef PHYSICALLY_CORRECT_LIGHTS\n\t\tirradiance *= PI;\n\t#endif\n\treturn irradiance;\n}\n#if NUM_DIR_LIGHTS > 0\n\tstruct DirectionalLight {\n\t\tvec3 direction;\n\t\tvec3 color;\n\t};\n\tuniform DirectionalLight directionalLights[ NUM_DIR_LIGHTS ];\n\t#if defined( USE_SHADOWMAP ) && NUM_DIR_LIGHT_SHADOWS > 0\n\t\tstruct DirectionalLightShadow {\n\t\t\tfloat shadowBias;\n\t\t\tfloat shadowRadius;\n\t\t\tvec2 shadowMapSize;\n\t\t};\n\t\tuniform DirectionalLightShadow directionalLightShadows[ NUM_DIR_LIGHT_SHADOWS ];\n\t#endif\n\tvoid getDirectionalDirectLightIrradiance( const in DirectionalLight directionalLight, const in GeometricContext geometry, out IncidentLight directLight ) {\n\t\tdirectLight.color = directionalLight.color;\n\t\tdirectLight.direction = directionalLight.direction;\n\t\tdirectLight.visible = true;\n\t}\n#endif\n#if NUM_POINT_LIGHTS > 0\n\tstruct PointLight {\n\t\tvec3 position;\n\t\tvec3 color;\n\t\tfloat distance;\n\t\tfloat decay;\n\t};\n\tuniform PointLight pointLights[ NUM_POINT_LIGHTS ];\n\t#if defined( USE_SHADOWMAP ) && NUM_POINT_LIGHT_SHADOWS > 0\n\t\tstruct PointLightShadow {\n\t\t\tfloat shadowBias;\n\t\t\tfloat shadowRadius;\n\t\t\tvec2 shadowMapSize;\n\t\t\tfloat shadowCameraNear;\n\t\t\tfloat shadowCameraFar;\n\t\t};\n\t\tuniform PointLightShadow pointLightShadows[ NUM_POINT_LIGHT_SHADOWS ];\n\t#endif\n\tvoid getPointDirectLightIrradiance( const in PointLight pointLight, const in GeometricContext geometry, out IncidentLight directLight ) {\n\t\tvec3 lVector = pointLight.position - geometry.position;\n\t\tdirectLight.direction = normalize( lVector );\n\t\tfloat lightDistance = length( lVector );\n\t\tdirectLight.color = pointLight.color;\n\t\tdirectLight.color *= punctualLightIntensityToIrradianceFactor( lightDistance, pointLight.distance, pointLight.decay );\n\t\tdirectLight.visible = ( directLight.color != vec3( 0.0 ) );\n\t}\n#endif\n#if NUM_SPOT_LIGHTS > 0\n\tstruct SpotLight {\n\t\tvec3 position;\n\t\tvec3 direction;\n\t\tvec3 color;\n\t\tfloat distance;\n\t\tfloat decay;\n\t\tfloat coneCos;\n\t\tfloat penumbraCos;\n\t};\n\tuniform SpotLight spotLights[ NUM_SPOT_LIGHTS ];\n\t#if defined( USE_SHADOWMAP ) && NUM_SPOT_LIGHT_SHADOWS > 0\n\t\tstruct SpotLightShadow {\n\t\t\tfloat shadowBias;\n\t\t\tfloat shadowRadius;\n\t\t\tvec2 shadowMapSize;\n\t\t};\n\t\tuniform SpotLightShadow spotLightShadows[ NUM_SPOT_LIGHT_SHADOWS ];\n\t#endif\n\tvoid getSpotDirectLightIrradiance( const in SpotLight spotLight, const in GeometricContext geometry, out IncidentLight directLight  ) {\n\t\tvec3 lVector = spotLight.position - geometry.position;\n\t\tdirectLight.direction = normalize( lVector );\n\t\tfloat lightDistance = length( lVector );\n\t\tfloat angleCos = dot( directLight.direction, spotLight.direction );\n\t\tif ( angleCos > spotLight.coneCos ) {\n\t\t\tfloat spotEffect = smoothstep( spotLight.coneCos, spotLight.penumbraCos, angleCos );\n\t\t\tdirectLight.color = spotLight.color;\n\t\t\tdirectLight.color *= spotEffect * punctualLightIntensityToIrradianceFactor( lightDistance, spotLight.distance, spotLight.decay );\n\t\t\tdirectLight.visible = true;\n\t\t} else {\n\t\t\tdirectLight.color = vec3( 0.0 );\n\t\t\tdirectLight.visible = false;\n\t\t}\n\t}\n#endif\n#if NUM_RECT_AREA_LIGHTS > 0\n\tstruct RectAreaLight {\n\t\tvec3 color;\n\t\tvec3 position;\n\t\tvec3 halfWidth;\n\t\tvec3 halfHeight;\n\t};\n\tuniform sampler2D ltc_1;\tuniform sampler2D ltc_2;\n\tuniform RectAreaLight rectAreaLights[ NUM_RECT_AREA_LIGHTS ];\n#endif\n#if NUM_HEMI_LIGHTS > 0\n\tstruct HemisphereLight {\n\t\tvec3 direction;\n\t\tvec3 skyColor;\n\t\tvec3 groundColor;\n\t};\n\tuniform HemisphereLight hemisphereLights[ NUM_HEMI_LIGHTS ];\n\tvec3 getHemisphereLightIrradiance( const in HemisphereLight hemiLight, const in GeometricContext geometry ) {\n\t\tfloat dotNL = dot( geometry.normal, hemiLight.direction );\n\t\tfloat hemiDiffuseWeight = 0.5 * dotNL + 0.5;\n\t\tvec3 irradiance = mix( hemiLight.groundColor, hemiLight.skyColor, hemiDiffuseWeight );\n\t\t#ifndef PHYSICALLY_CORRECT_LIGHTS\n\t\t\tirradiance *= PI;\n\t\t#endif\n\t\treturn irradiance;\n\t}\n#endif&quot;,
lights_toon_fragment:&quot;ToonMaterial material;\nmaterial.diffuseColor = diffuseColor.rgb;\nmaterial.specularColor = specular;\nmaterial.specularShininess = shininess;\nmaterial.specularStrength = specularStrength;&quot;,lights_toon_pars_fragment:&quot;varying vec3 vViewPosition;\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n#endif\nstruct ToonMaterial {\n\tvec3\tdiffuseColor;\n\tvec3\tspecularColor;\n\tfloat\tspecularShininess;\n\tfloat\tspecularStrength;\n};\nvoid RE_Direct_Toon( const in IncidentLight directLight, const in GeometricContext geometry, const in ToonMaterial material, inout ReflectedLight reflectedLight ) {\n\tvec3 irradiance = getGradientIrradiance( geometry.normal, directLight.direction ) * directLight.color;\n\t#ifndef PHYSICALLY_CORRECT_LIGHTS\n\t\tirradiance *= PI;\n\t#endif\n\treflectedLight.directDiffuse += irradiance * BRDF_Diffuse_Lambert( material.diffuseColor );\n\treflectedLight.directSpecular += irradiance * BRDF_Specular_BlinnPhong( directLight, geometry, material.specularColor, material.specularShininess ) * material.specularStrength;\n}\nvoid RE_IndirectDiffuse_Toon( const in vec3 irradiance, const in GeometricContext geometry, const in ToonMaterial material, inout ReflectedLight reflectedLight ) {\n\treflectedLight.indirectDiffuse += irradiance * BRDF_Diffuse_Lambert( material.diffuseColor );\n}\n#define RE_Direct\t\t\t\tRE_Direct_Toon\n#define RE_IndirectDiffuse\t\tRE_IndirectDiffuse_Toon\n#define Material_LightProbeLOD( material )\t(0)&quot;,
lights_phong_fragment:&quot;BlinnPhongMaterial material;\nmaterial.diffuseColor = diffuseColor.rgb;\nmaterial.specularColor = specular;\nmaterial.specularShininess = shininess;\nmaterial.specularStrength = specularStrength;&quot;,lights_phong_pars_fragment:&quot;varying vec3 vViewPosition;\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n#endif\nstruct BlinnPhongMaterial {\n\tvec3\tdiffuseColor;\n\tvec3\tspecularColor;\n\tfloat\tspecularShininess;\n\tfloat\tspecularStrength;\n};\nvoid RE_Direct_BlinnPhong( const in IncidentLight directLight, const in GeometricContext geometry, const in BlinnPhongMaterial material, inout ReflectedLight reflectedLight ) {\n\tfloat dotNL = saturate( dot( geometry.normal, directLight.direction ) );\n\tvec3 irradiance = dotNL * directLight.color;\n\t#ifndef PHYSICALLY_CORRECT_LIGHTS\n\t\tirradiance *= PI;\n\t#endif\n\treflectedLight.directDiffuse += irradiance * BRDF_Diffuse_Lambert( material.diffuseColor );\n\treflectedLight.directSpecular += irradiance * BRDF_Specular_BlinnPhong( directLight, geometry, material.specularColor, material.specularShininess ) * material.specularStrength;\n}\nvoid RE_IndirectDiffuse_BlinnPhong( const in vec3 irradiance, const in GeometricContext geometry, const in BlinnPhongMaterial material, inout ReflectedLight reflectedLight ) {\n\treflectedLight.indirectDiffuse += irradiance * BRDF_Diffuse_Lambert( material.diffuseColor );\n}\n#define RE_Direct\t\t\t\tRE_Direct_BlinnPhong\n#define RE_IndirectDiffuse\t\tRE_IndirectDiffuse_BlinnPhong\n#define Material_LightProbeLOD( material )\t(0)&quot;,
lights_physical_fragment:&quot;PhysicalMaterial material;\nmaterial.diffuseColor = diffuseColor.rgb * ( 1.0 - metalnessFactor );\nvec3 dxy = max( abs( dFdx( geometryNormal ) ), abs( dFdy( geometryNormal ) ) );\nfloat geometryRoughness = max( max( dxy.x, dxy.y ), dxy.z );\nmaterial.specularRoughness = max( roughnessFactor, 0.0525 );material.specularRoughness += geometryRoughness;\nmaterial.specularRoughness = min( material.specularRoughness, 1.0 );\n#ifdef REFLECTIVITY\n\tmaterial.specularColor = mix( vec3( MAXIMUM_SPECULAR_COEFFICIENT * pow2( reflectivity ) ), diffuseColor.rgb, metalnessFactor );\n#else\n\tmaterial.specularColor = mix( vec3( DEFAULT_SPECULAR_COEFFICIENT ), diffuseColor.rgb, metalnessFactor );\n#endif\n#ifdef CLEARCOAT\n\tmaterial.clearcoat = clearcoat;\n\tmaterial.clearcoatRoughness = clearcoatRoughness;\n\t#ifdef USE_CLEARCOATMAP\n\t\tmaterial.clearcoat *= texture2D( clearcoatMap, vUv ).x;\n\t#endif\n\t#ifdef USE_CLEARCOAT_ROUGHNESSMAP\n\t\tmaterial.clearcoatRoughness *= texture2D( clearcoatRoughnessMap, vUv ).y;\n\t#endif\n\tmaterial.clearcoat = saturate( material.clearcoat );\tmaterial.clearcoatRoughness = max( material.clearcoatRoughness, 0.0525 );\n\tmaterial.clearcoatRoughness += geometryRoughness;\n\tmaterial.clearcoatRoughness = min( material.clearcoatRoughness, 1.0 );\n#endif\n#ifdef USE_SHEEN\n\tmaterial.sheenColor = sheen;\n#endif&quot;,
lights_physical_pars_fragment:&quot;struct PhysicalMaterial {\n\tvec3\tdiffuseColor;\n\tfloat\tspecularRoughness;\n\tvec3\tspecularColor;\n#ifdef CLEARCOAT\n\tfloat clearcoat;\n\tfloat clearcoatRoughness;\n#endif\n#ifdef USE_SHEEN\n\tvec3 sheenColor;\n#endif\n};\n#define MAXIMUM_SPECULAR_COEFFICIENT 0.16\n#define DEFAULT_SPECULAR_COEFFICIENT 0.04\nfloat clearcoatDHRApprox( const in float roughness, const in float dotNL ) {\n\treturn DEFAULT_SPECULAR_COEFFICIENT + ( 1.0 - DEFAULT_SPECULAR_COEFFICIENT ) * ( pow( 1.0 - dotNL, 5.0 ) * pow( 1.0 - roughness, 2.0 ) );\n}\n#if NUM_RECT_AREA_LIGHTS > 0\n\tvoid RE_Direct_RectArea_Physical( const in RectAreaLight rectAreaLight, const in GeometricContext geometry, const in PhysicalMaterial material, inout ReflectedLight reflectedLight ) {\n\t\tvec3 normal = geometry.normal;\n\t\tvec3 viewDir = geometry.viewDir;\n\t\tvec3 position = geometry.position;\n\t\tvec3 lightPos = rectAreaLight.position;\n\t\tvec3 halfWidth = rectAreaLight.halfWidth;\n\t\tvec3 halfHeight = rectAreaLight.halfHeight;\n\t\tvec3 lightColor = rectAreaLight.color;\n\t\tfloat roughness = material.specularRoughness;\n\t\tvec3 rectCoords[ 4 ];\n\t\trectCoords[ 0 ] = lightPos + halfWidth - halfHeight;\t\trectCoords[ 1 ] = lightPos - halfWidth - halfHeight;\n\t\trectCoords[ 2 ] = lightPos - halfWidth + halfHeight;\n\t\trectCoords[ 3 ] = lightPos + halfWidth + halfHeight;\n\t\tvec2 uv = LTC_Uv( normal, viewDir, roughness );\n\t\tvec4 t1 = texture2D( ltc_1, uv );\n\t\tvec4 t2 = texture2D( ltc_2, uv );\n\t\tmat3 mInv = mat3(\n\t\t\tvec3( t1.x, 0, t1.y ),\n\t\t\tvec3(    0, 1,    0 ),\n\t\t\tvec3( t1.z, 0, t1.w )\n\t\t);\n\t\tvec3 fresnel = ( material.specularColor * t2.x + ( vec3( 1.0 ) - material.specularColor ) * t2.y );\n\t\treflectedLight.directSpecular += lightColor * fresnel * LTC_Evaluate( normal, viewDir, position, mInv, rectCoords );\n\t\treflectedLight.directDiffuse += lightColor * material.diffuseColor * LTC_Evaluate( normal, viewDir, position, mat3( 1.0 ), rectCoords );\n\t}\n#endif\nvoid RE_Direct_Physical( const in IncidentLight directLight, const in GeometricContext geometry, const in PhysicalMaterial material, inout ReflectedLight reflectedLight ) {\n\tfloat dotNL = saturate( dot( geometry.normal, directLight.direction ) );\n\tvec3 irradiance = dotNL * directLight.color;\n\t#ifndef PHYSICALLY_CORRECT_LIGHTS\n\t\tirradiance *= PI;\n\t#endif\n\t#ifdef CLEARCOAT\n\t\tfloat ccDotNL = saturate( dot( geometry.clearcoatNormal, directLight.direction ) );\n\t\tvec3 ccIrradiance = ccDotNL * directLight.color;\n\t\t#ifndef PHYSICALLY_CORRECT_LIGHTS\n\t\t\tccIrradiance *= PI;\n\t\t#endif\n\t\tfloat clearcoatDHR = material.clearcoat * clearcoatDHRApprox( material.clearcoatRoughness, ccDotNL );\n\t\treflectedLight.directSpecular += ccIrradiance * material.clearcoat * BRDF_Specular_GGX( directLight, geometry.viewDir, geometry.clearcoatNormal, vec3( DEFAULT_SPECULAR_COEFFICIENT ), material.clearcoatRoughness );\n\t#else\n\t\tfloat clearcoatDHR = 0.0;\n\t#endif\n\t#ifdef USE_SHEEN\n\t\treflectedLight.directSpecular += ( 1.0 - clearcoatDHR ) * irradiance * BRDF_Specular_Sheen(\n\t\t\tmaterial.specularRoughness,\n\t\t\tdirectLight.direction,\n\t\t\tgeometry,\n\t\t\tmaterial.sheenColor\n\t\t);\n\t#else\n\t\treflectedLight.directSpecular += ( 1.0 - clearcoatDHR ) * irradiance * BRDF_Specular_GGX( directLight, geometry.viewDir, geometry.normal, material.specularColor, material.specularRoughness);\n\t#endif\n\treflectedLight.directDiffuse += ( 1.0 - clearcoatDHR ) * irradiance * BRDF_Diffuse_Lambert( material.diffuseColor );\n}\nvoid RE_IndirectDiffuse_Physical( const in vec3 irradiance, const in GeometricContext geometry, const in PhysicalMaterial material, inout ReflectedLight reflectedLight ) {\n\treflectedLight.indirectDiffuse += irradiance * BRDF_Diffuse_Lambert( material.diffuseColor );\n}\nvoid RE_IndirectSpecular_Physical( const in vec3 radiance, const in vec3 irradiance, const in vec3 clearcoatRadiance, const in GeometricContext geometry, const in PhysicalMaterial material, inout ReflectedLight reflectedLight) {\n\t#ifdef CLEARCOAT\n\t\tfloat ccDotNV = saturate( dot( geometry.clearcoatNormal, geometry.viewDir ) );\n\t\treflectedLight.indirectSpecular += clearcoatRadiance * material.clearcoat * BRDF_Specular_GGX_Environment( geometry.viewDir, geometry.clearcoatNormal, vec3( DEFAULT_SPECULAR_COEFFICIENT ), material.clearcoatRoughness );\n\t\tfloat ccDotNL = ccDotNV;\n\t\tfloat clearcoatDHR = material.clearcoat * clearcoatDHRApprox( material.clearcoatRoughness, ccDotNL );\n\t#else\n\t\tfloat clearcoatDHR = 0.0;\n\t#endif\n\tfloat clearcoatInv = 1.0 - clearcoatDHR;\n\tvec3 singleScattering = vec3( 0.0 );\n\tvec3 multiScattering = vec3( 0.0 );\n\tvec3 cosineWeightedIrradiance = irradiance * RECIPROCAL_PI;\n\tBRDF_Specular_Multiscattering_Environment( geometry, material.specularColor, material.specularRoughness, singleScattering, multiScattering );\n\tvec3 diffuse = material.diffuseColor * ( 1.0 - ( singleScattering + multiScattering ) );\n\treflectedLight.indirectSpecular += clearcoatInv * radiance * singleScattering;\n\treflectedLight.indirectSpecular += multiScattering * cosineWeightedIrradiance;\n\treflectedLight.indirectDiffuse += diffuse * cosineWeightedIrradiance;\n}\n#define RE_Direct\t\t\t\tRE_Direct_Physical\n#define RE_Direct_RectArea\t\tRE_Direct_RectArea_Physical\n#define RE_IndirectDiffuse\t\tRE_IndirectDiffuse_Physical\n#define RE_IndirectSpecular\t\tRE_IndirectSpecular_Physical\nfloat computeSpecularOcclusion( const in float dotNV, const in float ambientOcclusion, const in float roughness ) {\n\treturn saturate( pow( dotNV + ambientOcclusion, exp2( - 16.0 * roughness - 1.0 ) ) - 1.0 + ambientOcclusion );\n}&quot;,
lights_fragment_begin:&quot;\nGeometricContext geometry;\ngeometry.position = - vViewPosition;\ngeometry.normal = normal;\ngeometry.viewDir = ( isOrthographic ) ? vec3( 0, 0, 1 ) : normalize( vViewPosition );\n#ifdef CLEARCOAT\n\tgeometry.clearcoatNormal = clearcoatNormal;\n#endif\nIncidentLight directLight;\n#if ( NUM_POINT_LIGHTS > 0 ) && defined( RE_Direct )\n\tPointLight pointLight;\n\t#if defined( USE_SHADOWMAP ) && NUM_POINT_LIGHT_SHADOWS > 0\n\tPointLightShadow pointLightShadow;\n\t#endif\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_POINT_LIGHTS; i ++ ) {\n\t\tpointLight = pointLights[ i ];\n\t\tgetPointDirectLightIrradiance( pointLight, geometry, directLight );\n\t\t#if defined( USE_SHADOWMAP ) && ( UNROLLED_LOOP_INDEX < NUM_POINT_LIGHT_SHADOWS )\n\t\tpointLightShadow = pointLightShadows[ i ];\n\t\tdirectLight.color *= all( bvec2( directLight.visible, receiveShadow ) ) ? getPointShadow( pointShadowMap[ i ], pointLightShadow.shadowMapSize, pointLightShadow.shadowBias, pointLightShadow.shadowRadius, vPointShadowCoord[ i ], pointLightShadow.shadowCameraNear, pointLightShadow.shadowCameraFar ) : 1.0;\n\t\t#endif\n\t\tRE_Direct( directLight, geometry, material, reflectedLight );\n\t}\n\t#pragma unroll_loop_end\n#endif\n#if ( NUM_SPOT_LIGHTS > 0 ) && defined( RE_Direct )\n\tSpotLight spotLight;\n\t#if defined( USE_SHADOWMAP ) && NUM_SPOT_LIGHT_SHADOWS > 0\n\tSpotLightShadow spotLightShadow;\n\t#endif\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_SPOT_LIGHTS; i ++ ) {\n\t\tspotLight = spotLights[ i ];\n\t\tgetSpotDirectLightIrradiance( spotLight, geometry, directLight );\n\t\t#if defined( USE_SHADOWMAP ) && ( UNROLLED_LOOP_INDEX < NUM_SPOT_LIGHT_SHADOWS )\n\t\tspotLightShadow = spotLightShadows[ i ];\n\t\tdirectLight.color *= all( bvec2( directLight.visible, receiveShadow ) ) ? getShadow( spotShadowMap[ i ], spotLightShadow.shadowMapSize, spotLightShadow.shadowBias, spotLightShadow.shadowRadius, vSpotShadowCoord[ i ] ) : 1.0;\n\t\t#endif\n\t\tRE_Direct( directLight, geometry, material, reflectedLight );\n\t}\n\t#pragma unroll_loop_end\n#endif\n#if ( NUM_DIR_LIGHTS > 0 ) && defined( RE_Direct )\n\tDirectionalLight directionalLight;\n\t#if defined( USE_SHADOWMAP ) && NUM_DIR_LIGHT_SHADOWS > 0\n\tDirectionalLightShadow directionalLightShadow;\n\t#endif\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_DIR_LIGHTS; i ++ ) {\n\t\tdirectionalLight = directionalLights[ i ];\n\t\tgetDirectionalDirectLightIrradiance( directionalLight, geometry, directLight );\n\t\t#if defined( USE_SHADOWMAP ) && ( UNROLLED_LOOP_INDEX < NUM_DIR_LIGHT_SHADOWS )\n\t\tdirectionalLightShadow = directionalLightShadows[ i ];\n\t\tdirectLight.color *= all( bvec2( directLight.visible, receiveShadow ) ) ? getShadow( directionalShadowMap[ i ], directionalLightShadow.shadowMapSize, directionalLightShadow.shadowBias, directionalLightShadow.shadowRadius, vDirectionalShadowCoord[ i ] ) : 1.0;\n\t\t#endif\n\t\tRE_Direct( directLight, geometry, material, reflectedLight );\n\t}\n\t#pragma unroll_loop_end\n#endif\n#if ( NUM_RECT_AREA_LIGHTS > 0 ) && defined( RE_Direct_RectArea )\n\tRectAreaLight rectAreaLight;\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_RECT_AREA_LIGHTS; i ++ ) {\n\t\trectAreaLight = rectAreaLights[ i ];\n\t\tRE_Direct_RectArea( rectAreaLight, geometry, material, reflectedLight );\n\t}\n\t#pragma unroll_loop_end\n#endif\n#if defined( RE_IndirectDiffuse )\n\tvec3 iblIrradiance = vec3( 0.0 );\n\tvec3 irradiance = getAmbientLightIrradiance( ambientLightColor );\n\tirradiance += getLightProbeIrradiance( lightProbe, geometry );\n\t#if ( NUM_HEMI_LIGHTS > 0 )\n\t\t#pragma unroll_loop_start\n\t\tfor ( int i = 0; i < NUM_HEMI_LIGHTS; i ++ ) {\n\t\t\tirradiance += getHemisphereLightIrradiance( hemisphereLights[ i ], geometry );\n\t\t}\n\t\t#pragma unroll_loop_end\n\t#endif\n#endif\n#if defined( RE_IndirectSpecular )\n\tvec3 radiance = vec3( 0.0 );\n\tvec3 clearcoatRadiance = vec3( 0.0 );\n#endif&quot;,
lights_fragment_maps:&quot;#if defined( RE_IndirectDiffuse )\n\t#ifdef USE_LIGHTMAP\n\t\tvec4 lightMapTexel= texture2D( lightMap, vUv2 );\n\t\tvec3 lightMapIrradiance = lightMapTexelToLinear( lightMapTexel ).rgb * lightMapIntensity;\n\t\t#ifndef PHYSICALLY_CORRECT_LIGHTS\n\t\t\tlightMapIrradiance *= PI;\n\t\t#endif\n\t\tirradiance += lightMapIrradiance;\n\t#endif\n\t#if defined( USE_ENVMAP ) && defined( STANDARD ) && defined( ENVMAP_TYPE_CUBE_UV )\n\t\tiblIrradiance += getLightProbeIndirectIrradiance( geometry, maxMipLevel );\n\t#endif\n#endif\n#if defined( USE_ENVMAP ) && defined( RE_IndirectSpecular )\n\tradiance += getLightProbeIndirectRadiance( geometry.viewDir, geometry.normal, material.specularRoughness, maxMipLevel );\n\t#ifdef CLEARCOAT\n\t\tclearcoatRadiance += getLightProbeIndirectRadiance( geometry.viewDir, geometry.clearcoatNormal, material.clearcoatRoughness, maxMipLevel );\n\t#endif\n#endif&quot;,
lights_fragment_end:&quot;#if defined( RE_IndirectDiffuse )\n\tRE_IndirectDiffuse( irradiance, geometry, material, reflectedLight );\n#endif\n#if defined( RE_IndirectSpecular )\n\tRE_IndirectSpecular( radiance, iblIrradiance, clearcoatRadiance, geometry, material, reflectedLight );\n#endif&quot;,logdepthbuf_fragment:&quot;#if defined( USE_LOGDEPTHBUF ) && defined( USE_LOGDEPTHBUF_EXT )\n\tgl_FragDepthEXT = vIsPerspective == 0.0 ? gl_FragCoord.z : log2( vFragDepth ) * logDepthBufFC * 0.5;\n#endif&quot;,logdepthbuf_pars_fragment:&quot;#if defined( USE_LOGDEPTHBUF ) && defined( USE_LOGDEPTHBUF_EXT )\n\tuniform float logDepthBufFC;\n\tvarying float vFragDepth;\n\tvarying float vIsPerspective;\n#endif&quot;,
logdepthbuf_pars_vertex:&quot;#ifdef USE_LOGDEPTHBUF\n\t#ifdef USE_LOGDEPTHBUF_EXT\n\t\tvarying float vFragDepth;\n\t\tvarying float vIsPerspective;\n\t#else\n\t\tuniform float logDepthBufFC;\n\t#endif\n#endif&quot;,logdepthbuf_vertex:&quot;#ifdef USE_LOGDEPTHBUF\n\t#ifdef USE_LOGDEPTHBUF_EXT\n\t\tvFragDepth = 1.0 + gl_Position.w;\n\t\tvIsPerspective = float( isPerspectiveMatrix( projectionMatrix ) );\n\t#else\n\t\tif ( isPerspectiveMatrix( projectionMatrix ) ) {\n\t\t\tgl_Position.z = log2( max( EPSILON, gl_Position.w + 1.0 ) ) * logDepthBufFC - 1.0;\n\t\t\tgl_Position.z *= gl_Position.w;\n\t\t}\n\t#endif\n#endif&quot;,
map_fragment:&quot;#ifdef USE_MAP\n\tvec4 texelColor = texture2D( map, vUv );\n\ttexelColor = mapTexelToLinear( texelColor );\n\tdiffuseColor *= texelColor;\n#endif&quot;,map_pars_fragment:&quot;#ifdef USE_MAP\n\tuniform sampler2D map;\n#endif&quot;,map_particle_fragment:&quot;#if defined( USE_MAP ) || defined( USE_ALPHAMAP )\n\tvec2 uv = ( uvTransform * vec3( gl_PointCoord.x, 1.0 - gl_PointCoord.y, 1 ) ).xy;\n#endif\n#ifdef USE_MAP\n\tvec4 mapTexel = texture2D( map, uv );\n\tdiffuseColor *= mapTexelToLinear( mapTexel );\n#endif\n#ifdef USE_ALPHAMAP\n\tdiffuseColor.a *= texture2D( alphaMap, uv ).g;\n#endif&quot;,
map_particle_pars_fragment:&quot;#if defined( USE_MAP ) || defined( USE_ALPHAMAP )\n\tuniform mat3 uvTransform;\n#endif\n#ifdef USE_MAP\n\tuniform sampler2D map;\n#endif\n#ifdef USE_ALPHAMAP\n\tuniform sampler2D alphaMap;\n#endif&quot;,metalnessmap_fragment:&quot;float metalnessFactor = metalness;\n#ifdef USE_METALNESSMAP\n\tvec4 texelMetalness = texture2D( metalnessMap, vUv );\n\tmetalnessFactor *= texelMetalness.b;\n#endif&quot;,metalnessmap_pars_fragment:&quot;#ifdef USE_METALNESSMAP\n\tuniform sampler2D metalnessMap;\n#endif&quot;,
morphnormal_vertex:&quot;#ifdef USE_MORPHNORMALS\n\tobjectNormal *= morphTargetBaseInfluence;\n\tobjectNormal += morphNormal0 * morphTargetInfluences[ 0 ];\n\tobjectNormal += morphNormal1 * morphTargetInfluences[ 1 ];\n\tobjectNormal += morphNormal2 * morphTargetInfluences[ 2 ];\n\tobjectNormal += morphNormal3 * morphTargetInfluences[ 3 ];\n#endif&quot;,morphtarget_pars_vertex:&quot;#ifdef USE_MORPHTARGETS\n\tuniform float morphTargetBaseInfluence;\n\t#ifndef USE_MORPHNORMALS\n\tuniform float morphTargetInfluences[ 8 ];\n\t#else\n\tuniform float morphTargetInfluences[ 4 ];\n\t#endif\n#endif&quot;,
morphtarget_vertex:&quot;#ifdef USE_MORPHTARGETS\n\ttransformed *= morphTargetBaseInfluence;\n\ttransformed += morphTarget0 * morphTargetInfluences[ 0 ];\n\ttransformed += morphTarget1 * morphTargetInfluences[ 1 ];\n\ttransformed += morphTarget2 * morphTargetInfluences[ 2 ];\n\ttransformed += morphTarget3 * morphTargetInfluences[ 3 ];\n\t#ifndef USE_MORPHNORMALS\n\ttransformed += morphTarget4 * morphTargetInfluences[ 4 ];\n\ttransformed += morphTarget5 * morphTargetInfluences[ 5 ];\n\ttransformed += morphTarget6 * morphTargetInfluences[ 6 ];\n\ttransformed += morphTarget7 * morphTargetInfluences[ 7 ];\n\t#endif\n#endif&quot;,
normal_fragment_begin:&quot;#ifdef FLAT_SHADED\n\tvec3 fdx = vec3( dFdx( vViewPosition.x ), dFdx( vViewPosition.y ), dFdx( vViewPosition.z ) );\n\tvec3 fdy = vec3( dFdy( vViewPosition.x ), dFdy( vViewPosition.y ), dFdy( vViewPosition.z ) );\n\tvec3 normal = normalize( cross( fdx, fdy ) );\n#else\n\tvec3 normal = normalize( vNormal );\n\t#ifdef DOUBLE_SIDED\n\t\tnormal = normal * ( float( gl_FrontFacing ) * 2.0 - 1.0 );\n\t#endif\n\t#ifdef USE_TANGENT\n\t\tvec3 tangent = normalize( vTangent );\n\t\tvec3 bitangent = normalize( vBitangent );\n\t\t#ifdef DOUBLE_SIDED\n\t\t\ttangent = tangent * ( float( gl_FrontFacing ) * 2.0 - 1.0 );\n\t\t\tbitangent = bitangent * ( float( gl_FrontFacing ) * 2.0 - 1.0 );\n\t\t#endif\n\t\t#if defined( TANGENTSPACE_NORMALMAP ) || defined( USE_CLEARCOAT_NORMALMAP )\n\t\t\tmat3 vTBN = mat3( tangent, bitangent, normal );\n\t\t#endif\n\t#endif\n#endif\nvec3 geometryNormal = normal;&quot;,
normal_fragment_maps:&quot;#ifdef OBJECTSPACE_NORMALMAP\n\tnormal = texture2D( normalMap, vUv ).xyz * 2.0 - 1.0;\n\t#ifdef FLIP_SIDED\n\t\tnormal = - normal;\n\t#endif\n\t#ifdef DOUBLE_SIDED\n\t\tnormal = normal * ( float( gl_FrontFacing ) * 2.0 - 1.0 );\n\t#endif\n\tnormal = normalize( normalMatrix * normal );\n#elif defined( TANGENTSPACE_NORMALMAP )\n\tvec3 mapN = texture2D( normalMap, vUv ).xyz * 2.0 - 1.0;\n\tmapN.xy *= normalScale;\n\t#ifdef USE_TANGENT\n\t\tnormal = normalize( vTBN * mapN );\n\t#else\n\t\tnormal = perturbNormal2Arb( -vViewPosition, normal, mapN );\n\t#endif\n#elif defined( USE_BUMPMAP )\n\tnormal = perturbNormalArb( -vViewPosition, normal, dHdxy_fwd() );\n#endif&quot;,
normalmap_pars_fragment:&quot;#ifdef USE_NORMALMAP\n\tuniform sampler2D normalMap;\n\tuniform vec2 normalScale;\n#endif\n#ifdef OBJECTSPACE_NORMALMAP\n\tuniform mat3 normalMatrix;\n#endif\n#if ! defined ( USE_TANGENT ) && ( defined ( TANGENTSPACE_NORMALMAP ) || defined ( USE_CLEARCOAT_NORMALMAP ) )\n\tvec3 perturbNormal2Arb( vec3 eye_pos, vec3 surf_norm, vec3 mapN ) {\n\t\tvec3 q0 = vec3( dFdx( eye_pos.x ), dFdx( eye_pos.y ), dFdx( eye_pos.z ) );\n\t\tvec3 q1 = vec3( dFdy( eye_pos.x ), dFdy( eye_pos.y ), dFdy( eye_pos.z ) );\n\t\tvec2 st0 = dFdx( vUv.st );\n\t\tvec2 st1 = dFdy( vUv.st );\n\t\tfloat scale = sign( st1.t * st0.s - st0.t * st1.s );\n\t\tvec3 S = normalize( ( q0 * st1.t - q1 * st0.t ) * scale );\n\t\tvec3 T = normalize( ( - q0 * st1.s + q1 * st0.s ) * scale );\n\t\tvec3 N = normalize( surf_norm );\n\t\tmat3 tsn = mat3( S, T, N );\n\t\tmapN.xy *= ( float( gl_FrontFacing ) * 2.0 - 1.0 );\n\t\treturn normalize( tsn * mapN );\n\t}\n#endif&quot;,
clearcoat_normal_fragment_begin:&quot;#ifdef CLEARCOAT\n\tvec3 clearcoatNormal = geometryNormal;\n#endif&quot;,clearcoat_normal_fragment_maps:&quot;#ifdef USE_CLEARCOAT_NORMALMAP\n\tvec3 clearcoatMapN = texture2D( clearcoatNormalMap, vUv ).xyz * 2.0 - 1.0;\n\tclearcoatMapN.xy *= clearcoatNormalScale;\n\t#ifdef USE_TANGENT\n\t\tclearcoatNormal = normalize( vTBN * clearcoatMapN );\n\t#else\n\t\tclearcoatNormal = perturbNormal2Arb( - vViewPosition, clearcoatNormal, clearcoatMapN );\n\t#endif\n#endif&quot;,clearcoat_pars_fragment:&quot;#ifdef USE_CLEARCOATMAP\n\tuniform sampler2D clearcoatMap;\n#endif\n#ifdef USE_CLEARCOAT_ROUGHNESSMAP\n\tuniform sampler2D clearcoatRoughnessMap;\n#endif\n#ifdef USE_CLEARCOAT_NORMALMAP\n\tuniform sampler2D clearcoatNormalMap;\n\tuniform vec2 clearcoatNormalScale;\n#endif&quot;,
packing:&quot;vec3 packNormalToRGB( const in vec3 normal ) {\n\treturn normalize( normal ) * 0.5 + 0.5;\n}\nvec3 unpackRGBToNormal( const in vec3 rgb ) {\n\treturn 2.0 * rgb.xyz - 1.0;\n}\nconst float PackUpscale = 256. / 255.;const float UnpackDownscale = 255. / 256.;\nconst vec3 PackFactors = vec3( 256. * 256. * 256., 256. * 256.,  256. );\nconst vec4 UnpackFactors = UnpackDownscale / vec4( PackFactors, 1. );\nconst float ShiftRight8 = 1. / 256.;\nvec4 packDepthToRGBA( const in float v ) {\n\tvec4 r = vec4( fract( v * PackFactors ), v );\n\tr.yzw -= r.xyz * ShiftRight8;\treturn r * PackUpscale;\n}\nfloat unpackRGBAToDepth( const in vec4 v ) {\n\treturn dot( v, UnpackFactors );\n}\nvec4 pack2HalfToRGBA( vec2 v ) {\n\tvec4 r = vec4( v.x, fract( v.x * 255.0 ), v.y, fract( v.y * 255.0 ));\n\treturn vec4( r.x - r.y / 255.0, r.y, r.z - r.w / 255.0, r.w);\n}\nvec2 unpackRGBATo2Half( vec4 v ) {\n\treturn vec2( v.x + ( v.y / 255.0 ), v.z + ( v.w / 255.0 ) );\n}\nfloat viewZToOrthographicDepth( const in float viewZ, const in float near, const in float far ) {\n\treturn ( viewZ + near ) / ( near - far );\n}\nfloat orthographicDepthToViewZ( const in float linearClipZ, const in float near, const in float far ) {\n\treturn linearClipZ * ( near - far ) - near;\n}\nfloat viewZToPerspectiveDepth( const in float viewZ, const in float near, const in float far ) {\n\treturn (( near + viewZ ) * far ) / (( far - near ) * viewZ );\n}\nfloat perspectiveDepthToViewZ( const in float invClipZ, const in float near, const in float far ) {\n\treturn ( near * far ) / ( ( far - near ) * invClipZ - far );\n}&quot;,
premultiplied_alpha_fragment:&quot;#ifdef PREMULTIPLIED_ALPHA\n\tgl_FragColor.rgb *= gl_FragColor.a;\n#endif&quot;,project_vertex:&quot;vec4 mvPosition = vec4( transformed, 1.0 );\n#ifdef USE_INSTANCING\n\tmvPosition = instanceMatrix * mvPosition;\n#endif\nmvPosition = modelViewMatrix * mvPosition;\ngl_Position = projectionMatrix * mvPosition;&quot;,dithering_fragment:&quot;#ifdef DITHERING\n\tgl_FragColor.rgb = dithering( gl_FragColor.rgb );\n#endif&quot;,dithering_pars_fragment:&quot;#ifdef DITHERING\n\tvec3 dithering( vec3 color ) {\n\t\tfloat grid_position = rand( gl_FragCoord.xy );\n\t\tvec3 dither_shift_RGB = vec3( 0.25 / 255.0, -0.25 / 255.0, 0.25 / 255.0 );\n\t\tdither_shift_RGB = mix( 2.0 * dither_shift_RGB, -2.0 * dither_shift_RGB, grid_position );\n\t\treturn color + dither_shift_RGB;\n\t}\n#endif&quot;,
roughnessmap_fragment:&quot;float roughnessFactor = roughness;\n#ifdef USE_ROUGHNESSMAP\n\tvec4 texelRoughness = texture2D( roughnessMap, vUv );\n\troughnessFactor *= texelRoughness.g;\n#endif&quot;,roughnessmap_pars_fragment:&quot;#ifdef USE_ROUGHNESSMAP\n\tuniform sampler2D roughnessMap;\n#endif&quot;,shadowmap_pars_fragment:&quot;#ifdef USE_SHADOWMAP\n\t#if NUM_DIR_LIGHT_SHADOWS > 0\n\t\tuniform sampler2D directionalShadowMap[ NUM_DIR_LIGHT_SHADOWS ];\n\t\tvarying vec4 vDirectionalShadowCoord[ NUM_DIR_LIGHT_SHADOWS ];\n\t#endif\n\t#if NUM_SPOT_LIGHT_SHADOWS > 0\n\t\tuniform sampler2D spotShadowMap[ NUM_SPOT_LIGHT_SHADOWS ];\n\t\tvarying vec4 vSpotShadowCoord[ NUM_SPOT_LIGHT_SHADOWS ];\n\t#endif\n\t#if NUM_POINT_LIGHT_SHADOWS > 0\n\t\tuniform sampler2D pointShadowMap[ NUM_POINT_LIGHT_SHADOWS ];\n\t\tvarying vec4 vPointShadowCoord[ NUM_POINT_LIGHT_SHADOWS ];\n\t#endif\n\tfloat texture2DCompare( sampler2D depths, vec2 uv, float compare ) {\n\t\treturn step( compare, unpackRGBAToDepth( texture2D( depths, uv ) ) );\n\t}\n\tvec2 texture2DDistribution( sampler2D shadow, vec2 uv ) {\n\t\treturn unpackRGBATo2Half( texture2D( shadow, uv ) );\n\t}\n\tfloat VSMShadow (sampler2D shadow, vec2 uv, float compare ){\n\t\tfloat occlusion = 1.0;\n\t\tvec2 distribution = texture2DDistribution( shadow, uv );\n\t\tfloat hard_shadow = step( compare , distribution.x );\n\t\tif (hard_shadow != 1.0 ) {\n\t\t\tfloat distance = compare - distribution.x ;\n\t\t\tfloat variance = max( 0.00000, distribution.y * distribution.y );\n\t\t\tfloat softness_probability = variance / (variance + distance * distance );\t\t\tsoftness_probability = clamp( ( softness_probability - 0.3 ) / ( 0.95 - 0.3 ), 0.0, 1.0 );\t\t\tocclusion = clamp( max( hard_shadow, softness_probability ), 0.0, 1.0 );\n\t\t}\n\t\treturn occlusion;\n\t}\n\tfloat getShadow( sampler2D shadowMap, vec2 shadowMapSize, float shadowBias, float shadowRadius, vec4 shadowCoord ) {\n\t\tfloat shadow = 1.0;\n\t\tshadowCoord.xyz /= shadowCoord.w;\n\t\tshadowCoord.z += shadowBias;\n\t\tbvec4 inFrustumVec = bvec4 ( shadowCoord.x >= 0.0, shadowCoord.x <= 1.0, shadowCoord.y >= 0.0, shadowCoord.y <= 1.0 );\n\t\tbool inFrustum = all( inFrustumVec );\n\t\tbvec2 frustumTestVec = bvec2( inFrustum, shadowCoord.z <= 1.0 );\n\t\tbool frustumTest = all( frustumTestVec );\n\t\tif ( frustumTest ) {\n\t\t#if defined( SHADOWMAP_TYPE_PCF )\n\t\t\tvec2 texelSize = vec2( 1.0 ) / shadowMapSize;\n\t\t\tfloat dx0 = - texelSize.x * shadowRadius;\n\t\t\tfloat dy0 = - texelSize.y * shadowRadius;\n\t\t\tfloat dx1 = + texelSize.x * shadowRadius;\n\t\t\tfloat dy1 = + texelSize.y * shadowRadius;\n\t\t\tfloat dx2 = dx0 / 2.0;\n\t\t\tfloat dy2 = dy0 / 2.0;\n\t\t\tfloat dx3 = dx1 / 2.0;\n\t\t\tfloat dy3 = dy1 / 2.0;\n\t\t\tshadow = (\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx0, dy0 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( 0.0, dy0 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx1, dy0 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx2, dy2 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( 0.0, dy2 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx3, dy2 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx0, 0.0 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx2, 0.0 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy, shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx3, 0.0 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx1, 0.0 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx2, dy3 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( 0.0, dy3 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx3, dy3 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx0, dy1 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( 0.0, dy1 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, shadowCoord.xy + vec2( dx1, dy1 ), shadowCoord.z )\n\t\t\t) * ( 1.0 / 17.0 );\n\t\t#elif defined( SHADOWMAP_TYPE_PCF_SOFT )\n\t\t\tvec2 texelSize = vec2( 1.0 ) / shadowMapSize;\n\t\t\tfloat dx = texelSize.x;\n\t\t\tfloat dy = texelSize.y;\n\t\t\tvec2 uv = shadowCoord.xy;\n\t\t\tvec2 f = fract( uv * shadowMapSize + 0.5 );\n\t\t\tuv -= f * texelSize;\n\t\t\tshadow = (\n\t\t\t\ttexture2DCompare( shadowMap, uv, shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, uv + vec2( dx, 0.0 ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, uv + vec2( 0.0, dy ), shadowCoord.z ) +\n\t\t\t\ttexture2DCompare( shadowMap, uv + texelSize, shadowCoord.z ) +\n\t\t\t\tmix( texture2DCompare( shadowMap, uv + vec2( -dx, 0.0 ), shadowCoord.z ), \n\t\t\t\t\t texture2DCompare( shadowMap, uv + vec2( 2.0 * dx, 0.0 ), shadowCoord.z ),\n\t\t\t\t\t f.x ) +\n\t\t\t\tmix( texture2DCompare( shadowMap, uv + vec2( -dx, dy ), shadowCoord.z ), \n\t\t\t\t\t texture2DCompare( shadowMap, uv + vec2( 2.0 * dx, dy ), shadowCoord.z ),\n\t\t\t\t\t f.x ) +\n\t\t\t\tmix( texture2DCompare( shadowMap, uv + vec2( 0.0, -dy ), shadowCoord.z ), \n\t\t\t\t\t texture2DCompare( shadowMap, uv + vec2( 0.0, 2.0 * dy ), shadowCoord.z ),\n\t\t\t\t\t f.y ) +\n\t\t\t\tmix( texture2DCompare( shadowMap, uv + vec2( dx, -dy ), shadowCoord.z ), \n\t\t\t\t\t texture2DCompare( shadowMap, uv + vec2( dx, 2.0 * dy ), shadowCoord.z ),\n\t\t\t\t\t f.y ) +\n\t\t\t\tmix( mix( texture2DCompare( shadowMap, uv + vec2( -dx, -dy ), shadowCoord.z ), \n\t\t\t\t\t\t  texture2DCompare( shadowMap, uv + vec2( 2.0 * dx, -dy ), shadowCoord.z ),\n\t\t\t\t\t\t  f.x ),\n\t\t\t\t\t mix( texture2DCompare( shadowMap, uv + vec2( -dx, 2.0 * dy ), shadowCoord.z ), \n\t\t\t\t\t\t  texture2DCompare( shadowMap, uv + vec2( 2.0 * dx, 2.0 * dy ), shadowCoord.z ),\n\t\t\t\t\t\t  f.x ),\n\t\t\t\t\t f.y )\n\t\t\t) * ( 1.0 / 9.0 );\n\t\t#elif defined( SHADOWMAP_TYPE_VSM )\n\t\t\tshadow = VSMShadow( shadowMap, shadowCoord.xy, shadowCoord.z );\n\t\t#else\n\t\t\tshadow = texture2DCompare( shadowMap, shadowCoord.xy, shadowCoord.z );\n\t\t#endif\n\t\t}\n\t\treturn shadow;\n\t}\n\tvec2 cubeToUV( vec3 v, float texelSizeY ) {\n\t\tvec3 absV = abs( v );\n\t\tfloat scaleToCube = 1.0 / max( absV.x, max( absV.y, absV.z ) );\n\t\tabsV *= scaleToCube;\n\t\tv *= scaleToCube * ( 1.0 - 2.0 * texelSizeY );\n\t\tvec2 planar = v.xy;\n\t\tfloat almostATexel = 1.5 * texelSizeY;\n\t\tfloat almostOne = 1.0 - almostATexel;\n\t\tif ( absV.z >= almostOne ) {\n\t\t\tif ( v.z > 0.0 )\n\t\t\t\tplanar.x = 4.0 - v.x;\n\t\t} else if ( absV.x >= almostOne ) {\n\t\t\tfloat signX = sign( v.x );\n\t\t\tplanar.x = v.z * signX + 2.0 * signX;\n\t\t} else if ( absV.y >= almostOne ) {\n\t\t\tfloat signY = sign( v.y );\n\t\t\tplanar.x = v.x + 2.0 * signY + 2.0;\n\t\t\tplanar.y = v.z * signY - 2.0;\n\t\t}\n\t\treturn vec2( 0.125, 0.25 ) * planar + vec2( 0.375, 0.75 );\n\t}\n\tfloat getPointShadow( sampler2D shadowMap, vec2 shadowMapSize, float shadowBias, float shadowRadius, vec4 shadowCoord, float shadowCameraNear, float shadowCameraFar ) {\n\t\tvec2 texelSize = vec2( 1.0 ) / ( shadowMapSize * vec2( 4.0, 2.0 ) );\n\t\tvec3 lightToPosition = shadowCoord.xyz;\n\t\tfloat dp = ( length( lightToPosition ) - shadowCameraNear ) / ( shadowCameraFar - shadowCameraNear );\t\tdp += shadowBias;\n\t\tvec3 bd3D = normalize( lightToPosition );\n\t\t#if defined( SHADOWMAP_TYPE_PCF ) || defined( SHADOWMAP_TYPE_PCF_SOFT ) || defined( SHADOWMAP_TYPE_VSM )\n\t\t\tvec2 offset = vec2( - 1, 1 ) * shadowRadius * texelSize.y;\n\t\t\treturn (\n\t\t\t\ttexture2DCompare( shadowMap, cubeToUV( bd3D + offset.xyy, texelSize.y ), dp ) +\n\t\t\t\ttexture2DCompare( shadowMap, cubeToUV( bd3D + offset.yyy, texelSize.y ), dp ) +\n\t\t\t\ttexture2DCompare( shadowMap, cubeToUV( bd3D + offset.xyx, texelSize.y ), dp ) +\n\t\t\t\ttexture2DCompare( shadowMap, cubeToUV( bd3D + offset.yyx, texelSize.y ), dp ) +\n\t\t\t\ttexture2DCompare( shadowMap, cubeToUV( bd3D, texelSize.y ), dp ) +\n\t\t\t\ttexture2DCompare( shadowMap, cubeToUV( bd3D + offset.xxy, texelSize.y ), dp ) +\n\t\t\t\ttexture2DCompare( shadowMap, cubeToUV( bd3D + offset.yxy, texelSize.y ), dp ) +\n\t\t\t\ttexture2DCompare( shadowMap, cubeToUV( bd3D + offset.xxx, texelSize.y ), dp ) +\n\t\t\t\ttexture2DCompare( shadowMap, cubeToUV( bd3D + offset.yxx, texelSize.y ), dp )\n\t\t\t) * ( 1.0 / 9.0 );\n\t\t#else\n\t\t\treturn texture2DCompare( shadowMap, cubeToUV( bd3D, texelSize.y ), dp );\n\t\t#endif\n\t}\n#endif&quot;,
shadowmap_pars_vertex:&quot;#ifdef USE_SHADOWMAP\n\t#if NUM_DIR_LIGHT_SHADOWS > 0\n\t\tuniform mat4 directionalShadowMatrix[ NUM_DIR_LIGHT_SHADOWS ];\n\t\tvarying vec4 vDirectionalShadowCoord[ NUM_DIR_LIGHT_SHADOWS ];\n\t#endif\n\t#if NUM_SPOT_LIGHT_SHADOWS > 0\n\t\tuniform mat4 spotShadowMatrix[ NUM_SPOT_LIGHT_SHADOWS ];\n\t\tvarying vec4 vSpotShadowCoord[ NUM_SPOT_LIGHT_SHADOWS ];\n\t#endif\n\t#if NUM_POINT_LIGHT_SHADOWS > 0\n\t\tuniform mat4 pointShadowMatrix[ NUM_POINT_LIGHT_SHADOWS ];\n\t\tvarying vec4 vPointShadowCoord[ NUM_POINT_LIGHT_SHADOWS ];\n\t#endif\n#endif&quot;,
shadowmap_vertex:&quot;#ifdef USE_SHADOWMAP\n\t#if NUM_DIR_LIGHT_SHADOWS > 0\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_DIR_LIGHT_SHADOWS; i ++ ) {\n\t\tvDirectionalShadowCoord[ i ] = directionalShadowMatrix[ i ] * worldPosition;\n\t}\n\t#pragma unroll_loop_end\n\t#endif\n\t#if NUM_SPOT_LIGHT_SHADOWS > 0\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_SPOT_LIGHT_SHADOWS; i ++ ) {\n\t\tvSpotShadowCoord[ i ] = spotShadowMatrix[ i ] * worldPosition;\n\t}\n\t#pragma unroll_loop_end\n\t#endif\n\t#if NUM_POINT_LIGHT_SHADOWS > 0\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_POINT_LIGHT_SHADOWS; i ++ ) {\n\t\tvPointShadowCoord[ i ] = pointShadowMatrix[ i ] * worldPosition;\n\t}\n\t#pragma unroll_loop_end\n\t#endif\n#endif&quot;,
shadowmask_pars_fragment:&quot;float getShadowMask() {\n\tfloat shadow = 1.0;\n\t#ifdef USE_SHADOWMAP\n\t#if NUM_DIR_LIGHT_SHADOWS > 0\n\tDirectionalLightShadow directionalLight;\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_DIR_LIGHT_SHADOWS; i ++ ) {\n\t\tdirectionalLight = directionalLightShadows[ i ];\n\t\tshadow *= receiveShadow ? getShadow( directionalShadowMap[ i ], directionalLight.shadowMapSize, directionalLight.shadowBias, directionalLight.shadowRadius, vDirectionalShadowCoord[ i ] ) : 1.0;\n\t}\n\t#pragma unroll_loop_end\n\t#endif\n\t#if NUM_SPOT_LIGHT_SHADOWS > 0\n\tSpotLightShadow spotLight;\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_SPOT_LIGHT_SHADOWS; i ++ ) {\n\t\tspotLight = spotLightShadows[ i ];\n\t\tshadow *= receiveShadow ? getShadow( spotShadowMap[ i ], spotLight.shadowMapSize, spotLight.shadowBias, spotLight.shadowRadius, vSpotShadowCoord[ i ] ) : 1.0;\n\t}\n\t#pragma unroll_loop_end\n\t#endif\n\t#if NUM_POINT_LIGHT_SHADOWS > 0\n\tPointLightShadow pointLight;\n\t#pragma unroll_loop_start\n\tfor ( int i = 0; i < NUM_POINT_LIGHT_SHADOWS; i ++ ) {\n\t\tpointLight = pointLightShadows[ i ];\n\t\tshadow *= receiveShadow ? getPointShadow( pointShadowMap[ i ], pointLight.shadowMapSize, pointLight.shadowBias, pointLight.shadowRadius, vPointShadowCoord[ i ], pointLight.shadowCameraNear, pointLight.shadowCameraFar ) : 1.0;\n\t}\n\t#pragma unroll_loop_end\n\t#endif\n\t#endif\n\treturn shadow;\n}&quot;,
skinbase_vertex:&quot;#ifdef USE_SKINNING\n\tmat4 boneMatX = getBoneMatrix( skinIndex.x );\n\tmat4 boneMatY = getBoneMatrix( skinIndex.y );\n\tmat4 boneMatZ = getBoneMatrix( skinIndex.z );\n\tmat4 boneMatW = getBoneMatrix( skinIndex.w );\n#endif&quot;,skinning_pars_vertex:&quot;#ifdef USE_SKINNING\n\tuniform mat4 bindMatrix;\n\tuniform mat4 bindMatrixInverse;\n\t#ifdef BONE_TEXTURE\n\t\tuniform highp sampler2D boneTexture;\n\t\tuniform int boneTextureSize;\n\t\tmat4 getBoneMatrix( const in float i ) {\n\t\t\tfloat j = i * 4.0;\n\t\t\tfloat x = mod( j, float( boneTextureSize ) );\n\t\t\tfloat y = floor( j / float( boneTextureSize ) );\n\t\t\tfloat dx = 1.0 / float( boneTextureSize );\n\t\t\tfloat dy = 1.0 / float( boneTextureSize );\n\t\t\ty = dy * ( y + 0.5 );\n\t\t\tvec4 v1 = texture2D( boneTexture, vec2( dx * ( x + 0.5 ), y ) );\n\t\t\tvec4 v2 = texture2D( boneTexture, vec2( dx * ( x + 1.5 ), y ) );\n\t\t\tvec4 v3 = texture2D( boneTexture, vec2( dx * ( x + 2.5 ), y ) );\n\t\t\tvec4 v4 = texture2D( boneTexture, vec2( dx * ( x + 3.5 ), y ) );\n\t\t\tmat4 bone = mat4( v1, v2, v3, v4 );\n\t\t\treturn bone;\n\t\t}\n\t#else\n\t\tuniform mat4 boneMatrices[ MAX_BONES ];\n\t\tmat4 getBoneMatrix( const in float i ) {\n\t\t\tmat4 bone = boneMatrices[ int(i) ];\n\t\t\treturn bone;\n\t\t}\n\t#endif\n#endif&quot;,
skinning_vertex:&quot;#ifdef USE_SKINNING\n\tvec4 skinVertex = bindMatrix * vec4( transformed, 1.0 );\n\tvec4 skinned = vec4( 0.0 );\n\tskinned += boneMatX * skinVertex * skinWeight.x;\n\tskinned += boneMatY * skinVertex * skinWeight.y;\n\tskinned += boneMatZ * skinVertex * skinWeight.z;\n\tskinned += boneMatW * skinVertex * skinWeight.w;\n\ttransformed = ( bindMatrixInverse * skinned ).xyz;\n#endif&quot;,skinnormal_vertex:&quot;#ifdef USE_SKINNING\n\tmat4 skinMatrix = mat4( 0.0 );\n\tskinMatrix += skinWeight.x * boneMatX;\n\tskinMatrix += skinWeight.y * boneMatY;\n\tskinMatrix += skinWeight.z * boneMatZ;\n\tskinMatrix += skinWeight.w * boneMatW;\n\tskinMatrix  = bindMatrixInverse * skinMatrix * bindMatrix;\n\tobjectNormal = vec4( skinMatrix * vec4( objectNormal, 0.0 ) ).xyz;\n\t#ifdef USE_TANGENT\n\t\tobjectTangent = vec4( skinMatrix * vec4( objectTangent, 0.0 ) ).xyz;\n\t#endif\n#endif&quot;,
specularmap_fragment:&quot;float specularStrength;\n#ifdef USE_SPECULARMAP\n\tvec4 texelSpecular = texture2D( specularMap, vUv );\n\tspecularStrength = texelSpecular.r;\n#else\n\tspecularStrength = 1.0;\n#endif&quot;,specularmap_pars_fragment:&quot;#ifdef USE_SPECULARMAP\n\tuniform sampler2D specularMap;\n#endif&quot;,tonemapping_fragment:&quot;#if defined( TONE_MAPPING )\n\tgl_FragColor.rgb = toneMapping( gl_FragColor.rgb );\n#endif&quot;,tonemapping_pars_fragment:&quot;#ifndef saturate\n#define saturate(a) clamp( a, 0.0, 1.0 )\n#endif\nuniform float toneMappingExposure;\nuniform float toneMappingWhitePoint;\nvec3 LinearToneMapping( vec3 color ) {\n\treturn toneMappingExposure * color;\n}\nvec3 ReinhardToneMapping( vec3 color ) {\n\tcolor *= toneMappingExposure;\n\treturn saturate( color / ( vec3( 1.0 ) + color ) );\n}\n#define Uncharted2Helper( x ) max( ( ( x * ( 0.15 * x + 0.10 * 0.50 ) + 0.20 * 0.02 ) / ( x * ( 0.15 * x + 0.50 ) + 0.20 * 0.30 ) ) - 0.02 / 0.30, vec3( 0.0 ) )\nvec3 Uncharted2ToneMapping( vec3 color ) {\n\tcolor *= toneMappingExposure;\n\treturn saturate( Uncharted2Helper( color ) / Uncharted2Helper( vec3( toneMappingWhitePoint ) ) );\n}\nvec3 OptimizedCineonToneMapping( vec3 color ) {\n\tcolor *= toneMappingExposure;\n\tcolor = max( vec3( 0.0 ), color - 0.004 );\n\treturn pow( ( color * ( 6.2 * color + 0.5 ) ) / ( color * ( 6.2 * color + 1.7 ) + 0.06 ), vec3( 2.2 ) );\n}\nvec3 ACESFilmicToneMapping( vec3 color ) {\n\tcolor *= toneMappingExposure;\n\treturn saturate( ( color * ( 2.51 * color + 0.03 ) ) / ( color * ( 2.43 * color + 0.59 ) + 0.14 ) );\n}&quot;,
uv_pars_fragment:&quot;#if ( defined( USE_UV ) && ! defined( UVS_VERTEX_ONLY ) )\n\tvarying vec2 vUv;\n#endif&quot;,uv_pars_vertex:&quot;#ifdef USE_UV\n\t#ifdef UVS_VERTEX_ONLY\n\t\tvec2 vUv;\n\t#else\n\t\tvarying vec2 vUv;\n\t#endif\n\tuniform mat3 uvTransform;\n#endif&quot;,uv_vertex:&quot;#ifdef USE_UV\n\tvUv = ( uvTransform * vec3( uv, 1 ) ).xy;\n#endif&quot;,uv2_pars_fragment:&quot;#if defined( USE_LIGHTMAP ) || defined( USE_AOMAP )\n\tvarying vec2 vUv2;\n#endif&quot;,uv2_pars_vertex:&quot;#if defined( USE_LIGHTMAP ) || defined( USE_AOMAP )\n\tattribute vec2 uv2;\n\tvarying vec2 vUv2;\n\tuniform mat3 uv2Transform;\n#endif&quot;,
uv2_vertex:&quot;#if defined( USE_LIGHTMAP ) || defined( USE_AOMAP )\n\tvUv2 = ( uv2Transform * vec3( uv2, 1 ) ).xy;\n#endif&quot;,worldpos_vertex:&quot;#if defined( USE_ENVMAP ) || defined( DISTANCE ) || defined ( USE_SHADOWMAP )\n\tvec4 worldPosition = vec4( transformed, 1.0 );\n\t#ifdef USE_INSTANCING\n\t\tworldPosition = instanceMatrix * worldPosition;\n\t#endif\n\tworldPosition = modelMatrix * worldPosition;\n#endif&quot;,background_frag:&quot;uniform sampler2D t2D;\nvarying vec2 vUv;\nvoid main() {\n\tvec4 texColor = texture2D( t2D, vUv );\n\tgl_FragColor = mapTexelToLinear( texColor );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n}&quot;,
background_vert:&quot;varying vec2 vUv;\nuniform mat3 uvTransform;\nvoid main() {\n\tvUv = ( uvTransform * vec3( uv, 1 ) ).xy;\n\tgl_Position = vec4( position.xy, 1.0, 1.0 );\n}&quot;,cube_frag:&quot;#include <envmap_common_pars_fragment>\nuniform float opacity;\nvarying vec3 vWorldDirection;\n#include <cube_uv_reflection_fragment>\nvoid main() {\n\tvec3 vReflect = vWorldDirection;\n\t#include <envmap_fragment>\n\tgl_FragColor = envColor;\n\tgl_FragColor.a *= opacity;\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n}&quot;,
cube_vert:&quot;varying vec3 vWorldDirection;\n#include <common>\nvoid main() {\n\tvWorldDirection = transformDirection( position, modelMatrix );\n\t#include <begin_vertex>\n\t#include <project_vertex>\n\tgl_Position.z = gl_Position.w;\n}&quot;,depth_frag:&quot;#if DEPTH_PACKING == 3200\n\tuniform float opacity;\n#endif\n#include <common>\n#include <packing>\n#include <uv_pars_fragment>\n#include <map_pars_fragment>\n#include <alphamap_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvarying vec2 vHighPrecisionZW;\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tvec4 diffuseColor = vec4( 1.0 );\n\t#if DEPTH_PACKING == 3200\n\t\tdiffuseColor.a = opacity;\n\t#endif\n\t#include <map_fragment>\n\t#include <alphamap_fragment>\n\t#include <alphatest_fragment>\n\t#include <logdepthbuf_fragment>\n\tfloat fragCoordZ = 0.5 * vHighPrecisionZW[0] / vHighPrecisionZW[1] + 0.5;\n\t#if DEPTH_PACKING == 3200\n\t\tgl_FragColor = vec4( vec3( 1.0 - fragCoordZ ), opacity );\n\t#elif DEPTH_PACKING == 3201\n\t\tgl_FragColor = packDepthToRGBA( fragCoordZ );\n\t#endif\n}&quot;,
depth_vert:&quot;#include <common>\n#include <uv_pars_vertex>\n#include <displacementmap_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <skinning_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvarying vec2 vHighPrecisionZW;\nvoid main() {\n\t#include <uv_vertex>\n\t#include <skinbase_vertex>\n\t#ifdef USE_DISPLACEMENTMAP\n\t\t#include <beginnormal_vertex>\n\t\t#include <morphnormal_vertex>\n\t\t#include <skinnormal_vertex>\n\t#endif\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <skinning_vertex>\n\t#include <displacementmap_vertex>\n\t#include <project_vertex>\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n\tvHighPrecisionZW = gl_Position.zw;\n}&quot;,
distanceRGBA_frag:&quot;#define DISTANCE\nuniform vec3 referencePosition;\nuniform float nearDistance;\nuniform float farDistance;\nvarying vec3 vWorldPosition;\n#include <common>\n#include <packing>\n#include <uv_pars_fragment>\n#include <map_pars_fragment>\n#include <alphamap_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main () {\n\t#include <clipping_planes_fragment>\n\tvec4 diffuseColor = vec4( 1.0 );\n\t#include <map_fragment>\n\t#include <alphamap_fragment>\n\t#include <alphatest_fragment>\n\tfloat dist = length( vWorldPosition - referencePosition );\n\tdist = ( dist - nearDistance ) / ( farDistance - nearDistance );\n\tdist = saturate( dist );\n\tgl_FragColor = packDepthToRGBA( dist );\n}&quot;,
distanceRGBA_vert:&quot;#define DISTANCE\nvarying vec3 vWorldPosition;\n#include <common>\n#include <uv_pars_vertex>\n#include <displacementmap_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <skinning_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <uv_vertex>\n\t#include <skinbase_vertex>\n\t#ifdef USE_DISPLACEMENTMAP\n\t\t#include <beginnormal_vertex>\n\t\t#include <morphnormal_vertex>\n\t\t#include <skinnormal_vertex>\n\t#endif\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <skinning_vertex>\n\t#include <displacementmap_vertex>\n\t#include <project_vertex>\n\t#include <worldpos_vertex>\n\t#include <clipping_planes_vertex>\n\tvWorldPosition = worldPosition.xyz;\n}&quot;,
equirect_frag:&quot;uniform sampler2D tEquirect;\nvarying vec3 vWorldDirection;\n#include <common>\nvoid main() {\n\tvec3 direction = normalize( vWorldDirection );\n\tvec2 sampleUV;\n\tsampleUV.y = asin( clamp( direction.y, - 1.0, 1.0 ) ) * RECIPROCAL_PI + 0.5;\n\tsampleUV.x = atan( direction.z, direction.x ) * RECIPROCAL_PI2 + 0.5;\n\tvec4 texColor = texture2D( tEquirect, sampleUV );\n\tgl_FragColor = mapTexelToLinear( texColor );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n}&quot;,
equirect_vert:&quot;varying vec3 vWorldDirection;\n#include <common>\nvoid main() {\n\tvWorldDirection = transformDirection( position, modelMatrix );\n\t#include <begin_vertex>\n\t#include <project_vertex>\n}&quot;,linedashed_frag:&quot;uniform vec3 diffuse;\nuniform float opacity;\nuniform float dashSize;\nuniform float totalSize;\nvarying float vLineDistance;\n#include <common>\n#include <color_pars_fragment>\n#include <fog_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tif ( mod( vLineDistance, totalSize ) > dashSize ) {\n\t\tdiscard;\n\t}\n\tvec3 outgoingLight = vec3( 0.0 );\n\tvec4 diffuseColor = vec4( diffuse, opacity );\n\t#include <logdepthbuf_fragment>\n\t#include <color_fragment>\n\toutgoingLight = diffuseColor.rgb;\n\tgl_FragColor = vec4( outgoingLight, diffuseColor.a );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n\t#include <premultiplied_alpha_fragment>\n}&quot;,
linedashed_vert:&quot;uniform float scale;\nattribute float lineDistance;\nvarying float vLineDistance;\n#include <common>\n#include <color_pars_vertex>\n#include <fog_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\tvLineDistance = scale * lineDistance;\n\t#include <color_vertex>\n\t#include <begin_vertex>\n\t#include <project_vertex>\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n\t#include <fog_vertex>\n}&quot;,meshbasic_frag:&quot;uniform vec3 diffuse;\nuniform float opacity;\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n#endif\n#include <common>\n#include <color_pars_fragment>\n#include <uv_pars_fragment>\n#include <uv2_pars_fragment>\n#include <map_pars_fragment>\n#include <alphamap_pars_fragment>\n#include <aomap_pars_fragment>\n#include <lightmap_pars_fragment>\n#include <envmap_common_pars_fragment>\n#include <envmap_pars_fragment>\n#include <cube_uv_reflection_fragment>\n#include <fog_pars_fragment>\n#include <specularmap_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tvec4 diffuseColor = vec4( diffuse, opacity );\n\t#include <logdepthbuf_fragment>\n\t#include <map_fragment>\n\t#include <color_fragment>\n\t#include <alphamap_fragment>\n\t#include <alphatest_fragment>\n\t#include <specularmap_fragment>\n\tReflectedLight reflectedLight = ReflectedLight( vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ) );\n\t#ifdef USE_LIGHTMAP\n\t\n\t\tvec4 lightMapTexel= texture2D( lightMap, vUv2 );\n\t\treflectedLight.indirectDiffuse += lightMapTexelToLinear( lightMapTexel ).rgb * lightMapIntensity;\n\t#else\n\t\treflectedLight.indirectDiffuse += vec3( 1.0 );\n\t#endif\n\t#include <aomap_fragment>\n\treflectedLight.indirectDiffuse *= diffuseColor.rgb;\n\tvec3 outgoingLight = reflectedLight.indirectDiffuse;\n\t#include <envmap_fragment>\n\tgl_FragColor = vec4( outgoingLight, diffuseColor.a );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n\t#include <premultiplied_alpha_fragment>\n}&quot;,
meshbasic_vert:&quot;#include <common>\n#include <uv_pars_vertex>\n#include <uv2_pars_vertex>\n#include <envmap_pars_vertex>\n#include <color_pars_vertex>\n#include <fog_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <skinning_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <uv_vertex>\n\t#include <uv2_vertex>\n\t#include <color_vertex>\n\t#include <skinbase_vertex>\n\t#ifdef USE_ENVMAP\n\t#include <beginnormal_vertex>\n\t#include <morphnormal_vertex>\n\t#include <skinnormal_vertex>\n\t#include <defaultnormal_vertex>\n\t#endif\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <skinning_vertex>\n\t#include <project_vertex>\n\t#include <logdepthbuf_vertex>\n\t#include <worldpos_vertex>\n\t#include <clipping_planes_vertex>\n\t#include <envmap_vertex>\n\t#include <fog_vertex>\n}&quot;,
meshlambert_frag:&quot;uniform vec3 diffuse;\nuniform vec3 emissive;\nuniform float opacity;\nvarying vec3 vLightFront;\nvarying vec3 vIndirectFront;\n#ifdef DOUBLE_SIDED\n\tvarying vec3 vLightBack;\n\tvarying vec3 vIndirectBack;\n#endif\n#include <common>\n#include <packing>\n#include <dithering_pars_fragment>\n#include <color_pars_fragment>\n#include <uv_pars_fragment>\n#include <uv2_pars_fragment>\n#include <map_pars_fragment>\n#include <alphamap_pars_fragment>\n#include <aomap_pars_fragment>\n#include <lightmap_pars_fragment>\n#include <emissivemap_pars_fragment>\n#include <envmap_common_pars_fragment>\n#include <envmap_pars_fragment>\n#include <cube_uv_reflection_fragment>\n#include <bsdfs>\n#include <lights_pars_begin>\n#include <fog_pars_fragment>\n#include <shadowmap_pars_fragment>\n#include <shadowmask_pars_fragment>\n#include <specularmap_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tvec4 diffuseColor = vec4( diffuse, opacity );\n\tReflectedLight reflectedLight = ReflectedLight( vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ) );\n\tvec3 totalEmissiveRadiance = emissive;\n\t#include <logdepthbuf_fragment>\n\t#include <map_fragment>\n\t#include <color_fragment>\n\t#include <alphamap_fragment>\n\t#include <alphatest_fragment>\n\t#include <specularmap_fragment>\n\t#include <emissivemap_fragment>\n\treflectedLight.indirectDiffuse = getAmbientLightIrradiance( ambientLightColor );\n\t#ifdef DOUBLE_SIDED\n\t\treflectedLight.indirectDiffuse += ( gl_FrontFacing ) ? vIndirectFront : vIndirectBack;\n\t#else\n\t\treflectedLight.indirectDiffuse += vIndirectFront;\n\t#endif\n\t#include <lightmap_fragment>\n\treflectedLight.indirectDiffuse *= BRDF_Diffuse_Lambert( diffuseColor.rgb );\n\t#ifdef DOUBLE_SIDED\n\t\treflectedLight.directDiffuse = ( gl_FrontFacing ) ? vLightFront : vLightBack;\n\t#else\n\t\treflectedLight.directDiffuse = vLightFront;\n\t#endif\n\treflectedLight.directDiffuse *= BRDF_Diffuse_Lambert( diffuseColor.rgb ) * getShadowMask();\n\t#include <aomap_fragment>\n\tvec3 outgoingLight = reflectedLight.directDiffuse + reflectedLight.indirectDiffuse + totalEmissiveRadiance;\n\t#include <envmap_fragment>\n\tgl_FragColor = vec4( outgoingLight, diffuseColor.a );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n\t#include <premultiplied_alpha_fragment>\n\t#include <dithering_fragment>\n}&quot;,
meshlambert_vert:&quot;#define LAMBERT\nvarying vec3 vLightFront;\nvarying vec3 vIndirectFront;\n#ifdef DOUBLE_SIDED\n\tvarying vec3 vLightBack;\n\tvarying vec3 vIndirectBack;\n#endif\n#include <common>\n#include <uv_pars_vertex>\n#include <uv2_pars_vertex>\n#include <envmap_pars_vertex>\n#include <bsdfs>\n#include <lights_pars_begin>\n#include <color_pars_vertex>\n#include <fog_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <skinning_pars_vertex>\n#include <shadowmap_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <uv_vertex>\n\t#include <uv2_vertex>\n\t#include <color_vertex>\n\t#include <beginnormal_vertex>\n\t#include <morphnormal_vertex>\n\t#include <skinbase_vertex>\n\t#include <skinnormal_vertex>\n\t#include <defaultnormal_vertex>\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <skinning_vertex>\n\t#include <project_vertex>\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n\t#include <worldpos_vertex>\n\t#include <envmap_vertex>\n\t#include <lights_lambert_vertex>\n\t#include <shadowmap_vertex>\n\t#include <fog_vertex>\n}&quot;,
meshmatcap_frag:&quot;#define MATCAP\nuniform vec3 diffuse;\nuniform float opacity;\nuniform sampler2D matcap;\nvarying vec3 vViewPosition;\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n#endif\n#include <common>\n#include <color_pars_fragment>\n#include <uv_pars_fragment>\n#include <map_pars_fragment>\n#include <alphamap_pars_fragment>\n#include <fog_pars_fragment>\n#include <bumpmap_pars_fragment>\n#include <normalmap_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tvec4 diffuseColor = vec4( diffuse, opacity );\n\t#include <logdepthbuf_fragment>\n\t#include <map_fragment>\n\t#include <color_fragment>\n\t#include <alphamap_fragment>\n\t#include <alphatest_fragment>\n\t#include <normal_fragment_begin>\n\t#include <normal_fragment_maps>\n\tvec3 viewDir = normalize( vViewPosition );\n\tvec3 x = normalize( vec3( viewDir.z, 0.0, - viewDir.x ) );\n\tvec3 y = cross( viewDir, x );\n\tvec2 uv = vec2( dot( x, normal ), dot( y, normal ) ) * 0.495 + 0.5;\n\t#ifdef USE_MATCAP\n\t\tvec4 matcapColor = texture2D( matcap, uv );\n\t\tmatcapColor = matcapTexelToLinear( matcapColor );\n\t#else\n\t\tvec4 matcapColor = vec4( 1.0 );\n\t#endif\n\tvec3 outgoingLight = diffuseColor.rgb * matcapColor.rgb;\n\tgl_FragColor = vec4( outgoingLight, diffuseColor.a );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n\t#include <premultiplied_alpha_fragment>\n}&quot;,
meshmatcap_vert:&quot;#define MATCAP\nvarying vec3 vViewPosition;\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n#endif\n#include <common>\n#include <uv_pars_vertex>\n#include <color_pars_vertex>\n#include <displacementmap_pars_vertex>\n#include <fog_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <skinning_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <uv_vertex>\n\t#include <color_vertex>\n\t#include <beginnormal_vertex>\n\t#include <morphnormal_vertex>\n\t#include <skinbase_vertex>\n\t#include <skinnormal_vertex>\n\t#include <defaultnormal_vertex>\n\t#ifndef FLAT_SHADED\n\t\tvNormal = normalize( transformedNormal );\n\t#endif\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <skinning_vertex>\n\t#include <displacementmap_vertex>\n\t#include <project_vertex>\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n\t#include <fog_vertex>\n\tvViewPosition = - mvPosition.xyz;\n}&quot;,
meshtoon_frag:&quot;#define TOON\nuniform vec3 diffuse;\nuniform vec3 emissive;\nuniform vec3 specular;\nuniform float shininess;\nuniform float opacity;\n#include <common>\n#include <packing>\n#include <dithering_pars_fragment>\n#include <color_pars_fragment>\n#include <uv_pars_fragment>\n#include <uv2_pars_fragment>\n#include <map_pars_fragment>\n#include <alphamap_pars_fragment>\n#include <aomap_pars_fragment>\n#include <lightmap_pars_fragment>\n#include <emissivemap_pars_fragment>\n#include <gradientmap_pars_fragment>\n#include <fog_pars_fragment>\n#include <bsdfs>\n#include <lights_pars_begin>\n#include <lights_toon_pars_fragment>\n#include <shadowmap_pars_fragment>\n#include <bumpmap_pars_fragment>\n#include <normalmap_pars_fragment>\n#include <specularmap_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tvec4 diffuseColor = vec4( diffuse, opacity );\n\tReflectedLight reflectedLight = ReflectedLight( vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ) );\n\tvec3 totalEmissiveRadiance = emissive;\n\t#include <logdepthbuf_fragment>\n\t#include <map_fragment>\n\t#include <color_fragment>\n\t#include <alphamap_fragment>\n\t#include <alphatest_fragment>\n\t#include <specularmap_fragment>\n\t#include <normal_fragment_begin>\n\t#include <normal_fragment_maps>\n\t#include <emissivemap_fragment>\n\t#include <lights_toon_fragment>\n\t#include <lights_fragment_begin>\n\t#include <lights_fragment_maps>\n\t#include <lights_fragment_end>\n\t#include <aomap_fragment>\n\tvec3 outgoingLight = reflectedLight.directDiffuse + reflectedLight.indirectDiffuse + reflectedLight.directSpecular + reflectedLight.indirectSpecular + totalEmissiveRadiance;\n\tgl_FragColor = vec4( outgoingLight, diffuseColor.a );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n\t#include <premultiplied_alpha_fragment>\n\t#include <dithering_fragment>\n}&quot;,
meshtoon_vert:&quot;#define TOON\nvarying vec3 vViewPosition;\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n#endif\n#include <common>\n#include <uv_pars_vertex>\n#include <uv2_pars_vertex>\n#include <displacementmap_pars_vertex>\n#include <color_pars_vertex>\n#include <fog_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <skinning_pars_vertex>\n#include <shadowmap_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <uv_vertex>\n\t#include <uv2_vertex>\n\t#include <color_vertex>\n\t#include <beginnormal_vertex>\n\t#include <morphnormal_vertex>\n\t#include <skinbase_vertex>\n\t#include <skinnormal_vertex>\n\t#include <defaultnormal_vertex>\n#ifndef FLAT_SHADED\n\tvNormal = normalize( transformedNormal );\n#endif\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <skinning_vertex>\n\t#include <displacementmap_vertex>\n\t#include <project_vertex>\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n\tvViewPosition = - mvPosition.xyz;\n\t#include <worldpos_vertex>\n\t#include <shadowmap_vertex>\n\t#include <fog_vertex>\n}&quot;,
meshphong_frag:&quot;#define PHONG\nuniform vec3 diffuse;\nuniform vec3 emissive;\nuniform vec3 specular;\nuniform float shininess;\nuniform float opacity;\n#include <common>\n#include <packing>\n#include <dithering_pars_fragment>\n#include <color_pars_fragment>\n#include <uv_pars_fragment>\n#include <uv2_pars_fragment>\n#include <map_pars_fragment>\n#include <alphamap_pars_fragment>\n#include <aomap_pars_fragment>\n#include <lightmap_pars_fragment>\n#include <emissivemap_pars_fragment>\n#include <envmap_common_pars_fragment>\n#include <envmap_pars_fragment>\n#include <cube_uv_reflection_fragment>\n#include <fog_pars_fragment>\n#include <bsdfs>\n#include <lights_pars_begin>\n#include <lights_phong_pars_fragment>\n#include <shadowmap_pars_fragment>\n#include <bumpmap_pars_fragment>\n#include <normalmap_pars_fragment>\n#include <specularmap_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tvec4 diffuseColor = vec4( diffuse, opacity );\n\tReflectedLight reflectedLight = ReflectedLight( vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ) );\n\tvec3 totalEmissiveRadiance = emissive;\n\t#include <logdepthbuf_fragment>\n\t#include <map_fragment>\n\t#include <color_fragment>\n\t#include <alphamap_fragment>\n\t#include <alphatest_fragment>\n\t#include <specularmap_fragment>\n\t#include <normal_fragment_begin>\n\t#include <normal_fragment_maps>\n\t#include <emissivemap_fragment>\n\t#include <lights_phong_fragment>\n\t#include <lights_fragment_begin>\n\t#include <lights_fragment_maps>\n\t#include <lights_fragment_end>\n\t#include <aomap_fragment>\n\tvec3 outgoingLight = reflectedLight.directDiffuse + reflectedLight.indirectDiffuse + reflectedLight.directSpecular + reflectedLight.indirectSpecular + totalEmissiveRadiance;\n\t#include <envmap_fragment>\n\tgl_FragColor = vec4( outgoingLight, diffuseColor.a );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n\t#include <premultiplied_alpha_fragment>\n\t#include <dithering_fragment>\n}&quot;,
meshphong_vert:&quot;#define PHONG\nvarying vec3 vViewPosition;\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n#endif\n#include <common>\n#include <uv_pars_vertex>\n#include <uv2_pars_vertex>\n#include <displacementmap_pars_vertex>\n#include <envmap_pars_vertex>\n#include <color_pars_vertex>\n#include <fog_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <skinning_pars_vertex>\n#include <shadowmap_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <uv_vertex>\n\t#include <uv2_vertex>\n\t#include <color_vertex>\n\t#include <beginnormal_vertex>\n\t#include <morphnormal_vertex>\n\t#include <skinbase_vertex>\n\t#include <skinnormal_vertex>\n\t#include <defaultnormal_vertex>\n#ifndef FLAT_SHADED\n\tvNormal = normalize( transformedNormal );\n#endif\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <skinning_vertex>\n\t#include <displacementmap_vertex>\n\t#include <project_vertex>\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n\tvViewPosition = - mvPosition.xyz;\n\t#include <worldpos_vertex>\n\t#include <envmap_vertex>\n\t#include <shadowmap_vertex>\n\t#include <fog_vertex>\n}&quot;,
meshphysical_frag:&quot;#define STANDARD\n#ifdef PHYSICAL\n\t#define REFLECTIVITY\n\t#define CLEARCOAT\n\t#define TRANSPARENCY\n#endif\nuniform vec3 diffuse;\nuniform vec3 emissive;\nuniform float roughness;\nuniform float metalness;\nuniform float opacity;\n#ifdef TRANSPARENCY\n\tuniform float transparency;\n#endif\n#ifdef REFLECTIVITY\n\tuniform float reflectivity;\n#endif\n#ifdef CLEARCOAT\n\tuniform float clearcoat;\n\tuniform float clearcoatRoughness;\n#endif\n#ifdef USE_SHEEN\n\tuniform vec3 sheen;\n#endif\nvarying vec3 vViewPosition;\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n\t#ifdef USE_TANGENT\n\t\tvarying vec3 vTangent;\n\t\tvarying vec3 vBitangent;\n\t#endif\n#endif\n#include <common>\n#include <packing>\n#include <dithering_pars_fragment>\n#include <color_pars_fragment>\n#include <uv_pars_fragment>\n#include <uv2_pars_fragment>\n#include <map_pars_fragment>\n#include <alphamap_pars_fragment>\n#include <aomap_pars_fragment>\n#include <lightmap_pars_fragment>\n#include <emissivemap_pars_fragment>\n#include <bsdfs>\n#include <cube_uv_reflection_fragment>\n#include <envmap_common_pars_fragment>\n#include <envmap_physical_pars_fragment>\n#include <fog_pars_fragment>\n#include <lights_pars_begin>\n#include <lights_physical_pars_fragment>\n#include <shadowmap_pars_fragment>\n#include <bumpmap_pars_fragment>\n#include <normalmap_pars_fragment>\n#include <clearcoat_pars_fragment>\n#include <roughnessmap_pars_fragment>\n#include <metalnessmap_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tvec4 diffuseColor = vec4( diffuse, opacity );\n\tReflectedLight reflectedLight = ReflectedLight( vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ), vec3( 0.0 ) );\n\tvec3 totalEmissiveRadiance = emissive;\n\t#include <logdepthbuf_fragment>\n\t#include <map_fragment>\n\t#include <color_fragment>\n\t#include <alphamap_fragment>\n\t#include <alphatest_fragment>\n\t#include <roughnessmap_fragment>\n\t#include <metalnessmap_fragment>\n\t#include <normal_fragment_begin>\n\t#include <normal_fragment_maps>\n\t#include <clearcoat_normal_fragment_begin>\n\t#include <clearcoat_normal_fragment_maps>\n\t#include <emissivemap_fragment>\n\t#include <lights_physical_fragment>\n\t#include <lights_fragment_begin>\n\t#include <lights_fragment_maps>\n\t#include <lights_fragment_end>\n\t#include <aomap_fragment>\n\tvec3 outgoingLight = reflectedLight.directDiffuse + reflectedLight.indirectDiffuse + reflectedLight.directSpecular + reflectedLight.indirectSpecular + totalEmissiveRadiance;\n\t#ifdef TRANSPARENCY\n\t\tdiffuseColor.a *= saturate( 1. - transparency + linearToRelativeLuminance( reflectedLight.directSpecular + reflectedLight.indirectSpecular ) );\n\t#endif\n\tgl_FragColor = vec4( outgoingLight, diffuseColor.a );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n\t#include <premultiplied_alpha_fragment>\n\t#include <dithering_fragment>\n}&quot;,
meshphysical_vert:&quot;#define STANDARD\nvarying vec3 vViewPosition;\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n\t#ifdef USE_TANGENT\n\t\tvarying vec3 vTangent;\n\t\tvarying vec3 vBitangent;\n\t#endif\n#endif\n#include <common>\n#include <uv_pars_vertex>\n#include <uv2_pars_vertex>\n#include <displacementmap_pars_vertex>\n#include <color_pars_vertex>\n#include <fog_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <skinning_pars_vertex>\n#include <shadowmap_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <uv_vertex>\n\t#include <uv2_vertex>\n\t#include <color_vertex>\n\t#include <beginnormal_vertex>\n\t#include <morphnormal_vertex>\n\t#include <skinbase_vertex>\n\t#include <skinnormal_vertex>\n\t#include <defaultnormal_vertex>\n#ifndef FLAT_SHADED\n\tvNormal = normalize( transformedNormal );\n\t#ifdef USE_TANGENT\n\t\tvTangent = normalize( transformedTangent );\n\t\tvBitangent = normalize( cross( vNormal, vTangent ) * tangent.w );\n\t#endif\n#endif\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <skinning_vertex>\n\t#include <displacementmap_vertex>\n\t#include <project_vertex>\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n\tvViewPosition = - mvPosition.xyz;\n\t#include <worldpos_vertex>\n\t#include <shadowmap_vertex>\n\t#include <fog_vertex>\n}&quot;,
normal_frag:&quot;#define NORMAL\nuniform float opacity;\n#if defined( FLAT_SHADED ) || defined( USE_BUMPMAP ) || defined( TANGENTSPACE_NORMALMAP )\n\tvarying vec3 vViewPosition;\n#endif\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n\t#ifdef USE_TANGENT\n\t\tvarying vec3 vTangent;\n\t\tvarying vec3 vBitangent;\n\t#endif\n#endif\n#include <packing>\n#include <uv_pars_fragment>\n#include <bumpmap_pars_fragment>\n#include <normalmap_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\t#include <logdepthbuf_fragment>\n\t#include <normal_fragment_begin>\n\t#include <normal_fragment_maps>\n\tgl_FragColor = vec4( packNormalToRGB( normal ), opacity );\n}&quot;,
normal_vert:&quot;#define NORMAL\n#if defined( FLAT_SHADED ) || defined( USE_BUMPMAP ) || defined( TANGENTSPACE_NORMALMAP )\n\tvarying vec3 vViewPosition;\n#endif\n#ifndef FLAT_SHADED\n\tvarying vec3 vNormal;\n\t#ifdef USE_TANGENT\n\t\tvarying vec3 vTangent;\n\t\tvarying vec3 vBitangent;\n\t#endif\n#endif\n#include <common>\n#include <uv_pars_vertex>\n#include <displacementmap_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <skinning_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <uv_vertex>\n\t#include <beginnormal_vertex>\n\t#include <morphnormal_vertex>\n\t#include <skinbase_vertex>\n\t#include <skinnormal_vertex>\n\t#include <defaultnormal_vertex>\n#ifndef FLAT_SHADED\n\tvNormal = normalize( transformedNormal );\n\t#ifdef USE_TANGENT\n\t\tvTangent = normalize( transformedTangent );\n\t\tvBitangent = normalize( cross( vNormal, vTangent ) * tangent.w );\n\t#endif\n#endif\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <skinning_vertex>\n\t#include <displacementmap_vertex>\n\t#include <project_vertex>\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n#if defined( FLAT_SHADED ) || defined( USE_BUMPMAP ) || defined( TANGENTSPACE_NORMALMAP )\n\tvViewPosition = - mvPosition.xyz;\n#endif\n}&quot;,
points_frag:&quot;uniform vec3 diffuse;\nuniform float opacity;\n#include <common>\n#include <color_pars_fragment>\n#include <map_particle_pars_fragment>\n#include <fog_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tvec3 outgoingLight = vec3( 0.0 );\n\tvec4 diffuseColor = vec4( diffuse, opacity );\n\t#include <logdepthbuf_fragment>\n\t#include <map_particle_fragment>\n\t#include <color_fragment>\n\t#include <alphatest_fragment>\n\toutgoingLight = diffuseColor.rgb;\n\tgl_FragColor = vec4( outgoingLight, diffuseColor.a );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n\t#include <premultiplied_alpha_fragment>\n}&quot;,
points_vert:&quot;uniform float size;\nuniform float scale;\n#include <common>\n#include <color_pars_vertex>\n#include <fog_pars_vertex>\n#include <morphtarget_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <color_vertex>\n\t#include <begin_vertex>\n\t#include <morphtarget_vertex>\n\t#include <project_vertex>\n\tgl_PointSize = size;\n\t#ifdef USE_SIZEATTENUATION\n\t\tbool isPerspective = isPerspectiveMatrix( projectionMatrix );\n\t\tif ( isPerspective ) gl_PointSize *= ( scale / - mvPosition.z );\n\t#endif\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n\t#include <worldpos_vertex>\n\t#include <fog_vertex>\n}&quot;,
shadow_frag:&quot;uniform vec3 color;\nuniform float opacity;\n#include <common>\n#include <packing>\n#include <fog_pars_fragment>\n#include <bsdfs>\n#include <lights_pars_begin>\n#include <shadowmap_pars_fragment>\n#include <shadowmask_pars_fragment>\nvoid main() {\n\tgl_FragColor = vec4( color, opacity * ( 1.0 - getShadowMask() ) );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n}&quot;,shadow_vert:&quot;#include <fog_pars_vertex>\n#include <shadowmap_pars_vertex>\nvoid main() {\n\t#include <begin_vertex>\n\t#include <project_vertex>\n\t#include <worldpos_vertex>\n\t#include <shadowmap_vertex>\n\t#include <fog_vertex>\n}&quot;,
sprite_frag:&quot;uniform vec3 diffuse;\nuniform float opacity;\n#include <common>\n#include <uv_pars_fragment>\n#include <map_pars_fragment>\n#include <alphamap_pars_fragment>\n#include <fog_pars_fragment>\n#include <logdepthbuf_pars_fragment>\n#include <clipping_planes_pars_fragment>\nvoid main() {\n\t#include <clipping_planes_fragment>\n\tvec3 outgoingLight = vec3( 0.0 );\n\tvec4 diffuseColor = vec4( diffuse, opacity );\n\t#include <logdepthbuf_fragment>\n\t#include <map_fragment>\n\t#include <alphamap_fragment>\n\t#include <alphatest_fragment>\n\toutgoingLight = diffuseColor.rgb;\n\tgl_FragColor = vec4( outgoingLight, diffuseColor.a );\n\t#include <tonemapping_fragment>\n\t#include <encodings_fragment>\n\t#include <fog_fragment>\n}&quot;,
sprite_vert:&quot;uniform float rotation;\nuniform vec2 center;\n#include <common>\n#include <uv_pars_vertex>\n#include <fog_pars_vertex>\n#include <logdepthbuf_pars_vertex>\n#include <clipping_planes_pars_vertex>\nvoid main() {\n\t#include <uv_vertex>\n\tvec4 mvPosition = modelViewMatrix * vec4( 0.0, 0.0, 0.0, 1.0 );\n\tvec2 scale;\n\tscale.x = length( vec3( modelMatrix[ 0 ].x, modelMatrix[ 0 ].y, modelMatrix[ 0 ].z ) );\n\tscale.y = length( vec3( modelMatrix[ 1 ].x, modelMatrix[ 1 ].y, modelMatrix[ 1 ].z ) );\n\t#ifndef USE_SIZEATTENUATION\n\t\tbool isPerspective = isPerspectiveMatrix( projectionMatrix );\n\t\tif ( isPerspective ) scale *= - mvPosition.z;\n\t#endif\n\tvec2 alignedPosition = ( position.xy - ( center - vec2( 0.5 ) ) ) * scale;\n\tvec2 rotatedPosition;\n\trotatedPosition.x = cos( rotation ) * alignedPosition.x - sin( rotation ) * alignedPosition.y;\n\trotatedPosition.y = sin( rotation ) * alignedPosition.x + cos( rotation ) * alignedPosition.y;\n\tmvPosition.xy += rotatedPosition;\n\tgl_Position = projectionMatrix * mvPosition;\n\t#include <logdepthbuf_vertex>\n\t#include <clipping_planes_vertex>\n\t#include <fog_vertex>\n}&quot;},
eb={basic:{uniforms:va([D.common,D.specularmap,D.envmap,D.aomap,D.lightmap,D.fog]),vertexShader:O.meshbasic_vert,fragmentShader:O.meshbasic_frag},lambert:{uniforms:va([D.common,D.specularmap,D.envmap,D.aomap,D.lightmap,D.emissivemap,D.fog,D.lights,{emissive:{value:new A(0)}}]),vertexShader:O.meshlambert_vert,fragmentShader:O.meshlambert_frag},phong:{uniforms:va([D.common,D.specularmap,D.envmap,D.aomap,D.lightmap,D.emissivemap,D.bumpmap,D.normalmap,D.displacementmap,D.fog,D.lights,{emissive:{value:new A(0)},
specular:{value:new A(1118481)},shininess:{value:30}}]),vertexShader:O.meshphong_vert,fragmentShader:O.meshphong_frag},standard:{uniforms:va([D.common,D.envmap,D.aomap,D.lightmap,D.emissivemap,D.bumpmap,D.normalmap,D.displacementmap,D.roughnessmap,D.metalnessmap,D.fog,D.lights,{emissive:{value:new A(0)},roughness:{value:.5},metalness:{value:.5},envMapIntensity:{value:1}}]),vertexShader:O.meshphysical_vert,fragmentShader:O.meshphysical_frag},toon:{uniforms:va([D.common,D.specularmap,D.aomap,D.lightmap,
D.emissivemap,D.bumpmap,D.normalmap,D.displacementmap,D.gradientmap,D.fog,D.lights,{emissive:{value:new A(0)},specular:{value:new A(1118481)},shininess:{value:30}}]),vertexShader:O.meshtoon_vert,fragmentShader:O.meshtoon_frag},matcap:{uniforms:va([D.common,D.bumpmap,D.normalmap,D.displacementmap,D.fog,{matcap:{value:null}}]),vertexShader:O.meshmatcap_vert,fragmentShader:O.meshmatcap_frag},points:{uniforms:va([D.points,D.fog]),vertexShader:O.points_vert,fragmentShader:O.points_frag},dashed:{uniforms:va([D.common,
D.fog,{scale:{value:1},dashSize:{value:1},totalSize:{value:2}}]),vertexShader:O.linedashed_vert,fragmentShader:O.linedashed_frag},depth:{uniforms:va([D.common,D.displacementmap]),vertexShader:O.depth_vert,fragmentShader:O.depth_frag},normal:{uniforms:va([D.common,D.bumpmap,D.normalmap,D.displacementmap,{opacity:{value:1}}]),vertexShader:O.normal_vert,fragmentShader:O.normal_frag},sprite:{uniforms:va([D.sprite,D.fog]),vertexShader:O.sprite_vert,fragmentShader:O.sprite_frag},background:{uniforms:{uvTransform:{value:new wa},
t2D:{value:null}},vertexShader:O.background_vert,fragmentShader:O.background_frag},cube:{uniforms:va([D.envmap,{opacity:{value:1}}]),vertexShader:O.cube_vert,fragmentShader:O.cube_frag},equirect:{uniforms:{tEquirect:{value:null}},vertexShader:O.equirect_vert,fragmentShader:O.equirect_frag},distanceRGBA:{uniforms:va([D.common,D.displacementmap,{referencePosition:{value:new n},nearDistance:{value:1},farDistance:{value:1E3}}]),vertexShader:O.distanceRGBA_vert,fragmentShader:O.distanceRGBA_frag},shadow:{uniforms:va([D.lights,
D.fog,{color:{value:new A(0)},opacity:{value:1}}]),vertexShader:O.shadow_vert,fragmentShader:O.shadow_frag}};eb.physical={uniforms:va([eb.standard.uniforms,{clearcoat:{value:0},clearcoatMap:{value:null},clearcoatRoughness:{value:0},clearcoatRoughnessMap:{value:null},clearcoatNormalScale:{value:new t(1,1)},clearcoatNormalMap:{value:null},sheen:{value:new A(0)},transparency:{value:0}}]),vertexShader:O.meshphysical_vert,fragmentShader:O.meshphysical_frag};qb.prototype=Object.create(V.prototype);qb.prototype.constructor=
qb;qb.prototype.isCubeTexture=!0;Object.defineProperty(qb.prototype,&quot;images&quot;,{get:function(){return this.image},set:function(a){this.image=a}});Ic.prototype=Object.create(V.prototype);Ic.prototype.constructor=Ic;Ic.prototype.isDataTexture2DArray=!0;Jc.prototype=Object.create(V.prototype);Jc.prototype.constructor=Jc;Jc.prototype.isDataTexture3D=!0;var Hh=new V,Pj=new Ic,Rj=new Jc,Ih=new qb,Bh=[],Dh=[],Gh=new Float32Array(16),Fh=new Float32Array(9),Eh=new Float32Array(4);Jh.prototype.updateCache=function(a){var b=
this.cache;a instanceof Float32Array&&b.length!==a.length&&(this.cache=new Float32Array(a.length));Ia(b,a)};Kh.prototype.setValue=function(a,b,c){for(var d=this.seq,e=0,f=d.length;e!==f;++e){var g=d[e];g.setValue(a,b[g.id],c)}};var gg=/([\w\d_]+)(\])?(\[|\.)?/g;Eb.prototype.setValue=function(a,b,c,d){b=this.map[b];void 0!==b&&b.setValue(a,c,d)};Eb.prototype.setOptional=function(a,b,c){b=b[c];void 0!==b&&this.setValue(a,c,b)};Eb.upload=function(a,b,c,d){for(var e=0,f=b.length;e!==f;++e){var g=b[e],
h=c[g.id];!1!==h.needsUpdate&&g.setValue(a,h.value,d)}};Eb.seqWithValue=function(a,b){for(var c=[],d=0,e=a.length;d!==e;++d){var f=a[d];f.id in b&&c.push(f)}return c};var wk=0,ig=/^[ \t]*#include +<([\w\d./]+)>/gm,Th=/#pragma unroll_loop[\s]+?for \( int i = (\d+); i < (\d+); i \+\+ \) \{([\s\S]+?)(?=\})\}/g,Sh=/#pragma unroll_loop_start[\s]+?for \( int i = (\d+); i < (\d+); i \+\+ \) \{([\s\S]+?)(?=\})\}[\s]+?#pragma unroll_loop_end/g,Gk=0;Fb.prototype=Object.create(K.prototype);Fb.prototype.constructor=
Fb;Fb.prototype.isMeshDepthMaterial=!0;Fb.prototype.copy=function(a){K.prototype.copy.call(this,a);this.depthPacking=a.depthPacking;this.skinning=a.skinning;this.morphTargets=a.morphTargets;this.map=a.map;this.alphaMap=a.alphaMap;this.displacementMap=a.displacementMap;this.displacementScale=a.displacementScale;this.displacementBias=a.displacementBias;this.wireframe=a.wireframe;this.wireframeLinewidth=a.wireframeLinewidth;return this};Gb.prototype=Object.create(K.prototype);Gb.prototype.constructor=
Gb;Gb.prototype.isMeshDistanceMaterial=!0;Gb.prototype.copy=function(a){K.prototype.copy.call(this,a);this.referencePosition.copy(a.referencePosition);this.nearDistance=a.nearDistance;this.farDistance=a.farDistance;this.skinning=a.skinning;this.morphTargets=a.morphTargets;this.map=a.map;this.alphaMap=a.alphaMap;this.displacementMap=a.displacementMap;this.displacementScale=a.displacementScale;this.displacementBias=a.displacementBias;return this};Pe.prototype=Object.assign(Object.create(aa.prototype),
{constructor:Pe,isArrayCamera:!0});Mc.prototype=Object.assign(Object.create(F.prototype),{constructor:Mc,isGroup:!0});Object.assign($h.prototype,Ea.prototype);Object.assign(Qe.prototype,{isFogExp2:!0,clone:function(){return new Qe(this.color,this.density)},toJSON:function(){return{type:&quot;FogExp2&quot;,color:this.color.getHex(),density:this.density}}});Object.assign(Re.prototype,{isFog:!0,clone:function(){return new Re(this.color,this.near,this.far)},toJSON:function(){return{type:&quot;Fog&quot;,color:this.color.getHex(),
near:this.near,far:this.far}}});Object.defineProperty(rb.prototype,&quot;needsUpdate&quot;,{set:function(a){!0===a&&this.version++}});Object.assign(rb.prototype,{isInterleavedBuffer:!0,onUploadCallback:function(){},setUsage:function(a){this.usage=a;return this},copy:function(a){this.array=new a.array.constructor(a.array);this.count=a.count;this.stride=a.stride;this.usage=a.usage;return this},copyAt:function(a,b,c){a*=this.stride;c*=b.stride;for(var d=0,e=this.stride;d<e;d++)this.array[a+d]=b.array[c+d];return this},
set:function(a,b){void 0===b&&(b=0);this.array.set(a,b);return this},clone:function(){return(new this.constructor).copy(this)},onUpload:function(a){this.onUploadCallback=a;return this}});var uc=new n;Object.defineProperties(Od.prototype,{count:{get:function(){return this.data.count}},array:{get:function(){return this.data.array}}});Object.assign(Od.prototype,{isInterleavedBufferAttribute:!0,applyMatrix4:function(a){for(var b=0,c=this.data.count;b<c;b++)uc.x=this.getX(b),uc.y=this.getY(b),uc.z=this.getZ(b),
uc.applyMatrix4(a),this.setXYZ(b,uc.x,uc.y,uc.z);return this},setX:function(a,b){this.data.array[a*this.data.stride+this.offset]=b;return this},setY:function(a,b){this.data.array[a*this.data.stride+this.offset+1]=b;return this},setZ:function(a,b){this.data.array[a*this.data.stride+this.offset+2]=b;return this},setW:function(a,b){this.data.array[a*this.data.stride+this.offset+3]=b;return this},getX:function(a){return this.data.array[a*this.data.stride+this.offset]},getY:function(a){return this.data.array[a*
this.data.stride+this.offset+1]},getZ:function(a){return this.data.array[a*this.data.stride+this.offset+2]},getW:function(a){return this.data.array[a*this.data.stride+this.offset+3]},setXY:function(a,b,c){a=a*this.data.stride+this.offset;this.data.array[a+0]=b;this.data.array[a+1]=c;return this},setXYZ:function(a,b,c,d){a=a*this.data.stride+this.offset;this.data.array[a+0]=b;this.data.array[a+1]=c;this.data.array[a+2]=d;return this},setXYZW:function(a,b,c,d,e){a=a*this.data.stride+this.offset;this.data.array[a+
0]=b;this.data.array[a+1]=c;this.data.array[a+2]=d;this.data.array[a+3]=e;return this}});Ib.prototype=Object.create(K.prototype);Ib.prototype.constructor=Ib;Ib.prototype.isSpriteMaterial=!0;Ib.prototype.copy=function(a){K.prototype.copy.call(this,a);this.color.copy(a.color);this.map=a.map;this.alphaMap=a.alphaMap;this.rotation=a.rotation;this.sizeAttenuation=a.sizeAttenuation;return this};var Nc,Ce=new n,xd=new n,yd=new n,Oc=new t,Qd=new t,bi=new P,Mf=new n,De=new n,Nf=new n,Ri=new t,oh=new t,Si=
new t;Pd.prototype=Object.assign(Object.create(F.prototype),{constructor:Pd,isSprite:!0,raycast:function(a,b){null===a.camera&&console.error('THREE.Sprite: &quot;Raycaster.camera&quot; needs to be set in order to raycast against sprites.');xd.setFromMatrixScale(this.matrixWorld);bi.copy(a.camera.matrixWorld);this.modelViewMatrix.multiplyMatrices(a.camera.matrixWorldInverse,this.matrixWorld);yd.setFromMatrixPosition(this.modelViewMatrix);a.camera.isPerspectiveCamera&&!1===this.material.sizeAttenuation&&xd.multiplyScalar(-yd.z);
var c=this.material.rotation;if(0!==c){var d=Math.cos(c);var e=Math.sin(c)}c=this.center;Se(Mf.set(-.5,-.5,0),yd,c,xd,e,d);Se(De.set(.5,-.5,0),yd,c,xd,e,d);Se(Nf.set(.5,.5,0),yd,c,xd,e,d);Ri.set(0,0);oh.set(1,0);Si.set(1,1);var f=a.ray.intersectTriangle(Mf,De,Nf,!1,Ce);if(null===f&&(Se(De.set(-.5,.5,0),yd,c,xd,e,d),oh.set(0,1),f=a.ray.intersectTriangle(Mf,Nf,De,!1,Ce),null===f))return;e=a.ray.origin.distanceTo(Ce);e<a.near||e>a.far||b.push({distance:e,point:Ce.clone(),uv:oa.getUV(Ce,Mf,De,Nf,Ri,oh,
Si,new t),face:null,object:this})},clone:function(){return(new this.constructor(this.material)).copy(this)},copy:function(a){F.prototype.copy.call(this,a);void 0!==a.center&&this.center.copy(a.center);return this}});var Of=new n,Ti=new n;Rd.prototype=Object.assign(Object.create(F.prototype),{constructor:Rd,isLOD:!0,copy:function(a){F.prototype.copy.call(this,a,!1);for(var b=a.levels,c=0,d=b.length;c<d;c++){var e=b[c];this.addLevel(e.object.clone(),e.distance)}this.autoUpdate=a.autoUpdate;return this},
addLevel:function(a,b){void 0===b&&(b=0);b=Math.abs(b);for(var c=this.levels,d=0;d<c.length&&!(b<c[d].distance);d++);c.splice(d,0,{distance:b,object:a});this.add(a);return this},getCurrentLevel:function(){return this._currentLevel},getObjectForDistance:function(a){var b=this.levels;if(0<b.length){for(var c=1,d=b.length;c<d&&!(a<b[c].distance);c++);return b[c-1].object}return null},raycast:function(a,b){if(0<this.levels.length){Of.setFromMatrixPosition(this.matrixWorld);var c=a.ray.origin.distanceTo(Of);
this.getObjectForDistance(c).raycast(a,b)}},update:function(a){var b=this.levels;if(1<b.length){Of.setFromMatrixPosition(a.matrixWorld);Ti.setFromMatrixPosition(this.matrixWorld);a=Of.distanceTo(Ti)/a.zoom;b[0].object.visible=!0;for(var c=1,d=b.length;c<d;c++)if(a>=b[c].distance)b[c-1].object.visible=!1,b[c].object.visible=!0;else break;for(this._currentLevel=c-1;c<d;c++)b[c].object.visible=!1}},toJSON:function(a){a=F.prototype.toJSON.call(this,a);!1===this.autoUpdate&&(a.object.autoUpdate=!1);a.object.levels=
[];for(var b=this.levels,c=0,d=b.length;c<d;c++){var e=b[c];a.object.levels.push({object:e.object.uuid,distance:e.distance})}return a}});Te.prototype=Object.assign(Object.create(S.prototype),{constructor:Te,isSkinnedMesh:!0,bind:function(a,b){this.skeleton=a;void 0===b&&(this.updateMatrixWorld(!0),this.skeleton.calculateInverses(),b=this.matrixWorld);this.bindMatrix.copy(b);this.bindMatrixInverse.getInverse(b)},pose:function(){this.skeleton.pose()},normalizeSkinWeights:function(){for(var a=new ka,
b=this.geometry.attributes.skinWeight,c=0,d=b.count;c<d;c++){a.x=b.getX(c);a.y=b.getY(c);a.z=b.getZ(c);a.w=b.getW(c);var e=1/a.manhattanLength();Infinity!==e?a.multiplyScalar(e):a.set(1,0,0,0);b.setXYZW(c,a.x,a.y,a.z,a.w)}},updateMatrixWorld:function(a){S.prototype.updateMatrixWorld.call(this,a);&quot;attached&quot;===this.bindMode?this.bindMatrixInverse.getInverse(this.matrixWorld):&quot;detached&quot;===this.bindMode?this.bindMatrixInverse.getInverse(this.bindMatrix):console.warn(&quot;THREE.SkinnedMesh: Unrecognized bindMode: &quot;+
this.bindMode)},clone:function(){return(new this.constructor(this.geometry,this.material)).copy(this)}});var Ui=new P,$k=new P;Object.assign(Ue.prototype,{calculateInverses:function(){this.boneInverses=[];for(var a=0,b=this.bones.length;a<b;a++){var c=new P;this.bones[a]&&c.getInverse(this.bones[a].matrixWorld);this.boneInverses.push(c)}},pose:function(){var a,b;var c=0;for(b=this.bones.length;c<b;c++)(a=this.bones[c])&&a.matrixWorld.getInverse(this.boneInverses[c]);c=0;for(b=this.bones.length;c<
b;c++)if(a=this.bones[c])a.parent&&a.parent.isBone?(a.matrix.getInverse(a.parent.matrixWorld),a.matrix.multiply(a.matrixWorld)):a.matrix.copy(a.matrixWorld),a.matrix.decompose(a.position,a.quaternion,a.scale)},update:function(){for(var a=this.bones,b=this.boneInverses,c=this.boneMatrices,d=this.boneTexture,e=0,f=a.length;e<f;e++)Ui.multiplyMatrices(a[e]?a[e].matrixWorld:$k,b[e]),Ui.toArray(c,16*e);void 0!==d&&(d.needsUpdate=!0)},clone:function(){return new Ue(this.bones,this.boneInverses)},getBoneByName:function(a){for(var b=
0,c=this.bones.length;b<c;b++){var d=this.bones[b];if(d.name===a)return d}},dispose:function(){this.boneTexture&&(this.boneTexture.dispose(),this.boneTexture=void 0)}});pg.prototype=Object.assign(Object.create(F.prototype),{constructor:pg,isBone:!0});var Vi=new P,Wi=new P,zd=[],Ee=new S;Ve.prototype=Object.assign(Object.create(S.prototype),{constructor:Ve,isInstancedMesh:!0,getMatrixAt:function(a,b){b.fromArray(this.instanceMatrix.array,16*a)},raycast:function(a,b){var c=this.matrixWorld,d=this.count;
Ee.geometry=this.geometry;Ee.material=this.material;if(void 0!==Ee.material)for(var e=0;e<d;e++)this.getMatrixAt(e,Vi),Wi.multiplyMatrices(c,Vi),Ee.matrixWorld=Wi,Ee.raycast(a,zd),0<zd.length&&(zd[0].instanceId=e,zd[0].object=this,b.push(zd[0]),zd.length=0)},setMatrixAt:function(a,b){b.toArray(this.instanceMatrix.array,16*a)},updateMorphTargets:function(){}});la.prototype=Object.create(K.prototype);la.prototype.constructor=la;la.prototype.isLineBasicMaterial=!0;la.prototype.copy=function(a){K.prototype.copy.call(this,
a);this.color.copy(a.color);this.linewidth=a.linewidth;this.linecap=a.linecap;this.linejoin=a.linejoin;return this};var Xi=new n,Yi=new n,Zi=new P,Pf=new Vb,Fe=new pb;Ka.prototype=Object.assign(Object.create(F.prototype),{constructor:Ka,isLine:!0,computeLineDistances:function(){var a=this.geometry;if(a.isBufferGeometry)if(null===a.index){for(var b=a.attributes.position,c=[0],d=1,e=b.count;d<e;d++)Xi.fromBufferAttribute(b,d-1),Yi.fromBufferAttribute(b,d),c[d]=c[d-1],c[d]+=Xi.distanceTo(Yi);a.setAttribute(&quot;lineDistance&quot;,
new y(c,1))}else console.warn(&quot;THREE.Line.computeLineDistances(): Computation only possible with non-indexed BufferGeometry.&quot;);else if(a.isGeometry)for(b=a.vertices,c=a.lineDistances,c[0]=0,d=1,e=b.length;d<e;d++)c[d]=c[d-1],c[d]+=b[d-1].distanceTo(b[d]);return this},raycast:function(a,b){var c=this.geometry,d=this.matrixWorld,e=a.params.Line.threshold;null===c.boundingSphere&&c.computeBoundingSphere();Fe.copy(c.boundingSphere);Fe.applyMatrix4(d);Fe.radius+=e;if(!1!==a.ray.intersectsSphere(Fe)){Zi.getInverse(d);
Pf.copy(a.ray).applyMatrix4(Zi);d=e/((this.scale.x+this.scale.y+this.scale.z)/3);d*=d;var f=new n,g=new n;e=new n;var h=new n,l=this&&this.isLineSegments?2:1;if(c.isBufferGeometry){var k=c.index,u=c.attributes.position.array;if(null!==k){k=k.array;c=0;for(var p=k.length-1;c<p;c+=l){var x=k[c+1];f.fromArray(u,3*k[c]);g.fromArray(u,3*x);x=Pf.distanceSqToSegment(f,g,h,e);x>d||(h.applyMatrix4(this.matrixWorld),x=a.ray.origin.distanceTo(h),x<a.near||x>a.far||b.push({distance:x,point:e.clone().applyMatrix4(this.matrixWorld),
index:c,face:null,faceIndex:null,object:this}))}}else for(c=0,p=u.length/3-1;c<p;c+=l)f.fromArray(u,3*c),g.fromArray(u,3*c+3),x=Pf.distanceSqToSegment(f,g,h,e),x>d||(h.applyMatrix4(this.matrixWorld),x=a.ray.origin.distanceTo(h),x<a.near||x>a.far||b.push({distance:x,point:e.clone().applyMatrix4(this.matrixWorld),index:c,face:null,faceIndex:null,object:this}))}else if(c.isGeometry)for(f=c.vertices,g=f.length,c=0;c<g-1;c+=l)x=Pf.distanceSqToSegment(f[c],f[c+1],h,e),x>d||(h.applyMatrix4(this.matrixWorld),
x=a.ray.origin.distanceTo(h),x<a.near||x>a.far||b.push({distance:x,point:e.clone().applyMatrix4(this.matrixWorld),index:c,face:null,faceIndex:null,object:this}))}},clone:function(){return(new this.constructor(this.geometry,this.material)).copy(this)}});var Qf=new n,Rf=new n;ma.prototype=Object.assign(Object.create(Ka.prototype),{constructor:ma,isLineSegments:!0,computeLineDistances:function(){var a=this.geometry;if(a.isBufferGeometry)if(null===a.index){for(var b=a.attributes.position,c=[],d=0,e=b.count;d<
e;d+=2)Qf.fromBufferAttribute(b,d),Rf.fromBufferAttribute(b,d+1),c[d]=0===d?0:c[d-1],c[d+1]=c[d]+Qf.distanceTo(Rf);a.setAttribute(&quot;lineDistance&quot;,new y(c,1))}else console.warn(&quot;THREE.LineSegments.computeLineDistances(): Computation only possible with non-indexed BufferGeometry.&quot;);else if(a.isGeometry)for(b=a.vertices,c=a.lineDistances,d=0,e=b.length;d<e;d+=2)Qf.copy(b[d]),Rf.copy(b[d+1]),c[d]=0===d?0:c[d-1],c[d+1]=c[d]+Qf.distanceTo(Rf);return this}});We.prototype=Object.assign(Object.create(Ka.prototype),
{constructor:We,isLineLoop:!0});Va.prototype=Object.create(K.prototype);Va.prototype.constructor=Va;Va.prototype.isPointsMaterial=!0;Va.prototype.copy=function(a){K.prototype.copy.call(this,a);this.color.copy(a.color);this.map=a.map;this.alphaMap=a.alphaMap;this.size=a.size;this.sizeAttenuation=a.sizeAttenuation;this.morphTargets=a.morphTargets;return this};var $i=new P,rg=new Vb,Ge=new pb,Sf=new n;Pc.prototype=Object.assign(Object.create(F.prototype),{constructor:Pc,isPoints:!0,raycast:function(a,
b){var c=this.geometry,d=this.matrixWorld,e=a.params.Points.threshold;null===c.boundingSphere&&c.computeBoundingSphere();Ge.copy(c.boundingSphere);Ge.applyMatrix4(d);Ge.radius+=e;if(!1!==a.ray.intersectsSphere(Ge))if($i.getInverse(d),rg.copy(a.ray).applyMatrix4($i),e/=(this.scale.x+this.scale.y+this.scale.z)/3,e*=e,c.isBufferGeometry){var f=c.index;c=c.attributes.position.array;if(null!==f){var g=f.array;f=0;for(var h=g.length;f<h;f++){var l=g[f];Sf.fromArray(c,3*l);qg(Sf,l,e,d,a,b,this)}}else for(f=
0,g=c.length/3;f<g;f++)Sf.fromArray(c,3*f),qg(Sf,f,e,d,a,b,this)}else for(c=c.vertices,f=0,g=c.length;f<g;f++)qg(c[f],f,e,d,a,b,this)},updateMorphTargets:function(){var a=this.geometry;if(a.isBufferGeometry){a=a.morphAttributes;var b=Object.keys(a);if(0<b.length){var c=a[b[0]];if(void 0!==c)for(this.morphTargetInfluences=[],this.morphTargetDictionary={},a=0,b=c.length;a<b;a++){var d=c[a].name||String(a);this.morphTargetInfluences.push(0);this.morphTargetDictionary[d]=a}}}else a=a.morphTargets,void 0!==
a&&0<a.length&&console.error(&quot;THREE.Points.updateMorphTargets() does not support THREE.Geometry. Use THREE.BufferGeometry instead.&quot;)},clone:function(){return(new this.constructor(this.geometry,this.material)).copy(this)}});sg.prototype=Object.assign(Object.create(V.prototype),{constructor:sg,isVideoTexture:!0,update:function(){var a=this.image;a.readyState>=a.HAVE_CURRENT_DATA&&(this.needsUpdate=!0)}});Qc.prototype=Object.create(V.prototype);Qc.prototype.constructor=Qc;Qc.prototype.isCompressedTexture=
!0;Sd.prototype=Object.create(V.prototype);Sd.prototype.constructor=Sd;Sd.prototype.isCanvasTexture=!0;Td.prototype=Object.create(V.prototype);Td.prototype.constructor=Td;Td.prototype.isDepthTexture=!0;Rc.prototype=Object.create(C.prototype);Rc.prototype.constructor=Rc;Ud.prototype=Object.create(N.prototype);Ud.prototype.constructor=Ud;Sc.prototype=Object.create(C.prototype);Sc.prototype.constructor=Sc;Vd.prototype=Object.create(N.prototype);Vd.prototype.constructor=Vd;Ga.prototype=Object.create(C.prototype);
Ga.prototype.constructor=Ga;Wd.prototype=Object.create(N.prototype);Wd.prototype.constructor=Wd;Tc.prototype=Object.create(Ga.prototype);Tc.prototype.constructor=Tc;Xd.prototype=Object.create(N.prototype);Xd.prototype.constructor=Xd;cc.prototype=Object.create(Ga.prototype);cc.prototype.constructor=cc;Yd.prototype=Object.create(N.prototype);Yd.prototype.constructor=Yd;Uc.prototype=Object.create(Ga.prototype);Uc.prototype.constructor=Uc;Zd.prototype=Object.create(N.prototype);Zd.prototype.constructor=
Zd;Vc.prototype=Object.create(Ga.prototype);Vc.prototype.constructor=Vc;$d.prototype=Object.create(N.prototype);$d.prototype.constructor=$d;dc.prototype=Object.create(C.prototype);dc.prototype.constructor=dc;dc.prototype.toJSON=function(){var a=C.prototype.toJSON.call(this);a.path=this.parameters.path.toJSON();return a};ae.prototype=Object.create(N.prototype);ae.prototype.constructor=ae;Wc.prototype=Object.create(C.prototype);Wc.prototype.constructor=Wc;be.prototype=Object.create(N.prototype);be.prototype.constructor=
be;Xc.prototype=Object.create(C.prototype);Xc.prototype.constructor=Xc;var al={triangulate:function(a,b,c){c=c||2;var d=b&&b.length,e=d?b[0]*c:a.length,f=ci(a,0,e,c,!0),g=[];if(!f||f.next===f.prev)return g;var h;if(d){var l=c;d=[];var k;var n=0;for(k=b.length;n<k;n++){var p=b[n]*l;var x=n<k-1?b[n+1]*l:a.length;p=ci(a,p,x,l,!1);p===p.next&&(p.steiner=!0);d.push(Nk(p))}d.sort(Lk);for(n=0;n<d.length;n++){b=d[n];l=f;if(l=Mk(b,l))b=fi(l,b),de(b,b.next);f=de(f,f.next)}}if(a.length>80*c){var r=h=a[0];var q=
d=a[1];for(l=c;l<e;l+=c)n=a[l],b=a[l+1],n<r&&(r=n),b<q&&(q=b),n>h&&(h=n),b>d&&(d=b);h=Math.max(h-r,d-q);h=0!==h?1/h:0}ee(f,g,c,r,q,h);return g}},sb={area:function(a){for(var b=a.length,c=0,d=b-1,e=0;e<b;d=e++)c+=a[d].x*a[e].y-a[e].x*a[d].y;return.5*c},isClockWise:function(a){return 0>sb.area(a)},triangulateShape:function(a,b){var c=[],d=[],e=[];gi(a);hi(c,a);var f=a.length;b.forEach(gi);for(a=0;a<b.length;a++)d.push(f),f+=b[a].length,hi(c,b[a]);b=al.triangulate(c,d);for(a=0;a<b.length;a+=3)e.push(b.slice(a,
a+3));return e}};fc.prototype=Object.create(N.prototype);fc.prototype.constructor=fc;fc.prototype.toJSON=function(){var a=N.prototype.toJSON.call(this);return ii(this.parameters.shapes,this.parameters.options,a)};fb.prototype=Object.create(C.prototype);fb.prototype.constructor=fb;fb.prototype.toJSON=function(){var a=C.prototype.toJSON.call(this);return ii(this.parameters.shapes,this.parameters.options,a)};var Ok={generateTopUV:function(a,b,c,d,e){a=b[3*d];d=b[3*d+1];var f=b[3*e];e=b[3*e+1];return[new t(b[3*
c],b[3*c+1]),new t(a,d),new t(f,e)]},generateSideWallUV:function(a,b,c,d,e,f){a=b[3*c];var g=b[3*c+1];c=b[3*c+2];var h=b[3*d],l=b[3*d+1];d=b[3*d+2];var k=b[3*e],n=b[3*e+1];e=b[3*e+2];var p=b[3*f],x=b[3*f+1];b=b[3*f+2];return.01>Math.abs(g-l)?[new t(a,1-c),new t(h,1-d),new t(k,1-e),new t(p,1-b)]:[new t(g,1-c),new t(l,1-d),new t(n,1-e),new t(x,1-b)]}};ge.prototype=Object.create(N.prototype);ge.prototype.constructor=ge;Zc.prototype=Object.create(fb.prototype);Zc.prototype.constructor=Zc;he.prototype=
Object.create(N.prototype);he.prototype.constructor=he;gc.prototype=Object.create(C.prototype);gc.prototype.constructor=gc;ie.prototype=Object.create(N.prototype);ie.prototype.constructor=ie;$c.prototype=Object.create(C.prototype);$c.prototype.constructor=$c;je.prototype=Object.create(N.prototype);je.prototype.constructor=je;ad.prototype=Object.create(C.prototype);ad.prototype.constructor=ad;hc.prototype=Object.create(N.prototype);hc.prototype.constructor=hc;hc.prototype.toJSON=function(){var a=N.prototype.toJSON.call(this);
return ji(this.parameters.shapes,a)};ic.prototype=Object.create(C.prototype);ic.prototype.constructor=ic;ic.prototype.toJSON=function(){var a=C.prototype.toJSON.call(this);return ji(this.parameters.shapes,a)};bd.prototype=Object.create(C.prototype);bd.prototype.constructor=bd;jc.prototype=Object.create(N.prototype);jc.prototype.constructor=jc;tb.prototype=Object.create(C.prototype);tb.prototype.constructor=tb;ke.prototype=Object.create(jc.prototype);ke.prototype.constructor=ke;le.prototype=Object.create(tb.prototype);
le.prototype.constructor=le;me.prototype=Object.create(N.prototype);me.prototype.constructor=me;cd.prototype=Object.create(C.prototype);cd.prototype.constructor=cd;var ua=Object.freeze({__proto__:null,WireframeGeometry:Rc,ParametricGeometry:Ud,ParametricBufferGeometry:Sc,TetrahedronGeometry:Wd,TetrahedronBufferGeometry:Tc,OctahedronGeometry:Xd,OctahedronBufferGeometry:cc,IcosahedronGeometry:Yd,IcosahedronBufferGeometry:Uc,DodecahedronGeometry:Zd,DodecahedronBufferGeometry:Vc,PolyhedronGeometry:Vd,
PolyhedronBufferGeometry:Ga,TubeGeometry:$d,TubeBufferGeometry:dc,TorusKnotGeometry:ae,TorusKnotBufferGeometry:Wc,TorusGeometry:be,TorusBufferGeometry:Xc,TextGeometry:ge,TextBufferGeometry:Zc,SphereGeometry:he,SphereBufferGeometry:gc,RingGeometry:ie,RingBufferGeometry:$c,PlaneGeometry:Id,PlaneBufferGeometry:bc,LatheGeometry:je,LatheBufferGeometry:ad,ShapeGeometry:hc,ShapeBufferGeometry:ic,ExtrudeGeometry:fc,ExtrudeBufferGeometry:fb,EdgesGeometry:bd,ConeGeometry:ke,ConeBufferGeometry:le,CylinderGeometry:jc,
CylinderBufferGeometry:tb,CircleGeometry:me,CircleBufferGeometry:cd,BoxGeometry:nh,BoxBufferGeometry:Jd});kc.prototype=Object.create(K.prototype);kc.prototype.constructor=kc;kc.prototype.isShadowMaterial=!0;kc.prototype.copy=function(a){K.prototype.copy.call(this,a);this.color.copy(a.color);return this};ub.prototype=Object.create(Ba.prototype);ub.prototype.constructor=ub;ub.prototype.isRawShaderMaterial=!0;gb.prototype=Object.create(K.prototype);gb.prototype.constructor=gb;gb.prototype.isMeshStandardMaterial=
!0;gb.prototype.copy=function(a){K.prototype.copy.call(this,a);this.defines={STANDARD:&quot;&quot;};this.color.copy(a.color);this.roughness=a.roughness;this.metalness=a.metalness;this.map=a.map;this.lightMap=a.lightMap;this.lightMapIntensity=a.lightMapIntensity;this.aoMap=a.aoMap;this.aoMapIntensity=a.aoMapIntensity;this.emissive.copy(a.emissive);this.emissiveMap=a.emissiveMap;this.emissiveIntensity=a.emissiveIntensity;this.bumpMap=a.bumpMap;this.bumpScale=a.bumpScale;this.normalMap=a.normalMap;this.normalMapType=
a.normalMapType;this.normalScale.copy(a.normalScale);this.displacementMap=a.displacementMap;this.displacementScale=a.displacementScale;this.displacementBias=a.displacementBias;this.roughnessMap=a.roughnessMap;this.metalnessMap=a.metalnessMap;this.alphaMap=a.alphaMap;this.envMap=a.envMap;this.envMapIntensity=a.envMapIntensity;this.refractionRatio=a.refractionRatio;this.wireframe=a.wireframe;this.wireframeLinewidth=a.wireframeLinewidth;this.wireframeLinecap=a.wireframeLinecap;this.wireframeLinejoin=
a.wireframeLinejoin;this.skinning=a.skinning;this.morphTargets=a.morphTargets;this.morphNormals=a.morphNormals;this.vertexTangents=a.vertexTangents;return this};lc.prototype=Object.create(gb.prototype);lc.prototype.constructor=lc;lc.prototype.isMeshPhysicalMaterial=!0;lc.prototype.copy=function(a){gb.prototype.copy.call(this,a);this.defines={STANDARD:&quot;&quot;,PHYSICAL:&quot;&quot;};this.clearcoat=a.clearcoat;this.clearcoatMap=a.clearcoatMap;this.clearcoatRoughness=a.clearcoatRoughness;this.clearcoatRoughnessMap=
a.clearcoatRoughnessMap;this.clearcoatNormalMap=a.clearcoatNormalMap;this.clearcoatNormalScale.copy(a.clearcoatNormalScale);this.reflectivity=a.reflectivity;this.sheen=a.sheen?(this.sheen||new A).copy(a.sheen):null;this.transparency=a.transparency;return this};Jb.prototype=Object.create(K.prototype);Jb.prototype.constructor=Jb;Jb.prototype.isMeshPhongMaterial=!0;Jb.prototype.copy=function(a){K.prototype.copy.call(this,a);this.color.copy(a.color);this.specular.copy(a.specular);this.shininess=a.shininess;
this.map=a.map;this.lightMap=a.lightMap;this.lightMapIntensity=a.lightMapIntensity;this.aoMap=a.aoMap;this.aoMapIntensity=a.aoMapIntensity;this.emissive.copy(a.emissive);this.emissiveMap=a.emissiveMap;this.emissiveIntensity=a.emissiveIntensity;this.bumpMap=a.bumpMap;this.bumpScale=a.bumpScale;this.normalMap=a.normalMap;this.normalMapType=a.normalMapType;this.normalScale.copy(a.normalScale);this.displacementMap=a.displacementMap;this.displacementScale=a.displacementScale;this.displacementBias=a.displacementBias;
this.specularMap=a.specularMap;this.alphaMap=a.alphaMap;this.envMap=a.envMap;this.combine=a.combine;this.reflectivity=a.reflectivity;this.refractionRatio=a.refractionRatio;this.wireframe=a.wireframe;this.wireframeLinewidth=a.wireframeLinewidth;this.wireframeLinecap=a.wireframeLinecap;this.wireframeLinejoin=a.wireframeLinejoin;this.skinning=a.skinning;this.morphTargets=a.morphTargets;this.morphNormals=a.morphNormals;return this};mc.prototype=Object.create(K.prototype);mc.prototype.constructor=mc;mc.prototype.isMeshToonMaterial=
!0;mc.prototype.copy=function(a){K.prototype.copy.call(this,a);this.color.copy(a.color);this.specular.copy(a.specular);this.shininess=a.shininess;this.map=a.map;this.gradientMap=a.gradientMap;this.lightMap=a.lightMap;this.lightMapIntensity=a.lightMapIntensity;this.aoMap=a.aoMap;this.aoMapIntensity=a.aoMapIntensity;this.emissive.copy(a.emissive);this.emissiveMap=a.emissiveMap;this.emissiveIntensity=a.emissiveIntensity;this.bumpMap=a.bumpMap;this.bumpScale=a.bumpScale;this.normalMap=a.normalMap;this.normalMapType=
a.normalMapType;this.normalScale.copy(a.normalScale);this.displacementMap=a.displacementMap;this.displacementScale=a.displacementScale;this.displacementBias=a.displacementBias;this.specularMap=a.specularMap;this.alphaMap=a.alphaMap;this.wireframe=a.wireframe;this.wireframeLinewidth=a.wireframeLinewidth;this.wireframeLinecap=a.wireframeLinecap;this.wireframeLinejoin=a.wireframeLinejoin;this.skinning=a.skinning;this.morphTargets=a.morphTargets;this.morphNormals=a.morphNormals;return this};nc.prototype=
Object.create(K.prototype);nc.prototype.constructor=nc;nc.prototype.isMeshNormalMaterial=!0;nc.prototype.copy=function(a){K.prototype.copy.call(this,a);this.bumpMap=a.bumpMap;this.bumpScale=a.bumpScale;this.normalMap=a.normalMap;this.normalMapType=a.normalMapType;this.normalScale.copy(a.normalScale);this.displacementMap=a.displacementMap;this.displacementScale=a.displacementScale;this.displacementBias=a.displacementBias;this.wireframe=a.wireframe;this.wireframeLinewidth=a.wireframeLinewidth;this.skinning=
a.skinning;this.morphTargets=a.morphTargets;this.morphNormals=a.morphNormals;return this};oc.prototype=Object.create(K.prototype);oc.prototype.constructor=oc;oc.prototype.isMeshLambertMaterial=!0;oc.prototype.copy=function(a){K.prototype.copy.call(this,a);this.color.copy(a.color);this.map=a.map;this.lightMap=a.lightMap;this.lightMapIntensity=a.lightMapIntensity;this.aoMap=a.aoMap;this.aoMapIntensity=a.aoMapIntensity;this.emissive.copy(a.emissive);this.emissiveMap=a.emissiveMap;this.emissiveIntensity=
a.emissiveIntensity;this.specularMap=a.specularMap;this.alphaMap=a.alphaMap;this.envMap=a.envMap;this.combine=a.combine;this.reflectivity=a.reflectivity;this.refractionRatio=a.refractionRatio;this.wireframe=a.wireframe;this.wireframeLinewidth=a.wireframeLinewidth;this.wireframeLinecap=a.wireframeLinecap;this.wireframeLinejoin=a.wireframeLinejoin;this.skinning=a.skinning;this.morphTargets=a.morphTargets;this.morphNormals=a.morphNormals;return this};pc.prototype=Object.create(K.prototype);pc.prototype.constructor=
pc;pc.prototype.isMeshMatcapMaterial=!0;pc.prototype.copy=function(a){K.prototype.copy.call(this,a);this.defines={MATCAP:&quot;&quot;};this.color.copy(a.color);this.matcap=a.matcap;this.map=a.map;this.bumpMap=a.bumpMap;this.bumpScale=a.bumpScale;this.normalMap=a.normalMap;this.normalMapType=a.normalMapType;this.normalScale.copy(a.normalScale);this.displacementMap=a.displacementMap;this.displacementScale=a.displacementScale;this.displacementBias=a.displacementBias;this.alphaMap=a.alphaMap;this.skinning=a.skinning;
this.morphTargets=a.morphTargets;this.morphNormals=a.morphNormals;return this};qc.prototype=Object.create(la.prototype);qc.prototype.constructor=qc;qc.prototype.isLineDashedMaterial=!0;qc.prototype.copy=function(a){la.prototype.copy.call(this,a);this.scale=a.scale;this.dashSize=a.dashSize;this.gapSize=a.gapSize;return this};var bl=Object.freeze({__proto__:null,ShadowMaterial:kc,SpriteMaterial:Ib,RawShaderMaterial:ub,ShaderMaterial:Ba,PointsMaterial:Va,MeshPhysicalMaterial:lc,MeshStandardMaterial:gb,
MeshPhongMaterial:Jb,MeshToonMaterial:mc,MeshNormalMaterial:nc,MeshLambertMaterial:oc,MeshDepthMaterial:Fb,MeshDistanceMaterial:Gb,MeshBasicMaterial:Oa,MeshMatcapMaterial:pc,LineDashedMaterial:qc,LineBasicMaterial:la,Material:K}),R={arraySlice:function(a,b,c){return R.isTypedArray(a)?new a.constructor(a.subarray(b,void 0!==c?c:a.length)):a.slice(b,c)},convertArray:function(a,b,c){return!a||!c&&a.constructor===b?a:&quot;number&quot;===typeof b.BYTES_PER_ELEMENT?new b(a):Array.prototype.slice.call(a)},isTypedArray:function(a){return ArrayBuffer.isView(a)&&
!(a instanceof DataView)},getKeyframeOrder:function(a){for(var b=a.length,c=Array(b),d=0;d!==b;++d)c[d]=d;c.sort(function(b,c){return a[b]-a[c]});return c},sortedArray:function(a,b,c){for(var d=a.length,e=new a.constructor(d),f=0,g=0;g!==d;++f)for(var h=c[f]*b,l=0;l!==b;++l)e[g++]=a[h+l];return e},flattenJSON:function(a,b,c,d){for(var e=1,f=a[0];void 0!==f&&void 0===f[d];)f=a[e++];if(void 0!==f){var g=f[d];if(void 0!==g)if(Array.isArray(g)){do g=f[d],void 0!==g&&(b.push(f.time),c.push.apply(c,g)),
f=a[e++];while(void 0!==f)}else if(void 0!==g.toArray){do g=f[d],void 0!==g&&(b.push(f.time),g.toArray(c,c.length)),f=a[e++];while(void 0!==f)}else{do g=f[d],void 0!==g&&(b.push(f.time),c.push(g)),f=a[e++];while(void 0!==f)}}},subclip:function(a,b,c,d,e){e=e||30;a=a.clone();a.name=b;var f=[];for(b=0;b<a.tracks.length;++b){for(var g=a.tracks[b],h=g.getValueSize(),l=[],k=[],n=0;n<g.times.length;++n){var p=g.times[n]*e;if(!(p<c||p>=d))for(l.push(g.times[n]),p=0;p<h;++p)k.push(g.values[n*h+p])}0!==l.length&&
(g.times=R.convertArray(l,g.times.constructor),g.values=R.convertArray(k,g.values.constructor),f.push(g))}a.tracks=f;c=Infinity;for(b=0;b<a.tracks.length;++b)c>a.tracks[b].times[0]&&(c=a.tracks[b].times[0]);for(b=0;b<a.tracks.length;++b)a.tracks[b].shift(-1*c);a.resetDuration();return a}};Object.assign(La.prototype,{evaluate:function(a){var b=this.parameterPositions,c=this._cachedIndex,d=b[c],e=b[c-1];a:{b:{c:{d:if(!(a<d)){for(var f=c+2;;){if(void 0===d){if(a<e)break d;this._cachedIndex=c=b.length;
return this.afterEnd_(c-1,a,e)}if(c===f)break;e=d;d=b[++c];if(a<d)break b}d=b.length;break c}if(a>=e)break a;else{f=b[1];a<f&&(c=2,e=f);for(f=c-2;;){if(void 0===e)return this._cachedIndex=0,this.beforeStart_(0,a,d);if(c===f)break;d=e;e=b[--c-1];if(a>=e)break b}d=c;c=0}}for(;c<d;)e=c+d>>>1,a<b[e]?d=e:c=e+1;d=b[c];e=b[c-1];if(void 0===e)return this._cachedIndex=0,this.beforeStart_(0,a,d);if(void 0===d)return this._cachedIndex=c=b.length,this.afterEnd_(c-1,e,a)}this._cachedIndex=c;this.intervalChanged_(c,
e,d)}return this.interpolate_(c,e,a,d)},settings:null,DefaultSettings_:{},getSettings_:function(){return this.settings||this.DefaultSettings_},copySampleValue_:function(a){var b=this.resultBuffer,c=this.sampleValues,d=this.valueSize;a*=d;for(var e=0;e!==d;++e)b[e]=c[a+e];return b},interpolate_:function(){throw Error(&quot;call to abstract method&quot;);},intervalChanged_:function(){}});Object.assign(La.prototype,{beforeStart_:La.prototype.copySampleValue_,afterEnd_:La.prototype.copySampleValue_});Xe.prototype=
Object.assign(Object.create(La.prototype),{constructor:Xe,DefaultSettings_:{endingStart:2400,endingEnd:2400},intervalChanged_:function(a,b,c){var d=this.parameterPositions,e=a-2,f=a+1,g=d[e],h=d[f];if(void 0===g)switch(this.getSettings_().endingStart){case 2401:e=a;g=2*b-c;break;case 2402:e=d.length-2;g=b+d[e]-d[e+1];break;default:e=a,g=c}if(void 0===h)switch(this.getSettings_().endingEnd){case 2401:f=a;h=2*c-b;break;case 2402:f=1;h=c+d[1]-d[0];break;default:f=a-1,h=b}a=.5*(c-b);d=this.valueSize;
this._weightPrev=a/(b-g);this._weightNext=a/(h-c);this._offsetPrev=e*d;this._offsetNext=f*d},interpolate_:function(a,b,c,d){var e=this.resultBuffer,f=this.sampleValues,g=this.valueSize;a*=g;var h=a-g,l=this._offsetPrev,k=this._offsetNext,n=this._weightPrev,p=this._weightNext,x=(c-b)/(d-b);c=x*x;d=c*x;b=-n*d+2*n*c-n*x;n=(1+n)*d+(-1.5-2*n)*c+(-.5+n)*x+1;x=(-1-p)*d+(1.5+p)*c+.5*x;p=p*d-p*c;for(c=0;c!==g;++c)e[c]=b*f[l+c]+n*f[h+c]+x*f[a+c]+p*f[k+c];return e}});ne.prototype=Object.assign(Object.create(La.prototype),
{constructor:ne,interpolate_:function(a,b,c,d){var e=this.resultBuffer,f=this.sampleValues,g=this.valueSize;a*=g;var h=a-g;b=(c-b)/(d-b);c=1-b;for(d=0;d!==g;++d)e[d]=f[h+d]*c+f[a+d]*b;return e}});Ye.prototype=Object.assign(Object.create(La.prototype),{constructor:Ye,interpolate_:function(a){return this.copySampleValue_(a-1)}});Object.assign(sa,{toJSON:function(a){var b=a.constructor;if(void 0!==b.toJSON)b=b.toJSON(a);else{b={name:a.name,times:R.convertArray(a.times,Array),values:R.convertArray(a.values,
Array)};var c=a.getInterpolation();c!==a.DefaultInterpolation&&(b.interpolation=c)}b.type=a.ValueTypeName;return b}});Object.assign(sa.prototype,{constructor:sa,TimeBufferType:Float32Array,ValueBufferType:Float32Array,DefaultInterpolation:2301,InterpolantFactoryMethodDiscrete:function(a){return new Ye(this.times,this.values,this.getValueSize(),a)},InterpolantFactoryMethodLinear:function(a){return new ne(this.times,this.values,this.getValueSize(),a)},InterpolantFactoryMethodSmooth:function(a){return new Xe(this.times,
this.values,this.getValueSize(),a)},setInterpolation:function(a){switch(a){case 2300:var b=this.InterpolantFactoryMethodDiscrete;break;case 2301:b=this.InterpolantFactoryMethodLinear;break;case 2302:b=this.InterpolantFactoryMethodSmooth}if(void 0===b){b=&quot;unsupported interpolation for &quot;+this.ValueTypeName+&quot; keyframe track named &quot;+this.name;if(void 0===this.createInterpolant)if(a!==this.DefaultInterpolation)this.setInterpolation(this.DefaultInterpolation);else throw Error(b);console.warn(&quot;THREE.KeyframeTrack:&quot;,
b);return this}this.createInterpolant=b;return this},getInterpolation:function(){switch(this.createInterpolant){case this.InterpolantFactoryMethodDiscrete:return 2300;case this.InterpolantFactoryMethodLinear:return 2301;case this.InterpolantFactoryMethodSmooth:return 2302}},getValueSize:function(){return this.values.length/this.times.length},shift:function(a){if(0!==a)for(var b=this.times,c=0,d=b.length;c!==d;++c)b[c]+=a;return this},scale:function(a){if(1!==a)for(var b=this.times,c=0,d=b.length;c!==
d;++c)b[c]*=a;return this},trim:function(a,b){for(var c=this.times,d=c.length,e=0,f=d-1;e!==d&&c[e]<a;)++e;for(;-1!==f&&c[f]>b;)--f;++f;if(0!==e||f!==d)e>=f&&(f=Math.max(f,1),e=f-1),a=this.getValueSize(),this.times=R.arraySlice(c,e,f),this.values=R.arraySlice(this.values,e*a,f*a);return this},validate:function(){var a=!0,b=this.getValueSize();0!==b-Math.floor(b)&&(console.error(&quot;THREE.KeyframeTrack: Invalid value size in track.&quot;,this),a=!1);var c=this.times;b=this.values;var d=c.length;0===d&&(console.error(&quot;THREE.KeyframeTrack: Track is empty.&quot;,
this),a=!1);for(var e=null,f=0;f!==d;f++){var g=c[f];if(&quot;number&quot;===typeof g&&isNaN(g)){console.error(&quot;THREE.KeyframeTrack: Time is not a valid number.&quot;,this,f,g);a=!1;break}if(null!==e&&e>g){console.error(&quot;THREE.KeyframeTrack: Out of order keys.&quot;,this,f,g,e);a=!1;break}e=g}if(void 0!==b&&R.isTypedArray(b))for(f=0,c=b.length;f!==c;++f)if(d=b[f],isNaN(d)){console.error(&quot;THREE.KeyframeTrack: Value is not a valid number.&quot;,this,f,d);a=!1;break}return a},optimize:function(){for(var a=R.arraySlice(this.times),
b=R.arraySlice(this.values),c=this.getValueSize(),d=2302===this.getInterpolation(),e=1,f=a.length-1,g=1;g<f;++g){var h=!1,l=a[g];if(l!==a[g+1]&&(1!==g||l!==l[0]))if(d)h=!0;else{var k=g*c,n=k-c,p=k+c;for(l=0;l!==c;++l){var x=b[k+l];if(x!==b[n+l]||x!==b[p+l]){h=!0;break}}}if(h){if(g!==e)for(a[e]=a[g],h=g*c,k=e*c,l=0;l!==c;++l)b[k+l]=b[h+l];++e}}if(0<f){a[e]=a[f];h=f*c;k=e*c;for(l=0;l!==c;++l)b[k+l]=b[h+l];++e}e!==a.length?(this.times=R.arraySlice(a,0,e),this.values=R.arraySlice(b,0,e*c)):(this.times=
a,this.values=b);return this},clone:function(){var a=R.arraySlice(this.times,0),b=R.arraySlice(this.values,0);a=new this.constructor(this.name,a,b);a.createInterpolant=this.createInterpolant;return a}});Ze.prototype=Object.assign(Object.create(sa.prototype),{constructor:Ze,ValueTypeName:&quot;bool&quot;,ValueBufferType:Array,DefaultInterpolation:2300,InterpolantFactoryMethodLinear:void 0,InterpolantFactoryMethodSmooth:void 0});$e.prototype=Object.assign(Object.create(sa.prototype),{constructor:$e,ValueTypeName:&quot;color&quot;});
dd.prototype=Object.assign(Object.create(sa.prototype),{constructor:dd,ValueTypeName:&quot;number&quot;});af.prototype=Object.assign(Object.create(La.prototype),{constructor:af,interpolate_:function(a,b,c,d){var e=this.resultBuffer,f=this.sampleValues,g=this.valueSize;a*=g;b=(c-b)/(d-b);for(c=a+g;a!==c;a+=4)Aa.slerpFlat(e,0,f,a-g,f,a,b);return e}});oe.prototype=Object.assign(Object.create(sa.prototype),{constructor:oe,ValueTypeName:&quot;quaternion&quot;,DefaultInterpolation:2301,InterpolantFactoryMethodLinear:function(a){return new af(this.times,
this.values,this.getValueSize(),a)},InterpolantFactoryMethodSmooth:void 0});bf.prototype=Object.assign(Object.create(sa.prototype),{constructor:bf,ValueTypeName:&quot;string&quot;,ValueBufferType:Array,DefaultInterpolation:2300,InterpolantFactoryMethodLinear:void 0,InterpolantFactoryMethodSmooth:void 0});ed.prototype=Object.assign(Object.create(sa.prototype),{constructor:ed,ValueTypeName:&quot;vector&quot;});Object.assign(Qa,{parse:function(a){for(var b=[],c=a.tracks,d=1/(a.fps||1),e=0,f=c.length;e!==f;++e)b.push(Qk(c[e]).scale(d));
return new Qa(a.name,a.duration,b)},toJSON:function(a){var b=[],c=a.tracks;a={name:a.name,duration:a.duration,tracks:b,uuid:a.uuid};for(var d=0,e=c.length;d!==e;++d)b.push(sa.toJSON(c[d]));return a},CreateFromMorphTargetSequence:function(a,b,c,d){for(var e=b.length,f=[],g=0;g<e;g++){var h=[],l=[];h.push((g+e-1)%e,g,(g+1)%e);l.push(0,1,0);var k=R.getKeyframeOrder(h);h=R.sortedArray(h,1,k);l=R.sortedArray(l,1,k);d||0!==h[0]||(h.push(e),l.push(l[0]));f.push((new dd(&quot;.morphTargetInfluences[&quot;+b[g].name+
&quot;]&quot;,h,l)).scale(1/c))}return new Qa(a,-1,f)},findByName:function(a,b){var c=a;Array.isArray(a)||(c=a.geometry&&a.geometry.animations||a.animations);for(a=0;a<c.length;a++)if(c[a].name===b)return c[a];return null},CreateClipsFromMorphTargetSequences:function(a,b,c){for(var d={},e=/^([\w-]*?)([\d]+)$/,f=0,g=a.length;f<g;f++){var h=a[f],l=h.name.match(e);if(l&&1<l.length){var k=l[1];(l=d[k])||(d[k]=l=[]);l.push(h)}}a=[];for(k in d)a.push(Qa.CreateFromMorphTargetSequence(k,d[k],b,c));return a},parseAnimation:function(a,
b){if(!a)return console.error(&quot;THREE.AnimationClip: No animation in JSONLoader data.&quot;),null;var c=function(a,b,c,d,e){if(0!==c.length){var f=[],g=[];R.flattenJSON(c,f,g,d);0!==f.length&&e.push(new a(b,f,g))}},d=[],e=a.name||&quot;default&quot;,f=a.length||-1,g=a.fps||30;a=a.hierarchy||[];for(var h=0;h<a.length;h++){var l=a[h].keys;if(l&&0!==l.length)if(l[0].morphTargets){f={};for(var k=0;k<l.length;k++)if(l[k].morphTargets)for(var n=0;n<l[k].morphTargets.length;n++)f[l[k].morphTargets[n]]=-1;for(var p in f){var x=
[],r=[];for(n=0;n!==l[k].morphTargets.length;++n){var q=l[k];x.push(q.time);r.push(q.morphTarget===p?1:0)}d.push(new dd(&quot;.morphTargetInfluence[&quot;+p+&quot;]&quot;,x,r))}f=f.length*(g||1)}else k=&quot;.bones[&quot;+b[h].name+&quot;]&quot;,c(ed,k+&quot;.position&quot;,l,&quot;pos&quot;,d),c(oe,k+&quot;.quaternion&quot;,l,&quot;rot&quot;,d),c(ed,k+&quot;.scale&quot;,l,&quot;scl&quot;,d)}return 0===d.length?null:new Qa(e,f,d)}});Object.assign(Qa.prototype,{resetDuration:function(){for(var a=0,b=0,c=this.tracks.length;b!==c;++b){var d=this.tracks[b];a=Math.max(a,d.times[d.times.length-1])}this.duration=
a;return this},trim:function(){for(var a=0;a<this.tracks.length;a++)this.tracks[a].trim(0,this.duration);return this},validate:function(){for(var a=!0,b=0;b<this.tracks.length;b++)a=a&&this.tracks[b].validate();return a},optimize:function(){for(var a=0;a<this.tracks.length;a++)this.tracks[a].optimize();return this},clone:function(){for(var a=[],b=0;b<this.tracks.length;b++)a.push(this.tracks[b].clone());return new Qa(this.name,this.duration,a)}});var vc={enabled:!1,files:{},add:function(a,b){!1!==
this.enabled&&(this.files[a]=b)},get:function(a){if(!1!==this.enabled)return this.files[a]},remove:function(a){delete this.files[a]},clear:function(){this.files={}}},ki=new vg;Object.assign(W.prototype,{load:function(){},parse:function(){},setCrossOrigin:function(a){this.crossOrigin=a;return this},setPath:function(a){this.path=a;return this},setResourcePath:function(a){this.resourcePath=a;return this}});var cb={};Ra.prototype=Object.assign(Object.create(W.prototype),{constructor:Ra,load:function(a,
b,c,d){void 0===a&&(a=&quot;&quot;);void 0!==this.path&&(a=this.path+a);a=this.manager.resolveURL(a);var e=this,f=vc.get(a);if(void 0!==f)return e.manager.itemStart(a),setTimeout(function(){b&&b(f);e.manager.itemEnd(a)},0),f;if(void 0!==cb[a])cb[a].push({onLoad:b,onProgress:c,onError:d});else{var g=a.match(/^data:(.*?)(;base64)?,(.*)$/);if(g){c=g[1];var h=!!g[2];g=g[3];g=decodeURIComponent(g);h&&(g=atob(g));try{var l=(this.responseType||&quot;&quot;).toLowerCase();switch(l){case &quot;arraybuffer&quot;:case &quot;blob&quot;:var k=new Uint8Array(g.length);
for(h=0;h<g.length;h++)k[h]=g.charCodeAt(h);var n=&quot;blob&quot;===l?new Blob([k.buffer],{type:c}):k.buffer;break;case &quot;document&quot;:n=(new DOMParser).parseFromString(g,c);break;case &quot;json&quot;:n=JSON.parse(g);break;default:n=g}setTimeout(function(){b&&b(n);e.manager.itemEnd(a)},0)}catch(x){setTimeout(function(){d&&d(x);e.manager.itemError(a);e.manager.itemEnd(a)},0)}}else{cb[a]=[];cb[a].push({onLoad:b,onProgress:c,onError:d});var p=new XMLHttpRequest;p.open(&quot;GET&quot;,a,!0);p.addEventListener(&quot;load&quot;,function(b){var c=
this.response,d=cb[a];delete cb[a];if(200===this.status||0===this.status){0===this.status&&console.warn(&quot;THREE.FileLoader: HTTP Status 0 received.&quot;);vc.add(a,c);for(var f=0,g=d.length;f<g;f++){var h=d[f];if(h.onLoad)h.onLoad(c)}}else{f=0;for(g=d.length;f<g;f++)if(h=d[f],h.onError)h.onError(b);e.manager.itemError(a)}e.manager.itemEnd(a)},!1);p.addEventListener(&quot;progress&quot;,function(b){for(var c=cb[a],d=0,e=c.length;d<e;d++){var f=c[d];if(f.onProgress)f.onProgress(b)}},!1);p.addEventListener(&quot;error&quot;,
function(b){var c=cb[a];delete cb[a];for(var d=0,f=c.length;d<f;d++){var g=c[d];if(g.onError)g.onError(b)}e.manager.itemError(a);e.manager.itemEnd(a)},!1);p.addEventListener(&quot;abort&quot;,function(b){var c=cb[a];delete cb[a];for(var d=0,f=c.length;d<f;d++){var g=c[d];if(g.onError)g.onError(b)}e.manager.itemError(a);e.manager.itemEnd(a)},!1);void 0!==this.responseType&&(p.responseType=this.responseType);void 0!==this.withCredentials&&(p.withCredentials=this.withCredentials);p.overrideMimeType&&p.overrideMimeType(void 0!==
this.mimeType?this.mimeType:&quot;text/plain&quot;);for(h in this.requestHeader)p.setRequestHeader(h,this.requestHeader[h]);p.send(null)}e.manager.itemStart(a);return p}},setResponseType:function(a){this.responseType=a;return this},setWithCredentials:function(a){this.withCredentials=a;return this},setMimeType:function(a){this.mimeType=a;return this},setRequestHeader:function(a){this.requestHeader=a;return this}});wg.prototype=Object.assign(Object.create(W.prototype),{constructor:wg,load:function(a,b,c,d){var e=
this,f=new Ra(e.manager);f.setPath(e.path);f.load(a,function(a){b(e.parse(JSON.parse(a)))},c,d)},parse:function(a){for(var b=[],c=0;c<a.length;c++){var d=Qa.parse(a[c]);b.push(d)}return b}});xg.prototype=Object.assign(Object.create(W.prototype),{constructor:xg,load:function(a,b,c,d){function e(e){l.load(a[e],function(a){a=f.parse(a,!0);g[e]={width:a.width,height:a.height,format:a.format,mipmaps:a.mipmaps};k+=1;6===k&&(1===a.mipmapCount&&(h.minFilter=1006),h.format=a.format,h.needsUpdate=!0,b&&b(h))},
c,d)}var f=this,g=[],h=new Qc;h.image=g;var l=new Ra(this.manager);l.setPath(this.path);l.setResponseType(&quot;arraybuffer&quot;);if(Array.isArray(a))for(var k=0,n=0,p=a.length;n<p;++n)e(n);else l.load(a,function(a){a=f.parse(a,!0);if(a.isCubemap)for(var c=a.mipmaps.length/a.mipmapCount,d=0;d<c;d++){g[d]={mipmaps:[]};for(var e=0;e<a.mipmapCount;e++)g[d].mipmaps.push(a.mipmaps[d*a.mipmapCount+e]),g[d].format=a.format,g[d].width=a.width,g[d].height=a.height}else h.image.width=a.width,h.image.height=a.height,
h.mipmaps=a.mipmaps;1===a.mipmapCount&&(h.minFilter=1006);h.format=a.format;h.needsUpdate=!0;b&&b(h)},c,d);return h}});cf.prototype=Object.assign(Object.create(W.prototype),{constructor:cf,load:function(a,b,c,d){var e=this,f=new ac,g=new Ra(this.manager);g.setResponseType(&quot;arraybuffer&quot;);g.setPath(this.path);g.load(a,function(a){if(a=e.parse(a))void 0!==a.image?f.image=a.image:void 0!==a.data&&(f.image.width=a.width,f.image.height=a.height,f.image.data=a.data),f.wrapS=void 0!==a.wrapS?a.wrapS:1001,
f.wrapT=void 0!==a.wrapT?a.wrapT:1001,f.magFilter=void 0!==a.magFilter?a.magFilter:1006,f.minFilter=void 0!==a.minFilter?a.minFilter:1006,f.anisotropy=void 0!==a.anisotropy?a.anisotropy:1,void 0!==a.format&&(f.format=a.format),void 0!==a.type&&(f.type=a.type),void 0!==a.mipmaps&&(f.mipmaps=a.mipmaps,f.minFilter=1008),1===a.mipmapCount&&(f.minFilter=1006),f.needsUpdate=!0,b&&b(f,a)},c,d);return f}});fd.prototype=Object.assign(Object.create(W.prototype),{constructor:fd,load:function(a,b,c,d){function e(){l.removeEventListener(&quot;load&quot;,
e,!1);l.removeEventListener(&quot;error&quot;,f,!1);vc.add(a,this);b&&b(this);g.manager.itemEnd(a)}function f(b){l.removeEventListener(&quot;load&quot;,e,!1);l.removeEventListener(&quot;error&quot;,f,!1);d&&d(b);g.manager.itemError(a);g.manager.itemEnd(a)}void 0!==this.path&&(a=this.path+a);a=this.manager.resolveURL(a);var g=this,h=vc.get(a);if(void 0!==h)return g.manager.itemStart(a),setTimeout(function(){b&&b(h);g.manager.itemEnd(a)},0),h;var l=document.createElementNS(&quot;http://www.w3.org/1999/xhtml&quot;,&quot;img&quot;);l.addEventListener(&quot;load&quot;,
e,!1);l.addEventListener(&quot;error&quot;,f,!1);&quot;data:&quot;!==a.substr(0,5)&&void 0!==this.crossOrigin&&(l.crossOrigin=this.crossOrigin);g.manager.itemStart(a);l.src=a;return l}});df.prototype=Object.assign(Object.create(W.prototype),{constructor:df,load:function(a,b,c,d){function e(c){g.load(a[c],function(a){f.images[c]=a;h++;6===h&&(f.needsUpdate=!0,b&&b(f))},void 0,d)}var f=new qb,g=new fd(this.manager);g.setCrossOrigin(this.crossOrigin);g.setPath(this.path);var h=0;for(c=0;c<a.length;++c)e(c);return f}});
ef.prototype=Object.assign(Object.create(W.prototype),{constructor:ef,load:function(a,b,c,d){var e=new V,f=new fd(this.manager);f.setCrossOrigin(this.crossOrigin);f.setPath(this.path);f.load(a,function(c){e.image=c;c=0<a.search(/\.jpe?g($|\?)/i)||0===a.search(/^data:image\/jpeg/);e.format=c?1022:1023;e.needsUpdate=!0;void 0!==b&&b(e)},c,d);return e}});Object.assign(G.prototype,{getPoint:function(){console.warn(&quot;THREE.Curve: .getPoint() not implemented.&quot;);return null},getPointAt:function(a,b){a=this.getUtoTmapping(a);
return this.getPoint(a,b)},getPoints:function(a){void 0===a&&(a=5);for(var b=[],c=0;c<=a;c++)b.push(this.getPoint(c/a));return b},getSpacedPoints:function(a){void 0===a&&(a=5);for(var b=[],c=0;c<=a;c++)b.push(this.getPointAt(c/a));return b},getLength:function(){var a=this.getLengths();return a[a.length-1]},getLengths:function(a){void 0===a&&(a=this.arcLengthDivisions);if(this.cacheArcLengths&&this.cacheArcLengths.length===a+1&&!this.needsUpdate)return this.cacheArcLengths;this.needsUpdate=!1;var b=
[],c=this.getPoint(0),d,e=0;b.push(0);for(d=1;d<=a;d++){var f=this.getPoint(d/a);e+=f.distanceTo(c);b.push(e);c=f}return this.cacheArcLengths=b},updateArcLengths:function(){this.needsUpdate=!0;this.getLengths()},getUtoTmapping:function(a,b){var c=this.getLengths(),d=c.length;b=b?b:a*c[d-1];for(var e=0,f=d-1,g;e<=f;)if(a=Math.floor(e+(f-e)/2),g=c[a]-b,0>g)e=a+1;else if(0<g)f=a-1;else{f=a;break}a=f;if(c[a]===b)return a/(d-1);e=c[a];return(a+(b-e)/(c[a+1]-e))/(d-1)},getTangent:function(a){var b=a-1E-4;
a+=1E-4;0>b&&(b=0);1<a&&(a=1);b=this.getPoint(b);return this.getPoint(a).clone().sub(b).normalize()},getTangentAt:function(a){a=this.getUtoTmapping(a);return this.getTangent(a)},computeFrenetFrames:function(a,b){var c=new n,d=[],e=[],f=[],g=new n,h=new P,l;for(l=0;l<=a;l++){var k=l/a;d[l]=this.getTangentAt(k);d[l].normalize()}e[0]=new n;f[0]=new n;l=Number.MAX_VALUE;k=Math.abs(d[0].x);var u=Math.abs(d[0].y),p=Math.abs(d[0].z);k<=l&&(l=k,c.set(1,0,0));u<=l&&(l=u,c.set(0,1,0));p<=l&&c.set(0,0,1);g.crossVectors(d[0],
c).normalize();e[0].crossVectors(d[0],g);f[0].crossVectors(d[0],e[0]);for(l=1;l<=a;l++)e[l]=e[l-1].clone(),f[l]=f[l-1].clone(),g.crossVectors(d[l-1],d[l]),g.length()>Number.EPSILON&&(g.normalize(),c=Math.acos(L.clamp(d[l-1].dot(d[l]),-1,1)),e[l].applyMatrix4(h.makeRotationAxis(g,c))),f[l].crossVectors(d[l],e[l]);if(!0===b)for(c=Math.acos(L.clamp(e[0].dot(e[a]),-1,1)),c/=a,0<d[0].dot(g.crossVectors(e[0],e[a]))&&(c=-c),l=1;l<=a;l++)e[l].applyMatrix4(h.makeRotationAxis(d[l],c*l)),f[l].crossVectors(d[l],
e[l]);return{tangents:d,normals:e,binormals:f}},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.arcLengthDivisions=a.arcLengthDivisions;return this},toJSON:function(){var a={metadata:{version:4.5,type:&quot;Curve&quot;,generator:&quot;Curve.toJSON&quot;}};a.arcLengthDivisions=this.arcLengthDivisions;a.type=this.type;return a},fromJSON:function(a){this.arcLengthDivisions=a.arcLengthDivisions;return this}});Ma.prototype=Object.create(G.prototype);Ma.prototype.constructor=Ma;Ma.prototype.isEllipseCurve=
!0;Ma.prototype.getPoint=function(a,b){b=b||new t;for(var c=2*Math.PI,d=this.aEndAngle-this.aStartAngle,e=Math.abs(d)<Number.EPSILON;0>d;)d+=c;for(;d>c;)d-=c;d<Number.EPSILON&&(d=e?0:c);!0!==this.aClockwise||e||(d=d===c?-c:d-c);c=this.aStartAngle+a*d;a=this.aX+this.xRadius*Math.cos(c);var f=this.aY+this.yRadius*Math.sin(c);0!==this.aRotation&&(c=Math.cos(this.aRotation),d=Math.sin(this.aRotation),e=a-this.aX,f-=this.aY,a=e*c-f*d+this.aX,f=e*d+f*c+this.aY);return b.set(a,f)};Ma.prototype.copy=function(a){G.prototype.copy.call(this,
a);this.aX=a.aX;this.aY=a.aY;this.xRadius=a.xRadius;this.yRadius=a.yRadius;this.aStartAngle=a.aStartAngle;this.aEndAngle=a.aEndAngle;this.aClockwise=a.aClockwise;this.aRotation=a.aRotation;return this};Ma.prototype.toJSON=function(){var a=G.prototype.toJSON.call(this);a.aX=this.aX;a.aY=this.aY;a.xRadius=this.xRadius;a.yRadius=this.yRadius;a.aStartAngle=this.aStartAngle;a.aEndAngle=this.aEndAngle;a.aClockwise=this.aClockwise;a.aRotation=this.aRotation;return a};Ma.prototype.fromJSON=function(a){G.prototype.fromJSON.call(this,
a);this.aX=a.aX;this.aY=a.aY;this.xRadius=a.xRadius;this.yRadius=a.yRadius;this.aStartAngle=a.aStartAngle;this.aEndAngle=a.aEndAngle;this.aClockwise=a.aClockwise;this.aRotation=a.aRotation;return this};gd.prototype=Object.create(Ma.prototype);gd.prototype.constructor=gd;gd.prototype.isArcCurve=!0;var Tf=new n,ph=new yg,qh=new yg,rh=new yg;pa.prototype=Object.create(G.prototype);pa.prototype.constructor=pa;pa.prototype.isCatmullRomCurve3=!0;pa.prototype.getPoint=function(a,b){b=b||new n;var c=this.points,
d=c.length;a*=d-(this.closed?0:1);var e=Math.floor(a);a-=e;this.closed?e+=0<e?0:(Math.floor(Math.abs(e)/d)+1)*d:0===a&&e===d-1&&(e=d-2,a=1);if(this.closed||0<e)var f=c[(e-1)%d];else Tf.subVectors(c[0],c[1]).add(c[0]),f=Tf;var g=c[e%d];var h=c[(e+1)%d];this.closed||e+2<d?c=c[(e+2)%d]:(Tf.subVectors(c[d-1],c[d-2]).add(c[d-1]),c=Tf);if(&quot;centripetal&quot;===this.curveType||&quot;chordal&quot;===this.curveType){var l=&quot;chordal&quot;===this.curveType?.5:.25;d=Math.pow(f.distanceToSquared(g),l);e=Math.pow(g.distanceToSquared(h),
l);l=Math.pow(h.distanceToSquared(c),l);1E-4>e&&(e=1);1E-4>d&&(d=e);1E-4>l&&(l=e);ph.initNonuniformCatmullRom(f.x,g.x,h.x,c.x,d,e,l);qh.initNonuniformCatmullRom(f.y,g.y,h.y,c.y,d,e,l);rh.initNonuniformCatmullRom(f.z,g.z,h.z,c.z,d,e,l)}else&quot;catmullrom&quot;===this.curveType&&(ph.initCatmullRom(f.x,g.x,h.x,c.x,this.tension),qh.initCatmullRom(f.y,g.y,h.y,c.y,this.tension),rh.initCatmullRom(f.z,g.z,h.z,c.z,this.tension));b.set(ph.calc(a),qh.calc(a),rh.calc(a));return b};pa.prototype.copy=function(a){G.prototype.copy.call(this,
a);this.points=[];for(var b=0,c=a.points.length;b<c;b++)this.points.push(a.points[b].clone());this.closed=a.closed;this.curveType=a.curveType;this.tension=a.tension;return this};pa.prototype.toJSON=function(){var a=G.prototype.toJSON.call(this);a.points=[];for(var b=0,c=this.points.length;b<c;b++)a.points.push(this.points[b].toArray());a.closed=this.closed;a.curveType=this.curveType;a.tension=this.tension;return a};pa.prototype.fromJSON=function(a){G.prototype.fromJSON.call(this,a);this.points=[];
for(var b=0,c=a.points.length;b<c;b++){var d=a.points[b];this.points.push((new n).fromArray(d))}this.closed=a.closed;this.curveType=a.curveType;this.tension=a.tension;return this};Wa.prototype=Object.create(G.prototype);Wa.prototype.constructor=Wa;Wa.prototype.isCubicBezierCurve=!0;Wa.prototype.getPoint=function(a,b){b=b||new t;var c=this.v0,d=this.v1,e=this.v2,f=this.v3;b.set(qe(a,c.x,d.x,e.x,f.x),qe(a,c.y,d.y,e.y,f.y));return b};Wa.prototype.copy=function(a){G.prototype.copy.call(this,a);this.v0.copy(a.v0);
this.v1.copy(a.v1);this.v2.copy(a.v2);this.v3.copy(a.v3);return this};Wa.prototype.toJSON=function(){var a=G.prototype.toJSON.call(this);a.v0=this.v0.toArray();a.v1=this.v1.toArray();a.v2=this.v2.toArray();a.v3=this.v3.toArray();return a};Wa.prototype.fromJSON=function(a){G.prototype.fromJSON.call(this,a);this.v0.fromArray(a.v0);this.v1.fromArray(a.v1);this.v2.fromArray(a.v2);this.v3.fromArray(a.v3);return this};hb.prototype=Object.create(G.prototype);hb.prototype.constructor=hb;hb.prototype.isCubicBezierCurve3=
!0;hb.prototype.getPoint=function(a,b){b=b||new n;var c=this.v0,d=this.v1,e=this.v2,f=this.v3;b.set(qe(a,c.x,d.x,e.x,f.x),qe(a,c.y,d.y,e.y,f.y),qe(a,c.z,d.z,e.z,f.z));return b};hb.prototype.copy=function(a){G.prototype.copy.call(this,a);this.v0.copy(a.v0);this.v1.copy(a.v1);this.v2.copy(a.v2);this.v3.copy(a.v3);return this};hb.prototype.toJSON=function(){var a=G.prototype.toJSON.call(this);a.v0=this.v0.toArray();a.v1=this.v1.toArray();a.v2=this.v2.toArray();a.v3=this.v3.toArray();return a};hb.prototype.fromJSON=
function(a){G.prototype.fromJSON.call(this,a);this.v0.fromArray(a.v0);this.v1.fromArray(a.v1);this.v2.fromArray(a.v2);this.v3.fromArray(a.v3);return this};Da.prototype=Object.create(G.prototype);Da.prototype.constructor=Da;Da.prototype.isLineCurve=!0;Da.prototype.getPoint=function(a,b){b=b||new t;1===a?b.copy(this.v2):(b.copy(this.v2).sub(this.v1),b.multiplyScalar(a).add(this.v1));return b};Da.prototype.getPointAt=function(a,b){return this.getPoint(a,b)};Da.prototype.getTangent=function(){return this.v2.clone().sub(this.v1).normalize()};
Da.prototype.copy=function(a){G.prototype.copy.call(this,a);this.v1.copy(a.v1);this.v2.copy(a.v2);return this};Da.prototype.toJSON=function(){var a=G.prototype.toJSON.call(this);a.v1=this.v1.toArray();a.v2=this.v2.toArray();return a};Da.prototype.fromJSON=function(a){G.prototype.fromJSON.call(this,a);this.v1.fromArray(a.v1);this.v2.fromArray(a.v2);return this};Xa.prototype=Object.create(G.prototype);Xa.prototype.constructor=Xa;Xa.prototype.isLineCurve3=!0;Xa.prototype.getPoint=function(a,b){b=b||
new n;1===a?b.copy(this.v2):(b.copy(this.v2).sub(this.v1),b.multiplyScalar(a).add(this.v1));return b};Xa.prototype.getPointAt=function(a,b){return this.getPoint(a,b)};Xa.prototype.copy=function(a){G.prototype.copy.call(this,a);this.v1.copy(a.v1);this.v2.copy(a.v2);return this};Xa.prototype.toJSON=function(){var a=G.prototype.toJSON.call(this);a.v1=this.v1.toArray();a.v2=this.v2.toArray();return a};Xa.prototype.fromJSON=function(a){G.prototype.fromJSON.call(this,a);this.v1.fromArray(a.v1);this.v2.fromArray(a.v2);
return this};Ya.prototype=Object.create(G.prototype);Ya.prototype.constructor=Ya;Ya.prototype.isQuadraticBezierCurve=!0;Ya.prototype.getPoint=function(a,b){b=b||new t;var c=this.v0,d=this.v1,e=this.v2;b.set(pe(a,c.x,d.x,e.x),pe(a,c.y,d.y,e.y));return b};Ya.prototype.copy=function(a){G.prototype.copy.call(this,a);this.v0.copy(a.v0);this.v1.copy(a.v1);this.v2.copy(a.v2);return this};Ya.prototype.toJSON=function(){var a=G.prototype.toJSON.call(this);a.v0=this.v0.toArray();a.v1=this.v1.toArray();a.v2=
this.v2.toArray();return a};Ya.prototype.fromJSON=function(a){G.prototype.fromJSON.call(this,a);this.v0.fromArray(a.v0);this.v1.fromArray(a.v1);this.v2.fromArray(a.v2);return this};ib.prototype=Object.create(G.prototype);ib.prototype.constructor=ib;ib.prototype.isQuadraticBezierCurve3=!0;ib.prototype.getPoint=function(a,b){b=b||new n;var c=this.v0,d=this.v1,e=this.v2;b.set(pe(a,c.x,d.x,e.x),pe(a,c.y,d.y,e.y),pe(a,c.z,d.z,e.z));return b};ib.prototype.copy=function(a){G.prototype.copy.call(this,a);
this.v0.copy(a.v0);this.v1.copy(a.v1);this.v2.copy(a.v2);return this};ib.prototype.toJSON=function(){var a=G.prototype.toJSON.call(this);a.v0=this.v0.toArray();a.v1=this.v1.toArray();a.v2=this.v2.toArray();return a};ib.prototype.fromJSON=function(a){G.prototype.fromJSON.call(this,a);this.v0.fromArray(a.v0);this.v1.fromArray(a.v1);this.v2.fromArray(a.v2);return this};Za.prototype=Object.create(G.prototype);Za.prototype.constructor=Za;Za.prototype.isSplineCurve=!0;Za.prototype.getPoint=function(a,b){b=
b||new t;var c=this.points,d=(c.length-1)*a;a=Math.floor(d);d-=a;var e=c[0===a?a:a-1],f=c[a],g=c[a>c.length-2?c.length-1:a+1];c=c[a>c.length-3?c.length-1:a+2];b.set(li(d,e.x,f.x,g.x,c.x),li(d,e.y,f.y,g.y,c.y));return b};Za.prototype.copy=function(a){G.prototype.copy.call(this,a);this.points=[];for(var b=0,c=a.points.length;b<c;b++)this.points.push(a.points[b].clone());return this};Za.prototype.toJSON=function(){var a=G.prototype.toJSON.call(this);a.points=[];for(var b=0,c=this.points.length;b<c;b++)a.points.push(this.points[b].toArray());
return a};Za.prototype.fromJSON=function(a){G.prototype.fromJSON.call(this,a);this.points=[];for(var b=0,c=a.points.length;b<c;b++){var d=a.points[b];this.points.push((new t).fromArray(d))}return this};var sh=Object.freeze({__proto__:null,ArcCurve:gd,CatmullRomCurve3:pa,CubicBezierCurve:Wa,CubicBezierCurve3:hb,EllipseCurve:Ma,LineCurve:Da,LineCurve3:Xa,QuadraticBezierCurve:Ya,QuadraticBezierCurve3:ib,SplineCurve:Za});vb.prototype=Object.assign(Object.create(G.prototype),{constructor:vb,add:function(a){this.curves.push(a)},
closePath:function(){var a=this.curves[0].getPoint(0),b=this.curves[this.curves.length-1].getPoint(1);a.equals(b)||this.curves.push(new Da(b,a))},getPoint:function(a){var b=a*this.getLength(),c=this.getCurveLengths();for(a=0;a<c.length;){if(c[a]>=b)return b=c[a]-b,a=this.curves[a],c=a.getLength(),a.getPointAt(0===c?0:1-b/c);a++}return null},getLength:function(){var a=this.getCurveLengths();return a[a.length-1]},updateArcLengths:function(){this.needsUpdate=!0;this.cacheLengths=null;this.getCurveLengths()},
getCurveLengths:function(){if(this.cacheLengths&&this.cacheLengths.length===this.curves.length)return this.cacheLengths;for(var a=[],b=0,c=0,d=this.curves.length;c<d;c++)b+=this.curves[c].getLength(),a.push(b);return this.cacheLengths=a},getSpacedPoints:function(a){void 0===a&&(a=40);for(var b=[],c=0;c<=a;c++)b.push(this.getPoint(c/a));this.autoClose&&b.push(b[0]);return b},getPoints:function(a){a=a||12;for(var b=[],c,d=0,e=this.curves;d<e.length;d++){var f=e[d];f=f.getPoints(f&&f.isEllipseCurve?
2*a:f&&(f.isLineCurve||f.isLineCurve3)?1:f&&f.isSplineCurve?a*f.points.length:a);for(var g=0;g<f.length;g++){var h=f[g];c&&c.equals(h)||(b.push(h),c=h)}}this.autoClose&&1<b.length&&!b[b.length-1].equals(b[0])&&b.push(b[0]);return b},copy:function(a){G.prototype.copy.call(this,a);this.curves=[];for(var b=0,c=a.curves.length;b<c;b++)this.curves.push(a.curves[b].clone());this.autoClose=a.autoClose;return this},toJSON:function(){var a=G.prototype.toJSON.call(this);a.autoClose=this.autoClose;a.curves=
[];for(var b=0,c=this.curves.length;b<c;b++)a.curves.push(this.curves[b].toJSON());return a},fromJSON:function(a){G.prototype.fromJSON.call(this,a);this.autoClose=a.autoClose;this.curves=[];for(var b=0,c=a.curves.length;b<c;b++){var d=a.curves[b];this.curves.push((new sh[d.type]).fromJSON(d))}return this}});$a.prototype=Object.assign(Object.create(vb.prototype),{constructor:$a,setFromPoints:function(a){this.moveTo(a[0].x,a[0].y);for(var b=1,c=a.length;b<c;b++)this.lineTo(a[b].x,a[b].y);return this},
moveTo:function(a,b){this.currentPoint.set(a,b);return this},lineTo:function(a,b){var c=new Da(this.currentPoint.clone(),new t(a,b));this.curves.push(c);this.currentPoint.set(a,b);return this},quadraticCurveTo:function(a,b,c,d){a=new Ya(this.currentPoint.clone(),new t(a,b),new t(c,d));this.curves.push(a);this.currentPoint.set(c,d);return this},bezierCurveTo:function(a,b,c,d,e,f){a=new Wa(this.currentPoint.clone(),new t(a,b),new t(c,d),new t(e,f));this.curves.push(a);this.currentPoint.set(e,f);return this},
splineThru:function(a){var b=[this.currentPoint.clone()].concat(a);b=new Za(b);this.curves.push(b);this.currentPoint.copy(a[a.length-1]);return this},arc:function(a,b,c,d,e,f){this.absarc(a+this.currentPoint.x,b+this.currentPoint.y,c,d,e,f);return this},absarc:function(a,b,c,d,e,f){this.absellipse(a,b,c,c,d,e,f);return this},ellipse:function(a,b,c,d,e,f,g,h){this.absellipse(a+this.currentPoint.x,b+this.currentPoint.y,c,d,e,f,g,h);return this},absellipse:function(a,b,c,d,e,f,g,h){a=new Ma(a,b,c,d,
e,f,g,h);0<this.curves.length&&(b=a.getPoint(0),b.equals(this.currentPoint)||this.lineTo(b.x,b.y));this.curves.push(a);a=a.getPoint(1);this.currentPoint.copy(a);return this},copy:function(a){vb.prototype.copy.call(this,a);this.currentPoint.copy(a.currentPoint);return this},toJSON:function(){var a=vb.prototype.toJSON.call(this);a.currentPoint=this.currentPoint.toArray();return a},fromJSON:function(a){vb.prototype.fromJSON.call(this,a);this.currentPoint.fromArray(a.currentPoint);return this}});Kb.prototype=
Object.assign(Object.create($a.prototype),{constructor:Kb,getPointsHoles:function(a){for(var b=[],c=0,d=this.holes.length;c<d;c++)b[c]=this.holes[c].getPoints(a);return b},extractPoints:function(a){return{shape:this.getPoints(a),holes:this.getPointsHoles(a)}},copy:function(a){$a.prototype.copy.call(this,a);this.holes=[];for(var b=0,c=a.holes.length;b<c;b++)this.holes.push(a.holes[b].clone());return this},toJSON:function(){var a=$a.prototype.toJSON.call(this);a.uuid=this.uuid;a.holes=[];for(var b=
0,c=this.holes.length;b<c;b++)a.holes.push(this.holes[b].toJSON());return a},fromJSON:function(a){$a.prototype.fromJSON.call(this,a);this.uuid=a.uuid;this.holes=[];for(var b=0,c=a.holes.length;b<c;b++){var d=a.holes[b];this.holes.push((new $a).fromJSON(d))}return this}});da.prototype=Object.assign(Object.create(F.prototype),{constructor:da,isLight:!0,copy:function(a){F.prototype.copy.call(this,a);this.color.copy(a.color);this.intensity=a.intensity;return this},toJSON:function(a){a=F.prototype.toJSON.call(this,
a);a.object.color=this.color.getHex();a.object.intensity=this.intensity;void 0!==this.groundColor&&(a.object.groundColor=this.groundColor.getHex());void 0!==this.distance&&(a.object.distance=this.distance);void 0!==this.angle&&(a.object.angle=this.angle);void 0!==this.decay&&(a.object.decay=this.decay);void 0!==this.penumbra&&(a.object.penumbra=this.penumbra);void 0!==this.shadow&&(a.object.shadow=this.shadow.toJSON());return a}});ff.prototype=Object.assign(Object.create(da.prototype),{constructor:ff,
isHemisphereLight:!0,copy:function(a){da.prototype.copy.call(this,a);this.groundColor.copy(a.groundColor);return this}});Object.assign(jb.prototype,{_projScreenMatrix:new P,_lightPositionWorld:new n,_lookTarget:new n,getViewportCount:function(){return this._viewportCount},getFrustum:function(){return this._frustum},updateMatrices:function(a){var b=this.camera,c=this.matrix,d=this._projScreenMatrix,e=this._lookTarget,f=this._lightPositionWorld;f.setFromMatrixPosition(a.matrixWorld);b.position.copy(f);
e.setFromMatrixPosition(a.target.matrixWorld);b.lookAt(e);b.updateMatrixWorld();d.multiplyMatrices(b.projectionMatrix,b.matrixWorldInverse);this._frustum.setFromProjectionMatrix(d);c.set(.5,0,0,.5,0,.5,0,.5,0,0,.5,.5,0,0,0,1);c.multiply(b.projectionMatrix);c.multiply(b.matrixWorldInverse)},getViewport:function(a){return this._viewports[a]},getFrameExtents:function(){return this._frameExtents},copy:function(a){this.camera=a.camera.clone();this.bias=a.bias;this.radius=a.radius;this.mapSize.copy(a.mapSize);
return this},clone:function(){return(new this.constructor).copy(this)},toJSON:function(){var a={};0!==this.bias&&(a.bias=this.bias);1!==this.radius&&(a.radius=this.radius);if(512!==this.mapSize.x||512!==this.mapSize.y)a.mapSize=this.mapSize.toArray();a.camera=this.camera.toJSON(!1).object;delete a.camera.matrix;return a}});gf.prototype=Object.assign(Object.create(jb.prototype),{constructor:gf,isSpotLightShadow:!0,updateMatrices:function(a){var b=this.camera,c=2*L.RAD2DEG*a.angle,d=this.mapSize.width/
this.mapSize.height,e=a.distance||b.far;if(c!==b.fov||d!==b.aspect||e!==b.far)b.fov=c,b.aspect=d,b.far=e,b.updateProjectionMatrix();jb.prototype.updateMatrices.call(this,a)}});hf.prototype=Object.assign(Object.create(da.prototype),{constructor:hf,isSpotLight:!0,copy:function(a){da.prototype.copy.call(this,a);this.distance=a.distance;this.angle=a.angle;this.penumbra=a.penumbra;this.decay=a.decay;this.target=a.target.clone();this.shadow=a.shadow.clone();return this}});zg.prototype=Object.assign(Object.create(jb.prototype),
{constructor:zg,isPointLightShadow:!0,updateMatrices:function(a,b){void 0===b&&(b=0);var c=this.camera,d=this.matrix,e=this._lightPositionWorld,f=this._lookTarget,g=this._projScreenMatrix;e.setFromMatrixPosition(a.matrixWorld);c.position.copy(e);f.copy(c.position);f.add(this._cubeDirections[b]);c.up.copy(this._cubeUps[b]);c.lookAt(f);c.updateMatrixWorld();d.makeTranslation(-e.x,-e.y,-e.z);g.multiplyMatrices(c.projectionMatrix,c.matrixWorldInverse);this._frustum.setFromProjectionMatrix(g)}});jf.prototype=
Object.assign(Object.create(da.prototype),{constructor:jf,isPointLight:!0,copy:function(a){da.prototype.copy.call(this,a);this.distance=a.distance;this.decay=a.decay;this.shadow=a.shadow.clone();return this}});hd.prototype=Object.assign(Object.create(db.prototype),{constructor:hd,isOrthographicCamera:!0,copy:function(a,b){db.prototype.copy.call(this,a,b);this.left=a.left;this.right=a.right;this.top=a.top;this.bottom=a.bottom;this.near=a.near;this.far=a.far;this.zoom=a.zoom;this.view=null===a.view?
null:Object.assign({},a.view);return this},setViewOffset:function(a,b,c,d,e,f){null===this.view&&(this.view={enabled:!0,fullWidth:1,fullHeight:1,offsetX:0,offsetY:0,width:1,height:1});this.view.enabled=!0;this.view.fullWidth=a;this.view.fullHeight=b;this.view.offsetX=c;this.view.offsetY=d;this.view.width=e;this.view.height=f;this.updateProjectionMatrix()},clearViewOffset:function(){null!==this.view&&(this.view.enabled=!1);this.updateProjectionMatrix()},updateProjectionMatrix:function(){var a=(this.right-
this.left)/(2*this.zoom),b=(this.top-this.bottom)/(2*this.zoom),c=(this.right+this.left)/2,d=(this.top+this.bottom)/2,e=c-a;c+=a;a=d+b;b=d-b;null!==this.view&&this.view.enabled&&(d=(this.right-this.left)/this.view.fullWidth/this.zoom,b=(this.top-this.bottom)/this.view.fullHeight/this.zoom,e+=d*this.view.offsetX,c=e+d*this.view.width,a-=b*this.view.offsetY,b=a-b*this.view.height);this.projectionMatrix.makeOrthographic(e,c,a,b,this.near,this.far);this.projectionMatrixInverse.getInverse(this.projectionMatrix)},
toJSON:function(a){a=F.prototype.toJSON.call(this,a);a.object.zoom=this.zoom;a.object.left=this.left;a.object.right=this.right;a.object.top=this.top;a.object.bottom=this.bottom;a.object.near=this.near;a.object.far=this.far;null!==this.view&&(a.object.view=Object.assign({},this.view));return a}});kf.prototype=Object.assign(Object.create(jb.prototype),{constructor:kf,isDirectionalLightShadow:!0,updateMatrices:function(a){jb.prototype.updateMatrices.call(this,a)}});lf.prototype=Object.assign(Object.create(da.prototype),
{constructor:lf,isDirectionalLight:!0,copy:function(a){da.prototype.copy.call(this,a);this.target=a.target.clone();this.shadow=a.shadow.clone();return this}});mf.prototype=Object.assign(Object.create(da.prototype),{constructor:mf,isAmbientLight:!0});nf.prototype=Object.assign(Object.create(da.prototype),{constructor:nf,isRectAreaLight:!0,copy:function(a){da.prototype.copy.call(this,a);this.width=a.width;this.height=a.height;return this},toJSON:function(a){a=da.prototype.toJSON.call(this,a);a.object.width=
this.width;a.object.height=this.height;return a}});of.prototype=Object.assign(Object.create(W.prototype),{constructor:of,load:function(a,b,c,d){var e=this,f=new Ra(e.manager);f.setPath(e.path);f.load(a,function(a){b(e.parse(JSON.parse(a)))},c,d)},parse:function(a){function b(a){void 0===c[a]&&console.warn(&quot;THREE.MaterialLoader: Undefined texture&quot;,a);return c[a]}var c=this.textures,d=new bl[a.type];void 0!==a.uuid&&(d.uuid=a.uuid);void 0!==a.name&&(d.name=a.name);void 0!==a.color&&d.color.setHex(a.color);
void 0!==a.roughness&&(d.roughness=a.roughness);void 0!==a.metalness&&(d.metalness=a.metalness);void 0!==a.sheen&&(d.sheen=(new A).setHex(a.sheen));void 0!==a.emissive&&d.emissive.setHex(a.emissive);void 0!==a.specular&&d.specular.setHex(a.specular);void 0!==a.shininess&&(d.shininess=a.shininess);void 0!==a.clearcoat&&(d.clearcoat=a.clearcoat);void 0!==a.clearcoatRoughness&&(d.clearcoatRoughness=a.clearcoatRoughness);void 0!==a.fog&&(d.fog=a.fog);void 0!==a.flatShading&&(d.flatShading=a.flatShading);
void 0!==a.blending&&(d.blending=a.blending);void 0!==a.combine&&(d.combine=a.combine);void 0!==a.side&&(d.side=a.side);void 0!==a.opacity&&(d.opacity=a.opacity);void 0!==a.transparent&&(d.transparent=a.transparent);void 0!==a.alphaTest&&(d.alphaTest=a.alphaTest);void 0!==a.depthTest&&(d.depthTest=a.depthTest);void 0!==a.depthWrite&&(d.depthWrite=a.depthWrite);void 0!==a.colorWrite&&(d.colorWrite=a.colorWrite);void 0!==a.stencilWrite&&(d.stencilWrite=a.stencilWrite);void 0!==a.stencilWriteMask&&(d.stencilWriteMask=
a.stencilWriteMask);void 0!==a.stencilFunc&&(d.stencilFunc=a.stencilFunc);void 0!==a.stencilRef&&(d.stencilRef=a.stencilRef);void 0!==a.stencilFuncMask&&(d.stencilFuncMask=a.stencilFuncMask);void 0!==a.stencilFail&&(d.stencilFail=a.stencilFail);void 0!==a.stencilZFail&&(d.stencilZFail=a.stencilZFail);void 0!==a.stencilZPass&&(d.stencilZPass=a.stencilZPass);void 0!==a.wireframe&&(d.wireframe=a.wireframe);void 0!==a.wireframeLinewidth&&(d.wireframeLinewidth=a.wireframeLinewidth);void 0!==a.wireframeLinecap&&
(d.wireframeLinecap=a.wireframeLinecap);void 0!==a.wireframeLinejoin&&(d.wireframeLinejoin=a.wireframeLinejoin);void 0!==a.rotation&&(d.rotation=a.rotation);1!==a.linewidth&&(d.linewidth=a.linewidth);void 0!==a.dashSize&&(d.dashSize=a.dashSize);void 0!==a.gapSize&&(d.gapSize=a.gapSize);void 0!==a.scale&&(d.scale=a.scale);void 0!==a.polygonOffset&&(d.polygonOffset=a.polygonOffset);void 0!==a.polygonOffsetFactor&&(d.polygonOffsetFactor=a.polygonOffsetFactor);void 0!==a.polygonOffsetUnits&&(d.polygonOffsetUnits=
a.polygonOffsetUnits);void 0!==a.skinning&&(d.skinning=a.skinning);void 0!==a.morphTargets&&(d.morphTargets=a.morphTargets);void 0!==a.morphNormals&&(d.morphNormals=a.morphNormals);void 0!==a.dithering&&(d.dithering=a.dithering);void 0!==a.vertexTangents&&(d.vertexTangents=a.vertexTangents);void 0!==a.visible&&(d.visible=a.visible);void 0!==a.toneMapped&&(d.toneMapped=a.toneMapped);void 0!==a.userData&&(d.userData=a.userData);void 0!==a.vertexColors&&(d.vertexColors=&quot;number&quot;===typeof a.vertexColors?
0<a.vertexColors?!0:!1:a.vertexColors);if(void 0!==a.uniforms)for(var e in a.uniforms){var f=a.uniforms[e];d.uniforms[e]={};switch(f.type){case &quot;t&quot;:d.uniforms[e].value=b(f.value);break;case &quot;c&quot;:d.uniforms[e].value=(new A).setHex(f.value);break;case &quot;v2&quot;:d.uniforms[e].value=(new t).fromArray(f.value);break;case &quot;v3&quot;:d.uniforms[e].value=(new n).fromArray(f.value);break;case &quot;v4&quot;:d.uniforms[e].value=(new ka).fromArray(f.value);break;case &quot;m3&quot;:d.uniforms[e].value=(new wa).fromArray(f.value);case &quot;m4&quot;:d.uniforms[e].value=
(new P).fromArray(f.value);break;default:d.uniforms[e].value=f.value}}void 0!==a.defines&&(d.defines=a.defines);void 0!==a.vertexShader&&(d.vertexShader=a.vertexShader);void 0!==a.fragmentShader&&(d.fragmentShader=a.fragmentShader);if(void 0!==a.extensions)for(var g in a.extensions)d.extensions[g]=a.extensions[g];void 0!==a.shading&&(d.flatShading=1===a.shading);void 0!==a.size&&(d.size=a.size);void 0!==a.sizeAttenuation&&(d.sizeAttenuation=a.sizeAttenuation);void 0!==a.map&&(d.map=b(a.map));void 0!==
a.matcap&&(d.matcap=b(a.matcap));void 0!==a.alphaMap&&(d.alphaMap=b(a.alphaMap));void 0!==a.bumpMap&&(d.bumpMap=b(a.bumpMap));void 0!==a.bumpScale&&(d.bumpScale=a.bumpScale);void 0!==a.normalMap&&(d.normalMap=b(a.normalMap));void 0!==a.normalMapType&&(d.normalMapType=a.normalMapType);void 0!==a.normalScale&&(e=a.normalScale,!1===Array.isArray(e)&&(e=[e,e]),d.normalScale=(new t).fromArray(e));void 0!==a.displacementMap&&(d.displacementMap=b(a.displacementMap));void 0!==a.displacementScale&&(d.displacementScale=
a.displacementScale);void 0!==a.displacementBias&&(d.displacementBias=a.displacementBias);void 0!==a.roughnessMap&&(d.roughnessMap=b(a.roughnessMap));void 0!==a.metalnessMap&&(d.metalnessMap=b(a.metalnessMap));void 0!==a.emissiveMap&&(d.emissiveMap=b(a.emissiveMap));void 0!==a.emissiveIntensity&&(d.emissiveIntensity=a.emissiveIntensity);void 0!==a.specularMap&&(d.specularMap=b(a.specularMap));void 0!==a.envMap&&(d.envMap=b(a.envMap));void 0!==a.envMapIntensity&&(d.envMapIntensity=a.envMapIntensity);
void 0!==a.reflectivity&&(d.reflectivity=a.reflectivity);void 0!==a.refractionRatio&&(d.refractionRatio=a.refractionRatio);void 0!==a.lightMap&&(d.lightMap=b(a.lightMap));void 0!==a.lightMapIntensity&&(d.lightMapIntensity=a.lightMapIntensity);void 0!==a.aoMap&&(d.aoMap=b(a.aoMap));void 0!==a.aoMapIntensity&&(d.aoMapIntensity=a.aoMapIntensity);void 0!==a.gradientMap&&(d.gradientMap=b(a.gradientMap));void 0!==a.clearcoatMap&&(d.clearcoatMap=b(a.clearcoatMap));void 0!==a.clearcoatRoughnessMap&&(d.clearcoatRoughnessMap=
b(a.clearcoatRoughnessMap));void 0!==a.clearcoatNormalMap&&(d.clearcoatNormalMap=b(a.clearcoatNormalMap));void 0!==a.clearcoatNormalScale&&(d.clearcoatNormalScale=(new t).fromArray(a.clearcoatNormalScale));return d},setTextures:function(a){this.textures=a;return this}});var th={decodeText:function(a){if(&quot;undefined&quot;!==typeof TextDecoder)return(new TextDecoder).decode(a);for(var b=&quot;&quot;,c=0,d=a.length;c<d;c++)b+=String.fromCharCode(a[c]);try{return decodeURIComponent(escape(b))}catch(e){return b}},extractUrlBase:function(a){var b=
a.lastIndexOf(&quot;/&quot;);return-1===b?&quot;./&quot;:a.substr(0,b+1)}};pf.prototype=Object.assign(Object.create(C.prototype),{constructor:pf,isInstancedBufferGeometry:!0,copy:function(a){C.prototype.copy.call(this,a);this.maxInstancedCount=a.maxInstancedCount;return this},clone:function(){return(new this.constructor).copy(this)},toJSON:function(){var a=C.prototype.toJSON.call(this);a.maxInstancedCount=this.maxInstancedCount;a.isInstancedBufferGeometry=!0;return a}});qf.prototype=Object.assign(Object.create(M.prototype),
{constructor:qf,isInstancedBufferAttribute:!0,copy:function(a){M.prototype.copy.call(this,a);this.meshPerAttribute=a.meshPerAttribute;return this},toJSON:function(){var a=M.prototype.toJSON.call(this);a.meshPerAttribute=this.meshPerAttribute;a.isInstancedBufferAttribute=!0;return a}});rf.prototype=Object.assign(Object.create(W.prototype),{constructor:rf,load:function(a,b,c,d){var e=this,f=new Ra(e.manager);f.setPath(e.path);f.load(a,function(a){b(e.parse(JSON.parse(a)))},c,d)},parse:function(a){var b=
a.isInstancedBufferGeometry?new pf:new C,c=a.data.index;if(void 0!==c){var d=new uh[c.type](c.array);b.setIndex(new M(d,1))}c=a.data.attributes;for(var e in c){var f=c[e];d=new uh[f.type](f.array);d=new (f.isInstancedBufferAttribute?qf:M)(d,f.itemSize,f.normalized);void 0!==f.name&&(d.name=f.name);b.setAttribute(e,d)}var g=a.data.morphAttributes;if(g)for(e in g){var h=g[e],l=[];c=0;for(var k=h.length;c<k;c++)f=h[c],d=new uh[f.type](f.array),d=new M(d,f.itemSize,f.normalized),void 0!==f.name&&(d.name=
f.name),l.push(d);b.morphAttributes[e]=l}a.data.morphTargetsRelative&&(b.morphTargetsRelative=!0);e=a.data.groups||a.data.drawcalls||a.data.offsets;if(void 0!==e)for(c=0,f=e.length;c!==f;++c)d=e[c],b.addGroup(d.start,d.count,d.materialIndex);c=a.data.boundingSphere;void 0!==c&&(e=new n,void 0!==c.center&&e.fromArray(c.center),b.boundingSphere=new pb(e,c.radius));a.name&&(b.name=a.name);a.userData&&(b.userData=a.userData);return b}});var uh={Int8Array:Int8Array,Uint8Array:Uint8Array,Uint8ClampedArray:&quot;undefined&quot;!==
typeof Uint8ClampedArray?Uint8ClampedArray:Uint8Array,Int16Array:Int16Array,Uint16Array:Uint16Array,Int32Array:Int32Array,Uint32Array:Uint32Array,Float32Array:Float32Array,Float64Array:Float64Array};sf.prototype=Object.assign(Object.create(W.prototype),{constructor:sf,load:function(a,b,c,d){var e=this,f=&quot;&quot;===this.path?th.extractUrlBase(a):this.path;this.resourcePath=this.resourcePath||f;f=new Ra(e.manager);f.setPath(this.path);f.load(a,function(c){var f=null;try{f=JSON.parse(c)}catch(l){void 0!==
d&&d(l);console.error(&quot;THREE:ObjectLoader: Can't parse &quot;+a+&quot;.&quot;,l.message);return}c=f.metadata;void 0===c||void 0===c.type||&quot;geometry&quot;===c.type.toLowerCase()?console.error(&quot;THREE.ObjectLoader: Can't load &quot;+a):e.parse(f,b)},c,d)},parse:function(a,b){var c=this.parseShape(a.shapes);c=this.parseGeometries(a.geometries,c);var d=this.parseImages(a.images,function(){void 0!==b&&b(e)});d=this.parseTextures(a.textures,d);d=this.parseMaterials(a.materials,d);var e=this.parseObject(a.object,c,d);a.animations&&
(e.animations=this.parseAnimations(a.animations));void 0!==a.images&&0!==a.images.length||void 0===b||b(e);return e},parseShape:function(a){var b={};if(void 0!==a)for(var c=0,d=a.length;c<d;c++){var e=(new Kb).fromJSON(a[c]);b[e.uuid]=e}return b},parseGeometries:function(a,b){var c={};if(void 0!==a)for(var d=new rf,e=0,f=a.length;e<f;e++){var g=a[e];switch(g.type){case &quot;PlaneGeometry&quot;:case &quot;PlaneBufferGeometry&quot;:var h=new ua[g.type](g.width,g.height,g.widthSegments,g.heightSegments);break;case &quot;BoxGeometry&quot;:case &quot;BoxBufferGeometry&quot;:case &quot;CubeGeometry&quot;:h=
new ua[g.type](g.width,g.height,g.depth,g.widthSegments,g.heightSegments,g.depthSegments);break;case &quot;CircleGeometry&quot;:case &quot;CircleBufferGeometry&quot;:h=new ua[g.type](g.radius,g.segments,g.thetaStart,g.thetaLength);break;case &quot;CylinderGeometry&quot;:case &quot;CylinderBufferGeometry&quot;:h=new ua[g.type](g.radiusTop,g.radiusBottom,g.height,g.radialSegments,g.heightSegments,g.openEnded,g.thetaStart,g.thetaLength);break;case &quot;ConeGeometry&quot;:case &quot;ConeBufferGeometry&quot;:h=new ua[g.type](g.radius,g.height,g.radialSegments,
g.heightSegments,g.openEnded,g.thetaStart,g.thetaLength);break;case &quot;SphereGeometry&quot;:case &quot;SphereBufferGeometry&quot;:h=new ua[g.type](g.radius,g.widthSegments,g.heightSegments,g.phiStart,g.phiLength,g.thetaStart,g.thetaLength);break;case &quot;DodecahedronGeometry&quot;:case &quot;DodecahedronBufferGeometry&quot;:case &quot;IcosahedronGeometry&quot;:case &quot;IcosahedronBufferGeometry&quot;:case &quot;OctahedronGeometry&quot;:case &quot;OctahedronBufferGeometry&quot;:case &quot;TetrahedronGeometry&quot;:case &quot;TetrahedronBufferGeometry&quot;:h=new ua[g.type](g.radius,g.detail);
break;case &quot;RingGeometry&quot;:case &quot;RingBufferGeometry&quot;:h=new ua[g.type](g.innerRadius,g.outerRadius,g.thetaSegments,g.phiSegments,g.thetaStart,g.thetaLength);break;case &quot;TorusGeometry&quot;:case &quot;TorusBufferGeometry&quot;:h=new ua[g.type](g.radius,g.tube,g.radialSegments,g.tubularSegments,g.arc);break;case &quot;TorusKnotGeometry&quot;:case &quot;TorusKnotBufferGeometry&quot;:h=new ua[g.type](g.radius,g.tube,g.tubularSegments,g.radialSegments,g.p,g.q);break;case &quot;TubeGeometry&quot;:case &quot;TubeBufferGeometry&quot;:h=new ua[g.type]((new sh[g.path.type]).fromJSON(g.path),
g.tubularSegments,g.radius,g.radialSegments,g.closed);break;case &quot;LatheGeometry&quot;:case &quot;LatheBufferGeometry&quot;:h=new ua[g.type](g.points,g.segments,g.phiStart,g.phiLength);break;case &quot;PolyhedronGeometry&quot;:case &quot;PolyhedronBufferGeometry&quot;:h=new ua[g.type](g.vertices,g.indices,g.radius,g.details);break;case &quot;ShapeGeometry&quot;:case &quot;ShapeBufferGeometry&quot;:h=[];for(var l=0,k=g.shapes.length;l<k;l++){var n=b[g.shapes[l]];h.push(n)}h=new ua[g.type](h,g.curveSegments);break;case &quot;ExtrudeGeometry&quot;:case &quot;ExtrudeBufferGeometry&quot;:h=
[];l=0;for(k=g.shapes.length;l<k;l++)n=b[g.shapes[l]],h.push(n);l=g.options.extrudePath;void 0!==l&&(g.options.extrudePath=(new sh[l.type]).fromJSON(l));h=new ua[g.type](h,g.options);break;case &quot;BufferGeometry&quot;:case &quot;InstancedBufferGeometry&quot;:h=d.parse(g);break;case &quot;Geometry&quot;:console.error('THREE.ObjectLoader: Loading &quot;Geometry&quot; is not supported anymore.');break;default:console.warn('THREE.ObjectLoader: Unsupported geometry type &quot;'+g.type+'&quot;');continue}h.uuid=g.uuid;void 0!==g.name&&(h.name=g.name);
!0===h.isBufferGeometry&&void 0!==g.userData&&(h.userData=g.userData);c[g.uuid]=h}return c},parseMaterials:function(a,b){var c={},d={};if(void 0!==a){var e=new of;e.setTextures(b);b=0;for(var f=a.length;b<f;b++){var g=a[b];if(&quot;MultiMaterial&quot;===g.type){for(var h=[],l=0;l<g.materials.length;l++){var k=g.materials[l];void 0===c[k.uuid]&&(c[k.uuid]=e.parse(k));h.push(c[k.uuid])}d[g.uuid]=h}else void 0===c[g.uuid]&&(c[g.uuid]=e.parse(g)),d[g.uuid]=c[g.uuid]}}return d},parseAnimations:function(a){for(var b=
[],c=0;c<a.length;c++){var d=a[c],e=Qa.parse(d);void 0!==d.uuid&&(e.uuid=d.uuid);b.push(e)}return b},parseImages:function(a,b){function c(a){d.manager.itemStart(a);return f.load(a,function(){d.manager.itemEnd(a)},void 0,function(){d.manager.itemError(a);d.manager.itemEnd(a)})}var d=this,e={};if(void 0!==a&&0<a.length){b=new vg(b);var f=new fd(b);f.setCrossOrigin(this.crossOrigin);b=0;for(var g=a.length;b<g;b++){var h=a[b],l=h.url;if(Array.isArray(l)){e[h.uuid]=[];for(var k=0,n=l.length;k<n;k++){var p=
l[k];p=/^(\/\/)|([a-z]+:(\/\/)?)/i.test(p)?p:d.resourcePath+p;e[h.uuid].push(c(p))}}else p=/^(\/\/)|([a-z]+:(\/\/)?)/i.test(h.url)?h.url:d.resourcePath+h.url,e[h.uuid]=c(p)}}return e},parseTextures:function(a,b){function c(a,b){if(&quot;number&quot;===typeof a)return a;console.warn(&quot;THREE.ObjectLoader.parseTexture: Constant should be in numeric form.&quot;,a);return b[a]}var d={};if(void 0!==a)for(var e=0,f=a.length;e<f;e++){var g=a[e];void 0===g.image&&console.warn('THREE.ObjectLoader: No &quot;image&quot; specified for',
g.uuid);void 0===b[g.image]&&console.warn(&quot;THREE.ObjectLoader: Undefined image&quot;,g.image);var h=Array.isArray(b[g.image])?new qb(b[g.image]):new V(b[g.image]);h.needsUpdate=!0;h.uuid=g.uuid;void 0!==g.name&&(h.name=g.name);void 0!==g.mapping&&(h.mapping=c(g.mapping,cl));void 0!==g.offset&&h.offset.fromArray(g.offset);void 0!==g.repeat&&h.repeat.fromArray(g.repeat);void 0!==g.center&&h.center.fromArray(g.center);void 0!==g.rotation&&(h.rotation=g.rotation);void 0!==g.wrap&&(h.wrapS=c(g.wrap[0],aj),
h.wrapT=c(g.wrap[1],aj));void 0!==g.format&&(h.format=g.format);void 0!==g.type&&(h.type=g.type);void 0!==g.encoding&&(h.encoding=g.encoding);void 0!==g.minFilter&&(h.minFilter=c(g.minFilter,bj));void 0!==g.magFilter&&(h.magFilter=c(g.magFilter,bj));void 0!==g.anisotropy&&(h.anisotropy=g.anisotropy);void 0!==g.flipY&&(h.flipY=g.flipY);void 0!==g.premultiplyAlpha&&(h.premultiplyAlpha=g.premultiplyAlpha);void 0!==g.unpackAlignment&&(h.unpackAlignment=g.unpackAlignment);d[g.uuid]=h}return d},parseObject:function(a,
b,c){function d(a){void 0===b[a]&&console.warn(&quot;THREE.ObjectLoader: Undefined geometry&quot;,a);return b[a]}function e(a){if(void 0!==a){if(Array.isArray(a)){for(var b=[],d=0,e=a.length;d<e;d++){var f=a[d];void 0===c[f]&&console.warn(&quot;THREE.ObjectLoader: Undefined material&quot;,f);b.push(c[f])}return b}void 0===c[a]&&console.warn(&quot;THREE.ObjectLoader: Undefined material&quot;,a);return c[a]}}switch(a.type){case &quot;Scene&quot;:var f=new ob;void 0!==a.background&&Number.isInteger(a.background)&&(f.background=new A(a.background));
void 0!==a.fog&&(&quot;Fog&quot;===a.fog.type?f.fog=new Re(a.fog.color,a.fog.near,a.fog.far):&quot;FogExp2&quot;===a.fog.type&&(f.fog=new Qe(a.fog.color,a.fog.density)));break;case &quot;PerspectiveCamera&quot;:f=new aa(a.fov,a.aspect,a.near,a.far);void 0!==a.focus&&(f.focus=a.focus);void 0!==a.zoom&&(f.zoom=a.zoom);void 0!==a.filmGauge&&(f.filmGauge=a.filmGauge);void 0!==a.filmOffset&&(f.filmOffset=a.filmOffset);void 0!==a.view&&(f.view=Object.assign({},a.view));break;case &quot;OrthographicCamera&quot;:f=new hd(a.left,a.right,a.top,a.bottom,
a.near,a.far);void 0!==a.zoom&&(f.zoom=a.zoom);void 0!==a.view&&(f.view=Object.assign({},a.view));break;case &quot;AmbientLight&quot;:f=new mf(a.color,a.intensity);break;case &quot;DirectionalLight&quot;:f=new lf(a.color,a.intensity);break;case &quot;PointLight&quot;:f=new jf(a.color,a.intensity,a.distance,a.decay);break;case &quot;RectAreaLight&quot;:f=new nf(a.color,a.intensity,a.width,a.height);break;case &quot;SpotLight&quot;:f=new hf(a.color,a.intensity,a.distance,a.angle,a.penumbra,a.decay);break;case &quot;HemisphereLight&quot;:f=new ff(a.color,a.groundColor,
a.intensity);break;case &quot;SkinnedMesh&quot;:console.warn(&quot;THREE.ObjectLoader.parseObject() does not support SkinnedMesh yet.&quot;);case &quot;Mesh&quot;:f=d(a.geometry);var g=e(a.material);f=new S(f,g);break;case &quot;InstancedMesh&quot;:f=d(a.geometry);g=e(a.material);var h=a.instanceMatrix;f=new Ve(f,g,a.count);f.instanceMatrix=new M(new Float32Array(h.array),16);break;case &quot;LOD&quot;:f=new Rd;break;case &quot;Line&quot;:f=new Ka(d(a.geometry),e(a.material),a.mode);break;case &quot;LineLoop&quot;:f=new We(d(a.geometry),e(a.material));break;case &quot;LineSegments&quot;:f=
new ma(d(a.geometry),e(a.material));break;case &quot;PointCloud&quot;:case &quot;Points&quot;:f=new Pc(d(a.geometry),e(a.material));break;case &quot;Sprite&quot;:f=new Pd(e(a.material));break;case &quot;Group&quot;:f=new Mc;break;default:f=new F}f.uuid=a.uuid;void 0!==a.name&&(f.name=a.name);void 0!==a.matrix?(f.matrix.fromArray(a.matrix),void 0!==a.matrixAutoUpdate&&(f.matrixAutoUpdate=a.matrixAutoUpdate),f.matrixAutoUpdate&&f.matrix.decompose(f.position,f.quaternion,f.scale)):(void 0!==a.position&&f.position.fromArray(a.position),void 0!==
a.rotation&&f.rotation.fromArray(a.rotation),void 0!==a.quaternion&&f.quaternion.fromArray(a.quaternion),void 0!==a.scale&&f.scale.fromArray(a.scale));void 0!==a.castShadow&&(f.castShadow=a.castShadow);void 0!==a.receiveShadow&&(f.receiveShadow=a.receiveShadow);a.shadow&&(void 0!==a.shadow.bias&&(f.shadow.bias=a.shadow.bias),void 0!==a.shadow.radius&&(f.shadow.radius=a.shadow.radius),void 0!==a.shadow.mapSize&&f.shadow.mapSize.fromArray(a.shadow.mapSize),void 0!==a.shadow.camera&&(f.shadow.camera=
this.parseObject(a.shadow.camera)));void 0!==a.visible&&(f.visible=a.visible);void 0!==a.frustumCulled&&(f.frustumCulled=a.frustumCulled);void 0!==a.renderOrder&&(f.renderOrder=a.renderOrder);void 0!==a.userData&&(f.userData=a.userData);void 0!==a.layers&&(f.layers.mask=a.layers);if(void 0!==a.children)for(h=a.children,g=0;g<h.length;g++)f.add(this.parseObject(h[g],b,c));if(&quot;LOD&quot;===a.type)for(void 0!==a.autoUpdate&&(f.autoUpdate=a.autoUpdate),a=a.levels,h=0;h<a.length;h++){g=a[h];var l=f.getObjectByProperty(&quot;uuid&quot;,
g.object);void 0!==l&&f.addLevel(l,g.distance)}return f}});var cl={UVMapping:300,CubeReflectionMapping:301,CubeRefractionMapping:302,EquirectangularReflectionMapping:303,EquirectangularRefractionMapping:304,SphericalReflectionMapping:305,CubeUVReflectionMapping:306,CubeUVRefractionMapping:307},aj={RepeatWrapping:1E3,ClampToEdgeWrapping:1001,MirroredRepeatWrapping:1002},bj={NearestFilter:1003,NearestMipmapNearestFilter:1004,NearestMipmapLinearFilter:1005,LinearFilter:1006,LinearMipmapNearestFilter:1007,
LinearMipmapLinearFilter:1008};Ag.prototype=Object.assign(Object.create(W.prototype),{constructor:Ag,setOptions:function(a){this.options=a;return this},load:function(a,b,c,d){void 0===a&&(a=&quot;&quot;);void 0!==this.path&&(a=this.path+a);a=this.manager.resolveURL(a);var e=this,f=vc.get(a);if(void 0!==f)return e.manager.itemStart(a),setTimeout(function(){b&&b(f);e.manager.itemEnd(a)},0),f;fetch(a).then(function(a){return a.blob()}).then(function(a){return void 0===e.options?createImageBitmap(a):createImageBitmap(a,
e.options)}).then(function(c){vc.add(a,c);b&&b(c);e.manager.itemEnd(a)}).catch(function(b){d&&d(b);e.manager.itemError(a);e.manager.itemEnd(a)});e.manager.itemStart(a)}});Object.assign(Bg.prototype,{moveTo:function(a,b){this.currentPath=new $a;this.subPaths.push(this.currentPath);this.currentPath.moveTo(a,b);return this},lineTo:function(a,b){this.currentPath.lineTo(a,b);return this},quadraticCurveTo:function(a,b,c,d){this.currentPath.quadraticCurveTo(a,b,c,d);return this},bezierCurveTo:function(a,
b,c,d,e,f){this.currentPath.bezierCurveTo(a,b,c,d,e,f);return this},splineThru:function(a){this.currentPath.splineThru(a);return this},toShapes:function(a,b){function c(a){for(var b=[],c=0,d=a.length;c<d;c++){var e=a[c],f=new Kb;f.curves=e.curves;b.push(f)}return b}function d(a,b){for(var c=b.length,d=!1,e=c-1,f=0;f<c;e=f++){var g=b[e],h=b[f],k=h.x-g.x,l=h.y-g.y;if(Math.abs(l)>Number.EPSILON){if(0>l&&(g=b[f],k=-k,h=b[e],l=-l),!(a.y<g.y||a.y>h.y))if(a.y===g.y){if(a.x===g.x)return!0}else{e=l*(a.x-g.x)-
k*(a.y-g.y);if(0===e)return!0;0>e||(d=!d)}}else if(a.y===g.y&&(h.x<=a.x&&a.x<=g.x||g.x<=a.x&&a.x<=h.x))return!0}return d}var e=sb.isClockWise,f=this.subPaths;if(0===f.length)return[];if(!0===b)return c(f);b=[];if(1===f.length){var g=f[0];var h=new Kb;h.curves=g.curves;b.push(h);return b}var l=!e(f[0].getPoints());l=a?!l:l;h=[];var k=[],n=[],p=0;k[p]=void 0;n[p]=[];for(var t=0,r=f.length;t<r;t++){g=f[t];var q=g.getPoints();var v=e(q);(v=a?!v:v)?(!l&&k[p]&&p++,k[p]={s:new Kb,p:q},k[p].s.curves=g.curves,
l&&p++,n[p]=[]):n[p].push({h:g,p:q[0]})}if(!k[0])return c(f);if(1<k.length){t=!1;a=[];e=0;for(f=k.length;e<f;e++)h[e]=[];e=0;for(f=k.length;e<f;e++)for(g=n[e],v=0;v<g.length;v++){l=g[v];p=!0;for(q=0;q<k.length;q++)d(l.p,k[q].p)&&(e!==q&&a.push({froms:e,tos:q,hole:v}),p?(p=!1,h[q].push(l)):t=!0);p&&h[e].push(l)}0<a.length&&(t||(n=h))}t=0;for(e=k.length;t<e;t++)for(h=k[t].s,b.push(h),a=n[t],f=0,g=a.length;f<g;f++)h.holes.push(a[f].h);return b}});Object.assign(Cg.prototype,{isFont:!0,generateShapes:function(a,
b){void 0===b&&(b=100);var c=[],d=b;b=this.data;var e=Array.from?Array.from(a):String(a).split(&quot;&quot;);d/=b.resolution;var f=(b.boundingBox.yMax-b.boundingBox.yMin+b.underlineThickness)*d;a=[];for(var g=0,h=0,l=0;l<e.length;l++){var k=e[l];if(&quot;\n&quot;===k)g=0,h-=f;else{var n=k;k=d;var p=g,t=h,r=b,q=r.glyphs[n]||r.glyphs[&quot;?&quot;];if(q){n=new Bg;if(q.o){r=q._cachedOutline||(q._cachedOutline=q.o.split(&quot; &quot;));for(var v=0,y=r.length;v<y;)switch(r[v++]){case &quot;m&quot;:var w=r[v++]*k+p;var z=r[v++]*k+t;n.moveTo(w,z);break;
case &quot;l&quot;:w=r[v++]*k+p;z=r[v++]*k+t;n.lineTo(w,z);break;case &quot;q&quot;:var A=r[v++]*k+p;var C=r[v++]*k+t;var D=r[v++]*k+p;var B=r[v++]*k+t;n.quadraticCurveTo(D,B,A,C);break;case &quot;b&quot;:A=r[v++]*k+p,C=r[v++]*k+t,D=r[v++]*k+p,B=r[v++]*k+t,w=r[v++]*k+p,z=r[v++]*k+t,n.bezierCurveTo(D,B,w,z,A,C)}}k={offsetX:q.ha*k,path:n}}else console.error('THREE.Font: character &quot;'+n+'&quot; does not exists in font family '+r.familyName+&quot;.&quot;),k=void 0;g+=k.offsetX;a.push(k.path)}}b=0;for(e=a.length;b<e;b++)Array.prototype.push.apply(c,
a[b].toShapes());return c}});Dg.prototype=Object.assign(Object.create(W.prototype),{constructor:Dg,load:function(a,b,c,d){var e=this,f=new Ra(this.manager);f.setPath(this.path);f.load(a,function(a){try{var c=JSON.parse(a)}catch(l){console.warn(&quot;THREE.FontLoader: typeface.js support is being deprecated. Use typeface.json instead.&quot;),c=JSON.parse(a.substring(65,a.length-2))}a=e.parse(c);b&&b(a)},c,d)},parse:function(a){return new Cg(a)}});var Uf,Ig={getContext:function(){void 0===Uf&&(Uf=new (window.AudioContext||
window.webkitAudioContext));return Uf},setContext:function(a){Uf=a}};tf.prototype=Object.assign(Object.create(W.prototype),{constructor:tf,load:function(a,b,c,d){var e=new Ra(this.manager);e.setResponseType(&quot;arraybuffer&quot;);e.setPath(this.path);e.load(a,function(a){a=a.slice(0);Ig.getContext().decodeAudioData(a,function(a){b(a)})},c,d)}});Object.assign(uf.prototype,{isSphericalHarmonics3:!0,set:function(a){for(var b=0;9>b;b++)this.coefficients[b].copy(a[b]);return this},zero:function(){for(var a=0;9>
a;a++)this.coefficients[a].set(0,0,0);return this},getAt:function(a,b){var c=a.x,d=a.y;a=a.z;var e=this.coefficients;b.copy(e[0]).multiplyScalar(.282095);b.addScaledVector(e[1],.488603*d);b.addScaledVector(e[2],.488603*a);b.addScaledVector(e[3],.488603*c);b.addScaledVector(e[4],1.092548*c*d);b.addScaledVector(e[5],1.092548*d*a);b.addScaledVector(e[6],.315392*(3*a*a-1));b.addScaledVector(e[7],1.092548*c*a);b.addScaledVector(e[8],.546274*(c*c-d*d));return b},getIrradianceAt:function(a,b){var c=a.x,
d=a.y;a=a.z;var e=this.coefficients;b.copy(e[0]).multiplyScalar(.886227);b.addScaledVector(e[1],1.023328*d);b.addScaledVector(e[2],1.023328*a);b.addScaledVector(e[3],1.023328*c);b.addScaledVector(e[4],.858086*c*d);b.addScaledVector(e[5],.858086*d*a);b.addScaledVector(e[6],.743125*a*a-.247708);b.addScaledVector(e[7],.858086*c*a);b.addScaledVector(e[8],.429043*(c*c-d*d));return b},add:function(a){for(var b=0;9>b;b++)this.coefficients[b].add(a.coefficients[b]);return this},addScaledSH:function(a,b){for(var c=
0;9>c;c++)this.coefficients[c].addScaledVector(a.coefficients[c],b);return this},scale:function(a){for(var b=0;9>b;b++)this.coefficients[b].multiplyScalar(a);return this},lerp:function(a,b){for(var c=0;9>c;c++)this.coefficients[c].lerp(a.coefficients[c],b);return this},equals:function(a){for(var b=0;9>b;b++)if(!this.coefficients[b].equals(a.coefficients[b]))return!1;return!0},copy:function(a){return this.set(a.coefficients)},clone:function(){return(new this.constructor).copy(this)},fromArray:function(a,
b){void 0===b&&(b=0);for(var c=this.coefficients,d=0;9>d;d++)c[d].fromArray(a,b+3*d);return this},toArray:function(a,b){void 0===a&&(a=[]);void 0===b&&(b=0);for(var c=this.coefficients,d=0;9>d;d++)c[d].toArray(a,b+3*d);return a}});Object.assign(uf,{getBasisAt:function(a,b){var c=a.x,d=a.y;a=a.z;b[0]=.282095;b[1]=.488603*d;b[2]=.488603*a;b[3]=.488603*c;b[4]=1.092548*c*d;b[5]=1.092548*d*a;b[6]=.315392*(3*a*a-1);b[7]=1.092548*c*a;b[8]=.546274*(c*c-d*d)}});ab.prototype=Object.assign(Object.create(da.prototype),
{constructor:ab,isLightProbe:!0,copy:function(a){da.prototype.copy.call(this,a);this.sh.copy(a.sh);this.intensity=a.intensity;return this},toJSON:function(a){return da.prototype.toJSON.call(this,a)}});Eg.prototype=Object.assign(Object.create(ab.prototype),{constructor:Eg,isHemisphereLightProbe:!0,copy:function(a){ab.prototype.copy.call(this,a);return this},toJSON:function(a){return ab.prototype.toJSON.call(this,a)}});Fg.prototype=Object.assign(Object.create(ab.prototype),{constructor:Fg,isAmbientLightProbe:!0,
copy:function(a){ab.prototype.copy.call(this,a);return this},toJSON:function(a){return ab.prototype.toJSON.call(this,a)}});var cj=new P,dj=new P;Object.assign(mi.prototype,{update:function(a){var b=this._cache;if(b.focus!==a.focus||b.fov!==a.fov||b.aspect!==a.aspect*this.aspect||b.near!==a.near||b.far!==a.far||b.zoom!==a.zoom||b.eyeSep!==this.eyeSep){b.focus=a.focus;b.fov=a.fov;b.aspect=a.aspect*this.aspect;b.near=a.near;b.far=a.far;b.zoom=a.zoom;b.eyeSep=this.eyeSep;var c=a.projectionMatrix.clone(),
d=b.eyeSep/2,e=d*b.near/b.focus,f=b.near*Math.tan(L.DEG2RAD*b.fov*.5)/b.zoom;dj.elements[12]=-d;cj.elements[12]=d;d=-f*b.aspect+e;var g=f*b.aspect+e;c.elements[0]=2*b.near/(g-d);c.elements[8]=(g+d)/(g-d);this.cameraL.projectionMatrix.copy(c);d=-f*b.aspect-e;g=f*b.aspect-e;c.elements[0]=2*b.near/(g-d);c.elements[8]=(g+d)/(g-d);this.cameraR.projectionMatrix.copy(c)}this.cameraL.matrixWorld.copy(a.matrixWorld).multiply(dj);this.cameraR.matrixWorld.copy(a.matrixWorld).multiply(cj)}});Object.assign(Gg.prototype,
{start:function(){this.oldTime=this.startTime=(&quot;undefined&quot;===typeof performance?Date:performance).now();this.elapsedTime=0;this.running=!0},stop:function(){this.getElapsedTime();this.autoStart=this.running=!1},getElapsedTime:function(){this.getDelta();return this.elapsedTime},getDelta:function(){var a=0;if(this.autoStart&&!this.running)return this.start(),0;if(this.running){var b=(&quot;undefined&quot;===typeof performance?Date:performance).now();a=(b-this.oldTime)/1E3;this.oldTime=b;this.elapsedTime+=a}return a}});
var wc=new n,ej=new Aa,dl=new n,xc=new n;Hg.prototype=Object.assign(Object.create(F.prototype),{constructor:Hg,getInput:function(){return this.gain},removeFilter:function(){null!==this.filter&&(this.gain.disconnect(this.filter),this.filter.disconnect(this.context.destination),this.gain.connect(this.context.destination),this.filter=null);return this},getFilter:function(){return this.filter},setFilter:function(a){null!==this.filter?(this.gain.disconnect(this.filter),this.filter.disconnect(this.context.destination)):
this.gain.disconnect(this.context.destination);this.filter=a;this.gain.connect(this.filter);this.filter.connect(this.context.destination);return this},getMasterVolume:function(){return this.gain.gain.value},setMasterVolume:function(a){this.gain.gain.setTargetAtTime(a,this.context.currentTime,.01);return this},updateMatrixWorld:function(a){F.prototype.updateMatrixWorld.call(this,a);a=this.context.listener;var b=this.up;this.timeDelta=this._clock.getDelta();this.matrixWorld.decompose(wc,ej,dl);xc.set(0,
0,-1).applyQuaternion(ej);if(a.positionX){var c=this.context.currentTime+this.timeDelta;a.positionX.linearRampToValueAtTime(wc.x,c);a.positionY.linearRampToValueAtTime(wc.y,c);a.positionZ.linearRampToValueAtTime(wc.z,c);a.forwardX.linearRampToValueAtTime(xc.x,c);a.forwardY.linearRampToValueAtTime(xc.y,c);a.forwardZ.linearRampToValueAtTime(xc.z,c);a.upX.linearRampToValueAtTime(b.x,c);a.upY.linearRampToValueAtTime(b.y,c);a.upZ.linearRampToValueAtTime(b.z,c)}else a.setPosition(wc.x,wc.y,wc.z),a.setOrientation(xc.x,
xc.y,xc.z,b.x,b.y,b.z)}});id.prototype=Object.assign(Object.create(F.prototype),{constructor:id,getOutput:function(){return this.gain},setNodeSource:function(a){this.hasPlaybackControl=!1;this.sourceType=&quot;audioNode&quot;;this.source=a;this.connect();return this},setMediaElementSource:function(a){this.hasPlaybackControl=!1;this.sourceType=&quot;mediaNode&quot;;this.source=this.context.createMediaElementSource(a);this.connect();return this},setMediaStreamSource:function(a){this.hasPlaybackControl=!1;this.sourceType=
&quot;mediaStreamNode&quot;;this.source=this.context.createMediaStreamSource(a);this.connect();return this},setBuffer:function(a){this.buffer=a;this.sourceType=&quot;buffer&quot;;this.autoplay&&this.play();return this},play:function(a){void 0===a&&(a=0);if(!0===this.isPlaying)console.warn(&quot;THREE.Audio: Audio is already playing.&quot;);else if(!1===this.hasPlaybackControl)console.warn(&quot;THREE.Audio: this Audio has no playback control.&quot;);else return this._startedAt=this.context.currentTime+a,a=this.context.createBufferSource(),
a.buffer=this.buffer,a.loop=this.loop,a.loopStart=this.loopStart,a.loopEnd=this.loopEnd,a.onended=this.onEnded.bind(this),a.start(this._startedAt,this._pausedAt+this.offset,this.duration),this.isPlaying=!0,this.source=a,this.setDetune(this.detune),this.setPlaybackRate(this.playbackRate),this.connect()},pause:function(){if(!1===this.hasPlaybackControl)console.warn(&quot;THREE.Audio: this Audio has no playback control.&quot;);else return!0===this.isPlaying&&(this._pausedAt+=Math.max(this.context.currentTime-
this._startedAt,0)*this.playbackRate,this.source.stop(),this.source.onended=null,this.isPlaying=!1),this},stop:function(){if(!1===this.hasPlaybackControl)console.warn(&quot;THREE.Audio: this Audio has no playback control.&quot;);else return this._pausedAt=0,this.source.stop(),this.source.onended=null,this.isPlaying=!1,this},connect:function(){if(0<this.filters.length){this.source.connect(this.filters[0]);for(var a=1,b=this.filters.length;a<b;a++)this.filters[a-1].connect(this.filters[a]);this.filters[this.filters.length-
1].connect(this.getOutput())}else this.source.connect(this.getOutput());return this},disconnect:function(){if(0<this.filters.length){this.source.disconnect(this.filters[0]);for(var a=1,b=this.filters.length;a<b;a++)this.filters[a-1].disconnect(this.filters[a]);this.filters[this.filters.length-1].disconnect(this.getOutput())}else this.source.disconnect(this.getOutput());return this},getFilters:function(){return this.filters},setFilters:function(a){a||(a=[]);!0===this.isPlaying?(this.disconnect(),this.filters=
a,this.connect()):this.filters=a;return this},setDetune:function(a){this.detune=a;if(void 0!==this.source.detune)return!0===this.isPlaying&&this.source.detune.setTargetAtTime(this.detune,this.context.currentTime,.01),this},getDetune:function(){return this.detune},getFilter:function(){return this.getFilters()[0]},setFilter:function(a){return this.setFilters(a?[a]:[])},setPlaybackRate:function(a){if(!1===this.hasPlaybackControl)console.warn(&quot;THREE.Audio: this Audio has no playback control.&quot;);else return this.playbackRate=
a,!0===this.isPlaying&&this.source.playbackRate.setTargetAtTime(this.playbackRate,this.context.currentTime,.01),this},getPlaybackRate:function(){return this.playbackRate},onEnded:function(){this.isPlaying=!1},getLoop:function(){return!1===this.hasPlaybackControl?(console.warn(&quot;THREE.Audio: this Audio has no playback control.&quot;),!1):this.loop},setLoop:function(a){if(!1===this.hasPlaybackControl)console.warn(&quot;THREE.Audio: this Audio has no playback control.&quot;);else return this.loop=a,!0===this.isPlaying&&
(this.source.loop=this.loop),this},setLoopStart:function(a){this.loopStart=a;return this},setLoopEnd:function(a){this.loopEnd=a;return this},getVolume:function(){return this.gain.gain.value},setVolume:function(a){this.gain.gain.setTargetAtTime(a,this.context.currentTime,.01);return this}});var yc=new n,fj=new Aa,el=new n,zc=new n;Jg.prototype=Object.assign(Object.create(id.prototype),{constructor:Jg,getOutput:function(){return this.panner},getRefDistance:function(){return this.panner.refDistance},
setRefDistance:function(a){this.panner.refDistance=a;return this},getRolloffFactor:function(){return this.panner.rolloffFactor},setRolloffFactor:function(a){this.panner.rolloffFactor=a;return this},getDistanceModel:function(){return this.panner.distanceModel},setDistanceModel:function(a){this.panner.distanceModel=a;return this},getMaxDistance:function(){return this.panner.maxDistance},setMaxDistance:function(a){this.panner.maxDistance=a;return this},setDirectionalCone:function(a,b,c){this.panner.coneInnerAngle=
a;this.panner.coneOuterAngle=b;this.panner.coneOuterGain=c;return this},updateMatrixWorld:function(a){F.prototype.updateMatrixWorld.call(this,a);if(!0!==this.hasPlaybackControl||!1!==this.isPlaying)if(this.matrixWorld.decompose(yc,fj,el),zc.set(0,0,1).applyQuaternion(fj),a=this.panner,a.positionX){var b=this.context.currentTime+this.listener.timeDelta;a.positionX.linearRampToValueAtTime(yc.x,b);a.positionY.linearRampToValueAtTime(yc.y,b);a.positionZ.linearRampToValueAtTime(yc.z,b);a.orientationX.linearRampToValueAtTime(zc.x,
b);a.orientationY.linearRampToValueAtTime(zc.y,b);a.orientationZ.linearRampToValueAtTime(zc.z,b)}else a.setPosition(yc.x,yc.y,yc.z),a.setOrientation(zc.x,zc.y,zc.z)}});Object.assign(Kg.prototype,{getFrequencyData:function(){this.analyser.getByteFrequencyData(this.data);return this.data},getAverageFrequency:function(){for(var a=0,b=this.getFrequencyData(),c=0;c<b.length;c++)a+=b[c];return a/b.length}});Object.assign(Lg.prototype,{accumulate:function(a,b){var c=this.buffer,d=this.valueSize;a=a*d+d;
var e=this.cumulativeWeight;if(0===e){for(e=0;e!==d;++e)c[a+e]=c[e];e=b}else e+=b,this._mixBufferRegion(c,a,0,b/e,d);this.cumulativeWeight=e},apply:function(a){var b=this.valueSize,c=this.buffer;a=a*b+b;var d=this.cumulativeWeight,e=this.binding;this.cumulativeWeight=0;1>d&&this._mixBufferRegion(c,a,3*b,1-d,b);d=b;for(var f=b+b;d!==f;++d)if(c[d]!==c[d+b]){e.setValue(c,a);break}},saveOriginalState:function(){var a=this.buffer,b=this.valueSize,c=3*b;this.binding.getValue(a,c);for(var d=b;d!==c;++d)a[d]=
a[c+d%b];this.cumulativeWeight=0},restoreOriginalState:function(){this.binding.setValue(this.buffer,3*this.valueSize)},_select:function(a,b,c,d,e){if(.5<=d)for(d=0;d!==e;++d)a[b+d]=a[c+d]},_slerp:function(a,b,c,d){Aa.slerpFlat(a,b,a,b,a,c,d)},_lerp:function(a,b,c,d,e){for(var f=1-d,g=0;g!==e;++g){var h=b+g;a[h]=a[h]*f+a[c+g]*d}}});var fl=/[\[\]\.:\/]/g,gl=&quot;[^&quot;+&quot;\\[\\]\\.:\\/&quot;.replace(&quot;\\.&quot;,&quot;&quot;)+&quot;]&quot;,hl=/((?:WC+[\/:])*)/.source.replace(&quot;WC&quot;,&quot;[^\\[\\]\\.:\\/]&quot;),il=/(WCOD+)?/.source.replace(&quot;WCOD&quot;,gl),
jl=/(?:\.(WC+)(?:\[(.+)\])?)?/.source.replace(&quot;WC&quot;,&quot;[^\\[\\]\\.:\\/]&quot;),kl=/\.(WC+)(?:\[(.+)\])?/.source.replace(&quot;WC&quot;,&quot;[^\\[\\]\\.:\\/]&quot;),ll=new RegExp(&quot;^&quot;+hl+il+jl+kl+&quot;$&quot;),ml=[&quot;material&quot;,&quot;materials&quot;,&quot;bones&quot;];Object.assign(ni.prototype,{getValue:function(a,b){this.bind();var c=this._bindings[this._targetGroup.nCachedObjects_];void 0!==c&&c.getValue(a,b)},setValue:function(a,b){for(var c=this._bindings,d=this._targetGroup.nCachedObjects_,e=c.length;d!==e;++d)c[d].setValue(a,b)},bind:function(){for(var a=
this._bindings,b=this._targetGroup.nCachedObjects_,c=a.length;b!==c;++b)a[b].bind()},unbind:function(){for(var a=this._bindings,b=this._targetGroup.nCachedObjects_,c=a.length;b!==c;++b)a[b].unbind()}});Object.assign(ya,{Composite:ni,create:function(a,b,c){return a&&a.isAnimationObjectGroup?new ya.Composite(a,b,c):new ya(a,b,c)},sanitizeNodeName:function(a){return a.replace(/\s/g,&quot;_&quot;).replace(fl,&quot;&quot;)},parseTrackName:function(a){var b=ll.exec(a);if(!b)throw Error(&quot;PropertyBinding: Cannot parse trackName: &quot;+
a);b={nodeName:b[2],objectName:b[3],objectIndex:b[4],propertyName:b[5],propertyIndex:b[6]};var c=b.nodeName&&b.nodeName.lastIndexOf(&quot;.&quot;);if(void 0!==c&&-1!==c){var d=b.nodeName.substring(c+1);-1!==ml.indexOf(d)&&(b.nodeName=b.nodeName.substring(0,c),b.objectName=d)}if(null===b.propertyName||0===b.propertyName.length)throw Error(&quot;PropertyBinding: can not parse propertyName from trackName: &quot;+a);return b},findNode:function(a,b){if(!b||&quot;&quot;===b||&quot;.&quot;===b||-1===b||b===a.name||b===a.uuid)return a;if(a.skeleton){var c=
a.skeleton.getBoneByName(b);if(void 0!==c)return c}if(a.children){var d=function(a){for(var c=0;c<a.length;c++){var e=a[c];if(e.name===b||e.uuid===b||(e=d(e.children)))return e}return null};if(a=d(a.children))return a}return null}});Object.assign(ya.prototype,{_getValue_unavailable:function(){},_setValue_unavailable:function(){},BindingType:{Direct:0,EntireArray:1,ArrayElement:2,HasFromToArray:3},Versioning:{None:0,NeedsUpdate:1,MatrixWorldNeedsUpdate:2},GetterByBindingType:[function(a,b){a[b]=this.node[this.propertyName]},
function(a,b){for(var c=this.resolvedProperty,d=0,e=c.length;d!==e;++d)a[b++]=c[d]},function(a,b){a[b]=this.resolvedProperty[this.propertyIndex]},function(a,b){this.resolvedProperty.toArray(a,b)}],SetterByBindingTypeAndVersioning:[[function(a,b){this.targetObject[this.propertyName]=a[b]},function(a,b){this.targetObject[this.propertyName]=a[b];this.targetObject.needsUpdate=!0},function(a,b){this.targetObject[this.propertyName]=a[b];this.targetObject.matrixWorldNeedsUpdate=!0}],[function(a,b){for(var c=
this.resolvedProperty,d=0,e=c.length;d!==e;++d)c[d]=a[b++]},function(a,b){for(var c=this.resolvedProperty,d=0,e=c.length;d!==e;++d)c[d]=a[b++];this.targetObject.needsUpdate=!0},function(a,b){for(var c=this.resolvedProperty,d=0,e=c.length;d!==e;++d)c[d]=a[b++];this.targetObject.matrixWorldNeedsUpdate=!0}],[function(a,b){this.resolvedProperty[this.propertyIndex]=a[b]},function(a,b){this.resolvedProperty[this.propertyIndex]=a[b];this.targetObject.needsUpdate=!0},function(a,b){this.resolvedProperty[this.propertyIndex]=
a[b];this.targetObject.matrixWorldNeedsUpdate=!0}],[function(a,b){this.resolvedProperty.fromArray(a,b)},function(a,b){this.resolvedProperty.fromArray(a,b);this.targetObject.needsUpdate=!0},function(a,b){this.resolvedProperty.fromArray(a,b);this.targetObject.matrixWorldNeedsUpdate=!0}]],getValue:function(a,b){this.bind();this.getValue(a,b)},setValue:function(a,b){this.bind();this.setValue(a,b)},bind:function(){var a=this.node,b=this.parsedPath,c=b.objectName,d=b.propertyName,e=b.propertyIndex;a||(this.node=
a=ya.findNode(this.rootNode,b.nodeName)||this.rootNode);this.getValue=this._getValue_unavailable;this.setValue=this._setValue_unavailable;if(a){if(c){var f=b.objectIndex;switch(c){case &quot;materials&quot;:if(!a.material){console.error(&quot;THREE.PropertyBinding: Can not bind to material as node does not have a material.&quot;,this);return}if(!a.material.materials){console.error(&quot;THREE.PropertyBinding: Can not bind to material.materials as node.material does not have a materials array.&quot;,this);return}a=a.material.materials;
break;case &quot;bones&quot;:if(!a.skeleton){console.error(&quot;THREE.PropertyBinding: Can not bind to bones as node does not have a skeleton.&quot;,this);return}a=a.skeleton.bones;for(c=0;c<a.length;c++)if(a[c].name===f){f=c;break}break;default:if(void 0===a[c]){console.error(&quot;THREE.PropertyBinding: Can not bind to objectName of node undefined.&quot;,this);return}a=a[c]}if(void 0!==f){if(void 0===a[f]){console.error(&quot;THREE.PropertyBinding: Trying to bind to objectIndex of objectName, but is undefined.&quot;,this,a);return}a=
a[f]}}f=a[d];if(void 0===f)console.error(&quot;THREE.PropertyBinding: Trying to update property for track: &quot;+b.nodeName+&quot;.&quot;+d+&quot; but it wasn't found.&quot;,a);else{b=this.Versioning.None;this.targetObject=a;void 0!==a.needsUpdate?b=this.Versioning.NeedsUpdate:void 0!==a.matrixWorldNeedsUpdate&&(b=this.Versioning.MatrixWorldNeedsUpdate);c=this.BindingType.Direct;if(void 0!==e){if(&quot;morphTargetInfluences&quot;===d){if(!a.geometry){console.error(&quot;THREE.PropertyBinding: Can not bind to morphTargetInfluences because node does not have a geometry.&quot;,
this);return}if(a.geometry.isBufferGeometry){if(!a.geometry.morphAttributes){console.error(&quot;THREE.PropertyBinding: Can not bind to morphTargetInfluences because node does not have a geometry.morphAttributes.&quot;,this);return}for(c=0;c<this.node.geometry.morphAttributes.position.length;c++)if(a.geometry.morphAttributes.position[c].name===e){e=c;break}}else{if(!a.geometry.morphTargets){console.error(&quot;THREE.PropertyBinding: Can not bind to morphTargetInfluences because node does not have a geometry.morphTargets.&quot;,
this);return}for(c=0;c<this.node.geometry.morphTargets.length;c++)if(a.geometry.morphTargets[c].name===e){e=c;break}}}c=this.BindingType.ArrayElement;this.resolvedProperty=f;this.propertyIndex=e}else void 0!==f.fromArray&&void 0!==f.toArray?(c=this.BindingType.HasFromToArray,this.resolvedProperty=f):Array.isArray(f)?(c=this.BindingType.EntireArray,this.resolvedProperty=f):this.propertyName=d;this.getValue=this.GetterByBindingType[c];this.setValue=this.SetterByBindingTypeAndVersioning[c][b]}}else console.error(&quot;THREE.PropertyBinding: Trying to update node for track: &quot;+
this.path+&quot; but it wasn't found.&quot;)},unbind:function(){this.node=null;this.getValue=this._getValue_unbound;this.setValue=this._setValue_unbound}});Object.assign(ya.prototype,{_getValue_unbound:ya.prototype.getValue,_setValue_unbound:ya.prototype.setValue});Object.assign(oi.prototype,{isAnimationObjectGroup:!0,add:function(){for(var a=this._objects,b=a.length,c=this.nCachedObjects_,d=this._indicesByUUID,e=this._paths,f=this._parsedPaths,g=this._bindings,h=g.length,k=void 0,m=0,n=arguments.length;m!==
n;++m){var p=arguments[m],t=p.uuid,r=d[t];if(void 0===r){r=b++;d[t]=r;a.push(p);t=0;for(var q=h;t!==q;++t)g[t].push(new ya(p,e[t],f[t]))}else if(r<c){k=a[r];var v=--c;q=a[v];d[q.uuid]=r;a[r]=q;d[t]=v;a[v]=p;t=0;for(q=h;t!==q;++t){var y=g[t],w=y[r];y[r]=y[v];void 0===w&&(w=new ya(p,e[t],f[t]));y[v]=w}}else a[r]!==k&&console.error(&quot;THREE.AnimationObjectGroup: Different objects with the same UUID detected. Clean the caches or recreate your infrastructure when reloading scenes.&quot;)}this.nCachedObjects_=
c},remove:function(){for(var a=this._objects,b=this.nCachedObjects_,c=this._indicesByUUID,d=this._bindings,e=d.length,f=0,g=arguments.length;f!==g;++f){var h=arguments[f],k=h.uuid,m=c[k];if(void 0!==m&&m>=b){var n=b++,p=a[n];c[p.uuid]=m;a[m]=p;c[k]=n;a[n]=h;h=0;for(k=e;h!==k;++h){p=d[h];var t=p[m];p[m]=p[n];p[n]=t}}}this.nCachedObjects_=b},uncache:function(){for(var a=this._objects,b=a.length,c=this.nCachedObjects_,d=this._indicesByUUID,e=this._bindings,f=e.length,g=0,h=arguments.length;g!==h;++g){var k=
arguments[g].uuid,m=d[k];if(void 0!==m)if(delete d[k],m<c){k=--c;var n=a[k],p=--b,t=a[p];d[n.uuid]=m;a[m]=n;d[t.uuid]=k;a[k]=t;a.pop();n=0;for(t=f;n!==t;++n){var r=e[n],q=r[p];r[m]=r[k];r[k]=q;r.pop()}}else for(p=--b,t=a[p],d[t.uuid]=m,a[m]=t,a.pop(),n=0,t=f;n!==t;++n)r=e[n],r[m]=r[p],r.pop()}this.nCachedObjects_=c},subscribe_:function(a,b){var c=this._bindingsIndicesByPath,d=c[a],e=this._bindings;if(void 0!==d)return e[d];var f=this._paths,g=this._parsedPaths,h=this._objects,k=this.nCachedObjects_,
m=Array(h.length);d=e.length;c[a]=d;f.push(a);g.push(b);e.push(m);c=k;for(d=h.length;c!==d;++c)m[c]=new ya(h[c],a,b);return m},unsubscribe_:function(a){var b=this._bindingsIndicesByPath,c=b[a];if(void 0!==c){var d=this._paths,e=this._parsedPaths,f=this._bindings,g=f.length-1,h=f[g];b[a[g]]=c;f[c]=h;f.pop();e[c]=e[g];e.pop();d[c]=d[g];d.pop()}}});Object.assign(pi.prototype,{play:function(){this._mixer._activateAction(this);return this},stop:function(){this._mixer._deactivateAction(this);return this.reset()},
reset:function(){this.paused=!1;this.enabled=!0;this.time=0;this._loopCount=-1;this._startTime=null;return this.stopFading().stopWarping()},isRunning:function(){return this.enabled&&!this.paused&&0!==this.timeScale&&null===this._startTime&&this._mixer._isActiveAction(this)},isScheduled:function(){return this._mixer._isActiveAction(this)},startAt:function(a){this._startTime=a;return this},setLoop:function(a,b){this.loop=a;this.repetitions=b;return this},setEffectiveWeight:function(a){this.weight=a;
this._effectiveWeight=this.enabled?a:0;return this.stopFading()},getEffectiveWeight:function(){return this._effectiveWeight},fadeIn:function(a){return this._scheduleFading(a,0,1)},fadeOut:function(a){return this._scheduleFading(a,1,0)},crossFadeFrom:function(a,b,c){a.fadeOut(b);this.fadeIn(b);if(c){c=this._clip.duration;var d=a._clip.duration,e=c/d;a.warp(1,d/c,b);this.warp(e,1,b)}return this},crossFadeTo:function(a,b,c){return a.crossFadeFrom(this,b,c)},stopFading:function(){var a=this._weightInterpolant;
null!==a&&(this._weightInterpolant=null,this._mixer._takeBackControlInterpolant(a));return this},setEffectiveTimeScale:function(a){this.timeScale=a;this._effectiveTimeScale=this.paused?0:a;return this.stopWarping()},getEffectiveTimeScale:function(){return this._effectiveTimeScale},setDuration:function(a){this.timeScale=this._clip.duration/a;return this.stopWarping()},syncWith:function(a){this.time=a.time;this.timeScale=a.timeScale;return this.stopWarping()},halt:function(a){return this.warp(this._effectiveTimeScale,
0,a)},warp:function(a,b,c){var d=this._mixer,e=d.time,f=this._timeScaleInterpolant,g=this.timeScale;null===f&&(this._timeScaleInterpolant=f=d._lendControlInterpolant());d=f.parameterPositions;f=f.sampleValues;d[0]=e;d[1]=e+c;f[0]=a/g;f[1]=b/g;return this},stopWarping:function(){var a=this._timeScaleInterpolant;null!==a&&(this._timeScaleInterpolant=null,this._mixer._takeBackControlInterpolant(a));return this},getMixer:function(){return this._mixer},getClip:function(){return this._clip},getRoot:function(){return this._localRoot||
this._mixer._root},_update:function(a,b,c,d){if(this.enabled){var e=this._startTime;if(null!==e){b=(a-e)*c;if(0>b||0===c)return;this._startTime=null;b*=c}b*=this._updateTimeScale(a);c=this._updateTime(b);a=this._updateWeight(a);if(0<a){b=this._interpolants;e=this._propertyBindings;for(var f=0,g=b.length;f!==g;++f)b[f].evaluate(c),e[f].accumulate(d,a)}}else this._updateWeight(a)},_updateWeight:function(a){var b=0;if(this.enabled){b=this.weight;var c=this._weightInterpolant;if(null!==c){var d=c.evaluate(a)[0];
b*=d;a>c.parameterPositions[1]&&(this.stopFading(),0===d&&(this.enabled=!1))}}return this._effectiveWeight=b},_updateTimeScale:function(a){var b=0;if(!this.paused){b=this.timeScale;var c=this._timeScaleInterpolant;if(null!==c){var d=c.evaluate(a)[0];b*=d;a>c.parameterPositions[1]&&(this.stopWarping(),0===b?this.paused=!0:this.timeScale=b)}}return this._effectiveTimeScale=b},_updateTime:function(a){var b=this.time+a,c=this._clip.duration,d=this.loop,e=this._loopCount,f=2202===d;if(0===a)return-1===
e?b:f&&1===(e&1)?c-b:b;if(2200===d)a:{if(-1===e&&(this._loopCount=0,this._setEndings(!0,!0,!1)),b>=c)b=c;else if(0>b)b=0;else{this.time=b;break a}this.clampWhenFinished?this.paused=!0:this.enabled=!1;this.time=b;this._mixer.dispatchEvent({type:&quot;finished&quot;,action:this,direction:0>a?-1:1})}else{-1===e&&(0<=a?(e=0,this._setEndings(!0,0===this.repetitions,f)):this._setEndings(0===this.repetitions,!0,f));if(b>=c||0>b){d=Math.floor(b/c);b-=c*d;e+=Math.abs(d);var g=this.repetitions-e;0>=g?(this.clampWhenFinished?
this.paused=!0:this.enabled=!1,this.time=b=0<a?c:0,this._mixer.dispatchEvent({type:&quot;finished&quot;,action:this,direction:0<a?1:-1})):(1===g?(a=0>a,this._setEndings(a,!a,f)):this._setEndings(!1,!1,f),this._loopCount=e,this.time=b,this._mixer.dispatchEvent({type:&quot;loop&quot;,action:this,loopDelta:d}))}else this.time=b;if(f&&1===(e&1))return c-b}return b},_setEndings:function(a,b,c){var d=this._interpolantSettings;c?(d.endingStart=2401,d.endingEnd=2401):(d.endingStart=a?this.zeroSlopeAtStart?2401:2400:2402,d.endingEnd=
b?this.zeroSlopeAtEnd?2401:2400:2402)},_scheduleFading:function(a,b,c){var d=this._mixer,e=d.time,f=this._weightInterpolant;null===f&&(this._weightInterpolant=f=d._lendControlInterpolant());d=f.parameterPositions;f=f.sampleValues;d[0]=e;f[0]=b;d[1]=e+a;f[1]=c;return this}});Mg.prototype=Object.assign(Object.create(Ea.prototype),{constructor:Mg,_bindAction:function(a,b){var c=a._localRoot||this._root,d=a._clip.tracks,e=d.length,f=a._propertyBindings;a=a._interpolants;var g=c.uuid,h=this._bindingsByRootAndName,
k=h[g];void 0===k&&(k={},h[g]=k);for(h=0;h!==e;++h){var m=d[h],n=m.name,p=k[n];if(void 0===p){p=f[h];if(void 0!==p){null===p._cacheIndex&&(++p.referenceCount,this._addInactiveBinding(p,g,n));continue}p=new Lg(ya.create(c,n,b&&b._propertyBindings[h].binding.parsedPath),m.ValueTypeName,m.getValueSize());++p.referenceCount;this._addInactiveBinding(p,g,n)}f[h]=p;a[h].resultBuffer=p.buffer}},_activateAction:function(a){if(!this._isActiveAction(a)){if(null===a._cacheIndex){var b=(a._localRoot||this._root).uuid,
c=a._clip.uuid,d=this._actionsByClip[c];this._bindAction(a,d&&d.knownActions[0]);this._addInactiveAction(a,c,b)}b=a._propertyBindings;c=0;for(d=b.length;c!==d;++c){var e=b[c];0===e.useCount++&&(this._lendBinding(e),e.saveOriginalState())}this._lendAction(a)}},_deactivateAction:function(a){if(this._isActiveAction(a)){for(var b=a._propertyBindings,c=0,d=b.length;c!==d;++c){var e=b[c];0===--e.useCount&&(e.restoreOriginalState(),this._takeBackBinding(e))}this._takeBackAction(a)}},_initMemoryManager:function(){this._actions=
[];this._nActiveActions=0;this._actionsByClip={};this._bindings=[];this._nActiveBindings=0;this._bindingsByRootAndName={};this._controlInterpolants=[];this._nActiveControlInterpolants=0;var a=this;this.stats={actions:{get total(){return a._actions.length},get inUse(){return a._nActiveActions}},bindings:{get total(){return a._bindings.length},get inUse(){return a._nActiveBindings}},controlInterpolants:{get total(){return a._controlInterpolants.length},get inUse(){return a._nActiveControlInterpolants}}}},
_isActiveAction:function(a){a=a._cacheIndex;return null!==a&&a<this._nActiveActions},_addInactiveAction:function(a,b,c){var d=this._actions,e=this._actionsByClip,f=e[b];void 0===f?(f={knownActions:[a],actionByRoot:{}},a._byClipCacheIndex=0,e[b]=f):(b=f.knownActions,a._byClipCacheIndex=b.length,b.push(a));a._cacheIndex=d.length;d.push(a);f.actionByRoot[c]=a},_removeInactiveAction:function(a){var b=this._actions,c=b[b.length-1],d=a._cacheIndex;c._cacheIndex=d;b[d]=c;b.pop();a._cacheIndex=null;b=a._clip.uuid;
c=this._actionsByClip;d=c[b];var e=d.knownActions,f=e[e.length-1],g=a._byClipCacheIndex;f._byClipCacheIndex=g;e[g]=f;e.pop();a._byClipCacheIndex=null;delete d.actionByRoot[(a._localRoot||this._root).uuid];0===e.length&&delete c[b];this._removeInactiveBindingsForAction(a)},_removeInactiveBindingsForAction:function(a){a=a._propertyBindings;for(var b=0,c=a.length;b!==c;++b){var d=a[b];0===--d.referenceCount&&this._removeInactiveBinding(d)}},_lendAction:function(a){var b=this._actions,c=a._cacheIndex,
d=this._nActiveActions++,e=b[d];a._cacheIndex=d;b[d]=a;e._cacheIndex=c;b[c]=e},_takeBackAction:function(a){var b=this._actions,c=a._cacheIndex,d=--this._nActiveActions,e=b[d];a._cacheIndex=d;b[d]=a;e._cacheIndex=c;b[c]=e},_addInactiveBinding:function(a,b,c){var d=this._bindingsByRootAndName,e=d[b],f=this._bindings;void 0===e&&(e={},d[b]=e);e[c]=a;a._cacheIndex=f.length;f.push(a)},_removeInactiveBinding:function(a){var b=this._bindings,c=a.binding,d=c.rootNode.uuid;c=c.path;var e=this._bindingsByRootAndName,
f=e[d],g=b[b.length-1];a=a._cacheIndex;g._cacheIndex=a;b[a]=g;b.pop();delete f[c];0===Object.keys(f).length&&delete e[d]},_lendBinding:function(a){var b=this._bindings,c=a._cacheIndex,d=this._nActiveBindings++,e=b[d];a._cacheIndex=d;b[d]=a;e._cacheIndex=c;b[c]=e},_takeBackBinding:function(a){var b=this._bindings,c=a._cacheIndex,d=--this._nActiveBindings,e=b[d];a._cacheIndex=d;b[d]=a;e._cacheIndex=c;b[c]=e},_lendControlInterpolant:function(){var a=this._controlInterpolants,b=this._nActiveControlInterpolants++,
c=a[b];void 0===c&&(c=new ne(new Float32Array(2),new Float32Array(2),1,this._controlInterpolantsResultBuffer),c.__cacheIndex=b,a[b]=c);return c},_takeBackControlInterpolant:function(a){var b=this._controlInterpolants,c=a.__cacheIndex,d=--this._nActiveControlInterpolants,e=b[d];a.__cacheIndex=d;b[d]=a;e.__cacheIndex=c;b[c]=e},_controlInterpolantsResultBuffer:new Float32Array(1),clipAction:function(a,b){var c=b||this._root,d=c.uuid;c=&quot;string&quot;===typeof a?Qa.findByName(c,a):a;a=null!==c?c.uuid:a;var e=
this._actionsByClip[a],f=null;if(void 0!==e){f=e.actionByRoot[d];if(void 0!==f)return f;f=e.knownActions[0];null===c&&(c=f._clip)}if(null===c)return null;b=new pi(this,c,b);this._bindAction(b,f);this._addInactiveAction(b,a,d);return b},existingAction:function(a,b){var c=b||this._root;b=c.uuid;c=&quot;string&quot;===typeof a?Qa.findByName(c,a):a;a=this._actionsByClip[c?c.uuid:a];return void 0!==a?a.actionByRoot[b]||null:null},stopAllAction:function(){for(var a=this._actions,b=this._nActiveActions,c=this._bindings,
d=this._nActiveBindings,e=this._nActiveBindings=this._nActiveActions=0;e!==b;++e)a[e].reset();for(e=0;e!==d;++e)c[e].useCount=0;return this},update:function(a){a*=this.timeScale;for(var b=this._actions,c=this._nActiveActions,d=this.time+=a,e=Math.sign(a),f=this._accuIndex^=1,g=0;g!==c;++g)b[g]._update(d,a,e,f);a=this._bindings;b=this._nActiveBindings;for(g=0;g!==b;++g)a[g].apply(f);return this},setTime:function(a){for(var b=this.time=0;b<this._actions.length;b++)this._actions[b].time=0;return this.update(a)},
getRoot:function(){return this._root},uncacheClip:function(a){var b=this._actions;a=a.uuid;var c=this._actionsByClip,d=c[a];if(void 0!==d){d=d.knownActions;for(var e=0,f=d.length;e!==f;++e){var g=d[e];this._deactivateAction(g);var h=g._cacheIndex,k=b[b.length-1];g._cacheIndex=null;g._byClipCacheIndex=null;k._cacheIndex=h;b[h]=k;b.pop();this._removeInactiveBindingsForAction(g)}delete c[a]}},uncacheRoot:function(a){a=a.uuid;var b=this._actionsByClip;for(d in b){var c=b[d].actionByRoot[a];void 0!==c&&
(this._deactivateAction(c),this._removeInactiveAction(c))}var d=this._bindingsByRootAndName[a];if(void 0!==d)for(var e in d)a=d[e],a.restoreOriginalState(),this._removeInactiveBinding(a)},uncacheAction:function(a,b){a=this.existingAction(a,b);null!==a&&(this._deactivateAction(a),this._removeInactiveAction(a))}});vf.prototype.clone=function(){return new vf(void 0===this.value.clone?this.value:this.value.clone())};Ng.prototype=Object.assign(Object.create(rb.prototype),{constructor:Ng,isInstancedInterleavedBuffer:!0,
copy:function(a){rb.prototype.copy.call(this,a);this.meshPerAttribute=a.meshPerAttribute;return this}});Object.assign(Og.prototype,{set:function(a,b){this.ray.set(a,b)},setFromCamera:function(a,b){b&&b.isPerspectiveCamera?(this.ray.origin.setFromMatrixPosition(b.matrixWorld),this.ray.direction.set(a.x,a.y,.5).unproject(b).sub(this.ray.origin).normalize(),this.camera=b):b&&b.isOrthographicCamera?(this.ray.origin.set(a.x,a.y,(b.near+b.far)/(b.near-b.far)).unproject(b),this.ray.direction.set(0,0,-1).transformDirection(b.matrixWorld),
this.camera=b):console.error(&quot;THREE.Raycaster: Unsupported camera type.&quot;)},intersectObject:function(a,b,c){c=c||[];Pg(a,this,c,b);c.sort(qi);return c},intersectObjects:function(a,b,c){c=c||[];if(!1===Array.isArray(a))return console.warn(&quot;THREE.Raycaster.intersectObjects: objects is not an Array.&quot;),c;for(var d=0,e=a.length;d<e;d++)Pg(a[d],this,c,b);c.sort(qi);return c}});Object.assign(ri.prototype,{set:function(a,b,c){this.radius=a;this.phi=b;this.theta=c;return this},clone:function(){return(new this.constructor).copy(this)},
copy:function(a){this.radius=a.radius;this.phi=a.phi;this.theta=a.theta;return this},makeSafe:function(){this.phi=Math.max(1E-6,Math.min(Math.PI-1E-6,this.phi));return this},setFromVector3:function(a){return this.setFromCartesianCoords(a.x,a.y,a.z)},setFromCartesianCoords:function(a,b,c){this.radius=Math.sqrt(a*a+b*b+c*c);0===this.radius?this.phi=this.theta=0:(this.theta=Math.atan2(a,c),this.phi=Math.acos(L.clamp(b/this.radius,-1,1)));return this}});Object.assign(si.prototype,{set:function(a,b,c){this.radius=
a;this.theta=b;this.y=c;return this},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.radius=a.radius;this.theta=a.theta;this.y=a.y;return this},setFromVector3:function(a){return this.setFromCartesianCoords(a.x,a.y,a.z)},setFromCartesianCoords:function(a,b,c){this.radius=Math.sqrt(a*a+c*c);this.theta=Math.atan2(a,c);this.y=b;return this}});var gj=new t;Object.assign(Qg.prototype,{set:function(a,b){this.min.copy(a);this.max.copy(b);return this},setFromPoints:function(a){this.makeEmpty();
for(var b=0,c=a.length;b<c;b++)this.expandByPoint(a[b]);return this},setFromCenterAndSize:function(a,b){b=gj.copy(b).multiplyScalar(.5);this.min.copy(a).sub(b);this.max.copy(a).add(b);return this},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.min.copy(a.min);this.max.copy(a.max);return this},makeEmpty:function(){this.min.x=this.min.y=Infinity;this.max.x=this.max.y=-Infinity;return this},isEmpty:function(){return this.max.x<this.min.x||this.max.y<this.min.y},getCenter:function(a){void 0===
a&&(console.warn(&quot;THREE.Box2: .getCenter() target is now required&quot;),a=new t);return this.isEmpty()?a.set(0,0):a.addVectors(this.min,this.max).multiplyScalar(.5)},getSize:function(a){void 0===a&&(console.warn(&quot;THREE.Box2: .getSize() target is now required&quot;),a=new t);return this.isEmpty()?a.set(0,0):a.subVectors(this.max,this.min)},expandByPoint:function(a){this.min.min(a);this.max.max(a);return this},expandByVector:function(a){this.min.sub(a);this.max.add(a);return this},expandByScalar:function(a){this.min.addScalar(-a);
this.max.addScalar(a);return this},containsPoint:function(a){return a.x<this.min.x||a.x>this.max.x||a.y<this.min.y||a.y>this.max.y?!1:!0},containsBox:function(a){return this.min.x<=a.min.x&&a.max.x<=this.max.x&&this.min.y<=a.min.y&&a.max.y<=this.max.y},getParameter:function(a,b){void 0===b&&(console.warn(&quot;THREE.Box2: .getParameter() target is now required&quot;),b=new t);return b.set((a.x-this.min.x)/(this.max.x-this.min.x),(a.y-this.min.y)/(this.max.y-this.min.y))},intersectsBox:function(a){return a.max.x<
this.min.x||a.min.x>this.max.x||a.max.y<this.min.y||a.min.y>this.max.y?!1:!0},clampPoint:function(a,b){void 0===b&&(console.warn(&quot;THREE.Box2: .clampPoint() target is now required&quot;),b=new t);return b.copy(a).clamp(this.min,this.max)},distanceToPoint:function(a){return gj.copy(a).clamp(this.min,this.max).sub(a).length()},intersect:function(a){this.min.max(a.min);this.max.min(a.max);return this},union:function(a){this.min.min(a.min);this.max.max(a.max);return this},translate:function(a){this.min.add(a);
this.max.add(a);return this},equals:function(a){return a.min.equals(this.min)&&a.max.equals(this.max)}});var hj=new n,Vf=new n;Object.assign(Rg.prototype,{set:function(a,b){this.start.copy(a);this.end.copy(b);return this},clone:function(){return(new this.constructor).copy(this)},copy:function(a){this.start.copy(a.start);this.end.copy(a.end);return this},getCenter:function(a){void 0===a&&(console.warn(&quot;THREE.Line3: .getCenter() target is now required&quot;),a=new n);return a.addVectors(this.start,this.end).multiplyScalar(.5)},
delta:function(a){void 0===a&&(console.warn(&quot;THREE.Line3: .delta() target is now required&quot;),a=new n);return a.subVectors(this.end,this.start)},distanceSq:function(){return this.start.distanceToSquared(this.end)},distance:function(){return this.start.distanceTo(this.end)},at:function(a,b){void 0===b&&(console.warn(&quot;THREE.Line3: .at() target is now required&quot;),b=new n);return this.delta(b).multiplyScalar(a).add(this.start)},closestPointToPointParameter:function(a,b){hj.subVectors(a,this.start);Vf.subVectors(this.end,
this.start);a=Vf.dot(Vf);a=Vf.dot(hj)/a;b&&(a=L.clamp(a,0,1));return a},closestPointToPoint:function(a,b,c){a=this.closestPointToPointParameter(a,b);void 0===c&&(console.warn(&quot;THREE.Line3: .closestPointToPoint() target is now required&quot;),c=new n);return this.delta(c).multiplyScalar(a).add(this.start)},applyMatrix4:function(a){this.start.applyMatrix4(a);this.end.applyMatrix4(a);return this},equals:function(a){return a.start.equals(this.start)&&a.end.equals(this.end)}});re.prototype=Object.create(F.prototype);
re.prototype.constructor=re;re.prototype.isImmediateRenderObject=!0;var ij=new n;jd.prototype=Object.create(F.prototype);jd.prototype.constructor=jd;jd.prototype.dispose=function(){this.cone.geometry.dispose();this.cone.material.dispose()};jd.prototype.update=function(){this.light.updateMatrixWorld();var a=this.light.distance?this.light.distance:1E3,b=a*Math.tan(this.light.angle);this.cone.scale.set(b,b,a);ij.setFromMatrixPosition(this.light.target.matrixWorld);this.cone.lookAt(ij);void 0!==this.color?
this.cone.material.color.set(this.color):this.cone.material.color.copy(this.light.color)};var Qb=new n,Wf=new P,vh=new P;rc.prototype=Object.create(ma.prototype);rc.prototype.constructor=rc;rc.prototype.isSkeletonHelper=!0;rc.prototype.updateMatrixWorld=function(a){var b=this.bones,c=this.geometry,d=c.getAttribute(&quot;position&quot;);vh.getInverse(this.root.matrixWorld);for(var e=0,f=0;e<b.length;e++){var g=b[e];g.parent&&g.parent.isBone&&(Wf.multiplyMatrices(vh,g.matrixWorld),Qb.setFromMatrixPosition(Wf),
d.setXYZ(f,Qb.x,Qb.y,Qb.z),Wf.multiplyMatrices(vh,g.parent.matrixWorld),Qb.setFromMatrixPosition(Wf),d.setXYZ(f+1,Qb.x,Qb.y,Qb.z),f+=2)}c.getAttribute(&quot;position&quot;).needsUpdate=!0;F.prototype.updateMatrixWorld.call(this,a)};kd.prototype=Object.create(S.prototype);kd.prototype.constructor=kd;kd.prototype.dispose=function(){this.geometry.dispose();this.material.dispose()};kd.prototype.update=function(){void 0!==this.color?this.material.color.set(this.color):this.material.color.copy(this.light.color)};
var nl=new n,jj=new A,kj=new A;ld.prototype=Object.create(F.prototype);ld.prototype.constructor=ld;ld.prototype.dispose=function(){this.children[0].geometry.dispose();this.children[0].material.dispose()};ld.prototype.update=function(){var a=this.children[0];if(void 0!==this.color)this.material.color.set(this.color);else{var b=a.geometry.getAttribute(&quot;color&quot;);jj.copy(this.light.color);kj.copy(this.light.groundColor);for(var c=0,d=b.count;c<d;c++){var e=c<d/2?jj:kj;b.setXYZ(c,e.r,e.g,e.b)}b.needsUpdate=
!0}a.lookAt(nl.setFromMatrixPosition(this.light.matrixWorld).negate())};wf.prototype=Object.assign(Object.create(ma.prototype),{constructor:wf,copy:function(a){ma.prototype.copy.call(this,a);this.geometry.copy(a.geometry);this.material.copy(a.material);return this},clone:function(){return(new this.constructor).copy(this)}});xf.prototype=Object.create(ma.prototype);xf.prototype.constructor=xf;var lj=new n,Xf=new n,mj=new n;md.prototype=Object.create(F.prototype);md.prototype.constructor=md;md.prototype.dispose=
function(){this.lightPlane.geometry.dispose();this.lightPlane.material.dispose();this.targetLine.geometry.dispose();this.targetLine.material.dispose()};md.prototype.update=function(){lj.setFromMatrixPosition(this.light.matrixWorld);Xf.setFromMatrixPosition(this.light.target.matrixWorld);mj.subVectors(Xf,lj);this.lightPlane.lookAt(Xf);void 0!==this.color?(this.lightPlane.material.color.set(this.color),this.targetLine.material.color.set(this.color)):(this.lightPlane.material.color.copy(this.light.color),
this.targetLine.material.color.copy(this.light.color));this.targetLine.lookAt(Xf);this.targetLine.scale.z=mj.length()};var yf=new n,ja=new db;se.prototype=Object.create(ma.prototype);se.prototype.constructor=se;se.prototype.update=function(){var a=this.geometry,b=this.pointMap;ja.projectionMatrixInverse.copy(this.camera.projectionMatrixInverse);na(&quot;c&quot;,b,a,ja,0,0,-1);na(&quot;t&quot;,b,a,ja,0,0,1);na(&quot;n1&quot;,b,a,ja,-1,-1,-1);na(&quot;n2&quot;,b,a,ja,1,-1,-1);na(&quot;n3&quot;,b,a,ja,-1,1,-1);na(&quot;n4&quot;,b,a,ja,1,1,-1);na(&quot;f1&quot;,b,a,ja,
-1,-1,1);na(&quot;f2&quot;,b,a,ja,1,-1,1);na(&quot;f3&quot;,b,a,ja,-1,1,1);na(&quot;f4&quot;,b,a,ja,1,1,1);na(&quot;u1&quot;,b,a,ja,.7,1.1,-1);na(&quot;u2&quot;,b,a,ja,-.7,1.1,-1);na(&quot;u3&quot;,b,a,ja,0,2,-1);na(&quot;cf1&quot;,b,a,ja,-1,0,1);na(&quot;cf2&quot;,b,a,ja,1,0,1);na(&quot;cf3&quot;,b,a,ja,0,-1,1);na(&quot;cf4&quot;,b,a,ja,0,1,1);na(&quot;cn1&quot;,b,a,ja,-1,0,-1);na(&quot;cn2&quot;,b,a,ja,1,0,-1);na(&quot;cn3&quot;,b,a,ja,0,-1,-1);na(&quot;cn4&quot;,b,a,ja,0,1,-1);a.getAttribute(&quot;position&quot;).needsUpdate=!0};var Yf=new Sa;wb.prototype=Object.create(ma.prototype);wb.prototype.constructor=wb;wb.prototype.update=function(a){void 0!==
a&&console.warn(&quot;THREE.BoxHelper: .update() has no longer arguments.&quot;);void 0!==this.object&&Yf.setFromObject(this.object);if(!Yf.isEmpty()){a=Yf.min;var b=Yf.max,c=this.geometry.attributes.position,d=c.array;d[0]=b.x;d[1]=b.y;d[2]=b.z;d[3]=a.x;d[4]=b.y;d[5]=b.z;d[6]=a.x;d[7]=a.y;d[8]=b.z;d[9]=b.x;d[10]=a.y;d[11]=b.z;d[12]=b.x;d[13]=b.y;d[14]=a.z;d[15]=a.x;d[16]=b.y;d[17]=a.z;d[18]=a.x;d[19]=a.y;d[20]=a.z;d[21]=b.x;d[22]=a.y;d[23]=a.z;c.needsUpdate=!0;this.geometry.computeBoundingSphere()}};wb.prototype.setFromObject=
function(a){this.object=a;this.update();return this};wb.prototype.copy=function(a){ma.prototype.copy.call(this,a);this.object=a.object;return this};wb.prototype.clone=function(){return(new this.constructor).copy(this)};te.prototype=Object.create(ma.prototype);te.prototype.constructor=te;te.prototype.updateMatrixWorld=function(a){var b=this.box;b.isEmpty()||(b.getCenter(this.position),b.getSize(this.scale),this.scale.multiplyScalar(.5),F.prototype.updateMatrixWorld.call(this,a))};ue.prototype=Object.create(Ka.prototype);
ue.prototype.constructor=ue;ue.prototype.updateMatrixWorld=function(a){var b=-this.plane.constant;1E-8>Math.abs(b)&&(b=1E-8);this.scale.set(.5*this.size,.5*this.size,b);this.children[0].material.side=0>b?1:0;this.lookAt(this.plane.normal);F.prototype.updateMatrixWorld.call(this,a)};var nj=new n,zf,Sg;xb.prototype=Object.create(F.prototype);xb.prototype.constructor=xb;xb.prototype.setDirection=function(a){.99999<a.y?this.quaternion.set(0,0,0,1):-.99999>a.y?this.quaternion.set(1,0,0,0):(nj.set(a.z,
0,-a.x).normalize(),this.quaternion.setFromAxisAngle(nj,Math.acos(a.y)))};xb.prototype.setLength=function(a,b,c){void 0===b&&(b=.2*a);void 0===c&&(c=.2*b);this.line.scale.set(1,Math.max(1E-4,a-b),1);this.line.updateMatrix();this.cone.scale.set(c,b,c);this.cone.position.y=a;this.cone.updateMatrix()};xb.prototype.setColor=function(a){this.line.material.color.set(a);this.cone.material.color.set(a)};xb.prototype.copy=function(a){F.prototype.copy.call(this,a,!1);this.line.copy(a.line);this.cone.copy(a.cone);
return this};xb.prototype.clone=function(){return(new this.constructor).copy(this)};ve.prototype=Object.create(ma.prototype);ve.prototype.constructor=ve;var kb=Math.pow(2,8),oj=[.125,.215,.35,.446,.526,.582],yi=5+oj.length,lb={3E3:0,3001:1,3002:2,3004:3,3005:4,3006:5,3007:6},Wg=new hd,Af=function(a){var b=new Float32Array(a),c=new n(0,1,0);a=new ub({defines:{n:a},uniforms:{envMap:{value:null},samples:{value:1},weights:{value:b},latitudinal:{value:!1},dTheta:{value:0},mipInt:{value:0},poleAxis:{value:c},
inputEncoding:{value:lb[3E3]},outputEncoding:{value:lb[3E3]}},vertexShader:Yg(),fragmentShader:&quot;\nprecision mediump float;\nprecision mediump int;\nvarying vec3 vOutputDirection;\nuniform sampler2D envMap;\nuniform int samples;\nuniform float weights[n];\nuniform bool latitudinal;\nuniform float dTheta;\nuniform float mipInt;\nuniform vec3 poleAxis;\n\n&quot;+Zg()+&quot;\n\n#define ENVMAP_TYPE_CUBE_UV\n#include <cube_uv_reflection_fragment>\n\nvec3 getSample(float theta, vec3 axis) {\n\tfloat cosTheta = cos(theta);\n\t// Rodrigues' axis-angle rotation\n\tvec3 sampleDirection = vOutputDirection * cosTheta\n\t\t+ cross(axis, vOutputDirection) * sin(theta)\n\t\t+ axis * dot(axis, vOutputDirection) * (1.0 - cosTheta);\n\treturn bilinearCubeUV(envMap, sampleDirection, mipInt);\n}\n\nvoid main() {\n\tvec3 axis = latitudinal ? poleAxis : cross(poleAxis, vOutputDirection);\n\tif (all(equal(axis, vec3(0.0))))\n\t\taxis = vec3(vOutputDirection.z, 0.0, - vOutputDirection.x);\n\taxis = normalize(axis);\n\tgl_FragColor = vec4(0.0);\n\tgl_FragColor.rgb += weights[0] * getSample(0.0, axis);\n\tfor (int i = 1; i < n; i++) {\n\t\tif (i >= samples)\n\t\t\tbreak;\n\t\tfloat theta = dTheta * float(i);\n\t\tgl_FragColor.rgb += weights[i] * getSample(-1.0 * theta, axis);\n\t\tgl_FragColor.rgb += weights[i] * getSample(theta, axis);\n\t}\n\tgl_FragColor = linearToOutputTexel(gl_FragColor);\n}\n\t\t&quot;,
blending:0,depthTest:!1,depthWrite:!1});a.type=&quot;SphericalGaussianBlur&quot;;return a}(20),Rb=null,Sb=null,wh=function(){for(var a=[],b=[],c=[],d=8,e=0;e<yi;e++){var f=Math.pow(2,d);b.push(f);var g=1/f;4<e?g=oj[e-8+4-1]:0==e&&(g=0);c.push(g);g=1/(f-1);f=-g/2;g=1+g/2;var h=[f,f,g,f,g,g,f,f,g,g,f,g];f=new Float32Array(108);g=new Float32Array(72);for(var k=new Float32Array(36),m=0;6>m;m++){var n=m%3*2/3-1,p=2<m?0:-1;f.set([n,p,0,n+2/3,p,0,n+2/3,p+1,0,n,p,0,n+2/3,p+1,0,n,p+1,0],18*m);g.set(h,12*m);k.set([m,
m,m,m,m,m],6*m)}h=new C;h.setAttribute(&quot;position&quot;,new M(f,3));h.setAttribute(&quot;uv&quot;,new M(g,2));h.setAttribute(&quot;faceIndex&quot;,new M(k,1));a.push(h);4<d&&d--}return{_lodPlanes:a,_sizeLods:b,_sigmas:c}}(),we=wh._lodPlanes,Ci=wh._sizeLods,Cf=wh._sigmas,Bf=null,T=null,Vg=null,Ac=(1+Math.sqrt(5))/2,Ad=1/Ac,Ai=[new n(1,1,1),new n(-1,1,1),new n(1,1,-1),new n(-1,1,-1),new n(0,Ac,Ad),new n(0,Ac,-Ad),new n(Ad,0,Ac),new n(-Ad,0,Ac),new n(Ac,Ad,0),new n(-Ac,Ad,0)];Tg.prototype={constructor:Tg,fromScene:function(a,
b,c,d){void 0===b&&(b=0);void 0===c&&(c=.1);void 0===d&&(d=100);Vg=T.getRenderTarget();var e=ui();c=new aa(90,1,c,d);d=[1,1,1,1,-1,1];var f=[1,1,-1,-1,-1,1],g=T.outputEncoding,h=T.toneMapping,k=T.toneMappingExposure,m=T.getClearColor(),n=T.getClearAlpha();T.toneMapping=1;T.toneMappingExposure=1;T.outputEncoding=3E3;a.scale.z*=-1;var p=a.background;if(p&&p.isColor){p.convertSRGBToLinear();var t=Math.min(Math.max(Math.ceil(Math.log2(Math.max(p.r,p.g,p.b))),-128),127);p=p.multiplyScalar(Math.pow(2,-t));
T.setClearColor(p,(t+128)/255);a.background=null}for(p=0;6>p;p++)t=p%3,0==t?(c.up.set(0,d[p],0),c.lookAt(f[p],0,0)):1==t?(c.up.set(0,0,d[p]),c.lookAt(0,f[p],0)):(c.up.set(0,d[p],0),c.lookAt(0,0,f[p])),Xg(e,t*kb,2<p?kb:0,kb,kb),T.setRenderTarget(e),T.render(a,c);T.toneMapping=h;T.toneMappingExposure=k;T.outputEncoding=g;T.setClearColor(m,n);a.scale.z*=-1;0<b&&zi(e,0,0,b);xi(e);wi(e);return e},fromEquirectangular:function(a){a.magFilter=1003;a.minFilter=1003;a.generateMipmaps=!1;return this.fromCubemap(a)},
fromCubemap:function(a){Vg=T.getRenderTarget();var b=ui(a),c=new ob;a.isCubeTexture?null==Sb&&(Sb=Ei()):null==Rb&&(Rb=Di());var d=a.isCubeTexture?Sb:Rb;c.add(new S(we[0],d));d=d.uniforms;d.envMap.value=a;a.isCubeTexture||d.texelSize.value.set(1/a.image.width,1/a.image.height);d.inputEncoding.value=lb[a.encoding];d.outputEncoding.value=lb[a.encoding];Xg(b,0,0,3*kb,2*kb);T.setRenderTarget(b);T.render(c,Wg);xi(b);wi(b);return b},compileCubemapShader:function(){null==Sb&&(Sb=Ei(),Ug(Sb))},compileEquirectangularShader:function(){null==
Rb&&(Rb=Di(),Ug(Rb))},dispose:function(){Af.dispose();null!=Sb&&Sb.dispose();null!=Rb&&Rb.dispose();for(var a=0;a<we.length;a++)we[a].dispose()}};G.create=function(a,b){console.log(&quot;THREE.Curve.create() has been deprecated&quot;);a.prototype=Object.create(G.prototype);a.prototype.constructor=a;a.prototype.getPoint=b;return a};Object.assign(vb.prototype,{createPointsGeometry:function(a){console.warn(&quot;THREE.CurvePath: .createPointsGeometry() has been removed. Use new THREE.Geometry().setFromPoints( points ) instead.&quot;);
a=this.getPoints(a);return this.createGeometry(a)},createSpacedPointsGeometry:function(a){console.warn(&quot;THREE.CurvePath: .createSpacedPointsGeometry() has been removed. Use new THREE.Geometry().setFromPoints( points ) instead.&quot;);a=this.getSpacedPoints(a);return this.createGeometry(a)},createGeometry:function(a){console.warn(&quot;THREE.CurvePath: .createGeometry() has been removed. Use new THREE.Geometry().setFromPoints( points ) instead.&quot;);for(var b=new N,c=0,d=a.length;c<d;c++){var e=a[c];b.vertices.push(new n(e.x,
e.y,e.z||0))}return b}});Object.assign($a.prototype,{fromPoints:function(a){console.warn(&quot;THREE.Path: .fromPoints() has been renamed to .setFromPoints().&quot;);return this.setFromPoints(a)}});Fi.prototype=Object.create(pa.prototype);Gi.prototype=Object.create(pa.prototype);$g.prototype=Object.create(pa.prototype);Object.assign($g.prototype,{initFromArray:function(){console.error(&quot;THREE.Spline: .initFromArray() has been removed.&quot;)},getControlPointsArray:function(){console.error(&quot;THREE.Spline: .getControlPointsArray() has been removed.&quot;)},
reparametrizeByArcLength:function(){console.error(&quot;THREE.Spline: .reparametrizeByArcLength() has been removed.&quot;)}});wf.prototype.setColors=function(){console.error(&quot;THREE.GridHelper: setColors() has been deprecated, pass them in the constructor instead.&quot;)};rc.prototype.update=function(){console.error(&quot;THREE.SkeletonHelper: update() no longer needs to be called.&quot;)};Object.assign(W.prototype,{extractUrlBase:function(a){console.warn(&quot;THREE.Loader: .extractUrlBase() has been deprecated. Use THREE.LoaderUtils.extractUrlBase() instead.&quot;);
return th.extractUrlBase(a)}});W.Handlers={add:function(){console.error(&quot;THREE.Loader: Handlers.add() has been removed. Use LoadingManager.addHandler() instead.&quot;)},get:function(){console.error(&quot;THREE.Loader: Handlers.get() has been removed. Use LoadingManager.getHandler() instead.&quot;)}};Object.assign(sf.prototype,{setTexturePath:function(a){console.warn(&quot;THREE.ObjectLoader: .setTexturePath() has been renamed to .setResourcePath().&quot;);return this.setResourcePath(a)}});Object.assign(Qg.prototype,{center:function(a){console.warn(&quot;THREE.Box2: .center() has been renamed to .getCenter().&quot;);
return this.getCenter(a)},empty:function(){console.warn(&quot;THREE.Box2: .empty() has been renamed to .isEmpty().&quot;);return this.isEmpty()},isIntersectionBox:function(a){console.warn(&quot;THREE.Box2: .isIntersectionBox() has been renamed to .intersectsBox().&quot;);return this.intersectsBox(a)},size:function(a){console.warn(&quot;THREE.Box2: .size() has been renamed to .getSize().&quot;);return this.getSize(a)}});Object.assign(Sa.prototype,{center:function(a){console.warn(&quot;THREE.Box3: .center() has been renamed to .getCenter().&quot;);
return this.getCenter(a)},empty:function(){console.warn(&quot;THREE.Box3: .empty() has been renamed to .isEmpty().&quot;);return this.isEmpty()},isIntersectionBox:function(a){console.warn(&quot;THREE.Box3: .isIntersectionBox() has been renamed to .intersectsBox().&quot;);return this.intersectsBox(a)},isIntersectionSphere:function(a){console.warn(&quot;THREE.Box3: .isIntersectionSphere() has been renamed to .intersectsSphere().&quot;);return this.intersectsSphere(a)},size:function(a){console.warn(&quot;THREE.Box3: .size() has been renamed to .getSize().&quot;);
return this.getSize(a)}});Hc.prototype.setFromMatrix=function(a){console.warn(&quot;THREE.Frustum: .setFromMatrix() has been renamed to .setFromProjectionMatrix().&quot;);return this.setFromProjectionMatrix(a)};Rg.prototype.center=function(a){console.warn(&quot;THREE.Line3: .center() has been renamed to .getCenter().&quot;);return this.getCenter(a)};Object.assign(L,{random16:function(){console.warn(&quot;THREE.Math: .random16() has been deprecated. Use Math.random() instead.&quot;);return Math.random()},nearestPowerOfTwo:function(a){console.warn(&quot;THREE.Math: .nearestPowerOfTwo() has been renamed to .floorPowerOfTwo().&quot;);
return L.floorPowerOfTwo(a)},nextPowerOfTwo:function(a){console.warn(&quot;THREE.Math: .nextPowerOfTwo() has been renamed to .ceilPowerOfTwo().&quot;);return L.ceilPowerOfTwo(a)}});Object.assign(wa.prototype,{flattenToArrayOffset:function(a,b){console.warn(&quot;THREE.Matrix3: .flattenToArrayOffset() has been deprecated. Use .toArray() instead.&quot;);return this.toArray(a,b)},multiplyVector3:function(a){console.warn(&quot;THREE.Matrix3: .multiplyVector3() has been removed. Use vector.applyMatrix3( matrix ) instead.&quot;);return a.applyMatrix3(this)},
multiplyVector3Array:function(){console.error(&quot;THREE.Matrix3: .multiplyVector3Array() has been removed.&quot;)},applyToBufferAttribute:function(a){console.warn(&quot;THREE.Matrix3: .applyToBufferAttribute() has been removed. Use attribute.applyMatrix3( matrix ) instead.&quot;);return a.applyMatrix3(this)},applyToVector3Array:function(){console.error(&quot;THREE.Matrix3: .applyToVector3Array() has been removed.&quot;)}});Object.assign(P.prototype,{extractPosition:function(a){console.warn(&quot;THREE.Matrix4: .extractPosition() has been renamed to .copyPosition().&quot;);
return this.copyPosition(a)},flattenToArrayOffset:function(a,b){console.warn(&quot;THREE.Matrix4: .flattenToArrayOffset() has been deprecated. Use .toArray() instead.&quot;);return this.toArray(a,b)},getPosition:function(){console.warn(&quot;THREE.Matrix4: .getPosition() has been removed. Use Vector3.setFromMatrixPosition( matrix ) instead.&quot;);return(new n).setFromMatrixColumn(this,3)},setRotationFromQuaternion:function(a){console.warn(&quot;THREE.Matrix4: .setRotationFromQuaternion() has been renamed to .makeRotationFromQuaternion().&quot;);
return this.makeRotationFromQuaternion(a)},multiplyToArray:function(){console.warn(&quot;THREE.Matrix4: .multiplyToArray() has been removed.&quot;)},multiplyVector3:function(a){console.warn(&quot;THREE.Matrix4: .multiplyVector3() has been removed. Use vector.applyMatrix4( matrix ) instead.&quot;);return a.applyMatrix4(this)},multiplyVector4:function(a){console.warn(&quot;THREE.Matrix4: .multiplyVector4() has been removed. Use vector.applyMatrix4( matrix ) instead.&quot;);return a.applyMatrix4(this)},multiplyVector3Array:function(){console.error(&quot;THREE.Matrix4: .multiplyVector3Array() has been removed.&quot;)},
rotateAxis:function(a){console.warn(&quot;THREE.Matrix4: .rotateAxis() has been removed. Use Vector3.transformDirection( matrix ) instead.&quot;);a.transformDirection(this)},crossVector:function(a){console.warn(&quot;THREE.Matrix4: .crossVector() has been removed. Use vector.applyMatrix4( matrix ) instead.&quot;);return a.applyMatrix4(this)},translate:function(){console.error(&quot;THREE.Matrix4: .translate() has been removed.&quot;)},rotateX:function(){console.error(&quot;THREE.Matrix4: .rotateX() has been removed.&quot;)},rotateY:function(){console.error(&quot;THREE.Matrix4: .rotateY() has been removed.&quot;)},
rotateZ:function(){console.error(&quot;THREE.Matrix4: .rotateZ() has been removed.&quot;)},rotateByAxis:function(){console.error(&quot;THREE.Matrix4: .rotateByAxis() has been removed.&quot;)},applyToBufferAttribute:function(a){console.warn(&quot;THREE.Matrix4: .applyToBufferAttribute() has been removed. Use attribute.applyMatrix4( matrix ) instead.&quot;);return a.applyMatrix4(this)},applyToVector3Array:function(){console.error(&quot;THREE.Matrix4: .applyToVector3Array() has been removed.&quot;)},makeFrustum:function(a,b,c,d,e,f){console.warn(&quot;THREE.Matrix4: .makeFrustum() has been removed. Use .makePerspective( left, right, top, bottom, near, far ) instead.&quot;);
return this.makePerspective(a,b,d,c,e,f)}});Ta.prototype.isIntersectionLine=function(a){console.warn(&quot;THREE.Plane: .isIntersectionLine() has been renamed to .intersectsLine().&quot;);return this.intersectsLine(a)};Aa.prototype.multiplyVector3=function(a){console.warn(&quot;THREE.Quaternion: .multiplyVector3() has been removed. Use is now vector.applyQuaternion( quaternion ) instead.&quot;);return a.applyQuaternion(this)};Object.assign(Vb.prototype,{isIntersectionBox:function(a){console.warn(&quot;THREE.Ray: .isIntersectionBox() has been renamed to .intersectsBox().&quot;);
return this.intersectsBox(a)},isIntersectionPlane:function(a){console.warn(&quot;THREE.Ray: .isIntersectionPlane() has been renamed to .intersectsPlane().&quot;);return this.intersectsPlane(a)},isIntersectionSphere:function(a){console.warn(&quot;THREE.Ray: .isIntersectionSphere() has been renamed to .intersectsSphere().&quot;);return this.intersectsSphere(a)}});Object.assign(oa.prototype,{area:function(){console.warn(&quot;THREE.Triangle: .area() has been renamed to .getArea().&quot;);return this.getArea()},barycoordFromPoint:function(a,
b){console.warn(&quot;THREE.Triangle: .barycoordFromPoint() has been renamed to .getBarycoord().&quot;);return this.getBarycoord(a,b)},midpoint:function(a){console.warn(&quot;THREE.Triangle: .midpoint() has been renamed to .getMidpoint().&quot;);return this.getMidpoint(a)},normal:function(a){console.warn(&quot;THREE.Triangle: .normal() has been renamed to .getNormal().&quot;);return this.getNormal(a)},plane:function(a){console.warn(&quot;THREE.Triangle: .plane() has been renamed to .getPlane().&quot;);return this.getPlane(a)}});Object.assign(oa,
{barycoordFromPoint:function(a,b,c,d,e){console.warn(&quot;THREE.Triangle: .barycoordFromPoint() has been renamed to .getBarycoord().&quot;);return oa.getBarycoord(a,b,c,d,e)},normal:function(a,b,c,d){console.warn(&quot;THREE.Triangle: .normal() has been renamed to .getNormal().&quot;);return oa.getNormal(a,b,c,d)}});Object.assign(Kb.prototype,{extractAllPoints:function(a){console.warn(&quot;THREE.Shape: .extractAllPoints() has been removed. Use .extractPoints() instead.&quot;);return this.extractPoints(a)},extrude:function(a){console.warn(&quot;THREE.Shape: .extrude() has been removed. Use ExtrudeGeometry() instead.&quot;);
return new fc(this,a)},makeGeometry:function(a){console.warn(&quot;THREE.Shape: .makeGeometry() has been removed. Use ShapeGeometry() instead.&quot;);return new hc(this,a)}});Object.assign(t.prototype,{fromAttribute:function(a,b,c){console.warn(&quot;THREE.Vector2: .fromAttribute() has been renamed to .fromBufferAttribute().&quot;);return this.fromBufferAttribute(a,b,c)},distanceToManhattan:function(a){console.warn(&quot;THREE.Vector2: .distanceToManhattan() has been renamed to .manhattanDistanceTo().&quot;);return this.manhattanDistanceTo(a)},
lengthManhattan:function(){console.warn(&quot;THREE.Vector2: .lengthManhattan() has been renamed to .manhattanLength().&quot;);return this.manhattanLength()}});Object.assign(n.prototype,{setEulerFromRotationMatrix:function(){console.error(&quot;THREE.Vector3: .setEulerFromRotationMatrix() has been removed. Use Euler.setFromRotationMatrix() instead.&quot;)},setEulerFromQuaternion:function(){console.error(&quot;THREE.Vector3: .setEulerFromQuaternion() has been removed. Use Euler.setFromQuaternion() instead.&quot;)},getPositionFromMatrix:function(a){console.warn(&quot;THREE.Vector3: .getPositionFromMatrix() has been renamed to .setFromMatrixPosition().&quot;);
return this.setFromMatrixPosition(a)},getScaleFromMatrix:function(a){console.warn(&quot;THREE.Vector3: .getScaleFromMatrix() has been renamed to .setFromMatrixScale().&quot;);return this.setFromMatrixScale(a)},getColumnFromMatrix:function(a,b){console.warn(&quot;THREE.Vector3: .getColumnFromMatrix() has been renamed to .setFromMatrixColumn().&quot;);return this.setFromMatrixColumn(b,a)},applyProjection:function(a){console.warn(&quot;THREE.Vector3: .applyProjection() has been removed. Use .applyMatrix4( m ) instead.&quot;);return this.applyMatrix4(a)},
fromAttribute:function(a,b,c){console.warn(&quot;THREE.Vector3: .fromAttribute() has been renamed to .fromBufferAttribute().&quot;);return this.fromBufferAttribute(a,b,c)},distanceToManhattan:function(a){console.warn(&quot;THREE.Vector3: .distanceToManhattan() has been renamed to .manhattanDistanceTo().&quot;);return this.manhattanDistanceTo(a)},lengthManhattan:function(){console.warn(&quot;THREE.Vector3: .lengthManhattan() has been renamed to .manhattanLength().&quot;);return this.manhattanLength()}});Object.assign(ka.prototype,
{fromAttribute:function(a,b,c){console.warn(&quot;THREE.Vector4: .fromAttribute() has been renamed to .fromBufferAttribute().&quot;);return this.fromBufferAttribute(a,b,c)},lengthManhattan:function(){console.warn(&quot;THREE.Vector4: .lengthManhattan() has been renamed to .manhattanLength().&quot;);return this.manhattanLength()}});Object.assign(N.prototype,{computeTangents:function(){console.error(&quot;THREE.Geometry: .computeTangents() has been removed.&quot;)},computeLineDistances:function(){console.error(&quot;THREE.Geometry: .computeLineDistances() has been removed. Use THREE.Line.computeLineDistances() instead.&quot;)},
applyMatrix:function(a){console.warn(&quot;THREE.Geometry: .applyMatrix() has been renamed to .applyMatrix4().&quot;);return this.applyMatrix4(a)}});Object.assign(F.prototype,{getChildByName:function(a){console.warn(&quot;THREE.Object3D: .getChildByName() has been renamed to .getObjectByName().&quot;);return this.getObjectByName(a)},renderDepth:function(){console.warn(&quot;THREE.Object3D: .renderDepth has been removed. Use .renderOrder, instead.&quot;)},translate:function(a,b){console.warn(&quot;THREE.Object3D: .translate() has been removed. Use .translateOnAxis( axis, distance ) instead.&quot;);
return this.translateOnAxis(b,a)},getWorldRotation:function(){console.error(&quot;THREE.Object3D: .getWorldRotation() has been removed. Use THREE.Object3D.getWorldQuaternion( target ) instead.&quot;)},applyMatrix:function(a){console.warn(&quot;THREE.Object3D: .applyMatrix() has been renamed to .applyMatrix4().&quot;);return this.applyMatrix4(a)}});Object.defineProperties(F.prototype,{eulerOrder:{get:function(){console.warn(&quot;THREE.Object3D: .eulerOrder is now .rotation.order.&quot;);return this.rotation.order},set:function(a){console.warn(&quot;THREE.Object3D: .eulerOrder is now .rotation.order.&quot;);
this.rotation.order=a}},useQuaternion:{get:function(){console.warn(&quot;THREE.Object3D: .useQuaternion has been removed. The library now uses quaternions by default.&quot;)},set:function(){console.warn(&quot;THREE.Object3D: .useQuaternion has been removed. The library now uses quaternions by default.&quot;)}}});Object.assign(S.prototype,{setDrawMode:function(){console.error(&quot;THREE.Mesh: .setDrawMode() has been removed. The renderer now always assumes THREE.TrianglesDrawMode. Transform your geometry via BufferGeometryUtils.toTrianglesDrawMode() if necessary.&quot;)}});
Object.defineProperties(S.prototype,{drawMode:{get:function(){console.error(&quot;THREE.Mesh: .drawMode has been removed. The renderer now always assumes THREE.TrianglesDrawMode.&quot;);return 0},set:function(){console.error(&quot;THREE.Mesh: .drawMode has been removed. The renderer now always assumes THREE.TrianglesDrawMode. Transform your geometry via BufferGeometryUtils.toTrianglesDrawMode() if necessary.&quot;)}}});Object.defineProperties(Rd.prototype,{objects:{get:function(){console.warn(&quot;THREE.LOD: .objects has been renamed to .levels.&quot;);
return this.levels}}});Object.defineProperty(Ue.prototype,&quot;useVertexTexture&quot;,{get:function(){console.warn(&quot;THREE.Skeleton: useVertexTexture has been removed.&quot;)},set:function(){console.warn(&quot;THREE.Skeleton: useVertexTexture has been removed.&quot;)}});Te.prototype.initBones=function(){console.error(&quot;THREE.SkinnedMesh: initBones() has been removed.&quot;)};Object.defineProperty(G.prototype,&quot;__arcLengthDivisions&quot;,{get:function(){console.warn(&quot;THREE.Curve: .__arcLengthDivisions is now .arcLengthDivisions.&quot;);return this.arcLengthDivisions},
set:function(a){console.warn(&quot;THREE.Curve: .__arcLengthDivisions is now .arcLengthDivisions.&quot;);this.arcLengthDivisions=a}});aa.prototype.setLens=function(a,b){console.warn(&quot;THREE.PerspectiveCamera.setLens is deprecated. Use .setFocalLength and .filmGauge for a photographic setup.&quot;);void 0!==b&&(this.filmGauge=b);this.setFocalLength(a)};Object.defineProperties(da.prototype,{onlyShadow:{set:function(){console.warn(&quot;THREE.Light: .onlyShadow has been removed.&quot;)}},shadowCameraFov:{set:function(a){console.warn(&quot;THREE.Light: .shadowCameraFov is now .shadow.camera.fov.&quot;);
this.shadow.camera.fov=a}},shadowCameraLeft:{set:function(a){console.warn(&quot;THREE.Light: .shadowCameraLeft is now .shadow.camera.left.&quot;);this.shadow.camera.left=a}},shadowCameraRight:{set:function(a){console.warn(&quot;THREE.Light: .shadowCameraRight is now .shadow.camera.right.&quot;);this.shadow.camera.right=a}},shadowCameraTop:{set:function(a){console.warn(&quot;THREE.Light: .shadowCameraTop is now .shadow.camera.top.&quot;);this.shadow.camera.top=a}},shadowCameraBottom:{set:function(a){console.warn(&quot;THREE.Light: .shadowCameraBottom is now .shadow.camera.bottom.&quot;);
this.shadow.camera.bottom=a}},shadowCameraNear:{set:function(a){console.warn(&quot;THREE.Light: .shadowCameraNear is now .shadow.camera.near.&quot;);this.shadow.camera.near=a}},shadowCameraFar:{set:function(a){console.warn(&quot;THREE.Light: .shadowCameraFar is now .shadow.camera.far.&quot;);this.shadow.camera.far=a}},shadowCameraVisible:{set:function(){console.warn(&quot;THREE.Light: .shadowCameraVisible has been removed. Use new THREE.CameraHelper( light.shadow.camera ) instead.&quot;)}},shadowBias:{set:function(a){console.warn(&quot;THREE.Light: .shadowBias is now .shadow.bias.&quot;);
this.shadow.bias=a}},shadowDarkness:{set:function(){console.warn(&quot;THREE.Light: .shadowDarkness has been removed.&quot;)}},shadowMapWidth:{set:function(a){console.warn(&quot;THREE.Light: .shadowMapWidth is now .shadow.mapSize.width.&quot;);this.shadow.mapSize.width=a}},shadowMapHeight:{set:function(a){console.warn(&quot;THREE.Light: .shadowMapHeight is now .shadow.mapSize.height.&quot;);this.shadow.mapSize.height=a}}});Object.defineProperties(M.prototype,{length:{get:function(){console.warn(&quot;THREE.BufferAttribute: .length has been deprecated. Use .count instead.&quot;);
return this.array.length}},dynamic:{get:function(){console.warn(&quot;THREE.BufferAttribute: .dynamic has been deprecated. Use .usage instead.&quot;);return 35048===this.usage},set:function(){console.warn(&quot;THREE.BufferAttribute: .dynamic has been deprecated. Use .usage instead.&quot;);this.setUsage(35048)}}});Object.assign(M.prototype,{setDynamic:function(a){console.warn(&quot;THREE.BufferAttribute: .setDynamic() has been deprecated. Use .setUsage() instead.&quot;);this.setUsage(!0===a?35048:35044);return this},copyIndicesArray:function(){console.error(&quot;THREE.BufferAttribute: .copyIndicesArray() has been removed.&quot;)},
setArray:function(){console.error(&quot;THREE.BufferAttribute: .setArray has been removed. Use BufferGeometry .setAttribute to replace/resize attribute buffers&quot;)}});Object.assign(C.prototype,{addIndex:function(a){console.warn(&quot;THREE.BufferGeometry: .addIndex() has been renamed to .setIndex().&quot;);this.setIndex(a)},addAttribute:function(a,b,c){console.warn(&quot;THREE.BufferGeometry: .addAttribute() has been renamed to .setAttribute().&quot;);return b&&b.isBufferAttribute||b&&b.isInterleavedBufferAttribute?&quot;index&quot;===
a?(console.warn(&quot;THREE.BufferGeometry.addAttribute: Use .setIndex() for index attribute.&quot;),this.setIndex(b),this):this.setAttribute(a,b):(console.warn(&quot;THREE.BufferGeometry: .addAttribute() now expects ( name, attribute ).&quot;),this.setAttribute(a,new M(b,c)))},addDrawCall:function(a,b,c){void 0!==c&&console.warn(&quot;THREE.BufferGeometry: .addDrawCall() no longer supports indexOffset.&quot;);console.warn(&quot;THREE.BufferGeometry: .addDrawCall() is now .addGroup().&quot;);this.addGroup(a,b)},clearDrawCalls:function(){console.warn(&quot;THREE.BufferGeometry: .clearDrawCalls() is now .clearGroups().&quot;);
this.clearGroups()},computeTangents:function(){console.warn(&quot;THREE.BufferGeometry: .computeTangents() has been removed.&quot;)},computeOffsets:function(){console.warn(&quot;THREE.BufferGeometry: .computeOffsets() has been removed.&quot;)},removeAttribute:function(a){console.warn(&quot;THREE.BufferGeometry: .removeAttribute() has been renamed to .deleteAttribute().&quot;);return this.deleteAttribute(a)},applyMatrix:function(a){console.warn(&quot;THREE.BufferGeometry: .applyMatrix() has been renamed to .applyMatrix4().&quot;);return this.applyMatrix4(a)}});
Object.defineProperties(C.prototype,{drawcalls:{get:function(){console.error(&quot;THREE.BufferGeometry: .drawcalls has been renamed to .groups.&quot;);return this.groups}},offsets:{get:function(){console.warn(&quot;THREE.BufferGeometry: .offsets has been renamed to .groups.&quot;);return this.groups}}});Object.defineProperties(Og.prototype,{linePrecision:{get:function(){console.warn(&quot;THREE.Raycaster: .linePrecision has been deprecated. Use .params.Line.threshold instead.&quot;);return this.params.Line.threshold},set:function(a){console.warn(&quot;THREE.Raycaster: .linePrecision has been deprecated. Use .params.Line.threshold instead.&quot;);
this.params.Line.threshold=a}}});Object.defineProperties(rb.prototype,{dynamic:{get:function(){console.warn(&quot;THREE.InterleavedBuffer: .length has been deprecated. Use .usage instead.&quot;);return 35048===this.usage},set:function(a){console.warn(&quot;THREE.InterleavedBuffer: .length has been deprecated. Use .usage instead.&quot;);this.setUsage(a)}}});Object.assign(rb.prototype,{setDynamic:function(a){console.warn(&quot;THREE.InterleavedBuffer: .setDynamic() has been deprecated. Use .setUsage() instead.&quot;);this.setUsage(!0===
a?35048:35044);return this},setArray:function(){console.error(&quot;THREE.InterleavedBuffer: .setArray has been removed. Use BufferGeometry .setAttribute to replace/resize attribute buffers&quot;)}});Object.assign(fb.prototype,{getArrays:function(){console.error(&quot;THREE.ExtrudeBufferGeometry: .getArrays() has been removed.&quot;)},addShapeList:function(){console.error(&quot;THREE.ExtrudeBufferGeometry: .addShapeList() has been removed.&quot;)},addShape:function(){console.error(&quot;THREE.ExtrudeBufferGeometry: .addShape() has been removed.&quot;)}});
Object.defineProperties(vf.prototype,{dynamic:{set:function(){console.warn(&quot;THREE.Uniform: .dynamic has been removed. Use object.onBeforeRender() instead.&quot;)}},onUpdate:{value:function(){console.warn(&quot;THREE.Uniform: .onUpdate() has been removed. Use object.onBeforeRender() instead.&quot;);return this}}});Object.defineProperties(K.prototype,{wrapAround:{get:function(){console.warn(&quot;THREE.Material: .wrapAround has been removed.&quot;)},set:function(){console.warn(&quot;THREE.Material: .wrapAround has been removed.&quot;)}},
overdraw:{get:function(){console.warn(&quot;THREE.Material: .overdraw has been removed.&quot;)},set:function(){console.warn(&quot;THREE.Material: .overdraw has been removed.&quot;)}},wrapRGB:{get:function(){console.warn(&quot;THREE.Material: .wrapRGB has been removed.&quot;);return new A}},shading:{get:function(){console.error(&quot;THREE.&quot;+this.type+&quot;: .shading has been removed. Use the boolean .flatShading instead.&quot;)},set:function(a){console.warn(&quot;THREE.&quot;+this.type+&quot;: .shading has been removed. Use the boolean .flatShading instead.&quot;);
this.flatShading=1===a}},stencilMask:{get:function(){console.warn(&quot;THREE.&quot;+this.type+&quot;: .stencilMask has been removed. Use .stencilFuncMask instead.&quot;);return this.stencilFuncMask},set:function(a){console.warn(&quot;THREE.&quot;+this.type+&quot;: .stencilMask has been removed. Use .stencilFuncMask instead.&quot;);this.stencilFuncMask=a}}});Object.defineProperties(Jb.prototype,{metal:{get:function(){console.warn(&quot;THREE.MeshPhongMaterial: .metal has been removed. Use THREE.MeshStandardMaterial instead.&quot;);return!1},set:function(){console.warn(&quot;THREE.MeshPhongMaterial: .metal has been removed. Use THREE.MeshStandardMaterial instead&quot;)}}});
Object.defineProperties(Ba.prototype,{derivatives:{get:function(){console.warn(&quot;THREE.ShaderMaterial: .derivatives has been moved to .extensions.derivatives.&quot;);return this.extensions.derivatives},set:function(a){console.warn(&quot;THREE. ShaderMaterial: .derivatives has been moved to .extensions.derivatives.&quot;);this.extensions.derivatives=a}}});Object.assign(og.prototype,{clearTarget:function(a,b,c,d){console.warn(&quot;THREE.WebGLRenderer: .clearTarget() has been deprecated. Use .setRenderTarget() and .clear() instead.&quot;);
this.setRenderTarget(a);this.clear(b,c,d)},animate:function(a){console.warn(&quot;THREE.WebGLRenderer: .animate() is now .setAnimationLoop().&quot;);this.setAnimationLoop(a)},getCurrentRenderTarget:function(){console.warn(&quot;THREE.WebGLRenderer: .getCurrentRenderTarget() is now .getRenderTarget().&quot;);return this.getRenderTarget()},getMaxAnisotropy:function(){console.warn(&quot;THREE.WebGLRenderer: .getMaxAnisotropy() is now .capabilities.getMaxAnisotropy().&quot;);return this.capabilities.getMaxAnisotropy()},getPrecision:function(){console.warn(&quot;THREE.WebGLRenderer: .getPrecision() is now .capabilities.precision.&quot;);
return this.capabilities.precision},resetGLState:function(){console.warn(&quot;THREE.WebGLRenderer: .resetGLState() is now .state.reset().&quot;);return this.state.reset()},supportsFloatTextures:function(){console.warn(&quot;THREE.WebGLRenderer: .supportsFloatTextures() is now .extensions.get( 'OES_texture_float' ).&quot;);return this.extensions.get(&quot;OES_texture_float&quot;)},supportsHalfFloatTextures:function(){console.warn(&quot;THREE.WebGLRenderer: .supportsHalfFloatTextures() is now .extensions.get( 'OES_texture_half_float' ).&quot;);
return this.extensions.get(&quot;OES_texture_half_float&quot;)},supportsStandardDerivatives:function(){console.warn(&quot;THREE.WebGLRenderer: .supportsStandardDerivatives() is now .extensions.get( 'OES_standard_derivatives' ).&quot;);return this.extensions.get(&quot;OES_standard_derivatives&quot;)},supportsCompressedTextureS3TC:function(){console.warn(&quot;THREE.WebGLRenderer: .supportsCompressedTextureS3TC() is now .extensions.get( 'WEBGL_compressed_texture_s3tc' ).&quot;);return this.extensions.get(&quot;WEBGL_compressed_texture_s3tc&quot;)},
supportsCompressedTexturePVRTC:function(){console.warn(&quot;THREE.WebGLRenderer: .supportsCompressedTexturePVRTC() is now .extensions.get( 'WEBGL_compressed_texture_pvrtc' ).&quot;);return this.extensions.get(&quot;WEBGL_compressed_texture_pvrtc&quot;)},supportsBlendMinMax:function(){console.warn(&quot;THREE.WebGLRenderer: .supportsBlendMinMax() is now .extensions.get( 'EXT_blend_minmax' ).&quot;);return this.extensions.get(&quot;EXT_blend_minmax&quot;)},supportsVertexTextures:function(){console.warn(&quot;THREE.WebGLRenderer: .supportsVertexTextures() is now .capabilities.vertexTextures.&quot;);
return this.capabilities.vertexTextures},supportsInstancedArrays:function(){console.warn(&quot;THREE.WebGLRenderer: .supportsInstancedArrays() is now .extensions.get( 'ANGLE_instanced_arrays' ).&quot;);return this.extensions.get(&quot;ANGLE_instanced_arrays&quot;)},enableScissorTest:function(a){console.warn(&quot;THREE.WebGLRenderer: .enableScissorTest() is now .setScissorTest().&quot;);this.setScissorTest(a)},initMaterial:function(){console.warn(&quot;THREE.WebGLRenderer: .initMaterial() has been removed.&quot;)},addPrePlugin:function(){console.warn(&quot;THREE.WebGLRenderer: .addPrePlugin() has been removed.&quot;)},
addPostPlugin:function(){console.warn(&quot;THREE.WebGLRenderer: .addPostPlugin() has been removed.&quot;)},updateShadowMap:function(){console.warn(&quot;THREE.WebGLRenderer: .updateShadowMap() has been removed.&quot;)},setFaceCulling:function(){console.warn(&quot;THREE.WebGLRenderer: .setFaceCulling() has been removed.&quot;)},allocTextureUnit:function(){console.warn(&quot;THREE.WebGLRenderer: .allocTextureUnit() has been removed.&quot;)},setTexture:function(){console.warn(&quot;THREE.WebGLRenderer: .setTexture() has been removed.&quot;)},setTexture2D:function(){console.warn(&quot;THREE.WebGLRenderer: .setTexture2D() has been removed.&quot;)},
setTextureCube:function(){console.warn(&quot;THREE.WebGLRenderer: .setTextureCube() has been removed.&quot;)},getActiveMipMapLevel:function(){console.warn(&quot;THREE.WebGLRenderer: .getActiveMipMapLevel() is now .getActiveMipmapLevel().&quot;);return this.getActiveMipmapLevel()}});Object.defineProperties(og.prototype,{shadowMapEnabled:{get:function(){return this.shadowMap.enabled},set:function(a){console.warn(&quot;THREE.WebGLRenderer: .shadowMapEnabled is now .shadowMap.enabled.&quot;);this.shadowMap.enabled=a}},shadowMapType:{get:function(){return this.shadowMap.type},
set:function(a){console.warn(&quot;THREE.WebGLRenderer: .shadowMapType is now .shadowMap.type.&quot;);this.shadowMap.type=a}},shadowMapCullFace:{get:function(){console.warn(&quot;THREE.WebGLRenderer: .shadowMapCullFace has been removed. Set Material.shadowSide instead.&quot;)},set:function(){console.warn(&quot;THREE.WebGLRenderer: .shadowMapCullFace has been removed. Set Material.shadowSide instead.&quot;)}},context:{get:function(){console.warn(&quot;THREE.WebGLRenderer: .context has been removed. Use .getContext() instead.&quot;);return this.getContext()}},
vr:{get:function(){console.warn(&quot;THREE.WebGLRenderer: .vr has been renamed to .xr&quot;);return this.xr}},gammaInput:{get:function(){console.warn(&quot;THREE.WebGLRenderer: .gammaInput has been removed. Set the encoding for textures via Texture.encoding instead.&quot;);return!1},set:function(){console.warn(&quot;THREE.WebGLRenderer: .gammaInput has been removed. Set the encoding for textures via Texture.encoding instead.&quot;)}},gammaOutput:{get:function(){console.warn(&quot;THREE.WebGLRenderer: .gammaOutput has been removed. Set WebGLRenderer.outputEncoding instead.&quot;);
return!1},set:function(a){console.warn(&quot;THREE.WebGLRenderer: .gammaOutput has been removed. Set WebGLRenderer.outputEncoding instead.&quot;);this.outputEncoding=!0===a?3001:3E3}}});Object.defineProperties(Xh.prototype,{cullFace:{get:function(){console.warn(&quot;THREE.WebGLRenderer: .shadowMap.cullFace has been removed. Set Material.shadowSide instead.&quot;)},set:function(){console.warn(&quot;THREE.WebGLRenderer: .shadowMap.cullFace has been removed. Set Material.shadowSide instead.&quot;)}},renderReverseSided:{get:function(){console.warn(&quot;THREE.WebGLRenderer: .shadowMap.renderReverseSided has been removed. Set Material.shadowSide instead.&quot;)},
set:function(){console.warn(&quot;THREE.WebGLRenderer: .shadowMap.renderReverseSided has been removed. Set Material.shadowSide instead.&quot;)}},renderSingleSided:{get:function(){console.warn(&quot;THREE.WebGLRenderer: .shadowMap.renderSingleSided has been removed. Set Material.shadowSide instead.&quot;)},set:function(){console.warn(&quot;THREE.WebGLRenderer: .shadowMap.renderSingleSided has been removed. Set Material.shadowSide instead.&quot;)}}});Object.defineProperties(Ha.prototype,{wrapS:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .wrapS is now .texture.wrapS.&quot;);
return this.texture.wrapS},set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .wrapS is now .texture.wrapS.&quot;);this.texture.wrapS=a}},wrapT:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .wrapT is now .texture.wrapT.&quot;);return this.texture.wrapT},set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .wrapT is now .texture.wrapT.&quot;);this.texture.wrapT=a}},magFilter:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .magFilter is now .texture.magFilter.&quot;);return this.texture.magFilter},
set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .magFilter is now .texture.magFilter.&quot;);this.texture.magFilter=a}},minFilter:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .minFilter is now .texture.minFilter.&quot;);return this.texture.minFilter},set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .minFilter is now .texture.minFilter.&quot;);this.texture.minFilter=a}},anisotropy:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .anisotropy is now .texture.anisotropy.&quot;);return this.texture.anisotropy},
set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .anisotropy is now .texture.anisotropy.&quot;);this.texture.anisotropy=a}},offset:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .offset is now .texture.offset.&quot;);return this.texture.offset},set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .offset is now .texture.offset.&quot;);this.texture.offset=a}},repeat:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .repeat is now .texture.repeat.&quot;);return this.texture.repeat},set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .repeat is now .texture.repeat.&quot;);
this.texture.repeat=a}},format:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .format is now .texture.format.&quot;);return this.texture.format},set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .format is now .texture.format.&quot;);this.texture.format=a}},type:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .type is now .texture.type.&quot;);return this.texture.type},set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .type is now .texture.type.&quot;);this.texture.type=a}},generateMipmaps:{get:function(){console.warn(&quot;THREE.WebGLRenderTarget: .generateMipmaps is now .texture.generateMipmaps.&quot;);
return this.texture.generateMipmaps},set:function(a){console.warn(&quot;THREE.WebGLRenderTarget: .generateMipmaps is now .texture.generateMipmaps.&quot;);this.texture.generateMipmaps=a}}});Object.defineProperties(id.prototype,{load:{value:function(a){console.warn(&quot;THREE.Audio: .load has been deprecated. Use THREE.AudioLoader instead.&quot;);var b=this;(new tf).load(a,function(a){b.setBuffer(a)});return this}},startTime:{set:function(){console.warn(&quot;THREE.Audio: .startTime is now .play( delay ).&quot;)}}});Kg.prototype.getData=
function(){console.warn(&quot;THREE.AudioAnalyser: .getData() is now .getFrequencyData().&quot;);return this.getFrequencyData()};Gc.prototype.updateCubeMap=function(a,b){console.warn(&quot;THREE.CubeCamera: .updateCubeMap() is now .update().&quot;);return this.update(a,b)};Lb.crossOrigin=void 0;Lb.loadTexture=function(a,b,c,d){console.warn(&quot;THREE.ImageUtils.loadTexture has been deprecated. Use THREE.TextureLoader() instead.&quot;);var e=new ef;e.setCrossOrigin(this.crossOrigin);a=e.load(a,c,void 0,d);b&&(a.mapping=b);return a};
Lb.loadTextureCube=function(a,b,c,d){console.warn(&quot;THREE.ImageUtils.loadTextureCube has been deprecated. Use THREE.CubeTextureLoader() instead.&quot;);var e=new df;e.setCrossOrigin(this.crossOrigin);a=e.load(a,c,void 0,d);b&&(a.mapping=b);return a};Lb.loadCompressedTexture=function(){console.error(&quot;THREE.ImageUtils.loadCompressedTexture has been removed. Use THREE.DDSLoader instead.&quot;)};Lb.loadCompressedTextureCube=function(){console.error(&quot;THREE.ImageUtils.loadCompressedTextureCube has been removed. Use THREE.DDSLoader instead.&quot;)};
&quot;undefined&quot;!==typeof __THREE_DEVTOOLS__&&__THREE_DEVTOOLS__.dispatchEvent(new CustomEvent(&quot;register&quot;,{detail:{revision:&quot;115&quot;}}));k.ACESFilmicToneMapping=5;k.AddEquation=100;k.AddOperation=2;k.AdditiveBlending=2;k.AlphaFormat=1021;k.AlwaysDepth=1;k.AlwaysStencilFunc=519;k.AmbientLight=mf;k.AmbientLightProbe=Fg;k.AnimationClip=Qa;k.AnimationLoader=wg;k.AnimationMixer=Mg;k.AnimationObjectGroup=oi;k.AnimationUtils=R;k.ArcCurve=gd;k.ArrayCamera=Pe;k.ArrowHelper=xb;k.Audio=id;k.AudioAnalyser=Kg;k.AudioContext=
Ig;k.AudioListener=Hg;k.AudioLoader=tf;k.AxesHelper=ve;k.AxisHelper=function(a){console.warn(&quot;THREE.AxisHelper has been renamed to THREE.AxesHelper.&quot;);return new ve(a)};k.BackSide=1;k.BasicDepthPacking=3200;k.BasicShadowMap=0;k.BinaryTextureLoader=function(a){console.warn(&quot;THREE.BinaryTextureLoader has been renamed to THREE.DataTextureLoader.&quot;);return new cf(a)};k.Bone=pg;k.BooleanKeyframeTrack=Ze;k.BoundingBoxHelper=function(a,b){console.warn(&quot;THREE.BoundingBoxHelper has been deprecated. Creating a THREE.BoxHelper instead.&quot;);
return new wb(a,b)};k.Box2=Qg;k.Box3=Sa;k.Box3Helper=te;k.BoxBufferGeometry=Jd;k.BoxGeometry=nh;k.BoxHelper=wb;k.BufferAttribute=M;k.BufferGeometry=C;k.BufferGeometryLoader=rf;k.ByteType=1010;k.Cache=vc;k.Camera=db;k.CameraHelper=se;k.CanvasRenderer=function(){console.error(&quot;THREE.CanvasRenderer has been removed&quot;)};k.CanvasTexture=Sd;k.CatmullRomCurve3=pa;k.CineonToneMapping=4;k.CircleBufferGeometry=cd;k.CircleGeometry=me;k.ClampToEdgeWrapping=1001;k.Clock=Gg;k.ClosedSplineCurve3=Fi;k.Color=A;k.ColorKeyframeTrack=
$e;k.CompressedTexture=Qc;k.CompressedTextureLoader=xg;k.ConeBufferGeometry=le;k.ConeGeometry=ke;k.CubeCamera=Gc;k.CubeGeometry=nh;k.CubeReflectionMapping=301;k.CubeRefractionMapping=302;k.CubeTexture=qb;k.CubeTextureLoader=df;k.CubeUVReflectionMapping=306;k.CubeUVRefractionMapping=307;k.CubicBezierCurve=Wa;k.CubicBezierCurve3=hb;k.CubicInterpolant=Xe;k.CullFaceBack=1;k.CullFaceFront=2;k.CullFaceFrontBack=3;k.CullFaceNone=0;k.Curve=G;k.CurvePath=vb;k.CustomBlending=5;k.CylinderBufferGeometry=tb;k.CylinderGeometry=
jc;k.Cylindrical=si;k.DataTexture=ac;k.DataTexture2DArray=Ic;k.DataTexture3D=Jc;k.DataTextureLoader=cf;k.DecrementStencilOp=7683;k.DecrementWrapStencilOp=34056;k.DefaultLoadingManager=ki;k.DepthFormat=1026;k.DepthStencilFormat=1027;k.DepthTexture=Td;k.DirectionalLight=lf;k.DirectionalLightHelper=md;k.DirectionalLightShadow=kf;k.DiscreteInterpolant=Ye;k.DodecahedronBufferGeometry=Vc;k.DodecahedronGeometry=Zd;k.DoubleSide=2;k.DstAlphaFactor=206;k.DstColorFactor=208;k.DynamicBufferAttribute=function(a,
b){console.warn(&quot;THREE.DynamicBufferAttribute has been removed. Use new THREE.BufferAttribute().setUsage( THREE.DynamicDrawUsage ) instead.&quot;);return(new M(a,b)).setUsage(35048)};k.DynamicCopyUsage=35050;k.DynamicDrawUsage=35048;k.DynamicReadUsage=35049;k.EdgesGeometry=bd;k.EdgesHelper=function(a,b){console.warn(&quot;THREE.EdgesHelper has been removed. Use THREE.EdgesGeometry instead.&quot;);return new ma(new bd(a.geometry),new la({color:void 0!==b?b:16777215}))};k.EllipseCurve=Ma;k.EqualDepth=4;k.EqualStencilFunc=
514;k.EquirectangularReflectionMapping=303;k.EquirectangularRefractionMapping=304;k.Euler=Tb;k.EventDispatcher=Ea;k.ExtrudeBufferGeometry=fb;k.ExtrudeGeometry=fc;k.Face3=Bc;k.Face4=function(a,b,c,d,e,f,g){console.warn(&quot;THREE.Face4 has been removed. A THREE.Face3 will be created instead.&quot;);return new Bc(a,b,c,e,f,g)};k.FaceColors=1;k.FileLoader=Ra;k.FlatShading=1;k.Float32Attribute=function(a,b){console.warn(&quot;THREE.Float32Attribute has been removed. Use new THREE.Float32BufferAttribute() instead.&quot;);
return new y(a,b)};k.Float32BufferAttribute=y;k.Float64Attribute=function(a,b){console.warn(&quot;THREE.Float64Attribute has been removed. Use new THREE.Float64BufferAttribute() instead.&quot;);return new Gd(a,b)};k.Float64BufferAttribute=Gd;k.FloatType=1015;k.Fog=Re;k.FogExp2=Qe;k.Font=Cg;k.FontLoader=Dg;k.FrontFaceDirectionCCW=1;k.FrontFaceDirectionCW=0;k.FrontSide=0;k.Frustum=Hc;k.GammaEncoding=3007;k.Geometry=N;k.GeometryUtils={merge:function(a,b,c){console.warn(&quot;THREE.GeometryUtils: .merge() has been moved to Geometry. Use geometry.merge( geometry2, matrix, materialIndexOffset ) instead.&quot;);
if(b.isMesh){b.matrixAutoUpdate&&b.updateMatrix();var d=b.matrix;b=b.geometry}a.merge(b,d,c)},center:function(a){console.warn(&quot;THREE.GeometryUtils: .center() has been moved to Geometry. Use geometry.center() instead.&quot;);return a.center()}};k.GreaterDepth=6;k.GreaterEqualDepth=5;k.GreaterEqualStencilFunc=518;k.GreaterStencilFunc=516;k.GridHelper=wf;k.Group=Mc;k.HalfFloatType=1016;k.HemisphereLight=ff;k.HemisphereLightHelper=ld;k.HemisphereLightProbe=Eg;k.IcosahedronBufferGeometry=Uc;k.IcosahedronGeometry=
Yd;k.ImageBitmapLoader=Ag;k.ImageLoader=fd;k.ImageUtils=Lb;k.ImmediateRenderObject=re;k.IncrementStencilOp=7682;k.IncrementWrapStencilOp=34055;k.InstancedBufferAttribute=qf;k.InstancedBufferGeometry=pf;k.InstancedInterleavedBuffer=Ng;k.InstancedMesh=Ve;k.Int16Attribute=function(a,b){console.warn(&quot;THREE.Int16Attribute has been removed. Use new THREE.Int16BufferAttribute() instead.&quot;);return new Ed(a,b)};k.Int16BufferAttribute=Ed;k.Int32Attribute=function(a,b){console.warn(&quot;THREE.Int32Attribute has been removed. Use new THREE.Int32BufferAttribute() instead.&quot;);
return new Fd(a,b)};k.Int32BufferAttribute=Fd;k.Int8Attribute=function(a,b){console.warn(&quot;THREE.Int8Attribute has been removed. Use new THREE.Int8BufferAttribute() instead.&quot;);return new Bd(a,b)};k.Int8BufferAttribute=Bd;k.IntType=1013;k.InterleavedBuffer=rb;k.InterleavedBufferAttribute=Od;k.Interpolant=La;k.InterpolateDiscrete=2300;k.InterpolateLinear=2301;k.InterpolateSmooth=2302;k.InvertStencilOp=5386;k.JSONLoader=function(){console.error(&quot;THREE.JSONLoader has been removed.&quot;)};k.KeepStencilOp=7680;
k.KeyframeTrack=sa;k.LOD=Rd;k.LatheBufferGeometry=ad;k.LatheGeometry=je;k.Layers=He;k.LensFlare=function(){console.error(&quot;THREE.LensFlare has been moved to /examples/jsm/objects/Lensflare.js&quot;)};k.LessDepth=2;k.LessEqualDepth=3;k.LessEqualStencilFunc=515;k.LessStencilFunc=513;k.Light=da;k.LightProbe=ab;k.LightShadow=jb;k.Line=Ka;k.Line3=Rg;k.LineBasicMaterial=la;k.LineCurve=Da;k.LineCurve3=Xa;k.LineDashedMaterial=qc;k.LineLoop=We;k.LinePieces=1;k.LineSegments=ma;k.LineStrip=0;k.LinearEncoding=3E3;
k.LinearFilter=1006;k.LinearInterpolant=ne;k.LinearMipMapLinearFilter=1008;k.LinearMipMapNearestFilter=1007;k.LinearMipmapLinearFilter=1008;k.LinearMipmapNearestFilter=1007;k.LinearToneMapping=1;k.Loader=W;k.LoaderUtils=th;k.LoadingManager=vg;k.LogLuvEncoding=3003;k.LoopOnce=2200;k.LoopPingPong=2202;k.LoopRepeat=2201;k.LuminanceAlphaFormat=1025;k.LuminanceFormat=1024;k.MOUSE={LEFT:0,MIDDLE:1,RIGHT:2,ROTATE:0,DOLLY:1,PAN:2};k.Material=K;k.MaterialLoader=of;k.Math=L;k.MathUtils=L;k.Matrix3=wa;k.Matrix4=
P;k.MaxEquation=104;k.Mesh=S;k.MeshBasicMaterial=Oa;k.MeshDepthMaterial=Fb;k.MeshDistanceMaterial=Gb;k.MeshFaceMaterial=function(a){console.warn(&quot;THREE.MeshFaceMaterial has been removed. Use an Array instead.&quot;);return a};k.MeshLambertMaterial=oc;k.MeshMatcapMaterial=pc;k.MeshNormalMaterial=nc;k.MeshPhongMaterial=Jb;k.MeshPhysicalMaterial=lc;k.MeshStandardMaterial=gb;k.MeshToonMaterial=mc;k.MinEquation=103;k.MirroredRepeatWrapping=1002;k.MixOperation=1;k.MultiMaterial=function(a){void 0===a&&(a=[]);
console.warn(&quot;THREE.MultiMaterial has been removed. Use an Array instead.&quot;);a.isMultiMaterial=!0;a.materials=a;a.clone=function(){return a.slice()};return a};k.MultiplyBlending=4;k.MultiplyOperation=0;k.NearestFilter=1003;k.NearestMipMapLinearFilter=1005;k.NearestMipMapNearestFilter=1004;k.NearestMipmapLinearFilter=1005;k.NearestMipmapNearestFilter=1004;k.NeverDepth=0;k.NeverStencilFunc=512;k.NoBlending=0;k.NoColors=0;k.NoToneMapping=0;k.NormalBlending=1;k.NotEqualDepth=7;k.NotEqualStencilFunc=517;
k.NumberKeyframeTrack=dd;k.Object3D=F;k.ObjectLoader=sf;k.ObjectSpaceNormalMap=1;k.OctahedronBufferGeometry=cc;k.OctahedronGeometry=Xd;k.OneFactor=201;k.OneMinusDstAlphaFactor=207;k.OneMinusDstColorFactor=209;k.OneMinusSrcAlphaFactor=205;k.OneMinusSrcColorFactor=203;k.OrthographicCamera=hd;k.PCFShadowMap=1;k.PCFSoftShadowMap=2;k.PMREMGenerator=Tg;k.ParametricBufferGeometry=Sc;k.ParametricGeometry=Ud;k.Particle=function(a){console.warn(&quot;THREE.Particle has been renamed to THREE.Sprite.&quot;);return new Pd(a)};
k.ParticleBasicMaterial=function(a){console.warn(&quot;THREE.ParticleBasicMaterial has been renamed to THREE.PointsMaterial.&quot;);return new Va(a)};k.ParticleSystem=function(a,b){console.warn(&quot;THREE.ParticleSystem has been renamed to THREE.Points.&quot;);return new Pc(a,b)};k.ParticleSystemMaterial=function(a){console.warn(&quot;THREE.ParticleSystemMaterial has been renamed to THREE.PointsMaterial.&quot;);return new Va(a)};k.Path=$a;k.PerspectiveCamera=aa;k.Plane=Ta;k.PlaneBufferGeometry=bc;k.PlaneGeometry=Id;k.PlaneHelper=
ue;k.PointCloud=function(a,b){console.warn(&quot;THREE.PointCloud has been renamed to THREE.Points.&quot;);return new Pc(a,b)};k.PointCloudMaterial=function(a){console.warn(&quot;THREE.PointCloudMaterial has been renamed to THREE.PointsMaterial.&quot;);return new Va(a)};k.PointLight=jf;k.PointLightHelper=kd;k.Points=Pc;k.PointsMaterial=Va;k.PolarGridHelper=xf;k.PolyhedronBufferGeometry=Ga;k.PolyhedronGeometry=Vd;k.PositionalAudio=Jg;k.PropertyBinding=ya;k.PropertyMixer=Lg;k.QuadraticBezierCurve=Ya;k.QuadraticBezierCurve3=
ib;k.Quaternion=Aa;k.QuaternionKeyframeTrack=oe;k.QuaternionLinearInterpolant=af;k.REVISION=&quot;115&quot;;k.RGBADepthPacking=3201;k.RGBAFormat=1023;k.RGBAIntegerFormat=1033;k.RGBA_ASTC_10x10_Format=37819;k.RGBA_ASTC_10x5_Format=37816;k.RGBA_ASTC_10x6_Format=37817;k.RGBA_ASTC_10x8_Format=37818;k.RGBA_ASTC_12x10_Format=37820;k.RGBA_ASTC_12x12_Format=37821;k.RGBA_ASTC_4x4_Format=37808;k.RGBA_ASTC_5x4_Format=37809;k.RGBA_ASTC_5x5_Format=37810;k.RGBA_ASTC_6x5_Format=37811;k.RGBA_ASTC_6x6_Format=37812;k.RGBA_ASTC_8x5_Format=
37813;k.RGBA_ASTC_8x6_Format=37814;k.RGBA_ASTC_8x8_Format=37815;k.RGBA_BPTC_Format=36492;k.RGBA_ETC2_EAC_Format=37496;k.RGBA_PVRTC_2BPPV1_Format=35843;k.RGBA_PVRTC_4BPPV1_Format=35842;k.RGBA_S3TC_DXT1_Format=33777;k.RGBA_S3TC_DXT3_Format=33778;k.RGBA_S3TC_DXT5_Format=33779;k.RGBDEncoding=3006;k.RGBEEncoding=3002;k.RGBEFormat=1023;k.RGBFormat=1022;k.RGBIntegerFormat=1032;k.RGBM16Encoding=3005;k.RGBM7Encoding=3004;k.RGB_ETC1_Format=36196;k.RGB_ETC2_Format=37492;k.RGB_PVRTC_2BPPV1_Format=35841;k.RGB_PVRTC_4BPPV1_Format=
35840;k.RGB_S3TC_DXT1_Format=33776;k.RGFormat=1030;k.RGIntegerFormat=1031;k.RawShaderMaterial=ub;k.Ray=Vb;k.Raycaster=Og;k.RectAreaLight=nf;k.RedFormat=1028;k.RedIntegerFormat=1029;k.ReinhardToneMapping=2;k.RepeatWrapping=1E3;k.ReplaceStencilOp=7681;k.ReverseSubtractEquation=102;k.RingBufferGeometry=$c;k.RingGeometry=ie;k.SRGB8_ALPHA8_ASTC_10x10_Format=37851;k.SRGB8_ALPHA8_ASTC_10x5_Format=37848;k.SRGB8_ALPHA8_ASTC_10x6_Format=37849;k.SRGB8_ALPHA8_ASTC_10x8_Format=37850;k.SRGB8_ALPHA8_ASTC_12x10_Format=
37852;k.SRGB8_ALPHA8_ASTC_12x12_Format=37853;k.SRGB8_ALPHA8_ASTC_4x4_Format=37840;k.SRGB8_ALPHA8_ASTC_5x4_Format=37841;k.SRGB8_ALPHA8_ASTC_5x5_Format=37842;k.SRGB8_ALPHA8_ASTC_6x5_Format=37843;k.SRGB8_ALPHA8_ASTC_6x6_Format=37844;k.SRGB8_ALPHA8_ASTC_8x5_Format=37845;k.SRGB8_ALPHA8_ASTC_8x6_Format=37846;k.SRGB8_ALPHA8_ASTC_8x8_Format=37847;k.Scene=ob;k.SceneUtils={createMultiMaterialObject:function(){console.error(&quot;THREE.SceneUtils has been moved to /examples/jsm/utils/SceneUtils.js&quot;)},detach:function(){console.error(&quot;THREE.SceneUtils has been moved to /examples/jsm/utils/SceneUtils.js&quot;)},
attach:function(){console.error(&quot;THREE.SceneUtils has been moved to /examples/jsm/utils/SceneUtils.js&quot;)}};k.ShaderChunk=O;k.ShaderLib=eb;k.ShaderMaterial=Ba;k.ShadowMaterial=kc;k.Shape=Kb;k.ShapeBufferGeometry=ic;k.ShapeGeometry=hc;k.ShapePath=Bg;k.ShapeUtils=sb;k.ShortType=1011;k.Skeleton=Ue;k.SkeletonHelper=rc;k.SkinnedMesh=Te;k.SmoothShading=2;k.Sphere=pb;k.SphereBufferGeometry=gc;k.SphereGeometry=he;k.Spherical=ri;k.SphericalHarmonics3=uf;k.SphericalReflectionMapping=305;k.Spline=$g;k.SplineCurve=
Za;k.SplineCurve3=Gi;k.SpotLight=hf;k.SpotLightHelper=jd;k.SpotLightShadow=gf;k.Sprite=Pd;k.SpriteMaterial=Ib;k.SrcAlphaFactor=204;k.SrcAlphaSaturateFactor=210;k.SrcColorFactor=202;k.StaticCopyUsage=35046;k.StaticDrawUsage=35044;k.StaticReadUsage=35045;k.StereoCamera=mi;k.StreamCopyUsage=35042;k.StreamDrawUsage=35040;k.StreamReadUsage=35041;k.StringKeyframeTrack=bf;k.SubtractEquation=101;k.SubtractiveBlending=3;k.TOUCH={ROTATE:0,PAN:1,DOLLY_PAN:2,DOLLY_ROTATE:3};k.TangentSpaceNormalMap=0;k.TetrahedronBufferGeometry=
Tc;k.TetrahedronGeometry=Wd;k.TextBufferGeometry=Zc;k.TextGeometry=ge;k.Texture=V;k.TextureLoader=ef;k.TorusBufferGeometry=Xc;k.TorusGeometry=be;k.TorusKnotBufferGeometry=Wc;k.TorusKnotGeometry=ae;k.Triangle=oa;k.TriangleFanDrawMode=2;k.TriangleStripDrawMode=1;k.TrianglesDrawMode=0;k.TubeBufferGeometry=dc;k.TubeGeometry=$d;k.UVMapping=300;k.Uint16Attribute=function(a,b){console.warn(&quot;THREE.Uint16Attribute has been removed. Use new THREE.Uint16BufferAttribute() instead.&quot;);return new Wb(a,b)};k.Uint16BufferAttribute=
Wb;k.Uint32Attribute=function(a,b){console.warn(&quot;THREE.Uint32Attribute has been removed. Use new THREE.Uint32BufferAttribute() instead.&quot;);return new Xb(a,b)};k.Uint32BufferAttribute=Xb;k.Uint8Attribute=function(a,b){console.warn(&quot;THREE.Uint8Attribute has been removed. Use new THREE.Uint8BufferAttribute() instead.&quot;);return new Cd(a,b)};k.Uint8BufferAttribute=Cd;k.Uint8ClampedAttribute=function(a,b){console.warn(&quot;THREE.Uint8ClampedAttribute has been removed. Use new THREE.Uint8ClampedBufferAttribute() instead.&quot;);
return new Dd(a,b)};k.Uint8ClampedBufferAttribute=Dd;k.Uncharted2ToneMapping=3;k.Uniform=vf;k.UniformsLib=D;k.UniformsUtils=Uh;k.UnsignedByteType=1009;k.UnsignedInt248Type=1020;k.UnsignedIntType=1014;k.UnsignedShort4444Type=1017;k.UnsignedShort5551Type=1018;k.UnsignedShort565Type=1019;k.UnsignedShortType=1012;k.VSMShadowMap=3;k.Vector2=t;k.Vector3=n;k.Vector4=ka;k.VectorKeyframeTrack=ed;k.Vertex=function(a,b,c){console.warn(&quot;THREE.Vertex has been removed. Use THREE.Vector3 instead.&quot;);return new n(a,
b,c)};k.VertexColors=2;k.VideoTexture=sg;k.WebGLCubeRenderTarget=Db;k.WebGLMultisampleRenderTarget=Zf;k.WebGLRenderTarget=Ha;k.WebGLRenderTargetCube=function(a,b,c){console.warn(&quot;THREE.WebGLRenderTargetCube( width, height, options ) is now WebGLCubeRenderTarget( size, options ).&quot;);return new Db(a,c)};k.WebGLRenderer=og;k.WebGLUtils=Zh;k.WireframeGeometry=Rc;k.WireframeHelper=function(a,b){console.warn(&quot;THREE.WireframeHelper has been removed. Use THREE.WireframeGeometry instead.&quot;);return new ma(new Rc(a.geometry),
new la({color:void 0!==b?b:16777215}))};k.WrapAroundEnding=2402;k.XHRLoader=function(a){console.warn(&quot;THREE.XHRLoader has been renamed to THREE.FileLoader.&quot;);return new Ra(a)};k.ZeroCurvatureEnding=2400;k.ZeroFactor=200;k.ZeroSlopeEnding=2401;k.ZeroStencilOp=0;k.sRGBEncoding=3001;Object.defineProperty(k,&quot;__esModule&quot;,{value:!0})});
THREE.TrackballControls=function(object,domElement){if(domElement===undefined)console.warn('THREE.TrackballControls: The second parameter &quot;domElement&quot; is now mandatory.');if(domElement===document)console.error('THREE.TrackballControls: &quot;document&quot; should not be used as the target &quot;domElement&quot;. Please use &quot;renderer.domElement&quot; instead.');var _this=this;var STATE={NONE:-1,ROTATE:0,ZOOM:1,PAN:2,TOUCH_ROTATE:3,TOUCH_ZOOM_PAN:4};this.object=object;this.domElement=domElement;this.enabled=true;this.screen={left:0,top:0,width:0,height:0};this.rotateSpeed=1.0;this.zoomSpeed=1.2;this.panSpeed=0.3;this.noRotate=false;this.noZoom=false;this.noPan=false;this.staticMoving=false;this.dynamicDampingFactor=0.2;this.minDistance=0;this.maxDistance=Infinity;this.keys=[65 ,83 ,68 ];this.mouseButtons={LEFT:THREE.MOUSE.ROTATE,MIDDLE:THREE.MOUSE.ZOOM,RIGHT:THREE.MOUSE.PAN};this.target=new THREE.Vector3();var EPS=0.000001;var lastPosition=new THREE.Vector3();var lastZoom=1;var _state=STATE.NONE,_keyState=STATE.NONE,_eye=new THREE.Vector3(),_movePrev=new THREE.Vector2(),_moveCurr=new THREE.Vector2(),_lastAxis=new THREE.Vector3(),_lastAngle=0,_zoomStart=new THREE.Vector2(),_zoomEnd=new THREE.Vector2(),_touchZoomDistanceStart=0,_touchZoomDistanceEnd=0,_panStart=new THREE.Vector2(),_panEnd=new THREE.Vector2();this.target0=this.target.clone();this.position0=this.object.position.clone();this.up0=this.object.up.clone();this.zoom0=this.object.zoom;var changeEvent={type:'change'};var startEvent={type:'start'};var endEvent={type:'end'};this.handleResize=function(){var box=this.domElement.getBoundingClientRect();var d=this.domElement.ownerDocument.documentElement;this.screen.left=box.left+window.pageXOffset-d.clientLeft;this.screen.top=box.top+window.pageYOffset-d.clientTop;this.screen.width=box.width;this.screen.height=box.height;};var getMouseOnScreen=(function(){var vector=new THREE.Vector2();return function getMouseOnScreen(pageX,pageY){vector.set((pageX-_this.screen.left)/_this.screen.width,(pageY-_this.screen.top)/_this.screen.height);return vector;};}());var getMouseOnCircle=(function(){var vector=new THREE.Vector2();return function getMouseOnCircle(pageX,pageY){vector.set(((pageX-_this.screen.width*0.5-_this.screen.left)/(_this.screen.width*0.5)),((_this.screen.height+2*(_this.screen.top-pageY))/_this.screen.width));return vector;};}());this.rotateCamera=(function(){var axis=new THREE.Vector3(),quaternion=new THREE.Quaternion(),eyeDirection=new THREE.Vector3(),objectUpDirection=new THREE.Vector3(),objectSidewaysDirection=new THREE.Vector3(),moveDirection=new THREE.Vector3(),angle;return function rotateCamera(){moveDirection.set(_moveCurr.x-_movePrev.x,_moveCurr.y-_movePrev.y,0);angle=moveDirection.length();if(angle){_eye.copy(_this.object.position).sub(_this.target);eyeDirection.copy(_eye).normalize();objectUpDirection.copy(_this.object.up).normalize();objectSidewaysDirection.crossVectors(objectUpDirection,eyeDirection).normalize();objectUpDirection.setLength(_moveCurr.y-_movePrev.y);objectSidewaysDirection.setLength(_moveCurr.x-_movePrev.x);moveDirection.copy(objectUpDirection.add(objectSidewaysDirection));axis.crossVectors(moveDirection,_eye).normalize();angle*=_this.rotateSpeed;quaternion.setFromAxisAngle(axis,angle);_eye.applyQuaternion(quaternion);_this.object.up.applyQuaternion(quaternion);_lastAxis.copy(axis);_lastAngle=angle;}else if(!_this.staticMoving&&_lastAngle){_lastAngle*=Math.sqrt(1.0-_this.dynamicDampingFactor);_eye.copy(_this.object.position).sub(_this.target);quaternion.setFromAxisAngle(_lastAxis,_lastAngle);_eye.applyQuaternion(quaternion);_this.object.up.applyQuaternion(quaternion);}
_movePrev.copy(_moveCurr);};}());this.zoomCamera=function(){var factor;if(_state===STATE.TOUCH_ZOOM_PAN){factor=_touchZoomDistanceStart/_touchZoomDistanceEnd;_touchZoomDistanceStart=_touchZoomDistanceEnd;if(_this.object.isPerspectiveCamera){_eye.multiplyScalar(factor);}else if(_this.object.isOrthographicCamera){_this.object.zoom*=factor;_this.object.updateProjectionMatrix();}else{console.warn('THREE.TrackballControls: Unsupported camera type');}}else{factor=1.0+(_zoomEnd.y-_zoomStart.y)*_this.zoomSpeed;if(factor!==1.0&&factor>0.0){if(_this.object.isPerspectiveCamera){_eye.multiplyScalar(factor);}else if(_this.object.isOrthographicCamera){_this.object.zoom/=factor;_this.object.updateProjectionMatrix();}else{console.warn('THREE.TrackballControls: Unsupported camera type');}}
if(_this.staticMoving){_zoomStart.copy(_zoomEnd);}else{_zoomStart.y+=(_zoomEnd.y-_zoomStart.y)*this.dynamicDampingFactor;}}};this.panCamera=(function(){var mouseChange=new THREE.Vector2(),objectUp=new THREE.Vector3(),pan=new THREE.Vector3();return function panCamera(){mouseChange.copy(_panEnd).sub(_panStart);if(mouseChange.lengthSq()){if(_this.object.isOrthographicCamera){var scale_x=(_this.object.right-_this.object.left)/_this.object.zoom/_this.domElement.clientWidth;var scale_y=(_this.object.top-_this.object.bottom)/_this.object.zoom/_this.domElement.clientWidth;mouseChange.x*=scale_x;mouseChange.y*=scale_y;}
mouseChange.multiplyScalar(_eye.length()*_this.panSpeed);pan.copy(_eye).cross(_this.object.up).setLength(mouseChange.x);pan.add(objectUp.copy(_this.object.up).setLength(mouseChange.y));_this.object.position.add(pan);_this.target.add(pan);if(_this.staticMoving){_panStart.copy(_panEnd);}else{_panStart.add(mouseChange.subVectors(_panEnd,_panStart).multiplyScalar(_this.dynamicDampingFactor));}}};}());this.checkDistances=function(){if(!_this.noZoom||!_this.noPan){if(_eye.lengthSq()>_this.maxDistance*_this.maxDistance){_this.object.position.addVectors(_this.target,_eye.setLength(_this.maxDistance));_zoomStart.copy(_zoomEnd);}
if(_eye.lengthSq()<_this.minDistance*_this.minDistance){_this.object.position.addVectors(_this.target,_eye.setLength(_this.minDistance));_zoomStart.copy(_zoomEnd);}}};this.update=function(){_eye.subVectors(_this.object.position,_this.target);if(!_this.noRotate){_this.rotateCamera();}
if(!_this.noZoom){_this.zoomCamera();}
if(!_this.noPan){_this.panCamera();}
_this.object.position.addVectors(_this.target,_eye);if(_this.object.isPerspectiveCamera){_this.checkDistances();_this.object.lookAt(_this.target);if(lastPosition.distanceToSquared(_this.object.position)>EPS){_this.dispatchEvent(changeEvent);lastPosition.copy(_this.object.position);}}else if(_this.object.isOrthographicCamera){_this.object.lookAt(_this.target);if(lastPosition.distanceToSquared(_this.object.position)>EPS||lastZoom!==_this.object.zoom){_this.dispatchEvent(changeEvent);lastPosition.copy(_this.object.position);lastZoom=_this.object.zoom;}}else{console.warn('THREE.TrackballControls: Unsupported camera type');}};this.reset=function(){_state=STATE.NONE;_keyState=STATE.NONE;_this.target.copy(_this.target0);_this.object.position.copy(_this.position0);_this.object.up.copy(_this.up0);_this.object.zoom=_this.zoom0;_this.object.updateProjectionMatrix();_eye.subVectors(_this.object.position,_this.target);_this.object.lookAt(_this.target);_this.dispatchEvent(changeEvent);lastPosition.copy(_this.object.position);lastZoom=_this.object.zoom;};function keydown(event){if(_this.enabled===false)return;window.removeEventListener('keydown',keydown);if(_keyState!==STATE.NONE){return;}else if(event.keyCode===_this.keys[STATE.ROTATE]&&!_this.noRotate){_keyState=STATE.ROTATE;}else if(event.keyCode===_this.keys[STATE.ZOOM]&&!_this.noZoom){_keyState=STATE.ZOOM;}else if(event.keyCode===_this.keys[STATE.PAN]&&!_this.noPan){_keyState=STATE.PAN;}}
function keyup(){if(_this.enabled===false)return;_keyState=STATE.NONE;window.addEventListener('keydown',keydown,false);}
function mousedown(event){if(_this.enabled===false)return;event.preventDefault();event.stopPropagation();if(_state===STATE.NONE){switch(event.button){case _this.mouseButtons.LEFT:_state=STATE.ROTATE;break;case _this.mouseButtons.MIDDLE:_state=STATE.ZOOM;break;case _this.mouseButtons.RIGHT:_state=STATE.PAN;break;default:_state=STATE.NONE;}}
var state=(_keyState!==STATE.NONE)?_keyState:_state;if(state===STATE.ROTATE&&!_this.noRotate){_moveCurr.copy(getMouseOnCircle(event.pageX,event.pageY));_movePrev.copy(_moveCurr);}else if(state===STATE.ZOOM&&!_this.noZoom){_zoomStart.copy(getMouseOnScreen(event.pageX,event.pageY));_zoomEnd.copy(_zoomStart);}else if(state===STATE.PAN&&!_this.noPan){_panStart.copy(getMouseOnScreen(event.pageX,event.pageY));_panEnd.copy(_panStart);}
document.addEventListener('mousemove',mousemove,false);document.addEventListener('mouseup',mouseup,false);_this.dispatchEvent(startEvent);}
function mousemove(event){if(_this.enabled===false)return;event.preventDefault();event.stopPropagation();var state=(_keyState!==STATE.NONE)?_keyState:_state;if(state===STATE.ROTATE&&!_this.noRotate){_movePrev.copy(_moveCurr);_moveCurr.copy(getMouseOnCircle(event.pageX,event.pageY));}else if(state===STATE.ZOOM&&!_this.noZoom){_zoomEnd.copy(getMouseOnScreen(event.pageX,event.pageY));}else if(state===STATE.PAN&&!_this.noPan){_panEnd.copy(getMouseOnScreen(event.pageX,event.pageY));}}
function mouseup(event){if(_this.enabled===false)return;event.preventDefault();event.stopPropagation();_state=STATE.NONE;document.removeEventListener('mousemove',mousemove);document.removeEventListener('mouseup',mouseup);_this.dispatchEvent(endEvent);}
function mousewheel(event){if(_this.enabled===false)return;if(_this.noZoom===true)return;event.preventDefault();event.stopPropagation();switch(event.deltaMode){case 2:_zoomStart.y-=event.deltaY*0.025;break;case 1:_zoomStart.y-=event.deltaY*0.01;break;default:_zoomStart.y-=event.deltaY*0.00025;break;}
_this.dispatchEvent(startEvent);_this.dispatchEvent(endEvent);}
function touchstart(event){if(_this.enabled===false)return;event.preventDefault();switch(event.touches.length){case 1:_state=STATE.TOUCH_ROTATE;_moveCurr.copy(getMouseOnCircle(event.touches[0].pageX,event.touches[0].pageY));_movePrev.copy(_moveCurr);break;default:_state=STATE.TOUCH_ZOOM_PAN;var dx=event.touches[0].pageX-event.touches[1].pageX;var dy=event.touches[0].pageY-event.touches[1].pageY;_touchZoomDistanceEnd=_touchZoomDistanceStart=Math.sqrt(dx*dx+dy*dy);var x=(event.touches[0].pageX+event.touches[1].pageX)/2;var y=(event.touches[0].pageY+event.touches[1].pageY)/2;_panStart.copy(getMouseOnScreen(x,y));_panEnd.copy(_panStart);break;}
_this.dispatchEvent(startEvent);}
function touchmove(event){if(_this.enabled===false)return;event.preventDefault();event.stopPropagation();switch(event.touches.length){case 1:_movePrev.copy(_moveCurr);_moveCurr.copy(getMouseOnCircle(event.touches[0].pageX,event.touches[0].pageY));break;default:var dx=event.touches[0].pageX-event.touches[1].pageX;var dy=event.touches[0].pageY-event.touches[1].pageY;_touchZoomDistanceEnd=Math.sqrt(dx*dx+dy*dy);var x=(event.touches[0].pageX+event.touches[1].pageX)/2;var y=(event.touches[0].pageY+event.touches[1].pageY)/2;_panEnd.copy(getMouseOnScreen(x,y));break;}}
function touchend(event){if(_this.enabled===false)return;switch(event.touches.length){case 0:_state=STATE.NONE;break;case 1:_state=STATE.TOUCH_ROTATE;_moveCurr.copy(getMouseOnCircle(event.touches[0].pageX,event.touches[0].pageY));_movePrev.copy(_moveCurr);break;}
_this.dispatchEvent(endEvent);}
function contextmenu(event){if(_this.enabled===false)return;event.preventDefault();}
this.dispose=function(){this.domElement.removeEventListener('contextmenu',contextmenu,false);this.domElement.removeEventListener('mousedown',mousedown,false);this.domElement.removeEventListener('wheel',mousewheel,false);this.domElement.removeEventListener('touchstart',touchstart,false);this.domElement.removeEventListener('touchend',touchend,false);this.domElement.removeEventListener('touchmove',touchmove,false);document.removeEventListener('mousemove',mousemove,false);document.removeEventListener('mouseup',mouseup,false);window.removeEventListener('keydown',keydown,false);window.removeEventListener('keyup',keyup,false);};this.domElement.addEventListener('contextmenu',contextmenu,false);this.domElement.addEventListener('mousedown',mousedown,false);this.domElement.addEventListener('wheel',mousewheel,false);this.domElement.addEventListener('touchstart',touchstart,false);this.domElement.addEventListener('touchend',touchend,false);this.domElement.addEventListener('touchmove',touchmove,false);window.addEventListener('keydown',keydown,false);window.addEventListener('keyup',keyup,false);this.handleResize();this.update();};THREE.TrackballControls.prototype=Object.create(THREE.EventDispatcher.prototype);THREE.TrackballControls.prototype.constructor=THREE.TrackballControls;THREE.GLTFLoader=(function(){function GLTFLoader(manager){THREE.Loader.call(this,manager);this.dracoLoader=null;this.ddsLoader=null;}
GLTFLoader.prototype=Object.assign(Object.create(THREE.Loader.prototype),{constructor:GLTFLoader,load:function(url,onLoad,onProgress,onError){var scope=this;var resourcePath;if(this.resourcePath!==''){resourcePath=this.resourcePath;}else if(this.path!==''){resourcePath=this.path;}else{resourcePath=THREE.LoaderUtils.extractUrlBase(url);}
scope.manager.itemStart(url);var _onError=function(e){if(onError){onError(e);}else{console.error(e);}
scope.manager.itemError(url);scope.manager.itemEnd(url);};var loader=new THREE.FileLoader(scope.manager);loader.setPath(this.path);loader.setResponseType('arraybuffer');if(scope.crossOrigin==='use-credentials'){loader.setWithCredentials(true);}
loader.load(url,function(data){try{scope.parse(data,resourcePath,function(gltf){onLoad(gltf);scope.manager.itemEnd(url);},_onError);}catch(e){_onError(e);}},onProgress,_onError);},setDRACOLoader:function(dracoLoader){this.dracoLoader=dracoLoader;return this;},setDDSLoader:function(ddsLoader){this.ddsLoader=ddsLoader;return this;},parse:function(data,path,onLoad,onError){var content;var extensions={};if(typeof data==='string'){content=data;}else{var magic=THREE.LoaderUtils.decodeText(new Uint8Array(data,0,4));if(magic===BINARY_EXTENSION_HEADER_MAGIC){try{extensions[EXTENSIONS.KHR_BINARY_GLTF]=new GLTFBinaryExtension(data);}catch(error){if(onError)onError(error);return;}
content=extensions[EXTENSIONS.KHR_BINARY_GLTF].content;}else{content=THREE.LoaderUtils.decodeText(new Uint8Array(data));}}
var json=JSON.parse(content);if(json.asset===undefined||json.asset.version[0]<2){if(onError)onError(new Error('THREE.GLTFLoader: Unsupported asset. glTF versions >=2.0 are supported.'));return;}
if(json.extensionsUsed){for(var i=0;i<json.extensionsUsed.length;++i){var extensionName=json.extensionsUsed[i];var extensionsRequired=json.extensionsRequired||[];switch(extensionName){case EXTENSIONS.KHR_LIGHTS_PUNCTUAL:extensions[extensionName]=new GLTFLightsExtension(json);break;case EXTENSIONS.KHR_MATERIALS_CLEARCOAT:extensions[extensionName]=new GLTFMaterialsClearcoatExtension();break;case EXTENSIONS.KHR_MATERIALS_UNLIT:extensions[extensionName]=new GLTFMaterialsUnlitExtension();break;case EXTENSIONS.KHR_MATERIALS_PBR_SPECULAR_GLOSSINESS:extensions[extensionName]=new GLTFMaterialsPbrSpecularGlossinessExtension();break;case EXTENSIONS.KHR_DRACO_MESH_COMPRESSION:extensions[extensionName]=new GLTFDracoMeshCompressionExtension(json,this.dracoLoader);break;case EXTENSIONS.MSFT_TEXTURE_DDS:extensions[extensionName]=new GLTFTextureDDSExtension(this.ddsLoader);break;case EXTENSIONS.KHR_TEXTURE_TRANSFORM:extensions[extensionName]=new GLTFTextureTransformExtension();break;case EXTENSIONS.KHR_MESH_QUANTIZATION:extensions[extensionName]=new GLTFMeshQuantizationExtension();break;default:if(extensionsRequired.indexOf(extensionName)>=0){console.warn('THREE.GLTFLoader: Unknown extension &quot;'+extensionName+'&quot;.');}}}}
var parser=new GLTFParser(json,extensions,{path:path||this.resourcePath||'',crossOrigin:this.crossOrigin,manager:this.manager});parser.parse(onLoad,onError);}});function GLTFRegistry(){var objects={};return{get:function(key){return objects[key];},add:function(key,object){objects[key]=object;},remove:function(key){delete objects[key];},removeAll:function(){objects={};}};}
var EXTENSIONS={KHR_BINARY_GLTF:'KHR_binary_glTF',KHR_DRACO_MESH_COMPRESSION:'KHR_draco_mesh_compression',KHR_LIGHTS_PUNCTUAL:'KHR_lights_punctual',KHR_MATERIALS_CLEARCOAT:'KHR_materials_clearcoat',KHR_MATERIALS_PBR_SPECULAR_GLOSSINESS:'KHR_materials_pbrSpecularGlossiness',KHR_MATERIALS_UNLIT:'KHR_materials_unlit',KHR_TEXTURE_TRANSFORM:'KHR_texture_transform',KHR_MESH_QUANTIZATION:'KHR_mesh_quantization',MSFT_TEXTURE_DDS:'MSFT_texture_dds'};function GLTFTextureDDSExtension(ddsLoader){if(!ddsLoader){throw new Error('THREE.GLTFLoader: Attempting to load .dds texture without importing THREE.DDSLoader');}
this.name=EXTENSIONS.MSFT_TEXTURE_DDS;this.ddsLoader=ddsLoader;}
function GLTFLightsExtension(json){this.name=EXTENSIONS.KHR_LIGHTS_PUNCTUAL;var extension=(json.extensions&&json.extensions[EXTENSIONS.KHR_LIGHTS_PUNCTUAL])||{};this.lightDefs=extension.lights||[];}
GLTFLightsExtension.prototype.loadLight=function(lightIndex){var lightDef=this.lightDefs[lightIndex];var lightNode;var color=new THREE.Color(0xffffff);if(lightDef.color!==undefined)color.fromArray(lightDef.color);var range=lightDef.range!==undefined?lightDef.range:0;switch(lightDef.type){case'directional':lightNode=new THREE.DirectionalLight(color);lightNode.target.position.set(0,0,-1);lightNode.add(lightNode.target);break;case'point':lightNode=new THREE.PointLight(color);lightNode.distance=range;break;case'spot':lightNode=new THREE.SpotLight(color);lightNode.distance=range;lightDef.spot=lightDef.spot||{};lightDef.spot.innerConeAngle=lightDef.spot.innerConeAngle!==undefined?lightDef.spot.innerConeAngle:0;lightDef.spot.outerConeAngle=lightDef.spot.outerConeAngle!==undefined?lightDef.spot.outerConeAngle:Math.PI/4.0;lightNode.angle=lightDef.spot.outerConeAngle;lightNode.penumbra=1.0-lightDef.spot.innerConeAngle/lightDef.spot.outerConeAngle;lightNode.target.position.set(0,0,-1);lightNode.add(lightNode.target);break;default:throw new Error('THREE.GLTFLoader: Unexpected light type, &quot;'+lightDef.type+'&quot;.');}
lightNode.position.set(0,0,0);lightNode.decay=2;if(lightDef.intensity!==undefined)lightNode.intensity=lightDef.intensity;lightNode.name=lightDef.name||('light_'+lightIndex);return Promise.resolve(lightNode);};function GLTFMaterialsUnlitExtension(){this.name=EXTENSIONS.KHR_MATERIALS_UNLIT;}
GLTFMaterialsUnlitExtension.prototype.getMaterialType=function(){return THREE.MeshBasicMaterial;};GLTFMaterialsUnlitExtension.prototype.extendParams=function(materialParams,materialDef,parser){var pending=[];materialParams.color=new THREE.Color(1.0,1.0,1.0);materialParams.opacity=1.0;var metallicRoughness=materialDef.pbrMetallicRoughness;if(metallicRoughness){if(Array.isArray(metallicRoughness.baseColorFactor)){var array=metallicRoughness.baseColorFactor;materialParams.color.fromArray(array);materialParams.opacity=array[3];}
if(metallicRoughness.baseColorTexture!==undefined){pending.push(parser.assignTexture(materialParams,'map',metallicRoughness.baseColorTexture));}}
return Promise.all(pending);};function GLTFMaterialsClearcoatExtension(){this.name=EXTENSIONS.KHR_MATERIALS_CLEARCOAT;}
GLTFMaterialsClearcoatExtension.prototype.getMaterialType=function(){return THREE.MeshPhysicalMaterial;};GLTFMaterialsClearcoatExtension.prototype.extendParams=function(materialParams,materialDef,parser){var pending=[];var extension=materialDef.extensions[this.name];if(extension.clearcoatFactor!==undefined){materialParams.clearcoat=extension.clearcoatFactor;}
if(extension.clearcoatTexture!==undefined){pending.push(parser.assignTexture(materialParams,'clearcoatMap',extension.clearcoatTexture));}
if(extension.clearcoatRoughnessFactor!==undefined){materialParams.clearcoatRoughness=extension.clearcoatRoughnessFactor;}
if(extension.clearcoatRoughnessTexture!==undefined){pending.push(parser.assignTexture(materialParams,'clearcoatRoughnessMap',extension.clearcoatRoughnessTexture));}
if(extension.clearcoatNormalTexture!==undefined){pending.push(parser.assignTexture(materialParams,'clearcoatNormalMap',extension.clearcoatNormalTexture));if(extension.clearcoatNormalTexture.scale!==undefined){var scale=extension.clearcoatNormalTexture.scale;materialParams.clearcoatNormalScale=new THREE.Vector2(scale,scale);}}
return Promise.all(pending);};var BINARY_EXTENSION_HEADER_MAGIC='glTF';var BINARY_EXTENSION_HEADER_LENGTH=12;var BINARY_EXTENSION_CHUNK_TYPES={JSON:0x4E4F534A,BIN:0x004E4942};function GLTFBinaryExtension(data){this.name=EXTENSIONS.KHR_BINARY_GLTF;this.content=null;this.body=null;var headerView=new DataView(data,0,BINARY_EXTENSION_HEADER_LENGTH);this.header={magic:THREE.LoaderUtils.decodeText(new Uint8Array(data.slice(0,4))),version:headerView.getUint32(4,true),length:headerView.getUint32(8,true)};if(this.header.magic!==BINARY_EXTENSION_HEADER_MAGIC){throw new Error('THREE.GLTFLoader: Unsupported glTF-Binary header.');}else if(this.header.version<2.0){throw new Error('THREE.GLTFLoader: Legacy binary file detected.');}
var chunkView=new DataView(data,BINARY_EXTENSION_HEADER_LENGTH);var chunkIndex=0;while(chunkIndex<chunkView.byteLength){var chunkLength=chunkView.getUint32(chunkIndex,true);chunkIndex+=4;var chunkType=chunkView.getUint32(chunkIndex,true);chunkIndex+=4;if(chunkType===BINARY_EXTENSION_CHUNK_TYPES.JSON){var contentArray=new Uint8Array(data,BINARY_EXTENSION_HEADER_LENGTH+chunkIndex,chunkLength);this.content=THREE.LoaderUtils.decodeText(contentArray);}else if(chunkType===BINARY_EXTENSION_CHUNK_TYPES.BIN){var byteOffset=BINARY_EXTENSION_HEADER_LENGTH+chunkIndex;this.body=data.slice(byteOffset,byteOffset+chunkLength);}
chunkIndex+=chunkLength;}
if(this.content===null){throw new Error('THREE.GLTFLoader: JSON content not found.');}}
function GLTFDracoMeshCompressionExtension(json,dracoLoader){if(!dracoLoader){throw new Error('THREE.GLTFLoader: No DRACOLoader instance provided.');}
this.name=EXTENSIONS.KHR_DRACO_MESH_COMPRESSION;this.json=json;this.dracoLoader=dracoLoader;this.dracoLoader.preload();}
GLTFDracoMeshCompressionExtension.prototype.decodePrimitive=function(primitive,parser){var json=this.json;var dracoLoader=this.dracoLoader;var bufferViewIndex=primitive.extensions[this.name].bufferView;var gltfAttributeMap=primitive.extensions[this.name].attributes;var threeAttributeMap={};var attributeNormalizedMap={};var attributeTypeMap={};for(var attributeName in gltfAttributeMap){var threeAttributeName=ATTRIBUTES[attributeName]||attributeName.toLowerCase();threeAttributeMap[threeAttributeName]=gltfAttributeMap[attributeName];}
for(attributeName in primitive.attributes){var threeAttributeName=ATTRIBUTES[attributeName]||attributeName.toLowerCase();if(gltfAttributeMap[attributeName]!==undefined){var accessorDef=json.accessors[primitive.attributes[attributeName]];var componentType=WEBGL_COMPONENT_TYPES[accessorDef.componentType];attributeTypeMap[threeAttributeName]=componentType;attributeNormalizedMap[threeAttributeName]=accessorDef.normalized===true;}}
return parser.getDependency('bufferView',bufferViewIndex).then(function(bufferView){return new Promise(function(resolve){dracoLoader.decodeDracoFile(bufferView,function(geometry){for(var attributeName in geometry.attributes){var attribute=geometry.attributes[attributeName];var normalized=attributeNormalizedMap[attributeName];if(normalized!==undefined)attribute.normalized=normalized;}
resolve(geometry);},threeAttributeMap,attributeTypeMap);});});};function GLTFTextureTransformExtension(){this.name=EXTENSIONS.KHR_TEXTURE_TRANSFORM;}
GLTFTextureTransformExtension.prototype.extendTexture=function(texture,transform){texture=texture.clone();if(transform.offset!==undefined){texture.offset.fromArray(transform.offset);}
if(transform.rotation!==undefined){texture.rotation=transform.rotation;}
if(transform.scale!==undefined){texture.repeat.fromArray(transform.scale);}
if(transform.texCoord!==undefined){console.warn('THREE.GLTFLoader: Custom UV sets in &quot;'+this.name+'&quot; extension not yet supported.');}
texture.needsUpdate=true;return texture;};function GLTFMeshStandardSGMaterial(params){THREE.MeshStandardMaterial.call(this);this.isGLTFSpecularGlossinessMaterial=true;var specularMapParsFragmentChunk=['#ifdef USE_SPECULARMAP','	uniform sampler2D specularMap;','#endif'].join('\n');var glossinessMapParsFragmentChunk=['#ifdef USE_GLOSSINESSMAP','	uniform sampler2D glossinessMap;','#endif'].join('\n');var specularMapFragmentChunk=['vec3 specularFactor = specular;','#ifdef USE_SPECULARMAP','	vec4 texelSpecular = texture2D( specularMap, vUv );','	texelSpecular = sRGBToLinear( texelSpecular );','	// reads channel RGB, compatible with a glTF Specular-Glossiness (RGBA) texture','	specularFactor *= texelSpecular.rgb;','#endif'].join('\n');var glossinessMapFragmentChunk=['float glossinessFactor = glossiness;','#ifdef USE_GLOSSINESSMAP','	vec4 texelGlossiness = texture2D( glossinessMap, vUv );','	// reads channel A, compatible with a glTF Specular-Glossiness (RGBA) texture','	glossinessFactor *= texelGlossiness.a;','#endif'].join('\n');var lightPhysicalFragmentChunk=['PhysicalMaterial material;','material.diffuseColor = diffuseColor.rgb;','vec3 dxy = max( abs( dFdx( geometryNormal ) ), abs( dFdy( geometryNormal ) ) );','float geometryRoughness = max( max( dxy.x, dxy.y ), dxy.z );','material.specularRoughness = max( 1.0 - glossinessFactor, 0.0525 );// 0.0525 corresponds to the base mip of a 256 cubemap.','material.specularRoughness += geometryRoughness;','material.specularRoughness = min( material.specularRoughness, 1.0 );','material.specularColor = specularFactor.rgb;',].join('\n');var uniforms={specular:{value:new THREE.Color().setHex(0xffffff)},glossiness:{value:1},specularMap:{value:null},glossinessMap:{value:null}};this._extraUniforms=uniforms;this.onBeforeCompile=function(shader){for(var uniformName in uniforms){shader.uniforms[uniformName]=uniforms[uniformName];}
shader.fragmentShader=shader.fragmentShader.replace('uniform float roughness;','uniform vec3 specular;');shader.fragmentShader=shader.fragmentShader.replace('uniform float metalness;','uniform float glossiness;');shader.fragmentShader=shader.fragmentShader.replace('#include <roughnessmap_pars_fragment>',specularMapParsFragmentChunk);shader.fragmentShader=shader.fragmentShader.replace('#include <metalnessmap_pars_fragment>',glossinessMapParsFragmentChunk);shader.fragmentShader=shader.fragmentShader.replace('#include <roughnessmap_fragment>',specularMapFragmentChunk);shader.fragmentShader=shader.fragmentShader.replace('#include <metalnessmap_fragment>',glossinessMapFragmentChunk);shader.fragmentShader=shader.fragmentShader.replace('#include <lights_physical_fragment>',lightPhysicalFragmentChunk);};Object.defineProperties(this,{specular:{get:function(){return uniforms.specular.value;},set:function(v){uniforms.specular.value=v;}},specularMap:{get:function(){return uniforms.specularMap.value;},set:function(v){uniforms.specularMap.value=v;}},glossiness:{get:function(){return uniforms.glossiness.value;},set:function(v){uniforms.glossiness.value=v;}},glossinessMap:{get:function(){return uniforms.glossinessMap.value;},set:function(v){uniforms.glossinessMap.value=v;if(v){this.defines.USE_GLOSSINESSMAP='';this.defines.USE_ROUGHNESSMAP='';}else{delete this.defines.USE_ROUGHNESSMAP;delete this.defines.USE_GLOSSINESSMAP;}}}});delete this.metalness;delete this.roughness;delete this.metalnessMap;delete this.roughnessMap;this.setValues(params);}
GLTFMeshStandardSGMaterial.prototype=Object.create(THREE.MeshStandardMaterial.prototype);GLTFMeshStandardSGMaterial.prototype.constructor=GLTFMeshStandardSGMaterial;GLTFMeshStandardSGMaterial.prototype.copy=function(source){THREE.MeshStandardMaterial.prototype.copy.call(this,source);this.specularMap=source.specularMap;this.specular.copy(source.specular);this.glossinessMap=source.glossinessMap;this.glossiness=source.glossiness;delete this.metalness;delete this.roughness;delete this.metalnessMap;delete this.roughnessMap;return this;};function GLTFMaterialsPbrSpecularGlossinessExtension(){return{name:EXTENSIONS.KHR_MATERIALS_PBR_SPECULAR_GLOSSINESS,specularGlossinessParams:['color','map','lightMap','lightMapIntensity','aoMap','aoMapIntensity','emissive','emissiveIntensity','emissiveMap','bumpMap','bumpScale','normalMap','normalMapType','displacementMap','displacementScale','displacementBias','specularMap','specular','glossinessMap','glossiness','alphaMap','envMap','envMapIntensity','refractionRatio',],getMaterialType:function(){return GLTFMeshStandardSGMaterial;},extendParams:function(materialParams,materialDef,parser){var pbrSpecularGlossiness=materialDef.extensions[this.name];materialParams.color=new THREE.Color(1.0,1.0,1.0);materialParams.opacity=1.0;var pending=[];if(Array.isArray(pbrSpecularGlossiness.diffuseFactor)){var array=pbrSpecularGlossiness.diffuseFactor;materialParams.color.fromArray(array);materialParams.opacity=array[3];}
if(pbrSpecularGlossiness.diffuseTexture!==undefined){pending.push(parser.assignTexture(materialParams,'map',pbrSpecularGlossiness.diffuseTexture));}
materialParams.emissive=new THREE.Color(0.0,0.0,0.0);materialParams.glossiness=pbrSpecularGlossiness.glossinessFactor!==undefined?pbrSpecularGlossiness.glossinessFactor:1.0;materialParams.specular=new THREE.Color(1.0,1.0,1.0);if(Array.isArray(pbrSpecularGlossiness.specularFactor)){materialParams.specular.fromArray(pbrSpecularGlossiness.specularFactor);}
if(pbrSpecularGlossiness.specularGlossinessTexture!==undefined){var specGlossMapDef=pbrSpecularGlossiness.specularGlossinessTexture;pending.push(parser.assignTexture(materialParams,'glossinessMap',specGlossMapDef));pending.push(parser.assignTexture(materialParams,'specularMap',specGlossMapDef));}
return Promise.all(pending);},createMaterial:function(materialParams){var material=new GLTFMeshStandardSGMaterial(materialParams);material.fog=true;material.color=materialParams.color;material.map=materialParams.map===undefined?null:materialParams.map;material.lightMap=null;material.lightMapIntensity=1.0;material.aoMap=materialParams.aoMap===undefined?null:materialParams.aoMap;material.aoMapIntensity=1.0;material.emissive=materialParams.emissive;material.emissiveIntensity=1.0;material.emissiveMap=materialParams.emissiveMap===undefined?null:materialParams.emissiveMap;material.bumpMap=materialParams.bumpMap===undefined?null:materialParams.bumpMap;material.bumpScale=1;material.normalMap=materialParams.normalMap===undefined?null:materialParams.normalMap;material.normalMapType=THREE.TangentSpaceNormalMap;if(materialParams.normalScale)material.normalScale=materialParams.normalScale;material.displacementMap=null;material.displacementScale=1;material.displacementBias=0;material.specularMap=materialParams.specularMap===undefined?null:materialParams.specularMap;material.specular=materialParams.specular;material.glossinessMap=materialParams.glossinessMap===undefined?null:materialParams.glossinessMap;material.glossiness=materialParams.glossiness;material.alphaMap=null;material.envMap=materialParams.envMap===undefined?null:materialParams.envMap;material.envMapIntensity=1.0;material.refractionRatio=0.98;return material;},};}
function GLTFMeshQuantizationExtension(){this.name=EXTENSIONS.KHR_MESH_QUANTIZATION;}
function GLTFCubicSplineInterpolant(parameterPositions,sampleValues,sampleSize,resultBuffer){THREE.Interpolant.call(this,parameterPositions,sampleValues,sampleSize,resultBuffer);}
GLTFCubicSplineInterpolant.prototype=Object.create(THREE.Interpolant.prototype);GLTFCubicSplineInterpolant.prototype.constructor=GLTFCubicSplineInterpolant;GLTFCubicSplineInterpolant.prototype.copySampleValue_=function(index){var result=this.resultBuffer,values=this.sampleValues,valueSize=this.valueSize,offset=index*valueSize*3+valueSize;for(var i=0;i!==valueSize;i++){result[i]=values[offset+i];}
return result;};GLTFCubicSplineInterpolant.prototype.beforeStart_=GLTFCubicSplineInterpolant.prototype.copySampleValue_;GLTFCubicSplineInterpolant.prototype.afterEnd_=GLTFCubicSplineInterpolant.prototype.copySampleValue_;GLTFCubicSplineInterpolant.prototype.interpolate_=function(i1,t0,t,t1){var result=this.resultBuffer;var values=this.sampleValues;var stride=this.valueSize;var stride2=stride*2;var stride3=stride*3;var td=t1-t0;var p=(t-t0)/td;var pp=p*p;var ppp=pp*p;var offset1=i1*stride3;var offset0=offset1-stride3;var s2=-2*ppp+3*pp;var s3=ppp-pp;var s0=1-s2;var s1=s3-pp+p;for(var i=0;i!==stride;i++){var p0=values[offset0+i+stride];var m0=values[offset0+i+stride2]*td;var p1=values[offset1+i+stride];var m1=values[offset1+i]*td;result[i]=s0*p0+s1*m0+s2*p1+s3*m1;}
return result;};var WEBGL_CONSTANTS={FLOAT:5126,FLOAT_MAT3:35675,FLOAT_MAT4:35676,FLOAT_VEC2:35664,FLOAT_VEC3:35665,FLOAT_VEC4:35666,LINEAR:9729,REPEAT:10497,SAMPLER_2D:35678,POINTS:0,LINES:1,LINE_LOOP:2,LINE_STRIP:3,TRIANGLES:4,TRIANGLE_STRIP:5,TRIANGLE_FAN:6,UNSIGNED_BYTE:5121,UNSIGNED_SHORT:5123};var WEBGL_COMPONENT_TYPES={5120:Int8Array,5121:Uint8Array,5122:Int16Array,5123:Uint16Array,5125:Uint32Array,5126:Float32Array};var WEBGL_FILTERS={9728:THREE.NearestFilter,9729:THREE.LinearFilter,9984:THREE.NearestMipmapNearestFilter,9985:THREE.LinearMipmapNearestFilter,9986:THREE.NearestMipmapLinearFilter,9987:THREE.LinearMipmapLinearFilter};var WEBGL_WRAPPINGS={33071:THREE.ClampToEdgeWrapping,33648:THREE.MirroredRepeatWrapping,10497:THREE.RepeatWrapping};var WEBGL_TYPE_SIZES={'SCALAR':1,'VEC2':2,'VEC3':3,'VEC4':4,'MAT2':4,'MAT3':9,'MAT4':16};var ATTRIBUTES={POSITION:'position',NORMAL:'normal',TANGENT:'tangent',TEXCOORD_0:'uv',TEXCOORD_1:'uv2',COLOR_0:'color',WEIGHTS_0:'skinWeight',JOINTS_0:'skinIndex',};var PATH_PROPERTIES={scale:'scale',translation:'position',rotation:'quaternion',weights:'morphTargetInfluences'};var INTERPOLATION={CUBICSPLINE:undefined,LINEAR:THREE.InterpolateLinear,STEP:THREE.InterpolateDiscrete};var ALPHA_MODES={OPAQUE:'OPAQUE',MASK:'MASK',BLEND:'BLEND'};var MIME_TYPE_FORMATS={'image/png':THREE.RGBAFormat,'image/jpeg':THREE.RGBFormat};function resolveURL(url,path){if(typeof url!=='string'||url==='')return'';if(/^https?:\/\//i.test(path)&&/^\//.test(url)){path=path.replace(/(^https?:\/\/[^\/]+).*/i,'$1');}
if(/^(https?:)?\/\//i.test(url))return url;if(/^data:.*,.*$/i.test(url))return url;if(/^blob:.*$/i.test(url))return url;return path+url;}
function createDefaultMaterial(cache){if(cache['DefaultMaterial']===undefined){cache['DefaultMaterial']=new THREE.MeshStandardMaterial({color:0xFFFFFF,emissive:0x000000,metalness:1,roughness:1,transparent:false,depthTest:true,side:THREE.FrontSide});}
return cache['DefaultMaterial'];}
function addUnknownExtensionsToUserData(knownExtensions,object,objectDef){for(var name in objectDef.extensions){if(knownExtensions[name]===undefined){object.userData.gltfExtensions=object.userData.gltfExtensions||{};object.userData.gltfExtensions[name]=objectDef.extensions[name];}}}
function assignExtrasToUserData(object,gltfDef){if(gltfDef.extras!==undefined){if(typeof gltfDef.extras==='object'){Object.assign(object.userData,gltfDef.extras);}else{console.warn('THREE.GLTFLoader: Ignoring primitive type .extras, '+gltfDef.extras);}}}
function addMorphTargets(geometry,targets,parser){var hasMorphPosition=false;var hasMorphNormal=false;for(var i=0,il=targets.length;i<il;i++){var target=targets[i];if(target.POSITION!==undefined)hasMorphPosition=true;if(target.NORMAL!==undefined)hasMorphNormal=true;if(hasMorphPosition&&hasMorphNormal)break;}
if(!hasMorphPosition&&!hasMorphNormal)return Promise.resolve(geometry);var pendingPositionAccessors=[];var pendingNormalAccessors=[];for(var i=0,il=targets.length;i<il;i++){var target=targets[i];if(hasMorphPosition){var pendingAccessor=target.POSITION!==undefined?parser.getDependency('accessor',target.POSITION):geometry.attributes.position;pendingPositionAccessors.push(pendingAccessor);}
if(hasMorphNormal){var pendingAccessor=target.NORMAL!==undefined?parser.getDependency('accessor',target.NORMAL):geometry.attributes.normal;pendingNormalAccessors.push(pendingAccessor);}}
return Promise.all([Promise.all(pendingPositionAccessors),Promise.all(pendingNormalAccessors)]).then(function(accessors){var morphPositions=accessors[0];var morphNormals=accessors[1];if(hasMorphPosition)geometry.morphAttributes.position=morphPositions;if(hasMorphNormal)geometry.morphAttributes.normal=morphNormals;geometry.morphTargetsRelative=true;return geometry;});}
function updateMorphTargets(mesh,meshDef){mesh.updateMorphTargets();if(meshDef.weights!==undefined){for(var i=0,il=meshDef.weights.length;i<il;i++){mesh.morphTargetInfluences[i]=meshDef.weights[i];}}
if(meshDef.extras&&Array.isArray(meshDef.extras.targetNames)){var targetNames=meshDef.extras.targetNames;if(mesh.morphTargetInfluences.length===targetNames.length){mesh.morphTargetDictionary={};for(var i=0,il=targetNames.length;i<il;i++){mesh.morphTargetDictionary[targetNames[i]]=i;}}else{console.warn('THREE.GLTFLoader: Invalid extras.targetNames length. Ignoring names.');}}}
function createPrimitiveKey(primitiveDef){var dracoExtension=primitiveDef.extensions&&primitiveDef.extensions[EXTENSIONS.KHR_DRACO_MESH_COMPRESSION];var geometryKey;if(dracoExtension){geometryKey='draco:'+dracoExtension.bufferView
+':'+dracoExtension.indices
+':'+createAttributesKey(dracoExtension.attributes);}else{geometryKey=primitiveDef.indices+':'+createAttributesKey(primitiveDef.attributes)+':'+primitiveDef.mode;}
return geometryKey;}
function createAttributesKey(attributes){var attributesKey='';var keys=Object.keys(attributes).sort();for(var i=0,il=keys.length;i<il;i++){attributesKey+=keys[i]+':'+attributes[keys[i]]+';';}
return attributesKey;}
function GLTFParser(json,extensions,options){this.json=json||{};this.extensions=extensions||{};this.options=options||{};this.cache=new GLTFRegistry();this.primitiveCache={};this.textureLoader=new THREE.TextureLoader(this.options.manager);this.textureLoader.setCrossOrigin(this.options.crossOrigin);this.fileLoader=new THREE.FileLoader(this.options.manager);this.fileLoader.setResponseType('arraybuffer');if(this.options.crossOrigin==='use-credentials'){this.fileLoader.setWithCredentials(true);}}
GLTFParser.prototype.parse=function(onLoad,onError){var parser=this;var json=this.json;var extensions=this.extensions;this.cache.removeAll();this.markDefs();Promise.all([this.getDependencies('scene'),this.getDependencies('animation'),this.getDependencies('camera'),]).then(function(dependencies){var result={scene:dependencies[0][json.scene||0],scenes:dependencies[0],animations:dependencies[1],cameras:dependencies[2],asset:json.asset,parser:parser,userData:{}};addUnknownExtensionsToUserData(extensions,result,json);assignExtrasToUserData(result,json);onLoad(result);}).catch(onError);};GLTFParser.prototype.markDefs=function(){var nodeDefs=this.json.nodes||[];var skinDefs=this.json.skins||[];var meshDefs=this.json.meshes||[];var meshReferences={};var meshUses={};for(var skinIndex=0,skinLength=skinDefs.length;skinIndex<skinLength;skinIndex++){var joints=skinDefs[skinIndex].joints;for(var i=0,il=joints.length;i<il;i++){nodeDefs[joints[i]].isBone=true;}}
for(var nodeIndex=0,nodeLength=nodeDefs.length;nodeIndex<nodeLength;nodeIndex++){var nodeDef=nodeDefs[nodeIndex];if(nodeDef.mesh!==undefined){if(meshReferences[nodeDef.mesh]===undefined){meshReferences[nodeDef.mesh]=meshUses[nodeDef.mesh]=0;}
meshReferences[nodeDef.mesh]++;if(nodeDef.skin!==undefined){meshDefs[nodeDef.mesh].isSkinnedMesh=true;}}}
this.json.meshReferences=meshReferences;this.json.meshUses=meshUses;};GLTFParser.prototype.getDependency=function(type,index){var cacheKey=type+':'+index;var dependency=this.cache.get(cacheKey);if(!dependency){switch(type){case'scene':dependency=this.loadScene(index);break;case'node':dependency=this.loadNode(index);break;case'mesh':dependency=this.loadMesh(index);break;case'accessor':dependency=this.loadAccessor(index);break;case'bufferView':dependency=this.loadBufferView(index);break;case'buffer':dependency=this.loadBuffer(index);break;case'material':dependency=this.loadMaterial(index);break;case'texture':dependency=this.loadTexture(index);break;case'skin':dependency=this.loadSkin(index);break;case'animation':dependency=this.loadAnimation(index);break;case'camera':dependency=this.loadCamera(index);break;case'light':dependency=this.extensions[EXTENSIONS.KHR_LIGHTS_PUNCTUAL].loadLight(index);break;default:throw new Error('Unknown type: '+type);}
this.cache.add(cacheKey,dependency);}
return dependency;};GLTFParser.prototype.getDependencies=function(type){var dependencies=this.cache.get(type);if(!dependencies){var parser=this;var defs=this.json[type+(type==='mesh'?'es':'s')]||[];dependencies=Promise.all(defs.map(function(def,index){return parser.getDependency(type,index);}));this.cache.add(type,dependencies);}
return dependencies;};GLTFParser.prototype.loadBuffer=function(bufferIndex){var bufferDef=this.json.buffers[bufferIndex];var loader=this.fileLoader;if(bufferDef.type&&bufferDef.type!=='arraybuffer'){throw new Error('THREE.GLTFLoader: '+bufferDef.type+' buffer type is not supported.');}
if(bufferDef.uri===undefined&&bufferIndex===0){return Promise.resolve(this.extensions[EXTENSIONS.KHR_BINARY_GLTF].body);}
var options=this.options;return new Promise(function(resolve,reject){loader.load(resolveURL(bufferDef.uri,options.path),resolve,undefined,function(){reject(new Error('THREE.GLTFLoader: Failed to load buffer &quot;'+bufferDef.uri+'&quot;.'));});});};GLTFParser.prototype.loadBufferView=function(bufferViewIndex){var bufferViewDef=this.json.bufferViews[bufferViewIndex];return this.getDependency('buffer',bufferViewDef.buffer).then(function(buffer){var byteLength=bufferViewDef.byteLength||0;var byteOffset=bufferViewDef.byteOffset||0;return buffer.slice(byteOffset,byteOffset+byteLength);});};GLTFParser.prototype.loadAccessor=function(accessorIndex){var parser=this;var json=this.json;var accessorDef=this.json.accessors[accessorIndex];if(accessorDef.bufferView===undefined&&accessorDef.sparse===undefined){return Promise.resolve(null);}
var pendingBufferViews=[];if(accessorDef.bufferView!==undefined){pendingBufferViews.push(this.getDependency('bufferView',accessorDef.bufferView));}else{pendingBufferViews.push(null);}
if(accessorDef.sparse!==undefined){pendingBufferViews.push(this.getDependency('bufferView',accessorDef.sparse.indices.bufferView));pendingBufferViews.push(this.getDependency('bufferView',accessorDef.sparse.values.bufferView));}
return Promise.all(pendingBufferViews).then(function(bufferViews){var bufferView=bufferViews[0];var itemSize=WEBGL_TYPE_SIZES[accessorDef.type];var TypedArray=WEBGL_COMPONENT_TYPES[accessorDef.componentType];var elementBytes=TypedArray.BYTES_PER_ELEMENT;var itemBytes=elementBytes*itemSize;var byteOffset=accessorDef.byteOffset||0;var byteStride=accessorDef.bufferView!==undefined?json.bufferViews[accessorDef.bufferView].byteStride:undefined;var normalized=accessorDef.normalized===true;var array,bufferAttribute;if(byteStride&&byteStride!==itemBytes){var ibSlice=Math.floor(byteOffset/byteStride);var ibCacheKey='InterleavedBuffer:'+accessorDef.bufferView+':'+accessorDef.componentType+':'+ibSlice+':'+accessorDef.count;var ib=parser.cache.get(ibCacheKey);if(!ib){array=new TypedArray(bufferView,ibSlice*byteStride,accessorDef.count*byteStride/elementBytes);ib=new THREE.InterleavedBuffer(array,byteStride/elementBytes);parser.cache.add(ibCacheKey,ib);}
bufferAttribute=new THREE.InterleavedBufferAttribute(ib,itemSize,(byteOffset%byteStride)/elementBytes,normalized);}else{if(bufferView===null){array=new TypedArray(accessorDef.count*itemSize);}else{array=new TypedArray(bufferView,byteOffset,accessorDef.count*itemSize);}
bufferAttribute=new THREE.BufferAttribute(array,itemSize,normalized);}
if(accessorDef.sparse!==undefined){var itemSizeIndices=WEBGL_TYPE_SIZES.SCALAR;var TypedArrayIndices=WEBGL_COMPONENT_TYPES[accessorDef.sparse.indices.componentType];var byteOffsetIndices=accessorDef.sparse.indices.byteOffset||0;var byteOffsetValues=accessorDef.sparse.values.byteOffset||0;var sparseIndices=new TypedArrayIndices(bufferViews[1],byteOffsetIndices,accessorDef.sparse.count*itemSizeIndices);var sparseValues=new TypedArray(bufferViews[2],byteOffsetValues,accessorDef.sparse.count*itemSize);if(bufferView!==null){bufferAttribute=new THREE.BufferAttribute(bufferAttribute.array.slice(),bufferAttribute.itemSize,bufferAttribute.normalized);}
for(var i=0,il=sparseIndices.length;i<il;i++){var index=sparseIndices[i];bufferAttribute.setX(index,sparseValues[i*itemSize]);if(itemSize>=2)bufferAttribute.setY(index,sparseValues[i*itemSize+1]);if(itemSize>=3)bufferAttribute.setZ(index,sparseValues[i*itemSize+2]);if(itemSize>=4)bufferAttribute.setW(index,sparseValues[i*itemSize+3]);if(itemSize>=5)throw new Error('THREE.GLTFLoader: Unsupported itemSize in sparse BufferAttribute.');}}
return bufferAttribute;});};GLTFParser.prototype.loadTexture=function(textureIndex){var parser=this;var json=this.json;var options=this.options;var textureLoader=this.textureLoader;var URL=self.URL||self.webkitURL;var textureDef=json.textures[textureIndex];var textureExtensions=textureDef.extensions||{};var source;if(textureExtensions[EXTENSIONS.MSFT_TEXTURE_DDS]){source=json.images[textureExtensions[EXTENSIONS.MSFT_TEXTURE_DDS].source];}else{source=json.images[textureDef.source];}
var sourceURI=source.uri;var isObjectURL=false;if(source.bufferView!==undefined){sourceURI=parser.getDependency('bufferView',source.bufferView).then(function(bufferView){isObjectURL=true;var blob=new Blob([bufferView],{type:source.mimeType});sourceURI=URL.createObjectURL(blob);return sourceURI;});}
return Promise.resolve(sourceURI).then(function(sourceURI){var loader=options.manager.getHandler(sourceURI);if(!loader){loader=textureExtensions[EXTENSIONS.MSFT_TEXTURE_DDS]?parser.extensions[EXTENSIONS.MSFT_TEXTURE_DDS].ddsLoader:textureLoader;}
return new Promise(function(resolve,reject){loader.load(resolveURL(sourceURI,options.path),resolve,undefined,reject);});}).then(function(texture){if(isObjectURL===true){URL.revokeObjectURL(sourceURI);}
texture.flipY=false;if(textureDef.name)texture.name=textureDef.name;if(source.mimeType in MIME_TYPE_FORMATS){texture.format=MIME_TYPE_FORMATS[source.mimeType];}
var samplers=json.samplers||{};var sampler=samplers[textureDef.sampler]||{};texture.magFilter=WEBGL_FILTERS[sampler.magFilter]||THREE.LinearFilter;texture.minFilter=WEBGL_FILTERS[sampler.minFilter]||THREE.LinearMipmapLinearFilter;texture.wrapS=WEBGL_WRAPPINGS[sampler.wrapS]||THREE.RepeatWrapping;texture.wrapT=WEBGL_WRAPPINGS[sampler.wrapT]||THREE.RepeatWrapping;return texture;});};GLTFParser.prototype.assignTexture=function(materialParams,mapName,mapDef){var parser=this;return this.getDependency('texture',mapDef.index).then(function(texture){if(!texture.isCompressedTexture){switch(mapName){case'aoMap':case'emissiveMap':case'metalnessMap':case'normalMap':case'roughnessMap':texture.format=THREE.RGBFormat;break;}}
if(mapDef.texCoord!==undefined&&mapDef.texCoord!=0&&!(mapName==='aoMap'&&mapDef.texCoord==1)){console.warn('THREE.GLTFLoader: Custom UV set '+mapDef.texCoord+' for texture '+mapName+' not yet supported.');}
if(parser.extensions[EXTENSIONS.KHR_TEXTURE_TRANSFORM]){var transform=mapDef.extensions!==undefined?mapDef.extensions[EXTENSIONS.KHR_TEXTURE_TRANSFORM]:undefined;if(transform){texture=parser.extensions[EXTENSIONS.KHR_TEXTURE_TRANSFORM].extendTexture(texture,transform);}}
materialParams[mapName]=texture;});};GLTFParser.prototype.assignFinalMaterial=function(mesh){var geometry=mesh.geometry;var material=mesh.material;var useVertexTangents=geometry.attributes.tangent!==undefined;var useVertexColors=geometry.attributes.color!==undefined;var useFlatShading=geometry.attributes.normal===undefined;var useSkinning=mesh.isSkinnedMesh===true;var useMorphTargets=Object.keys(geometry.morphAttributes).length>0;var useMorphNormals=useMorphTargets&&geometry.morphAttributes.normal!==undefined;if(mesh.isPoints){var cacheKey='PointsMaterial:'+material.uuid;var pointsMaterial=this.cache.get(cacheKey);if(!pointsMaterial){pointsMaterial=new THREE.PointsMaterial();THREE.Material.prototype.copy.call(pointsMaterial,material);pointsMaterial.color.copy(material.color);pointsMaterial.map=material.map;pointsMaterial.sizeAttenuation=false;this.cache.add(cacheKey,pointsMaterial);}
material=pointsMaterial;}else if(mesh.isLine){var cacheKey='LineBasicMaterial:'+material.uuid;var lineMaterial=this.cache.get(cacheKey);if(!lineMaterial){lineMaterial=new THREE.LineBasicMaterial();THREE.Material.prototype.copy.call(lineMaterial,material);lineMaterial.color.copy(material.color);this.cache.add(cacheKey,lineMaterial);}
material=lineMaterial;}
if(useVertexTangents||useVertexColors||useFlatShading||useSkinning||useMorphTargets){var cacheKey='ClonedMaterial:'+material.uuid+':';if(material.isGLTFSpecularGlossinessMaterial)cacheKey+='specular-glossiness:';if(useSkinning)cacheKey+='skinning:';if(useVertexTangents)cacheKey+='vertex-tangents:';if(useVertexColors)cacheKey+='vertex-colors:';if(useFlatShading)cacheKey+='flat-shading:';if(useMorphTargets)cacheKey+='morph-targets:';if(useMorphNormals)cacheKey+='morph-normals:';var cachedMaterial=this.cache.get(cacheKey);if(!cachedMaterial){cachedMaterial=material.clone();if(useSkinning)cachedMaterial.skinning=true;if(useVertexTangents)cachedMaterial.vertexTangents=true;if(useVertexColors)cachedMaterial.vertexColors=true;if(useFlatShading)cachedMaterial.flatShading=true;if(useMorphTargets)cachedMaterial.morphTargets=true;if(useMorphNormals)cachedMaterial.morphNormals=true;this.cache.add(cacheKey,cachedMaterial);}
material=cachedMaterial;}
if(material.aoMap&&geometry.attributes.uv2===undefined&&geometry.attributes.uv!==undefined){geometry.setAttribute('uv2',new THREE.BufferAttribute(geometry.attributes.uv.array,2));}
if(material.normalScale&&!useVertexTangents){material.normalScale.y=-material.normalScale.y;}
if(material.clearcoatNormalScale&&!useVertexTangents){material.clearcoatNormalScale.y=-material.clearcoatNormalScale.y;}
mesh.material=material;};GLTFParser.prototype.loadMaterial=function(materialIndex){var parser=this;var json=this.json;var extensions=this.extensions;var materialDef=json.materials[materialIndex];var materialType;var materialParams={};var materialExtensions=materialDef.extensions||{};var pending=[];if(materialExtensions[EXTENSIONS.KHR_MATERIALS_PBR_SPECULAR_GLOSSINESS]){var sgExtension=extensions[EXTENSIONS.KHR_MATERIALS_PBR_SPECULAR_GLOSSINESS];materialType=sgExtension.getMaterialType();pending.push(sgExtension.extendParams(materialParams,materialDef,parser));}else if(materialExtensions[EXTENSIONS.KHR_MATERIALS_UNLIT]){var kmuExtension=extensions[EXTENSIONS.KHR_MATERIALS_UNLIT];materialType=kmuExtension.getMaterialType();pending.push(kmuExtension.extendParams(materialParams,materialDef,parser));}else{materialType=THREE.MeshStandardMaterial;var metallicRoughness=materialDef.pbrMetallicRoughness||{};materialParams.color=new THREE.Color(1.0,1.0,1.0);materialParams.opacity=1.0;if(Array.isArray(metallicRoughness.baseColorFactor)){var array=metallicRoughness.baseColorFactor;materialParams.color.fromArray(array);materialParams.opacity=array[3];}
if(metallicRoughness.baseColorTexture!==undefined){pending.push(parser.assignTexture(materialParams,'map',metallicRoughness.baseColorTexture));}
materialParams.metalness=metallicRoughness.metallicFactor!==undefined?metallicRoughness.metallicFactor:1.0;materialParams.roughness=metallicRoughness.roughnessFactor!==undefined?metallicRoughness.roughnessFactor:1.0;if(metallicRoughness.metallicRoughnessTexture!==undefined){pending.push(parser.assignTexture(materialParams,'metalnessMap',metallicRoughness.metallicRoughnessTexture));pending.push(parser.assignTexture(materialParams,'roughnessMap',metallicRoughness.metallicRoughnessTexture));}}
if(materialDef.doubleSided===true){materialParams.side=THREE.DoubleSide;}
var alphaMode=materialDef.alphaMode||ALPHA_MODES.OPAQUE;if(alphaMode===ALPHA_MODES.BLEND){materialParams.transparent=true;materialParams.depthWrite=false;}else{materialParams.transparent=false;if(alphaMode===ALPHA_MODES.MASK){materialParams.alphaTest=materialDef.alphaCutoff!==undefined?materialDef.alphaCutoff:0.5;}}
if(materialDef.normalTexture!==undefined&&materialType!==THREE.MeshBasicMaterial){pending.push(parser.assignTexture(materialParams,'normalMap',materialDef.normalTexture));materialParams.normalScale=new THREE.Vector2(1,1);if(materialDef.normalTexture.scale!==undefined){materialParams.normalScale.set(materialDef.normalTexture.scale,materialDef.normalTexture.scale);}}
if(materialDef.occlusionTexture!==undefined&&materialType!==THREE.MeshBasicMaterial){pending.push(parser.assignTexture(materialParams,'aoMap',materialDef.occlusionTexture));if(materialDef.occlusionTexture.strength!==undefined){materialParams.aoMapIntensity=materialDef.occlusionTexture.strength;}}
if(materialDef.emissiveFactor!==undefined&&materialType!==THREE.MeshBasicMaterial){materialParams.emissive=new THREE.Color().fromArray(materialDef.emissiveFactor);}
if(materialDef.emissiveTexture!==undefined&&materialType!==THREE.MeshBasicMaterial){pending.push(parser.assignTexture(materialParams,'emissiveMap',materialDef.emissiveTexture));}
if(materialExtensions[EXTENSIONS.KHR_MATERIALS_CLEARCOAT]){var clearcoatExtension=extensions[EXTENSIONS.KHR_MATERIALS_CLEARCOAT];materialType=clearcoatExtension.getMaterialType();pending.push(clearcoatExtension.extendParams(materialParams,{extensions:materialExtensions},parser));}
return Promise.all(pending).then(function(){var material;if(materialType===GLTFMeshStandardSGMaterial){material=extensions[EXTENSIONS.KHR_MATERIALS_PBR_SPECULAR_GLOSSINESS].createMaterial(materialParams);}else{material=new materialType(materialParams);}
if(materialDef.name)material.name=materialDef.name;if(material.map)material.map.encoding=THREE.sRGBEncoding;if(material.emissiveMap)material.emissiveMap.encoding=THREE.sRGBEncoding;assignExtrasToUserData(material,materialDef);if(materialDef.extensions)addUnknownExtensionsToUserData(extensions,material,materialDef);return material;});};function computeBounds(geometry,primitiveDef,parser){var attributes=primitiveDef.attributes;var box=new THREE.Box3();if(attributes.POSITION!==undefined){var accessor=parser.json.accessors[attributes.POSITION];var min=accessor.min;var max=accessor.max;if(min!==undefined&&max!==undefined){box.set(new THREE.Vector3(min[0],min[1],min[2]),new THREE.Vector3(max[0],max[1],max[2]));}else{console.warn('THREE.GLTFLoader: Missing min/max properties for accessor POSITION.');return;}}else{return;}
var targets=primitiveDef.targets;if(targets!==undefined){var maxDisplacement=new THREE.Vector3();var vector=new THREE.Vector3();for(var i=0,il=targets.length;i<il;i++){var target=targets[i];if(target.POSITION!==undefined){var accessor=parser.json.accessors[target.POSITION];var min=accessor.min;var max=accessor.max;if(min!==undefined&&max!==undefined){vector.setX(Math.max(Math.abs(min[0]),Math.abs(max[0])));vector.setY(Math.max(Math.abs(min[1]),Math.abs(max[1])));vector.setZ(Math.max(Math.abs(min[2]),Math.abs(max[2])));maxDisplacement.max(vector);}else{console.warn('THREE.GLTFLoader: Missing min/max properties for accessor POSITION.');}}}
box.expandByVector(maxDisplacement);}
geometry.boundingBox=box;var sphere=new THREE.Sphere();box.getCenter(sphere.center);sphere.radius=box.min.distanceTo(box.max)/2;geometry.boundingSphere=sphere;}
function addPrimitiveAttributes(geometry,primitiveDef,parser){var attributes=primitiveDef.attributes;var pending=[];function assignAttributeAccessor(accessorIndex,attributeName){return parser.getDependency('accessor',accessorIndex).then(function(accessor){geometry.setAttribute(attributeName,accessor);});}
for(var gltfAttributeName in attributes){var threeAttributeName=ATTRIBUTES[gltfAttributeName]||gltfAttributeName.toLowerCase();if(threeAttributeName in geometry.attributes)continue;pending.push(assignAttributeAccessor(attributes[gltfAttributeName],threeAttributeName));}
if(primitiveDef.indices!==undefined&&!geometry.index){var accessor=parser.getDependency('accessor',primitiveDef.indices).then(function(accessor){geometry.setIndex(accessor);});pending.push(accessor);}
assignExtrasToUserData(geometry,primitiveDef);computeBounds(geometry,primitiveDef,parser);return Promise.all(pending).then(function(){return primitiveDef.targets!==undefined?addMorphTargets(geometry,primitiveDef.targets,parser):geometry;});}
function toTrianglesDrawMode(geometry,drawMode){var index=geometry.getIndex();if(index===null){var indices=[];var position=geometry.getAttribute('position');if(position!==undefined){for(var i=0;i<position.count;i++){indices.push(i);}
geometry.setIndex(indices);index=geometry.getIndex();}else{console.error('THREE.GLTFLoader.toTrianglesDrawMode(): Undefined position attribute. Processing not possible.');return geometry;}}
var numberOfTriangles=index.count-2;var newIndices=[];if(drawMode===THREE.TriangleFanDrawMode){for(var i=1;i<=numberOfTriangles;i++){newIndices.push(index.getX(0));newIndices.push(index.getX(i));newIndices.push(index.getX(i+1));}}else{for(var i=0;i<numberOfTriangles;i++){if(i%2===0){newIndices.push(index.getX(i));newIndices.push(index.getX(i+1));newIndices.push(index.getX(i+2));}else{newIndices.push(index.getX(i+2));newIndices.push(index.getX(i+1));newIndices.push(index.getX(i));}}}
if((newIndices.length/3)!==numberOfTriangles){console.error('THREE.GLTFLoader.toTrianglesDrawMode(): Unable to generate correct amount of triangles.');}
var newGeometry=geometry.clone();newGeometry.setIndex(newIndices);return newGeometry;}
GLTFParser.prototype.loadGeometries=function(primitives){var parser=this;var extensions=this.extensions;var cache=this.primitiveCache;function createDracoPrimitive(primitive){return extensions[EXTENSIONS.KHR_DRACO_MESH_COMPRESSION].decodePrimitive(primitive,parser).then(function(geometry){return addPrimitiveAttributes(geometry,primitive,parser);});}
var pending=[];for(var i=0,il=primitives.length;i<il;i++){var primitive=primitives[i];var cacheKey=createPrimitiveKey(primitive);var cached=cache[cacheKey];if(cached){pending.push(cached.promise);}else{var geometryPromise;if(primitive.extensions&&primitive.extensions[EXTENSIONS.KHR_DRACO_MESH_COMPRESSION]){geometryPromise=createDracoPrimitive(primitive);}else{geometryPromise=addPrimitiveAttributes(new THREE.BufferGeometry(),primitive,parser);}
cache[cacheKey]={primitive:primitive,promise:geometryPromise};pending.push(geometryPromise);}}
return Promise.all(pending);};GLTFParser.prototype.loadMesh=function(meshIndex){var parser=this;var json=this.json;var meshDef=json.meshes[meshIndex];var primitives=meshDef.primitives;var pending=[];for(var i=0,il=primitives.length;i<il;i++){var material=primitives[i].material===undefined?createDefaultMaterial(this.cache):this.getDependency('material',primitives[i].material);pending.push(material);}
pending.push(parser.loadGeometries(primitives));return Promise.all(pending).then(function(results){var materials=results.slice(0,results.length-1);var geometries=results[results.length-1];var meshes=[];for(var i=0,il=geometries.length;i<il;i++){var geometry=geometries[i];var primitive=primitives[i];var mesh;var material=materials[i];if(primitive.mode===WEBGL_CONSTANTS.TRIANGLES||primitive.mode===WEBGL_CONSTANTS.TRIANGLE_STRIP||primitive.mode===WEBGL_CONSTANTS.TRIANGLE_FAN||primitive.mode===undefined){mesh=meshDef.isSkinnedMesh===true?new THREE.SkinnedMesh(geometry,material):new THREE.Mesh(geometry,material);if(mesh.isSkinnedMesh===true&&!mesh.geometry.attributes.skinWeight.normalized){mesh.normalizeSkinWeights();}
if(primitive.mode===WEBGL_CONSTANTS.TRIANGLE_STRIP){mesh.geometry=toTrianglesDrawMode(mesh.geometry,THREE.TriangleStripDrawMode);}else if(primitive.mode===WEBGL_CONSTANTS.TRIANGLE_FAN){mesh.geometry=toTrianglesDrawMode(mesh.geometry,THREE.TriangleFanDrawMode);}}else if(primitive.mode===WEBGL_CONSTANTS.LINES){mesh=new THREE.LineSegments(geometry,material);}else if(primitive.mode===WEBGL_CONSTANTS.LINE_STRIP){mesh=new THREE.Line(geometry,material);}else if(primitive.mode===WEBGL_CONSTANTS.LINE_LOOP){mesh=new THREE.LineLoop(geometry,material);}else if(primitive.mode===WEBGL_CONSTANTS.POINTS){mesh=new THREE.Points(geometry,material);}else{throw new Error('THREE.GLTFLoader: Primitive mode unsupported: '+primitive.mode);}
if(Object.keys(mesh.geometry.morphAttributes).length>0){updateMorphTargets(mesh,meshDef);}
mesh.name=meshDef.name||('mesh_'+meshIndex);if(geometries.length>1)mesh.name+='_'+i;assignExtrasToUserData(mesh,meshDef);parser.assignFinalMaterial(mesh);meshes.push(mesh);}
if(meshes.length===1){return meshes[0];}
var group=new THREE.Group();for(var i=0,il=meshes.length;i<il;i++){group.add(meshes[i]);}
return group;});};GLTFParser.prototype.loadCamera=function(cameraIndex){var camera;var cameraDef=this.json.cameras[cameraIndex];var params=cameraDef[cameraDef.type];if(!params){console.warn('THREE.GLTFLoader: Missing camera parameters.');return;}
if(cameraDef.type==='perspective'){camera=new THREE.PerspectiveCamera(THREE.MathUtils.radToDeg(params.yfov),params.aspectRatio||1,params.znear||1,params.zfar||2e6);}else if(cameraDef.type==='orthographic'){camera=new THREE.OrthographicCamera(params.xmag/-2,params.xmag/2,params.ymag/2,params.ymag/-2,params.znear,params.zfar);}
if(cameraDef.name)camera.name=cameraDef.name;assignExtrasToUserData(camera,cameraDef);return Promise.resolve(camera);};GLTFParser.prototype.loadSkin=function(skinIndex){var skinDef=this.json.skins[skinIndex];var skinEntry={joints:skinDef.joints};if(skinDef.inverseBindMatrices===undefined){return Promise.resolve(skinEntry);}
return this.getDependency('accessor',skinDef.inverseBindMatrices).then(function(accessor){skinEntry.inverseBindMatrices=accessor;return skinEntry;});};GLTFParser.prototype.loadAnimation=function(animationIndex){var json=this.json;var animationDef=json.animations[animationIndex];var pendingNodes=[];var pendingInputAccessors=[];var pendingOutputAccessors=[];var pendingSamplers=[];var pendingTargets=[];for(var i=0,il=animationDef.channels.length;i<il;i++){var channel=animationDef.channels[i];var sampler=animationDef.samplers[channel.sampler];var target=channel.target;var name=target.node!==undefined?target.node:target.id;var input=animationDef.parameters!==undefined?animationDef.parameters[sampler.input]:sampler.input;var output=animationDef.parameters!==undefined?animationDef.parameters[sampler.output]:sampler.output;pendingNodes.push(this.getDependency('node',name));pendingInputAccessors.push(this.getDependency('accessor',input));pendingOutputAccessors.push(this.getDependency('accessor',output));pendingSamplers.push(sampler);pendingTargets.push(target);}
return Promise.all([Promise.all(pendingNodes),Promise.all(pendingInputAccessors),Promise.all(pendingOutputAccessors),Promise.all(pendingSamplers),Promise.all(pendingTargets)]).then(function(dependencies){var nodes=dependencies[0];var inputAccessors=dependencies[1];var outputAccessors=dependencies[2];var samplers=dependencies[3];var targets=dependencies[4];var tracks=[];for(var i=0,il=nodes.length;i<il;i++){var node=nodes[i];var inputAccessor=inputAccessors[i];var outputAccessor=outputAccessors[i];var sampler=samplers[i];var target=targets[i];if(node===undefined)continue;node.updateMatrix();node.matrixAutoUpdate=true;var TypedKeyframeTrack;switch(PATH_PROPERTIES[target.path]){case PATH_PROPERTIES.weights:TypedKeyframeTrack=THREE.NumberKeyframeTrack;break;case PATH_PROPERTIES.rotation:TypedKeyframeTrack=THREE.QuaternionKeyframeTrack;break;case PATH_PROPERTIES.position:case PATH_PROPERTIES.scale:default:TypedKeyframeTrack=THREE.VectorKeyframeTrack;break;}
var targetName=node.name?node.name:node.uuid;var interpolation=sampler.interpolation!==undefined?INTERPOLATION[sampler.interpolation]:THREE.InterpolateLinear;var targetNames=[];if(PATH_PROPERTIES[target.path]===PATH_PROPERTIES.weights){node.traverse(function(object){if(object.isMesh===true&&object.morphTargetInfluences){targetNames.push(object.name?object.name:object.uuid);}});}else{targetNames.push(targetName);}
var outputArray=outputAccessor.array;if(outputAccessor.normalized){var scale;if(outputArray.constructor===Int8Array){scale=1/127;}else if(outputArray.constructor===Uint8Array){scale=1/255;}else if(outputArray.constructor==Int16Array){scale=1/32767;}else if(outputArray.constructor===Uint16Array){scale=1/65535;}else{throw new Error('THREE.GLTFLoader: Unsupported output accessor component type.');}
var scaled=new Float32Array(outputArray.length);for(var j=0,jl=outputArray.length;j<jl;j++){scaled[j]=outputArray[j]*scale;}
outputArray=scaled;}
for(var j=0,jl=targetNames.length;j<jl;j++){var track=new TypedKeyframeTrack(targetNames[j]+'.'+PATH_PROPERTIES[target.path],inputAccessor.array,outputArray,interpolation);if(sampler.interpolation==='CUBICSPLINE'){track.createInterpolant=function InterpolantFactoryMethodGLTFCubicSpline(result){return new GLTFCubicSplineInterpolant(this.times,this.values,this.getValueSize()/3,result);};track.createInterpolant.isInterpolantFactoryMethodGLTFCubicSpline=true;}
tracks.push(track);}}
var name=animationDef.name?animationDef.name:'animation_'+animationIndex;return new THREE.AnimationClip(name,undefined,tracks);});};GLTFParser.prototype.loadNode=function(nodeIndex){var json=this.json;var extensions=this.extensions;var parser=this;var meshReferences=json.meshReferences;var meshUses=json.meshUses;var nodeDef=json.nodes[nodeIndex];return(function(){var pending=[];if(nodeDef.mesh!==undefined){pending.push(parser.getDependency('mesh',nodeDef.mesh).then(function(mesh){var node;if(meshReferences[nodeDef.mesh]>1){var instanceNum=meshUses[nodeDef.mesh]++;node=mesh.clone();node.name+='_instance_'+instanceNum;}else{node=mesh;}
if(nodeDef.weights!==undefined){node.traverse(function(o){if(!o.isMesh)return;for(var i=0,il=nodeDef.weights.length;i<il;i++){o.morphTargetInfluences[i]=nodeDef.weights[i];}});}
return node;}));}
if(nodeDef.camera!==undefined){pending.push(parser.getDependency('camera',nodeDef.camera));}
if(nodeDef.extensions&&nodeDef.extensions[EXTENSIONS.KHR_LIGHTS_PUNCTUAL]&&nodeDef.extensions[EXTENSIONS.KHR_LIGHTS_PUNCTUAL].light!==undefined){pending.push(parser.getDependency('light',nodeDef.extensions[EXTENSIONS.KHR_LIGHTS_PUNCTUAL].light));}
return Promise.all(pending);}()).then(function(objects){var node;if(nodeDef.isBone===true){node=new THREE.Bone();}else if(objects.length>1){node=new THREE.Group();}else if(objects.length===1){node=objects[0];}else{node=new THREE.Object3D();}
if(node!==objects[0]){for(var i=0,il=objects.length;i<il;i++){node.add(objects[i]);}}
if(nodeDef.name){node.userData.name=nodeDef.name;node.name=THREE.PropertyBinding.sanitizeNodeName(nodeDef.name);}
assignExtrasToUserData(node,nodeDef);if(nodeDef.extensions)addUnknownExtensionsToUserData(extensions,node,nodeDef);if(nodeDef.matrix!==undefined){var matrix=new THREE.Matrix4();matrix.fromArray(nodeDef.matrix);node.applyMatrix4(matrix);}else{if(nodeDef.translation!==undefined){node.position.fromArray(nodeDef.translation);}
if(nodeDef.rotation!==undefined){node.quaternion.fromArray(nodeDef.rotation);}
if(nodeDef.scale!==undefined){node.scale.fromArray(nodeDef.scale);}}
return node;});};GLTFParser.prototype.loadScene=function(){function buildNodeHierachy(nodeId,parentObject,json,parser){var nodeDef=json.nodes[nodeId];return parser.getDependency('node',nodeId).then(function(node){if(nodeDef.skin===undefined)return node;var skinEntry;return parser.getDependency('skin',nodeDef.skin).then(function(skin){skinEntry=skin;var pendingJoints=[];for(var i=0,il=skinEntry.joints.length;i<il;i++){pendingJoints.push(parser.getDependency('node',skinEntry.joints[i]));}
return Promise.all(pendingJoints);}).then(function(jointNodes){node.traverse(function(mesh){if(!mesh.isMesh)return;var bones=[];var boneInverses=[];for(var j=0,jl=jointNodes.length;j<jl;j++){var jointNode=jointNodes[j];if(jointNode){bones.push(jointNode);var mat=new THREE.Matrix4();if(skinEntry.inverseBindMatrices!==undefined){mat.fromArray(skinEntry.inverseBindMatrices.array,j*16);}
boneInverses.push(mat);}else{console.warn('THREE.GLTFLoader: Joint &quot;%s&quot; could not be found.',skinEntry.joints[j]);}}
mesh.bind(new THREE.Skeleton(bones,boneInverses),mesh.matrixWorld);});return node;});}).then(function(node){parentObject.add(node);var pending=[];if(nodeDef.children){var children=nodeDef.children;for(var i=0,il=children.length;i<il;i++){var child=children[i];pending.push(buildNodeHierachy(child,node,json,parser));}}
return Promise.all(pending);});}
return function loadScene(sceneIndex){var json=this.json;var extensions=this.extensions;var sceneDef=this.json.scenes[sceneIndex];var parser=this;var scene=new THREE.Group();if(sceneDef.name)scene.name=sceneDef.name;assignExtrasToUserData(scene,sceneDef);if(sceneDef.extensions)addUnknownExtensionsToUserData(extensions,scene,sceneDef);var nodeIds=sceneDef.nodes||[];var pending=[];for(var i=0,il=nodeIds.length;i<il;i++){pending.push(buildNodeHierachy(nodeIds[i],scene,json,parser));}
return Promise.all(pending).then(function(){return scene;});};}();return GLTFLoader;})();var camera,controls,scene,renderer,tracklight;function autoFit(obj,camera,controls){const boundingBox=new THREE.Box3().setFromObject(obj);const boundingSphere=new THREE.Sphere();boundingBox.getBoundingSphere((target=boundingSphere));const scale=1.0;const angularSize=((camera.fov*Math.PI)/180)*scale;const distanceToCamera=boundingSphere.radius/Math.tan(angularSize);const len=Math.sqrt(Math.pow(distanceToCamera,2)+
Math.pow(distanceToCamera,2)+
Math.pow(distanceToCamera,2));camera.position.set(len,len,len);controls.update();camera.lookAt(boundingSphere.center);controls.target.set(boundingSphere.center.x,boundingSphere.center.y,boundingSphere.center.z);camera.updateProjectionMatrix();}
function centerControls(obj,camera,controls){const boundingBox=new THREE.Box3().setFromObject(obj);const boundingSphere=new THREE.Sphere();boundingBox.getBoundingSphere((target=boundingSphere));controls.update();controls.target.set(boundingSphere.center.x,boundingSphere.center.y,boundingSphere.center.z);}
function init(){scene=new THREE.Scene();scene.background=new THREE.Color(0xffffff);tracklight=new THREE.DirectionalLight(0xffffff,1.75);scene.add(tracklight);base64_data=&quot;Z2xURgIAAAA4JQIATBsAAEpTT057InNjZW5lIjowLCJzY2VuZXMiOlt7Im5vZGVzIjpbMF0sImV4dHJhcyI6eyJmaWxlX3BhdGgiOiJhbC5vYmoiLCJmaWxlX25hbWUiOiJhbC5vYmoifX1dLCJhc3NldCI6eyJ2ZXJzaW9uIjoiMi4wIiwiZ2VuZXJhdG9yIjoiaHR0cHM6Ly9naXRodWIuY29tL21pa2VkaC90cmltZXNoIn0sImFjY2Vzc29ycyI6W3siY29tcG9uZW50VHlwZSI6NTEyNSwidHlwZSI6IlNDQUxBUiIsImJ1ZmZlclZpZXciOjAsImNvdW50IjoxNDQsIm1heCI6WzQ3XSwibWluIjpbMF19LHsiY29tcG9uZW50VHlwZSI6NTEyNiwidHlwZSI6IlZFQzMiLCJieXRlT2Zmc2V0IjowLCJidWZmZXJWaWV3IjoxLCJjb3VudCI6NDgsIm1heCI6WzAuMjQxNDExOTk4ODY3OTg4NiwyLjA1NjM2NDA1OTQ0ODI0MiwwLjE5NjIzMDk5MjY3NDgyNzU4XSwibWluIjpbLTAuMjQxNDExOTk4ODY3OTg4NiwxLjkzMDUyMDA1NzY3ODIyMjcsMC4xNDIwNTUwMDQ4MzUxMjg3OF19LHsiY29tcG9uZW50VHlwZSI6NTEyNSwidHlwZSI6IlNDQUxBUiIsImJ1ZmZlclZpZXciOjIsImNvdW50IjoyMTAsIm1heCI6WzY2XSwibWluIjpbMF19LHsiY29tcG9uZW50VHlwZSI6NTEyNiwidHlwZSI6IlZFQzMiLCJieXRlT2Zmc2V0IjowLCJidWZmZXJWaWV3IjozLCJjb3VudCI6NjcsIm1heCI6WzAuMzY3MTQ4OTk1Mzk5NDc1MSwyLjE3NTg0Mzk1NDA4NjMwMzcsMC4yMjQ4NDk5OTg5NTA5NTgyNV0sIm1pbiI6Wy0wLjM2NzE0ODk5NTM5OTQ3NTEsMS43ODIwMjkwMzI3MDcyMTQ0LC0wLjU5NDM0MzAwNjYxMDg3MDRdfSx7ImNvbXBvbmVudFR5cGUiOjUxMjUsInR5cGUiOiJTQ0FMQVIiLCJidWZmZXJWaWV3Ijo0LCJjb3VudCI6MTcyOCwibWF4IjpbMzE1XSwibWluIjpbMF19LHsiY29tcG9uZW50VHlwZSI6NTEyNiwidHlwZSI6IlZFQzMiLCJieXRlT2Zmc2V0IjowLCJidWZmZXJWaWV3Ijo1LCJjb3VudCI6MzE2LCJtYXgiOlswLjU5MDE3OTAyNjEyNjg2MTYsMi41OTc5NDIxMTM4NzYzNDI4LDAuNTkwNjA2OTg3NDc2MzQ4OV0sIm1pbiI6Wy0wLjU5MDE3OTAyNjEyNjg2MTYsMS45ODU3OTIwNDA4MjQ4OTAxLC0wLjg4ODU0NDk3NjcxMTI3MzJdfSx7ImNvbXBvbmVudFR5cGUiOjUxMjUsInR5cGUiOiJTQ0FMQVIiLCJidWZmZXJWaWV3Ijo2LCJjb3VudCI6MjQzLCJtYXgiOls1MF0sIm1pbiI6WzBdfSx7ImNvbXBvbmVudFR5cGUiOjUxMjYsInR5cGUiOiJWRUMzIiwiYnl0ZU9mZnNldCI6MCwiYnVmZmVyVmlldyI6NywiY291bnQiOjUxLCJtYXgiOlswLjI2Mzg1MzAxMzUxNTQ3MjQsMC43MzU0NTAwMjkzNzMxNjksMC44NzM0NDkwMjc1MzgyOTk2XSwibWluIjpbLTAuMjUzMTc0MDA2OTM4OTM0MywwLjEzNzIyODk5NTU2MTU5OTczLDAuNTY3MjU0MDA2ODYyNjQwNF19LHsiY29tcG9uZW50VHlwZSI6NTEyNSwidHlwZSI6IlNDQUxBUiIsImJ1ZmZlclZpZXciOjgsImNvdW50Ijo4MDg4LCJtYXgiOlsxMzc5XSwibWluIjpbMF19LHsiY29tcG9uZW50VHlwZSI6NTEyNiwidHlwZSI6IlZFQzMiLCJieXRlT2Zmc2V0IjowLCJidWZmZXJWaWV3Ijo5LCJjb3VudCI6MTM4MCwibWF4IjpbMi43MTIzMDg4ODM2NjY5OTIsMi40MTE5MTkxMTY5NzM4NzcsMS4xMTE5NTAwMzk4NjM1ODY0XSwibWluIjpbLTIuNzEyMzA4ODgzNjY2OTkyLC0xLjQ0NDQ4NjAyMTk5NTU0NDQsLTAuNzcyNTc2OTg3NzQzMzc3N119LHsiY29tcG9uZW50VHlwZSI6NTEyNSwidHlwZSI6IlNDQUxBUiIsImJ1ZmZlclZpZXciOjEwLCJjb3VudCI6Njg1OCwibWF4IjpbMTE4Nl0sIm1pbiI6WzBdfSx7ImNvbXBvbmVudFR5cGUiOjUxMjYsInR5cGUiOiJWRUMzIiwiYnl0ZU9mZnNldCI6MCwiYnVmZmVyVmlldyI6MTEsImNvdW50IjoxMTg3LCJtYXgiOlsyLjUwNjAzNTA4OTQ5Mjc5OCwxLjUwOTU1NjA1NTA2ODk2OTcsMC45MjAwODE5NzMwNzU4NjY3XSwibWluIjpbLTIuNTA2MDM1MDg5NDkyNzk4LC0yLjg5MTY2MzA3NDQ5MzQwOCwtMS4xMTE5NTAwMzk4NjM1ODY0XX0seyJjb21wb25lbnRUeXBlIjo1MTI1LCJ0eXBlIjoiU0NBTEFSIiwiYnVmZmVyVmlldyI6MTIsImNvdW50Ijo1ODIsIm1heCI6WzExN10sIm1pbiI6WzBdfSx7ImNvbXBvbmVudFR5cGUiOjUxMjYsInR5cGUiOiJWRUMzIiwiYnl0ZU9mZnNldCI6MCwiYnVmZmVyVmlldyI6MTMsImNvdW50IjoxMTgsIm1heCI6WzEuMDgxMTA0OTkzODIwMTkwNCwtMy4yMDE0NzQ5MDUwMTQwMzgsMC44OTc1OTIwMDgxMTM4NjExXSwibWluIjpbLTEuMDgxMTA0OTkzODIwMTkwNCwtMy4yOTQyMTkwMTcwMjg4MDg2LC0wLjE1OTgxMjAwMzM3NDA5OTczXX0seyJjb21wb25lbnRUeXBlIjo1MTI1LCJ0eXBlIjoiU0NBTEFSIiwiYnVmZmVyVmlldyI6MTQsImNvdW50IjoxNjExLCJtYXgiOlszNjRdLCJtaW4iOlswXX0seyJjb21wb25lbnRUeXBlIjo1MTI2LCJ0eXBlIjoiVkVDMyIsImJ5dGVPZmZzZXQiOjAsImJ1ZmZlclZpZXciOjE1LCJjb3VudCI6MzY1LCJtYXgiOlsyLjQ5MjIzMjA4NDI3NDI5MiwyLjExNDUwMDA0NTc3NjM2NywwLjc2MDY1NzAxMjQ2MjYxNl0sIm1pbiI6Wy0yLjQ5MjIzMjA4NDI3NDI5MiwtMy4yMzgzODA5MDg5NjYwNjQ1LC0wLjc1MDU1NzAwNTQwNTQyNl19LHsiY29tcG9uZW50VHlwZSI6NTEyNSwidHlwZSI6IlNDQUxBUiIsImJ1ZmZlclZpZXciOjE2LCJjb3VudCI6MTk5MiwibWF4IjpbNDQ3XSwibWluIjpbMF19LHsiY29tcG9uZW50VHlwZSI6NTEyNiwidHlwZSI6IlZFQzMiLCJieXRlT2Zmc2V0IjowLCJidWZmZXJWaWV3IjoxNywiY291bnQiOjQ0OCwibWF4IjpbMS4xMDY5Mjg5NDQ1ODc3MDc1LDIuMzMxNDc4MTE4ODk2NDg0NCwwLjk1ODMxNTAxNDgzOTE3MjRdLCJtaW4iOlstMS4xMDY5Mjg5NDQ1ODc3MDc1LC0zLjIzODM4OTAxNTE5Nzc1NCwtMC42MDg0Nzk5NzY2NTQwNTI3XX1dLCJtZXNoZXMiOlt7Im5hbWUiOiJkZWVwZ3JlZW4iLCJleHRyYXMiOnsicHJvY2Vzc2VkIjp0cnVlfSwicHJpbWl0aXZlcyI6W3siYXR0cmlidXRlcyI6eyJQT1NJVElPTiI6MX0sImluZGljZXMiOjAsIm1vZGUiOjQsIm1hdGVyaWFsIjowfV19LHsibmFtZSI6ImJybmhhaXIiLCJleHRyYXMiOnsicHJvY2Vzc2VkIjp0cnVlfSwicHJpbWl0aXZlcyI6W3siYXR0cmlidXRlcyI6eyJQT1NJVElPTiI6M30sImluZGljZXMiOjIsIm1vZGUiOjQsIm1hdGVyaWFsIjoxfV19LHsibmFtZSI6ImJyb3duIiwiZXh0cmFzIjp7InByb2Nlc3NlZCI6dHJ1ZX0sInByaW1pdGl2ZXMiOlt7ImF0dHJpYnV0ZXMiOnsiUE9TSVRJT04iOjV9LCJpbmRpY2VzIjo0LCJtb2RlIjo0LCJtYXRlcmlhbCI6Mn1dfSx7Im5hbWUiOiJmbGRrcmVkIiwiZXh0cmFzIjp7InByb2Nlc3NlZCI6dHJ1ZX0sInByaW1pdGl2ZXMiOlt7ImF0dHJpYnV0ZXMiOnsiUE9TSVRJT04iOjd9LCJpbmRpY2VzIjo2LCJtb2RlIjo0LCJtYXRlcmlhbCI6M31dfSx7Im5hbWUiOiJza2luIiwiZXh0cmFzIjp7InByb2Nlc3NlZCI6dHJ1ZX0sInByaW1pdGl2ZXMiOlt7ImF0dHJpYnV0ZXMiOnsiUE9TSVRJT04iOjl9LCJpbmRpY2VzIjo4LCJtb2RlIjo0LCJtYXRlcmlhbCI6NH1dfSx7Im5hbWUiOiJka2JsdWVfcHVyZSIsImV4dHJhcyI6eyJwcm9jZXNzZWQiOnRydWV9LCJwcmltaXRpdmVzIjpbeyJhdHRyaWJ1dGVzIjp7IlBPU0lUSU9OIjoxMX0sImluZGljZXMiOjEwLCJtb2RlIjo0LCJtYXRlcmlhbCI6NX1dfSx7Im5hbWUiOiJka2dyZXkiLCJleHRyYXMiOnsicHJvY2Vzc2VkIjp0cnVlfSwicHJpbWl0aXZlcyI6W3siYXR0cmlidXRlcyI6eyJQT1NJVElPTiI6MTN9LCJpbmRpY2VzIjoxMiwibW9kZSI6NCwibWF0ZXJpYWwiOjZ9XX0seyJuYW1lIjoid2hpdGUiLCJleHRyYXMiOnsicHJvY2Vzc2VkIjp0cnVlfSwicHJpbWl0aXZlcyI6W3siYXR0cmlidXRlcyI6eyJQT1NJVElPTiI6MTV9LCJpbmRpY2VzIjoxNCwibW9kZSI6NCwibWF0ZXJpYWwiOjd9XX0seyJuYW1lIjoiYmxhY2siLCJleHRyYXMiOnsicHJvY2Vzc2VkIjp0cnVlfSwicHJpbWl0aXZlcyI6W3siYXR0cmlidXRlcyI6eyJQT1NJVElPTiI6MTd9LCJpbmRpY2VzIjoxNiwibW9kZSI6NCwibWF0ZXJpYWwiOjh9XX1dLCJtYXRlcmlhbHMiOlt7InBick1ldGFsbGljUm91Z2huZXNzIjp7ImJhc2VDb2xvckZhY3RvciI6WzAuMCwwLjAxOTYwNzg0MzEzNzI1NDksMC4wMDM5MjE1Njg2Mjc0NTA5OCwxLjBdLCJyb3VnaG5lc3NGYWN0b3IiOjAuMzYzMDA1NTIxMDMxNTQ5Mn0sImRvdWJsZVNpZGVkIjpmYWxzZX0seyJwYnJNZXRhbGxpY1JvdWdobmVzcyI6eyJiYXNlQ29sb3JGYWN0b3IiOlswLjA5MDE5NjA3ODQzMTM3MjU1LDAuMDMxMzcyNTQ5MDE5NjA3ODQsMC4wMTE3NjQ3MDU4ODIzNTI5NDEsMS4wXSwicm91Z2huZXNzRmFjdG9yIjowLjQyMjk5OTI3NDQyODUwMDU1fSwiZG91YmxlU2lkZWQiOmZhbHNlfSx7InBick1ldGFsbGljUm91Z2huZXNzIjp7ImJhc2VDb2xvckZhY3RvciI6WzAuMDg2Mjc0NTA5ODAzOTIxNTcsMC4wMTU2ODYyNzQ1MDk4MDM5MiwwLjAsMS4wXSwicm91Z2huZXNzRmFjdG9yIjowLjQxNDI5MTUyMjY2MzcxNjh9LCJkb3VibGVTaWRlZCI6ZmFsc2V9LHsicGJyTWV0YWxsaWNSb3VnaG5lc3MiOnsiYmFzZUNvbG9yRmFjdG9yIjpbMC4yMjc0NTA5ODAzOTIxNTY4NiwwLjAwNzg0MzEzNzI1NDkwMTk2LDAuMDA3ODQzMTM3MjU0OTAxOTYsMS4wXSwicm91Z2huZXNzRmFjdG9yIjowLjkwMzYwMjAwMzYwOTg0NDh9LCJkb3VibGVTaWRlZCI6ZmFsc2V9LHsicGJyTWV0YWxsaWNSb3VnaG5lc3MiOnsiYmFzZUNvbG9yRmFjdG9yIjpbMC4xMDk4MDM5MjE1Njg2Mjc0NSwwLjAzMTM3MjU0OTAxOTYwNzg0LDAuMDE1Njg2Mjc0NTA5ODAzOTIsMS4wXSwicm91Z2huZXNzRmFjdG9yIjowLjU2MzU0MjY4OTU2NDczMTZ9LCJkb3VibGVTaWRlZCI6ZmFsc2V9LHsicGJyTWV0YWxsaWNSb3VnaG5lc3MiOnsiYmFzZUNvbG9yRmFjdG9yIjpbMC4wLDAuMCwwLjEzMzMzMzMzMzMzMzMzMzMzLDEuMF0sInJvdWdobmVzc0ZhY3RvciI6MC40MTQyOTE1MjI2NjM3MTY4fSwiZG91YmxlU2lkZWQiOmZhbHNlfSx7InBick1ldGFsbGljUm91Z2huZXNzIjp7ImJhc2VDb2xvckZhY3RvciI6WzAuMDIzNTI5NDExNzY0NzA1ODgyLDAuMDIzNTI5NDExNzY0NzA1ODgyLDAuMDIzNTI5NDExNzY0NzA1ODgyLDEuMF0sInJvdWdobmVzc0ZhY3RvciI6MC40MjM3OTg2NTc0MTUwMjE2NX0sImRvdWJsZVNpZGVkIjpmYWxzZX0seyJwYnJNZXRhbGxpY1JvdWdobmVzcyI6eyJiYXNlQ29sb3JGYWN0b3IiOlsxLjAsMS4wLDEuMCwxLjBdLCJyb3VnaG5lc3NGYWN0b3IiOjAuNDIzNzk4NjU3NDE1MDIxNjV9LCJkb3VibGVTaWRlZCI6ZmFsc2V9LHsicGJyTWV0YWxsaWNSb3VnaG5lc3MiOnsiYmFzZUNvbG9yRmFjdG9yIjpbMC4wMDM5MjE1Njg2Mjc0NTA5OCwwLjAwMzkyMTU2ODYyNzQ1MDk4LDAuMDAzOTIxNTY4NjI3NDUwOTgsMS4wXSwicm91Z2huZXNzRmFjdG9yIjowLjMzNDcwNTM2MTA4MjAzNDY0fSwiZG91YmxlU2lkZWQiOmZhbHNlfV0sImNhbWVyYXMiOlt7Im5hbWUiOiJjYW1lcmFfRTA1Q0IxIiwidHlwZSI6InBlcnNwZWN0aXZlIiwicGVyc3BlY3RpdmUiOnsiYXNwZWN0UmF0aW8iOjEuMzMzMzMzMzMzMzMzMzMzMywieWZvdiI6MC43ODUzOTgxNjMzOTc0NDgzLCJ6bmVhciI6MC4wMX19XSwibm9kZXMiOlt7Im5hbWUiOiJ3b3JsZCIsImNoaWxkcmVuIjpbMSwyLDMsNCw1LDYsNyw4LDksMTBdfSx7Im5hbWUiOiJkZWVwZ3JlZW4iLCJtZXNoIjowfSx7Im5hbWUiOiJicm5oYWlyIiwibWVzaCI6MX0seyJuYW1lIjoiYnJvd24iLCJtZXNoIjoyfSx7Im5hbWUiOiJmbGRrcmVkIiwibWVzaCI6M30seyJuYW1lIjoic2tpbiIsIm1lc2giOjR9LHsibmFtZSI6ImRrYmx1ZV9wdXJlIiwibWVzaCI6NX0seyJuYW1lIjoiZGtncmV5IiwibWVzaCI6Nn0seyJuYW1lIjoid2hpdGUiLCJtZXNoIjo3fSx7Im5hbWUiOiJibGFjayIsIm1lc2giOjh9LHsibmFtZSI6ImNhbWVyYV9FMDVDQjEiLCJjYW1lcmEiOjAsIm1hdHJpeCI6WzEuMCwwLjAsLTAuMCwwLjAsMC4wLDEuMCwwLjAsMC4wLDAuMCwwLjAsMS4wLDAuMCwwLjAsLTAuMzQ4MTM4NTAwMDAwMDAwMTMsOC4yMjQ0MTc0OTg5NDI5MDksMS4wXX1dLCJidWZmZXJzIjpbeyJieXRlTGVuZ3RoIjoxMzM1ODR9XSwiYnVmZmVyVmlld3MiOlt7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6MCwiYnl0ZUxlbmd0aCI6NTc2fSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6NTc2LCJieXRlTGVuZ3RoIjo1NzZ9LHsiYnVmZmVyIjowLCJieXRlT2Zmc2V0IjoxMTUyLCJieXRlTGVuZ3RoIjo4NDB9LHsiYnVmZmVyIjowLCJieXRlT2Zmc2V0IjoxOTkyLCJieXRlTGVuZ3RoIjo4MDR9LHsiYnVmZmVyIjowLCJieXRlT2Zmc2V0IjoyNzk2LCJieXRlTGVuZ3RoIjo2OTEyfSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6OTcwOCwiYnl0ZUxlbmd0aCI6Mzc5Mn0seyJidWZmZXIiOjAsImJ5dGVPZmZzZXQiOjEzNTAwLCJieXRlTGVuZ3RoIjo5NzJ9LHsiYnVmZmVyIjowLCJieXRlT2Zmc2V0IjoxNDQ3MiwiYnl0ZUxlbmd0aCI6NjEyfSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6MTUwODQsImJ5dGVMZW5ndGgiOjMyMzUyfSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6NDc0MzYsImJ5dGVMZW5ndGgiOjE2NTYwfSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6NjM5OTYsImJ5dGVMZW5ndGgiOjI3NDMyfSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6OTE0MjgsImJ5dGVMZW5ndGgiOjE0MjQ0fSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6MTA1NjcyLCJieXRlTGVuZ3RoIjoyMzI4fSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6MTA4MDAwLCJieXRlTGVuZ3RoIjoxNDE2fSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6MTA5NDE2LCJieXRlTGVuZ3RoIjo2NDQ0fSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6MTE1ODYwLCJieXRlTGVuZ3RoIjo0MzgwfSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6MTIwMjQwLCJieXRlTGVuZ3RoIjo3OTY4fSx7ImJ1ZmZlciI6MCwiYnl0ZU9mZnNldCI6MTI4MjA4LCJieXRlTGVuZ3RoIjo1Mzc2fV19ICAg0AkCAEJJTgAUAAAAEwAAAAkAAAAQAAAADwAAAAUAAAAMAAAAFwAAAAEAAAANAAAADAAAAAIAAAARAAAAEAAAAAYAAAAVAAAAFAAAAAoAAAATAAAAEgAAAAgAAAAPAAAADgAAAAQAAAAXAAAAFgAAAAAAAAAOAAAADQAAAAMAAAASAAAAEQAAAAcAAAAAAAAAFgAAABUAAAAjAAAAIgAAACwAAAAfAAAAHgAAACgAAAAbAAAAGgAAACQAAAAaAAAAGQAAAC8AAAAeAAAAHQAAACcAAAAiAAAAIQAAACsAAAAjAAAALQAAAC4AAAAgAAAAHwAAACkAAAAcAAAAGwAAACUAAAAZAAAAGAAAAC4AAAAdAAAAHAAAACYAAAAhAAAAIAAAACoAAAAJAAAACgAAABQAAAAFAAAABgAAABAAAAABAAAAAgAAAAwAAAACAAAAAwAAAA0AAAAGAAAABwAAABEAAAAKAAAACwAAABUAAAAIAAAACQAAABMAAAAEAAAABQAAAA8AAAAAAAAAAQAAABcAAAADAAAABAAAAA4AAAAHAAAACAAAABIAAAAVAAAACwAAAAAAAAAsAAAALQAAACMAAAAoAAAAKQAAAB8AAAAkAAAAJQAAABsAAAAvAAAAJAAAABoAAAAnAAAAKAAAAB4AAAArAAAALAAAACIAAAAuAAAAGAAAACMAAAApAAAAKgAAACAAAAAlAAAAJgAAABwAAAAuAAAALwAAABkAAAAmAAAAJwAAAB0AAAAqAAAAKwAAACEAAADaHj2+eJsDQFwAKj7jNlq+UwUDQLiPHD6FeW++LIMBQHWUEz61NHe+Afn+P952ET4hVm++g/j6Py7IFj5I+Vm+dhf4PzsbIj6M1zy+SBv3P4xnMD7Gvx++kUf4PzDYPT7hfAq+4Ev7P3TTRj6xwQK+Pln/P8fwSD6IoAq+2qwBQLqfQz5h/R++YB0DQK1MOD7Y8lq+ppcAQB/4KD5c5F6+VB7/P9LjJz7B4Fq+6BP9P/qZKj7O+0++Bpz7P3hgMD4lIUG+cRv7P++qNz40SzK+mrT7P3eFPj5ccye+cD79P8YZQz5egiO+ak//PxIuRD75hSe+5qwAQKd3QT6pajK+12gBQGyxOz5SRUG+IqkBQPVmND6GG1C+klwBQLCMLT7aHj0+eJsDQFwAKj7jNlo+UwUDQLiPHD6FeW8+LIMBQHWUEz61NHc+Afn+P952ET4hVm8+g/j6Py7IFj5I+Vk+dhf4PzsbIj6M1zw+SBv3P4xnMD7Gvx8+kUf4PzDYPT7hfAo+4Ev7P3TTRj6xwQI+Pln/P8fwSD6IoAo+2qwBQLqfQz5h/R8+YB0DQK1MOD7Y8lo+ppcAQB/4KD5c5F4+VB7/P9LjJz7B4Fo+6BP9P/qZKj7O+08+Bpz7P3hgMD4lIUE+cRv7P++qNz40SzI+mrT7P3eFPj5ccyc+cD79P8YZQz5egiM+ak//PxIuRD75hSc+5qwAQKd3QT6pajI+12gBQGyxOz5SRUE+IqkBQPVmND6GG1A+klwBQLCMLT48AAAAOwAAADMAAAAzAAAAJAAAADwAAAAvAAAAKgAAADcAAAAuAAAAMAAAADEAAAAxAAAALQAAAC4AAAAIAAAABwAAACkAAAApAAAAJwAAAAgAAAA5AAAAOAAAAEIAAAA5AAAAQgAAAEEAAAA5AAAAQQAAACUAAAA5AAAAJQAAACYAAAAUAAAAFQAAABEAAAARAAAAEgAAABQAAAAMAAAAEgAAABEAAAARAAAADQAAAAwAAAAFAAAABAAAAAgAAAAIAAAACQAAAAUAAAAeAAAAHwAAAAQAAAAEAAAABQAAAB4AAAADAAAAAgAAACEAAAADAAAAIQAAACIAAAADAAAAIgAAABgAAAADAAAAGAAAABkAAAAEAAAABgAAAAcAAAAHAAAACAAAAAQAAAANAAAAEQAAABAAAAAQAAAADgAAAA0AAAAXAAAACgAAAA8AAAABAAAAEwAAABsAAAAbAAAAHAAAAAEAAABAAAAAKQAAACUAAAAlAAAAQQAAAEAAAAAvAAAAKwAAACoAAAAwAAAALwAAADcAAAA3AAAANgAAADAAAAAyAAAAMQAAADUAAAA1AAAANAAAADIAAAAtAAAAMQAAADIAAAAyAAAALAAAAC0AAAAJAAAACAAAACcAAAAnAAAAKAAAAAkAAAAoAAAAJwAAAD8AAAA/AAAAPgAAACgAAAAVAAAAFgAAABAAAAAQAAAAEQAAABUAAAADAAAAGQAAABoAAAADAAAAGgAAABMAAAADAAAAEwAAAAEAAAALAAAADgAAABAAAAAQAAAADwAAAAsAAAAgAAAABgAAAAQAAAAEAAAAHwAAACAAAAAAAAAAAQAAABwAAAAcAAAAHQAAAAAAAAAhAAAAAgAAAAYAAAAGAAAAIAAAACEAAAAKAAAACwAAAA8AAAAWAAAAFwAAAA8AAAAPAAAAEAAAABYAAAA9AAAAPAAAACQAAAAkAAAAIwAAAD0AAAA/AAAAJwAAACkAAAApAAAAQAAAAD8AAAAvAAAAMAAAAC4AAAAuAAAAKwAAAC8AAAAkAAAAMwAAADoAAAAkAAAAOgAAADkAAAAkAAAAOQAAACYAAAAxAAAAMAAAADYAAAA2AAAANQAAADEAAAC3DbO+jBIAQGBYHr6EKbK+PgMJQBmsaL5g556+t5YIQC/f6r4Z5qy+Lq4IQOoku77lY0e+kLzzP8V2D7/oLmm+hxnkP6slEb+RuFe+3rAIQHU6CL8AAAAAp5EIQHzxEb8AAAAAG0vyP8h7Fb8AAAAAM4zlP90mGL8a96a+lPgCQKLrAr32soW+4bYEQCAJez34io696iUEQBQ/Zj5jRBK+hNkEQLOzSD58nk++w7gFQApl4T3/IqC+S3cHQFA2ZT2Re2q+4h0JQPPk2j2QMRe+uAUHQHfbVT40EmG9xjUIQCpzYz5Wnri+KSD9P1gdmb5QjZe9B0ELQGk6Sz6d8S2+2xgKQPWiJj4Cf3i+YcUKQJM5lj0Oo5i+AYkJQGKh1jwDzrq+51H/P3Zrwb5VhLu+Y2MAQDJXvr70+ru+jpD/PzXSur4HC7e+uw75P3XKc75GlLa+Y0X5P8k+aL7zdLa+xO34PwxYYr4kKpS+ZVblP7IRDL//6Ya+6rHvP3RABr/ptpS+tYv1PwETAL96w6W+RYD5Pw1t6L6Rnae+qvP4P2Zr5b63DbM+jBIAQGBYHr6EKbI+PgMJQBmsaL5g554+t5YIQC/f6r4Z5qw+Lq4IQOoku77lY0c+kLzzP8V2D7/oLmk+hxnkP6slEb+RuFc+3rAIQHU6CL8a96Y+lPgCQKLrAr32soU+4bYEQCAJez34io496iUEQBQ/Zj5jRBI+hNkEQLOzSD58nk8+w7gFQApl4T3/IqA+S3cHQFA2ZT2Re2o+4h0JQPPk2j2QMRc+uAUHQHfbVT40EmE9xjUIQCpzYz5Wnrg+KSD9P1gdmb5QjZc9B0ELQGk6Sz6d8S0+2xgKQPWiJj4Cf3g+YcUKQJM5lj0Oo5g+AYkJQGKh1jwDzro+51H/P3Zrwb5VhLs+Y2MAQDJXvr70+rs+jpD/PzXSur4HC7c+uw75P3XKc75GlLY+Y0X5P8k+aL7zdLY+xO34PwxYYr4kKpQ+ZVblP7IRDL//6YY+6rHvP3RABr/ptpQ+tYv1PwETAL96w6U+RYD5Pw1t6L6Rnac+qvP4P2Zr5b7RAAAA0AAAAMwAAADMAAAAywAAANEAAAC8AAAAuwAAADoBAAA6AQAAOwEAALwAAAC2AAAAtQAAALQAAAC0AAAAswAAALYAAAAjAAAAIgAAANIAAADSAAAAzwAAACMAAADLAAAAzAAAAMgAAADIAAAAxwAAAMsAAADDAAAAugAAAL0AAAC9AAAAxAAAAMMAAAC7AAAAugAAAL4AAAC1AAAAtgAAAA0AAAANAAAADAAAALUAAACvAAAAsAAAAK0AAACtAAAAqgAAAK8AAACXAAAAlgAAAB4AAAAeAAAAHAAAAJcAAACRAAAAjwAAAI4AAACOAAAAkAAAAJEAAAADAAAACwAAAGkAAABpAAAAaAAAAAMAAAAfAAAAIQAAACIAAAAiAAAAIwAAAB8AAAAaAAAAGQAAAJcAAACXAAAAnAAAABoAAAAWAAAAEQAAABIAAAASAAAAFwAAABYAAAATAAAAEgAAABAAAAAMAAAADQAAAGkAAABpAAAAagAAAAwAAAACAAAABQAAAAYAAAAGAAAABwAAAAIAAAABAAAABAAAAAUAAAAFAAAAAgAAAAEAAAAHAAAACwAAAAMAAAADAAAAAgAAAAcAAAARAAAADwAAABAAAAAQAAAAEgAAABEAAAASAAAAEwAAABQAAAAUAAAAFQAAABIAAAAXAAAAGAAAAKIAAAAXAAAAogAAABsAAAAXAAAAGwAAABkAAAAcAAAAHgAAAKUAAAAcAAAApQAAACEAAAAcAAAAIQAAAB8AAAAAAAAAAwAAAGgAAABoAAAAZwAAAAAAAACQAAAAjgAAABQAAAAUAAAAEwAAAJAAAACTAAAAkQAAAJAAAACQAAAAkgAAAJMAAACqAAAArQAAAK4AAACuAAAAqwAAAKoAAACqAAAAqQAAALMAAACzAAAArwAAAKoAAAC6AAAAuwAAALwAAAC8AAAAvQAAALoAAADBAAAAwgAAAL4AAAC+AAAAugAAAMEAAADIAAAAyQAAAMoAAADIAAAAygAAAMUAAADIAAAAxQAAAMMAAADPAAAA0gAAANMAAADPAAAA0wAAANQAAADPAAAA1AAAANAAAABkAAAADgAAALcAAAC3AAAAJQEAAGQAAAC7AAAAvgAAADkBAAA5AQAAOgEAALsAAAA7AQAAOgEAALkAAAC5AAAAuAAAADsBAADQAAAA1AAAAM0AAADNAAAAzAAAANAAAAA5AQAAOAEAAMAAAADAAAAAvwAAADkBAAC3AAAAtgAAALMAAACzAAAAqQAAALcAAAAjAAAAzwAAAM4AAADOAAAAJAAAACMAAADJAAAAyAAAAMwAAADMAAAAzQAAAMkAAADFAAAAxgAAAMEAAADFAAAAwQAAALoAAADFAAAAugAAAMMAAACtAAAAsAAAALkAAAC5AAAAvwAAAK0AAAC2AAAAtwAAAA4AAAAOAAAADQAAALYAAACxAAAAsgAAALAAAACwAAAArwAAALEAAACcAAAAlwAAABwAAAAcAAAAHQAAAJwAAACVAAAAkgAAABAAAAAQAAAADwAAAJUAAAALAAAACgAAAGoAAABqAAAAaQAAAAsAAAAkAAAAIAAAAB8AAAAfAAAAIwAAACQAAACWAAAAlwAAABkAAAAZAAAAGwAAAJYAAAAXAAAAEgAAABUAAAAXAAAAFQAAAKEAAAAXAAAAoQAAABgAAACRAAAAkwAAAAYAAAAGAAAABQAAAJEAAAANAAAADgAAAGgAAABoAAAAaQAAAA0AAAAHAAAABgAAAAgAAAAIAAAACQAAAAcAAAAAAAAAAQAAAAIAAAACAAAAAwAAAAAAAAAJAAAACgAAAAsAAAALAAAABwAAAAkAAACTAAAAlAAAAAgAAAAIAAAABgAAAJMAAACPAAAAkQAAAAUAAAAFAAAABAAAAI8AAAAWAAAAFwAAABkAAAAZAAAAGgAAABYAAAAdAAAAHAAAAB8AAAAfAAAAIAAAAB0AAABnAAAAaAAAAA4AAAAOAAAAZAAAAGcAAACSAAAAkAAAABMAAAATAAAAEAAAAJIAAACUAAAAkwAAAJIAAACSAAAAlQAAAJQAAACpAAAAqgAAAKsAAACrAAAArAAAAKkAAACvAAAAswAAALQAAAC0AAAAsQAAAK8AAACwAAAAsgAAALgAAAC4AAAAuQAAALAAAACuAAAArQAAAL8AAAC/AAAAwAAAAK4AAADHAAAAyAAAAMMAAADDAAAAxAAAAMcAAADOAAAAzwAAANAAAADQAAAA0QAAAM4AAAAlAQAAtwAAAKkAAACpAAAArAAAACUBAAC+AAAAwgAAADgBAAA4AQAAOQEAAL4AAAA6AQAAOQEAAL8AAAC/AAAAuQAAADoBAADiAAAAMAAAACUAAAAlAAAA4QAAAOIAAAD4AAAA9wAAAP8AAAD/AAAA+gAAAPgAAAAEAQAAAwEAAAEBAAABAQAAAAEAAAQBAAAEAQAABQEAAFEAAABRAAAAUAAAAAQBAAD+AAAA+wAAAPoAAAD6AAAA/wAAAP4AAAD0AAAA5wAAAPYAAAD2AAAA9QAAAPQAAADzAAAA4wAAAOIAAADiAAAA8gAAAPMAAADvAAAA3QAAANsAAADbAAAA7gAAAO8AAADrAAAA7AAAANwAAADcAAAA6AAAAOsAAAAwAAAA4gAAAOMAAADjAAAAMgAAADAAAADdAAAA1wAAANkAAADZAAAA2wAAAN0AAACeAAAAnQAAACcAAAAnAAAALgAAAJ4AAABFAAAAUgAAADgAAAA4AAAANwAAAEUAAABKAAAATQAAAE4AAABOAAAASwAAAEoAAABGAAAAPAAAADkAAAA5AAAASAAAAEYAAAA7AAAAOgAAAD0AAAA7AAAAPQAAAD4AAAA7AAAAPgAAAD8AAAA7AAAAPwAAAEAAAAA7AAAAQAAAAEEAAAA7AAAAQQAAAEIAAAA7AAAAQgAAAEQAAAA7AAAARAAAAEMAAAA/AAAALgAAAC8AAAAvAAAAQAAAAD8AAAAqAAAAKwAAADwAAAA1AAAANgAAADcAAAA3AAAAOAAAADUAAAAuAAAAJwAAACYAAAAmAAAALwAAAC4AAAArAAAAKgAAACkAAAApAAAALAAAACsAAAAvAAAAJgAAACUAAAAlAAAApgAAAC8AAAA5AAAAKwAAADYAAAA2AAAANQAAADkAAAA6AAAAKwAAACwAAAAsAAAAPQAAADoAAABAAAAALwAAAKYAAABAAAAApgAAADEAAABAAAAAMQAAAEEAAABCAAAAqAAAAEUAAABFAAAARAAAAEIAAACaAAAASQAAAEoAAABKAAAAmQAAAJoAAABPAAAAUAAAAGsAAABrAAAAbAAAAE8AAABNAAAATAAAAGwAAABsAAAAawAAAE0AAACZAAAAmAAAAEcAAABHAAAARgAAAJkAAACmAAAAJQAAADAAAAAwAAAAMQAAAKYAAADbAAAA2QAAANoAAADaAAAA3AAAANsAAADhAAAAJQAAANUAAADVAAAA3wAAAOEAAADpAAAA6AAAANwAAADcAAAA6gAAAOkAAADuAAAA2wAAANwAAADcAAAA7AAAAO4AAADyAAAA4gAAAOEAAADyAAAA4QAAAN8AAADyAAAA3wAAAPAAAAD1AAAA9gAAAOQAAADkAAAA8wAAAPUAAAD6AAAA+wAAAPwAAAD8AAAA/QAAAPoAAAADAQAABAEAAFAAAABQAAAATwAAAAMBAAD5AAAA+AAAAPoAAAD6AAAA/QAAAPkAAADvAAAA8QAAAN4AAADgAAAA1gAAANgAAADYAAAA3gAAAOAAAAD2AAAABgEAAOUAAADlAAAA5AAAAPYAAAD7AAAAAAEAAAEBAAABAQAA/AAAAPsAAAD3AAAA+AAAAO0AAADtAAAA2gAAAPcAAADrAAAA6AAAAOcAAADnAAAA9AAAAOsAAADeAAAA3QAAAO8AAADtAAAA6gAAANwAAADjAAAA5AAAAOUAAADjAAAA5QAAADQAAADjAAAANAAAADIAAADeAAAA2AAAANcAAADXAAAA3QAAAN4AAACoAAAAQgAAAKcAAAA/AAAAngAAAC4AAABOAAAATQAAAGsAAABrAAAAbQAAAE4AAACoAAAAMwAAAFIAAABSAAAARQAAAKgAAABJAAAATAAAAE0AAABNAAAASgAAAEkAAAAqAAAAPAAAAEYAAABGAAAARwAAACoAAABDAAAANwAAADYAAAA2AAAAOwAAAEMAAAA+AAAALQAAAJ4AAAArAAAAOQAAADwAAAAyAAAANAAAADMAAAAyAAAAMwAAAKgAAAAyAAAAqAAAAKcAAAAtAAAAKAAAAJ0AAACdAAAAngAAAC0AAAAsAAAAKQAAACgAAAAoAAAALQAAACwAAAAyAAAApwAAADEAAAAxAAAAMAAAADIAAAA2AAAAKwAAADoAAAA6AAAAOwAAADYAAAA9AAAALAAAAC0AAAAtAAAAPgAAAD0AAABBAAAAMQAAAKcAAACnAAAAQgAAAEEAAABEAAAARQAAADcAAAA3AAAAQwAAAEQAAACYAAAAmQAAAEoAAABKAAAASwAAAJgAAABQAAAAUQAAAG0AAABtAAAAawAAAFAAAACaAAAAmQAAAEYAAABGAAAASAAAAJoAAACeAAAAPwAAAD4AAADfAAAA1QAAANYAAADWAAAA4AAAAN8AAADmAAAA5wAAAOgAAADoAAAA6QAAAOYAAADtAAAA3AAAANoAAADwAAAA3wAAAOAAAADgAAAA8QAAAPAAAAD0AAAA9QAAAPMAAAD0AAAA8wAAAPIAAAD0AAAA8gAAAPAAAAD0AAAA8AAAAPEAAAD0AAAA8QAAAO8AAAD0AAAA7wAAAO4AAAD0AAAA7gAAAOwAAAD0AAAA7AAAAOsAAAD5AAAA6gAAAO0AAADtAAAA+AAAAPkAAAD+AAAAAgEAAAABAAAAAQAA+wAAAP4AAADnAAAA5gAAAAYBAAAGAQAA9gAAAOcAAAAFAQAABAEAAAABAAAAAQAAAgEAAAUBAADgAAAA3gAAAPEAAADjAAAA8wAAAOQAAAAQAQAA+QAAAP0AAAD9AAAABwEAABABAADpAAAA6gAAAA8BAAAPAQAADgEAAOkAAAABAQAAAwEAAAoBAAAKAQAACQEAAAEBAAAHAQAACAEAAAkBAAAHAQAACQEAAAoBAAAHAQAACgEAAIQAAAAHAQAAhAAAAIMAAAAHAQAAgwAAAAsBAAAHAQAACwEAAAwBAAAHAQAADAEAAA0BAAAHAQAADQEAAA4BAAAHAQAADgEAAA8BAAAHAQAADwEAABABAACJAAAAiAAAAEgAAABIAAAAOQAAAIkAAACFAAAAhAAAAE8AAABPAAAAbAAAAIUAAACIAAAAiQAAAIoAAACIAAAAigAAAIsAAACIAAAAiwAAAIEAAACIAAAAgQAAAIIAAACIAAAAggAAAIMAAACIAAAAgwAAAIQAAACIAAAAhAAAAIUAAACIAAAAhQAAAIYAAACIAAAAhgAAAIcAAACIAAAAhwAAAJsAAACGAAAAhQAAAGwAAABsAAAATAAAAIYAAACKAAAAiQAAADkAAAA5AAAANQAAAIoAAACbAAAAmgAAAEgAAABIAAAAiAAAAJsAAAADAQAATwAAAIQAAACEAAAACgEAAAMBAADqAAAA+QAAABABAAAQAQAADwEAAOoAAADmAAAA6QAAAA4BAAAOAQAADQEAAOYAAAD8AAAAAQEAAAkBAAAJAQAACAEAAPwAAADlAAAABgEAAAwBAAAMAQAACwEAAOUAAACBAAAAiwAAADgAAAA4AAAAUgAAAIEAAACaAAAAmwAAAIcAAACHAAAASQAAAJoAAACDAAAAggAAADMAAAAzAAAANAAAAIMAAACCAAAAgQAAAFIAAABSAAAAMwAAAIIAAACHAAAAhgAAAEwAAABMAAAASQAAAIcAAACLAAAAigAAADUAAAA1AAAAOAAAAIsAAAA0AAAA5QAAAAsBAAALAQAAgwAAADQAAAD8AAAACAEAAAcBAAAHAQAA/QAAAPwAAAAGAQAA5gAAAA0BAAANAQAADAEAAAYBAAAmAQAArAAAAKsAAACrAAAAFwEAACYBAAATAQAAwQAAAMYAAADGAAAAHQEAABMBAAChAAAAoAAAAGAAAABgAAAAGAAAAKEAAABUAAAAAAAAAGcAAABnAAAAUwAAAFQAAACfAAAAlgAAABsAAAAbAAAAYQAAAJ8AAAAUAAAAVwAAAFgAAABYAAAAFQAAABQAAACjAAAAYQAAABsAAAAbAAAAogAAAKMAAABVAAAAAQAAAAAAAAAAAAAAVAAAAFUAAACjAAAAogAAABgAAAAYAAAAYAAAAKMAAAAXAQAAqwAAAK4AAACuAAAAEgEAABcBAADCAAAAFAEAABoBAAAaAQAAOAEAAMIAAAAWAQAAJQEAAKwAAACsAAAAJgEAABYBAADAAAAAEQEAABIBAAASAQAArgAAAMAAAACPAAAAjQAAAIwAAACMAAAAjgAAAI8AAABkAAAAWgAAAFMAAABTAAAAZwAAAGQAAAAEAAAAVgAAAI0AAACNAAAAjwAAAAQAAACgAAAAoQAAABUAAAAVAAAAWAAAAKAAAABWAAAABAAAAAEAAAABAAAAVQAAAFYAAACOAAAAjAAAAFcAAABXAAAAFAAAAI4AAADBAAAAEwEAABQBAAAUAQAAwgAAAMEAAAAlAQAAFgEAAFoAAABaAAAAZAAAACUBAAA4AQAAGgEAABEBAAARAQAAwAAAADgBAAAzAQAANAEAAC4BAAAuAQAALQEAADMBAAA3AQAAJAEAACgBAAAoAQAAMQEAADcBAAAjAQAAZgAAAHgAAAB4AAAAKwEAACMBAAAbAQAAHAEAADcBAAA3AQAANgEAABsBAAAVAQAAJwEAADMBAAAzAQAAMgEAABUBAAAVAQAAFgEAACYBAAAmAQAAJwEAABUBAACAAAAAfwAAAF0AAABdAAAAXAAAAIAAAAB8AAAAewAAAGEAAAB8AAAAYQAAAKMAAAB8AAAAowAAAGAAAAB3AAAAdgAAAFkAAABZAAAAWwAAAHcAAABdAAAAXgAAAHMAAABzAAAAcgAAAF0AAABbAAAAWQAAAG8AAABvAAAAbgAAAFsAAABgAAAAoAAAAFgAAABYAAAAdAAAAGAAAABTAAAAWgAAAG4AAABuAAAAbwAAAFMAAABYAAAAVwAAAHMAAABzAAAAdAAAAFgAAAB9AAAAfgAAAH8AAAB9AAAAfwAAAIAAAAB9AAAAgAAAAHUAAAB9AAAAdQAAAHYAAAB9AAAAdgAAAHcAAAB9AAAAdwAAAHgAAAB9AAAAeAAAAHkAAAB9AAAAeQAAAHoAAAB9AAAAegAAAHsAAAB9AAAAewAAAHwAAABcAAAAXQAAAHIAAAByAAAAcQAAAFwAAAB2AAAAdQAAAGUAAABlAAAAWQAAAHYAAAB7AAAAegAAAGIAAAB7AAAAYgAAAJ8AAAB7AAAAnwAAAGEAAAB/AAAAfgAAAF4AAABeAAAAXQAAAH8AAAAVAQAAbgAAAFoAAABaAAAAFgEAABUBAAAZAQAAJwEAACYBAAAmAQAAFwEAABkBAAAnAQAAGQEAADQBAAA0AQAAMwEAACcBAAAkAQAANwEAABwBAAAhAQAAIgEAACMBAAAhAQAAIwEAACsBAAAhAQAAKwEAACoBAAA2AQAANwEAADEBAAAxAQAAMAEAADYBAAA0AQAANQEAAC8BAAAvAQAALgEAADQBAAAkAQAAHwEAAB4BAAAkAQAAHgEAACkBAAAkAQAAKQEAACgBAABbAAAAMgEAACwBAAAsAQAAdwAAAFsAAAAYAQAAGwEAADYBAAA2AQAANQEAABgBAABuAAAAFQEAADIBAAAyAQAAWwAAAG4AAAASAQAAEQEAABoBAAASAQAAGgEAABQBAAASAQAAFAEAABsBAAASAQAAGwEAABgBAAB1AAAAgAAAAFwAAABcAAAAZQAAAHUAAAB9AAAAfAAAAGAAAABgAAAAXwAAAH0AAAB5AAAAeAAAAGYAAABmAAAAYwAAAHkAAABeAAAAXwAAAHQAAAB0AAAAcwAAAF4AAABZAAAAZQAAAHAAAABwAAAAbwAAAFkAAABwAAAAVAAAAFMAAABTAAAAbwAAAHAAAABxAAAAcgAAAFYAAABWAAAAVQAAAHEAAAByAAAAcwAAAFcAAAByAAAAVwAAAIwAAAByAAAAjAAAAI0AAAByAAAAjQAAAFYAAABVAAAAVAAAAHAAAABwAAAAcQAAAFUAAABlAAAAXAAAAHEAAABxAAAAcAAAAGUAAAB0AAAAXwAAAGAAAAB6AAAAeQAAAGMAAAB6AAAAYwAAAKQAAAB6AAAApAAAAGIAAAB+AAAAfQAAAF8AAABfAAAAXgAAAH4AAAAXAQAAEgEAABgBAAAYAQAAGQEAABcBAAAoAQAAKQEAACoBAAAoAQAAKgEAACsBAAAoAQAAKwEAAHgAAAAoAQAAeAAAAHcAAAAoAQAAdwAAACwBAAAoAQAALAEAAC0BAAAoAQAALQEAAC4BAAAoAQAALgEAAC8BAAAoAQAALwEAADABAAAoAQAAMAEAADEBAAAZAQAAGAEAADUBAAA1AQAANAEAABkBAAAyAQAAMwEAAC0BAAAtAQAALAEAADIBAAAeAQAAIAEAACEBAAAeAQAAIQEAACoBAAAeAQAAKgEAACkBAAA1AQAANgEAADABAAAwAQAALwEAADUBAAAiAAAAZgAAACMBAAAjAQAA0gAAACIAAAClAAAApAAAAGMAAABjAAAAIQAAAKUAAAAhAAAAYwAAAGYAAABmAAAAIgAAACEAAADKAAAAyQAAAB4BAAAeAQAAHwEAAMoAAADSAAAAIwEAACIBAAAiAQAA0wAAANIAAAAhAQAA1AAAANMAAADTAAAAIgEAACEBAACWAAAAnwAAAGIAAABiAAAAHgAAAJYAAACkAAAApQAAAB4AAAAeAAAAYgAAAKQAAAAeAQAAyQAAAM0AAADNAAAAIAEAAB4BAADUAAAAIQEAACABAAAgAQAAzQAAANQAAAAkAQAAxQAAAMoAAADKAAAAHwEAACQBAAAcAQAAEwEAAB0BAAAdAQAAJAEAABwBAADFAAAAJAEAAB0BAAAdAQAAxgAAAMUAAAAcAQAAGwEAABQBAAAUAQAAEwEAABwBAABZiz++NA4RQGe5VL+7RJ2+b2YQQJW3S78bL42+YFYIQA1RNb9Fgim+uYwKQPnAPr8IBdW+F/IOQAQhNb+LFsC+kdAGQBy2Ib/tEaq+9S8HQAIME78Eq4K+F2EHQOZ4Ib+syJC+KVoIQCvCAb+Fz2a+fSUIQAvPC7+HURC+bLIHQMjQFb9WZR++dqYHQCMQK7+cU0m2O+IHQFjFG7+cU0m2uw4IQGH/Lb+cU0m2iQwMQNDtQb+H/ba+6DIJQB2Opr62uem+aOsFQN4etL63KLu+YhEKQGGnOL6Xj/S+/dsEQM4XO760OPu+CFoGQAR1ur4xPw+/XMYIQJS/w778jhW/a0YFQJ6WP76Wsam+g94KQKG6ubxWYeO+rtQDQA6CjjsBhRK/OdQBQCcwXT3iy7y+4LsEQHGuIT7GpYq+/HALQD3Ulj33cwa/3H4BQOFFfz4sRYq+KEYGQCMtnT5Vhzy+kNoLQOELIz5HIbm+zxICQDLl6z4llQm+X+8GQIwsyT56/rS9SBUMQIifTz6FlzC+XoEDQDdwDz+cU0m2GsQEQAUyFz+cU0m2BB0IQPBr3D6cU0m23CwMQFWGYT6cU0m28DYVQFFp5D3+tXy9QC8VQMe62D2KrRC+WyMVQKKWxj0VHG6+fNQUQBjuHD1St5O+lGcUQOLJLr3FxqS+NbgTQE1pPb7fpYy+eJgcQJo+C76ry4m+o5AXQHQKcr3L2GC+FTcWQE0Ttjx7ExO+3ZUWQLQ8rz2Zuqu9eqUYQEZCuz2cU0m2kGkeQPMBoT3mzgy93CweQDjXkD2cU0m2YtwhQKXzgT2+LVi9MbMlQMzR4zycU0m2avclQOawOz0Cfm2++kYjQIfgGL4/xXG+6pEhQHb6Ab5M3yu++oAjQLVvjr2V9DC+UkYkQG9Kub2AZIq+VUsiQE4NZL718lu+O+McQDp2sL3FyU2+q9EeQNwQw70QXZi+GvcbQNMvUb7H2Fm+mggbQEcccr0vbkO+oDIaQGCRH7228hK+kQoaQFN7kTxtj969eEcbQDcaAD3rGq292LwdQM0f0zyVgLi9Pl0fQHwNQTwogCK+zEQgQLyyi700htm9sFYgQGAA4bxvSe693UQkQJMbxbzhKZS++fcaQPuxob5i2Zy+e/YSQMgIoL7shIe+i8EgQCOjo74eN0y+00wfQMHi6L5nX2m+JCgaQHOC7r6Qvnm+zy4SQIdu7r7TwO+9gqseQIi8Ab+u1gm+fEcZQDz2A793SxK+9N8RQFg7Br+9Nwa2eGIeQEusCL+cU0m21EUXQNuFCr+cU0m2v7oRQLlQDb+3C+29rP4kQODy2Lydgpy9YFkRQMA+Yr818T6+CoYQQE34Xb9cq6W+N8QOQHv3U7+EhOC+fO8MQN3qPb8/dBW/T5UGQILix775FRe/ByMDQN19Tr7Fc5a9aqMCQBQlMb+cU8m2vLERQK93Y7+cU0m2lj8DQFJGNL9vn42+kC0CQBJsKL/EtbK+XFkBQL+YGb+Z8vG+l4wAQC3Qzr6y1wu/IjQAQCUhQb5mhhG/kSn/P3Ztbz1Eowe/by7+Pz1heT7Ny7m+q1v/P8Xm6z6JIy++WyUBQN0LED+cU8m26WMSQMo3W7+mfym+3XsCQKnaLr+cU0m2zCgCQJojFz95B5i93QcSQMwJWr+nP3u9/tQLQLX/Qb92wHW9xecHQN9SLr/7eVO9jKAHQIiEG789SE+9pkYYQKrXCb8GDii9kWQeQFYLCL/CMGC9wccRQLuAC7+9N4a2x/YJQHRFTb+BzZm9dsYJQB43TL93ojS+KJoIQP2+R7+S6pu+BJAGQFmKPL/x18y+WrcEQBxdJb9TJQa/kJ8CQPoM2L4DzxW/VKsAQCPcRL7HY+a9Bp4EQMEbDr93K0u9KQUFQBBADr8AAAAA6KIEQDyfDb8AAAAAyogEQKm/pj7Ay+y9E0UEQFBwoT7hCX2+BwcDQEeuez5OfL2+2sgCQApI2z0Sh8y+aOgCQIVf6rznjr6+M4sDQFnaSb6F0aS+gscDQLgfsL49X3O+LlMEQJp9/r47x0C+gJoEQM1yCb/Y19q9kWElQKoKTb3iPUe9LQYmQNj0oLmcU0m2r0QmQN21hDycU0m2DVIfQFrYA7/qJBu9/FMfQOVEA7/MRdy9+5QfQC/6+r70azu+1CggQB+d4r7NBX6+X34hQHU6oL5f7H2+OuciQJFja76a6lm+0c0jQBeAJr43cSK+9bskQAKc3r0FTQ+/OncJQNQKC7/MCwS/oggLQFPOI7+13ge/k4sLQKyQBr8WM/q+GhQNQHnrHL9klu2+cvgGQNFA/L7Ar9m+guQGQAosEL+fHdi+l1cGQOLm7L4g8cO+xsMGQJawBr8i46G+ZqIIQFex6L4nS62+q+oIQJgxzb6rIO6+IqsBQJKvvD4qrKS+AoEFQBDLdj72RZK+p5ISQKvryL5r1om+EZAaQFzKyb55c3i+NQcgQAbzx74AjGe+ntMgQN8bw775MGu+xqULQJMBAD5YrEG+7PsUQIMUnD2rPTy+e2YWQFncfz1LXO++mIn+P2Q6vD67XxW/WioBQPfojb1xGxW/xFsDQMmQg73vjQ2/7YEBQKoPJD55AQ6/vFz+P4S7Iz6nPY2+xVUAQOTaBD9BRY2+RrYCQLWMBD8M51q9h9saQBrbqz2fcgy9RpUiQFH3QT2HFEO9glUjQJXTHj1Fgik+uYwKQPnAPr8bL40+YFYIQA1RNb+7RJ0+b2YQQJW3S79Ziz8+NA4RQGe5VL+LFsA+kdAGQBy2Ib8IBdU+F/IOQAQhNb8Eq4I+F2EHQOZ4Ib/tEao+9S8HQAIME7+Fz2Y+fSUIQAvPC7+syJA+KVoIQCvCAb9WZR8+dqYHQCMQK7+HURA+bLIHQMjQFb/7eVM9jKAHQIiEG792wHU9xecHQN9SLr+nP3s9/tQLQLX/Qb8i46E+ZqIIQFex6L4g8cM+xsMGQJawBr+Xj/Q+/dsEQM4XO762uek+aOsFQN4etL6H/bY+6DIJQB2Opr63KLs+YhEKQGGnOL60OPs+CFoGQAR1ur7Ar9k+guQGQAosEL8WM/o+GhQNQHnrHL/8jhU/a0YFQJ6WP74xPw8/XMYIQJS/w75WYeM+rtQDQA6CjjuWsak+g94KQKG6ubwBhRI/OdQBQCcwXT1xGxU/xFsDQMmQg73GpYo+/HALQD3Ulj3iy7w+4LsEQHGuIT73cwY/3H4BQOFFfz7vjQ0/7YEBQKoPJD75MGs+xqULQJMBAD4qrKQ+AoEFQBDLdj6rIO4+IqsBQJKvvD56/rQ9SBUMQIifTz4llQk+X+8GQIwsyT4sRYo+KEYGQCMtnT5Vhzw+kNoLQOELIz6FlzA+XoEDQDdwDz9BRY0+RrYCQLWMBD9HIbk+zxICQDLl6z7+tXw9QC8VQMe62D2KrRA+WyMVQKKWxj0VHG4+fNQUQBjuHD1YrEE+7PsUQIMUnD1St5M+lGcUQOLJLr3FxqQ+NbgTQE1pPb6ry4k+o5AXQHQKcr3fpYw+eJgcQJo+C77L2GA+FTcWQE0TtjyrPTw+e2YWQFncfz2Zuqs9eqUYQEZCuz17ExM+3ZUWQLQ8rz0M51o9h9saQBrbqz3mzgw93CweQDjXkD2fcgw9RpUiQFH3QT2HFEM9glUjQJXTHj2+LVg9MbMlQMzR4zyV9DA+UkYkQG9Kub1M3ys++oAjQLVvjr0/xXE+6pEhQHb6Ab4Cfm0++kYjQIfgGL6AZIo+VUsiQE4NZL7FyU0+q9EeQNwQw7318ls+O+McQDp2sL0QXZg+GvcbQNMvUb7H2Fk+mggbQEcccr0vbkM+oDIaQGCRH71tj949eEcbQDcaAD228hI+kQoaQFN7kTzrGq092LwdQM0f0zyVgLg9Pl0fQHwNQTwogCI+zEQgQLyyi700htk9sFYgQGAA4bxvSe493UQkQJMbxbxi2Zw+e/YSQMgIoL7hKZQ++fcaQPuxob7shIc+i8EgQCOjo75r1ok+EZAaQFzKyb5nX2k+JCgaQHOC7r4eN0w+00wfQMHi6L55c3g+NQcgQAbzx76Qvnk+zy4SQIdu7r72RZI+p5ISQKvryL6u1gk+fEcZQDz2A7/TwO89gqseQIi8Ab93SxI+9N8RQFg7Br8GDig9kWQeQFYLCL89SE89pkYYQKrXCb/CMGA9wccRQLuAC7+3C+09rP4kQODy2LwAjGc+ntMgQN8bw770azs+1CggQB+d4r7MRdw9+5QfQC/6+r7qJBs9/FMfQOVEA7/iPUc9LQYmQNj0oLnY19o9kWElQKoKTb03cSI+9bskQAKc3r2a6lk+0c0jQBeAJr5f7H0+OuciQJFja77NBX4+X34hQHU6oL7MCwQ/oggLQFPOI7+EhOA+fO8MQN3qPb/5FRc/ByMDQN19Tr4/dBU/T5UGQILix76BzZk9dsYJQB43TL+dgpw9YFkRQMA+Yr9cq6U+N8QOQHv3U7/x18w+WrcEQBxdJb+S6ps+BJAGQFmKPL8FTQ8/OncJQNQKC79TJQY/kJ8CQPoM2L4DzxU/VKsAQCPcRL67XxU/WioBQPfojb1Eowc/by7+Pz1heT55AQ4/vFz+P4S7Iz5LXO8+mIn+P2Q6vD7Ny7k+q1v/P8Xm6z6nPY0+xVUAQOTaBD+JIy8+WyUBQN0LED9mhhE/kSn/P3Ztbz15B5g93QcSQMwJWr818T4+CoYQQE34Xb93ojQ+KJoIQP2+R78Sh8w+aOgCQIVf6rxOfL0+2sgCQApI2z3hCX0+BwcDQEeuez7Ay+w9E0UEQFBwoT53K0s9KQUFQBBADr/HY+Y9Bp4EQMEbDr87x0A+gJoEQM1yCb89X3M+LlMEQJp9/r6F0aQ+gscDQLgfsL7njr4+M4sDQFnaSb7Fc5Y9aqMCQBQlMb+mfyk+3XsCQKnaLr9vn40+kC0CQBJsKL/EtbI+XFkBQL+YGb+Z8vE+l4wAQC3Qzr6y1ws/IjQAQCUhQb613gc/k4sLQKyQBr9klu0+cvgGQNFA/L6fHdg+l1cGQOLm7L4nS60+q+oIQJgxzb4KAAAACwAAACcAAAAjAAAAIgAAAB4AAAAeAAAALgAAACMAAAAoAAAAKgAAACsAAAAlAAAAEwAAADAAAAAwAAAAJAAAACUAAAAqAAAAJgAAACEAAAAhAAAAHQAAACoAAAApAAAAHgAAACIAAAATAAAAEgAAAC8AAAAvAAAAMAAAABMAAAAJAAAABgAAAAMAAAADAAAADAAAAAkAAAAFAAAAAQAAAAIAAAALAAAACgAAAAwAAAASAAAAGQAAABYAAAAWAAAALwAAABIAAAAoAAAACgAAACcAAAAPAAAADQAAABEAAAAoAAAAJwAAACoAAAAyAAAABwAAAAgAAAAIAAAALAAAADIAAAAJAAAADAAAAAoAAAAVAAAAIwAAAC4AAAAuAAAAMQAAABUAAAAaAAAAGwAAABgAAAAYAAAAFwAAABoAAAAOAAAAFwAAABgAAAAYAAAAEAAAAA4AAAAIAAAABwAAAAYAAAAGAAAACQAAAAgAAAAFAAAAAAAAAAEAAAAOAAAAEAAAABEAAAARAAAADQAAAA4AAAAEAAAAFgAAABkAAAApAAAAHAAAACAAAAAgAAAAHwAAACkAAAAmAAAADwAAACEAAAAPAAAAJgAAAA0AAAAgAAAAJQAAACQAAAAkAAAAHwAAACAAAAAVAAAAMQAAABQAAAAAAAAABwAAADIAAAAUAAAAMQAAADAAAAAwAAAALwAAABQAAAAjAAAALQAAACIAAAALAAAAJgAAACcAAAAYAAAAGQAAABIAAAASAAAAEAAAABgAAAAIAAAACQAAAAoAAAAGAAAABQAAAAIAAAACAAAAAwAAAAYAAAAQAAAAEgAAABMAAAATAAAAEQAAABAAAAAcAAAAHQAAACEAAAAhAAAAIAAAABwAAAARAAAAEwAAACUAAAAlAAAADwAAABEAAAAjAAAAFQAAAC0AAAAEAAAAGQAAABgAAAAYAAAAGwAAAAQAAAAUAAAALwAAABYAAAAxAAAALgAAACQAAAAkAAAAMAAAADEAAAAhAAAADwAAACUAAAAlAAAAIAAAACEAAAAnAAAAJgAAACoAAAAfAAAAJAAAAC4AAAAuAAAAHgAAAB8AAAApAAAAHwAAAB4AAAALAAAADAAAAA4AAAAOAAAADQAAAAsAAAAAAAAABQAAAAYAAAAGAAAABwAAAAAAAAAMAAAAFwAAAA4AAAAMAAAAAwAAABoAAAAaAAAAFwAAAAwAAAAoAAAAKwAAACwAAAALAAAADQAAACYAAAAsAAAACAAAAAoAAAAKAAAAKAAAACwAAACrr647dEY8P57TGD8GoIG+SSsuPznUFz+P+mu+dckYP483ET81Qg++hPT8PokjIz9zKlm+zuGCPp+uQj8I5FK+PBciP/MbJj+Hpye+iUIXP2/VLT9JEK47AFYrP1VnKT8tza07jgUdP+j2Mj8PuO692cwNP07uNz+fq607dlQRP3KnOD/03a07w9YAP1FoOT8SF8C9D0UBP4wSPD9oJa07EwzvPhzsST9oQai9AvLlPpzDVT/KxG09SdbRPjp2WD8J35u9K/e6PvaVXz+Tp6w7Pj7JPuV9WD+5q1e9vTeOPrWIWD+hFK07c0eXPn44WD8dOCe9lfEfPisVQD850a47TU0SPq66Qj/lffy9mN0zPuW5Pj9moCK+WHXOPhe7UT8lkii+AoGmPlXZUz9agBa+YrqAPsZuSz88aVG+fjfFPmh2OT/j4GK+F/ObPltAQD9z2mM+gQixPidMQD82ck0+CMzTPuCBOT+twCA+1ht1Phh3Sz+sxjI+71SgPoLiUz9jYjM+M+G3Pk0sVD8Ykx4+cAjdPgDEUT/cLQk+++UzPujAPj/D81I9d/QfPqkXQD8DeoE9kzmOPtuLWD/sLrA9SBaoPluaXz+eztU9okYBP+wXPD8ROAI+0c4NP+30Nz/biTI+SkUXP63eLT9zEGQ+wOiCPmu6Qj8BNBo+Gvn8PncrIz+/9HY+Qs0YP1ZEET++F4c+ey8uPz/iFz9aR5U9woUMPtlBTT9ANJM95WNXPspRTD9xcj+9iDBePuhOTD+fq607NXl6PpkOUT/ReKI8eZM/PqTkST+GxiM+VdkzP05DGD++AAAAvQAAAAABAAAAAQAAAQEAAL4AAAC8AAAAuwAAAMgAAADIAAAA/wAAALwAAADsAAAA/gAAAOcAAADnAAAA6QAAAOwAAADlAAAA5AAAALIAAACyAAAA9wAAAOUAAAD8AAAA+wAAAPoAAAD6AAAA/QAAAPwAAADyAAAA+QAAAPgAAAD4AAAA8AAAAPIAAACxAAAAsAAAAPUAAAD1AAAA9gAAALEAAADSAAAA1QAAAPMAAADzAAAA9AAAANIAAADtAAAA3QAAAOEAAADhAAAA7wAAAO0AAADpAAAA5wAAAN4AAADeAAAA4AAAAOkAAADmAAAAzQAAAMwAAADMAAAA0AAAAOYAAADiAAAA4wAAAOQAAADkAAAA3wAAAOIAAADcAAAA3gAAAOEAAADhAAAA3QAAANwAAADZAAAA2AAAANoAAADaAAAA1AAAANkAAADRAAAA1AAAANUAAADVAAAA0gAAANEAAADKAAAAzQAAAM4AAADOAAAAyQAAAMoAAADKAAAAwwAAAMYAAADGAAAAywAAAMoAAAC5AAAAwQAAALoAAACjAAAAogAAAMUAAADFAAAAwAAAAKMAAAC6AAAAwQAAAMIAAAC1AAAAtgAAAL0AAAC9AAAAvgAAALUAAAC4AAAApwAAAKgAAACoAAAAuQAAALgAAAC0AAAAqQAAAKUAAAClAAAAtQAAALQAAACtAAAArAAAAK4AAACuAAAArwAAAK0AAACYAAAAkQAAAKUAAAClAAAAqQAAAJgAAACRAAAAkAAAAJIAAACRAAAAkgAAAKQAAACRAAAApAAAAKUAAACGAAAAhwAAAKAAAACgAAAAoQAAAIYAAACKAAAAjAAAAJwAAACcAAAAnQAAAIoAAACPAAAAeQAAAJcAAACXAAAAmQAAAI8AAACCAAAAgwAAAJUAAACVAAAAlAAAAIIAAAB7AAAAfAAAAJAAAACQAAAAkQAAAHsAAACMAAAAdAAAAHUAAAB1AAAAjQAAAIwAAACKAAAAiwAAAIwAAACIAAAAgQAAAEsAAABLAAAAiQAAAIgAAACFAAAAhAAAACIAAAAiAAAAIwAAAIUAAACCAAAAIAAAAB8AAAAfAAAAgwAAAIIAAAB6AAAAgAAAAE4AAABOAAAASQAAAHoAAAB9AAAARwAAAH4AAAB3AAAAegAAAHsAAAB2AAAAdQAAAG0AAABtAAAAawAAAHYAAABnAAAAbAAAAHMAAABzAAAAbwAAAGcAAABiAAAAYQAAAG4AAABuAAAAbAAAAGIAAABiAAAAZgAAAFgAAABYAAAAWQAAAGIAAABlAAAAaAAAAGkAAABpAAAAZAAAAGUAAABkAAAAYwAAAF0AAABdAAAAXgAAAGQAAABcAAAAWgAAAFQAAABUAAAAUwAAAFwAAABbAAAAXAAAAFMAAABTAAAAUgAAAFsAAABPAAAAUAAAAFEAAABPAAAAUQAAAFIAAABPAAAAUgAAAFMAAABPAAAAUwAAAFQAAABPAAAAVAAAAFUAAABPAAAAVQAAAFYAAABMAAAAQAAAAD8AAAA/AAAATQAAAEwAAABEAAAARQAAAEcAAABHAAAASAAAAEQAAABDAAAAPgAAADkAAAA5AAAARAAAAEMAAAA8AAAAOwAAAD8AAAA/AAAAQAAAADwAAAAxAAAAMAAAADwAAAA8AAAAPQAAADEAAAAnAAAANgAAAC8AAAAvAAAAKAAAACcAAAAsAAAAKwAAADIAAAAyAAAAMwAAACwAAAAnAAAAKAAAACkAAAAnAAAAKQAAACoAAAAnAAAAKgAAACsAAAAnAAAAKwAAACwAAAAnAAAALAAAAC0AAAAnAAAALQAAAC4AAAAdAAAAHAAAACQAAAAkAAAAIwAAAB0AAAAYAAAAFwAAAB8AAAAfAAAAIAAAABgAAAAOAAAAGwAAABwAAAAcAAAADwAAAA4AAAATAAAAFgAAABcAAAAXAAAAGAAAABMAAAANAAAAFQAAAA4AAAAOAAAACAAAAA0AAAAHAAAADwAAABAAAAAQAAAACQAAAAcAAAAEAAAACgAAAAsAAAALAAAABQAAAAQAAAAAAAAAAQAAAAIAAAAAAAAAAgAAAAMAAAAAAAAAAwAAAAQAAAAAAAAABAAAAAUAAAAAAAAABQAAAAYAAAADAAAACQAAAAoAAAAKAAAABAAAAAMAAAAIAAAADgAAAA8AAAAPAAAABwAAAAgAAAANAAAAEgAAABQAAAAUAAAAFQAAAA0AAAANAAAAAAAAAAYAAAANAAAABgAAAAwAAAANAAAADAAAABIAAAAOAAAAFQAAABoAAAAaAAAAGwAAAA4AAAARAAAAHgAAABcAAAAXAAAAFgAAABEAAAAeAAAAHQAAACMAAAAjAAAAIgAAAB4AAAAmAAAAGgAAABkAAAAZAAAAIQAAACYAAAArAAAAKgAAADEAAAAxAAAAMgAAACsAAAA1AAAANgAAACcAAAAnAAAALgAAADUAAAA2AAAAOgAAADsAAAA7AAAALwAAADYAAAA+AAAANQAAADQAAAA0AAAAOQAAAD4AAABCAAAAOgAAAD4AAAA+AAAAQwAAAEIAAABGAAAANwAAAD0AAAA9AAAAQQAAAEYAAABKAAAARgAAAEEAAABBAAAASwAAAEoAAABMAAAASwAAAEEAAABBAAAAQAAAAEwAAABZAAAAVQAAAFQAAABUAAAAWgAAAFkAAABPAAAAVwAAAF4AAABeAAAAUAAAAE8AAABbAAAAXQAAAGMAAABjAAAAXwAAAFsAAABmAAAAZwAAAGgAAABoAAAAZQAAAGYAAABiAAAAbAAAAGcAAABnAAAAZgAAAGIAAABhAAAAYAAAAG0AAABtAAAAbgAAAGEAAABqAAAAaQAAAHEAAABxAAAAcgAAAGoAAAB2AAAAawAAAGoAAABqAAAAcgAAAHYAAABvAAAAegAAAHcAAAB8AAAASAAAAEcAAABHAAAAfQAAAHwAAAB6AAAAbwAAAHMAAABzAAAAgAAAAHoAAACCAAAAfwAAACEAAAAhAAAAIAAAAIIAAACGAAAAJAAAACUAAAAlAAAAhwAAAIYAAACHAAAAJQAAAIEAAACBAAAAiAAAAIcAAACKAAAATQAAAIsAAACLAAAATQAAAE4AAABOAAAAgAAAAIsAAAByAAAAcQAAAHkAAAB5AAAAjwAAAHIAAAB/AAAAggAAAJQAAACUAAAAkwAAAH8AAACYAAAAlgAAAHgAAAB4AAAAdwAAAJgAAACMAAAAjQAAAJsAAACbAAAAnAAAAIwAAACHAAAAiAAAAJ8AAACfAAAAoAAAAIcAAACDAAAAhAAAAKMAAACjAAAAlQAAAIMAAACnAAAAlAAAAJUAAACVAAAAqAAAAKcAAACrAAAAqgAAAKwAAACsAAAArQAAAKsAAACqAAAAqQAAALQAAAC0AAAArAAAAKoAAAC3AAAApgAAAKcAAACnAAAAuAAAALcAAAC2AAAAtwAAALwAAAC8AAAAvQAAALYAAACVAAAAowAAAMAAAADAAAAAqAAAAJUAAADGAAAAxQAAAKIAAACiAAAAoQAAAMYAAACoAAAAwAAAAMEAAADBAAAAuQAAAKgAAADJAAAAxAAAAMMAAADDAAAAygAAAMkAAADLAAAAzAAAAM0AAADNAAAAygAAAMsAAADOAAAA0QAAANIAAADSAAAA0wAAAM4AAADWAAAA1wAAANgAAADYAAAA2QAAANYAAADeAAAA3AAAAN8AAADfAAAA4AAAAN4AAADiAAAA1wAAAJwAAACcAAAAmwAAAOIAAADRAAAAzgAAAM0AAADNAAAA5gAAANEAAADeAAAA5wAAAOgAAADoAAAA4QAAAN4AAADbAAAA3QAAAO0AAADtAAAA7gAAANsAAADtAAAAsAAAAK8AAACvAAAA7gAAAO0AAADzAAAArgAAAL8AAAC/AAAA9AAAAPMAAADwAAAA+AAAAPUAAAD1AAAA7wAAAPAAAAD3AAAA9gAAAPsAAAD7AAAA/AAAAPcAAACZAAAAswAAAOMAAADjAAAAmgAAAJkAAADnAAAA/gAAAPEAAADxAAAA6AAAAOcAAAA7AAAAPAAAADAAAAAwAAAALwAAADsAAAC9AAAAvAAAAP8AAAD/AAAAAAEAAL0AAAC/AAAAvgAAAAEBAAABAQAA9AAAAL8AAAD/AAAAyAAAAMcAAADHAAAAzwAAAP8AAADTAAAAzwAAAMkAAADJAAAAzgAAANMAAADqAAAA5QAAAPcAAAD3AAAA/AAAAOoAAACzAAAAsgAAAOQAAADkAAAA4wAAALMAAAD5AAAA+gAAAPsAAAD7AAAA+AAAAPkAAACyAAAAsQAAAPYAAAD2AAAA9wAAALIAAADVAAAA2wAAAO4AAADuAAAA8wAAANUAAADvAAAA4QAAAOgAAADoAAAA8AAAAO8AAADpAAAA4AAAAOUAAADlAAAA6gAAAOkAAADQAAAA1gAAANkAAADZAAAA5gAAANAAAADjAAAA4gAAAJsAAACbAAAAmgAAAOMAAADcAAAA2gAAANgAAADYAAAA3wAAANwAAADUAAAA2gAAANsAAADbAAAA1QAAANQAAADWAAAA0AAAAJ4AAACeAAAAnQAAANYAAADPAAAAxwAAAMQAAADEAAAAyQAAAM8AAADLAAAAxgAAAKEAAAChAAAAoAAAAMsAAADHAAAAyAAAALoAAADAAAAAxQAAAMIAAADCAAAAwQAAAMAAAAC6AAAAwgAAAMMAAADDAAAAxAAAALoAAAC0AAAAtQAAAL4AAAC+AAAAvwAAALQAAAC4AAAAuQAAALoAAAC6AAAAuwAAALgAAAC1AAAApQAAAKQAAACkAAAAtgAAALUAAACtAAAArwAAALAAAACwAAAAsQAAAK0AAACWAAAAmAAAAKkAAACpAAAAqgAAAJYAAACSAAAAkwAAAKYAAACmAAAApAAAAJIAAACFAAAAhgAAAKEAAAChAAAAogAAAIUAAACJAAAAigAAAJ0AAACdAAAAngAAAIkAAACOAAAAjwAAAJkAAACZAAAAmgAAAI4AAAB5AAAAeAAAAJYAAACWAAAAlwAAAHkAAAB8AAAAfQAAAJIAAACSAAAAkAAAAHwAAACNAAAAdQAAAHYAAAB2AAAAjgAAAI0AAACMAAAAiwAAAHQAAACJAAAASwAAAEwAAAAlAAAAJgAAAIEAAACDAAAAHwAAACIAAAAiAAAAhAAAAIMAAAB+AAAARwAAAEoAAABKAAAAgQAAAH4AAAB9AAAAfgAAAH8AAAB7AAAAegAAAEkAAABwAAAAbwAAAHcAAAB3AAAAeAAAAHAAAABsAAAAbgAAAHQAAAB0AAAAcwAAAGwAAABoAAAAZwAAAG8AAABvAAAAcAAAAGgAAABgAAAAYQAAAFoAAABaAAAAXAAAAGAAAABkAAAAaQAAAGoAAABqAAAAYwAAAGQAAABlAAAAZAAAAF4AAABeAAAAVwAAAGUAAABfAAAAYAAAAFwAAABcAAAAWwAAAF8AAABRAAAAXQAAAFsAAABbAAAAUgAAAFEAAABXAAAATwAAAFYAAABWAAAAWAAAAFcAAABNAAAAPwAAAEIAAABCAAAATgAAAE0AAABDAAAARAAAAEgAAABIAAAASQAAAEMAAABEAAAAOQAAADgAAAA4AAAARQAAAEQAAAA9AAAAPAAAAEAAAABAAAAAQQAAAD0AAAAyAAAAMQAAAD0AAAA9AAAANwAAADIAAAAyAAAANwAAADgAAAA4AAAAMwAAADIAAAAtAAAALAAAADMAAAAzAAAANAAAAC0AAAAvAAAAMAAAACkAAAApAAAAKAAAAC8AAAAcAAAAGwAAACUAAAAlAAAAJAAAABwAAAAYAAAAIAAAACEAAAAhAAAAGQAAABgAAAAPAAAAHAAAAB0AAAAdAAAAEAAAAA8AAAAUAAAAEwAAABgAAAAYAAAAGQAAABQAAAAMAAAACwAAABYAAAAWAAAAEwAAAAwAAAAJAAAAEAAAABEAAAARAAAACgAAAAkAAAAFAAAACwAAAAwAAAAMAAAABgAAAAUAAAAHAAAAAgAAAAEAAAABAAAACAAAAAcAAAAHAAAACQAAAAMAAAADAAAAAgAAAAcAAAAAAAAADQAAAAgAAAAIAAAAAQAAAAAAAAASAAAADAAAABMAAAATAAAAFAAAABIAAAAWAAAACwAAAAoAAAAKAAAAEQAAABYAAAAVAAAAFAAAABkAAAAZAAAAGgAAABUAAAAQAAAAHQAAAB4AAAAeAAAAEQAAABAAAAAXAAAAHgAAACIAAAAiAAAAHwAAABcAAAAlAAAAGwAAABoAAAAaAAAAJgAAACUAAAAqAAAAKQAAADAAAAAwAAAAMQAAACoAAAAuAAAALQAAADQAAAA0AAAANQAAAC4AAAAzAAAAOAAAADkAAAA5AAAANAAAADMAAAA6AAAANgAAADUAAAA1AAAAPgAAADoAAAA7AAAAOgAAAEIAAABCAAAAPwAAADsAAABFAAAAOAAAADcAAAA3AAAARgAAAEUAAABFAAAARgAAAEoAAABKAAAARwAAAEUAAABOAAAAQgAAAEMAAABDAAAASQAAAE4AAABYAAAAVgAAAFUAAABVAAAAWQAAAFgAAABQAAAAXgAAAF0AAABdAAAAUQAAAFAAAABZAAAAWgAAAGEAAABhAAAAYgAAAFkAAABYAAAAZgAAAGUAAABlAAAAVwAAAFgAAABjAAAAagAAAGsAAABrAAAAXwAAAGMAAABrAAAAbQAAAGAAAABgAAAAXwAAAGsAAABpAAAAaAAAAHAAAABwAAAAcQAAAGkAAAB0AAAAbgAAAG0AAABtAAAAdQAAAHQAAABxAAAAcAAAAHgAAAB4AAAAeQAAAHEAAAB7AAAASQAAAEgAAABIAAAAfAAAAHsAAAB/AAAAfgAAACEAAAB+AAAAgQAAACYAAAAmAAAAIQAAAH4AAACFAAAAIwAAACQAAAAkAAAAhgAAAIUAAACBAAAASgAAAEsAAACJAAAATAAAAE0AAABNAAAAigAAAIkAAAB0AAAAiwAAAIAAAACAAAAAcwAAAHQAAACOAAAAdgAAAHIAAAByAAAAjwAAAI4AAAB9AAAAfwAAAJMAAACTAAAAkgAAAH0AAACRAAAAmAAAAHcAAAB3AAAAewAAAJEAAACNAAAAjgAAAJoAAACaAAAAmwAAAI0AAACIAAAAiQAAAJ4AAACeAAAAnwAAAIgAAACEAAAAhQAAAKIAAACiAAAAowAAAIQAAACmAAAAkwAAAJQAAACUAAAApwAAAKYAAACXAAAAlgAAAKoAAACqAAAAqwAAAJcAAACrAAAArQAAALEAAACrAAAAsQAAALIAAACrAAAAsgAAALMAAAC2AAAApAAAAKYAAACmAAAAtwAAALYAAAC3AAAAuAAAALsAAAC7AAAAvAAAALcAAACsAAAAtAAAAL8AAAC/AAAArgAAAKwAAADDAAAAwgAAAMUAAADFAAAAxgAAAMMAAAC6AAAAxAAAAMcAAAC6AAAAyAAAALsAAACgAAAAnwAAAMwAAADMAAAAywAAAKAAAACfAAAAngAAANAAAADQAAAAzAAAAJ8AAACdAAAAnAAAANcAAADXAAAA1gAAAJ0AAADbAAAA2gAAANwAAADcAAAA3QAAANsAAADfAAAA2AAAANcAAADXAAAA4gAAAN8AAADgAAAA3wAAAOQAAADkAAAA5QAAAOAAAADmAAAA2QAAANQAAADUAAAA0QAAAOYAAADrAAAA7AAAAOkAAADpAAAA6gAAAOsAAADwAAAA6AAAAPEAAADxAAAA8gAAAPAAAADzAAAA7gAAAK8AAACvAAAArgAAAPMAAAD1AAAAsAAAAO0AAADtAAAA7wAAAPUAAAD4AAAA+wAAAPYAAAD2AAAA9QAAAPgAAACXAAAAqwAAALMAAACzAAAAmQAAAJcAAADrAAAA6gAAAPwAAAD8AAAA/QAAAOsAAADrAAAA/QAAAPoAAADrAAAA+gAAAPkAAADrAAAA+QAAAPIAAADrAAAA8gAAAPEAAADrAAAA8QAAAP4AAADrAAAA/gAAAOwAAAD/AAAAzwAAANMAAADTAAAAAAEAAP8AAAD0AAAAAQEAAAABAAD0AAAAAAEAANMAAAD0AAAA0wAAANIAAAD2AQAAAwIAAMABAADAAQAAwQEAAPYBAADRAQAAyQEAAMoBAADKAQAAAQIAANEBAADQAQAAywEAANEBAADRAQAA1QEAANABAAD+AQAA+QEAAOcBAADnAQAA7AEAAP4BAADlAQAA5gEAALQBAAC0AQAAtQEAAOUBAAD6AQAA/QEAAPwBAAD8AQAA+wEAAPoBAAD5AQAA+AEAALMBAACzAQAAtAEAAPkBAAD1AQAA8AEAAN0BAADdAQAA1wEAAPUBAADyAQAA6gEAAOMBAADjAQAA8QEAAPIBAADsAQAA5wEAAOIBAADiAQAA6wEAAOwBAADoAQAA2wEAANgBAADYAQAA0gEAAOgBAACcAQAAnQEAAOQBAADkAQAA5QEAAJwBAADhAQAA2gEAANwBAADcAQAA3gEAAOEBAADXAQAA3QEAANwBAADcAQAA1gEAANcBAACfAQAAoAEAANIBAADSAQAA2AEAAJ8BAADLAQAAxgEAAMkBAADJAQAA0QEAAMsBAACiAQAAowEAAMgBAADIAQAAzQEAAKIBAAC8AQAAygEAAMkBAADDAQAAxAEAAMcBAADHAQAAwgEAAMMBAADGAQAAxQEAAMQBAADEAQAAvAEAAMYBAADBAQAAwAEAALcBAAC3AQAAtgEAAMEBAAC9AQAAvAEAALsBAAC7AQAAugEAAL0BAAC4AQAApgEAAKcBAACnAQAAtwEAALgBAACzAQAAsgEAALEBAACxAQAArwEAALMBAACsAQAAqwEAAJoBAACaAQAAmAEAAKwBAACmAQAAqAEAAJUBAACVAQAAlAEAAKYBAACkAQAAowEAAIgBAACIAQAAhwEAAKQBAACgAQAAnwEAAIwBAACMAQAAiwEAAKABAACcAQAAmwEAAJEBAACRAQAAkAEAAJwBAACZAQAAmAEAAHoBAAB6AQAAewEAAJkBAACSAQAAlAEAAH8BAAB/AQAAfgEAAJIBAACQAQAAeAEAAHcBAAB3AQAAjwEAAJABAAB2AQAAjQEAAI4BAABOAQAATQEAAIsBAACDAQAAKAEAACcBAACGAQAAJAEAACEBAAAhAQAAhQEAAIYBAACDAQAATAEAAEkBAABJAQAAgAEAAIMBAACBAQAAgAEAAH8BAABLAQAAfAEAAH0BAAB6AQAAeQEAAHEBAABxAQAAcgEAAHoBAAB1AQAAdgEAAHABAABwAQAAbgEAAHUBAAByAQAAcQEAAGkBAABpAQAAagEAAHIBAABeAQAAXAEAAGMBAABjAQAAYgEAAF4BAABlAQAAbAEAAGsBAABrAQAAZgEAAGUBAABZAQAAYAEAAGYBAABmAQAAZwEAAFkBAABdAQAAXgEAAGIBAABiAQAAYQEAAF0BAABUAQAAXQEAAF8BAABfAQAAUwEAAFQBAABaAQAAWAEAAFEBAABRAQAAWQEAAFoBAABQAQAARAEAAEEBAABBAQAATwEAAFABAABLAQAASgEAAEYBAABGAQAARQEAAEsBAABHAQAAOgEAADsBAAA7AQAARgEAAEcBAABDAQAAQgEAAD4BAAA+AQAAPwEAAEMBAAA5AQAAPwEAADMBAAAzAQAANAEAADkBAAA1AQAAOgEAADkBAAA5AQAANAEAADUBAAA2AQAANQEAAC4BAAAuAQAALwEAADYBAAAqAQAAKwEAADIBAAAyAQAAMQEAACoBAAAmAQAAJwEAAB0BAAAdAQAAHgEAACYBAAAbAQAAIwEAACIBAAAiAQAAGgEAABsBAAASAQAAHwEAAB4BAAAeAQAAEQEAABIBAAAbAQAAGgEAABUBAAAVAQAAFgEAABsBAAAVAQAAGAEAAA0BAAANAQAADgEAABUBAAAMAQAAEwEAABIBAAASAQAACwEAAAwBAAAIAQAADgEAAA0BAAANAQAABwEAAAgBAAAKAQAAAwEAAAQBAAAEAQAACQEAAAoBAAAEAQAABQEAAAsBAAALAQAACQEAAAQBAAADAQAACgEAAA8BAAAPAQAAAgEAAAMBAAAWAQAAFQEAAA4BAAAOAQAAFAEAABYBAAATAQAADAEAAA0BAAANAQAAGAEAABMBAAAcAQAAGwEAABYBAAAWAQAAFwEAABwBAAATAQAAIAEAAB8BAAAfAQAAEgEAABMBAAAhAQAAJAEAACABAAAgAQAAGQEAACEBAAAoAQAAHAEAAB0BAAAdAQAAJwEAACgBAAAzAQAAMgEAACsBAAArAQAALAEAADMBAAA3AQAANgEAAC8BAAAvAQAAMAEAADcBAAA2AQAAOwEAADoBAAA6AQAANQEAADYBAABAAQAANwEAADgBAAA4AQAAPAEAAEABAABBAQAARAEAADwBAAA8AQAAPQEAAEEBAABIAQAAOQEAADoBAAA6AQAARwEAAEgBAABJAQAATAEAAEgBAABIAQAARwEAAEkBAABLAQAARQEAAEQBAABEAQAAUAEAAEsBAABbAQAAVwEAAFgBAABYAQAAWgEAAFsBAABTAQAAXwEAAGABAABgAQAAUgEAAFMBAABkAQAAYwEAAFwBAABcAQAAWwEAAGQBAABZAQAAZwEAAGgBAABoAQAAWgEAAFkBAABhAQAAbQEAAGwBAABsAQAAZQEAAGEBAABhAQAAYgEAAG8BAABvAQAAbQEAAGEBAABzAQAAcgEAAGoBAABqAQAAawEAAHMBAAB3AQAAbwEAAHABAABwAQAAdgEAAHcBAAB7AQAAegEAAHIBAAByAQAAcwEAAHsBAAB+AQAASgEAAEsBAABLAQAAfQEAAH4BAAAjAQAAgAEAAIEBAAAjAQAAKAEAAIMBAACDAQAAgAEAACMBAACIAQAAJgEAACUBAAAlAQAAhwEAAIgBAABNAQAATAEAAIMBAACMAQAATwEAAE4BAABOAQAAiwEAAIwBAAB1AQAAggEAAI0BAACNAQAAdgEAAHUBAACRAQAAdAEAAHgBAAB4AQAAkAEAAJEBAACUAQAAlQEAAIEBAACBAQAAfwEAAJQBAAB9AQAAeQEAAJoBAACaAQAAkwEAAH0BAACdAQAAnAEAAJABAACQAQAAjwEAAJ0BAAChAQAAoAEAAIsBAACLAQAAigEAAKEBAAClAQAApAEAAIcBAACHAQAAhgEAAKUBAACpAQAAlgEAAJUBAACVAQAAqAEAAKkBAACtAQAArAEAAJgBAACYAQAAmQEAAK0BAAC1AQAAtAEAALMBAAC1AQAAswEAAK8BAAC1AQAArwEAAK0BAAC5AQAAqAEAAKYBAACmAQAAuAEAALkBAAC+AQAAvQEAALoBAAC6AQAAuQEAAL4BAACwAQAAwQEAALYBAAC2AQAArgEAALABAADIAQAAxwEAAMQBAADEAQAAxQEAAMgBAADJAQAAxgEAALwBAAC9AQAAygEAALwBAADNAQAAzgEAAKEBAAChAQAAogEAAM0BAADOAQAA0gEAAKABAACgAQAAoQEAAM4BAADYAQAA2QEAAJ4BAACeAQAAnwEAANgBAADfAQAA3gEAANwBAADcAQAA3QEAAN8BAADkAQAA2QEAANoBAADaAQAA4QEAAOQBAADnAQAA5gEAAOEBAADhAQAA4gEAAOcBAADTAQAA1gEAANsBAADbAQAA6AEAANMBAADsAQAA6wEAAO4BAADuAQAA7QEAAOwBAAD0AQAA8wEAAOoBAADqAQAA8gEAAPQBAACwAQAAsQEAAPABAADwAQAA9QEAALABAADxAQAA7wEAALIBAACyAQAA9wEAAPEBAAD3AQAA+AEAAP0BAAD9AQAA+gEAAPcBAACbAQAAtQEAAK0BAACtAQAAmQEAAJsBAAD/AQAA/gEAAOwBAADsAQAA7QEAAP8BAADuAQAAAAIAAPMBAADuAQAA8wEAAPQBAADuAQAA9AEAAPsBAADuAQAA+wEAAPwBAADuAQAA/AEAAP8BAADuAQAA/wEAAO0BAAACAgAA1QEAANEBAADRAQAAAQIAAAICAADUAQAA1QEAAAICAADUAQAAAgIAAAMCAADUAQAAAwIAAPYBAAADAgAAAgIAAL8BAAC/AQAAwAEAAAMCAAABAgAAygEAAL0BAAC9AQAAvgEAAAECAADrAQAA6QEAAAACAAAAAgAA7gEAAOsBAAD5AQAAtAEAAOYBAADmAQAA5wEAAPkBAAD/AQAA/AEAAP0BAAD9AQAA/gEAAP8BAADyAQAA+gEAAPsBAAD7AQAA9AEAAPIBAAD4AQAA9wEAALIBAACyAQAAswEAAPgBAAD2AQAA9QEAANcBAADXAQAA1AEAAPYBAADxAQAA4wEAAN8BAADfAQAA7wEAAPEBAADiAQAA4AEAAOkBAADpAQAA6wEAAOIBAADSAQAAzgEAAM8BAADPAQAA6AEAANIBAADhAQAA5gEAAOUBAADlAQAA5AEAAOEBAADfAQAA4wEAAOABAADgAQAA3gEAAN8BAADWAQAA3AEAANoBAADaAQAA2wEAANYBAADUAQAA1wEAANYBAADWAQAA0wEAANQBAADLAQAA0AEAAM8BAADPAQAAzAEAAMsBAADNAQAAyAEAAMUBAADFAQAAzAEAAM0BAAC8AQAAwwEAALsBAADCAQAAxwEAAKQBAACkAQAApQEAAMIBAADEAQAAwwEAALwBAADAAQAAvwEAALgBAAC4AQAAtwEAAMABAAC7AQAAqgEAAKkBAACpAQAAugEAALsBAAC3AQAApwEAAKsBAACrAQAAtgEAALcBAACxAQAAsAEAAK4BAACuAQAArwEAALEBAACrAQAApwEAAJMBAACTAQAAmgEAAKsBAACnAQAApgEAAJQBAACnAQAAlAEAAJIBAACnAQAAkgEAAJMBAACjAQAAogEAAIkBAACJAQAAiAEAAKMBAACfAQAAngEAAI4BAACOAQAAjAEAAJ8BAACbAQAAmQEAAHsBAAB7AQAAkQEAAJsBAACWAQAAlwEAAIUBAACFAQAAhAEAAJYBAACTAQAAkgEAAH4BAAB+AQAAfQEAAJMBAACPAQAAdwEAAHYBAAB2AQAAjgEAAI8BAACOAQAAjQEAAIwBAACLAQAATQEAAIMBAACDAQAAigEAAIsBAAAlAQAAJAEAAIYBAACGAQAAhwEAACUBAACFAQAAIQEAACIBAAAiAQAAhAEAAIUBAABLAQAAUAEAAIIBAACCAQAAfAEAAEsBAACAAQAASQEAAH8BAAB9AQAAfAEAAHkBAABtAQAAbwEAAHcBAAB3AQAAeAEAAG0BAABxAQAAdQEAAG4BAABuAQAAaQEAAHEBAABuAQAAcAEAAGMBAABjAQAAZAEAAG4BAABbAQAAWgEAAGgBAABoAQAAZAEAAFsBAABmAQAAawEAAGoBAABqAQAAZwEAAGYBAABgAQAAXwEAAGUBAABlAQAAZgEAAGABAABVAQAAVgEAAFwBAABcAQAAXgEAAFUBAABUAQAAVQEAAF4BAABeAQAAXQEAAFQBAABYAQAAVwEAAFYBAABYAQAAVgEAAFUBAABYAQAAVQEAAFQBAABYAQAAVAEAAFMBAABYAQAAUwEAAFIBAABYAQAAUgEAAFEBAABPAQAAQQEAAEIBAABCAQAATgEAAE8BAABKAQAASQEAAEcBAABHAQAARgEAAEoBAABGAQAAOwEAAEABAABAAQAARQEAAEYBAABCAQAAQQEAAD0BAAA9AQAAPgEAAEIBAAA/AQAAPgEAADIBAAAyAQAAMwEAAD8BAAAqAQAAMQEAADgBAAA4AQAAKQEAACoBAAA1AQAANAEAAC0BAAAtAQAALgEAADUBAAAwAQAALwEAAC4BAAAwAQAALgEAAC0BAAAwAQAALQEAACwBAAAwAQAALAEAACsBAAAwAQAAKwEAACoBAAAwAQAAKgEAACkBAAAlAQAAJgEAAB4BAAAeAQAAHwEAACUBAAAiAQAAIQEAABkBAAAZAQAAGgEAACIBAAARAQAAHgEAAB0BAAAdAQAAEAEAABEBAAAaAQAAGQEAABgBAAAYAQAAFQEAABoBAAAKAQAAEAEAABcBAAAXAQAADwEAAAoBAAALAQAAEgEAABEBAAARAQAACQEAAAsBAAAHAQAADQEAAAwBAAAMAQAABgEAAAcBAAAIAQAABwEAAAYBAAAIAQAABgEAAAUBAAAIAQAABQEAAAQBAAAIAQAABAEAAAMBAAAIAQAAAwEAAAIBAAAGAQAADAEAAAsBAAALAQAABQEAAAYBAAAJAQAAEQEAABABAAAQAQAACgEAAAkBAAAXAQAAFgEAABQBAAAUAQAADwEAABcBAAAUAQAADgEAAAgBAAAUAQAACAEAAAIBAAAUAQAAAgEAAA8BAAAdAQAAHAEAABcBAAAXAQAAEAEAAB0BAAAYAQAAGQEAACABAAAgAQAAEwEAABgBAAAkAQAAJQEAAB8BAAAfAQAAIAEAACQBAAAjAQAAGwEAABwBAAAcAQAAKAEAACMBAAA0AQAAMwEAACwBAAAsAQAALQEAADQBAAAwAQAAKQEAADgBAAA4AQAANwEAADABAAAxAQAAPQEAADwBAAA8AQAAOAEAADEBAAA7AQAANgEAADcBAAA3AQAAQAEAADsBAABFAQAAQAEAADwBAAA8AQAARAEAAEUBAABDAQAAPwEAADkBAAA5AQAASAEAAEMBAABNAQAAQwEAAEgBAABIAQAATAEAAE0BAABCAQAAQwEAAE0BAABNAQAATgEAAEIBAABcAQAAVgEAAFcBAABXAQAAWwEAAFwBAABSAQAAYAEAAFkBAABZAQAAUQEAAFIBAABhAQAAZQEAAF8BAABfAQAAXQEAAGEBAABnAQAAagEAAGkBAABpAQAAaAEAAGcBAABoAQAAaQEAAG4BAABuAQAAZAEAAGgBAABwAQAAbwEAAGIBAABiAQAAYwEAAHABAAB0AQAAcwEAAGsBAABrAQAAbAEAAHQBAAB0AQAAbAEAAG0BAABtAQAAeAEAAHQBAAB5AQAAfAEAAHEBAAB/AQAASQEAAEoBAABKAQAAfgEAAH8BAACCAQAAdQEAAHEBAABxAQAAfAEAAIIBAAAiAQAAIwEAAIEBAACBAQAAhAEAACIBAACJAQAAJwEAACYBAAAmAQAAiAEAAIkBAACKAQAAgwEAACcBAAAnAQAAiQEAAIoBAACNAQAATwEAAIwBAACCAQAAUAEAAE8BAABPAQAAjQEAAIIBAACRAQAAewEAAHMBAABzAQAAdAEAAJEBAACVAQAAlgEAAIQBAACEAQAAgQEAAJUBAAB5AQAAegEAAJgBAACYAQAAmgEAAHkBAACeAQAAnQEAAI8BAACPAQAAjgEAAJ4BAACiAQAAoQEAAIoBAACKAQAAiQEAAKIBAACXAQAApQEAAIYBAACGAQAAhQEAAJcBAACqAQAAlwEAAJYBAACWAQAAqQEAAKoBAACvAQAArgEAAKwBAACsAQAArQEAAK8BAACuAQAAtgEAAKsBAACrAQAArAEAAK4BAAC6AQAAqQEAAKgBAACoAQAAuQEAALoBAAC/AQAAvgEAALkBAAC5AQAAuAEAAL8BAACqAQAAwgEAAKUBAAClAQAAlwEAAKoBAACjAQAApAEAAMcBAADHAQAAyAEAAKMBAAC7AQAAwwEAAMIBAADCAQAAqgEAALsBAADMAQAAxQEAAMYBAADGAQAAywEAAMwBAADMAQAAzwEAAM4BAADOAQAAzQEAAMwBAADVAQAA1AEAANMBAADTAQAA0AEAANUBAADbAQAA2gEAANkBAADZAQAA2AEAANsBAADiAQAA4QEAAN4BAADeAQAA4AEAAOIBAACdAQAAngEAANkBAADZAQAA5AEAAJ0BAADoAQAAzwEAANABAADQAQAA0wEAAOgBAADjAQAA6gEAAOkBAADpAQAA4AEAAOMBAADwAQAA7wEAAN8BAADfAQAA3QEAAPABAADwAQAAsQEAALIBAACyAQAA7wEAAPABAAD2AQAAwQEAALABAACwAQAA9QEAAPYBAADxAQAA9wEAAPoBAAD6AQAA8gEAAPEBAAD+AQAA/QEAAPgBAAD4AQAA+QEAAP4BAACcAQAA5QEAALUBAAC1AQAAmwEAAJwBAADqAQAA8wEAAAACAAAAAgAA6QEAAOoBAAAxAQAAMgEAAD4BAAA+AQAAPQEAADEBAAACAgAAAQIAAL4BAAC+AQAAvwEAAAICAAA9AwAAOwMAADwDAAA8AwAAPgMAAD0DAADlBAAARQMAAEYDAABGAwAA5gQAAOUEAABFAwAAQwMAAEQDAABEAwAARgMAAEUDAABVBQAAYQUAAGMFAABjBQAAVgUAAFUFAABZBQAAVQUAAFcFAABXBQAAWAUAAFkFAABIBQAASgUAAE0FAABNBQAASQUAAEgFAABCBQAARAUAAEUFAABFBQAAQwUAAEIFAAD7BAAAPAUAAD0FAAA9BQAA/AQAAPsEAAAyBQAANAUAADUFAAA1BQAAMwUAADIFAAAoBQAALAUAAC0FAAAtBQAAKQUAACgFAAAFBQAAJAUAACUFAAAlBQAABgUAAAUFAAAYBQAAEAUAABMFAAATBQAAFAUAABgFAAAIBQAAAQUAAAAFAAAABQAABwUAAAgFAAD5BAAA+wQAAP0EAAD9BAAA+gQAAPkEAABWBQAAVwUAAFUFAAAcBQAAHQUAAB4FAAAcBQAAHgUAAB8FAAAcBQAAHwUAABsFAAAFBQAABgUAAAQFAAD0BAAA8gQAAPEEAADxBAAA8wQAAPQEAADqBAAA6AQAAOcEAADnBAAA6QQAAOoEAAAwBAAA2AQAANkEAADZBAAA1wQAADAEAADVBAAA1gQAAAsFAAALBQAACQUAANUEAADRBAAA0gQAAP8EAAD/BAAAAwUAANEEAADNBAAAzgQAACcFAAAnBQAADQUAAM0EAAA8AgAAMgIAAL8EAAC/BAAA/AMAADwCAAC+BAAA9AMAAMUEAADFBAAAxgQAAL4EAAD4AwAA+QQAAPgEAAD4BAAAwgQAAPgDAAC4BAAAKAQAANsEAADbBAAA3gQAALgEAAD0AwAA/gMAALwEAAC8BAAAwQQAAPQDAAAuBAAALQQAALYEAAC2BAAAtQQAAC4EAAA1BAAA9AQAAPYEAAD2BAAArgQAADUEAAC0BAAALgQAAOQEAADkBAAA5gQAALQEAAD1AwAAvwQAAMYEAADGBAAAxQQAAPUDAAAvBAAANAQAAKwEAACsBAAAsQQAAC8EAAA6BAAAQQQAALAEAACwBAAArwQAADoEAAA5BAAANAQAAKsEAACrBAAApwQAADkEAAA9BAAAagQAAJ4EAACeBAAAoAQAAD0EAABqBAAAPQQAAKMEAACjBAAAnwQAAGoEAABgBAAAXgQAAI4EAACOBAAAlQQAAGAEAABXBAAAWAQAAHwEAAB8BAAAdQQAAFcEAABeBAAAYAQAAJQEAACUBAAAjQQAAF4EAABDBAAAbwQAAEYEAABDBAAARgQAAIwEAABDBAAAjAQAAJMEAABQBAAAUwQAAHkEAAB5BAAAewQAAFAEAABTBAAAUAQAAHoEAAB6BAAAeAQAAFMEAABJBAAARgQAAIsEAACLBAAAhgQAAEkEAABtBAAAPwQAAJoEAACaBAAAmAQAAG0EAACXAgAAmAIAAIoEAACKBAAAjwQAAJcCAACgAgAAoQIAAHcEAAB3BAAAfgQAAKACAABHBAAAiwQAAJAEAACQBAAAbgQAAEcEAACKAgAAxQIAAJ8EAACfBAAAowQAAIoCAABIBAAAZQQAAIQEAACEBAAAhgQAAEgEAADPAwAA1QMAAGQEAAC5BAAAYgQAANwEAABXBAAA2QMAAHYEAAA1BQAANwUAAFIFAABSBQAAUAUAADUFAAAyBAAArQQAAK8EAABKBAAAiAQAAI0EAACNBAAARQQAAEoEAACJBAAAXQQAAF8EAABfBAAAjgQAAIkEAABRBAAAUgQAAFYEAABRBAAAVgQAAHUEAABRBAAAdQQAAHwEAABWBAAAdAQAAFUEAABWBAAAVQQAAOoDAABWBAAA6gMAANsDAAB4BAAAegQAAE4EAABOBAAATwQAAHgEAABOBAAATAQAAIMEAACDBAAAfgQAAE4EAABlBAAATAQAAH8EAAB/BAAAhAQAAGUEAABuBAAAQgQAAJcEAACXBAAAkQQAAG4EAACaBAAAPgQAAGwEAABsBAAAmAQAAJoEAACeBAAAagQAAD4EAAA+BAAAmgQAAJ4EAAA8BAAAoQQAAKcEAAA8BAAApwQAADgEAAA8BAAAOAQAAGkEAACuBAAAgAIAAIICAACCAgAAqQQAAK4EAAAzBAAAMAQAALEEAACxBAAArAQAADMEAAC3BAAALAQAAPAEAAC3BAAA8AQAAO8EAAC3BAAA7wQAALIEAAB3AgAAtAQAAOYEAADmBAAARgMAAHcCAAAsBAAAKQQAALUEAAC1BAAAtgQAACwEAAA9BQAAPwUAAN8EAADfBAAA3QQAAD0FAAAmBAAAKgQAACUEAAAmBAAAJQQAAAMEAAAmBAAAAwQAABIEAAAmBAAAEgQAACcEAAAiBAAAZwIAAGsCAABrAgAAIwQAACIEAADoAwAAIwIAAGcCAABnAgAAIgQAAOgDAADxAwAAIQQAACAEAAAgBAAA7gMAAPEDAAAdBAAAXwIAAF4CAABeAgAAGwQAAB0EAAAXBAAAFgQAABUEAAAVBAAAGgQAABcEAAAXBAAAGQQAAMkEAADJBAAAygQAABcEAAAVBAAAFgQAABQEAAAJBAAASAUAAEcFAABHBQAACAQAAAkEAAC5AgAAEAQAABEEAAARBAAAuAIAALkCAADOBAAAzQQAANYEAADOBAAA1gQAANUEAADOBAAA1QQAANQEAAANBAAAyAQAAA8EAAAPBAAADgQAAA0EAAAJBAAACAUAAAoFAAAKBQAACgQAAAkEAAACBAAAAQQAAMcEAADHBAAABgQAAAIEAADXAwAAAQQAAAIEAAACBAAA1gMAANcDAABGBQAA+QMAAMIEAADCBAAA+AQAAEYFAAA8BQAAuwQAAP0DAAD9AwAAPgUAADwFAAD1AwAA/AMAAL8EAADDBAAAwgQAAPkDAAD5AwAA9wMAAMMEAADuAwAA7wMAAPIDAADyAwAA8QMAAO4DAADcAwAA5gMAAOwDAADsAwAA6wMAANwDAABwBAAAJQIAACQCAABwBAAAJAIAAOcDAABwBAAA5wMAAOkDAAAhBQAA4QMAAOMDAADjAwAAGgUAACEFAABgBQAA5AMAAOIDAADiAwAAXQUAAGAFAADhAwAA4gMAAOQDAADkAwAA4wMAAOEDAABhBQAA3QMAAFEFAABhBQAAUQUAAFQFAABhBQAAVAUAAGIFAAA7BQAAOgUAANgDAAA7BQAA2AMAANcDAAA7BQAA1wMAAM8DAADOAwAAZAQAANUDAADUAwAAywMAAMoDAADTAwAAzwMAANIDAADSAwAA0QMAANMDAADOAwAAKAUAACsFAAArBQAAZAQAAM4DAAAyBQAAygMAAMsDAADLAwAANAUAADIFAAATAgAA0wIAAK4CAACuAgAAFAIAABMCAADAAwAAvgMAAL8DAAC/AwAAwQMAAMADAAC0AwAAhwMAAIYDAACGAwAAtQMAALQDAACpAwAAqwMAAKoDAACqAwAAqAMAAKkDAAChAwAAowMAAKIDAACiAwAAoAMAAKEDAACZAwAAmwMAAJoDAACaAwAAmAMAAJkDAACRAwAAkwMAAJIDAACSAwAAkAMAAJEDAACJAwAAiwMAAIoDAACKAwAAiAMAAIkDAAB/AwAAfgMAAH0DAAB9AwAAgwMAAH8DAABtAwAAawMAAGwDAABsAwAAbgMAAG0DAABkAwAAYwMAAGIDAABiAwAAZQMAAGQDAADEAwAAxgMAAMUDAACcAwAAnwMAAJ4DAACeAwAAnQMAAJwDAABxAwAAcwMAAHIDAABWAwAAWAMAAFcDAABXAwAAVQMAAFYDAABMAwAATgMAAE0DAABNAwAASwMAAEwDAAByAgAANwMAADUDAABvAwAAcQMAACsDAAArAwAANAMAAG8DAABjAwAAZAMAADEDAAAxAwAAMAMAAGMDAACLAwAAiQMAAC0DAAAtAwAALAMAAIsDAACxAwAAOwIAAEsCAABLAgAAsAMAALEDAAAjAwAAJAMAABsDAAAbAwAAMQIAACMDAAAgAwAAHwMAADcCAAA3AgAANQIAACADAAAUAwAAFQMAAHACAABwAgAAcwIAABQDAAAdAwAAIQMAACIDAAAiAwAANAIAAB0DAABCAwAAQAMAABIDAAASAwAAdgIAAEIDAABUAwAABwMAAH8CAAB/AgAAVgMAAFQDAAAQAwAAEQMAAHYCAAB2AgAAdQIAABADAAAaAwAAGwMAADMCAAAzAgAAPgIAABoDAAAXAwAADgMAAHkCAAB5AgAAcwIAABcDAAAKAwAADAMAAIYCAACGAgAAfQIAAAoDAAD+AgAABAMAAIUCAACFAgAAhwIAAP4CAAABAwAA/QIAAIkCAACJAgAAwgIAAAEDAAD6AgAA/AIAAMQCAADEAgAAjAIAAPoCAAAGAwAACwMAAH8CAAB/AgAAgwIAAAYDAADfAgAA2gIAAK0CAACtAgAArwIAAN8CAADlAgAA6gIAALMCAACzAgAAsQIAAOUCAAD4AgAA8AIAAJICAAD4AgAAkgIAAMkCAAD4AgAAyQIAAI0CAADdAgAA2AIAAKQCAACkAgAAnQIAAN0CAADQAgAA1QIAAKcCAACnAgAAqgIAANACAADhAgAA4wIAAJsCAACbAgAAvwIAAOECAADvAgAA9QIAAMkCAADJAgAAkgIAAO8CAADuAgAA7AIAAJcCAACXAgAAywIAAO4CAADgAgAA2wIAAKACAACgAgAAmQIAAOACAADsAgAA7gIAAMoCAADKAgAAlgIAAOwCAADGAgAA+wIAAP0CAAD9AgAAiAIAAMYCAADAAgAAjgMAAJwDAACPAwAAvAIAAAkCAAAJAgAAnwMAAI8DAAC6AgAAOAMAAGADAAC6AgAAYAMAAGEDAAC6AgAAYQMAAG8CAACsAgAArQIAANoCAADaAgAA0wIAAKwCAAAHAwAAVAMAAFADAABQAwAAgQIAAAcDAAB3AwAAuQMAALoDAAB3AwAAugMAAMgDAAB3AwAAyAMAAMkDAAB3AwAAyQMAAHYDAADyAgAADQMAALYCAADlAgAA3gIAAJ8CAACfAgAAnAIAAOUCAADmAgAAsgIAALACAACwAgAA3wIAAOYCAADSAgAArAIAABMCAADSAgAAEwIAABUCAADSAgAAFQIAAKsCAACmAgAAqwIAANECAADRAgAA1gIAAKYCAADWAgAA2AIAAKUCAAClAgAApgIAANYCAACiAgAAowIAANQCAADUAgAA2wIAAKICAADkAgAA6QIAAJQCAACUAgAAnAIAAOQCAADKAgAAkAIAAO8CAADvAgAA7QIAAMoCAAD2AgAAxwIAAIsCAACLAgAA+gIAAPYCAAD6AgAAiwIAAMUCAADFAgAA/AIAAPoCAAAFAwAA/wIAAIcCAACHAgAAhQIAAAUDAAAPAwAACAMAAIECAAAPAwAAgQIAAFADAAAPAwAAUAMAAFEDAAAKAwAANgMAAHsCAAB7AgAAfAIAAAoDAABOAwAAEwMAAHQCAAB0AgAAUgMAAE4DAABDAwAARQMAAKsDAACrAwAAqQMAAEMDAABxAgAAFQMAABQDAAAUAwAAdAIAAHECAABvAgAAbgIAALoCAABtAgAARAIAABQCAAAUAgAArgIAAG0CAAAwAgAAagIAAGkCAABpAgAALQIAADACAABlAgAAYwIAAGQCAABkAgAAZgIAAGUCAABWAgAAXAIAAGMCAABjAgAAZQIAAFYCAABiAgAAXwIAAGACAABgAgAAYQIAAGICAABNAgAAUgIAAFkCAAAoAwAAWAIAAFcCAABXAgAAKQMAACgDAABVAgAAVgIAAFQCAABUAgAAQQIAAFUCAABKAgAArgMAALADAACwAwAASwIAAEoCAABRAgAAOwIAADwCAAA8AgAAuQIAAFECAACKAwAAJgMAAFACAABQAgAAiAMAAIoDAAAmAwAAcgMAAHMDAAAmAwAAcwMAAE0CAAAmAwAATQIAAE4CAABKAgAAbAMAAGUDAABlAwAASQIAAEoCAAAlAwAAaAMAAGkDAAAlAwAAaQMAAEYCAAAlAwAARgIAAEICAAARAgAAQgIAAEACAABAAgAAEgIAABECAABfAwAAOgIAADcCAAA3AgAAXQMAAF8DAAA/AgAAMQIAABsDAAAbAwAAGgMAAD8CAAA0AgAAOwIAALEDAAA0AgAAsQMAALIDAAA0AgAAsgMAAB0DAAAdAwAANgIAACADAAAgAwAAIQMAAB0DAAAqAgAAKQIAACwCAAAsAgAALQIAACoCAAC9AwAAvgMAACECAAB/AwAAHgIAAB0CAAAdAgAAfgMAAH8DAAB9AwAAhAMAAJMDAACTAwAAkQMAAH0DAAAaAgAAHAIAAMMDAADDAwAAwQMAABoCAAAbAgAAHAIAABoCAAAaAgAAGQIAABsCAACZAwAAlwMAALQDAAC0AwAAtgMAAJkDAAAJAgAAEQIAABICAAAJAgAAEgIAAJ4DAAAJAgAAngMAAJ8DAAAPAgAAvAIAAAgCAAAEAgAABQIAAA4CAAALAgAADAIAAAkCAAAJAgAADQIAAAsCAAC8AgAAjwMAAIwDAACMAwAACAIAALwCAACYAwAABQIAAAQCAAAEAgAAlgMAAJgDAAAHAgAACAIAAIwDAACMAwAAkAMAAAcCAAALAgAAeQMAAHoDAAB6AwAADAIAAAsCAAAOAgAABQIAAA0CAAAPAgAADgIAAA0CAAAQAgAADAIAAHoDAAAQAgAAegMAAHsDAAAQAgAAewMAABQCAACHAwAAlQMAAJMDAACTAwAAhAMAAIcDAAAZAgAAGgIAABgCAAAYAgAAFwIAABkCAAAYAgAAGgIAAMEDAADBAwAAvwMAABgCAACFAwAAGwIAABkCAACFAwAAGQIAALUDAACFAwAAtQMAAIYDAACAAwAAxQMAAMYDAACAAwAAxgMAAB4CAACAAwAAHgIAAH8DAAB1AwAAJwIAABUCAAAVAgAAdAMAAHUDAAArAgAALAIAACkCAAApAgAAKAIAACsCAAAkAwAAIwMAADICAAAyAgAAHAMAACQDAAA3AgAAOgIAADUCAAAZAwAAHgMAADUCAAA1AgAAPQIAABkDAACqAwAAPwIAABoDAAAaAwAAqAMAAKoDAABAAgAAQQIAABYCAAAWAgAAEgIAAEACAABCAgAARgIAAEUCAABFAgAAQAIAAEICAABJAgAAZQMAAGIDAABiAwAASAIAAEkCAABNAgAAcwMAAHADAABwAwAATAIAAE0CAABpAwAAiAMAAFACAABQAgAARgIAAGkDAAAtAwAAMQMAADIDAAAyAwAALAMAAC0DAABLAgAAOwIAAFECAABRAgAATAIAAEsCAABUAgAAKAIAABYCAAAWAgAAQQIAAFQCAABAAgAAKgMAAFUCAABVAgAAQQIAAEACAABZAgAAUgIAALgCAAC4AgAAtwIAAFkCAABbAgAAXgIAAF0CAABdAgAAWgIAAFsCAABhAgAAZAIAAGMCAABjAgAAYgIAAGECAABUAgAAKwIAACgCAAAtAgAAaQIAAGgCAABoAgAAKgIAAC0CAABgAgAAbAIAAGoCAABgAgAAagIAADACAABgAgAAMAIAAC8CAABgAgAALwIAAC4CAABgAgAALgIAAGYCAABgAgAAZgIAAGQCAABgAgAAZAIAAGECAABeAwAAWwMAAEgCAABIAgAAUwIAAF4DAAAXAwAAFgMAALoCAAC6AgAANQMAABcDAAA9AwAAPwMAAKUDAAClAwAAowMAAD0DAABHAwAAeAIAABADAAAQAwAASgMAAEcDAAB7AgAAcgIAAG0CAABTAwAATQMAAE8DAAAEAwAAhAIAAIECAACBAgAACAMAAAQDAADBAgAAggIAAAYDAAAGAwAAAgMAAMECAAD5AgAA/wIAAIYCAACGAgAAjwIAAPkCAADpAgAA8AIAAJMCAACTAgAAlAIAAOkCAADjAgAAmgIAAJYCAACWAgAA6AIAAOMCAACcAgAAnwIAAN0CAADdAgAA5AIAAJwCAACfAgAApQIAANgCAADYAgAA3QIAAJ8CAADQAgAAqQIAAKMCAACjAgAA1QIAANACAADNAgAAzwIAAKkCAACpAgAAJgIAAM0CAADaAgAA3wIAALACAACwAgAArgIAANoCAADxAgAA6gIAAJQCAACUAgAAkwIAAPECAACTAgAAjgIAAPkCAAD5AgAA8QIAAJMCAAAnAgAAdQMAAHYDAAAnAgAAdgMAAMkDAAAnAgAAyQMAALwDAABaAwAAWAMAAAsDAAALAwAAgAIAAFoDAAAJAwAABQMAAIUCAACFAgAAfgIAAAkDAAByAgAANQMAALoCAAC6AgAAbgIAAHICAAC7AgAAlAMAAJYDAACWAwAABAIAALsCAADIAgAA9QIAAO8CAADvAgAAkAIAAMgCAAD9AgAAAQMAAMMCAADDAgAAiAIAAP0CAADPAgAAzQIAACUCAAAlAgAAqAIAAM8CAADiAgAA4AIAAJkCAACZAgAAvQIAAOICAAD0AgAA7gIAAMsCAADLAgAAkQIAAPQCAADtAgAA7wIAAJICAACSAgAAzAIAAO0CAADcAgAA4QIAAL8CAAC/AgAAnQIAANwCAADOAgAA0AIAAKoCAACqAgAAJwIAAM4CAADkAgAA3QIAAJ0CAADkAgAAnQIAAL8CAADkAgAAvwIAAJsCAAD+AgAA+AIAAI0CAAD+AgAAjQIAAMYCAAD+AgAAxgIAAIgCAADeAgAA5QIAALECAACxAgAArwIAAN4CAADmAgAA3wIAAK8CAACvAgAAsQIAAOYCAAACAwAABgMAAIMCAACDAgAAwgIAAAIDAAD2AgAA+gIAAIwCAACMAgAAyAIAAPYCAAADAwAAAQMAAMICAADCAgAAgwIAAAMDAAD/AgAA+QIAAI4CAACOAgAAhwIAAP8CAAAFAwAACQMAAH0CAAB9AgAAhgIAAAUDAAAWAwAAFwMAAHMCAABzAgAAcAIAABYDAACoAwAAGgMAAD4CAAA+AgAApgMAAKgDAABKAwAAEAMAAHUCAAB1AgAATAMAAEoDAABNAwAAUwMAAFUDAABVAwAASwMAAE0DAAA/AwAAQQMAAKcDAACnAwAApQMAAD8DAACgAwAAGAMAADoCAAA6AgAAXwMAAKADAAAPAwAAFAMAAHMCAABzAgAAeQIAAA8DAAAhAwAAIAMAADUCAAA1AgAAHgMAACEDAACzAwAArwMAADkCAAAlAwAARwIAAGYDAABmAwAAaAMAACUDAACJAwAAagMAAC4DAAAuAwAALQMAAIkDAABkAwAAawMAADIDAAAyAwAAMQMAAGQDAAByAgAAewIAADYDAAA2AwAANwMAAHICAABPAwAAUgMAAFEDAABRAwAAUAMAAE8DAABPAwAATQMAAE4DAABOAwAAUgMAAE8DAABYAwAAWgMAAFkDAABZAwAAVwMAAFgDAAB0AwAAdwMAAHYDAAB2AwAAdQMAAHQDAACwAwAAswMAALIDAACyAwAAsQMAALADAADHAwAAyQMAAMgDAABjAwAAZwMAAGYDAABmAwAAYgMAAGMDAABvAwAAbQMAAG4DAABuAwAAcAMAAG8DAAB+AwAAhQMAAIQDAACEAwAAfQMAAH4DAACLAwAAcQMAAHIDAAByAwAAigMAAIsDAACTAwAAlQMAAJQDAACUAwAAkgMAAJMDAACbAwAAeAMAAHkDAAB5AwAAmgMAAJsDAACjAwAApQMAAKQDAACkAwAAogMAAKMDAACuAwAArQMAAKwDAACsAwAArwMAAK4DAAC1AwAAtwMAALYDAAC2AwAAtAMAALUDAADCAwAAwAMAAMEDAADBAwAAwwMAAMIDAADTAgAA2gIAAK4CAAAsBQAAKAUAAM4DAADOAwAAzQMAACwFAADSAwAAFgUAABUFAAAVBQAA0QMAANIDAADTAwAAywMAANQDAADTAwAA1AMAANUDAADaAwAAFwUAABYFAADaAwAAFgUAANIDAADaAwAA0gMAANYDAAApBQAALQUAABkFAAAZBQAAHwUAACkFAADeAwAA3QMAAGEFAABhBQAAVQUAAN4DAABZBQAA3gMAAFUFAABRBQAA3QMAAN8DAADfAwAATwUAAFEFAAApBQAAHwUAAB4FAAApBQAAHgUAAGcEAAApBQAAZwQAACoFAABxBAAA6QMAAOcDAABxBAAA5wMAAFcFAABxBAAAVwUAAFYFAABxBAAAVgUAAOoDAADtAwAA7AMAAOYDAADmAwAA5QMAAO0DAADvAwAA8AMAAPMDAADzAwAA8gMAAO8DAAD5AwAA+gMAAPcDAAD7AwAA9gMAAP0DAAD9AwAAuwQAAPsDAAA+BQAA/QMAALwEAAC8BAAAQAUAAD4FAAD6AwAA+QMAAEYFAABGBQAASQUAAPoDAADWAwAAAgQAAAMEAAADBAAA2gMAANYDAAAGBAAAAgUAAP4EAAD+BAAABwQAAAYEAAAKBAAACgUAAAwFAAAMBQAACwQAAAoEAAAOBAAADwQAAAUEAAAFBAAABAQAAA4EAADSBAAA0QQAANAEAADSBAAA0AQAAM8EAADSBAAAzwQAANMEAAAMBAAAEQQAABAEAAAQBAAACwQAAAwEAAASBAAAAwQAAAIEAAASBAAAAgQAAAYEAAASBAAABgQAAAcEAAAUBAAAFgQAAMsEAADLBAAAzAQAABQEAAC3AgAAGAQAAMkEAAC3AgAAyQQAABkEAAC3AgAAGQQAAFsCAAAXBAAAGgQAABsEAAAbBAAAGQQAABcEAAAbBAAAGgQAAB4EAAAeBAAAHQQAABsEAADuAwAAIAQAABUEAAAVBAAAEwQAAO4DAADlAwAA6AMAACIEAAAiBAAA7QMAAOUDAAAjBAAAawIAAGwCAABsAgAAJAQAACMEAAAnBAAAEgQAAPoEAAD6BAAA/QQAACcEAAApBAAAuAQAAN4EAADeBAAA4AQAACkEAAC1BAAAKQQAAOAEAADgBAAA4gQAALUEAAC0BAAAdwIAAHgCAAB4AgAAswQAALQEAAAwBAAA1wQAALoEAAC6BAAAsQQAADAEAAD1BAAA5wQAAEgDAABIAwAAWQMAAPUEAAA4BAAANgQAAKoEAACqBAAApgQAADgEAABpBAAAOAQAAKYEAACmBAAApAQAAGkEAABABAAAOwQAAKEEAAChBAAAmwQAAEAEAABEBAAAQAQAAJsEAACbBAAAkwQAAEQEAABIBAAARwQAAI8EAACPBAAAigQAAEgEAACDBAAATAQAAGUEAABlBAAAhQQAAIMEAAB6BAAAfwQAAEwEAABMBAAATgQAAHoEAABUBAAATwQAAHcEAAB3BAAAcgQAAFQEAABxBAAAcwQAAFQEAABUBAAA6QMAAHEEAABNBAAAUQQAAHwEAAB8BAAAgQQAAE0EAACOBAAAXwQAAGEEAABhBAAAlQQAAI4EAACwBAAAYQQAAJYEAACwBAAAlgQAADIEAACwBAAAMgQAAK8EAAAlBAAAMQQAADIEAAAlBAAAMgQAAJYEAAAlBAAAlgQAAGEEAAAlBAAAYQQAAF8EAAAlBAAAXwQAAF0EAAAlBAAAXQQAAFsEAAAlBAAAWwQAAFkEAABSBQAANwUAABQFAABSBQAAFAUAABMFAABSBQAAEwUAAFMFAADZAwAAGAUAABcFAAAXBQAA2gMAANkDAADWAwAA0gMAAM8DAADPAwAA1wMAANYDAAAzBQAATgUAACMFAAAjBQAAMQUAADMFAABIBAAAigQAAIUEAACFBAAAZQQAAEgEAACRAgAAlwQAAJkEAACZBAAAxwIAAJECAACKAgAAowQAAKUEAAClBAAAwQIAAIoCAAChAgAAqAIAAHIEAAByBAAAdwQAAKECAACYAgAAvQIAAIUEAACFBAAAigQAAJgCAAA/BAAAawQAAJ4EAACeBAAAmgQAAD8EAABGBAAAbwQAAJAEAACQBAAAiwQAAEYEAABQBAAASwQAAH8EAAB/BAAAegQAAFAEAABTBAAAVQQAAHQEAAB0BAAAeQQAAFMEAABGBAAASQQAAIcEAACHBAAAjAQAAEYEAABgBAAAQQQAAJwEAACcBAAAlAQAAGAEAABYBAAAWgQAAIEEAACBBAAAfAQAAFgEAABeBAAAXAQAAIkEAACJBAAAjgQAAF4EAAA9BAAAaAQAAKUEAAClBAAAowQAAD0EAABCBAAAbAQAAJkEAACZBAAAlwQAAEIEAAA1BAAANwQAAKYEAACmBAAAqgQAADUEAABBBAAAYAQAAJUEAACVBAAAsAQAAEEEAAA6BAAAqAQAAKIEAAAoBAAAuQQAANwEAADcBAAA2wQAACgEAABDBQAARQUAAOUEAADlBAAA4wQAAEMFAADzBAAA6QQAAOcEAADnBAAA9QQAAPMEAAAtBAAA6gQAAOwEAADsBAAAtgQAAC0EAAD+AwAAQgUAAEAFAABABQAAvAQAAP4DAAD8BAAAPQUAAN0EAADdBAAA2gQAAPwEAAA0BAAALwQAALIEAACyBAAAqwQAADQEAAD0AwAAwQQAAMQEAADEBAAAxQQAAPQDAAD3AwAA+gMAAE0FAAD3AwAATQUAAEwFAAD3AwAATAUAAMAEAAAmBQAAyAQAAA4FAADQBAAA0QQAAAMFAAADBQAABgUAANAEAADUBAAA1QQAAAkFAAAJBQAABwUAANQEAAAwBAAAMwQAAK0EAACtBAAA2AQAADAEAADoBAAARwMAAEgDAABIAwAA5wQAAOgEAADtBAAA8QQAAPIEAADyBAAA7gQAAO0EAAD8BAAA/QQAAPsEAAAVBQAAFgUAABcFAAAVBQAAFwUAABgFAAAVBQAAGAUAABQFAABTBQAAVAUAAFIFAAD6BAAA9wQAAPgEAAD4BAAA+QQAAPoEAAACBQAABAUAAAYFAAAGBQAAAwUAAAIFAAAPBQAADAUAAAsFAAALBQAADQUAAA8FAAAgBQAAIwUAACIFAAAiBQAAIQUAACAFAAArBQAAKAUAACkFAAApBQAAKgUAACsFAAAzBQAAMQUAADAFAAAwBQAAMgUAADMFAAA7BQAAKwUAACoFAAAqBQAAOAUAADsFAABDBQAAQQUAAEAFAABABQAAQgUAAEMFAABGBQAA+AQAAPcEAAD3BAAARwUAAEYFAABUBQAAUQUAAFAFAABQBQAAUgUAAFQFAABgBQAAXQUAAFwFAABcBQAAXgUAAGAFAAAIBAAARwUAAPcEAAD3BAAABwQAAAgEAADbBAAA3AQAAGIEAABiBAAA2gQAANsEAADkBAAA4gQAAOEEAADhBAAA4wQAAOQEAAA/AwAAPQMAAD4DAAA+AwAAQAMAAD8DAABDAwAAQQMAAEIDAABCAwAARAMAAEMDAAA7AwAAOAMAADkDAAA5AwAAPAMAADsDAADiBAAA4AQAAN4EAADiBAAA3gQAAN0EAADiBAAA3QQAAN8EAADiBAAA3wQAAOEEAADkBAAA4wQAAOUEAADlBAAA5gQAAOQEAADaAwAAWQQAAHYEAAB2BAAA2QMAANoDAABbBQAAWQUAAFgFAABYBQAAWgUAAFsFAABPBQAAIgUAACMFAAAjBQAATgUAAE8FAABEBQAAqgMAAKsDAACrAwAARQUAAEQFAAA8BQAAPgUAAD8FAAA/BQAAPQUAADwFAAA0BQAANgUAADcFAAA3BQAANQUAADQFAAAsBQAALgUAAC8FAAAvBQAALQUAACwFAAAkBQAAJgUAACcFAAAnBQAAJQUAACQFAAAfBQAAGQUAABoFAAAaBQAAGwUAAB8FAAAKBQAACAUAAAcFAAAHBQAACQUAAAoFAAABBQAA/gQAAP8EAAD/BAAAAAUAAAEFAABfBQAAYAUAAF4FAAA5BQAAOgUAADsFAAA7BQAAOAUAADkFAAAOBQAADwUAAA0FAADzBAAA9QQAAPYEAAD2BAAA9AQAAPMEAADpBAAA6wQAAOwEAADsBAAA6gQAAOkEAADXBAAA2QQAACoEAADWBAAAzQQAAA0FAAANBQAACwUAANYEAADSBAAA0wQAAAAFAAAABQAA/wQAANIEAADOBAAAzwQAACUFAAAlBQAAJwUAAM4EAABKBQAACgQAAPwDAAD8AwAASwUAAEoFAAAxAgAAvgQAAMYEAADGBAAAIwMAADECAAD2AwAA+AMAAMIEAADCBAAAwwQAAPYDAAArBAAAKAQAALgEAAC4BAAAtwQAACsEAAD1AwAAxQQAAMQEAADEBAAAwAQAAPUDAAAuBAAAtQQAAOIEAADiBAAA5AQAAC4EAAD0BAAANQQAAKoEAACqBAAA8gQAAPQEAAAtBAAALgQAALQEAAC0BAAAswQAAC0EAAD+AwAA9AMAAL4EAAC+BAAAvQQAAP4DAAArBAAALwQAALEEAACxBAAAugQAACsEAAAzBAAAOgQAAK8EAACvBAAArQQAADMEAAA7BAAAOQQAAKcEAACnBAAAoQQAADsEAABoBAAAPQQAAKAEAACgBAAApAQAAGgEAAA+BAAAagQAAJ8EAACfBAAAnQQAAD4EAAA3BAAANQQAAK4EAACuBAAAqQQAADcEAABaBAAAWAQAAH0EAAB9BAAAggQAAFoEAABcBAAAXgQAAI0EAACNBAAAiAQAAFwEAAA/BAAAbQQAAEMEAAA/BAAAQwQAAJMEAAA/BAAAkwQAAJsEAABLBAAAUAQAAHsEAAB7BAAAgAQAAEsEAABVBAAAUwQAAHgEAAB4BAAAcwQAAFUEAABmBAAASQQAAIYEAACGBAAAhAQAAGYEAABDBAAAbQQAAJgEAACYBAAAkgQAAEMEAADLAgAAlwIAAI8EAACPBAAAkQQAAMsCAACZAgAAoAIAAH4EAAB+BAAAgwQAAJkCAABHBAAAbgQAAJEEAACRBAAAjwQAAEcEAAA8BAAAoAQAAJ4EAACeBAAAawQAADwEAAA4BQAAKgUAAGcEAAA7BQAAzwMAAGQEAABkBAAAKwUAADsFAAAnBAAA/QQAAPwEAAAnBAAA/AQAANoEAAAnBAAA2gQAAGIEAAB2BAAAfQQAAFgEAABYBAAAVwQAAHYEAAA2BAAA7gQAAPIEAADyBAAAqgQAADYEAAASBQAAYwUAAGIFAAASBQAAYgUAAFQFAAASBQAAVAUAAFMFAAASBQAAUwUAABMFAABhBAAAsAQAAJUEAABKBAAATQQAAIEEAACBBAAAiAQAAEoEAACCBAAAWwQAAF0EAABdBAAAiQQAAIIEAABWBAAA2wMAANkDAABWBAAA2QMAAFcEAABWBAAAVwQAAHUEAAB5BAAAdAQAAFYEAABWBAAAUgQAAHkEAABSBAAAUQQAAHsEAAB7BAAAeQQAAFIEAAB+BAAAdwQAAE8EAABPBAAATgQAAH4EAABKBAAARQQAAIwEAACMBAAAhwQAAEoEAACQBAAAkgQAAEIEAABCBAAAbgQAAJAEAACdBAAAiwIAAMcCAADHAgAAmQQAAJ0EAACfBAAAxQIAAIsCAACLAgAAnQQAAJ8EAAA5BAAAOwQAAKIEAACiBAAAqAQAADkEAADvBAAA7gQAADYEAADvBAAANgQAAKsEAADvBAAAqwQAALIEAAAyBAAAMQQAANgEAADYBAAArQQAADIEAADwBAAALAQAALYEAAC2BAAA7AQAAPAEAABFBQAAqwMAAEUDAABFAwAA5QQAAEUFAAAsBAAAtwQAALgEAAC4BAAAKQQAACwEAABiBAAAJgQAACcEAABZBAAA2gMAAAMEAAADBAAAJQQAAFkEAADwAwAAIwQAACQEAAAkBAAA8wMAAPADAAAhBAAAHwQAAB4EAAAeBAAAIAQAACEEAAAgBAAAHgQAABoEAAAaBAAAFQQAACAEAAAcBAAAYAIAAF8CAABfAgAAHQQAABwEAAAYBAAAEQQAAAwEAADLBAAAFgQAABcEAAAXBAAAygQAAMsEAAAABAAAEwQAABUEAAAVBAAAFAQAAAAEAAAKBAAASgUAAEgFAABIBQAACQQAAAoEAAC5AgAAPAIAAPwDAAD8AwAAEAQAALkCAAAkBQAADwQAAMgEAADIBAAAJgUAACQFAAANBAAADAQAAA8FAAANBAAADwUAAA4FAAANBAAADgUAAMgEAAAIBAAAAQUAAAgFAAAIBQAACQQAAAgEAAABBAAABQQAAAUFAAABBAAABQUAAAQFAAABBAAABAUAAMcEAADYAwAA/wMAAAEEAAABBAAA1wMAANgDAAD5BAAA+AMAAPsDAAD7AwAA+wQAAPkEAAC9BAAAvgQAADECAAAxAgAAPwIAAL0EAADABAAATAUAAEsFAADABAAASwUAAPwDAADABAAA/AMAAPUDAADEBAAAwwQAAPcDAAD3AwAAwAQAAMQEAADwAwAA7wMAAOwDAADsAwAA7QMAAPADAADnAwAAWAUAAFcFAAAaBQAA4wMAAOQDAADkAwAAGwUAABoFAAAtBQAALwUAACAFAAAgBQAAGQUAAC0FAABbBQAAXQUAAOIDAADiAwAA4AMAAFsFAADfAwAA4AMAAOIDAADiAwAA4QMAAN8DAABQBQAATgUAADMFAAAzBQAANQUAAFAFAADZAwAA2wMAABAFAAAQBQAAGAUAANkDAADPAwAA0wMAANUDAADUAwAAygMAAGMEAABjBAAAzAMAANQDAADTAwAA0QMAANADAAA0BQAAywMAANADAADQAwAANgUAADQFAAAwBQAALgUAAMwDAADMAwAAYwQAADAFAAC8AwAAyQMAAMcDAADHAwAAuwMAALwDAAC+AwAAvQMAALsDAAC7AwAAvwMAAL4DAACvAwAAswMAALADAACwAwAArgMAAK8DAACnAwAAqQMAAKgDAACoAwAApgMAAKcDAABgAwAAoQMAAKADAACgAwAAXwMAAGADAACXAwAAmQMAAJgDAACYAwAAlgMAAJcDAACNAwAAkQMAAJADAACQAwAAjAMAAI0DAABqAwAAiQMAAIgDAACIAwAAaQMAAGoDAAB4AwAAdwMAAHQDAAB0AwAAfAMAAHgDAABrAwAAZAMAAGUDAABlAwAAbAMAAGsDAABeAwAAYQMAAF8DAABfAwAAXQMAAF4DAAC7AwAAvQMAALwDAAB/AwAAgwMAAIIDAAB/AwAAggMAAIEDAAB/AwAAgQMAAIADAABoAwAAagMAAGkDAABVAwAAUwMAAFQDAABUAwAAVgMAAFUDAABLAwAASQMAAEoDAABKAwAATAMAAEsDAAA1AwAANwMAADYDAAA2AwAAegIAADUDAABtAwAAbwMAADQDAAA0AwAAMwMAAG0DAABnAwAAYwMAADADAAAwAwAALwMAAGcDAABxAwAAiwMAACwDAAAsAwAAKwMAAHEDAAA7AgAAHAMAADICAAAyAgAAPAIAADsCAAAkAwAAIgMAADMCAAAzAgAAGwMAACQDAAAfAwAAXAMAAF0DAABdAwAANwIAAB8DAAA8AwAAOQMAAHACAABwAgAAFQMAADwDAAAeAwAAGQMAAD4CAAA+AgAAMwIAAB4DAAASAwAAEwMAAHUCAAB1AgAAdgIAABIDAAALAwAAWAMAAFYDAABWAwAAfwIAAAsDAABEAwAAQgMAAHYCAAB2AgAAEQMAAEQDAAAiAwAAJAMAABwDAAAcAwAANAIAACIDAAAOAwAACQMAAH4CAAB+AgAAeQIAAA4DAAAMAwAADQMAAI8CAACPAgAAhgIAAAwDAAAEAwAACAMAAH4CAAB+AgAAhQIAAAQDAAD9AgAA+wIAAMQCAADEAgAAiQIAAP0CAAD8AgAAAAMAAIkCAACJAgAAxAIAAPwCAADyAgAA6wIAALMCAACzAgAAtQIAAPICAADSAgAA2QIAAK0CAACtAgAArAIAANICAADqAgAA8QIAALUCAAC1AgAAswIAAOoCAADwAgAA6QIAAJUCAADwAgAAlQIAAMwCAADwAgAAzAIAAJICAADYAgAA1gIAAKcCAACnAgAApAIAANgCAADVAgAA1wIAAKQCAACkAgAApwIAANUCAADjAgAA6AIAAJUCAACVAgAAmwIAAOMCAAD1AgAA9wIAAI0CAACNAgAAyQIAAPUCAADsAgAA5wIAAJgCAACYAgAAlwIAAOwCAADbAgAA1AIAAKECAAChAgAAoAIAANsCAADKAgAA7QIAAOgCAADoAgAAlgIAAMoCAAAAAwAA/AIAAMUCAADFAgAAigIAAAADAADjAgAA4QIAAL4CAAC+AgAAmgIAAOMCAAC8AgAADwIAAAkCAAA6AwAAugIAABYDAADTAgAAEwIAAKwCAAC2AwAAuAMAAJsDAACbAwAAmQMAALYDAAAMAwAACgMAAHwCAACUAgAA6gIAAOUCAADlAgAAnAIAAJQCAADrAgAAtAIAALICAACyAgAA5gIAAOsCAADZAgAA0gIAAKsCAADZAgAAqwIAAKYCAADZAgAApgIAAKUCAAAVAgAAJwIAAKoCAAAVAgAAqgIAANECAAAVAgAA0QIAAKsCAACjAgAAogIAANcCAADXAgAA1QIAAKMCAADbAgAA4AIAAJ4CAACeAgAAogIAANsCAADhAgAA3AIAAJ4CAACeAgAAvgIAAOECAADuAgAA9AIAAJACAACQAgAAygIAAO4CAAD1AgAAyAIAAIwCAACMAgAA9wIAAPUCAAD3AgAAjAIAAMQCAADEAgAA+wIAAPcCAADDAgAAhAIAAAQDAADDAgAABAMAAP4CAADDAgAA/gIAAIgCAAAGAwAAggIAAIACAACAAgAACwMAAAYDAAAJAwAADgMAAHoCAAB6AgAAfQIAAAkDAAAPAwAAUQMAAFIDAAAPAwAAUgMAAHQCAAAPAwAAdAIAABQDAABGAwAARAMAABEDAAARAwAAdwIAAEYDAAATAwAAEgMAAHECAABxAgAAdAIAABMDAAA7AwAAPQMAAKMDAACjAwAAoQMAADsDAABvAgAAUwIAAEQCAABvAgAARAIAAG0CAABvAgAAbQIAAHICAABvAgAAcgIAAG4CAABpAgAAawIAAGcCAABnAgAAaAIAAGkCAABoAgAAZwIAACMCAAAjAgAAIgIAAGgCAAArAgAAZQIAAGYCAABmAgAALgIAACsCAABdAgAAXgIAAF8CAABfAgAAYgIAAF0CAABcAgAAVgIAAFcCAABXAgAAWAIAAFwCAAAoAwAAJwMAAFoCAABaAgAAWAIAACgDAABVAgAAVwIAAFYCAABJAgAArQMAAK4DAACuAwAASgIAAEkCAAC4AgAAUgIAAFECAABRAgAAuQIAALgCAAAyAwAAMwMAADQDAAAyAwAANAMAACsDAAAyAwAAKwMAACwDAABPAgAAUAIAACYDAAAmAwAATgIAAE8CAABLAgAAbgMAAGwDAABsAwAASgIAAEsCAABHAgAAJQMAAEICAABCAgAAQwIAAEcCAAAQAgAAQwIAAEICAABCAgAAEQIAABACAABcAwAAHwMAADgCAAA4AgAArAMAAFwDAACiAwAAPQIAABgDAAAYAwAAoAMAAKIDAAAcAwAAOwIAADQCAAA2AgAAOAIAAB8DAAAfAwAAIAMAADYCAAAuAgAALwIAACwCAAAsAgAAKwIAAC4CAAAoAgAAKQIAACACAAAgAgAAFgIAACgCAAAmAgAAIQIAACQCAAAmAgAAJAIAACUCAAAmAgAAJQIAAM0CAAB+AwAAHQIAABsCAAAbAgAAhQMAAH4DAADDAwAAHAIAAB4CAAAeAgAAxgMAAMMDAAAdAgAAHgIAABwCAAAcAgAAGwIAAB0CAADIAwAAugMAALcDAADIAwAAtwMAABcCAADIAwAAFwIAAMcDAAB8AwAAdAMAABUCAAAVAgAAEwIAAHwDAAAPAgAADQIAAAkCAAAGAgAAuwIAAAQCAAAEAgAADgIAAAYCAAAKAgAACwIAAA0CAACaAwAACgIAAAUCAAAFAgAAmAMAAJoDAAC7AgAABgIAAJIDAACSAwAAlAMAALsCAAAGAgAABwIAAJADAACQAwAAkgMAAAYCAAB5AwAACwIAAAoCAAAKAgAAmgMAAHkDAAAFAgAACgIAAA0CAAAHAgAABgIAAA4CAAAOAgAADwIAAAgCAAAIAgAABwIAAA4CAACDAwAAfQMAAJEDAACRAwAAjQMAAIMDAAC7AwAAxwMAABcCAAAXAgAAGAIAALsDAAC7AwAAGAIAAL8DAAC1AwAAGQIAABcCAAAXAgAAtwMAALUDAACOAwAAwAIAAIIDAACOAwAAggMAAIMDAACOAwAAgwMAAI0DAAAnAgAAvAMAAL0DAAAnAgAAvQMAACECAAAnAgAAIQIAACYCAAAnAgAAJgIAAM4CAAAfAgAAIAIAACkCAAApAgAAKgIAAB8CAAAvAgAAMAIAAC0CAAAtAgAALAIAAC8CAAA2AgAAOQIAADgCAAAYAwAAPQIAADUCAAA1AgAAOgIAABgDAACkAwAAGQMAAD0CAAA9AgAAogMAAKQDAACvAwAArAMAADgCAAA4AgAAOQIAAK8DAAAUAgAARAIAAEMCAABDAgAAEAIAABQCAABIAgAAYgMAAGYDAABmAwAARwIAAEgCAABMAgAAcAMAAG4DAABuAwAASwIAAEwCAABFAgAARgIAAFACAABQAgAATwIAAEUCAAAxAwAALQMAAC4DAAAxAwAALgMAAC8DAAAxAwAALwMAADADAABMAgAAUQIAAFICAABSAgAATQIAAEwCAABIAgAARwIAAEMCAABIAgAAQwIAAEQCAABIAgAARAIAAFMCAAAqAwAAKQMAAFcCAABXAgAAVQIAACoDAABbAgAAWgIAACcDAABbAgAAJwMAAFkCAABbAgAAWQIAALcCAABaAgAAXQIAAFwCAABcAgAAWAIAAFoCAABiAgAAYwIAAFwCAABcAgAAXQIAAGICAABUAgAAVgIAAGUCAABlAgAAKwIAAFQCAAAqAgAAaAIAACICAAAiAgAAHwIAACoCAABqAgAAbAIAAGsCAABrAgAAaQIAAGoCAABhAwAAXgMAAFMCAABTAgAAbwIAAGEDAAA+AwAAPAMAABUDAAAVAwAAcQIAAD4DAABAAwAAPgMAAHECAABxAgAAEgMAAEADAAAQAwAAeAIAAHcCAAB3AgAAEQMAABADAAAOAwAAFwMAADUDAAA1AwAAegIAAA4DAABZAwAASAMAAEkDAABJAwAAVwMAAFkDAAADAwAABwMAAIECAACBAgAAhAIAAAMDAAABAwAAAwMAAIQCAACEAgAAwwIAAAEDAAD4AgAA/gIAAIcCAACHAgAAjgIAAPgCAADwAgAA+AIAAI4CAACOAgAAkwIAAPACAADnAgAA7AIAAJYCAACWAgAAmgIAAOcCAADiAgAAvgIAAJ4CAACeAgAA4AIAAOICAACiAgAAngIAANwCAADcAgAA1wIAAKICAADPAgAA1AIAAKMCAACjAgAAqQIAAM8CAAAmAgAAqQIAANACAADQAgAAzgIAACYCAADeAgAA2QIAAKUCAAClAgAAnwIAAN4CAADyAgAAtgIAALQCAAC0AgAA6wIAAPICAAAMAwAAfAIAAPMCAAAMAwAA8wIAALYCAAAMAwAAtgIAAA0DAACuAgAAsAIAALICAACuAgAAsgIAALQCAACuAgAAtAIAALYCAACuAgAAtgIAAPMCAACuAgAA8wIAAHwCAACuAgAAfAIAAHsCAACuAgAAewIAAG0CAAC5AwAAdwMAAHgDAAC5AwAAeAMAAJsDAAC5AwAAmwMAALgDAAAUAgAAewMAAHwDAAB8AwAAEwIAABQCAAARAgAACQIAAAwCAAAMAgAAEAIAABECAACVAwAAhwMAALQDAAC0AwAAlwMAAJUDAAC+AgAA4gIAAOcCAADnAgAAmgIAAL4CAADHAgAA9gIAAPQCAAD0AgAAkQIAAMcCAADBAgAAAgMAAAADAAAAAwAAigIAAMECAADUAgAAzwIAAKgCAACoAgAAoQIAANQCAADnAgAA4gIAAL0CAAC9AgAAmAIAAOcCAAD3AgAA+wIAAMYCAADGAgAAjQIAAPcCAADoAgAA7QIAAMwCAADMAgAAlQIAAOgCAADXAgAA3AIAAJ0CAACdAgAApAIAANcCAADWAgAA0QIAAKoCAACqAgAApwIAANYCAADpAgAA5AIAAJsCAACbAgAAlQIAAOkCAADxAgAA+QIAAI8CAACPAgAAtQIAAPECAADZAgAA3gIAAK8CAACvAgAArQIAANkCAADrAgAA5gIAALECAACxAgAAswIAAOsCAAAAAwAAAgMAAMICAADCAgAAiQIAAAADAAD0AgAA9gIAAMgCAADIAgAAkAIAAPQCAAAHAwAAAwMAAIMCAACDAgAAfwIAAAcDAAANAwAA8gIAALUCAAC1AgAAjwIAAA0DAAD/AgAABQMAAIYCAAA5AwAAOgMAABYDAAAWAwAAcAIAADkDAABBAwAAQwMAAKkDAACpAwAApwMAAEEDAABXAwAASQMAAEsDAABLAwAAVQMAAFcDAAATAwAATgMAAEwDAABMAwAAdQIAABMDAAAZAwAApAMAAKYDAACmAwAAPgIAABkDAAA4AwAAOwMAAKEDAAChAwAAYAMAADgDAAAIAwAADwMAAHkCAAB5AgAAfgIAAAgDAAAiAwAAIQMAAB4DAAAeAwAAMwIAACIDAAAdAwAAsgMAALMDAAAdAwAAswMAADkCAAAdAwAAOQIAADYCAAByAwAAJgMAAIoDAABqAwAAZwMAAC8DAAAvAwAALgMAAGoDAABrAwAAbQMAADMDAAAzAwAAMgMAAGsDAAA2AwAACgMAAH0CAAB9AgAAegIAADYDAABJAwAASAMAAEcDAABHAwAASgMAAEkDAABQAwAAVAMAAFMDAABTAwAATwMAAFADAABfAwAAYQMAAGADAAB4AwAAfAMAAHsDAAB4AwAAewMAAHoDAAB4AwAAegMAAHkDAAC4AwAAugMAALkDAABdAwAAXAMAAFsDAABbAwAAXgMAAF0DAABnAwAAagMAAGgDAABoAwAAZgMAAGcDAABxAwAAbwMAAHADAABwAwAAcwMAAHEDAACFAwAAhgMAAIcDAACHAwAAhAMAAIUDAACOAwAAjQMAAIwDAACMAwAAjwMAAI4DAACWAwAAlAMAAJUDAACVAwAAlwMAAJYDAACcAwAAjgMAAI8DAACPAwAAnwMAAJwDAACmAwAApAMAAKUDAAClAwAApwMAAKYDAACtAwAAWwMAAFwDAABcAwAArAMAAK0DAAC4AwAAtgMAALcDAAC3AwAAugMAALgDAADEAwAAwgMAAMMDAADDAwAAxgMAAMQDAABIAgAAWwMAAK0DAACtAwAASQIAAEgCAAAuBQAALAUAAM0DAADNAwAAzAMAAC4FAAA2BQAA0AMAANEDAADRAwAAFQUAADYFAADTAwAA0AMAAMsDAADUAwAAzAMAAM0DAADNAwAAzgMAANUDAADVAwAA1AMAAM0DAAAgBQAALwUAADEFAAAxBQAAIwUAACAFAADdAwAA3gMAAOADAADgAwAA3wMAAN0DAABZBQAAWwUAAOADAADgAwAA3gMAAFkFAAAiBQAATwUAAN8DAAAiBQAA3wMAAOEDAAAiBQAA4QMAACEFAAAbBQAA5AMAAGAFAAAbBQAAYAUAAF8FAAAbBQAAXwUAABwFAAAQBQAA2wMAAOoDAADqAwAAEQUAABAFAADrAwAA7AMAAO8DAADvAwAA7gMAAOsDAAC/BAAAMgIAACMDAAAjAwAAxgQAAL8EAAD2AwAA+wMAAPgDAAD9AwAA9gMAAMEEAADBBAAAvAQAAP0DAABEBQAAvQQAAD8CAAA/AgAAqgMAAEQFAADYAwAA3AMAAAAEAAAABAAA/wMAANgDAAD/AwAABAQAAAUEAAAFBAAAAQQAAP8DAAAHBAAA/gQAAAEFAAABBQAACAQAAAcEAAALBAAADAUAAA8FAAAPBQAADAQAAAsEAAAFBAAADwQAACQFAAAkBQAABQUAAAUEAADOBAAA1AQAANMEAADTBAAAzwQAAM4EAAALBAAAEAQAAPwDAAD8AwAACgQAAAsEAAAABAAA3AMAAOsDAADrAwAAEwQAAAAEAAAABAAAFAQAAMwEAADMBAAA/wMAAAAEAAC3AgAAuAIAABEEAAARBAAAGAQAALcCAAAZBAAAGwQAAF4CAABeAgAAWwIAABkEAAAdBAAAHgQAAB8EAAAfBAAAHAQAAB0EAADrAwAA7gMAABMEAADtAwAAIgQAACMEAAAjBAAA8AMAAO0DAAAcBAAAHwQAACEEAAAcBAAAIQQAAPEDAAAcBAAA8QMAAPIDAAAcBAAA8gMAAPMDAAAcBAAA8wMAACQEAAAcBAAAJAQAAGwCAAAcBAAAbAIAAGACAAASBAAABwQAAPcEAAD3BAAA+gQAABIEAADXBAAAYgQAALkEAAC5BAAAugQAANcEAAA/BQAAQQUAAOEEAADhBAAA3wQAAD8FAADoBAAAswQAAHgCAAB4AgAARwMAAOgEAAAlBAAAKgQAADEEAADtBAAA6wQAAPEEAACrBAAANgQAADgEAAA4BAAApwQAAKsEAAClBAAAqQQAAIICAACCAgAAwQIAAKUEAABBBAAAOgQAAKIEAACiBAAAnAQAAEEEAABFBAAARAQAAJMEAACTBAAAjAQAAEUEAACLBAAARwQAAEgEAABIBAAAhgQAAIsEAACHBAAAgAQAAE0EAABNBAAASgQAAIcEAACABAAAewQAAFEEAABRBAAATQQAAIAEAAB4BAAATwQAAFQEAABUBAAAcwQAAHgEAADpAwAAVAQAAHIEAAByBAAAcAQAAOkDAABZBAAAWwQAAIIEAACCBAAAfQQAAFkEAABEBAAARQQAAI0EAACNBAAAlAQAAEQEAACUBAAAnAQAAEAEAABABAAARAQAAJQEAABWBQAAYwUAABIFAABWBQAAEgUAABEFAABWBQAAEQUAAOoDAACAAgAArgQAAPYEAAD2BAAAWgMAAIACAAA0BAAAOQQAAKgEAACoBAAArAQAADQEAAAmBAAAYgQAANcEAADXBAAAKgQAACYEAADKAwAAMgUAADAFAAAwBQAAYwQAAMoDAABCBAAAkgQAAJgEAACYBAAAbAQAAEIEAAA8BAAAaQQAAKQEAACkBAAAoAQAADwEAACoAgAAJQIAAHAEAABwBAAAcgQAAKgCAAC9AgAAmQIAAIMEAACDBAAAhQQAAL0CAACRAgAAywIAAJEEAACRBAAAlwQAAJECAABvBAAAQwQAAJIEAACSBAAAkAQAAG8EAABLBAAAZgQAAIQEAACEBAAAfwQAAEsEAADqAwAAVQQAAHMEAABzBAAAcQQAAOoDAABJBAAAZgQAAEsEAABJBAAASwQAAIAEAABJBAAAgAQAAIcEAAA8BAAAawQAAD8EAAA8BAAAPwQAAJsEAAA8BAAAmwQAAKEEAABaBAAAXAQAAIgEAACIBAAAgQQAAFoEAABcBAAAWgQAAIIEAACCBAAAiQQAAFwEAABoBAAANwQAAKkEAACpBAAApQQAAGgEAABsBAAAPgQAAJ0EAACdBAAAmQQAAGwEAAA3BAAAaAQAAKQEAACkBAAApgQAADcEAAA7BAAAQAQAAJwEAACcBAAAogQAADsEAAA6BAAAMwQAAKwEAACsBAAAqAQAADoEAAAoBAAAKwQAALoEAAC6BAAAuQQAACgEAABCBQAA/gMAAL0EAAC9BAAARAUAAEIFAADqBAAALQQAALMEAACzBAAA6AQAAOoEAADpBAAA8wQAAPEEAADxBAAA6wQAAOkEAABBBQAAQwUAAOMEAADjBAAA4QQAAEEFAAD7BAAA+wMAALsEAAC7BAAAPAUAAPsEAAAvBAAAKwQAALcEAAC3BAAAsgQAAC8EAADBBAAA9gMAAMMEAADDBAAAxAQAAMEEAAD6AwAASQUAAE0FAAAEBQAAAgUAAAYEAAAGBAAAxwQAAAQFAADPBAAA0AQAAAYFAAAGBQAAJQUAAM8EAADTBAAA1AQAAAcFAAAHBQAAAAUAANMEAADZBAAA2AQAADEEAAAxBAAAKgQAANkEAADuBAAA7wQAAPAEAADwBAAA7QQAAO4EAADwBAAA7AQAAOsEAADrBAAA7QQAAPAEAAD1BAAAWQMAAFoDAABaAwAA9gQAAPUEAAARBQAAEgUAABMFAAATBQAAEAUAABEFAABLBQAATAUAAE0FAABNBQAASgUAAEsFAABiBQAAYwUAAGEFAAD+BAAAAgUAAAMFAAADBQAA/wQAAP4EAAAMBQAACgUAAAkFAAAJBQAACwUAAAwFAAAZBQAAIAUAACEFAAAhBQAAGgUAABkFAAAmBQAADgUAAA0FAAANBQAAJwUAACYFAAAuBQAAMAUAADEFAAAxBQAALwUAAC4FAAA2BQAAFQUAABQFAAAUBQAANwUAADYFAAA+BQAAQAUAAEEFAABBBQAAPwUAAD4FAABJBQAARgUAAEcFAABHBQAASAUAAEkFAABOBQAAUAUAAFEFAABRBQAATwUAAE4FAABdBQAAWwUAAFoFAABaBQAAXAUAAF0FAABZBAAAfQQAAHYEAADeBAAA2wQAANoEAADaBAAA3QQAAN4EAAA4AwAAugIAADoDAAA6AwAAOQMAADgDAABBAwAAPwMAAEADAABAAwAAQgMAAEEDAAC9UR1AzNG1v6GCwz5YkBxARYG0v2owvT74xRxAOs61v5SjsD5DBB5AZye3v7UXqT6vzx5AfVu4v1isqT44MB9AF7q4v9cTrT6C4h5Az2e4v8uduT6x4RtAx4Gxv3edpT71gBtA7diwv7Qevj7aWB5A5/uzv1SpmT4cBiFA4Qi4v/GhnD6MvSFA6+S4vwk3qT4ydSFAFM+3vxhDwT7tER1AHCezvx8r0D4QkxtAumWlvyEjwD7DRx1AOXujv7a6lD4c8CBA7Pinv0KVgj4wECVAC7OsvwTKjj7+0h5AgJ21v72M0j4EPCRA1O+sv+VEwz57SyFA7Uqpv/IM4j5j1B5A3gGov+s22D7JriVAmE+uvz9voj70qSlAl8abvx6MoD6PxyhAje+Zv37jwz45DCVAmzeWv3Ui4T5z2SFAuI+Uvy4d2z5JSB5AcvyUv9yeyD7uJx9Ajs6Xv3akkj6ZECRAObmZv/JEgD76RChAN6mav0Pijj5ubixAwLGLv/McmT7PESxALLaJv4ihvT6LFyhAOj2Fv5Hz3j6+oSpAMiCNv61shz7/riZAFjGMv981gD4HtCFASZyLvwNAnT7q0B9A4QyIv3zuxD5kryNAUmOGv1cF2j7hYBtA2uW1vwSORD+zRBpANgK1vwnhPT9crhpAUyK1v4zWNT+qmRxA12e2vwlSMT82Wh5AQSy3v7g/Mz9p5R5AH2a3v/8lOT9Cdh5AOUO3vy8wPz9D4RxAcLG2v2HCRD+D3xZArYSsv8EaOz9hNBlAeCiwv5KXLT+thRxA4L2zvz/+Jj+OBCBAw7u0v2UcKz/LLiFAIsG0v5upND82lR9ARPeyv0seQz/uBR1Ab9Oxv5BmTD8YCBpAGVavvwPtSj+U9iBAMH+lv7TKHD8+dSNA/+mkvxjtKT8bLSFAqOGhvwovPT/vchlA7Iqdv0yMRT+bjxZAPBSfv3y0MD+ilxlAuvShv7jIHT++ox1AOlylv/iLFT+Bzx1AmIeev66DSz9nYRdApWmSv4cXKD/oSxpAIY+Uv/3dEz9klR5A4ueVvwzICj9AwhlAIv+Ov629Oz92wR5ApKaPv2+BQD+ZRiJA7UeSvzCfND+vzCRAgPOUvz/HHz+CHSJAsyWXv8dLDz+1GCZAfLSCv0WgEj8k0SNAout+v2ZrKT+W5x9ATrV6v0+vND+fyyJA+OOEvzP+AT9LHx9AnmCJvwFOAz8MARtAev6Iv70eED9CXRlA6daFv1GhHj/5LhpAnbmBv5z8Mj8HJRdAIsGSvwItjT87NhVAJQeSv2FUjj9+GhRA9IaTvyzwjT95kRNAm1WVv7ZKjD+mSBRASYKYv6nCiT9t4xVALqqZv4zahz9aERdA7ziZv/iKiD9QUhhArd6Vv05iij9mahhAl1WMvwq8iT/eIRpAbSCTv/34gz88vhdAjBGZvzkPfz+5iBRAS+aav2t/fz8wSxBAK4mUvyFZiD/t9RFA/iuZv4SBgz9v8xFASPyOv7LXiz/LvhRA8DSLv0ORjD/SqwxApPyMv/yOdT/RAw9AZaqSv5Ciaj95rhJAlgSUv0BoYT+DhRdAfouQvxFXYj/rHQ9Ati+GvyargD9S0xNAYMeBv3BEgT+1NRdAbAeDv6wBfj82AhpA02uJv8XIbj8Q6hhAcmp/vwkWWz81KBZAD0Vxv9KpZz94YBJArHNsv89IbD+A8g1AXflwv+ZbZz/saQtAQQuBv8HEVz+jeBdAdcuIv4fgTD+yRw5AeZSGv/BuTT/BUxJAjleKv7KBSD98RRlANGZqv9QrST+9+xVA/FZXv07QUj9D5xBArvVRv5S+VD+ESQ1Akndav4NQTj+wyRdAogp/v4lfPT/rVxJAZEB+v2WONT9UVA5Askd0v24ZOD/z5AtAEhJpvyXsPz9GJBtABMdRvy8ZNz/dshVABtpBv/NVQj9qoQ9ANUVEvxUaRD9fYRtA2sduv9cXOT9kliBAyHlfv3FWLD8FxCRAEodkvwezHT8jECdA48Vqv9EIAj8McyZAG0l+v4On8D5uiClAqFFwv5Ur3D5cAhlA8MJ+v0fIND9S0SFAR1eDv5G54j4YJi1AzQV2v1Sorj54li1A2zF9v+Pgij5h/CpAHoqCv0DdcD52iiVAZYmEv+ROaT56xh9AmiKGv4Qtjj6QoB1ALNODvx08uz7+uRxAcaqDv00v2T63fhhANA6Dv8BCAj+kVRVA0VmAv3v6FD9iERZA/ieBv8GpJz/eIBFATaJyvzaPIz8dBA5AAtNpv++qKz9JogtAhCtcv0gXMz+0Ag1A6IdNv2N9Pz+zKCNAUDdMvzYGCT9m3R9Ap1xFv4V4GD/7dSVAUTJRvwwB8D6M2SdAK4ZTv01nxz4LRypAoS1XvxdmkT4ukipAa7piv0LsXD5e2xNA2bUxv2QiMT/xSA5AMlc2v2XjNT/q6RpANNs5v98zJj+cwQtAFXQ/v1dgND+IvQpA1IJPvzv/Kj8ipwxAN2xbv9EDHz9J1Q9AFmtgvzjaET+KkhNA4X5kv2PvBT8qARdAHJZqv7KB9D7jGxtAY5xzv3+FzD4eNRxAU+l3vzBLqz5uFB9Ar5d6v8R3gj4iiCNAlKN4v3uHSz7TLihA+rdvv2FxOD6kbCJAKuYkv4Rjxj7onx5APuwdvy0h7z74FiVA14Uvv7nHmj5ssyZAuM47vyWtWD4XvCZAWhJIv2sRET4nwRlAtycYv5MYDD+FCBJANSgav0YHID8VGwxAqoIlv7vxJj+FPxBALZUHv8JoDj/Y1QlAOPMTvwsNHD+HbAtAUwP1vt1d3z41XgVATQ8Gv8DLAD/g1QBA4NYRv9l7DT8tPwVAdw8gvzM3Hz80aQZAGxMyv0z7Ij+OeQlAzXU2v1AXKT8PDBdARfQDv5zb7D5ybRtAzqUIv3FZvT7DKB5AAg4Nv2djnT67uSBA2UAWvytRZj49uCFAUn8lv0RNFD5ioCFAVcA1v4f7yD0woBpAVWouvzs3jT049BtAJ/oYv1n72z27fxpAZhEOv2XeGj58mRdA3QYFv2gHXD4SahRAeNP9viPYiD6PphBALSb2vv9eqj56OSNA091ZvzG32z1eSx9Ad4JFv7mIjz3ZJxtA4GlWvw7ZgD2ZnhZAC9Rev9AK7D2P+hRAUaBDv+tXmj34pR5Aympmv22N6D3B5hlAweRuv1XaMj65FhZA5C0jv0jhuj2EKhhAAkobvwExyT15khFAO8dAvy6pKj6ZRRNAgh9Zv0j7Xz46JBZAvDtqv10ziT4v3BVAtANav6naxj5maBJAnvBKv1ezpj6P4hBAI9c1vxtHfD4aNBRAQ1Qdv21VAj5HrhNAYvhcv+Pg2j4LYgtACk06v0HYiT5jDQpAmiQav/OPXj4JNhFA4CobvxlxMT4P7wZA5BQ9v8lZmD6oNQVAAMQZv8cvjD5SXxBA9tJcv0rs8j6LFg1Ak+Nav+BjBD+5+whAWWpVv4o64z6dEAxAnupUvz0MxT5TXQNAJcw8vybHvT6JDABAQSsgv1VntT6mDwBAldNCvyjV7j68lfc/hzYsv07u5z7bNfU/vHdQv7KgED/P9QNAem1Ov6q6Bz/LgPs/G6FXv2ivGj9vEu8/I004vzc4DT9qUApA0H1Rv/PmFD9ftAhA71dFvxSwIT8p6wVAdatDv2dEHT9KCABA66xOv8u6Kz80uQ9Av5tOvxFytj5X6uk/wVNYv4VdKD/NseQ/K4lEv3eEIz8dAO8/JbBdvyI5MT+7nfE/JHtYv0d2QT90Dec/aQJVv5jBSD+DUuQ/7iFZvxUDRD8baPg/Wz8Zv0DC+D6jygBA008Mv8k5yT4A4vA/rOEqvyHpFz+U9+c/6zk1vwhXMD8tlN4/p8lMv2FRPT8iOOA/tAFEvwX7Qz/pDAZA7YIFvyl6mD5DVwxAObQEv8cQcD5BnPU/N1Ejv+VFJj9F8/w/EAQsv7D+Mz/3AwBAa/I8v+1iNj/ymew/E0cuv6sJPj9GQeQ/7bg9vzBJTT9Sf+c/FLJDvw+aUT+/YPM/6bY0v2KCRj+FzPM/0CZHvw0zTD+YbOg/N1JKv0imTz/vruE/bW9XvxyZPz+kOBZARE4Pv0+UBD45thNAknQJvzf6KD7kLRBAixsHv0z6Wz69UR3AzNG1v6GCwz5YkBzARYG0v2owvT74xRzAOs61v5SjsD5DBB7AZye3v7UXqT6vzx7AfVu4v1isqT44MB/AF7q4v9cTrT6C4h7Az2e4v8uduT6x4RvAx4Gxv3edpT71gBvA7diwv7Qevj7aWB7A5/uzv1SpmT4cBiHA4Qi4v/GhnD6MvSHA6+S4vwk3qT4ydSHAFM+3vxhDwT7tER3AHCezvx8r0D4QkxvAumWlvyEjwD7DRx3AOXujv7a6lD4c8CDA7Pinv0KVgj4wECXAC7OsvwTKjj7+0h7AgJ21v72M0j4EPCTA1O+sv+VEwz57SyHA7Uqpv/IM4j5j1B7A3gGov+s22D7JriXAmE+uvz9voj70qSnAl8abvx6MoD6PxyjAje+Zv37jwz45DCXAmzeWv3Ui4T5z2SHAuI+Uvy4d2z5JSB7AcvyUv9yeyD7uJx/Ajs6Xv3akkj6ZECTAObmZv/JEgD76RCjAN6mav0Pijj5ubizAwLGLv/McmT7PESzALLaJv4ihvT6LFyjAOj2Fv5Hz3j6+oSrAMiCNv61shz7/ribAFjGMv981gD4HtCHASZyLvwNAnT7q0B/A4QyIv3zuxD5kryPAUmOGv1cF2j7hYBvA2uW1vwSORD+zRBrANgK1vwnhPT9crhrAUyK1v4zWNT+qmRzA12e2vwlSMT82Wh7AQSy3v7g/Mz9p5R7AH2a3v/8lOT9Cdh7AOUO3vy8wPz9D4RzAcLG2v2HCRD+D3xbArYSsv8EaOz9hNBnAeCiwv5KXLT+thRzA4L2zvz/+Jj+OBCDAw7u0v2UcKz/LLiHAIsG0v5upND82lR/ARPeyv0seQz/uBR3Ab9Oxv5BmTD8YCBrAGVavvwPtSj+U9iDAMH+lv7TKHD8+dSPA/+mkvxjtKT8bLSHAqOGhvwovPT/vchnA7Iqdv0yMRT+bjxbAPBSfv3y0MD+ilxnAuvShv7jIHT++ox3AOlylv/iLFT+Bzx3AmIeev66DSz9nYRfApWmSv4cXKD/oSxrAIY+Uv/3dEz9klR7A4ueVvwzICj9AwhnAIv+Ov629Oz92wR7ApKaPv2+BQD+ZRiLA7UeSvzCfND+vzCTAgPOUvz/HHz+CHSLAsyWXv8dLDz+1GCbAfLSCv0WgEj8k0SPAout+v2ZrKT+W5x/ATrV6v0+vND+fyyLA+OOEvzP+AT9LHx/AnmCJvwFOAz8MARvAev6Iv70eED9CXRnA6daFv1GhHj/5LhrAnbmBv5z8Mj8HJRfAIsGSvwItjT87NhXAJQeSv2FUjj9+GhTA9IaTvyzwjT95kRPAm1WVv7ZKjD+mSBTASYKYv6nCiT9t4xXALqqZv4zahz9aERfA7ziZv/iKiD9QUhjArd6Vv05iij9mahjAl1WMvwq8iT/eIRrAbSCTv/34gz88vhfAjBGZvzkPfz+5iBTAS+aav2t/fz8wSxDAK4mUvyFZiD/t9RHA/iuZv4SBgz9v8xHASPyOv7LXiz/LvhTA8DSLv0ORjD/SqwzApPyMv/yOdT/RAw/AZaqSv5Ciaj95rhLAlgSUv0BoYT+DhRfAfouQvxFXYj/rHQ/Ati+GvyargD9S0xPAYMeBv3BEgT+1NRfAbAeDv6wBfj82AhrA02uJv8XIbj8Q6hjAcmp/vwkWWz81KBbAD0Vxv9KpZz94YBLArHNsv89IbD+A8g3AXflwv+ZbZz/saQvAQQuBv8HEVz+jeBfAdcuIv4fgTD+yRw7AeZSGv/BuTT/BUxLAjleKv7KBSD98RRnANGZqv9QrST+9+xXA/FZXv07QUj9D5xDArvVRv5S+VD+ESQ3Akndav4NQTj+wyRfAogp/v4lfPT/rVxLAZEB+v2WONT9UVA7Askd0v24ZOD/z5AvAEhJpvyXsPz9GJBvABMdRvy8ZNz/dshXABtpBv/NVQj9qoQ/ANUVEvxUaRD9fYRvA2sduv9cXOT9kliDAyHlfv3FWLD8FxCTAEodkvwezHT8jECfA48Vqv9EIAj8McybAG0l+v4On8D5uiCnAqFFwv5Ur3D5cAhnA8MJ+v0fIND9S0SHAR1eDv5G54j4YJi3AzQV2v1Sorj54li3A2zF9v+Pgij5h/CrAHoqCv0DdcD52iiXAZYmEv+ROaT56xh/AmiKGv4Qtjj6QoB3ALNODvx08uz7+uRzAcaqDv00v2T63fhjANA6Dv8BCAj+kVRXA0VmAv3v6FD9iERbA/ieBv8GpJz/eIBHATaJyvzaPIz8dBA7AAtNpv++qKz9JogvAhCtcv0gXMz+0Ag3A6IdNv2N9Pz+zKCPAUDdMvzYGCT9m3R/Ap1xFv4V4GD/7dSXAUTJRvwwB8D6M2SfAK4ZTv01nxz4LRyrAoS1XvxdmkT4ukirAa7piv0LsXD5e2xPA2bUxv2QiMT/xSA7AMlc2v2XjNT/q6RrANNs5v98zJj+cwQvAFXQ/v1dgND+IvQrA1IJPvzv/Kj8ipwzAN2xbv9EDHz9J1Q/AFmtgvzjaET+KkhPA4X5kv2PvBT8qARfAHJZqv7KB9D7jGxvAY5xzv3+FzD4eNRzAU+l3vzBLqz5uFB/Ar5d6v8R3gj4iiCPAlKN4v3uHSz7TLijA+rdvv2FxOD6kbCLAKuYkv4Rjxj7onx7APuwdvy0h7z74FiXA14Uvv7nHmj5ssybAuM47vyWtWD4XvCbAWhJIv2sRET4nwRnAtycYv5MYDD+FCBLANSgav0YHID8VGwzAqoIlv7vxJj+FPxDALZUHv8JoDj/Y1QnAOPMTvwsNHD+HbAvAUwP1vt1d3z41XgXATQ8Gv8DLAD/g1QDA4NYRv9l7DT8tPwXAdw8gvzM3Hz80aQbAGxMyv0z7Ij+OeQnAzXU2v1AXKT8PDBfARfQDv5zb7D5ybRvAzqUIv3FZvT7DKB7AAg4Nv2djnT67uSDA2UAWvytRZj49uCHAUn8lv0RNFD5ioCHAVcA1v4f7yD0woBrAVWouvzs3jT049BvAJ/oYv1n72z27fxrAZhEOv2XeGj58mRfA3QYFv2gHXD4SahTAeNP9viPYiD6PphDALSb2vv9eqj56OSPA091ZvzG32z1eSx/Ad4JFv7mIjz3ZJxvA4GlWvw7ZgD2ZnhbAC9Rev9AK7D2P+hTAUaBDv+tXmj34pR7Aympmv22N6D3B5hnAweRuv1XaMj65FhbA5C0jv0jhuj2EKhjAAkobvwExyT15khHAO8dAvy6pKj6ZRRPAgh9Zv0j7Xz46JBbAvDtqv10ziT4v3BXAtANav6naxj5maBLAnvBKv1ezpj6P4hDAI9c1vxtHfD4aNBTAQ1Qdv21VAj5HrhPAYvhcv+Pg2j4LYgvACk06v0HYiT5jDQrAmiQav/OPXj4JNhHA4CobvxlxMT4P7wbA5BQ9v8lZmD6oNQXAAMQZv8cvjD5SXxDA9tJcv0rs8j6LFg3Ak+Nav+BjBD+5+wjAWWpVv4o64z6dEAzAnupUvz0MxT5TXQPAJcw8vybHvT6JDADAQSsgv1VntT6mDwDAldNCvyjV7j68lfe/hzYsv07u5z7bNfW/vHdQv7KgED/P9QPAem1Ov6q6Bz/LgPu/G6FXv2ivGj9vEu+/I004vzc4DT9qUArA0H1Rv/PmFD9ftAjA71dFvxSwIT8p6wXAdatDv2dEHT9KCADA66xOv8u6Kz80uQ/Av5tOvxFytj5X6um/wVNYv4VdKD/NseS/K4lEv3eEIz8dAO+/JbBdvyI5MT+7nfG/JHtYv0d2QT90Dee/aQJVv5jBSD+DUuS/7iFZvxUDRD8baPi/Wz8Zv0DC+D6jygDA008Mv8k5yT4A4vC/rOEqvyHpFz+U9+e/6zk1vwhXMD8tlN6/p8lMv2FRPT8iOOC/tAFEvwX7Qz/pDAbA7YIFvyl6mD5DVwzAObQEv8cQcD5BnPW/N1Ejv+VFJj9F8/y/EAQsv7D+Mz/3AwDAa/I8v+1iNj/ymey/E0cuv6sJPj9GQeS/7bg9vzBJTT9Sf+e/FLJDvw+aUT+/YPO/6bY0v2KCRj+FzPO/0CZHvw0zTD+YbOi/N1JKv0imTz/vruG/bW9XvxyZPz+kOBbARE4Pv0+UBD45thPAknQJvzf6KD7kLRDAixsHv0z6Wz5d3wO/9+jvP+0q7L6/KgO/mUrpPwLw375zEf++0nH7PwT+4L7OwfO+J0r8PwVvyL6tvuK+NSj6P7fUqb6Upca+8uzuPwFqgr4Ysvq+pl7lP83qxb5cyum+eETjP0Fmp76vfN6+9fXiP4Dzkr6+afK+qYnsP+lHs74TKti+NiPxPxTn0L7+gcq+hq/zP5/nv77pmd6+x2XmP1EuDb50s7++2GL1PyXpGr63DbO+jBIAQGBYHr76fBy/tcbMP7Cqjr5lqBq/mUnWP6gACL4AVBW/I03KP1T/2L6EKbK+PgMJQBmsaL4Medy+KT7mPxiwBL9mML6+rW3oP9iBC7/SAdm+CtrwP9xGB786PLy+CknwP6JDCL/83s6+HGD2P8xG/74z4be+1Cz0PyB+Ar/k88K+gsX5P/KV6L4rabW+8Z/4P5lH3r5g556+t5YIQC/f6r4Z5qy+Lq4IQOoku77oLmm+hxnkP6slEb+RuFe+3rAIQHU6CL8AAAAAp5EIQHzxEb8AAAAAM4zlP90mGL8AAAAA7S3VP1AXOb9KmZS+GF7RPxQ/Kr+bO+q+dVnQP8R7Er8o1JO+uRoSQOi8dr7KpI6+x4IRQAFRuL4ZVHu+QPwQQKpl676sNk++EaoWQGJnar4Fbk2+cw8WQBgip747/TC+uJUVQKKWzr6HwAG+BqAZQCzvar5qawS+CD0ZQMv4l75flta9cAYZQA4urb4AAAAA2VznPzxMLz8AAAAA58TyP5oGHT+nBBS+10vlP1sjIj+0yuy9cGHxP9vBDD9TQFq+zvzgP0/OBD9OKS++7NzqP+8e8D5rgIK+qFfeP2+9vj67C1S+NQjnP25ouj60VRm+F2XqP+m5xT5LdHa+SIzYP/660z7I03K902n3P53a6T4AAAAAZJD7Px78/D5W8Ua+XtjWP3oZ9T4xmP+9BHDXP60UEj8AAAAARu7XPxQEGz8a96a+lPgCQKLrAr3+naW+8N8KQBH9ur3niae+wOz6P96uFz2Ef8G+vW3sP8r5Ij3xSAi/32rbP7Jjoz32soW+4bYEQCAJez1Ktoq+TfkBQMx5xj1Jgpi+fLjuPyrJ2j3dtoe+38DsPzs6Hj5O1U2+t2HuP2OZXj7BOCi+44r1P8nHfj6mCsa9lQ76P6mfjz5n1Hy9aocAQNQPij74io696iUEQBQ/Zj5jRBK+hNkEQLOzSD58nk++w7gFQApl4T0hy3K+P/4BQORNLj5cxg29UKoBQIM1nj6rtAW902YEQDbKgj6toqe+24XcP3NmWz7JAZO+WHUSQJShKr7ZIpG+h4gMQGLWCzxUNny+swsTQGbYaL06QYO+uFoNQOymVD1osUS+zhkOQFfNAz40EmG9xjUIQCpzYz4Mkr69SUwOQKgBMz4AAAAAFF4PQCVbPT4Wpi++/3YTQLUzzDxjmb69nKITQL9Fhz0AAAAA3NQTQJS9xT0AAAAAoggYQOwWAbwAAAAA4lwaQOwv+70n+Ga9FTwaQNBHCb590LO9/tYXQP5gIL3CFRC+d4cXQANDtr0hIcq9VP8ZQPzFLL5ZTCy+MzUXQCRkEL6JC+C9/OIZQEPiPr4AAAAA41AQQKhSB7/irTO+iLsQQFM8Ar+Tw+e9t0MVQL4V6b5MOHS9XOMYQJ1Hvb4AAAAAih4VQK4N9b4AAAAAHeQYQCxKwb55Oyq//dzIP918Iz4/pwC/KlLJP3bjfT5RosW+QBjUP8SyiT59CdW+PRDBP7di3z5rDKq+kBDBP2ZqBj9avAS/6ZjHPzGUcz7sUeC+ujDAP9np5z7Kw7K+PPi/P6/qDD/VsRq+U1rHP4ffJT8o1Q6+lgnJP49uID8AAAAAJArLP1dbKT8AAAAA1GHJP7smMD+QZwe/o5KsP090/T70iie/oPu0P4rKpj6y1TW/A9LCP+nVID7XUEK/j8W8P5lK/z1uNTe/NLqvPzScmj6ZRxa/IZOeP4TwCD/VymS+SWOWP+EoVT8AAAAA5NiWP3pyYT9+4ti+n1mWP38WMz8AAAAA0jmLP4BFaj8Jb4e+Ko6LP2L5Vz/obPG+aeSPP/iLOT+ynR+/9DObP4QuCT/gE0O/w7irP7JikD44EjC/JeaVP1w++j5uFhO/MuaAP4QsHz8wYqe+QtJjPwUZQT8AAAAApkdXP7L2Uz8AAAAACKouP2qmHz9ihLi+FQBDP3hCEz+aXCC/T+ZzP5Q09z590D+/dXKSPzp6vD7ht02/rmSnPzRlVz4eUbm+lugwP2AFsD4AAAAAlBcdPy8wwz5DqiC/VHJqPyWwiT54fUa/9piQP8v0Oz59XEO/l2+TP1RXvjxvoCC/4Q11P1IPUT0tJMi+rkc5P3Cxoj0AAAAAFO0iP1cGtT0AAAAAcy0+P1GDKb4AAAAAwCCFP0BoAb8qxNO+yJpZP1QZFr4DeCe/lNmGP4HsFb4Zr0W/gGWZP+Y8A77QYCu/guKZP7fwvL5cWs2+w9iMP+Vf475gVkC/It+lP4bnnb4AAAAAlbqcP3k+I78AAAAAQKK1P1/uP7++Tsq+jkCePwd8Dr8MI8W+Tn20P+GZKL+Szye/ArmmP9eG8r4+zTm/WDmwP2K7u766LDa/m5K4P49t0b6JtB2/fuO1P9b8EL8AAAAAwojHP5vHRb/D9bC+AwjDP2oTL7/k9Qy/RnvAP5c7E78M5TC/oE69PwGG1b7vWTu/z0y+P4WUj74HKUC//Wm5P8Yxir4DzT+/hczBP5uv8r12bka/3bSzP+uoWr4BGEO/5Ia/P/Vlqb0+Qk2/yO6sP8NF7r2tTUe/DY68Pz9VBb3wpFG/zF6oP6Q0GzwfSUm/OPi6P0kwVTuMo1K/T3anPyhj3D27Y0m/1QO6P68Lfz0AAAAAkpYJQKDebD4AAAAA5ZkFQNArjj4AAAAAkgQDQLlwsD4wg/G+hlfGP9DthT7DnwG/Nq32PzI7674xstS+YHj1P5gxlb4AAAAAAfxjPzvju75Dj9C+rWp5P11Qp75hpCe//0GQPyoeh75Wnri+KSD9P1gdmb4AAAAAkdBuP9XqYz+daJe+OGp1P+dVUT8EOQi/6x+IP6iNMj9J86++XMdQP1KALD8AAAAACFdAPxghPD+EuRm/gjd4P/K1Dz8AAAAALQUkP7u0AT+e6ri+tRg4Px5w7T53gyC/qVBtP0Gdwj6musC+Yf8xP6x0Vz4AAAAAo+kcP5Q0bz5ZpSC/YadsP92UIj6gxBe+0t/TP5qWNL8mcMO+nWjRP/zgIL+QvzS+8pbFPwmIPb8TQ+q+P/3BPxO6I79jRCK/HxG/PxvaAL/LaDi/m+S9PzHSs76Bkz2/pgzAP0pATL4LXUm+GAe1P449N79O1AK/Yyi1P3hGH7/Lgi2/Az63PxpS/b7TTgm/KBCiP8fUBb9/hTS/3jyrP5oF2r6GWj+/uMu0P18JpL6sA0S/h8C9P97JR74vpU6+FvacPzFDG78U6Au/rp+SP0/n0r7arTm/MGafPzdwr75r1kW/MqqsP1lShr6j5Ui/sK25P0GcF77KwFG++waIPxiT9r5b0Qq/aHiFPxXEmL4UBlW+GApsP3C0s76hugm/ZQFzPwiTEr76lzi/r0SQPxfyCL4yAEy/aOuiP2qg+b0WiE6/B7S0P3gplb1eS1i+iH9IP6voH752FgW/4gJUPw/ujj3guzW/sfqFP1RVKD2VC02/kpedP1Q3lzxDyFG/l5CxPxaIHjxhbUy+mBUqPy/7tT2UNAO/UAFMP9k+RD4j3ES++UsjPwNdaz6xUgG/i1JKPzfHoT7eWTe/ntKBPwDKbz4IIk+/A7KbPzLkGD7aV1K/H52wP5cfuD1J2kW/rmS7P3tovz3kSj2+F9UiP/chvz7AJAG/zXRPP1Ex3j7G4Ty+7gkqP79F/z7g9gC/TmJYPzcXCz+8yTO/Bi6FP8wL4D4YQ0m/zJicP0/rlj6neDy+8gk1P0Z+HT9Dx/a+TZ9hP+Y9Ij+uuTO+whhFP/LqOD/ooOu+OjxwP4emND9gciO/9iaKP7qeED+VYzq/ezOgPzIhxj47+yq+bodaP6KZTz9BYdi+BkyAP2vVRj8uqRq+f4ZvP7cIYD+dY8C+/dyMP/PKTT/ylA2/kIOUP4o9JD/pDjK/6+KiP0NX0j4hVwq+JHuKP8GqZj/rHqm+o7CVP/4OST9RFAS/lN6ZP8glIj+A1ii/v/SmPx3o2T6EukC/kj22PwJHYj5osOm9pvGVP6q5YD+wyUa/EjK0P7bWVz6/1E+/HZSwPzPEET5CeRe/gAyxP7tf1T72ReK+I9yqP1RTCj9evJ+9Tu/IP5cDLT8S3ZO9F5vKP2nlJj+9/G6+3o3FP3psEz+2on2+GyrEPxtlGT/sism+Mli/P85vAD/0ir++HlTAP7Ab9j63l+S+9yHDPw2rsD7vV/G+LV7CPz/irz5Up2O+3nbXPwHe4j5Eaiq+u+/WP2yyBj+wVYK9o+rXP1iMFj8fFJS9Q8vmP71UKD+K52y9Le3yPzLkFD8yxxK+09ntP1iQAj+fIje+l3PjP914Ez8Fhmy+Mv/iPxi0uD7QtES+tW3mPz1jAz8K9SS+iSfpP+z4Dz8BNQW+3ULrP5uOHD8AAAAA+pnsP1TiKj/yQ4W9acjsP4C4Iz+nz5a+isj8PxqLpj22KyS+/fcBQBvVaT5QjZe9B0ELQGk6Sz6d8S2+2xgKQPWiJj4Cf3i+YcUKQJM5lj0Oo5i+AYkJQGKh1jzQ1Ou9lx8BQGJOID5GRSy+srgAQOFiFT7i5Wm+PuwAQBGK7T3/s4a+5EkAQNXrlj1PIoq+PiD4Pxwpuz2o5IS+N8P1P/vOzz0jElW+Wfr4PzfCEj7AJiu+9wb7P/T+Hz5hxvS98WP+P6hvKT5b6dW9I8AAQBE0Jj7oavu+0STFPwxYgj5TsC6/zea7P341dz7OqA2/sATCPw6+cD7y6rS+wtnDP6WfsD622bi+JuXCPy/CtD4u59K+Lc7EPzs4oD6T45a+FsLCP9qtxT5Plpq+IuHBP8Yxyj6skHK+p7HBP7x52j4/G3m+gufAP6eU3z5kPyu+K8LFP99P9T55AjG+whbFP8+++j7iOsa9pvHJP51ICD+V7s69DkvJP6RTCz9JgUW9sFnKP/FFCz8yHE+9+rPJP+VBDj8AAAAAVWvLP74zDj8AAAAAF7bKP/Q3ET8AAAAA34m5P3ZvPT8AAAAAB+23P1fQPD+jXJq9PPW2P5P+Oj+94Z69ZY24PxZsOz83Nxa+nNu0Px8uNT/Y1Bm+UG+2P7pnNT8s13u+xxCwP9xIJT9TBIC+9KmxP8qKJT9pc6S+iCysP6UtGj/dDLe+qiiqP4ApFz8kKbm+IOyqP+AsFT9GmbW+aOirP9emFT/PhoS+hLetP/1oID/Nc4S+TUmsPxmNID8hzSi+SMCuP9i5KT/Ymyi+glWtP8vXKT/zPq69PgOwP0HULT/Rea29hJ+uP1/wLT8AAAAA9yKwPy3oLT8AAAAAf7+uP+z2LT/j41O+tRvhPxN9fj5vu1C+L+HgP/yKhT5cjWy+EcbbPyehjD5lNHK+xTjbPz1+hz6vsWu+pprVP+IDqz4ewW2+oz3UP9e+qD5QAXC+8+bUP7H7pj4suoW+8670P7HAFz6o5IS+N8P1PxxhET7Kh1C+FTr5P2FPWz6tMlO+HAn4P9wQYz6AYoy+cJn3P3B7Aj7ZsYm+Tpf3P/PmAD4Og4m+aOr/P1ORyj1ozYe+OMAAQH/3zj3/s4a+5EkAQHCxwj32mCi+TS37P7yWcD54uSi+LST6Px77eT5hxvS98WP+P44GgD5gOu29zbD9P0vKhT5b6dW9I8AAQJ6Xej6v6cG9fewAQDczgj4A4eO9VRMBQGZMcT78VAW+tmcBQBaHcz4yc+G9wHUBQGRbdj75Zei+isnXP0Vi2r6doOW+3gPYPzP+3b7RzOO+fezYP/ZG3b7vIOa+WTbZP1sK4L5os+q+/S7aP+I9r74sSum+G57aPzI4or4/O+i+MXjaPww/oL5hqOu+Tn7ZP66fnr4ofuy+DhPZP8fVoL6In9++mpkBQA1x3L6Oztm+cy4BQHBE377Yt7u+RS3/P15pwb4Dzrq+51H/P3Zrwb5VhLu+Y2MAQDJXvr70+ru+jpD/PzXSur6vQb++h/z/P+cBvL4qHOm+sDn/P1xVBr9eveK+Xr3+Pwg+Br/UKee+aM/3PyhIDL+1G+2+ZRv4P1BtDL+6pGq+kUcBQKJ/Mj7vrmO+jPcAQKRVLT41miy+pyMBQDEGZj4bZii+uroAQHgMXz6mYeC+rmcAQBLcoL6mKt2+CMkAQN4gor4Cf8i+C9H9Px1yi77VXsy+0A/9P0g3ir7U8vO++FABQB12x74bZvC+qMYBQKgeyb7nbQS/ryf+P2ke8L61pAO/I/n+P9Wv9L6LcAe/GTj4P0g1/L686Qa/fXb4P7nBAL900wq/HAvwPxe5/75HWgq/jPXvP6GDAr/9vgu/EaflP+F++L6AKQu/oTHlP4GT/b60kgK/eT3eP6ey4L7HSQG/MSbdP14R5L4HC7e+uw75P3XKc75GlLa+Y0X5P8k+aL7zdLa+xO34PwxYYr6hMLi+m3T3P0rRar42ykq+eXnSP8swvj7/zUu+7C/RP8AhvD79Lye+3GPRP9Y71D4z+Sa+qwfQP7lu0j61pvm9RSzQP3Wt7T5MVfq97dLOP57r6z5nf6C9YHPQPxJnAT+mtJ69eCnPP+seAT+JRCG9AU/QP0OOBT+wrR+9CObOP6Q2BT8AAAAAKSPQP9C3CT8AAAAAx5/OP+xMCT8n9TW+og3mPzAtij66hzS+l/3mP+WYhD6rXRO+4+PrP96tnD720Rm+MZXqP9V1oD5cHNW99TDuP7mo1j6Doc69aEHuPwA63D5Epty94X3tPy+K3j64Beu948PsP7E21j5HHPK+4c/uP3svDr+u7+u+WafuP1wADr+nW/a+EAPlPzFECr9/oe++vQDlPxJPCr/VtOu+aM/bPybl7r402ee+QyDbPzMx7b6q1OS+MBDcPw4t8r4ctbK+eEThP3VbCr94QbS+RIjfP5SlCr+PUqm+K9zgPyUfC78kKpS+ZVblP7IRDL+Mo5q+e2fmP4czC7//6Ya+6rHvP3RABr8H7Y2+TWfvP12jBb/ptpS+tYv1PwETAL9ya5q+6X70P4EkAL96w6W+RYD5Pw1t6L6Rnae+qvP4P2Zr5b4jL6u+SYD4P5T45L6mQtS+KT3fP8Q//L4jZta+pfXdP48c+b7RWda+CFrdP4If9b5d3wM/9+jvP+0q7L6/KgM/mUrpPwLw375zEf8+0nH7PwT+4L7OwfM+J0r8PwVvyL6tvuI+NSj6P7fUqb6UpcY+8uzuPwFqgr4Ysvo+pl7lP83qxb5cyuk+eETjP0Fmp76vfN4+9fXiP4Dzkr6+afI+qYnsP+lHs74TKtg+NiPxPxTn0L7+gco+hq/zP5/nv77pmd4+x2XmP1EuDb50s78+2GL1PyXpGr63DbM+jBIAQGBYHr76fBw/tcbMP7Cqjr5lqBo/mUnWP6gACL4AVBU/I03KP1T/2L6EKbI+PgMJQBmsaL4Medw+KT7mPxiwBL9mML4+rW3oP9iBC7/SAdk+CtrwP9xGB786PLw+CknwP6JDCL/83s4+HGD2P8xG/74z4bc+1Cz0PyB+Ar/k88I+gsX5P/KV6L4rabU+8Z/4P5lH3r5g554+t5YIQC/f6r4Z5qw+Lq4IQOoku77oLmk+hxnkP6slEb+RuFc+3rAIQHU6CL9KmZQ+GF7RPxQ/Kr+bO+o+dVnQP8R7Er8o1JM+uRoSQOi8dr7KpI4+x4IRQAFRuL4ZVHs+QPwQQKpl676sNk8+EaoWQGJnar4Fbk0+cw8WQBgip747/TA+uJUVQKKWzr6HwAE+BqAZQCzvar5qawQ+CD0ZQMv4l75fltY9cAYZQA4urb6nBBQ+10vlP1sjIj+0yuw9cGHxP9vBDD9TQFo+zvzgP0/OBD9OKS8+7NzqP+8e8D5rgII+qFfeP2+9vj67C1Q+NQjnP25ouj60VRk+F2XqP+m5xT5LdHY+SIzYP/660z7I03I902n3P53a6T5W8UY+XtjWP3oZ9T4xmP89BHDXP60UEj8a96Y+lPgCQKLrAr3+naU+8N8KQBH9ur3niac+wOz6P96uFz2Ef8E+vW3sP8r5Ij3xSAg/32rbP7Jjoz32soU+4bYEQCAJez1Ktoo+TfkBQMx5xj1Jgpg+fLjuPyrJ2j3dtoc+38DsPzs6Hj5O1U0+t2HuP2OZXj7BOCg+44r1P8nHfj6mCsY9lQ76P6mfjz5n1Hw9aocAQNQPij74io496iUEQBQ/Zj5jRBI+hNkEQLOzSD58nk8+w7gFQApl4T0hy3I+P/4BQORNLj5cxg09UKoBQIM1nj6rtAU902YEQDbKgj6toqc+24XcP3NmWz7JAZM+WHUSQJShKr7ZIpE+h4gMQGLWCzxUNnw+swsTQGbYaL06QYM+uFoNQOymVD1osUQ+zhkOQFfNAz40EmE9xjUIQCpzYz4Mkr49SUwOQKgBMz4Wpi8+/3YTQLUzzDxjmb49nKITQL9Fhz0n+GY9FTwaQNBHCb590LM9/tYXQP5gIL3CFRA+d4cXQANDtr0hIco9VP8ZQPzFLL5ZTCw+MzUXQCRkEL6JC+A9/OIZQEPiPr7irTM+iLsQQFM8Ar+Tw+c9t0MVQL4V6b5MOHQ9XOMYQJ1Hvb55Oyo//dzIP918Iz4/pwA/KlLJP3bjfT5RosU+QBjUP8SyiT59CdU+PRDBP7di3z5rDKo+kBDBP2ZqBj9avAQ/6ZjHPzGUcz7sUeA+ujDAP9np5z7Kw7I+PPi/P6/qDD/VsRo+U1rHP4ffJT8o1Q4+lgnJP49uID+QZwc/o5KsP090/T70iic/oPu0P4rKpj6y1TU/A9LCP+nVID7XUEI/j8W8P5lK/z1uNTc/NLqvPzScmj6ZRxY/IZOeP4TwCD/VymQ+SWOWP+EoVT9+4tg+n1mWP38WMz8Jb4c+Ko6LP2L5Vz/obPE+aeSPP/iLOT+ynR8/9DObP4QuCT/gE0M/w7irP7JikD44EjA/JeaVP1w++j5uFhM/MuaAP4QsHz8wYqc+QtJjPwUZQT9ihLg+FQBDP3hCEz+aXCA/T+ZzP5Q09z590D8/dXKSPzp6vD7ht00/rmSnPzRlVz4eUbk+lugwP2AFsD5DqiA/VHJqPyWwiT54fUY/9piQP8v0Oz59XEM/l2+TP1RXvjxvoCA/4Q11P1IPUT0tJMg+rkc5P3Cxoj0qxNM+yJpZP1QZFr4DeCc/lNmGP4HsFb4Zr0U/gGWZP+Y8A77QYCs/guKZP7fwvL5cWs0+w9iMP+Vf475gVkA/It+lP4bnnb6+Tso+jkCePwd8Dr8MI8U+Tn20P+GZKL+Szyc/ArmmP9eG8r4+zTk/WDmwP2K7u766LDY/m5K4P49t0b6JtB0/fuO1P9b8EL/D9bA+AwjDP2oTL7/k9Qw/RnvAP5c7E78M5TA/oE69PwGG1b7vWTs/z0y+P4WUj74HKUA//Wm5P8Yxir4DzT8/hczBP5uv8r12bkY/3bSzP+uoWr4BGEM/5Ia/P/Vlqb0+Qk0/yO6sP8NF7r2tTUc/DY68Pz9VBb3wpFE/zF6oP6Q0GzwfSUk/OPi6P0kwVTuMo1I/T3anPyhj3D27Y0k/1QO6P68Lfz0wg/E+hlfGP9DthT7DnwE/Nq32PzI7674xstQ+YHj1P5gxlb5Dj9A+rWp5P11Qp75hpCc//0GQPyoeh75Wnrg+KSD9P1gdmb6daJc+OGp1P+dVUT8EOQg/6x+IP6iNMj9J868+XMdQP1KALD+EuRk/gjd4P/K1Dz+e6rg+tRg4Px5w7T53gyA/qVBtP0Gdwj6musA+Yf8xP6x0Vz5ZpSA/YadsP92UIj6gxBc+0t/TP5qWNL8mcMM+nWjRP/zgIL+QvzQ+8pbFPwmIPb8TQ+o+P/3BPxO6I79jRCI/HxG/PxvaAL/LaDg/m+S9PzHSs76Bkz0/pgzAP0pATL4LXUk+GAe1P449N79O1AI/Yyi1P3hGH7/Lgi0/Az63PxpS/b7TTgk/KBCiP8fUBb9/hTQ/3jyrP5oF2r6GWj8/uMu0P18JpL6sA0Q/h8C9P97JR74vpU4+FvacPzFDG78U6As/rp+SP0/n0r7arTk/MGafPzdwr75r1kU/MqqsP1lShr6j5Ug/sK25P0GcF77KwFE++waIPxiT9r5b0Qo/aHiFPxXEmL4UBlU+GApsP3C0s76hugk/ZQFzPwiTEr76lzg/r0SQPxfyCL4yAEw/aOuiP2qg+b0WiE4/B7S0P3gplb1eS1g+iH9IP6voH752FgU/4gJUPw/ujj3guzU/sfqFP1RVKD2VC00/kpedP1Q3lzxDyFE/l5CxPxaIHjxhbUw+mBUqPy/7tT2UNAM/UAFMP9k+RD4j3EQ++UsjPwNdaz6xUgE/i1JKPzfHoT7eWTc/ntKBPwDKbz4IIk8/A7KbPzLkGD7aV1I/H52wP5cfuD1J2kU/rmS7P3tovz3kSj0+F9UiP/chvz7AJAE/zXRPP1Ex3j7G4Tw+7gkqP79F/z7g9gA/TmJYPzcXCz+8yTM/Bi6FP8wL4D4YQ0k/zJicP0/rlj6neDw+8gk1P0Z+HT9Dx/Y+TZ9hP+Y9Ij+uuTM+whhFP/LqOD/ooOs+OjxwP4emND9gciM/9iaKP7qeED+VYzo/ezOgPzIhxj47+yo+bodaP6KZTz9BYdg+BkyAP2vVRj8uqRo+f4ZvP7cIYD+dY8A+/dyMP/PKTT/ylA0/kIOUP4o9JD/pDjI/6+KiP0NX0j4hVwo+JHuKP8GqZj/rHqk+o7CVP/4OST9RFAQ/lN6ZP8glIj+A1ig/v/SmPx3o2T6EukA/kj22PwJHYj5osOk9pvGVP6q5YD+wyUY/EjK0P7bWVz6/1E8/HZSwPzPEET5CeRc/gAyxP7tf1T72ReI+I9yqP1RTCj9evJ89Tu/IP5cDLT8S3ZM9F5vKP2nlJj+9/G4+3o3FP3psEz+2on0+GyrEPxtlGT/sisk+Mli/P85vAD/0ir8+HlTAP7Ab9j63l+Q+9yHDPw2rsD7vV/E+LV7CPz/irz5Up2M+3nbXPwHe4j5Eaio+u+/WP2yyBj+wVYI9o+rXP1iMFj8fFJQ9Q8vmP71UKD+K52w9Le3yPzLkFD8yxxI+09ntP1iQAj+fIjc+l3PjP914Ez8Fhmw+Mv/iPxi0uD7QtEQ+tW3mPz1jAz8K9SQ+iSfpP+z4Dz8BNQU+3ULrP5uOHD/yQ4U9acjsP4C4Iz+nz5Y+isj8PxqLpj22KyQ+/fcBQBvVaT5QjZc9B0ELQGk6Sz6d8S0+2xgKQPWiJj4Cf3g+YcUKQJM5lj0Oo5g+AYkJQGKh1jzQ1Os9lx8BQGJOID5GRSw+srgAQOFiFT7i5Wk+PuwAQBGK7T3/s4Y+5EkAQNXrlj1PIoo+PiD4Pxwpuz2o5IQ+N8P1P/vOzz0jElU+Wfr4PzfCEj7AJis+9wb7P/T+Hz5hxvQ98WP+P6hvKT5b6dU9I8AAQBE0Jj7oavs+0STFPwxYgj5TsC4/zea7P341dz7OqA0/sATCPw6+cD7y6rQ+wtnDP6WfsD622bg+JuXCPy/CtD4u59I+Lc7EPzs4oD6T45Y+FsLCP9qtxT5Plpo+IuHBP8Yxyj6skHI+p7HBP7x52j4/G3k+gufAP6eU3z5kPys+K8LFP99P9T55AjE+whbFP8+++j7iOsY9pvHJP51ICD+V7s49DkvJP6RTCz9JgUU9sFnKP/FFCz8yHE89+rPJP+VBDj+jXJo9PPW2P5P+Oj+94Z49ZY24PxZsOz83NxY+nNu0Px8uNT/Y1Bk+UG+2P7pnNT8s13s+xxCwP9xIJT9TBIA+9KmxP8qKJT9pc6Q+iCysP6UtGj/dDLc+qiiqP4ApFz8kKbk+IOyqP+AsFT9GmbU+aOirP9emFT/PhoQ+hLetP/1oID/Nc4Q+TUmsPxmNID8hzSg+SMCuP9i5KT/Ymyg+glWtP8vXKT/zPq49PgOwP0HULT/Rea09hJ+uP1/wLT/j41M+tRvhPxN9fj5vu1A+L+HgP/yKhT5cjWw+EcbbPyehjD5lNHI+xTjbPz1+hz6vsWs+pprVP+IDqz4ewW0+oz3UP9e+qD5QAXA+8+bUP7H7pj4suoU+8670P7HAFz6o5IQ+N8P1PxxhET7Kh1A+FTr5P2FPWz6tMlM+HAn4P9wQYz6AYow+cJn3P3B7Aj7ZsYk+Tpf3P/PmAD4Og4k+aOr/P1ORyj1ozYc+OMAAQH/3zj3/s4Y+5EkAQHCxwj32mCg+TS37P7yWcD54uSg+LST6Px77eT5hxvQ98WP+P44GgD5gOu09zbD9P0vKhT5b6dU9I8AAQJ6Xej6v6cE9fewAQDczgj4A4eM9VRMBQGZMcT78VAU+tmcBQBaHcz4yc+E9wHUBQGRbdj75Zeg+isnXP0Vi2r6doOU+3gPYPzP+3b7RzOM+fezYP/ZG3b7vIOY+WTbZP1sK4L5os+o+/S7aP+I9r74sSuk+G57aPzI4or4/O+g+MXjaPww/oL5hqOs+Tn7ZP66fnr4ofuw+DhPZP8fVoL6In98+mpkBQA1x3L6Oztk+cy4BQHBE377Yt7s+RS3/P15pwb4Dzro+51H/P3Zrwb5VhLs+Y2MAQDJXvr70+rs+jpD/PzXSur6vQb8+h/z/P+cBvL4qHOk+sDn/P1xVBr9eveI+Xr3+Pwg+Br/UKec+aM/3PyhIDL+1G+0+ZRv4P1BtDL+6pGo+kUcBQKJ/Mj7vrmM+jPcAQKRVLT41miw+pyMBQDEGZj4bZig+uroAQHgMXz6mYeA+rmcAQBLcoL6mKt0+CMkAQN4gor4Cf8g+C9H9Px1yi77VXsw+0A/9P0g3ir7U8vM++FABQB12x74bZvA+qMYBQKgeyb7nbQQ/ryf+P2ke8L61pAM/I/n+P9Wv9L6LcAc/GTj4P0g1/L686QY/fXb4P7nBAL900wo/HAvwPxe5/75HWgo/jPXvP6GDAr/9vgs/EaflP+F++L6AKQs/oTHlP4GT/b60kgI/eT3eP6ey4L7HSQE/MSbdP14R5L4HC7c+uw75P3XKc75GlLY+Y0X5P8k+aL7zdLY+xO34PwxYYr6hMLg+m3T3P0rRar42yko+eXnSP8swvj7/zUs+7C/RP8AhvD79Lyc+3GPRP9Y71D4z+SY+qwfQP7lu0j61pvk9RSzQP3Wt7T5MVfo97dLOP57r6z5nf6A9YHPQPxJnAT+mtJ49eCnPP+seAT+JRCE9AU/QP0OOBT+wrR89CObOP6Q2BT8n9TU+og3mPzAtij66hzQ+l/3mP+WYhD6rXRM+4+PrP96tnD720Rk+MZXqP9V1oD5cHNU99TDuP7mo1j6Doc49aEHuPwA63D5Eptw94X3tPy+K3j64Bes948PsP7E21j5HHPI+4c/uP3svDr+u7+s+WafuP1wADr+nW/Y+EAPlPzFECr9/oe8+vQDlPxJPCr/VtOs+aM/bPybl7r402ec+QyDbPzMx7b6q1OQ+MBDcPw4t8r4ctbI+eEThP3VbCr94QbQ+RIjfP5SlCr+PUqk+K9zgPyUfC78kKpQ+ZVblP7IRDL+Mo5o+e2fmP4czC7//6YY+6rHvP3RABr8H7Y0+TWfvP12jBb/ptpQ+tYv1PwETAL9ya5o+6X70P4EkAL96w6U+RYD5Pw1t6L6Rnac+qvP4P2Zr5b4jL6s+SYD4P5T45L6mQtQ+KT3fP8Q//L4jZtY+pfXdP48c+b7RWdY+CFrdP4If9b4EAAAAAgAAACAAAAAgAAAALwAAAAQAAAAMAAAACgAAACwAAAAsAAAAKwAAAAwAAAAUAAAAEgAAACgAAAAoAAAAJwAAABQAAAAcAAAAGgAAACQAAAAkAAAAIwAAABwAAAAfAAAAHgAAAAAAAAAAAAAAAQAAAB8AAAAXAAAAFgAAABgAAAAYAAAAGQAAABcAAAAPAAAADgAAABAAAAAQAAAAEQAAAA8AAAAHAAAABgAAAAgAAAAIAAAACQAAAAcAAAABAAAAAAAAAAIAAAACAAAAAwAAAAEAAAAJAAAACAAAAAoAAAAKAAAACwAAAAkAAAARAAAAEAAAABIAAAASAAAAEwAAABEAAAAZAAAAGAAAABoAAAAaAAAAGwAAABkAAAACAAAAAAAAACEAAAAhAAAAIAAAAAIAAAAaAAAAGAAAACUAAAAlAAAAJAAAABoAAAASAAAAEAAAACkAAAApAAAAKAAAABIAAAAKAAAACAAAAC0AAAAtAAAALAAAAAoAAAAqAAAAKwAAAD4AAAA+AAAAPQAAACoAAAAGAAAABAAAAC8AAAAvAAAALgAAAAYAAAAOAAAADAAAACsAAAArAAAAKgAAAA4AAAAWAAAAFAAAACcAAAAnAAAAJgAAABYAAAAeAAAAHAAAACMAAAAjAAAAIgAAAB4AAAAdAAAAHAAAAB4AAAAeAAAAHwAAAB0AAAAVAAAAFAAAABYAAAAWAAAAFwAAABUAAAANAAAADAAAAA4AAAAOAAAADwAAAA0AAAAFAAAABAAAAAYAAAAGAAAABwAAAAUAAAADAAAAAgAAAAQAAAAEAAAABQAAAAMAAAALAAAACgAAAAwAAAAMAAAADQAAAAsAAAATAAAAEgAAABQAAAAUAAAAFQAAABMAAAAbAAAAGgAAABwAAAAcAAAAHQAAABsAAAAAAAAAHgAAACIAAAAiAAAAIQAAAAAAAAAYAAAAFgAAACYAAAAmAAAAJQAAABgAAAAQAAAADgAAACoAAAAqAAAAKQAAABAAAAAIAAAABgAAAC4AAAAuAAAALQAAAAgAAAA9AAAAPgAAAFAAAABQAAAAUQAAAD0AAAA/AAAAKwAAACwAAABQAAAAPwAAAEIAAABCAAAAQQAAAFAAAAA5AAAAOgAAAE0AAABNAAAATAAAADkAAAA1AAAANgAAAEkAAABJAAAASAAAADUAAAAxAAAAMgAAAEUAAABFAAAARAAAADEAAAAoAAAAKQAAADwAAAA8AAAAOwAAACgAAAAkAAAAJQAAADgAAAA4AAAANwAAACQAAAAgAAAAIQAAADQAAAA0AAAAMwAAACAAAABOAAAATwAAAEAAAABOAAAAQAAAAEEAAABOAAAAQQAAAEIAAABOAAAAQgAAAEMAAABOAAAAQwAAAEQAAABOAAAARAAAAEUAAABOAAAARQAAAEYAAABOAAAARgAAAEcAAABOAAAARwAAAEgAAABOAAAASAAAAEkAAABOAAAASQAAAEoAAABOAAAASgAAAEsAAABOAAAASwAAAEwAAABOAAAATAAAAE0AAAAvAAAAIAAAADMAAAAzAAAAMgAAAC8AAAAjAAAAJAAAADcAAAA3AAAANgAAACMAAAAnAAAAKAAAADsAAAA7AAAAOgAAACcAAAA+AAAAPwAAAFAAAAAyAAAAMwAAAEYAAABGAAAARQAAADIAAAA2AAAANwAAAEoAAABKAAAASQAAADYAAAA6AAAAOwAAAE4AAABOAAAATQAAADoAAABRAAAAUAAAAEEAAABBAAAAQAAAAFEAAAAqAAAAPAAAACkAAAA8AAAAPQAAAFEAAAA4AAAAOQAAAEwAAABMAAAASwAAADgAAAA0AAAANQAAAEgAAABIAAAARwAAADQAAAAwAAAAMQAAAEQAAABEAAAAQwAAADAAAAAsAAAALQAAADAAAAAwAAAAPwAAACwAAAAlAAAAJgAAADkAAAA5AAAAOAAAACUAAAAhAAAAIgAAADUAAAA1AAAANAAAACEAAAAtAAAALgAAADEAAAAxAAAAMAAAAC0AAAAuAAAALwAAADIAAAAyAAAAMQAAAC4AAAAiAAAAIwAAADYAAAA2AAAANQAAACIAAAAmAAAAJwAAADoAAAA6AAAAOQAAACYAAAA/AAAAMAAAAEMAAABDAAAAQgAAAD8AAAAzAAAANAAAAEcAAABHAAAARgAAADMAAAA3AAAAOAAAAEsAAABLAAAASgAAADcAAAA7AAAAPAAAAE8AAABPAAAATgAAADsAAAA8AAAAUQAAAEAAAABAAAAATwAAADwAAAA8AAAAKgAAAD0AAAArAAAAPwAAAD4AAAAdAAAAHwAAAAEAAAAdAAAAAQAAAAMAAAAdAAAAAwAAAAUAAAAdAAAABQAAAAcAAAAdAAAABwAAAAkAAAAdAAAACQAAAAsAAAAdAAAACwAAAA0AAAAdAAAADQAAAA8AAAAdAAAADwAAABEAAAAdAAAAEQAAABMAAAAdAAAAEwAAABUAAAAdAAAAFQAAABcAAAAdAAAAFwAAABkAAAAdAAAAGQAAABsAAAC9AAAAwQAAAG0AAABtAAAAawAAAL0AAAC5AAAAvgAAAFUAAABVAAAAUwAAALkAAAC2AAAAtQAAAF0AAABdAAAAWwAAALYAAACyAAAAugAAAGUAAABlAAAAYwAAALIAAACOAAAAjwAAAKIAAACiAAAAsQAAAI4AAACKAAAAiwAAAK4AAACuAAAArQAAAIoAAACGAAAAhwAAAKoAAACqAAAAqQAAAIYAAACCAAAAgwAAAKYAAACmAAAApQAAAIIAAAB+AAAAfwAAAJIAAACSAAAAoQAAAH4AAAB6AAAAewAAAJ4AAACeAAAAnQAAAHoAAAB2AAAAdwAAAJoAAACaAAAAmQAAAHYAAAByAAAAcwAAAJYAAACWAAAAlQAAAHIAAABvAAAAcQAAAIIAAACCAAAAkQAAAG8AAABnAAAAaQAAAI4AAACOAAAAjQAAAGcAAABfAAAAYQAAAIoAAACKAAAAiQAAAF8AAABXAAAAWQAAAIYAAACGAAAAhQAAAFcAAABWAAAAVAAAAHIAAAByAAAAgQAAAFYAAABeAAAAXAAAAH4AAAB+AAAAfQAAAF4AAABmAAAAZAAAAHoAAAB6AAAAeQAAAGYAAABuAAAAbAAAAHYAAAB2AAAAdQAAAG4AAADAAAAAbgAAAHAAAABwAAAAvwAAAMAAAAC7AAAAZgAAAGgAAABoAAAAvAAAALsAAACzAAAAXgAAAGAAAABgAAAAtAAAALMAAAC4AAAAVgAAAFgAAABYAAAAtwAAALgAAAC+AAAAVAAAAFYAAABWAAAAuAAAAL4AAAC1AAAAXAAAAF4AAABeAAAAswAAALUAAAC6AAAAZAAAAGYAAABmAAAAuwAAALoAAADBAAAAbAAAAG4AAABuAAAAwAAAAMEAAABUAAAAUgAAAHMAAABzAAAAcgAAAFQAAABsAAAAagAAAHcAAAB3AAAAdgAAAGwAAABkAAAAYgAAAHsAAAB7AAAAegAAAGQAAABcAAAAWgAAAH8AAAB/AAAAfgAAAFwAAABxAAAAUwAAAIMAAACDAAAAggAAAHEAAABZAAAAWwAAAIcAAACHAAAAhgAAAFkAAABhAAAAYwAAAIsAAACLAAAAigAAAGEAAABpAAAAawAAAI8AAACPAAAAjgAAAGkAAAB/AAAAgAAAAJMAAACTAAAAkgAAAH8AAABzAAAAdAAAAJcAAACXAAAAlgAAAHMAAAB3AAAAeAAAAJsAAACbAAAAmgAAAHcAAAB7AAAAfAAAAJ8AAACfAAAAngAAAHsAAACPAAAAkAAAAKMAAACjAAAAogAAAI8AAACDAAAAhAAAAKcAAACnAAAApgAAAIMAAACHAAAAiAAAAKsAAACrAAAAqgAAAIcAAACLAAAAjAAAAK8AAACvAAAArgAAAIsAAAC8AAAAvQAAAGsAAABrAAAAaQAAALwAAAC0AAAAsgAAAGMAAABjAAAAYQAAALQAAAC3AAAAtgAAAFsAAABbAAAAWQAAALcAAAC/AAAAuQAAAFMAAABTAAAAcQAAAL8AAADBAAAAwAAAAG8AAABvAAAAbQAAAMEAAAC+AAAAuAAAAFcAAABXAAAAVQAAAL4AAAC1AAAAswAAAF8AAABfAAAAXQAAALUAAAC6AAAAuwAAAGcAAABnAAAAZQAAALoAAACNAAAAjgAAALEAAACxAAAAsAAAAI0AAACJAAAAigAAAK0AAACtAAAArAAAAIkAAACFAAAAhgAAAKkAAACpAAAAqAAAAIUAAACRAAAAggAAAKUAAAClAAAApAAAAJEAAAB9AAAAfgAAAKEAAAChAAAAoAAAAH0AAAB5AAAAegAAAJ0AAACdAAAAnAAAAHkAAAB1AAAAdgAAAJkAAACZAAAAmAAAAHUAAACBAAAAcgAAAJUAAACVAAAAlAAAAIEAAABtAAAAbwAAAJEAAACRAAAAkAAAAG0AAABlAAAAZwAAAI0AAACNAAAAjAAAAGUAAABdAAAAXwAAAIkAAACJAAAAiAAAAF0AAABVAAAAVwAAAIUAAACFAAAAhAAAAFUAAABYAAAAVgAAAIEAAACBAAAAgAAAAFgAAABgAAAAXgAAAH0AAAB9AAAAfAAAAGAAAABoAAAAZgAAAHkAAAB5AAAAeAAAAGgAAABwAAAAbgAAAHUAAAB1AAAAdAAAAHAAAAC/AAAAcAAAAFIAAABSAAAAuQAAAL8AAAC8AAAAaAAAAGoAAABqAAAAvQAAALwAAAC0AAAAYAAAAGIAAABiAAAAsgAAALQAAAC3AAAAWAAAAFoAAABaAAAAtgAAALcAAAC5AAAAUgAAAFQAAABUAAAAvgAAALkAAAC2AAAAWgAAAFwAAABcAAAAtQAAALYAAACyAAAAYgAAAGQAAABkAAAAugAAALIAAAC9AAAAagAAAGwAAABsAAAAwQAAAL0AAABSAAAAcAAAAHQAAAB0AAAAcwAAAFIAAABqAAAAaAAAAHgAAAB4AAAAdwAAAGoAAABiAAAAYAAAAHwAAAB8AAAAewAAAGIAAABaAAAAWAAAAIAAAACAAAAAfwAAAFoAAABTAAAAVQAAAIQAAACEAAAAgwAAAFMAAABbAAAAXQAAAIgAAACIAAAAhwAAAFsAAABjAAAAZQAAAIwAAACMAAAAiwAAAGMAAABrAAAAbQAAAJAAAACQAAAAjwAAAGsAAACAAAAAgQAAAJQAAACUAAAAkwAAAIAAAAB0AAAAdQAAAJgAAACYAAAAlwAAAHQAAAB4AAAAeQAAAJwAAACcAAAAmwAAAHgAAAB8AAAAfQAAAKAAAACgAAAAnwAAAHwAAACQAAAAkQAAAKQAAACkAAAAowAAAJAAAACEAAAAhQAAAKgAAACoAAAApwAAAIQAAACIAAAAiQAAAKwAAACsAAAAqwAAAIgAAACMAAAAjQAAALAAAACwAAAArwAAAIwAAAC7AAAAvAAAAGkAAABpAAAAZwAAALsAAACzAAAAtAAAAGEAAABhAAAAXwAAALMAAAC4AAAAtwAAAFkAAABZAAAAVwAAALgAAADAAAAAvwAAAHEAAABxAAAAbwAAAMAAAACgAAAAoQAAAJIAAACgAAAAkgAAAJMAAACgAAAAkwAAAJQAAACgAAAAlAAAAJUAAACgAAAAlQAAAJYAAACgAAAAlgAAAJcAAACgAAAAlwAAAJgAAACgAAAAmAAAAJkAAACgAAAAmQAAAJoAAACgAAAAmgAAAJsAAACgAAAAmwAAAJwAAACgAAAAnAAAAJ0AAACgAAAAnQAAAJ4AAACgAAAAngAAAJ8AAACwAAAAsQAAAKIAAACwAAAAogAAAKMAAACwAAAAowAAAKQAAACwAAAApAAAAKUAAACwAAAApQAAAKYAAACwAAAApgAAAKcAAACwAAAApwAAAKgAAACwAAAAqAAAAKkAAACwAAAAqQAAAKoAAACwAAAAqgAAAKsAAACwAAAAqwAAAKwAAACwAAAArAAAAK0AAACwAAAArQAAAK4AAACwAAAArgAAAK8AAADsAAAA/gAAAOsAAAD+AAAA/wAAABMBAAD6AAAA+wAAAA4BAAAOAQAADQEAAPoAAAD2AAAA9wAAAAoBAAAKAQAACQEAAPYAAADyAAAA8wAAAAYBAAAGAQAABQEAAPIAAADuAAAA7wAAAPIAAADyAAAAAQEAAO4AAADqAAAA6wAAAP4AAAD+AAAA/QAAAOoAAADmAAAA5wAAAPoAAAD6AAAA+QAAAOYAAADiAAAA4wAAAPYAAAD2AAAA9QAAAOIAAADGAAAAxAAAAOIAAADiAAAA8QAAAMYAAADOAAAAzAAAAO4AAADuAAAA7QAAAM4AAADWAAAA1AAAAOoAAADqAAAA6QAAANYAAADeAAAA3AAAAOYAAADmAAAA5QAAAN4AAAAQAQAAEQEAAAIBAAAQAQAAAgEAAAMBAAAQAQAAAwEAAAQBAAAQAQAABAEAAAUBAAAQAQAABQEAAAYBAAAQAQAABgEAAAcBAAAQAQAABwEAAAgBAAAQAQAACAEAAAkBAAAQAQAACQEAAAoBAAAQAQAACgEAAAsBAAAQAQAACwEAAAwBAAAQAQAADAEAAA0BAAAQAQAADQEAAA4BAAAQAQAADgEAAA8BAADbAAAA2gAAANwAAADcAAAA3QAAANsAAADTAAAA0gAAANQAAADUAAAA1QAAANMAAADLAAAAygAAAMwAAADMAAAAzQAAAMsAAADDAAAAwgAAAMQAAADEAAAAxQAAAMMAAADJAAAAyAAAAMoAAADKAAAAywAAAMkAAADRAAAA0AAAANIAAADSAAAA0wAAANEAAADZAAAA2AAAANoAAADaAAAA2wAAANkAAADhAAAA4AAAAMIAAADCAAAAwwAAAOEAAADgAAAA3gAAAOUAAADlAAAA5AAAAOAAAADYAAAA1gAAAOkAAADpAAAA6AAAANgAAADQAAAAzgAAAO0AAADtAAAA7AAAANAAAADIAAAAxgAAAPEAAADxAAAA8AAAAMgAAADxAAAA4gAAAPUAAAD1AAAA9AAAAPEAAADlAAAA5gAAAPkAAAD5AAAA+AAAAOUAAADpAAAA6gAAAP0AAAD9AAAA/AAAAOkAAADtAAAAAQEAAAABAAABAQAA8gAAAAUBAAAFAQAABAEAAAEBAAD1AAAA9gAAAAkBAAAJAQAACAEAAPUAAAD5AAAA+gAAAA0BAAANAQAADAEAAPkAAAD9AAAA/gAAABEBAAARAQAAEAEAAP0AAAD+AAAAEwEAAAIBAAACAQAAEQEAAP4AAAABAQAA7QAAAO4AAAASAQAAAQEAAAQBAAAEAQAAAwEAABIBAAD7AAAA/AAAAA8BAAAPAQAADgEAAPsAAAD3AAAA+AAAAAsBAAALAQAACgEAAPcAAADzAAAA9AAAAAcBAAAHAQAABgEAAPMAAAD/AAAAAAEAABIBAAASAQAAEwEAAP8AAAD+AAAA7AAAAP8AAADnAAAA6AAAAPsAAAD7AAAA+gAAAOcAAADjAAAA5AAAAPcAAAD3AAAA9gAAAOMAAADvAAAA8AAAAPMAAADzAAAA8gAAAO8AAADMAAAAygAAAO8AAADvAAAA7gAAAMwAAADUAAAA0gAAAOsAAADrAAAA6gAAANQAAADcAAAA2gAAAOcAAADnAAAA5gAAANwAAADEAAAAwgAAAOMAAADjAAAA4gAAAMQAAADdAAAA3AAAAN4AAADeAAAA3wAAAN0AAADVAAAA1AAAANYAAADWAAAA1wAAANUAAADNAAAAzAAAAM4AAADOAAAAzwAAAM0AAADFAAAAxAAAAMYAAADGAAAAxwAAAMUAAADHAAAAxgAAAMgAAADIAAAAyQAAAMcAAADPAAAAzgAAANAAAADQAAAA0QAAAM8AAADXAAAA1gAAANgAAADYAAAA2QAAANcAAADfAAAA3gAAAOAAAADgAAAA4QAAAN8AAADCAAAA4AAAAOQAAADkAAAA4wAAAMIAAADaAAAA2AAAAOgAAADoAAAA5wAAANoAAADSAAAA0AAAAOwAAADsAAAA6wAAANIAAADKAAAAyAAAAPAAAADwAAAA7wAAAMoAAADwAAAA8QAAAPQAAAD0AAAA8wAAAPAAAADkAAAA5QAAAPgAAAD4AAAA9wAAAOQAAADoAAAA6QAAAPwAAAD8AAAA+wAAAOgAAADsAAAA7QAAAAABAAAAAQAA/wAAAOwAAAAAAQAAAQEAABIBAAD0AAAA9QAAAAgBAAAIAQAABwEAAPQAAAD4AAAA+QAAAAwBAAAMAQAACwEAAPgAAAD8AAAA/QAAABABAAAQAQAADwEAAPwAAAATAQAAEgEAAAMBAAADAQAAAgEAABMBAADbAAAA3QAAACIBAAAiAQAAIQEAANsAAADTAAAA1QAAAB4BAAAeAQAAHQEAANMAAADLAAAAzQAAABoBAAAaAQAAGQEAAMsAAADDAAAAxQAAABYBAAAWAQAAFQEAAMMAAADFAAAAxwAAABcBAAAXAQAAFgEAAMUAAADNAAAAzwAAABsBAAAbAQAAGgEAAM0AAADVAAAA1wAAAB8BAAAfAQAAHgEAANUAAADdAAAA3wAAACMBAAAjAQAAIgEAAN0AAADZAAAA2wAAACEBAAAhAQAAIAEAANkAAADRAAAA0wAAAB0BAAAdAQAAHAEAANEAAADJAAAAywAAABkBAAAZAQAAGAEAAMkAAADhAAAAwwAAABUBAAAVAQAAFAEAAOEAAADHAAAAyQAAABgBAAAYAQAAFwEAAMcAAADPAAAA0QAAABwBAAAcAQAAGwEAAM8AAADXAAAA2QAAACABAAAgAQAAHwEAANcAAADfAAAA4QAAABQBAAAUAQAAIwEAAN8AAACPAQAAkwEAAD8BAAA/AQAAPQEAAI8BAACLAQAAkAEAACcBAAAnAQAAJQEAAIsBAACIAQAAhwEAAC8BAAAvAQAALQEAAIgBAACEAQAAjAEAADcBAAA3AQAANQEAAIQBAABgAQAAYQEAAHQBAAB0AQAAgwEAAGABAABcAQAAXQEAAIABAACAAQAAfwEAAFwBAABYAQAAWQEAAHwBAAB8AQAAewEAAFgBAABUAQAAVQEAAHgBAAB4AQAAdwEAAFQBAABQAQAAUQEAAGQBAABkAQAAcwEAAFABAABMAQAATQEAAHABAABwAQAAbwEAAEwBAABIAQAASQEAAGwBAABsAQAAawEAAEgBAABEAQAARQEAAGgBAABoAQAAZwEAAEQBAABBAQAAQwEAAFQBAABUAQAAYwEAAEEBAAA5AQAAOwEAAGABAABgAQAAXwEAADkBAAAxAQAAMwEAAFwBAABcAQAAWwEAADEBAAApAQAAKwEAAFgBAABYAQAAVwEAACkBAAAoAQAAJgEAAEQBAABEAQAAUwEAACgBAAAwAQAALgEAAFABAABQAQAATwEAADABAAA4AQAANgEAAEwBAABMAQAASwEAADgBAABAAQAAPgEAAEgBAABIAQAARwEAAEABAACSAQAAQAEAAEIBAABCAQAAkQEAAJIBAACNAQAAOAEAADoBAAA6AQAAjgEAAI0BAACFAQAAMAEAADIBAAAyAQAAhgEAAIUBAACKAQAAKAEAACoBAAAqAQAAiQEAAIoBAACQAQAAJgEAACgBAAAoAQAAigEAAJABAACHAQAALgEAADABAAAwAQAAhQEAAIcBAACMAQAANgEAADgBAAA4AQAAjQEAAIwBAACTAQAAPgEAAEABAABAAQAAkgEAAJMBAAAmAQAAJAEAAEUBAABFAQAARAEAACYBAAA+AQAAPAEAAEkBAABJAQAASAEAAD4BAAA2AQAANAEAAE0BAABNAQAATAEAADYBAAAuAQAALAEAAFEBAABRAQAAUAEAAC4BAABDAQAAJQEAAFUBAABVAQAAVAEAAEMBAAArAQAALQEAAFkBAABZAQAAWAEAACsBAAAzAQAANQEAAF0BAABdAQAAXAEAADMBAAA7AQAAPQEAAGEBAABhAQAAYAEAADsBAABRAQAAUgEAAGUBAABlAQAAZAEAAFEBAABFAQAARgEAAGkBAABpAQAAaAEAAEUBAABJAQAASgEAAG0BAABtAQAAbAEAAEkBAABNAQAATgEAAHEBAABxAQAAcAEAAE0BAABhAQAAYgEAAHUBAAB1AQAAdAEAAGEBAABVAQAAVgEAAHkBAAB5AQAAeAEAAFUBAABZAQAAWgEAAH0BAAB9AQAAfAEAAFkBAABdAQAAXgEAAIEBAACBAQAAgAEAAF0BAACOAQAAjwEAAD0BAAA9AQAAOwEAAI4BAACGAQAAhAEAADUBAAA1AQAAMwEAAIYBAACJAQAAiAEAAC0BAAAtAQAAKwEAAIkBAACRAQAAiwEAACUBAAAlAQAAQwEAAJEBAACTAQAAkgEAAEEBAABBAQAAPwEAAJMBAACQAQAAigEAACkBAAApAQAAJwEAAJABAACHAQAAhQEAADEBAAAxAQAALwEAAIcBAACMAQAAjQEAADkBAAA5AQAANwEAAIwBAABfAQAAYAEAAIMBAACDAQAAggEAAF8BAABbAQAAXAEAAH8BAAB/AQAAfgEAAFsBAABXAQAAWAEAAHsBAAB7AQAAegEAAFcBAABjAQAAVAEAAHcBAAB3AQAAdgEAAGMBAABPAQAAUAEAAHMBAABzAQAAcgEAAE8BAABLAQAATAEAAG8BAABvAQAAbgEAAEsBAABHAQAASAEAAGsBAABrAQAAagEAAEcBAABTAQAARAEAAGcBAABnAQAAZgEAAFMBAAA/AQAAQQEAAGMBAABjAQAAYgEAAD8BAAA3AQAAOQEAAF8BAABfAQAAXgEAADcBAAAvAQAAMQEAAFsBAABbAQAAWgEAAC8BAAAnAQAAKQEAAFcBAABXAQAAVgEAACcBAAAqAQAAKAEAAFMBAABTAQAAUgEAACoBAAAyAQAAMAEAAE8BAABPAQAATgEAADIBAAA6AQAAOAEAAEsBAABLAQAASgEAADoBAABCAQAAQAEAAEcBAABHAQAARgEAAEIBAACRAQAAQgEAACQBAAAkAQAAiwEAAJEBAACOAQAAOgEAADwBAAA8AQAAjwEAAI4BAACGAQAAMgEAADQBAAA0AQAAhAEAAIYBAACJAQAAKgEAACwBAAAsAQAAiAEAAIkBAACLAQAAJAEAACYBAAAmAQAAkAEAAIsBAACIAQAALAEAAC4BAAAuAQAAhwEAAIgBAACEAQAANAEAADYBAAA2AQAAjAEAAIQBAACPAQAAPAEAAD4BAAA+AQAAkwEAAI8BAAAkAQAAQgEAAEYBAABGAQAARQEAACQBAAA8AQAAOgEAAEoBAABKAQAASQEAADwBAAA0AQAAMgEAAE4BAABOAQAATQEAADQBAAAsAQAAKgEAAFIBAABSAQAAUQEAACwBAAAlAQAAJwEAAFYBAABWAQAAVQEAACUBAAAtAQAALwEAAFoBAABaAQAAWQEAAC0BAAA1AQAANwEAAF4BAABeAQAAXQEAADUBAAA9AQAAPwEAAGIBAABiAQAAYQEAAD0BAABSAQAAUwEAAGYBAABmAQAAZQEAAFIBAABGAQAARwEAAGoBAABqAQAAaQEAAEYBAABKAQAASwEAAG4BAABuAQAAbQEAAEoBAABOAQAATwEAAHIBAAByAQAAcQEAAE4BAABiAQAAYwEAAHYBAAB2AQAAdQEAAGIBAABWAQAAVwEAAHoBAAB6AQAAeQEAAFYBAABaAQAAWwEAAH4BAAB+AQAAfQEAAFoBAABeAQAAXwEAAIIBAACCAQAAgQEAAF4BAACNAQAAjgEAADsBAAA7AQAAOQEAAI0BAACFAQAAhgEAADMBAAAzAQAAMQEAAIUBAACKAQAAiQEAACsBAAArAQAAKQEAAIoBAACSAQAAkQEAAEMBAABDAQAAQQEAAJIBAAByAQAAcwEAAGQBAAByAQAAZAEAAGUBAAByAQAAZQEAAGYBAAByAQAAZgEAAGcBAAByAQAAZwEAAGgBAAByAQAAaAEAAGkBAAByAQAAaQEAAGoBAAByAQAAagEAAGsBAAByAQAAawEAAGwBAAByAQAAbAEAAG0BAAByAQAAbQEAAG4BAAByAQAAbgEAAG8BAAByAQAAbwEAAHABAAByAQAAcAEAAHEBAACCAQAAgwEAAHQBAACCAQAAdAEAAHUBAACCAQAAdQEAAHYBAACCAQAAdgEAAHcBAACCAQAAdwEAAHgBAACCAQAAeAEAAHkBAACCAQAAeQEAAHoBAACCAQAAegEAAHsBAACCAQAAewEAAHwBAACCAQAAfAEAAH0BAACCAQAAfQEAAH4BAACCAQAAfgEAAH8BAACCAQAAfwEAAIABAACCAQAAgAEAAIEBAADDAQAAtAEAAJYBAACWAQAAmAEAAMMBAAC/AQAAwAEAAJ4BAACeAQAAoAEAAL8BAAC7AQAAvAEAAKYBAACmAQAAqAEAALsBAAC3AQAAuAEAAK4BAACuAQAAsAEAALcBAACVAQAAlAEAALIBAACyAQAAswEAAJUBAACtAQAArAEAAKoBAACqAQAAqwEAAK0BAAClAQAApAEAAKIBAACiAQAAowEAAKUBAACdAQAAnAEAAJoBAACaAQAAmwEAAJ0BAACXAQAAlgEAAJQBAACUAQAAlQEAAJcBAACfAQAAngEAAJwBAACcAQAAnQEAAJ8BAACnAQAApgEAAKQBAACkAQAApQEAAKcBAACvAQAArgEAAKwBAACsAQAArQEAAK8BAAC0AQAAtQEAAJQBAACUAQAAlgEAALQBAAC4AQAAuQEAAKwBAACsAQAArgEAALgBAAC8AQAAvQEAAKQBAACkAQAApgEAALwBAADAAQAAwQEAAJwBAACcAQAAngEAAMABAADRAQAA0gEAAL8BAAC/AQAAvgEAANEBAADCAQAAwwEAAJgBAACYAQAAmgEAAMIBAAC+AQAAvwEAAKABAACgAQAAogEAAL4BAAC6AQAAuwEAAKgBAACoAQAAqgEAALoBAAC2AQAAtwEAALABAACwAQAAsgEAALYBAACzAQAAsgEAALABAACwAQAAsQEAALMBAACrAQAAqgEAAKgBAACoAQAAqQEAAKsBAACjAQAAogEAAKABAACgAQAAoQEAAKMBAACbAQAAmgEAAJgBAACYAQAAmQEAAJsBAACZAQAAmAEAAJYBAACWAQAAlwEAAJkBAAChAQAAoAEAAJ4BAACeAQAAnwEAAKEBAACpAQAAqAEAAKYBAACmAQAApwEAAKkBAACxAQAAsAEAAK4BAACuAQAArwEAALEBAAC1AQAAtgEAALIBAACyAQAAlAEAALUBAAC5AQAAugEAAKoBAACqAQAArAEAALkBAAC9AQAAvgEAAKIBAACiAQAApAEAAL0BAADBAQAAwgEAAJoBAACaAQAAnAEAAMEBAADlAQAA5AEAANIBAADSAQAA0QEAAOUBAADAAQAAvwEAANMBAADVAQAA1gEAANMBAADTAQAA5AEAANUBAADgAQAA4QEAAM4BAADOAQAAzQEAAOABAADcAQAA3QEAAMoBAADKAQAAyQEAANwBAADYAQAA2QEAAMYBAADGAQAAxQEAANgBAADPAQAA0AEAAL0BAAC9AQAAvAEAAM8BAADLAQAAzAEAALkBAAC5AQAAuAEAAMsBAADHAQAAyAEAALUBAAC1AQAAtAEAAMcBAADhAQAA4AEAAN8BAADhAQAA3wEAAN4BAADhAQAA3gEAAN0BAADhAQAA3QEAANwBAADhAQAA3AEAANsBAADhAQAA2wEAANoBAADhAQAA2gEAANkBAADhAQAA2QEAANgBAADhAQAA2AEAANcBAADhAQAA1wEAANYBAADhAQAA1gEAANUBAADhAQAA1QEAANQBAADhAQAA1AEAAOMBAADhAQAA4wEAAOIBAADGAQAAxwEAALQBAAC0AQAAwwEAAMYBAADKAQAAywEAALgBAAC4AQAAtwEAAMoBAADOAQAAzwEAALwBAAC8AQAAuwEAAM4BAADkAQAA0wEAANIBAADZAQAA2gEAAMcBAADHAQAAxgEAANkBAADdAQAA3gEAAMsBAADLAQAAygEAAN0BAADhAQAA4gEAAM8BAADPAQAAzgEAAOEBAADUAQAA1QEAAOQBAADkAQAA5QEAANQBAAC9AQAA0AEAAL4BAADlAQAA0QEAANABAADfAQAA4AEAAM0BAADNAQAAzAEAAN8BAADbAQAA3AEAAMkBAADJAQAAyAEAANsBAADXAQAA2AEAAMUBAADFAQAAxAEAANcBAADTAQAAxAEAAMEBAADBAQAAwAEAANMBAADMAQAAzQEAALoBAAC6AQAAuQEAAMwBAADIAQAAyQEAALYBAAC2AQAAtQEAAMgBAADEAQAAxQEAAMIBAADCAQAAwQEAAMQBAADFAQAAxgEAAMMBAADDAQAAwgEAAMUBAADJAQAAygEAALcBAAC3AQAAtgEAAMkBAADNAQAAzgEAALsBAAC7AQAAugEAAM0BAADWAQAA1wEAAMQBAADEAQAA0wEAANYBAADaAQAA2wEAAMgBAADIAQAAxwEAANoBAADeAQAA3wEAAMwBAADMAQAAywEAAN4BAADiAQAA4wEAANABAADQAQAAzwEAAOIBAADjAQAA1AEAAOUBAADlAQAA0AEAAOMBAADRAQAAvgEAANABAADSAQAA0wEAAL8BAACvAQAArQEAAKsBAACvAQAAqwEAAKkBAACvAQAAqQEAAKcBAACvAQAApwEAAKUBAACvAQAApQEAAKMBAACvAQAAowEAAKEBAACvAQAAoQEAAJ8BAACvAQAAnwEAAJ0BAACvAQAAnQEAAJsBAACvAQAAmwEAAJkBAACvAQAAmQEAAJcBAACvAQAAlwEAAJUBAACvAQAAlQEAALMBAACvAQAAswEAALEBAAD/AQAAAQIAAFUCAABVAgAAUQIAAP8BAADnAQAA6QEAAFICAABSAgAATQIAAOcBAADvAQAA8QEAAEkCAABJAgAASgIAAO8BAAD3AQAA+QEAAE4CAABOAgAARgIAAPcBAABFAgAANgIAACMCAAAjAgAAIgIAAEUCAABBAgAAQgIAAB8CAAAfAgAAHgIAAEECAAA9AgAAPgIAABsCAAAbAgAAGgIAAD0CAAA5AgAAOgIAABcCAAAXAgAAFgIAADkCAAA1AgAAJgIAABMCAAATAgAAEgIAADUCAAAxAgAAMgIAAA8CAAAPAgAADgIAADECAAAtAgAALgIAAAsCAAALAgAACgIAAC0CAAApAgAAKgIAAAcCAAAHAgAABgIAACkCAAAlAgAAFgIAAAUCAAAFAgAAAwIAACUCAAAhAgAAIgIAAP0BAAD9AQAA+wEAACECAAAdAgAAHgIAAPUBAAD1AQAA8wEAAB0CAAAZAgAAGgIAAO0BAADtAQAA6wEAABkCAAAVAgAABgIAAOgBAADoAQAA6gEAABUCAAARAgAAEgIAAPABAADwAQAA8gEAABECAAANAgAADgIAAPgBAAD4AQAA+gEAAA0CAAAJAgAACgIAAAACAAAAAgAAAgIAAAkCAABTAgAABAIAAAICAAACAgAAVAIAAFMCAABQAgAA/AEAAPoBAAD6AQAATwIAAFACAABIAgAA9AEAAPIBAADyAQAARwIAAEgCAABLAgAA7AEAAOoBAADqAQAATAIAAEsCAABMAgAA6gEAAOgBAADoAQAAUgIAAEwCAABHAgAA8gEAAPABAADwAQAASQIAAEcCAABPAgAA+gEAAPgBAAD4AQAATgIAAE8CAABUAgAAAgIAAAACAAAAAgAAVQIAAFQCAAAGAgAABwIAAOYBAADmAQAA6AEAAAYCAAAKAgAACwIAAP4BAAD+AQAAAAIAAAoCAAAOAgAADwIAAPYBAAD2AQAA+AEAAA4CAAASAgAAEwIAAO4BAADuAQAA8AEAABICAAAWAgAAFwIAAOcBAADnAQAABQIAABYCAAAaAgAAGwIAAO8BAADvAQAA7QEAABoCAAAeAgAAHwIAAPcBAAD3AQAA9QEAAB4CAAAiAgAAIwIAAP8BAAD/AQAA/QEAACICAAAmAgAAJwIAABQCAAAUAgAAEwIAACYCAAAqAgAAKwIAAAgCAAAIAgAABwIAACoCAAAuAgAALwIAAAwCAAAMAgAACwIAAC4CAAAyAgAAMwIAABACAAAQAgAADwIAADICAAA2AgAANwIAACQCAAAkAgAAIwIAADYCAAA6AgAAOwIAABgCAAAYAgAAFwIAADoCAAA+AgAAPwIAABwCAAAcAgAAGwIAAD4CAABCAgAAQwIAACACAAAgAgAAHwIAAEICAAD9AQAA/wEAAFECAABRAgAAUAIAAP0BAAD1AQAA9wEAAEYCAABGAgAASAIAAPUBAADtAQAA7wEAAEoCAABKAgAASwIAAO0BAAAFAgAA5wEAAE0CAABNAgAAUwIAAAUCAAABAgAAAwIAAFQCAABUAgAAVQIAAAECAADpAQAA6wEAAEwCAABMAgAAUgIAAOkBAADxAQAA8wEAAEcCAABHAgAASQIAAPEBAAD5AQAA+wEAAE8CAABPAgAATgIAAPkBAABEAgAARQIAACICAAAiAgAAIQIAAEQCAABAAgAAQQIAAB4CAAAeAgAAHQIAAEACAAA8AgAAPQIAABoCAAAaAgAAGQIAADwCAAA4AgAAOQIAABYCAAAWAgAAJQIAADgCAAA0AgAANQIAABICAAASAgAAEQIAADQCAAAwAgAAMQIAAA4CAAAOAgAADQIAADACAAAsAgAALQIAAAoCAAAKAgAACQIAACwCAAAoAgAAKQIAAAYCAAAGAgAAFQIAACgCAAAkAgAAJQIAAAMCAAADAgAAAQIAACQCAAAgAgAAIQIAAPsBAAD7AQAA+QEAACACAAAcAgAAHQIAAPMBAADzAQAA8QEAABwCAAAYAgAAGQIAAOsBAADrAQAA6QEAABgCAAAUAgAAFQIAAOoBAADqAQAA7AEAABQCAAAQAgAAEQIAAPIBAADyAQAA9AEAABACAAAMAgAADQIAAPoBAAD6AQAA/AEAAAwCAAAIAgAACQIAAAICAAACAgAABAIAAAgCAABNAgAA5gEAAAQCAAAEAgAAUwIAAE0CAABRAgAA/gEAAPwBAAD8AQAAUAIAAFECAABGAgAA9gEAAPQBAAD0AQAASAIAAEYCAABKAgAA7gEAAOwBAADsAQAASwIAAEoCAABSAgAA6AEAAOYBAADmAQAATQIAAFICAABJAgAA8AEAAO4BAADuAQAASgIAAEkCAABOAgAA+AEAAPYBAAD2AQAARgIAAE4CAABVAgAAAAIAAP4BAAD+AQAAUQIAAFUCAAAHAgAACAIAAAQCAAAEAgAA5gEAAAcCAAALAgAADAIAAPwBAAD8AQAA/gEAAAsCAAAPAgAAEAIAAPQBAAD0AQAA9gEAAA8CAAATAgAAFAIAAOwBAADsAQAA7gEAABMCAAAXAgAAGAIAAOkBAADpAQAA5wEAABcCAAAbAgAAHAIAAPEBAADxAQAA7wEAABsCAAAfAgAAIAIAAPkBAAD5AQAA9wEAAB8CAAAjAgAAJAIAAAECAAABAgAA/wEAACMCAAAnAgAAKAIAABUCAAAVAgAAFAIAACcCAAArAgAALAIAAAkCAAAJAgAACAIAACsCAAAvAgAAMAIAAA0CAAANAgAADAIAAC8CAAAzAgAANAIAABECAAARAgAAEAIAADMCAAA3AgAAOAIAACUCAAAlAgAAJAIAADcCAAA7AgAAPAIAABkCAAAZAgAAGAIAADsCAAA/AgAAQAIAAB0CAAAdAgAAHAIAAD8CAABDAgAARAIAACECAAAhAgAAIAIAAEMCAAD7AQAA/QEAAFACAABQAgAATwIAAPsBAADzAQAA9QEAAEgCAABIAgAARwIAAPMBAADrAQAA7QEAAEsCAABLAgAATAIAAOsBAAADAgAABQIAAFMCAABTAgAAVAIAAAMCAAAzAgAAMgIAADECAAAzAgAAMQIAADACAAAzAgAAMAIAAC8CAAAzAgAALwIAAC4CAAAzAgAALgIAAC0CAAAzAgAALQIAACwCAAAzAgAALAIAACsCAAAzAgAAKwIAACoCAAAzAgAAKgIAACkCAAAzAgAAKQIAACgCAAAzAgAAKAIAACcCAAAzAgAAJwIAACYCAAAzAgAAJgIAADUCAAAzAgAANQIAADQCAABDAgAAQgIAAEECAABDAgAAQQIAAEACAABDAgAAQAIAAD8CAABDAgAAPwIAAD4CAABDAgAAPgIAAD0CAABDAgAAPQIAADwCAABDAgAAPAIAADsCAABDAgAAOwIAADoCAABDAgAAOgIAADkCAABDAgAAOQIAADgCAABDAgAAOAIAADcCAABDAgAANwIAADYCAABDAgAANgIAAEUCAABDAgAARQIAAEQCAACXBAAAmAQAAGEEAABhBAAAYgQAAJcEAABiBAAAYQQAAGAEAABgBAAAXwQAAGIEAACMAgAAhAIAAIUCAACFAgAAiwIAAIwCAACEAgAAjAIAAIICAACCAgAAgwIAAIQCAACDAgAAggIAAJQCAACUAgAAkwIAAIMCAACSBAAAjAQAAF8EAABfBAAAYAQAAJIEAACQBAAAjwQAAI4EAACQBAAAjgQAAI0EAACQBAAAjQQAAJEEAACQBAAAkQQAAIYEAACGBAAAhQQAAJAEAACMBAAAjQQAAIEEAACBBAAAggQAAIwEAACFBAAAhgQAAHgEAAB4BAAAdgQAAIUEAACCBAAAgQQAAH8EAAB/BAAAfgQAAIIEAAB6BAAAewQAAGUEAABlBAAAYwQAAHoEAAB3BAAAdgQAAHQEAAB0BAAAcwQAAHcEAAByBAAAcwQAAFsCAABbAgAAXAIAAHIEAABjBAAAZQQAAFYCAABWAgAAVwIAAGMEAACKAgAAhgIAAIcCAACKAgAAhwIAAIgCAACKAgAAiAIAAIkCAAB7AgAAfAIAAIoCAACKAgAAiQIAAHsCAAB3AgAAeAIAAIYCAACGAgAAhQIAAHcCAABtAgAAbgIAAHwCAAB8AgAAewIAAG0CAAB0AgAAdQIAAHgCAAB4AgAAdwIAAHQCAABiAgAAYwIAAHECAABxAgAAcAIAAGICAABpAgAAagIAAG0CAABtAgAAbAIAAGkCAABcAgAAWwIAAGkCAABpAgAAaAIAAFwCAABYAgAAVwIAAGICAABiAgAAYQIAAFgCAABdAgAAXAIAAGgCAABoAgAAZwIAAF0CAABZAgAAWAIAAGECAABhAgAAawIAAFkCAABhAgAAYgIAAHACAABwAgAAbwIAAGECAABoAgAAaQIAAGwCAABsAgAAdgIAAGgCAABsAgAAbQIAAHsCAAB7AgAAegIAAGwCAABwAgAAcQIAAH8CAAB/AgAAfgIAAHACAAB6AgAAewIAAIkCAACJAgAAiAIAAHoCAAB+AgAAfwIAAIICAACCAgAAjAIAAH4CAABjAgAAVgIAAI4CAACOAgAAjQIAAGMCAABxAgAAYwIAAI0CAACNAgAAlgIAAHECAABxBAAAcgQAAFwCAABcAgAAXQIAAHEEAAB1BAAAZAQAAFgCAABYAgAAWQIAAHUEAAB5BAAAegQAAGMEAABjBAAAZAQAAHkEAACABAAAdwQAAHMEAABzBAAAcgQAAIAEAACEBAAAhQQAAHYEAAB2BAAAdwQAAIQEAACIBAAAiQQAAHsEAAB7BAAAegQAAIgEAACPBAAAkAQAAIUEAACFBAAAhAQAAI8EAABgBAAAYQQAAIkEAACJBAAAiAQAAGAEAACTBAAAjgIAAFYCAABWAgAAZQQAAJMEAACaBAAAkwQAAGUEAABlBAAAewQAAJoEAACYBAAAmQQAAIkEAACJBAAAYQQAAJgEAACRBAAAjQQAAIwEAACMBAAAkgQAAJEEAACRBAAAkgQAAIcEAACHBAAAhgQAAJEEAACNBAAAjgQAAIMEAACDBAAAgQQAAI0EAACGBAAAhwQAAHkEAAB5BAAAeAQAAIYEAACBBAAAgwQAAIAEAACABAAAfwQAAIEEAAB+BAAAfwQAAHEEAABxBAAAcAQAAH4EAAB2BAAAeAQAAHUEAAB1BAAAdAQAAHYEAABzBAAAdAQAAFoCAABaAgAAWwIAAHMEAABkBAAAYwQAAFcCAABXAgAAWAIAAGQEAACCAgAAfwIAAJUCAACVAgAAlAIAAIICAACLAgAAhQIAAIYCAACGAgAAigIAAIsCAAB8AgAAfQIAAIsCAACLAgAAigIAAHwCAAB4AgAAeQIAAIcCAACHAgAAhgIAAHgCAABuAgAAbwIAAH0CAAB9AgAAfAIAAG4CAAB1AgAAdgIAAHkCAAB5AgAAeAIAAHUCAABmAgAAZwIAAHUCAAB1AgAAdAIAAGYCAABqAgAAawIAAG4CAABuAgAAbQIAAGoCAABbAgAAWgIAAGoCAABqAgAAaQIAAFsCAABXAgAAVgIAAGMCAABjAgAAYgIAAFcCAABeAgAAXQIAAGcCAABnAgAAZgIAAF4CAABaAgAAWQIAAGsCAABrAgAAagIAAFoCAABrAgAAYQIAAG8CAABvAgAAbgIAAGsCAABnAgAAaAIAAHYCAAB2AgAAdQIAAGcCAAB2AgAAbAIAAHoCAAB6AgAAeQIAAHYCAABvAgAAcAIAAH4CAAB+AgAAfQIAAG8CAAB5AgAAegIAAIgCAACIAgAAhwIAAHkCAAB9AgAAfgIAAIwCAACMAgAAiwIAAH0CAAB/AgAAcQIAAJYCAACWAgAAlQIAAH8CAABwBAAAcQQAAF0CAABdAgAAXgIAAHAEAAB0BAAAdQQAAFkCAABZAgAAWgIAAHQEAAB4BAAAeQQAAGQEAABkBAAAdQQAAHgEAAB/BAAAgAQAAHIEAAByBAAAcQQAAH8EAACDBAAAhAQAAHcEAAB3BAAAgAQAAIMEAACHBAAAiAQAAHoEAAB6BAAAeQQAAIcEAACOBAAAjwQAAIQEAACEBAAAgwQAAI4EAACSBAAAYAQAAIgEAACIBAAAhwQAAJIEAACZBAAAmgQAAHsEAAB7BAAAiQQAAJkEAACpAgAAZgQAAJ8EAACfBAAAnAIAAKkCAABqBAAAawQAAJsEAACbBAAAnAQAAGoEAACiBAAAnAQAAJgEAACYBAAAlwQAAKIEAACcAgAAnwQAAJQEAACUBAAAjwIAAJwCAACcBAAAmwQAAJkEAACZBAAAmAQAAJwEAACPAgAAlAQAAG8EAABvBAAAYAIAAI8CAACLBAAAigQAAHwEAAB8BAAAfQQAAIsEAABvBAAAbgQAAF8CAABfAgAAYAIAAG8EAACZAgAAmgIAAKcCAACnAgAApgIAAJkCAACfAgAAoAIAAKMCAACjAgAAogIAAJ8CAACRAgAAkgIAAJ8CAACfAgAAngIAAJECAACWAgAAjQIAAJoCAACaAgAAmQIAAJYCAAByAgAAgAIAAJICAACSAgAAkQIAAHICAACAAgAAgQIAAIQCAACEAgAAgwIAAIACAABlAgAAZgIAAHQCAAB0AgAAcwIAAGUCAACpAgAAqgIAAKECAACpAgAAoQIAAKICAACpAgAAogIAAKMCAACpAgAAowIAAKQCAACpAgAApAIAAKUCAACpAgAApQIAAKYCAACpAgAApgIAAKcCAACpAgAApwIAAKgCAABkAgAAZQIAAHMCAABzAgAAcgIAAGQCAACBAgAAdwIAAIUCAACFAgAAhAIAAIECAACAAgAAgwIAAJMCAACTAgAAkgIAAIACAACNAgAAjgIAAJsCAACbAgAAmgIAAI0CAACSAgAAkwIAAKACAACgAgAAnwIAAJICAACgAgAAlwIAAKQCAACkAgAAowIAAKACAACaAgAAmwIAAKgCAACoAgAApwIAAJoCAACoAgAAbQQAAGwEAACoAgAAbAQAAGsEAACoAgAAawQAAGoEAACoAgAAagQAAGkEAACoAgAAaQQAAGgEAACoAgAAaAQAAGcEAACoAgAAZwQAAGYEAACoAgAAZgQAAKkCAAB9BAAAfAQAAG4EAABuBAAAbwQAAH0EAABfBAAAjAQAAIIEAACCBAAAigQAAF8EAACWBAAAlwQAAGIEAABiBAAAiwQAAJYEAACeBAAAmwIAAI4CAACOAgAAkwQAAJ4EAAChBAAAogQAAJcEAACXBAAAlgQAAKEEAABpBAAAagQAAJwEAACcBAAAogQAAGkEAABtBAAAqAIAAJsCAACbAgAAngQAAG0EAABmBAAAZwQAAKAEAACgBAAAnwQAAGYEAABrBAAAbAQAAJ0EAACdBAAAmwQAAGsEAABnBAAAaAQAAKEEAAChBAAAoAQAAGcEAACfBAAAoAQAAJUEAACVBAAAlAQAAJ8EAACbBAAAnQQAAJoEAACaBAAAmQQAAJsEAACUBAAAlQQAAH0EAAB9BAAAbwQAAJQEAACKBAAAggQAAH4EAAB+BAAAfAQAAIoEAABuBAAAcAQAAF4CAABeAgAAXwIAAG4EAACdAgAAngIAAKECAAChAgAAqgIAAJ0CAACYAgAAmQIAAKYCAACmAgAApQIAAJgCAACeAgAAnwIAAKICAACiAgAAoQIAAJ4CAACQAgAAkQIAAJ4CAACeAgAAnQIAAJACAACVAgAAlgIAAJkCAACZAgAAmAIAAJUCAABkAgAAcgIAAJECAACRAgAAkAIAAGQCAABzAgAAdAIAAHcCAAB3AgAAgQIAAHMCAABfAgAAXgIAAGYCAABmAgAAZQIAAF8CAABgAgAAXwIAAGUCAABlAgAAZAIAAGACAAByAgAAcwIAAIECAACBAgAAgAIAAHICAABgAgAAZAIAAJACAACQAgAAjwIAAGACAACUAgAAlQIAAJgCAACYAgAAlwIAAJQCAACPAgAAkAIAAJ0CAACdAgAAnAIAAI8CAACTAgAAlAIAAJcCAACXAgAAoAIAAJMCAACXAgAAmAIAAKUCAAClAgAApAIAAJcCAACcAgAAnQIAAKoCAACqAgAAqQIAAJwCAAB8BAAAfgQAAHAEAABwBAAAbgQAAHwEAABiBAAAXwQAAIoEAACKBAAAiwQAAGIEAACVBAAAlgQAAIsEAACLBAAAfQQAAJUEAACdBAAAngQAAJMEAACTBAAAmgQAAJ0EAACgBAAAoQQAAJYEAACWBAAAlQQAAKAEAABoBAAAaQQAAKIEAACiBAAAoQQAAGgEAABsBAAAbQQAAJ4EAACeBAAAnQQAAGwEAAD6AgAA6wIAAPwCAAD8AgAA5wIAAPoCAAD5AgAA+gIAAOcCAADnAgAA5gIAAPkCAAD1AgAA9gIAAOMCAADjAgAA4gIAAPUCAADxAgAA8gIAAN8CAADfAgAA3gIAAPECAADtAgAA7gIAANsCAADbAgAA6gIAAO0CAADpAgAA6gIAANYCAADlAgAA5gIAANMCAADTAgAA0gIAAOUCAADhAgAA4gIAAM8CAADPAgAAzgIAAOECAADdAgAA3gIAAMsCAADLAgAA2gIAAN0CAADZAgAA2gIAAK8CAACvAgAAsQIAANkCAADVAgAA1gIAALcCAAC3AgAAuQIAANUCAADRAgAA0gIAAL8CAAC/AgAAwQIAANECAADNAgAAzgIAAMcCAADHAgAAyQIAAM0CAACsAgAAqwIAAMkCAADJAgAAygIAAKwCAADEAgAAwwIAAMECAADBAgAAwgIAAMQCAAC8AgAAuwIAALkCAAC5AgAAugIAALwCAAC0AgAAswIAALECAACxAgAAsgIAALQCAACuAgAArQIAAKsCAACrAgAArAIAAK4CAAC2AgAAtQIAALMCAACzAgAAtAIAALYCAAC+AgAAvQIAALsCAAC7AgAAvAIAAL4CAADGAgAAxQIAAMMCAADDAgAAxAIAAMYCAAD4AgAA9wIAAPYCAAD4AgAA9gIAAPUCAAD4AgAA9QIAAPQCAAD4AgAA9AIAAPMCAAD4AgAA8wIAAPICAAD4AgAA8gIAAPECAAD4AgAA8QIAAPACAAD4AgAA8AIAAO8CAAD4AgAA7wIAAO4CAAD4AgAA7gIAAO0CAAD4AgAA7QIAAOwCAAD4AgAA7AIAAOsCAAD4AgAA6wIAAPoCAAD4AgAA+gIAAPkCAADOAgAAzwIAAMUCAADFAgAAxwIAAM4CAADSAgAA0wIAAL0CAAC9AgAAvwIAANICAADWAgAA1wIAALUCAAC1AgAAtwIAANYCAADaAgAAywIAAK0CAACtAgAArwIAANoCAADeAgAA3wIAAMwCAADMAgAAywIAAN4CAADiAgAA4wIAANACAADQAgAAzwIAAOICAADmAgAA5wIAANQCAADUAgAA0wIAAOYCAADqAgAA2wIAANgCAADYAgAA1wIAAOoCAADuAgAA7wIAANwCAADcAgAA2wIAAO4CAADyAgAA8wIAAOACAADgAgAA3wIAAPICAAD2AgAA9wIAAOQCAADkAgAA4wIAAPYCAAD8AgAA6AIAAOcCAADUAgAA5wIAANUCAADrAgAA7AIAAPsCAAD7AgAA/AIAAOsCAAD4AgAA+QIAAOYCAADmAgAA5QIAAPgCAAD0AgAA9QIAAOICAADiAgAA4QIAAPQCAADwAgAA8QIAAN4CAADeAgAA3QIAAPACAAD7AgAA6gIAAOkCAADoAgAA6QIAANYCAADWAgAA1QIAAOgCAADkAgAA5QIAANICAADSAgAA0QIAAOQCAADgAgAA4QIAAM4CAADOAgAAzQIAAOACAADcAgAA3QIAANoCAADaAgAA2QIAANwCAADYAgAA2QIAALECAACxAgAAswIAANgCAADUAgAA1QIAALkCAAC5AgAAuwIAANQCAADQAgAA0QIAAMECAADBAgAAwwIAANACAADMAgAAzQIAAMkCAADJAgAAqwIAAMwCAADKAgAAyQIAAMcCAADHAgAAyAIAAMoCAADCAgAAwQIAAL8CAAC/AgAAwAIAAMICAAC6AgAAuQIAALcCAAC3AgAAuAIAALoCAACyAgAAsQIAAK8CAACvAgAAsAIAALICAACwAgAArwIAAK0CAACtAgAArgIAALACAAC4AgAAtwIAALUCAAC1AgAAtgIAALgCAADAAgAAvwIAAL0CAAC9AgAAvgIAAMACAADIAgAAxwIAAMUCAADFAgAAxgIAAMgCAADLAgAAzAIAAKsCAACrAgAArQIAAMsCAADPAgAA0AIAAMMCAADDAgAAxQIAAM8CAADTAgAA1AIAALsCAAC7AgAAvQIAANMCAADXAgAA2AIAALMCAACzAgAAtQIAANcCAADbAgAA3AIAANkCAADZAgAA2AIAANsCAADfAgAA4AIAAM0CAADNAgAAzAIAAN8CAADjAgAA5AIAANECAADRAgAA0AIAAOMCAADoAgAA1QIAAOcCAAD8AgAA+wIAAOkCAADpAgAA6AIAAPwCAADvAgAA8AIAAN0CAADdAgAA3AIAAO8CAADzAgAA9AIAAOECAADhAgAA4AIAAPMCAAD3AgAA+AIAAOUCAADlAgAA5AIAAPcCAADsAgAA7QIAAOoCAADqAgAA+wIAAOwCAADXAgAA1gIAAOoCAAALAwAADAMAAMgCAADIAgAAxgIAAAsDAAAHAwAACAMAAMACAADAAgAAvgIAAAcDAAADAwAABAMAALgCAAC4AgAAtgIAAAMDAAD/AgAAAAMAALACAACwAgAArgIAAP8CAAD+AgAA/wIAAK4CAACuAgAArAIAAP4CAAACAwAAAwMAALYCAAC2AgAAtAIAAAIDAAAGAwAABwMAAL4CAAC+AgAAvAIAAAYDAAAKAwAACwMAAMYCAADGAgAAxAIAAAoDAAAMAwAA/QIAAMoCAADKAgAAyAIAAAwDAAAIAwAACQMAAMICAADCAgAAwAIAAAgDAAAEAwAABQMAALoCAAC6AgAAuAIAAAQDAAAAAwAAAQMAALICAACyAgAAsAIAAAADAAD9AgAA/gIAAKwCAACsAgAAygIAAP0CAAABAwAAAgMAALQCAAC0AgAAsgIAAAEDAAAFAwAABgMAALwCAAC8AgAAugIAAAUDAAAJAwAACgMAAMQCAADEAgAAwgIAAAkDAADrAwAA7AMAAIoDAACKAwAA6QMAAOsDAAATAwAAEQMAABQDAAAPAwAAEAMAABIDAAASAwAAhQMAAA8DAAAcAwAAKQMAABoDAAAaAwAAGwMAABwDAAATAwAAFAMAABUDAAAVAwAAFgMAABMDAAAOAwAADQMAABADAAAQAwAADwMAAA4DAAAXAwAAgQMAABkDAAAXAwAAGQMAABsDAAAXAwAAGwMAABoDAADpAwAAigMAAOoDAADqAwAAiQMAAOkDAAAdAwAAHgMAAC4DAAAuAwAALAMAAB0DAACBAwAAFwMAABYDAACBAwAAFgMAABUDAACBAwAAFQMAABgDAACFAwAAEgMAAIYDAACGAwAAEQMAAIUDAAAdAwAAHwMAAB4DAAAsAwAALgMAACkDAAApAwAAHAMAACwDAAAUAwAAEQMAAIYDAADnAwAAiQMAAOoDAAANAwAADgMAAOwDAADsAwAA6wMAAA0DAADQAwAAiwMAAKEDAADaAwAA5wMAAOYDAADmAwAA2QMAANoDAADZAwAAywMAAMoDAADKAwAA2gMAANkDAACIAwAAxwMAAMUDAADFAwAA2wMAAIgDAADDAwAA1wMAANYDAADWAwAAwAMAAMMDAAC8AwAAwwMAAMADAADAAwAAuwMAALwDAAC9AwAAugMAALYDAAC2AwAAtQMAAL0DAADIAwAAxgMAALADAACwAwAArwMAAMgDAACvAwAAqwMAAKoDAACqAwAArgMAAK8DAACnAwAAswMAALEDAACxAwAApgMAAKcDAAC/AwAA1QMAANADAACjAwAAoQMAAGIDAABiAwAAYAMAAKMDAAC6AwAAnQMAAJwDAACcAwAAtgMAALoDAACYAwAApQMAAJkDAACZAwAAlwMAAJgDAACWAwAAlAMAAJMDAACTAwAAmAMAAJYDAABzAwAAdwMAAJIDAACSAwAAlAMAAHMDAADcAwAAxAMAAHoDAAB6AwAAIQMAANwDAADaAwAAxwMAAIgDAAAjAwAAhwMAADYDAAB4AwAAUAMAADgDAAA4AwAAOQMAAHgDAAB4AwAAdgMAAFQDAABUAwAAUAMAAHgDAAB1AwAAbQMAAFgDAABYAwAAWQMAAHUDAABtAwAAbwMAAHADAABwAwAAbgMAAG0DAABpAwAARwMAAEwDAABMAwAAagMAAGkDAABeAwAAZAMAAGUDAABlAwAAYQMAAF4DAABfAwAAXQMAAGADAABgAwAAYgMAAF8DAAA+AwAAXAMAAC0DAAAtAwAAKAMAAD4DAABWAwAAUgMAAFMDAABTAwAAWQMAAFYDAABRAwAAUgMAAFYDAABWAwAAVQMAAFEDAAA1AwAANgMAAE8DAABPAwAATgMAADUDAABKAwAASAMAAEkDAABJAwAASwMAAEoDAABCAwAARAMAAEYDAABGAwAARQMAAEIDAAA8AwAAPwMAAEADAABAAwAAOwMAADwDAAA6AwAAJgMAACoDAAAqAwAAOwMAADoDAAAzAwAAIwMAADYDAAA2AwAANQMAADMDAAAVAwAAJAMAACUDAAAlAwAAGAMAABUDAAAQAwAADQMAACEDAAAhAwAAIAMAABADAAAUAwAAIwMAACQDAAAkAwAAFQMAABQDAABcAwAAhAMAAC0DAAAxAwAAMgMAADMDAAAzAwAANAMAADEDAAA4AwAAIgMAACADAAAgAwAAOQMAADgDAAAnAwAAPQMAAD4DAAA+AwAAKAMAACcDAABBAwAAQwMAAEQDAABEAwAAQgMAAEEDAABIAwAAQAMAAD8DAAA/AwAASQMAAEgDAABPAwAANgMAADgDAAA4AwAAUAMAAE8DAABTAwAATwMAAFADAABQAwAAVAMAAFMDAABXAwAAVgMAAFkDAABZAwAAWAMAAFcDAABFAwAAWgMAAFwDAABcAwAAPgMAAEUDAABdAwAAXgMAAGEDAABhAwAAYAMAAF0DAABbAwAAYwMAAGQDAABkAwAAXgMAAFsDAABoAwAAQwMAAEcDAABHAwAAaQMAAGgDAABYAwAAbQMAAG4DAABuAwAAbAMAAFgDAAB0AwAAbwMAAG0DAABtAwAAdQMAAHQDAAB3AwAAcwMAAHQDAAB0AwAAdgMAAHcDAAA5AwAAIAMAACEDAAAhAwAAegMAADkDAAAlAwAAgAMAAIEDAACBAwAAGAMAACUDAAAjAwAAFAMAAIYDAACGAwAAhwMAACMDAADnAwAA2gMAAIgDAACIAwAAiQMAAOcDAACNAwAA2AMAAOMDAADjAwAAjAMAAI0DAACOAwAAzwMAANQDAADUAwAAjwMAAI4DAACQAwAA4QMAAOQDAADEAwAAkQMAAHkDAAB5AwAAegMAAMQDAACUAwAAkgMAAKkDAACpAwAAkwMAAJQDAABxAwAAcwMAAJQDAACUAwAAlgMAAHEDAACmAwAAmgMAAJkDAACZAwAApQMAAKYDAAC5AwAAngMAAJ0DAACdAwAAugMAALkDAACiAwAAowMAAGADAABgAwAAYQMAAKIDAACsAwAAsAMAAMYDAACsAwAAxgMAAMkDAACsAwAAyQMAAMwDAACsAwAAzAMAAM0DAACsAwAAzQMAAM4DAACsAwAAzgMAAMEDAACsAwAAwQMAAL4DAACsAwAAvgMAALQDAACsAwAAtAMAALIDAACsAwAAsgMAAKgDAACnAwAAqAMAALIDAACyAwAAswMAAKcDAACvAwAAsAMAAKwDAACsAwAAqwMAAK8DAAC2AwAAsQMAALMDAACzAwAAtQMAALYDAAC7AwAAwAMAAL8DAAC/AwAAuAMAALsDAAC9AwAAwgMAAMMDAADDAwAAvAMAAL0DAADTAwAAwgMAAMEDAADBAwAAzgMAANMDAADKAwAAyQMAAMYDAADGAwAAyAMAAMoDAADYAwAA0gMAAMsDAADLAwAA2QMAANgDAADXAwAA0wMAANIDAADSAwAA2AMAANcDAADbAwAA6QMAAIkDAACJAwAAiAMAANsDAADQAwAAoQMAAL8DAABiAwAAoQMAAIsDAACLAwAA7gMAAGIDAAAdAwAAhAMAAIMDAACCAwAA3QMAAO0DAADtAwAAHwMAAIIDAAANAwAA6wMAANwDAADcAwAAIQMAAA0DAADZAwAA5gMAAOMDAADjAwAA2AMAANkDAADSAwAA0wMAAM4DAADOAwAAzQMAANIDAADLAwAAzAMAAMkDAADJAwAAygMAAMsDAADbAwAAxQMAAMQDAADEAwAA3AMAANsDAADAAwAA1gMAANUDAADVAwAAvwMAAMADAAC6AwAAvAMAALsDAAC7AwAAuQMAALoDAAC9AwAAtQMAALQDAAC0AwAAvgMAAL0DAADHAwAArgMAAK0DAACtAwAAxQMAAMcDAACuAwAAqgMAAKkDAACpAwAArQMAAK4DAACnAwAApgMAAKUDAAClAwAApAMAAKcDAAC3AwAAuAMAAKMDAACjAwAAogMAALcDAACgAwAAogMAAGEDAABhAwAAZQMAAKADAAC2AwAAnAMAAJsDAACbAwAAsQMAALYDAACWAwAAmAMAAJcDAACXAwAAlQMAAJYDAACYAwAAkwMAAKQDAACkAwAApQMAAJgDAACSAwAAkQMAAK0DAACtAwAAqQMAAJIDAACtAwAAkQMAAMQDAADEAwAAxQMAAK0DAADhAwAAkAMAAI8DAACPAwAA1AMAAOEDAACgAwAAnwMAALcDAAC3AwAAogMAAKADAACNAwAA1wMAANgDAABBAwAAQAMAAEMDAACAAwAAJQMAACYDAABmAwAAYwMAAFsDAABbAwAARgMAAGYDAAB5AwAAdwMAAHYDAAB2AwAAeAMAAHkDAABUAwAAdQMAAFkDAABZAwAAUwMAAFQDAABvAwAAcQMAAHIDAAByAwAAcAMAAG8DAABqAwAATAMAAFcDAABXAwAAawMAAGoDAABmAwAARgMAAEQDAABEAwAAZwMAAGYDAABcAwAAXwMAAIMDAACDAwAAhAMAAFwDAABaAwAAWwMAAF4DAABeAwAAXQMAAFoDAABJAwAAPwMAADwDAABJAwAAPAMAAC8DAABJAwAALwMAADADAABJAwAAMAMAADEDAABJAwAAMQMAADQDAABJAwAANAMAADcDAABJAwAANwMAAE0DAABJAwAATQMAAFEDAABJAwAAUQMAAFUDAABJAwAAVQMAAEsDAABVAwAAVgMAAEoDAABKAwAASwMAAFUDAABNAwAATgMAAFIDAABSAwAAUQMAAE0DAABMAwAARwMAAEgDAABIAwAASgMAAEwDAAA9AwAAQgMAAEUDAABFAwAAPgMAAD0DAAA7AwAAQAMAAEEDAABBAwAAOgMAADsDAAA7AwAAKgMAAC8DAAAvAwAAPAMAADsDAAA0AwAAMwMAADUDAAA1AwAANwMAADQDAAAwAwAAKwMAADIDAAAyAwAAMQMAADADAAAqAwAAJgMAACUDAAAlAwAAKwMAACoDAAAQAwAAIAMAACIDAAAiAwAAEgMAABADAAASAwAAIgMAAIcDAACHAwAAhgMAABIDAAAqAwAAKwMAADADAAAwAwAALwMAACoDAAA2AwAAhwMAACIDAAAiAwAAOAMAADYDAAAmAwAAOgMAAD0DAAA9AwAAJwMAACYDAAA6AwAAQQMAAEIDAABCAwAAPQMAADoDAABDAwAAQAMAAEgDAABIAwAARwMAAEMDAAA3AwAANQMAAE4DAABOAwAATQMAADcDAABSAwAATgMAAE8DAABPAwAAUwMAAFIDAABKAwAAVgMAAFcDAABXAwAATAMAAEoDAABFAwAARgMAAFsDAABbAwAAWgMAAEUDAABaAwAAXQMAAF8DAABfAwAAXAMAAFoDAAAfAwAAHQMAAIMDAACDAwAAggMAAB8DAABnAwAARAMAAEMDAABDAwAAaAMAAGcDAABrAwAAVwMAAFgDAABYAwAAbAMAAGsDAABzAwAAcQMAAG8DAABvAwAAdAMAAHMDAAB2AwAAdAMAAHUDAAB1AwAAVAMAAHYDAAB4AwAAOQMAAHoDAAB6AwAAeQMAAHgDAABfAwAAYgMAAO4DAADuAwAAgwMAAF8DAAC8AwAAugMAAL0DAADtAwAA3QMAAM8DAADPAwAAjgMAAO0DAAB3AwAAeQMAAJEDAACRAwAAkgMAAHcDAACTAwAAqQMAAKoDAACqAwAApAMAAJMDAABxAwAAlgMAAJUDAACVAwAAcgMAAHEDAACxAwAAmwMAAJoDAACaAwAApgMAALEDAAC3AwAAnwMAAJ4DAACeAwAAuQMAALcDAACjAwAAuAMAAL8DAAC/AwAAoQMAAKMDAACrAwAApwMAAKQDAACkAwAAqgMAAKsDAACrAwAArAMAAKgDAACoAwAApwMAAKsDAADHAwAAyAMAAK8DAACvAwAArgMAAMcDAAC1AwAAswMAALIDAACyAwAAtAMAALUDAAC5AwAAuwMAALgDAAC4AwAAtwMAALkDAAC+AwAAwQMAAMIDAADCAwAAvQMAAL4DAADXAwAAwwMAAMIDAADCAwAA0wMAANcDAADaAwAAygMAAMgDAADIAwAAxwMAANoDAADSAwAAzQMAAMwDAADMAwAAywMAANIDAADcAwAA6wMAAOkDAADpAwAA2wMAANwDAACAAwAAJgMAABkDAAAZAwAAgQMAAIADAAAZAwAAJgMAACcDAAAnAwAAGwMAABkDAAAdAwAALAMAAC0DAAAtAwAAhAMAAB0DAAAbAwAAJwMAACgDAAAoAwAAHAMAABsDAAAcAwAAKAMAAC0DAAAtAwAALAMAABwDAAArAwAAJQMAACQDAAAkAwAAMgMAACsDAAAyAwAAJAMAACMDAAAjAwAAMwMAADIDAAAeAwAAfQMAAH8DAAB/AwAALgMAAB4DAAB7AwAAFwMAABoDAAAuAwAAfwMAAHwDAAB8AwAAKQMAAC4DAAB+AwAAHwMAAO0DAAAfAwAAfgMAAH0DAAB9AwAAHgMAAB8DAAB7AwAAGgMAACkDAAApAwAAfAMAAHsDAABjAwAAZgMAAGcDAABjAwAAZwMAAGgDAABjAwAAaAMAAGkDAABjAwAAaQMAAGoDAABjAwAAagMAAGsDAABjAwAAawMAAGwDAABjAwAAbAMAAG4DAABjAwAAbgMAAHADAABjAwAAcAMAAHIDAABjAwAAcgMAAJUDAABjAwAAlQMAAJcDAABjAwAAlwMAAJkDAABjAwAAmQMAAJoDAABjAwAAmgMAAJsDAABjAwAAmwMAAJwDAABjAwAAnAMAAJ0DAABjAwAAnQMAAJ4DAABjAwAAngMAAJ8DAABjAwAAnwMAAKADAABjAwAAoAMAAGUDAABjAwAAZQMAAGQDAACLAwAA3gMAAO4DAADlAwAA5AMAAIwDAADlAwAAjAMAAOMDAADlAwAA4wMAAOYDAADXAwAA4gMAAOADAADgAwAA1gMAANcDAADVAwAA3wMAANEDAADRAwAA0AMAANUDAADXAwAAjQMAAIwDAACMAwAA4gMAANcDAADoAwAA5wMAAOoDAADPAwAA0QMAAN8DAADfAwAA1AMAAM8DAADnAwAA6AMAAOUDAADlAwAA5gMAAOcDAADWAwAA4AMAAN8DAADfAwAA1QMAANYDAADRAwAA3gMAAIsDAACLAwAA0AMAANEDAADuAwAA3gMAAIIDAADdAwAA3gMAANEDAADRAwAAzwMAAN0DAADUAwAA3wMAAOADAADgAwAA4QMAANQDAADiAwAAjAMAAOQDAADiAwAA5AMAAOEDAADiAwAA4QMAAOADAADeAwAA3QMAAIIDAACCAwAAgwMAAO4DAABMBAAASwQAAEoEAABMBAAASgQAAEkEAABMBAAASQQAAEgEAABMBAAASAQAAEcEAABMBAAARwQAAEYEAABMBAAARgQAAEUEAABMBAAARQQAAEQEAABMBAAARAQAAEMEAABMBAAAQwQAAEIEAABMBAAAQgQAAEEEAABMBAAAQQQAAEAEAABMBAAAQAQAAD8EAABMBAAAPwQAAE4EAABMBAAATgQAAE0EAAA8BAAAOwQAADoEAAA8BAAAOgQAADkEAAA8BAAAOQQAADgEAAA8BAAAOAQAADcEAAA8BAAANwQAADYEAAA8BAAANgQAADUEAAA8BAAANQQAADQEAAA8BAAANAQAADMEAAA8BAAAMwQAADIEAAA8BAAAMgQAADEEAAA8BAAAMQQAADAEAAA8BAAAMAQAAC8EAAA8BAAALwQAAD4EAAA8BAAAPgQAAD0EAAAMBAAADgQAAFwEAABcBAAAXQQAAAwEAAD0AwAA9gMAAFQEAABUBAAAVQQAAPQDAAD8AwAA/gMAAFEEAABRBAAAUAQAAPwDAAAEBAAABgQAAFkEAABZBAAAWAQAAAQEAABMBAAATQQAACoEAAAqBAAAKQQAAEwEAABIBAAASQQAACYEAAAmBAAAJQQAAEgEAABEBAAARQQAACIEAAAiBAAAIQQAAEQEAABABAAAQQQAAC4EAAAuBAAALQQAAEAEAAA8BAAAPQQAABoEAAAaBAAAGQQAADwEAAA4BAAAOQQAABYEAAAWBAAAFQQAADgEAAA0BAAANQQAABIEAAASBAAAEQQAADQEAAAwBAAAMQQAAB4EAAAeBAAAHQQAADAEAAAsBAAALQQAAAoEAAAKBAAACAQAACwEAAAoBAAAKQQAAAIEAAACBAAAAAQAACgEAAAkBAAAJQQAAPoDAAD6AwAA+AMAACQEAAAgBAAAIQQAAPIDAADyAwAA8AMAACAEAAAcBAAAHQQAAPUDAAD1AwAA9wMAABwEAAAYBAAAGQQAAP0DAAD9AwAA/wMAABgEAAAUBAAAFQQAAAUEAAAFBAAABwQAABQEAAAQBAAAEQQAAA0EAAANBAAA7wMAABAEAABeBAAACQQAAAcEAAAHBAAAWgQAAF4EAABXBAAAAQQAAP8DAAD/AwAATwQAAFcEAABSBAAA+QMAAPcDAAD3AwAAUwQAAFIEAABbBAAA8QMAAO8DAADvAwAAVgQAAFsEAABTBAAA9wMAAPUDAAD1AwAAVAQAAFMEAABPBAAA/wMAAP0DAAD9AwAAUQQAAE8EAABaBAAABwQAAAUEAAAFBAAAWQQAAFoEAABWBAAA7wMAAA0EAAANBAAAXAQAAFYEAAARBAAAEgQAAAsEAAALBAAADQQAABEEAAAVBAAAFgQAAAMEAAADBAAABQQAABUEAAAZBAAAGgQAAPsDAAD7AwAA/QMAABkEAAAdBAAAHgQAAPMDAADzAwAA9QMAAB0EAAAhBAAAIgQAAPQDAAD0AwAA8gMAACEEAAAlBAAAJgQAAPwDAAD8AwAA+gMAACUEAAApBAAAKgQAAAQEAAAEBAAAAgQAACkEAAAtBAAALgQAAAwEAAAMBAAACgQAAC0EAAAxBAAAMgQAAA8EAAAPBAAAHgQAADEEAAA1BAAANgQAABMEAAATBAAAEgQAADUEAAA5BAAAOgQAABcEAAAXBAAAFgQAADkEAAA9BAAAPgQAABsEAAAbBAAAGgQAAD0EAABBBAAAQgQAAB8EAAAfBAAALgQAAEEEAABFBAAARgQAACMEAAAjBAAAIgQAAEUEAABJBAAASgQAACcEAAAnBAAAJgQAAEkEAABNBAAATgQAACsEAAArBAAAKgQAAE0EAAACBAAABAQAAFgEAABYBAAAVwQAAAIEAAD6AwAA/AMAAFAEAABQBAAAUgQAAPoDAADyAwAA9AMAAFUEAABVBAAAWwQAAPIDAAAKBAAADAQAAF0EAABdBAAAXgQAAAoEAAAOBAAA8AMAAFYEAABWBAAAXAQAAA4EAAD2AwAA+AMAAFMEAABTBAAAVAQAAPYDAAD+AwAAAAQAAE8EAABPBAAAUQQAAP4DAAAGBAAACAQAAFoEAABaBAAAWQQAAAYEAABLBAAATAQAACkEAAApBAAAKAQAAEsEAABHBAAASAQAACUEAAAlBAAAJAQAAEcEAABDBAAARAQAACEEAAAhBAAAIAQAAEMEAAA/BAAAQAQAAC0EAAAtBAAALAQAAD8EAAA7BAAAPAQAABkEAAAZBAAAGAQAADsEAAA3BAAAOAQAABUEAAAVBAAAFAQAADcEAAAzBAAANAQAABEEAAARBAAAEAQAADMEAAAvBAAAMAQAAB0EAAAdBAAAHAQAAC8EAAArBAAALAQAAAgEAAAIBAAABgQAACsEAAAnBAAAKAQAAAAEAAAABAAA/gMAACcEAAAjBAAAJAQAAPgDAAD4AwAA9gMAACMEAAAfBAAAIAQAAPADAADwAwAADgQAAB8EAAAbBAAAHAQAAPcDAAD3AwAA+QMAABsEAAAXBAAAGAQAAP8DAAD/AwAAAQQAABcEAAATBAAAFAQAAAcEAAAHBAAACQQAABMEAAAPBAAAEAQAAO8DAADvAwAA8QMAAA8EAABdBAAACwQAAAkEAAAJBAAAXgQAAF0EAABYBAAAAwQAAAEEAAABBAAAVwQAAFgEAABQBAAA+wMAAPkDAAD5AwAAUgQAAFAEAABVBAAA8wMAAPEDAADxAwAAWwQAAFUEAABUBAAA9QMAAPMDAADzAwAAVQQAAFQEAABRBAAA/QMAAPsDAAD7AwAAUAQAAFEEAABZBAAABQQAAAMEAAADBAAAWAQAAFkEAABcBAAADQQAAAsEAAALBAAAXQQAAFwEAAASBAAAEwQAAAkEAAAJBAAACwQAABIEAAAWBAAAFwQAAAEEAAABBAAAAwQAABYEAAAaBAAAGwQAAPkDAAD5AwAA+wMAABoEAAAeBAAADwQAAPEDAADxAwAA8wMAAB4EAAAiBAAAIwQAAPYDAAD2AwAA9AMAACIEAAAmBAAAJwQAAP4DAAD+AwAA/AMAACYEAAAqBAAAKwQAAAYEAAAGBAAABAQAACoEAAAuBAAAHwQAAA4EAAAOBAAADAQAAC4EAAAyBAAAMwQAABAEAAAQBAAADwQAADIEAAA2BAAANwQAABQEAAAUBAAAEwQAADYEAAA6BAAAOwQAABgEAAAYBAAAFwQAADoEAAA+BAAALwQAABwEAAAcBAAAGwQAAD4EAABCBAAAQwQAACAEAAAgBAAAHwQAAEIEAABGBAAARwQAACQEAAAkBAAAIwQAAEYEAABKBAAASwQAACgEAAAoBAAAJwQAAEoEAABOBAAAPwQAACwEAAAsBAAAKwQAAE4EAAAABAAAAgQAAFcEAABXBAAATwQAAAAEAAD4AwAA+gMAAFIEAABSBAAAUwQAAPgDAADwAwAA8gMAAFsEAABbBAAAVgQAAPADAAAIBAAACgQAAF4EAABeBAAAWgQAAAgEAABnKZE/jbgEwKw5AD595ow/Nc81wILIgj1VhI8/1EYFwEAYkD7u6ok/8Uk2wIejOz5cWYY/DAQGwOpe1z6tNIE/kuw2wCbllj6PG24/Z9MGwMaLBT93Lmg/Xp43wLg7wz5K60c/T5UHwN6wET9wI0k/OUQ4wLEV3D5e8h8/QSwIwKJFDj8yAio/48Q4wC6q3T7jp/k+a4AIwFp++D4BiA8/0Aw5wNi7xz7Mz8U+aoYIwOATuz50efs+AhE5wEWhnT6zKKw+pz0IwBpvWz6bq+4+5dA4wPSGSz7YvLI+XK8HwBrhbT23mfo+KlY4wFyPoj3gaNc+H/IGwCgrpr1vuQ4/hbM3wEp7A72pSwo/ySIGwJSGOr5R9Cg/vQE3wHYY873uezA/4WAFwLEaa75Y/0c/5ls2wC1AK77adFg/88kEwH9tXb6GIGc/L9s1wORoLr6SIHw/73QEwO6TE75kzYA/R5M1wHqMAr5wCIs/0m4EwHBcxrwHs4k/EY81wE9dOb27t4w/hPX3v9L8kT4BT44/M+L2v+Y+CT5bYYg/oFP2v01LLLy5q3c/dF/2v26IAb7YKlU/2gP3v5j3SL7QgC4/4Sf4v/kxVr5tkAk/Ap/5v3I0J77l8tc+HzD7v7ZMhr06erQ+I578v1d6jT0hHa4+dbH9v+p3YT4fLsE+cGD+vx/yvj6R7fQ+cVT+v3hB/D66gR4/xY/9v5rrDD/DK0U/xmv8vzI6ED8mHGo/rvT6v9F6BD+Q2YM/iGP5v3Tu1j7BV0A/VObsvzxL8D4XuVs/StDrv/Tg3j70p3E//Kbqv+/LuT5bzX4/mpfpvz6xhj6SlIA/jsvov9WxGj5KX3g/4WHov8zsMz1cxmU/oWrovzs0LL0eM0w/huTovwAAwL1dii8/9bzpvw+c0732KBQ//tLqv3Pzjb1UdPw+VPzrv841TDtjKeI+pgvtv+rL0j3Scd0+utftv98WXD5JocQ+tJHzv4Lmuz5KX/g+AYbzvzwz+T7/riM/wr7tvzlk6z4YWhU/ZW3jv4SAhD7r/R4/12jjv3pRmz6zPyw/qinjv5oKqT6xGjs/gLnivxyVqz4BTEk/Winivw2Ooj52qlQ/P4/hv3pVjz72els/kwLhv+KvaT72s1w/1Jjgv9o8Lj6vJVg/C2Lgv9i86j3cgU4/kGbgv/94jz0UQEE/vqXgvwIpMT0XZTI/8BXhv+3UHD3XMyQ/Dabhv2sNZT1S1Rg/KUDivwFpvz3iBBI/3sziv5KvFD7iyxA/nDbjv1ciUD7b3wE/xOjpv58E3j7zPOA+BvHpv7b1sz56qI0+FJKmvyL+6T5enyE/66r6v/xvFT+Z9Dc+UU68v4rl1j5RSgA/Y9IBwHFzBj8nofQ9zZDQv3dIoT7sFtE+2VsGwJ6Y1T5pj9c99kPgvzChIj65ALw+EEEKwCBgjT42kA4+4gPpv4Fdzby6h8Q+NuoMwMPx/D3QXl0+nnvpv+upVb7MX+k+ke8NwL/Rjrz8GaY+2Zjhv9Fdur5TdhI/WikNwPc/AL7nF+U+947Sv0xU777UDjc/zLUKwAzqO74YCRE/Gqi+v0XUAL8U51w/WvQGwPzCO75Q4Ck/5uuov/OP7r4hPH4/cHcCwK2i/71SSTk/aamUv+DyuL5+/Yo/9dv7vyegibyJ6zw/QPaEv3v1Ub4DQ5A/iBH0v+J4/j1HOTQ/qWx4vw5JrbxCIY4/RL/uv+jBjT6yhSA/QX13v59VJj5G64Q/hbTsv1Pr1T5o0AQ/XqGDv4mzoj4OEGw/4UDuvyqPBj/koso+NquSvwSq1z6fd0c/Dyjzv8B5FT8oLLE9n6quv/yljT71Sik+GQOdv9AonT7+LIY+y9iMv6JFjj5pV7k+tqGAv2eBRj5XWuY+73N0vz2aij31LQM/cjZ1v5bLpr0YPgo/mraBv+JXbL6ZSgc/LXeOv4Ygt75tjPU+5uuev5es4r79Mc0+fZOwv2ov8r5Yqps+y73AvzxM474dAFE+2PTMv05HuL758+09jlzTv1Nab7523Vs9XfvSv9NOrb1PstU89N/Lv+GVhD08Exo9aB+/v9czRD7xulI/r0EDwGU5DT+T/TM/nE4GwHcxDT+e6hg/z/MJwDoFAT8SoQU/OKMNwFQe1T5kIfo+Mc0QwCV1mj5VhwA/c/YSwN/5NT7pfQ8/wcoTwEF/gT1LrSc/wykTwHxgx7zPZkU//isRwBnFkr0LJGQ/FR8OwC+Gkr0iN38/4XkKwBGPxLxGQIk/dcoGwL2Mgj12iI0/e6ADwP6YNj4tzYs/NncBwHLEmj7sUYQ/7KIAwLNh1T51dHA/6UMBwLUbAT9MbWk9fGS1v2QEBL4HXi09mZ2xv7/wSr2bdUY9odaqv8lykjzzcpg9wheivwovgT1HO+493bWYvxMooj25wis+kx6Qv0GCgj3uJGI+tqCJv+I8nDxI/og+RzmGv9KJRL2fAJo+7GyGv15JAr6qgqE+1zOKv9WRU76WX54+zvqQvy9OjL6uEZE+prmZv8Vyo74xP3c+ixujvwexq74cmkI+3bKrv5PHo76kNww+wjCyv9HqjL6LwLg9KZi1v04rVb6i1X0/mKELwHAlmz7j4XU/xjALwP1Luj6FB2k/WYYLwNEf0j5tO1k/QpUMwIoA3z5P5Ug/SDQOwCf43j7cgTo/OiQQwPYH0j7MQTA/mBkSwDUouj6ztCs/DMgTwF37mj7Wiy0/E+4UwK6Acj6VfzU/6l4VwNczND7zWUI/WwkVwOuLBD4cJlI/cvoTwPKU1T0ofGI/bFsSwIC21T2t33A/f2sQwKG7BD6sH3s/HXYOwCR7ND7WrH8/pMcMwBfVcj6lLkE/swzjv18JtL4mGuQ+9Iv4vzSAh753MRk/v5rvvyi6rr7KbaM+3IP8vyqoCL79bHQ+xef6v1Z/BD2fdV0+ZvbzvxN/VD71uYI+Wb7ovxYTuz7np/g+aLPMv6ETCj8t7mM/MsvUvwaflr6vJXw/wQHHv6vqNb6pEoM/tMm7v3Iaory1M4A/Zti0v6wbHz4cKbM+8fTav8e89j5EUSQ/WyXAv/RrBz+4dUs/LzS3v/Wd5z7my2s/Rzyzv9ZxpD7Jxs8/KSNuveZ3Gr4R4/U/OncPv3Dqsz4wn84/WTQtvtbghb4npfY/t9Ukv6SOhj5b6tQ/ICWGvkqYwb7w/Pw//5c3v5gwKj4gs+E/ayilvrpK976g+QNA8+JEv1w8nD06B/M/T+uuvstlD7907gpAZLBKv+I/3TufIQNAa/GhvnQLGb/GTRJARx5Iv35RAr1uFwxA9fGAvtVAF79R+BhA0ZA9v4hnCb35SxNAtCEfvtEICr9C6h1AL6Msv9c07zoAqRdAB+wqvTfh5r7hYiBAQukXv7whjT3RPBhAGuGNPTI8rr7WASBAtYoCv3dIIT48FxVAdQYmPsAJZb7x1RxA+5DfvpMagj7Zsg5ACw1kPr9J873JWhdAEvvEvsgjsD7MCAZA0ZJ3PpyMKr31ZRBAMWC5vuC90z6N1fg/x55dPo6QgbujBglASYS+vg995z4wn+Y/zCQaPqMBPLwYXAJAVp/TvtBf6D5VNNg/vahdPY0kgb1O1Po/eHr1vqpD1j426co/wxGEvTqRuL41B8w/R6ovPfPGgb4bLtQ/bRobPuWYLL42IOI/+pl6PvcC8735vfM/Y++dPj8c5L1zLANAnnyqPsxhF77rjQtAhQuhPprNY77DvBFAJAyDPmfXpb78xxRAxR4qPo2a374AORRAR61wPXsyC7+hoBBAgIFgvdgRI796bwlASrYavjlEML+aXQBAJclTvlySL79HIO4/m+NsvmA9Jr9fXd0/aQFavn0iE7+w/9A/ZAIevmFU8r5xAO0/D9OevT6uKb/Zk+A/kdSCveyEG79madc/DY2nvAJFCL8y5tI/ees8PUG55b4sutM/6boAPlwcvb4Nxdk/TiVTPirinL5UG+Q/ZveMPpnyib4gKvE/5SelPtwvh76V9P4/m3WuPksFlb6SsAVA+3WnPu9Xsb7QRQpAqDmRPuTX175qhwxA0EZePlRUAb9tHQxACMcMPraiFb+neA5AOZsOPDyiLr/DRwdALlOzvbPTO7/eyvo/ZTdzvfaYML8f9gBAVisTPoJVHb+KkPw/6PTcPeY9Ir/Fy/U/GM2qPRL1Ir/Ope4/TYKXPS1fH78mNeg/aAWmPf8HGL/NdOM/px/UPZkNDr8IHuE/a2YNPs/0Ar/ni+E/6KQ3Pnfc8L68reQ/jV1iPvon4L5wCeo/e4eDPlRX1r42zvA/TRGQPvvo1L4s9Pc/3uOUPoMU3L7UZP4/GEORPgHD6r6OkgFAqbyFPsy3/r75vQJAeSJoPp90Cr8JhwJA/OM9PkT7FL9d3QRAOJ5Pukn3O7+4zAlAaqSFPTHqMr/er/4/tmb7vmOayT7eVPo/e/cPv+NUqz7j//o/o8wiv5Jbgz53SwBA81Qzv3aILz4UIAVA8gs/v0zHvD2BQQtAFylEv5wZ/TzPwBFAGeVBv+6zSrvLoBdATpg4v/tal7t2/BtAEK0pv8aH2Tz6KR5A8WgXv/J3rz141B1AyZMEv5uuJz7uCBtAEhfovkjdfj5QNBZAN6nQvi4BqD7oEhBAym7GvkZhxz6akwlAx/bKvmnI2D6eswNAXJDdvkuQ2T63Rac/0O6wPg4VYz4FMfw/oG6gu6r08z23Rac/decZPphOOz7YSfE/RdYKvgFotD23Rac/v9J5vJgTlD2RC+g/d0l8vqinj7m3Rac/+OEAvqn1vr163uE/uwukvlOzB763Rac/Kqgovl+4k75Ks98/81qxvl+4k763Rac/+OEAvlOz97563uE/uwukvvKW4763Rac/v9J5vNI6Jr+RC+g/d0l8vlmmE7+3Rac/decZPgWMQr/YSfE/RdYKvm9FKr+3Rac/0O6wPpF9TL8FMfw/oG6gu/Q2Mr+uRac/83QKPwWMQr8ejANAGM8APm9FKr+3Rac/G9Y0P9I6Jr9BKwhAB0JyPlmmE7+3Rac/TidRP1Oz977EQQtAAwifPvKW476uRac/7BhbP1+4k75kVwxAXFesPl+4k763Rac/TidRP6n1vr3EQQtAAwifPlOzB763Rac/G9Y0P5gTlD1BKwhAB0JyPqinj7muRac/83QKP5hOOz4ejANAGM8APgFotD3knpA/cXI/Pu9XwT3knpA/0O6wPt/6AD7knpA/RRIBP+9XwT3knpA/gH4jPw1slTvknpA/pn46P44FBb7knpA/YJJCP1+4k77knpA/pn46P9Xt5L7knpA/gH4jP0fjFL/knpA/RRIBP23jK7/knpA/0O6wPgb3M7/knpA/cXI/Pm3jK7/cnpA//wRXPUfjFL/knpA/Xf0YvdXt5L7knpA/+BuNvV+4k77knpA/Xf0YvY4FBb7cnpA//wRXPQ1slTuHGgpAehnFPMtmjjzvrAVAVB+ovahuLj1bPwFAQMJAvstmjjy9/Po/S3SOvhnJXr1/+PU/hj2tvv+wJb6DNfQ/Gw24vl+4k75/+PU/hj2tvhyY1L69/Po/S3SOvs3LBb9bPwFAQMJAvoQrGL/vrAVAVB+ovUmfHr+HGgpAehnFPIQrGL9/2w1ADJPpPc3LBb+iXRBA+1syPhyY1L4cPxFAJvtHPl+4k76iXRBA+1syPv+wJb5/2w1ADJPpPRnJXr2IL38/K08APl+4k76IL38/iXoRPpIfUb53L38/Y19CPoDyB76IL38/O8aFPsUarr2IL38/0O6wPgnEi72IL38/ZRfcPsUarr2IL38/91YAP4DyB76IL38/P5AMP5IfUb53L38/FtsQP1+4k76IL38/P5AMP/Tgvr6IL38/91YAP313476IL38/ZRfcPurp+76IL38/0O6wPt0/Ar+IL38/O8aFPurp+753L38/Y19CPn13476IL38/iXoRPvTgvr7XhRFASRPvO1+4k74GDhFAraWAu3x8Yr7DuA9AAfkSvXkHKL4Pug1Ak26rvUD4AL6aXwtAnnsPvqqB5r0tBQlANUBJvkD4AL5wBgdA/Dh6vnkHKL4tsQVA5niNvnx8Yr5YOQVAyjeTvl+4k74tsQVA5niNvn8ytr5wBgdA/Dh6vgFt074tBQlANUBJvp305r6aXwtAnnsPvjHQ7b4Pug1Ak26rvZ305r7DuA9AAfkSvQFt074GDhFAraWAu38ytr6brtM/j4taPpQRQ7+SsMM/+gnHvU+RH7/uBss/kKM5PZnXOb9+yb4/3QZFvix98L7LEL0/cERnvl+4k75+yb4/3QZFvrzN272SsMM/+gnHvQyPPT2brtM/j4taPthkPT5HVtw/HVfDPpnXOb+srOM/BycGP0+RH7+4k+g/f4cePyx98L5zTOo/0xYnP1+4k77uBss/kKM5PSx9GD5HVtw/HVfDPix9GD6srOM/BycGPwyPPT24k+g/f4ceP7zN271nKZG/jbgEwKw5AD595oy/Nc81wILIgj1VhI+/1EYFwEAYkD7u6om/8Uk2wIejOz5cWYa/DAQGwOpe1z6tNIG/kuw2wCbllj6PG26/Z9MGwMaLBT93Lmi/Xp43wLg7wz5K60e/T5UHwN6wET9wI0m/OUQ4wLEV3D5e8h+/QSwIwKJFDj8yAiq/48Q4wC6q3T7jp/m+a4AIwFp++D4BiA+/0Aw5wNi7xz7Mz8W+aoYIwOATuz50efu+AhE5wEWhnT6zKKy+pz0IwBpvWz6bq+6+5dA4wPSGSz7YvLK+XK8HwBrhbT23mfq+KlY4wFyPoj3gaNe+H/IGwCgrpr1vuQ6/hbM3wEp7A72pSwq/ySIGwJSGOr5R9Ci/vQE3wHYY873uezC/4WAFwLEaa75Y/0e/5ls2wC1AK77adFi/88kEwH9tXb6GIGe/L9s1wORoLr6SIHy/73QEwO6TE75kzYC/R5M1wHqMAr5wCIu/0m4EwHBcxrwHs4m/EY81wE9dOb27t4y/hPX3v9L8kT4BT46/M+L2v+Y+CT5bYYi/oFP2v01LLLy5q3e/dF/2v26IAb7YKlW/2gP3v5j3SL7QgC6/4Sf4v/kxVr5tkAm/Ap/5v3I0J77l8te+HzD7v7ZMhr06erS+I578v1d6jT0hHa6+dbH9v+p3YT4fLsG+cGD+vx/yvj6R7fS+cVT+v3hB/D66gR6/xY/9v5rrDD/DK0W/xmv8vzI6ED8mHGq/rvT6v9F6BD+Q2YO/iGP5v3Tu1j7BV0C/VObsvzxL8D4XuVu/StDrv/Tg3j70p3G//Kbqv+/LuT5bzX6/mpfpvz6xhj6SlIC/jsvov9WxGj5KX3i/4WHov8zsMz1cxmW/oWrovzs0LL0eM0y/huTovwAAwL1dii+/9bzpvw+c0732KBS//tLqv3Pzjb1UdPy+VPzrv841TDtjKeK+pgvtv+rL0j3Scd2+utftv98WXD5JocS+tJHzv4Lmuz5KX/i+AYbzvzwz+T7/riO/wr7tvzlk6z4YWhW/ZW3jv4SAhD7r/R6/12jjv3pRmz6zPyy/qinjv5oKqT6xGju/gLnivxyVqz4BTEm/Winivw2Ooj52qlS/P4/hv3pVjz72elu/kwLhv+KvaT72s1y/1Jjgv9o8Lj6vJVi/C2Lgv9i86j3cgU6/kGbgv/94jz0UQEG/vqXgvwIpMT0XZTK/8BXhv+3UHD3XMyS/Dabhv2sNZT1S1Ri/KUDivwFpvz3iBBK/3sziv5KvFD7iyxC/nDbjv1ciUD7b3wG/xOjpv58E3j7zPOC+BvHpv7b1sz56qI2+FJKmvyL+6T5enyG/66r6v/xvFT+Z9De+UU68v4rl1j5RSgC/Y9IBwHFzBj8nofS9zZDQv3dIoT7sFtG+2VsGwJ6Y1T5pj9e99kPgvzChIj65ALy+EEEKwCBgjT42kA6+4gPpv4Fdzby6h8S+NuoMwMPx/D3QXl2+nnvpv+upVb7MX+m+ke8NwL/Rjrz8Gaa+2Zjhv9Fdur5TdhK/WikNwPc/AL7nF+W+947Sv0xU777UDje/zLUKwAzqO74YCRG/Gqi+v0XUAL8U51y/WvQGwPzCO75Q4Cm/5uuov/OP7r4hPH6/cHcCwK2i/71SSTm/aamUv+DyuL5+/Yq/9dv7vyegibyJ6zy/QPaEv3v1Ub4DQ5C/iBH0v+J4/j1HOTS/qWx4vw5JrbxCIY6/RL/uv+jBjT6yhSC/QX13v59VJj5G64S/hbTsv1Pr1T5o0AS/XqGDv4mzoj4OEGy/4UDuvyqPBj/kosq+NquSvwSq1z6fd0e/Dyjzv8B5FT8oLLG9n6quv/yljT71Sim+GQOdv9AonT7+LIa+y9iMv6JFjj5pV7m+tqGAv2eBRj5XWua+73N0vz2aij31LQO/cjZ1v5bLpr0YPgq/mraBv+JXbL6ZSge/LXeOv4Ygt75tjPW+5uuev5es4r79Mc2+fZOwv2ov8r5Yqpu+y73AvzxM474dAFG+2PTMv05HuL758+29jlzTv1Nab7523Vu9XfvSv9NOrb1PstW89N/Lv+GVhD08Exq9aB+/v9czRD7xulK/r0EDwGU5DT+T/TO/nE4GwHcxDT+e6hi/z/MJwDoFAT8SoQW/OKMNwFQe1T5kIfq+Mc0QwCV1mj5VhwC/c/YSwN/5NT7pfQ+/wcoTwEF/gT1LrSe/wykTwHxgx7zPZkW//isRwBnFkr0LJGS/FR8OwC+Gkr0iN3+/4XkKwBGPxLxGQIm/dcoGwL2Mgj12iI2/e6ADwP6YNj4tzYu/NncBwHLEmj7sUYS/7KIAwLNh1T51dHC/6UMBwLUbAT9MbWm9fGS1v2QEBL4HXi29mZ2xv7/wSr2bdUa9odaqv8lykjzzcpi9wheivwovgT1HO+693bWYvxMooj25wiu+kx6Qv0GCgj3uJGK+tqCJv+I8nDxI/oi+RzmGv9KJRL2fAJq+7GyGv15JAr6qgqG+1zOKv9WRU76WX56+zvqQvy9OjL6uEZG+prmZv8Vyo74xP3e+ixujvwexq74cmkK+3bKrv5PHo76kNwy+wjCyv9HqjL6LwLi9KZi1v04rVb6i1X2/mKELwHAlmz7j4XW/xjALwP1Luj6FB2m/WYYLwNEf0j5tO1m/QpUMwIoA3z5P5Ui/SDQOwCf43j7cgTq/OiQQwPYH0j7MQTC/mBkSwDUouj6ztCu/DMgTwF37mj7Wiy2/E+4UwK6Acj6VfzW/6l4VwNczND7zWUK/WwkVwOuLBD4cJlK/cvoTwPKU1T0ofGK/bFsSwIC21T2t33C/f2sQwKG7BD6sH3u/HXYOwCR7ND7WrH+/pMcMwBfVcj6lLkG/swzjv18JtL4mGuS+9Iv4vzSAh753MRm/v5rvvyi6rr7KbaO+3IP8vyqoCL79bHS+xef6v1Z/BD2fdV2+ZvbzvxN/VD71uYK+Wb7ovxYTuz7np/i+aLPMv6ETCj8t7mO/MsvUvwaflr6vJXy/wQHHv6vqNb6pEoO/tMm7v3Iaory1M4C/Zti0v6wbHz4cKbO+8fTav8e89j5EUSS/WyXAv/RrBz+4dUu/LzS3v/Wd5z7my2u/Rzyzv9ZxpD4AAAAAZTdXvzQUDz8AAAAABtqbvzQUDz8AAAAAHGG1v4ekFj8AAAAAGSDLv1DG8D4AAAAAsU7Zv0d3sD0AAAAA2xbdv+dSnL4AAAAAZLDSv8SwG78AAAAAMUW3vxPyOb8AAAAA/+eavxPyOb8AAAAAyO9xv0dYEL8AAAAAZTdXvyan/r4OTIa+HGG1v+uoFj9eTYa+Btqbv6gYDz9eTYa+ZTdXv6gYDz8HfIa+ZTdXvxue/r4Lf4a+yO9xv9JTEL8fhoa+/+eav57tOb8fhoa+MUW3v57tOb8CgYa+ZLDSv2CsG7+Cc4a+2xbdv9xJnL41Yoa+sU7Zv+2asD0rUYa+GSDLv1vP8D4xRPa+ccjWv6p9kr4zNPa+61TTv1+ykT28JPa+v2PGv1GGuj4EIPa+j4uyv4I37D4yIfa+2T+bvy7L3z4yIfa+YThXv8qmAD/LS/a+YThXv9I47L6LTva+w4J3v0ikBb8ZVfa+9mKav/KaK78ZVfa+TkW0v/KaK79AUPa+y0rNv8H+D78ZWSa/FhaUv9bg/b6OWSa/YMmrv4ofA7/JVya/Awe+v6z93b5DUya/QfLEv2msbb6QTSa/Im7Cvxgl6DztRya/zv64vwoTdj4XRia/VYeqv2NDnz6dRia/vYuZv9s0lj6dRia/ITtXv9s0lj4VVia/ITtXv1tAuL7NVia/fnRyv02gyL7jbEq/ITtXv+mBLz4qdEq/ITtXv0oMIr6wdEq/rBlxv7pKN77u0FC/BWmGv4Y3a74/5D2/WB+Xv4Y3a77aGzi/Ivyjv7R0Rb70GTi/Ht2ov9pYyb12Fzi/7Banvz56wzzG3D2/xm+gvw4wEz792z2/PdSSv7b0OD7jbEq/C9SCv+mBLz7fFXG+tFUxv5xQAD8AAAAAtFUxv51MAD8AAAAAtFUxvwDJ5L7CaXG+tFUxv9/A5L7+Dt2+n1YxvzI+1L6QShW/HVkxv2eZpb7utDW/HVkxvxDnEb5xrjW/HVkxv5kOHT6bPBW/HVkxvxZPnT646Ny+n1Yxv1ez5j4qG/6+a0YSv2ngxz2EutC+a0YSvyofUj6/fJq+t0QSv19inD6/mSi+D0QSvyeErj4AAAAAD0QSv6Z+rj4AAAAAD0QSv+3vpL541Ci+D0QSv0vqpL59l5q+t0QSv55emb4MztC+a0YSv06Acb55JP6+a0YSv67Y370xzTS+jusDv5t0G75pNXS+iewDv1ac6r3gnZS+iewDvwAdJr2imJS+iewDvye8pD1YHnS+iewDvzrNEj68rTS+jusDv1/UTj5mLcW9KesDvxwJZD4AAAAAKesDv44CZD4AAAAAKesDvw38KL4VcsW9KesDv3/1KL7Jxs+/KSNuveZ3Gr4R4/W/OncPv3Dqsz4wn86/WTQtvtbghb4npfa/t9Ukv6SOhj5b6tS/ICWGvkqYwb7w/Py//5c3v5gwKj4gs+G/ayilvrpK976g+QPA8+JEv1w8nD06B/O/T+uuvstlD7907grAZLBKv+I/3TufIQPAa/GhvnQLGb/GTRLARx5Iv35RAr1uFwzA9fGAvtVAF79R+BjA0ZA9v4hnCb35SxPAtCEfvtEICr9C6h3AL6Msv9c07zoAqRfAB+wqvTfh5r7hYiDAQukXv7whjT3RPBjAGuGNPTI8rr7WASDAtYoCv3dIIT48FxXAdQYmPsAJZb7x1RzA+5DfvpMagj7Zsg7ACw1kPr9J873JWhfAEvvEvsgjsD7MCAbA0ZJ3PpyMKr31ZRDAMWC5vuC90z6N1fi/x55dPo6QgbujBgnASYS+vg995z4wn+a/zCQaPqMBPLwYXALAVp/TvtBf6D5VNNi/vahdPY0kgb1O1Pq/eHr1vqpD1j426cq/wxGEvTqRuL41B8y/R6ovPfPGgb4bLtS/bRobPuWYLL42IOK/+pl6PvcC8735vfO/Y++dPj8c5L1zLAPAnnyqPsxhF77rjQvAhQuhPprNY77DvBHAJAyDPmfXpb78xxTAxR4qPo2a374AORTAR61wPXsyC7+hoBDAgIFgvdgRI796bwnASrYavjlEML+aXQDAJclTvlySL79HIO6/m+NsvmA9Jr9fXd2/aQFavn0iE7+w/9C/ZAIevmFU8r5xAO2/D9OevT6uKb/Zk+C/kdSCveyEG79made/DY2nvAJFCL8y5tK/ees8PUG55b4sutO/6boAPlwcvb4Nxdm/TiVTPirinL5UG+S/ZveMPpnyib4gKvG/5SelPtwvh76V9P6/m3WuPksFlb6SsAXA+3WnPu9Xsb7QRQrAqDmRPuTX175qhwzA0EZePlRUAb9tHQzACMcMPraiFb+neA7AOZsOPDyiLr/DRwfALlOzvbPTO7/eyvq/ZTdzvfaYML8f9gDAVisTPoJVHb+KkPy/6PTcPeY9Ir/Fy/W/GM2qPRL1Ir/Ope6/TYKXPS1fH78mNei/aAWmPf8HGL/NdOO/px/UPZkNDr8IHuG/a2YNPs/0Ar/ni+G/6KQ3Pnfc8L68reS/jV1iPvon4L5wCeq/e4eDPlRX1r42zvC/TRGQPvvo1L4s9Pe/3uOUPoMU3L7UZP6/GEORPgHD6r6OkgHAqbyFPsy3/r75vQLAeSJoPp90Cr8JhwLA/OM9PkT7FL9d3QTAOJ5Pukn3O7+4zAnAaqSFPTHqMr/er/6/tmb7vmOayT7eVPq/e/cPv+NUqz7j//q/o8wiv5Jbgz53SwDA81Qzv3aILz4UIAXA8gs/v0zHvD2BQQvAFylEv5wZ/TzPwBHAGeVBv+6zSrvLoBfATpg4v/tal7t2/BvAEK0pv8aH2Tz6KR7A8WgXv/J3rz141B3AyZMEv5uuJz7uCBvAEhfovkjdfj5QNBbAN6nQvi4BqD7oEhDAym7GvkZhxz6akwnAx/bKvmnI2D6eswPAXJDdvkuQ2T6EYcA7KNesPwwCY7+Y2707IjnBP4EkQL/Nrmu+c2e+P9xiMr/DEKG+VTKqPwweVr8a3C6/lUmzP1Ka9b4UQRS/l3CiPx1VOb8psES/jEiiP9Iceb6opni/gqqRP8TpvL4vboe/v9N4P06zgL1hU0u/XaiOP2ZsaD0C1Su/Ja9aP3r+3D7EI2W/af48P+1+lT58KTi/0lblPj8ADT9cA/u+chgEPwghPD/BchS/4nWFPlVRPD8TZNy+elbSPYj3UD92F+i9GsMcOlyrWT/lX6u9kiLyPR9nWj9Y4oG8tHO6PQNeVj+6MZ2+fNKfPzrnV7+TN8A72LmlP3dJXL8fESe//UyRP1hvLL+Dp3i//OGFP6xS0r6okIe/3LhhP+v9xr0noWi/r+wiP/tYgT56NDW/yt7KPgae8z4qARG/lzxOPj7nJj8Y7eG+nUmbPYrHOT+VYZy+omGZPkErVD+z06G/djTmPtQrjT6wAKS/xCUnP6t4Mz6J7HO+WKmgOzSDWD97LWi+FCMLvUlJRz8lkii+B3wePnhBVD/oorm/lSzXPosY1j35obi/JNQQP16DPj36Krm/eEQ1P7t8O77Q8qK/6gRUP33mjL3uIaW/v51cP38Uvb5W87q/dhwzP2udwL5TzaC/atkuP6BTLL9grnm/9DUzPzoDQ79k572/S+URP0QYD79p/C6//U1QP+rNZL/zBa2+hlRhP8HKg7/2l3m/iIKZPIi+0z5ma52/rU5OO9/gez6quru/YXFoPu/Gwj3lQVK/PGoMvURsJD+pThO/XvbLvcdMTj+yDb6/YOXQPb8Nsby1qaC/DcdjvqfslD3GiXu/k41HvjHrlT5sJVi/V5RyvjjYHz9Qj32/1ZICv+zfNT5yFlq/HJcFv+7p4j7XTxO/MWCZvuhpTD9TXg+/pPwQv5oKIT8nn4i/+Gz9vmTlt70RrKC/DjKZvrzOFr7YgMC/Qzq8PKoPVL4BF6a/A7aLvqyPx76+LcK/48VCPPYIxb7Tn4e/GsPsvgETqL6yaMC/AoLRPlVpJ7+NR6e/ZTXlPpaXTL/XoHu/oPnMPjOja7/W4TS/N/vzPtqSh78HzsG/taeEPtCXIr+In6u/dHxEPk4oUL8ewoC/+KSTPclWe7/6DSu/rDtWPco0jr/fpcK/YW/CPZBlCb8sYam/cmw9vtGWJ7/cvYC/PE3uvt2aBL+TAWS/Ia78vh9KQL+wrXe/f6Vjvqg5Zb8NjYe+NligvioAUj/rc4W+86wUvzC7Jz8+tF+++Z7xvWKGXj94CHO+s5WnvmghXT/JkGO+PDIWv4MxMj8X80O+HuAZvvuQZz9+xK87xHyhvqevWz+i0LI7ct8Wv9V6Lz/XMa47sagIvn6Kaz9rt32+uWyGvz1gFj+Gj2i+uWyGv+nUIT8rTrU7uWyGv5/NIj/OGAa/uWyGv5ZDBz+q8EO/uWyGv5pbsT4FGWm/uWyGv/yqvD0EOne/uWyGv5nXUb2Tq2y/uWyGv4Sejb6JeFe/uWyGv8B19b5d3C6/uWyGv9pxF7+cTxm/5GcDv8zRZ7/wNve+uWyGv1oNLb8Wao2+ZwsIv6LxWL+LG3e+uWyGv9y7Lr+TisY72NQFv7/SSb/a5sY7uWyGv6VoKb+9Osc7bTdhvqQabr/mdJG+o+WQvq/Rer9tNyG/wCKPvksChL85Xpm+Zw80PYtQir+ES8c7Mc6fPb/wgr82HaG+srrtPmFUjr8hrMY7eT3wPpSii79FZcM7HO9mP3vder8bneu+44wJPwDhKz9zKlm+zuGCPp+uQj8dOCe9lfEfPisVQD850a47TU0SPq66Qj/lffy9mN0zPuW5Pj8t7C2/5xgsP0IJ2z58fjC/8GpBP80i7D6j6IG8dF5jPcxAZT9ClM+9dhiTvLiSaT+afR6+ZAYqvUpAVD8BM+++J9q5P+KSH79r9Fa/ED+bP68+Fr+5FlW/Ff2LP4/eEL/hClg/zQOMPxSxEL+451k/2EWbP78QFr8SFvU+9N25PzF5H7/aWRQ+Tp1HveVHVD8+QDM/FHZBP0xu7D4UsTA/6iMsP5NT2z7cLQk+++UzPujAPj9zEGQ+wOiCPmu6Qj/kEPE+W5QJP0z6Kz8VVac+DcXtPrJLjr8OoJ8+C140PUFIir/mWiQ/bw6Pvgbxg78Ttpc+VdyQvtPBer96x4E+umqGv22uLr8apZM+0QYIv0ziWL/8b/0+y2iGv9DyLL/vbxw/MV4Dv/ewZ7889DE/MGeGv25MF79Qilo/6WWGv70Z9b5Xsm8/QWWGv1Q5jb6HNHo/5WSGv3KKTr3MC2w/YmWGv+s4vj2u1UY/iGaGv1yvsT7t8wg/d2iGv1tgBz915XM+22qGv37hIT8b1E4+idMZvnOBVz80u24+oS4Wv+Y9Mj+s/X0++I2nvm4UTT/WyHo/2GZjvtUEZb8vGWc/S5H8vm4ZQL8cQ4I/uizuvuBjBL/e56o/QBc9vphOJ78gJ8Q/djPDPakSCb9mMC4/o+dWPXkijr+pT4I//iaUPc0fe7+RJ60/s9JEPivfT7+gUMM/mdiEPk5FIr+g+jc/BhL0Pnx/h79Xs34/WRnNPnZta78Yzag/ml/lPkhQTL+j6sE/X7LRPmoXJ7/mH4k/5KDsvkWfp76Gq8M/XeJIPGtjxL4+l6c/M4yLvh0Cx761+cE/HEC/PLnHUr7hJaI/jQmZvsu8Fb4JGYo/gEr9vk0Ttr00LBI/mfMQv1YpIT+ADxY/J4SWvk3XPz+i7Vw/WYkFvx9H4z5kOoA/2IICv3O4Nj6J7Vo/tMpsvjtUEz/pY34/I05HvopWlj74HKI/T3ZjvogQlz3lgL8/vqTRPdzxprxfCxY/66vLvUxsTj+KBlU/X5ULvSiZJD+zKb0/wM9oPpFGxT0j2J4/LCpiO5HtfD4jaHw/2XmbPNQo1D6JJ7M+91lhP3LBg7/IBjI/EFlQP36oZL93Zb8/N/0RP1THDr9fs3w/rkUzP+LNQr+QTaI/mu0uPyQPLL9Za7w//TMzPyr+v76BmKY/dLJcP7qHvL50YaQ/bhlUP7+6ir2/nbo/zVs1PxxBOr6yD7o/V+sQP6htQz0sELs/PlvXPtaQ2D3BbzM+xoYePrvvWD9sInM+T+cKve9VRz9w0H4+qaCiO3GQWD8naqU/aTonP2CQND4MPKM/Ol3mPre1jT7Ny6E+mGuZPhk8VD+paec+w4KbPbnfOT9qwRM/jWFOPmAGJz83+zc/mfXKPn3r8z4mcGs/XfsiPzS8gT6o/4g/98lhP/kuxb3uk3s/2emFP6Po0b50Cio/RFKRP6FLLL8SM6M++tSfP0HWV79ZFcE9lS3yPQ4RXz+phdI9E3+UvKuwWT/l1eE+DY7SPUEPUT+QLRc/woiFPh1xPD9tOAA/cSAEP+Y7PD9P6zo/Jm7lPqInDT/27mc/1Aw9P7XglT7Eli4/BrpaPwJI3T5TJU4/ya6OP5ojaz2M24g/yeR4P9HKfb1Hj3s/V7KRP7t/vL7vj0c/vk6iPwx1eL4AOxc/R3WiP0Q1Ob+gxjE/Hk+zP1lP9b7UDqc+4zSqP64MVr9Ngnc+UWm+PxVWMr9aR5U9woUMPtlBTT9xkTu9rmJxvXobXz+3Rae/0O6wPg4VYz4FMfy/oG6gu6r08z23Rae/decZPphOOz7YSfG/RdYKvgFotD23Rae/v9J5vJgTlD2RC+i/d0l8vqinj7m3Rae/+OEAvqn1vr163uG/uwukvlOzB763Rae/Kqgovl+4k75Ks9+/81qxvl+4k763Rae/+OEAvlOz97563uG/uwukvvKW4763Rae/v9J5vNI6Jr+RC+i/d0l8vlmmE7+3Rae/decZPgWMQr/YSfG/RdYKvm9FKr+3Rae/0O6wPpF9TL8FMfy/oG6gu/Q2Mr+uRae/83QKPwWMQr8ejAPAGM8APm9FKr+3Rae/G9Y0P9I6Jr9BKwjAB0JyPlmmE7+3Rae/TidRP1Oz977EQQvAAwifPvKW476uRae/7BhbP1+4k75kVwzAXFesPl+4k763Rae/TidRP6n1vr3EQQvAAwifPlOzB763Rae/G9Y0P5gTlD1BKwjAB0JyPqinj7muRae/83QKP5hOOz4ejAPAGM8APgFotD3knpC/cXI/Pu9XwT3knpC/0O6wPt/6AD7knpC/RRIBP+9XwT3knpC/gH4jPw1slTvknpC/pn46P44FBb7knpC/YJJCP1+4k77knpC/pn46P9Xt5L7knpC/gH4jP0fjFL/knpC/RRIBP23jK7/knpC/0O6wPgb3M7/knpC/cXI/Pm3jK7/cnpC//wRXPUfjFL/knpC/Xf0YvdXt5L7knpC/+BuNvV+4k77knpC/Xf0YvY4FBb7cnpC//wRXPQ1slTuHGgrAehnFPMtmjjzvrAXAVB+ovahuLj1bPwHAQMJAvstmjjy9/Pq/S3SOvhnJXr1/+PW/hj2tvv+wJb6DNfS/Gw24vl+4k75/+PW/hj2tvhyY1L69/Pq/S3SOvs3LBb9bPwHAQMJAvoQrGL/vrAXAVB+ovUmfHr+HGgrAehnFPIQrGL9/2w3ADJPpPc3LBb+iXRDA+1syPhyY1L4cPxHAJvtHPl+4k76iXRDA+1syPv+wJb5/2w3ADJPpPRnJXr2IL3+/K08APl+4k76IL3+/iXoRPpIfUb53L3+/Y19CPoDyB76IL3+/O8aFPsUarr2IL3+/0O6wPgnEi72IL3+/ZRfcPsUarr2IL3+/91YAP4DyB76IL3+/P5AMP5IfUb53L3+/FtsQP1+4k76IL3+/P5AMP/Tgvr6IL3+/91YAP313476IL3+/ZRfcPurp+76IL3+/0O6wPt0/Ar+IL3+/O8aFPurp+753L3+/Y19CPn13476IL3+/iXoRPvTgvr7XhRHASRPvO1+4k74GDhHAraWAu3x8Yr7DuA/AAfkSvXkHKL4Pug3Ak26rvUD4AL6aXwvAnnsPvqqB5r0tBQnANUBJvkD4AL5wBgfA/Dh6vnkHKL4tsQXA5niNvnx8Yr5YOQXAyjeTvl+4k74tsQXA5niNvn8ytr5wBgfA/Dh6vgFt074tBQnANUBJvp305r6aXwvAnnsPvjHQ7b4Pug3Ak26rvZ305r7DuA/AAfkSvQFt074GDhHAraWAu38ytr6brtO/j4taPpQRQ7+SsMO/+gnHvU+RH7/uBsu/kKM5PZnXOb9+yb6/3QZFvix98L7LEL2/cERnvl+4k75+yb6/3QZFvrzN272SsMO/+gnHvQyPPT2brtO/j4taPthkPT5HVty/HVfDPpnXOb+srOO/BycGP0+RH7+4k+i/f4cePyx98L5zTOq/0xYnP1+4k77uBsu/kKM5PSx9GD5HVty/HVfDPix9GD6srOO/BycGPwyPPT24k+i/f4ceP7zN272wdEo/rBlxv7pKN77jbEo/C9SCv+mBLz7jbEo/ITtXv+mBLz4qdEo/ITtXv0oMIr5eTYY+Btqbv6gYDz8OTIY+HGG1v+uoFj9eTYY+ZTdXv6gYDz8VcsU9KesDv3/1KL4xzTQ+jusDv5t0G75pNXQ+iewDv1ac6r3gnZQ+iewDvwAdJr2imJQ+iewDvye8pD1YHnQ+iewDvzrNEj68rTQ+jusDv1/UTj5mLcU9KesDvxwJZD4Lf4Y+yO9xv9JTEL8HfIY+ZTdXvxue/r4fhoY+/+eav57tOb8fhoY+MUW3v57tOb8CgYY+ZLDSv2CsG7+Cc4Y+2xbdv9xJnL41YoY+sU7Zv+2asD0rUYY+GSDLv1vP8D4zNPY+61TTv1+ykT0xRPY+ccjWv6p9kr68JPY+v2PGv1GGuj4EIPY+j4uyv4I37D4yIfY+2T+bvy7L3z4yIfY+YThXv8qmAD+LTvY+w4J3v0ikBb/LS/Y+YThXv9I47L4ZVfY+9mKav/KaK78ZVfY+TkW0v/KaK79AUPY+y0rNv8H+D7+OWSY/YMmrv4ofA78ZWSY/FhaUv9bg/b7JVyY/Awe+v6z93b5DUyY/QfLEv2msbb6QTSY/Im7Cvxgl6DztRyY/zv64vwoTdj4XRiY/VYeqv2NDnz6dRiY/vYuZv9s0lj6dRiY/ITtXv9s0lj7NViY/fnRyv02gyL4VViY/ITtXv1tAuL7u0FA/BWmGv4Y3a74/5D0/WB+Xv4Y3a77aGzg/Ivyjv7R0Rb70GTg/Ht2ov9pYyb12Fzg/7Banvz56wzzG3D0/xm+gvw4wEz792z0/PdSSv7b0OD7fFXE+tFUxv5xQAD/CaXE+tFUxv9/A5L7+Dt0+n1YxvzI+1L6QShU/HVkxv2eZpb7utDU/HVkxvxDnEb5xrjU/HVkxv5kOHT6bPBU/HVkxvxZPnT646Nw+n1Yxv1ez5j6EutA+a0YSvyofUj4qG/4+a0YSv2ngxz2/fJo+t0QSv19inD6/mSg+D0QSvyeErj541Cg+D0QSv0vqpL59l5o+t0QSv55emb4MztA+a0YSv06Acb55JP4+a0YSv67Y370oAAAALwAAADAAAAAwAAAAJwAAACgAAAAuAAAAKQAAACoAAAAuAAAAKgAAACsAAAAuAAAAKwAAACwAAAAuAAAALAAAAC0AAAAmAAAAMQAAADIAAAAyAAAAMwAAACYAAAAnAAAAMAAAADEAAAAxAAAAJgAAACcAAAApAAAALgAAAC8AAAAvAAAAKAAAACkAAAABAAAAAAAAACQAAAAkAAAAIwAAAAEAAAAKAAAACQAAACAAAAAgAAAAHwAAAAoAAAAOAAAADQAAABwAAAAcAAAAGwAAAA4AAAARAAAAEgAAABgAAAAYAAAAFwAAABEAAAADAAAABQAAABQAAAAUAAAAEwAAAAMAAAAGAAAAEQAAABcAAAAXAAAAFgAAAAYAAAAPAAAADgAAABsAAAAbAAAAGgAAAA8AAAALAAAACgAAAB8AAAAfAAAAHgAAAAsAAAACAAAAAQAAACMAAAAjAAAAIgAAAAIAAAAAAAAABAAAACUAAAAlAAAAJAAAAAAAAAAJAAAACAAAACEAAAAhAAAAIAAAAAkAAAANAAAADAAAAB0AAAAdAAAAHAAAAA0AAAASAAAAEAAAABkAAAAZAAAAGAAAABIAAAAFAAAABwAAABUAAAAVAAAAFAAAAAUAAAAHAAAABgAAABYAAAAWAAAAFQAAAAcAAAAQAAAADwAAABoAAAAaAAAAGQAAABAAAAAMAAAACwAAAB4AAAAeAAAAHQAAAAwAAAAIAAAAAgAAACIAAAAiAAAAIQAAAAgAAAAEAAAAAwAAABMAAAATAAAAJQAAAAQAAAAhAAAAIgAAADMAAAAhAAAAMwAAADYAAAAhAAAANgAAADUAAAAjAAAAJAAAACcAAAAnAAAAJgAAACMAAAAgAAAAIQAAADUAAAA1AAAANAAAACAAAAAiAAAAIwAAACYAAAAmAAAAMwAAACIAAAAkAAAAJQAAACgAAAAoAAAAJwAAACQAAAAlAAAAEwAAACkAAAApAAAAKAAAACUAAAATAAAAFAAAACoAAAAqAAAAKQAAABMAAAAWAAAAFwAAAC0AAAAtAAAALAAAABYAAAAUAAAAFQAAACsAAAArAAAAKgAAABQAAAAVAAAAFgAAACwAAAAsAAAAKwAAABUAAAAcAAAAHQAAADgAAAAcAAAAOAAAADcAAAAcAAAANwAAADIAAAAYAAAAGQAAAC8AAAAvAAAALgAAABgAAAAZAAAAGgAAADAAAAAwAAAALwAAABkAAAAbAAAAHAAAADIAAAAyAAAAMQAAABsAAAAXAAAAGAAAAC4AAAAuAAAALQAAABcAAAAaAAAAGwAAADEAAAAxAAAAMAAAABoAAAAfAAAAIAAAADQAAAA0AAAAOgAAAB8AAAAeAAAAHwAAADoAAAA6AAAAOQAAAB4AAAA1AAAAOAAAADkAAAA1AAAAOQAAADoAAAA1AAAAOgAAADQAAAA2AAAANwAAADgAAAA4AAAANQAAADYAAAAdAAAAHgAAADkAAAA5AAAAOAAAAB0AAAAzAAAAMgAAADcAAAA3AAAANgAAADMAAABiAAAAawAAAGoAAABqAAAAYwAAAGIAAABoAAAAZwAAAGYAAABoAAAAZgAAAGUAAABoAAAAZQAAAGQAAABoAAAAZAAAAGkAAABuAAAAbQAAAGwAAABsAAAAYQAAAG4AAABhAAAAbAAAAGsAAABrAAAAYgAAAGEAAABjAAAAagAAAGkAAABpAAAAZAAAAGMAAABeAAAAXwAAADsAAAA7AAAAPAAAAF4AAABaAAAAWwAAAEQAAABEAAAARQAAAFoAAABWAAAAVwAAAEgAAABIAAAASQAAAFYAAABSAAAAUwAAAE0AAABNAAAATAAAAFIAAABOAAAATwAAAEAAAABAAAAAPgAAAE4AAABRAAAAUgAAAEwAAABMAAAAQQAAAFEAAABVAAAAVgAAAEkAAABJAAAASgAAAFUAAABZAAAAWgAAAEUAAABFAAAARgAAAFkAAABdAAAAXgAAADwAAAA8AAAAPQAAAF0AAABfAAAAYAAAAD8AAAA/AAAAOwAAAF8AAABbAAAAXAAAAEMAAABDAAAARAAAAFsAAABXAAAAWAAAAEcAAABHAAAASAAAAFcAAABTAAAAVAAAAEsAAABLAAAATQAAAFMAAABPAAAAUAAAAEIAAABCAAAAQAAAAE8AAABQAAAAUQAAAEEAAABBAAAAQgAAAFAAAABUAAAAVQAAAEoAAABKAAAASwAAAFQAAABYAAAAWQAAAEYAAABGAAAARwAAAFgAAABcAAAAXQAAAD0AAAA9AAAAQwAAAFwAAABgAAAATgAAAD4AAAA+AAAAPwAAAGAAAABwAAAAcQAAAG4AAABwAAAAbgAAAF0AAABwAAAAXQAAAFwAAABhAAAAYgAAAF8AAABfAAAAXgAAAGEAAABvAAAAcAAAAFwAAABcAAAAWwAAAG8AAABuAAAAYQAAAF4AAABeAAAAXQAAAG4AAABiAAAAYwAAAGAAAABgAAAAXwAAAGIAAABjAAAAZAAAAE4AAABOAAAAYAAAAGMAAABkAAAAZQAAAE8AAABPAAAATgAAAGQAAABnAAAAaAAAAFIAAABSAAAAUQAAAGcAAABlAAAAZgAAAFAAAABQAAAATwAAAGUAAABmAAAAZwAAAFEAAABRAAAAUAAAAGYAAABtAAAAcgAAAHMAAABtAAAAcwAAAFgAAABtAAAAWAAAAFcAAABpAAAAagAAAFQAAABUAAAAUwAAAGkAAABqAAAAawAAAFUAAABVAAAAVAAAAGoAAABsAAAAbQAAAFcAAABXAAAAVgAAAGwAAABoAAAAaQAAAFMAAABTAAAAUgAAAGgAAABrAAAAbAAAAFYAAABWAAAAVQAAAGsAAAB1AAAAbwAAAFsAAABbAAAAWgAAAHUAAAB0AAAAdQAAAFoAAABaAAAAWQAAAHQAAABvAAAAdQAAAHQAAABvAAAAdAAAAHMAAABvAAAAcwAAAHAAAABwAAAAcwAAAHIAAAByAAAAcQAAAHAAAABzAAAAdAAAAFkAAABZAAAAWAAAAHMAAABxAAAAcgAAAG0AAABtAAAAbgAAAHEAAAAuOok/ChBPwCU8wT5VT4g/gLlNwJjglD5VvYI/EeVMwAHBLD5dv4Q/q0FPwNCbQj/deok/nkFPwHoXBz9ljnE/jUFPwD5BWj+lhS8/okFPwAuWYj/DKVc/xEFPwEaYYj87jXw/Q+VMwMo0Wj3JOmQ/JuVMwCkF3b0Ji0o/HeVMwBO4Fb678S4/9+RMwD8C/73bpRk/AOVMwPQ3oTo6QRc/COVMwB/2Ij6IYxE/8MNNwDM0jj5jnAs/okFPwOhozT6H4Ag/q0FPwEWEEz/YvB4/nkFPwFQ4Wj+kbg8/hUFPwOGXQj+Dh4U/CHRPwEz9RD/kg3I/73NPwLA7XT+KdFc/IXRPwJfIZT/6zy4/BHRPwG3GZT+mmh0/+3NPwJQyXT9s6Q0/4nNPwE35RD/5MAc/CHRPwCS1FD92/gk/BHRPwOyGzT4H6w8/TfZNwGe5jD6a7hU/ahdNwCTvHD7DYhg/XRdNwPSnDbxlOC4/WRdNwCbfDL4WhEo/ehdNwH+lI74K2mQ/gxdNwATl9r3WyX0/oBdNwOIiNz15eIM/bhdNwPD4Jj6MLok/TfZNwOuQkz5UH4o/aEJPwIULwT6mYYo/BHRPwBL4Bz+MLok/QFBRwMmQkz5LH4o/7s1SwIULwT6eYYo/981SwAH4Bz+Dh4U//81SwEz9RD/Tg3I/3c1SwLA7XT95dFc/GM5SwJfIZT/6zy4/981SwFzGZT+mmh0/8s1SwIMyXT9s6Q0/2c1SwDz5RD/oMAc//81SwCS1FD9l/gk/981SwOyGzT736g8/RFBRwGe5jD6a7hU/XHFQwOHuHD55eIM/ZXFQwPD4Jj752WQ/fNRSwATl9r3FyX0/fNRSwOIiNz1weIM/fNRSwPD4Jj6J7hU/fNRSwCTvHD6iYhg/fNRSwPSnDbxDOC4/fNRSwCbfDL4GhEo/fNRSwMKlI74uOom/ChBPwCU8wT5VT4i/gLlNwJjglD5VvYK/EeVMwAHBLD5dv4S/q0FPwNCbQj/deom/nkFPwHoXBz9ljnG/jUFPwD5BWj+lhS+/okFPwAuWYj/DKVe/xEFPwEaYYj87jXy/Q+VMwMo0Wj3JOmS/JuVMwCkF3b0Ji0q/HeVMwBO4Fb678S6/9+RMwD8C/73bpRm/AOVMwPQ3oTo6QRe/COVMwB/2Ij6IYxG/8MNNwDM0jj5jnAu/okFPwOhozT6H4Ai/q0FPwEWEEz/YvB6/nkFPwFQ4Wj+kbg+/hUFPwOGXQj+Dh4W/CHRPwEz9RD/kg3K/73NPwLA7XT+KdFe/IXRPwJfIZT/6zy6/BHRPwG3GZT+mmh2/+3NPwJQyXT9s6Q2/4nNPwE35RD/5MAe/CHRPwCS1FD92/gm/BHRPwOyGzT4H6w+/TfZNwGe5jD6a7hW/ahdNwCTvHD7DYhi/XRdNwPSnDbxlOC6/WRdNwCbfDL4WhEq/ehdNwH+lI74K2mS/gxdNwATl9r3WyX2/oBdNwOIiNz15eIO/bhdNwPD4Jj6MLom/TfZNwOuQkz5UH4q/aEJPwIULwT6mYYq/BHRPwBL4Bz+MLom/QFBRwMmQkz5LH4q/7s1SwIULwT6eYYq/981SwAH4Bz+Dh4W//81SwEz9RD/Tg3K/3c1SwLA7XT95dFe/GM5SwJfIZT/6zy6/981SwFzGZT+mmh2/8s1SwIMyXT9s6Q2/2c1SwDz5RD/oMAe//81SwCS1FD9l/gm/981SwOyGzT736g+/RFBRwGe5jD6a7hW/XHFQwOHuHD55eIO/ZXFQwPD4Jj752WS/fNRSwATl9r3FyX2/fNRSwOIiNz1weIO/fNRSwPD4Jj6J7hW/fNRSwCTvHD6iYhi/fNRSwPSnDbxDOC6/fNRSwCbfDL4GhEq/fNRSwMKlI747AAAAPQAAADwAAAAUAAAALwAAABcAAAAUAAAAFwAAABYAAAAUAAAAFgAAABUAAAASAAAAIgAAACUAAAAlAAAAEwAAABIAAAAkAAAAJwAAAC4AAAAkAAAALgAAAEQAAAAkAAAARAAAAEUAAAAkAAAARQAAAEAAAAAkAAAAQAAAAEEAAAArAAAALAAAACkAAAApAAAAKgAAACsAAAAkAAAAJQAAACIAAAAiAAAAIwAAACQAAAAYAAAAAAAAADIAAAAyAAAARwAAABgAAAAWAAAAOAAAADkAAAA5AAAAFQAAABYAAAADAAAANgAAADEAAAAxAAAAAQAAAAMAAAAPAAAAEQAAABAAAAAQAAAADgAAAA8AAAAOAAAACQAAAAwAAAAMAAAADwAAAA4AAAAGAAAABwAAAAUAAAAFAAAABAAAAAYAAAAGAAAACAAAAAcAAAANAAAACQAAAA4AAAASAAAAEwAAABAAAAAQAAAAEQAAABIAAAA0AAAAPQAAADoAAAA6AAAAMAAAADQAAAAzAAAANwAAAD4AAAA+AAAARgAAADMAAAAmAAAAJwAAACQAAAAkAAAAIwAAACYAAAAtAAAALgAAACkAAAApAAAALAAAAC0AAAAvAAAAJQAAACQAAAAvAAAAJAAAAEEAAAAvAAAAQQAAAEIAAAAiAAAAEgAAABoAAAAiAAAAGgAAABsAAAAiAAAAGwAAAB8AAAAiAAAAHwAAACAAAABAAAAAQgAAAEEAAAA0AAAANgAAADUAAAAvAAAAFAAAABMAAAATAAAAJQAAAC8AAABEAAAALgAAAC0AAABEAAAALQAAABkAAABEAAAAGQAAAEMAAAAhAAAAKgAAACkAAAApAAAAKAAAACEAAABFAAAARgAAAD4AAAA+AAAAQAAAAEUAAAAVAAAAOQAAADsAAAA7AAAAFAAAABUAAAAyAAAAAAAAAAEAAAABAAAAMQAAADIAAAADAAAABQAAAA0AAAADAAAADQAAADUAAAADAAAANQAAADYAAAALAAAADAAAAAkAAAAJAAAACgAAAAsAAAAEAAAABQAAAAMAAAADAAAAAgAAAAQAAAAIAAAACgAAAAkAAAAJAAAABwAAAAgAAAA1AAAADQAAAA4AAAA1AAAADgAAABAAAAA1AAAAEAAAADwAAAA1AAAAPAAAAD0AAAA1AAAAPQAAADQAAAAUAAAAOwAAADwAAAAUAAAAPAAAABAAAAAUAAAAEAAAABMAAAAwAAAAOgAAADcAAAA3AAAAMwAAADAAAAAXAAAAPwAAADgAAAA4AAAAFgAAABcAAAAZAAAAGAAAAEcAAABHAAAAQwAAABkAAAAoAAAAKQAAACcAAAAnAAAAJgAAACgAAAAuAAAAJwAAACkAAAA/AAAAFwAAAC8AAAAvAAAAQgAAAD8AAAAJAAAADQAAAAUAAAAFAAAABwAAAAkAAAAfAAAAGwAAABwAAAAfAAAAHAAAAB0AAAAfAAAAHQAAAB4AAABDAAAARQAAAEQAAABSAAAAUwAAAGYAAABmAAAAZQAAAFIAAABOAAAATwAAAGIAAABiAAAAYQAAAE4AAABKAAAASwAAAF4AAABeAAAAXQAAAEoAAABWAAAAVwAAAFoAAABaAAAAWQAAAFYAAABVAAAAVgAAAFkAAABZAAAAWAAAAFUAAABJAAAASgAAAF0AAABdAAAAXAAAAEkAAABNAAAATgAAAGEAAABhAAAAYAAAAE0AAABRAAAAUgAAAGUAAABlAAAAZAAAAFEAAABTAAAAVAAAAGcAAABnAAAAZgAAAFMAAABPAAAAUAAAAGMAAABjAAAAYgAAAE8AAABLAAAATAAAAF8AAABfAAAAXgAAAEsAAABXAAAASAAAAFsAAABbAAAAWgAAAFcAAABmAAAAZwAAAFgAAABmAAAAWAAAAFkAAABmAAAAWQAAAFoAAABmAAAAWgAAAFsAAABmAAAAWwAAAFwAAABmAAAAXAAAAF0AAABmAAAAXQAAAF4AAABmAAAAXgAAAF8AAABmAAAAXwAAAGAAAABmAAAAYAAAAGEAAABmAAAAYQAAAGIAAABmAAAAYgAAAGMAAABmAAAAYwAAAGQAAABmAAAAZAAAAGUAAABIAAAASQAAAFwAAABcAAAAWwAAAEgAAABMAAAATQAAAGAAAABgAAAAXwAAAEwAAABQAAAAUQAAAGQAAABkAAAAYwAAAFAAAABUAAAAVQAAAFgAAABYAAAAZwAAAFQAAACkAAAApQAAAKMAAAB9AAAAfgAAAH8AAAB9AAAAfwAAAJcAAAB9AAAAlwAAAHwAAAB7AAAAjQAAAIoAAACKAAAAegAAAHsAAACpAAAAqAAAAK0AAACpAAAArQAAAKwAAACpAAAArAAAAJYAAACpAAAAlgAAAI8AAACpAAAAjwAAAIwAAACSAAAAkQAAAJQAAACUAAAAkwAAAJIAAACLAAAAigAAAI0AAACNAAAAjAAAAIsAAACvAAAAmgAAAGgAAABoAAAAgAAAAK8AAAB9AAAAoQAAAKAAAACgAAAAfgAAAH0AAABpAAAAmQAAAJ4AAACeAAAAawAAAGkAAAB2AAAAeAAAAHkAAAB5AAAAdwAAAHYAAAB3AAAAdAAAAHEAAABxAAAAdgAAAHcAAABsAAAAbQAAAG8AAABvAAAAbgAAAGwAAABvAAAAcAAAAG4AAAB2AAAAcQAAAHUAAAB5AAAAeAAAAHsAAAB7AAAAegAAAHkAAACYAAAAogAAAKUAAAClAAAAnAAAAJgAAACuAAAApgAAAJ8AAACfAAAAmwAAAK4AAACLAAAAjAAAAI8AAACPAAAAjgAAAIsAAACUAAAAkQAAAJYAAACWAAAAlQAAAJQAAACqAAAAqQAAAIwAAACqAAAAjAAAAI0AAACqAAAAjQAAAJcAAACIAAAAhwAAAIMAAACIAAAAgwAAAIIAAACIAAAAggAAAHoAAACIAAAAegAAAIoAAACpAAAAqgAAAKgAAACdAAAAngAAAJwAAACNAAAAewAAAHwAAAB8AAAAlwAAAI0AAACrAAAAgQAAAJUAAACrAAAAlQAAAJYAAACrAAAAlgAAAKwAAACQAAAAkQAAAJIAAACSAAAAiQAAAJAAAACoAAAApgAAAK4AAACuAAAArQAAAKgAAAB8AAAAowAAAKEAAAChAAAAfQAAAHwAAACZAAAAaQAAAGgAAABoAAAAmgAAAJkAAACeAAAAnQAAAHUAAACeAAAAdQAAAG0AAACeAAAAbQAAAGsAAAByAAAAcQAAAHQAAAB0AAAAcwAAAHIAAABqAAAAawAAAG0AAABtAAAAbAAAAGoAAABvAAAAcQAAAHIAAAByAAAAcAAAAG8AAACcAAAApQAAAKQAAACcAAAApAAAAHgAAACcAAAAeAAAAHYAAACcAAAAdgAAAHUAAACcAAAAdQAAAJ0AAAB7AAAAeAAAAKQAAAB7AAAApAAAAKMAAAB7AAAAowAAAHwAAACbAAAAnwAAAKIAAACiAAAAmAAAAJsAAAB+AAAAoAAAAKcAAACnAAAAfwAAAH4AAACrAAAArwAAAIAAAACAAAAAgQAAAKsAAACOAAAAjwAAAJEAAACRAAAAkAAAAI4AAACRAAAAjwAAAJYAAACqAAAAlwAAAH8AAAB/AAAApwAAAKoAAABvAAAAbQAAAHUAAAB1AAAAcQAAAG8AAACGAAAAhQAAAIQAAACGAAAAhAAAAIMAAACGAAAAgwAAAIcAAACsAAAArQAAAKsAAADOAAAAzwAAALwAAAC8AAAAuwAAAM4AAADKAAAAywAAALgAAAC4AAAAtwAAAMoAAADGAAAAxwAAALQAAAC0AAAAswAAAMYAAADCAAAAwwAAALAAAACwAAAAvwAAAMIAAADNAAAAzAAAAMsAAADNAAAAywAAAMoAAADNAAAAygAAAMkAAADNAAAAyQAAAMgAAADNAAAAyAAAAMcAAADNAAAAxwAAAMYAAADNAAAAxgAAAMUAAADNAAAAxQAAAMQAAADNAAAAxAAAAMMAAADNAAAAwwAAAMIAAADNAAAAwgAAAMEAAADNAAAAwQAAAMAAAADNAAAAwAAAAM8AAADNAAAAzwAAAM4AAADDAAAAxAAAALEAAACxAAAAsAAAAMMAAADHAAAAyAAAALUAAAC1AAAAtAAAAMcAAADLAAAAzAAAALkAAAC5AAAAuAAAAMsAAADPAAAAwAAAAL0AAAC9AAAAvAAAAM8AAADNAAAAzgAAALsAAAC7AAAAugAAAM0AAADJAAAAygAAALcAAAC3AAAAtgAAAMkAAADFAAAAxgAAALMAAACzAAAAsgAAAMUAAADBAAAAwgAAAL8AAAC/AAAAvgAAAMEAAADAAAAAwQAAAL4AAAC+AAAAvQAAAMAAAADEAAAAxQAAALIAAACyAAAAsQAAAMQAAADIAAAAyQAAALYAAAC2AAAAtQAAAMgAAADMAAAAzQAAALoAAAC6AAAAuQAAAMwAAADmAAAA6AAAAOwAAADZAAAA2AAAAN8AAADfAAAA3gAAANkAAADkAAAA5QAAAOwAAADdAAAA3AAAAN4AAADeAAAA1gAAAN0AAADfAAAA1gAAAN4AAADrAAAA6gAAAOUAAADlAAAA5AAAAOsAAADlAAAA5wAAAOYAAADmAAAA7AAAAOUAAADWAAAA3wAAANUAAADdAAAA1gAAANsAAADXAAAA0AAAANEAAADXAAAA0QAAAOAAAADXAAAA4AAAANIAAADXAAAA0gAAANMAAADXAAAA0wAAANQAAADXAAAA1AAAAOEAAADXAAAA4QAAANgAAADuAAAA4gAAAO0AAADqAAAA6QAAAOcAAADnAAAA5QAAAOoAAADuAAAA7wAAAPAAAADuAAAA8AAAAOMAAADuAAAA4wAAAPEAAADuAAAA8QAAANAAAADuAAAA0AAAANcAAADuAAAA1wAAAPIAAADuAAAA8gAAAOsAAADuAAAA6wAAAOIAAADUAAAA1QAAAOEAAADaAAAA2QAAAN4AAADeAAAA3AAAANoAAADVAAAA3wAAANgAAADYAAAA4QAAANUAAADsAAAA7QAAAOQAAADkAAAA7QAAAOIAAADiAAAA6wAAAOQAAAAvAQAALgEAAC0BAAAvAQAAKgEAACkBAAAvAQAAJgEAACUBAAAuAQAAIgEAACEBAAAhAQAALQEAAC4BAAAqAQAAHgEAAB0BAAAdAQAAKQEAACoBAAAmAQAAGgEAABkBAAAZAQAAJQEAACYBAAAiAQAAFgEAABUBAAAVAQAAIQEAACIBAAAeAQAAEgEAABEBAAARAQAAHQEAAB4BAAAaAQAADgEAAA0BAAANAQAAGQEAABoBAAAWAQAACgEAAAkBAAAJAQAAFQEAABYBAAASAQAABgEAAAUBAAAFAQAAEQEAABIBAAAOAQAAAgEAAAEBAAABAQAADQEAAA4BAAAKAQAA/gAAAP0AAAD9AAAACQEAAAoBAAAGAQAA+gAAAPkAAAD5AAAABQEAAAYBAAACAQAA9gAAAPUAAAD1AAAAAQEAAAIBAAABAQAA9QAAAPQAAAD0AAAAAAEAAAEBAAAFAQAA+QAAAPgAAAD4AAAABAEAAAUBAAAJAQAA/QAAAPwAAAD8AAAACAEAAAkBAAANAQAAAQEAAAABAAAAAQAADAEAAA0BAAARAQAABQEAAAQBAAAEAQAAEAEAABEBAAAVAQAACQEAAAgBAAAIAQAAFAEAABUBAAAZAQAADQEAAAwBAAAMAQAAGAEAABkBAAAdAQAAEQEAABABAAAQAQAAHAEAAB0BAAAhAQAAFQEAABQBAAAUAQAAIAEAACEBAAAlAQAAGQEAABgBAAAYAQAAJAEAACUBAAApAQAAHQEAABwBAAAcAQAAKAEAACkBAAAtAQAAIQEAACABAAAgAQAALAEAAC0BAAAvAQAAJQEAACQBAAAvAQAAKQEAACgBAAAvAQAALQEAACwBAAAvAQAAIwEAAC4BAAAvAQAAKwEAACoBAAAvAQAAJwEAACYBAAAjAQAAFwEAACIBAAAiAQAALgEAACMBAAArAQAAHwEAAB4BAAAeAQAAKgEAACsBAAAnAQAAGwEAABoBAAAaAQAAJgEAACcBAAAXAQAACwEAABYBAAAWAQAAIgEAABcBAAAfAQAAEwEAABIBAAASAQAAHgEAAB8BAAAbAQAADwEAAA4BAAAOAQAAGgEAABsBAAALAQAA/wAAAAoBAAAKAQAAFgEAAAsBAAATAQAABwEAAAYBAAAGAQAAEgEAABMBAAAPAQAAAwEAAAIBAAACAQAADgEAAA8BAAD/AAAA8wAAAP4AAAD+AAAACgEAAP8AAAAHAQAA+wAAAPoAAAD6AAAABgEAAAcBAAADAQAA9wAAAPYAAAD2AAAAAgEAAAMBAAAAAQAA9AAAAPMAAADzAAAA/wAAAAABAAAEAQAA+AAAAPcAAAD3AAAAAwEAAAQBAAAIAQAA/AAAAPsAAAD7AAAABwEAAAgBAAAMAQAAAAEAAP8AAAD/AAAACwEAAAwBAAAQAQAABAEAAAMBAAADAQAADwEAABABAAAUAQAACAEAAAcBAAAHAQAAEwEAABQBAAAYAQAADAEAAAsBAAALAQAAFwEAABgBAAAcAQAAEAEAAA8BAAAPAQAAGwEAABwBAAAgAQAAFAEAABMBAAATAQAAHwEAACABAAAkAQAAGAEAABcBAAAXAQAAIwEAACQBAAAoAQAAHAEAABsBAAAbAQAAJwEAACgBAAAsAQAAIAEAAB8BAAAfAQAAKwEAACwBAAAvAQAAJAEAACMBAAAvAQAAKAEAACcBAAAvAQAALAEAACsBAABpAQAAagEAAGwBAABlAQAAZgEAAGwBAABhAQAAYgEAAGwBAABpAQAAXQEAAF4BAABeAQAAagEAAGkBAABlAQAAWQEAAFoBAABaAQAAZgEAAGUBAABhAQAAVQEAAFYBAABWAQAAYgEAAGEBAABdAQAAUQEAAFIBAABSAQAAXgEAAF0BAABZAQAATQEAAE4BAABOAQAAWgEAAFkBAABVAQAASQEAAEoBAABKAQAAVgEAAFUBAABRAQAARQEAAEYBAABGAQAAUgEAAFEBAABNAQAAQQEAAEIBAABCAQAATgEAAE0BAABJAQAAPQEAAD4BAAA+AQAASgEAAEkBAABFAQAAOQEAADoBAAA6AQAARgEAAEUBAABBAQAANQEAADYBAAA2AQAAQgEAAEEBAAA9AQAAMQEAADIBAAAyAQAAPgEAAD0BAAA+AQAAMgEAADMBAAAzAQAAPwEAAD4BAABCAQAANgEAADcBAAA3AQAAQwEAAEIBAABGAQAAOgEAADsBAAA7AQAARwEAAEYBAABKAQAAPgEAAD8BAAA/AQAASwEAAEoBAABOAQAAQgEAAEMBAABDAQAATwEAAE4BAABSAQAARgEAAEcBAABHAQAAUwEAAFIBAABWAQAASgEAAEsBAABLAQAAVwEAAFYBAABaAQAATgEAAE8BAABPAQAAWwEAAFoBAABeAQAAUgEAAFMBAABTAQAAXwEAAF4BAABiAQAAVgEAAFcBAABXAQAAYwEAAGIBAABmAQAAWgEAAFsBAABbAQAAZwEAAGYBAABqAQAAXgEAAF8BAABfAQAAawEAAGoBAABiAQAAYwEAAGwBAABmAQAAZwEAAGwBAABqAQAAawEAAGwBAABoAQAAaQEAAGwBAABkAQAAZQEAAGwBAABgAQAAYQEAAGwBAABoAQAAXAEAAF0BAABdAQAAaQEAAGgBAABkAQAAWAEAAFkBAABZAQAAZQEAAGQBAABgAQAAVAEAAFUBAABVAQAAYQEAAGABAABcAQAAUAEAAFEBAABRAQAAXQEAAFwBAABYAQAATAEAAE0BAABNAQAAWQEAAFgBAABUAQAASAEAAEkBAABJAQAAVQEAAFQBAABQAQAARAEAAEUBAABFAQAAUQEAAFABAABMAQAAQAEAAEEBAABBAQAATQEAAEwBAABIAQAAPAEAAD0BAAA9AQAASQEAAEgBAABEAQAAOAEAADkBAAA5AQAARQEAAEQBAABAAQAANAEAADUBAAA1AQAAQQEAAEABAAA8AQAAMAEAADEBAAAxAQAAPQEAADwBAAA/AQAAMwEAADQBAAA0AQAAQAEAAD8BAABDAQAANwEAADgBAAA4AQAARAEAAEMBAABHAQAAOwEAADABAAAwAQAAPAEAAEcBAABLAQAAPwEAAEABAABAAQAATAEAAEsBAABPAQAAQwEAAEQBAABEAQAAUAEAAE8BAABTAQAARwEAADwBAAA8AQAASAEAAFMBAABXAQAASwEAAEwBAABMAQAAWAEAAFcBAABbAQAATwEAAFABAABQAQAAXAEAAFsBAABfAQAAUwEAAEgBAABIAQAAVAEAAF8BAABjAQAAVwEAAFgBAABYAQAAZAEAAGMBAABnAQAAWwEAAFwBAABcAQAAaAEAAGcBAABrAQAAXwEAAFQBAABUAQAAYAEAAGsBAABjAQAAZAEAAGwBAABnAQAAaAEAAGwBAABrAQAAYAEAAGwBAACLOEk/K207wEsEGj+r7WY/pmQ7wInQDD8VAIM/3jw9wKhu9j5AMYI/1xU+wChh9j5M/4w/hbFAwCRH4j6lMIs/hEhBwAtF4j7Zr40/5E1JwItSyj5OuIs/zEZJwFBQyj4uOok/ChBPwCU8wT5I3Ik/5Q5KwFckjj5VT4g/gLlNwJjglD5VvYI/EeVMwAHBLD7K4IQ/9+dJwLTlDD5Y4ok/HjRBwDgVoT4DmYk/aydBwBcRdT40ZYU/SPhBwEuQsT3WO4g/zVw8wBecUT7vyoI/g089wCXPNT0L1Xk/e8E2wHLArrvoiIQ/euE1wOKxLz5WgIE/Ol00wNRhdT6HwWQ/eHwzwENZkD5qMUg//+cywOwzjz5+5DI/z0gzwFPNjD645Cw/rwk8wPLOFT+BXw8/+ig+wIrNAz8riXw//cA2wBnjQ7zO/3M/1a42wNwOjb15I2M/kq42wCkI3r3tC1A/T642wDc4Ab6g+zo/ya02wEC/772iJi4/l602wFoQqr3o9h4/Er82wADkhLw6QRc/COVMwB/2Ij4SvCE/tr02wKHXH7wjFhE/93I8wIY4tj0XDw8/MbM7wAXhWj4cCxo/nN01wKN5ED6cag0/NShCwAAeAT5cygk/YMhBwOCgfT69NxI/6ztJwEDBFT4hdg4//7FJwBHDjj6IYxE/8MNNwDM0jj5jnAs/okFPwOhozT6j6gM/gSFKwNyb1z5SKgE/VORCwK7w7j4N4Qg/hGVCwOyhtT57oB0/Ozk0wBKgVj58YGc/mdY5wNpz0T743mc/XDw7wGnh6j4MWEo/0o46wDLm9j5pcUo/xlA5wJ3z2z7KU4I/oKU7wAfvsz4xQ4M/xAg9wAFrvT4IWYI/vMk8wGhAzT58Rko/n8c3wG9Jxj6EuUk/Et41wJxSrj7RBGo/DCA2wIbipj5Numk//Dc4wJaSvT4k8IE/8rQ4wMyyjz6SzoI/MQk6wHapkT4rNII/s5U6wI3Uoz69cC8/cJo3wAeVwD6HMjA/H9o1wM9ppj4/4Rg/eV05wO2asD7CaRk/0Lk4wMiVmj48wRo/A5k3wAXClj47xBM/8DM8wBOC5T5lbRM/inI8wInS1j6dRhY/jNY6wNqOyT6jrS4/G/Q4wJS82j7h8S0/fSU6wPLQ9z7er/4/tmb7vmOayT7eVPo/e/cPv+NUqz7j//o/o8wiv5Jbgz53SwBA81Qzv3aILz4UIAVA8gs/v0zHvD2BQQtAFylEv5wZ/TzPwBFAGeVBv+6zSrvLoBdATpg4v/tal7t2/BtAEK0pv8aH2Tz6KR5A8WgXv/J3rz141B1AyZMEv5uuJz7uCBtAEhfovkjdfj5QNBZAN6nQvi4BqD7oEhBAym7GvkZhxz6akwlAx/bKvmnI2D6eswNAXJDdvkuQ2T6paRFAl3Dgvr+Z4D5b6gpAcvjkvgQB8j5jCgVAKZL3vsXI8j60rgBAQrQKv/7S4j5gAv0/Yvgcv36NxD5krf0/is0vvwyUnD48ogFA2lVAv2r5YT7VdgZAyAxMv9zUED5GmAxA/ilRv9QopD2UFxNAAOZOv48YPT2M9xhANZlFv3vZNj07Ux1A9602v19Emz27gB9A12kkvzAtCj45Kx9Ar5QRv9EfWj6zXxxAbwwBvz+nmD4VixdA46rqvqg5wT6LOEm/K207wEsEGj+r7Wa/pmQ7wInQDD8VAIO/3jw9wKhu9j5AMYK/1xU+wChh9j5M/4y/hbFAwCRH4j6lMIu/hEhBwAtF4j7Zr42/5E1JwItSyj5OuIu/zEZJwFBQyj4uOom/ChBPwCU8wT5I3Im/5Q5KwFckjj5VT4i/gLlNwJjglD5VvYK/EeVMwAHBLD7K4IS/9+dJwLTlDD5Y4om/HjRBwDgVoT4DmYm/aydBwBcRdT40ZYW/SPhBwEuQsT3WO4i/zVw8wBecUT7vyoK/g089wCXPNT0L1Xm/e8E2wHLArrvoiIS/euE1wOKxLz5WgIG/Ol00wNRhdT6HwWS/eHwzwENZkD5qMUi//+cywOwzjz5+5DK/z0gzwFPNjD645Cy/rwk8wPLOFT+BXw+/+ig+wIrNAz8riXy//cA2wBnjQ7zO/3O/1a42wNwOjb15I2O/kq42wCkI3r3tC1C/T642wDc4Ab6g+zq/ya02wEC/772iJi6/l602wFoQqr3o9h6/Er82wADkhLw6QRe/COVMwB/2Ij4SvCG/tr02wKHXH7wjFhG/93I8wIY4tj0XDw+/MbM7wAXhWj4cCxq/nN01wKN5ED6cag2/NShCwAAeAT5cygm/YMhBwOCgfT69NxK/6ztJwEDBFT4hdg6//7FJwBHDjj6IYxG/8MNNwDM0jj5jnAu/okFPwOhozT6j6gO/gSFKwNyb1z5SKgG/VORCwK7w7j4N4Qi/hGVCwOyhtT57oB2/Ozk0wBKgVj58YGe/mdY5wNpz0T743me/XDw7wGnh6j4MWEq/0o46wDLm9j5pcUq/xlA5wJ3z2z7KU4K/oKU7wAfvsz4xQ4O/xAg9wAFrvT4IWYK/vMk8wGhAzT58Rkq/n8c3wG9Jxj6EuUm/Et41wJxSrj7RBGq/DCA2wIbipj5Numm//Dc4wJaSvT4k8IG/8rQ4wMyyjz6SzoK/MQk6wHapkT4rNIK/s5U6wI3Uoz69cC+/cJo3wAeVwD6HMjC/H9o1wM9ppj4/4Ri/eV05wO2asD7CaRm/0Lk4wMiVmj48wRq/A5k3wAXClj47xBO/8DM8wBOC5T5lbRO/inI8wInS1j6dRha/jNY6wNqOyT6jrS6/G/Q4wJS82j7h8S2/fSU6wPLQ9z7er/6/tmb7vmOayT7eVPq/e/cPv+NUqz7j//q/o8wiv5Jbgz53SwDA81Qzv3aILz4UIAXA8gs/v0zHvD2BQQvAFylEv5wZ/TzPwBHAGeVBv+6zSrvLoBfATpg4v/tal7t2/BvAEK0pv8aH2Tz6KR7A8WgXv/J3rz141B3AyZMEv5uuJz7uCBvAEhfovkjdfj5QNBbAN6nQvi4BqD7oEhDAym7GvkZhxz6akwnAx/bKvmnI2D6eswPAXJDdvkuQ2T6paRHAl3Dgvr+Z4D5b6grAcvjkvgQB8j5jCgXAKZL3vsXI8j60rgDAQrQKv/7S4j5gAv2/Yvgcv36NxD5krf2/is0vvwyUnD48ogHA2lVAv2r5YT7VdgbAyAxMv9zUED5GmAzA/ilRv9QopD2UFxPAAOZOv48YPT2M9xjANZlFv3vZNj07Ux3A9602v19Emz27gB/A12kkvzAtCj45Kx/Ar5QRv9EfWj6zXxzAbwwBvz+nmD4VixfA46rqvqg5wT6Y2707IjnBP4EkQL/Nrmu+c2e+P9xiMr8a3C6/lUmzP1Ka9b4psES/jEiiP9Iceb5hU0u/XaiOP2ZsaD0C1Su/Ja9aP3r+3D4bneu+44wJPwDhKz+rr647dEY8P57TGD8GoIG+SSsuPznUFz+P+mu+dckYP483ET81Qg++hPT8PokjIz9zKlm+zuGCPp+uQj88aVG+fjfFPmh2OT/j4GK+F/ObPltAQD+856i+jewSP6QaGj9d3ry+8aEcPzSGIT8BM+++J9q5P+KSH7+9/RG/Cr5lPyWtwD7KwRQ/R8dlP6vrwD4SFvU+9N25PzF5H7/8U8I+56ccP4SaIT/yYa4+7fESP+0sGj9z2mM+gQixPidMQD82ck0+CMzTPuCBOT9zEGQ+wOiCPmu6Qj8BNBo+Gvn8PncrIz+/9HY+Qs0YP1ZEET++F4c+ey8uPz/iFz/kEPE+W5QJP0z6Kz/Eli4/BrpaPwJI3T5TJU4/ya6OP5ojaz3vj0c/vk6iPwx1eL6gxjE/Hk+zP1lP9b5Ngnc+UWm+PxVWMr+GxiM+VdkzP05DGD/aHj2+eJsDQFwAKj7jNlq+UwUDQLiPHD6FeW++LIMBQHWUEz61NHe+Afn+P952ET4hVm++g/j6Py7IFj5I+Vm+dhf4PzsbIj6M1zy+SBv3P4xnMD7Gvx++kUf4PzDYPT7hfAq+4Ev7P3TTRj6xwQK+Pln/P8fwSD6IoAq+2qwBQLqfQz5h/R++YB0DQK1MOD4m4yi+BW0GQNQq+j2ER1u+9WgFQDCcyz09DYC+HcwCQLt/rD1Ev4a+dZP+P9QrpT3f3H++CaX3P0aXtz3T3Fq+ZajyPwPS3j3xZyi+ofPwP7IsCD4lB+y9wvvyP0d0Hz6zYKK9YTX4P4ICLz6WmIe9Ljr/P7isMj7o26K9TRQDQP92KT6H3Oy9n5IFQF7ZFT5oJQ2++FMHQBwkhD03VUe+qicGQBrDHD2+2nG+YCMDQA/UqTyPqIC+Aiz+P6D6hzy0k3G+Fyv2P40N3TwB2ka+7GjwPy4fST3Mlgy+l3DuP9vAnT37zaS9MsnwP/GD0z3XhR+9yNH2P3pw9z2XVsO8hez+P0/m/z0BoiC9uHYDQJSh6j1lxKW9zVcGQGFVvT0NqeK9dxIGQGhYDDyiuCO+Zw4FQLQcaLzbi0i+k3ECQDCA8Lzp71W+Wd79P+bnBr1ATki+7e/2PxwkxLzwTSO+SfPxP0TcnLscsuG9hT7wP47onjzM03m9nUbyP46RLD3QDc28RYD3P3nKaj222kO8EoX+P0dyeT2k+s68v7kCQG+cVD2Rfnu9ETgFQOolBj3JyLm9oP4CQJuq+7xi+fO9e2gCQNGWM71TPw++VOYAQOCDV72D+ha+Ub/9Pzz6X72sGw++0775P/q0Sr0sfvO9xt32P8doHb20Orm9j+H1P+xsyLxEFX694Q33P8zTObyvCSm9MBL6P3JtKLvvHAq9jh/+P/phBLpKmCm9BhABQEq2uruvC3+9jYACQPpFibxvnqq9I9f9P5ZaL73aHj0+eJsDQFwAKj7jNlo+UwUDQLiPHD6FeW8+LIMBQHWUEz61NHc+Afn+P952ET4hVm8+g/j6Py7IFj5I+Vk+dhf4PzsbIj6M1zw+SBv3P4xnMD7Gvx8+kUf4PzDYPT7hfAo+4Ev7P3TTRj6xwQI+Pln/P8fwSD6IoAo+2qwBQLqfQz5h/R8+YB0DQK1MOD4m4yg+BW0GQNQq+j2ER1s+9WgFQDCcyz09DYA+HcwCQLt/rD1Ev4Y+dZP+P9QrpT3f3H8+CaX3P0aXtz3T3Fo+ZajyPwPS3j3xZyg+ofPwP7IsCD4lB+w9wvvyP0d0Hz6zYKI9YTX4P4ICLz6WmIc9Ljr/P7isMj7o26I9TRQDQP92KT6H3Ow9n5IFQF7ZFT5oJQ0++FMHQBwkhD03VUc+qicGQBrDHD2+2nE+YCMDQA/UqTyPqIA+Aiz+P6D6hzy0k3E+Fyv2P40N3TwB2kY+7GjwPy4fST3Mlgw+l3DuP9vAnT37zaQ9MsnwP/GD0z3XhR89yNH2P3pw9z2XVsM8hez+P0/m/z0BoiA9uHYDQJSh6j1lxKU9zVcGQGFVvT0NqeI9dxIGQGhYDDyiuCM+Zw4FQLQcaLzbi0g+k3ECQDCA8Lzp71U+Wd79P+bnBr1ATkg+7e/2PxwkxLzwTSM+SfPxP0TcnLscsuE9hT7wP47onjzM03k9nUbyP46RLD3QDc08RYD3P3nKaj222kM8EoX+P0dyeT2k+s48v7kCQG+cVD2Rfns9ETgFQOolBj3JyLk9oP4CQJuq+7xi+fM9e2gCQNGWM71TPw8+VOYAQOCDV72D+hY+Ub/9Pzz6X72sGw8+0775P/q0Sr0sfvM9xt32P8doHb20Ork9j+H1P+xsyLxEFX494Q33P8zTObyvCSk9MBL6P3JtKLvvHAo9jh/+P/phBLpKmCk9BhABQEq2uruvC389jYACQPpFibxvnqo9I9f9P5ZaL70TAAAADgAAABsAAAAbAAAAIwAAABMAAABiAAAAZQAAAGQAAABkAAAALQAAAGIAAABjAAAAXgAAAGAAAABgAAAAYgAAAGMAAABfAAAAYAAAAF4AAABeAAAAXQAAAF8AAABZAAAAVAAAAFUAAABRAAAAUgAAAEwAAABMAAAATQAAAFEAAABGAAAATQAAAEwAAABMAAAARwAAAEYAAABKAAAASwAAAEkAAABJAAAAQgAAAEoAAABFAAAAPwAAAD4AAAA+AAAARAAAAEUAAAA/AAAAOAAAADkAAAA5AAAAPgAAAD8AAAA2AAAAOgAAADkAAAA5AAAANQAAADYAAAAMAAAANQAAADQAAAA0AAAADQAAAAwAAAAwAAAAIQAAAB8AAAAfAAAALwAAADAAAAAmAAAAKgAAACkAAAApAAAAJAAAACYAAAAjAAAAJAAAACIAAAAiAAAAEwAAACMAAAAeAAAAGwAAABoAAAAaAAAAHQAAAB4AAAAXAAAAGAAAABIAAAASAAAABwAAAAYAAAAGAAAAEQAAABIAAAACAAAAAwAAAAAAAAAAAAAAAQAAAAIAAAAPAAAAAgAAAAEAAAABAAAADgAAAA8AAAAUAAAADwAAAA4AAAAOAAAAEwAAABQAAAAIAAAABwAAABIAAAASAAAAGAAAAAgAAAAdAAAAGgAAABkAAAAZAAAAHAAAAB0AAAATAAAAIgAAACAAAAAgAAAAFAAAABMAAAAkAAAAKQAAACgAAAAoAAAAIgAAACQAAAAvAAAAHwAAACAAAAAgAAAAJwAAAC8AAAAhAAAAMQAAABcAAAAXAAAAFgAAACEAAAA1AAAAOQAAADgAAAA4AAAANAAAADUAAAA+AAAAOQAAADoAAAA6AAAAPQAAAD4AAABDAAAAPQAAAEAAAABAAAAAQgAAAEMAAABCAAAASQAAAEgAAABIAAAAQwAAAEIAAAA3AAAACgAAAAkAAABLAAAAUAAAAE8AAABPAAAASQAAAEsAAABQAAAAVQAAAFQAAABUAAAATwAAAFAAAABXAAAAUQAAAFMAAABTAAAAWAAAAFcAAABfAAAAWwAAAFoAAAAlAAAAHgAAAF4AAABeAAAAYwAAACUAAABkAAAAMgAAAC4AAAAuAAAALQAAAGQAAABdAAAAHQAAABwAAAAcAAAAXAAAAF0AAAAyAAAAMwAAADAAAAAyAAAAMAAAAC8AAAAyAAAALwAAACcAAAAyAAAAJwAAACgAAAAyAAAAKAAAACkAAAAyAAAAKQAAACoAAAAyAAAAKgAAACwAAAAyAAAALAAAAC4AAAAjAAAAGwAAAB4AAAAeAAAAJQAAACMAAABjAAAAYgAAAC0AAAAtAAAAKwAAAGMAAABgAAAAXwAAAFoAAABaAAAAYQAAAGAAAABeAAAAHgAAAB0AAAAdAAAAXQAAAF4AAABUAAAAWQAAAFgAAABYAAAAUwAAAFQAAABOAAAAUwAAAFEAAABRAAAATQAAAE4AAABIAAAATgAAAE0AAABNAAAARgAAAEgAAABCAAAAQAAAAEEAAABBAAAASgAAAEIAAABGAAAARwAAAEUAAABFAAAARAAAAEYAAAA7AAAAQAAAAD0AAAA9AAAAOgAAADsAAAA3AAAAOwAAADoAAAA6AAAANgAAADcAAAALAAAANgAAADUAAAA1AAAADAAAAAsAAAAwAAAAMwAAADEAAAAxAAAAIQAAADAAAAArAAAALAAAACoAAAAqAAAAJgAAACsAAAAlAAAAJgAAACQAAAAkAAAAIwAAACUAAAAUAAAAIAAAAB8AAAAfAAAAFQAAABQAAAAaAAAAAQAAAAAAAAAAAAAAGQAAABoAAAAWAAAAEQAAABAAAAAQAAAAFQAAABYAAAARAAAABgAAAAQAAAAEAAAAEAAAABEAAAAEAAAABQAAAAMAAAADAAAAAgAAAAQAAAAQAAAABAAAAAIAAAACAAAADwAAABAAAAAVAAAAEAAAAA8AAAAPAAAAFAAAABUAAAAbAAAADgAAAAEAAAABAAAAGgAAABsAAAAhAAAAFgAAABUAAAAVAAAAHwAAACEAAAAiAAAAKAAAACcAAAAnAAAAIAAAACIAAAAtAAAALgAAACwAAAAsAAAAKwAAAC0AAAARAAAAFgAAABcAAAAXAAAAEgAAABEAAAA2AAAACwAAAAoAAAAKAAAANwAAADYAAAAJAAAAPAAAADsAAAA7AAAANwAAAAkAAAA8AAAAQQAAAEAAAABAAAAAOwAAADwAAABDAAAASAAAAEYAAABGAAAARAAAAEMAAABEAAAAPgAAAD0AAAA9AAAAQwAAAEQAAABJAAAATwAAAE4AAABOAAAASAAAAEkAAABPAAAAVAAAAFMAAABTAAAATgAAAE8AAABRAAAAVwAAAFYAAABWAAAAUgAAAFEAAABdAAAAXAAAAFsAAABbAAAAXwAAAF0AAAAlAAAAYwAAACsAAAArAAAAJgAAACUAAABiAAAAYAAAAGEAAABhAAAAZQAAAGIAAABrAAAAagAAAH8AAAB/AAAAfgAAAGsAAAB7AAAAfAAAAIEAAACBAAAAggAAAHsAAAB9AAAAfgAAAH8AAAB/AAAAgAAAAH0AAACCAAAAgwAAAHoAAAB6AAAAewAAAIIAAAB8AAAAfQAAAIAAAACAAAAAgQAAAHwAAACFAAAAhgAAAI0AAACNAAAAhAAAAIUAAACHAAAAiAAAAIsAAACLAAAAjAAAAIcAAACGAAAAhwAAAIwAAACMAAAAjQAAAIYAAACIAAAAiQAAAIoAAACKAAAAiwAAAIgAAABoAAAAZwAAAH0AAAB9AAAAfAAAAGgAAAB5AAAAaAAAAHwAAAB8AAAAewAAAHkAAABnAAAAawAAAH4AAAB+AAAAfQAAAGcAAAB2AAAAeQAAAHsAAAB7AAAAegAAAHYAAABmAAAAaQAAAIEAAACBAAAAgAAAAGYAAABpAAAAeAAAAIIAAACCAAAAgQAAAGkAAABqAAAAZgAAAIAAAACAAAAAfwAAAGoAAAB4AAAAdwAAAIMAAACDAAAAggAAAHgAAAB1AAAAdAAAAIoAAACKAAAAiQAAAHUAAAB3AAAAdgAAAHoAAAB6AAAAgwAAAHcAAABxAAAAcAAAAIUAAACFAAAAhAAAAHEAAABwAAAAbgAAAIYAAACGAAAAhQAAAHAAAABzAAAAdQAAAIkAAACJAAAAiAAAAHMAAABuAAAAbQAAAIcAAACHAAAAhgAAAG4AAABtAAAAcwAAAIgAAACIAAAAhwAAAG0AAABsAAAAbwAAAI0AAACNAAAAjAAAAGwAAAByAAAAbAAAAIwAAACMAAAAiwAAAHIAAABvAAAAcQAAAIQAAACEAAAAjQAAAG8AAAB0AAAAcgAAAIsAAACLAAAAigAAAHQAAACxAAAAqQAAAJwAAACcAAAAoQAAALEAAAC7AAAA8gAAAPMAAADzAAAA8AAAALsAAADwAAAA7gAAAOwAAADsAAAA8QAAAPAAAADrAAAA7AAAAO4AAADuAAAA7QAAAOsAAADjAAAA4gAAAOcAAADbAAAA2gAAAOAAAADgAAAA3wAAANsAAADVAAAA2gAAANsAAADbAAAA1AAAANUAAADQAAAA1wAAANkAAADZAAAA2AAAANAAAADSAAAAzAAAAM0AAADNAAAA0wAAANIAAADMAAAAxwAAAMYAAADGAAAAzQAAAMwAAADDAAAAxwAAAMgAAADIAAAAxAAAAMMAAACbAAAAwgAAAMMAAADDAAAAmgAAAJsAAAC9AAAArQAAAK8AAACvAAAAvgAAAL0AAACyAAAAtwAAALgAAAC4AAAAtAAAALIAAAChAAAAsAAAALIAAACyAAAAsQAAAKEAAACrAAAAqAAAAKkAAACpAAAArAAAAKsAAACgAAAApgAAAKUAAACfAAAAlAAAAJUAAACVAAAAoAAAAJ8AAACPAAAAjgAAAJEAAACRAAAAkAAAAI8AAACcAAAAjwAAAJAAAACQAAAAnQAAAJwAAAChAAAAnAAAAJ0AAACdAAAAogAAAKEAAACmAAAAoAAAAJUAAACVAAAAlgAAAKYAAACqAAAApwAAAKgAAACoAAAAqwAAAKoAAACiAAAArgAAALAAAACwAAAAoQAAAKIAAACwAAAAtgAAALcAAAC3AAAAsgAAALAAAAC1AAAArgAAAK0AAACtAAAAvQAAALUAAACkAAAApQAAAL8AAAC/AAAArwAAAKQAAADCAAAAxgAAAMcAAADHAAAAwwAAAMIAAADLAAAAyAAAAMcAAADHAAAAzAAAAMsAAADQAAAAzgAAAMsAAADLAAAA0QAAANAAAADRAAAA1gAAANcAAADXAAAA0AAAANEAAACXAAAAmAAAAMUAAADXAAAA3QAAAN4AAADeAAAA2QAAANcAAADdAAAA4gAAAOMAAADjAAAA3gAAAN0AAADmAAAA4QAAAN8AAADfAAAA5QAAAOYAAADoAAAA6QAAAO0AAADxAAAA7AAAAKwAAACsAAAAswAAAPEAAAC7AAAAvAAAAMAAAADAAAAA8gAAALsAAADqAAAAqgAAAKsAAACrAAAA6wAAAOoAAAC8AAAAugAAALgAAAC8AAAAuAAAALcAAAC8AAAAtwAAALYAAAC8AAAAtgAAALUAAAC8AAAAtQAAAL0AAAC8AAAAvQAAAL4AAAC8AAAAvgAAAMEAAAC8AAAAwQAAAMAAAACzAAAArAAAAKkAAACpAAAAsQAAALMAAAC5AAAAuwAAAPAAAADwAAAA8QAAALkAAADvAAAA6AAAAO0AAADtAAAA7gAAAO8AAADrAAAAqwAAAKwAAACsAAAA7AAAAOsAAADhAAAA5gAAAOcAAADnAAAA4gAAAOEAAADbAAAA3wAAAOEAAADhAAAA3AAAANsAAADUAAAA2wAAANwAAADcAAAA1gAAANQAAADYAAAAzwAAAM4AAADOAAAA0AAAANgAAADSAAAA0wAAANUAAADVAAAA1AAAANIAAADIAAAAywAAAM4AAADOAAAAyQAAAMgAAADEAAAAyAAAAMkAAADJAAAAxQAAAMQAAACaAAAAwwAAAMQAAADEAAAAmQAAAJoAAACvAAAAvwAAAMEAAADBAAAAvgAAAK8AAAC0AAAAuAAAALoAAAC6AAAAuQAAALQAAACxAAAAsgAAALQAAAC0AAAAswAAALEAAACjAAAArQAAAK4AAACuAAAAogAAAKMAAACnAAAAjgAAAI8AAACPAAAAqAAAAKcAAACjAAAAngAAAJ8AAACfAAAApAAAAKMAAACeAAAAkgAAAJQAAACUAAAAnwAAAJ4AAACQAAAAkQAAAJMAAACTAAAAkgAAAJAAAACdAAAAkAAAAJIAAACSAAAAngAAAJ0AAACiAAAAnQAAAJ4AAACeAAAAowAAAKIAAACoAAAAjwAAAJwAAACcAAAAqQAAAKgAAACtAAAAowAAAKQAAACkAAAArwAAAK0AAACuAAAAtQAAALYAAAC2AAAAsAAAAK4AAAC5AAAAugAAALwAAAC8AAAAuwAAALkAAACgAAAApQAAAKQAAACkAAAAnwAAAKAAAADFAAAAmAAAAJkAAACZAAAAxAAAAMUAAADFAAAAyQAAAMoAAADKAAAAlwAAAMUAAADJAAAAzgAAAM8AAADPAAAAygAAAMkAAADSAAAA1AAAANYAAADWAAAA0QAAANIAAADRAAAAywAAAMwAAADMAAAA0gAAANEAAADWAAAA3AAAAN0AAADdAAAA1wAAANYAAADcAAAA4QAAAOIAAADiAAAA3QAAANwAAADgAAAA5AAAAOUAAADlAAAA3wAAAOAAAADtAAAA6QAAAOoAAADqAAAA6wAAAO0AAAC0AAAAuQAAAPEAAADxAAAAswAAALQAAADzAAAA7wAAAO4AAADuAAAA8AAAAPMAAAAMAQAADQEAAPgAAAD4AAAA+QAAAAwBAAAQAQAADwEAAAoBAAAKAQAACQEAABABAAAOAQAADQEAAAwBAAAMAQAACwEAAA4BAAAJAQAACAEAABEBAAARAQAAEAEAAAkBAAAPAQAADgEAAAsBAAALAQAACgEAAA8BAAASAQAAGwEAABQBAAAUAQAAEwEAABIBAAAaAQAAGQEAABYBAAAWAQAAFQEAABoBAAAbAQAAGgEAABUBAAAVAQAAFAEAABsBAAAZAQAAGAEAABcBAAAXAQAAFgEAABkBAAAKAQAACwEAAPUAAAD1AAAA9gAAAAoBAAAJAQAACgEAAPYAAAD2AAAABwEAAAkBAAALAQAADAEAAPkAAAD5AAAA9QAAAAsBAAAIAQAACQEAAAcBAAAHAQAABAEAAAgBAAAOAQAADwEAAPcAAAD3AAAA9AAAAA4BAAAPAQAAEAEAAAYBAAAGAQAA9wAAAA8BAAANAQAADgEAAPQAAAD0AAAA+AAAAA0BAAAQAQAAEQEAAAUBAAAFAQAABgEAABABAAAXAQAAGAEAAAIBAAACAQAAAwEAABcBAAARAQAACAEAAAQBAAAEAQAABQEAABEBAAASAQAAEwEAAP4AAAD+AAAA/wAAABIBAAATAQAAFAEAAPwAAAD8AAAA/gAAABMBAAAWAQAAFwEAAAMBAAADAQAAAQEAABYBAAAUAQAAFQEAAPsAAAD7AAAA/AAAABQBAAAVAQAAFgEAAAEBAAABAQAA+wAAABUBAAAaAQAAGwEAAP0AAAD9AAAA+gAAABoBAAAZAQAAGgEAAPoAAAD6AAAAAAEAABkBAAAbAQAAEgEAAP8AAAD/AAAA/QAAABsBAAAYAQAAGQEAAAABAAAAAQAAAgEAABgBAABLAQAAUAEAAE8BAABPAQAATAEAAEsBAABHAQAAVAEAAFMBAABTAQAASAEAAEcBAABOAQAATwEAAFABAABOAQAAUAEAAFEBAABOAQAAUQEAAFIBAABOAQAAUgEAAFMBAABOAQAAUwEAAFQBAABOAQAAVAEAAFUBAABOAQAAVQEAAFYBAABOAQAAVgEAAFcBAAA3AQAAPAEAADsBAAA7AQAAOAEAADcBAAAzAQAAQAEAAD8BAAA/AQAANAEAADMBAAA6AQAAOwEAADwBAAA6AQAAPAEAAD0BAAA6AQAAPQEAAD4BAAA6AQAAPgEAAD8BAAA6AQAAPwEAAEABAAA6AQAAQAEAAEEBAAA6AQAAQQEAAEIBAAA6AQAAQgEAAEMBAAAjAQAALQEAAC4BAAAuAQAAJAEAACMBAAAfAQAAKQEAACoBAAAqAQAAIAEAAB8BAAAvAQAALgEAAC0BAAAvAQAALQEAACwBAAAvAQAALAEAACsBAAAvAQAAKwEAACoBAAAvAQAAKgEAACkBAAAvAQAAKQEAACgBAAAvAQAAKAEAACcBAAAvAQAAJwEAACYBAAAcAQAAJgEAACcBAAAnAQAAHQEAABwBAAAgAQAAKgEAACsBAAArAQAAIQEAACABAAAkAQAALgEAAC8BAAAvAQAAJQEAACQBAAAwAQAAQwEAAEIBAABCAQAAMQEAADABAAA0AQAAPwEAAD4BAAA+AQAANQEAADQBAAA4AQAAOwEAADoBAAA6AQAAOQEAADgBAABEAQAAVwEAAFYBAABWAQAARQEAAEQBAABIAQAAUwEAAFIBAABSAQAASQEAAEgBAABMAQAATwEAAE4BAABOAQAATQEAAEwBAABKAQAAUQEAAFABAABQAQAASwEAAEoBAABGAQAAVQEAAFQBAABUAQAARwEAAEYBAABEAQAARQEAAEYBAABEAQAARgEAAEcBAABEAQAARwEAAEgBAABEAQAASAEAAEkBAABEAQAASQEAAEoBAABEAQAASgEAAEsBAABEAQAASwEAAEwBAABEAQAATAEAAE0BAAA2AQAAPQEAADwBAAA8AQAANwEAADYBAAAyAQAAQQEAAEABAABAAQAAMwEAADIBAAAwAQAAMQEAADIBAAAwAQAAMgEAADMBAAAwAQAAMwEAADQBAAAwAQAANAEAADUBAAAwAQAANQEAADYBAAAwAQAANgEAADcBAAAwAQAANwEAADgBAAAwAQAAOAEAADkBAAAiAQAALAEAAC0BAAAtAQAAIwEAACIBAAAeAQAAKAEAACkBAAApAQAAHwEAAB4BAAAcAQAAHQEAAB4BAAAcAQAAHgEAAB8BAAAcAQAAHwEAACABAAAcAQAAIAEAACEBAAAcAQAAIQEAACIBAAAcAQAAIgEAACMBAAAcAQAAIwEAACQBAAAcAQAAJAEAACUBAAAdAQAAJwEAACgBAAAoAQAAHgEAAB0BAAAhAQAAKwEAACwBAAAsAQAAIgEAACEBAAAlAQAALwEAACYBAAAmAQAAHAEAACUBAAAxAQAAQgEAAEEBAABBAQAAMgEAADEBAAA1AQAAPgEAAD0BAAA9AQAANgEAADUBAAA5AQAAOgEAAEMBAABDAQAAMAEAADkBAABFAQAAVgEAAFUBAABVAQAARgEAAEUBAABJAQAAUgEAAFEBAABRAQAASgEAAEkBAABNAQAATgEAAFcBAABXAQAARAEAAE0BAAClAQAApAEAAIYBAACGAQAAnwEAAKUBAACeAQAAgwEAAIIBAACeAQAAggEAAIQBAACeAQAAhAEAAKEBAACIAQAAhwEAAJsBAACbAQAAkwEAAIgBAACQAQAAlgEAAJcBAACXAQAAkQEAAJABAACJAQAAkgEAAJABAACQAQAAigEAAIkBAAB+AQAAfQEAAHwBAAB+AQAAfAEAAHABAABwAQAAbwEAAH4BAAB0AQAAcwEAAHkBAAB5AQAAeAEAAHQBAAB4AQAAdgEAAFsBAABbAQAAdwEAAHgBAAB6AQAAfQEAAHIBAAByAQAAWAEAAHoBAABmAQAAbAEAAG0BAABtAQAAZwEAAGYBAABeAQAAZwEAAGgBAABoAQAAXQEAAF4BAABiAQAAYwEAAGQBAABkAQAAYQEAAGIBAABfAQAAZgEAAGcBAABnAQAAXgEAAF8BAACAAQAAgQEAAGwBAABsAQAAZgEAAIABAABuAQAAbwEAAHABAABwAQAAaAEAAG4BAABxAQAAcwEAAHQBAAB0AQAAcgEAAHEBAAB9AQAAegEAAHsBAAB7AQAAfAEAAH0BAACMAQAAjgEAAGMBAABjAQAAYgEAAIwBAACIAQAAkwEAAJIBAACSAQAAiQEAAIgBAACSAQAAmAEAAJYBAACWAQAAkAEAAJIBAACeAQAAnAEAAJ0BAACdAQAAnwEAAJ4BAACiAQAAowEAAHYBAAB2AQAAdQEAAKIBAACjAQAAogEAAKABAACgAQAAoQEAAKMBAACaAQAAmwEAAKUBAAClAQAAnQEAAJoBAAClAQAAnwEAAJ0BAACPAQAAjQEAAIsBAACLAQAAkQEAAI8BAACbAQAAhwEAAKQBAACkAQAApQEAAJsBAACFAQAAowEAAKEBAAChAQAAhAEAAIUBAACeAQAAoQEAAKABAACgAQAAnAEAAJ4BAACTAQAAmwEAAJoBAACaAQAAmQEAAJMBAACPAQAAlQEAAJQBAACUAQAAjgEAAI8BAACKAQAAkAEAAJEBAACRAQAAiwEAAIoBAACBAQAAgAEAAGUBAABlAQAAawEAAIEBAAB1AQAAdgEAAHgBAAB4AQAAeQEAAHUBAAB9AQAAfgEAAHEBAABxAQAAcgEAAH0BAABnAQAAbQEAAG4BAABuAQAAaAEAAGcBAABjAQAAaQEAAGoBAABqAQAAZAEAAGMBAABhAQAAZAEAAGUBAABlAQAAYAEAAGEBAAB/AQAAgAEAAGYBAABmAQAAXwEAAH8BAABkAQAAagEAAGsBAABrAQAAZQEAAGQBAABoAQAAcAEAAFwBAABcAQAAXQEAAGgBAAB0AQAAWgEAAFkBAAB0AQAAWQEAAFgBAAB0AQAAWAEAAHIBAABaAQAAdAEAAHgBAAB4AQAAdwEAAFoBAAB8AQAAewEAAFwBAABcAQAAcAEAAHwBAACAAQAAfwEAAGABAABgAQAAZQEAAIABAACNAQAAjwEAAI4BAACOAQAAjAEAAI0BAACOAQAAlAEAAGkBAABpAQAAYwEAAI4BAACTAQAAmQEAAJgBAACYAQAAkgEAAJMBAACDAQAAngEAAJ8BAACfAQAAhgEAAIMBAACFAQAAWwEAAHYBAAB2AQAAowEAAIUBAACVAQAAjwEAAJEBAACRAQAAlwEAAJUBAACrAQAArAEAAKYBAACvAQAAsAEAAKYBAACnAQAAqAEAAKYBAACyAQAApwEAAKYBAACuAQAArwEAAKYBAACqAQAAqwEAAKYBAACsAQAArQEAAKYBAACwAQAAsQEAAKYBAACoAQAAqQEAAKYBAACxAQAAsgEAAKYBAACtAQAArgEAAKYBAACpAQAAqgEAAKYBAACzAQAAuAEAALcBAACzAQAAvAEAALsBAACzAQAAtAEAAL8BAACzAQAAtQEAALQBAACzAQAAvQEAALwBAACzAQAAuQEAALgBAACzAQAAtwEAALYBAACzAQAAuwEAALoBAACzAQAAvwEAAL4BAACzAQAAtgEAALUBAACzAQAAvgEAAL0BAACzAQAAugEAALkBAACLOEk/K207wEsEGj9WnUk/zAw6wLwFGj/NPWg/Uis6wLrYDD+r7WY/pmQ7wInQDD8VAIM/3jw9wKhu9j5AMYI/1xU+wChh9j5M/4w/hbFAwCRH4j7Zr40/5E1JwItSyj4uOok/ChBPwCU8wT5VvYI/EeVMwAHBLD7K4IQ/9+dJwLTlDD40ZYU/SPhBwEuQsT3vyoI/g089wCXPNT0L1Xk/e8E2wHLArru700U/RWY6wJ61Nz+ESmg/ERg7wK9cLz/Rk4I/Hck9wP6aKD96iow/BFpEwCUkFj+x3Iw/FypMwG9ICz/8VEU/aAc8wFltRj8yWmc/UKg9wEPLQj8w8X8/rb1BwC/EQj+78oU/+3ZKwM6nQj9dv4Q/q0FPwNCbQj/deok/nkFPwHoXBz+45Cw/rwk8wPLOFT9Ycik/kwA7wMwJFj8+dyo/gUI7wA7cNT+BXw8/+ig+wIrNAz+8Hws/Ets8wKrVAz+/Kw4/CRY+wIDxMD9v1G4/kblFwHhHWj+y1V0/JnNCwCxJWj8epXY/zCVKwOFGWj+DbEU/xY9AwB9IWj/BAiw/t+08wHvZQj9BnC8/Yf5CwBdGWj/fTxU/2ZVAwDfGQj9O8CE/eepFwOFCWj9IwlI/tcJHwHOdYj9LIEU/gUBGwGecYj+kxzc/RiZIwCyaYj/oFjI/2XhJwKaZYj/baxk/ByRJwBBAWj/bay0/xjZLwA+ZYj/ScBY/YTJMwGQ8Wj//ris/BMhNwAOYYj/XwVk/CthJwJmcYj/5TFo/CXFMwJ2bYj9ljnE/jUFPwD5BWj+lhS8/okFPwAuWYj/DKVc/xEFPwEaYYj8riXw//cA2wBnjQ7zsooQ/mZ09wNTuFz28W4c/FVdCwGB1pD0gCYc/Nj5KwLg8Bj7O/3M/1a42wNwOjb2dSHw/tks9wHk8bb1fe4I/DsBDwH5SLb3/QIE/9btKwEiHh7s7jXw/Q+VMwMo0Wj20x2c/CHJDwM14C74UIWU/W0Q9wL5QAL55I2M/kq42wCkI3r0LR2g/7ZxJwHIXEb7JOmQ/JuVMwCkF3b0ibEw/NEhJwLN7Mr43NE0/prlCwGzqLL6fkE0/7bY9wKGFFL7tC1A/T642wDc4Ab58DTU/6xk9wOfkBb6g+zo/ya02wEC/77074y8/hzZCwIjYIL5SgSs/xQBJwJLKFL4Ji0o/HeVMwBO4Fb678S4/9+RMwD8C/72iJi4/l602wFoQqr3v4Ro/uyg9wI/HjL0A5RM/tB1CwA+ZMr1jtRU/k1RJwGpsL7zbpRk/AOVMwPQ3oTpBnQo/F9k7wI9srj3o9h4/Er82wADkhLwAGwg/yehBwKSO7j2OPg4/CFpJwCh+DD46QRc/COVMwB/2Ij4SvCE/tr02wKHXH7wjFhE/93I8wIY4tj2cag0/NShCwAAeAT69NxI/6ztJwEDBFT5jnAs/okFPwOhozT6j6gM/gSFKwNyb1z5SKgE/VORCwK7w7j7CTPM+NjxCwPwB7z47/vM+51ZDwFZ9Ij938/w+FktKwEZ51j4Rp/M+DjFKwMEfFj+H4Ag/q0FPwEWEEz98ZAc/6DBLwOilQj/tSgc/OdVEwIW0Qj/YvB4/nkFPwFQ4Wj+kbg8/hUFPwOGXQj98YGc/mdY5wNpz0T743mc/XDw7wGnh6j4MWEo/0o46wDLm9j5pcUo/xlA5wJ3z2z7KU4I/oKU7wAfvsz4IWYI/vMk8wGhAzT58Rko/n8c3wG9Jxj6EuUk/Et41wJxSrj7RBGo/DCA2wIbipj5Numk//Dc4wJaSvT4k8IE/8rQ4wMyyjz4rNII/s5U6wI3Uoz69cC8/cJo3wAeVwD6HMjA/H9o1wM9ppj4/4Rg/eV05wO2asD48wRo/A5k3wAXClj47xBM/8DM8wBOC5T6dRhY/jNY6wNqOyT6jrS4/G/Q4wJS82j7h8S0/fSU6wPLQ9z5TBBQ/U3U7wIY84j7oMS4/4GY5wKiL9D4kmEo/LdA5wOig8z4QH2g/t306wP6b5z4MeYI/IAs8wB77yT4RboI/3zU7wKyNuT4UBGc/6IU5wGTl1z6DoUo/bvs4wKPI4j7FqS4/ZJQ4wDeM4j4O3BU/r306wGEb0T7vO4I/ob05wDtUoz4OFoI/63E4wJFklj6UUGo/Bd01wCmUrT42BUo/B5s1wGAEtT5KfjA/GJc1wHEbrT7+DBs/+FU3wMlznT7njxk/g244wP4srj6RuC8/T7I2wNrjvT5oWko/JNU2wN6SxD7NWmo/3UA3wKM/vD6LOEm/K207wEsEGj9WnUm/zAw6wLwFGj/NPWi/Uis6wLrYDD+r7Wa/pmQ7wInQDD8VAIO/3jw9wKhu9j5AMYK/1xU+wChh9j5M/4y/hbFAwCRH4j7Zr42/5E1JwItSyj4uOom/ChBPwCU8wT5VvYK/EeVMwAHBLD7K4IS/9+dJwLTlDD40ZYW/SPhBwEuQsT3vyoK/g089wCXPNT0L1Xm/e8E2wHLArru700W/RWY6wJ61Nz+ESmi/ERg7wK9cLz/Rk4K/Hck9wP6aKD96ioy/BFpEwCUkFj+x3Iy/FypMwG9ICz/8VEW/aAc8wFltRj8yWme/UKg9wEPLQj8w8X+/rb1BwC/EQj+78oW/+3ZKwM6nQj9dv4S/q0FPwNCbQj/deom/nkFPwHoXBz+45Cy/rwk8wPLOFT9Ycim/kwA7wMwJFj8+dyq/gUI7wA7cNT+BXw+/+ig+wIrNAz+8Hwu/Ets8wKrVAz+/Kw6/CRY+wIDxMD9v1G6/kblFwHhHWj+y1V2/JnNCwCxJWj8epXa/zCVKwOFGWj+DbEW/xY9AwB9IWj/BAiy/t+08wHvZQj9BnC+/Yf5CwBdGWj/fTxW/2ZVAwDfGQj9O8CG/eepFwOFCWj9IwlK/tcJHwHOdYj9LIEW/gUBGwGecYj+kxze/RiZIwCyaYj/oFjK/2XhJwKaZYj/baxm/ByRJwBBAWj/bay2/xjZLwA+ZYj/ScBa/YTJMwGQ8Wj//riu/BMhNwAOYYj/XwVm/CthJwJmcYj/5TFq/CXFMwJ2bYj9ljnG/jUFPwD5BWj+lhS+/okFPwAuWYj/DKVe/xEFPwEaYYj8riXy//cA2wBnjQ7zsooS/mZ09wNTuFz28W4e/FVdCwGB1pD0gCYe/Nj5KwLg8Bj7O/3O/1a42wNwOjb2dSHy/tks9wHk8bb1fe4K/DsBDwH5SLb3/QIG/9btKwEiHh7s7jXy/Q+VMwMo0Wj20x2e/CHJDwM14C74UIWW/W0Q9wL5QAL55I2O/kq42wCkI3r0LR2i/7ZxJwHIXEb7JOmS/JuVMwCkF3b0ibEy/NEhJwLN7Mr43NE2/prlCwGzqLL6fkE2/7bY9wKGFFL7tC1C/T642wDc4Ab58DTW/6xk9wOfkBb6g+zq/ya02wEC/77074y+/hzZCwIjYIL5SgSu/xQBJwJLKFL4Ji0q/HeVMwBO4Fb678S6/9+RMwD8C/72iJi6/l602wFoQqr3v4Rq/uyg9wI/HjL0A5RO/tB1CwA+ZMr1jtRW/k1RJwGpsL7zbpRm/AOVMwPQ3oTpBnQq/F9k7wI9srj3o9h6/Er82wADkhLwAGwi/yehBwKSO7j2OPg6/CFpJwCh+DD46QRe/COVMwB/2Ij4SvCG/tr02wKHXH7wjFhG/93I8wIY4tj2cag2/NShCwAAeAT69NxK/6ztJwEDBFT5jnAu/okFPwOhozT6j6gO/gSFKwNyb1z5SKgG/VORCwK7w7j7CTPO+NjxCwPwB7z47/vO+51ZDwFZ9Ij938/y+FktKwEZ51j4Rp/O+DjFKwMEfFj+H4Ai/q0FPwEWEEz98ZAe/6DBLwOilQj/tSge/OdVEwIW0Qj/YvB6/nkFPwFQ4Wj+kbg+/hUFPwOGXQj98YGe/mdY5wNpz0T743me/XDw7wGnh6j4MWEq/0o46wDLm9j5pcUq/xlA5wJ3z2z7KU4K/oKU7wAfvsz4IWYK/vMk8wGhAzT58Rkq/n8c3wG9Jxj6EuUm/Et41wJxSrj7RBGq/DCA2wIbipj5Numm//Dc4wJaSvT4k8IG/8rQ4wMyyjz4rNIK/s5U6wI3Uoz69cC+/cJo3wAeVwD6HMjC/H9o1wM9ppj4/4Ri/eV05wO2asD48wRq/A5k3wAXClj47xBO/8DM8wBOC5T6dRha/jNY6wNqOyT6jrS6/G/Q4wJS82j7h8S2/fSU6wPLQ9z5TBBS/U3U7wIY84j7oMS6/4GY5wKiL9D4kmEq/LdA5wOig8z4QH2i/t306wP6b5z4MeYK/IAs8wB77yT4RboK/3zU7wKyNuT4UBGe/6IU5wGTl1z6DoUq/bvs4wKPI4j7FqS6/ZJQ4wDeM4j4O3BW/r306wGEb0T7vO4K/ob05wDtUoz4OFoK/63E4wJFklj6UUGq/Bd01wCmUrT42BUq/B5s1wGAEtT5KfjC/GJc1wHEbrT7+DBu/+FU3wMlznT7njxm/g244wP4srj6RuC+/T7I2wNrjvT5oWkq/JNU2wN6SxD7NWmq/3UA3wKM/vD4AAAAA8wBGv0qXNj/7rHK9fehKv9IbNj/eVMS9cr9Xv5nYND9xVsS9jJ5nvxdJMz9TtHK9tHV0v80FMj+1/ha3kl15v1WKMT+kpXI9bHZ0v7wFMj/SU8Q9qZ9nv+RIMz+KWMQ9oMBXv3fYND+2vHI9NulKv8EbNj8AAAAA06REvxrDKD/7rHK9TIxJv6FHKD/eVMS9UmNWv2gEJz9xVsS9bEJmv9Z0JT9TtHK9pBlzv5wxJD+1/ha3cQF4vyS2Iz+kpXI9TBpzv4wxJD/SU8Q9mkNmv7R0JT+KWMQ9gGRWv0cEJz+2vHI9FY1Jv5FHKD8AAAAANA4FviJUdT/7rHK9w/IXvjXscz/eVMS912hJvt49cD9xVsS9eEaDvhixaz9TtHK9CAKcvsECaD+1/ha393SlvrOaZj+kpXI9eQOcvn4CaD/SU8Q9skiDvsSwaz+KWMQ9j21Jvps9cD+2vHI9YvUXvgPscz+2vHI9FRoIvpGaZj+KWMQ9vJE5vinsYj/SU8Q91LV2vlJfXj+kpXI9kBWUvhyxWj+1/ha3L4edvlJJWT9TtHK9QBSUvk+xWj9xVsS9YLF2vrdfXj/eVMS9i405vn3sYj/7rHK9MxcIvsOaZj8AAAAASWXqvbACaD8AAAAApzvnvh+GTj/7rHK99UfwvvORTD/eVMS9M/wDv4p0Rz9xVsS9VaASv+4hQT9TtHK9wHgev3UEPD+1/ha3O/8ivxcQOj+kpXI9aHkevyEEPD/SU8Q9YqESv3khQT+KWMQ9UP0DvxV0Rz+2vHI9ZknwvrCRTD+2vHI9u0TlvubPPz+KWMQ9Fvb8vluyOj/SU8Q9HR8Nv8BfND+kpXI9EvcYv2hCLz+1/ha39nwdv11OLT9TtHK9avYYv6tCLz9xVsS9AB4NvzVgND/eVMS93PP8vsCyOj/7rHK9bEPlvjnQPz8AAAAAHjfcvlTEQT+syJC+KVoIQCvCAb+Fz2a+fSUIQAvPC7+HURC+bLIHQMjQFb+cU0m2O+IHQFjFG7+H/ba+6DIJQB2Opr63KLu+YhEKQGGnOL6Wsam+g94KQKG6ubzGpYq+/HALQD3Ulj1Vhzy+kNoLQOELIz56/rS9SBUMQIifTz6cU0m23CwMQFWGYT6cU0m2Oh4UQGzrFz78+4y94BEUQDarDj5CPii+SfMTQGxB7z2MFYW+Z54TQBLYXD0wnKO+EysTQKW9Ab19PrK+Y3sSQLEaO76cU0m28DYVQFFp5D3+tXy9QC8VQMe62D2KrRC+WyMVQKKWxj0VHG6+fNQUQBjuHD1St5O+lGcUQOLJLr3FxqS+NbgTQE1pPb5i2Zy+e/YSQMgIoL5Uja6+x5sRQEZApb6Qvnm+zy4SQIdu7r65cIi+YrsQQIGT/b53SxK+9N8RQFg7Br8TZBS+LWAQQCcSEL+cU0m2v7oRQLlQDb+cU0m2xjYQQGPSF7/7eVO9jKAHQIiEG7/yCVm9EDsQQFERF7/CMGC9wccRQLuAC78i46G+ZqIIQFex6L4nS62+q+oIQJgxzb58Kqe++FARQLZkxb7Sppq+LQYRQHzS4b72RZK+p5ISQKvryL75MGu+xqULQJMBAD41fFu+2sgTQCRkwD1YrEG+7PsUQIMUnD2Fz2Y+fSUIQAvPC7+syJA+KVoIQCvCAb+HURA+bLIHQMjQFb/7eVM9jKAHQIiEG78i46E+ZqIIQFex6L6H/bY+6DIJQB2Opr63KLs+YhEKQGGnOL6Wsak+g94KQKG6ubzGpYo+/HALQD3Ulj35MGs+xqULQJMBAD56/rQ9SBUMQIifTz5Vhzw+kNoLQOELIz78+4w94BEUQDarDj5CPig+SfMTQGxB7z2MFYU+Z54TQBLYXD01fFs+2sgTQCRkwD0wnKM+EysTQKW9Ab19PrI+Y3sSQLEaO77+tXw9QC8VQMe62D2KrRA+WyMVQKKWxj0VHG4+fNQUQBjuHD1YrEE+7PsUQIMUnD1St5M+lGcUQOLJLr3FxqQ+NbgTQE1pPb5i2Zw+e/YSQMgIoL5Uja4+x5sRQEZApb6Qvnk+zy4SQIdu7r72RZI+p5ISQKvryL65cIg+YrsQQIGT/b7Sppo+LQYRQHzS4b53SxI+9N8RQFg7Br8TZBQ+LWAQQCcSEL/CMGA9wccRQLuAC7/yCVk9EDsQQFERF78nS60+q+oIQJgxzb58Kqc++FARQLZkxb79bES+W0H/PyHJPD7Y8lq+ppcAQB/4KD5c5F6+VB7/P9LjJz7B4Fq+6BP9P/qZKj7O+0++Bpz7P3hgMD4lIUG+cRv7P++qNz40SzK+mrT7P3eFPj5ccye+cD79P8YZQz5egiO+ak//PxIuRD75hSe+5qwAQKd3QT6pajK+12gBQGyxOz5SRUG+IqkBQPVmND6GG1C+klwBQLCMLT79bEQ+W0H/PyHJPD7Y8lo+ppcAQB/4KD5c5F4+VB7/P9LjJz7B4Fo+6BP9P/qZKj7O+08+Bpz7P3hgMD4lIUE+cRv7P++qNz40SzI+mrT7P3eFPj5ccyc+cD79P8YZQz5egiM+ak//PxIuRD75hSc+5qwAQKd3QT6pajI+12gBQGyxOz5SRUE+IqkBQPVmND6GG1A+klwBQLCMLT4=&quot;;;renderer=new THREE.WebGLRenderer({antialias:true});renderer.setPixelRatio(window.devicePixelRatio);renderer.setSize(window.innerWidth,window.innerHeight);document.body.appendChild(renderer.domElement);loader=new THREE.GLTFLoader();loader.load(&quot;data:text/plain;base64,&quot;+base64_data,function(gltf){scene.add(gltf.scene);camera=gltf.cameras[0];controls=new THREE.TrackballControls(camera,renderer.domElement);controls.rotateSpeed=1.0;controls.zoomSpeed=1.2;controls.panSpeed=0.8;controls.noZoom=false;controls.noPan=false;controls.staticMoving=true;controls.dynamicDampingFactor=0.3;controls.keys=[65,83,68];controls.addEventListener(&quot;change&quot;,render);centerControls(scene,camera,controls);render();window.addEventListener(&quot;resize&quot;,onWindowResize,false);animate();onWindowResize();});}
function onWindowResize(){camera.aspect=window.innerWidth/window.innerHeight;camera.updateProjectionMatrix();renderer.setSize(window.innerWidth,window.innerHeight);controls.handleResize();render();}
function animate(){requestAnimationFrame(animate);controls.update();}
function render(){tracklight.position.copy(camera.position);renderer.render(scene,camera);}
init();</script></body>
</html>" width="100%" height="500px" style="border:none;"></iframe>

## Point cloud representation

A point cloud is a representation of a 3D mesh. It is made by repeatedly and uniformly sampling points within the surface of the 3D body. Compared to the mesh representation, the point cloud is a fixed size `ndarray` and hence easier for deep learning algorithms to handle. 

### Load point cloud

!!! tip
    Check out our predefined [`PointCloud3D`](#getting-started-predefined-docs) to get started and play around with our 3D features.

In DocArray, loading a point cloud from a [`PointCloud3DUrl`][docarray.typing.url.url_3d.point_cloud_url.PointCloud3DUrl] instance will return a [`PointsAndColors`][docarray.documents.point_cloud.points_and_colors.PointsAndColors] instance. Such an object has a `points` attribute containing the information about the points in 3D space as well as an optional `colors` attribute.

First, let's define our class `MyPointCloud`, which extends [`BaseDoc`][docarray.base_doc.doc.BaseDoc] and provides attributes to store the point cloud information:

```python
from typing import Optional

from docarray import BaseDoc
from docarray.documents.point_cloud.points_and_colors import PointsAndColors
from docarray.typing import PointCloud3DUrl


class MyPointCloud(BaseDoc):
    url: PointCloud3DUrl
    tensors: Optional[PointsAndColors] = None


doc = MyPointCloud(url="https://people.sc.fsu.edu/~jburkardt/data/obj/al.obj")
```

Next, we can load a point cloud of size `samples` by simply calling [`.load()`][docarray.typing.url.url_3d.point_cloud_url.PointCloud3DUrl.load] on the [`PointCloud3DUrl`][docarray.typing.url.url_3d.point_cloud_url.PointCloud3DUrl] instance:

```python
doc.tensors = doc.url.load(samples=1000)
doc.summary()
```

<details>
    <summary>Output</summary>
    ``` { .text .no-copy }
    📄 MyPointCloud : a63374d ...
    ╭──────────────────────────────────┬───────────────────────────────────────────╮
    │ Attribute                        │ Value                                     │
    ├──────────────────────────────────┼───────────────────────────────────────────┤
    │ url: PointCloud3DUrl             │ https://people.sc.fsu.edu/~jburkardt/dat… │
    │                                  │ ... (length: 52)                          │
    ╰──────────────────────────────────┴───────────────────────────────────────────╯
    └── 🔶 tensors: PointsAndColors
        └── 📄 PointsAndColors : 70ae175 ...
            ╭─────────────────┬────────────────────────────────────────────────────╮
            │ Attribute       │ Value                                              │
            ├─────────────────┼────────────────────────────────────────────────────┤
            │ points: NdArray │ NdArray of shape (1000, 3), dtype: float64         │
            ╰─────────────────┴────────────────────────────────────────────────────╯
    
    ```
</details>

### Display 3D point cloud in notebook

You can display your point cloud and interact with it from its URL as well as from a PointsAndColors instance. The first will always display without color, whereas the display from [`PointsAndColors`][docarray.documents.point_cloud.points_and_colors.PointsAndColors] will show with color if `.colors` is not None.

``` { .python}
doc.url.display()
```

<iframe srcdoc="<!DOCTYPE html>
<html lang=&quot;en&quot;>
  <head>
    <title>trimesh: threejs viewer</title>
    <meta charset=&quot;utf-8&quot;>
    <meta name=&quot;viewport&quot; content=&quot;width=device-width, 
		   user-scalable=no, 
		   minimum-scale=1.0, 
		   maximum-scale=1.0&quot;>
    <style>
      body {
      margin: 0px;
      overflow: hidden;
      }
    </style>
  </head>
  <body>
    <div id=&quot;container&quot;></div>
    <script>// threejs.org/license
(function(k,Ea){&quot;object&quot;===typeof exports&&&quot;undefined&quot;!==typeof module?Ea(exports):&quot;function&quot;===typeof define&&define.amd?define([&quot;exports&quot;],Ea):(k=k||self,Ea(k.THREE={}))})(this,function(k){function Ea(){}function t(a,b){this.x=a||0;this.y=b||0}function wa(){this.elements=[1,0,0,0,1,0,0,0,1];0<arguments.length&&console.error(&quot;THREE.Matrix3: the constructor no longer reads arguments. use .set() instead.&quot;)}function V(a,b,c,d,e,f,g,h,l,m){Object.defineProperty(this,&quot;id&quot;,{value:pj++});this.uuid=L.generateUUID();
this.name=&quot;&quot;;this.image=void 0!==a?a:V.DEFAULT_IMAGE;this.mipmaps=[];this.mapping=void 0!==b?b:V.DEFAULT_MAPPING;this.wrapS=void 0!==c?c:1001;this.wrapT=void 0!==d?d:1001;this.magFilter=void 0!==e?e:1006;this.minFilter=void 0!==f?f:1008;this.anisotropy=void 0!==l?l:1;this.format=void 0!==g?g:1023;this.internalFormat=null;this.type=void 0!==h?h:1009;this.offset=new t(0,0);this.repeat=new t(1,1);this.center=new t(0,0);this.rotation=0;this.matrixAutoUpdate=!0;this.matrix=new wa;this.generateMipmaps=
!0;this.premultiplyAlpha=!1;this.flipY=!0;this.unpackAlignment=4;this.encoding=void 0!==m?m:3E3;this.version=0;this.onUpdate=null}function ka(a,b,c,d){this.x=a||0;this.y=b||0;this.z=c||0;this.w=void 0!==d?d:1}function Ha(a,b,c){this.width=a;this.height=b;this.scissor=new ka(0,0,a,b);this.scissorTest=!1;this.viewport=new ka(0,0,a,b);c=c||{};this.texture=new V(void 0,c.mapping,c.wrapS,c.wrapT,c.magFilter,c.minFilter,c.format,c.type,c.anisotropy,c.encoding);this.texture.image={};this.texture.image.width=
a;this.texture.image.height=b;this.texture.generateMipmaps=void 0!==c.generateMipmaps?c.generateMipmaps:!1;this.texture.minFilter=void 0!==c.minFilter?c.minFilter:1006;this.depthBuffer=void 0!==c.depthBuffer?c.depthBuffer:!0;this.stencilBuffer=void 0!==c.stencilBuffer?c.stencilBuffer:!0;this.depthTexture=void 0!==c.depthTexture?c.depthTexture:null}function Zf(a,b,c){Ha.call(this,a,b,c);this.samples=4}function Aa(a,b,c,d){this._x=a||0;this._y=b||0;this._z=c||0;this._w=void 0!==d?d:1}function n(a,b,
c){this.x=a||0;this.y=b||0;this.z=c||0}function P(){this.elements=[1,0,0,0,0,1,0,0,0,0,1,0,0,0,0,1];0<arguments.length&&console.error(&quot;THREE.Matrix4: the constructor no longer reads arguments. use .set() instead.&quot;)}function Tb(a,b,c,d){this._x=a||0;this._y=b||0;this._z=c||0;this._order=d||Tb.DefaultOrder}function He(){this.mask=1}function F(){Object.defineProperty(this,&quot;id&quot;,{value:qj++});this.uuid=L.generateUUID();this.name=&quot;&quot;;this.type=&quot;Object3D&quot;;this.parent=null;this.children=[];this.up=F.DefaultUp.clone();
var a=new n,b=new Tb,c=new Aa,d=new n(1,1,1);b._onChange(function(){c.setFromEuler(b,!1)});c._onChange(function(){b.setFromQuaternion(c,void 0,!1)});Object.defineProperties(this,{position:{configurable:!0,enumerable:!0,value:a},rotation:{configurable:!0,enumerable:!0,value:b},quaternion:{configurable:!0,enumerable:!0,value:c},scale:{configurable:!0,enumerable:!0,value:d},modelViewMatrix:{value:new P},normalMatrix:{value:new wa}});this.matrix=new P;this.matrixWorld=new P;this.matrixAutoUpdate=F.DefaultMatrixAutoUpdate;
this.matrixWorldNeedsUpdate=!1;this.layers=new He;this.visible=!0;this.receiveShadow=this.castShadow=!1;this.frustumCulled=!0;this.renderOrder=0;this.userData={}}function ob(){F.call(this);this.type=&quot;Scene&quot;;this.overrideMaterial=this.fog=this.environment=this.background=null;this.autoUpdate=!0;&quot;undefined&quot;!==typeof __THREE_DEVTOOLS__&&__THREE_DEVTOOLS__.dispatchEvent(new CustomEvent(&quot;observe&quot;,{detail:this}))}function Sa(a,b){this.min=void 0!==a?a:new n(Infinity,Infinity,Infinity);this.max=void 0!==
b?b:new n(-Infinity,-Infinity,-Infinity)}function $f(a,b,c,d,e){var f;var g=0;for(f=a.length-3;g<=f;g+=3){Ub.fromArray(a,g);var h=e.x*Math.abs(Ub.x)+e.y*Math.abs(Ub.y)+e.z*Math.abs(Ub.z),l=b.dot(Ub),m=c.dot(Ub),u=d.dot(Ub);if(Math.max(-Math.max(l,m,u),Math.min(l,m,u))>h)return!1}return!0}function pb(a,b){this.center=void 0!==a?a:new n;this.radius=void 0!==b?b:0}function Vb(a,b){this.origin=void 0!==a?a:new n;this.direction=void 0!==b?b:new n(0,0,-1)}function Ta(a,b){this.normal=void 0!==a?a:new n(1,
0,0);this.constant=void 0!==b?b:0}function oa(a,b,c){this.a=void 0!==a?a:new n;this.b=void 0!==b?b:new n;this.c=void 0!==c?c:new n}function A(a,b,c){return void 0===b&&void 0===c?this.set(a):this.setRGB(a,b,c)}function ag(a,b,c){0>c&&(c+=1);1<c&&--c;return c<1/6?a+6*(b-a)*c:.5>c?b:c<2/3?a+6*(b-a)*(2/3-c):a}function bg(a){return.04045>a?.0773993808*a:Math.pow(.9478672986*a+.0521327014,2.4)}function cg(a){return.0031308>a?12.92*a:1.055*Math.pow(a,.41666)-.055}function Bc(a,b,c,d,e,f){this.a=a;this.b=
b;this.c=c;this.normal=d&&d.isVector3?d:new n;this.vertexNormals=Array.isArray(d)?d:[];this.color=e&&e.isColor?e:new A;this.vertexColors=Array.isArray(e)?e:[];this.materialIndex=void 0!==f?f:0}function K(){Object.defineProperty(this,&quot;id&quot;,{value:rj++});this.uuid=L.generateUUID();this.name=&quot;&quot;;this.type=&quot;Material&quot;;this.fog=!0;this.blending=1;this.side=0;this.vertexColors=this.flatShading=!1;this.opacity=1;this.transparent=!1;this.blendSrc=204;this.blendDst=205;this.blendEquation=100;this.blendEquationAlpha=
this.blendDstAlpha=this.blendSrcAlpha=null;this.depthFunc=3;this.depthWrite=this.depthTest=!0;this.stencilWriteMask=255;this.stencilFunc=519;this.stencilRef=0;this.stencilFuncMask=255;this.stencilZPass=this.stencilZFail=this.stencilFail=7680;this.stencilWrite=!1;this.clippingPlanes=null;this.clipShadows=this.clipIntersection=!1;this.shadowSide=null;this.colorWrite=!0;this.precision=null;this.polygonOffset=!1;this.polygonOffsetUnits=this.polygonOffsetFactor=0;this.dithering=!1;this.alphaTest=0;this.premultipliedAlpha=
!1;this.toneMapped=this.visible=!0;this.userData={};this.version=0}function Oa(a){K.call(this);this.type=&quot;MeshBasicMaterial&quot;;this.color=new A(16777215);this.lightMap=this.map=null;this.lightMapIntensity=1;this.aoMap=null;this.aoMapIntensity=1;this.envMap=this.alphaMap=this.specularMap=null;this.combine=0;this.reflectivity=1;this.refractionRatio=.98;this.wireframe=!1;this.wireframeLinewidth=1;this.wireframeLinejoin=this.wireframeLinecap=&quot;round&quot;;this.morphTargets=this.skinning=!1;this.setValues(a)}
function M(a,b,c){if(Array.isArray(a))throw new TypeError(&quot;THREE.BufferAttribute: array should be a Typed Array.&quot;);this.name=&quot;&quot;;this.array=a;this.itemSize=b;this.count=void 0!==a?a.length/b:0;this.normalized=!0===c;this.usage=35044;this.updateRange={offset:0,count:-1};this.version=0}function Bd(a,b,c){M.call(this,new Int8Array(a),b,c)}function Cd(a,b,c){M.call(this,new Uint8Array(a),b,c)}function Dd(a,b,c){M.call(this,new Uint8ClampedArray(a),b,c)}function Ed(a,b,c){M.call(this,new Int16Array(a),
b,c)}function Wb(a,b,c){M.call(this,new Uint16Array(a),b,c)}function Fd(a,b,c){M.call(this,new Int32Array(a),b,c)}function Xb(a,b,c){M.call(this,new Uint32Array(a),b,c)}function y(a,b,c){M.call(this,new Float32Array(a),b,c)}function Gd(a,b,c){M.call(this,new Float64Array(a),b,c)}function xh(){this.vertices=[];this.normals=[];this.colors=[];this.uvs=[];this.uvs2=[];this.groups=[];this.morphTargets={};this.skinWeights=[];this.skinIndices=[];this.boundingSphere=this.boundingBox=null;this.groupsNeedUpdate=
this.uvsNeedUpdate=this.colorsNeedUpdate=this.normalsNeedUpdate=this.verticesNeedUpdate=!1}function yh(a){if(0===a.length)return-Infinity;for(var b=a[0],c=1,d=a.length;c<d;++c)a[c]>b&&(b=a[c]);return b}function C(){Object.defineProperty(this,&quot;id&quot;,{value:sj+=2});this.uuid=L.generateUUID();this.name=&quot;&quot;;this.type=&quot;BufferGeometry&quot;;this.index=null;this.attributes={};this.morphAttributes={};this.morphTargetsRelative=!1;this.groups=[];this.boundingSphere=this.boundingBox=null;this.drawRange={start:0,count:Infinity};
this.userData={}}function S(a,b){F.call(this);this.type=&quot;Mesh&quot;;this.geometry=void 0!==a?a:new C;this.material=void 0!==b?b:new Oa;this.updateMorphTargets()}function zh(a,b,c,d,e,f,g,h){if(null===(1===b.side?d.intersectTriangle(g,f,e,!0,h):d.intersectTriangle(e,f,g,2!==b.side,h)))return null;Ie.copy(h);Ie.applyMatrix4(a.matrixWorld);b=c.ray.origin.distanceTo(Ie);return b<c.near||b>c.far?null:{distance:b,point:Ie.clone(),object:a}}function Je(a,b,c,d,e,f,g,h,l,m,u,p){Yb.fromBufferAttribute(e,m);Zb.fromBufferAttribute(e,
u);$b.fromBufferAttribute(e,p);e=a.morphTargetInfluences;if(b.morphTargets&&f&&e){Ke.set(0,0,0);Le.set(0,0,0);Me.set(0,0,0);for(var x=0,r=f.length;x<r;x++){var k=e[x],v=f[x];0!==k&&(dg.fromBufferAttribute(v,m),eg.fromBufferAttribute(v,u),fg.fromBufferAttribute(v,p),g?(Ke.addScaledVector(dg,k),Le.addScaledVector(eg,k),Me.addScaledVector(fg,k)):(Ke.addScaledVector(dg.sub(Yb),k),Le.addScaledVector(eg.sub(Zb),k),Me.addScaledVector(fg.sub($b),k)))}Yb.add(Ke);Zb.add(Le);$b.add(Me)}if(a=zh(a,b,c,d,Yb,Zb,
$b,Hd))h&&(Cc.fromBufferAttribute(h,m),Dc.fromBufferAttribute(h,u),Ec.fromBufferAttribute(h,p),a.uv=oa.getUV(Hd,Yb,Zb,$b,Cc,Dc,Ec,new t)),l&&(Cc.fromBufferAttribute(l,m),Dc.fromBufferAttribute(l,u),Ec.fromBufferAttribute(l,p),a.uv2=oa.getUV(Hd,Yb,Zb,$b,Cc,Dc,Ec,new t)),h=new Bc(m,u,p),oa.getNormal(Yb,Zb,$b,h.normal),a.face=h;return a}function N(){Object.defineProperty(this,&quot;id&quot;,{value:tj+=2});this.uuid=L.generateUUID();this.name=&quot;&quot;;this.type=&quot;Geometry&quot;;this.vertices=[];this.colors=[];this.faces=[];
this.faceVertexUvs=[[]];this.morphTargets=[];this.morphNormals=[];this.skinWeights=[];this.skinIndices=[];this.lineDistances=[];this.boundingSphere=this.boundingBox=null;this.groupsNeedUpdate=this.lineDistancesNeedUpdate=this.colorsNeedUpdate=this.normalsNeedUpdate=this.uvsNeedUpdate=this.verticesNeedUpdate=this.elementsNeedUpdate=!1}function Fc(a){var b={},c;for(c in a){b[c]={};for(var d in a[c]){var e=a[c][d];e&&(e.isColor||e.isMatrix3||e.isMatrix4||e.isVector2||e.isVector3||e.isVector4||e.isTexture)?
b[c][d]=e.clone():Array.isArray(e)?b[c][d]=e.slice():b[c][d]=e}}return b}function va(a){for(var b={},c=0;c<a.length;c++){var d=Fc(a[c]),e;for(e in d)b[e]=d[e]}return b}function Ba(a){K.call(this);this.type=&quot;ShaderMaterial&quot;;this.defines={};this.uniforms={};this.vertexShader=&quot;void main() {\n\tgl_Position = projectionMatrix * modelViewMatrix * vec4( position, 1.0 );\n}&quot;;this.fragmentShader=&quot;void main() {\n\tgl_FragColor = vec4( 1.0, 0.0, 0.0, 1.0 );\n}&quot;;this.linewidth=1;this.wireframe=!1;this.wireframeLinewidth=
1;this.morphNormals=this.morphTargets=this.skinning=this.clipping=this.lights=this.fog=!1;this.extensions={derivatives:!1,fragDepth:!1,drawBuffers:!1,shaderTextureLOD:!1};this.defaultAttributeValues={color:[1,1,1],uv:[0,0],uv2:[0,0]};this.index0AttributeName=void 0;this.uniformsNeedUpdate=!1;void 0!==a&&(void 0!==a.attributes&&console.error(&quot;THREE.ShaderMaterial: attributes should now be defined in THREE.BufferGeometry instead.&quot;),this.setValues(a))}function db(){F.call(this);this.type=&quot;Camera&quot;;this.matrixWorldInverse=
new P;this.projectionMatrix=new P;this.projectionMatrixInverse=new P}function aa(a,b,c,d){db.call(this);this.type=&quot;PerspectiveCamera&quot;;this.fov=void 0!==a?a:50;this.zoom=1;this.near=void 0!==c?c:.1;this.far=void 0!==d?d:2E3;this.focus=10;this.aspect=void 0!==b?b:1;this.view=null;this.filmGauge=35;this.filmOffset=0;this.updateProjectionMatrix()}function Gc(a,b,c,d){F.call(this);this.type=&quot;CubeCamera&quot;;var e=new aa(90,1,a,b);e.up.set(0,-1,0);e.lookAt(new n(1,0,0));this.add(e);var f=new aa(90,1,a,b);f.up.set(0,
-1,0);f.lookAt(new n(-1,0,0));this.add(f);var g=new aa(90,1,a,b);g.up.set(0,0,1);g.lookAt(new n(0,1,0));this.add(g);var h=new aa(90,1,a,b);h.up.set(0,0,-1);h.lookAt(new n(0,-1,0));this.add(h);var l=new aa(90,1,a,b);l.up.set(0,-1,0);l.lookAt(new n(0,0,1));this.add(l);var m=new aa(90,1,a,b);m.up.set(0,-1,0);m.lookAt(new n(0,0,-1));this.add(m);d=d||{format:1022,magFilter:1006,minFilter:1006};this.renderTarget=new Db(c,d);this.renderTarget.texture.name=&quot;CubeCamera&quot;;this.update=function(a,b){null===this.parent&&
this.updateMatrixWorld();var c=a.getRenderTarget(),d=this.renderTarget,p=d.texture.generateMipmaps;d.texture.generateMipmaps=!1;a.setRenderTarget(d,0);a.render(b,e);a.setRenderTarget(d,1);a.render(b,f);a.setRenderTarget(d,2);a.render(b,g);a.setRenderTarget(d,3);a.render(b,h);a.setRenderTarget(d,4);a.render(b,l);d.texture.generateMipmaps=p;a.setRenderTarget(d,5);a.render(b,m);a.setRenderTarget(c)};this.clear=function(a,b,c,d){for(var e=a.getRenderTarget(),f=this.renderTarget,g=0;6>g;g++)a.setRenderTarget(f,
g),a.clear(b,c,d);a.setRenderTarget(e)}}function Db(a,b,c){Number.isInteger(b)&&(console.warn(&quot;THREE.WebGLCubeRenderTarget: constructor signature is now WebGLCubeRenderTarget( size, options )&quot;),b=c);Ha.call(this,a,a,b)}function ac(a,b,c,d,e,f,g,h,l,m,u,p){V.call(this,null,f,g,h,l,m,d,e,u,p);this.image={data:a||null,width:b||1,height:c||1};this.magFilter=void 0!==l?l:1003;this.minFilter=void 0!==m?m:1003;this.flipY=this.generateMipmaps=!1;this.unpackAlignment=1;this.needsUpdate=!0}function Hc(a,b,
c,d,e,f){this.planes=[void 0!==a?a:new Ta,void 0!==b?b:new Ta,void 0!==c?c:new Ta,void 0!==d?d:new Ta,void 0!==e?e:new Ta,void 0!==f?f:new Ta]}function Ah(){function a(e,f){!1!==c&&(d(e,f),b.requestAnimationFrame(a))}var b=null,c=!1,d=null;return{start:function(){!0!==c&&null!==d&&(b.requestAnimationFrame(a),c=!0)},stop:function(){c=!1},setAnimationLoop:function(a){d=a},setContext:function(a){b=a}}}function uj(a,b){function c(b,c){var d=b.array,e=b.usage,f=a.createBuffer();a.bindBuffer(c,f);a.bufferData(c,
d,e);b.onUploadCallback();c=5126;d instanceof Float32Array?c=5126:d instanceof Float64Array?console.warn(&quot;THREE.WebGLAttributes: Unsupported data buffer format: Float64Array.&quot;):d instanceof Uint16Array?c=5123:d instanceof Int16Array?c=5122:d instanceof Uint32Array?c=5125:d instanceof Int32Array?c=5124:d instanceof Int8Array?c=5120:d instanceof Uint8Array&&(c=5121);return{buffer:f,type:c,bytesPerElement:d.BYTES_PER_ELEMENT,version:b.version}}var d=b.isWebGL2,e=new WeakMap;return{get:function(a){a.isInterleavedBufferAttribute&&
(a=a.data);return e.get(a)},remove:function(b){b.isInterleavedBufferAttribute&&(b=b.data);var c=e.get(b);c&&(a.deleteBuffer(c.buffer),e.delete(b))},update:function(b,g){b.isInterleavedBufferAttribute&&(b=b.data);var f=e.get(b);if(void 0===f)e.set(b,c(b,g));else if(f.version<b.version){var l=b.array,m=b.updateRange;a.bindBuffer(g,f.buffer);-1===m.count?a.bufferSubData(g,0,l):(d?a.bufferSubData(g,m.offset*l.BYTES_PER_ELEMENT,l,m.offset,m.count):a.bufferSubData(g,m.offset*l.BYTES_PER_ELEMENT,l.subarray(m.offset,
m.offset+m.count)),m.count=-1);f.version=b.version}}}}function Id(a,b,c,d){N.call(this);this.type=&quot;PlaneGeometry&quot;;this.parameters={width:a,height:b,widthSegments:c,heightSegments:d};this.fromBufferGeometry(new bc(a,b,c,d));this.mergeVertices()}function bc(a,b,c,d){C.call(this);this.type=&quot;PlaneBufferGeometry&quot;;this.parameters={width:a,height:b,widthSegments:c,heightSegments:d};a=a||1;b=b||1;var e=a/2,f=b/2;c=Math.floor(c)||1;d=Math.floor(d)||1;var g=c+1,h=d+1,l=a/c,m=b/d,u=[],p=[],k=[],r=[];for(a=0;a<
h;a++){var q=a*m-f;for(b=0;b<g;b++)p.push(b*l-e,-q,0),k.push(0,0,1),r.push(b/c),r.push(1-a/d)}for(a=0;a<d;a++)for(b=0;b<c;b++)e=b+g*(a+1),f=b+1+g*(a+1),h=b+1+g*a,u.push(b+g*a,e,h),u.push(e,f,h);this.setIndex(u);this.setAttribute(&quot;position&quot;,new y(p,3));this.setAttribute(&quot;normal&quot;,new y(k,3));this.setAttribute(&quot;uv&quot;,new y(r,2))}function vj(a,b,c,d){function e(a,c){b.buffers.color.setClear(a.r,a.g,a.b,c,d)}var f=new A(0),g=0,h,l,m=null,u=0,p=null;return{getClearColor:function(){return f},setClearColor:function(a,
b){f.set(a);g=void 0!==b?b:1;e(f,g)},getClearAlpha:function(){return g},setClearAlpha:function(a){g=a;e(f,g)},render:function(b,d,k,v){d=d.background;k=a.xr;(k=k.getSession&&k.getSession())&&&quot;additive&quot;===k.environmentBlendMode&&(d=null);null===d?e(f,g):d&&d.isColor&&(e(d,1),v=!0);(a.autoClear||v)&&a.clear(a.autoClearColor,a.autoClearDepth,a.autoClearStencil);if(d&&(d.isCubeTexture||d.isWebGLCubeRenderTarget||306===d.mapping)){void 0===l&&(l=new S(new Jd(1,1,1),new Ba({type:&quot;BackgroundCubeMaterial&quot;,
uniforms:Fc(eb.cube.uniforms),vertexShader:eb.cube.vertexShader,fragmentShader:eb.cube.fragmentShader,side:1,depthTest:!1,depthWrite:!1,fog:!1})),l.geometry.deleteAttribute(&quot;normal&quot;),l.geometry.deleteAttribute(&quot;uv&quot;),l.onBeforeRender=function(a,b,c){this.matrixWorld.copyPosition(c.matrixWorld)},Object.defineProperty(l.material,&quot;envMap&quot;,{get:function(){return this.uniforms.envMap.value}}),c.update(l));v=d.isWebGLCubeRenderTarget?d.texture:d;l.material.uniforms.envMap.value=v;l.material.uniforms.flipEnvMap.value=
v.isCubeTexture?-1:1;if(m!==d||u!==v.version||p!==a.toneMapping)l.material.needsUpdate=!0,m=d,u=v.version,p=a.toneMapping;b.unshift(l,l.geometry,l.material,0,0,null)}else if(d&&d.isTexture){void 0===h&&(h=new S(new bc(2,2),new Ba({type:&quot;BackgroundMaterial&quot;,uniforms:Fc(eb.background.uniforms),vertexShader:eb.background.vertexShader,fragmentShader:eb.background.fragmentShader,side:0,depthTest:!1,depthWrite:!1,fog:!1})),h.geometry.deleteAttribute(&quot;normal&quot;),Object.defineProperty(h.material,&quot;map&quot;,{get:function(){return this.uniforms.t2D.value}}),
c.update(h));h.material.uniforms.t2D.value=d;!0===d.matrixAutoUpdate&&d.updateMatrix();h.material.uniforms.uvTransform.value.copy(d.matrix);if(m!==d||u!==d.version||p!==a.toneMapping)h.material.needsUpdate=!0,m=d,u=d.version,p=a.toneMapping;b.unshift(h,h.geometry,h.material,0,0,null)}}}}function wj(a,b,c,d){var e=d.isWebGL2,f;this.setMode=function(a){f=a};this.render=function(b,d){a.drawArrays(f,b,d);c.update(d,f)};this.renderInstances=function(d,h,l,m){if(0!==m){if(e){d=a;var g=&quot;drawArraysInstanced&quot;}else if(d=
b.get(&quot;ANGLE_instanced_arrays&quot;),g=&quot;drawArraysInstancedANGLE&quot;,null===d){console.error(&quot;THREE.WebGLBufferRenderer: using THREE.InstancedBufferGeometry but hardware does not support extension ANGLE_instanced_arrays.&quot;);return}d[g](f,h,l,m);c.update(l,f,m)}}}function xj(a,b,c){function d(b){if(&quot;highp&quot;===b){if(0<a.getShaderPrecisionFormat(35633,36338).precision&&0<a.getShaderPrecisionFormat(35632,36338).precision)return&quot;highp&quot;;b=&quot;mediump&quot;}return&quot;mediump&quot;===b&&0<a.getShaderPrecisionFormat(35633,36337).precision&&
0<a.getShaderPrecisionFormat(35632,36337).precision?&quot;mediump&quot;:&quot;lowp&quot;}var e,f=&quot;undefined&quot;!==typeof WebGL2RenderingContext&&a instanceof WebGL2RenderingContext||&quot;undefined&quot;!==typeof WebGL2ComputeRenderingContext&&a instanceof WebGL2ComputeRenderingContext,g=void 0!==c.precision?c.precision:&quot;highp&quot;,h=d(g);h!==g&&(console.warn(&quot;THREE.WebGLRenderer:&quot;,g,&quot;not supported, using&quot;,h,&quot;instead.&quot;),g=h);c=!0===c.logarithmicDepthBuffer;h=a.getParameter(34930);var l=a.getParameter(35660),m=a.getParameter(3379),u=
a.getParameter(34076),p=a.getParameter(34921),k=a.getParameter(36347),r=a.getParameter(36348),q=a.getParameter(36349),v=0<l,n=f||!!b.get(&quot;OES_texture_float&quot;),w=v&&n,z=f?a.getParameter(36183):0;return{isWebGL2:f,getMaxAnisotropy:function(){if(void 0!==e)return e;var c=b.get(&quot;EXT_texture_filter_anisotropic&quot;);return e=null!==c?a.getParameter(c.MAX_TEXTURE_MAX_ANISOTROPY_EXT):0},getMaxPrecision:d,precision:g,logarithmicDepthBuffer:c,maxTextures:h,maxVertexTextures:l,maxTextureSize:m,maxCubemapSize:u,
maxAttributes:p,maxVertexUniforms:k,maxVaryings:r,maxFragmentUniforms:q,vertexTextures:v,floatFragmentTextures:n,floatVertexTextures:w,maxSamples:z}}function yj(){function a(){m.value!==d&&(m.value=d,m.needsUpdate=0<e);c.numPlanes=e;c.numIntersection=0}function b(a,b,d,e){var f=null!==a?a.length:0,g=null;if(0!==f){g=m.value;if(!0!==e||null===g){e=d+4*f;b=b.matrixWorldInverse;l.getNormalMatrix(b);if(null===g||g.length<e)g=new Float32Array(e);for(e=0;e!==f;++e,d+=4)h.copy(a[e]).applyMatrix4(b,l),h.normal.toArray(g,
d),g[d+3]=h.constant}m.value=g;m.needsUpdate=!0}c.numPlanes=f;c.numIntersection=0;return g}var c=this,d=null,e=0,f=!1,g=!1,h=new Ta,l=new wa,m={value:null,needsUpdate:!1};this.uniform=m;this.numIntersection=this.numPlanes=0;this.init=function(a,c,g){var h=0!==a.length||c||0!==e||f;f=c;d=b(a,g,0);e=a.length;return h};this.beginShadows=function(){g=!0;b(null)};this.endShadows=function(){g=!1;a()};this.setState=function(c,h,l,k,q,v){if(!f||null===c||0===c.length||g&&!l)g?b(null):a();else{l=g?0:e;var p=
4*l,u=q.clippingState||null;m.value=u;u=b(c,k,p,v);for(c=0;c!==p;++c)u[c]=d[c];q.clippingState=u;this.numIntersection=h?this.numPlanes:0;this.numPlanes+=l}}}function zj(a){var b={};return{get:function(c){if(void 0!==b[c])return b[c];switch(c){case &quot;WEBGL_depth_texture&quot;:var d=a.getExtension(&quot;WEBGL_depth_texture&quot;)||a.getExtension(&quot;MOZ_WEBGL_depth_texture&quot;)||a.getExtension(&quot;WEBKIT_WEBGL_depth_texture&quot;);break;case &quot;EXT_texture_filter_anisotropic&quot;:d=a.getExtension(&quot;EXT_texture_filter_anisotropic&quot;)||a.getExtension(&quot;MOZ_EXT_texture_filter_anisotropic&quot;)||
a.getExtension(&quot;WEBKIT_EXT_texture_filter_anisotropic&quot;);break;case &quot;WEBGL_compressed_texture_s3tc&quot;:d=a.getExtension(&quot;WEBGL_compressed_texture_s3tc&quot;)||a.getExtension(&quot;MOZ_WEBGL_compressed_texture_s3tc&quot;)||a.getExtension(&quot;WEBKIT_WEBGL_compressed_texture_s3tc&quot;);break;case &quot;WEBGL_compressed_texture_pvrtc&quot;:d=a.getExtension(&quot;WEBGL_compressed_texture_pvrtc&quot;)||a.getExtension(&quot;WEBKIT_WEBGL_compressed_texture_pvrtc&quot;);break;default:d=a.getExtension(c)}null===d&&console.warn(&quot;THREE.WebGLRenderer: &quot;+c+&quot; extension not supported.&quot;);
return b[c]=d}}}function Aj(a,b,c){function d(a){var e=a.target;a=f.get(e);null!==a.index&&b.remove(a.index);for(var h in a.attributes)b.remove(a.attributes[h]);e.removeEventListener(&quot;dispose&quot;,d);f.delete(e);if(h=g.get(a))b.remove(h),g.delete(a);c.memory.geometries--}function e(a){var c=[],d=a.index,e=a.attributes.position;if(null!==d){var f=d.array;d=d.version;e=0;for(var h=f.length;e<h;e+=3){var k=f[e+0],q=f[e+1],v=f[e+2];c.push(k,q,q,v,v,k)}}else for(f=e.array,d=e.version,e=0,h=f.length/3-1;e<
h;e+=3)k=e+0,q=e+1,v=e+2,c.push(k,q,q,v,v,k);c=new (65535<yh(c)?Xb:Wb)(c,1);c.version=d;b.update(c,34963);(f=g.get(a))&&b.remove(f);g.set(a,c)}var f=new WeakMap,g=new WeakMap;return{get:function(a,b){var e=f.get(b);if(e)return e;b.addEventListener(&quot;dispose&quot;,d);b.isBufferGeometry?e=b:b.isGeometry&&(void 0===b._bufferGeometry&&(b._bufferGeometry=(new C).setFromObject(a)),e=b._bufferGeometry);f.set(b,e);c.memory.geometries++;return e},update:function(a){var c=a.index,d=a.attributes;null!==c&&b.update(c,
34963);for(var e in d)b.update(d[e],34962);a=a.morphAttributes;for(e in a){c=a[e];d=0;for(var f=c.length;d<f;d++)b.update(c[d],34962)}},getWireframeAttribute:function(a){var b=g.get(a);if(b){var c=a.index;null!==c&&b.version<c.version&&e(a)}else e(a);return g.get(a)}}}function Bj(a,b,c,d){var e=d.isWebGL2,f,g,h;this.setMode=function(a){f=a};this.setIndex=function(a){g=a.type;h=a.bytesPerElement};this.render=function(b,d){a.drawElements(f,d,g,b*h);c.update(d,f)};this.renderInstances=function(d,m,u,
p){if(0!==p){if(e){d=a;var l=&quot;drawElementsInstanced&quot;}else if(d=b.get(&quot;ANGLE_instanced_arrays&quot;),l=&quot;drawElementsInstancedANGLE&quot;,null===d){console.error(&quot;THREE.WebGLIndexedBufferRenderer: using THREE.InstancedBufferGeometry but hardware does not support extension ANGLE_instanced_arrays.&quot;);return}d[l](f,u,g,m*h,p);c.update(u,f,p)}}}function Cj(a){var b={frame:0,calls:0,triangles:0,points:0,lines:0};return{memory:{geometries:0,textures:0},render:b,programs:null,autoReset:!0,reset:function(){b.frame++;
b.calls=0;b.triangles=0;b.points=0;b.lines=0},update:function(a,d,e){e=e||1;b.calls++;switch(d){case 4:b.triangles+=a/3*e;break;case 1:b.lines+=a/2*e;break;case 3:b.lines+=e*(a-1);break;case 2:b.lines+=e*a;break;case 0:b.points+=e*a;break;default:console.error(&quot;THREE.WebGLInfo: Unknown draw mode:&quot;,d)}}}}function Dj(a,b){return Math.abs(b[1])-Math.abs(a[1])}function Ej(a){var b={},c=new Float32Array(8);return{update:function(d,e,f,g){var h=d.morphTargetInfluences,l=void 0===h?0:h.length;d=b[e.id];
if(void 0===d){d=[];for(var m=0;m<l;m++)d[m]=[m,0];b[e.id]=d}var u=f.morphTargets&&e.morphAttributes.position;f=f.morphNormals&&e.morphAttributes.normal;for(m=0;m<l;m++){var p=d[m];0!==p[1]&&(u&&e.deleteAttribute(&quot;morphTarget&quot;+m),f&&e.deleteAttribute(&quot;morphNormal&quot;+m))}for(m=0;m<l;m++)p=d[m],p[0]=m,p[1]=h[m];d.sort(Dj);for(m=h=0;8>m;m++){if(p=d[m])if(l=p[0],p=p[1]){u&&e.setAttribute(&quot;morphTarget&quot;+m,u[l]);f&&e.setAttribute(&quot;morphNormal&quot;+m,f[l]);c[m]=p;h+=p;continue}c[m]=0}e=e.morphTargetsRelative?1:
1-h;g.getUniforms().setValue(a,&quot;morphTargetBaseInfluence&quot;,e);g.getUniforms().setValue(a,&quot;morphTargetInfluences&quot;,c)}}}function Fj(a,b,c,d){var e=new WeakMap;return{update:function(a){var f=d.render.frame,h=a.geometry,l=b.get(a,h);e.get(l)!==f&&(h.isGeometry&&l.updateFromObject(a),b.update(l),e.set(l,f));a.isInstancedMesh&&c.update(a.instanceMatrix,34962);return l},dispose:function(){e=new WeakMap}}}function qb(a,b,c,d,e,f,g,h,l,m){a=void 0!==a?a:[];V.call(this,a,void 0!==b?b:301,c,d,e,f,void 0!==g?
g:1022,h,l,m);this.flipY=!1}function Ic(a,b,c,d){V.call(this,null);this.image={data:a||null,width:b||1,height:c||1,depth:d||1};this.minFilter=this.magFilter=1003;this.wrapR=1001;this.flipY=this.generateMipmaps=!1;this.needsUpdate=!0}function Jc(a,b,c,d){V.call(this,null);this.image={data:a||null,width:b||1,height:c||1,depth:d||1};this.minFilter=this.magFilter=1003;this.wrapR=1001;this.flipY=this.generateMipmaps=!1;this.needsUpdate=!0}function Kc(a,b,c){var d=a[0];if(0>=d||0<d)return a;var e=b*c,f=
Bh[e];void 0===f&&(f=new Float32Array(e),Bh[e]=f);if(0!==b)for(d.toArray(f,0),d=1,e=0;d!==b;++d)e+=c,a[d].toArray(f,e);return f}function Pa(a,b){if(a.length!==b.length)return!1;for(var c=0,d=a.length;c<d;c++)if(a[c]!==b[c])return!1;return!0}function Ia(a,b){for(var c=0,d=b.length;c<d;c++)a[c]=b[c]}function Ch(a,b){var c=Dh[b];void 0===c&&(c=new Int32Array(b),Dh[b]=c);for(var d=0;d!==b;++d)c[d]=a.allocateTextureUnit();return c}function Gj(a,b){var c=this.cache;c[0]!==b&&(a.uniform1f(this.addr,b),c[0]=
b)}function Hj(a,b){var c=this.cache;if(void 0!==b.x){if(c[0]!==b.x||c[1]!==b.y)a.uniform2f(this.addr,b.x,b.y),c[0]=b.x,c[1]=b.y}else Pa(c,b)||(a.uniform2fv(this.addr,b),Ia(c,b))}function Ij(a,b){var c=this.cache;if(void 0!==b.x){if(c[0]!==b.x||c[1]!==b.y||c[2]!==b.z)a.uniform3f(this.addr,b.x,b.y,b.z),c[0]=b.x,c[1]=b.y,c[2]=b.z}else if(void 0!==b.r){if(c[0]!==b.r||c[1]!==b.g||c[2]!==b.b)a.uniform3f(this.addr,b.r,b.g,b.b),c[0]=b.r,c[1]=b.g,c[2]=b.b}else Pa(c,b)||(a.uniform3fv(this.addr,b),Ia(c,b))}
function Jj(a,b){var c=this.cache;if(void 0!==b.x){if(c[0]!==b.x||c[1]!==b.y||c[2]!==b.z||c[3]!==b.w)a.uniform4f(this.addr,b.x,b.y,b.z,b.w),c[0]=b.x,c[1]=b.y,c[2]=b.z,c[3]=b.w}else Pa(c,b)||(a.uniform4fv(this.addr,b),Ia(c,b))}function Kj(a,b){var c=this.cache,d=b.elements;void 0===d?Pa(c,b)||(a.uniformMatrix2fv(this.addr,!1,b),Ia(c,b)):Pa(c,d)||(Eh.set(d),a.uniformMatrix2fv(this.addr,!1,Eh),Ia(c,d))}function Lj(a,b){var c=this.cache,d=b.elements;void 0===d?Pa(c,b)||(a.uniformMatrix3fv(this.addr,!1,
b),Ia(c,b)):Pa(c,d)||(Fh.set(d),a.uniformMatrix3fv(this.addr,!1,Fh),Ia(c,d))}function Mj(a,b){var c=this.cache,d=b.elements;void 0===d?Pa(c,b)||(a.uniformMatrix4fv(this.addr,!1,b),Ia(c,b)):Pa(c,d)||(Gh.set(d),a.uniformMatrix4fv(this.addr,!1,Gh),Ia(c,d))}function Nj(a,b,c){var d=this.cache,e=c.allocateTextureUnit();d[0]!==e&&(a.uniform1i(this.addr,e),d[0]=e);c.safeSetTexture2D(b||Hh,e)}function Oj(a,b,c){var d=this.cache,e=c.allocateTextureUnit();d[0]!==e&&(a.uniform1i(this.addr,e),d[0]=e);c.setTexture2DArray(b||
Pj,e)}function Qj(a,b,c){var d=this.cache,e=c.allocateTextureUnit();d[0]!==e&&(a.uniform1i(this.addr,e),d[0]=e);c.setTexture3D(b||Rj,e)}function Sj(a,b,c){var d=this.cache,e=c.allocateTextureUnit();d[0]!==e&&(a.uniform1i(this.addr,e),d[0]=e);c.safeSetTextureCube(b||Ih,e)}function Tj(a,b){var c=this.cache;c[0]!==b&&(a.uniform1i(this.addr,b),c[0]=b)}function Uj(a,b){var c=this.cache;Pa(c,b)||(a.uniform2iv(this.addr,b),Ia(c,b))}function Vj(a,b){var c=this.cache;Pa(c,b)||(a.uniform3iv(this.addr,b),Ia(c,
b))}function Wj(a,b){var c=this.cache;Pa(c,b)||(a.uniform4iv(this.addr,b),Ia(c,b))}function Xj(a,b){var c=this.cache;c[0]!==b&&(a.uniform1ui(this.addr,b),c[0]=b)}function Yj(a){switch(a){case 5126:return Gj;case 35664:return Hj;case 35665:return Ij;case 35666:return Jj;case 35674:return Kj;case 35675:return Lj;case 35676:return Mj;case 5124:case 35670:return Tj;case 35667:case 35671:return Uj;case 35668:case 35672:return Vj;case 35669:case 35673:return Wj;case 5125:return Xj;case 35678:case 36198:case 36298:case 36306:case 35682:return Nj;
case 35679:case 36299:case 36307:return Qj;case 35680:case 36300:case 36308:case 36293:return Sj;case 36289:case 36303:case 36311:case 36292:return Oj}}function Zj(a,b){a.uniform1fv(this.addr,b)}function ak(a,b){a.uniform1iv(this.addr,b)}function bk(a,b){a.uniform2iv(this.addr,b)}function ck(a,b){a.uniform3iv(this.addr,b)}function dk(a,b){a.uniform4iv(this.addr,b)}function ek(a,b){b=Kc(b,this.size,2);a.uniform2fv(this.addr,b)}function fk(a,b){b=Kc(b,this.size,3);a.uniform3fv(this.addr,b)}function gk(a,
b){b=Kc(b,this.size,4);a.uniform4fv(this.addr,b)}function hk(a,b){b=Kc(b,this.size,4);a.uniformMatrix2fv(this.addr,!1,b)}function ik(a,b){b=Kc(b,this.size,9);a.uniformMatrix3fv(this.addr,!1,b)}function jk(a,b){b=Kc(b,this.size,16);a.uniformMatrix4fv(this.addr,!1,b)}function kk(a,b,c){var d=b.length,e=Ch(c,d);a.uniform1iv(this.addr,e);for(a=0;a!==d;++a)c.safeSetTexture2D(b[a]||Hh,e[a])}function lk(a,b,c){var d=b.length,e=Ch(c,d);a.uniform1iv(this.addr,e);for(a=0;a!==d;++a)c.safeSetTextureCube(b[a]||
Ih,e[a])}function mk(a){switch(a){case 5126:return Zj;case 35664:return ek;case 35665:return fk;case 35666:return gk;case 35674:return hk;case 35675:return ik;case 35676:return jk;case 5124:case 35670:return ak;case 35667:case 35671:return bk;case 35668:case 35672:return ck;case 35669:case 35673:return dk;case 35678:case 36198:case 36298:case 36306:case 35682:return kk;case 35680:case 36300:case 36308:case 36293:return lk}}function nk(a,b,c){this.id=a;this.addr=c;this.cache=[];this.setValue=Yj(b.type)}
function Jh(a,b,c){this.id=a;this.addr=c;this.cache=[];this.size=b.size;this.setValue=mk(b.type)}function Kh(a){this.id=a;this.seq=[];this.map={}}function Eb(a,b){this.seq=[];this.map={};for(var c=a.getProgramParameter(b,35718),d=0;d<c;++d){var e=a.getActiveUniform(b,d),f=a.getUniformLocation(b,e.name),g=this,h=e.name,l=h.length;for(gg.lastIndex=0;;){var m=gg.exec(h),u=gg.lastIndex,p=m[1],k=m[3];&quot;]&quot;===m[2]&&(p|=0);if(void 0===k||&quot;[&quot;===k&&u+2===l){h=g;e=void 0===k?new nk(p,e,f):new Jh(p,e,f);h.seq.push(e);
h.map[e.id]=e;break}else k=g.map[p],void 0===k&&(k=new Kh(p),p=g,g=k,p.seq.push(g),p.map[g.id]=g),g=k}}}function Lh(a,b,c){b=a.createShader(b);a.shaderSource(b,c);a.compileShader(b);return b}function Mh(a){switch(a){case 3E3:return[&quot;Linear&quot;,&quot;( value )&quot;];case 3001:return[&quot;sRGB&quot;,&quot;( value )&quot;];case 3002:return[&quot;RGBE&quot;,&quot;( value )&quot;];case 3004:return[&quot;RGBM&quot;,&quot;( value, 7.0 )&quot;];case 3005:return[&quot;RGBM&quot;,&quot;( value, 16.0 )&quot;];case 3006:return[&quot;RGBD&quot;,&quot;( value, 256.0 )&quot;];case 3007:return[&quot;Gamma&quot;,&quot;( value, float( GAMMA_FACTOR ) )&quot;];
case 3003:return[&quot;LogLuv&quot;,&quot;( value )&quot;];default:throw Error(&quot;unsupported encoding: &quot;+a);}}function Nh(a,b,c){var d=a.getShaderParameter(b,35713),e=a.getShaderInfoLog(b).trim();if(d&&&quot;&quot;===e)return&quot;&quot;;a=a.getShaderSource(b).split(&quot;\n&quot;);for(b=0;b<a.length;b++)a[b]=b+1+&quot;: &quot;+a[b];a=a.join(&quot;\n&quot;);return&quot;THREE.WebGLShader: gl.getShaderInfoLog() &quot;+c+&quot;\n&quot;+e+a}function Kd(a,b){b=Mh(b);return&quot;vec4 &quot;+a+&quot;( vec4 value ) { return &quot;+b[0]+&quot;ToLinear&quot;+b[1]+&quot;; }&quot;}function ok(a,b){b=Mh(b);return&quot;vec4 &quot;+a+&quot;( vec4 value ) { return LinearTo&quot;+
b[0]+b[1]+&quot;; }&quot;}function pk(a,b){switch(b){case 1:b=&quot;Linear&quot;;break;case 2:b=&quot;Reinhard&quot;;break;case 3:b=&quot;Uncharted2&quot;;break;case 4:b=&quot;OptimizedCineon&quot;;break;case 5:b=&quot;ACESFilmic&quot;;break;default:throw Error(&quot;unsupported toneMapping: &quot;+b);}return&quot;vec3 &quot;+a+&quot;( vec3 color ) { return &quot;+b+&quot;ToneMapping( color ); }&quot;}function qk(a){var b=[],c;for(c in a){var d=a[c];!1!==d&&b.push(&quot;#define &quot;+c+&quot; &quot;+d)}return b.join(&quot;\n&quot;)}function Ld(a){return&quot;&quot;!==a}function Oh(a,b){return a.replace(/NUM_DIR_LIGHTS/g,b.numDirLights).replace(/NUM_SPOT_LIGHTS/g,
b.numSpotLights).replace(/NUM_RECT_AREA_LIGHTS/g,b.numRectAreaLights).replace(/NUM_POINT_LIGHTS/g,b.numPointLights).replace(/NUM_HEMI_LIGHTS/g,b.numHemiLights).replace(/NUM_DIR_LIGHT_SHADOWS/g,b.numDirLightShadows).replace(/NUM_SPOT_LIGHT_SHADOWS/g,b.numSpotLightShadows).replace(/NUM_POINT_LIGHT_SHADOWS/g,b.numPointLightShadows)}function Ph(a,b){return a.replace(/NUM_CLIPPING_PLANES/g,b.numClippingPlanes).replace(/UNION_CLIPPING_PLANES/g,b.numClippingPlanes-b.numClipIntersection)}function hg(a,b){a=
O[b];if(void 0===a)throw Error(&quot;Can not resolve #include <&quot;+b+&quot;>&quot;);return a.replace(ig,hg)}function Qh(a,b,c,d){console.warn(&quot;WebGLProgram: #pragma unroll_loop shader syntax is deprecated. Please use #pragma unroll_loop_start syntax instead.&quot;);return jg(a,b,c,d)}function jg(a,b,c,d){a=&quot;&quot;;for(b=parseInt(b);b<parseInt(c);b++)a+=d.replace(/\[ i \]/g,&quot;[ &quot;+b+&quot; ]&quot;).replace(/UNROLLED_LOOP_INDEX/g,b);return a}function Rh(a){var b=&quot;precision &quot;+a.precision+&quot; float;\nprecision &quot;+a.precision+&quot; int;&quot;;&quot;highp&quot;===
a.precision?b+=&quot;\n#define HIGH_PRECISION&quot;:&quot;mediump&quot;===a.precision?b+=&quot;\n#define MEDIUM_PRECISION&quot;:&quot;lowp&quot;===a.precision&&(b+=&quot;\n#define LOW_PRECISION&quot;);return b}function rk(a){var b=&quot;SHADOWMAP_TYPE_BASIC&quot;;1===a.shadowMapType?b=&quot;SHADOWMAP_TYPE_PCF&quot;:2===a.shadowMapType?b=&quot;SHADOWMAP_TYPE_PCF_SOFT&quot;:3===a.shadowMapType&&(b=&quot;SHADOWMAP_TYPE_VSM&quot;);return b}function sk(a){var b=&quot;ENVMAP_TYPE_CUBE&quot;;if(a.envMap)switch(a.envMapMode){case 301:case 302:b=&quot;ENVMAP_TYPE_CUBE&quot;;break;case 306:case 307:b=&quot;ENVMAP_TYPE_CUBE_UV&quot;;
break;case 303:case 304:b=&quot;ENVMAP_TYPE_EQUIREC&quot;;break;case 305:b=&quot;ENVMAP_TYPE_SPHERE&quot;}return b}function tk(a){var b=&quot;ENVMAP_MODE_REFLECTION&quot;;if(a.envMap)switch(a.envMapMode){case 302:case 304:b=&quot;ENVMAP_MODE_REFRACTION&quot;}return b}function uk(a){var b=&quot;ENVMAP_BLENDING_NONE&quot;;if(a.envMap)switch(a.combine){case 0:b=&quot;ENVMAP_BLENDING_MULTIPLY&quot;;break;case 1:b=&quot;ENVMAP_BLENDING_MIX&quot;;break;case 2:b=&quot;ENVMAP_BLENDING_ADD&quot;}return b}function vk(a,b,c){var d=a.getContext(),e=c.defines,f=c.vertexShader,g=c.fragmentShader,
h=rk(c),l=sk(c),m=tk(c),u=uk(c),p=0<a.gammaFactor?a.gammaFactor:1,k=c.isWebGL2?&quot;&quot;:[c.extensionDerivatives||c.envMapCubeUV||c.bumpMap||c.tangentSpaceNormalMap||c.clearcoatNormalMap||c.flatShading||&quot;physical&quot;===c.shaderID?&quot;#extension GL_OES_standard_derivatives : enable&quot;:&quot;&quot;,(c.extensionFragDepth||c.logarithmicDepthBuffer)&&c.rendererExtensionFragDepth?&quot;#extension GL_EXT_frag_depth : enable&quot;:&quot;&quot;,c.extensionDrawBuffers&&c.rendererExtensionDrawBuffers?&quot;#extension GL_EXT_draw_buffers : require&quot;:&quot;&quot;,(c.extensionShaderTextureLOD||
c.envMap)&&c.rendererExtensionShaderTextureLod?&quot;#extension GL_EXT_shader_texture_lod : enable&quot;:&quot;&quot;].filter(Ld).join(&quot;\n&quot;),r=qk(e),q=d.createProgram();c.isRawShaderMaterial?(e=[r].filter(Ld).join(&quot;\n&quot;),0<e.length&&(e+=&quot;\n&quot;),h=[k,r].filter(Ld).join(&quot;\n&quot;),0<h.length&&(h+=&quot;\n&quot;)):(e=[Rh(c),&quot;#define SHADER_NAME &quot;+c.shaderName,r,c.instancing?&quot;#define USE_INSTANCING&quot;:&quot;&quot;,c.supportsVertexTextures?&quot;#define VERTEX_TEXTURES&quot;:&quot;&quot;,&quot;#define GAMMA_FACTOR &quot;+p,&quot;#define MAX_BONES &quot;+c.maxBones,c.useFog&&c.fog?&quot;#define USE_FOG&quot;:
&quot;&quot;,c.useFog&&c.fogExp2?&quot;#define FOG_EXP2&quot;:&quot;&quot;,c.map?&quot;#define USE_MAP&quot;:&quot;&quot;,c.envMap?&quot;#define USE_ENVMAP&quot;:&quot;&quot;,c.envMap?&quot;#define &quot;+m:&quot;&quot;,c.lightMap?&quot;#define USE_LIGHTMAP&quot;:&quot;&quot;,c.aoMap?&quot;#define USE_AOMAP&quot;:&quot;&quot;,c.emissiveMap?&quot;#define USE_EMISSIVEMAP&quot;:&quot;&quot;,c.bumpMap?&quot;#define USE_BUMPMAP&quot;:&quot;&quot;,c.normalMap?&quot;#define USE_NORMALMAP&quot;:&quot;&quot;,c.normalMap&&c.objectSpaceNormalMap?&quot;#define OBJECTSPACE_NORMALMAP&quot;:&quot;&quot;,c.normalMap&&c.tangentSpaceNormalMap?&quot;#define TANGENTSPACE_NORMALMAP&quot;:&quot;&quot;,c.clearcoatMap?&quot;#define USE_CLEARCOATMAP&quot;:
&quot;&quot;,c.clearcoatRoughnessMap?&quot;#define USE_CLEARCOAT_ROUGHNESSMAP&quot;:&quot;&quot;,c.clearcoatNormalMap?&quot;#define USE_CLEARCOAT_NORMALMAP&quot;:&quot;&quot;,c.displacementMap&&c.supportsVertexTextures?&quot;#define USE_DISPLACEMENTMAP&quot;:&quot;&quot;,c.specularMap?&quot;#define USE_SPECULARMAP&quot;:&quot;&quot;,c.roughnessMap?&quot;#define USE_ROUGHNESSMAP&quot;:&quot;&quot;,c.metalnessMap?&quot;#define USE_METALNESSMAP&quot;:&quot;&quot;,c.alphaMap?&quot;#define USE_ALPHAMAP&quot;:&quot;&quot;,c.vertexTangents?&quot;#define USE_TANGENT&quot;:&quot;&quot;,c.vertexColors?&quot;#define USE_COLOR&quot;:&quot;&quot;,c.vertexUvs?&quot;#define USE_UV&quot;:&quot;&quot;,c.uvsVertexOnly?&quot;#define UVS_VERTEX_ONLY&quot;:
&quot;&quot;,c.flatShading?&quot;#define FLAT_SHADED&quot;:&quot;&quot;,c.skinning?&quot;#define USE_SKINNING&quot;:&quot;&quot;,c.useVertexTexture?&quot;#define BONE_TEXTURE&quot;:&quot;&quot;,c.morphTargets?&quot;#define USE_MORPHTARGETS&quot;:&quot;&quot;,c.morphNormals&&!1===c.flatShading?&quot;#define USE_MORPHNORMALS&quot;:&quot;&quot;,c.doubleSided?&quot;#define DOUBLE_SIDED&quot;:&quot;&quot;,c.flipSided?&quot;#define FLIP_SIDED&quot;:&quot;&quot;,c.shadowMapEnabled?&quot;#define USE_SHADOWMAP&quot;:&quot;&quot;,c.shadowMapEnabled?&quot;#define &quot;+h:&quot;&quot;,c.sizeAttenuation?&quot;#define USE_SIZEATTENUATION&quot;:&quot;&quot;,c.logarithmicDepthBuffer?&quot;#define USE_LOGDEPTHBUF&quot;:&quot;&quot;,c.logarithmicDepthBuffer&&
c.rendererExtensionFragDepth?&quot;#define USE_LOGDEPTHBUF_EXT&quot;:&quot;&quot;,&quot;uniform mat4 modelMatrix;&quot;,&quot;uniform mat4 modelViewMatrix;&quot;,&quot;uniform mat4 projectionMatrix;&quot;,&quot;uniform mat4 viewMatrix;&quot;,&quot;uniform mat3 normalMatrix;&quot;,&quot;uniform vec3 cameraPosition;&quot;,&quot;uniform bool isOrthographic;&quot;,&quot;#ifdef USE_INSTANCING&quot;,&quot; attribute mat4 instanceMatrix;&quot;,&quot;#endif&quot;,&quot;attribute vec3 position;&quot;,&quot;attribute vec3 normal;&quot;,&quot;attribute vec2 uv;&quot;,&quot;#ifdef USE_TANGENT&quot;,&quot;\tattribute vec4 tangent;&quot;,&quot;#endif&quot;,&quot;#ifdef USE_COLOR&quot;,&quot;\tattribute vec3 color;&quot;,
&quot;#endif&quot;,&quot;#ifdef USE_MORPHTARGETS&quot;,&quot;\tattribute vec3 morphTarget0;&quot;,&quot;\tattribute vec3 morphTarget1;&quot;,&quot;\tattribute vec3 morphTarget2;&quot;,&quot;\tattribute vec3 morphTarget3;&quot;,&quot;\t#ifdef USE_MORPHNORMALS&quot;,&quot;\t\tattribute vec3 morphNormal0;&quot;,&quot;\t\tattribute vec3 morphNormal1;&quot;,&quot;\t\tattribute vec3 morphNormal2;&quot;,&quot;\t\tattribute vec3 morphNormal3;&quot;,&quot;\t#else&quot;,&quot;\t\tattribute vec3 morphTarget4;&quot;,&quot;\t\tattribute vec3 morphTarget5;&quot;,&quot;\t\tattribute vec3 morphTarget6;&quot;,&quot;\t\tattribute vec3 morphTarget7;&quot;,&quot;\t#endif&quot;,&quot;#endif&quot;,
&quot;#ifdef USE_SKINNING&quot;,&quot;\tattribute vec4 skinIndex;&quot;,&quot;\tattribute vec4 skinWeight;&quot;,&quot;#endif&quot;,&quot;\n&quot;].filter(Ld).join(&quot;\n&quot;),h=[k,Rh(c),&quot;#define SHADER_NAME &quot;+c.shaderName,r,c.alphaTest?&quot;#define ALPHATEST &quot;+c.alphaTest+(c.alphaTest%1?&quot;&quot;:&quot;.0&quot;):&quot;&quot;,&quot;#define GAMMA_FACTOR &quot;+p,c.useFog&&c.fog?&quot;#define USE_FOG&quot;:&quot;&quot;,c.useFog&&c.fogExp2?&quot;#define FOG_EXP2&quot;:&quot;&quot;,c.map?&quot;#define USE_MAP&quot;:&quot;&quot;,c.matcap?&quot;#define USE_MATCAP&quot;:&quot;&quot;,c.envMap?&quot;#define USE_ENVMAP&quot;:&quot;&quot;,c.envMap?&quot;#define &quot;+l:&quot;&quot;,c.envMap?&quot;#define &quot;+m:&quot;&quot;,c.envMap?&quot;#define &quot;+
u:&quot;&quot;,c.lightMap?&quot;#define USE_LIGHTMAP&quot;:&quot;&quot;,c.aoMap?&quot;#define USE_AOMAP&quot;:&quot;&quot;,c.emissiveMap?&quot;#define USE_EMISSIVEMAP&quot;:&quot;&quot;,c.bumpMap?&quot;#define USE_BUMPMAP&quot;:&quot;&quot;,c.normalMap?&quot;#define USE_NORMALMAP&quot;:&quot;&quot;,c.normalMap&&c.objectSpaceNormalMap?&quot;#define OBJECTSPACE_NORMALMAP&quot;:&quot;&quot;,c.normalMap&&c.tangentSpaceNormalMap?&quot;#define TANGENTSPACE_NORMALMAP&quot;:&quot;&quot;,c.clearcoatMap?&quot;#define USE_CLEARCOATMAP&quot;:&quot;&quot;,c.clearcoatRoughnessMap?&quot;#define USE_CLEARCOAT_ROUGHNESSMAP&quot;:&quot;&quot;,c.clearcoatNormalMap?&quot;#define USE_CLEARCOAT_NORMALMAP&quot;:&quot;&quot;,
c.specularMap?&quot;#define USE_SPECULARMAP&quot;:&quot;&quot;,c.roughnessMap?&quot;#define USE_ROUGHNESSMAP&quot;:&quot;&quot;,c.metalnessMap?&quot;#define USE_METALNESSMAP&quot;:&quot;&quot;,c.alphaMap?&quot;#define USE_ALPHAMAP&quot;:&quot;&quot;,c.sheen?&quot;#define USE_SHEEN&quot;:&quot;&quot;,c.vertexTangents?&quot;#define USE_TANGENT&quot;:&quot;&quot;,c.vertexColors?&quot;#define USE_COLOR&quot;:&quot;&quot;,c.vertexUvs?&quot;#define USE_UV&quot;:&quot;&quot;,c.uvsVertexOnly?&quot;#define UVS_VERTEX_ONLY&quot;:&quot;&quot;,c.gradientMap?&quot;#define USE_GRADIENTMAP&quot;:&quot;&quot;,c.flatShading?&quot;#define FLAT_SHADED&quot;:&quot;&quot;,c.doubleSided?&quot;#define DOUBLE_SIDED&quot;:&quot;&quot;,c.flipSided?&quot;#define FLIP_SIDED&quot;:
&quot;&quot;,c.shadowMapEnabled?&quot;#define USE_SHADOWMAP&quot;:&quot;&quot;,c.shadowMapEnabled?&quot;#define &quot;+h:&quot;&quot;,c.premultipliedAlpha?&quot;#define PREMULTIPLIED_ALPHA&quot;:&quot;&quot;,c.physicallyCorrectLights?&quot;#define PHYSICALLY_CORRECT_LIGHTS&quot;:&quot;&quot;,c.logarithmicDepthBuffer?&quot;#define USE_LOGDEPTHBUF&quot;:&quot;&quot;,c.logarithmicDepthBuffer&&c.rendererExtensionFragDepth?&quot;#define USE_LOGDEPTHBUF_EXT&quot;:&quot;&quot;,(c.extensionShaderTextureLOD||c.envMap)&&c.rendererExtensionShaderTextureLod?&quot;#define TEXTURE_LOD_EXT&quot;:&quot;&quot;,&quot;uniform mat4 viewMatrix;&quot;,&quot;uniform vec3 cameraPosition;&quot;,
&quot;uniform bool isOrthographic;&quot;,0!==c.toneMapping?&quot;#define TONE_MAPPING&quot;:&quot;&quot;,0!==c.toneMapping?O.tonemapping_pars_fragment:&quot;&quot;,0!==c.toneMapping?pk(&quot;toneMapping&quot;,c.toneMapping):&quot;&quot;,c.dithering?&quot;#define DITHERING&quot;:&quot;&quot;,c.outputEncoding||c.mapEncoding||c.matcapEncoding||c.envMapEncoding||c.emissiveMapEncoding||c.lightMapEncoding?O.encodings_pars_fragment:&quot;&quot;,c.mapEncoding?Kd(&quot;mapTexelToLinear&quot;,c.mapEncoding):&quot;&quot;,c.matcapEncoding?Kd(&quot;matcapTexelToLinear&quot;,c.matcapEncoding):&quot;&quot;,c.envMapEncoding?Kd(&quot;envMapTexelToLinear&quot;,
c.envMapEncoding):&quot;&quot;,c.emissiveMapEncoding?Kd(&quot;emissiveMapTexelToLinear&quot;,c.emissiveMapEncoding):&quot;&quot;,c.lightMapEncoding?Kd(&quot;lightMapTexelToLinear&quot;,c.lightMapEncoding):&quot;&quot;,c.outputEncoding?ok(&quot;linearToOutputTexel&quot;,c.outputEncoding):&quot;&quot;,c.depthPacking?&quot;#define DEPTH_PACKING &quot;+c.depthPacking:&quot;&quot;,&quot;\n&quot;].filter(Ld).join(&quot;\n&quot;));f=f.replace(ig,hg);f=Oh(f,c);f=Ph(f,c);g=g.replace(ig,hg);g=Oh(g,c);g=Ph(g,c);f=f.replace(Sh,jg).replace(Th,Qh);g=g.replace(Sh,jg).replace(Th,Qh);c.isWebGL2&&!c.isRawShaderMaterial&&(l=
!1,m=/^\s*#version\s+300\s+es\s*\n/,c.isShaderMaterial&&null!==f.match(m)&&null!==g.match(m)&&(l=!0,f=f.replace(m,&quot;&quot;),g=g.replace(m,&quot;&quot;)),e=&quot;#version 300 es\n\n#define attribute in\n#define varying out\n#define texture2D texture\n&quot;+e,h=[&quot;#version 300 es\n\n#define varying in&quot;,l?&quot;&quot;:&quot;out highp vec4 pc_fragColor;&quot;,l?&quot;&quot;:&quot;#define gl_FragColor pc_fragColor&quot;,&quot;#define gl_FragDepthEXT gl_FragDepth\n#define texture2D texture\n#define textureCube texture\n#define texture2DProj textureProj\n#define texture2DLodEXT textureLod\n#define texture2DProjLodEXT textureProjLod\n#define textureCubeLodEXT textureLod\n#define texture2DGradEXT textureGrad\n#define texture2DProjGradEXT textureProjGrad\n#define textureCubeGradEXT textureGrad&quot;].join(&quot;\n&quot;)+
&quot;\n&quot;+h);g=h+g;f=Lh(d,35633,e+f);g=Lh(d,35632,g);d.attachShader(q,f);d.attachShader(q,g);void 0!==c.index0AttributeName?d.bindAttribLocation(q,0,c.index0AttributeName):!0===c.morphTargets&&d.bindAttribLocation(q,0,&quot;position&quot;);d.linkProgram(q);if(a.debug.checkShaderErrors){a=d.getProgramInfoLog(q).trim();l=d.getShaderInfoLog(f).trim();m=d.getShaderInfoLog(g).trim();p=u=!0;if(!1===d.getProgramParameter(q,35714))u=!1,k=Nh(d,f,&quot;vertex&quot;),r=Nh(d,g,&quot;fragment&quot;),console.error(&quot;THREE.WebGLProgram: shader error: &quot;,
d.getError(),&quot;35715&quot;,d.getProgramParameter(q,35715),&quot;gl.getProgramInfoLog&quot;,a,k,r);else if(&quot;&quot;!==a)console.warn(&quot;THREE.WebGLProgram: gl.getProgramInfoLog()&quot;,a);else if(&quot;&quot;===l||&quot;&quot;===m)p=!1;p&&(this.diagnostics={runnable:u,programLog:a,vertexShader:{log:l,prefix:e},fragmentShader:{log:m,prefix:h}})}d.detachShader(q,f);d.detachShader(q,g);d.deleteShader(f);d.deleteShader(g);var v;this.getUniforms=function(){void 0===v&&(v=new Eb(d,q));return v};var n;this.getAttributes=function(){if(void 0===n){for(var a=
{},b=d.getProgramParameter(q,35721),c=0;c<b;c++){var e=d.getActiveAttrib(q,c).name;a[e]=d.getAttribLocation(q,e)}n=a}return n};this.destroy=function(){d.deleteProgram(q);this.program=void 0};this.name=c.shaderName;this.id=wk++;this.cacheKey=b;this.usedTimes=1;this.program=q;this.vertexShader=f;this.fragmentShader=g;return this}function xk(a,b,c){function d(a){if(a)a.isTexture?b=a.encoding:a.isWebGLRenderTarget&&(console.warn(&quot;THREE.WebGLPrograms.getTextureEncodingFromMap: don't use render targets as textures. Use their .texture property instead.&quot;),
b=a.texture.encoding);else var b=3E3;return b}var e=[],f=c.isWebGL2,g=c.logarithmicDepthBuffer,h=c.floatVertexTextures,l=c.precision,m=c.maxVertexUniforms,u=c.vertexTextures,p={MeshDepthMaterial:&quot;depth&quot;,MeshDistanceMaterial:&quot;distanceRGBA&quot;,MeshNormalMaterial:&quot;normal&quot;,MeshBasicMaterial:&quot;basic&quot;,MeshLambertMaterial:&quot;lambert&quot;,MeshPhongMaterial:&quot;phong&quot;,MeshToonMaterial:&quot;toon&quot;,MeshStandardMaterial:&quot;physical&quot;,MeshPhysicalMaterial:&quot;physical&quot;,MeshMatcapMaterial:&quot;matcap&quot;,LineBasicMaterial:&quot;basic&quot;,LineDashedMaterial:&quot;dashed&quot;,
PointsMaterial:&quot;points&quot;,ShadowMaterial:&quot;shadow&quot;,SpriteMaterial:&quot;sprite&quot;},k=&quot;precision isWebGL2 supportsVertexTextures outputEncoding instancing map mapEncoding matcap matcapEncoding envMap envMapMode envMapEncoding envMapCubeUV lightMap lightMapEncoding aoMap emissiveMap emissiveMapEncoding bumpMap normalMap objectSpaceNormalMap tangentSpaceNormalMap clearcoatMap clearcoatRoughnessMap clearcoatNormalMap displacementMap specularMap roughnessMap metalnessMap gradientMap alphaMap combine vertexColors vertexTangents vertexUvs uvsVertexOnly fog useFog fogExp2 flatShading sizeAttenuation logarithmicDepthBuffer skinning maxBones useVertexTexture morphTargets morphNormals maxMorphTargets maxMorphNormals premultipliedAlpha numDirLights numPointLights numSpotLights numHemiLights numRectAreaLights numDirLightShadows numPointLightShadows numSpotLightShadows shadowMapEnabled shadowMapType toneMapping physicallyCorrectLights alphaTest doubleSided flipSided numClippingPlanes numClipIntersection depthPacking dithering sheen&quot;.split(&quot; &quot;);
this.getParameters=function(e,k,x,n,w,z,ha){var q=n.fog;n=e.isMeshStandardMaterial?n.environment:null;n=e.envMap||n;var r=p[e.type];if(ha.isSkinnedMesh){var v=ha.skeleton.bones;if(h)v=1024;else{var E=Math.min(Math.floor((m-20)/4),v.length);E<v.length?(console.warn(&quot;THREE.WebGLRenderer: Skeleton has &quot;+v.length+&quot; bones. This GPU supports &quot;+E+&quot;.&quot;),v=0):v=E}}else v=0;null!==e.precision&&(l=c.getMaxPrecision(e.precision),l!==e.precision&&console.warn(&quot;THREE.WebGLProgram.getParameters:&quot;,e.precision,&quot;not supported, using&quot;,
l,&quot;instead.&quot;));r?(E=eb[r],E={name:e.type,uniforms:Uh.clone(E.uniforms),vertexShader:E.vertexShader,fragmentShader:E.fragmentShader}):E={name:e.type,uniforms:e.uniforms,vertexShader:e.vertexShader,fragmentShader:e.fragmentShader};e.onBeforeCompile(E,a);var ia=a.getRenderTarget();return{isWebGL2:f,shaderID:r,shaderName:E.name,uniforms:E.uniforms,vertexShader:E.vertexShader,fragmentShader:E.fragmentShader,defines:e.defines,isRawShaderMaterial:e.isRawShaderMaterial,isShaderMaterial:e.isShaderMaterial,
precision:l,instancing:!0===ha.isInstancedMesh,supportsVertexTextures:u,outputEncoding:null!==ia?d(ia.texture):a.outputEncoding,map:!!e.map,mapEncoding:d(e.map),matcap:!!e.matcap,matcapEncoding:d(e.matcap),envMap:!!n,envMapMode:n&&n.mapping,envMapEncoding:d(n),envMapCubeUV:!!n&&(306===n.mapping||307===n.mapping),lightMap:!!e.lightMap,lightMapEncoding:d(e.lightMap),aoMap:!!e.aoMap,emissiveMap:!!e.emissiveMap,emissiveMapEncoding:d(e.emissiveMap),bumpMap:!!e.bumpMap,normalMap:!!e.normalMap,objectSpaceNormalMap:1===
e.normalMapType,tangentSpaceNormalMap:0===e.normalMapType,clearcoatMap:!!e.clearcoatMap,clearcoatRoughnessMap:!!e.clearcoatRoughnessMap,clearcoatNormalMap:!!e.clearcoatNormalMap,displacementMap:!!e.displacementMap,roughnessMap:!!e.roughnessMap,metalnessMap:!!e.metalnessMap,specularMap:!!e.specularMap,alphaMap:!!e.alphaMap,gradientMap:!!e.gradientMap,sheen:!!e.sheen,combine:e.combine,vertexTangents:e.normalMap&&e.vertexTangents,vertexColors:e.vertexColors,vertexUvs:!!e.map||!!e.bumpMap||!!e.normalMap||
!!e.specularMap||!!e.alphaMap||!!e.emissiveMap||!!e.roughnessMap||!!e.metalnessMap||!!e.clearcoatMap||!!e.clearcoatRoughnessMap||!!e.clearcoatNormalMap||!!e.displacementMap,uvsVertexOnly:!(e.map||e.bumpMap||e.normalMap||e.specularMap||e.alphaMap||e.emissiveMap||e.roughnessMap||e.metalnessMap||e.clearcoatNormalMap)&&!!e.displacementMap,fog:!!q,useFog:e.fog,fogExp2:q&&q.isFogExp2,flatShading:e.flatShading,sizeAttenuation:e.sizeAttenuation,logarithmicDepthBuffer:g,skinning:e.skinning&&0<v,maxBones:v,
useVertexTexture:h,morphTargets:e.morphTargets,morphNormals:e.morphNormals,maxMorphTargets:a.maxMorphTargets,maxMorphNormals:a.maxMorphNormals,numDirLights:k.directional.length,numPointLights:k.point.length,numSpotLights:k.spot.length,numRectAreaLights:k.rectArea.length,numHemiLights:k.hemi.length,numDirLightShadows:k.directionalShadowMap.length,numPointLightShadows:k.pointShadowMap.length,numSpotLightShadows:k.spotShadowMap.length,numClippingPlanes:w,numClipIntersection:z,dithering:e.dithering,shadowMapEnabled:a.shadowMap.enabled&&
0<x.length,shadowMapType:a.shadowMap.type,toneMapping:e.toneMapped?a.toneMapping:0,physicallyCorrectLights:a.physicallyCorrectLights,premultipliedAlpha:e.premultipliedAlpha,alphaTest:e.alphaTest,doubleSided:2===e.side,flipSided:1===e.side,depthPacking:void 0!==e.depthPacking?e.depthPacking:!1,index0AttributeName:e.index0AttributeName,extensionDerivatives:e.extensions&&e.extensions.derivatives,extensionFragDepth:e.extensions&&e.extensions.fragDepth,extensionDrawBuffers:e.extensions&&e.extensions.drawBuffers,
extensionShaderTextureLOD:e.extensions&&e.extensions.shaderTextureLOD,rendererExtensionFragDepth:f||null!==b.get(&quot;EXT_frag_depth&quot;),rendererExtensionDrawBuffers:f||null!==b.get(&quot;WEBGL_draw_buffers&quot;),rendererExtensionShaderTextureLod:f||null!==b.get(&quot;EXT_shader_texture_lod&quot;),onBeforeCompile:e.onBeforeCompile}};this.getProgramCacheKey=function(b){var c=[];b.shaderID?c.push(b.shaderID):(c.push(b.fragmentShader),c.push(b.vertexShader));if(void 0!==b.defines)for(var d in b.defines)c.push(d),c.push(b.defines[d]);
if(void 0===b.isRawShaderMaterial){for(d=0;d<k.length;d++)c.push(b[k[d]]);c.push(a.outputEncoding);c.push(a.gammaFactor)}c.push(b.onBeforeCompile.toString());return c.join()};this.acquireProgram=function(b,c){for(var d,f=0,g=e.length;f<g;f++){var h=e[f];if(h.cacheKey===c){d=h;++d.usedTimes;break}}void 0===d&&(d=new vk(a,c,b),e.push(d));return d};this.releaseProgram=function(a){if(0===--a.usedTimes){var b=e.indexOf(a);e[b]=e[e.length-1];e.pop();a.destroy()}};this.programs=e}function yk(){var a=new WeakMap;
return{get:function(b){var c=a.get(b);void 0===c&&(c={},a.set(b,c));return c},remove:function(b){a.delete(b)},update:function(b,c,d){a.get(b)[c]=d},dispose:function(){a=new WeakMap}}}function zk(a,b){return a.groupOrder!==b.groupOrder?a.groupOrder-b.groupOrder:a.renderOrder!==b.renderOrder?a.renderOrder-b.renderOrder:a.program!==b.program?a.program.id-b.program.id:a.material.id!==b.material.id?a.material.id-b.material.id:a.z!==b.z?a.z-b.z:a.id-b.id}function Ak(a,b){return a.groupOrder!==b.groupOrder?
a.groupOrder-b.groupOrder:a.renderOrder!==b.renderOrder?a.renderOrder-b.renderOrder:a.z!==b.z?b.z-a.z:a.id-b.id}function Vh(){function a(a,d,e,m,k,p){var g=b[c];void 0===g?(g={id:a.id,object:a,geometry:d,material:e,program:e.program||f,groupOrder:m,renderOrder:a.renderOrder,z:k,group:p},b[c]=g):(g.id=a.id,g.object=a,g.geometry=d,g.material=e,g.program=e.program||f,g.groupOrder=m,g.renderOrder=a.renderOrder,g.z=k,g.group=p);c++;return g}var b=[],c=0,d=[],e=[],f={id:-1};return{opaque:d,transparent:e,
init:function(){c=0;d.length=0;e.length=0},push:function(b,c,f,m,k,p){b=a(b,c,f,m,k,p);(!0===f.transparent?e:d).push(b)},unshift:function(b,c,f,m,k,p){b=a(b,c,f,m,k,p);(!0===f.transparent?e:d).unshift(b)},finish:function(){for(var a=c,d=b.length;a<d;a++){var e=b[a];if(null===e.id)break;e.id=null;e.object=null;e.geometry=null;e.material=null;e.program=null;e.group=null}},sort:function(a,b){1<d.length&&d.sort(a||zk);1<e.length&&e.sort(b||Ak)}}}function Bk(){function a(c){c=c.target;c.removeEventListener(&quot;dispose&quot;,
a);b.delete(c)}var b=new WeakMap;return{get:function(c,d){var e=b.get(c);if(void 0===e){var f=new Vh;b.set(c,new WeakMap);b.get(c).set(d,f);c.addEventListener(&quot;dispose&quot;,a)}else f=e.get(d),void 0===f&&(f=new Vh,e.set(d,f));return f},dispose:function(){b=new WeakMap}}}function Ck(){var a={};return{get:function(b){if(void 0!==a[b.id])return a[b.id];switch(b.type){case &quot;DirectionalLight&quot;:var c={direction:new n,color:new A};break;case &quot;SpotLight&quot;:c={position:new n,direction:new n,color:new A,distance:0,
coneCos:0,penumbraCos:0,decay:0};break;case &quot;PointLight&quot;:c={position:new n,color:new A,distance:0,decay:0};break;case &quot;HemisphereLight&quot;:c={direction:new n,skyColor:new A,groundColor:new A};break;case &quot;RectAreaLight&quot;:c={color:new A,position:new n,halfWidth:new n,halfHeight:new n}}return a[b.id]=c}}}function Dk(){var a={};return{get:function(b){if(void 0!==a[b.id])return a[b.id];switch(b.type){case &quot;DirectionalLight&quot;:var c={shadowBias:0,shadowRadius:1,shadowMapSize:new t};break;case &quot;SpotLight&quot;:c={shadowBias:0,
shadowRadius:1,shadowMapSize:new t};break;case &quot;PointLight&quot;:c={shadowBias:0,shadowRadius:1,shadowMapSize:new t,shadowCameraNear:1,shadowCameraFar:1E3}}return a[b.id]=c}}}function Ek(a,b){return(b.castShadow?1:0)-(a.castShadow?1:0)}function Fk(){for(var a=new Ck,b=Dk(),c={version:0,hash:{directionalLength:-1,pointLength:-1,spotLength:-1,rectAreaLength:-1,hemiLength:-1,numDirectionalShadows:-1,numPointShadows:-1,numSpotShadows:-1},ambient:[0,0,0],probe:[],directional:[],directionalShadow:[],directionalShadowMap:[],
directionalShadowMatrix:[],spot:[],spotShadow:[],spotShadowMap:[],spotShadowMatrix:[],rectArea:[],point:[],pointShadow:[],pointShadowMap:[],pointShadowMatrix:[],hemi:[]},d=0;9>d;d++)c.probe.push(new n);var e=new n,f=new P,g=new P;return{setup:function(d,l,m){for(var h=0,p=0,k=0,r=0;9>r;r++)c.probe[r].set(0,0,0);var q=l=0,v=0,n=0,w=0,z=0,ha=0,U=0;m=m.matrixWorldInverse;d.sort(Ek);r=0;for(var ca=d.length;r<ca;r++){var B=d[r],t=B.color,ia=B.intensity,Ca=B.distance,Ja=B.shadow&&B.shadow.map?B.shadow.map.texture:
null;if(B.isAmbientLight)h+=t.r*ia,p+=t.g*ia,k+=t.b*ia;else if(B.isLightProbe)for(Ja=0;9>Ja;Ja++)c.probe[Ja].addScaledVector(B.sh.coefficients[Ja],ia);else if(B.isDirectionalLight){var H=a.get(B);H.color.copy(B.color).multiplyScalar(B.intensity);H.direction.setFromMatrixPosition(B.matrixWorld);e.setFromMatrixPosition(B.target.matrixWorld);H.direction.sub(e);H.direction.transformDirection(m);B.castShadow&&(ia=B.shadow,t=b.get(B),t.shadowBias=ia.bias,t.shadowRadius=ia.radius,t.shadowMapSize=ia.mapSize,
c.directionalShadow[l]=t,c.directionalShadowMap[l]=Ja,c.directionalShadowMatrix[l]=B.shadow.matrix,z++);c.directional[l]=H;l++}else B.isSpotLight?(H=a.get(B),H.position.setFromMatrixPosition(B.matrixWorld),H.position.applyMatrix4(m),H.color.copy(t).multiplyScalar(ia),H.distance=Ca,H.direction.setFromMatrixPosition(B.matrixWorld),e.setFromMatrixPosition(B.target.matrixWorld),H.direction.sub(e),H.direction.transformDirection(m),H.coneCos=Math.cos(B.angle),H.penumbraCos=Math.cos(B.angle*(1-B.penumbra)),
H.decay=B.decay,B.castShadow&&(ia=B.shadow,t=b.get(B),t.shadowBias=ia.bias,t.shadowRadius=ia.radius,t.shadowMapSize=ia.mapSize,c.spotShadow[v]=t,c.spotShadowMap[v]=Ja,c.spotShadowMatrix[v]=B.shadow.matrix,U++),c.spot[v]=H,v++):B.isRectAreaLight?(H=a.get(B),H.color.copy(t).multiplyScalar(ia),H.position.setFromMatrixPosition(B.matrixWorld),H.position.applyMatrix4(m),g.identity(),f.copy(B.matrixWorld),f.premultiply(m),g.extractRotation(f),H.halfWidth.set(.5*B.width,0,0),H.halfHeight.set(0,.5*B.height,
0),H.halfWidth.applyMatrix4(g),H.halfHeight.applyMatrix4(g),c.rectArea[n]=H,n++):B.isPointLight?(H=a.get(B),H.position.setFromMatrixPosition(B.matrixWorld),H.position.applyMatrix4(m),H.color.copy(B.color).multiplyScalar(B.intensity),H.distance=B.distance,H.decay=B.decay,B.castShadow&&(ia=B.shadow,t=b.get(B),t.shadowBias=ia.bias,t.shadowRadius=ia.radius,t.shadowMapSize=ia.mapSize,t.shadowCameraNear=ia.camera.near,t.shadowCameraFar=ia.camera.far,c.pointShadow[q]=t,c.pointShadowMap[q]=Ja,c.pointShadowMatrix[q]=
B.shadow.matrix,ha++),c.point[q]=H,q++):B.isHemisphereLight&&(H=a.get(B),H.direction.setFromMatrixPosition(B.matrixWorld),H.direction.transformDirection(m),H.direction.normalize(),H.skyColor.copy(B.color).multiplyScalar(ia),H.groundColor.copy(B.groundColor).multiplyScalar(ia),c.hemi[w]=H,w++)}c.ambient[0]=h;c.ambient[1]=p;c.ambient[2]=k;d=c.hash;if(d.directionalLength!==l||d.pointLength!==q||d.spotLength!==v||d.rectAreaLength!==n||d.hemiLength!==w||d.numDirectionalShadows!==z||d.numPointShadows!==
ha||d.numSpotShadows!==U)c.directional.length=l,c.spot.length=v,c.rectArea.length=n,c.point.length=q,c.hemi.length=w,c.directionalShadow.length=z,c.directionalShadowMap.length=z,c.pointShadow.length=ha,c.pointShadowMap.length=ha,c.spotShadow.length=U,c.spotShadowMap.length=U,c.directionalShadowMatrix.length=z,c.pointShadowMatrix.length=ha,c.spotShadowMatrix.length=U,d.directionalLength=l,d.pointLength=q,d.spotLength=v,d.rectAreaLength=n,d.hemiLength=w,d.numDirectionalShadows=z,d.numPointShadows=ha,
d.numSpotShadows=U,c.version=Gk++},state:c}}function Wh(){var a=new Fk,b=[],c=[];return{init:function(){b.length=0;c.length=0},state:{lightsArray:b,shadowsArray:c,lights:a},setupLights:function(d){a.setup(b,c,d)},pushLight:function(a){b.push(a)},pushShadow:function(a){c.push(a)}}}function Hk(){function a(c){c=c.target;c.removeEventListener(&quot;dispose&quot;,a);b.delete(c)}var b=new WeakMap;return{get:function(c,d){if(!1===b.has(c)){var e=new Wh;b.set(c,new WeakMap);b.get(c).set(d,e);c.addEventListener(&quot;dispose&quot;,
a)}else!1===b.get(c).has(d)?(e=new Wh,b.get(c).set(d,e)):e=b.get(c).get(d);return e},dispose:function(){b=new WeakMap}}}function Fb(a){K.call(this);this.type=&quot;MeshDepthMaterial&quot;;this.depthPacking=3200;this.morphTargets=this.skinning=!1;this.displacementMap=this.alphaMap=this.map=null;this.displacementScale=1;this.displacementBias=0;this.wireframe=!1;this.wireframeLinewidth=1;this.fog=!1;this.setValues(a)}function Gb(a){K.call(this);this.type=&quot;MeshDistanceMaterial&quot;;this.referencePosition=new n;this.nearDistance=
1;this.farDistance=1E3;this.morphTargets=this.skinning=!1;this.displacementMap=this.alphaMap=this.map=null;this.displacementScale=1;this.displacementBias=0;this.fog=!1;this.setValues(a)}function Xh(a,b,c){function d(a,b,c){c=a<<0|b<<1|c<<2;var d=p[c];void 0===d&&(d=new Fb({depthPacking:3201,morphTargets:a,skinning:b}),p[c]=d);return d}function e(a,b,c){c=a<<0|b<<1|c<<2;var d=x[c];void 0===d&&(d=new Gb({morphTargets:a,skinning:b}),x[c]=d);return d}function f(b,c,f,g,h,l){var m=b.geometry,p=d,k=b.customDepthMaterial;
!0===f.isPointLight&&(p=e,k=b.customDistanceMaterial);void 0===k?(k=!1,!0===c.morphTargets&&(!0===m.isBufferGeometry?k=m.morphAttributes&&m.morphAttributes.position&&0<m.morphAttributes.position.length:!0===m.isGeometry&&(k=m.morphTargets&&0<m.morphTargets.length)),m=!1,!0===b.isSkinnedMesh&&(!0===c.skinning?m=!0:console.warn(&quot;THREE.WebGLShadowMap: THREE.SkinnedMesh with material.skinning set to false:&quot;,b)),b=p(k,m,!0===b.isInstancedMesh)):b=k;a.localClippingEnabled&&!0===c.clipShadows&&0!==c.clippingPlanes.length&&
(k=b.uuid,p=c.uuid,m=r[k],void 0===m&&(m={},r[k]=m),k=m[p],void 0===k&&(k=b.clone(),m[p]=k),b=k);b.visible=c.visible;b.wireframe=c.wireframe;b.side=3===l?null!==c.shadowSide?c.shadowSide:c.side:null!==c.shadowSide?c.shadowSide:q[c.side];b.clipShadows=c.clipShadows;b.clippingPlanes=c.clippingPlanes;b.clipIntersection=c.clipIntersection;b.wireframeLinewidth=c.wireframeLinewidth;b.linewidth=c.linewidth;!0===f.isPointLight&&!0===b.isMeshDistanceMaterial&&(b.referencePosition.setFromMatrixPosition(f.matrixWorld),
b.nearDistance=g,b.farDistance=h);return b}function g(c,d,e,l,m){if(!1!==c.visible){if(c.layers.test(d.layers)&&(c.isMesh||c.isLine||c.isPoints)&&(c.castShadow||c.receiveShadow&&3===m)&&(!c.frustumCulled||h.intersectsObject(c))){c.modelViewMatrix.multiplyMatrices(e.matrixWorldInverse,c.matrixWorld);var p=b.update(c),k=c.material;if(Array.isArray(k))for(var u=p.groups,x=0,q=u.length;x<q;x++){var r=u[x],v=k[r.materialIndex];v&&v.visible&&(v=f(c,v,l,e.near,e.far,m),a.renderBufferDirect(e,null,p,v,c,
r))}else k.visible&&(v=f(c,k,l,e.near,e.far,m),a.renderBufferDirect(e,null,p,v,c,null))}c=c.children;p=0;for(k=c.length;p<k;p++)g(c[p],d,e,l,m)}}var h=new Hc,l=new t,m=new t,k=new ka,p=[],x=[],r={},q={0:1,1:0,2:2},v=new Ba({defines:{SAMPLE_RATE:.25,HALF_SAMPLE_RATE:.125},uniforms:{shadow_pass:{value:null},resolution:{value:new t},radius:{value:4}},vertexShader:&quot;void main() {\n\tgl_Position = vec4( position, 1.0 );\n}&quot;,fragmentShader:&quot;uniform sampler2D shadow_pass;\nuniform vec2 resolution;\nuniform float radius;\n#include <packing>\nvoid main() {\n  float mean = 0.0;\n  float squared_mean = 0.0;\n\tfloat depth = unpackRGBAToDepth( texture2D( shadow_pass, ( gl_FragCoord.xy  ) / resolution ) );\n  for ( float i = -1.0; i < 1.0 ; i += SAMPLE_RATE) {\n    #ifdef HORIZONAL_PASS\n      vec2 distribution = unpackRGBATo2Half( texture2D( shadow_pass, ( gl_FragCoord.xy + vec2( i, 0.0 ) * radius ) / resolution ) );\n      mean += distribution.x;\n      squared_mean += distribution.y * distribution.y + distribution.x * distribution.x;\n    #else\n      float depth = unpackRGBAToDepth( texture2D( shadow_pass, ( gl_FragCoord.xy + vec2( 0.0,  i )  * radius ) / resolution ) );\n      mean += depth;\n      squared_mean += depth * depth;\n    #endif\n  }\n  mean = mean * HALF_SAMPLE_RATE;\n  squared_mean = squared_mean * HALF_SAMPLE_RATE;\n  float std_dev = sqrt( squared_mean - mean * mean );\n  gl_FragColor = pack2HalfToRGBA( vec2( mean, std_dev ) );\n}&quot;}),
n=v.clone();n.defines.HORIZONAL_PASS=1;var w=new C;w.setAttribute(&quot;position&quot;,new M(new Float32Array([-1,-1,.5,3,-1,.5,-1,3,.5]),3));var z=new S(w,v),ha=this;this.enabled=!1;this.autoUpdate=!0;this.needsUpdate=!1;this.type=1;this.render=function(d,e,f){if(!1!==ha.enabled&&(!1!==ha.autoUpdate||!1!==ha.needsUpdate)&&0!==d.length){var p=a.getRenderTarget(),u=a.getActiveCubeFace(),x=a.getActiveMipmapLevel(),q=a.state;q.setBlending(0);q.buffers.color.setClear(1,1,1,1);q.buffers.depth.setTest(!0);q.setScissorTest(!1);
for(var r=0,E=d.length;r<E;r++){var w=d[r],B=w.shadow;if(void 0===B)console.warn(&quot;THREE.WebGLShadowMap:&quot;,w,&quot;has no shadow.&quot;);else{l.copy(B.mapSize);var t=B.getFrameExtents();l.multiply(t);m.copy(B.mapSize);if(l.x>c||l.y>c)console.warn(&quot;THREE.WebGLShadowMap:&quot;,w,&quot;has shadow exceeding max texture size, reducing&quot;),l.x>c&&(m.x=Math.floor(c/t.x),l.x=m.x*t.x,B.mapSize.x=m.x),l.y>c&&(m.y=Math.floor(c/t.y),l.y=m.y*t.y,B.mapSize.y=m.y);null!==B.map||B.isPointLightShadow||3!==this.type||(t={minFilter:1006,magFilter:1006,
format:1023},B.map=new Ha(l.x,l.y,t),B.map.texture.name=w.name+&quot;.shadowMap&quot;,B.mapPass=new Ha(l.x,l.y,t),B.camera.updateProjectionMatrix());null===B.map&&(t={minFilter:1003,magFilter:1003,format:1023},B.map=new Ha(l.x,l.y,t),B.map.texture.name=w.name+&quot;.shadowMap&quot;,B.camera.updateProjectionMatrix());a.setRenderTarget(B.map);a.clear();t=B.getViewportCount();for(var ca=0;ca<t;ca++){var U=B.getViewport(ca);k.set(m.x*U.x,m.y*U.y,m.x*U.z,m.y*U.w);q.viewport(k);B.updateMatrices(w,ca);h=B.getFrustum();g(e,
f,B.camera,w,this.type)}B.isPointLightShadow||3!==this.type||(w=B,B=f,t=b.update(z),v.uniforms.shadow_pass.value=w.map.texture,v.uniforms.resolution.value=w.mapSize,v.uniforms.radius.value=w.radius,a.setRenderTarget(w.mapPass),a.clear(),a.renderBufferDirect(B,null,t,v,z,null),n.uniforms.shadow_pass.value=w.mapPass.texture,n.uniforms.resolution.value=w.mapSize,n.uniforms.radius.value=w.radius,a.setRenderTarget(w.map),a.clear(),a.renderBufferDirect(B,null,t,n,z,null))}}ha.needsUpdate=!1;a.setRenderTarget(p,
u,x)}}}function Ik(a,b,c){function d(b,c,d){var e=new Uint8Array(4),f=a.createTexture();a.bindTexture(b,f);a.texParameteri(b,10241,9728);a.texParameteri(b,10240,9728);for(b=0;b<d;b++)a.texImage2D(c+b,0,6408,1,1,0,6408,5121,e);return f}function e(c,d){n[c]=1;0===w[c]&&(a.enableVertexAttribArray(c),w[c]=1);z[c]!==d&&((x?a:b.get(&quot;ANGLE_instanced_arrays&quot;))[x?&quot;vertexAttribDivisor&quot;:&quot;vertexAttribDivisorANGLE&quot;](c,d),z[c]=d)}function f(b){!0!==t[b]&&(a.enable(b),t[b]=!0)}function g(b){!1!==t[b]&&(a.disable(b),
t[b]=!1)}function h(b,c,d,e,h,l,m,p){if(0===b)ca&&(g(3042),ca=!1);else if(ca||(f(3042),ca=!0),5!==b){if(b!==B||p!==F){if(100!==y||100!==Ja)a.blendEquation(32774),Ja=y=100;if(p)switch(b){case 1:a.blendFuncSeparate(1,771,1,771);break;case 2:a.blendFunc(1,1);break;case 3:a.blendFuncSeparate(0,0,769,771);break;case 4:a.blendFuncSeparate(0,768,0,770);break;default:console.error(&quot;THREE.WebGLState: Invalid blending: &quot;,b)}else switch(b){case 1:a.blendFuncSeparate(770,771,1,771);break;case 2:a.blendFunc(770,
1);break;case 3:a.blendFunc(0,769);break;case 4:a.blendFunc(0,768);break;default:console.error(&quot;THREE.WebGLState: Invalid blending: &quot;,b)}D=H=Ca=ia=null;B=b;F=p}}else{h=h||c;l=l||d;m=m||e;if(c!==y||h!==Ja)a.blendEquationSeparate(Lc[c],Lc[h]),y=c,Ja=h;if(d!==ia||e!==Ca||l!==H||m!==D)a.blendFuncSeparate(J[d],J[e],J[l],J[m]),ia=d,Ca=e,H=l,D=m;B=b;F=null}}function l(b){A!==b&&(b?a.frontFace(2304):a.frontFace(2305),A=b)}function m(b){0!==b?(f(2884),b!==C&&(1===b?a.cullFace(1029):2===b?a.cullFace(1028):
a.cullFace(1032))):g(2884);C=b}function k(b,c,d){if(b){if(f(32823),K!==c||M!==d)a.polygonOffset(c,d),K=c,M=d}else g(32823)}function p(b){void 0===b&&(b=33984+fa-1);L!==b&&(a.activeTexture(b),L=b)}var x=c.isWebGL2,r=new function(){var b=!1,c=new ka,d=null,e=new ka(0,0,0,0);return{setMask:function(c){d===c||b||(a.colorMask(c,c,c,c),d=c)},setLocked:function(a){b=a},setClear:function(b,d,f,g,h){!0===h&&(b*=g,d*=g,f*=g);c.set(b,d,f,g);!1===e.equals(c)&&(a.clearColor(b,d,f,g),e.copy(c))},reset:function(){b=
!1;d=null;e.set(-1,0,0,0)}}},q=new function(){var b=!1,c=null,d=null,e=null;return{setTest:function(a){a?f(2929):g(2929)},setMask:function(d){c===d||b||(a.depthMask(d),c=d)},setFunc:function(b){if(d!==b){if(b)switch(b){case 0:a.depthFunc(512);break;case 1:a.depthFunc(519);break;case 2:a.depthFunc(513);break;case 3:a.depthFunc(515);break;case 4:a.depthFunc(514);break;case 5:a.depthFunc(518);break;case 6:a.depthFunc(516);break;case 7:a.depthFunc(517);break;default:a.depthFunc(515)}else a.depthFunc(515);
d=b}},setLocked:function(a){b=a},setClear:function(b){e!==b&&(a.clearDepth(b),e=b)},reset:function(){b=!1;e=d=c=null}}},v=new function(){var b=!1,c=null,d=null,e=null,h=null,l=null,m=null,p=null,k=null;return{setTest:function(a){b||(a?f(2960):g(2960))},setMask:function(d){c===d||b||(a.stencilMask(d),c=d)},setFunc:function(b,c,f){if(d!==b||e!==c||h!==f)a.stencilFunc(b,c,f),d=b,e=c,h=f},setOp:function(b,c,d){if(l!==b||m!==c||p!==d)a.stencilOp(b,c,d),l=b,m=c,p=d},setLocked:function(a){b=a},setClear:function(b){k!==
b&&(a.clearStencil(b),k=b)},reset:function(){b=!1;k=p=m=l=h=e=d=c=null}}};c=a.getParameter(34921);var n=new Uint8Array(c),w=new Uint8Array(c),z=new Uint8Array(c),t={},U=null,ca=null,B=null,y=null,ia=null,Ca=null,Ja=null,H=null,D=null,F=!1,A=null,C=null,G=null,K=null,M=null,fa=a.getParameter(35661),N=!1;c=0;c=a.getParameter(7938);-1!==c.indexOf(&quot;WebGL&quot;)?(c=parseFloat(/^WebGL ([0-9])/.exec(c)[1]),N=1<=c):-1!==c.indexOf(&quot;OpenGL ES&quot;)&&(c=parseFloat(/^OpenGL ES ([0-9])/.exec(c)[1]),N=2<=c);var L=null,
Nd={},Z=new ka,Yh=new ka,ng={};ng[3553]=d(3553,3553,1);ng[34067]=d(34067,34069,6);r.setClear(0,0,0,1);q.setClear(1);v.setClear(0);f(2929);q.setFunc(3);l(!1);m(1);f(2884);h(0);var Lc={100:32774,101:32778,102:32779};x?(Lc[103]=32775,Lc[104]=32776):(c=b.get(&quot;EXT_blend_minmax&quot;),null!==c&&(Lc[103]=c.MIN_EXT,Lc[104]=c.MAX_EXT));var J={200:0,201:1,202:768,204:770,210:776,208:774,206:772,203:769,205:771,209:775,207:773};return{buffers:{color:r,depth:q,stencil:v},initAttributes:function(){for(var a=0,b=n.length;a<
b;a++)n[a]=0},enableAttribute:function(a){e(a,0)},enableAttributeAndDivisor:e,disableUnusedAttributes:function(){for(var b=0,c=w.length;b!==c;++b)w[b]!==n[b]&&(a.disableVertexAttribArray(b),w[b]=0)},enable:f,disable:g,useProgram:function(b){return U!==b?(a.useProgram(b),U=b,!0):!1},setBlending:h,setMaterial:function(a,b){2===a.side?g(2884):f(2884);var c=1===a.side;b&&(c=!c);l(c);1===a.blending&&!1===a.transparent?h(0):h(a.blending,a.blendEquation,a.blendSrc,a.blendDst,a.blendEquationAlpha,a.blendSrcAlpha,
a.blendDstAlpha,a.premultipliedAlpha);q.setFunc(a.depthFunc);q.setTest(a.depthTest);q.setMask(a.depthWrite);r.setMask(a.colorWrite);b=a.stencilWrite;v.setTest(b);b&&(v.setMask(a.stencilWriteMask),v.setFunc(a.stencilFunc,a.stencilRef,a.stencilFuncMask),v.setOp(a.stencilFail,a.stencilZFail,a.stencilZPass));k(a.polygonOffset,a.polygonOffsetFactor,a.polygonOffsetUnits)},setFlipSided:l,setCullFace:m,setLineWidth:function(b){b!==G&&(N&&a.lineWidth(b),G=b)},setPolygonOffset:k,setScissorTest:function(a){a?
f(3089):g(3089)},activeTexture:p,bindTexture:function(b,c){null===L&&p();var d=Nd[L];void 0===d&&(d={type:void 0,texture:void 0},Nd[L]=d);if(d.type!==b||d.texture!==c)a.bindTexture(b,c||ng[b]),d.type=b,d.texture=c},unbindTexture:function(){var b=Nd[L];void 0!==b&&void 0!==b.type&&(a.bindTexture(b.type,null),b.type=void 0,b.texture=void 0)},compressedTexImage2D:function(){try{a.compressedTexImage2D.apply(a,arguments)}catch(Q){console.error(&quot;THREE.WebGLState:&quot;,Q)}},texImage2D:function(){try{a.texImage2D.apply(a,
arguments)}catch(Q){console.error(&quot;THREE.WebGLState:&quot;,Q)}},texImage3D:function(){try{a.texImage3D.apply(a,arguments)}catch(Q){console.error(&quot;THREE.WebGLState:&quot;,Q)}},scissor:function(b){!1===Z.equals(b)&&(a.scissor(b.x,b.y,b.z,b.w),Z.copy(b))},viewport:function(b){!1===Yh.equals(b)&&(a.viewport(b.x,b.y,b.z,b.w),Yh.copy(b))},reset:function(){for(var b=0;b<w.length;b++)1===w[b]&&(a.disableVertexAttribArray(b),w[b]=0);t={};L=null;Nd={};C=A=B=U=null;r.reset();q.reset();v.reset()}}}function Jk(a,b,c,d,
e,f,g){function h(a,b){return K?new OffscreenCanvas(a,b):document.createElementNS(&quot;http://www.w3.org/1999/xhtml&quot;,&quot;canvas&quot;)}function l(a,b,c,d){var e=1;if(a.width>d||a.height>d)e=d/Math.max(a.width,a.height);if(1>e||!0===b){if(&quot;undefined&quot;!==typeof HTMLImageElement&&a instanceof HTMLImageElement||&quot;undefined&quot;!==typeof HTMLCanvasElement&&a instanceof HTMLCanvasElement||&quot;undefined&quot;!==typeof ImageBitmap&&a instanceof ImageBitmap)return d=b?L.floorPowerOfTwo:Math.floor,b=d(e*a.width),e=d(e*a.height),void 0===
G&&(G=h(b,e)),c=c?h(b,e):G,c.width=b,c.height=e,c.getContext(&quot;2d&quot;).drawImage(a,0,0,b,e),console.warn(&quot;THREE.WebGLRenderer: Texture has been resized from (&quot;+a.width+&quot;x&quot;+a.height+&quot;) to (&quot;+b+&quot;x&quot;+e+&quot;).&quot;),c;&quot;data&quot;in a&&console.warn(&quot;THREE.WebGLRenderer: Image in DataTexture is too big (&quot;+a.width+&quot;x&quot;+a.height+&quot;).&quot;)}return a}function m(a){return L.isPowerOfTwo(a.width)&&L.isPowerOfTwo(a.height)}function k(a,b){return a.generateMipmaps&&b&&1003!==a.minFilter&&1006!==a.minFilter}function p(b,c,e,f){a.generateMipmap(b);
d.get(c).__maxMipLevel=Math.log(Math.max(e,f))*Math.LOG2E}function x(c,d,e){if(!1===Ca)return d;if(null!==c){if(void 0!==a[c])return a[c];console.warn(&quot;THREE.WebGLRenderer: Attempt to use non-existing WebGL internal format '&quot;+c+&quot;'&quot;)}c=d;6403===d&&(5126===e&&(c=33326),5131===e&&(c=33325),5121===e&&(c=33321));6407===d&&(5126===e&&(c=34837),5131===e&&(c=34843),5121===e&&(c=32849));6408===d&&(5126===e&&(c=34836),5131===e&&(c=34842),5121===e&&(c=32856));33325!==c&&33326!==c&&34842!==c&&34836!==c||b.get(&quot;EXT_color_buffer_float&quot;);
return c}function r(a){return 1003===a||1004===a||1005===a?9728:9729}function q(b){b=b.target;b.removeEventListener(&quot;dispose&quot;,q);var c=d.get(b);void 0!==c.__webglInit&&(a.deleteTexture(c.__webglTexture),d.remove(b));b.isVideoTexture&&C.delete(b);g.memory.textures--}function v(b){b=b.target;b.removeEventListener(&quot;dispose&quot;,v);var c=d.get(b),e=d.get(b.texture);if(b){void 0!==e.__webglTexture&&a.deleteTexture(e.__webglTexture);b.depthTexture&&b.depthTexture.dispose();if(b.isWebGLCubeRenderTarget)for(e=
0;6>e;e++)a.deleteFramebuffer(c.__webglFramebuffer[e]),c.__webglDepthbuffer&&a.deleteRenderbuffer(c.__webglDepthbuffer[e]);else a.deleteFramebuffer(c.__webglFramebuffer),c.__webglDepthbuffer&&a.deleteRenderbuffer(c.__webglDepthbuffer),c.__webglMultisampledFramebuffer&&a.deleteFramebuffer(c.__webglMultisampledFramebuffer),c.__webglColorRenderbuffer&&a.deleteRenderbuffer(c.__webglColorRenderbuffer),c.__webglDepthRenderbuffer&&a.deleteRenderbuffer(c.__webglDepthRenderbuffer);d.remove(b.texture);d.remove(b)}g.memory.textures--}
function n(a,b){var e=d.get(a);if(a.isVideoTexture){var f=g.render.frame;C.get(a)!==f&&(C.set(a,f),a.update())}if(0<a.version&&e.__version!==a.version)if(f=a.image,void 0===f)console.warn(&quot;THREE.WebGLRenderer: Texture marked for update but image is undefined&quot;);else if(!1===f.complete)console.warn(&quot;THREE.WebGLRenderer: Texture marked for update but image is incomplete&quot;);else{ca(e,a,b);return}c.activeTexture(33984+b);c.bindTexture(3553,e.__webglTexture)}function w(b,e){if(6===b.image.length){var g=
d.get(b);if(0<b.version&&g.__version!==b.version){U(g,b);c.activeTexture(33984+e);c.bindTexture(34067,g.__webglTexture);a.pixelStorei(37440,b.flipY);var h=b&&(b.isCompressedTexture||b.image[0].isCompressedTexture);e=b.image[0]&&b.image[0].isDataTexture;for(var u=[],q=0;6>q;q++)u[q]=h||e?e?b.image[q].image:b.image[q]:l(b.image[q],!1,!0,H);var r=u[0],v=m(r)||Ca,n=f.convert(b.format),w=f.convert(b.type),E=x(b.internalFormat,n,w);t(34067,b,v);if(h){for(q=0;6>q;q++){var Z=u[q].mipmaps;for(h=0;h<Z.length;h++){var z=
Z[h];1023!==b.format&&1022!==b.format?null!==n?c.compressedTexImage2D(34069+q,h,E,z.width,z.height,0,z.data):console.warn(&quot;THREE.WebGLRenderer: Attempt to load unsupported compressed texture format in .setTextureCube()&quot;):c.texImage2D(34069+q,h,E,z.width,z.height,0,n,w,z.data)}}g.__maxMipLevel=Z.length-1}else{Z=b.mipmaps;for(q=0;6>q;q++)if(e)for(c.texImage2D(34069+q,0,E,u[q].width,u[q].height,0,n,w,u[q].data),h=0;h<Z.length;h++)z=Z[h],z=z.image[q].image,c.texImage2D(34069+q,h+1,E,z.width,z.height,
0,n,w,z.data);else for(c.texImage2D(34069+q,0,E,n,w,u[q]),h=0;h<Z.length;h++)z=Z[h],c.texImage2D(34069+q,h+1,E,n,w,z.image[q]);g.__maxMipLevel=Z.length}k(b,v)&&p(34067,b,r.width,r.height);g.__version=b.version;if(b.onUpdate)b.onUpdate(b)}else c.activeTexture(33984+e),c.bindTexture(34067,g.__webglTexture)}}function z(a,b){c.activeTexture(33984+b);c.bindTexture(34067,d.get(a).__webglTexture)}function t(c,f,g){g?(a.texParameteri(c,10242,N[f.wrapS]),a.texParameteri(c,10243,N[f.wrapT]),32879!==c&&35866!==
c||a.texParameteri(c,32882,N[f.wrapR]),a.texParameteri(c,10240,fa[f.magFilter]),a.texParameteri(c,10241,fa[f.minFilter])):(a.texParameteri(c,10242,33071),a.texParameteri(c,10243,33071),32879!==c&&35866!==c||a.texParameteri(c,32882,33071),1001===f.wrapS&&1001===f.wrapT||console.warn(&quot;THREE.WebGLRenderer: Texture is not power of two. Texture.wrapS and Texture.wrapT should be set to THREE.ClampToEdgeWrapping.&quot;),a.texParameteri(c,10240,r(f.magFilter)),a.texParameteri(c,10241,r(f.minFilter)),1003!==f.minFilter&&
1006!==f.minFilter&&console.warn(&quot;THREE.WebGLRenderer: Texture is not power of two. Texture.minFilter should be set to THREE.NearestFilter or THREE.LinearFilter.&quot;));!(g=b.get(&quot;EXT_texture_filter_anisotropic&quot;))||1015===f.type&&null===b.get(&quot;OES_texture_float_linear&quot;)||1016===f.type&&null===(Ca||b.get(&quot;OES_texture_half_float_linear&quot;))||!(1<f.anisotropy||d.get(f).__currentAnisotropy)||(a.texParameterf(c,g.TEXTURE_MAX_ANISOTROPY_EXT,Math.min(f.anisotropy,e.getMaxAnisotropy())),d.get(f).__currentAnisotropy=
f.anisotropy)}function U(b,c){void 0===b.__webglInit&&(b.__webglInit=!0,c.addEventListener(&quot;dispose&quot;,q),b.__webglTexture=a.createTexture(),g.memory.textures++)}function ca(b,d,e){var g=3553;d.isDataTexture2DArray&&(g=35866);d.isDataTexture3D&&(g=32879);U(b,d);c.activeTexture(33984+e);c.bindTexture(g,b.__webglTexture);a.pixelStorei(37440,d.flipY);a.pixelStorei(37441,d.premultiplyAlpha);a.pixelStorei(3317,d.unpackAlignment);e=Ca?!1:1001!==d.wrapS||1001!==d.wrapT||1003!==d.minFilter&&1006!==d.minFilter;
e=e&&!1===m(d.image);e=l(d.image,e,!1,F);var h=m(e)||Ca,u=f.convert(d.format),q=f.convert(d.type),r=x(d.internalFormat,u,q);t(g,d,h);var v=d.mipmaps;if(d.isDepthTexture)r=6402,Ca?r=1015===d.type?36012:1014===d.type?33190:1020===d.type?35056:33189:1015===d.type&&console.error(&quot;WebGLRenderer: Floating point depth texture requires WebGL2.&quot;),1026===d.format&&6402===r&&1012!==d.type&&1014!==d.type&&(console.warn(&quot;THREE.WebGLRenderer: Use UnsignedShortType or UnsignedIntType for DepthFormat DepthTexture.&quot;),
d.type=1012,q=f.convert(d.type)),1027===d.format&&6402===r&&(r=34041,1020!==d.type&&(console.warn(&quot;THREE.WebGLRenderer: Use UnsignedInt248Type for DepthStencilFormat DepthTexture.&quot;),d.type=1020,q=f.convert(d.type))),c.texImage2D(3553,0,r,e.width,e.height,0,u,q,null);else if(d.isDataTexture)if(0<v.length&&h){for(var n=0,w=v.length;n<w;n++){var E=v[n];c.texImage2D(3553,n,r,E.width,E.height,0,u,q,E.data)}d.generateMipmaps=!1;b.__maxMipLevel=v.length-1}else c.texImage2D(3553,0,r,e.width,e.height,0,u,
q,e.data),b.__maxMipLevel=0;else if(d.isCompressedTexture){n=0;for(w=v.length;n<w;n++)E=v[n],1023!==d.format&&1022!==d.format?null!==u?c.compressedTexImage2D(3553,n,r,E.width,E.height,0,E.data):console.warn(&quot;THREE.WebGLRenderer: Attempt to load unsupported compressed texture format in .uploadTexture()&quot;):c.texImage2D(3553,n,r,E.width,E.height,0,u,q,E.data);b.__maxMipLevel=v.length-1}else if(d.isDataTexture2DArray)c.texImage3D(35866,0,r,e.width,e.height,e.depth,0,u,q,e.data),b.__maxMipLevel=0;else if(d.isDataTexture3D)c.texImage3D(32879,
0,r,e.width,e.height,e.depth,0,u,q,e.data),b.__maxMipLevel=0;else if(0<v.length&&h){n=0;for(w=v.length;n<w;n++)E=v[n],c.texImage2D(3553,n,r,u,q,E);d.generateMipmaps=!1;b.__maxMipLevel=v.length-1}else c.texImage2D(3553,0,r,u,q,e),b.__maxMipLevel=0;k(d,h)&&p(g,d,e.width,e.height);b.__version=d.version;if(d.onUpdate)d.onUpdate(d)}function B(b,e,g,h){var l=f.convert(e.texture.format),m=f.convert(e.texture.type),p=x(e.texture.internalFormat,l,m);c.texImage2D(h,0,p,e.width,e.height,0,l,m,null);a.bindFramebuffer(36160,
b);a.framebufferTexture2D(36160,g,h,d.get(e.texture).__webglTexture,0);a.bindFramebuffer(36160,null)}function y(b,c,d){a.bindRenderbuffer(36161,b);if(c.depthBuffer&&!c.stencilBuffer){var e=33189;d?((d=c.depthTexture)&&d.isDepthTexture&&(1015===d.type?e=36012:1014===d.type&&(e=33190)),d=ia(c),a.renderbufferStorageMultisample(36161,d,e,c.width,c.height)):a.renderbufferStorage(36161,e,c.width,c.height);a.framebufferRenderbuffer(36160,36096,36161,b)}else c.depthBuffer&&c.stencilBuffer?(d?(d=ia(c),a.renderbufferStorageMultisample(36161,
d,35056,c.width,c.height)):a.renderbufferStorage(36161,34041,c.width,c.height),a.framebufferRenderbuffer(36160,33306,36161,b)):(b=f.convert(c.texture.format),e=f.convert(c.texture.type),e=x(c.texture.internalFormat,b,e),d?(d=ia(c),a.renderbufferStorageMultisample(36161,d,e,c.width,c.height)):a.renderbufferStorage(36161,e,c.width,c.height));a.bindRenderbuffer(36161,null)}function ia(a){return Ca&&a.isWebGLMultisampleRenderTarget?Math.min(A,a.samples):0}var Ca=e.isWebGL2,D=e.maxTextures,H=e.maxCubemapSize,
F=e.maxTextureSize,A=e.maxSamples,C=new WeakMap,G,K=!1;try{K=&quot;undefined&quot;!==typeof OffscreenCanvas&&null!==(new OffscreenCanvas(1,1)).getContext(&quot;2d&quot;)}catch(Nd){}var M=0,N={1E3:10497,1001:33071,1002:33648},fa={1003:9728,1004:9984,1005:9986,1006:9729,1007:9985,1008:9987},P=!1,O=!1;this.allocateTextureUnit=function(){var a=M;a>=D&&console.warn(&quot;THREE.WebGLTextures: Trying to use &quot;+a+&quot; texture units while this GPU supports only &quot;+D);M+=1;return a};this.resetTextureUnits=function(){M=0};this.setTexture2D=
n;this.setTexture2DArray=function(a,b){var e=d.get(a);0<a.version&&e.__version!==a.version?ca(e,a,b):(c.activeTexture(33984+b),c.bindTexture(35866,e.__webglTexture))};this.setTexture3D=function(a,b){var e=d.get(a);0<a.version&&e.__version!==a.version?ca(e,a,b):(c.activeTexture(33984+b),c.bindTexture(32879,e.__webglTexture))};this.setTextureCube=w;this.setTextureCubeDynamic=z;this.setupRenderTarget=function(b){var e=d.get(b),h=d.get(b.texture);b.addEventListener(&quot;dispose&quot;,v);h.__webglTexture=a.createTexture();
g.memory.textures++;var l=!0===b.isWebGLCubeRenderTarget,u=!0===b.isWebGLMultisampleRenderTarget,q=m(b)||Ca;!Ca||1022!==b.texture.format||1015!==b.texture.type&&1016!==b.texture.type||(b.texture.format=1023,console.warn(&quot;THREE.WebGLRenderer: Rendering to textures with RGB format is not supported. Using RGBA format instead.&quot;));if(l)for(e.__webglFramebuffer=[],u=0;6>u;u++)e.__webglFramebuffer[u]=a.createFramebuffer();else if(e.__webglFramebuffer=a.createFramebuffer(),u)if(Ca){e.__webglMultisampledFramebuffer=
a.createFramebuffer();e.__webglColorRenderbuffer=a.createRenderbuffer();a.bindRenderbuffer(36161,e.__webglColorRenderbuffer);u=f.convert(b.texture.format);var r=f.convert(b.texture.type);u=x(b.texture.internalFormat,u,r);r=ia(b);a.renderbufferStorageMultisample(36161,r,u,b.width,b.height);a.bindFramebuffer(36160,e.__webglMultisampledFramebuffer);a.framebufferRenderbuffer(36160,36064,36161,e.__webglColorRenderbuffer);a.bindRenderbuffer(36161,null);b.depthBuffer&&(e.__webglDepthRenderbuffer=a.createRenderbuffer(),
y(e.__webglDepthRenderbuffer,b,!0));a.bindFramebuffer(36160,null)}else console.warn(&quot;THREE.WebGLRenderer: WebGLMultisampleRenderTarget can only be used with WebGL2.&quot;);if(l){c.bindTexture(34067,h.__webglTexture);t(34067,b.texture,q);for(u=0;6>u;u++)B(e.__webglFramebuffer[u],b,36064,34069+u);k(b.texture,q)&&p(34067,b.texture,b.width,b.height);c.bindTexture(34067,null)}else c.bindTexture(3553,h.__webglTexture),t(3553,b.texture,q),B(e.__webglFramebuffer,b,36064,3553),k(b.texture,q)&&p(3553,b.texture,
b.width,b.height),c.bindTexture(3553,null);if(b.depthBuffer){e=d.get(b);h=!0===b.isWebGLCubeRenderTarget;if(b.depthTexture){if(h)throw Error(&quot;target.depthTexture not supported in Cube render targets&quot;);if(b&&b.isWebGLCubeRenderTarget)throw Error(&quot;Depth Texture with cube render targets is not supported&quot;);a.bindFramebuffer(36160,e.__webglFramebuffer);if(!b.depthTexture||!b.depthTexture.isDepthTexture)throw Error(&quot;renderTarget.depthTexture must be an instance of THREE.DepthTexture&quot;);d.get(b.depthTexture).__webglTexture&&
b.depthTexture.image.width===b.width&&b.depthTexture.image.height===b.height||(b.depthTexture.image.width=b.width,b.depthTexture.image.height=b.height,b.depthTexture.needsUpdate=!0);n(b.depthTexture,0);e=d.get(b.depthTexture).__webglTexture;if(1026===b.depthTexture.format)a.framebufferTexture2D(36160,36096,3553,e,0);else if(1027===b.depthTexture.format)a.framebufferTexture2D(36160,33306,3553,e,0);else throw Error(&quot;Unknown depthTexture format&quot;);}else if(h)for(e.__webglDepthbuffer=[],h=0;6>h;h++)a.bindFramebuffer(36160,
e.__webglFramebuffer[h]),e.__webglDepthbuffer[h]=a.createRenderbuffer(),y(e.__webglDepthbuffer[h],b,!1);else a.bindFramebuffer(36160,e.__webglFramebuffer),e.__webglDepthbuffer=a.createRenderbuffer(),y(e.__webglDepthbuffer,b,!1);a.bindFramebuffer(36160,null)}};this.updateRenderTargetMipmap=function(a){var b=a.texture,e=m(a)||Ca;if(k(b,e)){e=a.isWebGLCubeRenderTarget?34067:3553;var f=d.get(b).__webglTexture;c.bindTexture(e,f);p(e,b,a.width,a.height);c.bindTexture(e,null)}};this.updateMultisampleRenderTarget=
function(b){if(b.isWebGLMultisampleRenderTarget)if(Ca){var c=d.get(b);a.bindFramebuffer(36008,c.__webglMultisampledFramebuffer);a.bindFramebuffer(36009,c.__webglFramebuffer);var e=b.width,f=b.height,g=16384;b.depthBuffer&&(g|=256);b.stencilBuffer&&(g|=1024);a.blitFramebuffer(0,0,e,f,0,0,e,f,g,9728);a.bindFramebuffer(36160,c.__webglMultisampledFramebuffer)}else console.warn(&quot;THREE.WebGLRenderer: WebGLMultisampleRenderTarget can only be used with WebGL2.&quot;)};this.safeSetTexture2D=function(a,b){a&&a.isWebGLRenderTarget&&
(!1===P&&(console.warn(&quot;THREE.WebGLTextures.safeSetTexture2D: don't use render targets as textures. Use their .texture property instead.&quot;),P=!0),a=a.texture);n(a,b)};this.safeSetTextureCube=function(a,b){a&&a.isWebGLCubeRenderTarget&&(!1===O&&(console.warn(&quot;THREE.WebGLTextures.safeSetTextureCube: don't use cube render targets as textures. Use their .texture property instead.&quot;),O=!0),a=a.texture);a&&a.isCubeTexture||Array.isArray(a.image)&&6===a.image.length?w(a,b):z(a,b)}}function Zh(a,b,c){var d=
c.isWebGL2;return{convert:function(a){if(1009===a)return 5121;if(1017===a)return 32819;if(1018===a)return 32820;if(1019===a)return 33635;if(1010===a)return 5120;if(1011===a)return 5122;if(1012===a)return 5123;if(1013===a)return 5124;if(1014===a)return 5125;if(1015===a)return 5126;if(1016===a){if(d)return 5131;var c=b.get(&quot;OES_texture_half_float&quot;);return null!==c?c.HALF_FLOAT_OES:null}if(1021===a)return 6406;if(1022===a)return 6407;if(1023===a)return 6408;if(1024===a)return 6409;if(1025===a)return 6410;
if(1026===a)return 6402;if(1027===a)return 34041;if(1028===a)return 6403;if(1029===a)return 36244;if(1030===a)return 33319;if(1031===a)return 33320;if(1032===a)return 36248;if(1033===a)return 36249;if(33776===a||33777===a||33778===a||33779===a)if(c=b.get(&quot;WEBGL_compressed_texture_s3tc&quot;),null!==c){if(33776===a)return c.COMPRESSED_RGB_S3TC_DXT1_EXT;if(33777===a)return c.COMPRESSED_RGBA_S3TC_DXT1_EXT;if(33778===a)return c.COMPRESSED_RGBA_S3TC_DXT3_EXT;if(33779===a)return c.COMPRESSED_RGBA_S3TC_DXT5_EXT}else return null;
if(35840===a||35841===a||35842===a||35843===a)if(c=b.get(&quot;WEBGL_compressed_texture_pvrtc&quot;),null!==c){if(35840===a)return c.COMPRESSED_RGB_PVRTC_4BPPV1_IMG;if(35841===a)return c.COMPRESSED_RGB_PVRTC_2BPPV1_IMG;if(35842===a)return c.COMPRESSED_RGBA_PVRTC_4BPPV1_IMG;if(35843===a)return c.COMPRESSED_RGBA_PVRTC_2BPPV1_IMG}else return null;if(36196===a)return c=b.get(&quot;WEBGL_compressed_texture_etc1&quot;),null!==c?c.COMPRESSED_RGB_ETC1_WEBGL:null;if(37492===a||37496===a)if(c=b.get(&quot;WEBGL_compressed_texture_etc&quot;),
null!==c){if(37492===a)return c.COMPRESSED_RGB8_ETC2;if(37496===a)return c.COMPRESSED_RGBA8_ETC2_EAC}if(37808===a||37809===a||37810===a||37811===a||37812===a||37813===a||37814===a||37815===a||37816===a||37817===a||37818===a||37819===a||37820===a||37821===a||37840===a||37841===a||37842===a||37843===a||37844===a||37845===a||37846===a||37847===a||37848===a||37849===a||37850===a||37851===a||37852===a||37853===a)return c=b.get(&quot;WEBGL_compressed_texture_astc&quot;),null!==c?a:null;if(36492===a)return c=b.get(&quot;EXT_texture_compression_bptc&quot;),
null!==c?a:null;if(1020===a){if(d)return 34042;c=b.get(&quot;WEBGL_depth_texture&quot;);return null!==c?c.UNSIGNED_INT_24_8_WEBGL:null}}}}function Pe(a){aa.call(this);this.cameras=a||[]}function Mc(){F.call(this);this.type=&quot;Group&quot;}function $h(a,b){function c(a){var b=q.get(a.inputSource);b&&(b.targetRay&&b.targetRay.dispatchEvent({type:a.type}),b.grip&&b.grip.dispatchEvent({type:a.type}))}function d(){q.forEach(function(a,b){a.targetRay&&(a.targetRay.dispatchEvent({type:&quot;disconnected&quot;,data:b}),a.targetRay.visible=
!1);a.grip&&(a.grip.dispatchEvent({type:&quot;disconnected&quot;,data:b}),a.grip.visible=!1)});q.clear();a.setFramebuffer(null);a.setRenderTarget(a.getRenderTarget());y.stop();h.isPresenting=!1;h.dispatchEvent({type:&quot;sessionend&quot;})}function e(a){k=a;y.setContext(l);y.start();h.isPresenting=!0;h.dispatchEvent({type:&quot;sessionstart&quot;})}function f(a){for(var b=l.inputSources,c=0;c<r.length;c++)q.set(b[c],r[c]);for(c=0;c<a.removed.length;c++){b=a.removed[c];var d=q.get(b);d&&(d.targetRay&&d.targetRay.dispatchEvent({type:&quot;disconnected&quot;,
data:b}),d.grip&&d.grip.dispatchEvent({type:&quot;disconnected&quot;,data:b}),q.delete(b))}for(c=0;c<a.added.length;c++)if(b=a.added[c],d=q.get(b))d.targetRay&&d.targetRay.dispatchEvent({type:&quot;connected&quot;,data:b}),d.grip&&d.grip.dispatchEvent({type:&quot;connected&quot;,data:b})}function g(a,b){null===b?a.matrixWorld.copy(a.matrix):a.matrixWorld.multiplyMatrices(b.matrixWorld,a.matrix);a.matrixWorldInverse.getInverse(a.matrixWorld)}var h=this,l=null,m=1,k=null,p=&quot;local-floor&quot;,x=null,r=[],q=new Map,v=new aa;v.layers.enable(1);
v.viewport=new ka;var E=new aa;E.layers.enable(2);E.viewport=new ka;var w=new Pe([v,E]);w.layers.enable(1);w.layers.enable(2);var z=null,t=null;this.isPresenting=this.enabled=!1;this.getController=function(a){var b=r[a];void 0===b&&(b={},r[a]=b);void 0===b.targetRay&&(b.targetRay=new Mc,b.targetRay.matrixAutoUpdate=!1,b.targetRay.visible=!1);return b.targetRay};this.getControllerGrip=function(a){var b=r[a];void 0===b&&(b={},r[a]=b);void 0===b.grip&&(b.grip=new Mc,b.grip.matrixAutoUpdate=!1,b.grip.visible=
!1);return b.grip};this.setFramebufferScaleFactor=function(a){m=a;1==h.isPresenting&&console.warn(&quot;WebXRManager: Cannot change framebuffer scale while presenting VR content&quot;)};this.setReferenceSpaceType=function(a){p=a};this.getReferenceSpace=function(){return k};this.getSession=function(){return l};this.setSession=function(a){l=a;null!==l&&(l.addEventListener(&quot;select&quot;,c),l.addEventListener(&quot;selectstart&quot;,c),l.addEventListener(&quot;selectend&quot;,c),l.addEventListener(&quot;squeeze&quot;,c),l.addEventListener(&quot;squeezestart&quot;,
c),l.addEventListener(&quot;squeezeend&quot;,c),l.addEventListener(&quot;end&quot;,d),a=b.getContextAttributes(),a=new XRWebGLLayer(l,b,{antialias:a.antialias,alpha:a.alpha,depth:a.depth,stencil:a.stencil,framebufferScaleFactor:m}),l.updateRenderState({baseLayer:a}),l.requestReferenceSpace(p).then(e),l.addEventListener(&quot;inputsourceschange&quot;,f))};var U=new n,ca=new n;this.getCamera=function(a){w.near=E.near=v.near=a.near;w.far=E.far=v.far=a.far;if(z!==w.near||t!==w.far)l.updateRenderState({depthNear:w.near,depthFar:w.far}),
z=w.near,t=w.far;var b=a.parent,c=w.cameras;g(w,b);for(var d=0;d<c.length;d++)g(c[d],b);a.matrixWorld.copy(w.matrixWorld);a=a.children;d=0;for(b=a.length;d<b;d++)a[d].updateMatrixWorld(!0);U.setFromMatrixPosition(v.matrixWorld);ca.setFromMatrixPosition(E.matrixWorld);d=U.distanceTo(ca);var e=v.projectionMatrix.elements,f=E.projectionMatrix.elements,h=e[14]/(e[10]-1);a=e[14]/(e[10]+1);b=(e[9]+1)/e[5];c=(e[9]-1)/e[5];var m=(e[8]-1)/e[0],p=(f[8]+1)/f[0];f=h*m;e=h*p;p=d/(-m+p);m=p*-m;v.matrixWorld.decompose(w.position,
w.quaternion,w.scale);w.translateX(m);w.translateZ(p);w.matrixWorld.compose(w.position,w.quaternion,w.scale);w.matrixWorldInverse.getInverse(w.matrixWorld);h+=p;p=a+p;w.projectionMatrix.makePerspective(f-m,e+(d-m),b*a/p*h,c*a/p*h,h,p);return w};var B=null,y=new Ah;y.setAnimationLoop(function(b,c){x=c.getViewerPose(k);if(null!==x){var d=x.views,e=l.renderState.baseLayer;a.setFramebuffer(e.framebuffer);for(var f=0;f<d.length;f++){var g=d[f],h=e.getViewport(g),m=w.cameras[f];m.matrix.fromArray(g.transform.matrix);
m.projectionMatrix.fromArray(g.projectionMatrix);m.viewport.set(h.x,h.y,h.width,h.height);0===f&&w.matrix.copy(m.matrix)}}d=l.inputSources;for(f=0;f<r.length;f++)e=r[f],g=d[f],m=h=null,g&&(e.targetRay&&(h=c.getPose(g.targetRaySpace,k),null!==h&&(e.targetRay.matrix.fromArray(h.transform.matrix),e.targetRay.matrix.decompose(e.targetRay.position,e.targetRay.rotation,e.targetRay.scale))),e.grip&&g.gripSpace&&(m=c.getPose(g.gripSpace,k),null!==m&&(e.grip.matrix.fromArray(m.transform.matrix),e.grip.matrix.decompose(e.grip.position,
e.grip.rotation,e.grip.scale)))),e.targetRay&&(e.targetRay.visible=null!==h),e.grip&&(e.grip.visible=null!==m);B&&B(b,c)});this.setAnimationLoop=function(a){B=a};this.dispose=function(){}}function og(a){var b;function c(){ra=new zj(I);Fa=new xj(I,ra,a);!1===Fa.isWebGL2&&(ra.get(&quot;WEBGL_depth_texture&quot;),ra.get(&quot;OES_texture_float&quot;),ra.get(&quot;OES_texture_half_float&quot;),ra.get(&quot;OES_texture_half_float_linear&quot;),ra.get(&quot;OES_standard_derivatives&quot;),ra.get(&quot;OES_element_index_uint&quot;),ra.get(&quot;ANGLE_instanced_arrays&quot;));
ra.get(&quot;OES_texture_float_linear&quot;);qa=new Zh(I,ra,Fa);Y=new Ik(I,ra,Fa);Y.scissor(T.copy(ja).multiplyScalar(Q).floor());Y.viewport(Z.copy(R).multiplyScalar(Q).floor());aa=new Cj(I);X=new yk;ea=new Jk(I,ra,Y,X,Fa,qa,aa);oa=new uj(I,Fa);xa=new Aj(I,oa,aa);sa=new Fj(I,xa,oa,aa);ya=new Ej(I);ta=new xk(H,ra,Fa);wa=new Bk;va=new Hk;pa=new vj(H,Y,sa,ca);Aa=new wj(I,ra,aa,Fa);Ba=new Bj(I,ra,aa,Fa);aa.programs=ta.programs;H.capabilities=Fa;H.extensions=ra;H.properties=X;H.renderLists=wa;H.state=Y;H.info=aa}
function d(a){a.preventDefault();console.log(&quot;THREE.WebGLRenderer: Context Lost.&quot;);G=!0}function e(){console.log(&quot;THREE.WebGLRenderer: Context Restored.&quot;);G=!1;c()}function f(a){a=a.target;a.removeEventListener(&quot;dispose&quot;,f);g(a);X.remove(a)}function g(a){var b=X.get(a).program;a.program=void 0;void 0!==b&&ta.releaseProgram(b)}function h(a,b){a.render(function(a){H.renderBufferImmediate(a,b)})}function l(a,b,c,d){if(!1!==a.visible){if(a.layers.test(b.layers))if(a.isGroup)c=a.renderOrder;else if(a.isLOD)!0===
a.autoUpdate&&a.update(b);else if(a.isLight)A.pushLight(a),a.castShadow&&A.pushShadow(a);else if(a.isSprite){if(!a.frustumCulled||lg.intersectsSprite(a)){d&&Hb.setFromMatrixPosition(a.matrixWorld).applyMatrix4(Md);var e=sa.update(a),f=a.material;f.visible&&C.push(a,e,f,c,Hb.z,null)}}else if(a.isImmediateRenderObject)d&&Hb.setFromMatrixPosition(a.matrixWorld).applyMatrix4(Md),C.push(a,null,a.material,c,Hb.z,null);else if(a.isMesh||a.isLine||a.isPoints)if(a.isSkinnedMesh&&a.skeleton.frame!==aa.render.frame&&
(a.skeleton.update(),a.skeleton.frame=aa.render.frame),!a.frustumCulled||lg.intersectsObject(a))if(d&&Hb.setFromMatrixPosition(a.matrixWorld).applyMatrix4(Md),e=sa.update(a),f=a.material,Array.isArray(f))for(var g=e.groups,h=0,m=g.length;h<m;h++){var p=g[h],k=f[p.materialIndex];k&&k.visible&&C.push(a,e,k,c,Hb.z,p)}else f.visible&&C.push(a,e,f,c,Hb.z,null);a=a.children;h=0;for(m=a.length;h<m;h++)l(a[h],b,c,d)}}function m(a,b,c,d){for(var e=0,f=a.length;e<f;e++){var g=a[e],h=g.object,l=g.geometry,m=
void 0===d?g.material:d;g=g.group;if(c.isArrayCamera){ba=c;for(var p=c.cameras,u=0,q=p.length;u<q;u++){var x=p[u];h.layers.test(x.layers)&&(Y.viewport(Z.copy(x.viewport)),A.setupLights(x),k(h,b,x,l,m,g))}}else ba=null,k(h,b,c,l,m,g)}}function k(a,c,d,e,f,g){a.onBeforeRender(H,c,d,e,f,g);A=va.get(c,ba||d);a.modelViewMatrix.multiplyMatrices(d.matrixWorldInverse,a.matrixWorld);a.normalMatrix.getNormalMatrix(a.modelViewMatrix);if(a.isImmediateRenderObject){var l=x(d,c,f,a);Y.setMaterial(f);fa=b=null;
Oe=!1;h(a,l)}else H.renderBufferDirect(d,c,e,f,a,g);a.onAfterRender(H,c,d,e,f,g);A=va.get(c,ba||d)}function p(a,b,c){var d=X.get(a),e=A.state.lights,h=e.state.version;c=ta.getParameters(a,e.state,A.state.shadowsArray,b,Ua.numPlanes,Ua.numIntersection,c);var l=ta.getProgramCacheKey(c),m=d.program,p=!0;if(void 0===m)a.addEventListener(&quot;dispose&quot;,f);else if(m.cacheKey!==l)g(a);else{if(d.lightsStateVersion!==h)d.lightsStateVersion=h;else if(void 0!==c.shaderID)return;p=!1}p&&(m=ta.acquireProgram(c,l),
d.program=m,d.uniforms=c.uniforms,d.environment=a.isMeshStandardMaterial?b.environment:null,d.outputEncoding=H.outputEncoding,a.program=m);c=m.getAttributes();if(a.morphTargets)for(l=a.numSupportedMorphTargets=0;l<H.maxMorphTargets;l++)0<=c[&quot;morphTarget&quot;+l]&&a.numSupportedMorphTargets++;if(a.morphNormals)for(l=a.numSupportedMorphNormals=0;l<H.maxMorphNormals;l++)0<=c[&quot;morphNormal&quot;+l]&&a.numSupportedMorphNormals++;c=d.uniforms;if(!a.isShaderMaterial&&!a.isRawShaderMaterial||!0===a.clipping)d.numClippingPlanes=
Ua.numPlanes,d.numIntersection=Ua.numIntersection,c.clippingPlanes=Ua.uniform;d.fog=b.fog;d.needsLights=a.isMeshLambertMaterial||a.isMeshToonMaterial||a.isMeshPhongMaterial||a.isMeshStandardMaterial||a.isShadowMaterial||a.isShaderMaterial&&!0===a.lights;d.lightsStateVersion=h;d.needsLights&&(c.ambientLightColor.value=e.state.ambient,c.lightProbe.value=e.state.probe,c.directionalLights.value=e.state.directional,c.directionalLightShadows.value=e.state.directionalShadow,c.spotLights.value=e.state.spot,
c.spotLightShadows.value=e.state.spotShadow,c.rectAreaLights.value=e.state.rectArea,c.pointLights.value=e.state.point,c.pointLightShadows.value=e.state.pointShadow,c.hemisphereLights.value=e.state.hemi,c.directionalShadowMap.value=e.state.directionalShadowMap,c.directionalShadowMatrix.value=e.state.directionalShadowMatrix,c.spotShadowMap.value=e.state.spotShadowMap,c.spotShadowMatrix.value=e.state.spotShadowMatrix,c.pointShadowMap.value=e.state.pointShadowMap,c.pointShadowMatrix.value=e.state.pointShadowMatrix);
a=d.program.getUniforms();a=Eb.seqWithValue(a.seq,c);d.uniformsList=a}function x(a,b,c,d){ea.resetTextureUnits();var e=b.fog,f=c.isMeshStandardMaterial?b.environment:null,g=X.get(c),h=A.state.lights;na&&(mg||a!==V)&&Ua.setState(c.clippingPlanes,c.clipIntersection,c.clipShadows,a,g,a===V&&c.id===Ne);c.version===g.__version?void 0===g.program?p(c,b,d):c.fog&&g.fog!==e?p(c,b,d):g.environment!==f?p(c,b,d):g.needsLights&&g.lightsStateVersion!==h.state.version?p(c,b,d):void 0===g.numClippingPlanes||g.numClippingPlanes===
Ua.numPlanes&&g.numIntersection===Ua.numIntersection?g.outputEncoding!==H.outputEncoding&&p(c,b,d):p(c,b,d):(p(c,b,d),g.__version=c.version);var l=!1,m=!1,k=!1;b=g.program;h=b.getUniforms();var u=g.uniforms;Y.useProgram(b.program)&&(k=m=l=!0);c.id!==Ne&&(Ne=c.id,m=!0);if(l||V!==a){h.setValue(I,&quot;projectionMatrix&quot;,a.projectionMatrix);Fa.logarithmicDepthBuffer&&h.setValue(I,&quot;logDepthBufFC&quot;,2/(Math.log(a.far+1)/Math.LN2));V!==a&&(V=a,k=m=!0);if(c.isShaderMaterial||c.isMeshPhongMaterial||c.isMeshToonMaterial||
c.isMeshStandardMaterial||c.envMap)l=h.map.cameraPosition,void 0!==l&&l.setValue(I,Hb.setFromMatrixPosition(a.matrixWorld));(c.isMeshPhongMaterial||c.isMeshToonMaterial||c.isMeshLambertMaterial||c.isMeshBasicMaterial||c.isMeshStandardMaterial||c.isShaderMaterial)&&h.setValue(I,&quot;isOrthographic&quot;,!0===a.isOrthographicCamera);(c.isMeshPhongMaterial||c.isMeshToonMaterial||c.isMeshLambertMaterial||c.isMeshBasicMaterial||c.isMeshStandardMaterial||c.isShaderMaterial||c.skinning)&&h.setValue(I,&quot;viewMatrix&quot;,
a.matrixWorldInverse)}if(c.skinning&&(h.setOptional(I,d,&quot;bindMatrix&quot;),h.setOptional(I,d,&quot;bindMatrixInverse&quot;),a=d.skeleton))if(l=a.bones,Fa.floatVertexTextures){if(void 0===a.boneTexture){l=Math.sqrt(4*l.length);l=L.ceilPowerOfTwo(l);l=Math.max(l,4);var x=new Float32Array(l*l*4);x.set(a.boneMatrices);var v=new ac(x,l,l,1023,1015);a.boneMatrices=x;a.boneTexture=v;a.boneTextureSize=l}