/*
 * Copyright 2015-2021 the original author or authors.
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v2.0 which
 * accompanies this distribution and is available at
 *
 * https://www.eclipse.org/legal/epl-v20.html
 */

package org.junit.jupiter.engine.support;

import java.util.function.Predicate;
import java.util.function.Supplier;

import org.junit.platform.commons.logging.Logger;
import org.junit.platform.commons.logging.LoggerFactory;
import org.junit.platform.commons.util.ReflectionUtils;
import org.junit.platform.commons.util.UnrecoverableExceptions;
import org.junit.platform.engine.support.hierarchical.ThrowableCollector;
import org.opentest4j.TestAbortedException;

/**
 * Specialization of {@link ThrowableCollector} that treats instances of the
 * OTA's {@link org.opentest4j.TestAbortedException} and JUnit 4's
 * {@code org.junit.AssumptionViolatedException} as <em>aborting</em>.
 *
 * @since 5.4
 * @see ThrowableCollector
 * @see org.junit.platform.engine.support.hierarchical.OpenTest4JAwareThrowableCollector
 */
class OpenTest4JAndJUnit4AwareThrowableCollector extends ThrowableCollector {

	private static final Logger logger = LoggerFactory.getLogger(OpenTest4JAndJUnit4AwareThrowableCollector.class);

	private static final String ASSUMPTION_VIOLATED_EXCEPTION = "org.junit.internal.AssumptionViolatedException";

	private static final String COMMON_FAILURE_MESSAGE = "Failed to load class " + ASSUMPTION_VIOLATED_EXCEPTION
			+ ": only supporting " + TestAbortedException.class.getName() + " for aborted execution.";

	private static final Predicate<? super Throwable> abortedExecutionPredicate = createAbortedExecutionPredicate();

	OpenTest4JAndJUnit4AwareThrowableCollector() {
		super(abortedExecutionPredicate);
	}

	private static Predicate<? super Throwable> createAbortedExecutionPredicate() {
		Predicate<Throwable> otaPredicate = TestAbortedException.class::isInstance;

		// Additionally support JUnit 4's AssumptionViolatedException?
		try {
			Class<?> clazz = ReflectionUtils.tryToLoadClass(ASSUMPTION_VIOLATED_EXCEPTION).get();
			if (clazz != null) {
				return otaPredicate.or(clazz::isInstance);
			}
		}
		catch (Throwable throwable) {
			UnrecoverableExceptions.rethrowIfUnrecoverable(throwable);
			Supplier<String> messageSupplier = (throwable instanceof NoClassDefFoundError)
					? () -> COMMON_FAILURE_MESSAGE + " Note that " + ASSUMPTION_VIOLATED_EXCEPTION
							+ " requires that Hamcrest is on the classpath."
					: () -> COMMON_FAILURE_MESSAGE;
			logger.debug(throwable, messageSupplier);
		}

		// Else just OTA's TestAbortedException
		return otaPredicate;
	}

}
